########PKG_HANDLER_SCRIPT_BEGIN#########
#!/bin/bash

# the first and the last lines are markers for SWM component to be able to recognize this script in the stream
# DO NOT REMOVE THEM unless you change SWM accordingly

VERSION="1.18"
#CHANGELOG: https://confluence.dev.global-intra.net:8443/display/OS4KD/pkg.sh+release+page


# inner PP/RLC  package script which provides interface for Software management application
# supported requests
# --get_name              # print package name: OpenScape4000
# --get_type_structure    # print package structure type: PP|RLC
# --get_type              # print package type: Major|Minor|Fix|RMX-HF release
# --get_version           # print overall package verstion
# --get_script_version    # print script version
# --get_complete_version  # print inner versions
# --is_transferable       # evaluate if package is transferable to specific system version
# --beam_to_slice         # do beam to slice

# source for the data:
#  - location entered with --location
#  - otherwise dirname ARGV[0]


OPTS=`getopt -o '' --long get_name,get_type_structure,get_type,get_version,get_complete_version,get_script_version,get_progress,get_flags,is_transferable,beam_to_slice,location: -n "${0##*/}" -- "$@"`
if [ $? -ne 0 ]
then
  echo "Error: Unrecognized option $@"
  exit 1
fi

eval set -- "$OPTS"

VERSION_FILE=version.info
BLACKLIST_FILE=blacklist.info
PKG_FILE=pkg.info
PKG_SCRIPT=pkg.sh
PKG_METADATA="$VERSION_FILE $BLACKLIST_FILE $PKG_FILE $PKG_SCRIPT"
PKG_SCRIPT_DEFAULT=/opt/swm/bin/$PKG_SCRIPT

BEAM_TO_SLICE_PROGRESS_SUF=pkg.sh.beam_to_slice.progress
PROGRESS_LIST="$BEAM_TO_SLICE_PROGRESS_SUF"

VERSION_FILE_BEGIN="version:start"
VERSION_FILE_END="version:end"
BLACKLIST_FILE_BEGIN="blacklist:start"
BLACKLIST_FILE_END="blacklist:end"

SLICE_PP=/.AS/SWS/PP
SLICE_RLC=/.AS/SWS/RLC
SLICE_AS_HF=/var/hf
SLICE_CSTA_HF=/.AS/SWS/csta_hf
SLICE_PLT_HF=/.AS/SWS/os4kplt_hf
SLICE_LW_HF=/.AS/SWS/LW_HF

DATA_DIR=${0%/*}

IS_HF='n'
ACT_FLAGS="FLAGS:"

# define standard binaries (Linux)
SED=/usr/bin/sed
CAT=/bin/cat
CHMOD=/bin/chmod
CHOWN=/bin/chown
ECHO=/bin/echo
EXPR=/usr/bin/expr
GREP=/usr/bin/grep
LS=/bin/ls
MKDIR=/bin/mkdir
MV="/bin/mv -v"
RM=/bin/rm
SORT=/usr/bin/sort
FIND=/usr/bin/find
TAR=/bin/tar
FILE=/usr/bin/file
PV=/usr/bin/pv
USE_PV="n"
UUDECODE=/usr/bin/uudecode
GUNZIP=/usr/bin/gunzip

UNZIP=/usr/bin/unzip
TEE=/usr/bin/tee
GETHINF0=/opt/bin/gethinfo

# rate_limit 15MB/s
RATE_LIMIT=15000000

TMP_DIR=/var/tmp

# print type of package
get_type_structure () {
  # find the $PKG_FILE in $DATA_DIR
  PKG_INFO=`$FIND $DATA_DIR -type f -name $PKG_FILE -print`
  [ -z "$PKG_INFO" ] && echo "$PKG_FILE doesn't exist" >&2 && return

  TYPE=`grep "^TYPE:" $PKG_INFO | cut -f2 -d':'`
  echo "$TYPE" | grep -e '-HF\s*$' > /dev/null && IS_HF='y'
  echo -n $TYPE
}

# print overall package version
# fill in 
get_pkg_version () {
  get_type_structure >/dev/null
  # set the search key
  if [ "$PLT" = 'AS-L' ]
  then
    # check the package type on Assistant, can be RMX-HF, CSTA-HF, PLT-HF, Assist-HF
    case "$TYPE" in
        'RMX-HF')  VER_KEY="RMX" ;; 
        'AS-HF')   VER_KEY="Assistant" ;;
        'PLT-HF')  VER_KEY="Platform" ;;
        'CSTA-HF') VER_KEY="CSTA" ;;
        'LW-HF')   VER_KEY="LW" ;;
        *)         VER_KEY="Platform" ;; # PLT by default on Assistant
    esac
  #OSFOURK-20074
  elif [ "$PLT" = 'AS-N' ]
  then
    # check the package type on Manager, can be PLT-HF, MGR-HF
    case "$TYPE" in
        'MGR-HF')   VER_KEY="Manager" ;;
        'PLT-HF')  VER_KEY="Platform" ;;
        *)         VER_KEY="Platform" ;; # PLT by default on Manager
    esac
  #-OSFOURK-20074
  fi  
  PKG_VERSION=`$FIND $DATA_DIR -type f -name $VERSION_FILE -exec grep "$VER_KEY:" '{}' \; | cut -f2 -d':'`
  echo -n "$PKG_VERSION"

  PKG_VERSION=`echo "$PKG_VERSION" | tr [a-z] [A-Z]`
  PKG_MAJOR=`echo $PKG_VERSION | awk 'BEGIN {FS="[_ ]"} { print $1 }' | tr -d "V"`
  PKG_MINOR=`echo $PKG_VERSION | awk 'BEGIN {FS="[_. ]"} { print $2 }' | tr -d "R"`
  PKG_FIX=`echo $PKG_VERSION | awk 'BEGIN {FS="[_. ]"} { print $3 }'`
  PKG_HF=`echo $PKG_VERSION | awk 'BEGIN {FS="[_. ]"} { print $4 }'`
}

# fill in INSTALL_VERSION, INSTALL_MAJOR, INSTALL_MINOR, INSTALL_FIX
get_installed_version() {
  if [ -z "$1" ]; then
    get_type_structure
  fi
  if [ "$PLT" = 'AS-L' ]
  then
    # check the package type on Assistant, can be RMX-HF, CSTA-HF, PLT-HF, Assist-HF
    case "$TYPE" in
        'RMX-HF')  SYS_VER_KEY="RMXVERSION" ;; 
        'AS-HF')   SYS_VER_KEY="A&SVERSION" ;;
        'PLT-HF')  SYS_VER_KEY="OS4KVERSION" ;;
        'CSTA-HF') SYS_VER_KEY="CSTAVERSION" ;;
        'LW-HF')   SYS_VER_KEY="LWVERSION" ;;
        *)         SYS_VER_KEY="OS4KVERSION" ;; # PLT by default on Assistant
    esac
  #OSFOURK-20074
  elif [ "$PLT" = 'AS-N' ]
  then
    # check the package type on Manager, can be PLT-HF, MGR-HF
    case "$TYPE" in
        'MGR-HF')   SYS_VER_KEY="A&SVERSION" ;;
        'PLT-HF')  SYS_VER_KEY="OS4KVERSION" ;;
        *)         SYS_VER_KEY="OS4KVERSION" ;; # PLT by default on Manager
    esac
  #-OSFOURK-20074
  fi  

  # check $INSTALL_VERSION variable, can be exported from outside
  if [ -z "$INSTALL_VERSION" ]
  then  
    INSTALL_VERSION=`sudo -n /opt/bin/getosversion -s -c | grep $SYS_VER_KEY | cut -f2 -d'='`
  fi  

  #upparcase version
  INSTALL_VERSION=`echo "$INSTALL_VERSION" | tr [a-z] [A-Z]`

  #parsing installed version to major, minor and fix versions
  INSTALL_MAJOR=`echo $INSTALL_VERSION | awk 'BEGIN {FS="[_. ]"} { print $1 }' | tr -d "V"`
  INSTALL_MINOR=`echo $INSTALL_VERSION | awk 'BEGIN {FS="[_. ]"} { print $2 }' | tr -d "R"`
  INSTALL_FIX=`echo $INSTALL_VERSION | awk 'BEGIN {FS="[_. ]"} { print $3 }'`
  INSTALL_HF=`echo $INSTALL_VERSION | awk 'BEGIN {FS="[_. ]"} { print $4 }'`
  #echo $INSTALL_MAJOR
  #echo $INSTALL_MINOR
  #echo $INSTALL_FIX
}

# print type of package - Major, Minor or Fix release
get_type () {
  get_pkg_version >&2 

  if [ $IS_HF = 'y' ]
  then  
    # it's HF type,
    case "$TYPE" in
        'RMX-HF')  echo -n "RMX" ;;
        'AS-HF')   echo -n "Assistant" ;;
        'PLT-HF')  echo -n "Platform" ;;
        'CSTA-HF') echo -n "CSTA" ;;
        'LW-HF')   echo -n "Loadware" ;;
        'MGR-HF')  echo -n "Manager" ;;
    esac
    echo -n " Hotfix"
    return
  fi
  
  get_installed_version >&2
  #comparing major versions
  if [ $PKG_MAJOR -gt $INSTALL_MAJOR ]; then
    echo -n "Major Release"
  else
    #comparing minor versions
    if [ $PKG_MINOR -gt $INSTALL_MINOR ]; then
      echo -n "Minor Release"
    else
      if [ $PKG_FIX -gt $INSTALL_FIX ]; then
        echo -n "Fix Release"
      else
        echo -n "Fix Release"
      fi
    fi
  fi
}

#find the progress file and print the progress value to stdout
get_progress () {
  for pr_file in $PROGRESS_LIST
  do  
    # find the first non empty progress file
    progress_file=`$FIND $TMP_DIR -type f -name "*$pr_file" -print | head -1`
    [ -n "$progress_file" ] && [ -s $progress_file ] && break
  done  
  [ -n "$progress_file" ] && $CAT $progress_file
}

#evaluate if package is transferable to specific system version
is_transferable () {
  # inner definition
  set_flags() {
    # is it activated (e.g. the version is the same
    [ $PKG_MAJOR -eq $INSTALL_MAJOR -a $PKG_MINOR -eq $INSTALL_MINOR -a $PKG_FIX -eq $INSTALL_FIX -a $PKG_HF -eq $INSTALL_HF ] && ACT_FLAGS="$ACT_FLAGS ACTOK"
    # is it older?
    if [ $PKG_MAJOR -lt $INSTALL_MAJOR ] 
    then
       ACT_FLAGS="$ACT_FLAGS OLDER"
    elif [ $PKG_MAJOR -eq $INSTALL_MAJOR -a $PKG_MINOR -lt $INSTALL_MINOR ]
    then
      ACT_FLAGS="$ACT_FLAGS OLDER"
    elif [ $PKG_MAJOR -eq $INSTALL_MAJOR -a $PKG_MINOR -eq $INSTALL_MINOR -a $PKG_FIX -lt $INSTALL_FIX ]
    then
      ACT_FLAGS="$ACT_FLAGS OLDER"
    elif [ $PKG_MAJOR -eq $INSTALL_MAJOR -a $PKG_MINOR -eq $INSTALL_MINOR -a $PKG_FIX -eq $INSTALL_FIX -a $PKG_HF -lt $INSTALL_HF ]
    then
      ACT_FLAGS="$ACT_FLAGS OLDER"
    fi  
  }

  get_pkg_version >&2
  [ -n "$INSTALL_VERSION" ] && IV_FLAG="y"
  get_installed_version >&2

  if [ $IS_HF = 'y' ]
  then
    # it's HF
    if [ $PKG_MAJOR -eq $INSTALL_MAJOR -a $PKG_MINOR -eq $INSTALL_MINOR -a $PKG_FIX -eq $INSTALL_FIX ]
    then
	[ $PKG_HF -gt $INSTALL_HF ] && echo -n true && return
	[ $PKG_HF -le $INSTALL_HF ] && echo -n false && return
    fi
    # HF for not installed RLC, test if it's transferred (only when INSTALL_VERSION wasn't set directly)
    if [ -z "$IV_FLAG" ]; then
      # Only for LW-HF
      if echo "$TYPE" | grep -e '^\s*LW-HF\s*$' -e '^\s*CSTA-HF\s*$' -e '^\s*AS-HF\s*$' -e '^\s*PLT-HF\s*$' -e '^\s*RMX-HF\s*$' -e '^\s*MGR-HF\s*$' > /dev/null; then
        # Activate LW-HF handling in swa for older swa versions
        touch /tmp/SWA_LW_HF_ENABLED >/dev/null 2>&1
        for INSTALL_VERSION in `sudo /opt/swa/bin/swa -p 2>/dev/null |grep RLC |sed 's/^.*RMX=//;s/\s.*//'`; do
          get_installed_version no_print_hf >&2
          [ $PKG_MAJOR -eq $INSTALL_MAJOR -a $PKG_MINOR -eq $INSTALL_MINOR -a $PKG_FIX -eq $INSTALL_FIX -a $PKG_HF -gt $INSTALL_HF ] && echo -n true && return
        done
      fi
    fi
    set_flags
    echo -n false
    return
  fi

  # comparing major versions
  if [ $PKG_MAJOR -gt $INSTALL_MAJOR ]; then
    echo -n true
  else
    # comparing minor versions
    if [ $PKG_MAJOR -eq $INSTALL_MAJOR -a $PKG_MINOR -gt $INSTALL_MINOR ]; then
      echo -n true
    else
      if [ $PKG_MAJOR -eq $INSTALL_MAJOR -a $PKG_MINOR -eq $INSTALL_MINOR -a $PKG_FIX -gt $INSTALL_FIX ]; then
        echo -n true
      else
        set_flags
        echo -n false
      fi
    fi
  fi

  #echo $INSTALL_VERSION
  #echo $PKG_VERSION
}

# extract and install pv if it's not available yet
check_and_install_pv () {
	if ! $PV -V >/dev/null 2>&1; then
		PV=/opt/swm/bin/pv
		if ! $PV -V >/dev/null 2>&1; then
			sed '1,/^-----BEGIN PV-----/d;/^-----END PV-----/,$d' $0 | $UUDECODE -o - |$GUNZIP -c |$TAR xP 2>/dev/null
			[[ $? -eq 0 ]] || USE_PV="n"
		fi
	fi
}

# unzip package file by file with pv rate limiting
# $1 = file_to_unzip, $2 = directory_where_to_extract
# $3-# = files_which_have_to_be_extracted
unzip_pv () {
	ZIPFILE="$1"
	shift
	OUTDIR="$1"
	shift

	RC=0
	for FILE in `$UNZIP -Z1 "$ZIPFILE"`; do
	        if [[ $FILE =~ \/$ ]]; then
	                $MKDIR -p "${OUTDIR}/${FILE}"
		fi
	done
	for FILE in `$UNZIP -Z1 "$ZIPFILE"`; do
	        if ! [[ $FILE =~ \/$ ]]; then
	                $UNZIP -p "$ZIPFILE" "${FILE}" | $PV -q -L $RATE_LIMIT > "${OUTDIR}/${FILE}"
	        fi
		RC=`$EXPR $RC + $?`
	done

	return $RC
}

# cat with pv limiting
# $1 = file to cat
cat_pv () {
	$CAT "$1" | $PV -q -L $RATE_LIMIT
}

# beam_to_slice function
beam_to_slice () {

  pack_comp () { ####-> pack files for one component  #############################

    IDX=1

    $RM -f $1

    while [ -f $1.$IDX.part ]
      do
      if [ "$USE_PV" = 'y' ]; then
        cat_pv $1.$IDX.part >> $1
      else
        $CAT $1.$IDX.part >> $1
      fi
      echo "Merge $1.$IDX.part to $1"
      $RM -f $1.$IDX.part
      IDX=`$EXPR $IDX + 1`
    done

  }  ####-----------------------------------------------------------------------####

  extract_comp () { ####-> extract component to original file-structer ############

    if [ "$USE_PV" = 'y' ]; then
      unzip_pv $1 $DEST_DIR
    else
      $UNZIP $1 -d $DEST_DIR
    fi
    ERG=$?
    if [ $ERG -ne 0 ]
      then
      #$ECHO "072 Unable to unzip files/directories rc=$ERG" | $TEE -a $LOGFILE

      #$CAT /tmp/uriapi$$/prot
      #$CAT /tmp/uriapi$$/error
      #uri_start_exception 72
      exit -1
    fi

    $RM -f $1

  } ####-----------------------------------------------------------------------####
  howmany() { ####-> return number of parameters - used as number of items in the list
    echo $#; 
  } ####-----------------------------------------------------------------------####
  # remove obsolete progress files
  echo "Removing obsolete temporary progress file(s)"
  $FIND $TMP_DIR -type f -name "*$BEAM_TO_SLICE_PROGRESS_SUF" -print -delete
  echo "Creating new progress file"
  # create temporary progress file
  progress_file=`mktemp --suffix=$BEAM_TO_SLICE_PROGRESS_SUF --tmpdir=$TMP_DIR`
  echo 1 > $progress_file
  TYPE=$(get_type_structure)
  if [ "$TYPE" = "RLC" ]
  then  
    # clean and create SLICE structure
    $RM -fr $SLICE/*
    [ -f /opt/swm/bin/removePackage ] && sudo /opt/swm/bin/removePackage RLC
    $MKDIR -p $SLICE/rmx
    $MKDIR -p $SLICE/rmx/partfiles
    RLC_PDS_FINAL_PATH="$SLICE/rmx/partfiles"
    mkdir -p $RLC_PDS_FINAL_PATH
    
    # RMX area merge and store
    RMXCOMPONENTS=`$FIND $DATA_DIR -name "*-*.e.zip*" | $SED 's/\(.*\)\..*\.part$/\1/' | $SORT -du`
    echo "RMX components - $RMXCOMPONENTS" 
    for COMP in $RMXCOMPONENTS
      do
      pack_comp $COMP
      echo "Move $COMP.zip to $RLC_PDS_FINAL_PATH" 
      $MV $COMP $RLC_PDS_FINAL_PATH
    done
    echo 15 > $progress_file
  elif [ "$TYPE" = "PP" -o "$TYPE" = "RMX-HF" ]
  then
    $RM -fr $SLICE/*
  else
    echo "Unknown package type, don't know what to do with $TYPE"
    rm -vf $progress_file
    exit 1
  fi  

  # common part - process Assistnat/Manager data
  COMPONENTS=`$FIND $DATA_DIR -name "*.zip*" | $SED 's/\(.*\)\..*\.part$/\1/' | $SORT -du`
  echo "Components - $COMPONENTS" 
  # check how many components we have and set the progress step
  components_count=`howmany $COMPONENTS`
  [ "$components_count" -eq "0" ] && components_count=1  # avoid division by 0
  curr_progress=`cat $progress_file`
  [ -z "$curr_progress" ] && curr_progress=0  # set 0 by default
  # OK, let's split the remaining progress, we should keep 5% for the last task --> result = (95 - $curr_progress) / $components_count
  #                                                                             --> result = result / 2  --> 1. pack_comp, 2. extract_comp
  progress_step=$(((95 - $curr_progress)/$components_count/2))
  for COMP in $COMPONENTS
  do
    DEST_DIR=$SLICE
    # special handling needed for PP, create dirname of the zip
    # we need to get:
    #   5000 from  OpenScape4000/hhd300/pp/p30252n5000bh0103/5000/*.zip.*
    #    uw7 from  OpenScape4000/hhd300/pp/p30252n5000bh0103/uw7/*.zip.*
    # script from  OpenScape4000/hhd300/pp/p30252n5000bh0103/script.zip.*

    # OpenScape4000/hhd300/pp/p30252n5000bh0103/5000/(*).zip --> /.AS/SWS/PP/5000/\1
    [ $SLICE = $SLICE_PP ] && parent_dir=$COMP && sub_dir=${parent_dir##*/} && parent_dir=${parent_dir%/*} && DEST_DIR="$SLICE/${parent_dir##*/}/${sub_dir%.*}"
    # OpenScape4000/hhd300/pp/p30252n5000bh0103/uw7/*.zip --> /.AS/SWS/PP/uw7
    [ $SLICE = $SLICE_PP ] && parent_dir=${COMP%/*} && [ ${parent_dir##*/} = "uw7" ] && DEST_DIR="$SLICE/${parent_dir##*/}"
    # special handling for script
    [ $SLICE = $SLICE_PP ] && cmp_name=$COMP && [ ${cmp_name##*/} = "script.zip" ]  && DEST_DIR="$SLICE/script"
    mkdir -pv $DEST_DIR
    pack_comp $COMP
    curr_progress=$(($curr_progress + $progress_step))
    [ "$curr_progress" -gt 95 ] && curr_progress=95  # no more than 95
    echo $curr_progress > $progress_file
    echo "Unzip $COMP to $DEST_DIR" 
    extract_comp $COMP
    curr_progress=$(($curr_progress + $progress_step))
    [ "$curr_progress" -gt 95 ] && curr_progress=95  # no more than 95
    echo $curr_progress > $progress_file
  done

  # common part - process metadata
  for meta_file in $PKG_METADATA
  do
    $FIND $DATA_DIR -name $meta_file -exec cp -v '{}' $SLICE \;
  done

  # process pkg info as the latest step
  if [ "$TYPE" = "RLC" ]
  then  
    # Find info file location (hispa calls pkg.sh with different location than SWM)
    INFOFILE=`$FIND $DATA_DIR -name "hd300-*-*"`
    if [[ -z "$INFOFILE" ]]; then
      INFOFILE=`$FIND $DATA_DIR/../stored-kuhd -name "hd300-*-*"`
    fi
    if [[ -z "$INFOFILE" ]]; then
      INFOFILE=`$FIND $DATA_DIR/../../stored-kuhd -name "hd300-*-*"`
    fi
    if [[ -z "$INFOFILE" ]]; then
      # Failback - default dir from SWT
      INFOFILE=`$FIND /.AS/SWS/hfmswt/hhd300/stored-kuhd -name "hd300-*-*"`
    fi
    if [[ -z "$INFOFILE" ]]; then
      echo "ERROR: INFOFILE doesn't exists!!!"
    else
      echo "Moving INFOFILE $INFOFILE to destination dir"
      mv -v "$INFOFILE" $SLICE/rmx/
    fi
  elif [ "$TYPE" = "PP" -o "$TYPE" = "RMX-HF" ]  
  then
    INFOFILE=`$FIND $DATA_DIR -name "*.inf"`
    if [[ -z "$INFOFILE" ]]; then
      INFOFILE=`$FIND $DATA_DIR/.. -name "*.inf"`
    fi
    if [[ -z "$INFOFILE" ]]; then
      echo "ERROR: INFOFILE doesn't exists!!!"
    else
      mv -v "$INFOFILE" $SLICE/
    fi
  fi

  $CHOWN -R swm:unity $SLICE
  echo "Change own to swm:unity for everything under $SLICE"
  $CHMOD -R 770 $SLICE
  echo "Change rights to 770 for everything under $SLICE"

  # dummy swa -p execution due is_preparation_needed must be run for some platform initialization steps in combination with possible HF
  if [ "$TYPE" = "RLC" ]; then
    sudo /opt/swa/bin/swa -p >/dev/null 2>&1
  fi

  rm -vf $progress_file  # done
}

#
# Main main MAIN
#

# if we are calling from non default location
#   use the latest available version of itself
#   --get_script_option should not be escalated
echo "$@" | grep -- "--get_script_version" >/dev/null
if [ $? -ne 0 ]
then  
  if [ -z "$SUPRESS_VERSION_CHECK" -a "`readlink -e $0`" != "$PKG_SCRIPT_DEFAULT" ]
  then
    echo "Comparing script versions" >&2
    default_ver=`$PKG_SCRIPT_DEFAULT --get_script_version`
    if [ $? -eq 0 ]
    then  
      # use the latest available
      ver_major=`echo $VERSION | cut -f1 -d'.'`
      ver_minor=`echo $VERSION | cut -f2 -d'.'`
      def_major=`echo $default_ver | cut -f1 -d'.'`
      def_minor=`echo $default_ver | cut -f2 -d'.'`
      use_default=0
      if [ $def_major -gt $ver_major ] 
      then
        use_default=1
      elif [ $def_major -eq $ver_major -a $def_minor -gt $ver_minor ]
      then
        use_default=1
      fi

      if [ $use_default -eq 1 ]
      then
        echo "Using default package script ($PKG_SCRIPT_DEFAULT), it's newer" >&2
        exec $PKG_SCRIPT_DEFAULT --location $DATA_DIR $@
      fi
    fi  
  fi
fi
# process the parameters
echo "Call:$$: $@" >&2
PARAM_LIST=$*

while true
do
  case "$1" in
    --location) DATA_DIR=$2; shift 2;;
    --beam_to_slice) BEAM_TO_SLICE='y'; shift ;;
    --get_version) GET_VERSION='y'; shift;; 
    --get_script_version) echo $VERSION; exit 0;;
    --is_transferable) IS_TRANSFERABLE='y'; shift;;
    --get_type) GET_TYPE='y'; shift;;
    --get_type_structure) GET_TYPE_STRUCTURE='y'; shift;;
    --get_progress) GET_PROGRESS='y'; shift;;
    --get_flags) GET_FLAGS='y'; shift;;
    --) shift; break;;
    *) break ;;
  esac
done  

echo "Data Location:$DATA_DIR" >&2

# get the running platform (Assistant or Manager)
PLT=`$GETHINF0 -a`

# if this is a manager system, check if by mistake the folders /.AS/SWS/csta_hf, /.AS/SWS/as_hf have something inside - in this case -> empty the folders
if [ "$PLT" = 'AS-N' ]
then
    #OSFOURK-20074
    #PLT_HF_FOLDER_SIZE=`du -sm /.AS/SWS/os4kplt_hf | cut -f1`
    #if [ $PLT_HF_FOLDER_SIZE -gt 0 ]
    #then
	  #  rm /.AS/SWS/os4kplt_hf/*
    #fi
		#-OSFOURK-20074  
    CSTA_HF_FOLDER_SIZE=`du -sm /.AS/SWS/csta_hf | cut -f1`
    if [ $CSTA_HF_FOLDER_SIZE -gt 0 ]
    then
	    rm /.AS/SWS/csta_hf/* 2>/dev/null
    fi					

    ASSIST_HF_FOLDER_SIZE=`du -sm /.AS/SWS/as_hf | cut -f1`
    if [ $ASSIST_HF_FOLDER_SIZE -gt 0 ]
    then
	    rm /.AS/SWS/as_hf/* 2>/dev/null
    fi
fi  

# install pv for AS-L if it's not available on some older system
#OSFOURK-21775 - instal pv also on MGR - otherwise the unzip command executed without pv will return "2" causing the script to stop -> "Building activation package failed"
#if [ "$PLT" = 'AS-L' ]
#then
  USE_PV="y"
  check_and_install_pv
#fi
#-OSFOURK-21775
if [ "$GET_VERSION" = 'y' ]
then
  get_pkg_version
fi

if [ "$GET_TYPE_STRUCTURE" = 'y' ]
then
  get_type_structure
fi

if [ "$GET_TYPE" = 'y' ]
then
  get_type
fi

if [ "$IS_TRANSFERABLE" = 'y' ]
then
  is_transferable
  if [ "$GET_FLAGS" = 'y' ]
  then
    echo -e "\n$ACT_FLAGS"
  fi
fi

if [ "$GET_PROGRESS" = 'y' ]
then
  get_progress
fi

if [ "$BEAM_TO_SLICE" = 'y' ]
then
  echo "Beam to slice started" >&2
  get_type_structure >&2
  if [ -z "$TYPE" ]
  then  
    # check if we are calling from package location (DATA_DIR)
    echo ${0%/*} | grep $DATA_DIR > /dev/null
    if [ $? -ne 0 ]
    then
      # OK, seems to be default ASswm location, this is probably HF case, try to find the HF tarball (or zip file for LW HF) in DATADIR
      hf_tarball=`$FIND $DATA_DIR \( -name "*.tar" -o -name "*.zip" \) -print`
      [ -z "$hf_tarball" ] && echo "Cannot find any HF in $DATA_DIR" && exit 1
      # OK, let's extract HF metada from tarball
      # use the corresponding tool
      hf_file_type=`$FILE --mime-type $hf_tarball`
      echo "$hf_file_type" |  grep "/x-tar"
      if [ $? -eq 0 ]
      then  
        $TAR xvf $hf_tarball -C $DATA_DIR $PKG_FILE $PKG_SCRIPT $VERSION_FILE
      else
        echo "$hf_file_type" |  grep "/zip"
        if [ $? -eq 0 ]
        then
          $UNZIP -oj $hf_tarball "*/$PKG_FILE" "*/$PKG_SCRIPT" "*/$VERSION_FILE" -d $DATA_DIR
        fi
      fi  
      
      chmod 755 $DATA_DIR/$PKG_SCRIPT
      # execute extracted pkg script
      exec $DATA_DIR/$PKG_SCRIPT $PARAM_LIST
      # should not be here
      echo "can not execute $DATA_DIR/$PKG_SCRIPT" && exit 1
    fi
  fi
  case $TYPE in
    'AS-HF')   SLICE=$SLICE_AS_HF && ARCH=tar ;;
    'MGR-HF')  SLICE=$SLICE_AS_HF && ARCH=tar ;;
    'PLT-HF')  SLICE=$SLICE_PLT_HF && ARCH=tar ;;
    'CSTA-HF') SLICE=$SLICE_CSTA_HF && ARCH=tar ;;
    'LW-HF')   SLICE=$SLICE_LW_HF && ARCH=zip ;;
    'PP')      SLICE=$SLICE_PP ;;
    'RLC')     SLICE=$SLICE_RLC ;;
    'RMX-HF')  SLICE=$SLICE_PP ;;
       *)      echo "Unknown type $TYPE. Don't know what to do" && exit 1
  esac
  if [ "$TYPE" = "PP" -o "$TYPE" = "RLC" -o "$TYPE" = "RMX-HF" ]
  then  
    # use beam to slice function for PPs/RLCs
    beam_to_slice
#   elif [ "$TYPE" = "LW-HF" ]
#   then
#     # extract LWHF zip file to $SLICE
#     if [ "$USE_PV" = 'y' ]; then
#       $FIND $DATA_DIR -name "*.$ARCH" -print0 | while IFS= read -r -d '' file; do unzip_pv "$file" $SLICE; done
#     else
#       $FIND $DATA_DIR -name "*.$ARCH" -exec $UNZIP -d $SLICE '{}' \;
#     fi
#     # move Spec_LW_* directory to $SLICE dir
#     $FIND $SLICE -type d -name "Spec_LW_*" -exec mv -v '{}' $SLICE \;
#     # remove OpenScape4000* root dir
#     rm -rfv $SLICE/OpenScape4000*
#     $CHOWN -R swm:unity $SLICE
#     echo "Change own to swm:unity for everything under $SLICE"
#     $CHMOD -R 770 $SLICE
#     echo "Change rights to 770 for everything under $SLICE"
  else
    # find the archive and move it to the slice
    $FIND $DATA_DIR -name "*.$ARCH" -exec chmod 644 -v '{}' \; -exec chown -v swm:unity '{}' \; -exec mv -v '{}' $SLICE \;
  fi  

  $RM -fr $DATA_DIR/*
fi

exit 0

-----BEGIN PV-----
begin-base64 644 pv.tar.gz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====
-----END PV-----
########PKG_HANDLER_SCRIPT_END#########
