#!/bin/bash

# Hack in kernel-specific firmware directory so it can be processed along with
# the dracut default ones.  Note that upstream dracut currently has defaults
# that include kernel-version-specific directories (dracut commit
# 041-100-gb52cfbea).
#
# SPDX-License-Identifier: CC0-1.0

check() {
	return 0
}

DATA_DIR=/usr/share/microcode_ctl/ucode_with_caveats
check_caveats=/usr/libexec/microcode_ctl/check_caveats

install() {
	verbose_opt=
	[ 4 -gt "$stdloglvl" ] || verbose_opt="-v"

	# HACK: we override external fw_dir variable in order to get
	#       an additional ucode based on the kernel version.
	dinfo "  microcode_ctl module: hacking fw_dir"

	[ -z "$fw_dir_l" ] || {
		dinfo "    microcode_ctl module: avoid touching fw_dir as" \
		      "it has been changed (fw_dir_l is '$fw_dir_l')"

		return 0
	}

	ucode=$(get_ucode_file)

	for i in $(ls "$DATA_DIR"); do
		dinfo "    microcode_ctl: processing data directory " \
		      "\"$DATA_DIR/$i\"..."
		[ -e "$DATA_DIR/$i/readme" ] || {
			dinfo "      microcode_ctl: skipping \"$i\": no readme"
			continue
		}
		[ -e "$DATA_DIR/$i/config" ] || {
			dinfo "    microcode_ctl: skipping \"$i\": no config"
			continue
		}
		[ ! -e "/etc/microcode_ctl/ucode_with_caveats/disallow-$i" ] || {
			dinfo "    microcode_ctl: skipping \"$i\":" \
			      "\"/etc/microcode_ctl/ucode_with_caveats/disallow-$i\"" \
			      "present"
			continue
		}
		path=$($check_caveats -k "$kernel" -c "$i")
                fname=$(basename "$path")

		if [ "x" != "x$hostonly" ]; then
			[ "x$ucode" = "x$fname" ] || {
				dinfo "    microcode_ctl: Host-Only mode" \
                                      "is enabled and ucode name does not" \
                                      "match the expected one" \
                                      "(\"$ucode\" != \"$fname\")"
				continue
			}
		fi

		dinfo "    microcode_ctl: processing file \"$path\""
		if $check_caveats -k "$kernel" -c "$i" $verbose_opt > /dev/null
		then
			dinfo "      microcode_ctl: caveats check for kernel" \
			      "version \"$kernel\" passed, adding" \
			      "\"/lib/firmware/$kernel\" to fw_dir variable"
			fw_dir="/lib/firmware/$kernel $fw_dir"
			break
		else
			dinfo "      microcode_ctl: kernel version \"$kernel\"" \
			      "failed caveats check, skipping"
			continue
		fi
	done
}

