function Get-AsupRestData {
    [CmdletBinding()]
    [OutputType([System.Xml.XmlDocument])]
    param (
        [Parameter(Mandatory)]
        [uri]$Uri,

        [Parameter()]
        [int]$TimeoutSec = 300,

        [Parameter()]
        [int]$MaxRetryCount = 3,

        [Parameter()]
        [int]$RetryIntervalSec = 1
    )

    # Refresh the token if it is expired or about to expire
    if (($Script:AuthToken.ExpiresOn - [System.DateTimeOffset]::UtcNow).TotalMinutes -lt 5) {
        Invoke-SsoLogin -ErrorAction Stop
    }

    $headers = $(
        if ((Get-PSCallStack | Select-Object -Index 1).Command -eq 'Get-AsupObject') {
            @{
                'Authorization' = "Bearer $( $Script:AuthToken.AccessToken )"
                'ClientId'      = 'netappdocs_secure'
            }
        }
        else {
            @{
                'AuthorizationToken' = $Script:AuthToken.AccessToken
                'client_id'          = 'netappdocs_secure'
            }
        }
    )

    try {
        $retries = 0

        while ($retries -lt $MaxRetryCount) {
            try {
                $response = Invoke-RestMethod -Uri $Uri -Method Get -Headers $headers -TimeoutSec $TimeoutSec -Verbose:$false -ErrorAction Stop

                break
            }
            catch {
                $Global:Error.RemoveAt(0)

                $retries++

                if ($retries -ge $MaxRetryCount) {
                    Write-Warning -Message 'Reached the max retry count'

                    throw
                }
                else {
                    Write-Debug -Message "Retry attempt $retries of $MaxRetryCount"
                }

                Start-Sleep -Seconds $RetryIntervalSec
            }
        }

        # IRM will return a string if the XML cannot be successfully parsed (i.e. when invalid XML character is present)
        if ($response -is [System.String]) {
            # Attempt to strip the invalid XML characters and cast to XmlDocument
            [xml]($response -replace '[^\u0009\u000a\u000d\u0020-\ufffd]|([\ud800-\udbff](?![\udc00-\udfff]))|((?<![\ud800-\udbff])[\udc00-\udfff])')
        }
        else {
            $response
        }
    }
    catch {
        $Global:Error.RemoveAt(0)

        if ($_.CategoryInfo.Reason -eq 'WebException') {
            $errorRecord = New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("Unable to fetch ASUP URL. Reason: $( $_.Exception.Message )", $_.CategoryInfo.Reason, [System.Management.Automation.ErrorCategory]::ResourceUnavailable, $Uri)
        }
        elseif ($_.CategoryInfo.Reason -eq 'InvalidOperationException') {
            $errorRecord = New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("Unable to fetch ASUP URL. Reason: $( $_.Exception.Message )", $_.CategoryInfo.Reason, [System.Management.Automation.ErrorCategory]::InvalidOperation, $Uri)
        }
        else {
            $errorRecord = New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("Unable to fetch ASUP URL. Reason: $( $_.Exception.Message )", 'InvalidResult', [System.Management.Automation.ErrorCategory]::InvalidResult, $Uri)
        }

        $errorRecord.CategoryInfo.TargetType = 'URI'

        $PSCmdlet.ThrowTerminatingError($errorRecord)
    }
    finally {
        # Invoke-RestMethod may not properly dispose the connection
        $servicePoint = [System.Net.ServicePointManager]::FindServicePoint($Uri)
        [void]$servicePoint.CloseConnectionGroup('')
    }
}

function Convert-GenericAsupXmlNode {
    [CmdletBinding()]
    param (
        [Parameter(Mandatory, ValueFromPipeline)]
        [AllowNull()]
        [System.Xml.XmlNode]$Node,

        [Parameter()]
        [string]$ReplaceString
    )

    process {
        if ($Node) {
            $hashtable = @{ }

            foreach ($child in ($Node.ChildNodes | Where-Object { $_.LocalName -notin @( 'parents', 'childs', 'object_view', 'biz_key' ) })) {
                $correctedName = $(
                    if ($child.LocalName.Contains('_') -or $child.LocalName.Contains('-')) {
                        (Get-Culture).TextInfo.ToTitleCase($child.LocalName) -replace '[_]|[-]'
                    }
                    else {
                        [string]::Concat($child.LocalName.Substring(0, 1).ToUpper(), $child.LocalName.Substring(1))
                    }
                ) -replace $ReplaceString

                $value = $(
                    if ($correctedName -eq 'Options' -and $child.row) {
                        Get-ObjectFromKeyValuePair -KeyValuePair $child.row
                    }
                    elseif ($child.list) {
                        @(
                            $child.list.li | ForEach-Object {
                                ConvertTo-TypedValue -Text ($_ -replace '%3A', ':' -replace '%2F', '/' -replace '%05' -replace '%5C', '\' -replace '%24', '$' -replace '%2C', ',' -replace '%3C', '<' -replace '%3E', '>' -replace '%28', '(' -replace '%29', ')' -replace '%26', '&' -replace '%2B', '+' -replace '%0D' -replace '%40', '@' -replace '%25', '%' -replace '%22', '"')
                            }
                        )
                    }
                    else {
                        ConvertTo-TypedValue -Text ($child.InnerText -replace '%3A', ':' -replace '%2F', '/' -replace '%05' -replace '%5C', '\' -replace '%24', '$' -replace '%2C', ',' -replace '%3C', '<' -replace '%3E', '>' -replace '%28', '(' -replace '%29', ')' -replace '%26', '&' -replace '%2B', '+' -replace '%0D' -replace '%40', '@' -replace '%25', '%' -replace '%22', '"')
                    }
                )

                if ($hashtable.ContainsKey($correctedName)) {
                    $value = $hashtable[$correctedName], $value
                }

                $hashtable[$correctedName] = $value
            }

            if ($PSVersionTable.PSVersion -eq [version]::Parse('3.0')) {
                New-Object -TypeName System.Management.Automation.PSObject -Property $hashtable
            }
            else {
                [pscustomobject]$hashtable
            }
        }
    }
}

function Test-AsupSetIsValid {
    [CmdletBinding()]
    [OutputType([System.Boolean])]
    param (
        [Parameter(Mandatory)]
        [ValidateNotNullOrEmpty()]
        [System.Xml.XmlElement[]]$AsupSet,

        [Parameter(Mandatory)]
        [string[]]$SectionsToValidate
    )

    if (($AsupSet | Group-Object -Property hostname | Where-Object { $_.Count -gt 1 } | Measure-Object).Count -gt 0) {
        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('AsupSet contains more than one ASUP for a single node.', 'MultipleAsupForSingleNode', [System.Management.Automation.ErrorCategory]::InvalidData, "$( $AsupSet.Hostname -join ', ' )")))
    }

    if (($AsupSet | Where-Object { $_.asup_is_minimal -eq $true } | Measure-Object).Count -gt 0) {
        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('ASUP content type is set to minimal for one or more nodes. Minimal content ASUPs cannot be used for document generation.', 'AsupContentTypeMinimal', [System.Management.Automation.ErrorCategory]::InvalidData, "$( $AsupSet.Hostname -join ', ' )")))
    }

    if (($AsupSet | Where-Object { $_.asup_type -ne 'DOT-REGULAR' } | Measure-Object).Count -gt 0) {
        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('AsupSet contains one or more ASUPs with an unsupported asup_type (must be DOT-REGULAR).', 'InvalidAsupType', [System.Management.Automation.ErrorCategory]::InvalidData, "$( $AsupSet.Hostname -join ', ' )")))
    }

    if (($AsupSet | Measure-Object).Count -gt 1) {
        $asupDateRange = $AsupSet.asup_gen_date | ForEach-Object { Convert-DateTimeString -String $_ } | Sort-Object

        if (($asupDateRange[-1] - $asupDateRange[0]).TotalHours -gt 4) {
            Write-Error -Message 'AsupSet time difference is greater than 4 hours.'

            return $false
        }
    }

    if (($AsupSet | Group-Object -Property asup_gen_zone -NoElement | Select-Object -Property Name -Unique | Measure-Object).Count -gt 1) {
        Write-Warning -Message 'AsupSet contains different timezone settings for one or more nodes. The timezone from only one node will be used and therefore any datetime calculations may be incorrect.'
    }

    $validAsups = $true

    for ($i = 0; $i -lt $AsupSet.Count; $i++) {
        $systemAsup = $asupSet[$i]

        Write-Verbose -Message "Validating ASUP for node: $( $systemAsup.hostname ) (ASUPID: $( $systemAsup.asup_id ))"

        try {
            [void](Get-AsupObject -AsupObject $systemAsup -ObjectName 'SYSTEM' -ErrorAction Stop)

            if ($systemAsup.sys_operating_mode -eq 'Cluster-Mode') {
                if (((Get-AsupObject -AsupObject $systemAsup -ObjectName 'CLUSTER' -ReplaceString '^Cluster(?!Name|Identifier)' -ErrorAction Stop | Where-Object { -not [string]::IsNullOrWhiteSpace($_.ClusterName) -and -not [string]::IsNullOrWhiteSpace($_.ClusterIdentifier) }) | Measure-Object).Count -eq 0) {
                    $errorRecord = New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("ASUP 'CLUSTER' object incomplete", 'AsupNotFullyIngested', [System.Management.Automation.ErrorCategory]::ReadError, $AsupSet.cluster_name)

                    throw $errorRecord
                }

                $clusterMembers = (Get-AsupSection -AsupObject $systemAsup -SectionName 'CLUSTER-INFO.XML' -InnerProperty 'CLUSTER_INFO' -ErrorAction Stop).node

                if ($i -eq 0) {
                    $memberCompare = Compare-Object -ReferenceObject $clusterMembers -DifferenceObject $AsupSet.Hostname

                    if (($memberCompare | Where-Object { $_.SideIndicator -eq '<=' } | Measure-Object).Count -gt 0) {
                        $msg = New-Object -TypeName System.Text.StringBuilder

                        [void]$msg.AppendLine("ASUP missing for one or more nodes in the cluster: $( ($memberCompare | Where-Object { $_.SideIndicator -eq '<=' }).InputObject -join ', ' )")
                        [void]$msg.AppendLine('-----------------------------------------------------------------------------------------')
                        [void]$msg.AppendLine('Further analysis is required to ensure a complete set of ASUPs (one complete ASUP per')
                        [void]$msg.AppendLine('node in the cluster and all within a 4 hour time window). In most cases, this condition')
                        [void]$msg.AppendLine('is caused by incorrect autosupport settings on the affected node(s). In rare cases, this')
                        [void]$msg.AppendLine('could be due to different time zone settings on the nodes.')
                        [void]$msg.AppendLine()

                        $errorRecord = New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), 'AsupNotFound', [System.Management.Automation.ErrorCategory]::ObjectNotFound, $AsupSet.cluster_name)

                        $PSCmdlet.ThrowTerminatingError($errorRecord)
                    }
                }
            }
            else {
                if (($haGroup = Get-AsupObject -AsupObject $systemAsup -ObjectName 'HA_GROUP' | Where-Object { -not [string]::IsNullOrWhiteSpace($_.PartnerHostname) -and $_.PartnerHostname -ne '<unknown>' })) {
                    if ($haGroup.HaState -eq 'taken_over') {
                        Write-Error -Message "ASUP invalid for node: $( $systemAsup.hostname ) - (node is in a takeover state)"

                        $validAsups = $false
                    }

                    if (($AsupSet | Where-Object { $_.hostname -ceq $haGroup.PartnerHostname } | Measure-Object).Count -lt 1) {
                        $msg = New-Object -TypeName System.Text.StringBuilder

                        if ($haGroup.PartnerHostname -eq '-') {
                            [void]$msg.AppendLine("Partner ASUP invalid for node: $( $systemAsup.hostname )")
                            [void]$msg.AppendLine('-----------------------------------------------------------------------------------------')
                            [void]$msg.AppendLine('HA may have been disabled at the time this ASUP set was generated.')
                            [void]$msg.AppendLine()

                            $errorRecord = New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), 'AsupInvalid', [System.Management.Automation.ErrorCategory]::InvalidData, $systemAsup.hostname)
                        }
                        else {
                            [void]$msg.AppendLine("ASUP missing for controller: $( $haGroup.PartnerHostname )")
                            [void]$msg.AppendLine('-----------------------------------------------------------------------------------------')
                            [void]$msg.AppendLine('Further analysis is required to ensure a complete set of ASUPs (one complete ASUP per')
                            [void]$msg.AppendLine('controller in the HA pair and both within a 4 hour time window). In most cases, this')
                            [void]$msg.AppendLine('condition is caused by incorrect autosupport settings on the affected controller.')
                            [void]$msg.AppendLine('In rare cases, this could be due to different time zone settings on the controllers.')
                            [void]$msg.AppendLine()

                            $errorRecord = New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), 'AsupNotFound', [System.Management.Automation.ErrorCategory]::ObjectNotFound, $systemAsup.hostname)
                        }

                        $PSCmdlet.ThrowTerminatingError($errorRecord)
                    }
                }
            }
        }
        catch {
            $Global:Error.RemoveAt(0)

            $msg = New-Object -TypeName System.Text.StringBuilder

            if ($_.CategoryInfo.Category -eq 'ReadError' -or $_.Exception.Message -like '*Object data is not yet processed/available for the given*') {
                [void]$msg.AppendLine("An unexpected error occurred. Processing halted for this node: $( $systemAsup.hostname ) (ASUPID: $( $systemAsup.asup_id )).")
                [void]$msg.AppendLine('-----------------------------------------------------------------------------------------')
                [void]$msg.AppendLine('This ASUP does not appear to be fully ingested into the ASUP DW. It is possible that')
                [void]$msg.AppendLine('waiting up to a few hours or so may be enough time for the ASUP DW to finishing ingesting')
                [void]$msg.AppendLine('the data. However, in some cases, the only workaround is to try a different set of ASUPs.')
                [void]$msg.AppendLine()

                $errorRecord = New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), $_.CategoryInfo.Category, [System.Management.Automation.ErrorCategory]::ReadError, $systemAsup.hostname)
            }
            elseif ($_.CategoryInfo.Category -eq 'InvalidData') {
                [void]$msg.AppendLine("An unexpected error occurred. Processing halted for this node: $( $systemAsup.hostname ) (ASUPID: $( $systemAsup.asup_id )).")
                [void]$msg.AppendLine('-----------------------------------------------------------------------------------------')
                [void]$msg.AppendLine('This ASUP does not appear to have been ingested into the ASUP DW. In most cases, the only')
                [void]$msg.AppendLine('workaround is to try a different set of ASUPs.')
                [void]$msg.AppendLine()

                $errorRecord = New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), $_.CategoryInfo.Category, [System.Management.Automation.ErrorCategory]::InvalidData, $systemAsup.hostname)
            }
            else {
                [void]$msg.AppendLine("There is an unexpected issue with this ASUP for node: $( $systemAsup.hostname ) (ASUPID: $( $systemAsup.asup_id )).")
                [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
                [void]$msg.AppendLine('As this error is not expected, please consider contacting the support staff via e-mail or by')
                [void]$msg.AppendLine('starting a thread on the community site.')
                [void]$msg.AppendLine('            E-mail: ng-NetAppDocs-support@netapp.com')
                [void]$msg.AppendLine('    Community site: https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs')
                [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
                [void]$msg.AppendLine($( $_.Exception.GetBaseException().Message ))
                [void]$msg.AppendLine("Location: $( $_.ScriptStackTrace )")
                [void]$msg.AppendLine()

                $errorRecord = New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), $_.CategoryInfo.Category, [System.Management.Automation.ErrorCategory]::InvalidResult, $systemAsup.hostname)
            }

            $PSCmdlet.ThrowTerminatingError($errorRecord)
        }

        if (($manifest = Get-AsupSection -AsupObject $systemAsup -SectionName 'MANIFEST.XML' -InnerProperty 'Manifest' -ErrorAction Ignore)) {
            $ignoreIncompleteSections = @(
                [pscustomobject]@{
                    BodyFile = 'KENV'
                    Message  = 'Environment Variables table'
                }, @{
                    BodyFile = 'PLATFORM-SENSORS.XML'
                    Message  = 'Hardware Faults table'
                }
            )

            $incompleteSections = [System.Collections.Generic.List[pscustomobject]]@( $manifest | Where-Object { $_.BodyFile -in $SectionsToValidate -and ($_.Status -match '(?:collection|delivery)[-](?:truncated|skipped)[-](?:size|time)[-]limit' -or $_.Status -eq 'collection-incomplete') } )

            foreach ($ignoredSection in $ignoreIncompleteSections) {
                if (($entry = $incompleteSections | Where-Object { $_.BodyFile -eq $ignoredSection.BodyFile })) {
                    Write-Warning -Message "The ASUP section $( $ignoredSection.BodyFile ) is missing for node: $( $systemAsup.hostname ) (Reason: $( $entry.Status )).`r`n         This is common and only the $( $ignoredSection.Message ) will be incomplete due to this missing data."

                    [void]$incompleteSections.Remove($entry)
                }
            }

            if (($incompleteSections | Measure-Object).Count -gt 0) {
                $msg = New-Object -TypeName System.Text.StringBuilder

                [void]$msg.AppendLine("ASUP incomplete for node: $( $systemAsup.hostname ). Incomplete or missing sections:")

                $incompleteSections | ForEach-Object {
                    [void]$msg.AppendLine("         $( [string]::Format('{0} (Reason: {1})', $_.BodyFile, $_.Status) )")
                }

                [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
                [void]$msg.AppendLine('The ASUP settings may need to be modified in order for complete ASUPs to be received. The protocol')
                [void]$msg.AppendLine('should be set to HTTPS (not SMTP). This usually resolves the issue. If not, the maximum size')
                [void]$msg.AppendLine('settings may need to be adjusted and in some cases, the budget settings (size and time limits) for')
                [void]$msg.AppendLine('the affected sections may need to be increased.')

                Write-Error -Message $msg.ToString()

                $validAsups = $false
            }
        }
    }

    $validAsups
}

function Get-AsupObject {
    [CmdletBinding()]
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseDeclaredVarsMoreThanAssignments', '')]
    param (
        [Parameter(Mandatory, ValueFromPipeline)]
        [System.Object]$AsupObject,

        [Parameter(Mandatory)]
        [string]$ObjectName,

        [Parameter()]
        [AllowEmptyString()]
        [string]$MasterNodeName,

        [Parameter()]
        [string]$InnerProperty,

        [Parameter()]
        [switch]$IsRawZapiXml,

        [Parameter()]
        [string]$ReplaceString
    )

    process {
        $uris = @( )

        if ($PSCmdlet.MyInvocation.BoundParameters.ContainsKey('MasterNodeName') -and $PSCmdlet.MyInvocation.BoundParameters.ContainsKey('ObjectName')) {
            $sourceAsup = Get-FirstValidAsupObjectForSection -SourceAsup $AsupObject -SectionName $SectionName -MasterNodeName $MasterNodeName

            $uris += "$( $Script:AsupObjectRestUri )/object_view/$ObjectName/biz_key/$( $sourceAsup.BizKey )?service_tier=$( $Script:AiqServiceTierList )"
        }
        else {
            switch -Exact ($AsupObject.GetType().Name) {
                'Object[]' {
                    Convert-ListToBuckets -List $AsupObject -BucketSize 10 | ForEach-Object {
                        $uris += "$( $Script:AsupObjectRestUri )/object_view/$ObjectName/biz_key/$( ($_.Bizkey | Where-Object { -not [string]::IsNullOrWhiteSpace($_) }) -join ',' )?service_tier=$( $Script:AiqServiceTierList )"
                    }
                }
                'ArrayList' {
                    Convert-ListToBuckets -List $AsupObject -BucketSize 10 | ForEach-Object {
                        $uris += "$( $Script:AsupObjectRestUri )/object_view/$ObjectName/biz_key/$( ($_.Bizkey | Where-Object { -not [string]::IsNullOrWhiteSpace($_) }) -join ',' )?service_tier=$( $Script:AiqServiceTierList )"
                    }
                }
                'XmlElement' {
                    $uris += "$( $Script:AsupObjectRestUri )/object_view/$ObjectName/biz_key/$( $AsupObject.biz_key )?service_tier=$( $Script:AiqServiceTierList )"
                }
                'XmlElement[]' {
                    Convert-ListToBuckets -List $AsupObject -BucketSize 10 | ForEach-Object {
                        $uris += "$( $Script:AsupObjectRestUri )/object_view/$ObjectName/biz_key/$( ($_.biz_key | Where-Object { -not [string]::IsNullOrWhiteSpace($_) }) -join ',' )?service_tier=$( $Script:AiqServiceTierList )"
                    }
                }
                'String' {
                    $uris += $(
                        if ($AsupObject -like '*/object_view/*') {
                            $AsupObject
                        }
                        else {
                            $AsupObject -replace '/biz_key', "/object_view/$ObjectName/biz_key"
                        }
                    )
                }
                'PSCustomObject' {
                    $uris += "$( $Script:AsupObjectRestUri )/object_view/$ObjectName/biz_key/$( $AsupObject.BizKey )?service_tier=$( $Script:AiqServiceTierList )"
                }
                default {
                    $errorRecord = New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('An invalid AsupObject was passed to the Get-AsupObject function.', 'InvalidType', [System.Management.Automation.ErrorCategory]::InvalidType, $AsupObject)
                    $errorRecord.CategoryInfo.TargetType = $AsupObject.GetType().Name

                    $PSCmdlet.ThrowTerminatingError($errorRecord)
                }
            }
        }

        foreach ($uri in ($uris | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })) {
            $response = Get-AsupRestData -Uri $uri

            foreach (
                $failedResponse in (
                    $response.xml | Where-Object {
                        $_ -and (
                            ($_.status_code -and $_.status_code -ne 'SUCCESS') -or
                            (($_.status | Select-Object -First 1).status_code -ne 'SUCCESS')
                        )
                    }
                )
            ) {
                $category = $(
                    switch -Exact ($failedResponse.status_code) {
                        'not.found' {
                            'ObjectNotFound'
                        }
                        'no.data.found' {
                            'InvalidData'
                        }
                        'NO_DATA_FOUND' {
                            'InvalidData'
                        }
                        'INVALID_PARAMETER' {
                            'InvalidArgument'
                        }
                        'unknown.error' {
                            'ReadError'
                        }
                        'xml_error' {
                            'ReadError'
                        }
                        'Validation Exception' {
                            'ResourceUnavailable'
                        }
                        default {
                            'NotSpecified'
                        }
                    }
                )

                $errorTargetName = Get-ErrorTargetName @PSBoundParameters

                Write-Error -CategoryReason $failedResponse.status_code -Category $category -CategoryActivity 'Get-AsupObject' -Message $failedResponse.message -CategoryTargetName $errorTargetName -ErrorAction (($PSCmdlet.MyInvocation.BoundParameters['ErrorAction'], 'SilentlyContinue')[[byte](! $PSCmdlet.MyInvocation.BoundParameters['ErrorAction'])])

                if ($failedResponse.message -eq 'No systems available for the input data:[]' -or $failedResponse.message -like 'Invalid or No section(s) available for the request*' -or $failedResponse.message -like 'Entity not supported:*') {
                    $Global:Error.RemoveAt(0)
                }
            }

            foreach ($successResponse in ($response.xml | Where-Object { ($_.status | Select-Object -First 1).status_code -eq 'SUCCESS' })) {
                if (-not [string]::IsNullOrWhiteSpace($InnerProperty)) {
                    $successResponse.SelectNodes('//data') | ForEach-Object {
                        if ($_.InnerText -notlike "*$InnerProperty><asup:Abort>*") {
                            try {
                                ($xmlNode = New-Object -TypeName System.Xml.XmlDocument).LoadXml(($_.InnerText -replace "(?s)(</$InnerProperty>).*", '$1'))
                                $xmlNode.$InnerProperty.Row | Convert-GenericAsupXmlNode -ReplaceString $ReplaceString
                            }
                            catch {
                                $Global:Error.RemoveAt(0)

                                $errorTargetName = Get-ErrorTargetName @PSBoundParameters

                                Write-Error -CategoryReason 'INVALID_DATA' -Category 'InvalidData' -CategoryActivity 'Get-AsupObject' -Message "XML data for section is corrupt: $( $_.Exception.InnerException.Message )" -CategoryTargetName $errorTargetName -ErrorAction (($PSCmdlet.MyInvocation.BoundParameters['ErrorAction'], 'SilentlyContinue')[[byte](! $PSCmdlet.MyInvocation.BoundParameters['ErrorAction'])])

                                continue
                            }
                        }
                    }
                }
                elseif ($IsRawZapiXml.IsPresent) {
                    $successResponse.SelectNodes('//data') | ForEach-Object {
                        try {
                            ($xmlNode = New-Object -TypeName System.Xml.XmlDocument).LoadXml($_.'#cdata-section')
                            $xmlNode | Convert-GenericAsupXmlNode -ReplaceString $ReplaceString
                        }
                        catch {
                            $Global:Error.RemoveAt(0)

                            $errorTargetName = Get-ErrorTargetName @PSBoundParameters

                            Write-Error -CategoryReason 'INVALID_DATA' -Category 'InvalidData' -CategoryActivity 'Get-AsupObject' -Message "XML data for section is corrupt: $( $_.Exception.InnerException.Message )" -CategoryTargetName $errorTargetName -ErrorAction (($PSCmdlet.MyInvocation.BoundParameters['ErrorAction'], 'SilentlyContinue')[[byte](! $PSCmdlet.MyInvocation.BoundParameters['ErrorAction'])])

                            continue
                        }
                    }
                }
                else {
                    $successResponse.SelectNodes('//object/rows/row') | Convert-GenericAsupXmlNode -ReplaceString $ReplaceString
                }
            }
        }
    }
}

function Get-AsupSection {
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseDeclaredVarsMoreThanAssignments', '')]
    [CmdletBinding()]
    param (
        [Parameter(Mandatory, ValueFromPipeline)]
        [System.Object]$AsupObject,

        [Parameter(Mandatory)]
        [string]$SectionName,

        [Parameter()]
        [AllowEmptyString()]
        [string]$MasterNodeName,

        [Parameter()]
        [string]$InnerProperty,

        [Parameter()]
        [switch]$IsRawZapiXml,

        [Parameter()]
        [string]$ReplaceString
    )

    process {
        $uris = @( )

        if ($PSCmdlet.MyInvocation.BoundParameters.ContainsKey('MasterNodeName') -and $PSCmdlet.MyInvocation.BoundParameters.ContainsKey('SectionName')) {
            $sourceAsup = Get-FirstValidAsupObjectForSection -SourceAsup $AsupObject -SectionName $SectionName -MasterNodeName $MasterNodeName

            $uris += "$( $Script:AsupSectionRestUri )/service_tier/$( $Script:AiqServiceTierList )/biz_key/$( $sourceAsup.BizKey )"
        }
        else {
            switch -Exact ($AsupObject.GetType().Name) {
                'Object[]' {
                    Convert-ListToBuckets -List $AsupObject -BucketSize 10 | ForEach-Object {
                        $uris += "$( $Script:AsupSectionRestUri )/service_tier/$( $Script:AiqServiceTierList )/biz_key/$( ($_.Bizkey | Where-Object { -not [string]::IsNullOrWhiteSpace($_) }) -join ',' )"
                    }
                }
                'ArrayList' {
                    Convert-ListToBuckets -List $AsupObject -BucketSize 10 | ForEach-Object {
                        $uris += "$( $Script:AsupSectionRestUri )/service_tier/$( $Script:AiqServiceTierList )/biz_key/$( ($_.Bizkey | Where-Object { -not [string]::IsNullOrWhiteSpace($_) }) -join ',' )"
                    }
                }
                'XmlElement' {
                    $uris += "$( $Script:AsupSectionRestUri )/service_tier/$( $Script:AiqServiceTierList )/biz_key/$( $AsupObject.biz_key )"
                }
                'XmlElement[]' {
                    Convert-ListToBuckets -List $AsupObject -BucketSize 10 | ForEach-Object {
                        $uris += "$( $Script:AsupSectionRestUri )/service_tier/$( $Script:AiqServiceTierList )/biz_key/$( ($_.biz_key | Where-Object { -not [string]::IsNullOrWhiteSpace($_) }) -join ',' )"
                    }
                }
                'String' {
                    $uris += $AsupObject
                }
                'PSCustomObject' {
                    $uris += "$( $Script:AsupSectionRestUri )/service_tier/$( $Script:AiqServiceTierList )/biz_key/$( $AsupObject.BizKey )"
                }
                default {
                    $errorRecord = New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('An invalid AsupObject was passed to the Get-AsupSection function.', 'InvalidType', [System.Management.Automation.ErrorCategory]::InvalidType, $AsupObject)
                    $errorRecord.CategoryInfo.TargetType = $AsupObject.GetType().Name

                    $PSCmdlet.ThrowTerminatingError($errorRecord)
                }
            }
        }

        foreach ($uri in ($uris | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })) {
            $uri += "/section_view/$SectionName"

            $response = Get-AsupRestData -Uri $uri

            foreach ($failedResponse in ($response.xml | Where-Object { $_ -and ($_.status | Select-Object -First 1).status_code -ne 'SUCCESS' })) {
                $category = $(
                    switch -Exact ($failedResponse.status.error_code) {
                        'not.found' {
                            'ObjectNotFound'
                        }
                        'no.data.found' {
                            'InvalidData'
                        }
                        'NO_DATA_FOUND' {
                            'InvalidData'
                        }
                        'INVALID_PARAMETER' {
                            'InvalidArgument'
                        }
                        'unknown.error' {
                            'ReadError'
                        }
                        'xml_error' {
                            'ReadError'
                        }
                        default {
                            'NotSpecified'
                        }
                    }
                )

                $errorTargetName = Get-ErrorTargetName @PSBoundParameters

                Write-Error -CategoryReason $failedResponse.status.error_code -Category $category -CategoryActivity 'Get-AsupSection' -Message $failedResponse.status.message.InnerText -CategoryTargetName $errorTargetName -ErrorAction (($PSCmdlet.MyInvocation.BoundParameters['ErrorAction'], 'SilentlyContinue')[[byte](! $PSCmdlet.MyInvocation.BoundParameters['ErrorAction'])])

                if ($failedResponse.status.message.InnerText -eq 'No systems available for the input data:[]' -or $failedResponse.status.message.InnerText -like 'Invalid or No section(s) available for the request*') {
                    $Global:Error.RemoveAt(0)
                }
            }

            foreach ($successResponse in ($response.xml | Where-Object { ($_.status | Select-Object -First 1).status_code -eq 'SUCCESS' })) {
                if (-not [string]::IsNullOrWhiteSpace($InnerProperty)) {
                    $successResponse.SelectNodes('//data') | ForEach-Object {
                        if ($_.InnerText -notlike "*$InnerProperty><asup:Abort>*") {
                            try {
                                ($xmlNode = New-Object -TypeName System.Xml.XmlDocument).LoadXml(($_.InnerText -replace "(?s)(</$InnerProperty>).*", '$1'))
                                $xmlNode.$InnerProperty.Row | Convert-GenericAsupXmlNode -ReplaceString $ReplaceString
                            }
                            catch {
                                $Global:Error.RemoveAt(0)

                                $errorTargetName = Get-ErrorTargetName @PSBoundParameters

                                Write-Error -CategoryReason 'INVALID_DATA' -Category 'InvalidData' -CategoryActivity 'Get-AsupSection' -Message "XML data for section is corrupt: $( $_.Exception.InnerException.Message )" -CategoryTargetName $errorTargetName -ErrorAction (($PSCmdlet.MyInvocation.BoundParameters['ErrorAction'], 'SilentlyContinue')[[byte](! $PSCmdlet.MyInvocation.BoundParameters['ErrorAction'])])

                                continue
                            }
                        }
                    }
                }
                elseif ($IsRawZapiXml.IsPresent) {
                    $successResponse.SelectNodes('//data') | ForEach-Object {
                        try {
                            ($xmlNode = New-Object -TypeName System.Xml.XmlDocument).LoadXml($_.'#cdata-section')
                            $xmlNode | Convert-GenericAsupXmlNode -ReplaceString $ReplaceString
                        }
                        catch {
                            $Global:Error.RemoveAt(0)

                            $errorTargetName = Get-ErrorTargetName @PSBoundParameters

                            Write-Error -CategoryReason 'INVALID_DATA' -Category 'InvalidData' -CategoryActivity 'Get-AsupSection' -Message "XML data for section is corrupt: $( $_.Exception.InnerException.Message )" -CategoryTargetName $errorTargetName -ErrorAction (($PSCmdlet.MyInvocation.BoundParameters['ErrorAction'], 'SilentlyContinue')[[byte](! $PSCmdlet.MyInvocation.BoundParameters['ErrorAction'])])

                            continue
                        }
                    }
                }
                else {
                    $successResponse.SelectNodes('//object/rows/row') | Convert-GenericAsupXmlNode -ReplaceString $ReplaceString
                }
            }
        }
    }
}

function Get-AsupContent {
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseDeclaredVarsMoreThanAssignments', '')]
    [CmdletBinding()]
    param (
        [Parameter(Mandatory, ValueFromPipeline)]
        [System.Object]$AsupObject,

        [Parameter(Mandatory)]
        [string]$SectionName
    )

    process {
        $uris = @( )

        switch -Exact ($AsupObject.GetType().Name) {
            'Object[]' {
                Convert-ListToBuckets -List $AsupObject -BucketSize 10 | ForEach-Object {
                    $uris += "$( $Script:AsupSectionRestUri )/service_tier/$( $Script:AiqServiceTierList )/biz_key/$( ($_.Bizkey | Where-Object { -not [string]::IsNullOrWhiteSpace($_) }) -join ',' )/section_view/$SectionName"
                }
            }
            'ArrayList' {
                Convert-ListToBuckets -List $AsupObject -BucketSize 10 | ForEach-Object {
                    $uris += "$( $Script:AsupSectionRestUri )/service_tier/$( $Script:AiqServiceTierList )/biz_key/$( ($_.Bizkey | Where-Object { -not [string]::IsNullOrWhiteSpace($_) }) -join ',' )/section_view/$SectionName"
                }
            }
            'XmlElement' {
                $uris += "$( $Script:AsupSectionRestUri )/service_tier/$( $Script:AiqServiceTierList )/biz_key/$( $AsupObject.biz_key )"
            }
            'XmlElement[]' {
                Convert-ListToBuckets -List $AsupObject -BucketSize 10 | ForEach-Object {
                    $uris += "$( $Script:AsupSectionRestUri )/service_tier/$( $Script:AiqServiceTierList )/biz_key/$( ($_.biz_key | Where-Object { -not [string]::IsNullOrWhiteSpace($_) }) -join ',' )"
                }
            }
            'String' {
                $uris += "$AsupObject/section_view/$SectionName"
            }
            'PSCustomObject' {
                $uris += "$( $Script:AsupSectionRestUri )/service_tier/$( $Script:AiqServiceTierList )/biz_key/$( $AsupObject.BizKey )/section_view/$SectionName"
            }
            default {
                $errorRecord = New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('An invalid AsupObject was passed to the Get-AsupContent function.', 'InvalidType', [System.Management.Automation.ErrorCategory]::InvalidType, $AsupObject)
                $errorRecord.CategoryInfo.TargetType = $AsupObject.GetType().Name

                $PSCmdlet.ThrowTerminatingError($errorRecord)
            }
        }

        foreach ($uri in ($uris | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })) {
            $response = Get-AsupRestData -Uri $uri

            foreach ($failedResponse in ($response.xml | Where-Object { $_ -and ($_.status | Select-Object -First 1).status_code -ne 'SUCCESS' })) {
                $category = $(
                    switch -Exact ($failedResponse.status.error_code) {
                        'not.found' {
                            'ObjectNotFound'
                        }
                        'NO_DATA_FOUND' {
                            'InvalidData'
                        }
                        'INVALID_PARAMETER' {
                            'InvalidArgument'
                        }
                        'unknown.error' {
                            'ReadError'
                        }
                        'xml_error' {
                            'ReadError'
                        }
                        default {
                            'NotSpecified'
                        }
                    }
                )

                $errorTargetName = Get-ErrorTargetName @PSBoundParameters

                Write-Error -CategoryReason $failedResponse.status.error_code -Category $category -CategoryActivity 'Get-AsupContent' -Message $failedResponse.status.message.InnerText -CategoryTargetName $errorTargetName -ErrorAction (($PSCmdlet.MyInvocation.BoundParameters['ErrorAction'], 'SilentlyContinue')[[byte](! $PSCmdlet.MyInvocation.BoundParameters['ErrorAction'])])

                if ($failedResponse.status.message.InnerText -eq 'No systems available for the input data:[]') {
                    $Global:Error.RemoveAt(0)
                }
            }

            foreach ($successResponse in ($response.xml | Where-Object { ($_.status | Select-Object -First 1).status_code -eq 'SUCCESS' })) {
                $successResponse.SelectNodes('//data') | ForEach-Object {
                    $_.'#cdata-section'
                }
            }
        }
    }
}

function Get-ErrorTargetName {
    [CmdletBinding()]
    [OutputType([System.Text.StringBuilder])]
    param (
        [Parameter(Mandatory)]
        [System.Object]$AsupObject,

        [Parameter(Mandatory, ParameterSetName = 'ObjectName')]
        [string]$ObjectName,

        [Parameter(Mandatory, ParameterSetName = 'SectionName')]
        [string]$SectionName,

        [Parameter(ValueFromRemainingArguments)]
        [string[]]$Properties
    )

    $sbErrorTargetName = New-Object -TypeName System.Text.StringBuilder

    if (($AsupObject | Measure-Object).Count -gt 0) {
        if ($uri -match 'biz_key[/](?<BizKeys>.+?)[/]') {
            $bizKeys = $Matches.BizKeys -split '\s*,\s*'

            if ($Matches) {
                $Matches.Clear()
            }

            [void]$sbErrorTargetName.Append("$( ($AsupObject | Where-Object { $_.BizKey -in $bizKeys }).SystemName | Where-Object { -not [string]::IsNullOrWhiteSpace($_) } | Sort-Object ) ")
        }
    }
    elseif (-not [string]::IsNullOrWhiteSpace($AsupObject.SystemName)) {
        [void]$sbErrorTargetName.Append("$( $AsupObject.SystemName ) ")
    }

    if ($sbErrorTargetName.Length -gt 0) {
        $sbErrorTargetName.Length--

        if ($PSCmdlet.MyInvocation.BoundParameters.ContainsKey('ObjectName')) {
            [void]$sbErrorTargetName.Append(": $ObjectName")
        }
        else {
            [void]$sbErrorTargetName.Append(": $SectionName")
        }

        $sbErrorTargetName.ToString()
    }
}

function Get-AsupVolSpace {
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseLiteralInitializerForHashtable', '')]
    param (
        [Parameter(Mandatory)]
        [System.Object[]]$AsupObject
    )

    $powerOf = @{
        'B'  = 0
        'KB' = 1
        'MB' = 2
        'GB' = 3
        'TB' = 4
        'PB' = 5
        'EB' = 6
    }

    foreach ($nodeAsup in $AsupObject) {
        Write-Verbose -Message "Capturing VOL-STATUS-S (node: $( $nodeAsup.SystemName ))"
        $volStatusS = Get-AsupContent -AsupObject $nodeAsup -SectionName 'VOL-STATUS-S' -ErrorVariable +NetAppDocsError

        Write-Verbose -Message "Capturing VOL-STATUS-V (node: $( $nodeAsup.SystemName ))"
        $volStatusV = Get-AsupContent -AsupObject $nodeAsup -SectionName 'VOL-STATUS-V' -ErrorVariable +NetAppDocsError

        Write-Verbose -Message "Capturing VOL-STATUS-F (node: $( $nodeAsup.SystemName ))"
        $volStatusF = Get-AsupContent -AsupObject $nodeAsup -SectionName 'VOL-STATUS-F' -ErrorVariable +NetAppDocsError

        $hashtable = New-Object -TypeName System.Collections.Hashtable

        foreach ($line in ($VolStatusV -split "`n")) {
            switch -Regex ($line) {
                '^(?:Volume\sName[:]\s)?(?<VolumeName>[^\s\:]+).*$' {
                    $volumeName = $Matches.VolumeName

                    $hashtable[$volumeName] = [ordered]@{
                        VolumeName                                 = $Matches.VolumeName -replace '\(\d+\)$'
                        VolumeUuid                                 = $null
                        InstanceUuid                               = $null
                        ProvenanceUuid                             = $null
                        UserData                                   = $null
                        UserDataPercent                            = $null
                        SnapshotReserve                            = $null
                        SnapshotReservePercent                     = $null
                        SnapshotSpill                              = $null
                        SnapshotSpillPercent                       = $null
                        TotalUsed                                  = $null
                        TotalUsedPercent                           = $null
                        PhysicalUsed                               = $null
                        PhysicalUsedPercent                        = $null
                        LogicalUsed                                = $null
                        LogicalUsedPercent                         = $null
                        LogicalAvailable                           = $null
                        LogicalAvailablePercent                    = $null
                        VolumeBlocksFootprint                      = $null
                        VolumeBlocksFootprintPercent               = $null
                        VolumeBlocksFootprintBin0                  = $null
                        VolumeBlocksFootprintBin0Percent           = $null
                        VolumeBlocksFootprintBin1                  = $null
                        VolumeBlocksFootprintBin1Percent           = $null
                        FlexvolMetadataFootprint                   = $null
                        FlexvolMetadataFootprintPercent            = $null
                        DedupeMetafilesTotalFootprint              = $null
                        DedupeMetafilesTotalFootprintPercent       = $null
                        DedupeMetafilesFootprint                   = $null
                        DedupeMetafilesFootprintPercent            = $null
                        DedupeMetafilesTemporaryFootprint          = $null
                        DedupeMetafilesTemporaryFootprintPercent   = $null
                        CrossVolumeDedupeMetafilesFootprint        = $null
                        CrossVolumeDedupeMetafilesFootprintPercent = $null
                        TapeBackupMetafilesFootprint               = $null
                        TapeBackupMetafilesFootprintPercent        = $null
                        DelayedFreeFootprint                       = $null
                        DelayedFreeFootprintPercent                = $null
                        SnapmirrorDestinationFootprint             = $null
                        SnapmirrorDestinationFootprintPercent      = $null
                        FileOperationMetadata                      = $null
                        FileOperationMetadataPercent               = $null
                        TotalFootprint                             = $null
                        TotalFootprintPercent                      = $null
                    }

                    $Matches.Clear()
                }
                '^\s+Volume\sUUID[:]\s(?<VolumeUuid>.+?)\s*$' {
                    $hashtable[$volumeName].VolumeUuid = $Matches.VolumeUuid

                    $Matches.Clear()
                }
                '^\s+Volume\sInstance\sUUID[:]\s(?<InstanceUuid>.+?)\s*$' {
                    $hashtable[$volumeName].InstanceUuid = $Matches.InstanceUuid

                    $Matches.Clear()
                }
                '^\s+Volume\sProvenance\sUUID[:]\s(?<ProvenanceUuid>.+?)\s*$' {
                    $hashtable[$volumeName].ProvenanceUuid = $Matches.ProvenanceUuid

                    $Matches.Clear()
                    $volumeName = $null
                }
                default { }
            }
        }

        if ($volStatusS) {
            foreach ($line in ($volStatusS -split "`n")) {
                if ($line -match '^\s*$') {
                    $blanklines++

                    continue
                }

                switch -Regex ($line) {
                    '^Volume\s+[:]\s+(?<VolumeName>.+?)\s*$' {
                        $volumeName = $Matches.VolumeName

                        $Matches.Clear()

                        $blanklines = 0

                        break
                    }
                    '^\s*User Data\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].UserData = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].UserDataPercent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    '^\s*Snapshot Reserve\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].SnapshotReserve = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].SnapshotReservePercent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    '^\s*Snapshot Spill\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].SnapshotSpill = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].SnapshotSpillPercent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    '^\s*Total\s(?:Used)?\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        if ($blanklines -eq 1) {
                            $hashtable[$volumeName].TotalUsed = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                            $hashtable[$volumeName].TotalUsedPercent = $Matches.Percent -replace '\s*%'

                            $Matches.Clear()
                        }
                        elseif ($blankLines -eq 2) {
                            $hashtable[$volumeName].PhysicalUsed = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                            $hashtable[$volumeName].PhysicalUsedPercent = $Matches.Percent -replace '\s*%'

                            $Matches.Clear()
                        }

                        break
                    }
                    '^\s*Total\sPhysical\sUsed\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].PhysicalUsed = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].PhysicalUsedPercent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    '^\s*Logical\sUsed\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].LogicalUsed = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].LogicalUsedPercent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    '^\s*Logical\sAvailable\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].LogicalAvailable = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].LogicalAvailablePercent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    default { }
                }
            }
        }

        if ($volStatusF) {
            foreach ($line in $volStatusF.Split("`n", [System.StringSplitOptions]::RemoveEmptyEntries)) {
                switch -Regex ($line) {
                    '^Volume\s+[:]\s+(?<VolumeName>.+?)\s*$' {
                        $volumeName = $Matches.VolumeName

                        $Matches.Clear()

                        break
                    }
                    '^\s*Volume Data Footprint\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].VolumeBlocksFootprint = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].VolumeBlocksFootprintPercent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    '^\s*Footprint in Performance Tier\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].VolumeBlocksFootprintBin0 = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].VolumeBlocksFootprintBin0Percent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    '^\s*Footprint in (?!Performance Tier).+?\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].VolumeBlocksFootprintBin1 = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].VolumeBlocksFootprintBin1Percent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    '^\s*Flexible Volume Metadata\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].FlexvolMetadataFootprint = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].FlexvolMetadataFootprintPercent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    '^\s*Deduplication\s+(?:Metadata\s+)(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].DedupeMetafilesTotalFootprint = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].DedupeMetafilesTotalFootprintPercent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    '^\s*Deduplication\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].DedupeMetafilesFootprint = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].DedupeMetafilesFootprintPercent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    '^\s*Temporary Deduplication\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].DedupeMetafilesTemporaryFootprint = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].DedupeMetafilesTemporaryFootprintPercent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    '^\s*Cross Volume Deduplication\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].CrossVolumeDedupeMetafilesFootprint = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].CrossVolumeDedupeMetafilesFootprintPercent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    '^\s*Tape Backup Metadata\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].TapeBackupMetafilesFootprint = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].TapeBackupMetafilesFootprintPercent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    '^\s*Delayed Frees\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].DelayedFreeFootprint = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].DelayedFreeFootprintPercent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    '^\s*Snapmirror Destination\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].SnapmirrorDestinationFootprint = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].SnapmirrorDestinationFootprintPercent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    '^\s*File Operation\s+(?:Metadata\s+)?(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].FileOperationMetadata = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].FileOperationMetadataPercent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                    '^\s*Total\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashtable[$volumeName].TotalFootprint = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashtable[$volumeName].TotalFootprintPercent = $Matches.Percent -replace '\s*%'

                        $Matches.Clear()

                        break
                    }
                }
            }
        }

        $hashtable.GetEnumerator() | ForEach-Object {
            [pscustomobject]$_.Value
        }
    }
}

function Get-AggrSpace {
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseLiteralInitializerForHashtable', '')]
    param (
        [Parameter(Mandatory)]
        [System.Object[]]$AsupObject,

        [Parameter(Mandatory)]
        [System.Object[]]$AggrObjects
    )

    $powerOf = @{
        'B'  = 0
        'KB' = 1
        'MB' = 2
        'GB' = 3
        'TB' = 4
        'PB' = 5
        'EB' = 6
    }

    foreach ($nodeAsup in $AsupObject) {
        Write-Verbose -Message "Capturing AGGR-STATUS-S (node: $( $nodeAsup.SystemName ))"
        if (($aggrStatusS = Get-AsupContent -AsupObject $nodeAsup -SectionName 'AGGR-STATUS-S' -ErrorVariable +NetAppDocsError)) {
            $hashtable = New-Object -TypeName System.Collections.Hashtable

            foreach ($line in $aggrStatusS.Split("`n", [System.StringSplitOptions]::RemoveEmptyEntries)) {
                switch -Regex ($line) {
                    '^Aggregate\s+[:]\s+(?<AggregateName>.+?)\s*$' {
                        $aggregateName = $Matches.AggregateName
                        $aggrObject = $AggrObjects | Where-Object { $_.Name -ceq $Matches.AggregateName }

                        $hashTable[$aggregateName] = [ordered]@{
                            AggregateName            = $aggregateName
                            AggregateUuid            = $aggrObject.Uuid
                            SnapSizeUsed             = [long]$aggrObject.SusedKb * 1024
                            SnapSizeTotal            = [long]$aggrObject.SallocatedKb * 1024
                            VolumeFootprints         = $null
                            VolumeFootprintsPercent  = $null
                            AggregateMetadata        = $null
                            AggregateMetadataPercent = $null
                            SnapshotReserve          = $null
                            SnapshotReservePercent   = $null
                            TotalUsed                = $null
                            TotalUsedPercent         = $null
                            PhysicalUsed             = $null
                            PhysicalUsedPercent      = $null
                        }

                        $Matches.Clear()

                        break
                    }
                    '^\s*Volume Footprints\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashTable[$aggregateName].VolumeFootprints = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashTable[$aggregateName].VolumeFootprintsPercent = ($Matches.Percent -replace '\s*%')

                        $Matches.Clear()

                        break
                    }
                    '^\s*Aggregate Metadata\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashTable[$aggregateName].AggregateMetadata = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashTable[$aggregateName].AggregateMetadataPercent = ($Matches.Percent -replace '\s*%')

                        $Matches.Clear()

                        break
                    }
                    '^\s*Snapshot Reserve\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashTable[$aggregateName].SnapshotReserve = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashTable[$aggregateName].SnapshotReservePercent = ($Matches.Percent -replace '\s*%')

                        $Matches.Clear()

                        break
                    }
                    '^\s*Total Used\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashTable[$aggregateName].TotalUsed = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashTable[$aggregateName].TotalUsedPercent = ($Matches.Percent -replace '\s*%')

                        $Matches.Clear()

                        break
                    }
                    '^\s*Total Physical Used\s+(?<Size>.+?)(?<Unit>B|KB|MB|GB|TB|PB|EB)\s+(?<Percent>\S+)\s*$' {
                        $hashTable[$aggregateName].PhysicalUsed = [long]([double]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit]))
                        $hashTable[$aggregateName].PhysicalUsedPercent = ($Matches.Percent -replace '\s*%')

                        $Matches.Clear()

                        break
                    }
                    default { }
                }
            }

            $hashtable.GetEnumerator() | ForEach-Object {
                [pscustomobject]$_.Value
            }
        }
    }
}

function Get-FirstValidAsupObjectForSection {
    [CmdletBinding()]
    param (
        [Parameter(Mandatory)]
        [System.Object[]]$SourceAsup,

        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$MasterNodeName,

        [Parameter(Mandatory)]
        [string]$SectionName
    )

    $importance = @( $MasterNodeName, ($SourceAsup | Where-Object { $_.SystemName -ne $MasterNodeName } | Sort-Object -Property AsupId -Descending).SystemName | Where-Object { $_ } )

    $masterAsup = $SourceAsup | Sort-Object -Property { $importance.IndexOf($_.SystemName) } | Where-Object {
        -not $_.Manifest -or (
            $_.Manifest | Where-Object {
                ($_.BodyFile -eq $SectionName -or $_.BodyFile -eq $( $SectionName -replace '-', '_' )) -and (
                    $_.Status -notmatch '(?:collection|delivery)[-](?:truncated|skipped)[-](?:size|time)[-]limit' -and $_.Status -ne 'collection-incomplete' -and $_.Status -ne 'content-empty'
                )
            }
        )
    } | Select-Object -First 1

    if (-not $masterAsup) {
        $masterAsup = $SourceAsup | Where-Object { $_.SystemName -ceq $importance[0] }
    }

    $masterAsup
}

function Get-DiskPartitionOwnership {
    param (
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [AllowNull()]
        [string[]]$InputObject
    )

    if ($InputObject) {
        $diskPartitionOwnershipRegex = '(?s)^(?<DiskName>\S+)\s+(?<OwnerNodeName>(?:\S+|Not\sOwned))\((?<OwnerSystemId>\S+)\)\s+Pool(?<PoolName>\d)\s+(?<SerialNumber>\S+)\s+(?<HomeNodeName>\S+)\((?<HomeSystemId>\S+)\).*$'

        foreach ($diskPartitionOwnershipInfo in ($InputObject | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })) {
            $diskPartitionOwnershipInfo -split "`n" | Select-Object -Skip 2 | ForEach-Object {
                if ($_ -match $diskPartitionOwnershipRegex) {
                    if ($Matches.OwnerNodeName -eq 'Not Owned') {
                        DiskName = $Matches.DiskName
                        OwnerNodeName = [string]::Empty
                        OwnerSystemId = [string]::Empty
                        PoolName = [string]::Empty
                        SerialNumber = $Matches.SerialNumber
                        HomeNodeName = [string]::Empty
                        HomeSystemId = [string]::Empty
                    }
                    else {
                        [pscustomobject](Get-RegexNamedGroup -Hashtable $Matches)
                    }

                    $Matches.Clear()
                }
            }
        }
    }
}

function Get-StorageDiskObjectInfo {
    param (
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string[]]$InputObject
    )

    $storageDiskRegex = '(?s)^(?=.*\bDisk[:]\s(?<DiskName>\S+)\b)(?=.*\bShelf[:]\s(?<ShelfId>\d+)\b)(?=.*\bBay[:]\s(?<ShelfBay>\S+)\b)(?=.*\bSerial[:]\s(?<SerialNumber>\S+)\b)(?=.*\bUID[:]\s(?<DiskUid>\S+)\b)(?=.*\bDrawer[:]\s(?<Drawer>\S+)\b)?(?=.*\bSlot[:]\s(?<Slot>\S+)\b)?.*$'

    foreach ($storageDiskObjectInfo in ($InputObject | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })) {
        $storageDiskObjectInfo -split '(?=Disk[:]\s)' | Select-Object -Skip 1 | ForEach-Object {
            if ($_ -match $storageDiskRegex) {
                [pscustomobject](Get-RegexNamedGroup -Hashtable $Matches)

                $Matches.Clear()
            }
        }
    }
}

<#function Get-NodeOptions {
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseSingularNouns', '')]
    param (
        [Parameter(Mandatory, ValueFromPipeline)]
        [AllowNull()]
        [string[]]$InputObject
    )

    begin {
        $optionRegex = '(?s)^\s*(?<Name>\S+)\s+(?<Value>[^\(\s]*).*$'
    }
    process {
        foreach ($option in ($InputObject | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })) {
            $option -split "`n" | ForEach-Object {
                if ($_ -match $optionRegex) {
                    [pscustomobject](Get-RegexNamedGroup -Hashtable $Matches)

                    $Matches.Clear()
                }
            }
        }
    }
}#>

function Test-AsupRestConnectivity {
    try {
        [void](New-Object -TypeName System.Net.WebClient).DownloadString($Script:AsupSectionRestUri)

        $true
    }
    catch {
        $Global:Error.RemoveAt(0)

        $false
    }
}

function Get-AiqRestEndpoint {
    [CmdletBinding()]
    [OutputType([System.Management.Automation.PSCustomObject])]
    param (
        [Parameter()]
        [string]$PathAndQuery,

        [Parameter()]
        [int]$TimeoutSec = 300,

        [Parameter()]
        [int]$MaxRetryCount = 3,

        [Parameter()]
        [int]$RetryIntervalSec = 1
    )

    [Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12

    $uri = [string]::Format('{0}{1}', $Script:AiqPublicRestUri, $PathAndQuery)

    try {
        $retries = 0

        while ($retries -lt $MaxRetryCount) {
            try {
                $response = Invoke-RestMethod -Uri $Uri -Method Get -Headers @{
                    'authorizationToken' = $Script:AuthToken.AccessToken
                    'accept'             = 'application/json'
                } -TimeoutSec $TimeoutSec -Verbose:$false -ErrorAction Stop

                break
            }
            catch {
                $Global:Error.RemoveAt(0)

                $retries++

                if ($retries -eq $MaxRetryCount) {
                    Write-Warning -Message 'Reached the max retry count'

                    throw
                }

                Write-Debug -Message "Retry attempt $retries of $maxRetryCount"

                Start-Sleep -Seconds $RetryIntervalSec
            }
        }

        $response.results
    }
    catch {
        $Global:Error.RemoveAt(0)

        $errorRecord = New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("Unable to fetch AIQ URL. Reason: $( $_.Exception.Message )", 'InvalidResult', [System.Management.Automation.ErrorCategory]::InvalidResult, $uri)

        $errorRecord.CategoryInfo.TargetType = 'URI'

        $PSCmdlet.ThrowTerminatingError($errorRecord)
    }
    finally {
        # Invoke-RestMethod may not properly dispose the connection
        $servicePoint = [System.Net.ServicePointManager]::FindServicePoint($Uri)
        [void]$servicePoint.CloseConnectionGroup('')
    }
}

function Test-UserHasApiAccess {
    $uri = [NetAppDocs.AsupHelper]::GetAsupRbacTestUri($Script:UseStaging)

    $timeoutSec = 60
    $maxRetryCount = 3
    $retryIntervalSec = 1

    try {
        $retries = 0

        while ($retries -lt 3) {
            try {
                [void](
                    Invoke-RestMethod -Uri $Uri -Method Get -Headers @{
                        authorizationToken = $Script:AuthToken.AccessToken
                    } -TimeoutSec $timeoutSec -Verbose:$false -ErrorAction Stop
                )

                break
            }
            catch {
                $Global:Error.RemoveAt(0)

                $retries++

                if ($retries -ge $maxRetryCount) {
                    Write-Warning -Message 'Reached the max retry count'

                    throw
                }

                Write-Debug "Retry attempt $retries of $maxRetryCount"

                Start-Sleep -Seconds $retryIntervalSec
            }
        }

        return $true
    }
    catch {
        if ($_.Exception.Message -like '*Unauthorized*') {
            $Global:Error.RemoveAt(0)

            $msg = New-Object -TypeName System.Text.StringBuilder

            [void]$msg.AppendLine('User is not authorized to access ASUP REST APIs.')
            [void]$msg.AppendLine('-------------------------------------------------------------------------------------')
            [void]$msg.AppendLine('In accordance with NetApp Enterprise role-based access control (RBAC) requirements,')
            [void]$msg.AppendLine('user accounts outside pre-authorized jobcodes / functions have been disabled.')
            [void]$msg.AppendLine('Please read this FAQ (https://confluence.ngage.netapp.com/display/ACE/FAQ+for+Active+IQ+Least+Privilege+Access+changes)')
            [void]$msg.AppendLine('on who is affected and why. If you require access to NetAppDocs, please submit an')
            [void]$msg.AppendLine('Access Request via ServiceNow (https://netapp.service-now.com/esp?id=sc_cat_item&sys_id=b0d4a4e11b577010b65e968f034bcbfe)')
            [void]$msg.AppendLine('to (re)establish your login to NetAppDocs.')
            [void]$msg.AppendLine('-------------------------------------------------------------------------------------')
            [void]$msg.AppendLine()

            throw $msg.ToString()
        }
        else {
            throw 'User is not able to access ASUP REST APIs.'
        }
    }
    finally {
        # Invoke-RestMethod may not properly dispose the connection
        $servicePoint = [System.Net.ServicePointManager]::FindServicePoint($uri)
        [void]$servicePoint.CloseConnectionGroup('')
    }
}

function ConvertTo-AsupObject {
    param (
        [Parameter(Mandatory)]
        [System.Xml.XmlElement[]]$AsupSet
    )

    $asupObject = @(
        $AsupSet | Select-Object -Property @{
            Name       = 'ClusterName'
            Expression = { $_.cluster_name }
        }, @{
            Name       = 'ClusterIdentifier'
            Expression = { $_.cluster_identifier }
        }, @{
            Name       = 'SystemName'
            Expression = { $_.hostname }
        }, @{
            Name       = 'SystemSerialNumber'
            Expression = { $_.sys_serial_no }
        }, @{
            Name       = 'SystemId'
            Expression = { $_.system_id }
        }, @{
            Name       = 'SystemVersion'
            Expression = { $_.sys_version }
        }, @{
            Name       = 'SystemModel'
            Expression = { $_.sys_model }
        }, @{
            Name       = 'SystemOperatingMode'
            Expression = { $_.sys_operating_mode }
        }, @{
            Name       = 'CustomerName'
            Expression = { $_.customer_name }
        }, @{
            Name       = 'CustomerId'
            Expression = { $_.customer_id }
        }, @{
            Name       = 'GroupName'
            Expression = { $_.group_name }
        }, @{
            Name       = 'GroupId'
            Expression = { $_.group_id }
        }, @{
            Name       = 'SiteName'
            Expression = { $_.site_name }
        }, @{
            Name       = 'SiteId'
            Expression = { $_.site_id }
        }, @{
            Name       = 'IsMinimal'
            Expression = { $_.asup_is_minimal }
        }, @{
            Name       = 'Warranty'
            Expression = {
                $_.warranty | Select-Object -Unique | ForEach-Object {
                    [pscustomobject]@{
                        'system_contract'   = $_.sys_contracts.system_contract
                        'warranty_end_date' = $_.warranty_end_date
                    }
                }
            }
        }, @{
            Name       = 'HaGroup'
            Expression = { $_.ha_group | Convert-GenericAsupXmlNode }
        }, @{
            Name       = 'AsupGenerationDate'
            Expression = { [datetime]::Parse($_.asup_gen_date) }
        }, @{
            Name       = 'AsupGenerationZone'
            Expression = { $_.asup_gen_zone }
        }, @{
            Name       = 'AsupSubject'
            Expression = { $_.asup_subject }
        }, @{
            Name       = 'AsupId'
            Expression = { $_.asup_id }
        }, @{
            Name       = 'AsupType'
            Expression = { $_.asup_type }
        }, @{
            Name       = 'BizKey'
            Expression = { $_.biz_key }
        }, @{
            Name       = 'RiskCount'
            Expression = { $_.risk_count }
        }, @{
            Name       = 'EosCount'
            Expression = { $_.eos_count }
        }, @{
            Name       = 'BestPracticesCount'
            Expression = { $_.best_practices_count }
        }, @{
            Name       = 'RlmEnabledCount'
            Expression = { $_.rlm_enabled_count }
        }, @{
            Name       = 'AsupEnabledCount'
            Expression = { $_.asup_enabled_count }
        }, @{
            Name       = 'PerfRiskCount'
            Expression = { $_.perf_risk_count }
        }, @{
            Name       = 'Manifest'
            Expression = { Get-AsupSection -AsupObject $_ -SectionName 'MANIFEST.XML' -InnerProperty 'Manifest' -ErrorAction Ignore }
        }
    )

    , $asupObject
}

# SIG # Begin signature block
# MIIq0AYJKoZIhvcNAQcCoIIqwTCCKr0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCAH4JnMUPWk/xd
# c0UjP2qIWoUt5uO7sO7lyALHxkwSRqCCJGgwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggWNMIIEdaADAgECAhAOmxiO
# +dAt5+/bUOIIQBhaMA0GCSqGSIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYD
# VQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAi
# BgNVBAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAw
# MDBaFw0zMTExMDkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
# Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERp
# Z2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBAL/mkHNo3rvkXUo8MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsb
# hA3EMB/zG6Q4FutWxpdtHauyefLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iT
# cMKyunWZanMylNEQRBAu34LzB4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGb
# NOsFxl7sWxq868nPzaw0QF+xembud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclP
# XuU15zHL2pNe3I6PgNq2kZhAkHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCr
# VYJBMtfbBHMqbpEBfCFM1LyuGwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFP
# ObURWBf3JFxGj2T3wWmIdph2PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTv
# kpI6nj3cAORFJYm2mkQZK37AlLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWM
# cCxBYKqxYxhElRp2Yn72gLD76GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls
# 5Q5SUUd0viastkF13nqsX40/ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBR
# a2+xq4aLT8LWRV+dIPyhHsXAj6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6
# MIIBNjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qY
# rhwPTzAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8E
# BAMCAYYweQYIKwYBBQUHAQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5k
# aWdpY2VydC5jb20wQwYIKwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDig
# NoY0aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNybDARBgNVHSAECjAIMAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCg
# v0NcVec4X6CjdBs9thbX979XB72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQT
# SnovLbc47/T/gLn4offyct4kvFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh
# 65ZyoUi0mcudT6cGAxN3J0TU53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSw
# uKFWjuyk1T3osdz9HNj0d1pcVIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAO
# QGPFmCLBsln1VWvPJ6tsds5vIy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjD
# TZ9ztwGpn1eqXijiuZQwggXMMIIENKADAgECAhAg429sPxgagUb53pPffJfkMA0G
# CSqGSIb3DQEBDAUAMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBS
# MzYwHhcNMjEwOTA5MDAwMDAwWhcNMjMwOTA5MjM1OTU5WjBQMQswCQYDVQQGEwJV
# UzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UECgwMTmV0QXBwLCBJbmMuMRUw
# EwYDVQQDDAxOZXRBcHAsIEluYy4wggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGK
# AoIBgQC4kYYj/oViZD9pN03hrqFjtwOz1Gx4eDWVV8IYUYEr2qpLeNYvWz26B/+E
# mYLboAGVpkYg0Wske3hjRpooZlMORm1S4+2C2NoQCstZ+kmNTevbXHJe/w1VNJrm
# fKjpXOJEfx8GUGDqCMR30dChVpsdBEavrM7T0hnfJLv18i19SGv3a+nhvB3iOjLP
# SEg8+DFkjD5jTI6fQ83DBEiVR4UEYq7yyH578ZEylhsRfJmK+uIREycEUk/NpFTw
# g6/7lxh/WYabCrZzqI4Ep3QataRKaAlxt3BiOlZkX4WGX3DYvSYltWOM2UlCcGpP
# m/1/LN8Xuvf+YO6H472QecHe59XtXbHSMKvqDuOEJu8Wk2LFMNK732o3Fc5QIHie
# 6JtNsgNYf+Vsm5EMBD1ZHNE+C3zlmQbbWIU5uEU1nhiodBPKsu0SkS6pARyKBO05
# DSc2gWWv0aukKi04ZAn+hAdSl5s1dgmU5shBvKHqe15K9ZRN4FFO/CgHE0BoGYQS
# UQVKwa0CAwEAAaOCAZwwggGYMB8GA1UdIwQYMBaAFA8qyyCHKLjsb0iuK1SmKaoX
# pM0MMB0GA1UdDgQWBBQuH643KcBMmb/Q6IZt+H9IrnXFwDAOBgNVHQ8BAf8EBAMC
# B4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDAzARBglghkgBhvhC
# AQEEBAMCBBAwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwIwJTAjBggrBgEFBQcC
# ARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQBMEkGA1UdHwRCMEAw
# PqA8oDqGOGh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVT
# aWduaW5nQ0FSMzYuY3JsMHkGCCsGAQUFBwEBBG0wazBEBggrBgEFBQcwAoY4aHR0
# cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQVIz
# Ni5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqG
# SIb3DQEBDAUAA4IBgQCOoGdXjP8Sif0h3ZvoDsIVfrJvQUdP9pZExRJGGj/Te6ML
# XyqHNc/G2WklNv+BC4ryNJ4zp0oneHOub55ejrYPfhgP9oFWS8oOPJbrp3rAtINa
# OeKRC88WUYZSKYAxSX6kubEjnQD6cmazXjxVN6P8+6q9smeqF3xI+++c+ekMw3Xv
# 4EWRYXqN0/srfFp1WpoluwSQcZMqj9haSX0bgO6AAWH2AnVJMfXTIKz/0FQ/RW0y
# Ed5QYQqYInhA7IUz9dBziNpJlwONeF5j7xzgfYDY63WU6WrgJnGjYkQCOOsUx74j
# gUiMRe9zV48GS8Vxz22c/TQs0x4/1GmOSJvUHEk3GseBmB3v+yEhh/D6zWOfYP4X
# D/9b91CxmugTuepuaJSeRg+qUm3KarKsOCUF/CLqUrNh/JwKrWD1cghRaYLvMucs
# ScksHHe7ZDvb2OtvxWXjPk1d1NKvEwFJSS6hIVIbug9x28AJqOoP5Pn9wZvJSzvW
# uQJGCLscdGyKefdCo30wggYaMIIEAqADAgECAhBiHW0MUgGeO5B5FSCJIRwKMA0G
# CSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290
# IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFQxCzAJBgNVBAYT
# AkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28g
# UHVibGljIENvZGUgU2lnbmluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEBAQUAA4IB
# jwAwggGKAoIBgQCbK51T+jU/jmAGQ2rAz/V/9shTUxjIztNsfvxYB5UXeWUzCxEe
# AEZGbEN4QMgCsJLZUKhWThj/yPqy0iSZhXkZ6Pg2A2NVDgFigOMYzB2OKhdqfWGV
# oYW3haT29PSTahYkwmMv0b/83nbeECbiMXhSOtbam+/36F09fy1tsB8je/RV0mIk
# 8XL/tfCK6cPuYHE215wzrK0h1SWHTxPbPuYkRdkP05ZwmRmTnAO5/arnY83jeNzh
# P06ShdnRqtZlV59+8yv+KIhE5ILMqgOZYAENHNX9SJDm+qxp4VqpB3MV/h53yl41
# aHU5pledi9lCBbH9JeIkNFICiVHNkRmq4TpxtwfvjsUedyz8rNyfQJy/aOs5b4s+
# ac7IH60B+Ja7TVM+EKv1WuTGwcLmoU3FpOFMbmPj8pz44MPZ1f9+YEQIQty/NQd/
# 2yGgW+ufflcZ/ZE9o1M7a5Jnqf2i2/uMSWymR8r2oQBMdlyh2n5HirY4jKnFH/9g
# Rvd+QOfdRrJZb1sCAwEAAaOCAWQwggFgMB8GA1UdIwQYMBaAFDLrkpr/NZZILyhA
# QnAgNpFcF4XmMB0GA1UdDgQWBBQPKssghyi47G9IritUpimqF6TNDDAOBgNVHQ8B
# Af8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcD
# AzAbBgNVHSAEFDASMAYGBFUdIAAwCAYGZ4EMAQQBMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Um9vdFI0Ni5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ1Jvb3RSNDYu
# cDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQwFAAOCAgEABv+C4XdjNm57oRUgmxP/BP6YdURhw1aVcdGRP4Wh60BAscjW
# 4HL9hcpkOTz5jUug2oeunbYAowbFC2AKK+cMcXIBD0ZdOaWTsyNyBBsMLHqafvIh
# rCymlaS98+QpoBCyKppP0OcxYEdU0hpsaqBBIZOtBajjcw5+w/KeFvPYfLF/ldYp
# mlG+vd0xqlqd099iChnyIMvY5HexjO2AmtsbpVn0OhNcWbWDRF/3sBp6fWXhz7Dc
# ML4iTAWS+MVXeNLj1lJziVKEoroGs9Mlizg0bUMbOalOhOfCipnx8CaLZeVme5yE
# Lg09Jlo8BMe80jO37PU8ejfkP9/uPak7VLwELKxAMcJszkyeiaerlphwoKx1uHRz
# NyE6bxuSKcutisqmKL5OTunAvtONEoteSiabkPVSZ2z76mKnzAfZxCl/3dq3dUNw
# 4rg3sTCggkHSRqTqlLMS7gjrhTqBmzu1L90Y1KWN/Y5JKdGvspbOrTfOXyXvmPL6
# E52z1NZJ6ctuMFBQZH3pwWvqURR8AgQdULUvrxjUYbHHj95Ejza63zdrEcxWLDX6
# xWls/GDnVNueKjWUH3fTv1Y8Wdho698YADR7TNx8X8z2Bev6SivBBOHY+uqiirZt
# g0y9ShQoPzmCcn63Syatatvx157YK9hlcPmVoa1oDE5/L9Uo2bC5a4CH2Rwwggau
# MIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJ
# BgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5k
# aWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAe
# Fw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcw
# FQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3Rl
# ZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9Ge
# TKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0
# hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZl
# jZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAsh
# aG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVY
# TXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1
# biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCir
# c0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+
# DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA
# +bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42Pg
# puE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzS
# M7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU
# uhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6
# mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsG
# AQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29t
# MEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNl
# cnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3Js
# My5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAE
# GTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1Z
# jsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8d
# B+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVp
# P0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp8
# 76i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2
# nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3
# ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQ
# txMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc
# 4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+Y
# AN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZ
# vAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQr
# H4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIGwDCCBKigAwIBAgIQDE1p
# ckuU+jwqSj0pB4A9WjANBgkqhkiG9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4XDTIyMDkyMTAwMDAw
# MFoXDTMzMTEyMTIzNTk1OVowRjELMAkGA1UEBhMCVVMxETAPBgNVBAoTCERpZ2lD
# ZXJ0MSQwIgYDVQQDExtEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMiAtIDIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDP7KUmOsap8mu7jcENmtuh6BSFdDMa
# JqzQHFUeHjZtvJJVDGH0nQl3PRWWCC9rZKT9BoMW15GSOBwxApb7crGXOlWvM+xh
# iummKNuQY1y9iVPgOi2Mh0KuJqTku3h4uXoW4VbGwLpkU7sqFudQSLuIaQyIxvG+
# 4C99O7HKU41Agx7ny3JJKB5MgB6FVueF7fJhvKo6B332q27lZt3iXPUv7Y3UTZWE
# aOOAy2p50dIQkUYp6z4m8rSMzUy5Zsi7qlA4DeWMlF0ZWr/1e0BubxaompyVR4aF
# eT4MXmaMGgokvpyq0py2909ueMQoP6McD1AGN7oI2TWmtR7aeFgdOej4TJEQln5N
# 4d3CraV++C0bH+wrRhijGfY59/XBT3EuiQMRoku7mL/6T+R7Nu8GRORV/zbq5Xwx
# 5/PCUsTmFntafqUlc9vAapkhLWPlWfVNL5AfJ7fSqxTlOGaHUQhr+1NDOdBk+lbP
# 4PQK5hRtZHi7mP2Uw3Mh8y/CLiDXgazT8QfU4b3ZXUtuMZQpi+ZBpGWUwFjl5S4p
# kKa3YWT62SBsGFFguqaBDwklU/G/O+mrBw5qBzliGcnWhX8T2Y15z2LF7OF7ucxn
# EweawXjtxojIsG4yeccLWYONxu71LHx7jstkifGxxLjnU15fVdJ9GSlZA076XepF
# cxyEftfO4tQ6dwIDAQABo4IBizCCAYcwDgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB
# /wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwIAYDVR0gBBkwFzAIBgZngQwB
# BAIwCwYJYIZIAYb9bAcBMB8GA1UdIwQYMBaAFLoW2W1NhS9zKXaaL3WMaiCPnshv
# MB0GA1UdDgQWBBRiit7QYfyPMRTtlwvNPSqUFN9SnDBaBgNVHR8EUzBRME+gTaBL
# hklodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0
# MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3JsMIGQBggrBgEFBQcBAQSBgzCBgDAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFgGCCsGAQUFBzAC
# hkxodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRS
# U0E0MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3J0MA0GCSqGSIb3DQEBCwUAA4IC
# AQBVqioa80bzeFc3MPx140/WhSPx/PmVOZsl5vdyipjDd9Rk/BX7NsJJUSx4iGNV
# CUY5APxp1MqbKfujP8DJAJsTHbCYidx48s18hc1Tna9i4mFmoxQqRYdKmEIrUPwb
# tZ4IMAn65C3XCYl5+QnmiM59G7hqopvBU2AJ6KO4ndetHxy47JhB8PYOgPvk/9+d
# EKfrALpfSo8aOlK06r8JSRU1NlmaD1TSsht/fl4JrXZUinRtytIFZyt26/+YsiaV
# OBmIRBTlClmia+ciPkQh0j8cwJvtfEiy2JIMkU88ZpSvXQJT657inuTTH4YBZJwA
# wuladHUNPeF5iL8cAZfJGSOA1zZaX5YWsWMMxkZAO85dNdRZPkOaGK7DycvD+5sT
# X2q1x+DzBcNZ3ydiK95ByVO5/zQQZ/YmMph7/lxClIGUgp2sCovGSxVK05iQRWAz
# gOAj3vgDpPZFR+XOuANCR+hBNnF3rf2i6Jd0Ti7aHh2MWsgemtXC8MYiqE+bvdgc
# mlHEL5r2X6cnl7qWLoVXwGDneFZ/au/ClZpLEQLIgpzJGgV8unG1TnqZbPTontRa
# mMifv427GFxD9dAq6OJi7ngE273R+1sKqHB+8JeEeOMIA11HLGOoJTiXAdI/Otrl
# 5fbmm9x+LMz/F0xNAKLY1gEOuIvu5uByVYksJxlh9ncBjDGCBb4wggW6AgEBMGgw
# VDELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDErMCkGA1UE
# AxMiU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIFIzNgIQIONvbD8YGoFG
# +d6T33yX5DANBglghkgBZQMEAgEFAKCBhDAYBgorBgEEAYI3AgEMMQowCKACgACh
# AoAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAM
# BgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCBt73drxs6Lnvan567ZCsyYmz45
# dDoqXwVqYuVV2r+Z3jANBgkqhkiG9w0BAQEFAASCAYAaCZ+TFBFhPeEEpPMAli1X
# rmPXAZJsuOooFdZu/qiW8n//eyHgIW25HPkDed1Rsk7EQ1cBTDyLAl4nLJdOpd4F
# O80xiU417eGg80hf9N9biqCjbWpracsJcy9X+Z9HEGV4i8RS+XKiecgDk6yWQNeQ
# 0/H59nIsWS/fhUoLrd1zlxhg5Izq2Nr8s5kZmKYygitGSgrY9OOtzoP0emIzdTwW
# mxPuTkzDul7mBGe9Xo2Ss9q0SU8+V8zhFkaXR6xka9luV+BR7KlTLS96QbpECLso
# Bl7Tx6VfDppz0DdT+2PJn/RCOTNZb0+ocB8auN/CFZi1GnQHquXhuU8MDprM4u8d
# 5KrnwEaeeJUtn+dBA/GuGE7K0a3y6JNEqbTZeMfsOs745raWH5dSc8TQNu/uZEtB
# YWT07HdvrSzxaRyvtQMWZ+rf2F6pnH2u/rpbod9KuvUYs3JYjpptsghoRczDaM4P
# CtX1pWN4u+iWRMOHjnX+qkoqEKWS2hnnP9MIrXiZ67ihggMgMIIDHAYJKoZIhvcN
# AQkGMYIDDTCCAwkCAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
# cnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
# SEEyNTYgVGltZVN0YW1waW5nIENBAhAMTWlyS5T6PCpKPSkHgD1aMA0GCWCGSAFl
# AwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUx
# DxcNMjMwNDAyMTUyODI1WjAvBgkqhkiG9w0BCQQxIgQgf8tNFVQ9rhJNpfoL7J2B
# MgtmqbjDhbrCo6k7REkVILUwDQYJKoZIhvcNAQEBBQAEggIAZEvpc+yMrJsuX9Mm
# spFCICLh2wOc++YQOz0bv6SdLGLJchRRqUX6JJ04aDNUtZrlQ1ehSQSBLTZiUjCH
# Lbo/7Uah/WLd3Izq11oJ6kuiV8cVwVS127ooCtEyKWfxE36u/2hvVH2hfQ0N0sL0
# Ik7+ilZYGLYQry1AKHE1ci04Tj5f9b9mzuuR1SilVmBFCAIZllZRLwBRzX1Hxe3+
# 2EjvswWjF2oiyCfO4iOfSEyGVKhpPgAiMtu5Q1EcD87c1Y30Pd6tHUPGktFVdDme
# IrzAlUziGwya8JqMNMvqoHGrudDfaATOKUOpft5cqppOm3plc3Q/wla4eXoTjeti
# gEsNz7C+E1PDKDJAZ4tVAKNoGIixDVEauTHR8uNxVvHLXzPQuLUAzXc0hdNKEQVb
# m2Hh01/iZqtSrxzN42incTdv9mZBDi6tVqrjPMjYtomCRRYp8+89oO39EzogbJjE
# /odBrBdlrr72cj66Dhi/N7WGir/2gbg4q+Z9LsO0utg4Ir0vG7/x7+wjCNOmuopr
# eds0VCapIPNqn1mZX7988MnyAoHEz6Y87ww+CWQP2rqQVz5zRTBAOLirkbD6Y9M0
# oycl5gtqcr2ynqCONGrj+4Yuzmy29dVv4KHzByHfWVDmK8P+RvgnE/0jRvn+fe21
# OUPwqUGvflzJakpyx3yshwUCpNc=
# SIG # End signature block
