function Convert-GenericXmlNode {
    [CmdletBinding()]
    param (
        [Parameter(Mandatory, ValueFromPipeline)]
        [AllowNull()]
        [System.Xml.XmlNode]$Node,

        [Parameter()]
        [string]$ReplaceString
    )

    process {
        if ($Node) {
            if ($Node.NodeType -eq 'Text' -or (-not [string]::IsNullOrWhiteSpace($Node.'#text'))) {
                $text = $Node.InnerText -replace '%3A', ':' -replace '%2F', '/' -replace '%05' -replace '%5C', '\' -replace '%24', '$' -replace '%2C', ',' -replace '%3C', '<' -replace '%3E', '>' -replace '%28', '(' -replace '%29', ')' -replace '%26', '&' -replace '%2B', '+' -replace '%0D' -replace '%40', '@' -replace '%25', '%' -replace '%22', '"'

                return (ConvertTo-TypedValue -Text $text)
            }

            $outputObj = [pscustomobject]@{ }

            foreach ($child in $Node.ChildNodes) {
                if ($child) {
                    $correctedName = $(
                        if ($child.LocalName.Contains('_') -or $child.LocalName.Contains('-')) {
                            (Get-Culture).TextInfo.ToTitleCase($child.LocalName) -replace '[_]|[-]'
                        }
                        else {
                            [string]::Concat($child.LocalName.Substring(0, 1).ToUpper(), $child.LocalName.Substring(1))
                        }
                    ) -replace $ReplaceString

                    if ($outputObj.psobject.Properties.Name -contains $correctedName) {
                        $outputObj.$correctedName = $outputObj.$correctedName, (Convert-GenericXmlNode -Node $child)
                    }
                    else {
                        if ($child.ChildNodes.Count -eq 1) {
                            if ($child.FirstChild.ChildNodes.Count -eq 0 -or (($child.LocalName -like '*-info' -and $child.LocalName -ne 'initiator-group-info' -and $child.LocalName -ne 'portset-port-info' -and $child.LocalName -ne 'lif-service-name') -or $child.LocalName -like '*-attributes')) {
                                $outputObj | Add-Member -MemberType NoteProperty -Name $correctedName -Value (Convert-GenericXmlNode -Node $child)
                            }
                            elseif ($child.FirstChild.ChildNodes.Count -eq 1) {
                                $outputObj | Add-Member -MemberType NoteProperty -Name $correctedName -Value (Convert-GenericXmlNode -Node $child.FirstChild)
                            }
                            else {
                                $outputObj | Add-Member -MemberType NoteProperty -Name $correctedName -Value ($child.ChildNodes | Convert-GenericXmlNode)
                            }
                        }
                        elseif (($child.ChildNodes.LocalName | Select-Object -Unique | Measure-Object).Count -eq 1) {
                            $outputObj | Add-Member -MemberType NoteProperty -Name $correctedName -Value ($child.ChildNodes | Convert-GenericXmlNode)
                        }
                        else {
                            $outputObj | Add-Member -MemberType NoteProperty -Name $correctedName -Value (Convert-GenericXmlNode -Node $child)
                        }
                    }
                }
            }

            if (@( $outputObj.psobject.Properties ).Count -gt 0) {
                $outputObj
            }
        }
    }
}

function Get-SasExpanderMap {
    <#
        .SYNOPSIS
            Parses the raw 'sasadmin expander_map' output string to help identify the SAS shelf cabling.

        .DESCRIPTION
            This function returns shelf cabling 'level' by parsing the raw sasadmin expander_map output. It returns
            the channel and shelf SSN's/IDs by level.

        .NOTES
            CAVEAT : This function parses raw text output, so at any point this may break due to changes in the text.
    #>

    param (
        # The raw output string from the 'sasadmin expander_map' output.
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$SasExpanderMap
    )

    $sasExpanderRegex = [regex]"^\s*Level\s+(?<Level>\d+)[:]\s+WWN\s+(?<WWN>\w+)[,]\s+ID\s+(?<ShelfID>\d+)[,]\s+Serial Number\s+[']*\s*(?<SerialNumber>\w+)[']*[,]\s+Product\s+[']*\s*(?<Product>\w+)\s*[']*[,]\s+Rev\s*[']*\s*(?<FirmwareRevision>\w+)\s*[']*[,]\s*Slot\s*(?<Module>\w+)\s*$"

    foreach ($line in ($SasExpanderMap.Split("`n") | Select-String -Pattern 'Node', 'Expander', 'Level' -SimpleMatch)) {
        switch -Regex ($line) {
            '^\s*Node[:]\s*(?<Node>.*)' {
                $node = $Matches.Node.Trim()

                if ($Matches) {
                    $Matches.Clear()
                }

                break
            }
            'Expanders on channel (?<Channel>.*)[:]' {
                $channel = $Matches.Channel.Trim()

                if ($Matches) {
                    $Matches.Clear()
                }

                break
            }
            $sasExpanderRegex {
                $outputObj = [pscustomobject]@{
                    NodeName = $node
                    Channel  = $channel
                }

                $Matches.Keys | Where-Object { $_ -ne '0' } | ForEach-Object {
                    $outputObj | Add-Member -MemberType NoteProperty -Name $_.Trim() -Value $Matches[$_].Trim()
                }

                if ($Matches) {
                    $Matches.Clear()
                }

                $outputObj | Select-Object -Property NodeName, Channel, Level, WWN, ShelfID, SerialNumber, Product, FirmwareRevision, Module

                break
            }
            default { }
        }
    }
}

function Get-NtapExpansionSlotInventory {
    <#
        .SYNOPSIS
            Parses the raw sysconfig -ac string to gather PCI card information.

        .DESCRIPTION
            This function returns PCI card inventory information by parsing the raw sysconfig -ac output. It returns
            the slot number, part number and description.

        .NOTES
            CAVEAT : This function parses raw text output, so at any point this may break due to changes in the text.
    #>

    param (
        # The raw output string from the 'sysconfig -ac' output.
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$SysconfigAc,

        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$SysconfigVorA
    )

    $adapterCardRegex = [regex]'(?:.*slot\s(?<SlotNumber>\d+)\s.*\:\s+(?<PartNumber>.*)\:\s+(?<Description>.*)$?)'
    $sasAdapterCardRegex = [regex]'(?s)(?:.*slot\s(?<SlotNumber>\d+)[:](?:.*?MFG\sPart\sNumber[:].*?rev[.]\s*(?<Revision>.*?)\s*\n)?(?:.*?Date\sCode[:]\s*(?<DateCode>\S*?)\s*\n)?.*$)'

    $adapterCards = @( )

    foreach ($line in ($SysconfigAc.Split("`n") | Select-String -Pattern 'Node', 'slot' -SimpleMatch)) {
        switch -Regex ($line) {
            '^\s*Node[:]\s*(?<Node>.*)' {
                $node = $Matches.Node.Trim()

                if ($Matches) {
                    $Matches.Clear()
                }

                break
            }
            $adapterCardRegex {
                $outputObj = [pscustomobject]@{
                    NodeName    = $node
                    SlotNumber  = $Matches.SlotNumber.Trim()
                    PartNumber  = $Matches.PartNumber.Trim()
                    Description = $Matches.Description.Trim()
                }

                if ($Matches) {
                    $Matches.Clear()
                }

                if ($outputObj.Description -like '*SAS*') {
                    $cardDetails = $SysconfigVorA -split '(?=slot\s\d+[:])' | Where-Object { $_ -like "*slot $( $outputObj.SlotNumber ):*" } | Select-Object -First 1

                    if (($cardDetails -join '') -match $sasAdapterCardRegex) {
                        $outputObj.Description += " (Rev: $( $Matches.Revision ) / DateCode: $( $Matches.DateCode ))"

                        $Matches.Clear()
                    }
                }

                $adapterCards += $outputObj

                break
            }
            default { }
        }
    }

    $adapterCards | Select-Object -Property NodeName, SlotNumber, PartNumber, Description -Unique
}

function Get-NtapRemoteManagement {
    <#
        .SYNOPSIS
            Parses the raw sysconfig -v string to return the remote management information (RLM or BMC).

        .DESCRIPTION
            This returns the remote management related information by parsing the sysconfig -v output.

        .NOTES
            CAVEAT : This function parses raw text output, so at any point this may break due to changes in the text.
    #>

    param (
        # The raw output string from the sysconfig -v output.
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$Sysconfig
    )

    if (-not [string]::IsNullOrWhiteSpace($Sysconfig)) {
        $RemoteMgmtRegex = '\s*(?:Remote\sLAN\sModule)?(?:IPv4\sconfiguration:)?\s*(?<key>.+):\s+(?<value>.+)'

        $lines = $Sysconfig.Split("`n")

        $nodeLineNumbers = ($lines | Select-String -Pattern '^\s*Node[:]').LineNumber | ForEach-Object { $_ - 2 }

        if (-not $nodeLineNumbers) {
            $nodeLineNumbers = @( 0 )
        }

        for ($i = 0; $i -lt $nodeLineNumbers.Count; $i++) {
            if ($nodeLineNumbers[$i + 1]) {
                $nodeLines = $lines[$nodeLineNumbers[$i]..$nodeLineNumbers[$i + 1]]
            }
            else {
                $nodeLines = $lines[$nodeLineNumbers[$i]..($lines.Count + 1)]
            }

            $node = (($nodeLines | Select-String -Pattern '^\s*Node[:]') -split ':')[1]
            $startLine = ($nodeLines | Select-String -Pattern 'Remote LAN Module|Baseboard Management Controller|(?<!\[)Service Processor').LineNumber

            if (-not $startLine) {
                return $null
            }

            $endLine = ($nodeLines | Select-String -Pattern 'slot\s\d+:|IOXM' | Where-Object { $_.LineNumber -gt $startLine } | Select-Object -First 1).LineNumber

            if (-not $endLine) {
                $endLine = $nodeLines.Count + 1
            }

            $outputObj = [pscustomobject]@{
                NodeName = $( if (-not [string]::IsNullOrWhiteSpace($node)) { $node.Trim() } )
            }

            foreach ($line in ($nodeLines)[($startLine - 1)..($endLine - 2)]) {
                switch -Regex ($line) {
                    'Remote LAN Module' {
                        $outputObj | Add-Member -MemberType NoteProperty -Name Type -Value 'RLM'

                        if ($Matches) {
                            $Matches.Clear()
                        }

                        break
                    }
                    'Baseboard Management Controller' {
                        $outputObj | Add-Member -MemberType NoteProperty -Name Type -Value 'BMC'

                        if ($Matches) {
                            $Matches.Clear()
                        }

                        break
                    }
                    '(?<!\[)Service Processor.*' {
                        $outputObj | Add-Member -MemberType NoteProperty -Name Type -Value 'SP'

                        if ($Matches) {
                            $Matches.Clear()
                        }

                        break
                    }
                    $RemoteMgmtRegex {
                        #The -replace's are due to different property names returned from BMC
                        $outputObj | Add-Member -MemberType NoteProperty -Name $( $Matches.Key.Trim() -replace '\s' -replace 'IPaddress', 'IPAddress' -replace 'IPMask', 'Netmask' -replace 'GatewayIPaddress', 'Gateway' -replace '^DHCP', 'UsingDHCP' -replace 'BMCMACaddress', 'MgmtMACAddress' -replace 'ServiceProcessorStatus', 'Status' ) -Value $Matches.Value.Trim()

                        if ($Matches) {
                            $Matches.Clear()
                        }

                        break
                    }
                    default { }
                }
            }

            $outputObj | Add-Member -MemberType NoteProperty -Name NetmaskLength -Value $(
                if (-not [string]::IsNullOrWhiteSpace($outputObj.NetMask)) {
                    (ConvertTo-MaskLength -Netmask $outputObj.Netmask)
                }
            )

            $outputObj
        }
    }
}

function Get-AdditionalSysconfigProperties {
    <#
        .SYNOPSIS
            Parses the raw sysconfig string to return the IOXM module status and Loader version.

        .DESCRIPTION
            This returns the IOXM module status and Loader version by parsing the sysconfig output.

        .NOTES
            CAVEAT : This function parses raw text output, so at any point this may break due to changes in the text.
    #>

    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseSingularNouns', '')]
    param (
        # The raw output string from the sysconfig output.
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$Sysconfig
    )

    if (-not [string]::IsNullOrWhiteSpace($Sysconfig)) {
        $lines = $Sysconfig.Split("`n")

        $nodeLineNumbers = ($lines | Select-String -Pattern '^\s*Node[:]').LineNumber | ForEach-Object { $_ - 2 }

        if (-not $nodeLineNumbers) {
            $nodeLineNumbers = @( 0 )
        }

        for ($i = 0; $i -lt $nodeLineNumbers.Count; $i++) {
            if ($nodeLineNumbers[$i + 1]) {
                $nodeLines = $lines[$nodeLineNumbers[$i]..$nodeLineNumbers[$i + 1]]
            }
            else {
                $nodeLines = $lines[$nodeLineNumbers[$i]..($lines.Count + 1)]
            }

            $node = (($nodeLines | Select-String -Pattern '^\s*Node[:]') -split ':\s*')[1]
            [pscustomobject]@{
                NodeName      = $( if (-not [string]::IsNullOrWhiteSpace($node)) { $node.Trim() } )
                IoxmPresent   = [bool](Select-String -InputObject $nodeLines -Pattern 'IOXM\s+Status:\s+present' -Quiet)
                LoaderVersion = (($nodeLines | Select-String -Pattern 'Loader version[:]') -split ':\s*(.+?)\s*$')[1]
            }
        }
    }
}

function Get-AggrSnapInformation {
    param (
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$SnapList,

        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$SnapSched
    )

    if ([string]::IsNullOrWhiteSpace($SnapList) -or [string]::IsNullOrWhiteSpace($SnapSched)) {
        return $null
    }

    function Get-AggrSnapList {
        param (
            [Parameter(Mandatory)]
            [string]$SnapList
        )

        foreach ($line in ($SnapList.Split("`n", [System.StringSplitOptions]::RemoveEmptyEntries) | Where-Object { $_ -notmatch '\d+ entries were acted on' })) {
            switch -Regex ($line) {
                'Node[:]\s*(?<Node>.*)\s*$' {
                    $node = $Matches.Node.Trim()

                    break
                }
                'Aggregate\s(?<Aggregate>.+)\s*$' {
                    $aggregate = $Matches.Aggregate.Trim()

                    if ($outputObj) {
                        $outputObj
                    }

                    $outputObj = [pscustomobject]@{
                        Node      = $node
                        Aggregate = $aggregate
                        Lines     = @( )
                    }

                    break
                }
                default {
                    if ($outputObj) {
                        $outputObj.Lines += $line
                    }
                }
            }

            if ($Matches) {
                $Matches.Clear()
            }
        }

        $outputObj
    }

    $aggrSnapList = Get-AggrSnapList -SnapList $SnapList

    foreach ($line in ($SnapSched.Split("`n") | Where-Object { $_ -notmatch '\d+ entries were acted on' })) {
        switch -Regex ($line) {
            'Node[:]\s*(?<Node>.+)\s*$' {
                $node = $Matches.Node.Trim()

                break
            }
            'Aggregate\s(.+)\s*$' {
                $aggrName, $schedule = ($Matches[1].Trim() -split '[:]\s+')[0, 1]

                [pscustomobject]@{
                    NodeName      = $node
                    AggregateName = $aggrName
                    Schedule      = $schedule
                    Count         = (($aggrSnapList | Where-Object { $_.Node -ceq $node -and $_.Aggregate -ceq $aggrName }).Lines | Where-Object { $_ -notmatch '^(?:No snapshots exist|working|^\s*$|\s*[%]|\s*[-]{5,}|Aggregate)' } | Measure-Object).Count
                }

                break
            }
        }

        if ($Matches) {
            $Matches.Clear()
        }
    }
}

function Convert-SystemCliFieldDelimitedText {
    param (
        [Parameter(Mandatory, ValueFromPipeline)]
        [AllowEmptyString()]
        [string]$Data,

        [Parameter()]
        [string]$Delimiter = '~'
    )

    process {
        if (-not [string]::IsNullOrWhiteSpace($Data)) {
            $entries = $Data -split "`n" -replace [regex]::Escape($Delimiter), '|' -replace '[|]\s*$' | Where-Object { $_ -notmatch '^\s*$' -and $_ -notmatch '^\s*Last login time[:].*$' } | ConvertFrom-Csv -Delimiter '|' | Select-Object -Skip 1

            if ($entries) {
                foreach ($entry in $entries) {
                    $hashtable = @{ }

                    foreach ($property in $entry.psobject.Properties.Name) {
                        $name = ((Get-Culture).TextInfo.ToTitleCase($property.ToLower().Trim()) -replace '-')

                        $hashtable.Add($name, (ConvertTo-TypedValue -Text $entry.$property))
                    }

                    if ($PSVersionTable.PSVersion -eq [version]::Parse('3.0')) {
                        New-Object -TypeName System.Management.Automation.PSObject -Property $hashtable
                    }
                    else {
                        [pscustomobject]$hashtable
                    }
                }
            }
        }
    }
}

function Convert-SystemCliTextInstance {
    param (
        [Parameter(Mandatory, ValueFromPipeline)]
        [AllowEmptyString()]
        [string]$Data
    )

    process {
        if (-not [string]::IsNullOrWhiteSpace($Data)) {
            $outputObj = @( )
            $hashtable = @{ }
            $name = $null

            foreach ($line in ($Data -split "`n" | Where-Object { $_ -notmatch '--' })) {
                $line = $line -replace "`'"

                switch ($line) {
                    { $line -match '^\s*$|Last login time[:]|entries were displayed|operation completed successfully|Warning[:]' } {
                        if ($hashtable.Keys.Count -gt 0) {
                            $outputObj += $(
                                if ($PSVersionTable.PSVersion -eq [version]::Parse('3.0')) {
                                    New-Object -TypeName System.Management.Automation.PSObject -Property $hashtable
                                }
                                else {
                                    [pscustomobject]$hashtable
                                }
                            )

                            $hashtable = @{ }
                            $name = $null
                        }

                        break
                    }
                    { $line -like '*no entries matching your query*' } {
                        break
                    }
                    #This is a hack for wrapped/multiline raw text
                    { $line -notmatch '[:]\s+' } {
                        if (-not [string]::IsNullOrWhiteSpace($name)) {
                            if ($hashtable[$name] -is [System.Boolean]) {
                                $hashtable[$name] = $hashtable[$name].ToString()
                            }

                            $hashtable[$name] += [string]::Format("`n{0}", $line.Trim())
                        }

                        break
                    }
                    default {
                        $lineSplit = $line -split '[:]\s+', 2 | ForEach-Object { $_.TrimEnd().Trim('"').Trim("'") }
                        $name, $value = ((Get-Culture).TextInfo.ToTitleCase($lineSplit[0].ToLower()) -replace '\(s\)' -replace '\(DEPRECATED\)[-]' -replace '\s*' -replace '\((.*)\)[-]?', '$1' -replace '/' -replace '\\' -replace '\?$' -replace '-' -replace '_' -replace ','), $lineSplit[1]

                        $hashtable[$name] = ConvertTo-TypedValue -Text $value
                    }
                }
            }

            if ($hashtable.Keys.Count -gt 0) {
                $outputObj += $(
                    if ($PSVersionTable.PSVersion -eq [version]::Parse('3.0')) {
                        New-Object -TypeName System.Management.Automation.PSObject -Property $hashtable
                    }
                    else {
                        [pscustomobject]$hashtable
                    }
                )

                $hashtable = @{ }
            }

            $outputObj
        }
    }
}

function Convert-FailoverTarget {
    param (
        [Parameter(Mandatory, ValueFromPipeline)]
        [AllowEmptyString()]
        [string]$FailoverTarget
    )

    process {
        if (-not [string]::IsNullOrWhiteSpace($FailoverTarget)) {
            $failoverTargetLines = $FailoverTarget -split "`n" | Where-Object { -not [string]::IsNullOrWhiteSpace($_) }
            $vserver = $null

            $startLine = ($failoverTargetLines | Select-String -Pattern '^\s*-{5,}.+').LineNumber

            if ($startLine -gt 0) {
                for ($n = $startLine; $n -le ($failoverTargetLines | Measure-Object).Count; $n++) {
                    switch -Regex ($failoverTargetLines[$n]) {
                        '^\S+\s*$' {
                            if ($n -gt 4) {
                                $outputObj
                            }

                            $vserver = $_.Trim()

                            $outputObj = [pscustomobject]@{
                                Vserver       = $vserver
                                InterfaceName = $null
                                Targets       = @( )
                            }

                            break
                        }
                        '^\s{9,15}\w+.+$' {
                            if (-not [string]::IsNullOrWhiteSpace($outputObj.InterfaceName)) {
                                $outputObj

                                $outputObj = [pscustomobject]@{
                                    Vserver       = $vserver
                                    InterfaceName = $null
                                    Targets       = @( )
                                }
                            }

                            $outputObj.InterfaceName = $( $_ -replace '^\s{9}' ).TrimEnd() -split '(?<=\S)\s+' | Select-Object -First 1

                            break
                        }
                        'Failover Targets:' {
                            $outputObj.Targets += ($_ -replace 'Failover Targets:\s') -split '\s*,\s*' | Where-Object { -not [string]::IsNullOrWhiteSpace($_) } | ForEach-Object { $_.Trim() }

                            break
                        }
                        '^\s*$' {
                            $outputObj

                            break
                        }
                        'entries were displayed' {
                            return $outputObj
                        }
                        default {
                            if ($outputObj.Targets) {
                                $outputObj.Targets += $_ -split '\s*,\s*' | Where-Object { -not [string]::IsNullOrWhiteSpace($_) } | ForEach-Object { $_.Trim() }
                            }
                        }
                    }
                }
            }
        }
    }
}

function ConvertTo-TypedValue {
    param (
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$Text
    )

    if ([string]::IsNullOrWhiteSpace($Text)) {
        return $null
    }

    # Convert newline separated values to an array
    if ($Text -match "`n") {
        return ($Text -split '\n' | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })
    }

    # try to convert to booleans
    $bool = $false

    if ([bool]::TryParse($Text, [ref]$bool)) {
        return $bool
    }

    #try to convert to date
    $date = $null

    if (($date = Convert-DateTimeString -String $Text -Format 'ddd MMM d HH:mm:ss yyyy')) {
        return $date
    }

    return $Text
}

function Get-SantricityPlatformConfig {
    param (
        [Parameter(Mandatory, Position = 0)]
        [pscustomobject]$HardwareInventoryInfo,

        [Parameter(Mandatory, Position = 1)]
        [bool]$SimplexModeEnabled
    )

    [string]::Format('E{0}{1} {2} {3}GB',
        $Controller.ModelName.Substring(0, $Controller.ModelName.Length - 2),
        $HardwareInventoryInfo.Trays[0].NumDriveSlotsPerCompartment,
        ('Simplex', 'Duplex')[[byte](! $SimplexModeEnabled)],
        [System.Math]::Ceiling((($HardwareInventoryInfo.Controllers | Measure-Object -Property PhysicalCacheMemorySize -Sum).Sum / 1024))
    )
}

function Get-PlatformConfig {
    param (
        [Parameter(Mandatory, Position = 0)]
        [string]$SystemModel,

        [Parameter(Mandatory, Position = 1)]
        [bool]$IsClustered,

        [Parameter(Mandatory, Position = 2)]
        [bool]$IoxmPresent
    )

    $platformConfig = $SystemModel

    switch -Regex ($SystemModel) {
        'FAS270' {
            if ($IsClustered) {
                $platformConfig += 'C'
            }

            break
        }
        'FAS9[2|4|6|8]0' {
            if ($IsClustered) {
                $platformConfig += 'C'
            }

            break
        }
        'FAS20[2|4|5]0' {
            if ($IsClustered) {
                $platformConfig += 'HA'
            }

            break
        }
        'FAS2[2|6|7][2|5]0' {
            if ($IsClustered) {
                $platformConfig += 'A'
            }

            break
        }
        'FAS2240[-][2|4]' {
            if ($IsClustered) {
                $platformConfig += 'HA'
            }

            break
        }
        'FAS25[20|52|54]' {
            if ($IsClustered) {
                $platformConfig += 'HA'
            }

            break
        }
        '[FAS|V]30[2|4|5|7]' {
            if ($IsClustered) {
                $platformConfig += 'C'
            }

            break
        }
        '[FAS|V]31[4|6|7]0' {
            if ($IsClustered) {
                $platformConfig += 'HA'
            }

            break
        }
        '[FAS|V]3210' {
            if ($IsClustered) {
                $platformConfig += 'A'
            }

            break
        }
        '[FAS|V]3220' {
            if ($IsClustered) {
                $platformConfig += 'A'
            }

            if ($IoxmPresent) {
                $platformConfig += 'E'
            }

            break
        }
        '[FAS|V]32[4|5|7]0' {
            if ($IsClustered) {
                $platformConfig += 'A'
            }

            $platformConfig += 'E'

            break
        }
        '[FAS|V]60[3|4|7|8]0' {
            if ($IsClustered) {
                $platformConfig += 'HA'
            }

            break
        }
        '[FAS|V]62[1|2]0' {
            if ($IsClustered) {
                $platformConfig += 'A'
            }

            break
        }
        '[FAS|V]62[4|5|8|9]0' {
            if ($IsClustered) {
                $platformConfig += 'A'
            }

            break
        }
        '[FAS|AFF]80[2|4]0' {
            if ($IsClustered) {
                $platformConfig += 'A'
            }

            break
        }
        'FAS8060' {
            if ($IsClustered) {
                $platformConfig += 'A'
            }

            if ($IoxmPresent) {
                $platformConfig += 'E'
            }

            break
        }
        'FAS8080' {
            if ($IsClustered) {
                $platformConfig += 'A'
            }

            $platformConfig += 'E EX'

            break
        }
        'FAS8200' {
            if ($IsClustered) {
                $platformConfig += 'A'
            }

            break
        }
        'FAS9000' {
            if ($IsClustered) {
                $platformConfig += 'A'
            }

            break
        }
        'AFF80[4|6]0' {
            $platformConfig += 'A'

            if ($IoxmPresent) {
                $platformConfig += 'E'
            }

            break
        }
        'AFF8080' {
            $platformConfig += 'A'

            if ($IoxmPresent) {
                $platformConfig += 'E'
            }

            $platformConfig += ' EX'

            break
        }
        'AFF[\s+|-][A|C]\d+' {
            if ($platformConfig -notlike '*s') {
                $platformConfig += 'A'
            }

            break
        }
        default { }
    }

    return $platformConfig
}

function Convert-OntapUnixPermStringToOctal {
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseLiteralInitializerForHashtable', '')]
    param (
        [Parameter(Mandatory)]
        [string]$PermissionString
    )

    $hash = New-Object -TypeName System.Collections.Hashtable

    $hash.Add( 's1', 4000)
    $hash.Add( 's2', 2000)
    $hash.Add( 't3', 1000)
    $hash.Add( 'r4', 400)
    $hash.Add( 'w5', 200)
    $hash.Add( 'x6', 100)
    $hash.Add( 'r7', 40)
    $hash.Add( 'w8', 20)
    $hash.Add( 'x9', 10)
    $hash.Add('r10', 4)
    $hash.Add('w11', 2)
    $hash.Add('x12', 1)

    $value = 0

    for ($i = 0; $i -le 11; $i++) {
        $value += $hash["$( $PermissionString.Substring($i, 1) )$( $i + 1 )"]
    }

    return $value
}

function Set-AllNcTemplateProperties {
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseShouldProcessForStateChangingFunctions', '')]
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseSingularNouns', '')]
    param (
        [Parameter(Mandatory)]
        [System.Object]$Template,

        [Parameter()]
        [string]$PropertyName
    )

    $Local:ErrorActionPreference = 'SilentlyContinue'

    $object = $(
        if (-not [string]::IsNullOrWhiteSpace($PropertyName)) {
            $Template.$PropertyName
        }
        else {
            $Template
        }
    )

    foreach ($prop in ($object.psobject.Properties | Where-Object { $_.Name -ne 'NcController' -and ($_.IsGettable -and $_.IsSettable) })) {
        if ($prop.TypeNameOfValue -like 'DataONTAP*') {
            [void](Set-AllNcTemplateProperties -Template $object -PropertyName $prop.Name)
        }
        else {
            try {
                switch -Wildcard ($prop.TypeNameOfValue) {
                    '*System.DateTime*' {
                        $prop.Value = [datetime]::MinValue

                        break
                    }
                    default {
                        $prop.Value = $true
                    }
                }
            }
            catch {
                $Global:Error.RemoveAt(0)

                $prop.Value = $null
            }
        }
    }

    $Template
}

function ConvertFrom-JobScheduleDescription {
    param (
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$JobScheduleDescription
    )

    if ([string]::IsNullOrWhiteSpace($JobScheduleDescription)) {
        return $null
    }

    $cronFieldAliases = @{
        'Month'     = @{
            'Jan' = 0
            'Feb' = 1
            'Mar' = 2
            'Apr' = 3
            'May' = 4
            'Jun' = 5
            'Jul' = 6
            'Aug' = 7
            'Sep' = 8
            'Oct' = 9
            'Nov' = 10
            'Dec' = 11
        };
        'DayOfWeek' = @{
            'Sun' = 0
            'Mon' = 1
            'Tue' = 2
            'Wed' = 3
            'Thu' = 4
            'Fri' = 5
            'Sat' = 6
        };
    }

    $outputObj = [pscustomobject]@{
        JobScheduleCronMonth       = New-Object -TypeName System.Collections.Generic.List[System.Int32]
        JobScheduleCronDayOfWeek   = New-Object -TypeName System.Collections.Generic.List[System.Int32]
        JobScheduleCronDay         = New-Object -TypeName System.Collections.Generic.List[System.Int32]
        JobScheduleCronHour        = New-Object -TypeName System.Collections.Generic.List[System.Int32]
        JobScheduleCronMinute      = New-Object -TypeName System.Collections.Generic.List[System.Int32]
        JobScheduleIntervalDays    = $null
        JobScheduleIntervalHours   = $null
        JobScheduleIntervalMinutes = $null
        JobScheduleIntervalSeconds = $null
        JobScheduleDescription     = $JobScheduleDescription
    }

    switch -WildCard ($JobScheduleDescription) {
        'Every*' {
            $intervalRegex = '^Every\s(?:(?<Days>\d+)d)?(?:(?<Hours>\d+)h)?(?:(?<Minutes>\d+)m)?(?:(?<Seconds>\d+)s)?'

            if ($JobScheduleDescription -match $intervalRegex) {
                $outputObj.JobScheduleIntervalDays = $Matches['Days']
                $outputObj.JobScheduleIntervalHours = $Matches['Hours']
                $outputObj.JobScheduleIntervalMinutes = $Matches['Minutes']
                $outputObj.JobScheduleIntervalSeconds = $Matches['Seconds']

                $Matches.Clear()
            }
        }
        default {
            $dayAndMonthEntry, $hourAndMinuteEntry = $JobScheduleDescription -split '[@]'

            foreach ($entry in ($dayAndMonthEntry -split '[,|\s]')) {
                if ($entry -like '*-*') {
                    $first, $last = $entry -split '-'

                    if ([int]::TryParse($first, [ref]$null)) {
                        $first .. $last | ForEach-Object {
                            $outputObj.JobScheduleCronDay.Add($_)
                        }
                    }

                    if ($cronFieldAliases['Month'].ContainsKey($first)) {
                        $cronFieldAliases['Month'][$first] .. $cronFieldAliases['Month'][$last] | ForEach-Object {
                            $outputObj.JobScheduleCronMonth.Add($_)
                        }
                    }
                    elseif ($cronFieldAliases['DayOfWeek'].ContainsKey($first)) {
                        $cronFieldAliases['DayOfWeek'][$first] .. $cronFieldAliases['DayOfWeek'][$last] | ForEach-Object {
                            $outputObj.JobScheduleCronDayOfWeek.Add($_)
                        }
                    }
                }
                else {
                    if ([int]::TryParse($entry, [ref]$null)) {
                        $outputObj.JobScheduleCronDay.Add($entry)
                    }
                    else {
                        if ($cronFieldAliases['Month'].ContainsKey($entry)) {
                            $outputObj.JobScheduleCronMonth.Add($cronFieldAliases['Month'][$entry])
                        }
                        elseif ($cronFieldAliases['DayOfWeek'].ContainsKey($entry)) {
                            $outputObj.JobScheduleCronDayOfWeek.Add($cronFieldAliases['DayOfWeek'][$entry])
                        }
                    }
                }
            }

            foreach ($entry in ($hourAndMinuteEntry -split ',')) {
                $hour, $minute = $entry -split ':'

                if ([int]::TryParse($hour, [ref]$null) -and -not $outputObj.JobScheduleCronHour.Contains($hour)) {
                    $outputObj.JobScheduleCronHour.Add($hour)
                }

                if ([int]::TryParse($minute, [ref]$null) -and -not $outputObj.JobScheduleCronMinute.Contains($minute)) {
                    $outputObj.JobScheduleCronMinute.Add($minute)
                }
            }
        }
    }

    $outputObj
}

# SIG # Begin signature block
# MIIq0AYJKoZIhvcNAQcCoIIqwTCCKr0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAxGTKJXxmWjaF3
# 2RuYEBziMvWfJSvIgnIoyZHQ/8q7wKCCJGgwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggWNMIIEdaADAgECAhAOmxiO
# +dAt5+/bUOIIQBhaMA0GCSqGSIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYD
# VQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAi
# BgNVBAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAw
# MDBaFw0zMTExMDkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
# Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERp
# Z2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBAL/mkHNo3rvkXUo8MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsb
# hA3EMB/zG6Q4FutWxpdtHauyefLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iT
# cMKyunWZanMylNEQRBAu34LzB4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGb
# NOsFxl7sWxq868nPzaw0QF+xembud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclP
# XuU15zHL2pNe3I6PgNq2kZhAkHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCr
# VYJBMtfbBHMqbpEBfCFM1LyuGwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFP
# ObURWBf3JFxGj2T3wWmIdph2PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTv
# kpI6nj3cAORFJYm2mkQZK37AlLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWM
# cCxBYKqxYxhElRp2Yn72gLD76GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls
# 5Q5SUUd0viastkF13nqsX40/ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBR
# a2+xq4aLT8LWRV+dIPyhHsXAj6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6
# MIIBNjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qY
# rhwPTzAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8E
# BAMCAYYweQYIKwYBBQUHAQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5k
# aWdpY2VydC5jb20wQwYIKwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDig
# NoY0aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNybDARBgNVHSAECjAIMAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCg
# v0NcVec4X6CjdBs9thbX979XB72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQT
# SnovLbc47/T/gLn4offyct4kvFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh
# 65ZyoUi0mcudT6cGAxN3J0TU53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSw
# uKFWjuyk1T3osdz9HNj0d1pcVIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAO
# QGPFmCLBsln1VWvPJ6tsds5vIy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjD
# TZ9ztwGpn1eqXijiuZQwggXMMIIENKADAgECAhAg429sPxgagUb53pPffJfkMA0G
# CSqGSIb3DQEBDAUAMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBS
# MzYwHhcNMjEwOTA5MDAwMDAwWhcNMjMwOTA5MjM1OTU5WjBQMQswCQYDVQQGEwJV
# UzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UECgwMTmV0QXBwLCBJbmMuMRUw
# EwYDVQQDDAxOZXRBcHAsIEluYy4wggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGK
# AoIBgQC4kYYj/oViZD9pN03hrqFjtwOz1Gx4eDWVV8IYUYEr2qpLeNYvWz26B/+E
# mYLboAGVpkYg0Wske3hjRpooZlMORm1S4+2C2NoQCstZ+kmNTevbXHJe/w1VNJrm
# fKjpXOJEfx8GUGDqCMR30dChVpsdBEavrM7T0hnfJLv18i19SGv3a+nhvB3iOjLP
# SEg8+DFkjD5jTI6fQ83DBEiVR4UEYq7yyH578ZEylhsRfJmK+uIREycEUk/NpFTw
# g6/7lxh/WYabCrZzqI4Ep3QataRKaAlxt3BiOlZkX4WGX3DYvSYltWOM2UlCcGpP
# m/1/LN8Xuvf+YO6H472QecHe59XtXbHSMKvqDuOEJu8Wk2LFMNK732o3Fc5QIHie
# 6JtNsgNYf+Vsm5EMBD1ZHNE+C3zlmQbbWIU5uEU1nhiodBPKsu0SkS6pARyKBO05
# DSc2gWWv0aukKi04ZAn+hAdSl5s1dgmU5shBvKHqe15K9ZRN4FFO/CgHE0BoGYQS
# UQVKwa0CAwEAAaOCAZwwggGYMB8GA1UdIwQYMBaAFA8qyyCHKLjsb0iuK1SmKaoX
# pM0MMB0GA1UdDgQWBBQuH643KcBMmb/Q6IZt+H9IrnXFwDAOBgNVHQ8BAf8EBAMC
# B4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDAzARBglghkgBhvhC
# AQEEBAMCBBAwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwIwJTAjBggrBgEFBQcC
# ARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQBMEkGA1UdHwRCMEAw
# PqA8oDqGOGh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVT
# aWduaW5nQ0FSMzYuY3JsMHkGCCsGAQUFBwEBBG0wazBEBggrBgEFBQcwAoY4aHR0
# cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQVIz
# Ni5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqG
# SIb3DQEBDAUAA4IBgQCOoGdXjP8Sif0h3ZvoDsIVfrJvQUdP9pZExRJGGj/Te6ML
# XyqHNc/G2WklNv+BC4ryNJ4zp0oneHOub55ejrYPfhgP9oFWS8oOPJbrp3rAtINa
# OeKRC88WUYZSKYAxSX6kubEjnQD6cmazXjxVN6P8+6q9smeqF3xI+++c+ekMw3Xv
# 4EWRYXqN0/srfFp1WpoluwSQcZMqj9haSX0bgO6AAWH2AnVJMfXTIKz/0FQ/RW0y
# Ed5QYQqYInhA7IUz9dBziNpJlwONeF5j7xzgfYDY63WU6WrgJnGjYkQCOOsUx74j
# gUiMRe9zV48GS8Vxz22c/TQs0x4/1GmOSJvUHEk3GseBmB3v+yEhh/D6zWOfYP4X
# D/9b91CxmugTuepuaJSeRg+qUm3KarKsOCUF/CLqUrNh/JwKrWD1cghRaYLvMucs
# ScksHHe7ZDvb2OtvxWXjPk1d1NKvEwFJSS6hIVIbug9x28AJqOoP5Pn9wZvJSzvW
# uQJGCLscdGyKefdCo30wggYaMIIEAqADAgECAhBiHW0MUgGeO5B5FSCJIRwKMA0G
# CSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290
# IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFQxCzAJBgNVBAYT
# AkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28g
# UHVibGljIENvZGUgU2lnbmluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEBAQUAA4IB
# jwAwggGKAoIBgQCbK51T+jU/jmAGQ2rAz/V/9shTUxjIztNsfvxYB5UXeWUzCxEe
# AEZGbEN4QMgCsJLZUKhWThj/yPqy0iSZhXkZ6Pg2A2NVDgFigOMYzB2OKhdqfWGV
# oYW3haT29PSTahYkwmMv0b/83nbeECbiMXhSOtbam+/36F09fy1tsB8je/RV0mIk
# 8XL/tfCK6cPuYHE215wzrK0h1SWHTxPbPuYkRdkP05ZwmRmTnAO5/arnY83jeNzh
# P06ShdnRqtZlV59+8yv+KIhE5ILMqgOZYAENHNX9SJDm+qxp4VqpB3MV/h53yl41
# aHU5pledi9lCBbH9JeIkNFICiVHNkRmq4TpxtwfvjsUedyz8rNyfQJy/aOs5b4s+
# ac7IH60B+Ja7TVM+EKv1WuTGwcLmoU3FpOFMbmPj8pz44MPZ1f9+YEQIQty/NQd/
# 2yGgW+ufflcZ/ZE9o1M7a5Jnqf2i2/uMSWymR8r2oQBMdlyh2n5HirY4jKnFH/9g
# Rvd+QOfdRrJZb1sCAwEAAaOCAWQwggFgMB8GA1UdIwQYMBaAFDLrkpr/NZZILyhA
# QnAgNpFcF4XmMB0GA1UdDgQWBBQPKssghyi47G9IritUpimqF6TNDDAOBgNVHQ8B
# Af8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcD
# AzAbBgNVHSAEFDASMAYGBFUdIAAwCAYGZ4EMAQQBMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Um9vdFI0Ni5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ1Jvb3RSNDYu
# cDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQwFAAOCAgEABv+C4XdjNm57oRUgmxP/BP6YdURhw1aVcdGRP4Wh60BAscjW
# 4HL9hcpkOTz5jUug2oeunbYAowbFC2AKK+cMcXIBD0ZdOaWTsyNyBBsMLHqafvIh
# rCymlaS98+QpoBCyKppP0OcxYEdU0hpsaqBBIZOtBajjcw5+w/KeFvPYfLF/ldYp
# mlG+vd0xqlqd099iChnyIMvY5HexjO2AmtsbpVn0OhNcWbWDRF/3sBp6fWXhz7Dc
# ML4iTAWS+MVXeNLj1lJziVKEoroGs9Mlizg0bUMbOalOhOfCipnx8CaLZeVme5yE
# Lg09Jlo8BMe80jO37PU8ejfkP9/uPak7VLwELKxAMcJszkyeiaerlphwoKx1uHRz
# NyE6bxuSKcutisqmKL5OTunAvtONEoteSiabkPVSZ2z76mKnzAfZxCl/3dq3dUNw
# 4rg3sTCggkHSRqTqlLMS7gjrhTqBmzu1L90Y1KWN/Y5JKdGvspbOrTfOXyXvmPL6
# E52z1NZJ6ctuMFBQZH3pwWvqURR8AgQdULUvrxjUYbHHj95Ejza63zdrEcxWLDX6
# xWls/GDnVNueKjWUH3fTv1Y8Wdho698YADR7TNx8X8z2Bev6SivBBOHY+uqiirZt
# g0y9ShQoPzmCcn63Syatatvx157YK9hlcPmVoa1oDE5/L9Uo2bC5a4CH2Rwwggau
# MIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJ
# BgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5k
# aWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAe
# Fw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcw
# FQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3Rl
# ZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9Ge
# TKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0
# hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZl
# jZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAsh
# aG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVY
# TXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1
# biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCir
# c0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+
# DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA
# +bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42Pg
# puE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzS
# M7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU
# uhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6
# mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsG
# AQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29t
# MEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNl
# cnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3Js
# My5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAE
# GTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1Z
# jsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8d
# B+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVp
# P0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp8
# 76i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2
# nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3
# ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQ
# txMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc
# 4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+Y
# AN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZ
# vAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQr
# H4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIGwDCCBKigAwIBAgIQDE1p
# ckuU+jwqSj0pB4A9WjANBgkqhkiG9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4XDTIyMDkyMTAwMDAw
# MFoXDTMzMTEyMTIzNTk1OVowRjELMAkGA1UEBhMCVVMxETAPBgNVBAoTCERpZ2lD
# ZXJ0MSQwIgYDVQQDExtEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMiAtIDIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDP7KUmOsap8mu7jcENmtuh6BSFdDMa
# JqzQHFUeHjZtvJJVDGH0nQl3PRWWCC9rZKT9BoMW15GSOBwxApb7crGXOlWvM+xh
# iummKNuQY1y9iVPgOi2Mh0KuJqTku3h4uXoW4VbGwLpkU7sqFudQSLuIaQyIxvG+
# 4C99O7HKU41Agx7ny3JJKB5MgB6FVueF7fJhvKo6B332q27lZt3iXPUv7Y3UTZWE
# aOOAy2p50dIQkUYp6z4m8rSMzUy5Zsi7qlA4DeWMlF0ZWr/1e0BubxaompyVR4aF
# eT4MXmaMGgokvpyq0py2909ueMQoP6McD1AGN7oI2TWmtR7aeFgdOej4TJEQln5N
# 4d3CraV++C0bH+wrRhijGfY59/XBT3EuiQMRoku7mL/6T+R7Nu8GRORV/zbq5Xwx
# 5/PCUsTmFntafqUlc9vAapkhLWPlWfVNL5AfJ7fSqxTlOGaHUQhr+1NDOdBk+lbP
# 4PQK5hRtZHi7mP2Uw3Mh8y/CLiDXgazT8QfU4b3ZXUtuMZQpi+ZBpGWUwFjl5S4p
# kKa3YWT62SBsGFFguqaBDwklU/G/O+mrBw5qBzliGcnWhX8T2Y15z2LF7OF7ucxn
# EweawXjtxojIsG4yeccLWYONxu71LHx7jstkifGxxLjnU15fVdJ9GSlZA076XepF
# cxyEftfO4tQ6dwIDAQABo4IBizCCAYcwDgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB
# /wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwIAYDVR0gBBkwFzAIBgZngQwB
# BAIwCwYJYIZIAYb9bAcBMB8GA1UdIwQYMBaAFLoW2W1NhS9zKXaaL3WMaiCPnshv
# MB0GA1UdDgQWBBRiit7QYfyPMRTtlwvNPSqUFN9SnDBaBgNVHR8EUzBRME+gTaBL
# hklodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0
# MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3JsMIGQBggrBgEFBQcBAQSBgzCBgDAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFgGCCsGAQUFBzAC
# hkxodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRS
# U0E0MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3J0MA0GCSqGSIb3DQEBCwUAA4IC
# AQBVqioa80bzeFc3MPx140/WhSPx/PmVOZsl5vdyipjDd9Rk/BX7NsJJUSx4iGNV
# CUY5APxp1MqbKfujP8DJAJsTHbCYidx48s18hc1Tna9i4mFmoxQqRYdKmEIrUPwb
# tZ4IMAn65C3XCYl5+QnmiM59G7hqopvBU2AJ6KO4ndetHxy47JhB8PYOgPvk/9+d
# EKfrALpfSo8aOlK06r8JSRU1NlmaD1TSsht/fl4JrXZUinRtytIFZyt26/+YsiaV
# OBmIRBTlClmia+ciPkQh0j8cwJvtfEiy2JIMkU88ZpSvXQJT657inuTTH4YBZJwA
# wuladHUNPeF5iL8cAZfJGSOA1zZaX5YWsWMMxkZAO85dNdRZPkOaGK7DycvD+5sT
# X2q1x+DzBcNZ3ydiK95ByVO5/zQQZ/YmMph7/lxClIGUgp2sCovGSxVK05iQRWAz
# gOAj3vgDpPZFR+XOuANCR+hBNnF3rf2i6Jd0Ti7aHh2MWsgemtXC8MYiqE+bvdgc
# mlHEL5r2X6cnl7qWLoVXwGDneFZ/au/ClZpLEQLIgpzJGgV8unG1TnqZbPTontRa
# mMifv427GFxD9dAq6OJi7ngE273R+1sKqHB+8JeEeOMIA11HLGOoJTiXAdI/Otrl
# 5fbmm9x+LMz/F0xNAKLY1gEOuIvu5uByVYksJxlh9ncBjDGCBb4wggW6AgEBMGgw
# VDELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDErMCkGA1UE
# AxMiU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIFIzNgIQIONvbD8YGoFG
# +d6T33yX5DANBglghkgBZQMEAgEFAKCBhDAYBgorBgEEAYI3AgEMMQowCKACgACh
# AoAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAM
# BgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCBxqgCRSUUGsvK77uf+7DyK9vTD
# RwJJ8dXMmQVUuanYfDANBgkqhkiG9w0BAQEFAASCAYAO2BwSxDocbfe5qNHMrUWT
# A+4dnoYoU3q2bDOqI2uhv9hVqUgheWyBWWLBvYFdF4CpMIgjryI20D96sqGQMQcl
# MCGCUF1/1H3YIbMIZ1BQXcufQ2683s6s+j4bGOyow8DMRlb3JArMxJMEwfNX2SQZ
# sfVuvpZV3cHT7mjzQKTUjgm/5PXlI2Fm3ivBHWOplOOpjUJwWkIqOttISEKHAQC6
# +MFwQWKJYp2rjjLnI34OU8SYf5ADRMvvl58s8go8mPjs6YLOHwKmiEN4Ynoi9TtN
# duAsVQk0BfDcuE5DOwG3SNktgCbvfexRM+ZQwpVcz6UTAT0KvVhDYj8jAi3yS4Ut
# LW1F7z324oJWcQXh1ob0L8pihrlKhvCh0Sv/SGILxtWON+WQfCP4VxkmA6AVC5nG
# O89E7eu8w26DKd1TpDOE2HTuA7KgRk9JX9Jhy2e1Wl1sb0nlDN3MVWtYvdBqn0Ax
# ukTjx1bhYARqPgkNVQiYTUBYjCYjYmM03mV8TbVad+ChggMgMIIDHAYJKoZIhvcN
# AQkGMYIDDTCCAwkCAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
# cnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
# SEEyNTYgVGltZVN0YW1waW5nIENBAhAMTWlyS5T6PCpKPSkHgD1aMA0GCWCGSAFl
# AwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUx
# DxcNMjMwNDAyMTUyODI1WjAvBgkqhkiG9w0BCQQxIgQgJG+OLijtrLrrZ6fjc76F
# TGQjxFhzpGOWPsGWzNL1uNcwDQYJKoZIhvcNAQEBBQAEggIAqK8e3Hozk7SspiKl
# Y0/7VtJwaHFymsrFTuEC0M4c9FBhfh6b3Ty9/mxdurXPfTzy27ZkrDmYfLJYk2LR
# hYGHxKeAsiSwbJFvtLd6waEGpwW65B/ohtdbk9533cqF1Lsz6PnbVumC6ubq5BIN
# biRgcCdGLdOCdsqUATiGmcv1QeYhKmRKM5w70tO3rg8ZRRkff7nsB0OgH3LheHTM
# A+Nh84rsyVOrFJjqwtNrhol/7sCzPDUaKjo62joajxAGkK6RT7eXBGrm3lvstef9
# lXR87GQCZagKYI2AL5XZCrPzmYCqHiZCTp0VyOt4uLEbTV7f6kReOtmL2cV2ABra
# FTXHmd01as1ccUUekut3MR7bvxmgsmuUUC2WO5+UN6tDp9hZ62T7Fi95QQO/IbcC
# LhBv4SOFXSKx+nrdaYzsv5UfJCrD6I6NvlypdRfSCGNN+o+PkIcRHb1N8FfVMMXC
# FT0bf2XLihqpZB6Jns2yn4GWpbKVUfJF0syCVX2ptWbSsEsN0YuypX949YwWKJf9
# hs/SmoeAP4j4qhvDsvwvIPFe8fN5cmca0FD9YAV6/arIXzr0tYsAXahheM6D5huF
# xZRWbhcIdXe5wE+5x65JGtZOBnEW6DhPr9BxjcE9gaz1JhkHrS+8f1EmxhMUrV2I
# MOWC5JqOMbdD5+Qk3zjAiHqA9AI=
# SIG # End signature block
