function ConvertTo-NtapDataRow {
    [CmdletBinding()]
    [OutputType([System.Data.DataRow])]
    param (
        [Parameter(Mandatory, Position = 0, ValueFromPipeline)]
        [AllowNull()]
        [System.Object[]]$InputObject,

        [Parameter(Mandatory)]
        [System.Data.DataTable]$DataTable,

        [Parameter()]
        [switch]$UseCaption
    )

    process {
        foreach ($object in $InputObject) {
            #if ($null -eq $object -or ($object -is [System.Management.Automation.PSCustomObject] -and -not $object.PSStandardMembers.DefaultDisplayPropertySet)) {
            if ($null -eq $object) {
                continue
            }

            $dataRow = $DataTable.NewRow()

            for ($c = 0; $c -lt $DataTable.Columns.Count; $c++) {
                $dataColumn = $DataTable.Columns[$c]

                $propertyValue = $(
                    if (($dataColumn.ColumnName -match 'In[T|G|M|K]b$' -or $DataTable.Columns.Contains("$( $dataColumn.ColumnName )InBytes")) -and ($null -eq $object.psobject.Properties[$dataColumn.ColumnName])) {
                        $inBytesColumnName = "$( $dataColumn.ColumnName -replace 'In[T|G|M|K]b$' )InBytes"

                        if (-not [string]::IsNullOrWhiteSpace($object.$inBytesColumnName)) {
                            if ($dataColumn.ColumnName -like '*InTb') {
                                if ($object.$inBytesColumnName -ge 0) {
                                    $object.$inBytesColumnName / 1TB
                                }
                            }
                            elseif ($dataColumn.ColumnName -like '*InGb') {
                                if ($object.$inBytesColumnName -ge 0) {
                                    $object.$inBytesColumnName / 1GB
                                }
                            }
                            elseif ($dataColumn.ColumnName -like '*InMb') {
                                if ($object.$inBytesColumnName -ge 0) {
                                    $object.$inBytesColumnName / 1MB
                                }
                            }
                            elseif ($dataColumn.ColumnName -like '*InKb') {
                                if ($object.$inBytesColumnName -ge 0) {
                                    $object.$inBytesColumnName / 1KB
                                }
                            }
                            else {
                                if ($dataColumn.ExtendedProperties.ContainsKey('NumberFormat')) {
                                    ConvertTo-FormattedNumber -Value $object.$inBytesColumnName -Type DataSize -NumberFormatString $dataColumn.ExtendedProperties['NumberFormat']
                                }
                                else {
                                    ConvertTo-FormattedNumber -Value $object.$inBytesColumnName -Type DataSize -NumberFormatString '0.00'
                                }
                            }
                        }
                    }
                    else {
                        if ($UseCaption.IsPresent) {
                            $object.$( $dataColumn.Caption )
                        }
                        else {
                            $object.$( $dataColumn.ColumnName )
                        }
                    }
                )

                $dataRow.Item($c) = $(
                    if ([string]::IsNullOrWhiteSpace($propertyValue)) {
                        if ($dataColumn.DataType.Name -eq 'String') {
                            [string]::Empty
                        }
                        else {
                            [System.DBNull]::Value
                        }
                    }
                    else {
                        if ($dataColumn.ExtendedProperties['NumberFormat'] -like 'D*' -and ([decimal]::TryParse($propertyValue, [ref]$null) -and $propertyValue -gt 0)) {
                            [System.Math]::Round($propertyValue, ($dataColumn.ExtendedProperties['NumberFormat'] -replace 'D'))
                        }
                        else {
                            $propertyValue
                        }
                    }
                )
            }

            $dataRow
        }
    }
}

function Add-RowToDataTable {
    [CmdletBinding()]
    param (
        [Parameter(Mandatory)]
        [System.Data.DataRow]$DataRow
    )

    try {
        $DataRow.Table.Rows.Add($DataRow)
    }
    catch {
        $msg = New-Object -TypeName System.Text.StringBuilder

        if ($_.FullyQualifiedErrorId -eq 'NoNullAllowedException') {
            [void]$msg.AppendLine("$( (Get-PSCallStack)[1].Command ) : A null value was not expected (table: $( $DataRow.Table.TableName )).")
            [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
            [void]$msg.AppendLine('The data will be present in the table and this warning can be ignored (unless this output will be')
            [void]$msg.AppendLine('used for a differences report in the future). Please reference the FAQ entry for more information.')

            $DataRow.Table.Columns | Where-Object { -not $_.AllowDbNull } | ForEach-Object { $_.AllowDbNull = $true }
            $DataRow.Table.PrimaryKey = $null

            $DataRow.Table.Rows.Add($DataRow)
        }
        elseif ($_.FullyQualifiedErrorId -eq 'ConstraintException') {
            [void]$msg.AppendLine("$( (Get-PSCallStack)[1].Command ) : A duplicate row was not expected (table: $( $DataRow.Table.TableName )).")
            [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
            [void]$msg.AppendLine('The data will be present in the table and this warning can be ignored (unless this output will be')
            [void]$msg.AppendLine('used for a differences report in the future). Please reference the FAQ entry for more information.')

            $DataRow.Table.PrimaryKey = $null

            $DataRow.Table.Rows.Add($DataRow)
        }
        else {
            [void]$msg.AppendLine("$( (Get-PSCallStack)[1].Command ) : An unexpected error occurred adding rows to the table: $( $DataRow.Table.TableName ).")
        }

        if (-not $DataRow.Table.ExtendedProperties.ContainsKey('DiffError')) {
            $DataRow.Table.ExtendedProperties.Add('DiffError', $_.Exception.Message)
        }

        [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
        [void]$msg.AppendLine('As this warning is not expected, please consider contacting the support staff via e-mail or by')
        [void]$msg.AppendLine('starting a thread on the community site.')
        [void]$msg.AppendLine('            E-mail: ng-NetAppDocs-support@netapp.com')
        [void]$msg.AppendLine('    Community site: https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs')
        [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
        [void]$msg.AppendLine($( $_.Exception.GetBaseException().Message ))
        [void]$msg.AppendLine("Location: $( $_.ScriptStackTrace )")

        Show-WarningMessage -Message $msg.ToString()
    }
}

function Set-TablePrimaryKey {
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseShouldProcessForStateChangingFunctions', '')]
    param (
        [Parameter(Mandatory, Position = 0)]
        [System.Data.DataTable]$DataTable,

        [Parameter(Mandatory, Position = 1)]
        [string[]]$PrimaryKey
    )

    try {
        $DataTable.PrimaryKey = $PrimaryKey | ForEach-Object {
            if (-not $DataTable.Columns.Contains($_)) {
                throw "Column not found in table: $_"
            }

            $DataTable.Columns[$_]
        }
    }
    catch {
        $Global:Error.RemoveAt(0)

        Write-Warning -Message "There was an error setting the primary key on table: $( $DataTable.TableName ). Reason: $( $_.Exception.Message )"

        $DataTable.PrimaryKey = $null
        $DataTable.ExtendedProperties.Add('DiffError', $_.Exception.Message)
    }
}

function Read-DataSet {
    [CmdletBinding()]
    [OutputType([System.Data.DataSet])]
    param (
        [Parameter(Mandatory, Position = 0)]
        [ValidateScript({
                if ($_ -isnot [System.Data.DataSet]) {
                    if ($_ -is [System.IO.FileInfo] -or $_ -is [System.String]) {
                        if ([System.IO.Path]::GetExtension($_) -notmatch '.x(ml|sd)' -and [System.IO.Path]::GetExtension($_) -ne '.gz') {
                            throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]'Invalid file extension specified for the InputObject parameter.', 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $_))
                        }
                    }
                    else {
                        throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]'Invalid input object specified.', 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $_))
                    }
                }

                $true
            })]
        [Alias('InputFile', 'InputData')]
        [System.Object]$InputObject
    )

    if ($InputObject -is [System.Data.DataSet]) {
        Write-Verbose -Message 'Reading DataSet'

        $dataSet = $InputObject
    }
    else {
        try {
            $inputFileInfo = $(
                if ($InputObject -is [System.IO.FileInfo]) {
                    [System.IO.FileInfo]$ExecutionContext.SessionState.Path.GetResolvedPSPathFromPSPath($InputObject.FullName).ProviderPath
                }
                else {
                    [System.IO.FileInfo]$ExecutionContext.SessionState.Path.GetResolvedPSPathFromPSPath($InputObject).ProviderPath
                }
            )
        }
        catch {
            $Global:Error.RemoveAt(0)

            $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('Input file specified does not exist.', 'FileNotFound', [System.Management.Automation.ErrorCategory]::ObjectNotFound, $InputObject)))
        }

        if ($inputFileInfo.Extension -eq '.gz') {
            Write-Verbose -Message "Expanding FileName: $( $inputFileInfo.FullName )"

            $inputFile = Expand-GzipFile -InFile $inputFileInfo.FullName -PassThru
        }
        else {
            $inputFile = $inputFileInfo
        }

        $dataSet = New-Object -TypeName System.Data.DataSet

        try {
            if ($inputFile.Extension -eq '.xsd') {
                Write-Verbose -Message 'Reading DataSet Schema file'

                $dataSet.ReadXmlSchema($inputFile)
            }
            else {
                Write-Verbose -Message 'Reading DataSet file'

                $xsdFile = [System.IO.Path]::ChangeExtension($inputFile.FullName, '.xsd')

                if (Test-Path -Path $xsdFile -PathType Leaf) {
                    $dataSet.ReadXmlSchema($xsdFile)
                }

                [void]$dataSet.ReadXml($inputFile.FullName)
            }
        }
        catch [System.Management.Automation.MethodInvocationException] {
            $Global:Error.RemoveAt(0)

            $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('There was an error reading the DataSet file. Reason: The input data has not been formatted with the appropriate Format-Ntap*Data function.', 'InvalidOperation', [System.Management.Automation.ErrorCategory]::InvalidOperation, $inputXmlFile)))
        }
        catch {
            $Global:Error.RemoveAt(0)

            $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("There was an error reading the DataSet file. Reason: $( $_.Exception.Message )", 'InvalidData', [System.Management.Automation.ErrorCategory]::InvalidData, $inputXmlFile)))
        }
        finally {
            if ($inputFileInfo.Extension -eq '.gz') {
                Remove-Item -Path $inputXmlFile.FullName -ErrorAction SilentlyContinue
            }
        }
    }

    if ($dataSet.Tables.Count -eq 0) {
        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('The DataSet is empty.', 'InvalidData', [System.Management.Automation.ErrorCategory]::InvalidData, $dataSet)))
    }

    $dataSet
}

function Get-NtapWordTemplate {
    param (
        [Parameter(Mandatory, Position = 0)]
        [string]$DataSetName,

        [Parameter(Mandatory, Position = 1)]
        [string]$BaseTypeName,

        [string]$Locale
    )

    switch -Exact ($DataSetName) {
        'NetAppDocs.ONTAP.BuildDoc' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.NcController.Asup' {
                    $wordTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (ONTAP - Cluster View - ASUP).dotx")
                }
                'NetAppDocs.NcController' {
                    $wordTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (ONTAP - Cluster View).dotx")
                }

                #These left for legacy pre-4.0 lookups
                'NetAppDocs.NcVserver.Asup' {
                    $wordTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (ONTAP - SVM View - ASUP).dotx")
                }
                'NetAppDocs.NcVserver' {
                    $wordTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (ONTAP - SVM View).dotx")
                }
                'NetAppDocs.NaController.Asup' {
                    $wordTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (7-Mode - ASUP).dotx")
                }
                'NetAppDocs.NaController' {
                    $wordTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (7-Mode).dotx")
                }
            }
        }
        'NetAppDocs.Vserver.BuildDoc' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.NcVserver.Asup' {
                    $wordTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (ONTAP - SVM View - ASUP).dotx")
                }
                'NetAppDocs.NcVserver' {
                    $wordTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (ONTAP - SVM View).dotx")
                }
            }
        }
        'NetAppDocs.FSxOntap.BuildDoc' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.NcVserver.FSxOntap' {
                    $wordTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (AWS FSx for ONTAP).dotx")
                }
            }
        }
        'NetAppDocs.7Mode.BuildDoc' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.NaController.Asup' {
                    $wordTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (7-Mode - ASUP).dotx")
                }
                'NetAppDocs.NaController' {
                    $wordTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (7-Mode).dotx")
                }
            }
        }
        'NetAppDocs.SANtricity.BuildDoc' {
            switch -Exact ($BaseTypeName) {
                #'NetAppDocs.NeStorageArray.Asup' {
                #    $wordTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (SANtricity - ASUP).dotx")
                #}
                'NetAppDocs.NeStorageArray' {
                    $wordTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (SANtricity).dotx")
                }
                default {
                    $wordTemplate = $null
                }
            }
        }
        'NetAppDocs.SFCluster.BuildDoc' {
            switch -Exact ($BaseTypeName) {
                #'NetAppDocs.SFCluster.Asup' {
                #    $wordTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (SolidFire - ASUP).dotx")
                #}
                #'NetAppDocs.SFCluster' {
                #    $wordTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (SolidFire).dotx")
                #}
                default {
                    $wordTemplate = $null
                }
            }
        }
        'NetAppDocs.OciServer.BuildDoc' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.OciServer' {
                    $wordTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (OCI).dotx")
                }
                default {
                    $wordTemplate = $null
                }
            }
        }
        default {
            $wordTemplate = $null
        }
    }

    $wordTemplate
}

function Get-NtapExcelTemplate {
    param (
        [Parameter(Mandatory, Position = 0)]
        [string]$DataSetName,

        [Parameter(Mandatory, Position = 1)]
        [string]$BaseTypeName,

        [string]$Locale = 'en-US'
    )

    switch -Exact ($DataSetName) {
        'NetAppDocs.ONTAP.BuildDoc' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.NcController.Asup' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (ONTAP - Cluster View - ASUP).xltx")
                }
                'NetAppDocs.NcController' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (ONTAP - Cluster View).xltx")
                }

                #These left for legacy pre-4.0 lookups
                'NetAppDocs.NcVserver.Asup' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (ONTAP - SVM View - ASUP).xltx")
                }
                'NetAppDocs.NcVserver' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (ONTAP - SVM View).xltx")
                }
                'NetAppDocs.NaController.Asup' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (7-Mode - ASUP).xltx")
                }
                'NetAppDocs.NaController' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (7-Mode).xltx")
                }
            }
        }
        'NetAppDocs.ONTAP.HealthAssessment' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.NcController.Aiq' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Health Assessment Template (ONTAP - AIQ).xltx")
                }
            }
        }
        'NetAppDocs.Vserver.BuildDoc' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.NcVserver.Asup' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (ONTAP - SVM View - ASUP).xltx")
                }
                'NetAppDocs.NcVserver' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (ONTAP - SVM View).xltx")
                }
            }
        }
        'NetAppDocs.FSxOntap.BuildDoc' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.NcVserver.FSxOntap' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (AWS FSx for ONTAP).xltx")
                }
            }
        }
        'NetAppDocs.7Mode.BuildDoc' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.NaController.Asup' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (7-Mode - ASUP).xltx")
                }
                'NetAppDocs.NaController' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (7-Mode).xltx")
                }
            }
        }
        'NetAppDocs.ONTAP.HealthCheck' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.NcController.Asup' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Health Check Summary Template (ONTAP - Cluster View - ASUP).xltx")
                }
                'NetAppDocs.NcController' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Health Check Summary Template (ONTAP - Cluster View).xltx")
                }

                #These left for legacy pre-4.0 lookups
                'NetAppDocs.NaController.Asup' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Health Check Summary Template (7-Mode - ASUP).xltx")
                }
                'NetAppDocs.NaController' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Health Check Summary Template (7-Mode).xltx")
                }
                'NetAppDocs.NcVserver.Asup' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Health Check Summary Template (ONTAP - SVM View - ASUP).xltx")
                }
                'NetAppDocs.NcVserver' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Health Check Summary Template (ONTAP - SVM View).xltx")
                }
            }
        }
        'NetAppDocs.ONTAP.PerfAssessment' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.CloudInsights' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/AQoS Performance Analysis Template (Cloud Insights).xltx")
                }
                default {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/NetAppDocs Excel Template.xltx")
                }
            }
        }
        'NetAppDocs.Vserver.HealthCheck' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.NcVserver.Asup' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Health Check Summary Template (ONTAP - SVM View - ASUP).xltx")
                }
                'NetAppDocs.NcVserver' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Health Check Summary Template (ONTAP - SVM View).xltx")
                }
            }
        }
        'NetAppDocs.7Mode.HealthCheck' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.NaController.Asup' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Health Check Summary Template (7-Mode - ASUP).xltx")
                }
                'NetAppDocs.NaController' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Health Check Summary Template (7-Mode).xltx")
                }
            }
        }
        'NetAppDocs.7Mode.Transition' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.NaController.Asup' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Transition Summary Template (7-Mode - ASUP).xltx")
                }
                'NetAppDocs.NaController' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Transition Summary Template (7-Mode).xltx")
                }
                default { }
            }
        }
        #These left for legacy pre-4.0 lookups
        'NetAppDocs.ONTAP.Transition' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.NaController.Asup' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Transition Summary Template (7-Mode - ASUP).xltx")
                }
                'NetAppDocs.NaController' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Transition Summary Template (7-Mode).xltx")
                }
                default { }
            }
        }
        'NetAppDocs.SANtricity.BuildDoc' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.NeStorageArray' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (SANtricity).xltx")
                }
                default {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/NetAppDocs Excel Template.xltx")
                }
            }
        }
        'NetAppDocs.SANtricity.HealthCheck' {
            switch -Exact ($BaseTypeName) {
                #'NetAppDocs.NeStorageArray.Asup' {
                #    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Health Check Summary Template (SANtricity - ASUP).xltx")
                #}
                #'NetAppDocs.NeStorageArray' {
                #    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Health Check Summary Template (SANtricity).xltx")
                #}
                default {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/NetAppDocs Excel Template.xltx")
                }
            }
        }
        'NetAppDocs.SFCluster.BuildDoc' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.SFCluster' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (SolidFire).xltx")
                }
                #default {
                #    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/NetAppDocs Excel Template.xltx")
                #}
            }
        }
        'NetAppDocs.SFCluster.HealthCheck' {
            switch -Exact ($BaseTypeName) {
                #'NetAppDocs.SFCluster' {
                #    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Health Check Summary Template (SolidFire).xltx")
                #}
                default {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/NetAppDocs Excel Template.xltx")
                }
            }
        }
        'NetAppDocs.OciServer.BuildDoc' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.OciServer' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (OCI).xltx")
                }
                default {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/NetAppDocs Excel Template.xltx")
                }
            }
        }
        'NetAppDocs.OciServer.HealthCheck' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.OciServer' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Health Check Summary Template (OCI).xltx")
                }
                default {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/NetAppDocs Excel Template.xltx")
                }
            }
        }
        'NetAppDocs.SnapDrive.BuildDoc' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.SnapDrive' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (SnapDrive).xltx")
                }
                default {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/NetAppDocs Excel Template.xltx")
                }
            }
        }
        'NetAppDocs.VMware.BuildDoc' {
            switch -Exact ($BaseTypeName) {
                'NetAppDocs.VMware' {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Site Design Document Template (VMware).xltx")
                }
                default {
                    $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/NetAppDocs Excel Template.xltx")
                }
            }
        }
        'NetAppDocs.Sanitize.Mappings' {
            $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/Sanitize Mappings Template.xltx")
        }
        default {
            $excelTemplate = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Templates/$Locale/NetAppDocs Excel Template.xltx")
        }
    }

    $excelTemplate
}

function Get-NtapDocsSchemaFile {
    param (
        [Parameter(Mandatory, Position = 0)]
        [string]$BaseTypeName,

        [Parameter(Mandatory, Position = 1)]
        [string]$DocumentType
    )

    Get-ChildItem -Path $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$MyInvocation.MyCommand.Module.Path).DirectoryName )/Resources/Schema/$BaseTypeName.Document#$DocumentType.xsd") -ErrorAction Ignore
}

function Show-IncorrectTypeErrorMessage {
    param (
        [Parameter(Mandatory)]
        [string]$BaseTypeName,

        [Parameter(Mandatory)]
        [string]$SystemName
    )

    switch -Wildcard ($BaseTypeName) {
        'NetAppDocs.NcController*' {
            Show-ErrorMessage -Message "Use Format-NtapClusterData for ONTAP data. Input data will not be processed for system: $SystemName"
        }
        'NetAppDocs.NaController*' {
            Show-ErrorMessage -Message "Use Format-NtapFilerData for 7-Mode ONTAP data. Input data will not be processed for system: $SystemName"
        }
        'NetAppDocs.NcVserver*' {
            Show-ErrorMessage -Message "Use Format-NtapVserverData for ONTAP SVM data. Input data will not be processed for system: $SystemName"
        }
        'NetAppDocs.NeStorageArray*' {
            Show-ErrorMessage -Message "Use Format-NtapSantricityData for SANtricity data. Input data will not be processed for system: $SystemName"
        }
        'NetAppDocs.SFCluster*' {
            Show-ErrorMessage -Message "Use Format-NtapSolidFireData for SolidFire data. Input data will not be processed for system: $SystemName"
        }
        'NetAppDocs.CloudInsights*' {
            Show-ErrorMessage -Message "Use Format-NtapCloudInsightsData for CloudInsights data. Input data will not be processed for system: $SystemName"
        }
        'NetAppDocs.OciServer*' {
            Show-ErrorMessage -Message "Use Format-NtapOciServerData for OCI data. Input data will not be processed for system: $SystemName"
        }
        'NetAppDocs.SnapDrive*' {
            Show-ErrorMessage -Message "Use Format-NtapSnapDriveData for SnapDrive data. Input data will not be processed for system: $SystemName"
        }
        'NetAppDocs.Ucs*' {
            Show-ErrorMessage -Message "Use Format-NtapCiscoUcsData for Cisco UCS data. Input data will not be processed for system: $SystemName"
        }
        'NetAppDocs.VMware*' {
            Show-ErrorMessage -Message "Use Format-NtapVMwareData for VMware data. Input data will not be processed for system: $SystemName"
        }
    }
}

function Get-StringHash {
    [CmdletBinding()]
    param (
        # String to generate the hash from.
        [Parameter(Mandatory, ValueFromPipeline)]
        [ValidateNotNullOrEmpty()]
        [string]$InputString,

        # Type of hash algorithm to use. Default is SHA1.
        [Parameter()]
        [ValidateSet('MD5', 'SHA', 'SHA1', 'SHA256', 'SHA384', 'SHA512')]
        [string]$Algorithm = 'SHA1'
    )

    begin {
        $hashAlgorithm = [System.Security.Cryptography.HashAlgorithm]::Create($Algorithm)
    }
    process {
        $sb = New-Object -TypeName System.Text.StringBuilder

        $hashAlgorithm.ComputeHash([System.Text.Encoding]::UTF8.GetBytes($InputString)) | ForEach-Object {
            [void]$sb.Append($_.ToString('x2'))
        }

        $sb.ToString()
    }
}

function New-CombinedGuid {
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseShouldProcessForStateChangingFunctions', '')]
    param (
        [Parameter(Mandatory, Position = 0)]
        [AllowEmptyString()]
        [string]$Guid1,

        [Parameter(Mandatory, Position = 1)]
        [AllowEmptyString()]
        [string]$Guid2,

        [Parameter()]
        [string]$Guid3
    )

    $byteCount = 16

    $destByte = New-Object -TypeName System.Byte[] -ArgumentList $byteCount

    $guid1Byte = $(
        if (-not [string]::IsNullOrWhiteSpace($Guid1)) {
            ([guid]($Guid1)).ToByteArray()
        }
        else {
            [guid]::Empty.ToByteArray()
        }
    )

    $guid2Byte = $(
        if (-not [string]::IsNullOrWhiteSpace($Guid2)) {
            ([guid]($Guid2)).ToByteArray()
        }
        else {
            [guid]::Empty.ToByteArray()
        }
    )

    if (-not [string]::IsNullOrWhiteSpace($Guid3)) {
        $guid3Byte = $(
            if (-not [string]::IsNullOrWhiteSpace($Guid3)) {
                ([guid]($Guid3)).ToByteArray()
            }
            else {
                [guid]::Empty.ToByteArray()
            }
        )
    }

    for ($i = 0; $i -lt $byteCount; $i++) {
        if (-not [string]::IsNullOrWhiteSpace($Guid3)) {
            $destByte[$i] = [byte]([byte]($guid1Byte[$i] -bor $guid2Byte[$i]) -bor $guid3Byte[$i])
        }
        else {
            $destByte[$i] = [byte]($guid1Byte[$i] -bor $guid2Byte[$i])
        }
    }

    [guid]$destByte
}

function ConvertTo-FormattedNumber {
    [CmdletBinding()]
    [OutputType([System.String])]
    param (
        [Parameter(Mandatory, ValueFromPipeline)]
        [AllowNull()]
        $Value,

        [Parameter()]
        [ValidateSet('DataSize', 'Percent', 'RPM', 'Count')]
        [string]$Type = 'DataSize',

        [Parameter()]
        #[ValidateScript({ })]
        [AllowNull()]
        [string]$NumberFormatString = '0',

        [Parameter()]
        [switch]$UseBase2Suffix
    )

    process {
        $inputValue = [decimal]::Zero

        if ([decimal]::TryParse($Value, [ref]$inputValue)) {
            switch -Exact ($Type) {
                'DataSize' {
                    $base10Suffix = @( 'B', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB' )
                    $base2Suffix = @( 'B', 'KiB', 'MiB', 'GiB', 'TiB', 'PiB', 'EiB', 'ZiB', 'YiB' )

                    if ($inputValue -eq 0) {
                        [string]::Format("{0:$NumberFormatString}", 0)
                    }
                    elseif ($inputValue -lt 1024) {
                        [string]::Format("{0:$NumberFormatString} B", $inputValue)
                    }
                    else {
                        $digitGroup = [System.Math]::Floor(([System.Math]::Log($inputValue, 2) / [System.Math]::Log(1024, 2)))
                        $suffix = $( if ($UseBase2Suffix.IsPresent) { $base2Suffix[$digitGroup] } else { $base10Suffix[$digitGroup] } )

                        [string]::Format("{0:$NumberFormatString} {1}", ($inputValue / [System.Math]::Pow(1024, $digitGroup)), $suffix)
                    }
                }
                'Percent' {
                    [string]::Format("{0:$NumberFormatString}%", $inputValue)
                }
                'RPM' {
                    if ($inputValue -gt 0) {
                        [string]::Format('{0}k', [System.Math]::Round(($inputValue / 1000), 1))
                    }
                }
                'Count' {
                    if ($inputValue -lt 1000) {
                        [string]::Format("{0:$NumberFormatString}", $inputValue)
                    }
                    else {
                        $digitGroup = [System.Math]::Floor(([System.Math]::Log($inputValue, 10) / [System.Math]::Log(1000, 10)))

                        $suffix = @( 'H', 'k', 'M', 'G', 'T', 'P', 'E', 'Z', 'Y' )[$digitGroup]
                        [string]::Format("{0:$NumberFormatString}{1}", ($inputValue / [System.Math]::Pow(1000, $digitGroup)), $suffix)
                    }
                }
            }
        }
    }
}

function ConvertFrom-FormattedNumber {
    [CmdletBinding()]
    [OutputType([System.Decimal])]
    param (
        [Parameter(Mandatory, ValueFromPipeline)]
        [AllowNull()]
        [AllowEmptyString()]
        [string]$Value,

        [Parameter()]
        [ValidateSet('DataSize', 'Percent', 'RPM', 'Count')]
        [string]$Type = 'DataSize',

        [Parameter()]
        [string]$NumberFormatString = '0'
    )

    process {
        $outputValue = $null

        if (-not [string]::IsNullOrWhiteSpace($Value)) {
            $outputValue = [decimal]::Zero

            $numValue = $(
                if ($Value -eq 0) {
                    [string]::Format("{0:$NumberFormatString}", 0)
                }
                else {
                    switch -Exact ($Type) {
                        'DataSize' {
                            $powerOf = @{
                                'B'   = 0
                                'KiB' = 1
                                'KB'  = 1
                                'MiB' = 2
                                'MB'  = 2
                                'GiB' = 3
                                'GB'  = 3
                                'TiB' = 4
                                'TB'  = 4
                                'PiB' = 5
                                'PB'  = 5
                                'EiB' = 6
                                'EB'  = 6
                                'ZiB' = 7
                                'ZB'  = 7
                                'YiB' = 8
                                'YB'  = 8
                            }

                            if ($Value -match "(?<Size>.+?)\s*(?<Unit>$( $powerOf.Keys -join '|' ))") {
                                [string]::Format("{0:$NumberFormatString}", ([decimal]$Matches.Size * [System.Math]::Pow(1024, $powerOf[$Matches.Unit])))

                                $Matches.Clear()
                            }
                        }
                        'Percent' {
                            [string]::Format("{0:$NumberFormatString}", [double]($Value -replace '(.+?)\s*[%]', '$1'))
                        }
                        'RPM' {
                            [double]($Value -replace '\s*k') * 1000
                        }
                        'Count' {
                            $powerOf = @{
                                'H' = 0
                                'k' = 1
                                'M' = 2
                                'G' = 3
                                'T' = 4
                                'P' = 5
                                'E' = 6
                                'Z' = 7
                                'Y' = 8
                            }

                            if ([decimal]::TryParse($Value, [ref]$null)) {
                                $Value
                            }
                            elseif ($Value -match '(?<Size>.+?)\s*(?<Unit>H|k|M|G|T|P|E|Z|Y)') {
                                [string]::Format("{0:$NumberFormatString}", [long]([double]$Matches.Size * [System.Math]::Pow(1000, $powerOf[$Matches.Unit])))

                                $Matches.Clear()
                            }
                        }
                    }
                }
            )

            if ([decimal]::TryParse($numValue, [ref]$outputValue)) {
                $outputValue
            }
        }
    }
}

function Get-RawDiskCapacity {
    [CmdletBinding()]
    [OutputType([System.Int64])]
    param (
        [Parameter(Mandatory, ValueFromPipeline)]
        [System.Object]$Disk
    )

    process {
        if ($Script:HardwareListings -and (($hwListing = (Select-Xml -Xml $Script:HardwareListings.Hardware -XPath "(./ONTAP/Disks/Disk[Model=`"$( $Disk.Model )`"])[1]").Node))) {
            if (-not [string]::IsNullOrWhiteSpace($hwListing.RightSizedCapacity)) {
                [long]$hwListing.RightSizedCapacity
            }
        }
        elseif (-not [string]::IsNullOrWhiteSpace($Disk.StorageDiskInfo.DiskRaidInfo.PhysicalBlocks)) {
            [long]$Disk.StorageDiskInfo.DiskRaidInfo.PhysicalBlocks * 4096
        }
        elseif (-not [string]::IsNullOrWhiteSpace($Disk.StorageDiskInfo.DiskInventoryInfo.RightSizeSectors)) {
            [long]$Disk.StorageDiskInfo.DiskInventoryInfo.RightSizeSectors * [int]$Disk.StorageDiskInfo.DiskInventoryInfo.BytesPerSector
        }
        elseif (-not [string]::IsNullOrWhiteSpace($Disk.DiskRaidInfo.PhysicalBlocks)) {
            [long]$Disk.DiskRaidInfo.PhysicalBlocks * 4096
        }
        elseif (-not [string]::IsNullOrWhiteSpace($Disk.DiskInventoryInfo.RightSizeSectors)) {
            [long]$Disk.DiskInventoryInfo.RightSizeSectors * [int]$Disk.DiskInventoryInfo.BytesPerSector
        }
        elseif (-not [string]::IsNullOrWhiteSpace($Disk.PhySizeBlocks)) {
            ([long]$Disk.PhySizeBlocks / 8) * 4096
        }
        elseif (-not [string]::IsNullOrWhiteSpace($Disk.PhysicalBlocks)) {
            ([long]$Disk.PhysicalBlocks / 8) * 4096
        }
        elseif (-not [string]::IsNullOrWhiteSpace($Disk.Capacity)) {
            [long]$Disk.Capacity
        }
        elseif (-not [string]::IsNullOrWhiteSpace($Disk.PhysicalSize)) {
            [long]$Disk.PhysicalSize
        }
        elseif (-not [string]::IsNullOrWhiteSpace($Disk.UsableMb)) {
            [long]$Disk.UsableMb * [System.Math]::Pow(1024, 2)
        }
    }
}

function Get-MinimumRecommendedOntapVersion {
    param ($RecommendedVersions, $OntapVersion, $AvailableOntapVersion)

    $minimumRecommendedOntapVersion = 'no recommendation found'

    if (-not [string]::IsNullOrWhiteSpace($RecommendedVersions)) {
        $currentSortableVersion = ConvertTo-SortableVersionString -VersionString $OntapVersion

        foreach ($recommendedVersion in @( $RecommendedVersions -split ',' ).Where({ $_ })) {
            if ((ConvertTo-SortableVersionString -VersionString $recommendedVersion) -ge $currentSortableVersion) {
                $minimumRecommendedOntapVersion = $recommendedVersion

                break
            }
        }
    }

    <#if ($minimumRecommendedOntapVersion -eq 'no recommendation found') {
        if ($AvailableOntapVersion -ne 'UTD') {
            if ((ConvertTo-SortableVersionString -VersionString $OntapVersion) -ge (ConvertTo-SortableVersionString -Version $AvailableOntapVersion)) {
                $minimumRecommendedOntapVersion = $OntapVersion
            }
            else {
                $minimumRecommendedOntapVersion = $AvailableOntapVersion
            }
        }
    }#>

    $minimumRecommendedOntapVersion
}

# SIG # Begin signature block
# MIIq0AYJKoZIhvcNAQcCoIIqwTCCKr0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDn5Xk6LsFhe4if
# qWT4u5/YMgLsAGgpwvges714C8CRDaCCJGgwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggWNMIIEdaADAgECAhAOmxiO
# +dAt5+/bUOIIQBhaMA0GCSqGSIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYD
# VQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAi
# BgNVBAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAw
# MDBaFw0zMTExMDkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
# Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERp
# Z2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBAL/mkHNo3rvkXUo8MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsb
# hA3EMB/zG6Q4FutWxpdtHauyefLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iT
# cMKyunWZanMylNEQRBAu34LzB4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGb
# NOsFxl7sWxq868nPzaw0QF+xembud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclP
# XuU15zHL2pNe3I6PgNq2kZhAkHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCr
# VYJBMtfbBHMqbpEBfCFM1LyuGwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFP
# ObURWBf3JFxGj2T3wWmIdph2PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTv
# kpI6nj3cAORFJYm2mkQZK37AlLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWM
# cCxBYKqxYxhElRp2Yn72gLD76GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls
# 5Q5SUUd0viastkF13nqsX40/ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBR
# a2+xq4aLT8LWRV+dIPyhHsXAj6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6
# MIIBNjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qY
# rhwPTzAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8E
# BAMCAYYweQYIKwYBBQUHAQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5k
# aWdpY2VydC5jb20wQwYIKwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDig
# NoY0aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNybDARBgNVHSAECjAIMAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCg
# v0NcVec4X6CjdBs9thbX979XB72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQT
# SnovLbc47/T/gLn4offyct4kvFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh
# 65ZyoUi0mcudT6cGAxN3J0TU53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSw
# uKFWjuyk1T3osdz9HNj0d1pcVIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAO
# QGPFmCLBsln1VWvPJ6tsds5vIy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjD
# TZ9ztwGpn1eqXijiuZQwggXMMIIENKADAgECAhAg429sPxgagUb53pPffJfkMA0G
# CSqGSIb3DQEBDAUAMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBS
# MzYwHhcNMjEwOTA5MDAwMDAwWhcNMjMwOTA5MjM1OTU5WjBQMQswCQYDVQQGEwJV
# UzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UECgwMTmV0QXBwLCBJbmMuMRUw
# EwYDVQQDDAxOZXRBcHAsIEluYy4wggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGK
# AoIBgQC4kYYj/oViZD9pN03hrqFjtwOz1Gx4eDWVV8IYUYEr2qpLeNYvWz26B/+E
# mYLboAGVpkYg0Wske3hjRpooZlMORm1S4+2C2NoQCstZ+kmNTevbXHJe/w1VNJrm
# fKjpXOJEfx8GUGDqCMR30dChVpsdBEavrM7T0hnfJLv18i19SGv3a+nhvB3iOjLP
# SEg8+DFkjD5jTI6fQ83DBEiVR4UEYq7yyH578ZEylhsRfJmK+uIREycEUk/NpFTw
# g6/7lxh/WYabCrZzqI4Ep3QataRKaAlxt3BiOlZkX4WGX3DYvSYltWOM2UlCcGpP
# m/1/LN8Xuvf+YO6H472QecHe59XtXbHSMKvqDuOEJu8Wk2LFMNK732o3Fc5QIHie
# 6JtNsgNYf+Vsm5EMBD1ZHNE+C3zlmQbbWIU5uEU1nhiodBPKsu0SkS6pARyKBO05
# DSc2gWWv0aukKi04ZAn+hAdSl5s1dgmU5shBvKHqe15K9ZRN4FFO/CgHE0BoGYQS
# UQVKwa0CAwEAAaOCAZwwggGYMB8GA1UdIwQYMBaAFA8qyyCHKLjsb0iuK1SmKaoX
# pM0MMB0GA1UdDgQWBBQuH643KcBMmb/Q6IZt+H9IrnXFwDAOBgNVHQ8BAf8EBAMC
# B4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDAzARBglghkgBhvhC
# AQEEBAMCBBAwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwIwJTAjBggrBgEFBQcC
# ARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQBMEkGA1UdHwRCMEAw
# PqA8oDqGOGh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVT
# aWduaW5nQ0FSMzYuY3JsMHkGCCsGAQUFBwEBBG0wazBEBggrBgEFBQcwAoY4aHR0
# cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQVIz
# Ni5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqG
# SIb3DQEBDAUAA4IBgQCOoGdXjP8Sif0h3ZvoDsIVfrJvQUdP9pZExRJGGj/Te6ML
# XyqHNc/G2WklNv+BC4ryNJ4zp0oneHOub55ejrYPfhgP9oFWS8oOPJbrp3rAtINa
# OeKRC88WUYZSKYAxSX6kubEjnQD6cmazXjxVN6P8+6q9smeqF3xI+++c+ekMw3Xv
# 4EWRYXqN0/srfFp1WpoluwSQcZMqj9haSX0bgO6AAWH2AnVJMfXTIKz/0FQ/RW0y
# Ed5QYQqYInhA7IUz9dBziNpJlwONeF5j7xzgfYDY63WU6WrgJnGjYkQCOOsUx74j
# gUiMRe9zV48GS8Vxz22c/TQs0x4/1GmOSJvUHEk3GseBmB3v+yEhh/D6zWOfYP4X
# D/9b91CxmugTuepuaJSeRg+qUm3KarKsOCUF/CLqUrNh/JwKrWD1cghRaYLvMucs
# ScksHHe7ZDvb2OtvxWXjPk1d1NKvEwFJSS6hIVIbug9x28AJqOoP5Pn9wZvJSzvW
# uQJGCLscdGyKefdCo30wggYaMIIEAqADAgECAhBiHW0MUgGeO5B5FSCJIRwKMA0G
# CSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290
# IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFQxCzAJBgNVBAYT
# AkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28g
# UHVibGljIENvZGUgU2lnbmluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEBAQUAA4IB
# jwAwggGKAoIBgQCbK51T+jU/jmAGQ2rAz/V/9shTUxjIztNsfvxYB5UXeWUzCxEe
# AEZGbEN4QMgCsJLZUKhWThj/yPqy0iSZhXkZ6Pg2A2NVDgFigOMYzB2OKhdqfWGV
# oYW3haT29PSTahYkwmMv0b/83nbeECbiMXhSOtbam+/36F09fy1tsB8je/RV0mIk
# 8XL/tfCK6cPuYHE215wzrK0h1SWHTxPbPuYkRdkP05ZwmRmTnAO5/arnY83jeNzh
# P06ShdnRqtZlV59+8yv+KIhE5ILMqgOZYAENHNX9SJDm+qxp4VqpB3MV/h53yl41
# aHU5pledi9lCBbH9JeIkNFICiVHNkRmq4TpxtwfvjsUedyz8rNyfQJy/aOs5b4s+
# ac7IH60B+Ja7TVM+EKv1WuTGwcLmoU3FpOFMbmPj8pz44MPZ1f9+YEQIQty/NQd/
# 2yGgW+ufflcZ/ZE9o1M7a5Jnqf2i2/uMSWymR8r2oQBMdlyh2n5HirY4jKnFH/9g
# Rvd+QOfdRrJZb1sCAwEAAaOCAWQwggFgMB8GA1UdIwQYMBaAFDLrkpr/NZZILyhA
# QnAgNpFcF4XmMB0GA1UdDgQWBBQPKssghyi47G9IritUpimqF6TNDDAOBgNVHQ8B
# Af8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcD
# AzAbBgNVHSAEFDASMAYGBFUdIAAwCAYGZ4EMAQQBMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Um9vdFI0Ni5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ1Jvb3RSNDYu
# cDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQwFAAOCAgEABv+C4XdjNm57oRUgmxP/BP6YdURhw1aVcdGRP4Wh60BAscjW
# 4HL9hcpkOTz5jUug2oeunbYAowbFC2AKK+cMcXIBD0ZdOaWTsyNyBBsMLHqafvIh
# rCymlaS98+QpoBCyKppP0OcxYEdU0hpsaqBBIZOtBajjcw5+w/KeFvPYfLF/ldYp
# mlG+vd0xqlqd099iChnyIMvY5HexjO2AmtsbpVn0OhNcWbWDRF/3sBp6fWXhz7Dc
# ML4iTAWS+MVXeNLj1lJziVKEoroGs9Mlizg0bUMbOalOhOfCipnx8CaLZeVme5yE
# Lg09Jlo8BMe80jO37PU8ejfkP9/uPak7VLwELKxAMcJszkyeiaerlphwoKx1uHRz
# NyE6bxuSKcutisqmKL5OTunAvtONEoteSiabkPVSZ2z76mKnzAfZxCl/3dq3dUNw
# 4rg3sTCggkHSRqTqlLMS7gjrhTqBmzu1L90Y1KWN/Y5JKdGvspbOrTfOXyXvmPL6
# E52z1NZJ6ctuMFBQZH3pwWvqURR8AgQdULUvrxjUYbHHj95Ejza63zdrEcxWLDX6
# xWls/GDnVNueKjWUH3fTv1Y8Wdho698YADR7TNx8X8z2Bev6SivBBOHY+uqiirZt
# g0y9ShQoPzmCcn63Syatatvx157YK9hlcPmVoa1oDE5/L9Uo2bC5a4CH2Rwwggau
# MIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJ
# BgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5k
# aWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAe
# Fw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcw
# FQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3Rl
# ZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9Ge
# TKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0
# hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZl
# jZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAsh
# aG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVY
# TXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1
# biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCir
# c0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+
# DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA
# +bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42Pg
# puE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzS
# M7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU
# uhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6
# mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsG
# AQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29t
# MEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNl
# cnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3Js
# My5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAE
# GTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1Z
# jsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8d
# B+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVp
# P0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp8
# 76i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2
# nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3
# ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQ
# txMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc
# 4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+Y
# AN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZ
# vAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQr
# H4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIGwDCCBKigAwIBAgIQDE1p
# ckuU+jwqSj0pB4A9WjANBgkqhkiG9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4XDTIyMDkyMTAwMDAw
# MFoXDTMzMTEyMTIzNTk1OVowRjELMAkGA1UEBhMCVVMxETAPBgNVBAoTCERpZ2lD
# ZXJ0MSQwIgYDVQQDExtEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMiAtIDIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDP7KUmOsap8mu7jcENmtuh6BSFdDMa
# JqzQHFUeHjZtvJJVDGH0nQl3PRWWCC9rZKT9BoMW15GSOBwxApb7crGXOlWvM+xh
# iummKNuQY1y9iVPgOi2Mh0KuJqTku3h4uXoW4VbGwLpkU7sqFudQSLuIaQyIxvG+
# 4C99O7HKU41Agx7ny3JJKB5MgB6FVueF7fJhvKo6B332q27lZt3iXPUv7Y3UTZWE
# aOOAy2p50dIQkUYp6z4m8rSMzUy5Zsi7qlA4DeWMlF0ZWr/1e0BubxaompyVR4aF
# eT4MXmaMGgokvpyq0py2909ueMQoP6McD1AGN7oI2TWmtR7aeFgdOej4TJEQln5N
# 4d3CraV++C0bH+wrRhijGfY59/XBT3EuiQMRoku7mL/6T+R7Nu8GRORV/zbq5Xwx
# 5/PCUsTmFntafqUlc9vAapkhLWPlWfVNL5AfJ7fSqxTlOGaHUQhr+1NDOdBk+lbP
# 4PQK5hRtZHi7mP2Uw3Mh8y/CLiDXgazT8QfU4b3ZXUtuMZQpi+ZBpGWUwFjl5S4p
# kKa3YWT62SBsGFFguqaBDwklU/G/O+mrBw5qBzliGcnWhX8T2Y15z2LF7OF7ucxn
# EweawXjtxojIsG4yeccLWYONxu71LHx7jstkifGxxLjnU15fVdJ9GSlZA076XepF
# cxyEftfO4tQ6dwIDAQABo4IBizCCAYcwDgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB
# /wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwIAYDVR0gBBkwFzAIBgZngQwB
# BAIwCwYJYIZIAYb9bAcBMB8GA1UdIwQYMBaAFLoW2W1NhS9zKXaaL3WMaiCPnshv
# MB0GA1UdDgQWBBRiit7QYfyPMRTtlwvNPSqUFN9SnDBaBgNVHR8EUzBRME+gTaBL
# hklodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0
# MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3JsMIGQBggrBgEFBQcBAQSBgzCBgDAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFgGCCsGAQUFBzAC
# hkxodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRS
# U0E0MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3J0MA0GCSqGSIb3DQEBCwUAA4IC
# AQBVqioa80bzeFc3MPx140/WhSPx/PmVOZsl5vdyipjDd9Rk/BX7NsJJUSx4iGNV
# CUY5APxp1MqbKfujP8DJAJsTHbCYidx48s18hc1Tna9i4mFmoxQqRYdKmEIrUPwb
# tZ4IMAn65C3XCYl5+QnmiM59G7hqopvBU2AJ6KO4ndetHxy47JhB8PYOgPvk/9+d
# EKfrALpfSo8aOlK06r8JSRU1NlmaD1TSsht/fl4JrXZUinRtytIFZyt26/+YsiaV
# OBmIRBTlClmia+ciPkQh0j8cwJvtfEiy2JIMkU88ZpSvXQJT657inuTTH4YBZJwA
# wuladHUNPeF5iL8cAZfJGSOA1zZaX5YWsWMMxkZAO85dNdRZPkOaGK7DycvD+5sT
# X2q1x+DzBcNZ3ydiK95ByVO5/zQQZ/YmMph7/lxClIGUgp2sCovGSxVK05iQRWAz
# gOAj3vgDpPZFR+XOuANCR+hBNnF3rf2i6Jd0Ti7aHh2MWsgemtXC8MYiqE+bvdgc
# mlHEL5r2X6cnl7qWLoVXwGDneFZ/au/ClZpLEQLIgpzJGgV8unG1TnqZbPTontRa
# mMifv427GFxD9dAq6OJi7ngE273R+1sKqHB+8JeEeOMIA11HLGOoJTiXAdI/Otrl
# 5fbmm9x+LMz/F0xNAKLY1gEOuIvu5uByVYksJxlh9ncBjDGCBb4wggW6AgEBMGgw
# VDELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDErMCkGA1UE
# AxMiU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIFIzNgIQIONvbD8YGoFG
# +d6T33yX5DANBglghkgBZQMEAgEFAKCBhDAYBgorBgEEAYI3AgEMMQowCKACgACh
# AoAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAM
# BgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCBMEMD/hR5HV3Np1MO7Lprp6DDF
# aHD3ekXeAsLmD12RcTANBgkqhkiG9w0BAQEFAASCAYBXAJ+YuBDsuXwYCgzDSYHL
# WGtAM+nkH/GKIwx3kOcaldVziGBH57s5LNxWvvqYJG0Ll5hsjbYQSdSCriV1W9vh
# bIKBBxA0zmt0ezXYtgoA40QeTsVpdnkfVPrQBLOKKTfwTcg5uu+/LF2MYucGz2bR
# kvl4EdS16z6grt7oslxYmb6+FdMG9lq1ZJoU+WPtrn7tbwW9+tNKUwA5YFPTBuE1
# 5tpUyn/twvsZ4sgornpssyj46tQb5axYTOb+Kjy/vEPnw9MkeGXoI22zr0fYEMIP
# c773znT0q2ug1PvShj01OqCZ7pewE21eI9zZnQWEkSkmbHPafFENXOUM4O0ho2qs
# IsAD5RYZoey1xxarln/cBMo3+6kv8aGysEaqTaob4VjrV/8yz7lNFkuT04HL73BC
# d159OzKOdyKWiKEGpI5+gPT8/AeQiek6MwS2zHffy6UmI86j3mWI70JMSVhvZ7l4
# gd8BU16r6o1qHtRj3nHWfSHDnnSlNpeBSve3CXPsksehggMgMIIDHAYJKoZIhvcN
# AQkGMYIDDTCCAwkCAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
# cnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
# SEEyNTYgVGltZVN0YW1waW5nIENBAhAMTWlyS5T6PCpKPSkHgD1aMA0GCWCGSAFl
# AwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUx
# DxcNMjMwNDAyMTUyODI1WjAvBgkqhkiG9w0BCQQxIgQg9h078kignzOI+NUHligh
# SOPr2RjmqEtqtTOu6Js1v18wDQYJKoZIhvcNAQEBBQAEggIAGyWYTqnfiai7oDiT
# JbrLqz9TtloqAfl7fSvr2Tc4mr993X/IAhdrJtP9AWDn5QQsUEU806ARWz2mSzZr
# p5TwxRhst7pWtNJCP+ZMhmZ5MDj5MSBw08ImrdF/hJE4jB9iDfSkUKdelhxnx1/+
# 3e8lYcfOEmaQ2nK/m5Wr1enROQJOTzP2NaUDUdqNWHW4D21smzelwo9jTGFj992h
# 2K/u5kgv5XuO17j42f9BzlUGo9yg1GqVG5Bbye/2jn8ZWyKVqKwFNH2qYtgUlgGw
# cjHpMDQnVBfGIULY9zLhJEKGnTa1UtdWmqVqNv7k21+z0w1IzxdSGSczxOGU3gpJ
# WLnseuFrHC6hrfMuUSYvan9X+Y3TfEO1NpjnwJZLYCPKi7Rj5xJROzro/XkAbwIn
# rIf8I9MG/xEv21zvVqsUabs270Trs3XA2N2nojnDmKmWgbmbwj3kH5ki+gTjvqMg
# na5krnL2FcoDoM9hIweFKtu3ndWCY620kFXVnUWBZSPDa5gpweFcR40uJt86B/rs
# n1HFBGKkC/6iHGpRoVcmQsL9tAg76X3HX0bjRxVKtzMNHzXwKPZUjozLdmflsnN8
# qca/7lqkxAtAcF5NI5n9M12UWrqFtOQRVEHSbOCigIg4RuvSd3yzlhAvn6SiPpvF
# dcPHI5iAmRzo/eIfoNT5I5Zm/8U=
# SIG # End signature block
