function Get-Match {
    [CmdletBinding()]
    param (
        [Parameter(ValueFromPipeline)]
        [System.Object]$InputObject,

        [Parameter(Mandatory)]
        [string]$Pattern
    )

    begin {
        try {
            $regex = New-Object -TypeName System.Text.RegularExpressions.Regex -ArgumentList $Pattern
        }
        catch {
            $Global:Error.RemoveAt(0)

            $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("Get-Match: Pattern not correct. '$Pattern' is not a valid regular expression.", 'InvalidRegEx', [System.Management.Automation.ErrorCategory]::InvalidArgument, $InputObject)))
        }

        $groups = @(
            $regex.GetGroupNames() | Where-Object { -not [int]::TryParse($_, [ref]$null) } | ForEach-Object {
                $_.ToString()
            }
        )
    }
    process {
        foreach ($line in $InputObject) {
            foreach ($match in $regex.Matches($line)) {
                if ($groups.Count -eq 0) {
                    ([System.Object[]]$match.Groups)[-1].Value
                }
                else {
                    $rv = 1 | Select-Object -Property $groups

                    $groups | ForEach-Object {
                        $rv.$_ = $match.Groups[$_].Value
                    }

                    $rv
                }
            }
        }
    }
}

function Convert-DateTimeString {
    param (
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$String,

        [Parameter()]
        [string[]]$Format
    )

    if ([string]::IsNullOrWhiteSpace($String) -or $String -eq '-' -or $String -eq 'none' -or $String -eq '0') {
        return [string]::Empty
    }

    $dateTime = [datetime]::Now

    if ([string]::IsNullOrWhiteSpace($Format)) {
        $convertible = [datetime]::TryParse(
            $String,
            [cultureinfo]::InvariantCulture,
            [System.Globalization.DateTimeStyles]::None,
            [ref]$dateTime
        )
    }
    else {
        $convertible = [datetime]::TryParseExact(
            $String,
            $Format,
            [cultureinfo]::InvariantCulture,
            [System.Globalization.DateTimeStyles]::AllowWhiteSpaces,
            [ref]$dateTime
        )
    }

    if (-not $convertible) {
        return [string]::Empty
    }

    $dateTime
}

function Convert-TimeSpanString {
    param (
        [Parameter(Mandatory)]
        [string]$String,

        [Parameter()]
        [string]$Format
    )

    $timeSpan = [timespan]::Zero

    if (-not [string]::IsNullOrWhiteSpace($Format)) {
        $style = [System.Globalization.TimeSpanStyles]::None
        $timeSpanFormat = $Format

        if ($String.StartsWith('-')) {
            $timeSpanFormat = $Format.Insert(0, "'-'")
            $style = [System.Globalization.TimeSpanStyles]::AssumeNegative
        }

        $convertible = [timespan]::TryParseExact(
            $String,
            $timeSpanFormat,
            [cultureinfo]::InvariantCulture,
            $style,
            [ref]$timeSpan
        )
    }
    elseif (($convertible = $string -match '^(?<Negative>[-])?(?<Hours>\d+)[:](?<Minutes>\d+)[:](?<Seconds>\d+)$')) {
        $timeSpan = New-TimeSpan -Hours "$( $Matches.Negative )$( $Matches.Hours)" -Minutes "$( $Matches.Negative )$( $Matches.Minutes )" -Seconds "$( $Matches.Negative )$( $Matches.Seconds )"

        $Matches.Clear()
    }
    else {
        $convertible = [timespan]::TryParse(
            $String,
            [cultureinfo]::InvariantCulture,
            [ref]$timeSpan
        )
    }

    if ($convertible) {
        $timeSpan
    }
}

function ConvertFrom-UnixTimestamp {
    [CmdletBinding(DefaultParameterSetName = 'Seconds')]
    [OutputType([System.DateTime])]
    param (
        [Parameter(Mandatory, ParameterSetName = 'Seconds')]
        [AllowEmptyString()]
        [string]$Seconds,

        [Parameter(Mandatory, ParameterSetName = 'Milliseconds')]
        [AllowEmptyString()]
        [string]$Milliseconds
    )

    switch -Exact ($PSCmdlet.ParameterSetName) {
        'Milliseconds' {
            if (-not [long]::TryParse($Milliseconds, [ref]$null)) {
                return
            }

            [datetime]::Parse('1970-01-01T00:00:00.0000000').AddMilliseconds($Milliseconds)
        }
        'Seconds' {
            if (-not [long]::TryParse($Seconds, [ref]$null)) {
                return
            }

            [datetime]::Parse('1970-01-01T00:00:00.0000000').AddSeconds($Seconds)
        }
    }
}

function ConvertTo-UnixTimestamp {
    param (
        [Parameter(Mandatory)]
        [datetime]$DateTime,

        [Parameter()]
        [switch]$AsMilliseconds
    )

    if ($AsMilliseconds.IsPresent) {
        (New-TimeSpan -Start '1970-01-01T00:00:00.0000000' -End $DateTime).TotalMilliseconds
    }
    else {
        (New-TimeSpan -Start '1970-01-01T00:00:00.0000000' -End $DateTime).TotalSeconds
    }
}

function Import-WithRegex {
    [CmdletBinding()]
    [OutputType([System.Management.Automation.PSCustomObject])]
    param (
        [Parameter(Mandatory, ValueFromPipeline)]
        [string]$CliOutput,

        [Parameter(Mandatory)]
        [string]$RegexString,

        [int]$Skip = 0,

        [switch]$ByLine
    )

    process {
        if ($ByLine.IsPresent) {
            $CliOutput -split '\n' | Where-Object { -not [string]::IsNullOrWhiteSpace($_) } | Select-Object -Skip $Skip | ForEach-Object {
                if ($_ -match $RegexString) {
                    if ($PSVersionTable.PSVersion -eq [version]::Parse('3.0')) {
                        New-Object -TypeName System.Management.Automation.PSObject -Property (Get-RegexNamedGroup -Hashtable $Matches)
                    }
                    else {
                        [pscustomobject](Get-RegexNamedGroup -Hashtable $Matches)
                    }
                }
            }
        }
        else {
            if ($CliOutput -match $RegexString) {
                if ($PSVersionTable.PSVersion -eq [version]::Parse('3.0')) {
                    New-Object -TypeName System.Management.Automation.PSObject -Property (Get-RegexNamedGroup -Hashtable $Matches)
                }
                else {
                    [pscustomobject](Get-RegexNamedGroup -Hashtable $Matches)
                }
            }
        }
    }
}

function Get-RegexNamedGroup {
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseLiteralInitializerForHashtable', '')]
    param (
        [Parameter(Mandatory)]
        [hashtable]$Hashtable
    )

    $newHashtable = New-Object -TypeName System.Collections.Hashtable

    $Hashtable.Keys | Where-Object { $_ -and $_ -notmatch '^\d+$' } | ForEach-Object {
        $newHashtable[$_] = $Hashtable[$_].Trim()
    }

    return $newHashtable
}

function Get-ObjectFromKeyValuePair {
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseLiteralInitializerForHashtable', '')]
    [CmdletBinding()]
    [OutputType([System.Management.Automation.PSCustomObject])]
    param (
        [Parameter(Mandatory, ValueFromPipeline)]
        [System.Object]$KeyValuePair
    )

    begin {
        $hashtable = New-Object -TypeName System.Collections.Hashtable
    }
    process {
        $KeyValuePair | ForEach-Object {
            $hashtable.Add($_.Name, $_.Value)
        }
    }
    end {
        if ($PSVersionTable.PSVersion -eq [version]::Parse('3.0')) {
            New-Object -TypeName System.Management.Automation.PSObject -Property $hashtable
        }
        else {
            [pscustomobject]$hashtable
        }
    }
}

function Convert-ListToBuckets {
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseSingularNouns', '')]
    [CmdletBinding()]
    [OutputType([System.Object[]])]
    param (
        [Parameter(Mandatory)]
        [ValidateNotNullOrEmpty()]
        [object[]]$List,

        [Parameter(Mandatory, ParameterSetName = 'NumberOfBuckets')]
        [ValidateRange(0, [int]::MaxValue)]
        [int]$NumberOfBuckets,

        [Parameter(Mandatory, ParameterSetName = 'BucketSize')]
        [ValidateRange(0, [int]::MaxValue)]
        [int]$BucketSize
    )

    $aggregateList = @( )

    if ($NumberOfBuckets -gt 0) {
        $blockCount = [System.Math]::Ceiling($List.Count / $NumberOfBuckets)
        $loopCount = $NumberOfBuckets
        $endBlockCount = $blockCount
    }
    else {
        $blockCount = [System.Math]::Floor($List.Count / $BucketSize)
        $leftOver = $List.Count % $BucketSize
        $loopCount = $blockCount
        $endBlockCount = $BucketSize
    }

    for ($i = 0; $i -lt $loopCount; $i++) {
        $end = $endBlockCount * ($i + 1) - 1

        $aggregateList += , $List[$start..$end]
        $start = $end + 1
    }

    if ($leftOver -gt 0) {
        $aggregateList += , $List[$start..($end + $leftOver)]
    }

    $aggregateList
}

function Show-ErrorMessage {
    [CmdletBinding()]
    param (
        [Parameter(Mandatory)]
        [string]$Message
    )

    if ($Host.Name -in 'Default Host', 'ConsoleHost') {
        [System.Console]::ForegroundColor = [System.ConsoleColor]::Red
        [System.Console]::WriteLine("ERROR: $Message")
        [System.Console]::ResetColor()
    }
    else {
        $Host.UI.WriteLine([System.ConsoleColor]::Red, $Host.UI.RawUI.BackgroundColor, "ERROR: $Message")
    }
}

function Show-WarningMessage {
    [CmdletBinding()]
    param (
        [Parameter(Mandatory)]
        [string]$Message
    )

    if ($Host.Name -in 'Default Host', 'ConsoleHost') {
        [System.Console]::ForegroundColor = [System.ConsoleColor]::Yellow
        [System.Console]::WriteLine("WARNING: $Message")
        [System.Console]::ResetColor()
    }
    else {
        $Host.UI.WriteLine([System.ConsoleColor]::Yellow, $Host.UI.RawUI.BackgroundColor, "WARNING: $Message")
    }
}

function Show-InformationMessage {
    [CmdletBinding()]
    param (
        [Parameter(Mandatory)]
        [string]$Message
    )

    if ($Host.Name -in 'Default Host', 'ConsoleHost') {
        [System.Console]::ForegroundColor = [System.ConsoleColor]::White
        [System.Console]::WriteLine("INFORMATION: $Message")
        [System.Console]::ResetColor()
    }
    else {
        $Host.UI.WriteLine([System.ConsoleColor]::White, $Host.UI.RawUI.BackgroundColor, "INFORMATION: $Message")
    }
}

function Get-NtapDocsDataSetSchemaVersion {
    param (
        [Parameter(Mandatory)]
        [System.Data.DataSet]$DataSet
    )

    if ($DataSet.ExtendedProperties.ContainsKey('SchemaVersion')) {
        [version]::Parse($DataSet.ExtendedProperties['SchemaVersion'])
    }
    else {
        $ntapDocsVersionColumnName = ($DataSet.Tables['DocumentVersionInformation'].Columns | Where-Object { $_.ColumnName -like '*DocsVersion' }).ColumnName
        [version]::Parse(
            (
                $DataSet.Tables['DocumentVersionInformation'].Rows | Select-Object -Property @{
                    Name       = 'Version'
                    Expression = { ($_.$ntapDocsVersionColumnName -split '\s*[|]\s*')[1] -Replace '^[^:]*[:]\s*' }
                } -Unique | Sort-Object -Property Version | Select-Object -First 1
            ).Version
        )
    }
}

function Get-VersionString {
    [CmdletBinding(DefaultParameterSetName = 'none')]
    [OutputType([System.String])]
    param (
        [Parameter(Mandatory, Position = 0)]
        [object]$InputVersion,

        [Parameter(Mandatory, ParameterSetName = 'PSVersion')]
        [switch]$PSVersion,

        [Parameter(Mandatory, ParameterSetName = 'AsFeatureVersion')]
        [switch]$AsFeatureVersion,

        [Parameter(Mandatory, ParameterSetName = 'AsPatchVersion')]
        [switch]$AsPatchVersion
    )

    switch ($PSCmdlet.ParameterSetName) {
        'PSVersion' {
            if ($InputVersion.psobject.TypeNames[0] -like '*SemanticVersion') {
                $InputVersion.ToString()
            }
            else {
                [string]::Format('{0}.{1}', $InputVersion.Major, $InputVersion.Minor)
            }
        }
        'AsFeatureVersion' {
            [string]::Format('{0}.{1}', $InputVersion.Major, $InputVersion.Minor)
        }
        'AsPatchVersion' {
            if ($InputVersion.psobject.TypeNames[0] -like '*SemanticVersion') {
                [string]::Format('{0}.{1}.{2}', $InputVersion.Major, $InputVersion.Minor, ($InputVersion.Patch, 0)[[byte](! ($InputVersion.Patch -gt 0))])
            }
            else {
                [string]::Format('{0}.{1}.{2}', $InputVersion.Major, $InputVersion.Minor, ($InputVersion.Build, 0)[[byte](! ($InputVersion.Build -gt 0))])
            }
        }
        default {
            $InputVersion.ToString()
        }
    }
}

function ConvertFrom-ShelfLogicalUid {
    <#
        .SYNOPSIS
            Converts a LogicalUid string into the Int64 referenced by ONTAP (disk show output 'shelf-uid' field).

        .DESCRIPTION
            ONTAP lists the ShelfUid that the disk resides on as an Int64. This function calculates that Int64 from a given
            shelf UID.

        .EXAMPLE
            PS> ConvertFrom-ShelfLogicalUid -LogicalUid '50:05:0c:c1:02:03:2f:ca'
            14568866629930452304

        .LINK
            https://community.netapp.com/t5/Microsoft-Virtualization-Discussions/bd-p/microsoft-cloud-and-virtualization-discussions
    #>

    param (
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$LogicalUid
    )

    if ([string]::IsNullOrWhiteSpace($LogicalUid)) {
        return
    }

    # ONTAP sometimes adds an extra :00 element to the end of the UID during the calculation
    $uidHexElements = ($LogicalUid -split ':') + '00'

    # Reverse sort for Big Endian conversion
    [array]::Reverse($uidHexElements)

    # If the ':00' is added by ONTAP then the number becomes larger than the Int64.MaxValue, so we must use a BigInteger instead
    $bigint = [bigint]::Zero

    if (
        [bigint]::TryParse(
            ($uidHexElements -join ''),
            [System.Globalization.NumberStyles]::HexNumber,
            [cultureinfo]::InvariantCulture,
            [ref]$bigint
        )
    ) {
        $bigint
    }
}

function ConvertTo-ShelfLogicalUid {
    <#
        .SYNOPSIS
          Converts the Int64 value referenced by ONTAP back to the proper LogicalUID (disk show output 'shelf-uid' field).

        .DESCRIPTION
            ONTAP lists the ShelfUID that the disk resides on as an Int64. This function converts that Int64 value back to
            the proper shelf UID.

        .EXAMPLE
            PS> ConvertTo-ShelfLogicalUid -LogicalUid 14568866629930452304
            50:05:0c:c1:02:03:2f:ca

        .LINK
            https://community.netapp.com/t5/Microsoft-Virtualization-Discussions/bd-p/microsoft-cloud-and-virtualization-discussions
    #>

    param (
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$Number
    )

    if ([string]::IsNullOrWhiteSpace($Number)) {
        return
    }

    # We must use a BigInteger as the number may be larger than the Int64.MaxValue
    $bigint = [bigint]::Zero

    if ([bigint]::TryParse($Number, [ref]$bigint)) {
        (
            $bigint.ToByteArray() | Select-Object -First 8 | ForEach-Object {
                $_.ToString('x2')
            }
        ) -join ':'
    }
}

function Get-SessionInfo ([int]$HistoryId) {
    $getMachineGuid = {
        $machineGuid = [guid]::Empty

        if ($PSVersionTable.PSVersion.Major -lt 6 -or $IsWindows) {
            $machineGuidRegValue = $(
                if ([System.Environment]::Is64BitOperatingSystem -and -not [System.Environment]::Is64BitProcess) {
                    try {
                        $baseKey = [Microsoft.Win32.RegistryKey]::OpenBaseKey([Microsoft.Win32.RegistryHive]::LocalMachine, [Microsoft.Win32.RegistryView]::Registry64)
                        $subKey = $baseKey.OpenSubKey('Software\Microsoft\Cryptography', [Microsoft.Win32.RegistryKeyPermissionCheck]::ReadSubTree)
                        $subKey.GetValue('MachineGuid')
                    }
                    catch {
                        $Global:Error.RemoveAt(0)
                    }
                    finally {
                        $subKey.Close()
                        $baseKey.Close()
                    }
                }
                else {
                    (Get-ItemProperty -Path 'HKLM:SOFTWARE\Microsoft\Cryptography' -Name MachineGuid -ErrorAction Ignore).MachineGuid
                }
            )

            if (-not [guid]::TryParse($machineGuidRegValue, [ref]$machineGuid)) {
                [void][guid]::TryParse((Get-CimInstance -ClassName Win32_ComputerSystemProduct -Property UUID -ErrorAction Ignore).Uuid, [ref]$machineGuid)
            }
        }
        elseif ($IsLinux) {
            [void][guid]::TryParse((Get-Content -Path '/etc/machine-id'), [ref]$machineGuid)

            #if ($machineGuid -eq [guid]::Empty) {
            #    #suggest running: systemd-machine-id-setup
            #}
        }
        elseif ($IsMacOS) {
            [void][guid]::TryParse((& ioreg -d2 -c IOPlatformExpertDevice | Select-String -Pattern '"IOPlatformUUID"\s*[=]\s*"(.+?)"$').Matches.Groups[1].Value, [ref]$machineGuid)
        }

        $machineGuid
    }

    [pscustomobject]@{
        ModuleSessionGuid   = [guid]::NewGuid()
        PipelineSessionGuid = [guid]::Empty
        HistoryId           = (Get-PSCallStack)[0].InvocationInfo.HistoryId
        OSInfo              = [System.Environment]::OSVersion | Select-Object -ExcludeProperty Platform -Property @{
            Name       = 'Platform'
            Expression = {
                if (Test-Path -Path './.dockerenv' -ErrorAction Ignore) {
                    "$( $_.Platform ) (Docker)"
                }
                else {
                    $_.Platform
                }
            }
        }, Version, @{
            Name       = 'Caption'
            Expression = {
                if ($PSVersionTable.PSVersion.Major -lt 6 -or $IsWindows) {
                    (Get-CimInstance -ClassName Win32_OperatingSystem -Property Caption).Caption
                }
                elseif ($IsLinux) {
                    [regex]::Match((Get-Content -Path '/etc/os-release' -Raw), '(?s)(?=.*\bPRETTY_NAME\s*[:|=]\s*\"(?<Caption>.+?)\")(?=.*\bVERSION_ID\s*[:|=]\s*\"(?<Version>.+?)\").*$').Groups['Caption'].Value
                }
                elseif ($IsMacOS) {
                    [regex]::Match(((& sw_vers) -join "`n"), '(?s)^(?=.*ProductName\s*[:]\s*(?<ProductName>.+?)\n)(?=.*ProductVersion\s*[:]\s*(?<ProductVersion>.+?)\n)(?=.*BuildVersion\s*[:]\s*(?<BuildVersion>.+?)\b).*$').Groups['ProductName'].Value
                }
            }
        }, @{
            Name       = 'Architecture'
            Expression = {
                $(
                    if ([System.Environment]::Is64BitOperatingSystem) {
                        '64-bit'
                    }
                    else {
                        '32-bit'
                    }
                )
            }
        }
        NetAppDocsVersion   = (Test-ModuleManifest -Path "$( (Get-PSCallStack)[0].InvocationInfo.ScriptName -replace '[.]psm1', '.psd1' )" -ErrorAction Ignore) | ForEach-Object { [version]::Parse([string]::Format('{0}.{1}', $_.Version.ToString(3), $_.PrivateData.BuildNumber)) }
        PowerShellVersion   = $PSVersionTable.PSVersion
        MachineGuid         = $getMachineGuid.InvokeReturnAsIs()
        LocationInfo        = [pscustomobject]@{
            CountryCode = [System.Globalization.RegionInfo]::CurrentRegion.ThreeLetterISORegionName
        }
    }
}

function FilterByVserver {
    param ([string]$VserverName)

    process {
        if ($_.VolumeIdAttributes.OwningVserverName -ceq $VserverName -or $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName -or $_.Vs -ceq $VserverName -or $_.VserverId -ceq $VserverName) {
            $_
        }
    }
}

function FilterByVolume {
    param ([string]$VserverName, [string]$VolumeName)

    process {
        if (
            ([string]::IsNullOrWhiteSpace($VserverName) -or ($_.VolumeIdAttributes.OwningVserverName -ceq $VserverName -or $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName -or $_.Vs -ceq $VserverName -or $_.VserverId -ceq $VserverName) ) -and
            ($_.VolumeIdAttributes.Name -ceq $VolumeName -or $_.VolumeName -ceq $VolumeName -or $_.Volume -ceq $VolumeName -or $_.Vol -ceq $VolumeName -or $_.VolId -ceq $VolumeName -or $_.V -ceq $VolumeName)
        ) {
            $_
        }
    }
}

function FilterByQtree {
    param ([string]$VserverName, [string]$VolumeName, [string]$QtreeName)

    process {
        if (
            ([string]::IsNullOrWhiteSpace($VserverName) -or ($_.VolumeIdAttributes.OwningVserverName -ceq $VserverName -or $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName -or $_.Vs -ceq $VserverName -or $_.VserverId -ceq $VserverName) ) -and
            ($_.VolumeIdAttributes.Name -ceq $VolumeName -or $_.VolumeName -ceq $VolumeName -or $_.Volume -ceq $VolumeName -or $_.Vol -ceq $VolumeName -or $_.VolId -ceq $VolumeName -or $_.V -ceq $VolumeName) -and
            ($_.Qtree -ceq $QtreeName -or $_.Qt -ceq $QtreeName)
        ) {
            $_
        }
    }
}

function FilterBySnapshot {
    param ([string]$VserverName, [string]$VolumeName, [string]$SnapshotName)

    process {
        if (
            ($_.VolumeIdAttributes.OwningVserverName -ceq $VserverName -or $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName -or $_.Vs -ceq $VserverName -or $_.VserverId -ceq $VserverName) -and
            ($_.VolumeIdAttributes.Name -ceq $VolumeName -or $_.VolumeName -ceq $VolumeName -or $_.Volume -ceq $VolumeName -or $_.Vol -ceq $VolumeName -or $_.VolId -ceq $VolumeName -or $_.V -ceq $VolumeName) -and
            ($_.Name -eq $SnapshotName -or $_.Snapshot -eq $SnapshotName)
        ) {
            $_
        }
    }
}

function FilterByCifsShare {
    param ([string]$VserverName, [string]$CifsShareName)

    process {
        if (
            ($_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName -or $_.Vs -ceq $VserverName -or $_.VserverId -ceq $VserverName) -and
            ($_.ShareName -ceq $CifsShareName -or $_.Share -ceq $CifsShareName -or $_.CifsShareId -ceq $CifsShareName)
        ) {
            $_
        }
    }
}

function Test-IsLibGdiPlusPackageInstalled {
    [CmdletBinding()]
    [OutputType([System.Boolean])]
    param ( )

    if ($PSVersionTable.PSVersion -lt [version]::Parse('6.0') -or $IsWindows) {
        return $true
    }

    if ($IsLinux -or $IsMacOS) {
        if ($PSVersionTable.PSVersion -ge [version]::Parse('7.2')) {
            $msg = @'
NetAppDocs is currently not supported on non-Windows platforms
with PowerShell 7.2+. This is due to the current limitation with the
components used to generate the MS Office documentation. NetAppDocs will
be updated to use the new components as soon as they are available.
The only solution at this time is to install and use the older 7.1.x version
of PowerShell on your MAC and Linux platforms.

'@

            Show-ErrorMessage -Message $msg

            return $false
        }

        $excelFile = New-Object -TypeName Aspose.Cells.Workbook

        try {
            $excelFile.Worksheets[0].AutoFitColumns()

            return $true
        }
        catch {
            if ($IsLinux) {
                $msg = @'
The following dependency packages must be installed on your system using your
package manager:
libgdiplus
libc6-dev

ubuntu/debian:
apt-get -y update && apt-get install -y --no-install-recommends libgdiplus libc6-dev

fedora/centos:
yum install libgdiplus libc6-dev

openuse linux:
zypper install libgdiplus libc6-dev

'@
            }

            if ($IsMacOS) {
                $msg = @'
The following dependency package must be installed on your system using the Homebrew
package manager:
mono-libgdiplus

brew install mono-libgdiplus

'@
            }

            Show-ErrorMessage -Message $msg

            return $false
        }
        finally {
            $excelFile.Dispose()
        }
    }
}

function Join-Object {
    <#
        .SYNOPSIS
            Join data from two sets of objects based on a common value

        .DESCRIPTION
            Join data from two sets of objects based on a common value
            For more details, see the accompanying blog post:
                http://ramblingcookiemonster.github.io/Join-Object/
            For even more details, see the original code and discussions that this borrows from:
                Dave Wyatt's Join-Object - http://powershell.org/wp/forums/topic/merging-very-large-collections
                Lucio Silveira's Join-Object - http://blogs.msdn.com/b/powershell/archive/2012/07/13/join-object.aspx

        .PARAMETER Left
            'Left' collection of objects to join. You can use the pipeline for Left.
            The objects in this collection should be consistent.
            We look at the properties on the first object for a baseline.

        .PARAMETER Right
            'Right' collection of objects to join.
            The objects in this collection should be consistent.
            We look at the properties on the first object for a baseline.

        .PARAMETER LeftJoinProperty
            Property on Left collection objects that we match up with RightJoinProperty on the Right collection

        .PARAMETER RightJoinProperty
            Property on Right collection objects that we match up with LeftJoinProperty on the Left collection

        .PARAMETER LeftProperties
            One or more properties to keep from Left. Default is to keep all Left properties (*).
            Each property can:
                - Be a plain property name like "Name"
                - Contain wildcards like "*"
                - Be a Hashtable like @{Name="Product Name";Expression={$_.Name}}.
                    Name is the output property name
                    Expression is the property value ($_ as the current object)
                    Alternatively, use the Suffix or Prefix parameter to avoid collisions
                    Each property using this Hashtable syntax will be excluded from suffixes and prefixes

        .PARAMETER RightProperties
            One or more properties to keep from Right. Default is to keep all Right properties (*).
            Each property can:
                - Be a plain property name like "Name"
                - Contain wildcards like "*"
                - Be a Hashtable like @{Name="Product Name";Expression={$_.Name}}.
                    Name is the output property name
                    Expression is the property value ($_ as the current object)
                    Alternatively, use the Suffix or Prefix parameter to avoid collisions
                    Each property using this Hashtable syntax will be excluded from suffixes and prefixes

        .PARAMETER Prefix
            If specified, prepend Right object property names with this prefix to avoid collisions
            Example:
                Property Name                   = 'Name'
                Suffix                          = 'j_'
                Resulting Joined Property Name  = 'j_Name'

        .PARAMETER Suffix
            If specified, append Right object property names with this suffix to avoid collisions
            Example:
                Property Name                   = 'Name'
                Suffix                          = '_j'
                Resulting Joined Property Name  = 'Name_j'

        .PARAMETER Type
            Type of join. Default is AllInLeft.
            AllInLeft will have all elements from Left at least once in the output, and might appear more than once
            if the where clause is true for more than one element in right, Left elements with matches in Right are
            preceded by elements with no matches.
            SQL equivalent: outer left join (or simply left join)
            AllInRight is similar to AllInLeft.
            OnlyIfInBoth will cause all elements from Left to be placed in the output, only if there is at least one
            match in Right.
            SQL equivalent: inner join (or simply join)
            AllInBoth will have all entries in right and left in the output. Specifically, it will have all entries
            in right with at least one match in left, followed by all entries in Right with no matches in left,
            followed by all entries in Left with no matches in Right.
            SQL equivalent: full join

        .EXAMPLE
            #
            #Define some input data.
            $l = 1..5 | Foreach-Object {
                [PSCustomObject]@{
                    Name = "jsmith$_"
                    Birthday = (Get-Date).AddDays(-1)
                }
            }
            $r = 4..7 | Foreach-Object{
                [PSCustomObject]@{
                    Department = "Department $_"
                    Name = "Department $_"
                    Manager = "jsmith$_"
                }
            }
            #We have a name and Birthday for each manager, how do we find their department, using an inner join?
            Join-Object -Left $l -Right $r -LeftJoinProperty Name -RightJoinProperty Manager -Type OnlyIfInBoth -RightProperties Department
                # Name    Birthday             Department
                # ----    --------             ----------
                # jsmith4 4/14/2015 3:27:22 PM Department 4
                # jsmith5 4/14/2015 3:27:22 PM Department 5

        .EXAMPLE
            #
            #Define some input data.
            $l = 1..5 | Foreach-Object {
                [PSCustomObject]@{
                    Name = "jsmith$_"
                    Birthday = (Get-Date).AddDays(-1)
                }
            }
            $r = 4..7 | Foreach-Object{
                [PSCustomObject]@{
                    Department = "Department $_"
                    Name = "Department $_"
                    Manager = "jsmith$_"
                }
            }
            #We have a name and Birthday for each manager, how do we find all related department data, even if there are conflicting properties?
            $l | Join-Object -Right $r -LeftJoinProperty Name -RightJoinProperty Manager -Type AllInLeft -Prefix j_
                # Name    Birthday             j_Department j_Name       j_Manager
                # ----    --------             ------------ ------       ---------
                # jsmith1 4/14/2015 3:27:22 PM
                # jsmith2 4/14/2015 3:27:22 PM
                # jsmith3 4/14/2015 3:27:22 PM
                # jsmith4 4/14/2015 3:27:22 PM Department 4 Department 4 jsmith4
                # jsmith5 4/14/2015 3:27:22 PM Department 5 Department 5 jsmith5

        .EXAMPLE
            #
            #Hey! You know how to script right? Can you merge these two CSVs, where Path1's IP is equal to Path2's IP_ADDRESS?
            #Get CSV data
            $s1 = Import-CSV $Path1
            $s2 = Import-CSV $Path2
            #Merge the data, using a full outer join to avoid omitting anything, and export it
            Join-Object -Left $s1 -Right $s2 -LeftJoinProperty IP_ADDRESS -RightJoinProperty IP -Prefix 'j_' -Type AllInBoth |
                Export-CSV $MergePath -NoTypeInformation

        .EXAMPLE
            #
            # "Hey Warren, we need to match up SSNs to Active Directory users, and check if they are enabled or not.
            #  I'll e-mail you an unencrypted CSV with all the SSNs from gmail, what could go wrong?"
            # Import some SSNs.
            $SSNs = Import-CSV -Path D:\SSNs.csv
            #Get AD users, and match up by a common value, SamAccountName in this case:
            Get-ADUser -Filter "SamAccountName -like 'WFrame*'" |
                Join-Object -LeftJoinProperty SamAccountName -Right $SSNs `
                            -RightJoinProperty SamAccountName -RightProperties ssn `
                            -LeftProperties SamAccountName, enabled, ObjectClass

        .NOTES
            This borrows from:
                Dave Wyatt's Join-Object - http://powershell.org/wp/forums/topic/merging-very-large-collections/
                Lucio Silveira's Join-Object - http://blogs.msdn.com/b/powershell/archive/2012/07/13/join-object.aspx
            Changes:
                Always display full set of properties
                Display properties in order (left first, right second)
                If specified, add suffix or prefix to right object property names to avoid collisions
                Use a Hashtable rather than OrderedDictionary (avoid case sensitivity)

        .LINK
            http://ramblingcookiemonster.github.io/Join-Object/

        .FUNCTIONALITY
            PowerShell Language
    #>

    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseDeclaredVarsMoreThanAssignments', '')]
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseProcessBlockForPipelineCommand', '')]
    [CmdletBinding(DefaultParameterSetName = 'Default')]
    [OutputType([System.Array])]
    param (
        [Parameter(Mandatory = $true, ValueFromPipeLine = $true)]
        $Left,

        [Parameter(Mandatory = $true)]
        [AllowNull()]
        $Right,

        [Parameter(Mandatory = $true)]
        [string[]]$LeftJoinProperty,

        [Parameter(Mandatory = $true)]
        [string[]]$RightJoinProperty,

        $LeftJoinScript,

        $RightJoinScript,

        [ValidateScript({ $_ -is [Collections.Hashtable] -or $_ -is [string] -or $_ -is [Collections.Specialized.OrderedDictionary] })]
        $LeftProperties = '*',

        [ValidateScript({ $_ -is [Collections.Hashtable] -or $_ -is [string] -or $_ -is [Collections.Specialized.OrderedDictionary] })]
        $RightProperties = '*',

        [string[]]$ExcludeLeftProperties,

        [string[]]$ExcludeRightProperties,

        [switch]$KeepRightJoinProperty,

        [ValidateSet('AllInLeft', 'OnlyIfInBoth')] #, 'AllInBoth')]
        [Parameter(Mandatory = $false)]
        [string]$Type = 'AllInLeft',

        [string]$Prefix,

        [string]$Suffix,

        [Parameter(Mandatory, ParameterSetName = 'PassThru')]
        [switch]$PassThru,

        [Parameter(Mandatory, ParameterSetName = 'DataTable')]
        [switch]$DataTable,

        [Parameter(ParameterSetName = 'PassThru')]
        [Parameter(ParameterSetName = 'DataTable')]
        [Hashtable]$DataTableTypes,

        [ValidateSet('SingleOnly', 'DuplicateLines', 'SubGroups')]
        [string]$LeftMultiMode = 'SingleOnly',

        [ValidateSet('SingleOnly', 'DuplicateLines', 'SubGroups')]
        [string]$RightMultiMode = 'SingleOnly',

        [String]$AddKey,

        [switch]$AllowColumnsMerging,

        [Collections.Generic.IEqualityComparer[Object]]$Comparer
    )

    # if $Right is null then just return $Left as-is
    if (-not $Right) {
        return $Left
    }

    #region Validate Params

    #if ($PassThru -and ($Type -in @( 'AllInBoth', 'OnlyIfInBoth' ) -or ($Type -eq 'AllInLeft' -and $RightMultiMode -eq 'DuplicateLines'))) {
    if ($PassThru -and ($Type -eq 'OnlyIfInBoth' -or ($Type -eq 'AllInLeft' -and $RightMultiMode -eq 'DuplicateLines'))) {
        $PSCmdlet.ThrowTerminatingError(
            [Management.Automation.ErrorRecord]::new(
                [ArgumentException]::new('"-PassThru" compatible only with "-Type AllInLeft" with "-RightMultiMode" "SingleOnly" or "SubGroups"'),
                'Incompatible Arguments',
                [Management.Automation.ErrorCategory]::InvalidArgument,
                $Type
            )
        )
    }

    <#if ($AddKey -and $Type -ne 'AllInBoth') {
        $PSCmdlet.ThrowTerminatingError(
            [Management.Automation.ErrorRecord]::new(
                [ArgumentException]::new('"-AddKey" support only "-Type AllInBoth"'),
                'Incompatible Arguments',
                [Management.Automation.ErrorCategory]::InvalidArgument,
                $AddKey
            )
        )
    }#>

    if ($Type -in @( 'AllInLeft', 'OnlyIfInBoth' )) {
        if ($PSBoundParameters['LeftMultiMode'] -ne 'DuplicateLines' -and $null -ne $PSBoundParameters['LeftMultiMode']) {
            $PSCmdlet.ThrowTerminatingError(
                [Management.Automation.ErrorRecord]::new(
                    [ArgumentException]::new('"-Type AllInLeft" and "-Type OnlyIfInBoth" support only "-LeftMultiMode DuplicateLines"'),
                    'Incompatible Arguments',
                    [Management.Automation.ErrorCategory]::InvalidArgument,
                    $Type
                )
            )
        }

        $Attributes = (Get-Variable 'LeftMultiMode').Attributes
        [void]$Attributes.Remove($Attributes.Where({ $_.TypeId.Name -eq 'ValidateSetAttribute' })[0])
        $ValidateSetAttribute = [System.Management.Automation.ValidateSetAttribute]::new('SingleOnly', 'DuplicateLines', 'SubGroups', $null)
        $Attributes.Add($ValidateSetAttribute)
        $LeftMultiMode = $null
    }

    if ($Type -eq 'OnlyIfInBoth') {
        if ($PSBoundParameters['RightMultiMode'] -ne 'DuplicateLines' -and $null -ne $PSBoundParameters['RightMultiMode']) {
            $PSCmdlet.ThrowTerminatingError(
                [Management.Automation.ErrorRecord]::new(
                    [ArgumentException]::new('"-Type OnlyIfInBoth" support only "-RightMultiMode DuplicateLines"'),
                    'Incompatible Arguments',
                    [Management.Automation.ErrorCategory]::InvalidArgument,
                    $Type
                )
            )
        }

        $Attributes = (Get-Variable 'RightMultiMode').Attributes
        [void]$Attributes.Remove($Attributes.Where({ $_.TypeId.Name -eq 'ValidateSetAttribute' })[0])
        $ValidateSetAttribute = [System.Management.Automation.ValidateSetAttribute]::new('SingleOnly', 'DuplicateLines', 'SubGroups', $null)
        $Attributes.Add($ValidateSetAttribute)
        $RightMultiMode = $null
    }

    if ($AllowColumnsMerging -and !$DataTable -and !($PassThru -and $Left -is [DataTable])) {
        $PSCmdlet.ThrowTerminatingError(
            [Management.Automation.ErrorRecord]::new(
                [ArgumentException]::new('"-AllowColumnsMerging" support only on DataTable output'),
                'Incompatible Arguments',
                [Management.Automation.ErrorCategory]::InvalidArgument,
                $AllowColumnsMerging
            )
        )
    }

    #endregion Validate Params

    #region Set $SelectedLeftProperties and $SelectedRightProperties

    function Get-Properties {
        [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseSingularNouns', '')]
        [CmdletBinding()]
        [OutputType([System.Collections.Specialized.OrderedDictionary])]
        param (
            $Object,
            $SelectProperties,
            $ExcludeProperties,
            $Prefix,
            $Suffix
        )

        $Properties = [ordered]@{ }

        if ($Object -is [System.Data.DataTable]) {
            $ObjectProperties = $Object.Columns.ColumnName
        }
        else {
            #$ObjectProperties = $Object[0].PSObject.Properties.Name
            $ObjectProperties = ($Object | Group-Object -Property { @( $_.psobject.properties ).Count } | Select-Object -Last 1).Group[0].psobject.properties.name
        }

        if ($SelectProperties -is [Hashtable] -or $SelectProperties -is [Collections.Specialized.OrderedDictionary]) {
            $SelectProperties.GetEnumerator() | Where-Object { $_.Key -notin $ExcludeProperties } | ForEach-Object { $Properties.Add($_.Key, $Prefix + $_.Value + $Suffix) }
        }
        elseif ($SelectProperties -eq '*') {
            $ObjectProperties | Where-Object { $_ -notin $ExcludeProperties } | ForEach-Object { $Properties.Add($_, $Prefix + $_ + $Suffix) }
        }
        else {
            $SelectProperties | Where-Object { $_ -notin $ExcludeProperties } | ForEach-Object { $Properties.Add($_, $Prefix + $_ + $Suffix) }
        }

        $Properties
    }

    $SelectedLeftProperties = Get-Properties -Object $Left -SelectProperties $LeftProperties -ExcludeProperties $ExcludeLeftProperties

    if (!$KeepRightJoinProperty) {
        $ExcludeRightProperties = $null -ne @( $ExcludeRightProperties ) + @( $RightJoinProperty )
    }

    $SelectedRightProperties = Get-Properties -Object $Right -SelectProperties $RightProperties -ExcludeProperties $ExcludeRightProperties -Prefix $Prefix -Suffix $Suffix

    #endregion Set $SelectedLeftProperties and $SelectedRightProperties

    #region Importing package MoreLinq

    <#if ($Type -eq 'AllInBoth') {
        try {
            if ($PSScriptRoot) {
                $ScriptRoot = $PSScriptRoot
            }
            elseif ($psISE.CurrentFile.IsUntitled -eq $false) {
                $ScriptRoot = Split-Path -Path $psISE.CurrentFile.FullPath
            }
            elseif ($null -ne $psEditor.GetEditorContext().CurrentFile.Path -and $psEditor.GetEditorContext().CurrentFile.Path -notlike 'untitled:*') {
                $ScriptRoot = Split-Path -Path $psEditor.GetEditorContext().CurrentFile.Path
            }
            else {
                $ScriptRoot = '.'
            }

            if (!('MoreLinq.MoreEnumerable' -as [type])) {
                Add-Type -Path (Resolve-Path -Path "$ScriptRoot\morelinq.*\lib\Core\MoreLinq.dll")
            }
        }
        catch {
            $PSCmdlet.ThrowTerminatingError(
                [Management.Automation.ErrorRecord]::new(
                    [TypeLoadException]::new('Importing package MoreLinq failed: {0}' -f $_.Exception.Message, $_.Exception),
                    'Importing package',
                    [Management.Automation.ErrorCategory]::NotInstalled,
                    $null
                )
            )
        }
    }#>

    #endregion Importing package MoreLinq

    #region Set $RightJoinScript and $LeftJoinScript

    function Get-JoinScript {
        [CmdletBinding()]
        [OutputType([System.Func[Object, String]])]
        param (
            $JoinScript,
            $JoinProperty,
            $Side,
            $Object
        )

        if ($JoinScript) {
            if ($JoinScript.GetType().Name -eq 'Func`2') {
                $JoinScript #.GetNewClosure()
            }
            else {
                [System.Func[Object, String]]$JoinScript
            }
        }
        else {
            $JoinScript = if ($JoinProperty.Count -gt 1) {
                {
                    param ($_Side_Line)
                    ($_Side_Line | Select-Object -Property $_Side_JoinProperty).PSObject.Properties.Value
                }
            }
            else {
                if ($Object -is [Data.DataTable]) {
                    {
                        param ($_Side_Line)
                        $_Side_Line[$_Side_JoinProperty]
                    }
                }
                else {
                    {
                        param ($_Side_Line)
                        $_Side_Line.$_Side_JoinProperty
                    }
                }
            }

            [System.Func[Object, String]][Scriptblock]::Create($JoinScript.ToString().Replace('_Side_', $Side))
        }
    }

    $LeftJoinScript = Get-JoinScript -JoinScript $LeftJoinScript -JoinProperty $LeftJoinProperty -Side 'Left' -Object $Left
    $RightJoinScript = Get-JoinScript -JoinScript $RightJoinScript -JoinProperty $RightJoinProperty -Side 'Right' -Object $Right

    #endregion Set $RightJoinScript and $LeftJoinScript

    #region Prepare Data

    function Set-OutDataTable {
        [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseShouldProcessForStateChangingFunctions', '')]
        param (
            $OutDataTable,
            $Object,
            $SelectedProperties,
            $AllowColumnsMerging
        )

        # Create Columns
        foreach ($item in $SelectedProperties.GetEnumerator()) {
            if (!$AllowColumnsMerging -or !$OutDataTable.Columns.Item($item.Value)) {
                if ($Object -is [Data.DataTable]) {
                    [void]$OutDataTable.Columns.Add($item.Value, $Object.Columns.Item($item.Name).DataType)
                }
                else {
                    if ($null -ne $DataTableTypes.($item.Value)) {
                        [void]$OutDataTable.Columns.Add($item.Value, $DataTableTypes.($item.Value))
                    }
                    else {
                        [void]$OutDataTable.Columns.Add($item.Value)
                    }
                }
            }
        }
    }

    if ($DataTable) {
        $OutDataTable = [Data.DataTable]::new('Joined')

        if ($AddKey) {
            [void]$OutDataTable.Columns.Add($AddKey)
        }

        if ($LeftMultiMode -eq 'SubGroups') {
            $OutDataTableSubGroupTemplateLeft = [Data.DataTable]::new('LeftGroup')
            Set-OutDataTable -OutDataTable $OutDataTableSubGroupTemplateLeft -Object $Left -SelectedProperties $SelectedLeftProperties

            [void]$OutDataTable.Columns.Add('LeftGroup', [Object])
        }
        else {
            Set-OutDataTable -OutDataTable $OutDataTable -Object $Left -SelectedProperties $SelectedLeftProperties
        }

        if ($RightMultiMode -eq 'SubGroups') {
            $OutDataTableSubGroupTemplateRight = [Data.DataTable]::new('RightGroup')
            Set-OutDataTable -OutDataTable $OutDataTableSubGroupTemplateRight -Object $Right -SelectedProperties $SelectedRightProperties

            [void]$OutDataTable.Columns.Add('RightGroup', [Object])
        }
        else {
            Set-OutDataTable -OutDataTable $OutDataTable -Object $Right -SelectedProperties $SelectedRightProperties -AllowColumnsMerging $AllowColumnsMerging
        }
    }
    elseif ($PassThru) {
        if ($Left -is [Data.DataTable]) {
            # Remove LeftLine
            foreach ($ColumnName in $Left.Columns.ColumnName) {
                if ($ColumnName -notin $SelectedLeftProperties.Keys) {
                    $Left.Columns.Remove($ColumnName)
                }
            }

            # Rename LeftLine
            foreach ($item in $SelectedLeftProperties.GetEnumerator()) {
                if ($item.Key -ne $item.Value -and ($Column = $Left.Columns.Item($item.Key))) {
                    $Column.ColumnName = $item.Value
                }
            }

            if ($RightMultiMode -eq 'SubGroups') {
                [void]$Left.Columns.Add('RightGroup', [Object])
            }
            else {
                # Add RightLine to LeftLine
                foreach ($item in $SelectedRightProperties.GetEnumerator()) {
                    if (!$AllowColumnsMerging -or !$Left.Columns.Item($item.Value)) {
                        if ($null -ne $DataTableTypes.($item.Value)) {
                            [void]$Left.Columns.Add($item.Value, $DataTableTypes.($item.Value))
                        }
                        else {
                            [void]$Left.Columns.Add($item.Value)
                        }
                    }
                }
            }
        }

        if ($Right -is [Data.DataTable] -and $RightMultiMode -eq 'SubGroups') {
            $OutDataTableSubGroupTemplateRight = [Data.DataTable]::new('RightGroup')

            Set-OutDataTable -OutDataTable $OutDataTableSubGroupTemplateRight -Object $Right -SelectedProperties $SelectedRightProperties
        }
    }

    #endregion Prepare Data

    #region Main

    #region Main: Set $QueryParts

    $QueryParts = @{
        'IfSideLine'                       = {
            if ($_Side_Line) {
                _SideScript_
            }
        }.ToString()
        'DataTableFromAny'                 = {
            foreach ($item in $Selected_Side_Properties.GetEnumerator()) {
                $Value = $_Side_Line.($item.Key)

                if ($null -ne $Value) {
                    $_Row_[$item.Value] = $Value
                }
            }
        }.ToString()
        'DataTableFromDataTable'           = {
            foreach ($item in $Selected_Side_Properties.GetEnumerator()) {
                $Value = $_Side_Line[$item.Key]

                if ($Value -isnot [System.DBNull]) {
                    $_Row_[$item.Value] = $Value
                }
            }
        }.ToString()
        'DataTableFromSubGroup'            = {
            if ($_Side_Lines) {
                _SubGroup_
                $_Row_['_Side_Group'] = $OutSubGroup_Side_
            }
        }.ToString()
        'SubGroupFromDataTable'            = {
            $OutSubGroup_Side_ = $OutDataTableSubGroupTemplate_Side_.Clone()
            foreach ($_Side_Line in $_Side_Lines) {
                $RowSubGroup = $OutSubGroup_Side_.Rows.Add()

                _DataTable_
            }
        }.ToString()
        'SubGroupFromPSCustomObject'       = {
            $OutSubGroup_Side_ = @( )

            foreach ($_Side_Line in $_Side_Lines) {
                $RowSubGroup = [ordered]@{ }

                _PSCustomObject_

                $OutSubGroup_Side_ += [PSCustomObject]$RowSubGroup
            }
        }.ToString()
        'PSCustomObjectFromPSCustomObject' = {
            foreach ($item in $Selected_Side_Properties.GetEnumerator()) {
                $_Row_.Add($item.Value, $_Side_Line.($item.Key))
            }
        }.ToString()
        'PSCustomObjectFromAny'            = {
            foreach ($item in $Selected_Side_Properties.GetEnumerator()) {
                $Value = $_Side_Line.($item.Key)

                if ($Value -is [System.DBNull]) {
                    $Value = $null
                }

                $_Row_.Add($item.Value, $Value)
            }
        }.ToString()
        'PSCustomObjectFromSubGroup'       = {
            if ($_Side_Lines) {
                _SubGroup_

                $_Row_.Add('_Side_Group', $OutSubGroup_Side_)
            }
            else {
                $_Row_.Add('_Side_Group', $null)
            }
        }.ToString()
    }

    #endregion Main: Set $QueryParts

    #region Main: Set $Query

    $Query = if ($PassThru) {
        if ($Left -is [Data.DataTable]) {
            $QueryTemp = @{
                Main        = { _SidesScript_ }.ToString()
                SideReplace = '_Row_', 'LeftLine'
            }

            if ($RightMultiMode -eq 'SubGroups') {
                $QueryTemp['SideSubGroupBase'] = $QueryParts['DataTableFromSubGroup']

                if ($Right -is [Data.DataTable]) {
                    $QueryTemp['SideSubGroup'] = $QueryParts['SubGroupFromDataTable']
                }
                else {
                    $QueryTemp['SideSubGroup'] = $QueryParts['SubGroupFromPSCustomObject']
                }
            }
            else {
                $QueryTemp['Right'] = { _DataTable_ }.ToString()
            }

            $QueryTemp
        }
        else {
            # Left is PSCustomObject
            $QueryTemp = @{
                # Edit PSCustomObject
                Main        = {
                    # Add to LeftLine (Rename)
                    foreach ($item in $SelectedLeftProperties.GetEnumerator()) {
                        if ($item.Value -notin $LeftLine.PSObject.Properties.Name) {
                            $LeftLine.PSObject.Properties.Add([Management.Automation.PSNoteProperty]::new($item.Value, $LeftLine.($item.Key)))
                        }
                    }
                    # Remove from LeftLine
                    foreach ($item in $LeftLine.PSObject.Properties.Name) {
                        if ($item -notin $SelectedLeftProperties.Values) {
                            $LeftLine.PSObject.Properties.Remove($item)
                        }
                    }
                    _SidesScript_
                }.ToString()

                SideReplace = '_Row_\.Add([^\r\n]*)', 'LeftLine.PSObject.Properties.Add([Management.Automation.PSNoteProperty]::new$1)'
            }

            if ($RightMultiMode -eq 'SubGroups') {
                $QueryTemp['SideSubGroupBase'] = $QueryParts['PSCustomObjectFromSubGroup']

                if ($Right -is [Data.DataTable]) {
                    $QueryTemp['SideSubGroup'] = $QueryParts['SubGroupFromDataTable']
                }
                else {
                    $QueryTemp['SideSubGroup'] = $QueryParts['SubGroupFromPSCustomObject']
                }
            }
            else {
                $QueryTemp['Right'] = { _PSCustomObject_ }.ToString()
            }

            $QueryTemp
        }
    }
    elseif ($DataTable) {
        $QueryTemp = @{
            Main        = {
                $RowMain = $OutDataTable.Rows.Add()
                _SidesScript_
            }.ToString()

            SideReplace = '_Row_', 'RowMain'
        }

        if ($LeftMultiMode -eq 'SubGroups' -or $RightMultiMode -eq 'SubGroups') {
            $QueryTemp['SideSubGroupBase'] = $QueryParts['DataTableFromSubGroup']
            $QueryTemp['SideSubGroup'] = $QueryParts['SubGroupFromDataTable']
        }

        if ($LeftMultiMode -ne 'SubGroups' -or $RightMultiMode -ne 'SubGroups') {
            $QueryTemp['Side'] = { _DataTable_ }.ToString()
        }

        $QueryTemp
    }
    else {
        # PSCustomObject
        $QueryTemp = @{
            Main        = {
                $RowMain = [ordered]@{ }
                _SidesScript_
                [PSCustomObject]$RowMain
            }.ToString()

            SideReplace = '_Row_', 'RowMain'
        }

        if ($LeftMultiMode -eq 'SubGroups' -or $RightMultiMode -eq 'SubGroups') {
            $QueryTemp['SideSubGroupBase'] = $QueryParts['PSCustomObjectFromSubGroup']
            $QueryTemp['SideSubGroup'] = $QueryParts['SubGroupFromPSCustomObject']
        }

        if ($LeftMultiMode -ne 'SubGroups' -or $RightMultiMode -ne 'SubGroups') {
            $QueryTemp['Side'] = { _PSCustomObject_ }.ToString()
        }

        $QueryTemp
    }

    $Query['Base'] = {
        param (
            #_$Key_,
            $LeftLine,
            $RightLine
        )
    }.ToString()

    #endregion Main: Set $Query

    #region Main: Assemble $Query

    function Invoke-AssembledQuery {
        param (
            $MultiMode,
            $Side,
            $Object
        )

        if ($MultiMode -eq 'SingleOnly') {
            $Query[$Side + 'Enumerable'] = { $_Side_Line = [System.Linq.Enumerable]::SingleOrDefault($_Side_Line) }
        }
        elseif ($MultiMode -eq 'DuplicateLines') {
            $Query[$Side + 'Enumerable'] = { $_Side_Lines = [System.Linq.Enumerable]::DefaultIfEmpty($_Side_Line) }

            $Query['Main'] = {
                foreach ($_Side_Line in $_Side_Lines) {
                    _MainScript_
                }
            }.ToString().Replace('_Side_', $Side).Replace('_MainScript_', $Query['Main'])
        }

        if ($MultiMode -eq 'SubGroups') {
            $Query[$Side + 'Enumerable'] = { $_Side_Lines = if ($_Side_Line.Count) { $_Side_Line } }

            $Query[$Side] = if ($Object -is [Data.DataTable]) {
                ($Query['SideSubGroupBase'] -Replace $Query['SideReplace']).Replace('_SubGroup_', $Query['SideSubGroup']).Replace('_DataTable_', $QueryParts['DataTableFromDataTable']).Replace('_PSCustomObject_', $QueryParts['PSCustomObjectFromAny'])
            }
            else {
                ($Query['SideSubGroupBase'] -Replace $Query['SideReplace']).Replace('_SubGroup_', $Query['SideSubGroup']).Replace('_DataTable_', $QueryParts['DataTableFromAny']).Replace('_PSCustomObject_', $QueryParts['PSCustomObjectFromPSCustomObject'])
            }

            $Query[$Side] = $Query[$Side].Replace('_Row_', 'RowSubGroup')
        }
        else {
            if ($Query[$Side] -or $Query['Side']) {
                if ($null -eq $Query[$Side]) {
                    $Query[$Side] = $Query['Side']
                }

                if ($null -ne $MultiMode -and $Query[$Side] -like '*_DataTable_*') {
                    $Query[$Side] = $QueryParts['IfSideLine'].Replace('_SideScript_', $Query[$Side])
                }

                $Query[$Side] = if ($Object -is [Data.DataTable]) {
                    $Query[$Side].Replace('_DataTable_', $QueryParts['DataTableFromDataTable']).Replace('_PSCustomObject_', $QueryParts['PSCustomObjectFromAny'])
                }
                else {
                    $Query[$Side].Replace('_DataTable_', $QueryParts['DataTableFromAny']).Replace('_PSCustomObject_', $QueryParts['PSCustomObjectFromPSCustomObject'])
                }

                $Query[$Side] = $Query[$Side] -Replace $Query['SideReplace']
            }
        }

        if ($Query[$Side]) {
            $Query[$Side] = $Query[$Side].Replace('_Side_', $Side)
        }

        if ($Query[$Side + 'Enumerable']) {
            $Query[$Side + 'Enumerable'] = $Query[$Side + 'Enumerable'].ToString().Replace('_Side_', $Side)
        }
    }

    Invoke-AssembledQuery -MultiMode $LeftMultiMode -Side 'Left' -Object $Left
    Invoke-AssembledQuery -MultiMode $RightMultiMode -Side 'Right' -Object $Right

    if ($AddKey) {
        $KeyScript = {
            $RowMain._Key_ = $Key

            _SidesScript_
        }.ToString()

        $KeyScript = if ($DataTable) {
            $KeyScript.Replace('._Key_', '[$AddKey]')
        }
        else {
            $KeyScript.Replace('_Key_', '$AddKey')
        }

        $Query['Main'] = $Query['Main'].Replace('_SidesScript_', $KeyScript)
    }

    $Query['Main'] = $Query['Main'].Replace('_SidesScript_', $Query['Left'] + $Query['Right'])

    if ($Type -eq 'OnlyIfInBoth') {
        [System.Func[System.Object, System.Object, System.Object]]$Query = [Scriptblock]::Create($Query['Base'] + $Query['Main'])
    }
    elseif ($Type -eq 'AllInLeft') {
        [System.Func[System.Object, [Collections.Generic.IEnumerable[System.Object]], System.Object]]$Query = [Scriptblock]::Create($Query['Base'] + $Query['RightEnumerable'] + $Query['Main'])
    }
    # elseif ($Type -eq 'AllInBoth') {
    #     [System.Func[System.Object, [Collections.Generic.IEnumerable[System.Object]], [Collections.Generic.IEnumerable[System.Object]], System.Object]]$Query = [Scriptblock]::Create($Query['Base'].Replace('#_$Key_', '$Key') + $Query['LeftEnumerable'] + "`n" + $Query['RightEnumerable'] + $Query['Main'])
    # }

    #endregion Main: Assemble $Query

    #endregion Main

    #region Execute

    if ($Left -is [Data.DataTable]) {
        $LeftNew = [DataTableExtensions]::AsEnumerable($Left)
    }
    elseif ($Left -is [PSCustomObject] -or $Left -is [Collections.ArrayList]) {
        $LeftNew = @( $Left )
    }
    else {
        $LeftNew = $Left
    }

    if ($Right -is [Data.DataTable]) {
        $RightNew = [DataTableExtensions]::AsEnumerable($Right)
    }
    elseif ($Right -is [PSCustomObject] -or $Right -is [Collections.ArrayList]) {
        $RightNew = @( $Right )
    }
    else {
        $RightNew = $Right
    }

    try {
        $Result = if ($Type -eq 'OnlyIfInBoth') {
            [System.Linq.Enumerable]::ToArray(
                [System.Linq.Enumerable]::Join($LeftNew, $RightNew, $LeftJoinScript, $RightJoinScript, $query, $Comparer)
            )
        }
        # elseif ($Type -eq 'AllInBoth') {
        #     [System.Linq.Enumerable]::ToArray(
        #         [MoreLinq.MoreEnumerable]::FullGroupJoin($LeftNew, $RightNew, $LeftJoinScript, $RightJoinScript, $query, $Comparer)
        #     )
        # }
        else {
            [System.Linq.Enumerable]::ToArray(
                [System.Linq.Enumerable]::GroupJoin($LeftNew, $RightNew, $LeftJoinScript, $RightJoinScript, $query, $Comparer)
            )
        }

        if ($PassThru) {
            , $Left
        }
        elseif ($DataTable) {
            , $OutDataTable
        }
        elseif ($LeftMultiMode -eq 'DuplicateLines' -or $RightMultiMode -eq 'DuplicateLines') {
            $Result.ForEach({ $_ })
        }
        else {
            $Result
        }
    }
    catch {
        $PSCmdlet.ThrowTerminatingError($_)
    }

    #endregion Execute
}

function ConvertTo-MaskLength {
    <#
        .SYNOPSIS
            Convert from a netmask to the masklength

        .EXAMPLE
            PS> ConvertTo-MaskLength -NetMask 255.255.255.0
    #>

    param (
        # The subnet mask to convert.
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$Netmask
    )

    if ([uri]::CheckHostName($Netmask) -ne 'IPv4') {
        return $null
    }

    $maskLength = 0

    foreach ($octet in $Netmask.Split('.')) {
        0..7 | ForEach-Object {
            if (($octet - [System.Math]::Pow(2, (7 - $_))) -ge 0) {
                $octet = $octet - [System.Math]::Pow(2, (7 - $_))
                $maskLength++
            }
        }
    }

    return $maskLength
}

function ConvertTo-SortableVersionString {
    param (
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$VersionString
    )

    if (-not [string]::IsNullOrWhiteSpace($VersionString)) {
        (
            ($VersionString -replace '\s\(7-Mode\)' -replace '[.]*P', '.0.9' -replace '[.]*RC', '.0.0.7' -replace '[.]*D', '.0.0.0.5' -replace '[.]*X', '.0.0.0.0.3' -replace '[.]*L', '.0.0.0.0.0.2' -replace '[.]*[a-z]+', '.0.0.0.0.0.0.1') -split '[.]' | ForEach-Object {
                '{0:D5}' -f [int]$_
            }
        ) -join '.'
    }
}

# SIG # Begin signature block
# MIIq0AYJKoZIhvcNAQcCoIIqwTCCKr0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBnx0GCk+YK6CJO
# o0KpLSHPJ51dAxvXndcgr+xgOuXH9qCCJGgwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggWNMIIEdaADAgECAhAOmxiO
# +dAt5+/bUOIIQBhaMA0GCSqGSIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYD
# VQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAi
# BgNVBAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAw
# MDBaFw0zMTExMDkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
# Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERp
# Z2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBAL/mkHNo3rvkXUo8MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsb
# hA3EMB/zG6Q4FutWxpdtHauyefLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iT
# cMKyunWZanMylNEQRBAu34LzB4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGb
# NOsFxl7sWxq868nPzaw0QF+xembud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclP
# XuU15zHL2pNe3I6PgNq2kZhAkHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCr
# VYJBMtfbBHMqbpEBfCFM1LyuGwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFP
# ObURWBf3JFxGj2T3wWmIdph2PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTv
# kpI6nj3cAORFJYm2mkQZK37AlLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWM
# cCxBYKqxYxhElRp2Yn72gLD76GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls
# 5Q5SUUd0viastkF13nqsX40/ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBR
# a2+xq4aLT8LWRV+dIPyhHsXAj6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6
# MIIBNjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qY
# rhwPTzAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8E
# BAMCAYYweQYIKwYBBQUHAQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5k
# aWdpY2VydC5jb20wQwYIKwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDig
# NoY0aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNybDARBgNVHSAECjAIMAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCg
# v0NcVec4X6CjdBs9thbX979XB72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQT
# SnovLbc47/T/gLn4offyct4kvFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh
# 65ZyoUi0mcudT6cGAxN3J0TU53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSw
# uKFWjuyk1T3osdz9HNj0d1pcVIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAO
# QGPFmCLBsln1VWvPJ6tsds5vIy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjD
# TZ9ztwGpn1eqXijiuZQwggXMMIIENKADAgECAhAg429sPxgagUb53pPffJfkMA0G
# CSqGSIb3DQEBDAUAMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBS
# MzYwHhcNMjEwOTA5MDAwMDAwWhcNMjMwOTA5MjM1OTU5WjBQMQswCQYDVQQGEwJV
# UzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UECgwMTmV0QXBwLCBJbmMuMRUw
# EwYDVQQDDAxOZXRBcHAsIEluYy4wggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGK
# AoIBgQC4kYYj/oViZD9pN03hrqFjtwOz1Gx4eDWVV8IYUYEr2qpLeNYvWz26B/+E
# mYLboAGVpkYg0Wske3hjRpooZlMORm1S4+2C2NoQCstZ+kmNTevbXHJe/w1VNJrm
# fKjpXOJEfx8GUGDqCMR30dChVpsdBEavrM7T0hnfJLv18i19SGv3a+nhvB3iOjLP
# SEg8+DFkjD5jTI6fQ83DBEiVR4UEYq7yyH578ZEylhsRfJmK+uIREycEUk/NpFTw
# g6/7lxh/WYabCrZzqI4Ep3QataRKaAlxt3BiOlZkX4WGX3DYvSYltWOM2UlCcGpP
# m/1/LN8Xuvf+YO6H472QecHe59XtXbHSMKvqDuOEJu8Wk2LFMNK732o3Fc5QIHie
# 6JtNsgNYf+Vsm5EMBD1ZHNE+C3zlmQbbWIU5uEU1nhiodBPKsu0SkS6pARyKBO05
# DSc2gWWv0aukKi04ZAn+hAdSl5s1dgmU5shBvKHqe15K9ZRN4FFO/CgHE0BoGYQS
# UQVKwa0CAwEAAaOCAZwwggGYMB8GA1UdIwQYMBaAFA8qyyCHKLjsb0iuK1SmKaoX
# pM0MMB0GA1UdDgQWBBQuH643KcBMmb/Q6IZt+H9IrnXFwDAOBgNVHQ8BAf8EBAMC
# B4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDAzARBglghkgBhvhC
# AQEEBAMCBBAwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwIwJTAjBggrBgEFBQcC
# ARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQBMEkGA1UdHwRCMEAw
# PqA8oDqGOGh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVT
# aWduaW5nQ0FSMzYuY3JsMHkGCCsGAQUFBwEBBG0wazBEBggrBgEFBQcwAoY4aHR0
# cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQVIz
# Ni5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqG
# SIb3DQEBDAUAA4IBgQCOoGdXjP8Sif0h3ZvoDsIVfrJvQUdP9pZExRJGGj/Te6ML
# XyqHNc/G2WklNv+BC4ryNJ4zp0oneHOub55ejrYPfhgP9oFWS8oOPJbrp3rAtINa
# OeKRC88WUYZSKYAxSX6kubEjnQD6cmazXjxVN6P8+6q9smeqF3xI+++c+ekMw3Xv
# 4EWRYXqN0/srfFp1WpoluwSQcZMqj9haSX0bgO6AAWH2AnVJMfXTIKz/0FQ/RW0y
# Ed5QYQqYInhA7IUz9dBziNpJlwONeF5j7xzgfYDY63WU6WrgJnGjYkQCOOsUx74j
# gUiMRe9zV48GS8Vxz22c/TQs0x4/1GmOSJvUHEk3GseBmB3v+yEhh/D6zWOfYP4X
# D/9b91CxmugTuepuaJSeRg+qUm3KarKsOCUF/CLqUrNh/JwKrWD1cghRaYLvMucs
# ScksHHe7ZDvb2OtvxWXjPk1d1NKvEwFJSS6hIVIbug9x28AJqOoP5Pn9wZvJSzvW
# uQJGCLscdGyKefdCo30wggYaMIIEAqADAgECAhBiHW0MUgGeO5B5FSCJIRwKMA0G
# CSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290
# IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFQxCzAJBgNVBAYT
# AkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28g
# UHVibGljIENvZGUgU2lnbmluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEBAQUAA4IB
# jwAwggGKAoIBgQCbK51T+jU/jmAGQ2rAz/V/9shTUxjIztNsfvxYB5UXeWUzCxEe
# AEZGbEN4QMgCsJLZUKhWThj/yPqy0iSZhXkZ6Pg2A2NVDgFigOMYzB2OKhdqfWGV
# oYW3haT29PSTahYkwmMv0b/83nbeECbiMXhSOtbam+/36F09fy1tsB8je/RV0mIk
# 8XL/tfCK6cPuYHE215wzrK0h1SWHTxPbPuYkRdkP05ZwmRmTnAO5/arnY83jeNzh
# P06ShdnRqtZlV59+8yv+KIhE5ILMqgOZYAENHNX9SJDm+qxp4VqpB3MV/h53yl41
# aHU5pledi9lCBbH9JeIkNFICiVHNkRmq4TpxtwfvjsUedyz8rNyfQJy/aOs5b4s+
# ac7IH60B+Ja7TVM+EKv1WuTGwcLmoU3FpOFMbmPj8pz44MPZ1f9+YEQIQty/NQd/
# 2yGgW+ufflcZ/ZE9o1M7a5Jnqf2i2/uMSWymR8r2oQBMdlyh2n5HirY4jKnFH/9g
# Rvd+QOfdRrJZb1sCAwEAAaOCAWQwggFgMB8GA1UdIwQYMBaAFDLrkpr/NZZILyhA
# QnAgNpFcF4XmMB0GA1UdDgQWBBQPKssghyi47G9IritUpimqF6TNDDAOBgNVHQ8B
# Af8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcD
# AzAbBgNVHSAEFDASMAYGBFUdIAAwCAYGZ4EMAQQBMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Um9vdFI0Ni5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ1Jvb3RSNDYu
# cDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQwFAAOCAgEABv+C4XdjNm57oRUgmxP/BP6YdURhw1aVcdGRP4Wh60BAscjW
# 4HL9hcpkOTz5jUug2oeunbYAowbFC2AKK+cMcXIBD0ZdOaWTsyNyBBsMLHqafvIh
# rCymlaS98+QpoBCyKppP0OcxYEdU0hpsaqBBIZOtBajjcw5+w/KeFvPYfLF/ldYp
# mlG+vd0xqlqd099iChnyIMvY5HexjO2AmtsbpVn0OhNcWbWDRF/3sBp6fWXhz7Dc
# ML4iTAWS+MVXeNLj1lJziVKEoroGs9Mlizg0bUMbOalOhOfCipnx8CaLZeVme5yE
# Lg09Jlo8BMe80jO37PU8ejfkP9/uPak7VLwELKxAMcJszkyeiaerlphwoKx1uHRz
# NyE6bxuSKcutisqmKL5OTunAvtONEoteSiabkPVSZ2z76mKnzAfZxCl/3dq3dUNw
# 4rg3sTCggkHSRqTqlLMS7gjrhTqBmzu1L90Y1KWN/Y5JKdGvspbOrTfOXyXvmPL6
# E52z1NZJ6ctuMFBQZH3pwWvqURR8AgQdULUvrxjUYbHHj95Ejza63zdrEcxWLDX6
# xWls/GDnVNueKjWUH3fTv1Y8Wdho698YADR7TNx8X8z2Bev6SivBBOHY+uqiirZt
# g0y9ShQoPzmCcn63Syatatvx157YK9hlcPmVoa1oDE5/L9Uo2bC5a4CH2Rwwggau
# MIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJ
# BgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5k
# aWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAe
# Fw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcw
# FQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3Rl
# ZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9Ge
# TKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0
# hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZl
# jZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAsh
# aG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVY
# TXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1
# biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCir
# c0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+
# DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA
# +bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42Pg
# puE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzS
# M7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU
# uhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6
# mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsG
# AQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29t
# MEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNl
# cnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3Js
# My5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAE
# GTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1Z
# jsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8d
# B+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVp
# P0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp8
# 76i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2
# nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3
# ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQ
# txMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc
# 4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+Y
# AN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZ
# vAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQr
# H4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIGwDCCBKigAwIBAgIQDE1p
# ckuU+jwqSj0pB4A9WjANBgkqhkiG9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4XDTIyMDkyMTAwMDAw
# MFoXDTMzMTEyMTIzNTk1OVowRjELMAkGA1UEBhMCVVMxETAPBgNVBAoTCERpZ2lD
# ZXJ0MSQwIgYDVQQDExtEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMiAtIDIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDP7KUmOsap8mu7jcENmtuh6BSFdDMa
# JqzQHFUeHjZtvJJVDGH0nQl3PRWWCC9rZKT9BoMW15GSOBwxApb7crGXOlWvM+xh
# iummKNuQY1y9iVPgOi2Mh0KuJqTku3h4uXoW4VbGwLpkU7sqFudQSLuIaQyIxvG+
# 4C99O7HKU41Agx7ny3JJKB5MgB6FVueF7fJhvKo6B332q27lZt3iXPUv7Y3UTZWE
# aOOAy2p50dIQkUYp6z4m8rSMzUy5Zsi7qlA4DeWMlF0ZWr/1e0BubxaompyVR4aF
# eT4MXmaMGgokvpyq0py2909ueMQoP6McD1AGN7oI2TWmtR7aeFgdOej4TJEQln5N
# 4d3CraV++C0bH+wrRhijGfY59/XBT3EuiQMRoku7mL/6T+R7Nu8GRORV/zbq5Xwx
# 5/PCUsTmFntafqUlc9vAapkhLWPlWfVNL5AfJ7fSqxTlOGaHUQhr+1NDOdBk+lbP
# 4PQK5hRtZHi7mP2Uw3Mh8y/CLiDXgazT8QfU4b3ZXUtuMZQpi+ZBpGWUwFjl5S4p
# kKa3YWT62SBsGFFguqaBDwklU/G/O+mrBw5qBzliGcnWhX8T2Y15z2LF7OF7ucxn
# EweawXjtxojIsG4yeccLWYONxu71LHx7jstkifGxxLjnU15fVdJ9GSlZA076XepF
# cxyEftfO4tQ6dwIDAQABo4IBizCCAYcwDgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB
# /wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwIAYDVR0gBBkwFzAIBgZngQwB
# BAIwCwYJYIZIAYb9bAcBMB8GA1UdIwQYMBaAFLoW2W1NhS9zKXaaL3WMaiCPnshv
# MB0GA1UdDgQWBBRiit7QYfyPMRTtlwvNPSqUFN9SnDBaBgNVHR8EUzBRME+gTaBL
# hklodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0
# MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3JsMIGQBggrBgEFBQcBAQSBgzCBgDAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFgGCCsGAQUFBzAC
# hkxodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRS
# U0E0MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3J0MA0GCSqGSIb3DQEBCwUAA4IC
# AQBVqioa80bzeFc3MPx140/WhSPx/PmVOZsl5vdyipjDd9Rk/BX7NsJJUSx4iGNV
# CUY5APxp1MqbKfujP8DJAJsTHbCYidx48s18hc1Tna9i4mFmoxQqRYdKmEIrUPwb
# tZ4IMAn65C3XCYl5+QnmiM59G7hqopvBU2AJ6KO4ndetHxy47JhB8PYOgPvk/9+d
# EKfrALpfSo8aOlK06r8JSRU1NlmaD1TSsht/fl4JrXZUinRtytIFZyt26/+YsiaV
# OBmIRBTlClmia+ciPkQh0j8cwJvtfEiy2JIMkU88ZpSvXQJT657inuTTH4YBZJwA
# wuladHUNPeF5iL8cAZfJGSOA1zZaX5YWsWMMxkZAO85dNdRZPkOaGK7DycvD+5sT
# X2q1x+DzBcNZ3ydiK95ByVO5/zQQZ/YmMph7/lxClIGUgp2sCovGSxVK05iQRWAz
# gOAj3vgDpPZFR+XOuANCR+hBNnF3rf2i6Jd0Ti7aHh2MWsgemtXC8MYiqE+bvdgc
# mlHEL5r2X6cnl7qWLoVXwGDneFZ/au/ClZpLEQLIgpzJGgV8unG1TnqZbPTontRa
# mMifv427GFxD9dAq6OJi7ngE273R+1sKqHB+8JeEeOMIA11HLGOoJTiXAdI/Otrl
# 5fbmm9x+LMz/F0xNAKLY1gEOuIvu5uByVYksJxlh9ncBjDGCBb4wggW6AgEBMGgw
# VDELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDErMCkGA1UE
# AxMiU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIFIzNgIQIONvbD8YGoFG
# +d6T33yX5DANBglghkgBZQMEAgEFAKCBhDAYBgorBgEEAYI3AgEMMQowCKACgACh
# AoAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAM
# BgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCB7WKnuuCyo2hITqRo5zqSsO4Q6
# fiOL7gH3uSCFZn/SUDANBgkqhkiG9w0BAQEFAASCAYAPj38/oN3LeDBRuIgWWkWm
# fHCgyTt5HjQtROZ0Tiyd8KUsw5v6MiUjAm9Ppt2Tm4j3ei27Kw3BTE1z0gdou0rb
# yofzQpVuvYQZSLhdMia4kTwtBTjMn67JF+ESzB8NVzjn/JR+AlXgIGa/VhVl4vbT
# C8gvpkBmjX/Ovj6Rlc3gy37tON3SaMGia3ZKxv3j8CxVhbxKY4LpK7cducaTcsQ0
# 72MX5OpBjdqq0bZf/LGVz46Fr2UbndRIstlyXVzCbaGxiG/p0nRza+460BBlpoUC
# 5xfwv/5hxEULBe6mT6RK3WmJZxbxc1XywX4ZxsGobOIV9+oZeugBZTdjOFRmRMz/
# CMULhia4jI144YKDjQ6rIhTgTRupddJASInC+lzW/rRmvqGzLuu3IQqe2b834gR6
# 8tOLkxeMl2sXQwj5Kq6/QckcnnFP55SXlth9BgbeRUVa4Pup6WChvStU9orqu2Q5
# JJMinhZbWW9UD9bOM+M4Ej68luKeuWyuJIcGdpUOLcChggMgMIIDHAYJKoZIhvcN
# AQkGMYIDDTCCAwkCAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
# cnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
# SEEyNTYgVGltZVN0YW1waW5nIENBAhAMTWlyS5T6PCpKPSkHgD1aMA0GCWCGSAFl
# AwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUx
# DxcNMjMwNDAyMTUyODI1WjAvBgkqhkiG9w0BCQQxIgQgPsci6AWd4kjRGHuZf5Cx
# Ue03ILiub8GEAfZkMsnhzDUwDQYJKoZIhvcNAQEBBQAEggIAT3yGgzhTE4ElgEtd
# pgWWKM8I6wHcMC6ZF/NB5Flp5/mA81WjssJTmLbwAcFOeWZVAlPdeOq8xEKTGwqi
# BssG0mC9urEvP9su9Kwg8ryPUxFsIAwLk8TJvp+Hc2jWBaXWryCui9InpFwEaLas
# I1iFNq+uzF8w+xOYP8U46fSH98lw37iXktIqz9kGH0JVbCbrMFLqaUZ8oZBiqcx/
# IcAshHuuSNx/RJrYk0A2/Ydx7ja98W71RQHfNeBiRrLBYTsPEKDMHaFQcPKTUc2d
# npm14JNLJuDR9JxtSB7/45b48XpHoXA2FvQjf3ZON+f5sRaSyqL8KhbTyvTu9ic2
# +uJk6jkbJGLDoYI2/LKlmXfi82BE1ukkoUij8q95+KJk0hkQwVJhkgCX4zOYR+Un
# nJdDKWHRh5fUV2P4Ebv0uxGuZz2NiQZel4KQLIG/lbcp+RSr86M+FpJIZPpkeSx0
# MgVELgLYSF/67brDkezqECqSmno6gw5jUUaPMJD+iZxj2sOK+FTfH+BSp99mhlR8
# keu/ntjJsYgY3xDPy0TbDt6wcjmc4vF6qh6p3Xt9BVngib0oGjog+wWMnBcClork
# xb0/P5VhLbKBn+UuZ/c+VEZl/eM8UiuQpISlGP3YpmNsvSlNEZWdjCaYL5v0i3Xw
# BA38b6ROqyy3W+hy8yqxbgsb5b4=
# SIG # End signature block
