function Invoke-SanitizeDataSet {
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseLiteralInitializerForHashtable', '')]
    [CmdletBinding()]
    param (
        [Parameter(Mandatory)]
        [System.Data.DataSet]$DataSet,

        [Parameter(Mandatory)]
        [int]$SanitizeLevel,

        [Parameter(Mandatory)]
        [xml]$SanitizeMappings
    )

    Write-Verbose -Message 'Sanitizing DataSet'

    $functionStartTime = [datetime]::Now

    for ($t = 0; $t -le $DataSet.Tables.Count; $t++) {
        if ($DataSet.Tables[$t].Rows.Count -eq 0) {
            continue
        }

        $mode = $(
            switch -Wildcard ($DataSet.ExtendedProperties['BaseTypeName']) {
                'NetAppDocs.NcController*' {
                    'DataONTAP'

                    break
                }
                'NetAppDocs.NaController*' {
                    '7-Mode'

                    break
                }
                'NetAppDocs.NcVserver*' {
                    'DataONTAP'

                    break
                }
                'NetAppDocs.CloudInsights*' {
                    'DataONTAP'

                    break
                }
                'NetAppDocs.NeStorageArray*' {
                    'SANtricity'

                    break
                }
                'NetAppDocs.SnapDrive*' {
                    'SnapDrive'

                    break
                }
            }
        )

        $dataColumnsToSanitize = $DataSet.Tables[$t].Columns | Where-Object {
            ($_.ExtendedProperties.ContainsKey('SanitizeLevel') -and -not [string]::IsNullOrWhiteSpace($_.ExtendedProperties['SanitizeLevel'])) -and (
                ([int]::TryParse($_.ExtendedProperties['SanitizeLevel'], [ref]$null) -and $_.ExtendedProperties['SanitizeLevel'] -band $SanitizeLevel) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'IpOrFqdnOrHostname' -and $SanitizeLevel -band 2057) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'SplitString' -and $SanitizeLevel -band ([regex]::Matches($_.ExtendedProperties['SanitizeFormatString'], '\(((?:\d+[,]*)+)\)') | ForEach-Object { $_.Groups[1].Value -split ',' } | Measure-Object -Sum).Sum) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'Email' -and $SanitizeLevel -band 8) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'DataOntapPath' -and $SanitizeLevel -band 384) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq '7ModePath' -and $SanitizeLevel -band 384) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'UserName' -and $SanitizeLevel -band 16393) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'DataOntapSmPath' -and $SanitizeLevel -band 32913) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq '7ModeSmLocation' -and $SanitizeLevel -band 397) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'SubnetName' -and $SanitizeLevel -band 1) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'RoutingGroupName' -and $SanitizeLevel -band 1) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'LunPath' -and $SanitizeLevel -band 896) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'IpRange' -and $SanitizeLevel -band 1) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'ClientMatch' -and $SanitizeLevel -band 2057) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'QuotaTarget' -and $SanitizeLevel -band 16768) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'StorageQosWorkloadName' -and $SanitizeLevel -band 912) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'EnvironmentVariableValue' -and $SanitizeLevel -band 32901) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'DnsZone' -and $SanitizeLevel -band 8) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq '7ModeOptions' -and $SanitizeLevel -band 2057) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq '7ModeSmMultiPath' -and $SanitizeLevel -band 384) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq '7ModeDpSystem' -and $SanitizeLevel -band 2057) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'Traphost' -and $SanitizeLevel -band 2057) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq '7ModeNfsExportRule' -and $SanitizeLevel -band 2057) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq '7ModeVscanScanner' -and $SanitizeLevel -band 2057) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq '7ModeSecureAdminAccessHost' -and $SanitizeLevel -band 2057) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'DebugVreportObject' -and $SanitizeLevel -band 208) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq '7ModeNfsExportString' -and $SanitizeLevel -band 2441) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq '7ModeAdDomainUser' -and $SanitizeLevel -band 16404) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'EventNotificationDestination' -and $SanitizeLevel -band 2057) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'InitiatorName' -and $SanitizeLevel -band 2056) -or
                ($_.ExtendedProperties['SanitizeLevel'] -eq 'SecuritySessionLimit' -and $SanitizeLevel -band 17)
            )
        }

        if (($dataColumnsToSanitize | Measure-Object).Count -gt 0) {
            Write-Verbose -Message "Sanitizing table: $( $DataSet.Tables[$t].TableName )"

            foreach ($dataRow in $DataSet.Tables[$t].Rows) {
                foreach ($dataColumn in $dataColumnsToSanitize) {
                    $secureString = $dataRow.Item($dataColumn) -replace '^\s*|\s*$'

                    if ([string]::IsNullOrWhiteSpace($secureString) -or $secureString -eq 'UTD' -or $secureString -eq 'n/a' -or $secureString -eq '-' -or $secureString -eq '*' -or $secureString -eq 'all' -or $secureString -eq '<unknown>') {
                        continue
                    }

                    $prefix = $(
                        if ($dataColumn.ExtendedProperties.ContainsKey('SanitizePrefix')) {
                            $dataColumn.ExtendedProperties['SanitizePrefix']
                        }
                        else {
                            [string]::Empty
                        }
                    )

                    $delimiter = $(
                        if ($dataColumn.ExtendedProperties.ContainsKey('Delimiter')) {
                            $dataColumn.ExtendedProperties['Delimiter'] -replace 'Comma', ',' -replace 'Space', ' ' -replace 'NewLine', "`n" -replace 'Colon', ':' -replace 'AtSign', '@' -replace 'ForwardSlash', '\' -replace 'Dot', '.'
                        }
                        else {
                            [string]::Empty
                        }
                    )

                    switch -Exact ($dataColumn.ExtendedProperties['SanitizeLevel']) {
                        'EventNotificationDestination' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    if ($secureEntry.Contains('pii_mask')) {
                                        $secureEntry
                                    }
                                    else {
                                        if ($secureEntry.Contains('@')) {
                                            $user, $domain = $secureEntry -split '@', 2

                                            if (-not [string]::IsNullOrWhiteSpace($user) -and -not [string]::IsNullOrWhiteSpace($domain)) {
                                                [string]::Format('{0}@{1}', $user, (ConvertTo-SanitizedString -InputObject $domain -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                            }
                                            else {
                                                $secureEntry
                                            }
                                        }
                                        elseif ($secureEntry.StartsWith('http')) {
                                            $uri = [uri]''

                                            if ([uri]::TryCreate($secureEntry, [System.UriKind]::Absolute, [ref]$uri)) {
                                                switch -WildCard ([uri]::CheckHostName($uri.DnsSafeHost)) {
                                                    'IPv[4|6]' {
                                                        ConvertTo-SanitizedString -InputObject $uri.DnsSafeHost -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                                        break
                                                    }
                                                    'Dns' {
                                                        $hostName, $domainName = $uri.DnsSafeHost -split '[.]', 2

                                                        if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                                            [string]::Format('{0}//{1}.{2}{3}', $uri.Scheme, (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix $prefix -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), $uri.AbsolutePath)
                                                        }
                                                        else {
                                                            [string]::Format('{0}//{1}{2}', $uri.Scheme, (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix $prefix -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), $uri.AbsolutePath)
                                                        }

                                                        break
                                                    }
                                                    default {
                                                        ConvertTo-SanitizedString -InputObject $secureEntry -OtherServer -Prefix $prefix -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                    }
                                                }
                                            }
                                        }
                                        else {
                                            switch -WildCard ([uri]::CheckHostName($secureEntry)) {
                                                'IPv[4|6]' {
                                                    ConvertTo-SanitizedString -InputObject $secureEntry -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                                    break
                                                }
                                                'Dns' {
                                                    $hostName, $domainName = $secureEntry -split '[.]', 2

                                                    if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                                        [string]::Format('{0}.{1}', (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix $prefix -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                                    }
                                                    else {
                                                        ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix $prefix -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                    }

                                                    break
                                                }
                                                default {
                                                    ConvertTo-SanitizedString -InputObject $secureEntry -OtherServer -Prefix $prefix -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                }
                                            }
                                        }
                                    }
                                }
                            ) -join $delimiter
                        }
                        'IpOrFqdnOrHostname' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    switch -Wildcard ([uri]::CheckHostName($secureEntry)) {
                                        'IPv[4|6]' {
                                            ConvertTo-SanitizedString -InputObject $secureEntry -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                            break
                                        }
                                        'Dns' {
                                            $hostName, $domainName = $secureEntry -split '[.]', 2

                                            if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                                [string]::Format('{0}.{1}', (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix $prefix -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                            }
                                            else {
                                                ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix $prefix -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                            }

                                            break
                                        }
                                        default {
                                            ConvertTo-SanitizedString -InputObject $secureEntry -OtherServer -Prefix $prefix -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                        }
                                    }
                                }
                            ) -join $delimiter
                        }
                        'SplitString' {
                            $sanitizeElements = @( [regex]::Matches($dataColumn.ExtendedProperties['SanitizeFormatString'], '(?:\(?\{\d+\}\)?(?:\((?:\d+[,]*)+\))?)').Value )

                            $separatorString = $dataColumn.ExtendedProperties['SanitizeFormatString']
                            $sanitizeElements | ForEach-Object {
                                $separatorString = $separatorString.Replace($_, '~!~')
                            }

                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    $secureEntryStrings = $secureEntry.Split(($separatorString -split '~!~'), $sanitizeElements.Count, [System.StringSplitOptions]::None)

                                    if ($sanitizeElements.Count -ne $secureEntryStrings.Count) {
                                        $secureEntry
                                    }
                                    else {
                                        [string]::Format($( $dataColumn.ExtendedProperties['SanitizeFormatString'] -replace '\((?:\d+[,]*)+\)' ),
                                            $(
                                                :SanitizeElementLoop for ($i = 0; $i -lt $sanitizeElements.Count; $i++) {
                                                    $secureEntryString = $(
                                                        if ($sanitizeElements[$i] -like '(*)') {
                                                            $secureEntryStrings[$i].TrimStart('(').TrimEnd(')')
                                                        }
                                                        else {
                                                            $secureEntryStrings[$i]
                                                        }
                                                    )

                                                    if ([string]::IsNullOrWhiteSpace($secureEntryString)) {
                                                        $secureEntryString

                                                        continue
                                                    }

                                                    $sanitizedString = $(
                                                        if ($dataColumn.ExtendedProperties.ContainsKey('SanitizeExcludeRegex')) {
                                                            $secureEntryString -replace $dataColumn.ExtendedProperties['SanitizeExcludeRegex']
                                                        }
                                                        else {
                                                            $secureEntryString
                                                        }
                                                    )

                                                    $params = @{
                                                        InputObject      = [string]::Empty
                                                        SanitizeMappings = $SanitizeMappings
                                                        SanitizeLevel    = $SanitizeLevel
                                                    }

                                                    if ($dataColumn.ExtendedProperties.ContainsKey('SanitizeMethod')) {
                                                        $params.Add($dataColumn.ExtendedProperties['SanitizeMethod'], $true)

                                                        if ($params.StringMatch) {
                                                            $params.Add('ReturnMappingOnly', [switch]::Present)
                                                        }
                                                    }

                                                    if ($sanitizeElements[$i] -match '\(((?:\d+[,]*)+)\)') {
                                                        $sanitizeValues = $Matches[1] -split ','

                                                        if ($Matches) {
                                                            $Matches.Clear()
                                                        }

                                                        $mappingEntries = New-Object -TypeName System.Collections.Hashtable
                                                        $mappingEntry = $null

                                                        :SanitizeValueLoop foreach ($sanitizeValue in $sanitizeValues) {
                                                            switch -Exact ($sanitizeValue) {
                                                                '1' {
                                                                    if ($dataColumn.ExtendedProperties['SanitizeMethod'] -eq 'StringMatch') {
                                                                        if ($sanitizedString -match '(?:(?:1\d\d|2[0-5][0-5]|2[0-4]\d|0?[1-9]\d|0?0?\d)\.){3}(?:1\d\d|2[0-5][0-5]|2[0-4]\d|0?[1-9]\d|0?0?\d)') {
                                                                            $sanitizedString = $sanitizedString.Replace($Matches[0], (ConvertTo-SanitizedString -InputObject $Matches[0] -SanitizeMappings $SanitizeMappings -SanitizeLevel $SanitizeLevel -IpAddress))

                                                                            $Matches.Clear()
                                                                        }
                                                                    }
                                                                    else {
                                                                        $sanitizedString = ConvertTo-SanitizedString -InputObject $sanitizedString -SanitizeMappings $SanitizeMappings -SanitizeLevel $SanitizeLevel -IpAddress
                                                                    }

                                                                    if (-not [string]::IsNullOrWhiteSpace($sanitizedString) -and $sanitizedString -cne $secureEntryString) {
                                                                        $sanitizedString

                                                                        continue SanitizeElementLoop
                                                                    }
                                                                }
                                                                '2' {
                                                                    $sanitizedString = ConvertTo-SanitizedString -InputObject $sanitizedString -SanitizeMappings $SanitizeMappings -SanitizeLevel $SanitizeLevel -MacAddress

                                                                    if (-not [string]::IsNullOrWhiteSpace($sanitizedString) -and $sanitizedString -cne $secureEntryString) {
                                                                        $sanitizedString

                                                                        continue SanitizeElementLoop
                                                                    }
                                                                }
                                                                '32' {
                                                                    $sanitizedString = ConvertTo-SanitizedString -InputObject $sanitizedString -SanitizeMappings $SanitizeMappings -SanitizeLevel $SanitizeLevel -LicenseCode

                                                                    if (-not [string]::IsNullOrWhiteSpace($sanitizedString) -and $sanitizedString -cne $secureEntryString) {
                                                                        $sanitizedString

                                                                        continue SanitizeElementLoop
                                                                    }
                                                                }
                                                                default {
                                                                    $sanitizeSection = $SanitizeMappings.Sanitization.Sections.SelectSingleNode("./Section[SanitizeLevel=`"$sanitizeValue`" and (not(Mode) or Mode=`"$mode`")]")

                                                                    $params.Add($sanitizeSection.Name, [switch]::Present)
                                                                    $params.InputObject = $sanitizedString

                                                                    if ($params.StringMatch) {
                                                                        if (($mappingEntry = ConvertTo-SanitizedString @params)) {
                                                                            if ($mappingEntry -is [System.String]) {
                                                                                $sanitizedString = $mappingEntry
                                                                            }
                                                                            else {
                                                                                $mappingEntries.Add($mappingEntry.SecureString, $mappingEntry.SanitizedString)

                                                                                $sanitizedString = $sanitizedString -replace $mappingEntry.SecureString
                                                                            }
                                                                        }

                                                                        if ($sanitizedString -eq [string]::Empty) {
                                                                            break SanitizeValueLoop
                                                                        }
                                                                    }
                                                                    else {
                                                                        $sanitizedString = ConvertTo-SanitizedString @params

                                                                        if (-not [string]::IsNullOrWhiteSpace($sanitizedString) -and $sanitizedString -cne $secureEntryString) {
                                                                            $sanitizedString

                                                                            continue SanitizeElementLoop
                                                                        }
                                                                    }
                                                                }
                                                            }

                                                            if (-not [string]::IsNullOrWhiteSpace($sanitizeSection.Name) -and $params.ContainsKey($sanitizeSection.Name)) {
                                                                $params.Remove($sanitizeSection.Name)
                                                            }
                                                        }

                                                        if ($mappingEntries.Count -gt 0) {
                                                            [regex]::Replace($secureEntryString, '(' + $( $mappingEntries.Keys -join '|' ) + ')', { param ($m) return $mappingEntries[$m.Value] })
                                                        }
                                                        else {
                                                            $secureEntryString
                                                        }
                                                    }
                                                    else {
                                                        $secureEntryString
                                                    }
                                                }
                                            )
                                        )
                                    }
                                }
                            ) -join $delimiter
                        }
                        'InitiatorName' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    if ($secureEntry -match '^\S{2}([:]\S{2}){7}$' -or -not $secureEntry.StartsWith('iqn.')) {
                                        $secureEntry

                                        continue
                                    }

                                    $secureElements = $secureEntry -split ':'

                                    $secureElements[1] = $(
                                        if ([uri]::CheckHostName($secureElements[1] -eq 'Dns')) {
                                            $hostName, $domainName = $secureElements[1] -split '[.]', 2

                                            if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                                [string]::Format('{0}.{1}', (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix $prefix -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                            }
                                            else {
                                                ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix $prefix -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                            }
                                        }
                                    )

                                    $secureElements -join ':'
                                }
                            ) -join $delimiter
                        }
                        'Email' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ($secureEntry.Contains('pii_mask')) {
                                        $secureEntry
                                    }
                                    else {
                                        $user, $domain = $secureEntry -split '@', 2

                                        if (-not [string]::IsNullOrWhiteSpace($user) -and -not [string]::IsNullOrWhiteSpace($domain)) {
                                            [string]::Format('{0}@{1}', $user, (ConvertTo-SanitizedString -InputObject $domain -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                        }
                                        else {
                                            $secureEntry
                                        }
                                    }
                                }
                            ) -join $delimiter
                        }
                        '7ModePath' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    $sb = New-Object -TypeName System.Text.StringBuilder
                                    $pathElements = New-Object -TypeName System.Collections.Generic.List[System.String]

                                    #We will have to loop through each element with a string match (which would finally end in subfolder). This assumes all vols and qtrees have already been mapped
                                    $secureEntry -split '/' | Select-Object -Skip 1 | ForEach-Object { $pathElements.Add($_) }

                                    if ($pathElements.Count -eq 0) {
                                        [void]$sb.Append([string]::Format('{0}', (ConvertTo-SanitizedString -InputObject $secureEntry -VolumeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                    }
                                    else {
                                        if ($pathElements[0] -eq 'vol' -or $pathElements[0] -eq 'etc') {
                                            [void]$sb.Append([string]::Format('/{0}', $pathElements[0]))
                                            $pathElements.RemoveAt(0)
                                        }

                                        for ($i = 0; $i -lt $pathElements.Count; $i++) {
                                            switch -Exact ($i) {
                                                '0' {
                                                    [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -VolumeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                }
                                                '1' {
                                                    [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -QtreeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                }
                                                default {
                                                    [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -QtreeName -Prefix 'subfolder' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                }
                                            }
                                        }
                                    }

                                    $sb.ToString()
                                }
                            ) -join $delimiter
                        }
                        'DataOntapPath' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    $sb = New-Object -TypeName System.Text.StringBuilder
                                    $pathElements = New-Object -TypeName System.Collections.Generic.List[System.String]

                                    #We will have to loop through each element with a string match (which would finally end in subfolder). This assumes all vols and qtrees have already been mapped
                                    $secureEntry -split '/' | Select-Object -Skip 1 | ForEach-Object { $pathElements.Add($_) }

                                    if ($pathElements[0] -eq 'vol') {
                                        [void]$sb.Append([string]::Format('/{0}', $pathElements[0]))
                                        $pathElements.RemoveAt(0)
                                    }

                                    foreach ($pathElement in $pathElements) {
                                        $pathString = ConvertTo-SanitizedString -InputObject $pathElement -VolumeName -StringMatchExact -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                        if ($pathString -ceq $pathElement) {
                                            $pathString = ConvertTo-SanitizedString -InputObject $pathElement -QtreeName -StringMatchExact -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                            if ($pathString -ceq $pathElement) {
                                                $pathString = ConvertTo-SanitizedString -InputObject $pathElement -QtreeName -Prefix 'subfolder' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                            }
                                        }

                                        [void]$sb.Append([string]::Format('/{0}', $pathString))
                                    }

                                    $sb.ToString()
                                }
                            ) -join $delimiter
                        }
                        'UserName' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    if ($secureEntry -match '(.+)\\\\(.+)') {
                                        [string]::Format('{0}\\{1}', (ConvertTo-SanitizedString -InputObject $Matches[1] -ADNetBiosName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), $Matches[2])
                                    }
                                    elseif ($secureEntry -match '(.+)\\(.+)') {
                                        switch -Wildcard ([uri]::CheckHostName($Matches[1])) {
                                            'IPv[4|6]' {
                                                [string]::Format('{0}\{1}', (ConvertTo-SanitizedString -InputObject $Matches[1] -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), $Matches[2])

                                                break
                                            }
                                            default {
                                                [string]::Format('{0}\{1}', (ConvertTo-SanitizedString -InputObject $Matches[1] -ADNetBiosName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), $Matches[2])
                                            }
                                        }
                                    }
                                    elseif ($secureEntry -match '(.+)\@(.+)') {
                                        [string]::Format('{0}@{1}', $Matches[1], (ConvertTo-SanitizedString -InputObject $Matches[2] -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                    }
                                    else {
                                        $secureEntry
                                    }

                                    if ($Matches) {
                                        $Matches.Clear()
                                    }
                                }
                            ) -join $delimiter
                        }
                        'DataOntapSmPath' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    if ($secureEntry -match '(?<Cluster>.+)[:][/][/](?<Vserver>.+)[/](?<Volume>.+)') {
                                        [string]::Format('{0}://{1}/{2}', (ConvertTo-SanitizedString -InputObject $Matches.Cluster -ClusterName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $Matches.Vserver -VserverName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $Matches.Volume -VolumeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                    }
                                    elseif ($secureEntry -match '(?<Vserver>.+)[:](?<Volume>.+)') {
                                        switch -Wildcard ([uri]::CheckHostName($Matches.Vserver)) {
                                            'IPv[4|6]' {
                                                [string]::Format('{0}:{1}', (ConvertTo-SanitizedString -InputObject $Matches.Vserver -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $Matches.Volume -VolumeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))

                                                break
                                            }
                                            default {
                                                [string]::Format('{0}:{1}', (ConvertTo-SanitizedString -InputObject $Matches.Vserver -VserverName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $Matches.Volume -VolumeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                            }
                                        }
                                    }
                                    elseif ($secureEntry -match '(?<Vserver>.+)[:]') {
                                        [string]::Format('{0}:', (ConvertTo-SanitizedString -InputObject $Matches.Vserver -VserverName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                    }
                                }

                                if ($Matches) {
                                    $Matches.Clear()
                                }
                            ) -join $delimiter
                        }
                        '7ModeSmLocation' {
                            $dataRow.Item($dataColumn) = $(
                                if (-not [string]::IsNullOrWhiteSpace($secureString)) {
                                    if ($secureString -like '*[:]*') {
                                        $sb = New-Object -TypeName System.Text.StringBuilder

                                        $server, $path = $secureString -split ':', 2

                                        [void]$sb.Append(
                                            [string]::Format(
                                                '{0}:', $(
                                                    switch -Wildcard ([uri]::CheckHostName($server)) {
                                                        'IPv[4|6]' {
                                                            ConvertTo-SanitizedString -InputObject $server -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                                            break
                                                        }
                                                        'Dns' {
                                                            $hostName, $domainName = $server -split '[.]', 2

                                                            if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                                                [string]::Format('{0}.{1}', (ConvertTo-SanitizedString -InputObject $hostName -ControllerName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                                            }
                                                            else {
                                                                ConvertTo-SanitizedString -InputObject $hostName -ControllerName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                            }

                                                            break
                                                        }
                                                        default {
                                                            $server
                                                        }
                                                    }
                                                )
                                            )
                                        )

                                        if ($path -notlike '*/*') {
                                            [void]$sb.Append((ConvertTo-SanitizedString -InputObject $path -VolumeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                        }
                                        else {
                                            $pathElements = New-Object -TypeName System.Collections.Generic.List[System.String]

                                            $path -split '/' | Select-Object -Skip 1 | ForEach-Object { $pathElements.Add($_) }

                                            for ($i = 0; $i -lt $pathElements.Count; $i++) {
                                                switch -Exact ($i) {
                                                    '0' {
                                                        [void]$sb.Append([string]::Format('/{0}', ($pathElements[$i])))
                                                    }
                                                    '1' {
                                                        [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -VolumeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                    }
                                                    '2' {
                                                        [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -QtreeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                    }
                                                    default {
                                                        [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -QtreeName -Prefix 'subfolder' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                    }
                                                }
                                            }
                                        }
                                        $sb.ToString()
                                    }
                                    else {
                                        $secureString
                                    }
                                }
                            )
                        }
                        '7ModeDpSystem' {
                            $dataRow.Item($dataColumn) = $(
                                switch -Wildcard ([uri]::CheckHostName($secureString)) {
                                    'IPv[4|6]' {
                                        ConvertTo-SanitizedString -InputObject $secureString -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                        break
                                    }
                                    'Dns' {
                                        $hostName, $domainName = $secureString -split '[.]', 2

                                        if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                            [string]::Format('{0}.{1}', (ConvertTo-SanitizedString -InputObject $hostName -ControllerName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                        }
                                        else {
                                            ConvertTo-SanitizedString -InputObject $hostName -ControllerName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                        }

                                        break
                                    }
                                    default {
                                        ConvertTo-SanitizedString -InputObject $secureString -OtherServer -Prefix 'dpserver' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                    }
                                }
                            )
                        }
                        'SubnetName' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    if ($secureEntry -match '^(\w)(\D+)(.*)\/(\d+)$') {
                                        [string]::Format('{0}{1}{2}/{4}', $Matches[1], $Matches[2], (ConvertTo-SanitizedString -InputObject $Matches[3] -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), $Matches[4])

                                        if ($Matches) {
                                            $Matches.Clear()
                                        }
                                    }
                                    else {
                                        switch -Wildcard ([uri]::CheckHostName($secureEntry)) {
                                            'IPv[4|6]' {
                                                ConvertTo-SanitizedString -InputObject $secureEntry -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                                break
                                            }
                                            default {
                                                $secureEntry
                                            }
                                        }
                                    }
                                }
                            ) -join $delimiter
                        }
                        'RoutingGroupName' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    if ($secureEntry -match '^(\D+)(.*)\/(\d+)$') {
                                        [string]::Format('{0}{1}/{2}', $Matches[1], (ConvertTo-SanitizedString -InputObject $Matches[2] -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), $Matches[3])

                                        if ($Matches) {
                                            $Matches.Clear()
                                        }
                                    }
                                    else {
                                        $secureEntry
                                    }
                                }
                            ) -join $delimiter
                        }
                        'LunPath' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    $sb = New-Object -TypeName System.Text.StringBuilder
                                    $pathElements = New-Object -TypeName System.Collections.Generic.List[System.String]

                                    #This should be fine since path will always start with /vol (additional hook in here to process the lun name)
                                    $secureEntry -split '/' | Select-Object -Skip 1 | ForEach-Object { $pathElements.Add($_) }

                                    if ($pathElements[0] -eq 'vol') {
                                        [void]$sb.Append([string]::Format('/{0}', $pathElements[0]))
                                        $pathElements.RemoveAt(0)
                                    }

                                    for ($i = 0; $i -lt $pathElements.Count - 1; $i++) {
                                        switch -Exact ($i) {
                                            '0' {
                                                [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -VolumeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                            }
                                            '1' {
                                                [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -QtreeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                            }
                                            default {
                                                [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -QtreeName -Prefix 'subfolder' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                            }
                                        }
                                    }

                                    [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[-1] -LunName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))

                                    $sb.ToString()
                                }
                            ) -join $delimiter
                        }
                        'IpRange' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    $ip1, $ip2 = $secureEntry -split '[-]', 2

                                    if (-not [string]::IsNullOrWhiteSpace($ip2)) {
                                        [string]::Format('{0}-{1}', (ConvertTo-SanitizedString -InputObject $ip1 -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $ip2 -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                    }
                                    else {
                                        ConvertTo-SanitizedString -InputObject $ip1 -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                    }
                                }
                            ) -join $delimiter
                        }
                        'ClientMatch' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ($secureEntry.StartsWith('@')) {
                                        $secureEntry
                                    }
                                    elseif ($secureEntry.Contains('/')) {
                                        $address, $netmaskLength = $secureEntry -split '/'

                                        $ip = [ipaddress]::Any

                                        if ([ipaddress]::TryParse($address, [ref]$ip)) {
                                            if ($ip.AddressFamily -eq 'InterNetworkV6') {
                                                ConvertTo-SanitizedString -InputObject $ip.IPAddressToString -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                            }
                                            else {
                                                [string]::Format('{0}/{1}', (ConvertTo-SanitizedString -InputObject $ip.IPAddressToString -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), $netmaskLength)
                                            }
                                        }
                                    }
                                    else {
                                        switch -Wildcard ([uri]::CheckHostName($secureEntry)) {
                                            'IPv[4|6]' {
                                                ConvertTo-SanitizedString -InputObject $secureEntry -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                                break
                                            }
                                            'Dns' {
                                                $hostName, $domainName = $secureEntry -split '[.]', 2

                                                if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                                    [string]::Format('{0}.{1}', (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix $prefix -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                                }
                                                else {
                                                    ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix $prefix -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                }

                                                break
                                            }
                                            default {
                                                ConvertTo-SanitizedString -InputObject $secureEntry -OtherServer -Prefix $prefix -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                            }
                                        }
                                    }
                                }
                            ) -join $delimiter
                        }
                        'QuotaTarget' {
                            $dataRow.Item($dataColumn) = $(
                                if ($dataRow.Item('QuotaType') -eq 'tree') {
                                    if ($secureString -like '*[/]*') {
                                        $sb = New-Object -TypeName System.Text.StringBuilder
                                        $pathElements = New-Object -TypeName System.Collections.Generic.List[System.String]

                                        $secureString -split '/' | Select-Object -Skip 1 | ForEach-Object { $pathElements.Add($_) }

                                        if ($pathElements[0] -eq 'vol') {
                                            [void]$sb.Append([string]::Format('/{0}', $pathElements[0]))
                                            $pathElements.RemoveAt(0)
                                        }

                                        for ($i = 0; $i -lt $pathElements.Count; $i++) {
                                            switch -Exact ($i) {
                                                '0' {
                                                    [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -VolumeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                }
                                                '1' {
                                                    [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -QtreeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                }
                                                default {
                                                    #This one probably not needed since quotatarget can only be /vol/volname/qtreename format
                                                    [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -QtreeName -Prefix 'subfolder' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                }
                                            }
                                        }

                                        $sb.ToString()
                                    }
                                    else {
                                        ConvertTo-SanitizedString -InputObject $secureString -QtreeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                    }
                                }
                                else {
                                    if ($secureString -match '(.+)\\(.+)') {
                                        [string]::Format('{0}\{1}', (ConvertTo-SanitizedString -InputObject $Matches[1] -ADNetBiosName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), $Matches[2])

                                        if ($Matches) {
                                            $Matches.Clear()
                                        }
                                    }
                                    else {
                                        $secureString
                                    }
                                }
                            )
                        }
                        'StorageQosWorkloadName' {
                            $entity, $wid = $secureString -split '-', 2

                            $dataRow.Item($dataColumn) = $(
                                [string]::Format(
                                    '{0}-{1}', $(
                                        if (-not [string]::IsNullOrWhiteSpace($dataRow.Item('Lun'))) {
                                            ConvertTo-SanitizedString -InputObject $entity -LunName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                        }
                                        elseif (-not [string]::IsNullOrWhiteSpace($dataRow.Item('Qtree'))) {
                                            ConvertTo-SanitizedString -InputObject $entity -QtreeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                        }
                                        elseif (-not [string]::IsNullOrWhiteSpace($dataRow.Item('Volume'))) {
                                            ConvertTo-SanitizedString -InputObject $entity -VolumeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                        }
                                        elseif (-not [string]::IsNullOrWhiteSpace($dataRow.Item('Vserver'))) {
                                            ConvertTo-SanitizedString -InputObject $entity -VserverName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                        }
                                        else {
                                            $entity
                                        }
                                    ), $wid
                                )
                            )
                        }
                        'EnvironmentVariableValue' {
                            $variable = $dataRow.Item('Variable')
                            $value = $secureString.ToString().Trim()

                            $dataRow.Item($dataColumn) = $(
                                switch -Exact ($variable) {
                                    'bootarg.dblade.root_volume.local_name' {
                                        ConvertTo-SanitizedString -InputObject $value -VolumeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                    }
                                    'bootarg.mgwd.cluster_name' {
                                        ConvertTo-SanitizedString -InputObject $value -ClusterName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                    }
                                    'failoverToken' {
                                        ConvertTo-SanitizedString -InputObject $value -NodeName -StringMatch -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                    }
                                    'kmip.init.gateway' {
                                        ConvertTo-SanitizedString -InputObject $value -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                    }
                                    'kmip.init.ipaddr' {
                                        ConvertTo-SanitizedString -InputObject $value -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                    }
                                    'NET_GATEWAY' {
                                        ConvertTo-SanitizedString -InputObject $value -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                    }
                                    'NET_IPADDR' {
                                        ConvertTo-SanitizedString -InputObject $value -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                    }
                                    'NET_NAMESERVER' {
                                        ConvertTo-SanitizedString -InputObject $value -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                    }
                                    default {
                                        $value
                                    }
                                }
                            )
                        }
                        '7ModeSmMultiPath' {
                            $srcAddr, $destAddr = $secureString -split ', ', 2

                            $dataRow.Item($dataColumn) = $(
                                [string]::Format(
                                    '{0}, {1}', $(
                                        $controller, $path = $srcAddr -split ':', 2

                                        if (-not [string]::IsNullOrWhiteSpace($path)) {
                                            [string]::Format(
                                                '{0}:{1}', $(
                                                    (ConvertTo-SanitizedString -InputObject $controller -ControllerName -StringMatch -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings),
                                                    $(
                                                        $sb = New-Object -TypeName System.Text.StringBuilder
                                                        $pathElements = New-Object -TypeName System.Collections.Generic.List[System.String]

                                                        $path -split '/' | Select-Object -Skip 1 | ForEach-Object { $pathElements.Add($_) }

                                                        for ($i = 0; $i -lt $pathElements.Count; $i++) {
                                                            switch -Exact ($i) {
                                                                '0' {
                                                                    [void]$sb.Append([string]::Format('/{0}', ($pathElements[$i])))
                                                                }
                                                                '1' {
                                                                    [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -VolumeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                                }
                                                                '2' {
                                                                    [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -QtreeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                                }
                                                                default {
                                                                    [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -QtreeName -Prefix 'subfolder' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                                }
                                                            }
                                                        }

                                                        $sb.ToString()
                                                    )
                                                )
                                            )
                                        }
                                        else {
                                            (ConvertTo-SanitizedString -InputObject $controller -ControllerName -StringMatch -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)
                                        }
                                    ),
                                    $(
                                        $controller, $path = $destAddr -split ':', 2

                                        if (-not [string]::IsNullOrWhiteSpace($path)) {
                                            [string]::Format(
                                                '{0}:{1}', $(
                                                    (ConvertTo-SanitizedString -InputObject $controller -ControllerName -StringMatch -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings),
                                                    $(
                                                        $sb = New-Object -TypeName System.Text.StringBuilder
                                                        $pathElements = New-Object -TypeName System.Collections.Generic.List[System.String]

                                                        $path -split '/' | Select-Object -Skip 1 | ForEach-Object { $pathElements.Add($_) }

                                                        for ($i = 0; $i -lt $pathElements.Count; $i++) {
                                                            switch -Exact ($i) {
                                                                '0' {
                                                                    [void]$sb.Append([string]::Format('/{0}', ($pathElements[$i])))
                                                                }
                                                                '1' {
                                                                    [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -VolumeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                                }
                                                                '2' {
                                                                    [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -QtreeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                                }
                                                                default {
                                                                    [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -QtreeName -Prefix 'subfolder' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                                }
                                                            }
                                                        }

                                                        $sb.ToString()
                                                    )
                                                )
                                            )
                                        }
                                        else {
                                            (ConvertTo-SanitizedString -InputObject $controller -ControllerName -StringMatch -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)
                                        }
                                    )
                                )
                            )
                        }
                        '7ModeOptions' {
                            $variable = $dataRow.Item('OptionName')
                            $value = $secureString.ToString().Trim()

                            $dataRow.Item($dataColumn) = $(
                                switch -Regex ($variable) {
                                    'autosupport[.]from' {
                                        $user, $domain = $value -split '@', 2

                                        if (-not [string]::IsNullOrWhiteSpace($user) -and -not [string]::IsNullOrWhiteSpace($domain)) {
                                            [string]::Format('{0}@{1}', $user, (ConvertTo-SanitizedString -InputObject $domain -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                        }
                                        else {
                                            $value
                                        }

                                        break
                                    }
                                    'autosupport[.](?:note|partner[.])?to' {
                                        $(
                                            foreach ($entry in ($value -split "`n")) {
                                                $entry -split '\s*,\s*' | Where-Object { -not [string]::IsNullOrWhiteSpace($_) } | Sort-Object | ForEach-Object {
                                                    $emailEntry = $_.Trim()
                                                    $user, $domain = $emailEntry -split '@', 2

                                                    if (-not [string]::IsNullOrWhiteSpace($user) -and -not [string]::IsNullOrWhiteSpace($domain)) {
                                                        [string]::Format('{0}@{1}', $user, (ConvertTo-SanitizedString -InputObject $domain -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                                    }
                                                    else {
                                                        $emailEntry
                                                    }
                                                }
                                            }
                                        ) -join "`n"

                                        break
                                    }
                                    'autosupport[.]mailhost' {
                                        $(
                                            foreach ($entry in ($value -split "`n")) {
                                                $entry -split '\s*,\s*' | Where-Object { -not [string]::IsNullOrWhiteSpace($_) } | Sort-Object | ForEach-Object {
                                                    $mailhostEntry = $_.Trim()

                                                    switch -Wildcard ([uri]::CheckHostName($mailhostEntry)) {
                                                        'IPv[4|6]' {
                                                            ConvertTo-SanitizedString -InputObject $mailhostEntry -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                                            break
                                                        }
                                                        'Dns' {
                                                            $hostName, $domainName = $mailhostEntry -split '[.]', 2

                                                            if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                                                [string]::Format('{0}.{1}', (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'asupmailhost' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                                            }
                                                            else {
                                                                ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'asupmailhost' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                            }

                                                            break
                                                        }
                                                        default {
                                                            $mailhostEntry
                                                        }
                                                    }
                                                }
                                            }
                                        ) -join "`n"

                                        break
                                    }
                                    'autosupport[.]support[.]proxy' {
                                        switch -Wildcard ([uri]::CheckHostName($value)) {
                                            'IPv[4|6]' {
                                                ConvertTo-SanitizedString -InputObject $value -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                                break
                                            }
                                            'Dns' {
                                                $hostName, $domainName = $value -split '[.]', 2

                                                if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                                    [string]::Format('{0}.{1}', (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'asupproxyserver' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                                }
                                                else {
                                                    ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'asupproxyserver' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                }

                                                break
                                            }
                                            default {
                                                ConvertTo-SanitizedString -InputObject $value -OtherServer -Prefix 'proxysrvr' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                            }
                                        }

                                        break
                                    }
                                    'dns[.]domainname' {
                                        ConvertTo-SanitizedString -InputObject $value -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                        break
                                    }
                                    'nfs[.]v4[.]id[.]domain' {
                                        switch -Wildcard ([uri]::CheckHostName($value)) {
                                            'Dns' {
                                                ConvertTo-SanitizedString -InputObject $value -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                                break
                                            }
                                            default {
                                                $value
                                            }
                                        }

                                        break
                                    }
                                    'timed[.]servers' {
                                        $(
                                            foreach ($entry in ($value -split "`n")) {
                                                $entry -split '\s*,\s*' | Where-Object { -not [string]::IsNullOrWhiteSpace($_) } | Sort-Object | ForEach-Object {
                                                    $timeServerEntry = $_.Trim()

                                                    switch -Wildcard ([uri]::CheckHostName($timeServerEntry)) {
                                                        'IPv[4|6]' {
                                                            ConvertTo-SanitizedString -InputObject $timeServerEntry -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                                            break
                                                        }
                                                        'Dns' {
                                                            $hostName, $domainName = $timeServerEntry -split '[.]', 2

                                                            if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                                                [string]::Format('{0}.{1}', (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'ntpserver' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                                            }
                                                            else {
                                                                ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'ntpserver' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                            }

                                                            break
                                                        }
                                                        default {
                                                            $timeServerEntry
                                                        }
                                                    }
                                                }
                                            }
                                        ) -join "`n"

                                        break
                                    }
                                    'cf[.]hw_assist[.]partner[.]address' {
                                        ConvertTo-SanitizedString -InputObject $value -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                    }
                                    'cifs[.]wins_servers' {
                                        $(
                                            foreach ($hostEntry in ($value -split '\s*,\s*')) {
                                                switch -Wildcard ([uri]::CheckHostName($hostEntry)) {
                                                    'IPv[4|6]' {
                                                        ConvertTo-SanitizedString -InputObject $hostEntry -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                                        break
                                                    }
                                                    'Dns' {
                                                        $hostName, $domainName = $hostEntry -split '[.]', 2

                                                        if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                                            [string]::Format('{0}.{1}', (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'winsserver' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                                        }
                                                        else {
                                                            ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'winsserver' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                        }

                                                        break
                                                    }
                                                    default {
                                                        ConvertTo-SanitizedString -InputObject $hostEntry -OtherServer -Prefix 'winsserver' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                    }
                                                }
                                            }
                                        ) -join ','
                                    }
                                    'nis[.]servers' {
                                        $(
                                            foreach ($hostEntry in ($value -split '\s*,\s*')) {
                                                switch -Wildcard ([uri]::CheckHostName($hostEntry)) {
                                                    'IPv[4|6]' {
                                                        ConvertTo-SanitizedString -InputObject $hostEntry -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                                        break
                                                    }
                                                    'Dns' {
                                                        $hostName, $domainName = $hostEntry -split '[.]', 2

                                                        if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                                            [string]::Format('{0}.{1}', (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'nisserver' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                                        }
                                                        else {
                                                            ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'nisserver' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                        }

                                                        break
                                                    }
                                                    default {
                                                        ConvertTo-SanitizedString -InputObject $hostEntry -OtherServer -Prefix 'nisserver' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                    }
                                                }
                                            }
                                        ) -join ','
                                    }
                                    'snapvault[.]access' {
                                        if ($value -like 'host=*' -and $value -ne 'host=all') {
                                            $type, $hosts = $value -split '=', 2

                                            [string]::Format(
                                                '{0}={1}', $type, $(
                                                    (
                                                        $hosts -split '\s*,\s*' | Where-Object { -not [string]::IsNullOrWhiteSpace($_) } | ForEach-Object {
                                                            $hostEntry = $_

                                                            switch -Wildcard ([uri]::CheckHostName($hostEntry)) {
                                                                'IPv[4|6]' {
                                                                    ConvertTo-SanitizedString -InputObject $hostEntry -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                                                    break
                                                                }
                                                                'Dns' {
                                                                    $hostName, $domainName = $hostEntry -split '[.]', 2

                                                                    if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                                                        [string]::Format('{0}.{1}', (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'svaccessserver' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                                                    }
                                                                    else {
                                                                        ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'svaccessserver' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                                    }

                                                                    break
                                                                }
                                                                default {
                                                                    ConvertTo-SanitizedString -InputObject $hostEntry -OtherServer -Prefix 'svaccessserver' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                                }
                                                            }
                                                        }
                                                    ) -join ','
                                                )
                                            )
                                        }
                                        else {
                                            $value
                                        }
                                    }
                                    default {
                                        $value
                                    }
                                }
                            )
                        }
                        'DnsZone' {
                            $dataRow.Item($dataColumn) = $(
                                if ($secureString -like '*[.]*') {
                                    ConvertTo-SanitizedString -InputObject $secureString -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                }
                                else {
                                    $secureString
                                }
                            )
                        }
                        'Traphost' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    $trapHostname, $trapIpAddress = $secureEntry -split ' : ', 2

                                    switch -Wildcard ([uri]::CheckHostName($trapHostname)) {
                                        'IPv[4|6]' {
                                            [string]::Format('{0} : {1}', (ConvertTo-SanitizedString -InputObject $trapHostname -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $trapIpAddress -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))

                                            break
                                        }
                                        'Dns' {
                                            $hostName, $domainName = $trapHostname -split '[.]', 2

                                            if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                                [string]::Format('{0}.{1} : {2}', (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'snmptraphost' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $trapIpAddress -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                            }
                                            else {
                                                [string]::Format('{0} : {1}', (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'snmptraphost' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $trapIpAddress -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                            }

                                            break
                                        }
                                        default {
                                            ConvertTo-SanitizedString -InputObject $trapHostname -OtherServer -Prefix 'snmptraphost' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                        }
                                    }
                                }
                            ) -join $delimiter
                        }
                        'DebugVreportObject' {
                            $dataRow.Item($dataColumn) = $(
                                if ($secureString -match '(?:\w+[:]){2}') {
                                    $clus, $vs, $dsid = $secureString -split ':', 3

                                    [string]::Format('{0}:{1}:{2}', (ConvertTo-SanitizedString -InputObject $clus -ClusterName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $vs -VserverName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), $dsid)
                                }
                                elseif ($secureString -like '*:*') {
                                    $vs, $vol = $secureString -split '\s*:\s*', 2

                                    [string]::Format('{0}:{1}', (ConvertTo-SanitizedString -InputObject $vs -VserverName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $vol -VolumeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                }
                                else {
                                    ConvertTo-SanitizedString -InputObject $secureString -AggregateName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                }
                            )
                        }
                        '7ModeVscanScanner' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    $scannerName, $scannerPriOrSec = $secureEntry -split ': ', 2

                                    switch -Wildcard ([uri]::CheckHostName($scannerName)) {
                                        'IPv[4|6]' {
                                            [string]::Format('{0}: {1}', (ConvertTo-SanitizedString -InputObject $scannerName -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), $scannerPriOrSec)

                                            break
                                        }
                                        'Dns' {
                                            $hostName, $domainName = $scannerName -split '[.]', 2

                                            if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                                [string]::Format('{0}.{1} : {2}', (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'vscanserver' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), $scannerPriOrSec)
                                            }
                                            else {
                                                [string]::Format('{0} : {1}', (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'vscanserver' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), $scannerPriOrSec)
                                            }

                                            break
                                        }
                                        default {
                                            [string]::Format('{0}: {1}', $scannerName, $scannerPriOrSec)
                                        }
                                    }
                                }
                            ) -join $delimiter
                        }
                        '7ModeSecureAdminAccessHost' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    $protocol, $hostsEntry = $secureEntry -split ': ', 2

                                    [string]::Format(
                                        '{0}: {1}', $protocol, $(
                                            if ($hostsEntry -match 'host=|if=' -and $hostsEntry -ne 'host=all') {
                                                $type, $hosts = $hostsEntry -split '=', 2

                                                [string]::Format(
                                                    '{0}={1}', $type, $(
                                                        (
                                                            $hosts -split '\s*,\s*' | Where-Object { -not [string]::IsNullOrWhiteSpace($_) } | Sort-Object | ForEach-Object {
                                                                $hostEntry = $_

                                                                switch -Wildcard ([uri]::CheckHostName($hostEntry)) {
                                                                    'IPv[4|6]' {
                                                                        ConvertTo-SanitizedString -InputObject $hostEntry -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                                                        break
                                                                    }
                                                                    'Dns' {
                                                                        $hostName, $domainName = $hostEntry -split '[.]', 2

                                                                        if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                                                            [string]::Format('{0}.{1}', (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix "$( $protocol )server" -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                                                        }
                                                                        else {
                                                                            ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix "$( $protocol )server" -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                                        }

                                                                        break
                                                                    }
                                                                    default {
                                                                        $hostEntry
                                                                    }
                                                                }
                                                            }
                                                        ) -join ','
                                                    )
                                                )
                                            }
                                            else {
                                                $hostsEntry
                                            }
                                        )
                                    )
                                }
                            ) -join $delimiter
                        }
                        '7ModeNfsExportRule' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    if ($secureEntry -eq 'all-hosts') {
                                        $secureEntry

                                        continue
                                    }

                                    $negate, $hostEntry = [regex]::Matches($secureEntry, '([-])?(.+)').Groups[1, 2]

                                    [string]::Format(
                                        '{0}{1}', $( if ($negate.Success) { '-' } ), $(
                                            if ($hostEntry.Value.Contains('/')) {
                                                $address, $netmaskLength = $hostEntry.Value -split '/'

                                                $ip = [ipaddress]::Any

                                                if ([ipaddress]::TryParse($address, [ref]$ip)) {
                                                    if ($ip.AddressFamily -eq 'InterNetworkV6') {
                                                        ConvertTo-SanitizedString -InputObject $ip.IPAddressToString -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                    }
                                                    else {
                                                        [string]::Format('{0}/{1}', (ConvertTo-SanitizedString -InputObject $ip.IPAddressToString -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), $netmaskLength)
                                                    }
                                                }
                                            }
                                            else {
                                                switch -Wildcard ([uri]::CheckHostName($hostEntry.Value)) {
                                                    'IPv[4|6]' {
                                                        ConvertTo-SanitizedString -InputObject $hostEntry.Value -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                                        break
                                                    }
                                                    'Dns' {
                                                        $hostName, $domainName = $hostEntry.Value -split '[.]', 2

                                                        if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                                            [string]::Format('{0}.{1}', (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'nfshost' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                                        }
                                                        else {
                                                            ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'nfshost' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                        }

                                                        break
                                                    }
                                                    default {
                                                        $hostEntry.Value
                                                    }
                                                }
                                            }
                                        )
                                    )
                                }
                            ) -join $delimiter
                        }
                        '7ModeAdDomainUser' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    try {
                                        New-Object -TypeName System.Security.Principal.SecurityIdentifier -ArgumentList $secureEntry
                                    }
                                    catch {
                                        $Global:Error.RemoveAt(0)

                                        $domainUserName = $secureEntry
                                        $userElements = New-Object -TypeName System.Collections.Generic.List[System.String]

                                        $domainUserName -split '\\' | Where-Object { -not [string]::IsNullOrWhiteSpace($_) } | ForEach-Object { $userElements.Add($_.ToLower()) }

                                        if (($userElements | Measure-Object).Count -gt 1) {
                                            $domain = ConvertTo-SanitizedString -InputObject $userElements[0] -ControllerName -StringMatchExact -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                            if ($domain -ceq $userElements[0]) {
                                                $domain = ConvertTo-SanitizedString -InputObject $userElements[0] -VfilerName -StringMatchExact -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                                if ($domain -ceq $userElements[0]) {
                                                    $domain = ConvertTo-SanitizedString -InputObject $userElements[0] -ADNetBiosName -Prefix 'addomain' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                }
                                            }

                                            [string]::Format('{0}\{1}', $domain, $userElements[1])
                                        }
                                        else {
                                            $domainUserName
                                        }
                                    }
                                }
                            ) -join $delimiter
                        }
                        '7ModeNfsExportString' {
                            $dataRow.Item($dataColumn) = $(
                                foreach ($ruleElement in ($secureString -split '\s*,\s*')) {
                                    if ($ruleElement -match '((?:[-]actual)(?>[=]))+(.+)') {
                                        [string]::Format(
                                            '{0}{1}', $Matches[1], $(
                                                $sb = New-Object -TypeName System.Text.StringBuilder
                                                $pathElements = New-Object -TypeName System.Collections.Generic.List[System.String]

                                                #We will have to loop through each element with a string match (which would finally end in subfolder). This assumes all vols and qtrees have already been mapped
                                                $Matches[2] -split '/' | Select-Object -Skip 1 | ForEach-Object { $pathElements.Add($_) }

                                                if ($pathElements[0] -eq 'vol' -or $pathElements[0] -eq 'etc') {
                                                    [void]$sb.Append([string]::Format('/{0}', $pathElements[0]))
                                                    $pathElements.RemoveAt(0)
                                                }

                                                for ($i = 0; $i -lt $pathElements.Count; $i++) {
                                                    switch -Exact ($i) {
                                                        '0' {
                                                            [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -VolumeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                        }
                                                        '1' {
                                                            [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -QtreeName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                        }
                                                        default {
                                                            [void]$sb.Append([string]::Format('/{0}', (ConvertTo-SanitizedString -InputObject $pathElements[$i] -QtreeName -Prefix 'subfolder' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings)))
                                                        }
                                                    }
                                                }

                                                $sb.ToString()
                                            )
                                        )
                                    }
                                    elseif ($ruleElement -match '((?:root|rw|ro)(?>[=]))+(.+)') {
                                        $type, $hosts = $Matches[1, 2]

                                        [string]::Format(
                                            '{0}{1}', $type, $(
                                                foreach ($hostEntry in ($hosts -split ':')) {
                                                    if ($hostEntry -like '-*') {
                                                        '-'
                                                    }

                                                    $hostEntry = $hostEntry -replace '-'

                                                    if ($hostEntry.Contains('/')) {
                                                        $address, $netmaskLength = $hostEntry -split '/'

                                                        $ip = [ipaddress]::Any

                                                        if ([ipaddress]::TryParse($address, [ref]$ip)) {
                                                            if ($ip.AddressFamily -eq 'InterNetworkV6') {
                                                                ConvertTo-SanitizedString -InputObject $ip.IPAddressToString -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                            }
                                                            else {
                                                                [string]::Format('{0}/{1}', (ConvertTo-SanitizedString -InputObject $ip.IPAddressToString -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), $netmaskLength)
                                                            }
                                                        }
                                                    }
                                                    else {
                                                        switch -Wildcard ([uri]::CheckHostName($hostEntry)) {
                                                            'IPv[4|6]' {
                                                                ConvertTo-SanitizedString -InputObject $hostEntry -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                                                                break
                                                            }
                                                            'Dns' {
                                                                $hostName, $domainName = $hostEntry -split '[.]', 2

                                                                if (-not [string]::IsNullOrWhiteSpace($domainName)) {
                                                                    [string]::Format('{0}.{1}', (ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'nfshost' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings), (ConvertTo-SanitizedString -InputObject $domainName -DnsDomainName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings))
                                                                }
                                                                else {
                                                                    ConvertTo-SanitizedString -InputObject $hostName -OtherServer -Prefix 'nfshost' -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                                                }

                                                                break
                                                            }
                                                            default {
                                                                $hostEntry
                                                            }
                                                        }
                                                    }
                                                }
                                            ) -join ':'
                                        )
                                    }
                                    else {
                                        $ruleElement
                                    }
                                }
                            ) -join ','
                        }
                        'SecuritySessionLimit' {
                            $dataRow.Item($dataColumn) = $(
                                if ($dataRow.Item('Category') -eq 'Location') {
                                    $location, $ipSpace = $secureString -split '\s:\s', 2

                                    [string]::Format(
                                        '{0} : {1}', $(
                                            if ($location -ne '-default-') {
                                                ConvertTo-SanitizedString -InputObject $location -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                            }
                                            else {
                                                $location
                                            }
                                        ), $(
                                            if ($ipSpace -ne 'Default') {
                                                ConvertTo-SanitizedString -InputObject $ipSpace -IpAddress -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                            }
                                            else {
                                                $ipSpace
                                            }
                                        )
                                    )
                                }
                                elseif ($dataRow.Item('Category') -eq 'User') {
                                    $vserver, $user = $secureString -split '\s:\s', 2

                                    [string]::Format(
                                        '{0} : {1}', $(
                                            if ($vserver -ne 'Cluster') {
                                                ConvertTo-SanitizedString -InputObject $vserver -VserverName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                            }
                                            else {
                                                $vserver
                                            }
                                        ), $user
                                    )
                                }
                                elseif ($dataRow.Item('Category') -eq 'Vserver' -and $secureString -ne 'Cluster') {
                                    ConvertTo-SanitizedString -InputObject $secureString -VserverName -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings
                                }
                                else {
                                    $secureString
                                }
                            )
                        }
                        default {
                            $sanitizeSection = $SanitizeMappings.Sanitization.Sections.SelectSingleNode("./Section[SanitizeLevel=`"$( $dataColumn.ExtendedProperties['SanitizeLevel'] )`" and (not(Mode) or Mode=`"$mode`")]")

                            $dataRow.Item($dataColumn) = $(
                                foreach ($secureEntry in $( ($secureString, $( $secureString -split $delimiter ))[[byte](! [string]::IsNullOrEmpty($delimiter))] -replace '^\s*|\s*$' )) {
                                    if ([string]::IsNullOrWhiteSpace($secureEntry)) {
                                        $null

                                        continue
                                    }

                                    $params = @{
                                        'InputObject'         = $secureEntry
                                        'SanitizeMappings'    = $SanitizeMappings
                                        'SanitizeLevel'       = $SanitizeLevel
                                        $sanitizeSection.Name = [switch]::Present
                                        'Prefix'              = ($sanitizeSection.Prefix, $prefix)[[byte](! [string]::IsNullOrWhiteSpace($prefix))]
                                    }

                                    if ($dataColumn.ExtendedProperties.ContainsKey('SanitizeMethod')) {
                                        $params.Add($dataColumn.ExtendedProperties['SanitizeMethod'], $true)
                                    }

                                    ConvertTo-SanitizedString @params
                                }
                            ) -join $delimiter
                        }
                    }
                }
            }
        }
    }

    $statsParams = @{
        FunctionPSCallStack = Get-PSCallStack
        ElapsedTicks        = ([datetime]::Now - $functionStartTime).Ticks
        ExcludeParameter    = 'DataSet', 'SanitizeMappings'
    }

    Write-Statistics @statsParams
}

function ConvertTo-SanitizedString {
    [CmdletBinding()]
    [OutputType([System.String])]
    param (
        [Parameter(Mandatory, Position = 0, ValueFromPipeline)]
        [AllowEmptyString()]
        [string]$InputObject,

        [Parameter(Mandatory, Position = 1)]
        [xml]$SanitizeMappings,

        [Parameter()]
        [string]$Prefix,

        [Parameter()]
        [int]$SanitizeLevel = 0,

        [Parameter()]
        [switch]$StringMatch,

        [Parameter()]
        [switch]$StringMatchExact,

        [Parameter(ParameterSetName = 'IpAddress')]
        [switch]$IpAddress,

        [Parameter(ParameterSetName = 'MacAddress')]
        [switch]$MacAddress,

        [Parameter(ParameterSetName = 'ControllerName')]
        [switch]$ControllerName,

        [Parameter(ParameterSetName = 'NodeName')]
        [switch]$NodeName,

        [Parameter(ParameterSetName = 'ComputerName')]
        [switch]$ComputerName,

        [Parameter(ParameterSetName = 'DnsDomainName')]
        [switch]$DnsDomainName,

        [Parameter(ParameterSetName = 'VfilerName')]
        [switch]$VfilerName,

        [Parameter(ParameterSetName = 'VserverName')]
        [switch]$VserverName,

        [Parameter(ParameterSetName = 'LicenseCode')]
        [switch]$LicenseCode,

        [Parameter(ParameterSetName = 'AggregateName')]
        [switch]$AggregateName,

        [Parameter(ParameterSetName = 'VolumeGroupName')]
        [switch]$VolumeGroupName,

        [Parameter(ParameterSetName = 'VolumeName')]
        [switch]$VolumeName,

        [Parameter(ParameterSetName = 'QtreeName')]
        [switch]$QtreeName,

        [Parameter(ParameterSetName = 'LunName')]
        [switch]$LunName,

        [Parameter(ParameterSetName = 'IgroupName')]
        [switch]$IgroupName,

        [Parameter(ParameterSetName = 'OtherServer')]
        [switch]$OtherServer,

        [Parameter(ParameterSetName = 'SnmpCommunityString')]
        [switch]$SnmpCommunityString,

        [Parameter(ParameterSetName = 'CifsShareName')]
        [switch]$CifsShareName,

        [Parameter(ParameterSetName = 'ADNetBiosName')]
        [switch]$ADNetBiosName,

        [Parameter(ParameterSetName = 'ClusterName')]
        [switch]$ClusterName,

        [Parameter(ParameterSetName = 'StorageSystemId')]
        [switch]$StorageSystemId,

        [Parameter(ParameterSetName = 'OciServerName')]
        [switch]$OciServerName,

        [Parameter(ParameterSetName = 'HostName')]
        [switch]$HostName,

        [Parameter()]
        [switch]$ReturnMappingOnly
    )

    process {
        if ($SanitizeLevel -eq 0) {
            return $InputObject
        }

        $sanitizeSection = $SanitizeMappings.Sanitization.Sections.SelectSingleNode("./Section[@Name=`"$( $PSCmdlet.ParameterSetName )`"]")

        if (-not ($sanitizeSection.SanitizeLevel -band $SanitizeLevel)) {
            return $InputObject
        }

        if ([string]::IsNullOrWhiteSpace($InputObject)) {
            return $InputObject
        }

        $string = $InputObject.Trim()

        if ($string -eq 'UTD' -or $string -eq 'n/a' -or $string -eq '-' -or $string -eq '*' -or $string -eq 'all' -or $string -eq '<unknown>') {
            return $string
        }

        $outputString = $null

        switch -Exact ($PSCmdlet.ParameterSetName) {
            'IpAddress' {
                if ($string -eq '::') {
                    return $string
                }

                $address, $netmaskLength = $string -split '[/]', 2

                if ($address -eq '0.0.0.0') {
                    return $string
                }

                $ip = [ipaddress]::Any

                if ([ipaddress]::TryParse($address, [ref]$ip)) {
                    if ($sanitizeSection.SanitizeMethod -eq 'Octets') {
                        return $(
                            if ($ip.AddressFamily -eq 'InterNetworkV6') {
                                $bytes = [byte[]]([ipaddress]::Parse('2001:DB8::').GetAddressBytes() | Select-Object -First $sanitizeSection.Ipv6BytesToSanitize) + ($ip.GetAddressBytes() | Select-Object -Last (16 - $sanitizeSection.Ipv6BytesToSanitize))

                                (New-Object -TypeName System.Net.IpAddress -ArgumentList $bytes, 2).IPAddressToString
                            }
                            else {
                                $sanitizedAddress = $(
                                    switch -Exact ($sanitizeSection.Ipv4OctetsToSanitize) {
                                        1 {
                                            [string]::Format('xxx.{0}.{1}.{2}', ($ip.GetAddressBytes() | Select-Object -Last 3))
                                        }
                                        2 {
                                            [string]::Format('xxx.xxx.{0}.{1}', ($ip.GetAddressBytes() | Select-Object -Last 2))
                                        }
                                        3 {
                                            [string]::Format('xxx.xxx.xxx.{0}', ($ip.GetAddressBytes() | Select-Object -Last 1))
                                        }
                                        4 {
                                            'xxx.xxx.xxx.xxx'
                                        }
                                    }
                                )

                                if (-not [string]::IsNullOrWhiteSpace($netmaskLength)) {
                                    [string]::Format('{0}/{1}', $sanitizedAddress, $netmaskLength)
                                }
                                else {
                                    $sanitizedAddress
                                }
                            }
                        )
                    }
                    elseif ($sanitizeSection.SanitizeMethod -eq 'Mapping') {
                        do {
                            if ($ip.AddressFamily -eq 'InterNetworkV6') {
                                $nextIpAddress = Get-NextIpAddress -Address $(
                                    $lastIpAddress = [ipaddress[]]($sanitizeSection.SelectNodes('Mappings/Mapping').SanitizedString) | Select-Object -Last 1

                                    if (-not [string]::IsNullOrWhiteSpace($lastIpAddress)) {
                                        $lastIpAddress
                                    }
                                    else {
                                        '2001::db8:9:0:0'
                                    }
                                )
                                $outputString = $nextIpAddress.IpAddressToString
                            }
                            else {
                                $nextIpAddress = Get-NextIpAddress -Address $(
                                    $lastIpAddress = [ipaddress[]](
                                        $sanitizeSection.SelectNodes('Mappings/Mapping').SanitizedString | ForEach-Object {
                                            $_ -replace 'xxx.xxx.', '10.10.'
                                        }
                                    ) | Select-Object -Last 1

                                    if (-not [string]::IsNullOrWhiteSpace($lastIpAddress)) {
                                        $lastIpAddress
                                    }
                                    else {
                                        '10.10.0.0'
                                    }
                                )
                                $outputString = [string]::Format('xxx.xxx.{0}.{1}', ($nextIpAddress.GetAddressBytes() | Select-Object -Last 2))
                            }
                        }
                        until (-not $sanitizeSection.SelectSingleNode("Mappings/Mapping[SecureString=`"$outputString`"]"))

                        $newMappingEntry = $SanitizeMappings.CreateElement('Mapping')
                        $newMappingEntry.set_InnerXml("<SecureString>$( [System.Security.SecurityElement]::Escape($string) )</SecureString><SanitizedString>$outputString</SanitizedString>")
                        [void]$sanitizeSection.SelectSingleNode('Mappings').AppendChild($newMappingEntry)
                    }
                }
                else {
                    return $string
                }
            }
            'MacAddress' {
                return 'XX:XX:XX:XX:XX:XX'
            }
            'LicenseCode' {
                return 'XXXXXXX'
            }
            'DnsDomainName' {
                if ($string -eq 'netapp.com' -and -not $sanitizeSection.SelectSingleNode("Mappings/Mapping[SecureString=`"$string`"]")) {
                    return $string
                }
            }
            'AggregateName' {
                if ($string -ceq 'aggr0' -and -not $sanitizeSection.SelectSingleNode("Mappings/Mapping[SecureString=`"$string`"]")) {
                    return $string
                }
            }
            'VolumeName' {
                if ($string -ceq 'vol0' -and -not $sanitizeSection.SelectSingleNode("Mappings/Mapping[SecureString=`"$string`"]")) {
                    return $string
                }
            }
            'VserverName' {
                if (($string -match '^vs\d+$' -or $string -ceq 'Cluster') -and -not $sanitizeSection.SelectSingleNode("Mappings/Mapping[SecureString=`"$string`"]")) {
                    return $string
                }
            }
            'VfilerName' {
                if ($string -ceq 'vfiler0' -and -not $sanitizeSection.SelectSingleNode("Mappings/Mapping[SecureString=`"$string`"]")) {
                    return $string
                }
            }
            'SnmpCommunityString' {
                if ($string -ceq 'public' -and -not $sanitizeSection.SelectSingleNode("Mappings/Mapping[SecureString=`"$string`"]")) {
                    return $string
                }
            }
            'ADNetBiosName' {
                if ($string -eq 'BUILTIN') {
                    return $string
                }
            }
            'StorageSystemId' {
                if ([guid]::TryParse($string, [ref]([guid]::Empty))) {
                    return $string
                }
            }
            'CifsShareName' {
                if ($string -like '<Share Count:*') {
                    return $string
                }
            }
            default { }
        }

        if (-not [string]::IsNullOrWhiteSpace($Prefix)) {
            $mappingEntry = $sanitizeSection.SelectSingleNode("Mappings/Mapping[SecureString=`"$( [System.Security.SecurityElement]::Escape($string) )`" and starts-with(SanitizedString, `"$( [System.Security.SecurityElement]::Escape($Prefix) )`")]")
        }
        else {
            $mappingEntry = $sanitizeSection.SelectSingleNode("Mappings/Mapping[SecureString=`"$( [System.Security.SecurityElement]::Escape($string) )`"]")
        }

        if ($mappingEntry) {
            if ($ReturnMappingOnly.IsPresent) {
                return $mappingEntry
            }
            else {
                return $mappingEntry.SanitizedString
            }
        }
        else {
            if ($StringMatchExact.IsPresent -or $SanitizeMappings.Sanitization.SelectSingleNode("Options/Option[Name='CurrentEntriesOnly']").Value -eq $true) {
                return $string
            }

            if ($StringMatch.IsPresent) {
                if (($mappingEntry = $sanitizeSection.Mappings.Mapping | Where-Object { $string -like "*$( $_.SecureString )*" } | Sort-Object -Property { [NetAppDocs.Utilities.DiceSimilarity]::CompareStrings($string, $_.SecureString, $true) } | Select-Object -Last 1)) {
                    if ($ReturnMappingOnly.IsPresent) {
                        return $mappingEntry
                    }
                    else {
                        return ($string -replace $mappingEntry.SecureString, $mappingEntry.SanitizedString)
                    }
                }
                else {
                    return
                }
            }

            if ($sanitizeSection.SelectSingleNode("Mappings/Mapping[SecureString=`"$( [System.Security.SecurityElement]::Escape($string) )`"]")) {
                [void]$sanitizeSection.AppendChild($SanitizeMappings.CreateElement('Mappings'))
            }

            do {
                switch -Exact ($SanitizeMappings.Sanitization.SelectSingleNode("Options/Option[Name='NumberingMethod']").Value) {
                    'Sequential' {
                        $lastNum = 0

                        if (($sanitizeSection.SelectNodes('Mappings/Mapping').SanitizedString | Sort-Object | Select-Object -Last 1) -match '\d{4}$') {
                            $lastNum = [int]$Matches[0]
                        }

                        $outputString = [string]::Format(
                            '{0}{1:D4}', $(
                                if (-not [string]::IsNullOrWhiteSpace($Prefix)) {
                                    $Prefix -replace '^\s*$'
                                }
                                else {
                                    $sanitizeSection.Prefix -replace '^\s*$'
                                }
                            ), ($lastNum + 1)
                        )
                    }
                    'Random' {
                        $outputString = [string]::Format(
                            '{0}{1:D4}', $(
                                if (-not [string]::IsNullOrWhiteSpace($Prefix)) {
                                    $Prefix -replace '^\s*$'
                                }
                                else {
                                    $sanitizeSection.Prefix -replace '^\s*$'
                                }
                            ), (Get-Random -Minimum 1 -Maximum 9999)
                        )
                    }
                }
            }
            until (-not $sanitizeSection.SelectSingleNode("Mappings/Mapping[SanitizedString=`"$outputString`"]"))

            $newMappingEntry = $SanitizeMappings.CreateElement('Mapping')
            $newMappingEntry.set_InnerXml("<SecureString>$( [System.Security.SecurityElement]::Escape($string) )</SecureString><SanitizedString>$outputString</SanitizedString>")
            [void]$sanitizeSection.SelectSingleNode('Mappings').AppendChild($newMappingEntry)

            return $outputString
        }
    }
}

function Export-SanitizeMappingsList {
    [CmdletBinding()]
    param (
        [Parameter(Mandatory, Position = 0)]
        [xml]$SanitizeMappings,

        [Parameter(Mandatory, Position = 1)]
        [System.IO.FileInfo]$SanitizeMappingsXmlFile,

        [Parameter()]
        [ValidateSet('en-US')]
        [string]$Locale = 'en-US'
    )

    $dataSet = New-Object -TypeName System.Data.DataSet
    $dataSet.DataSetName = 'NetAppDocs.Sanitize.Mappings'
    $dataSet.CaseSensitive = $true
    $dataSet.ExtendedProperties.Add('BaseTypeName', 'None')

    foreach ($section in $SanitizeMappings.Sanitization.SelectNodes('Sections/Section[Mappings/Mapping/node()]')) {
        $dataTable = $dataSet.Tables.Add("$( $section.Name )")

        foreach ($columnName in 'SecureString', 'SanitizedString') {
            $dataTable.Columns.Add((New-Object -TypeName System.Data.DataColumn -ArgumentList $columnName))
            $dataTable.Columns[$columnName].Caption = [regex]::Replace($columnName, '([a-z])([A-Z])', '$1 $2')
        }

        $dataTable.ExtendedProperties.Add('SortOrder', 'SecureString')

        foreach ($mapping in $section.SelectNodes('Mappings/Mapping[node()]')) {
            $dataRow = $dataTable.NewRow()

            $dataRow.SecureString = $mapping.SecureString
            $dataRow.SanitizedString = $mapping.SanitizedString

            $dataTable.Rows.Add($dataRow)
        }
    }

    if ($dataSet.HasChanges()) {
        $sortedDataSet = $dataSet.Clone()

        foreach ($dataTable in $dataSet.Tables) {
            if ($dataTable.Rows.Count -gt 0) {
                Write-Verbose -Message "Sorting mappings table: $( $dataTable.TableName )"

                $sortedDataSet.Tables[$dataTable.TableName].Merge([NetAppDocs.Utilities.DataSetMethods]::SortDataTable($dataTable, [NetAppDocs.Utilities.SortMethod]::$( $dataSet.ExtendedProperties['SortMethod'] )))
            }
        }

        if (-not $sortedDataSet.ExtendedProperties.ContainsKey('SortMethod')) {
            $sortedDataSet.ExtendedProperties.Add('SortMethod', [NetAppDocs.Utilities.SortMethod]::$( $dataSet.ExtendedProperties['SortMethod'] ))
        }

        $ntapDocsWriter = New-Object -TypeName NetAppDocs.Writer -ArgumentList $sortedDataSet

        try {
            Write-Verbose -Message 'Generating SanitizeMappings XML document'

            $SanitizeMappings.Save($SanitizeMappingsXmlFile.FullName)
        }
        catch {
            $Global:Error.RemoveAt(0)

            Write-Warning -Message "SanitizeMappings XML file creation failed. Reason: $( $_.Exception.Message )"
        }

        try {
            Write-Verbose -Message 'Generating SanitizeMappings Excel document'

            $ntapDocsWriter.ExcelFile = [System.IO.Path]::ChangeExtension($SanitizeMappingsXmlFile.FullName, '.xlsx')
            $ntapDocsWriter.ExcelTemplate = Get-NtapExcelTemplate -DataSetName $sortedDataSet.DataSetName -BaseTypeName $sortedDataSet.ExtendedProperties['BaseTypeName'] -Locale $Locale
            $ntapDocsWriter.ExcelDocumentTitle = 'Sanitize Mappings'
            $ntapDocsWriter.ExcelDocumentSubject = [string]::Empty

            $ntapDocsWriter.CreateExcelFile()
        }
        catch {
            $Global:Error.RemoveAt(0)

            $ntapDocsWriter.XmlFile = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( $SanitizeMappingsXmlFile.DirectoryName )/$( $SanitizeMappingsXmlFile.BaseName )-DS.xml")
            $ntapDocsWriter.CreateXmlFile()

            $msg = New-Object -TypeName System.Text.StringBuilder

            [void]$msg.AppendLine("Sanitization Mappings Excel document output creation failed. Reason: $( $_.Exception.Message ).")
            [void]$msg.AppendLine('       XML file may be used to complete document generation.')
            [void]$msg.AppendLine("       Example: C:\> Get-ChildItem -Path $( $ntapDocsWriter.XmlFile ) | Out-NtapDocument -ExcelFile <ExcelFile>")
            [void]$msg.AppendLine()

            Write-Error -ErrorRecord (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), 'WriteError', [System.Management.Automation.ErrorCategory]::WriteError, $ntapDocsWriter.ExcelFile))
        }
    }
}

function Reset-SanitizeMappingsList {
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseShouldProcessForStateChangingFunctions', '')]
    param (
        [Parameter(Mandatory)]
        [System.IO.FileInfo]$SanitizeMappingsXmlFile
    )

    ($SanitizeMappings = New-Object -TypeName System.Xml.XmlDocument).Load($SanitizeMappingsXmlFile.FullName)

    foreach ($section in $SanitizeMappings.Sanitization.SelectNodes('Sections/Section[Mappings/Mapping/node()]')) {
        $section.SelectNodes('Mappings/Mapping') | Where-Object {
            ($section.Name -eq 'ControllerName' -and $_.SecureString -cne 'NA') -or
            ($section.Name -eq 'AggregateName' -and $_.SecureString -cne 'aggr0') -or
            ($section.Name -eq 'VolumeName' -and $_.SecureString -cne 'vol0') -or
            ($section.Name -eq 'VserverName' -and $_.SecureString -cnotmatch '^vs\d+$') -or
            ($section.Name -eq 'VfilerName' -and $_.SecureString -cne 'vfiler0') -or
            ($section.Name -eq 'SnmpCommunityString' -and $_.SecureString -cne 'public') -or
            ($section.Name -notmatch 'AggregateName|VolumeName|VfilerName|SnmpCommunityString')
        } | ForEach-Object {
            $child = $section.SelectSingleNode("Mappings/Mapping[SecureString=`"$( $_.SecureString )`"]")
            [void]$section.Mappings.RemoveChild($child)
        }
    }

    $SanitizeMappings.Save($SanitizeMappingsXmlFile.FullName)
}

function Get-NextIpAddress {
    param (
        [Parameter(Mandatory)]
        [string]$Address
    )

    $a = [ipaddress]::Parse($Address)
    $z = $a.GetAddressBytes()

    if ($z[-1] -eq 255) {
        $z[-1] = 1

        if ($z[-2] -eq 255) {
            $z[-2] = 0
            $z[-3] += 1
        }
        else {
            $z[-3] += 1
        }
    }
    else {
        $z[-1] += 1
    }

    return ([ipaddress]($z))
}

# SIG # Begin signature block
# MIIq0AYJKoZIhvcNAQcCoIIqwTCCKr0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDWOz8BKogTIcfs
# JDlpnDHX634oycKyhtxfWYMPi368xaCCJGgwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggWNMIIEdaADAgECAhAOmxiO
# +dAt5+/bUOIIQBhaMA0GCSqGSIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYD
# VQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAi
# BgNVBAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAw
# MDBaFw0zMTExMDkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
# Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERp
# Z2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBAL/mkHNo3rvkXUo8MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsb
# hA3EMB/zG6Q4FutWxpdtHauyefLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iT
# cMKyunWZanMylNEQRBAu34LzB4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGb
# NOsFxl7sWxq868nPzaw0QF+xembud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclP
# XuU15zHL2pNe3I6PgNq2kZhAkHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCr
# VYJBMtfbBHMqbpEBfCFM1LyuGwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFP
# ObURWBf3JFxGj2T3wWmIdph2PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTv
# kpI6nj3cAORFJYm2mkQZK37AlLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWM
# cCxBYKqxYxhElRp2Yn72gLD76GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls
# 5Q5SUUd0viastkF13nqsX40/ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBR
# a2+xq4aLT8LWRV+dIPyhHsXAj6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6
# MIIBNjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qY
# rhwPTzAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8E
# BAMCAYYweQYIKwYBBQUHAQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5k
# aWdpY2VydC5jb20wQwYIKwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDig
# NoY0aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNybDARBgNVHSAECjAIMAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCg
# v0NcVec4X6CjdBs9thbX979XB72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQT
# SnovLbc47/T/gLn4offyct4kvFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh
# 65ZyoUi0mcudT6cGAxN3J0TU53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSw
# uKFWjuyk1T3osdz9HNj0d1pcVIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAO
# QGPFmCLBsln1VWvPJ6tsds5vIy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjD
# TZ9ztwGpn1eqXijiuZQwggXMMIIENKADAgECAhAg429sPxgagUb53pPffJfkMA0G
# CSqGSIb3DQEBDAUAMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBS
# MzYwHhcNMjEwOTA5MDAwMDAwWhcNMjMwOTA5MjM1OTU5WjBQMQswCQYDVQQGEwJV
# UzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UECgwMTmV0QXBwLCBJbmMuMRUw
# EwYDVQQDDAxOZXRBcHAsIEluYy4wggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGK
# AoIBgQC4kYYj/oViZD9pN03hrqFjtwOz1Gx4eDWVV8IYUYEr2qpLeNYvWz26B/+E
# mYLboAGVpkYg0Wske3hjRpooZlMORm1S4+2C2NoQCstZ+kmNTevbXHJe/w1VNJrm
# fKjpXOJEfx8GUGDqCMR30dChVpsdBEavrM7T0hnfJLv18i19SGv3a+nhvB3iOjLP
# SEg8+DFkjD5jTI6fQ83DBEiVR4UEYq7yyH578ZEylhsRfJmK+uIREycEUk/NpFTw
# g6/7lxh/WYabCrZzqI4Ep3QataRKaAlxt3BiOlZkX4WGX3DYvSYltWOM2UlCcGpP
# m/1/LN8Xuvf+YO6H472QecHe59XtXbHSMKvqDuOEJu8Wk2LFMNK732o3Fc5QIHie
# 6JtNsgNYf+Vsm5EMBD1ZHNE+C3zlmQbbWIU5uEU1nhiodBPKsu0SkS6pARyKBO05
# DSc2gWWv0aukKi04ZAn+hAdSl5s1dgmU5shBvKHqe15K9ZRN4FFO/CgHE0BoGYQS
# UQVKwa0CAwEAAaOCAZwwggGYMB8GA1UdIwQYMBaAFA8qyyCHKLjsb0iuK1SmKaoX
# pM0MMB0GA1UdDgQWBBQuH643KcBMmb/Q6IZt+H9IrnXFwDAOBgNVHQ8BAf8EBAMC
# B4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDAzARBglghkgBhvhC
# AQEEBAMCBBAwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwIwJTAjBggrBgEFBQcC
# ARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQBMEkGA1UdHwRCMEAw
# PqA8oDqGOGh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVT
# aWduaW5nQ0FSMzYuY3JsMHkGCCsGAQUFBwEBBG0wazBEBggrBgEFBQcwAoY4aHR0
# cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQVIz
# Ni5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqG
# SIb3DQEBDAUAA4IBgQCOoGdXjP8Sif0h3ZvoDsIVfrJvQUdP9pZExRJGGj/Te6ML
# XyqHNc/G2WklNv+BC4ryNJ4zp0oneHOub55ejrYPfhgP9oFWS8oOPJbrp3rAtINa
# OeKRC88WUYZSKYAxSX6kubEjnQD6cmazXjxVN6P8+6q9smeqF3xI+++c+ekMw3Xv
# 4EWRYXqN0/srfFp1WpoluwSQcZMqj9haSX0bgO6AAWH2AnVJMfXTIKz/0FQ/RW0y
# Ed5QYQqYInhA7IUz9dBziNpJlwONeF5j7xzgfYDY63WU6WrgJnGjYkQCOOsUx74j
# gUiMRe9zV48GS8Vxz22c/TQs0x4/1GmOSJvUHEk3GseBmB3v+yEhh/D6zWOfYP4X
# D/9b91CxmugTuepuaJSeRg+qUm3KarKsOCUF/CLqUrNh/JwKrWD1cghRaYLvMucs
# ScksHHe7ZDvb2OtvxWXjPk1d1NKvEwFJSS6hIVIbug9x28AJqOoP5Pn9wZvJSzvW
# uQJGCLscdGyKefdCo30wggYaMIIEAqADAgECAhBiHW0MUgGeO5B5FSCJIRwKMA0G
# CSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290
# IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFQxCzAJBgNVBAYT
# AkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28g
# UHVibGljIENvZGUgU2lnbmluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEBAQUAA4IB
# jwAwggGKAoIBgQCbK51T+jU/jmAGQ2rAz/V/9shTUxjIztNsfvxYB5UXeWUzCxEe
# AEZGbEN4QMgCsJLZUKhWThj/yPqy0iSZhXkZ6Pg2A2NVDgFigOMYzB2OKhdqfWGV
# oYW3haT29PSTahYkwmMv0b/83nbeECbiMXhSOtbam+/36F09fy1tsB8je/RV0mIk
# 8XL/tfCK6cPuYHE215wzrK0h1SWHTxPbPuYkRdkP05ZwmRmTnAO5/arnY83jeNzh
# P06ShdnRqtZlV59+8yv+KIhE5ILMqgOZYAENHNX9SJDm+qxp4VqpB3MV/h53yl41
# aHU5pledi9lCBbH9JeIkNFICiVHNkRmq4TpxtwfvjsUedyz8rNyfQJy/aOs5b4s+
# ac7IH60B+Ja7TVM+EKv1WuTGwcLmoU3FpOFMbmPj8pz44MPZ1f9+YEQIQty/NQd/
# 2yGgW+ufflcZ/ZE9o1M7a5Jnqf2i2/uMSWymR8r2oQBMdlyh2n5HirY4jKnFH/9g
# Rvd+QOfdRrJZb1sCAwEAAaOCAWQwggFgMB8GA1UdIwQYMBaAFDLrkpr/NZZILyhA
# QnAgNpFcF4XmMB0GA1UdDgQWBBQPKssghyi47G9IritUpimqF6TNDDAOBgNVHQ8B
# Af8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcD
# AzAbBgNVHSAEFDASMAYGBFUdIAAwCAYGZ4EMAQQBMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Um9vdFI0Ni5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ1Jvb3RSNDYu
# cDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQwFAAOCAgEABv+C4XdjNm57oRUgmxP/BP6YdURhw1aVcdGRP4Wh60BAscjW
# 4HL9hcpkOTz5jUug2oeunbYAowbFC2AKK+cMcXIBD0ZdOaWTsyNyBBsMLHqafvIh
# rCymlaS98+QpoBCyKppP0OcxYEdU0hpsaqBBIZOtBajjcw5+w/KeFvPYfLF/ldYp
# mlG+vd0xqlqd099iChnyIMvY5HexjO2AmtsbpVn0OhNcWbWDRF/3sBp6fWXhz7Dc
# ML4iTAWS+MVXeNLj1lJziVKEoroGs9Mlizg0bUMbOalOhOfCipnx8CaLZeVme5yE
# Lg09Jlo8BMe80jO37PU8ejfkP9/uPak7VLwELKxAMcJszkyeiaerlphwoKx1uHRz
# NyE6bxuSKcutisqmKL5OTunAvtONEoteSiabkPVSZ2z76mKnzAfZxCl/3dq3dUNw
# 4rg3sTCggkHSRqTqlLMS7gjrhTqBmzu1L90Y1KWN/Y5JKdGvspbOrTfOXyXvmPL6
# E52z1NZJ6ctuMFBQZH3pwWvqURR8AgQdULUvrxjUYbHHj95Ejza63zdrEcxWLDX6
# xWls/GDnVNueKjWUH3fTv1Y8Wdho698YADR7TNx8X8z2Bev6SivBBOHY+uqiirZt
# g0y9ShQoPzmCcn63Syatatvx157YK9hlcPmVoa1oDE5/L9Uo2bC5a4CH2Rwwggau
# MIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJ
# BgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5k
# aWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAe
# Fw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcw
# FQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3Rl
# ZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9Ge
# TKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0
# hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZl
# jZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAsh
# aG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVY
# TXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1
# biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCir
# c0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+
# DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA
# +bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42Pg
# puE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzS
# M7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU
# uhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6
# mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsG
# AQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29t
# MEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNl
# cnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3Js
# My5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAE
# GTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1Z
# jsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8d
# B+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVp
# P0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp8
# 76i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2
# nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3
# ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQ
# txMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc
# 4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+Y
# AN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZ
# vAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQr
# H4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIGwDCCBKigAwIBAgIQDE1p
# ckuU+jwqSj0pB4A9WjANBgkqhkiG9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4XDTIyMDkyMTAwMDAw
# MFoXDTMzMTEyMTIzNTk1OVowRjELMAkGA1UEBhMCVVMxETAPBgNVBAoTCERpZ2lD
# ZXJ0MSQwIgYDVQQDExtEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMiAtIDIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDP7KUmOsap8mu7jcENmtuh6BSFdDMa
# JqzQHFUeHjZtvJJVDGH0nQl3PRWWCC9rZKT9BoMW15GSOBwxApb7crGXOlWvM+xh
# iummKNuQY1y9iVPgOi2Mh0KuJqTku3h4uXoW4VbGwLpkU7sqFudQSLuIaQyIxvG+
# 4C99O7HKU41Agx7ny3JJKB5MgB6FVueF7fJhvKo6B332q27lZt3iXPUv7Y3UTZWE
# aOOAy2p50dIQkUYp6z4m8rSMzUy5Zsi7qlA4DeWMlF0ZWr/1e0BubxaompyVR4aF
# eT4MXmaMGgokvpyq0py2909ueMQoP6McD1AGN7oI2TWmtR7aeFgdOej4TJEQln5N
# 4d3CraV++C0bH+wrRhijGfY59/XBT3EuiQMRoku7mL/6T+R7Nu8GRORV/zbq5Xwx
# 5/PCUsTmFntafqUlc9vAapkhLWPlWfVNL5AfJ7fSqxTlOGaHUQhr+1NDOdBk+lbP
# 4PQK5hRtZHi7mP2Uw3Mh8y/CLiDXgazT8QfU4b3ZXUtuMZQpi+ZBpGWUwFjl5S4p
# kKa3YWT62SBsGFFguqaBDwklU/G/O+mrBw5qBzliGcnWhX8T2Y15z2LF7OF7ucxn
# EweawXjtxojIsG4yeccLWYONxu71LHx7jstkifGxxLjnU15fVdJ9GSlZA076XepF
# cxyEftfO4tQ6dwIDAQABo4IBizCCAYcwDgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB
# /wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwIAYDVR0gBBkwFzAIBgZngQwB
# BAIwCwYJYIZIAYb9bAcBMB8GA1UdIwQYMBaAFLoW2W1NhS9zKXaaL3WMaiCPnshv
# MB0GA1UdDgQWBBRiit7QYfyPMRTtlwvNPSqUFN9SnDBaBgNVHR8EUzBRME+gTaBL
# hklodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0
# MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3JsMIGQBggrBgEFBQcBAQSBgzCBgDAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFgGCCsGAQUFBzAC
# hkxodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRS
# U0E0MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3J0MA0GCSqGSIb3DQEBCwUAA4IC
# AQBVqioa80bzeFc3MPx140/WhSPx/PmVOZsl5vdyipjDd9Rk/BX7NsJJUSx4iGNV
# CUY5APxp1MqbKfujP8DJAJsTHbCYidx48s18hc1Tna9i4mFmoxQqRYdKmEIrUPwb
# tZ4IMAn65C3XCYl5+QnmiM59G7hqopvBU2AJ6KO4ndetHxy47JhB8PYOgPvk/9+d
# EKfrALpfSo8aOlK06r8JSRU1NlmaD1TSsht/fl4JrXZUinRtytIFZyt26/+YsiaV
# OBmIRBTlClmia+ciPkQh0j8cwJvtfEiy2JIMkU88ZpSvXQJT657inuTTH4YBZJwA
# wuladHUNPeF5iL8cAZfJGSOA1zZaX5YWsWMMxkZAO85dNdRZPkOaGK7DycvD+5sT
# X2q1x+DzBcNZ3ydiK95ByVO5/zQQZ/YmMph7/lxClIGUgp2sCovGSxVK05iQRWAz
# gOAj3vgDpPZFR+XOuANCR+hBNnF3rf2i6Jd0Ti7aHh2MWsgemtXC8MYiqE+bvdgc
# mlHEL5r2X6cnl7qWLoVXwGDneFZ/au/ClZpLEQLIgpzJGgV8unG1TnqZbPTontRa
# mMifv427GFxD9dAq6OJi7ngE273R+1sKqHB+8JeEeOMIA11HLGOoJTiXAdI/Otrl
# 5fbmm9x+LMz/F0xNAKLY1gEOuIvu5uByVYksJxlh9ncBjDGCBb4wggW6AgEBMGgw
# VDELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDErMCkGA1UE
# AxMiU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIFIzNgIQIONvbD8YGoFG
# +d6T33yX5DANBglghkgBZQMEAgEFAKCBhDAYBgorBgEEAYI3AgEMMQowCKACgACh
# AoAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAM
# BgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCDddh/IkmX10EPq6KEC9s31VuSp
# tqMRoJRc3ixaD6TTsDANBgkqhkiG9w0BAQEFAASCAYCam5N2Fgu0SvVNjC2QCDkT
# bMeejLUl+QM0egKLOWh2hctTBLmR5JkYklOLBDwonB1GsyYP6W5RQ/codKzwV+El
# siDta5BqZIYxT8VttwJijaVMHdHcBgKunT9PtnaSyftthfmeGqwPW+8hfURBsSl6
# acK/lHHHaf3CSgtAzMOb55axs5pPggsroG/zacD9W4YRq/UQYbB9qEs5VkKlQXT7
# PGNF9aNPkd3qKWKzA/cnWyZ3pdeOteZsHdzQwEIQKv8L9uOz1JpN1L6AW6LXstGE
# p5mVvmhbYhwEn58SDeJzNt3Y6T+kh6dXLrs/tLp6FOidFGlhGaC1P5+KsM3K22lA
# FK707jW3TecjKerywdzwgC6fhBuDkyU+QvkAeSW0E+aPnMYQRH/emRiYhgstcpH/
# URIuvah+8V77MteJv54JZkEQ+h0nanZyOojYdBtROohAJoQBkQPo1zctiELDjv0P
# KC1zYevcezM5PFH2pceC8uSYSxwkor6DKcLBv+wzkpOhggMgMIIDHAYJKoZIhvcN
# AQkGMYIDDTCCAwkCAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
# cnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
# SEEyNTYgVGltZVN0YW1waW5nIENBAhAMTWlyS5T6PCpKPSkHgD1aMA0GCWCGSAFl
# AwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUx
# DxcNMjMwNDAyMTUyODI1WjAvBgkqhkiG9w0BCQQxIgQggKrT26fVls+s3aJbpPth
# xQVfuwJGnFkw1SZ8nRstswgwDQYJKoZIhvcNAQEBBQAEggIArYZS0pP2jGEx8i0L
# G8RWui4UOwxBVx6JQEMkIDIJURg7BEP/EHJRw70KQpYFEXpnKwlLBYLPa+Csj4OW
# 4xpYYW9AYWQKK6kCCj4DIdKOedSftykVhS495xXNEfogDACbuZ0z+5HJBrgVChcz
# fyGR845FUmUP5vjQ5e/t4ok1KRf4rN8300WWp8f8WLV5MXt62PQEowyAHsupn3/f
# sEMLXGG4QeNNR0ka5/jsUDcxN3lr20qiIJ+r4ZmYQI9XTQfF/JurJKoJiBqWqBVq
# YZ4hqVsNvuyJNKSXPT2uuHa/lMvmtARS9meEbCMaKwXTongsV8a2OpReIIzCeDQS
# r0O+PMiUJiZ2lYS0VAc0DW7gP929dgU7lYyY2Zbq8fse6C3DwFp8Sjt2moTa2ZBI
# xn77WH4k9z3D0yvqPzILRew+X+v9Kk6tfQ+22tbEeIDxCZAfu79CEhvKJUc9+3V2
# QjuLwJBckQ1/p20oekgXbQNlxT2VfLk3zmNLFoozeG1E9t8/P+mLGEEurhEmcZWA
# dkfLMlI8+aKDUmLT0gpVrO1aqtvI/vdAxMACvg7MgXUy5Xp1i8qn1HVAkIhYcOCf
# iq+kUFG7Did18JGg2lCdMTLk0Cxep5TMuG3zMAXhFsPlduaEPxweZWj/yT9msXTO
# OLYdk/Rc9pR5DQPkgjgcSyl0GYc=
# SIG # End signature block
