function Compare-NtapDocsData {
    <#
        .SYNOPSIS
            Creates a differences report from two outputs generated using the Format-Ntap*Data functions (using
            the -XmlFile parameter with the Out-NtapDocument function).

        .DESCRIPTION
            Compares output from two separate formatted NetAppDocs DataSets. The input for this function can be
            the output files from the Format functions that are saved using using the -XmlFile parameter of the
            Out-NtapDocument function. This will return a new DataSet highlighting any rows that have been added, deleted
            or modified. The updated columns in the modified rows will include a comment showing the original value.

        .EXAMPLE
            PS> Compare-NtapDocsData -DataSet1 'D:\XmlFile1.xml' -DataSet2 'D:\XmlFile2.xml' | Out-NtapDocument -ExcelFile 'D:\diff.xlsx'

            This example compares the data between the 2 formatted XML files and outputs the results to the specified file.

        .EXAMPLE
            PS> Compare-NtapDocsData -DataSet1 'D:\XmlFile1.xml' -DataSet2 'D:\XmlFile2.xml' -ChangesOnly | Out-NtapDocument -ExcelFile 'D:\diff_changes-only.xlsx'

            This example compares the data between the 2 formatted XML files and outputs only the changes to the specified file.

        .LINK
            https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs

        .INPUTS
            None

        .OUTPUTS
            System.Data.DataSet

        .NOTES
            AUTHOR : Jason Cole
    #>

    [CmdletBinding()]
    [OutputType([System.Data.DataSet])]
    param (
        # The first DataSet generated by the Out-NtapDocument function. This can be a path to a file or a DataSet object.
        [Parameter(Mandatory, Position = 0)]
        [ValidateScript({
                if ($_ -isnot [System.Data.DataSet]) {
                    if ($_ -is [System.IO.FileInfo] -or $_ -is [System.String]) {
                        if ([System.IO.Path]::GetExtension($_) -ne '.xml' -and [System.IO.Path]::GetExtension($_) -ne '.gz') {
                            throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]'Invalid file extension specified for the DataSet1 parameter.', 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $_))
                        }
                    }
                    else {
                        throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]'Invalid input object specified for the DataSet1 parameter.', 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $_))
                    }
                }

                $true
            })]
        [Alias('XmlFile1')]
        [System.Object]$DataSet1,

        # The second DataSet generated by the Out-NtapDocument function. This can be a path to a file or a DataSet object.
        [Parameter(Mandatory, Position = 1)]
        [ValidateScript({
                if ($_ -isnot [System.Data.DataSet]) {
                    if ($_ -is [System.IO.FileInfo] -or $_ -is [System.String]) {
                        if ([System.IO.Path]::GetExtension($_) -ne '.xml' -and [System.IO.Path]::GetExtension($_) -ne '.gz') {
                            throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]'Invalid file extension specified for the DataSet2 parameter.', 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $_))
                        }
                    }
                    else {
                        throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]'Invalid input object specified for the DataSet2 parameter.', 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $_))
                    }
                }

                $true
            })]
        [Alias('XmlFile2')]
        [System.Object]$DataSet2,

        # This parameter forces the order of the DataSet1 and DataSet2 parameters. As in, DataSet1 will be considered the
        # oldest data and DataSet2 will be considered the newest data. This is useful in scenarios where the earliest
        # raw data was formatted after the newest data was formatted. By default, the DataSet timestamps are used to
        # determine the order.
        [Parameter()]
        [switch]$ForceOrder,

        # When this parameter is specified, only the changed rows will be returned.
        [Parameter()]
        [switch]$ChangesOnly,

        # This parameter will prevent the output document from being generated when there are no differences found in any of the tables.
        [Parameter()]
        [switch]$SkipOutputOnNullDifferences
    )

    Write-Verbose -Message "Initializing function: $( $PSCmdlet.MyInvocation.MyCommand.Name ) ($( $PSCmdlet.MyInvocation.MyCommand.ModuleName ) v$( (Get-NtapDocsVersion -AsVersion).ToString() ))"

    $functionStartTime = [datetime]::Now

    try {
        $inputDataSet1 = Read-DataSet -InputObject $DataSet1
        $inputDataSet2 = Read-DataSet -InputObject $DataSet2
    }
    catch {
        $PSCmdlet.ThrowTerminatingError($_)
    }

    if ($inputDataSet1.ExtendedProperties['BaseTypeName'] -ne $inputDataSet2.ExtendedProperties['BaseTypeName'] -or $inputDataSet1.DataSetName -ne $inputDataSet2.DataSetName) {
        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('Both DataSets need to be the same system type and document type.', 'InvalidOperation', [System.Management.Automation.ErrorCategory]::InvalidOperation, $null)))
    }

    $inputDataSet1Version = Get-NtapDocsDataSetSchemaVersion -DataSet $inputDataSet1
    $inputDataSet2Version = Get-NtapDocsDataSetSchemaVersion -DataSet $inputDataSet2

    if ($inputDataSet1Version -lt [version]::Parse('3.2.0') -or $inputDataSet2Version -lt [version]::Parse('3.2.0')) {
        $msg = New-Object -TypeName System.Text.StringBuilder

        [void]$msg.AppendLine('An error was encountered while generating the differences report.')
        [void]$msg.AppendLine('-------------------------------------------------------------------------------------')
        [void]$msg.AppendLine('Due to some significant changes in the schemas for v3.2.0, it is not possible to')
        [void]$msg.AppendLine('generate a valid differences report for data formatted using NetAppDocs versions')
        [void]$msg.AppendLine('prior to v3.2.0. Please see the FAQ entry for more details.')
        [void]$msg.AppendLine('-------------------------------------------------------------------------------------')

        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), 'InvalidOperation', [System.Management.Automation.ErrorCategory]::InvalidOperation, $null)))
    }

    if ($inputDataSet1Version -ne $inputDataSet2Version) {
        Write-Warning -Message 'The DataSets were generated using different versions of NetAppDocs. This may cause some issues with the markup in the differences report.'
    }

    $inputDataSet1Timestamp = Convert-DateTimeString -String $inputDataSet1.ExtendedProperties['Timestamp']
    $inputDataSet2Timestamp = Convert-DateTimeString -String $inputDataSet2.ExtendedProperties['Timestamp']

    $originalDataSet, $updatedDataSet = $(
        if ($ForceOrder.IsPresent) {
            Write-Warning -Message 'ForceOrder parameters specified, so ignoring the timestamp in the DataSets.'

            $inputDataSet1, $inputDataSet2
        }
        else {
            if ($inputDataSet1Timestamp -lt $inputDataSet2Timestamp) {
                $inputDataSet1, $inputDataSet2
            }
            else {
                $inputDataSet2, $inputDataSet1
            }
        }
    )

    if ($originalDataSet.HasChanges()) {
        $originalDataSet.AcceptChanges()
    }

    if ($updatedDataSet.HasChanges()) {
        $updatedDataSet.AcceptChanges()
    }

    $baseTypeName = $updatedDataSet.ExtendedProperties['BaseTypeName']
    $documentType = $updatedDataSet.DataSetName -replace '^.+[.]'

    $dsSchemaFile = Get-NtapDocsSchemaFile -BaseTypeName $baseTypeName -DocumentType $documentType

    if (-not $dsSchemaFile) {
        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("An unexpected error occurred. DataSet schema file not found. SystemType: $baseTypeName#$documentType", 'FileNotFound', [System.Management.Automation.ErrorCategory]::InvalidResult, $dsSchemaFile)))
    }

    try {
        $outputDataSet = Read-DataSet -InputObject $dsSchemaFile
    }
    catch {
        $PSCmdlet.ThrowTerminatingError($_)
    }

    foreach ($dataTable in $outputDataSet.Tables) {
        Write-Verbose -Message "Processing table: $( $dataTable.TableName )"

        $performDiff = $true

        if ($dataTable.PrimaryKey.Count -eq 0) {
            $performDiff = $false
        }
        elseif (-not $originalDataSet.Tables.Contains($dataTable.TableName)) {
            $performDiff = $false

            $dataTable.ExtendedProperties.Add('DiffError', 'This table has been added since the oldest data was generated.')
        }
        elseif (-not $updatedDataSet.Tables.Contains($dataTable.TableName)) {
            $performDiff = $false

            if (-not $dataTable.ExtendedProperties.ContainsKey('RemoveTableIfEmpty')) {
                $dataTable.ExtendedProperties.Add('RemoveTableIfEmpty', $true)
            }
        }
        elseif ($updatedDataSet.Tables[$dataTable.TableName].ExtendedProperties.ContainsKey('DiffError')) {
            $performDiff = $false

            if ($originalDataSet.Tables[$dataTable.TableName].ExtendedProperties.ContainsKey('DiffError')) {
                $dataTable.ExtendedProperties.Add('DiffError', "Both tables contain errors. (Original table: $( $originalDataSet.Tables[$dataTable.TableName].ExtendedProperties['DiffError'] ) / Updated table: $( $updatedDataSet.Tables[$dataTable.TableName].ExtendedProperties['DiffError'] )).")
            }
            else {
                $dataTable.ExtendedProperties.Add('DiffError', "The newest table contained the following error: $( $updatedDataSet.Tables[$dataTable.TableName].ExtendedProperties['DiffError'] )" )
            }

            $dataTable.PrimaryKey | Where-Object { $updatedDataSet.Tables[$dataTable.TableName].Columns[$_.ColumnName].AllowDBNull } | ForEach-Object {
                $dataTable.Columns[$_.ColumnName].AllowDBNull = $true
            }

            $dataTable.PrimaryKey = $null
        }
        elseif ($originalDataSet.Tables[$dataTable.TableName].ExtendedProperties.ContainsKey('DiffError')) {
            $performDiff = $false

            $dataTable.ExtendedProperties.Add('DiffError', "The oldest table contained the following error: $( $originalDataSet.Tables[$dataTable.TableName].ExtendedProperties['DiffError'] )")
        }
        else {
            if (((Compare-Object -ReferenceObject $dataTable.PrimaryKey.ColumnName -DifferenceObject @( $originalDataSet.Tables[$dataTable.TableName].PrimaryKey.ColumnName | Out-String -Stream )) | Measure-Object).Count -gt 0) {
                Set-TablePrimaryKey -DataTable $originalDataSet.Tables[$dataTable.TableName] -PrimaryKey $dataTable.PrimaryKey.ColumnName -WarningAction SilentlyContinue

                $originalDataSet.Tables[$dataTable.TableName].ExtendedProperties['PrimaryKey'] = $dataTable.PrimaryKey.ColumnName -join ', '

                foreach ($dataColumn in $originalDataSet.Tables[$dataTable.TableName].Columns) {
                    if ($dataTable.PrimaryKey.Contains($dataColumn) -and $dataColumn.AllowDbNull) {
                        $dataColumn.AllowDbNull = $false
                    }
                    elseif (-not $dataTable.PrimaryKey.Contains($dataColumn) -and -not $dataColumn.AllowDbNull) {
                        $dataColumn.AllowDbNull = $true
                    }
                }

                if ($originalDataSet.Tables[$dataTable.TableName].ExtendedProperties.ContainsKey('DiffError')) {
                    $dataTable.ExtendedProperties.Add('DiffError', "Unable to update PrimaryKey on the oldest table. Error: $( $originalDataSet.Tables[$dataTable.TableName].ExtendedProperties['DiffError'] )")

                    $performDiff = $false
                }
            }

            if (((Compare-Object -ReferenceObject $dataTable.PrimaryKey.ColumnName -DifferenceObject @( $updatedDataSet.Tables[$dataTable.TableName].PrimaryKey.ColumnName | Out-String -Stream )) | Measure-Object).Count -gt 0) {
                Set-TablePrimaryKey -DataTable $updatedDataSet.Tables[$dataTable.TableName] -PrimaryKey $dataTable.PrimaryKey.ColumnName -WarningAction SilentlyContinue

                $updatedDataSet.Tables[$dataTable.TableName].ExtendedProperties['PrimaryKey'] = $dataTable.PrimaryKey.ColumnName -join ', '

                foreach ($dataColumn in $updatedDataSet.Tables[$dataTable.TableName].Columns) {
                    if ($dataTable.PrimaryKey.Contains($dataColumn) -and $dataColumn.AllowDbNull) {
                        $dataColumn.AllowDbNull = $false
                    }
                    elseif (-not $dataTable.PrimaryKey.Contains($dataColumn) -and -not $dataColumn.AllowDbNull) {
                        $dataColumn.AllowDbNull = $true
                    }
                }

                if ($updatedDataSet.Tables[$dataTable.TableName].ExtendedProperties.ContainsKey('DiffError')) {
                    $dataTable.ExtendedProperties.Add('DiffError', "Unable to update PrimaryKey on the newest table. Error: $( $updatedDataSet.Tables[$dataTable.TableName].ExtendedProperties['DiffError'] )")

                    $performDiff = $false
                }
            }
        }

        if ($performDiff) {
            try {
                $dataTable.Load($originalDataSet.Tables[$dataTable.TableName].CreateDataReader(), [System.Data.LoadOption]::Upsert)
                $dataTable.AcceptChanges()
            }
            catch {
                $Global:Error.RemoveAt(0)

                $dataTable.Clear()

                $performDiff = $false

                $dataTable.ExtendedProperties.Add('DiffError', "An error was encountered when loading the oldest data: $( $_.Exception.Message ).")
            }
        }

        if ($updatedDataSet.Tables.Contains($dataTable.TableName) -and $updatedDataSet.Tables[$dataTable.TableName].Rows.Count -gt 0) {
            try {
                $dataTable.Load($updatedDataSet.Tables[$dataTable.TableName].CreateDataReader(), [System.Data.LoadOption]::Upsert)
            }
            catch {
                $Global:Error.RemoveAt(0)

                $performDiff = $false

                if (-not $dataTable.ExtendedProperties.ContainsKey('DiffError')) {
                    $dataTable.ExtendedProperties.Add('DiffError', "An error was encountered when loading the newest data: $( $_.Exception.Message ).")
                }
            }
        }

        if ($sortedDataTable = [NetAppDocs.Utilities.DataSetMethods]::SortDataTable($dataTable.Copy(), [NetAppDocs.Utilities.SortMethod]::$( $updatedDataSet.ExtendedProperties['SortMethod'] ))) {
            $dataTable.ExtendedProperties.Keys | ForEach-Object {
                if (-not $sortedDataTable.ExtendedProperties.ContainsKey($_)) {
                    $sortedDataTable.ExtendedProperties.Add($_, $dataTable.ExtendedProperties[$_])
                }
            }

            if ($performDiff) {
                if (-not $dataTable.ExtendedProperties.Contains('DiffError')) {
                    $sortedDataTable.PrimaryKey = $dataTable.PrimaryKey | ForEach-Object {
                        $sortedDataTable.Columns[$_.ColumnName]
                    }
                }

                $sortedDataTable.AcceptChanges()

                if ($originalDataSet.Tables.Contains($dataTable.TableName) -and $dataTable.PrimaryKey.Count -gt 0) {
                    foreach ($sortedDataRow in $sortedDataTable.Rows) {
                        $combinedDataSetPrimaryKeys = $dataTable.PrimaryKey | ForEach-Object {
                            $sortedDataRow[$_.ColumnName]
                        }

                        $dataRow = $dataTable.Rows.Find($combinedDataSetPrimaryKeys)

                        if ($originalDataSet.Tables[$dataTable.TableName].Rows.Count -gt 0) {
                            $originalDataSetPrimaryKeys = $originalDataSet.Tables[$dataTable.TableName].PrimaryKey | ForEach-Object {
                                $dataRow[$_.ColumnName]
                            }

                            if (-not $updatedDataSet.Tables[$dataTable.TableName].Rows.Find($originalDataSetPrimaryKeys)) {
                                $dataRow.Delete()
                            }
                        }

                        switch -Exact ($dataRow.RowState) {
                            'Deleted' {
                                $sortedDataRow.SetModified()
                                $sortedDataRow.RowError = 'Deleted'
                            }
                            'Modified' {
                                $sortedDataRow.SetModified()

                                foreach ($dataColumn in $dataTable.Columns) {
                                    if ($dataColumn.ExtendedProperties.ContainsKey('DiffProperty') -and [System.Convert]::ToBoolean($dataColumn.ExtendedProperties['DiffProperty'])) {
                                        if ($dataRow[$dataColumn.ColumnName, [System.Data.DataRowVersion]::Original] -ne $dataRow[$dataColumn.ColumnName, [System.Data.DataRowVersion]::Current]) {
                                            $sortedDataRow.SetColumnError($dataColumn.ColumnName, ($dataRow[$dataColumn.ColumnName, [System.Data.DataRowVersion]::Original]))

                                            if ($sortedDataRow.RowError -ne 'Modified') {
                                                $sortedDataRow.RowError = 'Modified'
                                            }
                                        }
                                    }
                                }

                                if ($sortedDataRow.RowError -ne 'Modified') {
                                    $sortedDataRow.AcceptChanges()
                                }
                            }
                            'Added' {
                                $sortedDataRow.SetAdded()
                                $sortedDataRow.RowError = $dataRow.RowState
                            }
                        }
                    }
                }
            }

            $outputDataSet.Tables[$dataTable.TableName].Merge($sortedDataTable)
        }
    }

    $skipOutput = $false

    if (($outputDataSet.Tables | Where-Object { $_.PrimaryKey -and -not $_.ExtendedProperties.ContainsKey('DiffError') -and $_.GetChanges() } | Measure-Object).Count -eq 0) {
        $warningMsg = $(
            if ($SkipOutputOnNullDifferences.IsPresent) {
                $skipOutput = $true

                'No differences found between the two DataSets. No output will be generated (-SkipOutputOnNullDifferences specified).'
            }
            else {
                'No differences found between the two DataSets.'
            }
        )

        Write-Warning -Message $warningMsg
    }

    if (-not $skipOutput) {
        $outputDataSet.ExtendedProperties.Add('DiffReport', $true)

        if ($ChangesOnly.IsPresent) {
            $outputDataSet.ExtendedProperties.Add('ChangesOnly', $true)

            $outputDataSet.GetChanges()
        }
        else {
            $outputDataSet.ExtendedProperties.Add('ChangesOnly', $false)

            $outputDataSet
        }
    }

    $statsParams = @{
        FunctionPSCallStack = Get-PSCallStack
        ElapsedTicks        = ([datetime]::Now - $functionStartTime).Ticks
        ExcludeParameter    = 'DataSet1', 'DataSet2'
    }

    Write-Statistics @statsParams
}

# SIG # Begin signature block
# MIIq0AYJKoZIhvcNAQcCoIIqwTCCKr0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCkB8mw8Z+G9+H5
# W3OYBe/aOQj/M3AuFVs6wav2miQmEqCCJGgwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggWNMIIEdaADAgECAhAOmxiO
# +dAt5+/bUOIIQBhaMA0GCSqGSIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYD
# VQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAi
# BgNVBAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAw
# MDBaFw0zMTExMDkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
# Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERp
# Z2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBAL/mkHNo3rvkXUo8MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsb
# hA3EMB/zG6Q4FutWxpdtHauyefLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iT
# cMKyunWZanMylNEQRBAu34LzB4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGb
# NOsFxl7sWxq868nPzaw0QF+xembud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclP
# XuU15zHL2pNe3I6PgNq2kZhAkHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCr
# VYJBMtfbBHMqbpEBfCFM1LyuGwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFP
# ObURWBf3JFxGj2T3wWmIdph2PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTv
# kpI6nj3cAORFJYm2mkQZK37AlLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWM
# cCxBYKqxYxhElRp2Yn72gLD76GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls
# 5Q5SUUd0viastkF13nqsX40/ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBR
# a2+xq4aLT8LWRV+dIPyhHsXAj6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6
# MIIBNjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qY
# rhwPTzAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8E
# BAMCAYYweQYIKwYBBQUHAQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5k
# aWdpY2VydC5jb20wQwYIKwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDig
# NoY0aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNybDARBgNVHSAECjAIMAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCg
# v0NcVec4X6CjdBs9thbX979XB72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQT
# SnovLbc47/T/gLn4offyct4kvFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh
# 65ZyoUi0mcudT6cGAxN3J0TU53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSw
# uKFWjuyk1T3osdz9HNj0d1pcVIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAO
# QGPFmCLBsln1VWvPJ6tsds5vIy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjD
# TZ9ztwGpn1eqXijiuZQwggXMMIIENKADAgECAhAg429sPxgagUb53pPffJfkMA0G
# CSqGSIb3DQEBDAUAMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBS
# MzYwHhcNMjEwOTA5MDAwMDAwWhcNMjMwOTA5MjM1OTU5WjBQMQswCQYDVQQGEwJV
# UzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UECgwMTmV0QXBwLCBJbmMuMRUw
# EwYDVQQDDAxOZXRBcHAsIEluYy4wggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGK
# AoIBgQC4kYYj/oViZD9pN03hrqFjtwOz1Gx4eDWVV8IYUYEr2qpLeNYvWz26B/+E
# mYLboAGVpkYg0Wske3hjRpooZlMORm1S4+2C2NoQCstZ+kmNTevbXHJe/w1VNJrm
# fKjpXOJEfx8GUGDqCMR30dChVpsdBEavrM7T0hnfJLv18i19SGv3a+nhvB3iOjLP
# SEg8+DFkjD5jTI6fQ83DBEiVR4UEYq7yyH578ZEylhsRfJmK+uIREycEUk/NpFTw
# g6/7lxh/WYabCrZzqI4Ep3QataRKaAlxt3BiOlZkX4WGX3DYvSYltWOM2UlCcGpP
# m/1/LN8Xuvf+YO6H472QecHe59XtXbHSMKvqDuOEJu8Wk2LFMNK732o3Fc5QIHie
# 6JtNsgNYf+Vsm5EMBD1ZHNE+C3zlmQbbWIU5uEU1nhiodBPKsu0SkS6pARyKBO05
# DSc2gWWv0aukKi04ZAn+hAdSl5s1dgmU5shBvKHqe15K9ZRN4FFO/CgHE0BoGYQS
# UQVKwa0CAwEAAaOCAZwwggGYMB8GA1UdIwQYMBaAFA8qyyCHKLjsb0iuK1SmKaoX
# pM0MMB0GA1UdDgQWBBQuH643KcBMmb/Q6IZt+H9IrnXFwDAOBgNVHQ8BAf8EBAMC
# B4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDAzARBglghkgBhvhC
# AQEEBAMCBBAwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwIwJTAjBggrBgEFBQcC
# ARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQBMEkGA1UdHwRCMEAw
# PqA8oDqGOGh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVT
# aWduaW5nQ0FSMzYuY3JsMHkGCCsGAQUFBwEBBG0wazBEBggrBgEFBQcwAoY4aHR0
# cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQVIz
# Ni5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqG
# SIb3DQEBDAUAA4IBgQCOoGdXjP8Sif0h3ZvoDsIVfrJvQUdP9pZExRJGGj/Te6ML
# XyqHNc/G2WklNv+BC4ryNJ4zp0oneHOub55ejrYPfhgP9oFWS8oOPJbrp3rAtINa
# OeKRC88WUYZSKYAxSX6kubEjnQD6cmazXjxVN6P8+6q9smeqF3xI+++c+ekMw3Xv
# 4EWRYXqN0/srfFp1WpoluwSQcZMqj9haSX0bgO6AAWH2AnVJMfXTIKz/0FQ/RW0y
# Ed5QYQqYInhA7IUz9dBziNpJlwONeF5j7xzgfYDY63WU6WrgJnGjYkQCOOsUx74j
# gUiMRe9zV48GS8Vxz22c/TQs0x4/1GmOSJvUHEk3GseBmB3v+yEhh/D6zWOfYP4X
# D/9b91CxmugTuepuaJSeRg+qUm3KarKsOCUF/CLqUrNh/JwKrWD1cghRaYLvMucs
# ScksHHe7ZDvb2OtvxWXjPk1d1NKvEwFJSS6hIVIbug9x28AJqOoP5Pn9wZvJSzvW
# uQJGCLscdGyKefdCo30wggYaMIIEAqADAgECAhBiHW0MUgGeO5B5FSCJIRwKMA0G
# CSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290
# IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFQxCzAJBgNVBAYT
# AkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28g
# UHVibGljIENvZGUgU2lnbmluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEBAQUAA4IB
# jwAwggGKAoIBgQCbK51T+jU/jmAGQ2rAz/V/9shTUxjIztNsfvxYB5UXeWUzCxEe
# AEZGbEN4QMgCsJLZUKhWThj/yPqy0iSZhXkZ6Pg2A2NVDgFigOMYzB2OKhdqfWGV
# oYW3haT29PSTahYkwmMv0b/83nbeECbiMXhSOtbam+/36F09fy1tsB8je/RV0mIk
# 8XL/tfCK6cPuYHE215wzrK0h1SWHTxPbPuYkRdkP05ZwmRmTnAO5/arnY83jeNzh
# P06ShdnRqtZlV59+8yv+KIhE5ILMqgOZYAENHNX9SJDm+qxp4VqpB3MV/h53yl41
# aHU5pledi9lCBbH9JeIkNFICiVHNkRmq4TpxtwfvjsUedyz8rNyfQJy/aOs5b4s+
# ac7IH60B+Ja7TVM+EKv1WuTGwcLmoU3FpOFMbmPj8pz44MPZ1f9+YEQIQty/NQd/
# 2yGgW+ufflcZ/ZE9o1M7a5Jnqf2i2/uMSWymR8r2oQBMdlyh2n5HirY4jKnFH/9g
# Rvd+QOfdRrJZb1sCAwEAAaOCAWQwggFgMB8GA1UdIwQYMBaAFDLrkpr/NZZILyhA
# QnAgNpFcF4XmMB0GA1UdDgQWBBQPKssghyi47G9IritUpimqF6TNDDAOBgNVHQ8B
# Af8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcD
# AzAbBgNVHSAEFDASMAYGBFUdIAAwCAYGZ4EMAQQBMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Um9vdFI0Ni5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ1Jvb3RSNDYu
# cDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQwFAAOCAgEABv+C4XdjNm57oRUgmxP/BP6YdURhw1aVcdGRP4Wh60BAscjW
# 4HL9hcpkOTz5jUug2oeunbYAowbFC2AKK+cMcXIBD0ZdOaWTsyNyBBsMLHqafvIh
# rCymlaS98+QpoBCyKppP0OcxYEdU0hpsaqBBIZOtBajjcw5+w/KeFvPYfLF/ldYp
# mlG+vd0xqlqd099iChnyIMvY5HexjO2AmtsbpVn0OhNcWbWDRF/3sBp6fWXhz7Dc
# ML4iTAWS+MVXeNLj1lJziVKEoroGs9Mlizg0bUMbOalOhOfCipnx8CaLZeVme5yE
# Lg09Jlo8BMe80jO37PU8ejfkP9/uPak7VLwELKxAMcJszkyeiaerlphwoKx1uHRz
# NyE6bxuSKcutisqmKL5OTunAvtONEoteSiabkPVSZ2z76mKnzAfZxCl/3dq3dUNw
# 4rg3sTCggkHSRqTqlLMS7gjrhTqBmzu1L90Y1KWN/Y5JKdGvspbOrTfOXyXvmPL6
# E52z1NZJ6ctuMFBQZH3pwWvqURR8AgQdULUvrxjUYbHHj95Ejza63zdrEcxWLDX6
# xWls/GDnVNueKjWUH3fTv1Y8Wdho698YADR7TNx8X8z2Bev6SivBBOHY+uqiirZt
# g0y9ShQoPzmCcn63Syatatvx157YK9hlcPmVoa1oDE5/L9Uo2bC5a4CH2Rwwggau
# MIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJ
# BgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5k
# aWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAe
# Fw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcw
# FQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3Rl
# ZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9Ge
# TKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0
# hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZl
# jZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAsh
# aG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVY
# TXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1
# biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCir
# c0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+
# DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA
# +bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42Pg
# puE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzS
# M7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU
# uhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6
# mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsG
# AQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29t
# MEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNl
# cnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3Js
# My5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAE
# GTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1Z
# jsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8d
# B+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVp
# P0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp8
# 76i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2
# nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3
# ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQ
# txMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc
# 4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+Y
# AN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZ
# vAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQr
# H4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIGwDCCBKigAwIBAgIQDE1p
# ckuU+jwqSj0pB4A9WjANBgkqhkiG9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4XDTIyMDkyMTAwMDAw
# MFoXDTMzMTEyMTIzNTk1OVowRjELMAkGA1UEBhMCVVMxETAPBgNVBAoTCERpZ2lD
# ZXJ0MSQwIgYDVQQDExtEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMiAtIDIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDP7KUmOsap8mu7jcENmtuh6BSFdDMa
# JqzQHFUeHjZtvJJVDGH0nQl3PRWWCC9rZKT9BoMW15GSOBwxApb7crGXOlWvM+xh
# iummKNuQY1y9iVPgOi2Mh0KuJqTku3h4uXoW4VbGwLpkU7sqFudQSLuIaQyIxvG+
# 4C99O7HKU41Agx7ny3JJKB5MgB6FVueF7fJhvKo6B332q27lZt3iXPUv7Y3UTZWE
# aOOAy2p50dIQkUYp6z4m8rSMzUy5Zsi7qlA4DeWMlF0ZWr/1e0BubxaompyVR4aF
# eT4MXmaMGgokvpyq0py2909ueMQoP6McD1AGN7oI2TWmtR7aeFgdOej4TJEQln5N
# 4d3CraV++C0bH+wrRhijGfY59/XBT3EuiQMRoku7mL/6T+R7Nu8GRORV/zbq5Xwx
# 5/PCUsTmFntafqUlc9vAapkhLWPlWfVNL5AfJ7fSqxTlOGaHUQhr+1NDOdBk+lbP
# 4PQK5hRtZHi7mP2Uw3Mh8y/CLiDXgazT8QfU4b3ZXUtuMZQpi+ZBpGWUwFjl5S4p
# kKa3YWT62SBsGFFguqaBDwklU/G/O+mrBw5qBzliGcnWhX8T2Y15z2LF7OF7ucxn
# EweawXjtxojIsG4yeccLWYONxu71LHx7jstkifGxxLjnU15fVdJ9GSlZA076XepF
# cxyEftfO4tQ6dwIDAQABo4IBizCCAYcwDgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB
# /wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwIAYDVR0gBBkwFzAIBgZngQwB
# BAIwCwYJYIZIAYb9bAcBMB8GA1UdIwQYMBaAFLoW2W1NhS9zKXaaL3WMaiCPnshv
# MB0GA1UdDgQWBBRiit7QYfyPMRTtlwvNPSqUFN9SnDBaBgNVHR8EUzBRME+gTaBL
# hklodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0
# MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3JsMIGQBggrBgEFBQcBAQSBgzCBgDAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFgGCCsGAQUFBzAC
# hkxodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRS
# U0E0MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3J0MA0GCSqGSIb3DQEBCwUAA4IC
# AQBVqioa80bzeFc3MPx140/WhSPx/PmVOZsl5vdyipjDd9Rk/BX7NsJJUSx4iGNV
# CUY5APxp1MqbKfujP8DJAJsTHbCYidx48s18hc1Tna9i4mFmoxQqRYdKmEIrUPwb
# tZ4IMAn65C3XCYl5+QnmiM59G7hqopvBU2AJ6KO4ndetHxy47JhB8PYOgPvk/9+d
# EKfrALpfSo8aOlK06r8JSRU1NlmaD1TSsht/fl4JrXZUinRtytIFZyt26/+YsiaV
# OBmIRBTlClmia+ciPkQh0j8cwJvtfEiy2JIMkU88ZpSvXQJT657inuTTH4YBZJwA
# wuladHUNPeF5iL8cAZfJGSOA1zZaX5YWsWMMxkZAO85dNdRZPkOaGK7DycvD+5sT
# X2q1x+DzBcNZ3ydiK95ByVO5/zQQZ/YmMph7/lxClIGUgp2sCovGSxVK05iQRWAz
# gOAj3vgDpPZFR+XOuANCR+hBNnF3rf2i6Jd0Ti7aHh2MWsgemtXC8MYiqE+bvdgc
# mlHEL5r2X6cnl7qWLoVXwGDneFZ/au/ClZpLEQLIgpzJGgV8unG1TnqZbPTontRa
# mMifv427GFxD9dAq6OJi7ngE273R+1sKqHB+8JeEeOMIA11HLGOoJTiXAdI/Otrl
# 5fbmm9x+LMz/F0xNAKLY1gEOuIvu5uByVYksJxlh9ncBjDGCBb4wggW6AgEBMGgw
# VDELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDErMCkGA1UE
# AxMiU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIFIzNgIQIONvbD8YGoFG
# +d6T33yX5DANBglghkgBZQMEAgEFAKCBhDAYBgorBgEEAYI3AgEMMQowCKACgACh
# AoAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAM
# BgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCD0cSFSn//3rfK59cqWOdpTrpQ0
# 5KNDWZaFlsJwNJTo6DANBgkqhkiG9w0BAQEFAASCAYAMZS1t5UjMTNm+uHvDZh1w
# B3iGsI/UVYlFALOKFx/c/DJlcblj36/kKZJSxvmgCnpOS++u2+WzUPh4/cL6vyAg
# rMXUy8yRDfZRmVhD0QjM4cmo2F3eqIRtPDboeMkfiNBurWsnYWbktficcKjrygsn
# xkSalhS3CMJ9SW3Lxq4G5CQq21N8QnXQ433AYwfR7V+ldyT0y38r31WUcq8PnBed
# qtvzHq8KX0CP0ZSoF3yEt+7NFTG6JdcMqdDKf/V3SSPKXw/9o+tZMVKOxmqnPLtX
# ytl5x9N313q83hDZoKs/yZeBaun+rOItkHYIeWHLpBrOZpUnQdsruw8sEpYHwHVV
# lRdyrc1tmAojU+MCoKJ/NxlI6ugMgMVa+J22G0tjY06SwoxL5o/CZATDGerrGq3q
# r/jTO2cI2gPjJxpT92QQOVAhnQnE5Zy1NCVdJfaqsXodLC8cfwqqQ/n4dg0bi8i2
# +xsoXqLfoXLIA+k7OW/SOo7rMpXG7UAmHSFWZmwk2CqhggMgMIIDHAYJKoZIhvcN
# AQkGMYIDDTCCAwkCAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
# cnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
# SEEyNTYgVGltZVN0YW1waW5nIENBAhAMTWlyS5T6PCpKPSkHgD1aMA0GCWCGSAFl
# AwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUx
# DxcNMjMwNDAyMTUyODI2WjAvBgkqhkiG9w0BCQQxIgQgzPT6YZ95/3KwbAvkFUFJ
# Otx1Ix/CFyDjoe+fkChhJqAwDQYJKoZIhvcNAQEBBQAEggIAD3vkjDZ3M/i1c5lI
# KZRqVSlN0kni8geLGhxf8MULChpOKQcjEzXWQwDGKMiA6vAECAJ+U+FzEYa9kwlF
# d14uupjD4jpHAuO8qPIva2xTTTXZLy8YwYPnBvrU751RI+Hb5UY8J+fabD6gij3L
# 7zzz+PDoMWF56x8jDk0xfZQadLvbxVIO2p4yYQijvFKeb7ZRzVSddcSqGTNl7MpK
# lDUPWE1nTs9q7C9m0wVj05YJXQ/agWqrUoO4SoTLsk3MF2029BvEC8kVGJ6Hj9fe
# 6EwTdqANI7rhOYgq2M3Pt7RInicWFg8nY9ifvaejUT7FDx9QfdKRd85KFKRCxuMM
# xY+sRmxgauNK/ukjABdLaJJ/2ErkHinrJGP/EhNN4y/Fvjkj/tmyuzE310CyDWZw
# lM0zKgqwdAU2z2hH2b0RgZxz4N3Cju4FqAKspwPiYNyLMsp39kqQbVfcpcIttDZH
# lVmVrRsu0jZOkRtILLXEtKFfn2gD7KxdwWylzfd8aiiiwZIBEVuBiCVxHdF30p9I
# rlSJPObDuuTUZestrptqUymf706WL+d2uDTcFoeaMCgimNuJZDG9NqGR2FloEhHA
# 2zUihtKXroZE4/6ur8vwy9DhSKZKTplGrvrEUdt8Ckge5rwjxe3rgiVPnCeFryUf
# QL+wKzR6Ujhc6C3frJBbURrn/Cs=
# SIG # End signature block
