function ConvertTo-NtapVserverData {
    <#
        .SYNOPSIS
            Converts ONTAP data from the Get-NtapClusterData function to SVM specific data.

        .DESCRIPTION
            Produces SVM specific data from the raw output generated by Get-NtapClusterData. This is useful for collecting the
            data one time from the cluster and generating both the cluster documentation as well as the SVM specific documentation.

        .EXAMPLE
            PS> Get-ChildItem -Path 'D:\Output\cluster1.xml' | ConvertTo-NtapVserverData | Format-NtapVserverData | Out-NtapDocument -WordFile 'D:\Output\cluster1-AllSVMs.docx'

            Reads the raw cluster data from the file cluster1.xml (generated via Get-NtapClusterData) and outputs the data for
            all the data SVMs to the Word document specified.

        .EXAMPLE
            PS> Get-ChildItem -Path 'D:\Output\cluster1.xml' | ConvertTo-NtapVserverData -VserverName vs_nfs, vs_cifs | Format-NtapVserverData | Out-NtapDocument -WordFile 'D:\Output\SVMs.docx'

            Reads the raw cluster data from the file cluster1.xml (generated via Get-NtapClusterData) and outputs the data for
            the SVMs 'vs_nfs' and 'vs_cifs' to the single Word document specified.

        .EXAMPLE
            PS> Get-ChildItem -Path 'D:\Output\cluster1.xml' | ConvertTo-NtapVserverData -VserverName vs_nfs, vs_cifs | ForEach-Object { $_ | Format-NtapVserverData | Out-NtapDocument -WordFile "D:\Output\SVM-$( $_.SystemName ).docx" }

            Reads the raw cluster data from the file cluster1.xml (generated via Get-NtapClusterData) and outputs the data for
            the SVMs 'vs_nfs' and 'vs_cifs' to separate Word documents.

        .EXAMPLE
            PS> Get-NtapClusterData -Name cluster1 | ConvertTo-NtapVserverData | Format-NtapVserverData | Out-NtapDocument -WordFile 'D:\Output\cluster1-AllSVMs.docx'

            Collects the data from the cluster 'cluster1' and outputs the information for all the data SVMs to the single
            document specified. This use case is not common, since you'd probably use Get-NtapVserverData instead.

        .EXAMPLE
            PS> Get-ChildItem -Path 'D:\Output\cluster1.xml' | ConvertTo-NtapVserverData | Format-NtapVserverData -SanitizeLevel 1 | Out-NtapDocument -WordFile 'D:\Output\cluster1-AllSVMs.docx'

            Reads the raw cluster data from the file cluster1.xml (generated via Get-NtapClusterData) and outputs the data for
            all the data SVMs to the Word document specified. Additionally, this will sanitize the IP address information.

        .LINK
            https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs

        .LINK
            Get-NtapClusterData

        .LINK
            Format-NtapVserverData

        .INPUTS
            System.Management.Automation.PSCustomObject

        .INPUTS
            System.IO.FileInfo

        .INPUTS
            System.String

        .OUTPUTS
            System.Management.Automation.PSCustomObject

        .NOTES
            AUTHOR : Jason Cole
    #>

    [CmdletBinding()]
    [OutputType([System.Management.Automation.PSCustomObject])]
    param (
        # An Xml file created by the Get-NtapClusterData function (can also be *.xml.gz). Also accepts pipeline input from
        # the Get-NtapClusterData function.
        [Parameter(Mandatory, Position = 0, ValueFromPipeline)]
        [ValidateScript({
                if ($_ -isnot [System.Management.Automation.PSCustomObject]) {
                    if ($_ -is [System.IO.FileInfo] -or $_ -is [System.String]) {
                        if ([System.IO.Path]::GetExtension($_) -ne '.xml' -and [System.IO.Path]::GetExtension($_) -ne '.gz') {
                            throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]'Invalid file extension specified for the InputObject parameter.', 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $_))
                        }
                    }
                    else {
                        throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]'Invalid input object specified.', 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $_))
                    }
                }

                $true
            })]
        [System.Object]$InputObject,

        # The list of data SVM names to include in the output. If blank, all data SVMs will be included.
        [Parameter()]
        [Alias('SvmName')]
        [string[]]$VserverName
    )

    process {
        Write-Verbose -Message "Initializing function: $( $PSCmdlet.MyInvocation.MyCommand.Name ) ($( $PSCmdlet.MyInvocation.MyCommand.ModuleName ) v$( (Get-NtapDocsVersion -AsVersion).ToString() ))"

        if ($InputObject -is [System.Management.Automation.PSCustomObject]) {
            Write-Verbose -Message 'Reading input data from pipeline'

            $inputData = $InputObject

            if (-not $inputData) {
                Show-ErrorMessage -Message 'No data found in input object.'
            }
        }
        else {
            try {
                $inputFileInfo = $(
                    if ($InputObject -is [System.IO.FileInfo]) {
                        [System.IO.FileInfo]$ExecutionContext.SessionState.Path.GetResolvedPSPathFromPSPath($InputObject.FullName).ProviderPath
                    }
                    else {
                        [System.IO.FileInfo]$ExecutionContext.SessionState.Path.GetResolvedPSPathFromPSPath($InputObject).ProviderPath
                    }
                )
            }
            catch {
                $Global:Error.RemoveAt(0)

                Write-Error -ErrorRecord (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]'Input file specified does not exist.', $_.FullyQualifiedErrorId, [System.Management.Automation.ErrorCategory]::InvalidArgument, $InputObject))

                return
            }

            if ($inputFileInfo.Extension -eq '.gz') {
                Write-Verbose -Message "Expanding FileName: $( $inputFileInfo.FullName )"

                $inputXmlFile = Expand-GzipFile -InFile $inputFileInfo.FullName -PassThru
            }
            else {
                $inputXmlFile = $inputFileInfo
            }

            Write-Verbose -Message "Reading input data from FileName: $( $inputXmlFile.Name )"

            try {
                $inputData = Import-Clixml -Path $inputXmlFile.FullName -ErrorAction Stop

                if (-not $inputData) {
                    Show-ErrorMessage -Message "No data found in input file: $( $inputFileInfo.FullName )"
                }
            }
            catch [System.Xml.XmlException] {
                $Global:Error.RemoveAt(0)

                Write-Warning -Message "Data in the XML file has already been processed by this $( $PSCmdlet.MyInvocation.MyCommand.Name ) function. File: $( $inputXmlFile.FullName )"

                return
            }
            catch {
                $Global:Error.RemoveAt(0)

                Write-Error -ErrorRecord (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("There was an error reading the XML file. Reason: $( $_.Exception.Message )", $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $inputXmlFile.FullName))

                return
            }
            finally {
                if ($inputFileInfo.Extension -eq '.gz') {
                    Remove-Item -Path $inputXmlFile.FullName -ErrorAction SilentlyContinue
                }
            }
        }

        foreach ($clusterData in $inputData) {
            if ($null -eq $clusterData) {
                continue
            }

            $baseTypeName = $clusterData.pstypenames[0] -replace 'Deserialized.' -replace '.Data'

            if ($baseTypeName -notlike 'NetAppDocs.NcController*') {
                Show-ErrorMessage -Message 'ConvertTo-NtapVserverData only supports data generated with the Get-NtapClusterData or the Get-NtapClusterAsupData functions.'

                continue
            }

            $dataCollectVersion = ($clusterData.NetAppDocsVersion, $clusterData.NetAppDocsVersionDataCollection)[[byte](! ($clusterData.psobject.Properties.Name -contains 'NetAppDocsVersion'))]

            if ((Get-NtapDocsVersion -AsFeatureVersion) -ne [version]::Parse((Get-VersionString -InputVersion $dataCollectVersion -AsFeatureVersion))) {
                Write-Warning -Message "The NetAppDocs version ($( (Get-NtapDocsVersion -AsVersion).ToString() )) being used to convert the data is not the same as the version ($( $dataCollectVersion.ToString() )) used to collect the data for this system: $( $clusterData.SystemName ). It is recommended that the versions be the same, as there could be missing or incomplete data."
            }

            if ($clusterData.System.OntapiVersion -isnot [System.Version]) {
                $clusterData.System.OntapiVersion = [version]::Parse($clusterData.System.OntapiVersion.ToString())
            }

            Write-Verbose -Message "Converting input data for SystemName: $( $clusterData.SystemName )"

            $allVservers = $clusterData.Vserver.Info | Where-Object { $_.VserverType -in @( 'data', 'cluster' ) -or $_.Type -in @( 'data', 'cluster' ) }

            if ($baseTypeName -eq 'NetAppDocs.NcController.Asup') {
                $allVservers | ForEach-Object { $_ | Add-Member -MemberType NoteProperty -Name VserverName -Value $_.Vserver -ErrorAction Ignore }
            }

            if ($VserverName) {
                $diffObject = Compare-Object -ReferenceObject $allVservers.Vserver -DifferenceObject $VserverName -IncludeEqual

                ($diffObject | Where-Object { $_.SideIndicator -eq '=>' }).InputObject | Where-Object { $_ } | ForEach-Object {
                    Write-Warning -Message "Vserver does not exist in cluster: $_"
                }

                $vservers = $allVservers | Where-Object { $_.Vserver -cin ($diffObject | Where-Object { $_.SideIndicator -eq '==' }).InputObject }
            }
            else {
                $vservers = $allVservers
            }

            foreach ($vs in $vservers) {
                try {
                    $functionStartTime = [datetime]::Now

                    $ntapData = Convert-VserverData -ClusterData $clusterData -VserverName $vs.VserverName -VserverUuid $vs.Uuid

                    $ntapData

                    $statsParams = @{
                        FunctionPSCallStack = Get-PSCallStack
                        TargetDetails       = [pscustomobject]@{
                            SystemUid     = [string]::Join('_', ($ntapData.Cluster.Info.ClusterUuid, $ntapData.Cluster.Info.ClusterIdentifier)[[byte](! ($ntapData.pstypenames[0] -like 'NetAppDocs.NcVserver*'))], $ntapData.Vserver.Info.Uuid)
                            SystemType    = $ntapData.pstypenames[0]
                            SystemVersion = ([regex]::Split(([regex]::Split($ntapData.System.VersionInfo.Version, 'Release\s(?:\w+[__])?')[1]), '\s+'))[0].Replace(':', '')
                        }
                        ElapsedTicks        = ([datetime]::Now - $functionStartTime).Ticks
                        ExcludeParameter    = 'VserverName'
                    }

                    Write-Statistics @statsParams
                }
                catch {
                    Write-Error -ErrorRecord $_
                }
            }
        }
    }
}

function Convert-VserverData {
    [CmdletBinding()]
    param (
        [Parameter(Mandatory)]
        [ValidateNotNullOrEmpty()]
        [pscustomobject]$ClusterData,

        [Parameter(Mandatory)]
        [ValidateNotNullOrEmpty()]
        [string]$VserverName,

        [Parameter(Mandatory)]
        [ValidateNotNullOrEmpty()]
        [string]$VserverUuid
    )

    $baseTypeName = $ClusterData.pstypenames[0] -replace 'Deserialized.' -replace '.Data'

    $vserverData = [pscustomobject]@{ }
    $vserverData.pstypenames.Insert(0, "$( $baseTypeName -replace 'NcController', 'NcVserver' )")

    Write-Verbose -Message "- Converting data for Vserver: $VserverName"

    Write-Verbose -Message '- Identifying module version data -'
    $vserverData | Add-Member -MemberType NoteProperty -Name NetAppDocsVersion -Value $ClusterData.NetAppDocsVersion

    Write-Verbose -Message '- Identifying PowerShell version data -'
    $vserverData | Add-Member -MemberType NoteProperty -Name PowerShellVersion -Value $ClusterData.PowerShellVersion

    try {
        if ($baseTypeName -eq 'NetAppDocs.NcController.Asup') {
            Write-Verbose -Message '- Identifying ASUP information -'
            $vserverData | Add-Member -MemberType NoteProperty -Name SourceAsup -Value $ClusterData.SourceAsup
        }

        #region System/Node Information

        Write-Verbose -Message '- Converting basic system information -'
        $vserverData | Add-Member -MemberType NoteProperty -Name System -Value ($ClusterData.System | Select-Object -Property VersionInfo, OntapiVersion)

        Write-Verbose -Message '- Converting node information -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Node -Value $ClusterData.Node

        #endregion System/Node Information

        #region Timezone Information

        Write-Verbose -Message '- Converting timezone information'
        $vserverData | Add-Member -MemberType NoteProperty -Name Timezone -Value $ClusterData.Timezone

        #endregion Timezone Information

        #region Cluster Information

        Write-Verbose -Message '- Converting cluster information -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Cluster -Value ([pscustomobject]@{ })

        $vserverData.Cluster | Add-Member -MemberType NoteProperty -Name Info -Value $ClusterData.Cluster.Info

        #endregion Cluster Information

        #region Options Information

        Write-Verbose -Message '- Converting options -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Option -Value ($ClusterData.Option | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        #endregion Options Information

        #region Vserver Information

        Write-Verbose -Message '- Converting Vserver information -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Vserver -Value ([pscustomobject]@{ })

        $vserverData.Vserver | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.Vserver.Info | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName -or $_.VserverName -ceq $VserverName })

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.31')) {
            #Not in ASUP data
            $vserverData.Vserver | Add-Member -MemberType NoteProperty -Name LoginBanner -Value ($ClusterData.Vserver.LoginBanner | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            #Not in ASUP data
            $vserverData.Vserver | Add-Member -MemberType NoteProperty -Name Motd -Value ($ClusterData.Vserver.Motd | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        #endregion Vserver Information

        #region Vserver Peer Information

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Converting Vserver Peer information -'
            $vserverData.Vserver | Add-Member -MemberType NoteProperty -Name Peer -Value ([pscustomobject]@{ })

            $vserverData.Vserver.Peer | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.Vserver.Peer.Info | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName -or $_.LocalVserverName -ceq $VserverName })
        }

        #endregion Vserver Peer Information

        #region Volume Information

        Write-Verbose -Message '- Converting detailed volume information -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Vol -Value ([pscustomobject]@{ })

        $vserverData.Vol | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.Vol.Info | Where-Object { $_.VolumeIdAttributes.OwningVserverName -ceq $VserverName -or $_.Vs -ceq $VserverName })
        if ($ClusterData.Vol | Get-Member -MemberType NoteProperty -Name CloneVol) {
            $vserverData.Vol | Add-Member -MemberType NoteProperty -Name CloneVol -Value ($ClusterData.Vol.CloneVol | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }
        else {
            $vserverData.Vol | Add-Member -MemberType NoteProperty -Name Clone -Value ($ClusterData.Vol.Clone | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }
        $vserverData.Vol | Add-Member -MemberType NoteProperty -Name Space -Value $(
            if ($baseTypeName -eq 'NetAppDocs.NcController.Asup') {
                foreach ($vol in $vserverData.Vol.Info) {
                    $ClusterData.Vol.Space | Where-Object {
                        $_.VolumeUuid -eq $vol.VolUuid -and $_.InstanceUuid -eq $vol.Iuuid -and $_.ProvenanceUuid -eq $vol.Puuid
                    }
                }
            }
            else {
                $ClusterData.Vol.Space | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName }
            }
        )

        if ($clusterData.Vol | Get-Member -MemberType NoteProperty -Name Footprint) {
            $vserverData.Vol | Add-Member -MemberType NoteProperty -Name Footprint -Value ($ClusterData.Vol.Footprint | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        if ($clusterData.Vol | Get-Member -MemberType NoteProperty -Name TieringInfo) {
            $vserverData.Vol | Add-Member -MemberType NoteProperty -Name TieringInfo -Value ($ClusterData.Vol.TieringInfo | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        #endregion Volume Information

        #region SIS Configuration

        Write-Verbose -Message '- Converting SIS configuration -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Sis -Value ([pscustomobject]@{ })

        $vserverData.Sis | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.Sis.Info | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName -or $_.Vs -ceq $VserverName })
        $vserverData.Sis | Add-Member -MemberType NoteProperty -Name Policy -Value ($ClusterData.Sis.Policy | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName -or $_.Vs -ceq $VserverName })

        if ($baseTypeName -eq 'NetAppDocs.NcController.Asup') {
            $vserverData.Sis | Add-Member -MemberType NoteProperty -Name Config -Value ($ClusterData.Sis.Config | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName -or $_.Vs -ceq $VserverName })
        }

        #endregion SIS Configuration

        #region Snapshot Information

        Write-Verbose -Message '- Converting detailed Snapshot information -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Snapshot -Value ([pscustomobject]@{ })

        if ($baseTypeName -eq 'NetAppDocs.NcController.Asup') {
            $volDsids = ($ClusterData.Vol.Info | Where-Object { $_.Vs -ceq $VserverName }).Dsid

            $vserverData.Snapshot | Add-Member -MemberType NoteProperty -Name Info -Value (
                $ClusterData.Snapshot.Info | Where-Object { $_.V -in $volDsids } | Select-Object -ExcludeProperty Vs -Property *, @{
                    Name       = 'Vs'
                    Expression = { $VserverName }
                }
            )

            $vserverData.Snapshot | Add-Member -MemberType NoteProperty -Name Policy -Value (New-Object -TypeName System.Collections.Generic.List[pscustomobject])

            foreach ($policy in ($ClusterData.Snapshot.Policy | Where-Object { $_ -and ([string]::IsNullOrWhiteSpace($_.V) -or $_.V -ceq $vserverData.Cluster.Info.ClusterName -or $_.V -ceq $VserverName) })) {
                $policyObj = $policy | Select-Object -Property *

                if ([string]::IsNullOrWhiteSpace($policyObj.V) -or $policyObj.V -ceq $vserverData.Cluster.Info.ClusterName) {
                    $policyObj | Add-Member -MemberType NoteProperty -Name V -Value $VserverName -ErrorAction Ignore -Force
                }

                $vserverData.Snapshot.Policy.Add($policyObj)
            }
        }
        else {
            #Prior to 8.2, snapshot policies could not be created at the Vserver level (so, VserverName should always be $null)
            $vserverData.Snapshot | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.Snapshot.Info | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            $vserverData.Snapshot | Add-Member -MemberType NoteProperty -Name Policy -Value (New-Object -TypeName System.Collections.Generic.List[pscustomobject])

            foreach ($policy in ($ClusterData.Snapshot.Policy | Where-Object { [string]::IsNullOrWhiteSpace($_.VserverName) -or $_.VserverName -ceq $vserverData.Cluster.Info.ClusterName -or $_.VserverName -ceq $VserverName })) {
                $policyObj = $policy | Select-Object -Property *

                if ([string]::IsNullOrWhiteSpace($policyObj.VserverName) -or $policyObj.VserverName -ceq $vserverData.Cluster.Info.ClusterName) {
                    $policyObj | Add-Member -MemberType NoteProperty -Name VserverName -Value $VserverName -ErrorAction Ignore -Force
                }

                $vserverData.Snapshot.Policy.Add($policyObj)
            }
        }

        #endregion Snapshot Information

        #region Qtree Information

        Write-Verbose -Message '- Converting qtree data -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Qtree -Value ($ClusterData.Qtree | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName -or $_.Vs -ceq $VserverName })

        #endregion Qtree Information

        #region User Data

        Write-Verbose -Message '- Converting user data -'
        #Not in ASUP data
        $vserverData | Add-Member -MemberType NoteProperty -Name User -Value ($ClusterData.User | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        #endregion User Data

        #region Role Data

        Write-Verbose -Message '- Converting role data -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Role -Value ([pscustomobject]@{ })

        #Not in ASUP data
        $vserverData.Role | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.Role.Info | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        #Not in ASUP data
        $vserverData.Role | Add-Member -MemberType NoteProperty -Name Config -Value ($ClusterData.Role.Config | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        #endregion Role Data

        #region SNMP User Data

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.100')) {
            Write-Verbose -Message '- Converting SNMP user data -'

            #Not in ASUP data
            $vserverData | Add-Member -MemberType NoteProperty -Name SnmpUser -Value ($ClusterData.SnmpUser | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        #endregion SNMP User Data

        #region Network Information

        Write-Verbose -Message '- Converting network details -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Net -Value ([pscustomobject]@{ })

        $vserverData.Net | Add-Member -MemberType NoteProperty -Name Interface -Value ($ClusterData.Net.Interface | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName -or $_.VserverId -ceq $VserverName })
        $vserverData.Net | Add-Member -MemberType NoteProperty -Name Dns -Value ($ClusterData.Net.Dns | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        $vserverData.Net | Add-Member -MemberType NoteProperty -Name DnsHost -Value ($ClusterData.Net.DnsHost | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        #Not in ASUP data
        $vserverData.Net | Add-Member -MemberType NoteProperty -Name DnsZone -Value ($ClusterData.Net.DnsZone | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        $vserverData.Net | Add-Member -MemberType NoteProperty -Name FailoverGroup -Value ($ClusterData.Net.FailoverGroup | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        if ($baseTypeName -ne 'NetAppDocs.NcController.Asup') {
            $vserverData.Net | Add-Member -MemberType NoteProperty -Name FailoverTarget -Value ($ClusterData.Net.FailoverTarget | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            $vserverData.Net | Add-Member -MemberType NoteProperty -Name Route -Value ($ClusterData.Net.Route | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            #Not in ASUP data
            $vserverData.Net | Add-Member -MemberType NoteProperty -Name RouteLif -Value ($ClusterData.Net.RouteLif | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            $vserverData.Net | Add-Member -MemberType NoteProperty -Name FirewallPolicy -Value ($ClusterData.Net.FirewallPolicy | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }
        else {
            $vserverData.Net | Add-Member -MemberType NoteProperty -Name RoutingGroup -Value ($ClusterData.Net.RoutingGroup | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            $vserverData.Net | Add-Member -MemberType NoteProperty -Name RoutingGroupRoute -Value ($ClusterData.Net.RoutingGroupRoute | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.31')) {
            $vserverData.Net | Add-Member -MemberType NoteProperty -Name Ddns -Value ($ClusterData.Net.Ddns | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

            if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.140')) {
                $vserverData.Net | Add-Member -MemberType NoteProperty -Name ServicePolicy -Value ([pscustomobject]@{ })

                $vserverData.Net.ServicePolicy | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.Net.ServicePolicy.Info | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
                $vserverData.Net.ServicePolicy | Add-Member -MemberType NoteProperty -Name Entry -Value ($ClusterData.Net.ServicePolicy.Entry | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            }
        }

        #endregion Network Information

        #region Export Policy Information

        Write-Verbose -Message '- Converting export policy information -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Export -Value ([pscustomobject]@{ })

        $vserverData.Export | Add-Member -MemberType NoteProperty -Name Policy -Value ($ClusterData.Export.Policy | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        $vserverData.Export | Add-Member -MemberType NoteProperty -Name Rule -Value ($ClusterData.Export.Rule | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        #endregion Export Policy Information

        #region FCP Information

        Write-Verbose -Message '- Converting FCP configuration data -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Fcp -Value ([pscustomobject]@{ })

        $vserverData.Fcp | Add-Member -MemberType NoteProperty -Name Service -Value ($ClusterData.Fcp.Service | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        $vserverData.Fcp | Add-Member -MemberType NoteProperty -Name Initiator -Value ($ClusterData.Fcp.Initiator | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        #Not in ASUP data
        $vserverData.Fcp | Add-Member -MemberType NoteProperty -Name Interface -Value ($ClusterData.Fcp.Interface | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        #endregion FCP Information

        #region iSCSI Information

        Write-Verbose -Message '- Converting iSCSI configuration data -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Iscsi -Value ([pscustomobject]@{ })

        $vserverData.Iscsi | Add-Member -MemberType NoteProperty -Name Service -Value ($ClusterData.Iscsi.Service | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        $vserverData.Iscsi | Add-Member -MemberType NoteProperty -Name Interface -Value ([pscustomobject]@{ })
        $vserverData.Iscsi.Interface | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.Iscsi.Interface.Info | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        $vserverData.Iscsi | Add-Member -MemberType NoteProperty -Name Initiator -Value ($ClusterData.Iscsi.Initiator | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        #Not in ASUP data
        $vserverData.Iscsi | Add-Member -MemberType NoteProperty -Name TargetPortalGroup -Value ($ClusterData.Iscsi.TargetPortalGroup | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        #endregion iSCSI Information

        #region LUN Information

        Write-Verbose -Message '- Converting LUN information -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Lun -Value ([pscustomobject]@{ })

        $vserverData.Lun | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.Lun.Info | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        $vserverData.Lun | Add-Member -MemberType NoteProperty -Name Map -Value ($ClusterData.Lun.Map | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName -or $_.VserverUuid -eq $VserverUuid })

        #endregion LUN Information

        #region iGroup Information

        Write-Verbose -Message '- Converting iGroup information -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Igroup -Value ($ClusterData.Igroup | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        #endregion iGroup Information

        #region Portset Information

        Write-Verbose -Message '- Converting Portset information -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Portset -Value ($ClusterData.Portset | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        #endregion Portset Information

        #region CIFS Information

        Write-Verbose -Message '- Converting CIFS configuration information -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Cifs -Value ([pscustomobject]@{ })

        $vserverData.Cifs | Add-Member -MemberType NoteProperty -Name Server -Value ($ClusterData.Cifs.Server | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        $vserverData.Cifs | Add-Member -MemberType NoteProperty -Name Symlink -Value ($ClusterData.Cifs.Symlink | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        $vserverData.Cifs | Add-Member -MemberType NoteProperty -Name Share -Value ($ClusterData.Cifs.Share | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        if ($ClusterData.Cifs.psobject.Properties.Name -contains 'ShareAcl') {
            $vserverData.Cifs | Add-Member -MemberType NoteProperty -Name ShareAcl -Value ($ClusterData.Cifs.ShareAcl | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }
        $vserverData.Cifs | Add-Member -MemberType NoteProperty -Name HomeDirectorySearchPath -Value ($ClusterData.Cifs.HomeDirectorySearchPath | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        $vserverData.Cifs | Add-Member -MemberType NoteProperty -Name Option -Value ($ClusterData.Cifs.Option | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        $vserverData.Cifs | Add-Member -MemberType NoteProperty -Name PreferredDomainController -Value ($ClusterData.Cifs.PreferredDomainController | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        $vserverData.Cifs | Add-Member -MemberType NoteProperty -Name Security -Value ($ClusterData.Cifs.Security | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        $vserverData.Cifs | Add-Member -MemberType NoteProperty -Name UserAndGroup -Value ([pscustomobject]@{ })

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            #Not in ASUP data
            $vserverData.Cifs.UserAndGroup | Add-Member -MemberType NoteProperty -Name LocalUser -Value ($ClusterData.Cifs.UserAndGroup.LocalUser | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            #Not in ASUP data
            $vserverData.Cifs.UserAndGroup | Add-Member -MemberType NoteProperty -Name LocalGroup -Value ($ClusterData.Cifs.UserAndGroup.LocalGroup | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            #Not in ASUP data
            $vserverData.Cifs.UserAndGroup | Add-Member -MemberType NoteProperty -Name LocalGroupMember -Value ($ClusterData.Cifs.UserAndGroup.LocalGroupMember | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            #Not in ASUP data
            $vserverData.Cifs.UserAndGroup | Add-Member -MemberType NoteProperty -Name Privilege -Value ($ClusterData.Cifs.UserAndGroup.Privilege | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            #Not in ASUP data
            $vserverData.Cifs | Add-Member -MemberType NoteProperty -Name BranchCache -Value ($ClusterData.Cifs.BranchCache | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

            if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.30')) {
                $vserverData.Cifs | Add-Member -MemberType NoteProperty -Name HomeDirectoryConfig -Value ($ClusterData.Cifs.HomeDirectoryConfig | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

                if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.130')) {
                    $vserverData.Cifs | Add-Member -MemberType NoteProperty -Name DomainServerDiscoveryMode -Value ($ClusterData.Cifs.DomainServerDiscoveryMode | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
                }
            }
        }

        #endregion CIFS Information

        #region GPO Information

        Write-Verbose -Message '- Converting GPO configuration information -'
        $vserverData.Cifs | Add-Member -MemberType NoteProperty -Name Gpo -Value ([pscustomobject]@{ })

        $vserverData.Cifs.Gpo | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.Cifs.Gpo.Info | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        #endregion GPO Information

        #region FPolicy Information

        Write-Verbose -Message '- Converting Fpolicy configuration -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Fpolicy -Value ([pscustomobject]@{ })

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            $vserverData.Fpolicy | Add-Member -MemberType NoteProperty -Name ServerStatus -Value ($ClusterData.Fpolicy.ServerStatus | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            $vserverData.Fpolicy | Add-Member -MemberType NoteProperty -Name ExternalEngine -Value ($ClusterData.Fpolicy.ExternalEngine | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            $vserverData.Fpolicy | Add-Member -MemberType NoteProperty -Name Policy -Value ($ClusterData.Fpolicy.Policy | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            $vserverData.Fpolicy | Add-Member -MemberType NoteProperty -Name Event -Value ($ClusterData.Fpolicy.Event | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            $vserverData.Fpolicy | Add-Member -MemberType NoteProperty -Name Scope -Value ($ClusterData.Fpolicy.Scope | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        #endregion FPolicy Information

        #region NFS Information

        Write-Verbose -Message '- Converting NFS configuration information -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Nfs -Value ([pscustomobject]@{ })

        $vserverData.Nfs | Add-Member -MemberType NoteProperty -Name Service -Value ($ClusterData.Nfs.Service | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        #endregion NFS Information

        #region Name Services Information

        Write-Verbose -Message '- Converting name services information -'
        $vserverData | Add-Member -MemberType NoteProperty -Name NameService -Value ([pscustomobject]@{ })

        if ($ClusterData.NameService.Nis -and $ClusterData.NameService.Nis.psobject.Properties.Name -contains 'Info') {
            $vserverData.NameService | Add-Member -MemberType NoteProperty -Name Nis -Value ([pscustomobject]@{ })

            $vserverData.NameService.Nis | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.NameService.Nis.Info | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }
        else {
            $vserverData.NameService | Add-Member -MemberType NoteProperty -Name Nis -Value ($ClusterData.NameService.Nis | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        $vserverData.NameService | Add-Member -MemberType NoteProperty -Name Kerberos -Value ([pscustomobject]@{ })

        $vserverData.NameService.Kerberos | Add-Member -MemberType NoteProperty -Name Config -Value ($ClusterData.NameService.Kerberos.Config | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            $vserverData.NameService.Kerberos | Add-Member -MemberType NoteProperty -Name Realm -Value ($ClusterData.NameService.Kerberos.Realm | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        $vserverData.NameService | Add-Member -MemberType NoteProperty -Name Ldap -Value ([pscustomobject]@{ })

        $vserverData.NameService.Ldap | Add-Member -MemberType NoteProperty -Name Client -Value (
            $ClusterData.NameService.Ldap.Client | Where-Object { $_ -and ([string]::IsNullOrWhiteSpace($_.Vserver) -or $_.Vserver -ceq $VserverName -or $_.Vserver -ceq $vserverData.Cluster.Info.ClusterName) }
        )

        foreach ($ldapClient in $vserverData.NameService.Ldap.Client) {
            if ([string]::IsNullOrWhiteSpace($ldapClient.Vserver) -or $ldapClient.Vserver -ceq $vserverData.Cluster.Info.ClusterName) {
                $ldapClient | Add-Member -MemberType NoteProperty -Name IsOwner -Value $false -ErrorAction Ignore -Force
            }

            $ldapClient | Add-Member -MemberType NoteProperty -Name Vserver -Value $VserverName -ErrorAction Ignore -Force
        }

        $vserverData.NameService.Ldap | Add-Member -MemberType NoteProperty -Name ClientSchema -Value (
            $ClusterData.NameService.Ldap.ClientSchema | Group-Object -Property Schema | ForEach-Object {
                $_.Group | Where-Object { $_ -and ([string]::IsNullOrWhiteSpace($_.Vserver) -or $_.Vserver -ceq $VserverName -or $_.Vserver -ceq $vserverData.Cluster.Info.ClusterName) } | Select-Object -First 1 -Property *
            }
        )

        foreach ($ldapClientSchema in $vserverData.NameService.Ldap.ClientSchema) {
            if ([string]::IsNullOrWhiteSpace($ldapClientSchema.Vserver) -or $ldapClientSchema.Vserver -ceq $vserverData.Cluster.Info.ClusterName) {
                $ldapClientSchema | Add-Member -MemberType NoteProperty -Name IsOwner -Value $false -ErrorAction Ignore -Force
            }

            $ldapClientSchema | Add-Member -MemberType NoteProperty -Name Vserver -Value $VserverName -ErrorAction Ignore -Force
        }

        $vserverData.NameService.Ldap | Add-Member -MemberType NoteProperty -Name Config -Value ($ClusterData.NameService.Ldap.Config | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            $vserverData.NameService | Add-Member -MemberType NoteProperty -Name NsSwitch -Value ($ClusterData.NameService.NsSwitch | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        #endregion Name Services Information

        #region Netgroup Information

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message '- Converting netgroup information -'
            $vserverData | Add-Member -MemberType NoteProperty -Name Netgroup -Value ([pscustomobject]@{ })

            #Not in ASUP data
            $vserverData.Netgroup | Add-Member -MemberType NoteProperty -Name File -Value ($ClusterData.Netgroup.File | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        #endregion Netgroup Information

        #region Name Mapping Information

        Write-Verbose -Message '- Converting name mapping data -'
        $vserverData | Add-Member -MemberType NoteProperty -Name NameMapping -Value ([pscustomobject]@{ })

        $vserverData.NameMapping | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.NameMapping.Info | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        $vserverData.NameMapping | Add-Member -MemberType NoteProperty -Name UnixGroup -Value ($ClusterData.NameMapping.UnixGroup | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        $vserverData.NameMapping | Add-Member -MemberType NoteProperty -Name UnixUser -Value ($ClusterData.NameMapping.UnixUser | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        #endregion Name Mapping Information

        #region Group Mapping Information

        Write-Verbose -Message '- Converting group mapping data -'
        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            #Not in ASUP data
            $vserverData | Add-Member -MemberType NoteProperty -Name GroupMapping -Value ($ClusterData.Groupmapping | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        #endregion Group Mapping Information

        #region Vscan Configuration

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.21')) {
            Write-Verbose -Message '- Converting Vscan information -'
            $vserverData | Add-Member -MemberType NoteProperty -Name Vscan -Value ([pscustomobject]@{ })

            $vserverData.Vscan | Add-Member -MemberType NoteProperty -Name OnAccessPolicy -Value ($ClusterData.Vscan.OnAccessPolicy | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            $vserverData.Vscan | Add-Member -MemberType NoteProperty -Name ScannerPool -Value ($ClusterData.Vscan.ScannerPool | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            $vserverData.Vscan | Add-Member -MemberType NoteProperty -Name Status -Value ($ClusterData.Vscan.Status | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        #endregion Vscan Configuration

        #region Job Information

        Write-Verbose -Message '- Converting job information -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Job -Value ([pscustomobject]@{ })

        $vserverData.Job | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.Job.Info | Where-Object { $_.JobVserver -ceq $VserverName -or $_.OwningVserver -ceq $VserverName })
        $vserverData.Job | Add-Member -MemberType NoteProperty -Name Schedule -Value $ClusterData.Job.Schedule

        #endregion Job Information

        #region File Directory Security Information

        Write-Verbose -Message '- Converting File Directory Security configuration -'
        $vserverData | Add-Member -MemberType NoteProperty -Name FileDirectorySecurity -Value ([pscustomobject]@{ })

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            #Not in ASUP data
            $vserverData.FileDirectorySecurity | Add-Member -MemberType NoteProperty -Name PolicyTask -Value ($ClusterData.FileDirectorySecurity.PolicyTask | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        #endregion File Directory Security Information

        #region File Service Audit Information

        Write-Verbose -Message '- Converting File Service Audit configuration -'
        $vserverData | Add-Member -MemberType NoteProperty -Name FileServiceAudit -Value ([pscustomobject]@{ })

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            #Not in ASUP data
            $vserverData.FileServiceAudit | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.FileServiceAudit.Info | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        #endregion File Service Audit Information

        #region FlexCache Information

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Converting FlexCache data -'
            $vserverData | Add-Member -MemberType NoteProperty -Name Flexcache -Value ([pscustomobject]@{ })

            if ($vserverData.System.OntapiVersion -lt [version]::Parse('1.30')) {
                $vserverData.Flexcache | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.Flexcache.Info | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
                $vserverData.Flexcache | Add-Member -MemberType NoteProperty -Name Policy -Value ($ClusterData.Flexcache.Policy | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            }

            if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.150')) {
                $vserverData.Flexcache | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.Flexcache.Info | Where-Object { $_.Vserver -ceq $VserverName -or $_.VserverName -ceq $VserverName })
            }
        }

        #endregion FlexCache Information

        #region SnapMirror Information

        Write-Verbose -Message '- Converting Snapmirror configuration -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Snapmirror -Value ([pscustomobject]@{ })

        $vserverData.Snapmirror | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.Snapmirror.Info | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            if ($baseTypeName -eq 'NetAppDocs.NcController.Asup') {
                $vserverData.Snapmirror | Add-Member -MemberType NoteProperty -Name Destination -Value ($ClusterData.Snapmirror.Destination | Where-Object { ($_.SourcePath -split ':' | Select-Object -First 1) -ceq $VserverName })

                $vserverData.Snapmirror | Add-Member -MemberType NoteProperty -Name Policy -Value (New-Object -TypeName System.Collections.Generic.List[pscustomobject])

                foreach ($policy in ($ClusterData.Snapmirror.Policy | Where-Object { [string]::IsNullOrWhiteSpace($_.Vserver) -or $_.Vserver -ceq $vserverData.Cluster.Info.ClusterName -or $_.Vserver -ceq $VserverName })) {
                    $policyObj = $policy | Select-Object -Property *

                    if ((-not $policyObj.Vserver) -or $policyObj.Vserver -ceq $vserverData.Cluster.Info.ClusterName) {
                        $policyObj.Vserver = $VserverName
                    }

                    $vserverData.Snapmirror.Policy.Add($policyObj)
                }
            }
            else {
                $vserverData.Snapmirror | Add-Member -MemberType NoteProperty -Name Destination -Value ($ClusterData.Snapmirror.Destination | Where-Object { $_.SourceVserver -ceq $VserverName })

                $vserverData.Snapmirror | Add-Member -MemberType NoteProperty -Name Policy -Value (New-Object -TypeName System.Collections.Generic.List[pscustomobject])

                foreach ($policy in ($ClusterData.Snapmirror.Policy | Where-Object { [string]::IsNullOrWhiteSpace($_.VserverName) -or $_.VserverName -ceq $vserverData.Cluster.Info.ClusterName -or $_.VserverName -ceq $VserverName })) {
                    $policyObj = $policy | Select-Object -Property *

                    if ((-not $policyObj.VserverName) -or $policyObj.VserverName -ceq $vserverData.Cluster.Info.ClusterName) {
                        $policyObj.VserverName = $VserverName
                    }

                    $vserverData.Snapmirror.Policy.Add($policyObj)
                }
            }
        }

        #endregion SnapMirror Information

        #region Quota Information

        Write-Verbose -Message '- Converting quota configuration -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Quota -Value ([pscustomobject]@{ })

        $vserverData.Quota | Add-Member -MemberType NoteProperty -Name Rule -Value ($ClusterData.Quota.Rule | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName -or $_.Vs -ceq $VserverName })
        $vserverData.Quota | Add-Member -MemberType NoteProperty -Name Status -Value ($ClusterData.Quota.Status | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName -or $_.Vs -ceq $VserverName })

        #endregion Quota Information

        #region Security Information

        Write-Verbose -Message '- Converting security related information -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Security -Value ([pscustomobject]@{ })

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.17')) {
            #not in ASUP data
            $vserverData.Security | Add-Member -MemberType NoteProperty -Name DomainTunnelVserver -Value $ClusterData.Security.DomainTunnelVserver

            if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.20')) {
                $vserverData.Security | Add-Member -MemberType NoteProperty -Name Certificate -Value ($ClusterData.Security.Certificate | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })

                if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.21')) {
                    $vserverData.Security | Add-Member -MemberType NoteProperty -Name Ssh -Value ($ClusterData.Security.Ssh | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
                }
            }
        }

        #endregion Security Information

        #region Snaplock Information

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.100')) {
            Write-Verbose -Message '- Converting Snaplock information -'
            $vserverData | Add-Member -MemberType NoteProperty -Name Snaplock -Value ([pscustomobject]@{ })

            $vserverData.Snaplock | Add-Member -MemberType NoteProperty -Name Volume -Value ($ClusterData.Snaplock.Volume | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        #endregion Snaplock Information

        #region NVMe Information

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.140')) {
            Write-Verbose -Message '- Converting NVMe information -'
            $vserverData | Add-Member -MemberType NoteProperty -Name Nvme -Value (
                [pscustomobject]@{
                    Subsystem = [pscustomobject]@{ }
                }
            )

            $vserverData.Nvme | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.Nvme.Info | Where-Object { $_.VserverUuid -eq $VserverUuid -or $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            $vserverData.Nvme | Add-Member -MemberType NoteProperty -Name Interface -Value ($ClusterData.Nvme.Interface | Where-Object { $_.VserverUuid -eq $VserverUuid -or $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            $vserverData.Nvme | Add-Member -MemberType NoteProperty -Name Namespace -Value ($ClusterData.Nvme.Namespace | Where-Object { $_.VserverUuid -eq $VserverUuid -or $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            $vserverData.Nvme.Subsystem | Add-Member -MemberType NoteProperty -Name Info -Value ($ClusterData.Nvme.Subsystem.Info | Where-Object { $_.VserverUuid -eq $VserverUuid -or $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
            $vserverData.Nvme.Subsystem | Add-Member -MemberType NoteProperty -Name Host -Value $(
                if (($ClusterData.pstypenames[0] -replace 'Deserialized.' -replace '.Data' -replace '.OneCollect') -notlike '*Asup*') {
                    $ClusterData.Nvme.Subsystem.Host | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName }
                }
                else {
                    if ($vserverData.Nvme.Info) {
                        foreach ($subsystemHost in ($ClusterData.Nvme.Subsystem.Host | Where-Object { $_ })) {
                            $subsystemInfo = $ClusterData.Nvme.Subsystem.Info | Where-Object { $_.SubsystemUuid -eq $subsystemHost.SubsystemUuid }

                            $subsystemHost | Select-Object -Property *, @{
                                Name       = 'VserverUuid'
                                Expression = { $subsystemInfo.VserverUuid }
                            }, @{
                                Name       = 'Subsystem'
                                Expression = { $subsystemInfo.Subsystem }
                            }
                        }
                    }
                }
            )
            $vserverData.Nvme.Subsystem | Add-Member -MemberType NoteProperty -Name Map -Value ($ClusterData.Nvme.Subsystem.Map | Where-Object { $_.VserverUuid -eq $VserverUuid -or $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        #endregion NVMe Information

        #region QOS Information

        Write-Verbose -Message '- Converting QOS information -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Qos -Value ([pscustomobject]@{ })
        #Not in ASUP data (vserver not specified and no way to determine that info)
        $vserverData.Qos | Add-Member -MemberType NoteProperty -Name PolicyGroup -Value ($ClusterData.Qos.PolicyGroup | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        $vserverData.Qos | Add-Member -MemberType NoteProperty -Name Workload -Value ($ClusterData.Qos.Workload | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName -or $_.VserverId -ceq $VserverName })

        #endregion QOS Information

        #region NDMP Information

        Write-Verbose -Message '- Converting NDMP information -'
        $vserverData | Add-Member -MemberType NoteProperty -Name Ndmp -Value ([pscustomobject]@{ })

        if ($vserverData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            $vserverData.Ndmp | Add-Member -MemberType NoteProperty -Name VserverServices -Value ($ClusterData.Ndmp.VserverServices | Where-Object { $_.VserverName -ceq $VserverName -or $_.Vserver -ceq $VserverName })
        }

        #endregion NDMP Information

        Write-Verbose -Message '- Identifying Reference Time -'
        $vserverData | Add-Member -MemberType NoteProperty -Name ReferenceDT -Value $ClusterData.ReferenceDT

        Write-Verbose -Message '- Identifying SystemName -'
        $vserverData | Add-Member -MemberType NoteProperty -Name SystemName -Value $VserverName

        Write-Verbose -Message '- Identifying any errors during processing -'
        $vserverData | Add-Member -MemberType NoteProperty -Name DiscoveryError -Value @(
            $ClusterData.DiscoveryError | Where-Object { $_ } | Select-Object -Property *, @{
                Name       = 'VserverName'
                Expression = { $VserverName }
            }
        )

        $vserverData
    }
    catch {
        $Global:Error.RemoveAt(0)

        $msg = New-Object -TypeName System.Text.StringBuilder

        [void]$msg.AppendLine('An unexpected error occurred. Processing halted.')
        [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
        [void]$msg.AppendLine('As this error is not expected, please consider contacting the support staff via e-mail or by')
        [void]$msg.AppendLine('starting a thread on the community site.')
        [void]$msg.AppendLine('            E-mail: ng-NetAppDocs-support@netapp.com')
        [void]$msg.AppendLine('    Community site: https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs')
        [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
        [void]$msg.AppendLine($( $_.Exception.GetBaseException().Message ))
        [void]$msg.AppendLine("Location: $( $_.ScriptStackTrace )")
        [void]$msg.AppendLine()

        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $_.CategoryInfo.TargetName)))
    }
}

# SIG # Begin signature block
# MIIq0AYJKoZIhvcNAQcCoIIqwTCCKr0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBmwjQZlr5y5JWd
# KVvGUIWNg3ZJnJW4PS4aRpAMwpt3oKCCJGgwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggWNMIIEdaADAgECAhAOmxiO
# +dAt5+/bUOIIQBhaMA0GCSqGSIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYD
# VQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAi
# BgNVBAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAw
# MDBaFw0zMTExMDkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
# Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERp
# Z2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBAL/mkHNo3rvkXUo8MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsb
# hA3EMB/zG6Q4FutWxpdtHauyefLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iT
# cMKyunWZanMylNEQRBAu34LzB4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGb
# NOsFxl7sWxq868nPzaw0QF+xembud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclP
# XuU15zHL2pNe3I6PgNq2kZhAkHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCr
# VYJBMtfbBHMqbpEBfCFM1LyuGwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFP
# ObURWBf3JFxGj2T3wWmIdph2PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTv
# kpI6nj3cAORFJYm2mkQZK37AlLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWM
# cCxBYKqxYxhElRp2Yn72gLD76GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls
# 5Q5SUUd0viastkF13nqsX40/ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBR
# a2+xq4aLT8LWRV+dIPyhHsXAj6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6
# MIIBNjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qY
# rhwPTzAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8E
# BAMCAYYweQYIKwYBBQUHAQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5k
# aWdpY2VydC5jb20wQwYIKwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDig
# NoY0aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNybDARBgNVHSAECjAIMAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCg
# v0NcVec4X6CjdBs9thbX979XB72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQT
# SnovLbc47/T/gLn4offyct4kvFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh
# 65ZyoUi0mcudT6cGAxN3J0TU53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSw
# uKFWjuyk1T3osdz9HNj0d1pcVIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAO
# QGPFmCLBsln1VWvPJ6tsds5vIy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjD
# TZ9ztwGpn1eqXijiuZQwggXMMIIENKADAgECAhAg429sPxgagUb53pPffJfkMA0G
# CSqGSIb3DQEBDAUAMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBS
# MzYwHhcNMjEwOTA5MDAwMDAwWhcNMjMwOTA5MjM1OTU5WjBQMQswCQYDVQQGEwJV
# UzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UECgwMTmV0QXBwLCBJbmMuMRUw
# EwYDVQQDDAxOZXRBcHAsIEluYy4wggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGK
# AoIBgQC4kYYj/oViZD9pN03hrqFjtwOz1Gx4eDWVV8IYUYEr2qpLeNYvWz26B/+E
# mYLboAGVpkYg0Wske3hjRpooZlMORm1S4+2C2NoQCstZ+kmNTevbXHJe/w1VNJrm
# fKjpXOJEfx8GUGDqCMR30dChVpsdBEavrM7T0hnfJLv18i19SGv3a+nhvB3iOjLP
# SEg8+DFkjD5jTI6fQ83DBEiVR4UEYq7yyH578ZEylhsRfJmK+uIREycEUk/NpFTw
# g6/7lxh/WYabCrZzqI4Ep3QataRKaAlxt3BiOlZkX4WGX3DYvSYltWOM2UlCcGpP
# m/1/LN8Xuvf+YO6H472QecHe59XtXbHSMKvqDuOEJu8Wk2LFMNK732o3Fc5QIHie
# 6JtNsgNYf+Vsm5EMBD1ZHNE+C3zlmQbbWIU5uEU1nhiodBPKsu0SkS6pARyKBO05
# DSc2gWWv0aukKi04ZAn+hAdSl5s1dgmU5shBvKHqe15K9ZRN4FFO/CgHE0BoGYQS
# UQVKwa0CAwEAAaOCAZwwggGYMB8GA1UdIwQYMBaAFA8qyyCHKLjsb0iuK1SmKaoX
# pM0MMB0GA1UdDgQWBBQuH643KcBMmb/Q6IZt+H9IrnXFwDAOBgNVHQ8BAf8EBAMC
# B4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDAzARBglghkgBhvhC
# AQEEBAMCBBAwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwIwJTAjBggrBgEFBQcC
# ARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQBMEkGA1UdHwRCMEAw
# PqA8oDqGOGh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVT
# aWduaW5nQ0FSMzYuY3JsMHkGCCsGAQUFBwEBBG0wazBEBggrBgEFBQcwAoY4aHR0
# cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQVIz
# Ni5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqG
# SIb3DQEBDAUAA4IBgQCOoGdXjP8Sif0h3ZvoDsIVfrJvQUdP9pZExRJGGj/Te6ML
# XyqHNc/G2WklNv+BC4ryNJ4zp0oneHOub55ejrYPfhgP9oFWS8oOPJbrp3rAtINa
# OeKRC88WUYZSKYAxSX6kubEjnQD6cmazXjxVN6P8+6q9smeqF3xI+++c+ekMw3Xv
# 4EWRYXqN0/srfFp1WpoluwSQcZMqj9haSX0bgO6AAWH2AnVJMfXTIKz/0FQ/RW0y
# Ed5QYQqYInhA7IUz9dBziNpJlwONeF5j7xzgfYDY63WU6WrgJnGjYkQCOOsUx74j
# gUiMRe9zV48GS8Vxz22c/TQs0x4/1GmOSJvUHEk3GseBmB3v+yEhh/D6zWOfYP4X
# D/9b91CxmugTuepuaJSeRg+qUm3KarKsOCUF/CLqUrNh/JwKrWD1cghRaYLvMucs
# ScksHHe7ZDvb2OtvxWXjPk1d1NKvEwFJSS6hIVIbug9x28AJqOoP5Pn9wZvJSzvW
# uQJGCLscdGyKefdCo30wggYaMIIEAqADAgECAhBiHW0MUgGeO5B5FSCJIRwKMA0G
# CSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290
# IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFQxCzAJBgNVBAYT
# AkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28g
# UHVibGljIENvZGUgU2lnbmluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEBAQUAA4IB
# jwAwggGKAoIBgQCbK51T+jU/jmAGQ2rAz/V/9shTUxjIztNsfvxYB5UXeWUzCxEe
# AEZGbEN4QMgCsJLZUKhWThj/yPqy0iSZhXkZ6Pg2A2NVDgFigOMYzB2OKhdqfWGV
# oYW3haT29PSTahYkwmMv0b/83nbeECbiMXhSOtbam+/36F09fy1tsB8je/RV0mIk
# 8XL/tfCK6cPuYHE215wzrK0h1SWHTxPbPuYkRdkP05ZwmRmTnAO5/arnY83jeNzh
# P06ShdnRqtZlV59+8yv+KIhE5ILMqgOZYAENHNX9SJDm+qxp4VqpB3MV/h53yl41
# aHU5pledi9lCBbH9JeIkNFICiVHNkRmq4TpxtwfvjsUedyz8rNyfQJy/aOs5b4s+
# ac7IH60B+Ja7TVM+EKv1WuTGwcLmoU3FpOFMbmPj8pz44MPZ1f9+YEQIQty/NQd/
# 2yGgW+ufflcZ/ZE9o1M7a5Jnqf2i2/uMSWymR8r2oQBMdlyh2n5HirY4jKnFH/9g
# Rvd+QOfdRrJZb1sCAwEAAaOCAWQwggFgMB8GA1UdIwQYMBaAFDLrkpr/NZZILyhA
# QnAgNpFcF4XmMB0GA1UdDgQWBBQPKssghyi47G9IritUpimqF6TNDDAOBgNVHQ8B
# Af8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcD
# AzAbBgNVHSAEFDASMAYGBFUdIAAwCAYGZ4EMAQQBMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Um9vdFI0Ni5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ1Jvb3RSNDYu
# cDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQwFAAOCAgEABv+C4XdjNm57oRUgmxP/BP6YdURhw1aVcdGRP4Wh60BAscjW
# 4HL9hcpkOTz5jUug2oeunbYAowbFC2AKK+cMcXIBD0ZdOaWTsyNyBBsMLHqafvIh
# rCymlaS98+QpoBCyKppP0OcxYEdU0hpsaqBBIZOtBajjcw5+w/KeFvPYfLF/ldYp
# mlG+vd0xqlqd099iChnyIMvY5HexjO2AmtsbpVn0OhNcWbWDRF/3sBp6fWXhz7Dc
# ML4iTAWS+MVXeNLj1lJziVKEoroGs9Mlizg0bUMbOalOhOfCipnx8CaLZeVme5yE
# Lg09Jlo8BMe80jO37PU8ejfkP9/uPak7VLwELKxAMcJszkyeiaerlphwoKx1uHRz
# NyE6bxuSKcutisqmKL5OTunAvtONEoteSiabkPVSZ2z76mKnzAfZxCl/3dq3dUNw
# 4rg3sTCggkHSRqTqlLMS7gjrhTqBmzu1L90Y1KWN/Y5JKdGvspbOrTfOXyXvmPL6
# E52z1NZJ6ctuMFBQZH3pwWvqURR8AgQdULUvrxjUYbHHj95Ejza63zdrEcxWLDX6
# xWls/GDnVNueKjWUH3fTv1Y8Wdho698YADR7TNx8X8z2Bev6SivBBOHY+uqiirZt
# g0y9ShQoPzmCcn63Syatatvx157YK9hlcPmVoa1oDE5/L9Uo2bC5a4CH2Rwwggau
# MIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJ
# BgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5k
# aWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAe
# Fw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcw
# FQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3Rl
# ZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9Ge
# TKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0
# hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZl
# jZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAsh
# aG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVY
# TXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1
# biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCir
# c0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+
# DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA
# +bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42Pg
# puE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzS
# M7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU
# uhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6
# mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsG
# AQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29t
# MEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNl
# cnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3Js
# My5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAE
# GTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1Z
# jsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8d
# B+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVp
# P0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp8
# 76i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2
# nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3
# ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQ
# txMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc
# 4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+Y
# AN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZ
# vAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQr
# H4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIGwDCCBKigAwIBAgIQDE1p
# ckuU+jwqSj0pB4A9WjANBgkqhkiG9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4XDTIyMDkyMTAwMDAw
# MFoXDTMzMTEyMTIzNTk1OVowRjELMAkGA1UEBhMCVVMxETAPBgNVBAoTCERpZ2lD
# ZXJ0MSQwIgYDVQQDExtEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMiAtIDIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDP7KUmOsap8mu7jcENmtuh6BSFdDMa
# JqzQHFUeHjZtvJJVDGH0nQl3PRWWCC9rZKT9BoMW15GSOBwxApb7crGXOlWvM+xh
# iummKNuQY1y9iVPgOi2Mh0KuJqTku3h4uXoW4VbGwLpkU7sqFudQSLuIaQyIxvG+
# 4C99O7HKU41Agx7ny3JJKB5MgB6FVueF7fJhvKo6B332q27lZt3iXPUv7Y3UTZWE
# aOOAy2p50dIQkUYp6z4m8rSMzUy5Zsi7qlA4DeWMlF0ZWr/1e0BubxaompyVR4aF
# eT4MXmaMGgokvpyq0py2909ueMQoP6McD1AGN7oI2TWmtR7aeFgdOej4TJEQln5N
# 4d3CraV++C0bH+wrRhijGfY59/XBT3EuiQMRoku7mL/6T+R7Nu8GRORV/zbq5Xwx
# 5/PCUsTmFntafqUlc9vAapkhLWPlWfVNL5AfJ7fSqxTlOGaHUQhr+1NDOdBk+lbP
# 4PQK5hRtZHi7mP2Uw3Mh8y/CLiDXgazT8QfU4b3ZXUtuMZQpi+ZBpGWUwFjl5S4p
# kKa3YWT62SBsGFFguqaBDwklU/G/O+mrBw5qBzliGcnWhX8T2Y15z2LF7OF7ucxn
# EweawXjtxojIsG4yeccLWYONxu71LHx7jstkifGxxLjnU15fVdJ9GSlZA076XepF
# cxyEftfO4tQ6dwIDAQABo4IBizCCAYcwDgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB
# /wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwIAYDVR0gBBkwFzAIBgZngQwB
# BAIwCwYJYIZIAYb9bAcBMB8GA1UdIwQYMBaAFLoW2W1NhS9zKXaaL3WMaiCPnshv
# MB0GA1UdDgQWBBRiit7QYfyPMRTtlwvNPSqUFN9SnDBaBgNVHR8EUzBRME+gTaBL
# hklodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0
# MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3JsMIGQBggrBgEFBQcBAQSBgzCBgDAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFgGCCsGAQUFBzAC
# hkxodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRS
# U0E0MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3J0MA0GCSqGSIb3DQEBCwUAA4IC
# AQBVqioa80bzeFc3MPx140/WhSPx/PmVOZsl5vdyipjDd9Rk/BX7NsJJUSx4iGNV
# CUY5APxp1MqbKfujP8DJAJsTHbCYidx48s18hc1Tna9i4mFmoxQqRYdKmEIrUPwb
# tZ4IMAn65C3XCYl5+QnmiM59G7hqopvBU2AJ6KO4ndetHxy47JhB8PYOgPvk/9+d
# EKfrALpfSo8aOlK06r8JSRU1NlmaD1TSsht/fl4JrXZUinRtytIFZyt26/+YsiaV
# OBmIRBTlClmia+ciPkQh0j8cwJvtfEiy2JIMkU88ZpSvXQJT657inuTTH4YBZJwA
# wuladHUNPeF5iL8cAZfJGSOA1zZaX5YWsWMMxkZAO85dNdRZPkOaGK7DycvD+5sT
# X2q1x+DzBcNZ3ydiK95ByVO5/zQQZ/YmMph7/lxClIGUgp2sCovGSxVK05iQRWAz
# gOAj3vgDpPZFR+XOuANCR+hBNnF3rf2i6Jd0Ti7aHh2MWsgemtXC8MYiqE+bvdgc
# mlHEL5r2X6cnl7qWLoVXwGDneFZ/au/ClZpLEQLIgpzJGgV8unG1TnqZbPTontRa
# mMifv427GFxD9dAq6OJi7ngE273R+1sKqHB+8JeEeOMIA11HLGOoJTiXAdI/Otrl
# 5fbmm9x+LMz/F0xNAKLY1gEOuIvu5uByVYksJxlh9ncBjDGCBb4wggW6AgEBMGgw
# VDELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDErMCkGA1UE
# AxMiU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIFIzNgIQIONvbD8YGoFG
# +d6T33yX5DANBglghkgBZQMEAgEFAKCBhDAYBgorBgEEAYI3AgEMMQowCKACgACh
# AoAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAM
# BgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCDI2sSuApQ93nWnopomhxeS/5W7
# JltygA28mb2RfiTFqzANBgkqhkiG9w0BAQEFAASCAYAHkkR1+1z8E/Yy8lIvu5je
# heVPWW/mbrfIjYhbPMWjaxC0synWM/uXCxk/Vn/7MmJsu1H+pttIqkogU9zbIQQS
# opaVXO2SJ5e/1G4P2AtiRGgMnGtkHllOc9fXLdIe3pS5NtNx2ziaWJW/XMtfsyY1
# /uq5CeD83a7GljRYjmawWxw82zgBEgyPeevXUhkman6Wy2W9WZBYzNHpp6BUmjkP
# zExTs66SfpnKIbc3mqaU9HJ4fDIIOyClxKCoK7GMGwE3LR9AqdXHJsCxjvKNNrPy
# 5WexDoNgFx9pgez2Jllb9YR5vhIANEM59VI2z9InHFpk/7EEirQ/bgn5sMzscsQk
# H4dcjNak5Bo71XE+7VdSq4iHrUDwYAHj4fbkNxJZ6mf1q8cR4Tmxu3HTG+w6Vwrc
# 8pukdct9t5yS85Xe/TaLBLPCCZBpuMOAUxWp+238nTdgOWipj0x14t39WDtDza+t
# TFpLwBXrQ3GwLwYU3KrqPbbacs5GF35N8w4AVPltFjGhggMgMIIDHAYJKoZIhvcN
# AQkGMYIDDTCCAwkCAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
# cnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
# SEEyNTYgVGltZVN0YW1waW5nIENBAhAMTWlyS5T6PCpKPSkHgD1aMA0GCWCGSAFl
# AwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUx
# DxcNMjMwNDAyMTUyODI2WjAvBgkqhkiG9w0BCQQxIgQg2hOV4etjOhtGPQtBoy+G
# VXJ/HUxfj+bdRFuD3xUTiAswDQYJKoZIhvcNAQEBBQAEggIAtq92D8xkC6Bea63H
# 6YqxMcsR90UIkGZLLNUKRlXXIbujW2jk8h+q8R78iFEHKGPmkwc4zWwR+6VJKcAl
# rURfZnQ+5FugjOfxSpkMPPJmV57CyizYeMzZ+SrlUFuJYjGNtJ17qmOeo3c2Uese
# IFPVK2q2/eY5MqpfacfzN2rIw2LIvoWeYBn8J0WMu6qiSg2iW4loZVrq+rtElpjn
# 6htQgnza6ZUn7fQrzeuRvH857b13elck5DXz43F8otWB3Nj2/LrvC7nYXYYTxvDy
# 9zKfI+0rBI4grQL1waclhg5mIO6MvoRvSY2ro66DpvGbGcvTzz/hBHqKC7TLDWy4
# w5els+vwzRctOy/x8KP91ULV18aJYTasCcYW1T5vPRRYF0Obv1PSqmoFfqD82Ba1
# ly0whYcJ4T15NEuXGDhKmtCeuu7fNzg4HJPpdJTrC1ZMKJU6catdZNgqFjMEdm6l
# RIwbEYy903mFxAlcMBgeAmrXaguxni+2xfb8mjesREo8vCldDHNZFmYpCpDniDAl
# cYgU0lx0HrD5FdOiZ5yaCC45phqMQoJwl8nc9QqWy82uKAHQlizgpShMubI8K5QK
# KhKk/6QIUMAKGD+TkxyHGnwwScQfi1cSNExmqPY6AYCUPMd7l1dx37QJE0uuBg6z
# 7fO1PEMKzbWOWGbETVxOQLy7TeA=
# SIG # End signature block
