function Format-NtapVserverData {
    <#
        .SYNOPSIS
            Parses the raw data from the Get-NtapVserverData / ConvertTo-NtapVserverData functions and optionally sanitizes
            the data for secure environments.

        .DESCRIPTION
            Takes raw configuration information from Get-NtapVserverData and parses it into relevant documentation data.

            The parsed output data can optionally be sanitized via the -SanitizeLevel parameter. This will create an XML file that
            contains the sanitized mappings (-SanitizedMappingsXmlFile <filename>) and this file can be used each time so the
            mappings will stay consistent, or the file can be reset each time (-ResetSanitizeMappings). In addition to the
            XML file, an XLSX file will be created that contains the mappings generated in an easier to read format. This file will
            be regenerated with each run regardless of the XML file parameters. Currently, the XLSX file will be output to the same
            path as the XML file.

            There is a sanitize mappings XML template file in this modules 'Templates' folder. Within that file are some options
            to define the number of octets of the IP address to 'X' out ('SanitizeMethod' option set to 'Octets' and
            'Ipv4OctetsToSanitize' option) or to provide a mapping for the IP addresses ('SanitizeMethod' option set to 'Mapping').
            Another option within this file is to only sanitize the mappings currently contained within the XML file. This is
            useful in the case where only a certain subset of the items on the controllers may be sensitve. You can prepopulate
            the mappings file by making a copy of the template, changing the option 'CurrentEntriesOnly' to 'True' and adding
            the entries desired in the appropriate location. Then, pass this filename as the parameter for -SanitizeMappingsXmlFile
            and ensure -ResetSanitizeMappings is $false (default).

        .PARAMETER SanitizeLevel
            You can specify various levels of sanitization for the output documentation. The levels are defined as follows:

                1 = IP Address
                2 = MAC Address
                8 = DNS Domain Name
               16 = SVM Name
              128 = Volume Name
              256 = Qtree Name
              512 = LUN Name
             1024 = LUN Initiator Group Name
             2048 = Other Servers
             4096 = SNMP Community String
             8192 = CIFS Share Name
            16384 = Active Directory NetBIOS Name

            To include multiple entries, simply add up the numbers. For example, for IP Address, SVM Name and DNS Domain
            Name sanitization the level should be 25 (1 + 16 + 8).

        .EXAMPLE
            PS> Get-NtapVserverData -ClusterName cluster1 | Format-NtapVserverData | Out-NtapDocument -WordFile 'D:\Output\cluster1-allVservers.docx'

            Collects the data for all data Vservers in cluster 'cluster1' and produces the output Word documents.

        .EXAMPLE
            PS> Get-NtapVserverData -Name vs_nfs, vs_cifs -ClusterName cluster1 | Format-NtapVserverData | Out-NtapDocument -WordFile 'D:\Output\cluster1-NasVservers.docx' -ExcelFile 'D:\Output\cluster1-NasVservers.xlsx'

            Collects the data for the 2 specified Vservers in cluster 'cluster1' and produces the output documents specified.

        .EXAMPLE
            PS> Get-NtapVserverData -ClusterName cluster1 | Format-NtapVserverData -SanitizeLevel 4097 -SanitizeMappingsXmlFile 'D:\Output\Site-SanitizeMappings.xml' | Out-NtapDocument -WordFile 'D:\Output\Site.docx'

            Collects the data for all data Vservers in cluster 'cluster1' and produces the output Word document. Additionally,
            this will sanitize both IP addresses and SNMP community strings (1 + 4096) and create the specified sanitized
            mappings XML file.

        .EXAMPLE
            PS> Get-ChildItem -Path 'D:\Output\cluster1.xml' | ConvertTo-NtapVserverData | Format-NtapVserverData | Out-NtapDocument -WordFile 'D:\Output\cluster1-allVservers.docx' -ExcelFile 'D:\Output\cluster1-allVservers.xlsx'

            Reads the raw cluster data from the file cluster1.xml (generated via Get-NtapClusterData) and outputs the data for
            all the data Vservers to the Word and Excel documents specified.

        .LINK
            https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs

        .LINK
            Get-NtapVserverData

        .INPUTS
            System.Management.Automation.PSCustomObject

        .INPUTS
            System.IO.FileInfo

        .INPUTS
            System.String

        .OUTPUTS
            System.Data.DataSet

        .NOTES
            AUTHOR : Jason Cole
    #>

    [CmdletBinding(DefaultParameterSetName = 'BuildDoc')]
    [OutputType([System.Data.DataSet])]
    param (
        # An Xml file created by the Get-NtapVserverData function (can also be *.xml.gz). Also accepts pipeline input from
        # the Get-NtapVserverData function.
        [Parameter(Mandatory, Position = 0, ValueFromPipeline)]
        [ValidateScript({
                if ($_ -isnot [System.Management.Automation.PSCustomObject]) {
                    if ($_ -is [System.IO.FileInfo] -or $_ -is [System.String]) {
                        if ([System.IO.Path]::GetExtension($_) -ne '.xml' -and [System.IO.Path]::GetExtension($_) -ne '.gz') {
                            throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]'Invalid file extension specified for the InputObject parameter.', 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $_))
                        }
                    }
                    else {
                        throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]'Invalid input object specified.', 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $_))
                    }
                }

                $true
            })]
        [System.Object[]]$InputObject,

        # Excludes the specified table(s) from the output documentation. This is useful for tables that might be quite large and
        # will take a long time to process (such as CifsShares).
        [Parameter()]
        [Alias('ExcludeSection')]
        [string[]]$ExcludeTable,

        # This will exclude a pre-defined list of tables that have the potential to process slowly due to a large number of rows. See the FAQ entry for more information.
        [Parameter()]
        [switch]$Terse,

        # If specified, the rows in the DataTable will be sorted with a NaturalSort routine (this may increase the time necessary to complete the sort)
        [Parameter()]
        [switch]$UseNaturalSort,

        # See PARAMETER entry above.
        [Parameter()]
        [int]$SanitizeLevel = 0,

        # The path to Xml file containing sanitization options and string mappings. If this file does not currently exist, a
        # copy of the template will be used to generate a new file.
        #
        # Note: If the sanitize level specified contains only IP Address, MAC Address and/or License Code then this parameter
        # is not necessary. Also, if this parameter is not specified then a temporary file will be created and subsequently
        # deleted upon completion.
        [Parameter()]
        [ValidateScript({
                if ([System.IO.Path]::GetExtension($_) -ne '.xml') {
                    throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]'Invalid file extension specified for the SanitizeMappingsXmlFile parameter.', 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $_))
                }

                $true
            })]
        [string]$SanitizeMappingsXmlFile,

        # If set to true then the sanitize mappings file specified will be reset to the defaults. (Default: False)
        [Parameter()]
        [switch]$ResetSanitizeMappings
    )

    dynamicparam {
        $paramDictionary = New-Object -TypeName System.Management.Automation.RuntimeDefinedParameterDictionary

        if ($PSCmdlet.MyInvocation.BoundParameters['Debug'].IsPresent) {
            ($attributeCollection = New-Object -TypeName System.Collections.ObjectModel.Collection[System.Attribute]).Add(
                [Parameter]@{
                    ParameterSetName = '__AllParameterSets'
                    HelpMessage      = 'Process only the specified tables.'
                }
            )

            $attributeCollection.Add((New-Object -TypeName System.Management.Automation.AliasAttribute -ArgumentList 'IncludeSection'))

            $paramDictionary.Add(
                'IncludeTable', (New-Object -TypeName System.Management.Automation.RuntimeDefinedParameter -ArgumentList ('IncludeTable', [string[]], $attributeCollection))
            )
        }

        if ($paramDictionary.Count -gt 0) {
            return $paramDictionary
        }
    }

    begin {
        if ($PSCmdlet.MyInvocation.BoundParameters['Debug'].IsPresent) {
            $Local:DebugPreference = 'Continue'
        }

        if ($SanitizeLevel -gt 0) {
            if (($SanitizeLevel -band 10112) -gt 0) {
                Write-Warning -Message 'The specified SanitizeLevel options may significantly increase the amount of time required to perform the sanitization (depending on the number of entries to process per table).'
            }

            if (($SanitizeLevel -bor 34) -ne 34) {
                $defaultSanitizeMappingsXmlFile = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath("$( ([System.IO.FileInfo]$PSCmdlet.MyInvocation.MyCommand.Module.Path).DirectoryName )/Resources/SanitizeMappings_Template-ONTAP.xml")

                if ([string]::IsNullOrWhiteSpace($SanitizeMappingsXmlFile)) {
                    if (($SanitizeLevel -bor 35) -ne 35) {
                        Write-Warning -Message 'No sanitization mappings file specified so mappings will be discarded upon completion.'
                    }

                    ($SanitizeMappings = New-Object -TypeName System.Xml.XmlDocument).Load($defaultSanitizeMappingsXmlFile)
                }
                else {
                    try {
                        $sanitizeMappingsXmlFileInfo = [System.IO.FileInfo]$ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath($SanitizeMappingsXmlFile)
                    }
                    catch {
                        $Global:Error.RemoveAt(0)

                        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]'Invalid file path specified for the SanitizeMappingsXmlFile parameter.', 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $SanitizeMappingsXmlFile)))
                    }

                    if ($sanitizeMappingsXmlFileInfo.Exists) {
                        ($SanitizeMappings = New-Object -TypeName System.Xml.XmlDocument).Load($sanitizeMappingsXmlFileInfo.FullName)

                        if ((($SanitizeLevel -bor 35) -eq 35) -and ($SanitizeMappings.Sanitization.Sections.SelectSingleNode("//Section[@Name=`"IpAddress`"]").SanitizeMethod -eq 'Octets')) {
                            Write-Warning -Message 'No sanitization mappings file necessary with specified sanitization options.'
                        }
                        else {
                            if ($ResetSanitizeMappings.IsPresent) {
                                Write-Verbose -Message 'Resetting all sanitize mappings to the defaults.'

                                Reset-SanitizeMappingsList -SanitizeMappingsXmlFile $sanitizeMappingsXmlFileInfo
                            }
                        }
                    }
                    else {
                        if (($SanitizeLevel -bor 35) -eq 35) {
                            Write-Warning -Message 'No sanitization mappings file necessary with specified sanitization options.'
                        }
                        else {
                            if (-not $sanitizeMappingsXmlFileInfo.Directory.Exists) {
                                try {
                                    [void](New-Item -Path $sanitizeMappingsXmlFileInfo.DirectoryName -ItemType Directory -ErrorAction Stop)
                                }
                                catch {
                                    $Global:Error.RemoveAt(0)

                                    $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('Unable to create destination folder.', $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $sanitizeMappingsXmlFileInfo.DirectoryName)))
                                }
                            }

                            Write-Verbose -Message 'SanitizeMappingsXmlFile not found, copying from default template.'

                            Copy-Item -Path $defaultSanitizeMappingsXmlFile -Destination $sanitizeMappingsXmlFileInfo.FullName

                            ($SanitizeMappings = New-Object -TypeName System.Xml.XmlDocument).Load($sanitizeMappingsXmlFileInfo.FullName)
                        }
                    }
                }
            }
        }

        $dataSet = $null
    }
    process {
        Write-Verbose -Message "Initializing function: $( $PSCmdlet.MyInvocation.MyCommand.Name ) ($( $PSCmdlet.MyInvocation.MyCommand.ModuleName ) v$( (Get-NtapDocsVersion -AsVersion).ToString() ))"

        :inputObjectLoop foreach ($object in $InputObject) {
            if ($null -eq $object) {
                continue
            }

            if ($object -is [System.Management.Automation.PSCustomObject]) {
                Write-Verbose -Message 'Reading input data from pipeline'

                $inputData = $object

                if (-not $inputData) {
                    Show-ErrorMessage -Message 'No data found in input object.'

                    continue
                }
            }
            #elseif ($object -is [System.Xml.XmlDocument]) {
            #    Write-Verbose -Message 'Converting OneCollect input data from pipeline'

            #    $inputData = Convert-NtapOneCollectClusterData -InputObject $object

            #    if (-not $inputData) {
            #        Show-ErrorMessage -Message 'No data found in input object.'

            #        continue
            #    }
            #}
            else {
                try {
                    $inputFileInfo = $(
                        if ($object -is [System.IO.FileInfo]) {
                            [System.IO.FileInfo]$ExecutionContext.SessionState.Path.GetResolvedPSPathFromPSPath($object.FullName).ProviderPath
                        }
                        else {
                            [System.IO.FileInfo]$ExecutionContext.SessionState.Path.GetResolvedPSPathFromPSPath($object).ProviderPath
                        }
                    )
                }
                catch {
                    $Global:Error.RemoveAt(0)

                    Write-Error -ErrorRecord (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]'Input file specified does not exist.', $_.FullyQualifiedErrorId, [System.Management.Automation.ErrorCategory]::InvalidArgument, $object))

                    continue
                }

                if ($inputFileInfo.Extension -eq '.gz') {
                    Write-Verbose -Message "Expanding FileName: $( $inputFileInfo.FullName )"

                    $inputXmlFile = Expand-GzipFile -InFile $inputFileInfo.FullName -PassThru
                }
                else {
                    $inputXmlFile = $inputFileInfo
                }

                try {
                    switch -Wildcard (Get-Content -Path $inputXmlFile.FullName -TotalCount 3) {
                        '<Objs*' {
                            Write-Verbose -Message "Reading input data from FileName: $( $inputXmlFile.Name )"

                            $inputData = Import-Clixml -Path $inputXmlFile.FullName -ErrorAction Stop
                        }
                        '*<netapp><onecollect>*' {
                            Write-Warning -Message 'OneCollect data not yet supported for Format-NtapVserverData function'

                            continue inputObjectLoop
                            #Write-Verbose -Message "Converting OneCollect data from FileName: $( $inputXmlFile.Name )"

                            #$inputData = Convert-NtapClusterOneCollectData -InputObject $inputXmlFile -ErrorAction Stop
                        }
                        '*<NetAppDocs.*' {
                            Write-Warning -Message "Data in the XML file has already been processed by this $( $PSCmdlet.MyInvocation.MyCommand.Name ) function. File: $( $inputXmlFile.FullName )"

                            continue inputObjectLoop
                        }
                    }

                    if (-not $inputData) {
                        Show-ErrorMessage -Message "No data found in input file: $( $inputFileInfo.FullName )"

                        continue
                    }
                }
                catch {
                    $Global:Error.RemoveAt(0)

                    Write-Error -ErrorRecord (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("There was an error reading the XML file. Reason: $( $_.Exception.Message )", $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $inputXmlFile.FullName))

                    continue
                }
                finally {
                    if ($inputFileInfo.Extension -eq '.gz') {
                        Remove-Item -Path $inputXmlFile.FullName -ErrorAction SilentlyContinue
                    }
                }
            }

            foreach ($data in $inputData) {
                if ($null -eq $data) {
                    continue
                }

                $dataCollectVersion = ($data.NetAppDocsVersion, $data.NetAppDocsVersionDataCollection)[[byte](! ($data.psobject.Properties.Name -contains 'NetAppDocsVersion'))]

                if ((Get-NtapDocsVersion -AsFeatureVersion) -ne [version]::Parse((Get-VersionString -InputVersion $dataCollectVersion -AsFeatureVersion))) {
                    Write-Warning -Message "The NetAppDocs version ($( (Get-NtapDocsVersion -AsVersion).ToString() )) being used to format the data is not the same as the version ($( $dataCollectVersion.ToString() )) used to collect the data for this system: $( $data.SystemName ). It is recommended that the versions be the same, as there could be missing or incomplete data."
                }

                $baseTypeName = $data.pstypenames[0] -replace 'Deserialized.' -replace '.Data' -replace '.OneCollect'

                if ($baseTypeName -notlike 'NetAppDocs.NcVserver*') {
                    Show-IncorrectTypeErrorMessage -BaseTypeName $baseTypeName -SystemName $data.SystemName

                    continue
                }

                $functionStartTime = [datetime]::Now

                if ($data.System.OntapiVersion -isnot [System.Version]) {
                    $data.System.OntapiVersion = [version]::Parse($data.System.OntapiVersion.ToString())
                }

                if (-not $dataSet) {
                    Write-Verbose -Message 'Initializing DataSet'

                    $dsSchemaFile = Get-NtapDocsSchemaFile -BaseTypeName $baseTypeName -DocumentType $PSCmdlet.ParameterSetName

                    if (-not $dsSchemaFile) {
                        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("An unexpected error occurred. DataSet schema file not found. SystemType: $baseTypeName#$( $PSCmdlet.ParameterSetName )", 'FileNotFound', [System.Management.Automation.ErrorCategory]::InvalidResult, $dsSchemaFile)))
                    }

                    try {
                        $dataSet = Read-DataSet -InputObject $dsSchemaFile
                    }
                    catch {
                        $PSCmdlet.ThrowTerminatingError($_)
                    }

                    if ($Terse.IsPresent -and ($dataSet.Tables | Where-Object { $_.ExtendedProperties['ExcludeIfTersePresent'] } | Measure-Object).Count -gt 0) {
                        Write-Warning -Message "The -Terse parameter was specified, therefore the following tables will be excluded from the output (see the FAQ for additional information): $( ($dataSet.Tables | Where-Object { $_.ExtendedProperties['ExcludeIfTersePresent'] }).TableName -join ', ' )"
                    }

                    for ($t = 0; $t -lt $dataSet.Tables.Count; $t++) {
                        if ($Terse.IsPresent -and $dataSet.Tables[$t].ExtendedProperties['ExcludeIfTersePresent']) {
                            $dataSet.Tables[$t].ExtendedProperties.Add('Excluded', $true)
                            continue
                        }

                        if ($ExcludeTable -contains $dataSet.Tables[$t].TableName) {
                            $dataSet.Tables[$t].ExtendedProperties.Add('Excluded', $true)
                        }

                        if ($PSCmdlet.MyInvocation.BoundParameters.ContainsKey('IncludeTable') -and $PSCmdlet.MyInvocation.BoundParameters['IncludeTable'] -notcontains $dataSet.Tables[$t].TableName) {
                            $dataSet.Tables[$t].ExtendedProperties.Add('NotIncluded', $true)
                        }
                    }
                }

                Write-Verbose -Message "Formatting input data for SystemName: $( $data.SystemName )"

                $docObj = [pscustomobject]@{
                    Data = $data
                }

                $parentBaseTypeName = $baseTypeName -replace '[.]FSxOntap'
                $docObj.pstypenames.Insert(0, "$parentBaseTypeName.Document")
                $docObj.pstypenames.Insert(0, "$parentBaseTypeName.Document#$( $PSCmdlet.ParameterSetName )")

                foreach ($tableName in (@( $dataSet.Tables.TableName | Where-Object { $_ -ne 'FindingsSummary' -and $_ -ne 'DataDiscoveryErrors' } ) + 'DataDiscoveryErrors')) {
                    if ($dataSet.Tables[$tableName].ExtendedProperties['Excluded'] -or $dataSet.Tables[$tableName].ExtendedProperties['NotIncluded']) {
                        continue
                    }

                    Write-Verbose -Message $(
                        if ($dataSet.Tables[$tableName].ExtendedProperties['ShowExtendedTimeMessage']) {
                            "Formatting table: $tableName (this table may take an extended amount of time to process)"
                        }
                        else {
                            "Formatting table: $tableName"
                        }
                    )

                    $tableStartTime = [datetime]::Now

                    foreach ($dataRow in ($docObj.$tableName | ConvertTo-NtapDataRow -DataTable $dataSet.Tables[$tableName])) {
                        Add-RowToDataTable -DataRow $dataRow
                    }

                    Write-Debug -Message "Elapsed Time: $( [System.Math]::Round(([datetime]::Now - $tableStartTime).TotalMilliseconds, 0) ) milliseconds for table: $tableName"
                }

                $statsParams = @{
                    FunctionPSCallStack = $(
                        if ($PSCmdlet.MyInvocation.InvocationName -ne $PSCmdlet.MyInvocation.MyCommand) {
                            Get-PSCallStack | Select-Object -Skip 1
                        }
                        else {
                            Get-PSCallStack
                        }
                    )
                    TargetDetails       = [pscustomobject]@{
                        SystemUid     = [string]::Join('_', ($data.Cluster.Info.ClusterUuid, $data.Cluster.Info.ClusterIdentifier)[[byte](! ($data.pstypenames[0] -like 'NetAppDocs.NcVserver*'))], $data.Vserver.Info.Uuid)
                        SystemType    = $data.pstypenames[0]
                        SystemVersion = ([regex]::Split(([regex]::Split($data.System.VersionInfo.Version, 'Release\s(?:\w+[__])?')[1]), '\s+'))[0].Replace(':', '')
                    }
                    ElapsedTicks        = ([datetime]::Now - $functionStartTime).Ticks
                }

                Write-Statistics @statsParams

                Remove-Variable -Name data, docObj -ErrorAction Ignore

                [System.GC]::Collect()
                [System.GC]::WaitForPendingFinalizers()
            }
        }
    }
    end {
        if ($dataSet -and $dataSet.HasChanges()) {
            $dataSet.ExtendedProperties['Timestamp'] = [datetime]::UtcNow.ToString('u')
            $dataSet.ExtendedProperties['SanitizeLevel'] = $SanitizeLevel

            $sortedDataSet = [NetAppDocs.Utilities.DataSetMethods]::SortDataSet($dataSet, $( if ($UseNaturalSort.IsPresent) { [NetAppDocs.Utilities.SortMethod]::Natural } else { [NetAppDocs.Utilities.SortMethod]::Standard } ))

            if ($SanitizeLevel -gt 0) {
                Invoke-SanitizeDataSet -DataSet $sortedDataSet -SanitizeLevel $SanitizeLevel -SanitizeMappings $SanitizeMappings

                if (
                    -not [string]::IsNullOrWhiteSpace($SanitizeMappingsXmlFile) -and
                    ((($SanitizeLevel -bor 35) -ne 35) -or ($SanitizeMappings.Sanitization.Sections.SelectSingleNode("//Section[@Name=`"IpAddress`"]").SanitizeMethod -ne 'Octets'))
                ) {
                    Write-Verbose -Message 'Exporting sanitize mappings'

                    Export-SanitizeMappingsList -SanitizeMappings $SanitizeMappings -SanitizeMappingsXmlFile $sanitizeMappingsXmlFileInfo
                }
            }

            $sortedDataSet
        }
    }
}

# SIG # Begin signature block
# MIIq0AYJKoZIhvcNAQcCoIIqwTCCKr0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCD3CqDAmGJaK2/J
# atKRiAZS3ju+UtdZyYmcGIuYLksnU6CCJGgwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggWNMIIEdaADAgECAhAOmxiO
# +dAt5+/bUOIIQBhaMA0GCSqGSIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYD
# VQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAi
# BgNVBAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAw
# MDBaFw0zMTExMDkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
# Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERp
# Z2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBAL/mkHNo3rvkXUo8MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsb
# hA3EMB/zG6Q4FutWxpdtHauyefLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iT
# cMKyunWZanMylNEQRBAu34LzB4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGb
# NOsFxl7sWxq868nPzaw0QF+xembud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclP
# XuU15zHL2pNe3I6PgNq2kZhAkHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCr
# VYJBMtfbBHMqbpEBfCFM1LyuGwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFP
# ObURWBf3JFxGj2T3wWmIdph2PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTv
# kpI6nj3cAORFJYm2mkQZK37AlLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWM
# cCxBYKqxYxhElRp2Yn72gLD76GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls
# 5Q5SUUd0viastkF13nqsX40/ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBR
# a2+xq4aLT8LWRV+dIPyhHsXAj6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6
# MIIBNjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qY
# rhwPTzAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8E
# BAMCAYYweQYIKwYBBQUHAQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5k
# aWdpY2VydC5jb20wQwYIKwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDig
# NoY0aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNybDARBgNVHSAECjAIMAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCg
# v0NcVec4X6CjdBs9thbX979XB72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQT
# SnovLbc47/T/gLn4offyct4kvFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh
# 65ZyoUi0mcudT6cGAxN3J0TU53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSw
# uKFWjuyk1T3osdz9HNj0d1pcVIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAO
# QGPFmCLBsln1VWvPJ6tsds5vIy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjD
# TZ9ztwGpn1eqXijiuZQwggXMMIIENKADAgECAhAg429sPxgagUb53pPffJfkMA0G
# CSqGSIb3DQEBDAUAMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBS
# MzYwHhcNMjEwOTA5MDAwMDAwWhcNMjMwOTA5MjM1OTU5WjBQMQswCQYDVQQGEwJV
# UzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UECgwMTmV0QXBwLCBJbmMuMRUw
# EwYDVQQDDAxOZXRBcHAsIEluYy4wggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGK
# AoIBgQC4kYYj/oViZD9pN03hrqFjtwOz1Gx4eDWVV8IYUYEr2qpLeNYvWz26B/+E
# mYLboAGVpkYg0Wske3hjRpooZlMORm1S4+2C2NoQCstZ+kmNTevbXHJe/w1VNJrm
# fKjpXOJEfx8GUGDqCMR30dChVpsdBEavrM7T0hnfJLv18i19SGv3a+nhvB3iOjLP
# SEg8+DFkjD5jTI6fQ83DBEiVR4UEYq7yyH578ZEylhsRfJmK+uIREycEUk/NpFTw
# g6/7lxh/WYabCrZzqI4Ep3QataRKaAlxt3BiOlZkX4WGX3DYvSYltWOM2UlCcGpP
# m/1/LN8Xuvf+YO6H472QecHe59XtXbHSMKvqDuOEJu8Wk2LFMNK732o3Fc5QIHie
# 6JtNsgNYf+Vsm5EMBD1ZHNE+C3zlmQbbWIU5uEU1nhiodBPKsu0SkS6pARyKBO05
# DSc2gWWv0aukKi04ZAn+hAdSl5s1dgmU5shBvKHqe15K9ZRN4FFO/CgHE0BoGYQS
# UQVKwa0CAwEAAaOCAZwwggGYMB8GA1UdIwQYMBaAFA8qyyCHKLjsb0iuK1SmKaoX
# pM0MMB0GA1UdDgQWBBQuH643KcBMmb/Q6IZt+H9IrnXFwDAOBgNVHQ8BAf8EBAMC
# B4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDAzARBglghkgBhvhC
# AQEEBAMCBBAwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwIwJTAjBggrBgEFBQcC
# ARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQBMEkGA1UdHwRCMEAw
# PqA8oDqGOGh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVT
# aWduaW5nQ0FSMzYuY3JsMHkGCCsGAQUFBwEBBG0wazBEBggrBgEFBQcwAoY4aHR0
# cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQVIz
# Ni5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqG
# SIb3DQEBDAUAA4IBgQCOoGdXjP8Sif0h3ZvoDsIVfrJvQUdP9pZExRJGGj/Te6ML
# XyqHNc/G2WklNv+BC4ryNJ4zp0oneHOub55ejrYPfhgP9oFWS8oOPJbrp3rAtINa
# OeKRC88WUYZSKYAxSX6kubEjnQD6cmazXjxVN6P8+6q9smeqF3xI+++c+ekMw3Xv
# 4EWRYXqN0/srfFp1WpoluwSQcZMqj9haSX0bgO6AAWH2AnVJMfXTIKz/0FQ/RW0y
# Ed5QYQqYInhA7IUz9dBziNpJlwONeF5j7xzgfYDY63WU6WrgJnGjYkQCOOsUx74j
# gUiMRe9zV48GS8Vxz22c/TQs0x4/1GmOSJvUHEk3GseBmB3v+yEhh/D6zWOfYP4X
# D/9b91CxmugTuepuaJSeRg+qUm3KarKsOCUF/CLqUrNh/JwKrWD1cghRaYLvMucs
# ScksHHe7ZDvb2OtvxWXjPk1d1NKvEwFJSS6hIVIbug9x28AJqOoP5Pn9wZvJSzvW
# uQJGCLscdGyKefdCo30wggYaMIIEAqADAgECAhBiHW0MUgGeO5B5FSCJIRwKMA0G
# CSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290
# IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFQxCzAJBgNVBAYT
# AkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28g
# UHVibGljIENvZGUgU2lnbmluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEBAQUAA4IB
# jwAwggGKAoIBgQCbK51T+jU/jmAGQ2rAz/V/9shTUxjIztNsfvxYB5UXeWUzCxEe
# AEZGbEN4QMgCsJLZUKhWThj/yPqy0iSZhXkZ6Pg2A2NVDgFigOMYzB2OKhdqfWGV
# oYW3haT29PSTahYkwmMv0b/83nbeECbiMXhSOtbam+/36F09fy1tsB8je/RV0mIk
# 8XL/tfCK6cPuYHE215wzrK0h1SWHTxPbPuYkRdkP05ZwmRmTnAO5/arnY83jeNzh
# P06ShdnRqtZlV59+8yv+KIhE5ILMqgOZYAENHNX9SJDm+qxp4VqpB3MV/h53yl41
# aHU5pledi9lCBbH9JeIkNFICiVHNkRmq4TpxtwfvjsUedyz8rNyfQJy/aOs5b4s+
# ac7IH60B+Ja7TVM+EKv1WuTGwcLmoU3FpOFMbmPj8pz44MPZ1f9+YEQIQty/NQd/
# 2yGgW+ufflcZ/ZE9o1M7a5Jnqf2i2/uMSWymR8r2oQBMdlyh2n5HirY4jKnFH/9g
# Rvd+QOfdRrJZb1sCAwEAAaOCAWQwggFgMB8GA1UdIwQYMBaAFDLrkpr/NZZILyhA
# QnAgNpFcF4XmMB0GA1UdDgQWBBQPKssghyi47G9IritUpimqF6TNDDAOBgNVHQ8B
# Af8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcD
# AzAbBgNVHSAEFDASMAYGBFUdIAAwCAYGZ4EMAQQBMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Um9vdFI0Ni5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ1Jvb3RSNDYu
# cDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQwFAAOCAgEABv+C4XdjNm57oRUgmxP/BP6YdURhw1aVcdGRP4Wh60BAscjW
# 4HL9hcpkOTz5jUug2oeunbYAowbFC2AKK+cMcXIBD0ZdOaWTsyNyBBsMLHqafvIh
# rCymlaS98+QpoBCyKppP0OcxYEdU0hpsaqBBIZOtBajjcw5+w/KeFvPYfLF/ldYp
# mlG+vd0xqlqd099iChnyIMvY5HexjO2AmtsbpVn0OhNcWbWDRF/3sBp6fWXhz7Dc
# ML4iTAWS+MVXeNLj1lJziVKEoroGs9Mlizg0bUMbOalOhOfCipnx8CaLZeVme5yE
# Lg09Jlo8BMe80jO37PU8ejfkP9/uPak7VLwELKxAMcJszkyeiaerlphwoKx1uHRz
# NyE6bxuSKcutisqmKL5OTunAvtONEoteSiabkPVSZ2z76mKnzAfZxCl/3dq3dUNw
# 4rg3sTCggkHSRqTqlLMS7gjrhTqBmzu1L90Y1KWN/Y5JKdGvspbOrTfOXyXvmPL6
# E52z1NZJ6ctuMFBQZH3pwWvqURR8AgQdULUvrxjUYbHHj95Ejza63zdrEcxWLDX6
# xWls/GDnVNueKjWUH3fTv1Y8Wdho698YADR7TNx8X8z2Bev6SivBBOHY+uqiirZt
# g0y9ShQoPzmCcn63Syatatvx157YK9hlcPmVoa1oDE5/L9Uo2bC5a4CH2Rwwggau
# MIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJ
# BgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5k
# aWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAe
# Fw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcw
# FQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3Rl
# ZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9Ge
# TKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0
# hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZl
# jZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAsh
# aG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVY
# TXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1
# biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCir
# c0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+
# DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA
# +bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42Pg
# puE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzS
# M7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU
# uhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6
# mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsG
# AQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29t
# MEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNl
# cnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3Js
# My5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAE
# GTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1Z
# jsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8d
# B+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVp
# P0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp8
# 76i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2
# nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3
# ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQ
# txMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc
# 4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+Y
# AN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZ
# vAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQr
# H4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIGwDCCBKigAwIBAgIQDE1p
# ckuU+jwqSj0pB4A9WjANBgkqhkiG9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4XDTIyMDkyMTAwMDAw
# MFoXDTMzMTEyMTIzNTk1OVowRjELMAkGA1UEBhMCVVMxETAPBgNVBAoTCERpZ2lD
# ZXJ0MSQwIgYDVQQDExtEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMiAtIDIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDP7KUmOsap8mu7jcENmtuh6BSFdDMa
# JqzQHFUeHjZtvJJVDGH0nQl3PRWWCC9rZKT9BoMW15GSOBwxApb7crGXOlWvM+xh
# iummKNuQY1y9iVPgOi2Mh0KuJqTku3h4uXoW4VbGwLpkU7sqFudQSLuIaQyIxvG+
# 4C99O7HKU41Agx7ny3JJKB5MgB6FVueF7fJhvKo6B332q27lZt3iXPUv7Y3UTZWE
# aOOAy2p50dIQkUYp6z4m8rSMzUy5Zsi7qlA4DeWMlF0ZWr/1e0BubxaompyVR4aF
# eT4MXmaMGgokvpyq0py2909ueMQoP6McD1AGN7oI2TWmtR7aeFgdOej4TJEQln5N
# 4d3CraV++C0bH+wrRhijGfY59/XBT3EuiQMRoku7mL/6T+R7Nu8GRORV/zbq5Xwx
# 5/PCUsTmFntafqUlc9vAapkhLWPlWfVNL5AfJ7fSqxTlOGaHUQhr+1NDOdBk+lbP
# 4PQK5hRtZHi7mP2Uw3Mh8y/CLiDXgazT8QfU4b3ZXUtuMZQpi+ZBpGWUwFjl5S4p
# kKa3YWT62SBsGFFguqaBDwklU/G/O+mrBw5qBzliGcnWhX8T2Y15z2LF7OF7ucxn
# EweawXjtxojIsG4yeccLWYONxu71LHx7jstkifGxxLjnU15fVdJ9GSlZA076XepF
# cxyEftfO4tQ6dwIDAQABo4IBizCCAYcwDgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB
# /wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwIAYDVR0gBBkwFzAIBgZngQwB
# BAIwCwYJYIZIAYb9bAcBMB8GA1UdIwQYMBaAFLoW2W1NhS9zKXaaL3WMaiCPnshv
# MB0GA1UdDgQWBBRiit7QYfyPMRTtlwvNPSqUFN9SnDBaBgNVHR8EUzBRME+gTaBL
# hklodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0
# MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3JsMIGQBggrBgEFBQcBAQSBgzCBgDAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFgGCCsGAQUFBzAC
# hkxodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRS
# U0E0MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3J0MA0GCSqGSIb3DQEBCwUAA4IC
# AQBVqioa80bzeFc3MPx140/WhSPx/PmVOZsl5vdyipjDd9Rk/BX7NsJJUSx4iGNV
# CUY5APxp1MqbKfujP8DJAJsTHbCYidx48s18hc1Tna9i4mFmoxQqRYdKmEIrUPwb
# tZ4IMAn65C3XCYl5+QnmiM59G7hqopvBU2AJ6KO4ndetHxy47JhB8PYOgPvk/9+d
# EKfrALpfSo8aOlK06r8JSRU1NlmaD1TSsht/fl4JrXZUinRtytIFZyt26/+YsiaV
# OBmIRBTlClmia+ciPkQh0j8cwJvtfEiy2JIMkU88ZpSvXQJT657inuTTH4YBZJwA
# wuladHUNPeF5iL8cAZfJGSOA1zZaX5YWsWMMxkZAO85dNdRZPkOaGK7DycvD+5sT
# X2q1x+DzBcNZ3ydiK95ByVO5/zQQZ/YmMph7/lxClIGUgp2sCovGSxVK05iQRWAz
# gOAj3vgDpPZFR+XOuANCR+hBNnF3rf2i6Jd0Ti7aHh2MWsgemtXC8MYiqE+bvdgc
# mlHEL5r2X6cnl7qWLoVXwGDneFZ/au/ClZpLEQLIgpzJGgV8unG1TnqZbPTontRa
# mMifv427GFxD9dAq6OJi7ngE273R+1sKqHB+8JeEeOMIA11HLGOoJTiXAdI/Otrl
# 5fbmm9x+LMz/F0xNAKLY1gEOuIvu5uByVYksJxlh9ncBjDGCBb4wggW6AgEBMGgw
# VDELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDErMCkGA1UE
# AxMiU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIFIzNgIQIONvbD8YGoFG
# +d6T33yX5DANBglghkgBZQMEAgEFAKCBhDAYBgorBgEEAYI3AgEMMQowCKACgACh
# AoAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAM
# BgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCCfOAEsfXgEkp6qUNnQUevGJoKy
# 0Usg1/pHbhf4IAlGqzANBgkqhkiG9w0BAQEFAASCAYAa6sZG2Z46FQcweNGDbH1m
# Y9559MNcgYwhZBhklFiTqXwqrM8Bv5nHk7Lqd6IsLfRujgtw3ey/0WZs6tHssVuU
# 7MxV6+2j0WPPpBmOmKI5wvbpwfD1/fAm9ieUaV8QVFqNpnCA7YVFfa3U91nclrfR
# afiHXfD18yCKORqVdhIUrdlfwZnlJPdoIJG61pbQ3g9Sn3PLHILBFi+Ra3Agj0Cv
# RHCYyUMHCf/SvS9TbS8ZVwoM93RxSjRzltdzf9GtKr9Qri6lskyvj43xep3UrIAd
# GySQpKdYHIpC4Z5jTq0YyKNioJ8RPFEq3q/hFNTX+13idnBP+niuXxf1AfzqUbmu
# 2qh6Dt5mqXSg4tNLpGQ/xhvjTw7MrGJInxCkKAf0K3TVlWZcLv3HN7IKTiZFMMlO
# q36H932WtEu0lWTzs4pOJsaPPJZdbCfmVbLJEqU9zFOasl7Kue9zsm1fdbze5lsK
# O0cNGJtR5Cg7yHoNlavBNGHu7uH9WZktqW6xpP74XvahggMgMIIDHAYJKoZIhvcN
# AQkGMYIDDTCCAwkCAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
# cnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
# SEEyNTYgVGltZVN0YW1waW5nIENBAhAMTWlyS5T6PCpKPSkHgD1aMA0GCWCGSAFl
# AwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUx
# DxcNMjMwNDAyMTUyODI3WjAvBgkqhkiG9w0BCQQxIgQgeES5+EkDuRg7fS/NOtP1
# DOYwA97DfQ0mDKksF7DAiVQwDQYJKoZIhvcNAQEBBQAEggIAGXM+I6NDy+fWP1FZ
# YXQMNMYm28S92le6GMAE8lX9qGckzamVdHlVRKuZgmnQIoazc8SqawGXqInTfoaE
# +7akBFKKdfy0mnYU1TBsGKEwd0jqyleqw5FderFVPKoaAy083cXYEDhR60vMuSTf
# sEDy59gb4J7yKxKAAXVvwGEiF3+JdRyfVrymRDXg3WEfIuuaYdum4UlGUStnVeo+
# eTmVrWZS8ND3g2wbNimJhGX2THLWZ7BLJnY2lDNN2Oq6DraNyiqvW3TIZ+jXNyt9
# AJqGXvxhdPWVh2jnxtMHz7UFM2XqWPRIpx49cW7t095ti0mI/wwzS303mBqKc7P8
# 0ss8U7j7bg/l7u/rSXlu3r2GCgLQ1628sjqAI3QwIxbjBGnCNdd/5SuIEolCCnYu
# GRQ6zrBq4OITO0k1nx++EgfUW0/W6Y/KZmPu2t9jQxS2hpKrItiszzWo8BBH/m0H
# IsmaOm4cXxxQ60C1CH0w5FvLscDu32cAsZCkEBSUrkqN2fgS4YuOh2SHYqtoaiRp
# T8IdaZZkrlC+/NlFe/1Uy31We1MF0pQKAZvnNpBxvqgGjtQavEyp/IKdnteoC2/v
# uxSVnAeNZyalMDX+erZw/nJpveGKyhfWLCYLZRf71sSbGbJHrk6Ggn1jNWmnNMWY
# 47GP6fUlPaRwtglBGyOrA9V62vY=
# SIG # End signature block
