function Get-NtapClusterAsupData {
    <#
        .SYNOPSIS
            Collects data for ONTAP storage systems using ASUPs.

        .DESCRIPTION
            Uses the node ASUPs to gather as much configuration information as possible for the cluster. This
            data can be used to automate build docs and health checks. It can also be used for historical
            configuration information. If the -Cluster parameter is used, the latest ASUPs of the specified
            AsupSubject type will be used. If older or specified ASUPs are desired, use the Find-NtapClusterAsup
            function and pass those results to this function. The ASUPs will be validated to determine that an
            ASUP from each node in the cluster is present, complete and generated at similar times.

        .EXAMPLE
            PS> Get-NtapClusterAsupData -Cluster 'den-cdot:394e82bf-485c-11e2-86b3-123478563412' | Export-Clixml -Path 'D:\den-cdot_Asup_raw.xml'

            Collects the cluster information from the cluster name and cluster uuid specified and saves the results to a .xml file for
            further processing.

        .EXAMPLE
            PS> Get-NtapClusterAsupData -Cluster 'den-cdot:394e82bf-485c-11e2-86b3-123478563412' -AsupSubject user_triggered | Export-Clixml -Path 'D:\den-cdot_Asup_raw.xml'

            Collects the cluster information from the cluster name and cluster uuid specified, using the latest
            user_triggered ASUP set, and saves the results to a .xml file for further processing.

        .EXAMPLE
            PS> Get-NtapClusterAsupData -Cluster 'den-cdot:394e82bf-485c-11e2-86b3-123478563412' | Format-NtapClusterData | Out-NtapDocument -WordFile 'D:\Output\den-cdot_Asup.docx' -ExcelFile 'D:\Output\den-cdot_Asup.xlsx'

            Collects the cluster information from the cluster name and cluster uuid specified, formats the data and creates
            the specified Word and Excel documents.

        .EXAMPLE
            PS> Get-NtapClusterAsupData -AsupId 2015012500120554!2015012423440127 | Export-Clixml -Path 'D:\Output\den-cdot-Asup_raw.xml'

            Perform the data collection using specific ASUPIDs.

        .EXAMPLE
            PS> Find-NtapClusterAsup -Cluster 'den-cdot:394e82bf-485c-11e2-86b3-123478563412' -StartDate 10-15-2014 -EndDate 11-15-2014 | Get-NtapClusterAsupData | Export-Clixml -Path 'D:\Output\den-cdot_Asup_raw.xml'

            Finds the first validated weekly ASUP set for the specified cluster by searching between the dates
            specified, performs the data collection and saves the results to a .xml file for further processing.

        .LINK
            https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs

        .LINK
            Format-NtapClusterData

        .INPUTS
            System.Xml.XmlElement

        .OUTPUTS
            System.Management.Automation.PSCustomObject

        .NOTES
            AUTHOR : Jason Cole
            REQUIRES : Internal connection to NetApp
    #>

    [CmdletBinding(DefaultParameterSetName = 'ByClusterInfo')]
    [OutputType([System.Management.Automation.PSCustomObject])]
    param (
        # Searches the ASUP database for the specified ClusterName and ClusterIdentifier
        # (e.g., den-cdot:394e82bf-485c-11e2-86b3-123478563412).
        [Parameter(Mandatory, Position = 0, ParameterSetName = 'ByClusterInfo', ValueFromPipeline)]
        [ValidateScript({
                if ($_ -notmatch '^\s*[^!:]+[:][^!:]+\s*$') {
                    throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]"$_ is not a valid format for the Cluster parameter (e.g., ClusterName:ClusterIdentifier).", 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $_))
                }

                $true
            })]
        [string[]]$Cluster,

        # The ASUPID(s) to search for. An ASUPID from each node in the cluster is required. Each node's ASUPID must be
        # separated by an exclamation point (e.g., 2015012500120554!2015012423440127). Sets of ASUPIDs from multiple
        # clusters must be comma separated (e.g., 2015012500120554!2015012423440127, 2015062302150043!2015062302150041).
        [Parameter(Mandatory, Position = 0, ParameterSetName = 'ByAsupId')]
        [ValidateScript({
                if ($_ -notmatch '^\s*\d+(?:\s*[!]\s*\d+)*$') {
                    throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]"$_ is not a valid format for the AsupId parameter (e.g., 'ClusterNode1_AsupId!ClusterNode2_AsupId') or (single node cluster): 'ClusterNode1_AsupId'.", 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $_))
                }

                $true
            })]
        [string[]]$AsupId,

        # An ASUP object passed from the Find-NtapClusterAsup function. This is used when using older ASUPs to
        # generate the data.
        [Parameter(Mandatory, Position = 0, ParameterSetName = 'ByAsupObject', ValueFromPipeline)]
        [ValidateNotNullOrEmpty()]
        [System.Xml.XmlElement[]]$AsupObject,

        # The type of ASUP to search for (weekly or user_triggered).
        [Parameter(ParameterSetName = 'ByClusterInfo')]
        [ValidateSet('weekly', 'user_triggered')]
        [string]$AsupSubject = 'weekly',

        # Forces the use of an invalid ASUP (i.e., ASUP content missing or mismatched generated dates between
        # the nodes in the cluster).
        [Parameter()]
        [switch]$Force
    )

    begin {
        if ($PSCmdlet.MyInvocation.PipelinePosition -eq 1) {
            try {
                Invoke-SsoLogin -ErrorAction Stop
            }
            catch {
                $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($_.Exception.Message, 'InvalidResult', [System.Management.Automation.ErrorCategory]::InvalidResult, 'ASUP REST Server')))
            }

            Write-Verbose -Message 'Validating connectivity to ASUP REST server'

            if (-not (Test-AsupRestConnectivity)) {
                $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('Unable to contact NetApp ASUP REST server. Please ensure you have an internal connection to NetApp.', 'ResourceUnavailable', [System.Management.Automation.ErrorCategory]::ResourceUnavailable, 'ASUP REST Server')))
            }
        }
    }
    process {
        Write-Verbose -Message "Initializing function: $( $PSCmdlet.MyInvocation.MyCommand.Name ) ($( $PSCmdlet.MyInvocation.MyCommand.ModuleName ) v$( (Get-NtapDocsVersion -AsVersion).ToString() ))"

        switch -Exact ($PSCmdlet.ParameterSetName) {
            'ByClusterInfo' {
                foreach ($cdot in ($Cluster | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })) {
                    if (($asupSet = Find-NtapClusterAsup -Cluster $cdot -AsupSubject $AsupSubject -ValidateAsupSystemType Cluster)) {
                        if (($asupSet | Where-Object { $_.is_valid -eq $false } | Measure-Object).Count -eq 0 -or $Force.IsPresent) {
                            try {
                                $functionStartTime = [datetime]::Now

                                $asupData = Get-ClusterAsupData -AsupSet $asupSet

                                $asupData

                                $statsParams = @{
                                    FunctionPSCallStack = Get-PSCallStack
                                    TargetDetails       = [pscustomobject]@{
                                        SystemUid     = $asupData.Cluster.Info.ClusterIdentifier
                                        SystemType    = $asupData.pstypenames[0]
                                        SystemVersion = ([regex]::Split(([regex]::Split($asupData.System.VersionInfo.Version, 'Release\s(?:\w+[__])?')[1]), '\s+'))[0].Replace(':', '')
                                    }
                                    ElapsedTicks        = ([datetime]::Now - $functionStartTime).Ticks
                                    ExcludeParameter    = 'Cluster'
                                }

                                Write-Statistics @statsParams
                            }
                            catch {
                                Write-Error -ErrorRecord $_
                            }
                        }
                    }
                }
            }
            'ByAsupId' {
                foreach ($cdotAsupId in ($AsupId | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })) {
                    if (($asupSet = Find-NtapClusterAsup -AsupId $cdotAsupId -ValidateAsupSystemType Cluster)) {
                        if (($asupSet | Where-Object { $_.is_valid -eq $false } | Measure-Object).Count -eq 0 -or $Force.IsPresent) {
                            try {
                                $functionStartTime = [datetime]::Now

                                $asupData = Get-ClusterAsupData -AsupSet $asupSet

                                $asupData

                                $statsParams = @{
                                    FunctionPSCallStack = Get-PSCallStack
                                    TargetDetails       = [pscustomobject]@{
                                        SystemUid     = $asupData.Cluster.Info.ClusterIdentifier
                                        SystemType    = $asupData.pstypenames[0]
                                        SystemVersion = ([regex]::Split(([regex]::Split($asupData.System.VersionInfo.Version, 'Release\s(?:\w+[__])?')[1]), '\s+'))[0].Replace(':', '')
                                    }
                                    ElapsedTicks        = ([datetime]::Now - $functionStartTime).Ticks
                                    ExcludeParameter    = 'AsupId'
                                }

                                Write-Statistics @statsParams
                            }
                            catch {
                                Write-Error -ErrorRecord $_
                            }
                        }
                    }
                }
            }
            'ByAsupObject' {
                if ($AsupObject) {
                    if (($AsupObject | Where-Object { $_.sys_operating_mode -ne 'Cluster-Mode' } | Measure-Object).Count -gt 0) {
                        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('One or more ASUPS in the ASUP set is from a 7-Mode system. Please use Get-NtapFilerAsupData for 7-Mode ASUPs.', 'InvalidType', [System.Management.Automation.ErrorCategory]::InvalidType, $AsupObject)))
                    }

                    if (($AsupObject | Where-Object { $_.is_valid -eq $false } | Measure-Object).Count -eq 0 -or $Force.IsPresent) {
                        try {
                            $functionStartTime = [datetime]::Now

                            $asupData = Get-ClusterAsupData -AsupSet $AsupObject

                            $asupData

                            $statsParams = @{
                                FunctionPSCallStack = Get-PSCallStack
                                TargetDetails       = [pscustomobject]@{
                                    SystemUid     = $asupData.Cluster.Info.ClusterIdentifier
                                    SystemType    = $asupData.pstypenames[0]
                                    SystemVersion = ([regex]::Split(([regex]::Split($asupData.System.VersionInfo.Version, 'Release\s(?:\w+[__])?')[1]), '\s+'))[0].Replace(':', '')
                                }
                                ElapsedTicks        = ([datetime]::Now - $functionStartTime).Ticks
                                ExcludeParameter    = 'AsupObject'
                            }

                            Write-Statistics @statsParams
                        }
                        catch {
                            Write-Error -ErrorRecord $_
                        }
                    }
                }
            }
        }
    }
}

function Get-ClusterAsupData {
    [CmdletBinding()]
    param (
        [Parameter(Mandatory)]
        [ValidateNotNullOrEmpty()]
        [System.Xml.XmlElement[]]$AsupSet
    )

    $systemAsupData = [pscustomobject]@{ }
    $systemAsupData.pstypenames.Insert(0, 'NetAppDocs.NcController.Asup.Data')

    $NetAppDocsError = @( )

    Write-Verbose -Message '- Capturing module version -'
    $systemAsupData | Add-Member -MemberType NoteProperty -Name NetAppDocsVersion (Get-NtapDocsVersion -AsVersion)

    Write-Verbose -Message '- Capturing PowerShell version -'
    $systemAsupData | Add-Member -MemberType NoteProperty -Name PowerShellVersion -Value $PSVersionTable.PSVersion

    Write-Verbose -Message '- Capturing ASUP MetaData -'
    $systemAsupData | Add-Member -MemberType NoteProperty -Name SourceAsup -Value (ConvertTo-AsupObject -AsupSet $AsupSet)

    try {
        #region System/Node Information

        Write-Verbose -Message '- Gathering basic system information -'

        try {
            Write-Verbose -Message 'Capturing SYSCONFIG-A'
            $sysconfigA = $systemAsupData.SourceAsup | ForEach-Object {
                [pscustomobject]@{
                    NodeName = $_.SystemName
                    Text     = Get-AsupContent -AsupObject $_ -SectionName 'SYSCONFIG-A' -ErrorVariable +NetAppDocsError
                }
            }

            Write-Verbose -Message 'Capturing SYSCONFIG-AC'
            $sysconfigAc = $systemAsupData.SourceAsup | ForEach-Object {
                [pscustomobject]@{
                    NodeName = $_.SystemName
                    Text     = Get-AsupContent -AsupObject $_ -SectionName 'SYSCONFIG-AC' -ErrorVariable +NetAppDocsError
                }
            }

            Write-Verbose -Message 'Parsing SYSTEM'
            $system = $systemAsupData.SourceAsup | Get-AsupObject -ObjectName 'SYSTEM' -ReplaceString '^Sys(?!temId)' -ErrorAction Stop | Select-Object -Property *, @{
                Name       = 'SystemName'
                Expression = { $_.Hostname }
            }

            Write-Verbose -Message 'Parsing CLUSTER'
            $clusterObject = $systemAsupData.SourceAsup | Get-AsupObject -ObjectName 'CLUSTER' -ReplaceString '^Cluster(?!Name|Identifier)' -ErrorAction Stop | Where-Object { -not [string]::IsNullOrWhiteSpace($_.ClusterName) -and -not [string]::IsNullOrWhiteSpace($_.ClusterIdentifier) } | Select-Object -First 1

            if (-not $clusterObject) {
                throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("ASUP 'CLUSTER' object incomplete", 'ReadError', [System.Management.Automation.ErrorCategory]::ReadError, $AsupSet[0].cluster_name))
            }

            Write-Verbose -Message 'Parsing CLUSTER-INFO.XML'
            $clusterInfoXml = Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CLUSTER-INFO.XML' -InnerProperty 'CLUSTER_INFO' -ErrorVariable +NetAppDocsError | Where-Object { $_.IsSelf }

            if (-not $clusterInfoXml) {
                throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("ASUP 'CLUSTER-INFO.XML' section incomplete", 'ReadError', [System.Management.Automation.ErrorCategory]::ReadError, $AsupSet[0].cluster_name))
            }

            Write-Verbose -Message 'Parsing CLUSTER-RING.XML'
            $clusterRingXml = Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CLUSTER-RING.XML' -InnerProperty 'T_RDB' -ReplaceString '^Rdb' -ErrorVariable +NetAppDocsError | Group-Object -Property Node, UnitName | ForEach-Object {
                $_.Group | Sort-Object -Property DbTrnxs | Select-Object -Last 1
            }

            Write-Verbose -Message '- Identifying Cluster Ring Master Node -'
            $masterNodeName = $(
                $systemAsupData.SourceAsup | ForEach-Object {
                    $srcAsup = $_

                    ($clusterRingXml | Where-Object { $_.Node -ceq $srcAsup.Systemname -and $_.Unitname -eq 'mgmt' -and ($_.Local -ceq $_.Node -and $_.Node -ceq $_.Master) }).Master
                } | Select-Object -First 1
            )

            Write-Verbose -Message 'Parsing MOTHERBOARD'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name Motherboard -Value $(
                foreach ($srcAsup in $systemAsupData.SourceAsup) {
                    Get-AsupObject -AsupObject $srcAsup -ObjectName 'MOTHERBOARD' -ReplaceString '^Mb' -ErrorAction Stop | Select-Object -Property *, @{
                        Name       = 'SystemName'
                        Expression = { $srcAsup.SystemName }
                    }
                }
            )
        }
        catch {
            if ($_.CategoryInfo.Category -eq 'ReadError') {
                $Global:Error.RemoveAt(0)

                $msg = New-Object -TypeName System.Text.StringBuilder

                [void]$msg.AppendLine('An unexpected error occurred. Processing halted for this system.')
                [void]$msg.AppendLine('---------------------------------------------------------------------------------------------')
                [void]$msg.AppendLine('This error is not expected, but we do know about it. Unfortunately, this is an issue with the')
                [void]$msg.AppendLine('ASUP data warehouse and (usually) the ASUP not being fully ingested. It is possible that')
                [void]$msg.AppendLine('waiting up to a few hours or so may be enough time for the ASUP DW to finishing ingesting the')
                [void]$msg.AppendLine('data. However, in some cases, the only workaround is to try a different set of ASUPs.')
                [void]$msg.AppendLine('---------------------------------------------------------------------------------------------')
                [void]$msg.AppendLine("Target: $( $_.CategoryInfo.TargetName ). Reason: $( $_.Exception.GetBaseException().Message )")
                [void]$msg.AppendLine("Location: $( $_.ScriptStackTrace )")
                [void]$msg.AppendLine()

                $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $_.CategoryInfo.TargetName)))
            }
            else {
                $NetAppDocsError += $_
            }
        }

        $systemAsupData | Add-Member -MemberType NoteProperty -Name System -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing EFFECTIVE-CLUSTER-VERSION.XML'
        $clusterVersionXml = Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'EFFECTIVE-CLUSTER-VERSION.XML' -MasterNodeName $masterNodeName -InnerProperty 'effective_cluster_version' -ReplaceString '^Cluster' -ErrorVariable +NetAppDocsError

        $systemAsupData.System | Add-Member -MemberType NoteProperty -Name VersionInfo -Value (
            [pscustomobject]@{
                Version       = $clusterVersionXml.VersionString
                VersionTupleV = [version]::Parse([string]::Format('{0}.{1}.{2}', $clusterVersionXml.Generation, $clusterVersionXml.Major, $clusterVersionXml.Minor))
            }
        )

        $systemAsupData.System | Add-Member -MemberType NoteProperty -Name OntapiVersion -Value ([version]::Parse([string]::Format('{0}.{1}', $clusterVersionXml.OntapiMajor, $clusterVersionXml.OntapiMinor)))

        Write-Verbose -Message 'Parsing SOFTWARE-IMAGE.XML'
        $systemAsupData.System | Add-Member -MemberType NoteProperty -Name Image -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SOFTWARE-IMAGE.XML' -InnerProperty 'T_SOFTWARE_IMAGE' -ReplaceString '^Systemimage' -ErrorVariable +NetAppDocsError)

        $bcomdBootTimes = @(
            if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.30')) {
                Write-Verbose -Message 'Parsing BCOMD-BOOTTIME.XML'
                foreach ($srcAsup in $systemAsupData.SourceAsup) {
                    Get-AsupSection -AsupObject $srcAsup -SectionName 'BCOMD-BOOTTIME.XML' -InnerProperty 'T_BCOMD_BOOTTIME' -ErrorVariable +NetAppDocsError | Sort-Object -Property Sequence | Select-Object -Last 1 -Property @{
                        Name       = 'Node'
                        Expression = { $srcAsup.SystemName }
                    }, @{
                        Name       = 'NodeUptime'
                        Expression = {
                            if (-not [string]::IsNullOrWhiteSpace($_.StartTime) -and $_.StartTime -ne '0') {
                                [long]($srcAsup.AsupGenerationDate.ToLocalTime() - (Convert-DateTimeString -String $_.StartTime)).TotalSeconds
                            }
                        }
                    }
                }
            }
            else {
                Write-Verbose -Message 'Parsing BCOMD-BRINGUP-INFO.XML'
                foreach ($srcAsup in $systemAsupData.SourceAsup) {
                    Get-AsupSection -AsupObject $srcAsup -SectionName 'BCOMD-BRINGUP-INFO.XML' -InnerProperty 'T_BCOMD_BRINGUP_INFO' -ErrorVariable +NetAppDocsError | Where-Object { -not $_.FinishTime } | Select-Object -Last 1 -Property @{
                        Name       = 'Node'
                        Expression = { $srcAsup.SystemName }
                    }, @{
                        Name       = 'NodeUptime'
                        Expression = {
                            if (-not [string]::IsNullOrWhiteSpace($_.StartTime) -and $_.StartTime -ne '0') {
                                [long]($srcAsup.AsupGenerationDate.ToLocalTime() - (Convert-DateTimeString -String $_.StartTime)).TotalSeconds
                            }
                        }
                    }
                }
            }
        )

        Write-Verbose -Message '- Gathering node information -'

        $systemAsupData | Add-Member -MemberType NoteProperty -Name Node -Value ($system | Select-Object -Property * -ExcludeProperty Options)

        $systemAsupData | Add-Member -MemberType NoteProperty -Name Option -Value @( )
        foreach ($node in ($system | Where-Object { $_ })) {
            $node.Options.psobject.Properties | ForEach-Object {
                $systemAsupData.Option += [pscustomobject]@{
                    Name    = $_.Name
                    Value   = $_.Value
                    Vserver = $node.Hostname
                }
            }
        }

        # Write-Verbose -Message 'Parsing MOTHERBOARD-INFO.XML'
        # $motherboardInfoXml = $systemAsupData.SourceAsup | ForEach-Object {
        #     [pscustomobject]@{
        #         NodeName = $_.SystemName
        #         Object   = Get-AsupSection -AsupObject $_ -SectionName 'MOTHERBOARD-INFO.XML' -InnerProperty 'T_MBINFO' -ErrorVariable +NetAppDocsError
        #     }
        # }

        # Write-Verbose -Message 'Parsing SYSTEM-INFO.XML'
        # $systemAsupData | Add-Member -MemberType NoteProperty -Name Node -Value @(
        #     $systemAsupData.SourceAsup | ForEach-Object {
        #         $systemName = $_.SystemName
        #         $motherboardInfoXmlObject = ($motherboardInfoXml | Where-Object { $systemName -eq $_.NodeName }).Object

        #         Get-AsupSection -AsupObject $_ -SectionName 'SYSTEM-INFO.XML' -InnerProperty 'T_SYSINFO' -ErrorVariable +NetAppDocsError | Select-Object -ExcludeProperty OntapVersion -Property *, @{
        #             Name       = 'Hostname'
        #             Expression = { $_.SystemHostname }
        #         }, @{
        #             Name       = 'SerialNo'
        #             Expression = { $_.SystemSerialNumber }
        #         }, @{
        #             Name       = 'PartnerSystemId'
        #             Expression = { $_.PartnerId }
        #         }, @{
        #             Name       = 'FwRev'
        #             Expression = { $motherboardInfoXmlObject.BiosVersion }
        #         }, @{
        #             Name       = 'Model'
        #             Expression = { $motherboardInfoXmlObject.ModelName }
        #         }, @{
        #             Name       = 'OntapVersion'
        #             Expression = { ([regex]::Split(([regex]::Split($_.OntapVersion, 'Release\s(?:\w+[__])?')[1]), '\s+'))[0].Replace(':', '') }
        #         }
        #     }
        # )

        # Write-Verbose -Message 'Parsing OPTIONS'
        # $systemAsupData | Add-Member -MemberType NoteProperty -Name Option -Value @(
        #     $systemAsupData.SourceAsup | ForEach-Object {
        #         $systemName = $_.SystemName
        #         Get-AsupContent -AsupObject $_ -SectionName 'OPTIONS' -ErrorVariable +NetAppDocsError | Get-NodeOptions | Select-Object -Property *, @{
        #             Name       = 'Vserver'
        #             Expression = { $systemName }
        #         }
        #     }
        # )

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.21')) {
            Write-Verbose -Message 'Identifying IOXM status and PlatformConfig'
            foreach ($node in ($systemAsupData.Node | Where-Object { $_ })) {
                if (-not $node.ClusterMemberId) {
                    $node | Add-Member -MemberType NoteProperty -Name ClusterMemberId -Value ($clusterInfoXml | Where-Object { $_.Node -ceq $node.Hostname }).NodeUuid
                }

                $node | Add-Member -MemberType NoteProperty -Name IoxmPresent -Value ([bool](Select-String -InputObject ($sysconfigA | Where-Object { $_.NodeName -ceq $node.Hostname }).Text -Pattern 'IOXM\s+Status:\s+present' -Quiet))
                $node | Add-Member -MemberType NoteProperty -Name PlatformConfig -Value (Get-PlatformConfig -SystemModel $node.Model -IsClustered (! [string]::IsNullOrWhiteSpace($node.PartnerSystemId) -and ($node.PartnerSystemId -as [int]) -ne '0') -IoxmPresent $node.IoxmPresent)
                $node | Add-Member -MemberType NoteProperty -Name IsEpsilonNode -Value ($clusterInfoXml | Where-Object { $_.NodeUuid -eq $node.ClusterMemberId }).Epsilon
                $node | Add-Member -MemberType NoteProperty -Name IsClusterEligible -Value ($clusterInfoXml | Where-Object { $_.NodeUuid -eq $node.ClusterMemberId }).Eligibility
                $node | Add-Member -MemberType NoteProperty -Name NodeUptime -Value ($bcomdBootTimes | Where-Object { $_.Node -ceq $node.Hostname }).NodeUptime
            }
        }
        else {
            Write-Verbose -Message 'Capturing RDB-DUMP'
            $rdbDump = $systemAsupData.SourceAsup | ForEach-Object {
                [pscustomobject]@{
                    NodeName = $_.SystemName
                    Text     = Get-AsupContent -AsupObject $_ -SectionName 'RDB-DUMP' -ErrorVariable +NetAppDocsError
                }
            }

            Write-Verbose -Message 'Identifying IOXM status and PlatformConfig'
            foreach ($node in ($systemAsupData.Node | Where-Object { $_ })) {
                if (-not $node.ClusterMemberId) {
                    $node | Add-Member -MemberType NoteProperty -Name ClusterMemberId -Value ($clusterInfoXml | Where-Object { $_.Node -ceq $node.Hostname }).NodeUuid -Force
                }

                $node | Add-Member -MemberType NoteProperty -Name IoxmPresent -Value ([bool](Select-String -InputObject ($sysconfigA | Where-Object { $_.NodeName -ceq $node.Hostname }).Text -Pattern 'IOXM\s+Status:\s+present' -Quiet))
                $node | Add-Member -MemberType NoteProperty -Name PlatformConfig -Value (Get-PlatformConfig -SystemModel $node.Model -IsClustered (! [string]::IsNullOrWhiteSpace($node.PartnerSystemId) -and ($node.PartnerSystemId -as [int]) -ne '0') -IoxmPresent $node.IoxmPresent)
                $node | Add-Member -MemberType NoteProperty -Name NodeUptime -Value ($bcomdBootTimes | Where-Object { $_.Node -ceq $node.Hostname }).NodeUptime

                if (($rdbDump | Where-Object { $_.NodeName -ceq $node.Hostname }).Text -match "(?m)^.*host\s+`"(?<Node>$( $node.Hostname ))`".*(?<ClusterEligible>(?:not\s+)?eligible)\s*(?:w[/](?<Epsilon>epsilon)+)?.*$") {
                    $node | Add-Member -MemberType NoteProperty -Name IsEpsilonNode -Value ($Matches.Epsilon -eq 'epsilon')
                    $node | Add-Member -MemberType NoteProperty -Name IsClusterEligible -Value ($Matches.ClusterEligible -eq 'eligible')

                    if ($Matches) {
                        $Matches.Clear()
                    }
                }
            }
        }

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.120')) {
            Write-Verbose -Message 'Parsing EXTERNAL-CACHE-OPTIONS.XML'
            $systemAsupData.System | Add-Member -MemberType NoteProperty -Name ExternalCache -Value @(
                foreach ($srcAsup in $systemAsupData.SourceAsup) {
                    Get-AsupSection -AsupObject $srcAsup -SectionName 'EXTERNAL-CACHE-OPTIONS.XML' -InnerProperty 'T_EXTERNAL_CACHE_OPTIONS' -ReplaceString '^Ec' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                        Name       = 'Node'
                        Expression = { $srcAsup.SystemName }
                    }
                }
            )
        }

        #endregion System/Node Information

        #region Kenv Information

        Write-Verbose -Message '- Gathering KENV information -'

        Write-Verbose -Message 'Parsing KENV'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Kenv -Value $(
            foreach ($asup in $systemAsupData.SourceAsup) {
                $kenv = Get-AsupContent -AsupObject $asup -SectionName 'KENV' -ErrorVariable +NetAppDocsError

                $kenv -split "`n" | ForEach-Object {
                    $variable, $value = $_ -split '\s*[=]\s*', 2

                    [pscustomobject]@{
                        NodeName = $asup.SystemName
                        Variable = $variable
                        Value    = $(
                            if (-not [string]::IsNullOrWhiteSpace($value)) {
                                $value.Trim('"')
                            }
                        )
                    }
                }
            }
        )

        #endregion Kenv Information

        #region Cluster Information

        Write-Verbose -Message '- Gathering Cluster Info -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Cluster -Value ([pscustomobject]@{ })

        $systemAsupData.Cluster | Add-Member -MemberType NoteProperty -Name Info -Value ($clusterObject | Select-Object -ExcludeProperty Version*, OntapVersion* -Property *)

        Write-Verbose -Message 'Capturing CLUSTER-RING.XML'
        $systemAsupData.Cluster | Add-Member -MemberType NoteProperty -Name RingReplication -Value $clusterRingXml

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message 'Parsing CONTACT-INFO-VIEW.XML'
            $systemAsupData.Cluster | Add-Member -MemberType NoteProperty -Name Contact -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CONTACT-INFO-VIEW.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CONTACT_INFO' -ReplaceString '^ContactInfo(?:Site)?' -ErrorVariable +NetAppDocsError)
        }

        #endregion Cluster Information

        #region Cluster CLAM Information

        Write-Verbose -Message '- Gathering Cluster Kernel-Service Info -'
        $systemAsupData.Cluster | Add-Member -MemberType NoteProperty -Name KernelService -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing CLAM-NODE-STATUS.XML'
        $systemAsupData.Cluster.KernelService | Add-Member -MemberType NoteProperty -Name Status -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CLAM-NODE-STATUS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CLAM_NODE_STATUS' -ErrorVariable +NetAppDocsError | Where-Object { $_.IsLocalMaster })

        #endregion Cluster CLAM Information

        #region Cluster HA Information

        Write-Verbose -Message '- Gathering cluster HA information -'
        $systemAsupData.Cluster | Add-Member -MemberType NoteProperty -Name Ha -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing CLUSTER-HA.XML'
        $systemAsupData.Cluster.Ha | Add-Member -MemberType NoteProperty -Name Info -Value @(
            foreach ($srcAsup in $systemAsupData.SourceAsup) {
                Get-AsupSection -AsupObject $srcAsup -SectionName 'CLUSTER-HA.XML' -InnerProperty 'T_CLUS_HA' -ReplaceString '^Clusha' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                    Name       = 'Node'
                    Expression = { $srcAsup.SystemName }
                }
            }
        )

        Write-Verbose -Message 'Parsing STORAGE-FAILOVER.XML'
        $sfOptions = Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'STORAGE-FAILOVER.XML' -InnerProperty 'T_SFO' -ReplaceString '^Sfo' -ErrorVariable +NetAppDocsError

        foreach ($nodeHaInfo in $systemAsupData.Cluster.Ha.Info) {
            $nodeHaInfo | Add-Member -MemberType NoteProperty -Name SfOptions -Value ($sfOptions | Where-Object { $_.NodeName -ceq $nodeHaInfo.Node })
        }

        #endregion Cluster HA Information

        #region Cluster LogForward Information

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.31')) {
            Write-Verbose -Message '- Gathering Cluster LogForward information -'
            $systemAsupData.Cluster | Add-Member -MemberType NoteProperty -Name LogForward -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing LOG-FORWARDING.XML'
            $systemAsupData.Cluster.LogForward | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'LOG-FORWARDING.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_LOG_FORWARD' -ErrorVariable +NetAppDocsError)
        }

        #endregion Cluster LogForward Information

        #region HA Interconnect Information

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.100')) {
            Write-Verbose -Message '- Gathering HA interconnect information -'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name HaInterconnect -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing HA-INTERCONNECT-CONFIG.XML'
            $systemAsupData.HaInterconnect | Add-Member -MemberType NoteProperty -Name ConfigDetails -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'HA-INTERCONNECT-CONFIG.XML' -InnerProperty 'T_ICCONFIG' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing HA-INTERCONNECT-STATUS.XML'
            $systemAsupData.HaInterconnect | Add-Member -MemberType NoteProperty -Name ConnectionStatus -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'HA-INTERCONNECT-STATUS.XML' -InnerProperty 'T_ICSTATUS' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Capturing HA-INTERCONNECT-INFO.XML'
            $haInterconnectInfo = @(
                foreach ($srcAsup in $systemAsupData.SourceAsup) {
                    Get-AsupSection -AsupObject $srcAsup -SectionName 'HA-INTERCONNECT-INFO.XML' -InnerProperty 'T_ICINFO' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                        Name       = 'Node'
                        Expression = { $srcAsup.SystemName }
                    }
                }
            )

            if ($systemAsupData.HaInterconnect.ConnectionStatus -and $haInterconnectInfo) {
                foreach ($haInterconnectStatusObject in $systemAsupData.HaInterconnect.ConnectionStatus) {
                    $port1 = $haInterconnectInfo | Where-Object { $_.Node -ceq $haInterconnectStatusObject.Node -and $_.PortName -eq 'ib2a' }
                    $port2 = $haInterconnectInfo | Where-Object { $_.Node -ceq $haInterconnectStatusObject.Node -and $_.PortName -eq 'ib2b' }

                    $haInterconnectStatusObject | Add-Member -NotePropertyMembers @{
                        Slot                  = $port1.SlotNumber
                        DriverName            = $port1.DriverName
                        Firmware              = $port1.FirmwareRevision
                        DebugFirmware         = $port1.DebugFirmware -eq 'yes'
                        Port1PortName         = $port1.PortName
                        Port1Gid              = $port1.GlobalIdentifier
                        Port1BaseLid          = $port1.BaseLocalIdentifier
                        Port1Mtu              = $port1.Mtu
                        Port1DataRate         = $port1.DataRate
                        Port1LinkInfo         = $port1.LinkState
                        Port1QsfpVendor       = $port1.QsfpVendor
                        Port1QsfpPartNumber   = $port1.QsfpPartNumber
                        Port1QsfpType         = $port1.QsfpType
                        Port1QsfpSerialNumber = $port1.QsfpSerialNumber
                        Port2PortName         = $port2.PortName
                        Port2Gid              = $port2.GlobalIdentifier
                        Port2BaseLid          = $port2.BaseLocalIdentifier
                        Port2Mtu              = $port2.Mtu
                        Port2DataRate         = $port2.DataRate
                        Port2LinkInfo         = $port2.LinkState
                        Port2QsfpVendor       = $port2.QsfpVendor
                        Port2QsfpPartNumber   = $port2.QsfpPartNumber
                        Port2QsfpType         = $port2.QsfpType
                        Port2QsfpSerialNumber = $port2.QsfpSerialNumber
                    }
                }
            }
        }

        #endregion HA Interconnect Information

        #region License Information

        ##Unused Section: LICENSE-ENTITLEMENT-RISK.XML - (1.31+) Not needed for build docs (License.EntitlementRisk)
        ##Unused Section: LICENSES-CAPACITY.XML - (1.100+) Not needed for build docs (License.Capacity)
        ##Unused Section: LICENSES-SUBSCRIPTION.XML - (1.100+) Not needed for build docs (License.Subscription)

        #These should be the same for all nodes in the cluster
        Write-Verbose -Message '- Gathering licensing information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name License -Value ([pscustomobject]@{ })

        $systemAsupData.License | Add-Member -MemberType NoteProperty -Name Info -Value $(
            if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
                Write-Verbose -Message 'Parsing LICENSES.XML'
                (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'LICENSES.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_LIC_V2' -ErrorVariable +NetAppDocsError)
            }
            else {
                Write-Verbose -Message 'Parsing CLUSTER-LICENSES.XML'
                (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CLUSTER-LICENSES.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_LIC' -ReplaceString '^Lic' -ErrorVariable +NetAppDocsError)
            }
        )

        #endregion License Information

        #region Adapter Card Inventory

        Write-Verbose -Message '- Identifying adapter cards -'

        Write-Verbose -Message 'Calling Get-NtapExpansionSlotInventory'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name AdapterCard -Value $(
            foreach ($nodeEntry in ($sysconfigAc | Where-Object { $_ })) {
                Get-NtapExpansionSlotInventory -SysconfigAc $nodeEntry.Text -SysconfigVorA ($sysconfigA | Where-Object { $_.NodeName -eq $nodeEntry.NodeName }).Text | Add-Member -MemberType NoteProperty -Name NodeName -Value $nodeEntry.NodeName -Force -PassThru
            }
        )

        #endregion Adapter Card Inventory

        #region NDMP

        Write-Verbose -Message '- Gathering NDMP configuration -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Ndmp -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing NDMP-NODE-PROPERTIES.XML'
        $systemAsupData.Ndmp | Add-Member -MemberType NoteProperty -Name SystemServices -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NDMP-NODE-PROPERTIES.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_NDMP_NODE' -ErrorVariable +NetAppDocsError)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Parsing NDMP-NODE-SCOPE-MODE-STATUS'
            $ndmpNodeScopeMode = Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'NDMP-NODE-SCOPE-MODE-STATUS' -ErrorVariable +NetAppDocsError | Select-Object -First 1

            if ($ndmpNodeScopeMode -match 'enabled|disabled') {
                $systemAsupData.Ndmp | Add-Member -MemberType NoteProperty -Name NodeScopeMode -Value $Matches[0]

                if ($Matches) {
                    $Matches.Clear()
                }
            }

            if ($ndmpNodeScopeMode -like '*disabled*') {
                Write-Verbose -Message 'Parsing NDMP-VSERVER-PROPERTIES.XML'
                $systemAsupData.Ndmp | Add-Member -MemberType NoteProperty -Name VserverServices -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NDMP-VSERVER-PROPERTIES.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_NDMP_VSERV' -ErrorVariable +NetAppDocsError)
            }
        }

        #endregion NDMP

        #region Remote Management Configuration

        Write-Verbose -Message '- Gathering remote management configuration -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name ServiceProcessor -Value ([pscustomobject]@{ })

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.21')) {
            Write-Verbose -Message 'Parsing SP-INFO.XML'
            $systemAsupData.ServiceProcessor | Add-Member -MemberType NoteProperty -Name Info -Value @(
                foreach ($srcAsup in $systemAsupData.SourceAsup) {
                    Get-AsupSection -AsupObject $srcAsup -SectionName 'SP-INFO.XML' -InnerProperty 'T_SPINFO' -ReplaceString '^ServiceProcessor' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                        Name       = 'Node'
                        Expression = { $srcAsup.SystemName }
                    }
                }
            )

            Write-Verbose -Message 'Parsing SP-NETWORK-INFO.XML'
            $systemAsupData.ServiceProcessor | Add-Member -MemberType NoteProperty -Name Network -Value @(
                foreach ($srcAsup in $systemAsupData.SourceAsup) {
                    Get-AsupSection -AsupObject $srcAsup -SectionName 'SP-NETWORK-INFO.XML' -InnerProperty 'T_SPNETWORK' -ReplaceString '^ServiceProcessor' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                        Name       = 'Node'
                        Expression = { $srcAsup.SystemName }
                    }
                }
            )
        }
        else {
            Write-Verbose -Message 'Calling Get-NtapRemoteManagement'
            $systemAsupData.ServiceProcessor | Add-Member -MemberType NoteProperty -Name Info -Value @(
                foreach ($node in $sysconfigA) {
                    Get-NtapRemoteManagement -Sysconfig $node.Text | Select-Object -ExcludeProperty NodeName -Property *, @{
                        Name       = 'NodeName'
                        Expression = { $node.NodeName }
                    }
                }
            )
        }

        #endregion Remote Management Configuration

        #region System Services Data

        ##Unused Section: IPFILTERS-UI.XML - Not needed for build docs (SystemServices.Firewall)

        Write-Verbose -Message '- Gathering system services data -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name SystemServices -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing WEB-SERVER.XML'
        $systemAsupData.SystemServices | Add-Member -MemberType NoteProperty -Name WebNode -Value @(
            foreach ($srcAsup in $systemAsupData.SourceAsup) {
                Get-AsupSection -AsupObject $srcAsup -SectionName 'WEB-SERVER.XML' -InnerProperty 'T_WEB_NODE' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                    Name       = 'Node'
                    Expression = { $srcAsup.SystemName }
                }
            }
        )

        #endregion System Services Data

        #region Flash Device Details

        Write-Verbose -Message '- Gathering flash device data -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Flash -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing FMM_DATA'
        $fmmData = @(
            foreach ($srcAsup in $systemAsupData.SourceAsup) {
                Get-AsupObject -AsupObject $srcAsup -ObjectName 'FMM_DATA' -ReplaceString '^Fmm' -ErrorVariable +NetAppDocsError | Select-Object -ExcludeProperty FirmwareVersion -Property *, @{
                    Name       = 'NodeName'
                    Expression = { $srcAsup.SystemName }
                }, @{
                    Name       = 'AdvertisedCapacityInBytes'
                    Expression = { [long]$_.CapacityKb * 1024 }
                }, @{
                    Name       = 'FirmwareRevision'
                    Expression = { $_.FirmwareVersion }
                }
            }
        )

        $flashCacheInfoXml = @(
            if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.180')) {
                Write-Verbose -Message 'Parsing FLASH-CACHE-INFO2.XML'
                foreach ($srcAsup in $systemAsupData.SourceAsup) {
                    Get-AsupSection -AsupObject $srcAsup -SectionName 'FLASH-CACHE-INFO2.XML' -InnerProperty 'T_FCINFO2' -ErrorVariable +NetAppDocsError | Where-Object { $_.SerialNumber } | Select-Object -ExcludeProperty Slot, FirmwareVersion -Property *, @{
                        Name       = 'NodeName'
                        Expression = { $srcAsup.SystemName }
                    }, @{
                        Name       = 'ModelName'
                        Expression = { $_.ModelNumber }
                    }, @{
                        Name       = 'SlotNumber'
                        Expression = { $_.Slot }
                    }, @{
                        Name       = 'FirmwareRevision'
                        Expression = { $_.FirmwareVersion }
                    }, @{
                        Name       = 'Status'
                        Expression = { $_.DeviceState }
                    }, @{
                        Name       = 'AdvertisedCapacityInBytes'
                        Expression = { [long]$_.Capacity * 1GB }
                    }
                }
            }
            else {
                Write-Verbose -Message 'Parsing FLASH-CACHE-INFO.XML'
                foreach ($srcAsup in $systemAsupData.SourceAsup) {
                    Get-AsupSection -AsupObject $srcAsup -SectionName 'FLASH-CACHE-INFO.XML' -InnerProperty 'T_FCINFO' -ErrorVariable +NetAppDocsError | Where-Object { $_.SerialNumber } | Select-Object -ExcludeProperty Slot -Property *, @{
                        Name       = 'NodeName'
                        Expression = { $srcAsup.SystemName }
                    }, @{
                        Name       = 'SlotNumber'
                        Expression = { $_.Slot }
                    }
                }
            }
        )

        if ($flashCacheInfoXml) {
            $systemAsupData.Flash | Add-Member -MemberType NoteProperty -Name Device -Value $flashCacheInfoXml

            if ($systemAsupData.System.OntapiVersion -lt [version]::Parse('1.180')) {
                foreach ($flashDevice in ($systemAsupData.Flash.Device | Where-Object { $_ })) {
                    if (($fmmCardData = $fmmData | Where-Object { $_.NodeName -ceq $flashDevice.NodeName -and $_.SerialNumber -eq $flashDevice.SerialNumber })) {
                        $flashDevice | Add-Member -MemberType NoteProperty -Name FirmwareRevision -Value $fmmCardData.FirmwareRevision -ErrorAction Ignore -Force
                        $flashDevice | Add-Member -MemberType NoteProperty -Name Status -Value $fmmCardData.Status -ErrorAction Ignore -Force
                        $flashDevice | Add-Member -MemberType NoteProperty -Name AdvertisedCapacityInBytes -Value ([long]$fmmCardData.CapacityKb * 1024) -ErrorAction Ignore -Force
                    }
                }
            }
        }
        else {
            $systemAsupData.Flash | Add-Member -MemberType NoteProperty -Name Device -Value $fmmData
        }

        #endregion Flash Device Details

        #region FlexCache Data

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Gathering FlexCache data -'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name Flexcache -Value ([pscustomobject]@{ })

            if ($systemAsupData.System.OntapiVersion -lt [version]::Parse('1.30')) {
                Write-Verbose -Message 'Parsing FLEXCACHE-VOLUME.XML'
                $systemAsupData.Flexcache | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FLEXCACHE-VOLUME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FLEXCACHE_REPORT' -ReplaceString '^Fc' -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message 'Parsing FLEXCACHE-CACHE-POLICY.XML'
                $systemAsupData.Flexcache | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FLEXCACHE-CACHE-POLICY.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FLEXCACHE_CACHE_POLICY_MGMT' -ReplaceString '^Fc' -ErrorVariable +NetAppDocsError)
            }

            if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.150')) {
                Write-Verbose -Message 'Parsing FLEXCACHE-RELATIONS.XML'
                $systemAsupData.Flexcache | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FLEXCACHE-RELATIONS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FLEXCACHE' -ErrorVariable +NetAppDocsError)
            }
        }

        #endregion FlexCache Data

        #region Feature Usage Data

        ##Unused Section: MANAGED-FEATURE.XML - (1.20+) Data available in usage summary (Feature.Status)

        Write-Verbose -Message '- Gathering feature usage data -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Feature -Value ([pscustomobject]@{ })

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Parsing MANAGED-FEATURE-USAGE-SUMMARY.XML'
            $systemAsupData.Feature | Add-Member -MemberType NoteProperty -Name UsageSummary -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'MANAGED-FEATURE-USAGE-SUMMARY.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_MANAGED_FEATURE_USAGE_SUMMARY' -ErrorVariable +NetAppDocsError | Where-Object { $_.Owner -ne 'none' })
        }

        #endregion Feature Usage Data

        #region SNMP Configuration

        Write-Verbose -Message '- Gathering SNMP information -'

        Write-Verbose -Message 'Parsing SNMP-CONFIG.XML'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Snmp -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SNMP-CONFIG.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_snmp' -ErrorVariable +NetAppDocsError)

        #endregion SNMP Configuration

        #region ConfigBackup Information

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Gathering config backup information -'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name ConfigBackup -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing CONFIG-BACKUP-INFO.XML'
            $systemAsupData.ConfigBackup | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CONFIG-BACKUP-INFO.XML' -InnerProperty 'T_CONFIG_BACKUP_INFO' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing CONFIG-BACKUP-SETTINGS.XML'
            $configBackupSettings = Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CONFIG-BACKUP-SETTINGS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CONFIGURATION_BACKUP_SETTINGS' -ErrorVariable +NetAppDocsError

            $systemAsupData.ConfigBackup | Add-Member -MemberType NoteProperty -Name Count -Value ($configBackupSettings | Select-Object -Property Enabled*, Schedule*, Numbackups*)
            $systemAsupData.ConfigBackup | Add-Member -MemberType NoteProperty -Name Url -Value ($configBackupSettings | Select-Object -ExcludeProperty Enabled*, Schedule*, Numbackups*)
        }

        #endregion ConfigBackup Information

        #region Dashboard/Diagnosis Information

        #SYSTEM-HEALTH-ALERT-PERSISTENT.XML ??

        Write-Verbose -Message '- Gathering dashboard/diagnostics information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Diagnosis -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing SYSTEM-HEALTH-ALERT.XML'
        $systemAsupData.Diagnosis | Add-Member -MemberType NoteProperty -Name Alert -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SYSTEM-HEALTH-ALERT.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_SYSTEM_ALERT' -ErrorVariable +NetAppDocsError)

        #endregion Dashboard/Diagnosis Information

        #region Storage Configuration

        ##Unused Section: STORAGE-ACP.XML - (1.31+) Not needed for build docs (Storage.ShelfAcp.Info)
        ##Unused Section: STORAGE-PORT.XML - (1.100+) Not currently needed for build docs (Storage.Port)

        Write-Verbose -Message '- Gathering system storage configuration and system ACP connectivity -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Storage -Value ([pscustomobject]@{ })

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.31')) {
            Write-Verbose -Message '- Gathering ACP information -'
            $systemAsupData.Storage | Add-Member -MemberType NoteProperty -Name ShelfAcp -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing STORAGE-ACP-MODULE.XML'
            $systemAsupData.Storage.ShelfAcp | Add-Member -MemberType NoteProperty -Name Module -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'STORAGE-ACP-MODULE.XML' -InnerProperty 'T_STORAGE_SHELF_ACP_MOD' -ErrorVariable +NetAppDocsError)
        }

        $systemAsupData.Storage | Add-Member -MemberType NoteProperty -Name Configuration -Value @( )
        $systemAsupData.Storage | Add-Member -MemberType NoteProperty -Name SystemAcpConnectivity -Value @( )

        foreach ($node in ($systemAsupData.Node | Where-Object { $_ })) {
            $storageConfiguration = ($sysconfigA | Where-Object { $_.NodeName -ceq $node.Hostname }).Text -split "`n" | Select-String -Pattern 'System Storage Configuration:' -SimpleMatch
            $systemAcpConnectivity = ($sysconfigA | Where-Object { $_.NodeName -ceq $node.Hostname }).Text -split "`n" | Select-String -Pattern 'System ACP Connectivity:' -SimpleMatch

            if ($storageConfiguration) {
                $systemAsupData.Storage.Configuration += [pscustomobject]@{
                    NodeName      = $node.Hostname
                    Configuration = ($storageConfiguration -split ':')[1].Trim()
                }
            }
            else {
                $systemAsupData.Storage.Configuration += [pscustomobject]@{
                    NodeName      = $node.Hostname
                    Configuration = $node.StorageConfig
                }
            }

            if ($systemAcpConnectivity) {
                $systemAsupData.Storage.SystemAcpConnectivity += [pscustomobject]@{
                    NodeName      = $node.Hostname
                    Configuration = ($systemAcpConnectivity -split ':')[1].Trim()
                }
            }
            else {
                $systemAsupData.Storage | Add-Member -MemberType NoteProperty -Name SystemAcpConnectivity -Value 'NA' -Force
            }
        }

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message 'Parsing STORAGE-BRIDGE-VIEW.XML'
            $systemAsupData.Storage | Add-Member -MemberType NoteProperty -Name Bridge -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'STORAGE-BRIDGE-VIEW.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_STORAGE_BRIDGE_VIEW' -ReplaceString '^Bridge' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing STORAGE-SWITCH-VIEW.XML'
            $systemAsupData.Storage | Add-Member -MemberType NoteProperty -Name Switch -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'STORAGE-SWITCH-VIEW.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_STORAGE_SWITCH_VIEW' -ReplaceString '^Switch' -ErrorVariable +NetAppDocsError)

            if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.110')) {
                $systemAsupData.Storage | Add-Member -MemberType NoteProperty -Name ShelfDrawer -Value ([pscustomobject]@{ })

                Write-Verbose -Message 'Parsing STORAGE-SHELF-DRAWER.XML'
                $systemAsupData.Storage.ShelfDrawer | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'STORAGE-SHELF-DRAWER.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_STORAGE_SHELF_DRAWER' -ErrorVariable +NetAppDocsError)
            }
        }

        #endregion Storage Configuration

        #region StoragePool Configuration

        ##Unused Section: STORAGE-POOL-SPARE-CAPACITY.XML - (1.30+) Not currently needed for build docs (StoragePool.AvailableCapacity)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message '- Gathering StoragePool information -'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name StoragePool -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing STORAGE-POOL.XML'
            $systemAsupData.StoragePool | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'STORAGE-POOL.XML' -InnerProperty 'T_STORAGE_POOL' -ReplaceString '^StoragePool(?=.+)' -ErrorVariable +NetAppDocsError | Sort-Object -Property * -Unique)

            Write-Verbose -Message 'Parsing STORAGE-POOL-AGGR.XML'
            $systemAsupData.StoragePool | Add-Member -MemberType NoteProperty -Name Aggr -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'STORAGE-POOL-AGGR.XML' -InnerProperty 'T_SP_AGGR' -ReplaceString '^StoragePool(?=.+)' -ErrorVariable +NetAppDocsError | Sort-Object -Property * -Unique)

            Write-Verbose -Message 'Parsing STORAGE-POOL-DISK.XML'
            $systemAsupData.StoragePool | Add-Member -MemberType NoteProperty -Name Disk -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'STORAGE-POOL-DISK.XML' -InnerProperty 'T_SP_DISKS' -ReplaceString '^StoragePool(?=.+)' -ErrorVariable +NetAppDocsError | Sort-Object -Property * -Unique)
        }

        #endregion StoragePool Configuration

        #region StorageArray Configuration

        ##Unused Section: ARRAY-PROFILE-PORT-DB.XML - Not needed for build docs (StorageArray.Port)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message '- Gathering StorageArray information -'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name StorageArray -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing ARRAY-PROFILE-DB.XML'
            $systemAsupData.StorageArray | Add-Member -MemberType NoteProperty -Name Profile -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'ARRAY-PROFILE-DB.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_ARRAY_PROFILE_DB' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing ARRAY-CONFIG.XML'
            $systemAsupData.StorageArray | Add-Member -MemberType NoteProperty -Name Config -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'ARRAY-CONFIG.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_ARRAY_CONFIG' -ErrorVariable +NetAppDocsError)
        }

        #endregion StorageArray Configuration

        #region Shelf Information

        ##Unused Section: ACP_INFO - Not currently needed (may be used for cabling) (Shelf.AcpInfo)
        ##Unused Section: SAS-PORT-INFO.XML - Not currently needed (may be used for cabling) (Shelf.SasPortInfo)

        Write-Verbose -Message '- Gathering shelf information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Shelf -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing SHELF'
        $systemAsupData.Shelf | Add-Member -MemberType NoteProperty -Name Info -Value @(
            foreach ($srcAsup in $systemAsupData.SourceAsup) {
                Get-AsupObject -AsupObject $srcAsup -ObjectName 'SHELF' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                    Name       = 'NodeName'
                    Expression = { $srcAsup.SystemName }
                }, @{
                    Name       = 'ShelfUid'
                    Expression = { [regex]::Replace(($_.ShelfLogicalId -replace '0x'), '.{2}', '$&:').Trim(':') }
                }
            }
        )

        Write-Verbose -Message 'Parsing STORAGE-SHELF.XML'
        $storageShelfInfo = Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'STORAGE-SHELF.XML' -InnerProperty 'T_STORAGE_SHELF' -ErrorVariable +NetAppDocsError

        if ($storageShelfInfo) {
            foreach ($shelf in $systemAsupData.Shelf.Info) {
                $shelfStorageInfo = $storageShelfInfo | Where-Object { $_.SerialNumber -eq $shelf.ShelfSerialNo } | Select-Object -First 1

                $shelf | Add-Member -MemberType NoteProperty -Name ShelfState -Value ([regex]::Replace($shelfStorageInfo.State, '\w', { param ($m) if ($m) { $m.ToString().ToLower() } }))
                $shelf | Add-Member -MemberType NoteProperty -Name ShelfStatus -Value $shelfStorageInfo.OpStatus
                $shelf | Add-Member -MemberType NoteProperty -Name ShelfName -Value $shelfStorageInfo.ShelfName
            }
        }

        Write-Verbose -Message 'Parsing SHELF_MODULE'
        $systemAsupData.Shelf | Add-Member -MemberType NoteProperty -Name Module -Value @(
            foreach ($srcAsup in $systemAsupData.SourceAsup) {
                Get-AsupObject -AsupObject $srcAsup -ObjectName 'SHELF_MODULE' -ReplaceString '^Smod' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                    Name       = 'NodeName'
                    Expression = { $srcAsup.SystemName }
                }
            }
        )

        Write-Verbose -Message 'Parsing SHELF_ENVIRONMENT_INFO'
        $systemAsupData.Shelf | Add-Member -MemberType NoteProperty -Name Environment -Value @(
            foreach ($srcAsup in $systemAsupData.SourceAsup) {
                Get-AsupObject -AsupObject $srcAsup -ObjectName 'SHELF_ENVIRONMENT_INFO' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                    Name       = 'NodeName'
                    Expression = { $srcAsup.SystemName }
                }
            }
        )

        #endregion Shelf Information

        #region Tape/MC Details

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message '- Gathering tape/MC device information -'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name TapeMc -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing STORAGE-TAPE-MC.XML'
            $systemAsupData.TapeMc | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'STORAGE-TAPE-MC.XML' -InnerProperty 'T_STORAGE_TAPE_MC' -ErrorVariable +NetAppDocsError)
        }

        #endregion Tape/MC Details

        #region Audit Configuration

        ##Unused Section: AUDIT.XML' - Not needed for build docs (Audit.Info)

        #endregion Audit Configuration

        #region Autosupport Configuration

        Write-Verbose -Message '- Gathering autosupport configuration -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Autosupport -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing AUTOSUPPORT.XML'
        $systemAsupData.Autosupport | Add-Member -MemberType NoteProperty -Name Config -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'AUTOSUPPORT.XML' -InnerProperty 'T_AUTOSUPPORT' -ReplaceString '^Autosupport' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing AUTOSUPPORT-HISTORY.XML'
        $systemAsupData.Autosupport | Add-Member -MemberType NoteProperty -Name History -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'AUTOSUPPORT-HISTORY.XML' -InnerProperty 'T_AUTOSUPPORT_HISTORY' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing AUTOSUPPORT-BUDGET.XML'
        $systemAsupData.Autosupport | Add-Member -MemberType NoteProperty -Name Budget -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'AUTOSUPPORT-BUDGET.XML' -InnerProperty 'T_AUTOSUPPORT_BUDGET' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing AUTOSUPPORT-TRIGGERS.XML'
        $systemAsupData.Autosupport | Add-Member -MemberType NoteProperty -Name Trigger -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'AUTOSUPPORT-TRIGGERS.XML' -InnerProperty 'T_AUTOSUPPORT_TRIGGERS' -ErrorVariable +NetAppDocsError)

        #endregion Autosupport Configuration

        #region Vserver Configuration

        ##Unused Section: HEALTH-VSERVER.XML - Not currently needed for build docs (Vserver.Health)
        ##Unused Section: VSERVER-WEB-SERVICES.XML - Not currently needed for build docs (Vserver.WebService)

        Write-Verbose -Message '- Gathing Vserver information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Vserver -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing VSERVER-INFO.XML'
        $systemAsupData.Vserver | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VSERVER-INFO.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VSERV' -ReplaceString '^Vserv(?!er)' -ErrorVariable +NetAppDocsError)

        #endregion Vserver Configuration

        #region Export Policy Configuration

        ##Unused Section: EXPORTS-ACCESS-CACHE-CONFIG.XML - (1.100+) Not needed for build docs (Export.AccessCacheConfig)

        Write-Verbose -Message '- Gathering export policy information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Export -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing EXPORT-RULESET-UI-TABLE.XML'
        $systemAsupData.Export | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'EXPORT-RULESET-UI-TABLE.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_EXPORT_POL' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing EXPORT-RULE-TABLE.XML'
        $systemAsupData.Export | Add-Member -MemberType NoteProperty -Name Rule -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'EXPORT-RULE-TABLE.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_EXPORT_POL_RULE' -ReplaceString '^Exportpolrule' -ErrorVariable +NetAppDocsError)

        #endregion Export Policy Configuration

        #region QOS Configuration

        Write-Verbose -Message '- Gathering QOS configuration -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Qos -Value ([pscustomobject]@{ })

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Parsing QOS-POLICY-GROUP.XML'
            $systemAsupData.Qos | Add-Member -MemberType NoteProperty -Name PolicyGroup -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'QOS-POLICY-GROUP.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_QOS_POLICY_GROUP' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing QOS-WORKLOAD.XML'
            $systemAsupData.Qos | Add-Member -MemberType NoteProperty -Name Workload -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'QOS-WORKLOAD.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_QOS_WORKLOAD' -ErrorVariable +NetAppDocsError)

            if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.130')) {
                Write-Verbose -Message 'Parsing QOS-ADAPTIVE-POLICY-GROUP.XML'
                $systemAsupData.Qos | Add-Member -MemberType NoteProperty -Name AdaptivePolicyGroup -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'QOS-ADAPTIVE-POLICY-GROUP.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_QOS_ADAPTIVE_POLICY_GROUP' -ErrorVariable +NetAppDocsError)
            }
        }

        #endregion QOS Configuration

        #region Metrocluster Information

        ##Unused Section: METROCLUSTER-INTERCONNECT-MIRROR-MP-POLICY.XML - (1.30+) Not needed for build docs (Metrocluster.InterconnectMirrorMultipath)
        ##Unused Section: METROCLUSTER-CONFIGURATION-STATE-DB.XML - (1.30+) Not needed for build docs (Metrocluster.ConfigReplication)
        ##Unused Section: METROCLUSTER-INTERCONNECT-CONNECTIONS.XML - (1.30+) Not sure what API this maps to (seems to be covered in InterconnectMirror)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message '- Gathering Metrocluster information -'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name Metrocluster -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing METROCLUSTER.XML'
            $systemAsupData.Metrocluster | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'METROCLUSTER.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_METROCLUSTER' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing METROCLUSTER-INTERCONNECT-ADAPTER.XML'
            $systemAsupData.Metrocluster | Add-Member -MemberType NoteProperty -Name InterconnectAdapter -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'METROCLUSTER-INTERCONNECT-ADAPTER.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_MCC_INTERCONNECT_ADAPTER' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing METROCLUSTER-INTERCONNECT.XML'
            $systemAsupData.Metrocluster | Add-Member -MemberType NoteProperty -Name InterconnectMirror -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'METROCLUSTER-INTERCONNECT.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_MCC_INTERCONNECT' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing METROCLUSTER-NODES.XML'
            $systemAsupData.Metrocluster | Add-Member -MemberType NoteProperty -Name Node -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'METROCLUSTER-NODES.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_MCC_NODES' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing METROCLUSTER-VSERVER.XML'
            $systemAsupData.Metrocluster | Add-Member -MemberType NoteProperty -Name Vserver -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'METROCLUSTER-VSERVER.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_METROCLUSTER_VSERVER_UI' -ErrorVariable +NetAppDocsError)
        }

        #endregion Metrocluster Information

        #region Disk Information

        Write-Verbose -Message '- Gathering disk information -'

        Write-Verbose -Message 'Parsing DISK-PATHS.XML'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name DiskPath -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'DISK-PATHS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_DISK_PATHS' -ErrorVariable +NetAppDocsError)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.130')) {
            Write-Verbose -Message 'Parsing DISK-PARTITION-OWNERSHIP'
            $diskPartitionOwnershipInfo = Get-DiskPartitionOwnership -InputObject (Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'DISK-PARTITION-OWNERSHIP' -ErrorVariable +NetAppDocsError) # | Sort-Object -Property SerialNumber -Unique)
        }

        $validNodeSerialNumbers = $systemAsupData.Node.SerialNo | Where-Object { -not [string]::IsNullOrWhiteSpace($_) } | ForEach-Object { $_ -replace '^0' }

        Write-Verbose -Message 'Parsing DEVICE'
        $diskObjects = $systemAsupData.SourceAsup | Get-AsupObject -ObjectName 'DEVICE' -ReplaceString '^Dvc' -ErrorVariable +NetAppDocsError | Where-Object {
            ($_.Type -eq 'DISK' -or $_.Type -eq 'LUN' -or $_.Type -eq 'VMDISK') -and
            $_.SysId -in $validNodeSerialNumbers -and
            $_.Label -ne 'PARTNER'
        } | Group-Object -Property { $_.Uid[0] } | ForEach-Object {
            $_.Group | Sort-Object -Property { [string]::IsNullOrWhiteSpace($_.Label) }, PrimaryPort | Select-Object -First 1
        }

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.150')) {
            Write-Verbose -Message 'Parsing STORAGE-DISK.XML'
            $storageDisks = Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'STORAGE-DISK.XML' -InnerProperty 'T_DISK_INFO' -ErrorVariable +NetAppDocsError | Select-Object -ExcludeProperty DiskUniqueId, ShelfUid -Property *, @{
                Name       = 'DiskUid'
                Expression = { $_.DiskUniqueId }
            }, @{
                Name       = 'PowerHrs'
                Expression = { $_.PowerOnHours }
            }, @{
                Name       = 'SerialNo'
                Expression = { $_.SerialNumber }
            }, @{
                Name       = 'PoolName'
                Expression = { $_.Pool }
            }, @{
                Name       = 'Position'
                Expression = { $_.DiskPosition }
            }, @{
                Name       = 'FwRev'
                Expression = { $_.FirmwareRevision }
            }, @{
                Name       = 'ShelfUid'
                Expression = { ConvertTo-ShelfLogicalUid -Number $_.ShelfUid }
            }

            Write-Verbose -Message 'Parsing UNASSIGNED-DISKS.XML'
            $unassignedDisks = Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'UNASSIGNED-DISKS.XML' -InnerProperty 'UNASSIGNED_DISKS' -MasterNode $masterNodeName -ErrorVariable +NetAppDocsError | ForEach-Object {
                $unassignedDisk = $_

                $diskHwInfo = (Select-Xml -Xml $Script:HardwareListings.Hardware -XPath "(./ONTAP/Disks/Disk[Model = `"$( $unassignedDisk.Model )`"])[1]").Node

                $unassignedDisk | Select-Object -ExcludeProperty Disk, Type, Bay, ShelfUid -Property *, @{
                    Name       = 'DiskName'
                    Expression = { $_.Disk }
                }, @{
                    Name       = 'SerialNo'
                    Expression = { $_.SerialNumber }
                }, @{
                    Name       = 'DiskType'
                    Expression = { $_.Type }
                }, @{
                    Name       = 'ContainerType'
                    Expression = { 'unassigned' }
                }, @{
                    Name       = 'Position'
                    Expression = { 'present' }
                }, @{
                    Name       = 'ShelfUid'
                    Expression = { ConvertTo-ShelfLogicalUid -Number $_.ShelfUid }
                }, @{
                    Name       = 'ShelfBay'
                    Expression = { $_.Bay }
                }, @{
                    Name       = 'HomeNodeName'
                    Expression = { [string]::Empty }
                }, @{
                    Name       = 'OwnerNodeName'
                    Expression = { [string]::Empty }
                }, @{
                    Name       = 'DiskPathNames'
                    Expression = { ($systemAsupData.DiskPath | Where-Object { $_.DiskUid -eq $unassignedDisk.DiskUid }).LegacyDiskPathNames }
                }, @{
                    Name       = 'VendorName'
                    Expression = {
                        if ($diskHwInfo) {
                            $diskHwInfo.Vendor
                        }
                        else {
                            'UTD'
                        }
                    }
                }
            }

            $systemAsupData | Add-Member -MemberType NoteProperty -Name Disk -Value @( $storageDisks + $unassignedDisks )

            if ($systemAsupData.Metrocluster.Node) {
                $mccNodes = $systemAsupData.Metrocluster.Node | Where-Object { $_.NodeClusterUuid -eq $systemAsupData.Cluster.Info.ClusterIdentifier }

                $systemAsupData | Add-Member -MemberType NoteProperty -Name RemoteDisk -Value $(
                    foreach ($disk in ($diskObjects | Where-Object { -not [string]::IsNullOrWhiteSpace($_.HomeOwner) -and $_.HomeOwner -in $mccNodes.DrPartnerSystemid })) {
                        $disk | Add-Member -MemberType NoteProperty -Name VendorName -Value ([regex]::Replace($disk.Make, '^\s+|\s+$', ''))
                        $disk | Add-Member -MemberType NoteProperty -Name Node -Value ($systemAsupData.Node | Where-Object { ($_.SerialNo -replace '^0') -eq $disk.SysId }).Hostname
                        $disk | Add-Member -MemberType NoteProperty -Name DiskUid -Value ($disk.Uid | Where-Object { $_ -ne $disk.SerialNo })
                        $disk | Add-Member -MemberType NoteProperty -Name SerialNumber -Value $disk.SerialNo
                        $disk | Add-Member -MemberType NoteProperty -Name ShelfSerialNo -Value $disk.ShelfId
                        $disk | Add-Member -MemberType NoteProperty -Name HomeNodeName -Value ($mccNodes | Where-Object { $_.DrPartnerSystemId -eq $disk.HomeOwner }).DrPartnerName
                        $disk | Add-Member -MemberType NoteProperty -Name OwnerNodeName -Value $(
                            if (($mccNode = $mccNodes | Where-Object { $_.DrPartnerSystemId -eq $disk.CurrentOwner })) {
                                $mccNode.DrPartnerName
                            }
                            else {
                                ($systemAsupData.Node | Where-Object { ($_.SystemId -replace '^0') -eq $disk.HomeOwner }).Hostname
                            }
                        )

                        $disk | Add-Member -MemberType NoteProperty -Name DiskName -Value $(
                            if (($diskPath = $systemAsupData.DiskPath | Where-Object { $_.DiskUid -eq $disk.DiskUid -and ($_.DiskPathName -notlike '*:*' -or $_.DiskPathName -like "$( $disk.Node ):*") })) {
                                ($diskPath | Select-Object -First 1).DiskPathName
                            }
                            else {
                                [string]::Format('{0}:{1}', $disk.Node, $disk.PrimaryPath)
                            }
                        )

                        $disk
                    }
                )
            }
        }
        else {
            Write-Verbose -Message 'Parsing STORAGE-DISK'
            $storageDiskObjectInfo = Get-StorageDiskObjectInfo -InputObject (Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'STORAGE-DISK' -ErrorVariable +NetAppDocsError)

            foreach ($disk in $diskObjects) {
                $disk | Add-Member -MemberType NoteProperty -Name Node -Value ($systemAsupData.Node | Where-Object { ($_.SerialNo -replace '^0') -eq $disk.SysId }).Hostname
                $disk | Add-Member -MemberType NoteProperty -Name DiskUid -Value ($disk.Uid | Where-Object { $_ -ne $disk.SerialNo })
                $disk | Add-Member -MemberType NoteProperty -Name SerialNumber -Value $disk.SerialNo

                $storageDiskObject = $storageDiskObjectInfo | Where-Object { $_.DiskUid -eq $disk.DiskUid } | Select-Object -First 1
                $disk | Add-Member -MemberType NoteProperty -Name Drawer -Value $storageDiskObject.Drawer
                $disk | Add-Member -MemberType NoteProperty -Name Slot -Value $storageDiskObject.Slot

                $disk | Add-Member -MemberType NoteProperty -Name HomeNodeName -Value $(
                    if (($mccNode = $systemAsupData.Metrocluster.Node | Where-Object { $_.NodeClusterUuid -ceq $systemAsupData.Cluster.Info.ClusterIdentifier -and $_.DrPartnerSystemId -eq $disk.HomeOwner })) {
                        $mccNode.DrPartnerName
                    }
                    else {
                        ($systemAsupData.Node | Where-Object { ($_.SystemId -replace '^0') -eq $disk.HomeOwner }).Hostname
                    }
                )

                $disk | Add-Member -MemberType NoteProperty -Name OwnerNodeName -Value $(
                    if (($mccNode = $systemAsupData.Metrocluster.Node | Where-Object { $_.NodeClusterUuid -ceq $systemAsupData.Cluster.Info.ClusterIdentifier -and $_.DrPartnerSystemId -eq $disk.CurrentOwner })) {
                        $mccNode.DrPartnerName
                    }
                    else {
                        ($systemAsupData.Node | Where-Object { ($_.SystemId -replace '^0') -eq $disk.HomeOwner }).Hostname
                    }
                )

                $disk | Add-Member -MemberType NoteProperty -Name DiskName -Value $(
                    if (($diskPath = $systemAsupData.DiskPath | Where-Object { $_.DiskUid -eq $disk.DiskUid -and ($_.DiskPathName -notlike '*:*' -or $_.DiskPathName -like "$( $disk.Node ):*") })) {
                        ($diskPath | Select-Object -First 1).DiskPathName
                    }
                    else {
                        [string]::Format('{0}:{1}', $disk.Node, $disk.PrimaryPath)
                    }
                )

                if ([string]::IsNullOrWhiteSpace($disk.DiskType) -or $disk.DiskType -eq 'UNKNOWN') {
                    $disk | Add-Member -MemberType NoteProperty -Name DiskType -Value $(
                        if ($disk.Type -eq 'DISK' -and (($diskHwInfo = (Select-Xml -Xml $Script:HardwareListings.Hardware -XPath "(./ONTAP/Disks/Disk[Model = `"$( $disk.Model )`"])[1]").Node))) {
                            $diskHwInfo.OntapType
                        }
                        else {
                            'UNKNOWN'
                        }
                    ) -Force
                }

                if ($disk.SerialNo -eq 'StartupXfailedX' -or $disk.StateStor -eq 'FAILED') {
                    $disk | Add-Member -MemberType NoteProperty -Name Label -Value 'BROKEN' -Force
                }

                if ([string]::IsNullOrWhiteSpace($disk.PoolName) -and $diskPartitionOwnershipInfo) {
                    $disk | Add-Member -MemberType NoteProperty -Name PoolName -Value ($diskPartitionOwnershipInfo | Where-Object { $_.SerialNumber -eq $disk.SerialNo }).PoolName -Force
                }

                if ([string]::IsNullOrWhiteSpace($disk.PhySizeBlocks) -and $systemAsupData.PartitionDisk) {
                    $disk | Add-Member -MemberType NoteProperty -Name PhySizeBlocks -Value ($systemAsupData.PartitionDisk | Where-Object { $_.SerialNo -like "$( $disk.SerialNo )*" } | Measure-Object -Property PhySizeBlocks -Sum).Sum -Force
                }

                if ($disk.Label -eq 'BROKEN') {
                    $disk | Add-Member -MemberType NoteProperty -Name PoolName -Value 'broken' -Force
                }

                #?
                #if ([string]::IsNullOrWhiteSpace($disk.Label) -and [string]::IsNullOrWhiteSpace($disk.OwnerNodeName)) {
                #    $disk | Add-Member -MemberType NoteProperty -Name Label -Value 'UNOWNED' -Force
                #}
            }

            $systemAsupData | Add-Member -MemberType NoteProperty -Name Disk -Value $diskObjects
        }

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message 'Parsing PARTITION_DISK'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name PartitionDisk -Value ($systemAsupData.SourceAsup | Get-AsupObject -ObjectName 'PARTITION_DISK' -ReplaceString '^Dvc' -ErrorVariable +NetAppDocsError | Where-Object { $_.Label -ne 'PARTNER' })

            foreach ($partitionDisk in ($systemAsupData.PartitionDisk | Where-Object { $_ })) {
                if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.150')) {
                    if (($containerDisk = $systemAsupData.Disk | Where-Object { $_.SerialNumber -eq $partitionDisk.SerialNo })) {
                        $partitionDisk | Add-Member -MemberType NoteProperty -Name ContainerDisk -Value $containerDisk.DiskName
                        $partitionDisk | Add-Member -MemberType NoteProperty -Name ContainerDiskUid -Value $containerDisk.DiskUid
                        $partitionDisk | Add-Member -MemberType NoteProperty -Name DiskType -Value $containerDisk.DiskType
                    }
                }
                else {
                    if (($containerDisk = $diskObjects | Where-Object { $_.SerialNo -eq $partitionDisk.SerialNo } | Sort-Object -Property PrimaryPort | Select-Object -First 1)) {
                        $partitionDisk | Add-Member -MemberType NoteProperty -Name ContainerDisk -Value $containerDisk.DiskName
                        $partitionDisk | Add-Member -MemberType NoteProperty -Name ContainerDiskUid -Value $containerDisk.DiskUid
                        $partitionDisk | Add-Member -MemberType NoteProperty -Name DiskType -Value $containerDisk.DiskType
                    }
                }

                if ($diskPartitionOwnershipInfo) {
                    #$_.DiskName -eq $partitionDisk.PathName -or
                    $diskPartitionOwnership = $diskPartitionOwnershipInfo | Where-Object { $_.SerialNumber -like "$( $partitionDisk.SerialNo )NP00$( $partitionDisk.PartitionName.Substring(1,1) )" } | Select-Object -First 1

                    $partitionDisk | Add-Member -MemberType NoteProperty -Name PoolName -Value $diskPartitionOwnership.PoolName
                    $partitionDisk | Add-Member -MemberType NoteProperty -Name OwnerNodeName -Value $diskPartitionOwnership.OwnerNodeName
                    $partitionDisk | Add-Member -MemberType NoteProperty -Name HomeNodeName -Value $diskPartitionOwnership.HomeNodeName
                }
                else {
                    $partitionDisk | Add-Member -MemberType NoteProperty -Name PoolName -Value ($diskObjects | Where-Object { $_.SerialNo -eq $partitionDisk.SerialNo -and $_.Label -ne 'REMOTE' } | Select-Object -First 1).PoolName
                }
            }
        }

        #endregion Disk Information

        #region Disk Encrypt Information

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.140')) {
            Write-Verbose -Message '- Gathering disk encrypt information -'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name DiskEncrypt -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing STORAGE-ENCRYPTION-DISK.XML'
            $systemAsupData.DiskEncrypt | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'STORAGE-ENCRYPTION-DISK.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_DISK_ENCRYPT' -ErrorVariable +NetAppDocsError)
        }

        #endregion Disk Encrypt Information

        #region Fpolicy Configuration

        ##Unused Section: FPOLICY-POLICY-STATUS.XML - (1.20+) Not needed for build docs (Fpolicy.Status)

        Write-Verbose -Message '- Gathering Fpolicy configuration -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Fpolicy -Value ([pscustomobject]@{ })

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Parsing FPOLICY-SERVER-STATUS.XML'
            $systemAsupData.Fpolicy | Add-Member -MemberType NoteProperty -Name ServerStatus -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FPOLICY-SERVER-STATUS.XML' -InnerProperty 'T_FPOLICY_SERVER_STATUS' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing FPOLICY-EXT-ENGINE.XML'
            $systemAsupData.Fpolicy | Add-Member -MemberType NoteProperty -Name ExternalEngine -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FPOLICY-EXT-ENGINE.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FPOLICY_EXT_ENGINE' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing FPOLICY-POLICY.XML'
            $systemAsupData.Fpolicy | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FPOLICY-POLICY.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FPOLICY_POLICY' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing FPOLICY-EVENT.XML'
            $systemAsupData.Fpolicy | Add-Member -MemberType NoteProperty -Name Event -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FPOLICY-EVENT.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FPOLICY_EVENT' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing FPOLICY-SCOPE.XML'
            $systemAsupData.Fpolicy | Add-Member -MemberType NoteProperty -Name Scope -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FPOLICY-SCOPE.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FPOLICY_SCOPE' -ErrorVariable +NetAppDocsError)
        }

        #endregion Fpolicy Configuration

        #region Syslog (EMS) Data

        ##Unused Section: EMS-LOG-CONFIG.XML - Not needed for build docs (Ems.Config)
        ##Unused Section: EVENT-FILTER.XML - Not needed for build docs (Ems.Filter)
        ##Unused Section: EVENT-NOTIFICATION.XML - Not needed for build docs (Ems.EventNotification.Info)

        Write-Verbose -Message '- Gathering EMS information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Ems -Value ([pscustomobject]@{ })

        if ($systemAsupData.System.OntapiVersion -lt [version]::Parse('1.100')) {
            Write-Verbose -Message 'Parsing EMS-DESTINATION.XML'
            $systemAsupData.Ems | Add-Member -MemberType NoteProperty -Name Destination -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'EMS-DESTINATION.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_EMS_DESTINATIONS' -ErrorVariable +NetAppDocsError)
        }

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.100')) {
            Write-Verbose -Message 'Parsing EVENT-FILTER.XML'
            $systemAsupData.Ems | Add-Member -MemberType NoteProperty -Name EventFilter -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'EVENT-FILTER.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_EVENT_FILTERS' -ErrorVariable +NetAppDocsError)

            $systemAsupData.Ems | Add-Member -MemberType NoteProperty -Name EventNotification -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing EVENT-NOTIFICATION.XML'
            $systemAsupData.Ems.EventNotification | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'EVENT-NOTIFICATION.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_EVENT_NOTIFICATIONS' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing EVENT-NOTIFICATION-DESTINATION.XML'
            $systemAsupData.Ems.EventNotification | Add-Member -MemberType NoteProperty -Name Destination -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'EVENT-NOTIFICATION-DESTINATION.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_EVENT_NOTIFICATION_DESTINATIONS' -ErrorVariable +NetAppDocsError)
        }

        #endregion Syslog (EMS) Data

        #region Environment Information

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.21')) {
            Write-Verbose -Message '- Gathering environment information -'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name Environment -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing PLATFORM-SENSORS.XML'
            $systemAsupData.Environment | Add-Member -MemberType NoteProperty -Name Sensors -Value @(
                foreach ($srcAsup in $systemAsupData.SourceAsup) {
                    Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'PLATFORM-SENSORS.XML' -InnerProperty 'T_ENVIRONMENT' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                        Name       = 'NodeName'
                        Expression = { $srcAsup.SystemName }
                    }
                }
            )
        }

        #endregion Environment Information

        #region Aggregate Configuration

        Write-Verbose -Message '- Gathering aggregate information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Aggr -Value ([pscustomobject]@{ })

        $aggrRaidStatusAttrs = @(
            [pscustomobject]@{
                PropertyName = 'HasDegradedRg'
                StatusString = 'degraded'
            }
            [pscustomobject]@{
                PropertyName = 'HasReconsRg'
                StatusString = 'reconstruct'
            }
            [pscustomobject]@{
                PropertyName = 'HasUninitParity'
                StatusString = 'noparity'
            }
            [pscustomobject]@{
                PropertyName = 'IsSyncMirrored'
                StatusString = 'mirrored'
            }
            [pscustomobject]@{
                PropertyName = 'IsHybrid'
                StatusString = 'hybrid'
            }
            [pscustomobject]@{
                PropertyName = 'IsGrowing'
                StatusString = 'growing'
            }
            [pscustomobject]@{
                PropertyName = 'IsForeign'
                StatusString = 'foreign'
            }
            [pscustomobject]@{
                PropertyName = 'IsVerifiying'
                StatusString = 'verifying'
            }
            [pscustomobject]@{
                PropertyName = 'IsPartial'
                StatusString = 'partial'
            }
            [pscustomobject]@{
                PropertyName = 'IsNormal'
                StatusString = 'normal'
            }
        )

        Write-Verbose -Message 'Parsing AGGREGATE'
        $aggrObjects = @(
            foreach ($srcAsup in $systemAsupData.SourceAsup) {
                $clusterMemberId = ($systemAsupData.Node | Where-Object { $_.Hostname -ceq $srcAsup.SystemName }).ClusterMemberId

                Get-AsupObject -AsupObject $srcAsup -ObjectName 'AGGREGATE' -ReplaceString '^Aggr' -ErrorVariable +NetAppDocsError | Where-Object { $_.DbladeUuid -eq $clusterMemberId } | Add-Member -MemberType NoteProperty -Name Nodes -Value @( $srcAsup.SystemName ) -PassThru | Select-Object -Property *, @{
                    Name       = 'RaidStatus'
                    Expression = {
                        $(
                            foreach ($statusAttr in $aggrRaidStatusAttrs) {
                                if ($_.$( $statusAttr.PropertyName ) -eq $true) {
                                    $statusAttr.StatusString
                                }
                            }
                        ) -join ', '
                    }
                }
            }
        )

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            $systemAsupData.Aggr | Add-Member -MemberType NoteProperty -Name Autobalance -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing AGGREGATE-AUTOBALANCE-INFO.XML'
            $systemAsupData.Aggr.Autobalance | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'AGGREGATE-AUTOBALANCE-INFO.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_AUTO_BALANCER_CONFIGURATION' -ErrorVariable +NetAppDocsError)

            if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.100')) {
                Write-Verbose -Message 'Parsing AGGR-EFFICIENCY.XML'
                $systemAsupData.Aggr | Add-Member -MemberType NoteProperty -Name Efficiency -Value $(
                    foreach ($nodeAsup in ($systemAsupData.SourceAsup | Where-Object { $_ })) {
                        Get-AsupSection -AsupObject $nodeAsup -SectionName 'AGGR-EFFICIENCY.XML' -InnerProperty 'T_AGGR_EFFICIENCY' -ErrorVariable +NetAppDocsError | Where-Object { $_.Node -ceq $nodeAsup.SystemName }
                    }
                )

                Write-Verbose -Message 'Parsing AGGR-EFFICIENCY-CONFIG.XML'
                $systemAsupData.Aggr | Add-Member -MemberType NoteProperty -Name EfficiencyConfig -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'AGGR-EFFICIENCY-CONFIG.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_AGGR_SIS' -ErrorVariable +NetAppDocsError)

                if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.120')) {
                    $systemAsupData.Aggr | Add-Member -MemberType NoteProperty -Name ObjectStore -Value ([pscustomobject]@{ })

                    Write-Verbose -Message 'Parsing AGGR-BIN.XML'
                    $systemAsupData.Aggr.ObjectStore | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'AGGR-BIN.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_AGGR_BIN' -ErrorVariable +NetAppDocsError)

                    Write-Verbose -Message 'Parsing DATA-OBJECT-STORE-CONFIG.XML'
                    $systemAsupData.Aggr.ObjectStore | Add-Member -MemberType NoteProperty -Name Config -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'DATA-OBJECT-STORE-CONFIG.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_AGGR_OBJ_STORE_CONFIG' -ErrorVariable +NetAppDocsError)
                }

                if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.150')) {
                    Write-Verbose -Message 'Parsing AGGR-INFO.XML'
                    $aggrInfo = Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'AGGR-INFO.XML' -InnerProperty 'T_AGGR_INFO' -ErrorVariable +NetAppDocsError
                }
            }
        }

        Write-Verbose -Message 'Parsing AGGRTABLE.XML'
        $aggrTable = Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'AGGRTABLE.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_AGGR' -ErrorVariable +NetAppDocsError

        $aggrSpaceInfo = Get-AggrSpace -AsupObject $systemAsupData.SourceAsup -AggrObject $aggrObjects

        foreach ($aggr in $aggrObjects) {
            $aggrInfoEntry = $aggrInfo | Where-Object { $_.Uuid -eq $aggr.Uuid }
            $aggrTableEntry = $aggrTableEntry = $aggrTable | Where-Object { $_.Id -eq $aggr.Uuid }

            if ($aggrInfoEntry) {
                $aggr | Add-Member -MemberType NoteProperty -Name HomeName -Value $aggrInfoEntry.HomeName
                $aggr | Add-Member -MemberType NoteProperty -Name OwnerName -Value $aggrInfoEntry.OwnerName
                $aggr | Add-Member -MemberType NoteProperty -Name State -Value $aggrInfoEntry.State
                $aggr | Add-Member -MemberType NoteProperty -Name HaPolicy -Value $aggrInfoEntry.HaPolicy
                $aggr | Add-Member -MemberType NoteProperty -Name HasLocalRoot -Value ($aggrInfoEntry.HasMroot -eq $true)
                $aggr | Add-Member -MemberType NoteProperty -Name ResyncSnapTime -Value $aggrInfoEntry.ResyncSnapTime
                $aggr | Add-Member -MemberType NoteProperty -Name OptionNoSnap -Value $aggr.Options.nosnap
                $aggr | Add-Member -MemberType NoteProperty -Name FreeSpaceReallocation -Value $aggrInfoEntry.FreeSpaceReallocation
                $aggr | Add-Member -MemberType NoteProperty -Name PercentSnapshotSpace -Value ($aggrInfoEntry.PctReservedForSnapshots -replace '\s*%')
                $aggr | Add-Member -MemberType NoteProperty -Name HybridEnabled -Value $aggrInfoEntry.HybridEnabled
                $aggr | Add-Member -MemberType NoteProperty -Name AutobalanceState -Value $aggrInfoEntry.AutobalanceState
                $aggr | Add-Member -MemberType NoteProperty -Name PhysicalUsedPercent -Value ($aggrInfoEntry.PhysicalUsedPercent -replace '\s*%')
                $aggr | Add-Member -MemberType NoteProperty -Name SnaplockType -Value ($aggrInfoEntry.SnaplockType -replace '-', '_')
                $aggr | Add-Member -MemberType NoteProperty -Name DataCompactedCount -Value ([long]$aggrInfoEntry.DataCompactedCount)
                $aggr | Add-Member -MemberType NoteProperty -Name DataCompactionSpaceSavedPercent -Value ($aggrInfoEntry.DataCompactionSpaceSavedPercent -replace '\s*%')
                $aggr | Add-Member -MemberType NoteProperty -Name CompositeCapacityTierUsed -Value $(
                    if ($aggrInfoEntry.Composite -eq $true) {
                        [long]$aggrInfoEntry.CompositeCapacityTierUsed
                    }
                )

                $aggr | Add-Member -MemberType NoteProperty -Name IsEncrypted -Value $(
                    if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.160')) {
                        $aggrInfoEntry.EncryptWithAggrKey
                    }
                    elseif ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.31')) {
                        $aggrInfoEntry.IsEncrypted
                    }
                )

                if ($aggr.psobject.Properties.Name -notcontains 'IsSnaplock') {
                    $aggr | Add-Member -MemberType NoteProperty -Name IsSnaplock -Value $aggrInfoEntry.IsSnaplock
                }

                if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.120')) {
                    $aggr | Add-Member -MemberType NoteProperty -Name IsComposite -Value $aggrInfoEntry.Composite
                }
            }
            elseif ($aggrTableEntry) {
                $aggr | Add-Member -MemberType NoteProperty -Name HomeName -Value ($systemAsupData.Node | Where-Object { $_.ClusterMemberId -eq $aggrTableEntry.HomeBladeid }).Hostname
                $aggr | Add-Member -MemberType NoteProperty -Name OwnerName -Value ($systemAsupData.Node | Where-Object { $_.ClusterMemberId -eq $aggrTableEntry.BladeId }).Hostname
                $aggr | Add-Member -MemberType NoteProperty -Name HaPolicy -Value $aggr.Options.ha_policy
                $aggr | Add-Member -MemberType NoteProperty -Name HasLocalRoot -Value ($aggr.Options.diskroot -eq $true)
                $aggr | Add-Member -MemberType NoteProperty -Name ResyncSnapTime -Value $aggr.Options.resyncsnaptime
                $aggr | Add-Member -MemberType NoteProperty -Name OptionNoSnap -Value $aggr.Options.nosnap
                $aggr | Add-Member -MemberType NoteProperty -Name FreeSpaceReallocation -Value $aggr.Options.free_space_realloc
                $aggr | Add-Member -MemberType NoteProperty -Name PercentSnapshotSpace -Value ($aggr.Options.percent_snapshot_space -replace '\s*%')

                $aggr | Add-Member -MemberType NoteProperty -Name State -Value $(
                    if (-not [string]::IsNullOrWhiteSpace($aggr.StateAccess)) {
                        $aggr.StateAccess.ToLower()
                    }
                )

                $aggr | Add-Member -MemberType NoteProperty -Name HybridEnabled -Value $(
                    if (-not [string]::IsNullOrWhiteSpace($aggr.Options.hybrid_enabled)) {
                        $aggr.Options.hybrid_enabled -eq 'on'
                    }
                )

                $aggr | Add-Member -MemberType NoteProperty -Name SnaplockType -Value $(
                    if (-not [string]::IsNullOrWhiteSpace($aggrTableEntry.SnaplockType)) {
                        $aggrTableEntry.SnaplockType -replace '-', '_'
                    }
                    #else {
                    #    'non_snaplock'
                    #}
                )

                if ($aggr.psobject.Properties.Name -notcontains 'IsSnaplock') {
                    $aggr | Add-Member -MemberType NoteProperty -Name IsSnaplock -Value ($aggr.SnaplockType -in @( 'compliance', 'enterprise' ))
                }

                if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.120')) {
                    $aggr | Add-Member -MemberType NoteProperty -Name IsComposite -Value $(
                        if (-not [string]::IsNullOrWhiteSpace($aggr.Options.composite)) {
                            $aggr.Options.composite -eq 'on'
                        }
                    )
                }
            }

            if (($aggrSpaceEntry = $aggrSpaceInfo | Where-Object { $_.AggregateName -ceq $aggr.Name })) {
                if ($aggr.psobject.Properties.Name -notcontains 'PhysicalUsedPercent') {
                    $aggr | Add-Member -MemberType NoteProperty -Name PhysicalUsedPercent -Value $aggrSpaceEntry.PhysicalUsedPercent
                }

                $aggr | Add-Member -MemberType NoteProperty -Name SnapSizeUsed -Value $aggrSpaceEntry.SnapSizeUsed
                $aggr | Add-Member -MemberType NoteProperty -Name SnapSizeTotal -Value $aggrSpaceEntry.SnapSizeTotal
            }

            if (($aggrEffEntry = $systemAsupData.Aggr.Efficiency | Where-Object { $_.Aggr -ceq $aggr.Name })) {
                if (-not $aggr.Nodes) {
                    $aggr | Add-Member -MemberType NoteProperty -Name Nodes -Value @( $aggrEffEntry.Node ) -Force
                }

                $aggr | Add-Member -MemberType NoteProperty -Name DataCompactionSpaceSaved -Value ([long]$aggrEffEntry.Acs)
                $aggr | Add-Member -MemberType NoteProperty -Name SisSpaceSaved -Value ([long]$aggrEffEntry.Ves)
                $aggr | Add-Member -MemberType NoteProperty -Name AggrPhysicalUsed -Value ([long]$aggrEffEntry.Apu)
                $aggr | Add-Member -MemberType NoteProperty -Name PhysicalUsed -Value ([long]$aggrEffEntry.Tpu) -Force
            }
        }

        $systemAsupData.Aggr | Add-Member -MemberType NoteProperty -Name Info -Value $aggrObjects

        #endregion Aggregate Configuration

        #region Plex/RaidGroup Configuration

        Write-Verbose -Message '- Gathering plex/rg information -'

        Write-Verbose -Message 'Parsing PLEX'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Plex -Value ($systemAsupData.SourceAsup | Get-AsupObject -ObjectName 'PLEX' -ReplaceString '^Plex' -ErrorVariable +NetAppDocsError | Sort-Object -Property * -Unique)

        Write-Verbose -Message 'Parsing RAID_GROUP'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name RaidGroup -Value ($systemAsupData.SourceAsup | Get-AsupObject -ObjectName 'RAID_GROUP' -ReplaceString '^Rg' -ErrorVariable +NetAppDocsError | Sort-Object -Property * -Unique)

        foreach ($rg in ($systemAsupData.RaidGroup | Where-Object { $_ })) {
            $rg | Add-Member -MemberType NoteProperty -Name IsCacheTier -Value (
                ($systemAsupData.Disk | Where-Object { $_.DiskType -eq 'SSD' -and $_.RgId -ceq $rg.PathName } | Measure-Object).Count -gt 0 -or ($systemAsupData.PartitionDisk | Where-Object { $_.DiskType -eq 'SSD' -and $_.RgId -ceq $rg.PathName } | Measure-Object).Count -gt 0
            )
        }

        #endregion Plex/RaidGroup Configuration

        #region SIS Configuration

        Write-Verbose -Message '- Gathering SIS configuration -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Sis -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing SIS-STATUS-L.XML'
        $systemAsupData.Sis | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SIS-STATUS-L.XML' -InnerProperty 'T_SIS' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing SIS-POLICY-UI.XML'
        $systemAsupData.Sis | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SIS-POLICY-UI.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_SIS_POLICY_UI' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing SIS-STAT-L.XML'
        $sisStatL = Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SIS-STAT-L.XML' -InnerProperty 'T_SIS_STAT' -ErrorVariable +NetAppDocsError

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.150')) {
            Write-Verbose -Message 'Parsing SIS-CONFIG.XML'
            $systemAsupData.Sis | Add-Member -MemberType NoteProperty -Name Config -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SIS-CONFIG.XML' -InnerProperty 'T_SIS_CONFIG' -ErrorVariable +NetAppDocsError)
        }

        #endregion SIS Configuration

        #region Volume Configuration

        Write-Verbose -Message '- Gathering volume information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Vol -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing VOLUME.XML'
        $systemAsupData.Vol | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VOLUME.XML' -InnerProperty 'T_VOLUME' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing VOLUME'
        $volObjs = $systemAsupData.SourceAsup | Get-AsupObject -ObjectName 'VOLUME' -ReplaceString '^Vol' -ErrorVariable +NetAppDocsError

        foreach ($vol in ($systemAsupData.Vol.Info | Where-Object { $_ })) {
            $volObj = $volObjs | Where-Object { $_.VserverId -ceq $vol.Vs -and $_.InstanceUuid -eq $vol.IUuid }
            $volSisStatL = $sisStatL | Where-Object { $_.Vs -ceq $vol.Vs -and $_.Path -ceq "/vol/$( $vol.Vol )" }

            if ($volObj) {
                foreach ($prop in ($volObj | Get-Member -Name Asis*, Compress*, SAllocated*, SAvail*, Sused*, TotSaved*, IsOnline, Options)) {
                    $vol | Add-Member -MemberType NoteProperty -Name $prop.Name -Value $volObj.$( $prop.Name )
                }

                $vol | Add-Member -MemberType NoteProperty -Name IsNodeRoot -Value $volObj.IsRoot
                $vol | Add-Member -MemberType NoteProperty -Name LanguageString -Value $volObj.Language
                $vol | Add-Member -MemberType NoteProperty -Name CloneChildCount -Value $volObj.CloneCt
            }

            $vol | Add-Member -MemberType NoteProperty -Name IsSisVolume -Value $(
                if ($sisStatL) {
                    $null -ne $volSisStatL
                }
            )
        }

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Calling Get-AsupVolSpace'
            $systemAsupData.Vol | Add-Member -MemberType NoteProperty -Name Space -Value (Get-AsupVolSpace -AsupObject $systemAsupData.SourceAsup)

            if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.130')) {
                Write-Verbose -Message 'Parsing VOLUME-CLONE-SHOW.XML'
                $systemAsupData.Vol | Add-Member -MemberType NoteProperty -Name CloneVol -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VOLUME-CLONE-SHOW.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VOLUME_CLONE_SHOW' -ErrorVariable +NetAppDocsError)
            }
        }

        #endregion Volume Configuration

        #region Infinivol Repository Configuration

        ##Unused Section: REPOSITORY-INFO.XML - Not needed for build docs (Repository.Info)
        ##Unused Section: REPOSITORY-POLICY-RULES.XML - Not needed for build docs (Repository.PolicyRules)
        ##Unused Section: REPOSITORY-STORAGE-SERVICES.XML - Not needed for build docs (Repository.StorageServices)

        Write-Verbose -Message '- Gathering infinite volume repository information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Repository -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing REPOSITORY-CONSTITUENTS.XML'
        $systemAsupData.Repository | Add-Member -MemberType NoteProperty -Name Constituents -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'REPOSITORY-CONSTITUENTS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_REPOSITORY_CONSTITUENT' -ErrorVariable +NetAppDocsError)

        #endregion Infinivol Repository Configuration

        #region Qtree Configuration

        Write-Verbose -Message '- Gathering qtree information -'

        Write-Verbose -Message 'Parsing QTREE.XML'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Qtree -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'QTREE.XML' -InnerProperty 'T_QTREE' -ErrorVariable +NetAppDocsError)

        #endregion Qtree Configuration

        #region Name Mapping Data

        Write-Verbose -Message '- Gathering name mapping data -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name NameMapping -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing NAMEMAPPINGS-BYNAME.XML'
        $systemAsupData.NameMapping | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NAMEMAPPINGS-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VSERV_NAMEMAPPING' -ReplaceString '^Vservnamemap' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing UNIX-USERS-BYNAME.XML'
        $systemAsupData.NameMapping | Add-Member -MemberType NoteProperty -Name UnixUser -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'UNIX-USERS-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_UNIX_USERS' -ReplaceString '^Unixusers' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing UNIX-GROUPS-BYNAME.XML'
        $systemAsupData.NameMapping | Add-Member -MemberType NoteProperty -Name UnixGroup -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'UNIX-GROUPS-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_UNIX_GROUP' -ReplaceString '^Unixgroup' -ErrorVariable +NetAppDocsError)

        #endregion Name Mapping Data

        #region Group Mapping Data

        Write-Verbose -Message '- Gathering group mapping data -'

        #Unsure of InnerProperty
        Write-Verbose -Message 'Parsing GROUP-MAPPINGS.XML'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name GroupMapping -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'GROUP-MAPPINGS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_GROUP_MAPPING' -ErrorVariable +NetAppDocsError)

        #endregion Group Mapping Data

        #region Vscan Configuration

        ##Unused Section: VSCAN-ACTIVE-SCANNER-POOL.XML - (1.20+) Not needed for build docs (Vscan.ScannerPoolActive)
        ##Unused Section: VSCAN-EVENTS.XML - (1.31+) Not needed for build docs (Vscan.Events)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.21')) {
            Write-Verbose -Message '- Gathering Vscan information -'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name Vscan -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing VSCAN-CONNECTION-STATUS-ALL.XML'
            $systemAsupData.Vscan | Add-Member -MemberType NoteProperty -Name Connection -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VSCAN-CONNECTION-STATUS-ALL.XML' -InnerProperty 'T_VSCAN_CONNECTION_STATUS_ALL' -ReplaceString '^Vscanserver' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing VSCAN-STATUS.XML'
            $systemAsupData.Vscan | Add-Member -MemberType NoteProperty -Name Status -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VSCAN-STATUS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VSCAN_STATUS' -ReplaceString '^Vscan' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing VSCAN-ON-ACCESS-POLICY.XML'
            $systemAsupData.Vscan | Add-Member -MemberType NoteProperty -Name OnAccessPolicy -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VSCAN-ON-ACCESS-POLICY.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VSCAN_ON_ACCESS_POLICY' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing VSCAN-SCANNER-POOL.XML'
            $systemAsupData.Vscan | Add-Member -MemberType NoteProperty -Name ScannerPool -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VSCAN-SCANNER-POOL.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VSCAN_SCANNER_POOL' -ErrorVariable +NetAppDocsError)
        }

        #endregion Vscan Configuration

        #region Network Information

        ##Unused Section: DATA-LIF-CAPACITY.XML - (1.100+) Not needed for build docs (Net.DataLifCapacity)
        ##Unused Section: DATA-LIF-CAPACITY-DETAILS.XML - (1.100+) Not needed for build docs (Net.DataLifCapacityDetails)
        ##Unused Section: QOS-MARKING.XML - (1.100+) Not needed for build docs (Net.QosMarking)
        ##Unused Section: IPV6-OPTIONS.XML - (1.20+) Not needed for build docs (Net.Option.Ipv6Enabled)

        Write-Verbose -Message '- Gathering network information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Net -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing INTERFACE'
        $ports = @(
            foreach ($srcAsup in $systemAsupData.SourceAsup) {
                Get-AsupObject -AsupObject $srcAsup -ObjectName 'INTERFACE' -ReplaceString '^If' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                    Name       = 'Node'
                    Expression = { $srcAsup.SystemName }
                }
            }
        )

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Parsing NETWORK-PORTS.XML'
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Port -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NETWORK-PORTS.XML' -InnerProperty 'T_PORT' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing VLANS.XML'
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name PortVlan -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VLANS.XML' -InnerProperty 'T_VLAN' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing IFGRPS.XML'
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name PortIfgrp -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'IFGRPS.XML' -InnerProperty 'T_IFGRP' -ErrorVariable +NetAppDocsError)
        }
        else {
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Port -Value ($ports | Where-Object { $_.Type -eq 'ENET' -and $_.Uid -notmatch '^(?:e0P|losk|c0)' })
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name PortIfgrp -Value ($ports | Where-Object { $_.Type -eq 'ENET' -and $_.MediaType -like '*interface groups*' })
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name PortVlan -Value ($ports | Where-Object { $_.Type -eq 'ENET' -and $_.MediaType -like '*VLAN*' })
        }

        Write-Verbose -Message 'Parsing CLUSTER_LIF'
        $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Interface -Value ($systemAsupData.SourceAsup | Get-AsupObject -ObjectName 'CLUSTER_LIF' -ReplaceString '^Vif' -ErrorVariable +NetAppDocsError)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Option -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing NETOPTS-SWITCHLESS-CLUSTER.XML'
            $systemAsupData.Net.Option | Add-Member -MemberType NoteProperty -Name SwitchlessClusterEnabled -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NETOPTS-SWITCHLESS-CLUSTER.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_TNSC' -ErrorVariable +NetAppDocsError).Enabled

            if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.21')) {
                Write-Verbose -Message 'Parsing DEVICE-DISCOVERY.XML'
                $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name DeviceDiscovery -Value (
                    Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'DEVICE-DISCOVERY.XML' -InnerProperty 'T_NET_DEVICE_DISCOVERY' -ReplaceString '^Discovered' -ErrorVariable +NetAppDocsError | Select-Object -ExcludeProperty Version -Property *, @{
                        Name       = 'Version'
                        Expression = { $_.Version -join "`n" }
                    }
                )
            }
        }

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message 'Parsing IPSPACES.XML'
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Ipspace -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'IPSPACES.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_IPSPACES' -ReplaceString '^Ipspace' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing SUBNET.XML'
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Subnet -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SUBNET.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_SUBNET' -ReplaceString '^Subnet(?=.+)' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing BROADCAST-DOMAIN.XML'
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name PortBroadcastDomain -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'BROADCAST-DOMAIN.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_BROADCAST_DOMAIN' -ReplaceString '^BroadcastDomain' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing NETWORK-ROUTES.XML'
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Route -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NETWORK-ROUTES.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_ROUTES' -ReplaceString '^Route' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing VS-FAILOVER-GROUPS.XML'
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name FailoverGroup -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VS-FAILOVER-GROUPS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VS_FAILOVER_GROUPS' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing VS-FIREWALL-POLICIES.XML'
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name FirewallPolicy -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VS-FIREWALL-POLICIES.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_vs_firewall_policies' -ErrorVariable +NetAppDocsError)

            if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.31')) {
                Write-Verbose -Message 'Parsing DDNS.XML'
                $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Ddns -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'DDNS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_DDNS' -ErrorVariable +NetAppDocsError)

                if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.150')) {
                    $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name ServicePolicy -Value ([pscustomobject]@{ })

                    Write-Verbose -Message 'Parsing NETWORK-SERVICE-POLICY.XML'
                    $systemAsupData.Net.ServicePolicy | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NETWORK-SERVICE-POLICY.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_LIF_SERVICE_POLICIES' -ErrorVariable +NetAppDocsError)

                    Write-Verbose -Message 'Parsing NETWORK-SERVICE-POLICY-ENTRY.XML'
                    $systemAsupData.Net.ServicePolicy | Add-Member -MemberType NoteProperty -Name Entry -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NETWORK-SERVICE-POLICY-ENTRY.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_LIF_SERVICE_POLICY_ENTRIES' -ErrorVariable +NetAppDocsError)
                }
            }
        }
        else {
            Write-Verbose -Message 'Parsing FAILOVER-GROUPS.XML'
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name FailoverGroup -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FAILOVER-GROUPS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FAILOVER_GROUP' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing NETWORK-ROUTES.XML'
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name RoutingGroupRoute -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NETWORK-ROUTES.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_ROUTES' -ErrorVariable +NetAppDocsError)

            if ($systemAsupData.System.OntapiVersion -lt [version]::Parse('1.30')) {
                Write-Verbose -Message 'Parsing NETWORK-ROUTING-GROUPS.XML'
                $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name RoutingGroup -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NETWORK-ROUTING-GROUPS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_ROUTING_GROUPS' -ErrorVariable +NetAppDocsError)
            }
        }

        Write-Verbose -Message 'Parsing DNS.XML'
        $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Dns -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'DNS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_DNS' -ReplaceString '^Dns' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing HOSTS.XML'
        $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name DnsHost -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'HOSTS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_HOSTS' -ReplaceString '^Hosts' -ErrorVariable +NetAppDocsError)

        #endregion Network Information

        #region FC/FCP Information

        ##Unused section: FCP-TOPOLOGY-SWITCHES.XML - (1.120+) Not needed for build docs (Fcp.Topology.Info - T_FCP_TOPOLOGY_IE)
        ##Unused section: FCP-TOPOLOGY-ZONED-DEVICES.XML - (1.120+) Not needed for build docs (Fcp.Zone - T_FCP_TOPOLOGY_ZONED_DEVICE)

        Write-Verbose -Message '- Gathering FCP configuration data -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Fcp -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Identifying FC Ports'
        $systemAsupData.Fcp | Add-Member -MemberType NoteProperty -Name Adapter -Value ($ports | Where-Object { $_.Type -match '^FC' })

        Write-Verbose -Message 'Parsing FCP.XML'
        $systemAsupData.Fcp | Add-Member -MemberType NoteProperty -Name Service -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FCP.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FCP' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing FCP-INITIATOR.XML'
        $systemAsupData.Fcp | Add-Member -MemberType NoteProperty -Name Initiator -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FCP-INITIATOR.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FCP_INITIATOR' -ErrorVariable +NetAppDocsError)

        #endregion FC/FCP Information

        #region UCM Information

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.21')) {
            Write-Verbose -Message '- Gathering UCM information -'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name Ucm -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing UNIFIED-CONNECT.XML'
            $systemAsupData.Ucm | Add-Member -MemberType NoteProperty -Name Adapter -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'UNIFIED-CONNECT.XML' -InnerProperty 'T_UCM_DEVICE' -ErrorVariable +NetAppDocsError)
        }

        #endregion UCM Information

        #region PortSet Configuration

        Write-Verbose -Message '- Gathering PortSet information -'

        Write-Verbose -Message 'Parsing PORTSET.XML'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name PortSet -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'PORTSET.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_PORTSET' -ReplaceString '^PortSet' -ErrorVariable +NetAppDocsError)

        #endregion PortSet Configuration

        #region iSCSI Information

        ##Unused Section: ISCSI-INTERFACE.XML - Not needed for build docs (data in network-interface section) (Iscsi.Interface.Info)
        ##Unused Section: ISCSI-SECURITY.XML - Not needed for build docs (Iscsi.Interface.Access)

        Write-Verbose -Message '- Gathering iSCSI configuration data -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Iscsi -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing ISCSI.XML'
        $systemAsupData.Iscsi | Add-Member -MemberType NoteProperty -Name Service -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'ISCSI.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_ISCSI' -ReplaceString '^Iscsi' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing ISCSI-SESSION-BRIEF.XML'
        $systemAsupData.Iscsi | Add-Member -MemberType NoteProperty -Name Initiator -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'ISCSI-SESSION-BRIEF.XML' -InnerProperty 'T_ISCSI_SESSION_BRIEF' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing INITIATOR_SESSION'
        $initiatorSession = $systemAsupData.SourceAsup | Get-AsupObject -ObjectName 'INITIATOR_SESSION' -ReplaceString '^Ini' -ErrorVariable +NetAppDocsError

        foreach ($iscsiInitiator in ($systemAsupData.Iscsi.Initiator | Where-Object { $_ })) {
            $iscsiInitiator | Add-Member -MemberType NoteProperty -Name InitiatorGroupList -Value @(
                $initiatorSession | Where-Object { $_.Tsih -eq $iscsiInitiator.Tsih -and $_.Isid -eq $iscsiInitiator.Isid } | ForEach-Object {
                    [regex]::Replace($_.Igid, "^$( [regex]::Escape($iscsiInitiator.Vserver) )", '')
                } | Where-Object { $_ -ne 'null' } | Sort-Object -Unique
            )

            $iscsiInitiator | Add-Member -MemberType NoteProperty -Name TpgroupTag -Value ($systemAsupData.Net.Interface | Where-Object { $_.VserverId -ceq $iscsiInitiator.Vserver -and $_.LogicalInterface -eq $iscsiInitiator.Tpgroup }).CurrentTpgt
        }

        #endregion iSCSI Information

        #region Security Information

        ##Unused Section: CA-ISSUED-CERT.XML - Not needed for build docs (Security.CertificateIssued - not sure of inner property)
        ##Unused Section: OCSP-STATUS.XML - (1.120+) Not needed for build docs (Security.Ocsp - T_OCSP_STATUS)

        Write-Verbose -Message '- Gathering security information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Security -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing CERTIFICATE.XML'
        $systemAsupData.Security | Add-Member -MemberType NoteProperty -Name Certificate -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CERTIFICATE.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CERT' -ErrorVariable +NetAppDocsError)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.100')) {
            Write-Verbose -Message 'Parsing SECURITY-CONFIG.XML'
            $systemAsupData.Security | Add-Member -MemberType NoteProperty -Name Config -Value (
                Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SECURITY-CONFIG.XML' -InnerProperty 'T_FIPS_ENABLED' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                    Name       = 'Interface'
                    Expression = { 'ssl' }
                } -Unique
            )

            Write-Verbose -Message 'Parsing SECURITY-PROTOCOL.XML'
            $systemAsupData.Security | Add-Member -MemberType NoteProperty -Name Protocol -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SECURITY-PROTOCOL.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_security_protocol' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing SSH.XML'
            $systemAsupData.Security | Add-Member -MemberType NoteProperty -Name Ssh -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SSH.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_SSH' -ErrorVariable +NetAppDocsError)

            if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.120')) {
                Write-Verbose -Message 'Parsing SECURITY-PROTOCOL-SSH.XML'
                $systemAsupData.Security | Add-Member -MemberType NoteProperty -Name ProtocolSsh -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SECURITY-PROTOCOL-SSH.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_security_protocol_ssh' -ErrorVariable +NetAppDocsError)
            }
        }

        #endregion Security Information

        #region Security Session Information

        ##Unused Section: SESSION-APPLICATION-LIMITS.XML - (1.100+) Not needed for build docs (SecuritySession.AppLimit)
        ##Unused Section: SESSION-LOCATION-LIMITS.XML - (1.100+) Not needed for build docs (SecuritySession.LocLimit)
        ##Unused Section: SESSION-REQUEST-LIMITS.XML - (1.100+) Not needed for build docs (SecuritySession.ReqLimit)
        ##Unused Section: SESSION-USER-LIMITS.XML - (1.100+) Not needed for build docs (SecuritySession.UserLimit)
        ##Unused Section: SESSION-VSERVER-LIMITS.XML - (1.100+) Not needed for build docs (SecuritySession.VserverLimit)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.100')) {
            Write-Verbose -Message '- Gathering Security Session information -'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name SecuritySession -Value (
                [pscustomobject]@{
                    Limit = [pscustomobject]@{ }
                }
            )

            Write-Verbose -Message 'Parsing SESSION-APPLICATION-LIMITS.XML'
            $systemAsupData.SecuritySession.Limit | Add-Member -MemberType NoteProperty -Name Application -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SESSION-APPLICATION-LIMITS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_SESSION_APPLICATION_LIMITS' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing SESSION-LOCATION-LIMITS.XML'
            $systemAsupData.SecuritySession.Limit | Add-Member -MemberType NoteProperty -Name Location -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SESSION-LOCATION-LIMITS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_SESSION_LOCATION_LIMITS' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing SESSION-REQUEST-LIMITS.XML'
            $systemAsupData.SecuritySession.Limit | Add-Member -MemberType NoteProperty -Name Request -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SESSION-REQUEST-LIMITS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_SESSION_REQUEST_LIMITS' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing SESSION-USER-LIMITS.XML'
            $systemAsupData.SecuritySession.Limit | Add-Member -MemberType NoteProperty -Name User -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SESSION-USER-LIMITS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_SESSION_USER_LIMITS' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing SESSION-VSERVER-LIMITS.XML'
            $systemAsupData.SecuritySession.Limit | Add-Member -MemberType NoteProperty -Name Vserver -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SESSION-VSERVER-LIMITS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_SESSION_VSERVER_LIMITS' -ErrorVariable +NetAppDocsError)
        }

        #endregion Security Session Information

        #region Snaplock Information

        ##Unused Section: SNAPLOCK-LOG.XML - (1.100+) Not currently needed for build docs (Snaplock.LogConfig)
        ##Unused Section: SNAPLOCK-LOG-FILE.XML - (1.100+) Not currently needed for build docs (Snaplock.LogFiles)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.100')) {
            Write-Verbose -Message '- Gathering Snaplock information -'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name Snaplock -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing SNAPLOCK-VOLUME.XML'
            $systemAsupData.Snaplock | Add-Member -MemberType NoteProperty -Name Volume -Value (
                Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SNAPLOCK-VOLUME.XML' -InnerProperty 'T_VOLUME_SNAPLOCK' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                    Name       = 'SnaplockComplianceClock'
                    Expression = {
                        if (($compClockTime = Convert-DateTimeString -String $_.CompClockTime) -or ($compClockTime = Convert-DateTimeString -String (($_.CompClockTime -split '\s' | Select-Object -Index 0, 1, 2, 3, 5, 6) -join ' ') -Format 'ddd MMM d HH:mm:ss yyyy zzz')) {
                            ConvertTo-UnixTimestamp -DateTime $compClockTime.ToUniversalTime()
                        }
                    }
                }, @{
                    Name       = 'VolumeExpiryTime'
                    Expression = {
                        if (-not [string]::IsNullOrWhiteSpace($_.ExpiryTime)) {
                            if ($_.ExpiryTime -ne 'none' -and (($expiryTime = Convert-DateTimeString -String $_.ExpiryTime) -or ($expiryTime = Convert-DateTimeString -String $_.ExpiryTime -Format 'ddd MMM d HH:mm:ss GMT yyyy'))) {
                                ConvertTo-UnixTimestamp -DateTime $expiryTime.ToUniversalTime()
                            }
                            else {
                                #if we are here, this means the format of the string wasn't "GMT", so just take the string as is for now
                                $_.ExpiryTime
                            }
                        }
                    }
                } -Unique
            )
        }

        #endregion Snaplock Information

        #region NVMe Information

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.140')) {
            Write-Verbose -Message '- Gathering NVMe information'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name Nvme -Value (
                [pscustomobject]@{
                    Subsystem = [pscustomobject]@{ }
                }
            )

            Write-Verbose -Message 'Parsing NVMF-SERVICE.XML'
            $systemAsupData.Nvme | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NVMF-SERVICE.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_NVMF_SERVICE' -ErrorVariable +NetAppDocsError)

            #Write-Verbose -Message 'Parsing NVMF-FEATURES.XML'
            #$systemAsupData.Nvme | Add-Member -MemberType NoteProperty -Name Feature -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NVMF-FEATURES.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_NVMF_FEATURES' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing NVMF-SUBSYSTEM.XML'
            $systemAsupData.Nvme.Subsystem | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NVMF-SUBSYSTEM.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_NVMF_SUBSYSTEM' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing NVMF-SUBSYSTEM-HOST.XML'
            $systemAsupData.Nvme.Subsystem | Add-Member -MemberType NoteProperty -Name Host -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NVMF-SUBSYSTEM-HOST.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_NVMF_SUBSYSTEM_HOST' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing NVMF-SUBSYSTEM-MAP.XML'
            $systemAsupData.Nvme.Subsystem | Add-Member -MemberType NoteProperty -Name Map -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NVMF-SUBSYSTEM-MAP.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_NVMF_SUBSYSTEM_MAP' -ErrorVariable +NetAppDocsError)
        }

        #endregion NVMe Information

        #region VM/VM Services Information

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message '- Gathering VM information -'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name Vm -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing VM-HYPERVISOR-INFO.XML'
            $systemAsupData.Vm | Add-Member -MemberType NoteProperty -Name Hypervisor -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VM-HYPERVISOR-INFO.XML' -InnerProperty 'VM_HYPERVISOR_INFO' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing VM-INSTANCE-INFO.XML'
            $systemAsupData.Vm | Add-Member -MemberType NoteProperty -Name Instance -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VM-INSTANCE-INFO.XML' -InnerProperty 'VM_INSTANCE_INFO' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing STORAGE-DISK-VIRTUAL-MACHINE.XML'
            $systemAsupData.Vm | Add-Member -MemberType NoteProperty -Name SystemDisk -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'STORAGE-DISK-VIRTUAL-MACHINE.XML' -MasterNodeName $masterNodeName -InnerProperty 'VM_DATA_DISKS' -ErrorVariable +NetAppDocsError)
        }

        #endregion VM/VM Services Information

        #region Cluster Switch Information

        Write-Verbose -Message '- Gathering cluster switch information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name ClusterSwitch -Value ([pscustomobject]@{ })

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            $systemAsupData.ClusterSwitch | Add-Member -MemberType NoteProperty -Name Info -Value $(
                if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.30')) {
                    Write-Verbose -Message 'Parsing CSHM-SWITCH-CONFIG.XML'
                    (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CSHM-SWITCH-CONFIG.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_cshm_switch_config' -ErrorVariable +NetAppDocsError | Where-Object { $_.SoftwareVersion -notlike 'NetApp Release*' })
                }
                else {
                    Write-Verbose -Message 'Parsing CSHM-SWITCH-CONFIG.XML'
                    (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CSHM-SWITCH-CONFIG.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CSHM_SWITCH_VIEW' -ErrorVariable +NetAppDocsError | Where-Object { $_.SwVersion -notlike 'NetApp Release*' })
                }
            )

            if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.140')) {
                Write-Verbose -Message 'Parsing CLUSTER-SWITCH-INTERFACE.XML'
                $systemAsupData.ClusterSwitch | Add-Member -MemberType NoteProperty -Name Interface -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CLUSTER-SWITCH-INTERFACE.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_cluster_switch_interface' -ErrorVariable +NetAppDocsError)
            }
        }

        #endregion Cluster Switch Information

        #region FileService Audit Configuration

        Write-Verbose -Message '- Gathering File Service Audit configuration -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name FileServiceAudit -Value ([pscustomobject]@{ })

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Parsing FILE-SERVICES-AUDIT.XML'
            $systemAsupData.FileServiceAudit | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FILE-SERVICES-AUDIT.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VSERVER_AUDIT' -ReplaceString '^(?:Vserveraudit|LogFile)' -ErrorVariable +NetAppDocsError)
        }

        #endregion FileService Audit Configuration

        #region Cluster Peer Configuration

        ##Unused Section: CLUSTERPEER-HEALTH-MONITOR-CACHE.XML - Not needed for build docs (Cluster.Peer.Health)

        Write-Verbose -Message '- Gathering Cluster Peer configuration -'
        $systemAsupData.Cluster | Add-Member -MemberType NoteProperty -Name Peer -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing CLUSTERPEER-ITABLE.XML'
        $systemAsupData.Cluster.Peer | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CLUSTERPEER-ITABLE.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CLUSTER_PEER' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing PEER_CLUSTER_NODE'
        $peerClusterNode = $systemAsupData.SourceAsup | Get-AsupObject -ObjectName 'PEER_CLUSTER_NODE' -ErrorVariable +NetAppDocsError

        if ($peerClusterNode) {
            foreach ($clusterPeer in ($systemAsupData.Cluster.Peer.Info | Where-Object { $_ })) {
                $peerClusterNodeEntry = $peerClusterNode | Where-Object { $_.PeerClusterId -eq $clusterPeer.ClusterUUID } | Select-Object -Property Availability -Unique

                $clusterPeer | Add-Member -MemberType NoteProperty -Name Availability -Value $(
                    switch ($peerClusterNodeEntry.Availability.Count) {
                        '0' {
                            'unavailable'
                        }
                        '1' {
                            if ($peerClusterNodeEntry.Availability -contains 'true') {
                                'available'
                            }
                            else {
                                'unavailable'
                            }
                        }
                        '2' {
                            if ($peerClusterNodeEntry.Availability -contains 'true') {
                                'partial'
                            }
                            else {
                                'unavailable'
                            }
                        }
                    }
                )
            }
        }

        #endregion Cluster Peer Configuration

        #region Vserver Peer Configuration

        ##Unused Section: VSERVER-PEER-TRANSITION.XML - 8.2+ Not needed for build docs (Vserver.Peer.Transition)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Gathing Vserver Peer information -'
            $systemAsupData.Vserver | Add-Member -MemberType NoteProperty -Name Peer -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing VSERVER-PEER.XML'
            $systemAsupData.Vserver.Peer | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VSERVER-PEER.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VSERVERPEER_RDB' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing PEER_CLUSTER'
            $peerCluster = $systemAsupData.SourceAsup | Get-AsupObject -ObjectName 'PEER_CLUSTER' -ErrorVariable +NetAppDocsError

            foreach ($vserverPeer in ($systemAsupData.Vserver.Peer.Info | Where-Object { $_ })) {
                if ($vserverPeer.PeerClusterUUID -eq $systemAsupData.Cluster.Info.ClusterIdentifier) {
                    $clusterPeerName = $systemAsupData.Cluster.Info.ClusterName
                }
                else {
                    $clusterPeerName = ($systemAsupData.Cluster.Peer.Info | Where-Object { $_.ClusterUUID -eq $vserverPeer.PeerClusterUUID } | Select-Object -First 1).Cluster

                    if ([string]::IsNullOrWhiteSpace($clusterPeerName) -and $peerCluster) {
                        $clusterPeerName = ($peerCluster | Where-Object { $_.PeerClusterUuid -eq $vserverPeer.PeerClusterUUID } | Select-Object -First 1).PeerClusterName
                    }
                }

                $vserverPeer | Add-Member -MemberType NoteProperty -Name PeerCluster -Value $clusterPeerName -Force
            }
        }

        #endregion Vserver Peer Configuration

        #region Snapshot Information

        Write-Verbose -Message '- Gathering Snapshot information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Snapshot -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing SNAPSHOT.XML'
        $systemAsupData.Snapshot | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SNAPSHOT.XML' -InnerProperty 'T_SNAPSHOT' -ErrorVariable +NetAppDocsError | Sort-Object -Property * -Unique)

        Write-Verbose -Message 'Parsing SNAPSHOT-POLICY.XML'
        $systemAsupData.Snapshot | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SNAPSHOT-POLICY.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_SNAPSHOT_POLICY' -ReplaceString '^Snapshotpolicy' -ErrorVariable +NetAppDocsError)

        #endregion Snapshot Information

        #region Snapmirror Configuration

        Write-Verbose -Message '- Gathering SnapMirror information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Snapmirror -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing SNAPMIRROR.XML'
        $systemAsupData.Snapmirror | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SNAPMIRROR.XML' -InnerProperty 'T_SNAPMIRROR' -ErrorVariable +NetAppDocsError | Group-Object -Property SourcePath, DestinationPath, Type | ForEach-Object { $_.Group | Select-Object -First 1 })

        Write-Verbose -Message 'Parsing SNAPMIRROR-POLICY.XML'
        $systemAsupData.Snapmirror | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SNAPMIRROR-POLICY.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_SNAPMIRROR_POLICY' -ReplaceString '^Smpolicy' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing SNAPMIRROR-DESTINATION.XML'
        $systemAsupData.Snapmirror | Add-Member -MemberType NoteProperty -Name Destination -Value @(
            foreach ($srcAsup in $systemAsupData.SourceAsup) {
                Get-AsupSection -AsupObject $srcAsup -SectionName 'SNAPMIRROR-DESTINATION.XML' -InnerProperty 'T_SNAPMIRROR_DESTINATION' -ReplaceString '^Smpolicy' -ErrorVariable +NetAppDocsError | Where-Object { $_.SourceVolumeNode -ceq $srcAsup.SystemName }
            }
        )

        #endregion Snapmirror Configuration

        #region CIFS

        ##Unused Section: CIFS-DOMAIN-DISCOVERED-SERVER.XML - Not currently needed for build docs (Cifs.DomainServer)

        Write-Verbose -Message '- Gathering CIFS information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Cifs -Value ([pscustomobject]@{ })
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name Gpo -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing CIFS-SERVER-BYNAME.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name Server -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-SERVER-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_SERVER' -ReplaceString '^CifsServer' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-SERVER-OPTION.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name Option -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-SERVER-OPTION.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_SERVER_OPTIONS' -ReplaceString '^CifsServerOptions' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-SERVER-SECURITY.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name Security -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-SERVER-SECURITY.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_SERVER_SECURITY' -ReplaceString '^CifsServerSecurity' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-SYMLINK-PATHMAPPING.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name Symlink -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-SYMLINK-PATHMAPPING.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_SYMLINK' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-SHARE-BYNAME.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name Share -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-SHARE-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_SHARE' -ReplaceString '^CifsShare' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Capturing CIFS-SHARE-ACL.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name ShareAcl -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-SHARE-ACL.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_SHARE_ACL' -ReplaceString '^CifsShareAcl' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-GPO-DB.XML'
        $systemAsupData.Cifs.Gpo | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-GPO-DB.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_GPO_DB' -ReplaceString '^GpoDb' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-HOMEDIR.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name HomeDirectoryConfig -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-HOMEDIR.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_HOMEDIR' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-HOMEDIR-SEARCHPATH.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name HomeDirectorySearchPath -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-HOMEDIR-SEARCHPATH.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_HOMEDIR_SEARCHPATH' -ReplaceString '^CifsHomedirSearchPath' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-DOMAIN-PREFERRED-DC.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name PreferredDomainController -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-DOMAIN-PREFERRED-DC.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_PREFERRED_DCS' -ReplaceString '^CifsPreferredDc' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-DC-DISCOVERY-MODE.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name DomainServerDiscoveryMode -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-DC-DISCOVERY-MODE.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_DISCOVERY_MODE' -ErrorVariable +NetAppDocsError)

        #endregion CIFS

        #region NFS

        Write-Verbose -Message '- Gathering NFS configuration information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Nfs -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing NFS-SERVERS-BYNAME.XML'
        $systemAsupData.Nfs | Add-Member -MemberType NoteProperty -Name Service -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NFS-SERVERS-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VSERV_NFS' -ReplaceString '^VservNfs' -ErrorVariable +NetAppDocsError)

        #endregion NFS

        #region Lun Configuration

        ##Unused section: STORAGE-DISK-FOREIGN-LUN.XML - (1.30+) Not currently needed for build docs (Lun.Import - T_DISK_FLI)

        Write-Verbose -Message '- Gathering Lun information -'

        $systemAsupData | Add-Member -MemberType NoteProperty -Name Lun -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing LUN.XML'
        $systemAsupData.Lun | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'LUN.XML' -InnerProperty 'T_LUN' -ReplaceString '^Lun' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing LUN-MAPS.XML'
        $systemAsupData.Lun | Add-Member -MemberType NoteProperty -Name Map -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'LUN-MAPS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_LUN_MAP' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing IGROUP.XML'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Igroup -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'IGROUP.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_IGROUP' -ReplaceString '^Igroup' -ErrorVariable +NetAppDocsError)

        #endregion Lun Configuration

        #region Quota Configuration

        ##Unused Section: QUOTA-POLICY.XML - (1.20+) Not needed for build docs (Quota.Policy)
        ##Unused Section: QUOTA-POLICY-UI-TABLE.XML - (-lt 1.20) Not needed for build docs (Quota.Policy)

        Write-Verbose -Message '- Gathering quota configuration -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Quota -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing QUOTA-RULES.XML'
        $systemAsupData.Quota | Add-Member -MemberType NoteProperty -Name Rule -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'QUOTA-RULES.XML' -InnerProperty 'T_QUOTA_RULES' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing VOLUME-QUOTA.XML'
        $systemAsupData.Quota | Add-Member -MemberType NoteProperty -Name Status -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VOLUME-QUOTA.XML' -InnerProperty 'T_QUOTA' -ErrorVariable +NetAppDocsError)

        #endregion Quota Configuration

        #region Name Services Information (NIS/Kerberos/LDAP)

        ##Unused Section: UNIX-GROUPS-FILEONLY.XML - (1.101+) Not needed for build docs
        ##Unused Section: UNIX-USERS-FILEONLY.XML - (1.101+) Not needed for build docs

        Write-Verbose -Message '- Gathering name services information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name NameService -Value ([pscustomobject]@{ })
        $systemAsupData.NameService | Add-Member -MemberType NoteProperty -Name Kerberos -Value ([pscustomobject]@{ })
        $systemAsupData.NameService | Add-Member -MemberType NoteProperty -Name Ldap -Value ([pscustomobject]@{ })
        $systemAsupData.NameService | Add-Member -MemberType NoteProperty -Name Nis -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing NIS-DOMAINS-BYNAME.XML'
        $systemAsupData.NameService.Nis | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NIS-DOMAINS-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_NIS' -ReplaceString '^Nis' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing KERBEROS-IFCONFIG-BYNAME.XML'
        $systemAsupData.NameService.Kerberos | Add-Member -MemberType NoteProperty -Name Config -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'KERBEROS-IFCONFIG-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_KERBEROS_CONFIG' -ReplaceString '^Kerberosconfig' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing KERBREALM.XML'
        $systemAsupData.NameService.Kerberos | Add-Member -MemberType NoteProperty -Name Realm -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'KERBREALM.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_KERBEROS_REALM' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing LDAP-CONFIG.XML'
        $systemAsupData.NameService.Ldap | Add-Member -MemberType NoteProperty -Name Config -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'LDAP-CONFIG.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_LDAP' -ReplaceString '^Ldapconfig' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing LDAP-CLIENT-CONFIG.XML'
        $systemAsupData.NameService.Ldap | Add-Member -MemberType NoteProperty -Name Client -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'LDAP-CLIENT-CONFIG.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_LDAP_CLIENT' -ReplaceString '^Ldapclientconfig' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing LDAP-CLIENT-SCHEMA.XML'
        $systemAsupData.NameService.Ldap | Add-Member -MemberType NoteProperty -Name ClientSchema -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'LDAP-CLIENT-SCHEMA.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_LDAP_CLIENT_SCHEMA' -ReplaceString '^Ldapclientschema' -ErrorVariable +NetAppDocsError)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message 'Parsing NSSWITCH-BYNAME.XML'
            $systemAsupData.NameService | Add-Member -MemberType NoteProperty -Name NsSwitch -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NSSWITCH-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_NSSWITCH' -ReplaceString '^Nsswitch' -ErrorVariable +NetAppDocsError)
        }

        #endregion Name Services Information (NIS/Kerberos/LDAP)

        #region Job Information

        Write-Verbose -Message '- Gathering job information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Job -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing JM-CLUSTER-BACKUP.XML'
        $systemAsupData.Job | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'JM-CLUSTER-BACKUP.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_JOBS_TABLE' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing JM-SCHED.XML'
        $systemAsupData.Job | Add-Member -MemberType NoteProperty -Name Schedule -Value (
            Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'JM-SCHED.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_JOB_SCHEDULE' -ReplaceString '^JobSchedule' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                Name       = 'JobScheduleInfo'
                Expression = { ConvertFrom-JobScheduleDescription -JobScheduleDescription $_.Description }
            }
        )

        #endregion Job Information

        #region Disk Qual Information

        Write-Verbose -Message '- Gathering DQP information -'

        Write-Verbose -Message 'Parsing DQP_DETAILS'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name DiskQualPackage -Value @(
            foreach ($srcAsup in $systemAsupData.SourceAsup) {
                Get-AsupObject -AsupObject $srcAsup -ObjectName 'DQP_DETAILS' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                    Name       = 'Node'
                    Expression = { $srcAsup.SystemName }
                }
            }
        )

        #endregion Disk Qual Information

        #region Debug Information

        Write-Verbose -Message '- Gathering debug information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Debug -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing VREPORT.XML'
        $systemAsupData.Debug | Add-Member -MemberType NoteProperty -Name Vreport -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VREPORT.XML' -InnerProperty 'T_VREPORT_SHOW' -ErrorVariable +NetAppDocsError | Sort-Object -Property * -Unique)

        #endregion Debug Information

        #region AIQ Endpoints

        #Write-Verbose -Message 'Capturing AIQ CapacityForecast endpoint'
        #$systemAsupData | Add-Member -MemberType NoteProperty -Name CapacityForecast -Value (Get-AiqRestEndpoint -PathAndQuery "/v1/capacity2/forecast?platformType=ONTAP&serialNumbers=$( $systemAsupData.Node.SerialNo -join ',' )")

        #endregion AIQ Endpoints

        #region ARS Information

        Write-Verbose -Message '- Gathering ARS information -'

        $systemAsupData.SourceAsup | ForEach-Object {
            $_ | Add-Member -MemberType NoteProperty -Name 'AtRiskSignatures' -Value ((Get-AsupRestData -Uri "$( $Script:AsupArsRestUri )/service_tier/$( $Script:AiqServiceTierList )/sys_serial_no/$( $_.SystemSerialNumber )/system_id/$( $_.SystemId )" -ErrorVariable +NetAppDocsError).results.system.risks.risk)
        }

        #endregion ARS Information

        Write-Verbose -Message '- Identifying ASUP Generation Time -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name ReferenceDT -Value ($systemAsupData.SourceAsup | Select-Object -First 1).AsupGenerationDate.ToUniversalTime()

        Write-Verbose -Message '- Identifying SystemName -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name SystemName -Value $systemAsupData.Cluster.Info.ClusterName

        Write-Verbose -Message '- Identifying any errors during processing -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name DiscoveryError -Value @(
            foreach ($err in ($NetAppDocsError | Where-Object { $_ -and $_.Exception.Message -notlike 'Exception calling "LoadXml"*' -and $_.Exception.Message -notlike 'Entity not supported*' })) {
                $nodes, $section = $err.CategoryInfo.TargetName -split '\s*:\s*'

                foreach ($node in ($nodes -split '\s+')) {
                    $err.CategoryInfo | Select-Object -Property Category, Reason, @{
                        Name       = 'Activity'
                        Expression = { $_.Activity }
                    }, @{
                        Name       = 'TargetName'
                        Expression = { $node }
                    }, @{
                        Name       = 'TargetSection'
                        Expression = { $section }
                    }, @{
                        Name       = 'TargetType'
                        Expression = { 'NcController.Asup' }
                    }, @{
                        Name       = 'ManifestStatus'
                        Expression = {
                            (($systemAsupData.SourceAsup | Where-Object { $_.SystemName -eq $node }).Manifest | Where-Object {
                                $_.BodyFile -eq $section -or
                                $_.BodyFile -eq "$( $section ).txt" -or
                                $_.BodyFile -eq "$( $section -replace '-', '_' )"
                            }
                        ).Status }
                    }, @{
                        Name       = 'Message'
                        Expression = {
                            if ($err.Exception.Message -eq 'No systems available for the input data:[]') {
                                'ASUP section is missing or empty for this system'
                            }
                            else {
                                $err.Exception.Message -split '\s*[:]\s*' | Select-Object -First 1
                            }
                        }
                    }
                }
            }
        )

        if (($systemAsupData.DiscoveryError | Where-Object { $_.Reason -ne 'NO_DATA_FOUND' -or ($_.ManifestStatus -and $_.ManifestStatus -notin @( 'content-empty', 'content-not-collected-precheck' )) } | Measure-Object).Count -gt 0) {
            $msg = New-Object -TypeName System.Text.StringBuilder

            [void]$msg.AppendLine("Errors were encountered while collecting data from ASUPID: $( $systemAsupData.SourceAsup.AsupId )")
            [void]$msg.AppendLine('-------------------------------------------------------------------------------------')
            [void]$msg.AppendLine('Please review the output for accuracy and, if necessary, correct any error conditions')
            [void]$msg.AppendLine('then run these scripts again.')
            [void]$msg.AppendLine('-------------------------------------------------------------------------------------')
            [void]$msg.AppendLine()

            Write-Warning -Message $msg.ToString()
        }

        $systemAsupData
    }
    catch {
        $Global:Error.RemoveAt(0)

        $msg = New-Object -TypeName System.Text.StringBuilder

        if ($_.CategoryInfo.Category -eq 'ResourceUnavailable') {
            [void]$msg.AppendLine('The connection to the ASUP REST server has failed. Please check your connection and try the capture again.')
        }
        elseif ($_.CategoryInfo.Category -eq 'ReadError') {
            [void]$msg.AppendLine('An unexpected error occurred. Processing halted for this system.')
            [void]$msg.AppendLine('---------------------------------------------------------------------------------------------')
            [void]$msg.AppendLine('This error is not expected, but we do know about it. Unfortunately, this is an issue with the')
            [void]$msg.AppendLine('ASUP data warehouse and the only workaround at this time is to try a different set of ASUPs.')
            [void]$msg.AppendLine('---------------------------------------------------------------------------------------------')
            [void]$msg.AppendLine($( $_.Exception.GetBaseException().Message ))
            [void]$msg.AppendLine("Location: $( $_.ScriptStackTrace )")
            [void]$msg.AppendLine()
        }
        else {
            [void]$msg.AppendLine('An unexpected error occurred. Processing halted.')
            [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
            [void]$msg.AppendLine('As this error is not expected, please consider contacting the support staff via e-mail or by')
            [void]$msg.AppendLine('starting a thread on the community site.')
            [void]$msg.AppendLine('            E-mail: ng-NetAppDocs-support@netapp.com')
            [void]$msg.AppendLine('    Community site: https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs')
            [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
            [void]$msg.AppendLine($( $_.Exception.GetBaseException().Message ))
            [void]$msg.AppendLine("Location: $( $_.ScriptStackTrace )")
            [void]$msg.AppendLine()
        }

        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $_.CategoryInfo.TargetName)))
    }
}

# SIG # Begin signature block
# MIIq0AYJKoZIhvcNAQcCoIIqwTCCKr0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBU5o1wkQfuK93J
# bb/T73E7dAwPXh/QxqUikF8FIxsYo6CCJGgwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggWNMIIEdaADAgECAhAOmxiO
# +dAt5+/bUOIIQBhaMA0GCSqGSIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYD
# VQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAi
# BgNVBAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAw
# MDBaFw0zMTExMDkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
# Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERp
# Z2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBAL/mkHNo3rvkXUo8MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsb
# hA3EMB/zG6Q4FutWxpdtHauyefLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iT
# cMKyunWZanMylNEQRBAu34LzB4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGb
# NOsFxl7sWxq868nPzaw0QF+xembud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclP
# XuU15zHL2pNe3I6PgNq2kZhAkHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCr
# VYJBMtfbBHMqbpEBfCFM1LyuGwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFP
# ObURWBf3JFxGj2T3wWmIdph2PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTv
# kpI6nj3cAORFJYm2mkQZK37AlLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWM
# cCxBYKqxYxhElRp2Yn72gLD76GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls
# 5Q5SUUd0viastkF13nqsX40/ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBR
# a2+xq4aLT8LWRV+dIPyhHsXAj6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6
# MIIBNjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qY
# rhwPTzAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8E
# BAMCAYYweQYIKwYBBQUHAQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5k
# aWdpY2VydC5jb20wQwYIKwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDig
# NoY0aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNybDARBgNVHSAECjAIMAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCg
# v0NcVec4X6CjdBs9thbX979XB72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQT
# SnovLbc47/T/gLn4offyct4kvFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh
# 65ZyoUi0mcudT6cGAxN3J0TU53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSw
# uKFWjuyk1T3osdz9HNj0d1pcVIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAO
# QGPFmCLBsln1VWvPJ6tsds5vIy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjD
# TZ9ztwGpn1eqXijiuZQwggXMMIIENKADAgECAhAg429sPxgagUb53pPffJfkMA0G
# CSqGSIb3DQEBDAUAMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBS
# MzYwHhcNMjEwOTA5MDAwMDAwWhcNMjMwOTA5MjM1OTU5WjBQMQswCQYDVQQGEwJV
# UzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UECgwMTmV0QXBwLCBJbmMuMRUw
# EwYDVQQDDAxOZXRBcHAsIEluYy4wggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGK
# AoIBgQC4kYYj/oViZD9pN03hrqFjtwOz1Gx4eDWVV8IYUYEr2qpLeNYvWz26B/+E
# mYLboAGVpkYg0Wske3hjRpooZlMORm1S4+2C2NoQCstZ+kmNTevbXHJe/w1VNJrm
# fKjpXOJEfx8GUGDqCMR30dChVpsdBEavrM7T0hnfJLv18i19SGv3a+nhvB3iOjLP
# SEg8+DFkjD5jTI6fQ83DBEiVR4UEYq7yyH578ZEylhsRfJmK+uIREycEUk/NpFTw
# g6/7lxh/WYabCrZzqI4Ep3QataRKaAlxt3BiOlZkX4WGX3DYvSYltWOM2UlCcGpP
# m/1/LN8Xuvf+YO6H472QecHe59XtXbHSMKvqDuOEJu8Wk2LFMNK732o3Fc5QIHie
# 6JtNsgNYf+Vsm5EMBD1ZHNE+C3zlmQbbWIU5uEU1nhiodBPKsu0SkS6pARyKBO05
# DSc2gWWv0aukKi04ZAn+hAdSl5s1dgmU5shBvKHqe15K9ZRN4FFO/CgHE0BoGYQS
# UQVKwa0CAwEAAaOCAZwwggGYMB8GA1UdIwQYMBaAFA8qyyCHKLjsb0iuK1SmKaoX
# pM0MMB0GA1UdDgQWBBQuH643KcBMmb/Q6IZt+H9IrnXFwDAOBgNVHQ8BAf8EBAMC
# B4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDAzARBglghkgBhvhC
# AQEEBAMCBBAwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwIwJTAjBggrBgEFBQcC
# ARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQBMEkGA1UdHwRCMEAw
# PqA8oDqGOGh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVT
# aWduaW5nQ0FSMzYuY3JsMHkGCCsGAQUFBwEBBG0wazBEBggrBgEFBQcwAoY4aHR0
# cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQVIz
# Ni5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqG
# SIb3DQEBDAUAA4IBgQCOoGdXjP8Sif0h3ZvoDsIVfrJvQUdP9pZExRJGGj/Te6ML
# XyqHNc/G2WklNv+BC4ryNJ4zp0oneHOub55ejrYPfhgP9oFWS8oOPJbrp3rAtINa
# OeKRC88WUYZSKYAxSX6kubEjnQD6cmazXjxVN6P8+6q9smeqF3xI+++c+ekMw3Xv
# 4EWRYXqN0/srfFp1WpoluwSQcZMqj9haSX0bgO6AAWH2AnVJMfXTIKz/0FQ/RW0y
# Ed5QYQqYInhA7IUz9dBziNpJlwONeF5j7xzgfYDY63WU6WrgJnGjYkQCOOsUx74j
# gUiMRe9zV48GS8Vxz22c/TQs0x4/1GmOSJvUHEk3GseBmB3v+yEhh/D6zWOfYP4X
# D/9b91CxmugTuepuaJSeRg+qUm3KarKsOCUF/CLqUrNh/JwKrWD1cghRaYLvMucs
# ScksHHe7ZDvb2OtvxWXjPk1d1NKvEwFJSS6hIVIbug9x28AJqOoP5Pn9wZvJSzvW
# uQJGCLscdGyKefdCo30wggYaMIIEAqADAgECAhBiHW0MUgGeO5B5FSCJIRwKMA0G
# CSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290
# IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFQxCzAJBgNVBAYT
# AkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28g
# UHVibGljIENvZGUgU2lnbmluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEBAQUAA4IB
# jwAwggGKAoIBgQCbK51T+jU/jmAGQ2rAz/V/9shTUxjIztNsfvxYB5UXeWUzCxEe
# AEZGbEN4QMgCsJLZUKhWThj/yPqy0iSZhXkZ6Pg2A2NVDgFigOMYzB2OKhdqfWGV
# oYW3haT29PSTahYkwmMv0b/83nbeECbiMXhSOtbam+/36F09fy1tsB8je/RV0mIk
# 8XL/tfCK6cPuYHE215wzrK0h1SWHTxPbPuYkRdkP05ZwmRmTnAO5/arnY83jeNzh
# P06ShdnRqtZlV59+8yv+KIhE5ILMqgOZYAENHNX9SJDm+qxp4VqpB3MV/h53yl41
# aHU5pledi9lCBbH9JeIkNFICiVHNkRmq4TpxtwfvjsUedyz8rNyfQJy/aOs5b4s+
# ac7IH60B+Ja7TVM+EKv1WuTGwcLmoU3FpOFMbmPj8pz44MPZ1f9+YEQIQty/NQd/
# 2yGgW+ufflcZ/ZE9o1M7a5Jnqf2i2/uMSWymR8r2oQBMdlyh2n5HirY4jKnFH/9g
# Rvd+QOfdRrJZb1sCAwEAAaOCAWQwggFgMB8GA1UdIwQYMBaAFDLrkpr/NZZILyhA
# QnAgNpFcF4XmMB0GA1UdDgQWBBQPKssghyi47G9IritUpimqF6TNDDAOBgNVHQ8B
# Af8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcD
# AzAbBgNVHSAEFDASMAYGBFUdIAAwCAYGZ4EMAQQBMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Um9vdFI0Ni5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ1Jvb3RSNDYu
# cDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQwFAAOCAgEABv+C4XdjNm57oRUgmxP/BP6YdURhw1aVcdGRP4Wh60BAscjW
# 4HL9hcpkOTz5jUug2oeunbYAowbFC2AKK+cMcXIBD0ZdOaWTsyNyBBsMLHqafvIh
# rCymlaS98+QpoBCyKppP0OcxYEdU0hpsaqBBIZOtBajjcw5+w/KeFvPYfLF/ldYp
# mlG+vd0xqlqd099iChnyIMvY5HexjO2AmtsbpVn0OhNcWbWDRF/3sBp6fWXhz7Dc
# ML4iTAWS+MVXeNLj1lJziVKEoroGs9Mlizg0bUMbOalOhOfCipnx8CaLZeVme5yE
# Lg09Jlo8BMe80jO37PU8ejfkP9/uPak7VLwELKxAMcJszkyeiaerlphwoKx1uHRz
# NyE6bxuSKcutisqmKL5OTunAvtONEoteSiabkPVSZ2z76mKnzAfZxCl/3dq3dUNw
# 4rg3sTCggkHSRqTqlLMS7gjrhTqBmzu1L90Y1KWN/Y5JKdGvspbOrTfOXyXvmPL6
# E52z1NZJ6ctuMFBQZH3pwWvqURR8AgQdULUvrxjUYbHHj95Ejza63zdrEcxWLDX6
# xWls/GDnVNueKjWUH3fTv1Y8Wdho698YADR7TNx8X8z2Bev6SivBBOHY+uqiirZt
# g0y9ShQoPzmCcn63Syatatvx157YK9hlcPmVoa1oDE5/L9Uo2bC5a4CH2Rwwggau
# MIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJ
# BgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5k
# aWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAe
# Fw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcw
# FQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3Rl
# ZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9Ge
# TKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0
# hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZl
# jZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAsh
# aG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVY
# TXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1
# biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCir
# c0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+
# DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA
# +bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42Pg
# puE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzS
# M7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU
# uhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6
# mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsG
# AQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29t
# MEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNl
# cnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3Js
# My5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAE
# GTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1Z
# jsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8d
# B+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVp
# P0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp8
# 76i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2
# nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3
# ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQ
# txMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc
# 4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+Y
# AN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZ
# vAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQr
# H4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIGwDCCBKigAwIBAgIQDE1p
# ckuU+jwqSj0pB4A9WjANBgkqhkiG9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4XDTIyMDkyMTAwMDAw
# MFoXDTMzMTEyMTIzNTk1OVowRjELMAkGA1UEBhMCVVMxETAPBgNVBAoTCERpZ2lD
# ZXJ0MSQwIgYDVQQDExtEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMiAtIDIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDP7KUmOsap8mu7jcENmtuh6BSFdDMa
# JqzQHFUeHjZtvJJVDGH0nQl3PRWWCC9rZKT9BoMW15GSOBwxApb7crGXOlWvM+xh
# iummKNuQY1y9iVPgOi2Mh0KuJqTku3h4uXoW4VbGwLpkU7sqFudQSLuIaQyIxvG+
# 4C99O7HKU41Agx7ny3JJKB5MgB6FVueF7fJhvKo6B332q27lZt3iXPUv7Y3UTZWE
# aOOAy2p50dIQkUYp6z4m8rSMzUy5Zsi7qlA4DeWMlF0ZWr/1e0BubxaompyVR4aF
# eT4MXmaMGgokvpyq0py2909ueMQoP6McD1AGN7oI2TWmtR7aeFgdOej4TJEQln5N
# 4d3CraV++C0bH+wrRhijGfY59/XBT3EuiQMRoku7mL/6T+R7Nu8GRORV/zbq5Xwx
# 5/PCUsTmFntafqUlc9vAapkhLWPlWfVNL5AfJ7fSqxTlOGaHUQhr+1NDOdBk+lbP
# 4PQK5hRtZHi7mP2Uw3Mh8y/CLiDXgazT8QfU4b3ZXUtuMZQpi+ZBpGWUwFjl5S4p
# kKa3YWT62SBsGFFguqaBDwklU/G/O+mrBw5qBzliGcnWhX8T2Y15z2LF7OF7ucxn
# EweawXjtxojIsG4yeccLWYONxu71LHx7jstkifGxxLjnU15fVdJ9GSlZA076XepF
# cxyEftfO4tQ6dwIDAQABo4IBizCCAYcwDgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB
# /wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwIAYDVR0gBBkwFzAIBgZngQwB
# BAIwCwYJYIZIAYb9bAcBMB8GA1UdIwQYMBaAFLoW2W1NhS9zKXaaL3WMaiCPnshv
# MB0GA1UdDgQWBBRiit7QYfyPMRTtlwvNPSqUFN9SnDBaBgNVHR8EUzBRME+gTaBL
# hklodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0
# MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3JsMIGQBggrBgEFBQcBAQSBgzCBgDAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFgGCCsGAQUFBzAC
# hkxodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRS
# U0E0MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3J0MA0GCSqGSIb3DQEBCwUAA4IC
# AQBVqioa80bzeFc3MPx140/WhSPx/PmVOZsl5vdyipjDd9Rk/BX7NsJJUSx4iGNV
# CUY5APxp1MqbKfujP8DJAJsTHbCYidx48s18hc1Tna9i4mFmoxQqRYdKmEIrUPwb
# tZ4IMAn65C3XCYl5+QnmiM59G7hqopvBU2AJ6KO4ndetHxy47JhB8PYOgPvk/9+d
# EKfrALpfSo8aOlK06r8JSRU1NlmaD1TSsht/fl4JrXZUinRtytIFZyt26/+YsiaV
# OBmIRBTlClmia+ciPkQh0j8cwJvtfEiy2JIMkU88ZpSvXQJT657inuTTH4YBZJwA
# wuladHUNPeF5iL8cAZfJGSOA1zZaX5YWsWMMxkZAO85dNdRZPkOaGK7DycvD+5sT
# X2q1x+DzBcNZ3ydiK95ByVO5/zQQZ/YmMph7/lxClIGUgp2sCovGSxVK05iQRWAz
# gOAj3vgDpPZFR+XOuANCR+hBNnF3rf2i6Jd0Ti7aHh2MWsgemtXC8MYiqE+bvdgc
# mlHEL5r2X6cnl7qWLoVXwGDneFZ/au/ClZpLEQLIgpzJGgV8unG1TnqZbPTontRa
# mMifv427GFxD9dAq6OJi7ngE273R+1sKqHB+8JeEeOMIA11HLGOoJTiXAdI/Otrl
# 5fbmm9x+LMz/F0xNAKLY1gEOuIvu5uByVYksJxlh9ncBjDGCBb4wggW6AgEBMGgw
# VDELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDErMCkGA1UE
# AxMiU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIFIzNgIQIONvbD8YGoFG
# +d6T33yX5DANBglghkgBZQMEAgEFAKCBhDAYBgorBgEEAYI3AgEMMQowCKACgACh
# AoAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAM
# BgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCBDcUScIPd28hpDS9f0d+jMch/9
# iiPCNjLiKvZyZ6Tx/jANBgkqhkiG9w0BAQEFAASCAYBAUXXheb3UWKzUthLwWc05
# lGsx/jvb7N8jPnhDFO6Ixm5y8oPbdqtJJD+yj89A4GFDsEsXBmXo0riaQ+XcHw5H
# DD8YswLpkmaMtykPmkcBywFXxNNpiTwz0rya0hP68gCf99w0USmBnuPo4VK/f2N3
# fk4/eHDNQ/plNpcF8MHlsPz+99u+3NNbxCdmrEJAbhJRcENNwJNHf86Z9MjfNcy7
# gYYZvwa9vsIMdQrLocIR20BzeRr20gRwWbm0kPdVU5LmkyEDfIEFeAnggYLRGvgW
# wytbr4uEO+AY1hdk6GDVXLqQx73VCwydNvR7lBAyItUdo4CD31j45R67//ZKqpSr
# xfn+BDpBSZ/8ztOYhhBuXgKim77ktLZfBrBxzwIdYZVqchPmVwqk6ps4ipG8bpDF
# Mli1a81bdCnIyCOlyVaU2yDUzGY0UiPKTwED+0mcX2qr2BaBI730ZzNjfhIY4pwe
# 9+6f5xkFsadbXPIwe6NJBZN6ySOY1c/EYrHRM0CmsnehggMgMIIDHAYJKoZIhvcN
# AQkGMYIDDTCCAwkCAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
# cnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
# SEEyNTYgVGltZVN0YW1waW5nIENBAhAMTWlyS5T6PCpKPSkHgD1aMA0GCWCGSAFl
# AwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUx
# DxcNMjMwNDAyMTUyODI3WjAvBgkqhkiG9w0BCQQxIgQgSWq2R6Fd69r15ECpdNE5
# /RsJl7uL6yrwnn8T0DzA5HAwDQYJKoZIhvcNAQEBBQAEggIAqYxTmmrEObtkatka
# TcRsxbDOmvdgm2EXUND8qu2UWBcQ8TYGFf/eQfXUQOdMBfLVUSfc/TL7RRCwcaNd
# 6OWA6fCUWgu07+9Wc+5og93wS96Wfl/+fDaqXbFHcYiIPIpEJkCr0Yaqi1Q2XC2p
# DkBjw+AjA3OP3khWzv3m5muiCqfNt7dS4bYPln87BfFUjeIIsMLofN9xUS59Nlrx
# 6eI+P9BtDGRprtGihIM5sJnnqjdMUAAgGSwKF4zcy4Nri0dUloYIepodKFld8JZF
# ApQA46NwSEMxjasHkO1X9WKEyoxCP1IofdGb/UZ8bfY+8zOgxpE3wbSuRbmVO/rm
# oe5oiDcMhdtTlZ2aaMUzADJTuwSZCXc8Mw7ZOv3bBstcuBGO7l4RG5rIYXOxLJlg
# WzBEX//NFSSJGHfJTpQMyIJsWgWe2F7+3RnIZ8CaEIyEFaP4cwQfIOjWDmF1LFic
# uqM0OvFIt3NsOQ60HPM+3bGM8+iN+B8ew7PALgMoSMFNHaRTupvBziOnTsN1P5ce
# IhBpuWXHPrFraXJIitYEqVFuYwsEZQ9YenI9eSVg8EMI4udsBdwmKL8HwnjCnC2e
# o3dXU4nkmfwHNQO/GTFDeyTJ/TZUXd/qbWg/S+JBru/iinsyd5Da4Hc+uRtuiMeS
# JJ75jhyK2PdPyobnxMFqOrDGb00=
# SIG # End signature block
