function Get-NtapClusterData {
    <#
        .SYNOPSIS
            Collects data from ONTAP storage systems.

        .DESCRIPTION
            Uses the NetApp ONTAP PowerShell Toolkit to gather as much configuration information as possible. This
            data can be used to automate build docs and health checks. It can also be used for historical
            configuration information.

        .EXAMPLE
            PS> Get-NtapClusterData -Name cluster1 | Export-Clixml -Path 'D:\Output\cluster1.xml'

            Collects the configuration information off the cluster 'cluster1' and saves the results to a .xml file for further
            processing.

        .EXAMPLE
            PS> Get-NtapClusterData -Name cluster1 -Credential (Get-Credential) | Export-Clixml -Path 'D:\Output\cluster1.xml'

            Collects the configuration information off the cluster 'cluster1' using alternate credentials. The Add-NcCredential
            cmdlet may be used to cache credentials for the cluster to be accessed.

        .EXAMPLE
            PS> Get-NtapClusterData -Name cluster1 | Format-NtapClusterData | Out-NtapDocument -WordFile 'D:\Output\cluster1.docx' -ExcelFile 'D:\Output\cluster1.xlsx'

            Collects the configuration information from the cluster 'cluster1', formats the data and creates the specified
            Word and Excel documents.

        .LINK
            https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs

        .LINK
            Format-NtapClusterData

        .INPUTS
            NetApp.Ontapi.AbstractController

        .INPUTS
            System.String

        .OUTPUTS
            System.Management.Automation.PSCustomObject
    #>

    [CmdletBinding(DefaultParameterSetName = 'Name')]
    [OutputType([System.Management.Automation.PSCustomObject])]
    param (
        # The system name or IP address of the cluster admin SVM to gather the data from.
        [Parameter(Mandatory, Position = 0, ParameterSetName = 'Name', ValueFromPipeline, ValueFromPipelineByPropertyName)]
        [ValidateNotNullOrEmpty()]
        [Alias('ClusterName', 'SystemName')]
        [string[]]$Name,

        # A NetApp ONTAP PowerShell Toolkit controller object (NcController).
        [Parameter(Mandatory, Position = 0, ParameterSetName = 'ControllerObject', ValueFromPipeline)]
        [ValidateNotNullOrEmpty()]
        [NetApp.Ontapi.AbstractController[]]$AbstractController,

        # Specifies explicit credentials to connect to the storage system. (Alias: Cred)
        [Parameter(ParameterSetName = 'Name')]
        [ValidateNotNullOrEmpty()]
        [Alias('Cred')]
        [pscredential]
        [System.Management.Automation.Credential()]
        $Credential = [pscredential]::Empty,

        # The port on which to connect to the storage system. If not set, it defaults to 80 for HTTP and 443 for HTTPS.
        [Parameter(ParameterSetName = 'Name')]
        [ValidateRange([uint16]::MinValue, [uint16]::MaxValue)]
        [uint16]$Port,

        # Connection timeout in milliseconds. Defaults to 60000 milliseconds.
        [Parameter()]
        [ValidateRange(0, [int]::MaxValue)]
        [int]$Timeout,

        # Forces the use of HTTPS instead of HTTP for the connection to the storage system.
        [Parameter(ParameterSetName = 'Name')]
        [Alias('ForceHTTPS')]
        [switch]$ForceSecure,

        # During the initial connection to the storage system, this switch disables a check that is performed to ensure that the LIF
        # being used for the connection is a cluster mgmt or a node mgmt port. This should only be used for a very specific scenario
        # where port forwarding is configured on the host connecting to the storage system.
        [Parameter()]
        [switch]$DisableLifCheck,

        # Disables the ping test prior to connecting to the storage system (useful in the event ICMP is being blocked). (Deprecated)
        [Parameter()]
        [ValidateScript({
                Write-Warning -Message '-DisablePingCheck has been deprecated (no longer performing an ICMP ping to test connectivity).'

                $true
            })]
        [switch]$DisablePingCheck,

        # Disables the manual generation of an ASUP for the storage system. (Deprecated)
        [Parameter()]
        [ValidateScript({
                Write-Warning -Message 'The -DisableAsupInvoke switch has been deprecated. The function no longer triggers ASUPs by default (use the -InvokeAsup switch to explicitly trigger ASUPs)'

                $true
            })]
        [switch]$DisableAsupInvoke,

        # Invokes a user generated ASUP for the storage system.
        [Parameter()]
        [switch]$InvokeAsup
    )

    process {
        Write-Verbose -Message "Initializing function: $( $PSCmdlet.MyInvocation.MyCommand.Name ) ($( $PSCmdlet.MyInvocation.MyCommand.ModuleName ) v$( (Get-NtapDocsVersion -AsVersion).ToString() ))"

        switch -Exact ($PSCmdlet.ParameterSetName) {
            'Name' {
                foreach ($cluster in ($Name | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })) {
                    try {
                        Write-Verbose -Message "Validating connectivity to system: $cluster"

                        $ip = [System.Net.Dns]::GetHostAddresses($cluster)
                        $tcpClient = New-Object -TypeName System.Net.Sockets.TCPClient

                        if ($PSCmdlet.MyInvocation.BoundParameters.ContainsKey('Port')) {
                            $tcpConnect = $tcpClient.BeginConnect($ip, $Port, $null, $null)

                            if (-not $tcpConnect.AsyncWaitHandle.WaitOne(7500, $false)) {
                                $tcpClient.Close()

                                throw 'Connection failed (ResourceUnavailable).'
                            }
                        }
                        elseif ($ForceSecure.IsPresent) {
                            $tcpConnect = $tcpClient.BeginConnect($ip, 443, $null, $null)

                            if (-not $tcpConnect.AsyncWaitHandle.WaitOne(7500, $false)) {
                                $tcpClient.Close()

                                throw 'Connection failed (ResourceUnavailable).'
                            }
                        }
                        else {
                            $tcpConnect = $tcpClient.BeginConnect($ip, 443, $null, $null)

                            if (-not $tcpConnect.AsyncWaitHandle.WaitOne(7500, $false)) {
                                $tcpClient.Close()

                                $tcpClient = New-Object -TypeName System.Net.Sockets.TCPClient
                                $tcpConnect = $tcpClient.BeginConnect($ip, 80, $null, $null)

                                if (-not $tcpConnect.AsyncWaitHandle.WaitOne(7500, $false)) {
                                    $tcpClient.Close()

                                    throw 'Connection failed (ResourceUnavailable).'
                                }
                            }
                        }

                        [void]$tcpClient.EndConnect($tcpConnect)
                        $tcpClient.Close()

                        $params = @{
                            Name        = $cluster
                            ForceSecure = $ForceSecure.IsPresent
                        }

                        if ($Credential -ne [pscredential]::Empty) {
                            $params.Add('Credential', $Credential)
                        }

                        if ($PSCmdlet.MyInvocation.BoundParameters.ContainsKey('Port')) {
                            $params.Add('Port', $Port)
                        }

                        if ($PSCmdlet.MyInvocation.BoundParameters.ContainsKey('Timeout')) {
                            $params.Add('Timeout', $Timeout)
                        }

                        $controller = Connect-NtapStorageSystem @params -NcController -ErrorAction Stop

                        if ($controller.GetType().FullName -ne 'NetApp.Ontapi.Filer.C.NcController') {
                            throw 'Incorrect controller type specified for the Get-NtapClusterData function.'
                        }

                        if (-not (Test-NtapStorageSystemConnection -Controller $controller -DisableLifCheck:$DisableLifCheck.IsPresent -ErrorAction Stop)) {
                            throw 'Unspecified error.'
                        }

                        Write-Verbose -Message "Valid connection established to system: $cluster"
                    }
                    catch {
                        $Global:Error.RemoveAt(0)

                        Write-Error -ErrorRecord (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("Unable to gather data from system. Reason: $( $_.Exception.GetBaseException().Message )", $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $cluster))

                        continue
                    }

                    Write-Verbose -Message "Gathering data from system: $( $controller.Name )"

                    try {
                        $functionStartTime = [datetime]::Now

                        $ntapData = Get-ClusterData -Controller $controller

                        $ntapData

                        if ($ntapData -and $InvokeAsup.IsPresent) {
                            foreach ($node in ($ntapData.Node | Where-Object { $_ })) {
                                try {
                                    Write-Verbose -Message "Invoking ASUP on node: $( $node.Node )"
                                    [void](Invoke-NcAutoSupport -Node $node.Node -Message "NetAppDocs_$( $ntapData.ReferenceDT.ToString('u') )" -Type all -Controller $controller)
                                }
                                catch {
                                    $Global:Error.RemoveAt(0)

                                    Show-ErrorMessage -Message "Unable to invoke ASUP on node: $( $node.Node )"
                                }
                            }
                        }

                        $statsParams = @{
                            FunctionPSCallStack = Get-PSCallStack
                            TargetDetails       = [pscustomobject]@{
                                SystemUid     = $ntapData.Cluster.Info.ClusterUuid
                                SystemType    = $ntapData.pstypenames[0]
                                SystemVersion = ([regex]::Split(([regex]::Split($ntapData.System.VersionInfo.Version, 'Release\s(?:\w+[__])?')[1]), '\s+'))[0].Replace(':', '')
                            }
                            ElapsedTicks        = ([datetime]::Now - $functionStartTime).Ticks
                            ExcludeParameter    = 'Name'
                        }

                        Write-Statistics @statsParams
                    }
                    catch {
                        Write-Error -ErrorRecord $_
                    }
                }
            }
            'ControllerObject' {
                foreach ($controller in ($AbstractController | Where-Object { $_ })) {
                    Write-Verbose -Message "Validating connectivity to system: $( $controller.Name )"

                    try {
                        if ($controller.GetType().FullName -ne 'NetApp.Ontapi.Filer.C.NcController') {
                            throw 'Incorrect controller type specified for the Get-NtapClusterData function.'
                        }

                        if ($controller.Protocol -like 'HTTP*' -and $PSCmdlet.MyInvocation.BoundParameters.ContainsKey('Timeout')) {
                            $controller.TimeoutMsec = $Timeout
                        }

                        if (-not (Test-NtapStorageSystemConnection -Controller $controller -DisableLifCheck:$DisableLifCheck.IsPresent -ErrorAction Stop)) {
                            throw 'Unspecified error.'
                        }

                        Write-Verbose -Message "Valid connection established to system: $( $controller.Name )"
                    }
                    catch {
                        $Global:Error.RemoveAt(0)

                        Write-Error -ErrorRecord (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("Unable to gather data from system. Reason: $( $_.Exception.GetBaseException().Message )", $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $controller.Name))

                        continue
                    }

                    Write-Verbose -Message "Gathering data from system: $( $controller.Name )"

                    try {
                        $functionStartTime = [datetime]::Now

                        $ntapData = Get-ClusterData -Controller $controller

                        $ntapData

                        if ($ntapData -and $InvokeAsup.IsPresent) {
                            foreach ($node in ($ntapData.Node | Where-Object { $_ })) {
                                try {
                                    Write-Verbose -Message "Invoking ASUP on node: $( $node.Node )"
                                    [void](Invoke-NcAutoSupport -Node $node.Node -Message "NetAppDocs_$( $ntapData.ReferenceDT.ToString('u') )" -Type all -Controller $controller)
                                }
                                catch {
                                    $Global:Error.RemoveAt(0)

                                    Show-ErrorMessage -Message "Unable to invoke ASUP on node: $( $node.Node )"
                                }
                            }
                        }

                        $statsParams = @{
                            FunctionPSCallStack  = Get-PSCallStack
                            TargetDetails        = [pscustomobject]@{
                                SystemUid     = $ntapData.Cluster.Info.ClusterUuid
                                SystemType    = $ntapData.pstypenames[0]
                                SystemVersion = ([regex]::Split(([regex]::Split($ntapData.System.VersionInfo.Version, 'Release\s(?:\w+[__])?')[1]), '\s+'))[0].Replace(':', '')
                            }
                            ElapsedTicks         = ([datetime]::Now - $functionStartTime).Ticks
                            ParameterToSetAsBool = 'AbstractController'
                        }

                        Write-Statistics @statsParams
                    }
                    catch {
                        Write-Error -ErrorRecord $_
                    }
                }
            }
        }
    }
}

function Get-ClusterData {
    <#
        .NOTES
            AUTHOR : Jason Cole
            REQUIRES
            : NetApp ONTAP PowerShell Toolkit
            BURTS
            : 636873 - ldap-config-get-iter ZAPI doesn't provide which Vserver the config is attached to
            :   Workaround: Parsing 'vserver services ldap show -fields' cli output manually
            :   Fixed: SN.0
            : 640487 - gpo-get-iter - get-iters which are associated with vservers and callable from cluster context should return the vserver name
            :   Workaround: Loop through each vserver, calling the associated cmdlet
            :   Fixed: SN.0
            : 512073 - Need nfs-status-get-iter in cluster scope (nfs-service-get-iter)
            :   Workaround: Loop through each vserver, calling the associated cmdlet
            :   Fixed: RR.1
            : 520762 - The system-node-get-iter is missing values in the return like "is-node-healthy"
            :   Workaround: Parsing 'cluster show -fields' cli output manually
            :   Fixed: RR.1
            : 613219 - "fcadmin config" does not show storage HBA card information (Vespa platforms)
            :   Workaround: Ignore errors from ZAPI call for Vespa platforms
            :   Fixed: No target
            : 653291, 658442, 674884, 715998 - Loop detected in next() for table snapmirror
            :   Workaround: None
            :   Fixed: SN.0
            : 1140022 - template-management-template-parameter-get-iter API hangs when max-records <= num-records
            :  Workaround: Using CLI equivalent
            :  Fixed: None
    #>

    [CmdletBinding()]
    param (
        [Parameter(Mandatory)]
        [ValidateNotNullOrEmpty()]
        [NetApp.Ontapi.Filer.C.NcController]$Controller
    )

    $systemData = [pscustomobject]@{ }
    $systemData.pstypenames.Insert(0, 'NetAppDocs.NcController.Data')

    Write-Verbose -Message '- Capturing module version -'
    $systemData | Add-Member -MemberType NoteProperty -Name NetAppDocsVersion (Get-NtapDocsVersion -AsVersion)

    Write-Verbose -Message '- Capturing PowerShell version -'
    $systemData | Add-Member -MemberType NoteProperty -Name PowerShellVersion -Value $PSVersionTable.PSVersion

    $NetAppDocsError = @( )

    try {
        Write-Verbose -Message 'Calling Get-NcSystemApi'
        $useSystemCli = $null -ne (Get-NcSystemApi -Name 'system-cli*' -Controller $Controller -ErrorVariable +NetAppDocsError)

        #region Raw Sysconfig Data

        Write-Verbose -Message '- Collecting raw sysconfig data -'

        $sysconfigAc = $(
            if ($useSystemCli) {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (system node run -node * -command sysconfig -ac)'
                (Invoke-NcSystemApi -Request '<system-cli><args><arg>system node run -node * -command sysconfig -ac</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'
            }
            else {
                Write-Verbose -Message 'Calling Invoke-NcSsh (system node run -node * -command sysconfig -ac)'
                Invoke-NcSsh -Command 'system node run -node * -command sysconfig -ac' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false
            }
        )

        $sysconfigV = $(
            if ($useSystemCli) {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (system node run -node * -command sysconfig -v)'
                (Invoke-NcSystemApi -Request '<system-cli><args><arg>system node run -node * -command sysconfig -v</arg></args></system-cli>' -Controller $Controller -ErrorAction SilentlyContinue -ErrorVariable +NetAppDocsError).results.'cli-output'
            }
            else {
                Write-Verbose -Message 'Calling Invoke-NcSsh (system node run -node * -command sysconfig -v)'
                Invoke-NcSsh -Command 'system node run -node * -command sysconfig -v' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false
            }
        )

        Write-Verbose -Message 'Calling Get-AdditionalSysconfigProperties'
        $sysconfigProperties = Get-AdditionalSysconfigProperties -Sysconfig $sysconfigV

        #endregion Raw Sysconfig Data

        #region System Information

        ##Unused Cmdlet: Get-NcSystemImagePackage - Not needed for build docs
        ##Unused Cmdlet: Get-NcSystemImageExternalPackage - Not needed for build docs
        ##Unused Cmdlet: Get-NcSystemImageUpdateStatus - Not needed for build docs
        ##Unused Cmdlet: Get-NcSystemUserCapability - Not needed for build docs
        ##Unused Cmdlet: Get-NcSystemVendorInfo - Not needed for build docs
        ##Unused Cmdlet: Get-NcSystemVersion - Info in Get-NcSystemVersionInfo
        ##Unused Cmdlet: Get-NcExternalCache - (1.120+) Using CLI equivalent as ZAPI is missing one property
        ##Unused Cmdlet: Get-NcSystemHypervisorHostConfig - (1.120+) Not currently needed for build docs
        ##Unused Cmdlet: Get-NcSystemHypervisorSnmpv3Config - (1.120+) Not currently needed for build docs
        ##Unused Cmdlet: Get-NcSystemHypervisorUplinkMapping - (1.120+) Not currently needed for build docs
        ##Unused Cmdlet: Get-NcSystemStatusHypervisor - (1.130+) Not currently needed for build docs
        ##Unused Cmdlet: Get-NcSystemStatusService - (1.130+) Not currently needed for build docs

        Write-Verbose -Message '- Gathering basic system information -'
        $systemData | Add-Member -MemberType NoteProperty -Name System -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcSystemVersionInfo'
        $systemData.System | Add-Member -MemberType NoteProperty -Name VersionInfo -Value (Get-NcSystemVersionInfo -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcSystemOntapiVersion'
        $systemData.System | Add-Member -MemberType NoteProperty -Name OntapiVersion -Value ([version]::Parse((Get-NcSystemOntapiVersion -Controller $Controller -ErrorVariable +NetAppDocsError).ToString()))

        Write-Verbose -Message 'Calling Get-NcSystemImage'
        $systemData.System | Add-Member -MemberType NoteProperty -Name Image -Value (Get-NcSystemImage -Controller $Controller -ErrorVariable +NetAppDocsError)

        $systemData.System | Add-Member -MemberType NoteProperty -Name CliSessionTimeout -Value $(
            if ($useSystemCli) {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (system timeout show)'
                ((Invoke-NcSystemApi -Request '<system-cli><args><arg>system timeout show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliTextInstance).CliSessionTimeout
            }
            else {
                Write-Verbose -Message 'Calling Invoke-NcSsh (system timeout show)'
                (Invoke-NcSsh -Command 'system timeout show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText).CliSessionTimeout
            }
        )

        #endregion System Information

        #region Node Information

        ##Unused Cmdlet: Get-NcNodePower - (1.30+) Not needed for build docs

        Write-Verbose -Message '- Gathering node information-'

        Write-Verbose -Message 'Calling Get-NcNode'
        $systemData | Add-Member -MemberType NoteProperty -Name Node -Value (Get-NcNode -Controller $Controller -ErrorVariable +NetAppDocsError | Where-Object { -not [string]::IsNullOrWhiteSpace($_.Node) })

        #BURT 520762
        if ($systemData.System.OntapiVersion -lt [version]::Parse('1.17')) {
            $clusterShowCliOutput = $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (cluster show (priv advanced))'
                    (Invoke-NcSystemApi -Request '<system-cli><priv>advanced</priv><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>cluster show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (cluster show (priv advanced))'
                    Invoke-NcSsh -Command 'set -privilege advanced -confirmations off; set -showseparator ~; set -showallfields true; cluster show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            )

            foreach ($node in ($systemData.Node | Where-Object { $_ })) {
                $node | Add-Member -MemberType NoteProperty -Name IsEpsilonNode -Value ($clusterShowCliOutput | Where-Object { $_.NodeUuid -ceq $node.NodeUuid }).Epsilon
                $node | Add-Member -MemberType NoteProperty -Name IsNodeHealthy -Value ($clusterShowCliOutput | Where-Object { $_.NodeUuid -ceq $node.NodeUuid }).Health
                $node | Add-Member -MemberType NoteProperty -Name IsNodeClusterEligible -Value ($clusterShowCliOutput | Where-Object { $_.NodeUuid -ceq $node.NodeUuid }).Eligibility
            }
        }

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.120')) {
            $systemData.System | Add-Member -MemberType NoteProperty -Name ExternalCache -Value $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (system node external-cache show)'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>system node external-cache show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (system node external-cache show))'
                    Invoke-NcSsh -Command 'set -showseparator ~; set -showallfields true; system node external-cache show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            )
        }

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.140')) {
            #This API will throw an error if cluster is set to non_ha mode
            try {
                Write-Verbose -Message 'Calling Get-NcSystemCacheMirror'
                $systemCacheMirror = Get-NcSystemCacheMirror -Controller $Controller -ErrorAction Stop

                foreach ($node in ($systemData.Node | Where-Object { $_ })) {
                    $node | Add-Member -MemberType NoteProperty -Name IsCacheMirrorEnabled -Value ($systemCacheMirror | Where-Object { $_.Node -ceq $node.Node }).Enabled
                }
            }
            catch {
                if ($_.Exception.Message -like '*The cache mirror feature requires a HA configuration*') {
                    $Global:Error.RemoveAt(0)
                }
                else {
                    $NetAppDocsError += $_
                }
            }
        }

        Write-Verbose -Message 'Calling Get-NcNodeInfo'
        $systemData | Add-Member -MemberType NoteProperty -Name NodeInfo -Value (Get-NcNodeInfo -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Identifying additional properties from parsed sysconfig'
        foreach ($node in ($systemData.NodeInfo | Where-Object { $_ })) {
            $node | Add-Member -MemberType NoteProperty -Name IoxmPresent -Value ([bool]($sysconfigProperties | Where-Object { $_.NodeName -ceq $node.SystemName }).IoxmPresent)
            $node | Add-Member -MemberType NoteProperty -Name PlatformConfig -Value (Get-PlatformConfig -SystemModel $node.SystemModel -IsClustered (! [string]::IsNullOrWhiteSpace($node.PartnerSystemId)) -IoxmPresent $node.IoxmPresent)
            $node | Add-Member -MemberType NoteProperty -Name LoaderVersion -Value ($sysconfigProperties | Where-Object { $_.NodeName -ceq $node.SystemName }).LoaderVersion
        }

        #endregion Node Information

        #region Kenv Information

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Gathering KENV information -'

            $systemData | Add-Member -MemberType NoteProperty -Name Kenv -Value $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (debug kenv show (priv diag))'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -privilege diagnostic -confirmations off;</arg><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>debug kenv show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (debug kenv show (priv diag)))'
                    Invoke-NcSsh -Command 'set -privilege diagnostic -confirmations off; set -showseparator ~; set -showallfields true; debug kenv show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            )
        }

        #endregion Kenv Information

        #region Cluster Information

        ##Unused Cmdlet: Get-NcClusterHaPartner - Info in Get-NcClusterHa
        ##Unused Cmdlet: Get-NcClusterNode - Info in Get-NcNode
        ##Unused Cmdlet: Get-NcClusterImage - (1.30+) Info in Get-NcSystemImage
        ##Unused Cmdlet: Get-NcClusterImageDownload - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcClusterImagePackage - (1.30+) Info in Get-NcSystemImagePackage
        ##Unused Cmdlet: Get-NcClusterImageUpdateHistory - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcClusterImageUpdateLog - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcClusterImageUpdateProgress - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcClusterImageUpdateProgressOverview - (1.31+) Not needed for build docs
        ##Unused Cmdlet: Get-NcClusterCreateProgress - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcClusterImageUpgradePlan - (1.120+) Not needed for build docs
        ##Unused Cmdlet: Get-NcClusterZoneInfo -- (1.130+) Not needed for build docs
        ##Unused Cmdlet: Get-NcClusterNodeAddStatus -- (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcClusterImageMetroclusterUpdateProgressInfo - (1.150+) Not needed for build docs

        Write-Verbose -Message '- Gathering cluster information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Cluster -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcCluster'
        $systemData.Cluster | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcCluster -Controller $Controller -ErrorVariable +NetAppDocsError)

        $systemData.Cluster | Add-Member -MemberType NoteProperty -Name RingReplication -Value $(
            if ($useSystemCli) {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (cluster ring show (priv advanced))'
                (Invoke-NcSystemApi -Request '<system-cli><priv>advanced</priv><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>cluster ring show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
            }
            else {
                Write-Verbose -Message 'Calling Invoke-NcSsh (cluster ring show (priv advanced))'
                Invoke-NcSsh -Command 'set -privilege advanced -confirmations off; set -showseparator ~; set -showallfields true; cluster ring show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
            }
        )

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            $systemData.Cluster | Add-Member -MemberType NoteProperty -Name Contact -Value $(
                try {
                    Write-Verbose -Message 'Calling Get-NcClusterContact'
                    Get-NcClusterContact -Controller $Controller -ErrorAction Stop
                }
                catch {
                    if ($_.Exception.Message -like "*entry doesn't exist*") {
                        $Global:Error.RemoveAt(0)
                    }
                    else {
                        $NetAppDocsError += $_
                    }
                }
            )
        }

        #endregion Cluster Information

        #region Cluster CLAM Information

        Write-Verbose -Message '- Gathering Cluster Kernel-Service Info -'
        $systemData.Cluster | Add-Member -MemberType NoteProperty -Name KernelService -Value ([pscustomobject]@{ })

        #diag privilege not necessary for 8.3+ (advanced)
        $systemData.Cluster.KernelService | Add-Member -MemberType NoteProperty -Name Status -Value $(
            if ($useSystemCli) {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (cluster kernel-service show (priv diag))'
                (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -privilege diagnostic -confirmations off;</arg><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>cluster kernel-service show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
            }
            else {
                Write-Verbose -Message 'Calling Invoke-NcSsh (cluster kernel-service show (priv diag))'
                Invoke-NcSsh -Command 'set -privilege diagnostic -confirmations off; set -showseparator ~; set -showallfields true; cluster kernel-service show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
            }
        )

        #endregion Cluster CLAM Information

        #region Cluster HA Information

        ##Unused Cmdlet: Get-NcClusterHaHwAssistStats - Not needed
        ##Unused Cmdlet: Get-NcClusterHaInfo - Not currently needed, but cf-status may go unsupported.
        ##Unused Cmdlet: Get-NcClusterHaGiveback - (1.20+) Not needed for build docs (Cluster.Ha.Giveback - if Cluster.Ha.Info.CurrentMode -ne 'non_ha')
        ##Unused Cmdlet: Get-NcClusterHaTakeover - (1.20+) Not needed for build docs (Cluster.Ha.Takeover - if Cluster.Ha.Info.CurrentMode -ne 'non_ha')

        Write-Verbose -Message '- Gathering cluster HA information -'
        $systemData.Cluster | Add-Member -MemberType NoteProperty -Name Ha -Value ([pscustomobject]@{ })

        #If we use the new NcClusterHaInfo cmdlet in the future, we will need to change this 'Info' to 'Status' and update types files accordingly
        Write-Verbose -Message 'Calling Get-NcClusterHa'
        $systemData.Cluster.Ha | Add-Member -MemberType NoteProperty -Name Info -Value ($systemData.Node | Get-NcClusterHa -Controller $Controller -ErrorVariable +NetAppDocsError)

        #Check for single node cluster (non_ha mode)
        if (($systemData.Cluster.Ha.Info | Select-Object -First 1).CurrentMode -ne 'non_ha') {
            Write-Verbose -Message '- Gathering cluster HA HwAssist information -'
            $systemData.Cluster.Ha | Add-Member -MemberType NoteProperty -Name HwAssist -Value @( )

            foreach ($node in ($systemData.NodeInfo | Where-Object { $_ })) {
                Write-Verbose -Message "Calling Get-NcClusterHaHwAssist (Node: $( $node.SystemName ))"
                $systemData.Cluster.Ha.HwAssist += Get-NcClusterHaHwAssist -Node $node.SystemName -Controller $Controller -ErrorVariable +NetAppDocsError | Select-Object -Property *
                $systemData.Cluster.Ha.HwAssist | Select-Object -Last 1 | Add-Member -MemberType NoteProperty -Name NodeName -Value $node.SystemName
            }
        }

        #ZAPI: HA Configured property - 'cluster-ha-get' introduced in 8.2 (use system-cli to get this property for older releases) - This ZAPI may be removed in the future.
        $systemData.Cluster.Ha | Add-Member -MemberType NoteProperty -Name Configured -Value $(
            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (cluster-ha-get)'
                ((Invoke-NcSystemApi -Request '<cluster-ha-get />' -Controller $Controller -ErrorVariable +NetAppDocsError).results.attributes.'cluster-ha-info' | Convert-GenericXmlNode).HaConfigured
            }
            else {
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (cluster ha show (priv advanced))'
                    ((Invoke-NcSystemApi -Request '<system-cli><priv>advanced</priv><args><arg>off;</arg><arg>cluster ha show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliTextInstance).HighAvailabilityConfigured
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (cluster ha show (priv advanced))'
                    (Invoke-NcSsh -Command 'set -privilege advanced -confirmations off; cluster ha show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliTextInstance).HighAvailabilityConfigured
                }
            }
        )

        #endregion Cluster HA Information

        #region Cluster LogForward Information

        ##Unused Cmdlet: Get-NcClusterLogForwardStatistics - (1.31+) Not needed for build docs

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.31')) {
            Write-Verbose -Message '- Gathering Cluster LogForward information -'
            $systemData.Cluster | Add-Member -MemberType NoteProperty -Name LogForward -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcClusterLogForward'
            $systemData.Cluster.LogForward | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcClusterLogForward -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Cluster LogForward Information

        #region Node Health Validation

        Write-Verbose -Message '- Validating node health -'
        $nodesNotHealthy = ($systemData.Node | Where-Object { -not $_.IsNodeHealthy }).Node

        if ($nodesNotHealthy) {
            foreach ($node in ($nodesNotHealthy | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })) {
                $nodeHaStateMsg = [string]::Format(
                    "Node: {0} (HA State: {1})`nPartner: {2} (HA State: {3})", $node, $(
                        if ($systemData.Node | Where-Object { $_.Name -ceq $node }) {
                            ($systemData.Node | Where-Object { $_.Name -ceq $node }).State
                        }
                        else {
                            'UNKNOWN'
                        }
                    ),
                    $(
                        if ($systemData.Cluster.Ha.Info | Where-Object { $_.PartnerName -ceq $node }) {
                            ($systemData.Cluster.Ha.Info | Where-Object { $_.PartnerName -ceq $node }).Name
                        }
                        else {
                            'UNKNOWN'
                        }
                    ),
                    $(
                        if ($systemData.Cluster.Ha.Info | Where-Object { $_.PartnerName -ceq $node }) {
                            ($systemData.Cluster.Ha.Info | Where-Object { $_.PartnerName -ceq $node }).State
                        }
                        else {
                            'UNKNOWN'
                        }
                    )
                )

                $msg = New-Object -TypeName System.Text.StringBuilder

                [void]$msg.AppendLine("Node '$node' is not healthy. Errors may be encountered and data may be incomplete.")
                [void]$msg.AppendLine('----------------------------------------------------------------------------------')
                [void]$msg.AppendLine($nodeHaStateMsg)
                [void]$msg.AppendLine('All nodes in the cluster must be healthy to ensure accurate data is gathered.')
                [void]$msg.AppendLine('----------------------------------------------------------------------------------')
                [void]$msg.AppendLine()

                Show-ErrorMessage -Message $msg.ToString()

                $NetAppDocsError += New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("Node is not healthy. Reason: $nodeHaStateMsg", 'NODE_NOT_HEALTHY', [System.Management.Automation.ErrorCategory]::DeviceError, $node)
            }
        }

        #endregion Node Health Validation

        #region HA Interconnect Information

        ##Unused Cmdlet: Get-NcHaInterconnectAutoResetConfiguration - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcHaInterconnectChannelStatus - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcHaInterconnectFlintFlashDetails - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcHaInterconnectFlintImageDetails - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcHaInterconnectFlushDetails - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcHaInterconnectNvramConfiguration - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcHaInterconnectOodStatus - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcHaInterconnectPerformanceStatistics - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcHaInterconnectPortDetails - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcHaInterconnectSGLStatistics - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcHaInterconnectTimeoutValues - (1.100+) Not needed for build docs

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.100')) {
            Write-Verbose -Message '- Gathering HA interconnect information -'
            $systemData | Add-Member -MemberType NoteProperty -Name HaInterconnect -Value ([pscustomobject]@{ })

            try {
                Write-Verbose -Message 'Calling Get-NcHaInterconnectConfigDetails'
                $systemData.HaInterconnect | Add-Member -MemberType NoteProperty -Name ConfigDetails -Value (Get-NcHaInterconnectConfigDetails -Controller $Controller -ErrorAction Stop)

                Write-Verbose -Message 'Calling Get-NcHaInterconnectConnectionStatus'
                $systemData.HaInterconnect | Add-Member -MemberType NoteProperty -Name ConnectionStatus -Value (Get-NcHaInterconnectConnectionStatus -Controller $Controller -ErrorAction Stop)
            }
            catch {
                if ($_.Exception.Message -like '*This command is not supported on this platform*') {
                    $Global:Error.RemoveAt(0)
                }
                else {
                    $NetAppDocsError += $_
                }
            }
        }

        #endregion HA Interconnect Information

        #region Timezone Information

        Write-Verbose -Message 'Calling Get-NcTimezone'
        $systemData | Add-Member -MemberType NoteProperty -Name Timezone -Value (Get-NcTimezone -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion Timezone Information

        #region License Information

        ##Unused Cmdlet: Get-NcLicenseEntitlementRisk - (1.31+) Not needed for build docs (License.EntitlementRisk)
        ##Unused Cmdlet: Get-NcLicenseCapacity - (1.100+) Not needed for build docs (License.Capacity)
        ##Unused Cmdlet: Get-NcLicenseSubscription - (1.100+) Not needed for build docs (License.Subscription)
        ##Unused Cmdlet: Get-NcLicenseStatus - (1.120+ - license-v2-status-list-info for 1.20+) Not needed for build docs (License.Status)
        ##Unused Cmdlet: Get-NcLicenseV2CapacityPool - (Invalid Cmdlet) No corresponding API as of 1.150

        Write-Verbose -Message '- Gathering license information -'
        $systemData | Add-Member -MemberType NoteProperty -Name License -Value ([pscustomobject]@{ })

        #License v2.0 zapis introduced in 8.2. Prior to that, the unsupported interim-license-list-get zapi must be used to get license information
        $systemData.License | Add-Member -MemberType NoteProperty -Name Info -Value $(
            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
                Write-Verbose -Message 'Calling Get-NcLicense'
                Get-NcLicense -Controller $Controller -ErrorVariable +NetAppDocsError
            }
            else {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (interim-license-list-get)'
                ((Invoke-NcSystemApi -Request '<interim-license-list-get />' -Controller $Controller -ErrorVariable +NetAppDocsError).results.licenses.'interim-license-cluster-info' | Convert-GenericXmlNode)
            }
        )

        ##Unused cmdlet: Get-NcLicenseV2LicenseManagerStatus - (1.150+) Awaiting documentation
        ##Unused cmdlet: Get-NcLicenseV2LicenseManagerConfig - (1.150+) Awaiting documentation
        ##Unused cmdlet: Get-NcSystemLicenseAggregate - (1.150+) Awaiting documentation

        #endregion License Information

        #region NTP Server Information

        ##Unused Cmdlet: Get-NcNtpServerSecurity - (1.130+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNtpServerKey - (1.150+) Not needed for build docs

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message '- Gathering Ntp server information -'
            $systemData | Add-Member -MemberType NoteProperty -Name Ntp -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcNtpServer'
            $systemData.Ntp | Add-Member -MemberType NoteProperty -Name Server -Value (Get-NcNtpServer -Controller $Controller -ErrorVariable +NetAppDocsError)

            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.130')) {
                Write-Verbose -Message 'Calling Get-NcNtpServerStatus'
                $systemData.Ntp | Add-Member -MemberType NoteProperty -Name ServerStatus -Value (Get-NcNtpServerStatus -Controller $Controller -ErrorVariable +NetAppDocsError)
            }
        }

        #endregion NTP Server Information

        #region System Services Data

        Write-Verbose -Message '- Gathering system services data -'
        $systemData | Add-Member -MemberType NoteProperty -Name SystemServices -Value ([pscustomobject]@{ })

        if ($systemData.System.OntapiVersion -lt [version]::Parse('1.30')) {
            $systemData.SystemServices | Add-Member -MemberType NoteProperty -Name Ntp -Value ([pscustomobject]@{ })

            $systemData.SystemServices | Add-Member -MemberType NoteProperty -Name FirewallPolicy -Value $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (system services firewall policy show)'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>system services firewall policy show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (system services firewall policy show)'
                    Invoke-NcSsh -Command 'set -showseparator ~; set -showallfields true; system services firewall policy show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            )

            #This CLI command was changed to a diag level command in 8.2
            $systemData.SystemServices.Ntp | Add-Member -MemberType NoteProperty -Name Config -Value $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (system services ntp config show (priv diag))'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -privilege diagnostic -confirmations off;</arg><arg>system services ntp config show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliTextInstance
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (system services ntp config show (priv diag))'
                    Invoke-NcSsh -Command 'set -privilege diagnostic -confirmations off; system services ntp config show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliTextInstance
                }
            )

            $systemData.SystemServices.Ntp | Add-Member -MemberType NoteProperty -Name Server -Value $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (system services ntp server show)'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>system services ntp server show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (system services ntp server show)'
                    Invoke-NcSsh -Command 'set -showseparator ~; set -showallfields true; system services ntp server show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            )
        }

        $systemData.SystemServices | Add-Member -MemberType NoteProperty -Name Web -Value $(
            if ($systemData.System.OntapiVersion -lt [version]::Parse('1.31')) {
                $systemServicesWebShow = $(
                    if ($useSystemCli) {
                        Write-Verbose -Message 'Calling Invoke-NcSystemApi (system services web show)'
                        (Invoke-NcSystemApi -Request '<system-cli><args><arg>system services web show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliTextInstance
                    }
                    else {
                        Write-Verbose -Message 'Calling Invoke-NcSsh (system services web show)'
                        Invoke-NcSsh -Command 'system services web show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliTextInstance
                    }
                )

                $systemServicesWebShow | Select-Object -ExcludeProperty HttpProtocolPort, HttpsProtocolPort, ExternalWebServices, SslFips1402Enabled -Property *, @{
                    Name       = 'HttpPort'
                    Expression = { $_.HttpProtocolPort }
                }, @{
                    Name       = 'HttpsPort'
                    Expression = { $_.HttpsProtocolPort }
                }, @{
                    Name       = 'External'
                    Expression = { $_.ExternalWebServices }
                }, @{
                    Name       = 'SslFipsEnabled'
                    Expression = { $_.SslFips1402Enabled }
                }
            }
            else {
                Write-Verbose -Message 'Calling Get-NcSystemWebServices'
                Get-NcSystemWebServices -Controller $Controller -ErrorVariable +NetAppDocsError
            }
        )

        $systemData.SystemServices | Add-Member -MemberType NoteProperty -Name WebNode -Value $(
            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.100')) {
                Write-Verbose -Message 'Calling Get-NcSystemServicesWebNode'
                Get-NcSystemServicesWebNode -Controller $Controller -ErrorVariable +NetAppDocsError
            }
            else {
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (system services web node show)'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>system services web node show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (system services web node show)'
                    Invoke-NcSsh -Command 'set -showseparator ~; set -showallfields true; system services web node show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            }
        )

        #endregion System Services Data

        #region Adapter Card Inventory

        Write-Verbose -Message '- Identifying adapter cards -'

        Write-Verbose -Message 'Calling Get-NtapExpansionSlotInventory'
        $systemData | Add-Member -MemberType NoteProperty -Name AdapterCard -Value (Get-NtapExpansionSlotInventory -SysconfigAc $sysconfigAc -SysconfigVorA $sysconfigV | Sort-Object -Property NodeName, SlotNumber)

        #endregion Adapter Card Inventory

        #region NDMP

        ##Unused Cmdlet: Get-NcNdmpNodeSession - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNdmpVserverAttr - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNdmpVserverSession - (1.100+) Not needed for build docs

        Write-Verbose -Message '- Gathering NDMP configuration -'
        $systemData | Add-Member -MemberType NoteProperty -Name Ndmp -Value ([pscustomobject]@{ })

        $systemData.Ndmp | Add-Member -MemberType NoteProperty -Name SystemServices -Value $(
            if ($useSystemCli) {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (system services ndmp show)'
                (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>system services ndmp show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
            }
            else {
                Write-Verbose -Message 'Calling Invoke-NcSsh (system services ndmp show)'
                Invoke-NcSsh -Command 'set -showseparator ~; set -showallfields true; system services ndmp show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
            }
        )

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            $systemData.Ndmp | Add-Member -MemberType NoteProperty -Name NodeScopeMode -Value (
                [regex]::Matches(
                    $(
                        if ($useSystemCli) {
                            Write-Verbose -Message 'Calling Invoke-NcSystemApi (system services ndmp node-scope-mode status)'
                            (Invoke-NcSystemApi -Request '<system-cli><args><arg>system services ndmp node-scope-mode status</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'
                        }
                        else {
                            Write-Verbose -Message 'Calling Invoke-NcSsh (system services ndmp node-scope-mode status)'
                            Invoke-NcSsh -Command 'system services ndmp node-scope-mode status' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false
                        }
                    ),
                    'enabled|disabled'
                )[0].Value
            )

            if ($systemData.Ndmp.NodeScopeMode -eq 'disabled') {
                $systemData.Ndmp | Add-Member -MemberType NoteProperty -Name VserverServices -Value $(
                    if ($useSystemCli) {
                        Write-Verbose -Message 'Calling Invoke-NcSystemApi (vserver services ndmp show)'
                        (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>vserver services ndmp show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                    }
                    else {
                        Write-Verbose -Message 'Calling Invoke-NcSsh (vserver services ndmp show)'
                        Invoke-NcSsh -Command 'set -showseparator ~; set -showallfields true; vserver services ndmp show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                    }
                )
            }
        }

        #endregion NDMP

        #region Remote Management Configuration

        ##Unused Cmdlet: Get-NcServiceProcessorImageUpdate - Not needed for build docs
        ##Unused Cmdlet: Get-NcServiceProcessorLogAllocation - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcServiceProcessorApiService (1.30+) - Not needed for build docs (ServiceProcess.ApiService)
        ##Unused Cmdlet: Get-NcServiceProcessorAutoConfig (1.30+) - Not needed for build docs (ServiceProcess.AutoConfig)
        ##Unused Cmdlet: Get-NcServiceProcessorAutoSupport (1.30+) - Not needed for build docs (ServiceProcess.Autosupport)

        Write-Verbose -Message '- Gathering remote management configuration -'
        $systemData | Add-Member -MemberType NoteProperty -Name ServiceProcessor -Value ([pscustomobject]@{ })

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message 'Calling Get-NcServiceProcessor'
            $systemData.ServiceProcessor | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcServiceProcessor -Controller $Controller -ErrorVariable +NetAppDocsError)

            $systemData.ServiceProcessor | Add-Member -MemberType NoteProperty -Name Image -Value $(
                foreach ($node in ($systemData.NodeInfo | Where-Object { $_ })) {
                    foreach ($firmwareImage in 'primary', 'backup') {
                        Write-Verbose -Message "Calling Get-NcServiceProcessorImage (Node: $( $node.SystemName )) (FirmwareImage: $firmwareImage)"
                        try {
                            Get-NcServiceProcessorImage -Node $node.SystemName -FirmwareImage $firmwareImage -Controller $Controller -ErrorAction Stop | Where-Object { $_.FirmwareImage -eq $firmwareImage }
                        }
                        catch {
                            if ($_.Exception.Message -like "*entry doesn't exist*") {
                                $Global:Error.RemoveAt(0)
                            }
                            else {
                                $NetAppDocsError += $_
                            }
                        }
                    }
                }
            )

            Write-Verbose -Message 'Calling Get-NcServiceProcessorNetwork'
            $systemData.ServiceProcessor | Add-Member -MemberType NoteProperty -Name Network -Value (Get-NcServiceProcessorNetwork -Controller $Controller -ErrorVariable +NetAppDocsError)

            try {
                Write-Verbose -Message 'Calling Get-NcServiceProcessorSsh'
                $systemData.ServiceProcessor | Add-Member -MemberType NoteProperty -Name Ssh -Value (Get-NcServiceProcessorSsh -Controller $Controller -ErrorAction Stop)
            }
            catch {
                if ($_.Exception.Message -like "*entry doesn't exist*") {
                    $Global:Error.RemoveAt(0)
                }
                else {
                    $NetAppDocsError += $_
                }
            }
        }
        elseif ($systemData.System.OntapiVersion -ge [version]::Parse('1.21')) {
            $systemData.ServiceProcessor | Add-Member -MemberType NoteProperty -Name Info -Value $(
                foreach ($node in ($systemData.NodeInfo | Where-Object { $_ })) {
                    Write-Verbose -Message "Calling Get-NcServiceProcessor (Node: $( $node.SystemName ))"
                    Get-NcServiceProcessor -Node $node.SystemName -Controller $Controller -ErrorVariable +NetAppDocsError
                }
            )

            $systemData.ServiceProcessor | Add-Member -MemberType NoteProperty -Name Image -Value $(
                foreach ($node in ($systemData.NodeInfo | Where-Object { $_ })) {
                    foreach ($firmwareImage in 'primary', 'backup') {
                        Write-Verbose -Message "Calling Get-NcServiceProcessorImage (Node: $( $node.SystemName )) (FirmwareImage: $firmwareImage)"
                        Get-NcServiceProcessorImage -Node $node.SystemName -FirmwareImage $firmwareImage -Controller $Controller -ErrorVariable +NetAppDocsError | Where-Object { $_.FirmwareImage -eq $firmwareImage }
                    }
                }
            )

            $systemData.ServiceProcessor | Add-Member -MemberType NoteProperty -Name Network -Value $(
                foreach ($node in ($systemData.NodeInfo | Where-Object { $_ })) {
                    'Ipv4', 'Ipv6' | ForEach-Object {
                        Write-Verbose -Message "Calling Get-NcServiceProcessorNetwork (Node: $( $node.SystemName )) (AddressType: $_)"
                        Get-NcServiceProcessorNetwork -Node $node.SystemName -AddressType $_ -Controller $Controller -ErrorVariable +NetAppDocsError
                    }
                }
            )
        }
        else {
            if (-not [string]::IsNullOrWhiteSpace($sysconfigV)) {
                Write-Verbose -Message 'Calling Get-NtapRemoteManagement'
                $systemData.ServiceProcessor | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NtapRemoteManagement -Sysconfig $sysconfigV)
            }
            else {
                $sp = $(
                    switch -Wildcard ($sysconfigV) {
                        '*Service Processor*' {
                            if ($useSystemCli) {
                                Write-Verbose -Message 'Calling Invoke-NcSystemApi (system node run -node * -command sp status)'
                                (Invoke-NcSystemApi -Request '<system-cli><args><arg>system node run -node * -command sp status</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'
                            }
                            else {
                                Write-Verbose -Message 'Calling Invoke-NcSsh (system node run -node * -command sp status)'
                                Invoke-NcSsh -Command 'system node run -node * -command sp status' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false
                            }

                            break
                        }
                        '*Remote LAN Module*' {
                            if ($useSystemCli) {
                                Write-Verbose -Message 'Calling Invoke-NcSystemApi (system node run -node * -command rlm status)'
                                (Invoke-NcSystemApi -Request '<system-cli><args><arg>system node run -node * -command rlm status</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'
                            }
                            else {
                                Write-Verbose -Message 'Calling Invoke-NcSsh (system node run -node * -command rlm status)'
                                Invoke-NcSsh -Command 'system node run -node * -command rlm status' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false
                            }

                            break
                        }
                        '*Baseboard Management Controller*' {
                            if ($useSystemCli) {
                                Write-Verbose -Message 'Calling Invoke-NcSystemApi (system node run -node * -command bmc status)'
                                (Invoke-NcSystemApi -Request '<system-cli><args><arg>system node run -node * -command bmc status</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'
                            }
                            else {
                                Write-Verbose -Message 'Calling Invoke-NcSsh (system node run -node * -command bmc status)'
                                Invoke-NcSsh -Command 'system node run -node * -command bmc status' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false
                            }

                            break
                        }
                        default { }
                    }
                )

                if (-not [string]::IsNullOrWhiteSpace($sp)) {
                    Write-Verbose -Message 'Calling Get-NtapRemoteManagement'
                    $systemData.ServiceProcessor | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NtapRemoteManagement -Sysconfig $sp)
                }
                else {
                    $NetAppDocsError += New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('Unable to capture SP/RLM/BMC information.', 'NO_SP_INFORMATION', [System.Management.Automation.ErrorCategory]::InvalidData, $systemData.Info.ClusterName)
                }
            }
        }

        #endregion Remote Management Configuration

        #region Feature Usage Data

        ##Unused Cmdlet: Get-NcFeatureUsage - This level of detail not needed (summary info in Get-NcFeatureUsageSummary)
        ##Unused Cmdlet: Get-NcFeatureStatus - (1.20+) Data available in Get-NcFeatureUsageSummary (Feature.Status)

        Write-Verbose -Message '- Gathering feature usage data -'
        $systemData | Add-Member -MemberType NoteProperty -Name Feature -Value ([pscustomobject]@{ })

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message 'Calling Get-NcFeatureUsageSummary'
            $systemData.Feature | Add-Member -MemberType NoteProperty -Name UsageSummary -Value (Get-NcFeatureUsageSummary -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Feature Usage Data

        #region Platform Capability Data

        ##Unused Cmdlet: Get-NcPlatformCap - (1.101+) Not needed for build docs (PlatformCapability)

        #endregion Platform Capability Data

        #region SNMP Configuration

        Write-Verbose -Message '- Gathering SNMP information -'

        Write-Verbose -Message 'Calling Get-NcSnmp'
        $systemData | Add-Member -MemberType NoteProperty -Name Snmp -Value (Get-NcSnmp -Controller $Controller -ErrorVariable +NetAppDocsError)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.120')) {
            $systemData.Snmp | Add-Member -MemberType NoteProperty -Name SnmpV3Enable -Value $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (system snmp show -fields snmpv3-enable-value (priv advanced))'
                    ((Invoke-NcSystemApi -Request '<system-cli><priv>advanced</priv><args><arg>set -showseparator ~;</arg><arg>system snmp show -fields snmpv3-enable-value</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText).Snmpv3EnableValue
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (system snmp show -fields snmpv3-enable-value (priv advanced))'
                    (Invoke-NcSsh -Command 'set -privilege advanced -confirmations off; set -showseparator ~; system snmp show -fields snmpv3-enable-value' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText).Snmpv3EnableValue
                }
            )
        }

        #endregion SNMP Configuration

        #region Storage Configuration

        ##Unused Cmdlet: Get-NcStorageInitiatorDiskPath - Not needed for build docs
        ##Unused Cmdlet: Get-NcStorageInitiatorError - Not needed for build docs
        ##Unused Cmdlet: Get-NcStorageInitiatorLoad - Not needed for build docs
        ##Unused Cmdlet: Get-NcStorageInitiatorPath - Not needed for build docs
        ##Unused Cmdlet: Get-NcStorageIscsiInitiator - (1.100+) Not currently needed for build docs
        ##Unused Cmdlet: Get-NcStorageShelfError - (1.30+) Data available in Get-NcStorageShelf
        ##Unused Cmdlet: Get-NcStorageShelfFirmwareUpdateInfo - (1.31+) Not needed for build docs
        ##Unused Cmdlet: Get-NcStorageShelfLocationLed - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcStorageShelfAcp - (1.31+) Not needed for build docs (Storage.ShelfAcp.Info - $node loop)
        ##Unused Cmdlet: Get-NcStoragePort - (1.100+) Not currently needed for build docs (Storage.Port)
        ##Unused Cmdlet: Get-NcStorageAdapterInfo - This currently throws an error on some adapters (Storage.Adapter.Info - using zapi instead)
        ##Unused Cmdlet: Get-NcStorageShelfDrawer - (1.120+) Currently not using the get-iter API, so excluding for now
        ##Unused Cmdlet: Get-NcStorageShelfDrawerPhy - (1.120+) Currently not using the get-iter API, so excluding for now
        ##Unused Cmdlet: Get-NcStorageShelfDrawerSlot - (1.120+) Currently not using the get-iter API, so excluding for now
        ##Unused Cmdlet: Get-NcStorageBridgeCoredump - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcStorageShelfPort - (1.160+) Currently using environment info for this

        Write-Verbose -Message '- Identifying system storage configuration -'
        $systemData | Add-Member -MemberType NoteProperty -Name Storage -Value ([pscustomobject]@{ })

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.31')) {
            Write-Verbose -Message '- Gathering ACP information -'
            $systemData.Storage | Add-Member -MemberType NoteProperty -Name ShelfAcp -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcStorageShelfAcpModule'
            $systemData.Storage.ShelfAcp | Add-Member -MemberType NoteProperty -Name Module -Value (Get-NcStorageShelfAcpModule -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        $storageConfiguration = $sysconfigV -split '\n' | Select-String -Pattern 'Node', 'System Storage Configuration:' -SimpleMatch
        if ($storageConfiguration) {
            $systemData.Storage | Add-Member -MemberType NoteProperty -Name Configuration -Value @( )

            foreach ($line in ($storageConfiguration | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })) {
                switch -Regex ($line) {
                    '^\s*Node:' {
                        $systemData.Storage.Configuration += [pscustomobject]@{
                            Node          = ($line -split ':')[1].Trim()
                            Configuration = $null
                        }

                        break
                    }
                    '^\s*System Storage Configuration:' {
                        ($systemData.Storage.Configuration | Select-Object -Last 1).Configuration = ($line -split ':')[1].Trim()

                        break
                    }
                }
            }
        }
        else {
            $systemData.Storage | Add-Member -MemberType NoteProperty -Name Configuration -Value 'NA'
        }

        Write-Verbose -Message '- Identifying system ACP connectivity information -'

        $systemAcpConnectivity = $sysconfigV -split '\n' | Select-String -Pattern 'Node', 'System ACP Connectivity:' -SimpleMatch
        if ($systemAcpConnectivity) {
            $systemData.Storage | Add-Member -MemberType NoteProperty -Name SystemAcpConnectivity -Value @( )

            foreach ($line in ($systemAcpConnectivity | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })) {
                switch -Regex ($line) {
                    '^\s*Node:' {
                        $systemData.Storage.SystemAcpConnectivity += [pscustomobject]@{
                            Node          = ($line -split ':')[1].Trim()
                            Configuration = $null
                        }

                        break
                    }
                    '^\s*System ACP Connectivity:' {
                        ($systemData.Storage.SystemAcpConnectivity | Select-Object -Last 1).Configuration = ($line -split ':')[1].Trim()

                        break
                    }
                    default { }
                }
            }
        }
        else {
            $systemData.Storage | Add-Member -MemberType NoteProperty -Name SystemAcpConnectivity -Value 'NA'
        }

        Write-Verbose -Message '- Gathering system storage adapter configuration -'
        $systemData.Storage | Add-Member -MemberType NoteProperty -Name Adapter -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcStorageAdapter'
        $adapterList = Get-NcStorageAdapter -Controller $Controller -ErrorVariable +NetAppDocsError

        $systemData.Storage.Adapter | Add-Member -MemberType NoteProperty -Name Info -Value @( )
        $adapterList | Where-Object { $_ } | ForEach-Object {
            Write-Verbose -Message "Calling Invoke-NcSystemApi (storage-adapter-get-adapter-info) $( $_.NodeName ):$( $_.AdapterName )"
            $systemData.Storage.Adapter.Info += (Invoke-NcSystemApi -Request "<storage-adapter-get-adapter-info><node-name>$( $_.NodeName )</node-name><adapter-name>$( $_.AdapterName )</adapter-name></storage-adapter-get-adapter-info>" -Controller $Controller -ErrorVariable +NetAppDocsError).results.'adapter-details'.'adapter-detail-info' | Convert-GenericXmlNode
        }

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message 'Calling Get-NcStorageBridge'
            $systemData.Storage | Add-Member -MemberType NoteProperty -Name Bridge -Value (Get-NcStorageBridge -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcStorageSwitch'
            $systemData.Storage | Add-Member -MemberType NoteProperty -Name Switch -Value (Get-NcStorageSwitch -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcStorageShelf'
            $systemData.Storage | Add-Member -MemberType NoteProperty -Name Shelf -Value (Get-NcStorageShelf -Controller $Controller -ErrorVariable +NetAppDocsError)

            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.110')) {
                $systemData.Storage | Add-Member -MemberType NoteProperty -Name ShelfDrawer -Value ([pscustomobject]@{ })

                Write-Verbose -Message 'Calling Invoke-NcSystemApi (storage-shelf-drawer-get-iter)'
                $systemData.Storage.ShelfDrawer | Add-Member -MemberType NoteProperty -Name Info -Value ((Invoke-NcSystemApi -Request '<storage-shelf-drawer-get-iter />' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'attributes-list'.'storage-shelf-drawer-info' | Convert-GenericXmlNode)
            }
        }

        #endregion Storage Configuration

        #region StoragePool Configuration

        ##Unused Cmdlet: Get-NcStoragePoolAvailableCapacity - (1.30+) Not currently needed for build docs (StoragePool.AvailableCapacity)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message '- Gathering StoragePool information -'
            $systemData | Add-Member -MemberType NoteProperty -Name StoragePool -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcStoragePool'
            $systemData.StoragePool | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcStoragePool -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcStoragePoolAggr'
            $systemData.StoragePool | Add-Member -MemberType NoteProperty -Name Aggr -Value (Get-NcStoragePoolAggr -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcStoragePoolDisk'
            $systemData.StoragePool | Add-Member -MemberType NoteProperty -Name Disk -Value (Get-NcStoragePoolDisk -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion StoragePool Configuration

        #region StorageArray Configuration

        ##Unused Cmdlet: Get-NcStorageArrayPort - Not needed for build docs (StorageArray.Port - wrap in try..catch)

        ##TOOLKIT - These cmdlets currently throw errors for null entries (zapi issue)
        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message '- Gathering StorageArray information -'
            $systemData | Add-Member -MemberType NoteProperty -Name StorageArray -Value ([pscustomobject]@{ })

            try {
                Write-Verbose -Message 'Calling Get-NcStorageArray'
                $systemData.StorageArray | Add-Member -MemberType NoteProperty -Name Profile -Value (Get-NcStorageArray -Controller $Controller -ErrorAction Stop)
            }
            catch {
                if ($_.Exception.Message -like "*entry doesn't exist*") {
                    $Global:Error.RemoveAt(0)
                }
                else {
                    $NetAppDocsError += $_
                }
            }

            if ($systemData.StorageArray.Profile) {
                try {
                    Write-Verbose -Message 'Calling Get-NcStorageArrayConfig'
                    $systemData.StorageArray | Add-Member -MemberType NoteProperty -Name Config -Value (Get-NcStorageArrayConfig -Controller $Controller -ErrorAction Stop)
                }
                catch {
                    if ($_.Exception.Message -like '*Object reference not set to an instance of an object*') {
                        $Global:Error.RemoveAt(0)
                    }
                    else {
                        $NetAppDocsError += $_
                    }
                }
            }
        }

        #endregion StorageArray Configuration

        #region Syslog (EMS) Data

        ##Unused Cmdlet: Get-NcEmsMailHistory - Not needed for build docs (Deprecated in 9.0)
        ##Unused Cmdlet: Get-NcEmsRoute - Not currently needed for build docs (significant amount of data returned)(Deprecated in 9.0)
        ##Unused Cmdlet: Get-NcEmsSnmpHistory - Not needed for build docs(Deprecated in 9.0)
        ##Unused Cmdlet: Get-NcEmsStatus - Not currently needed for build docs (significant amount of data returned)
        ##Unused Cmdlet: Get-NcEmsEventCatalog - (1.100+) Not currently needed for build docs (significant amount of data returned)
        ##Unused Cmdlet: Get-NcEmsEventNotificationHistory - (1.100+) Not currently needed for build docs (significant amount of data returned)
        ##Unused Cmdlet: Get-NcEmsConfig - Not needed for build docs (Ems.Config)

        Write-Verbose -Message '- Gathering EMS information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Ems -Value ([pscustomobject]@{ })

        if ($systemData.System.OntapiVersion -lt [version]::Parse('1.100')) {
            Write-Verbose -Message 'Calling Get-NcEmsDestination'
            $systemData.Ems | Add-Member -MemberType NoteProperty -Name Destination -Value (Get-NcEmsDestination -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.100')) {
            Write-Verbose -Message 'Calling Get-NcEmsEventFilter'
            $systemData.Ems | Add-Member -MemberType NoteProperty -Name EventFilter -Value (Get-NcEmsEventFilter -Controller $Controller -ErrorVariable +NetAppDocsError)

            $systemData.Ems | Add-Member -MemberType NoteProperty -Name EventNotification -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcEmsEventNotification'
            $systemData.Ems.EventNotification | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcEmsEventNotification -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcEmsEventNotificationDestination'
            $systemData.Ems.EventNotification | Add-Member -MemberType NoteProperty -Name Destination -Value (Get-NcEmsEventNotificationDestination -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        if ($Script:EmsMessageNames) {
            Write-Verbose -Message '- Gathering limited EMS messages -'
            $systemData.Ems | Add-Member -MemberType NoteProperty -Name Message -Value (Get-NcEmsMessage -MessageName $Script:EmsMessageNames -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Syslog (EMS) Data

        #region Environment Information

        ##Unused Cmdlet: Get-NcEnvironmentSensors - (1.21+) Not currently needed (may be useful for an HC table) (Environment.Sensors)

        #endregion Environment Information

        #region Option Information

        Write-Verbose -Message '- Gathering options -'

        Write-Verbose -Message 'Calling Get-NcOption'
        $systemData | Add-Member -MemberType NoteProperty -Name Option -Value @( (Get-NcOption -Controller $Controller -ErrorVariable +NetAppDocsError) )

        $additionalOptions = @(
            'sis.max_active_ops'
            'sis.idedup_allow_non_aff_hya'
            'flexscale.flash_pool_caching'
            # The following properties must be explicitly called in 8.3.2 and below
            'raid.timeout'
            'raid.mirror_read_plex_pref'
        )

        $optionsRegex = '(?m)^Node[:]\s+(?<Node>.+?)(?:\s+Last login time[:]\s\S+\s\S+)?\s+(?<Name>.+?)\s+(?<Value>.+?)(?:\s+\(.*)?\s*$'

        foreach ($additionalOption in $additionalOptions) {
            if (-not ($systemData.Option | Where-Object { $_.Name -eq $additionalOption })) {
                $optionsCliOutput = $(
                    if ($useSystemCli) {
                        Write-Verbose -Message "Calling Invoke-NcSystemApi (system node run -node * -command options $additionalOption)"
                        (Invoke-NcSystemApi -Request "<system-cli><args><arg>system node run -node * -command options $additionalOption</arg></args></system-cli>" -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'
                    }
                    else {
                        Write-Verbose -Message "Calling Invoke-NcSsh (system node run -node * -command options $additionalOption)"
                        Invoke-NcSsh -Command "system node run -node * -command options $additionalOption" -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false
                    }
                )

                [regex]::Matches($optionsCliOutput, $optionsRegex) | ForEach-Object {
                    $option = New-Object -TypeName DataONTAP.C.Types.Options.OptionInfo
                    $option.ClusterConstraint = 'none'
                    $option.PrivLevel = 'admin'

                    $option.Vserver, $option.Name, $option.Value = $_.Groups[1, 2, 3].Value

                    $systemData.Option += $option
                }
            }
        }

        #endregion Option Information

        #region Aggregate Status

        ##Unused Cmdlet: Get-NcAggrNodeInfo - Not needed for build docs
        ##Unused Cmdlet: Get-NcAggrRelocation - Not needed for build docs
        ##Unused Cmdlet: Get-NcAggrResyncOptions - Not currently needed for build docs
        ##Unused Cmdlet: Get-NcAggrScrub - Not needed for build docs
        ##Unused Cmdlet: Get-NcAggrStatus - (1.30+) Not needed for build docs (Aggr.Status)
        ##Unused Cmdlet: Get-NcAggrVerify - (1.30+) Not CURRENTLY needed for build docs (investigate)
        ##Unused Cmdlet: Get-NcAggrSpare - (1.30+) Not CURRENTLY needed (check if data exists elsewhere)
        ##Unused Cmdlet: Get-NcAggrAutobalanceAggrState - (1.30+) Not needed for build docs (Aggr.Autobalance.AggrState)
        ##Unused Cmdlet: Get-NcAggrAutobalanceNotification - (1.30+) Not needed for build docs (Aggr.Autobalance.Notification)
        ##Unused Cmdlet: Get-NcAggrAutobalanceVolumeState - (1.30+) Not needed for build docs (Aggr.Autobalance.VolumeState)
        ##Unused Cmdlet: Get-NcAggrObjectStoreConfigProvider - (1.120+) Not currently needed for build docs
        ##Unused Cmdlet: Get-NcAggrObjectStoreVolBtuuids - (1.120+) Not currently needed for build docs (Aggr.ObjectStore.VolBtUuids)
        ##Unused Cmdlet: Get-NcAggrEfficiencyCumulated - (1.130+) Not currently needed for build docs (throws error on MCC)

        Write-Verbose -Message '- Gathering detailed aggregate information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Aggr -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcAggr'
        $systemData.Aggr | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcAggr -Controller $Controller -ErrorVariable +NetAppDocsError)

        $systemData.Aggr | Add-Member -MemberType NoteProperty -Name Option -Value @( )

        foreach ($aggr in ($systemData.Aggr.Info | Where-Object { $_ -and ($_.State -ne 'unknown' -and $_.State -ne 'remote_cluster') })) {
            Write-Verbose -Message "Calling Get-NcAggrOption (Aggr: $( $aggr.Name ))"
            $systemData.Aggr.Option += (Get-NcAggrOption -Name $aggr.Name -Controller $Controller -Hashtable -ErrorVariable +NetAppDocsError).Value | Add-Member -MemberType NoteProperty -Name AggregateName -Value $aggr.Name -PassThru
        }

        $aggrSnapSched = $(
            if ($useSystemCli) {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (system node run -node * -command snap sched -A)'
                (Invoke-NcSystemApi -Request '<system-cli><args><arg>system node run -node * -command snap sched -A</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'
            }
            else {
                Write-Verbose -Message 'Calling Invoke-NcSsh (system node run -node * -command snap sched -A)'
                Invoke-NcSsh -Command 'system node run -node * -command snap sched -A' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false
            }
        )

        $aggrSnapList = $(
            if ($useSystemCli) {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (system node run -node * -command snap list -A)'
                (Invoke-NcSystemApi -Request '<system-cli><args><arg>system node run -node * -command snap list -A</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'
            }
            else {
                Write-Verbose -Message 'Calling Invoke-NcSsh (system node run -node * -command snap list -A)'
                Invoke-NcSsh -Command 'system node run -node * -command snap list -A' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false
            }
        )

        Write-Verbose -Message 'Calling Get-AggrSnapInformation'
        $systemData.Aggr | Add-Member -MemberType NoteProperty -Name Snapshot -Value (Get-AggrSnapInformation -SnapSched $aggrSnapSched -SnapList $aggrSnapList)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message 'Calling Get-NcAggrSpace'
            $systemData.Aggr | Add-Member -MemberType NoteProperty -Name Space -Value (Get-NcAggrSpace -Controller $Controller -ErrorVariable +NetAppDocsError)

            $systemData.Aggr | Add-Member -MemberType NoteProperty -Name Autobalance -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcAggrAutobalance'
            $systemData.Aggr.Autobalance | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcAggrAutobalance -Controller $Controller -ErrorVariable +NetAppDocsError)

            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.100')) {
                Write-Verbose -Message 'Calling Get-NcAggrEfficiency'
                $systemData.Aggr | Add-Member -MemberType NoteProperty -Name Efficiency -Value (Get-NcAggrEfficiency -Controller $Controller -ErrorVariable +NetAppDocsError)

                if ($systemData.System.OntapiVersion -ge [version]::Parse('1.120')) {
                    $systemData.Aggr | Add-Member -MemberType NoteProperty -Name ObjectStore -Value ([pscustomobject]@{ })

                    Write-Verbose -Message 'Calling Get-NcAggrObjectStore'
                    $systemData.Aggr.ObjectStore | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcAggrObjectStore -Controller $Controller -ErrorVariable +NetAppDocsError)

                    Write-Verbose -Message 'Calling Get-NcAggrObjectStoreConfig'
                    $systemData.Aggr.ObjectStore | Add-Member -MemberType NoteProperty -Name Config -Value (Get-NcAggrObjectStoreConfig -Controller $Controller -ErrorVariable +NetAppDocsError)

                    if ($systemData.System.OntapiVersion -ge [version]::Parse('1.130')) {
                        $systemData.Aggr | Add-Member -MemberType NoteProperty -Name EfficiencyConfig -Value $(
                            if ($useSystemCli) {
                                Write-Verbose -Message 'Calling Invoke-NcSystemApi (storage aggregate efficiency show)'
                                (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>storage aggregate efficiency show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                            }
                            else {
                                Write-Verbose -Message 'Calling Invoke-NcSsh (storage aggregate efficiency show)'
                                Invoke-NcSsh -Command 'set -showseparator ~; set -showallfields true; storage aggregate efficiency show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                            }
                        )
                    }
                }
            }
        }

        #endregion Aggregate Status

        #region Flash Device Details

        ##Unused Cmdlet: Get-NcFlashThreshold - Not needed for build docs (Flash.Threshold)

        Write-Verbose -Message '- Gathering flash device data -'
        $systemData | Add-Member -MemberType NoteProperty -Name Flash -Value ([pscustomobject]@{ })

        ##TOOLKIT/ZAPI BURT (Since NcFlashDevice cmdlet throws a null error if no cards are present on any node, we have to loop through each node instead)
        $systemData.Flash | Add-Member -MemberType NoteProperty -Name Device @( )
        foreach ($node in ($systemData.Node | Where-Object { $_ })) {
            try {
                Write-Verbose -Message "Calling Get-NcFlashDevice (Node: $( $node.Node ))"
                $systemData.Flash.Device += Get-NcFlashDevice -Node $node.Node -Controller $Controller -ErrorAction Stop
            }
            catch {
                if ($_.Exception.Message -like '*No cards in the system*') {
                    $Global:Error.RemoveAt(0)
                }
                else {
                    $NetAppDocsError += $_
                }
            }
        }

        #endregion Flash Device Details

        #region FlexCache Data

        ##Unused Cmdlet: Get-NcFlexcacheOption - Deprecated functionality
        ##Unused Cmdlet: Get-NcFlexcacheUsage - (1.150+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFlexcacheConnectedCache - (1.150+) Not currently needed for build docs

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Gathering FlexCache data -'
            $systemData | Add-Member -MemberType NoteProperty -Name Flexcache -Value ([pscustomobject]@{ })

            if ($systemData.System.OntapiVersion -lt [version]::Parse('1.30')) {
                Write-Verbose -Message 'Calling Get-NcFlexcache'
                $systemData.Flexcache | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcFlexcache -Controller $Controller -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message 'Calling Get-NcFlexcachePolicy'
                $systemData.Flexcache | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-NcFlexcachePolicy -Controller $Controller -ErrorVariable +NetAppDocsError)
            }

            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.150')) {
                Write-Verbose -Message 'Calling Get-NcFlexcache'
                $systemData.Flexcache | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcFlexcache -Controller $Controller -ErrorVariable +NetAppDocsError)
            }
        }

        #endregion FlexCache Data

        #region Disk Information

        ##Unused Cmdlet: Get-NcDiskOwner - Info in Get-NcDisk
        ##Unused Cmdlet: Get-NcDiskOwnerFiler - Info in Get-NcDisk

        Write-Verbose -Message '- Gathering physical disk information -'

        Write-Verbose -Message 'Calling Get-NcDisk'
        $systemData | Add-Member -MemberType NoteProperty -Name Disk -Value (Get-NcDisk -Controller $Controller -ErrorVariable +NetAppDocsError)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            $systemData | Add-Member -MemberType NoteProperty -Name PartitionDisk -Value $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (storage disk partition show (priv diag))'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -privilege diagnostic -confirmations off;</arg><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>storage disk partition show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Where-Object { $_ -notmatch '^Error[:].+' } | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (storage disk partition show (priv diag))'
                    Invoke-NcSsh -Command 'set -privilege diagnostic -confirmations off; set -showseparator ~; set -showallfields true; storage disk partition show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Where-Object { $_ -notmatch '^Error[:].+' } | Convert-SystemCliFieldDelimitedText
                }
            )
        }

        #endregion Disk Information

        #region Disk Option Information

        $systemData | Add-Member -MemberType NoteProperty -Name DiskOption -Value $(
            if ($useSystemCli) {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (storage disk option show)'
                (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>storage disk option show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
            }
            else {
                Write-Verbose -Message 'Calling Invoke-NcSsh (storage disk option show)'
                Invoke-NcSsh -Command 'set -showseparator ~; set -showallfields true; storage disk option show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
            }
        )

        #endregion Disk Option Information

        #region Disk Encrypt Information

        ##Unused Cmdlet: Get-NcDiskEncryptStatus - (1.31+) - Not needed for build docs

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.31')) {
            Write-Verbose -Message '- Gathering disk encrypt information -'
            $systemData | Add-Member -MemberType NoteProperty -Name DiskEncrypt -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcDiskEncrypt'
            $systemData.DiskEncrypt | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcDiskEncrypt -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Disk Encrypt Information

        #region FC/FCP Configuration

        ##Unused Cmdlet: Get-NcFcpAdapterStatistics - Not needed for build docs
        ##Unused Cmdlet: Get-NcFcpNodeName - Info exists in Get-NcFcpInterface
        ##Unused Cmdlet: Get-NcFcPortLinkState - Info exists in Get-NcFcPortLinkState
        ##Unused Cmdlet: Get-NcFcpPortName - Info exists in Get-NcFcpInterface
        ##Unused Cmdlet: Get-NcFcpPingIgroup - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFcpPingInitiator - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFcpPortNameAlias - Not needed for build docs (Fcp.PortNameAlias)
        ##Unused Cmdlet: Get-NcFcpNameServer = (1.120+) Not currently needed for build docs (Fcp.NameServer)
        ##Unused Cmdlet: Get-NcFcpTopology = (1.120+) Not currently needed for build docs (Fcp.Topology.Info)
        ##Unused Cmdlet: Get-NcFcpTopologyPort = (1.120+) Not currently needed for build docs (Fcp.Topology.Port)
        ##Unused Cmdlet: Get-NcFcpZone = (1.120+) Not currently needed for build docs (Fcp.Zone)

        Write-Verbose -Message '- Gathering FC adapter data -'

        Write-Verbose -Message 'Calling Get-NcFcAdapter'
        #BURT 613219
        if (($systemData.Node | Where-Object { $_.NodeModel -match 'FAS2[0|2][2|4|5]0' } | Measure-Object).Count -gt 0) {
            #Known error, so do not log it
            $systemData | Add-Member -MemberType NoteProperty -Name FcAdapter -Value (Get-NcFcAdapter -Controller $Controller -ErrorAction SilentlyContinue)
        }
        else {
            $systemData | Add-Member -MemberType NoteProperty -Name FcAdapter -Value (Get-NcFcAdapter -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #Not sure if we need to test for this in cmode
        Write-Verbose -Message '- Gathering FCP configuration data -'
        $systemData | Add-Member -MemberType NoteProperty -Name Fcp -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcFcpService'
        $systemData.Fcp | Add-Member -MemberType NoteProperty -Name Service -Value (Get-NcFcpService -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcFcpAdapter'
        $systemData.Fcp | Add-Member -MemberType NoteProperty -Name Adapter -Value (Get-NcFcpAdapter -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcFcpInitiator'
        $systemData.Fcp | Add-Member -MemberType NoteProperty -Name Initiator -Value (Get-NcFcpInitiator -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcFcpInterface'
        $systemData.Fcp | Add-Member -MemberType NoteProperty -Name Interface -Value (Get-NcFcpInterface -Controller $Controller -ErrorVariable +NetAppDocsError)

        if ($systemData.System.OntapiVersion -lt [version]::Parse('1.30')) {
            $fcpStats = $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (fcp stats (priv diag))'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -privilege diagnostic -confirmations off;</arg><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>fcp stats</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (fcp stats (priv diag))'
                    Invoke-NcSsh -Command 'set -privilege diagnostic -confirmations off; set -showseparator ~; set -showallfields true; fcp stats' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            )

            if ($fcpStats) {
                foreach ($fcpAdapter in ($systemData.Fcp.Adapter | Where-Object { $_ })) {
                    if (($sfpDetails = $fcpStats | Where-Object { $_.Node -ceq $fcpAdapter.Node -and $_.Adapter -eq $fcpAdapter.Adapter })) {
                        $fcpAdapter.SfpFormfactor = $sfpDetails.SfpFormfactor
                        $fcpAdapter.SfpVendorName = $sfpDetails.SfpVendorName
                        $fcpAdapter.SfpPartNumber = $sfpDetails.SfpPartNumber
                        $fcpAdapter.SfpRev = $sfpDetails.SfpRev
                        $fcpAdapter.SfpSerialNumber = $sfpDetails.SfpSerialNumber
                        $fcpAdapter.SfpFcSpeedcapabilities = $sfpDetails.SfpFcSpeedCapabilities
                        $fcpAdapter.SfpVendorOui = $sfpDetails.SfpVendorOui
                        $fcpAdapter.SfpWavelength = $sfpDetails.SfpWaveLength
                        $fcpAdapter.SfpDateCode = $sfpDetails.SfpDateCode
                        $fcpAdapter.IsSfpOpticalTransceiverValid = $sfpDetails.IsSfpOpticalTransceiverValid
                        $fcpAdapter.SfpConnector = $sfpDetails.SfpConnector
                        $fcpAdapter.SfpEncoding = $sfpDetails.SfpEncoding
                        $fcpAdapter.IsSfpDiagnosticsInternallyCalibrated = $sfpDetails.IsSfpDiagnosticsInternallyCalibrated
                        $fcpAdapter.SfpRxPower = $sfpDetails.SfpRxPower -replace '\s+\(.+\)'
                        $fcpAdapter.IsSfpRxPowerInRange = $sfpDetails.IsSfpRxPowerInRange
                        $fcpAdapter.SfpTxPower = $sfpDetails.SfpTxPower -replace '\s+\(.+\)'
                        $fcpAdapter.IsSfpTxPowerInRange = $sfpDetails.IsSfpTxPowerInRange
                    }
                }
            }
        }

        #endregion FC/FCP Configuration

        #region iSCSI Configuration

        ##Unused Cmdlet: Get-NcIscsiConnection - Not needed for build docs
        ##Unused Cmdlet: Get-NcIscsiInitiatorAuth - Not needed for build docs ??
        ##Unused Cmdlet: Get-NcIscsiNodeName - Info in Get-NcIscsiService
        ##Unused Cmdlet: Get-NcIscsiSession - Not needed for build docs
        ##Unused Cmdlet: Get-NcIscsiStatistics - Not needed for build docs
        ##Unused Cmdlet: Get-NcIscsiTargetalias - Info in Get-NcIscsiService
        ##Unused Cmdlet: Get-NcIscsiInterfaceAccess - Not needed for build docs (Iscsi.Interface.Access)

        Write-Verbose -Message '- Gathering iSCSI configuration data -'
        $systemData | Add-Member -MemberType NoteProperty -Name Iscsi -Value ([pscustomobject]@{ })
        $systemData.Iscsi | Add-Member -MemberType NoteProperty -Name Interface -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcIscsiInterface'
        $systemData.Iscsi.Interface | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcIscsiInterface -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcIscsiService'
        $systemData.Iscsi | Add-Member -MemberType NoteProperty -Name Service -Value (Get-NcIscsiService -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcIscsiInitiator'
        $systemData.Iscsi | Add-Member -MemberType NoteProperty -Name Initiator -Value (Get-NcIscsiInitiator -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcIscsiTargetPortalGroup'
        $systemData.Iscsi | Add-Member -MemberType NoteProperty -Name TargetPortalGroup -Value (Get-NcIscsiTargetPortalGroup -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion iSCSI configuration

        #region iSNS Configuration

        ##Unused Cmdlet: Get-NcIsns - Not needed for build docs ??

        #endregion iSNS Configuration

        #region iGroup Configuration

        Write-Verbose -Message '- Gathering iGroup information -'

        ##Unused Cmdlet: Get-NcIgroupOsTypes - (1.31+) Not needed for build docs

        Write-Verbose -Message 'Calling Get-NcIgroup'
        $systemData | Add-Member -MemberType NoteProperty -Name Igroup -Value (Get-NcIgroup -Controller $Controller -ErrorVariable +NetAppDocsError)

        $igroupInitiatorDetails = $(
            if ($useSystemCli) {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (lun igroup show -fields init-details (priv diag))'
                (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -privilege diagnostic -confirmations off;</arg><arg>set -showseparator ~;</arg><arg>lun igroup show -fields init-details</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText | Select-Object -ExcludeProperty InitDetails -Property *, @{
                    Name       = 'Initiators'
                    Expression = {
                        @(
                            $_.InitDetails -replace '"' -split '\s*,\s*' | ForEach-Object {
                                [pscustomobject]@{
                                    InitiatorName = $_ -replace '\s+\(.+\)'
                                    IsLoggedIn    = $_ -like '*(logged in)'
                                }
                            }
                        )
                    }
                }
            }
            else {
                Write-Verbose -Message 'Calling Invoke-NcSsh (lun igroup show -fields init-details (priv diag))'
                Invoke-NcSsh -Command 'set -privilege diagnostic -confirmations off; set -showseparator ~; lun igroup show -fields init-details' -Controller $Controller -ErrorVariable +NetAppDocsError | Convert-SystemCliFieldDelimitedText | Select-Object -ExcludeProperty InitDetails -Property *, @{
                    Name       = 'Initiators'
                    Expression = {
                        @(
                            $_.InitDetails -replace '"' -split '\s*,\s*' | ForEach-Object {
                                [pscustomobject]@{
                                    InitiatorName = $_ -replace '\s+\(.+\)'
                                    IsLoggedIn    = $_ -like '*(logged in)'
                                }
                            }
                        )
                    }
                }
            }
        )

        foreach ($igroup in ($systemData.Igroup | Where-Object { $_.Initiators })) {
            $initiatorDetails = $igroupInitiatorDetails | Where-Object { $_.Vserver -ceq $igroup.Vserver -and $_.Igroup -ceq $igroup.InitiatorGroupName }

            foreach ($initiator in ($igroup.Initiators | Where-Object { -not [string]::IsNullOrWhiteSpace($_.InitiatorName) })) {
                $initiator | Add-Member -MemberType NoteProperty -Name IsLoggedIn -Value $(
                    if ($initiatorDetails) {
                        ($initiatorDetails.Initiators | Where-Object { $_.InitiatorName -ceq $initiator.InitiatorName }).IsLoggedIn
                    }
                )
            }
        }

        #endregion iGroup Information

        #region PortSet Configuration

        Write-Verbose -Message '- Gathering PortSet information -'

        Write-Verbose -Message 'Calling Get-NcPortSet'
        $systemData | Add-Member -MemberType NoteProperty -Name PortSet -Value (Get-NcPortset -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion PortSet Configuration

        #region Network Details

        ##Unused Cmdlet: Get-NcNetPlacementCache - (1.20+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNetSanInterfacePlacement - (1.20+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNetArpEntry - (1.30+) Not sure if needed for build docs
        ##Unused Cmdlet: Get-NcNetPortHealthMonitor - (1.110+) API not yet available in 9.1X5
        ##Unused Cmdlet: Get-NcNetFirewall - (1.30+) Not needed for build docs (Net.Firewall)
        ##Unused Cmdlet: Get-NcNetActiveRoutes - (1.30+) Not needed for build docs (Net.ActiveRoutes)
        ##Unused Cmdlet: Get-NcNetReceivedWindowSize - (1.21+) Not needed for build docs (Net.receivedWindowSize)
        ##Unused Cmdlet: Get-NcNetArpActiveEntry - (1.30+) Not needed for build docs (Net.ArpActiveEntry)
        ##Unused Cmdlet: Get-NcNetNdpActiveNeighbor - (1.30+) Not needed for build docs (Net.NdpActiveNeighbor)
        ##Unused Cmdlet: Get-NcNetNdpDefaultRouter - (1.30+) Not needed for build docs (Net.NdpDefaultRouter)
        ##Unused Cmdlet: Get-NcNetNdpNeighbor - (1.30+) Not needed for build docs (Net.NdpNeighbor)
        ##Unused Cmdlet: Get-NcNetNdpPrefix - (1.30+) Not needed for build docs (Net.NdpPrefix)
        ##Unused Cmdlet: Get-NcNetDataLifCapacity - (1.100+) Not needed for build docs (Net.DataLifCapacity)
        ##Unused Cmdlet: Get-NcNetDataLifCapacityDetails - (1.100+) Not needed for build docs (Net.DataLifCapacityDetails)
        ##Unused Cmdlet: Get-NcNetQosMarking - (1.100+) Not needed for build docs (Net.QosMarking)
        ##Unused Cmdlet: Get-NcNetOptionsHypervisorUplinkMonitoring - (1.120+) Not sure what this is yet
        ##Unused Cmdlet: Get-NcNetTuningIcmp - (1.120+) Not currently needed for build docs (per node) (Net.Tuning.Icmp)
        ##Unused Cmdlet: Get-NcNetTuningTcp - (1.120+) Not currently needed for build docs (per node) (Net.Tuning.Tcp)
        ##Unused Cmdlet: Get-NcNetTuningIcmp6 - (1.130+) Not currently needed for build docs (per node) (Net.Tuning.Icmp6)
        ##Unused Cmdlet: Get-NcNetSubnetsForBroadcastDomain - (1.140+) Not currently needed for build docs

        Write-Verbose -Message '- Gathering network details -'
        $systemData | Add-Member -MemberType NoteProperty -Name Net -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcNetPort'
        $systemData.Net | Add-Member -MemberType NoteProperty -Name Port -Value (Get-NcNetPort -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcNetPortIfgrp'
        $systemData.Net | Add-Member -MemberType NoteProperty -Name PortIfgrp -Value (Get-NcNetPortIfgrp -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcNetPortVlan'
        $systemData.Net | Add-Member -MemberType NoteProperty -Name PortVlan -Value (Get-NcNetPortVlan -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcNetInterface'
        $systemData.Net | Add-Member -MemberType NoteProperty -Name Interface -Value (Get-NcNetInterface -Controller $Controller -ErrorVariable +NetAppDocsError)

        $netInterfaceCli = $(
            if ($useSystemCli) {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (network interface show (priv advanced))'
                (Invoke-NcSystemApi -Request '<system-cli><priv>advanced</priv><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>network interface show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
            }
            else {
                Write-Verbose -Message 'Calling Invoke-NcSsh (network interface show (priv advanced))'
                Invoke-NcSsh -Command 'set -privilege advanced -confirmations off; set -showseparator ~; set -showallfields true; network interface show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
            }
        )

        foreach ($lif in ($systemData.Net.Interface | Where-Object { $_ })) {
            $netInterface = $netInterfaceCli | Where-Object { $_.Vserver -ceq $lif.Vserver -and $_.Lif -ceq $lif.InterfaceName }

            $lif | Add-Member -MemberType NoteProperty -Name NumericId -Value $netinterface.NumericId -Force
            $lif | Add-Member -MemberType NoteProperty -Name ServicePolicy -Value $netinterface.ServicePolicy -Force
        }

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.31')) {
            Write-Verbose -Message 'Calling Get-NcNetDdns'
            $systemData.Net | Add-Member -MemberType NoteProperty -Name Ddns -Value (Get-NcNetDdns -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        Write-Verbose -Message 'Calling Get-NcNetDns'
        $systemData.Net | Add-Member -MemberType NoteProperty -Name Dns -Value (Get-NcNetDns -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcNetDnsHost'
        $systemData.Net | Add-Member -MemberType NoteProperty -Name DnsHost -Value (Get-NcNetDnsHost -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Identifying Net.DnsZone data'
        $systemData.Net | Add-Member -MemberType NoteProperty -Name DnsZone -Value ($netInterfaceCli | Select-Object -Property Vserver, Lif, AllowLbMigrate, LbWeight)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message 'Calling Get-NcNetRoute'
            $systemData.Net | Add-Member -MemberType NoteProperty -Name Route -Value (Get-NcNetRoute -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcNetRouteLif'
            $systemData.Net | Add-Member -MemberType NoteProperty -Name RouteLif -Value (Get-NcNetRouteLif -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcNetSubnet'
            $systemData.Net | Add-Member -MemberType NoteProperty -Name Subnet -Value (Get-NcNetSubnet -Controller $Controller -ErrorVariable +NetAppDocsError)
        }
        else {
            $systemData.Net | Add-Member -MemberType NoteProperty -Name RoutingGroup -Value $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (network routing-groups show)'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>network routing-groups show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (network routing-groups show)'
                    Invoke-NcSsh -Command 'set -showseparator ~; set -showallfields true; network routing-groups show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            )

            Write-Verbose -Message 'Calling Get-NcNetRoutingGroupRoute'
            $systemData.Net | Add-Member -MemberType NoteProperty -Name RoutingGroupRoute -Value (Get-NcNetRoutingGroupRoute -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message 'Calling Get-NcNetFailoverGroup'
            $systemData.Net | Add-Member -MemberType NoteProperty -Name FailoverGroup -Value (Get-NcNetFailoverGroup -Controller $Controller -ErrorVariable +NetAppDocsError)
        }
        else {
            $systemData.Net | Add-Member -MemberType NoteProperty -Name FailoverGroup -Value $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (network interface failover-groups show)'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>network interface failover-groups show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (network interface failover-groups show)'
                    Invoke-NcSsh -Command 'set -showseparator ~; set -showallfields true; network interface failover-groups show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            )
        }

        $systemData.Net | Add-Member -MemberType NoteProperty -Name FailoverTarget -Value $(
            if ($useSystemCli) {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (network interface show -failover)'
                (Invoke-NcSystemApi -Request '<system-cli><args><arg>network interface show -failover</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-FailoverTarget
            }
            else {
                Write-Verbose -Message 'Calling Invoke-NcSsh (network interface show -failover)'
                Invoke-NcSsh -Command 'network interface show -failover' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-FailoverTarget
            }
        )

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Calling Get-NcNetOption'
            $systemData.Net | Add-Member -MemberType NoteProperty -Name Option -Value (Get-NcNetOption -Controller $Controller -ErrorVariable +NetAppDocsError)

            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.21')) {
                Write-Verbose -Message 'Calling Get-NcNetDeviceDiscovery'
                $systemData.Net | Add-Member -MemberType NoteProperty -Name DeviceDiscovery -Value (Get-NcNetDeviceDiscovery -Controller $Controller -ErrorVariable +NetAppDocsError)

                if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
                    Write-Verbose -Message 'Calling Get-NcNetFirewallPolicy'
                    $systemData.Net | Add-Member -MemberType NoteProperty -Name FirewallPolicy -Value (Get-NcNetFirewallPolicy -Controller $Controller -ErrorVariable +NetAppDocsError)

                    Write-Verbose -Message 'Calling Get-NcNetIpspace'
                    $systemData.Net | Add-Member -MemberType NoteProperty -Name Ipspace -Value (Get-NcNetIpspace -Controller $Controller -ErrorVariable +NetAppDocsError)

                    Write-Verbose -Message 'Calling Get-NcNetPortBroadcastDomain'
                    $systemData.Net | Add-Member -MemberType NoteProperty -Name PortBroadcastDomain -Value (Get-NcNetPortBroadcastDomain -Controller $Controller -ErrorVariable +NetAppDocsError)

                    if ($systemData.System.OntapiVersion -ge [version]::Parse('1.150')) {
                        $systemData.Net | Add-Member -MemberType NoteProperty -Name ServicePolicy -Value ([pscustomobject]@{ })

                        Write-Verbose -Message 'Calling Invoke-NcSystemApi (net-interface-service-policy-get-iter)'
                        $systemData.Net.ServicePolicy | Add-Member -MemberType NoteProperty -Name Info -Value ((Invoke-NcSystemApi -Request '<net-interface-service-policy-get-iter />' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'attributes-list'.'net-interface-service-policy-info' | Convert-GenericXmlNode)

                        Write-Verbose -Message 'Calling Invoke-NcSystemApi (net-interface-service-policy-entry-get-iter)'
                        $systemData.Net.ServicePolicy | Add-Member -MemberType NoteProperty -Name Entry -Value ((Invoke-NcSystemApi -Request '<net-interface-service-policy-entry-get-iter />' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'attributes-list'.'net-interface-service-policy-entry-info' | Convert-GenericXmlNode)
                    }
                }
            }
        }

        #endregion Network Details

        #region Shelf Details

        Write-Verbose -Message '- Gathering shelf data -'
        $systemData | Add-Member -MemberType NoteProperty -Name Shelf -Value (
            [pscustomobject]@{
                Info        = @( )
                Environment = @( )
                Bay         = @( )
            }
        )

        #Note: This will throw an error for an unhealthy node. Currently, this is desired. (Use $systemData.NodeInfo and $node.SystemName if we do not want this error).
        foreach ($node in ($systemData.Node | Where-Object { $_ -and $_.NodeModel -notmatch '^(?:CB|CD|DO|FD)vM\d{2,}|SIMBOX' })) {
            Write-Verbose -Message "Calling Get-NcShelf (Node: $( $node.Node ))"
            $systemData.Shelf.Info += Get-NcShelf -NodeName $node.Node -Controller $Controller -ErrorVariable +NetAppDocsError

            Write-Verbose -Message "Calling Get-NcShelfEnvironment (Node: $( $node.Node ))"
            $systemData.Shelf.Environment += Get-NcShelfEnvironment -NodeName $node.Node -Controller $Controller -ErrorVariable +NetAppDocsError

            Write-Verbose -Message "Calling Get-NcShelfBay (Node: $( $node.Node ))"
            $systemData.Shelf.Bay += Get-NcShelfBay -NodeName $node.Node -Controller $Controller -ErrorVariable +NetAppDocsError
        }

        #endregion Shelf Details

        #region Tape/MC Details

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message '- Gathering tape/MC device information -'
            $systemData | Add-Member -MemberType NoteProperty -Name TapeMc -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcTapeMc'
            $systemData.TapeMc | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcTapeMc -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Tape/MC Details

        #region Vserver Information

        ##Unused Cmdlet: Get-NcVserverConfigDiff - (1.30+) Not CURRENTLY needed for build docs
        ##Unused Cmdlet: Get-NcVserverAggr - (1.100+) Info in Get-NcVserver

        Write-Verbose -Message '- Gathering Vserver information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Vserver -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcVserver'
        $systemData.Vserver | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcVserver -Controller $Controller -ErrorVariable +NetAppDocsError)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.31')) {
            Write-Verbose -Message '- Gathering login banner/MOTD information -'

            Write-Verbose -Message 'Calling Get-NcVserverLoginBanner'
            $systemData.Vserver | Add-Member -MemberType NoteProperty -Name LoginBanner -Value (Get-NcVserverLoginBanner -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcVserverMotd'
            $systemData.Vserver | Add-Member -MemberType NoteProperty -Name Motd -Value (Get-NcVserverMotd -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Vserver Information

        #region Vserver Peer Information

        ##Unused Cmdlet: Get-NcVserverPeerTransition - (1.20+) Not needed for build docs (Vserver.Peer.Transition)
        ##Unused Cmdlet: Get-NcVserverPeerPermission - (1.120+) Not currently needed for build docs (Vserver.Peer.Permissions)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Gathering Vserver Peer information -'
            $systemData.Vserver | Add-Member -MemberType NoteProperty -Name Peer -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcVserverPeer'
            $systemData.Vserver.Peer | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcVserverPeer -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Vserver Peer Information

        #region Vserver Migration Information

        ##Unused Cmdlet: Get-NcVserverMigrate - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcVserverMigrateVolume - (1.140+) Not needed for build docs

        #endregion Vserver Migration Information

        #region Volume Configuration

        ##Unused Cmdlet: Get-NcVolAutosize - Info exists in Get-NcVol
        ##Unused Cmdlet: Get-NcVolCloneSplit - Not needed for build docs
        ##Unused Cmdlet: Get-NcVolCloneSplitEstimate - Not needed for build docs
        ##Unused Cmdlet: Get-NcVolContainer - Info exists in Get-NcAggr
        ##Unused Cmdlet: Get-NcVolLanguage - Info exists in Get-NcVol
        ##Unused Cmdlet: Get-NcVolLimit - Not currently needed (Infinite Volumes)
        ##Unused Cmdlet: Get-NcVolAutobalance - (1.30+) Not currently needed (Infinite Volumes)
        ##Unused Cmdlet: Get-NcVolMove - Not needed for build docs
        ##Unused Cmdlet: Get-NcVolMoveTargetAggr - Not needed for build docs
        ##Unused Cmdlet: Get-NcVolRoot - Info exists in Get-NcVol (& Get-NcVserver)
        ##Unused Cmdlet: Get-NcVolSize - Info exists in Get-NcVol
        ##Unused Cmdlet: Get-NcVolStorageService - Not needed for build docs ?? (not sure what this is)
        ##Unused Cmdlet: Get-NcVolFlexgroupAggrLayout - (1.110+) Not needed for build docs ?? (API not found?)
        ##Unused Cmdlet: Get-NcVolFlexgroupExpandLimits - (1.110+) Not needed for build docs ?? (API not found?)
        ##Unused Cmdlet: Get-NcVolFlexgroupLimits - (1.110+) Not needed for build docs ??
        ##Unused Cmdlet: Get-NcVolFlexgroupResizeLimits - (1.110+) Not needed for build docs ??
        ##Unused Cmdlet: Get-NcVolOption - Not needed for build docs (Vol.Option - loop through each cluster volume)
        ##Unused Cmdlet: Get-NcVolumeEncryption - (1.130+) Not currently needed for build docs (API issue in RC1)
        ##Unused Cmdlet: Get-NcVolumeEncryptionConversion - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcVolumeEncryptionRekey - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcVolumeTieringPolicies - (1.140+) Not needed for build docs

        Write-Verbose -Message '- Gathering detailed volume information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Vol -Value ([pscustomobject]@{ })

        #ZAPI BUG: Not all attributes are returned unless we inititalize the template (flexgroup aggr-list specifically)
        Write-Verbose -Message 'Calling Get-NcVol'
        $volTemplate = Set-AllNcTemplateProperties -Template (Get-NcVol -Template -Fill -Controller $Controller -ErrorVariable +NetAppDocsError)
        $systemData.Vol | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcVol -Attributes $volTemplate -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcVolClone'
        $systemData.Vol | Add-Member -MemberType NoteProperty -Name CloneVol -Value (Get-NcVolClone -Controller $Controller -ErrorVariable +NetAppDocsError)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Calling Get-NcVolSpace'
            $systemData.Vol | Add-Member -MemberType NoteProperty -Name Space -Value (Get-NcVolSpace -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcVolFootprint'
            $systemData.Vol | Add-Member -MemberType NoteProperty -Name Footprint -Value (Get-NcVolFootprint -Controller $Controller -ErrorVariable +NetAppDocsError)

            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.140')) {
                $systemData.Vol | Add-Member -MemberType NoteProperty -Name TieringInfo -Value $(
                    if ($useSystemCli) {
                        Write-Verbose -Message 'Calling Invoke-NcSystemApi (vol show -fields vserver, vserver-uuid, instance-uuid, tiering-minimum-cooling-days (priv diag))'
                        (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -privilege diagnostic -confirmations off;</arg><arg>set -showseparator ~;</arg><arg>vol show -fields vserver,vserver-uuid,instance-uuid,tiering-minimum-cooling-days</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                    }
                    else {
                        Write-Verbose -Message 'Calling Invoke-NcSsh (vol show -fields vserver,vserver-uuid,instance-uuid,tiering-minimum-cooling-days (priv diag))'
                        Invoke-NcSsh -Command 'set -privilege diagnostic -confirmations off; set -showseparator ~; set -showallfields true; vol show -fields vserver,vserver-uuid,instance-uuid,tiering-minimum-cooling-days' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                    }
                )
            }
        }

        #endregion Volume Configuration

        #region SIS Configuration

        Write-Verbose -Message '- Gathering SIS configuration -'
        $systemData | Add-Member -MemberType NoteProperty -Name Sis -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcSis'
        $systemData.Sis | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcSis -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcSisPolicy'
        $systemData.Sis | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-NcSisPolicy -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion SIS Configuration

        #region Snapshot Information

        ##Unused Cmdlet: Get-NcSnapshotAutoDelete - Info exists in Get-NcVol
        ##Unused Cmdlet: Get-NcSnapshotDelta - Not needed for build docs
        ##Unused Cmdlet: Get-NcSnapshotReclaimable - Not needed for build docs
        ##Unused Cmdlet: Get-NcSnapshotReserve - Info exists in Get-NcVol
        ##Unused Cmdlet: Get-NcSnapshotRestoreStatus - Not needed for build docs

        Write-Verbose -Message '- Gathering detailed Snapshot information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Snapshot -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcSnapshot'
        $snapshotTemplate = Set-AllNcTemplateProperties -Template (Get-NcSnapshot -Template -Controller $Controller -ErrorVariable +NetAppDocsError)
        $systemData.Snapshot | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcSnapshot -Attributes $snapshotTemplate -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcSnapshotPolicy'
        $systemData.Snapshot | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-NcSnapshotPolicy -Controller $Controller -ErrorVariable +NetAppDocsError)

        if ($systemData.System.OntapiVersion -lt [version]::Parse('1.20')) {
            $systemData.Snapshot.Policy | Add-Member -MemberType NoteProperty -Name VserverName -Value ($systemData.Cluster.Info.ClusterName) -ErrorAction Ignore -Force
        }

        #endregion Snapshot Information

        #region Qtree Configuration

        Write-Verbose -Message '- Gathering qtree data -'

        Write-Verbose -Message 'Calling Get-NcQtree'
        $systemData | Add-Member -MemberType NoteProperty -Name Qtree -Value (Get-NcQtree -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion Qtree Configuration

        #region User Data

        Write-Verbose -Message '- Gathering user data -'

        Write-Verbose -Message 'Calling Get-NcUser'
        $systemData | Add-Member -MemberType NoteProperty -Name User -Value (Get-NcUser -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcUser (admin vserver diag user)'
        $systemData.User += Get-NcUser -Name diag -Vserver ($systemData.Vserver.Info | Where-Object { $_.VserverType -eq 'admin' }).VserverName -Controller $Controller -ErrorVariable +NetAppDocsError

        #endregion User Data

        #region Role Data

        ##Unused Cmdlet: Get-NcRoleConfig - Using equivalent CLI command due to ZAPI issues not returning all properties

        Write-Verbose -Message '- Gathering role data -'
        $systemData | Add-Member -MemberType NoteProperty -Name Role -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcRole'
        $systemData.Role | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcRole -Controller $Controller -ErrorVariable +NetAppDocsError)

        #CLI BURT: diag needed for this CLI call only because admin/advanced does not add the appropriate newline between records
        $systemData.Role | Add-Member -MemberType NoteProperty -Name Config -Value $(
            $roles = $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (security login role config show (priv diag))'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -privilege diagnostic -confirmations off;</arg><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>security login role config show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (security login role config show (priv diag))'
                    Invoke-NcSsh -Command 'set -privilege diagnostic -confirmations off; set -showseparator ~; set -showallfields true; security login role config show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            )

            $roles | Select-Object -Property Vserver, @{
                Name       = 'RoleName'
                Expression = { $_.Role }
            }, @{
                Name       = 'MinUsernameSize'
                Expression = { $_.UsernameMinlength }
            }, @{
                Name       = 'RequireUsernameAlphaNumeric'
                Expression = { $_.UsernameAlphaNum -eq 'enabled' }
            }, @{
                Name       = 'MinPasswordSize'
                Expression = { $_.PasswdMinlength }
            }, @{
                Name       = 'RequirePasswordAlphaNumeric'
                Expression = { $_.PasswdAlphanum -eq 'enabled' }
            }, @{
                Name       = 'MinPasswdSpecialchar'
                Expression = { $_.PasswdMinSpecialChars }
            }, @{
                Name       = 'PasswordExpirationDuration'
                Expression = { $_.PasswdExpiryTime }
            }, @{
                Name       = 'RequireInitialPasswordUpdate'
                Expression = { $_.RequireInitialPasswdUpdate -eq 'enabled' }
            }, @{
                Name       = 'LastPasswordsDisallowedCount'
                Expression = { $_.DisallowedReuse }
            }, @{
                Name       = 'ChangePasswordDurationInDays'
                Expression = { $_.ChangeDelay }
            }, MaxFailedLoginAttempts, LockoutDuration, DelayAfterFailedLogin, PasswdMinLowercaseChars, PasswdMinUppercaseChars, PasswdMinDigits, PasswdExpiryWarnTime, AccountExpiryTime, AccountInactiveLimit
        )

        #endregion Role Data

        #region SNMP User Data

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.100')) {
            Write-Verbose -Message '- Gathering SNMP user data -'

            $systemData | Add-Member -MemberType NoteProperty -Name SnmpUser -Value $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (security snmpusers)'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>security snmpusers</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (security snmpusers)'
                    Invoke-NcSsh -Command 'set -showseparator ~; set -showallfields true; security snmpusers' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            )
        }

        #endregion SNMP User Data

        #region Name Mapping Data

        Write-Verbose -Message '- Gathering name mapping data -'
        $systemData | Add-Member -MemberType NoteProperty -Name NameMapping -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcNameMapping'
        $systemData.NameMapping | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcNameMapping -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcNameMappingUnixGroup'
        $systemData.NameMapping | Add-Member -MemberType NoteProperty -Name UnixGroup -Value (Get-NcNameMappingUnixGroup -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcNameMappingUnixUser'
        $systemData.NameMapping | Add-Member -MemberType NoteProperty -Name UnixUser -Value (Get-NcNameMappingUnixUser -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion Name Mapping Data

        #region Group Mapping Data

        Write-Verbose -Message '- Gathering group mapping data -'

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Calling Get-NcGroupMapping'
            $systemData | Add-Member -MemberType NoteProperty -Name GroupMapping -Value (Get-NcGroupMapping -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Group Mapping Data

        #region Antivirus Configuration

        ##Unused Cmdlet: Get-NcAntivirusEngine - AV ZAPIs deprecated
        ##Unused Cmdlet: Get-NcAntivirusEngineOption - AV ZAPIs deprecated
        ##Unused Cmdlet: Get-NcAntivirusLog - AV ZAPIs deprecated
        ##Unused Cmdlet: Get-NcAntivirusRemedy - AV ZAPIs deprecated
        ##Unused Cmdlet: Get-NcAntivirusVersion - AV ZAPIs deprecated

        #endregion Antivirus Configuration

        #region Vscan Configuration

        ##Unused Cmdlet: Get-NcVscanConnectionStats - Not needed for build docs
        ##Unused Cmdlet: Get-NcVscanOnDemandReport - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcVscanOnDemandTask - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcVscanScannerPoolActive - (1.21+) Not needed for build docs (Vscan.ScannerPoolActive)
        ##Unused Cmdlet: Get-NcVscanEvents - (1.31+) Not needed for build docs (Vscan.Events)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.21')) {
            Write-Verbose -Message '- Gathering Vscan information -'
            $systemData | Add-Member -MemberType NoteProperty -Name Vscan -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcVscanConnection'
            $systemData.Vscan | Add-Member -MemberType NoteProperty -Name Connection -Value (Get-NcVscanConnection -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcVscanOnAccessPolicy'
            $systemData.Vscan | Add-Member -MemberType NoteProperty -Name OnAccessPolicy -Value (Get-NcVscanOnAccessPolicy -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcVscanScannerPool'
            $systemData.Vscan | Add-Member -MemberType NoteProperty -Name ScannerPool -Value (Get-NcVscanScannerPool -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcVscanStatus'
            $systemData.Vscan | Add-Member -MemberType NoteProperty -Name Status -Value (Get-NcVscanStatus -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Vscan Configuration

        #region Audit Configuration

        ##Unused Cmdlet: Get-NcAudit - Not needed for build docs (Audit.Info)
        ##Unused Cmdlet: Get-NcAuditLog - Not needed for build docs

        #endregion Audit Configuration

        #region Autosupport Configuration

        ##Unused Cmdlet: Get-NcAutoSupportCompliant - Not needed for build docs
        ##Unused Cmdlet: Get-NcAutoSupportManifest - Not needed for build docs
        ##Unused Cmdlet: Get-NcAutoSupportDownload - Not needed for build docs
        ##Unused Cmdlet: Get-NcAutoSupportDestination - Not needed for build docs (Autosupport.Destination)

        Write-Verbose -Message '- Gathering autosupport configuration -'
        $systemData | Add-Member -MemberType NoteProperty -Name Autosupport -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcAutoSupportConfig'
        $systemData.Autosupport | Add-Member -MemberType NoteProperty -Name Config -Value (Get-NcAutoSupportConfig -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcAutoSupportHistory'
        $systemData.Autosupport | Add-Member -MemberType NoteProperty -Name History -Value (Get-NcAutoSupportHistory -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcAutoSupportBudget'
        $systemData.Autosupport | Add-Member -MemberType NoteProperty -Name Budget -Value (Get-NcAutoSupportBudget -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcAutoSupportTrigger'
        $systemData.Autosupport | Add-Member -MemberType NoteProperty -Name Trigger -Value (Get-NcAutoSupportTrigger -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion Autosupport Configuration

        #region Clone

        ##Unused Cmdlet: Get-NcCloneAutodelete - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcCloneDeletion - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcCloneLoadSplit - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcCloneToken - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcCloneSplitStatus - (1.120+) Not needed for build docs

        #endregion Clone

        #region Export Policy Configuration

        ##Unused Cmdlet: Get-NcExportAccessCacheConfig - (1.100+) Not needed for build docs (Export.AccessCacheConfig)
        ##Unused Cmdlet: Get-NcExportsAccessCache - (1.31+) Not currently need for build docs

        Write-Verbose -Message '- Gathering export policy information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Export -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcExportPolicy'
        $systemData.Export | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-NcExportPolicy -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcExportRule'
        $systemData.Export | Add-Member -MemberType NoteProperty -Name Rule -Value (Get-NcExportRule -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion Export Policy Configuration

        #region QOS Configuration

        ##Unused Cmdlet: Get-NcQosPolicy - Underlying ZAPI deprecated
        ##Unused Cmdlet: Get-NcQosStream - Underlying ZAPI deprecated
        ##Unused Cmdlet: Get-NcQosArchivedWorkloadConfiguration - Underlying ZAPI deprecated
        ##Unused Cmdlet: Get-NcQosControlConfiguration - (1.20+) Not needed for build docs
        ##Unused Cmdlet: Get-NcCachePolicy - (1.120+) Not currently needed for build docs (Qos.CachePolicy)
        ##Unused Cmdlet: Get-NcQosWorkloadPolicyCount - (1.130+) Not currently needed for build docs

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Gathering QOS configuration -'
            $systemData | Add-Member -MemberType NoteProperty -Name Qos -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcQosPolicyGroup'
            $systemData.Qos | Add-Member -MemberType NoteProperty -Name PolicyGroup -Value (Get-NcQosPolicyGroup -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcQosPolicyReadAhead'
            $systemData.Qos | Add-Member -MemberType NoteProperty -Name PolicyReadAhead -Value (Get-NcQosPolicyReadAhead -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcQosWorkload'
            $systemData.Qos | Add-Member -MemberType NoteProperty -Name Workload -Value (Get-NcQosWorkload -Controller $Controller -ErrorVariable +NetAppDocsError)

            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.130')) {
                Write-Verbose -Message 'Calling Get-NcQosAdaptivePolicyGroup'
                $systemData.Qos | Add-Member -MemberType NoteProperty -Name AdaptivePolicyGroup -Value (Get-NcQosAdaptivePolicyGroup -Controller $Controller -ErrorVariable +NetAppDocsError)
            }
        }

        #endregion QOS Configuration

        #region Fpolicy Configuration

        ##Unused Cmdlet: Get-NcFpolicyPassthroughReadConnection - (1.30+) Not needed for build docs (Fpolicy.PassthroughReadConnection)
        ##Unused Cmdlet: Get-NcFpolicyStatus - (1.20+) Not needed for build docs (Fpolicy.Status)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Gathering Fpolicy configuration -'
            $systemData | Add-Member -MemberType NoteProperty -Name Fpolicy -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcFpolicyServerStatus'
            $systemData.Fpolicy | Add-Member -MemberType NoteProperty -Name ServerStatus -Value (Get-NcFpolicyServerStatus -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcFpolicyExternalEngine'
            $systemData.Fpolicy | Add-Member -MemberType NoteProperty -Name ExternalEngine -Value (Get-NcFpolicyExternalEngine -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcFpolicyPolicy'
            $systemData.Fpolicy | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-NcFpolicyPolicy -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcFpolicyEvent'
            $systemData.Fpolicy | Add-Member -MemberType NoteProperty -Name Event -Value (Get-NcFpolicyEvent -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcFpolicyScope'
            $systemData.Fpolicy | Add-Member -MemberType NoteProperty -Name Scope -Value (Get-NcFpolicyScope -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Fpolicy Configuration

        #region File Information

        ##Unused Cmdlet: Get-NcFile - Not needed for build docs
        ##Unused Cmdlet: Get-NcFileHoles - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFileInodeInfo - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFileSpaceReservationInfo - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFileUsage - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFileUsageResult - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSymLinkTarget - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFileFingerprint - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFileFingerprintStatus - (1.100+) Not needed for build docs

        #endregion File Information

        #region Lock Information

        ##Unused Cmdlet: Get-NcLock - (1.30+) Not needed for build docs

        #endregion Lock Information

        #region FileDirectorySecurity Configuration

        ##Unused Cmdlet: Get-NcFileDirectorySecurity - Not needed for build docs
        ##Unused Cmdlet: Get-NcFileDirectorySecurityNtfs - Not needed for build docs
        ##Unused Cmdlet: Get-NcFileDirectorySecurityNtfsDacl - Not needed for build docs
        ##Unused Cmdlet: Get-NcFileDirectorySecurityNtfsSacl - Not needed for build docs
        ##Unused Cmdlet: Get-NcFileDirectorySecurityPolicy - (1.20+) Not needed for build docs (FileDirectorySecurity.Policy)
        ##Unused Cmdlet: Get-NcFileDirectoryEffectivePermissions - (1.120+) Not needed for build docs

        Write-Verbose -Message '- Gathering File Directory Security configuration -'
        $systemData | Add-Member -MemberType NoteProperty -Name FileDirectorySecurity -Value ([pscustomobject]@{ })

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Calling Get-NcFileDirectorySecurityPolicyTask'
            $systemData.FileDirectorySecurity | Add-Member -MemberType NoteProperty -Name PolicyTask -Value (Get-NcFileDirectorySecurityPolicyTask -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion FileDirectorySecurity Configuration

        #region FileService Audit Configuration

        Write-Verbose -Message '- Gathering File Service Audit configuration -'
        $systemData | Add-Member -MemberType NoteProperty -Name FileServiceAudit -Value ([pscustomobject]@{ })

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Calling Get-NcFileServiceAudit'
            $systemData.FileServiceAudit | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcFileserviceAudit -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion FileService Audit Configuration

        #region Cluster Peer Configuration

        ##Unused Cmdlet: Get-NcClusterPeerAuthOffer - Not needed for build docs
        ##Unused Cmdlet: Get-NcClusterPeerHealth - Not needed for build docs (Cluster.Peer.Health)
        ##Unused Cmdlet: Get-NcClusterPeerConnection - (1.30+) Not needed for build docs (Cluster.Peer.Connection)
        ##Unused Cmdlet: Get-NcClusterPeerPolicy - (1.30+) Not needed for build docs (Cluster.Peer.Policy)

        Write-Verbose -Message '- Gathering Cluster Peer configuration -'
        $systemData.Cluster | Add-Member -MemberType NoteProperty -Name Peer -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcClusterPeer'
        $systemData.Cluster.Peer | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcClusterPeer -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion Cluster Peer Configuration

        #region Metrocluster Information

        ##Unused Cmdlet: Get-NcMetroclusterAggregateCheck - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcMetroclusterAggregateEligibilityCheck - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcMetroclusterCheck - (1.30+) Not sure if this is needed
        ##Unused Cmdlet: Get-NcMetroclusterCheckCluster - (1.31+) Not sure if this is needed
        ##Unused Cmdlet: Get-NcMetroclusterCheckCaptureStatus - (1.30+) Not sure if this is needed
        ##Unused Cmdlet: Get-NcMetroclusterConfigDiff - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcMetroclusterConfigReplicationCheck - (1.30+) Not sure if this is needed
        ##Unused Cmdlet: Get-NcMetroclusterConfigReplicationResyncStatus - (1.30+) Not sure if this is needed
        ##Unused Cmdlet: Get-NcMetroclusterFailedLifPlacement - (1.30+) Not sure if this is needed
        ##Unused Cmdlet: Get-NcMetroclusterNodeCheck - (1.30+) Not sure if this is needed
        ##Unused Cmdlet: Get-NcMetroclusterInterconnectAdapterSwitch - (1.30+) API no longer exists
        ##Unused Cmdlet: Get-NcMetroclusterProgress - (1.30+) Not needed for build docs (Metrocluster.Progress)
        ##Unused Cmdlet: Get-NcMetroclusterOperationStatistics - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcMetroclusterOperation - (1.30+) Not needed for build docs (ErrorAction Ignore here due to a possible 'loop() detected in next' ZAPI error)
        ##Unused Cmdlet: Get-NcMetroclusterInterconnectIpfo - (1.100+) Not needed for build docs (Metrocluster.InterconnectIpfo)
        ##Unused Cmdlet: Get-NcMetroclusterInterconnectMirrorMultipath - (1.30+) Not needed for build docs (Metrocluster.InterconnectMirrorMultipath)
        ##Unused Cmdlet: Get-NcMetroclusterConfigReplication - (1.30+) Not needed for build docs (Metrocluster.ConfigReplication)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message '- Gathering Metrocluster information -'
            $systemData | Add-Member -MemberType NoteProperty -Name Metrocluster -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Test-NcMetrocluster'
            if (Test-NcMetrocluster -Controller $Controller -ErrorAction Ignore) {
                Write-Verbose -Message 'Calling Get-NcMetrocluster'
                $systemData.Metrocluster | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcMetrocluster -Controller $Controller -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message 'Calling Get-NcMetroclusterInterconnectAdapter'
                $systemData.Metrocluster | Add-Member -MemberType NoteProperty -Name InterconnectAdapter -Value (Get-NcMetroclusterInterconnectAdapter -Controller $Controller -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message 'Calling Get-NcMetroclusterInterconnectMirror'
                $systemData.Metrocluster | Add-Member -MemberType NoteProperty -Name InterconnectMirror -Value (Get-NcMetroclusterInterconnectMirror -Controller $Controller -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message 'Calling Get-NcMetroclusterNode'
                $systemData.Metrocluster | Add-Member -MemberType NoteProperty -Name Node -Value (Get-NcMetroclusterNode -Controller $Controller -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message 'Calling Get-NcMetroclusterVserver'
                $systemData.Metrocluster | Add-Member -MemberType NoteProperty -Name Vserver -Value (Get-NcMetroclusterVserver -Controller $Controller -ErrorVariable +NetAppDocsError)
            }
        }

        #endregion Metrocluster Information

        #region Snapmirror Configuration

        ##Unused Cmdlet: Get-NcSnapmirrorSnapshotOwner - Not currently needed for build docs
        ##Unused Cmdlet: Get-NcSnapmirrorHistory - (1.30+) Not currently needed for build docs
        ##Unused Cmdlet: Get-NcSnapmirrorDpPersonality - (1.120+) Currently not needed for build docs (Snapmirror.DpPersonality)
        ##Unused Cmdlet: Get-NcSnapmirrorUnprotected - (1.130+) Not currently needed for build docs
        ##Unused Cmdlet: Get-NcSnapmirrorCapabilities - (1.130+) Not currently needed for build docs

        Write-Verbose -Message '- Gathering Snapmirror configuration -'
        $systemData | Add-Member -MemberType NoteProperty -Name Snapmirror -Value ([pscustomobject]@{ })

        #BURTS 653291, 658442, 674884, 715998 (No way to get around this, there may be missing entries in the output)
        Write-Verbose -Message 'Calling Get-NcSnapmirror'
        $systemData.Snapmirror | Add-Member -MemberType NoteProperty -Name Info -Value $(
            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.100')) {
                Get-NcSnapmirror -Expand -Controller $Controller -ErrorVariable +NetAppDocsError
            }
            else {
                Get-NcSnapmirror -Controller $Controller -ErrorVariable +NetAppDocsError
            }
        )

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Calling Get-NcSnapmirrorDestination'
            $systemData.Snapmirror | Add-Member -MemberType NoteProperty -Name Destination -Value (Get-NcSnapmirrorDestination -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcSnapmirrorPolicy'
            $systemData.Snapmirror | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-NcSnapmirrorPolicy -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Snapmirror Configuration

        #region Snapmirror CR Configuration

        ##Move to Metrocluster section?
        ##Unused Cmdlet: Get-NcSnapmirrorCRStatusComm - (1.31+) Not needed for build docs (SnapmirrorCR.StatusComm)
        ##Unused Cmdlet: Get-NcSnapmirrorCRStatusAggregateEligibility - (1.31+) Not needed for build docs (SnapmirrorCR.StatusAggregateEligibility)
        ##Unused Cmdlet: Get-NcSnapmirrorCRClusterStorageConfiguration - (1.31+) Not needed for build docs (SnapmirrorCR.ClusterStorageConfiguration)
        ##Unused Cmdlet: Get-NcSnapmirrorCRStatus - (1.31+) Not needed for build docs (SnapmirrorCR.StatusAggregateElStatusigibility)

        #endregion Snapmirror CR Configuration

        #region Quota Configuration

        ##Unused Cmdlet: Get-NcQuotaReport - Not needed for build docs
        ##Unused Cmdlet: Get-NcQuotaPolicy - (1.31+) Not needed for build docs (Quota.Policy)
        ##Unused Cmdlet: Get-NcQuotaPolicyRuleCount - (1.100+) Not currently needed for build docs

        Write-Verbose -Message '- Gathering quota configuration -'
        $systemData | Add-Member -MemberType NoteProperty -Name Quota -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcQuota'
        $systemData.Quota | Add-Member -MemberType NoteProperty -Name Rule -Value (Get-NcQuota -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcQuotaStatus'
        $systemData.Quota | Add-Member -MemberType NoteProperty -Name Status -Value (Get-NcQuotaStatus -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion Quota Configuration

        #region ConfigBackup Information

        Write-Verbose -Message '- Gathering config backup information -'
        $systemData | Add-Member -MemberType NoteProperty -Name ConfigBackup -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcConfigBackup'
        $systemData.ConfigBackup | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcConfigBackup -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcConfigBackupCount'
        $systemData.ConfigBackup | Add-Member -MemberType NoteProperty -Name Count -Value (Get-NcConfigBackupCount -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcConfigBackupUrl'
        $systemData.ConfigBackup | Add-Member -MemberType NoteProperty -Name Url -Value (Get-NcConfigBackupUrl -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion ConfigBackup Information

        #region Dashboard/Diagnosis Information

        ##Unused Cmdlet: Get-NcDiagnosisSubscription - Not needed for build docs and was causing errors
        ##Unused Cmdlet: Get-NcDashboardAlarm - Deprecated in ONTAP 9.0, never used for output anyway
        ##Unused Cmdlet: Get-NcDashboardAlarmThreshold - Deprecated in ONTAP 9.0, never used for output anyway
        ##Unused Cmdlet: Get-NcDiagnosisAlertDefinition - Not needed for build docs (Diagnosis.AlertDefinition)
        ##Unused Cmdlet: Get-NcDiagnosisConfig - Not needed for build docs (Diagnosis.Config)
        ##Unused Cmdlet: Get-NcDiagnosisPolicyDefinition - Not needed for build docs (Diagnosis.PolicyDefinition)
        ##Unused Cmdlet: Get-NcDiagnosisStatus - Not needed for build docs (Diagnosis.Status)
        ##Unused Cmdlet: Get-NcDiagnosisSubsystem - (1.20+) Not needed for build docs (Diagnosis.Subsystem.Info)
        ##Unused Cmdlet: Get-NcDiagnosisSubsystemConfig - (1.31+) Not needed for build docs (Diagnosis.Subsystem.Config)

        Write-Verbose -Message '- Gathering diagnostics information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Diagnosis -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcDiagnosisAlert'
        $systemData.Diagnosis | Add-Member -MemberType NoteProperty -Name Alert -Value (Get-NcDiagnosisAlert -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion Diagnosis Information

        #region LUN Information

        ##Unused Cmdlet: Get-NcLunOsTypes - (1.31+) Not needed for build docs
        ##Unused Cmdlet: Get-NcLunAlignment - Info exists in Get-NcLun
        ##Unused Cmdlet: Get-NcLunAttribute - Info exists in Get-NcLun
        ##Unused Cmdlet: Get-NcLunByIgroup - Not needed
        ##Unused Cmdlet: Get-NcLunGeometry - Not needed for build docs
        ##Unused Cmdlet: Get-NcLunInquiry - Not needed for build docs
        ##Unused Cmdlet: Get-NcLunMapByInitiator - Not needed
        ##Unused Cmdlet: Get-NcLunMaxSize - Not needed for build docs
        ##Unused Cmdlet: Get-NcLunMinSize - Not needed for build docs
        ##Unused Cmdlet: Get-NcLunPersistentReservation - Not needed for build docs
        ##Unused Cmdlet: Get-NcLunStatistics - Not needed for build docs
        ##Unused Cmdlet: Get-NcLunSelect - Not sure if this is needed or not
        ##Unused Cmdlet: Get-NcLunSignature - Not sure if this is needed or not
        ##Unused Cmdlet: Get-NcLunSerialNumber - Info exists in Get-NcLun
        ##Unused Cmdlet: Get-NcLunCopy - (1.30+) Not needed for build docs (Lun.Copy)
        ##Unused Cmdlet: Get-NcLunMove - (1.30+) Not needed for build docs (Lun.Move)
        ##Unused Cmdlet: Get-NcLunImport - (1.30+) Not currently needed for build docs (Lun.Import)

        Write-Verbose -Message '- Gathering LUN information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Lun -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcLun'
        $systemData.Lun | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcLun -Controller $Controller -WarningAction SilentlyContinue -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcLunMap'
        $systemData.Lun | Add-Member -MemberType NoteProperty -Name Map -Value (Get-NcLunMap -Controller $Controller -WarningAction SilentlyContinue -ErrorVariable +NetAppDocsError)

        #endregion LUN Information

        #region Name Services Information (NIS/Kerberos/LDAP)

        ##Unused Cmdlet: Get-NcNameServiceDNSStatistics - (1.30+) Deprecated in 9.0
        ##Unused Cmdlet: Get-NcNameServiceHostnameFromIP - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNameServiceIPFromHostname - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNameServiceNisStatistics - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNameserviceBoundNisServer - (1.31+) Not currently needed
        ##Unused Cmdlet: Get-NcNameServiceNisBindingDetail - (1.100+) Not needed for build docs (Nameservice.Nis.BindingDetail)
        ##Unused Cmdlet: Get-NcNameServiceUnixGroupFileOnly - (1.110+) Not needed for build docs (NameService.Unix.GroupFileOnly - loop through NFS vservers and use -VserverContext)
        ##Unused Cmdlet: Get-NcNameServiceUnixUserFileOnly - (1.110+) Not needed for build docs (NameService.Unix.UserFileOnly - loop through NFS vservers and use -VserverContext)
        ##Unused Cmdlet: Get-NcNameServiceCacheGroupMembershipSettings - (1.130+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNameServiceCacheHostsSettings - (1.130+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNameServiceCacheNetgroupSettings - (1.130+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNameServiceCacheUnixGroupSettings - (1.130+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNameServiceCacheUnixUserSettings - (1.130+) Not needed for build docs

        Write-Verbose -Message '- Gathering name services information -'
        $systemData | Add-Member -MemberType NoteProperty -Name NameService -Value ([pscustomobject]@{ })
        $systemData.NameService | Add-Member -MemberType NoteProperty -Name Kerberos -Value ([pscustomobject]@{ })
        $systemData.NameService | Add-Member -MemberType NoteProperty -Name Ldap -Value ([pscustomobject]@{ })
        $systemData.NameService | Add-Member -MemberType NoteProperty -Name Nis -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcNis'
        $systemData.NameService.Nis | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcNis -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcKerberosConfig'
        $systemData.NameService.Kerberos | Add-Member -MemberType NoteProperty -Name Config -Value (Get-NcKerberosConfig -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcKerberosRealm'
        $systemData.NameService.Kerberos | Add-Member -MemberType NoteProperty -Name Realm -Value (Get-NcKerberosRealm -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcLdapClient'
        $systemData.NameService.Ldap | Add-Member -MemberType NoteProperty -Name Client -Value (Get-NcLdapClient -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcLdapClientSchema'
        $systemData.NameService.Ldap | Add-Member -MemberType NoteProperty -Name ClientSchema -Value (Get-NcLdapClientSchema -Controller $Controller -ErrorVariable +NetAppDocsError)

        #BURT 636873
        $systemData.NameService.Ldap | Add-Member -MemberType NoteProperty -Name Config -Value $(
            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
                Write-Verbose -Message 'Calling Get-NcLdapConfig'
                Get-NcLdapConfig -Controller $Controller -ErrorVariable +NetAppDocsError
            }
            else {
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (vserver services ldap show)'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>vserver services ldap show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (vserver services ldap show)'
                    Invoke-NcSsh -Command 'set -showseparator ~; set -showallfields true; vserver services ldap show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            }
        )

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message 'Calling Get-NcNameServiceNsSwitch'
            $systemData.NameService | Add-Member -MemberType NoteProperty -Name NsSwitch -Value (Get-NcNameServiceNsSwitch -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Name Service Information (NIS/Kerberos/LDAP)

        #region Netgroup Information

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message '- Gathering netgroup information -'
            $systemData | Add-Member -MemberType NoteProperty -Name Netgroup -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcNetgroupFile'
            $systemData.Netgroup | Add-Member -MemberType NoteProperty -Name File -Value (Get-NcNetgroupFile -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Netgroup Information

        #region Job Information

        ##Unused Cmdlet: Get-NcJobCronSchedule - Not needed, info in Get-NcJobSchedule
        ##Unused Cmdlet: Get-NcJobIntervalSchedule - Not needed, info in Get-NcJobSchedule
        ##Unused Cmdlet: Get-NcJobHistory - Not needed

        Write-Verbose -Message '- Gathering job information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Job -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcJob'
        $systemData.Job | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcJob -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcJobSchedule'
        $systemData.Job | Add-Member -MemberType NoteProperty -Name Schedule -Value (
            Get-NcJobSchedule -Controller $Controller -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                Name       = 'JobScheduleInfo'
                Expression = { ConvertFrom-JobScheduleDescription -JobScheduleDescription $_.JobScheduleDescription }
            }
        )

        #endregion Job Information

        #region Active Directory

        ##Unused Cmdlet: Get-NcActiveDirectoryAccount - (1.21+) Not needed for build docs (ActiveDirectoryAccount)

        #endregion Active Directory

        #region CIFS Configuration

        ##Unused Cmdlet: Get-NcCifsConnection - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcCifsNbtStat - Not needed for build docs
        ##Unused Cmdlet: Get-NcCifsSession - Not needed for build docs (MCC BURT: 969339)
        ##Unused Cmdlet: Get-NcCifsSessionFile - Not needed for build docs
        ##Unused Cmdlet: Get-NcCifsShadowCopyEmsMessage - Not needed for build docs
        ##Unused Cmdlet: Get-NcCifsUserHomeDirectory - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcCifsUserAndGroupMember - (1.100+) Not sure what this does differently
        ##Unused Cmdlet: Get-NcCifsUserAndUserMembership - (1.100+) Not sure what this does differently
        ##Unused Cmdlet: Get-NcCifsDomainTrust - (1.21+) Not needed for build docs (Cifs.DomainTrust)
        ##Unused Cmdlet: Get-Get-NcCifsTrustedDomain - (1.21+) Not needed for build docs (Cifs.TrustedDomain)
        ##Unused Cmdlet: Get-NcCifsCharacterMapping - (1.30+) Not needed for build docs (Cifs.CharacterMapping)
        ##Unused Cmdlet: Get-NcCifsDomainPasswordSchedule - (1.30+) Not needed for build docs (Cifs.DomainPasswordSchedule)
        ##Unused Cmdlet: Get-NcCifsDomainServer - Caused timeout issues in some cases (Cifs.DomainServer)
        ##Unused Cmdlet: Get-NcCifsPreferredDcStatus - (1.150+) Not needed for build docs
        ##Unused Cmdlet: Get-NcCifsServerStatus - (1.150+) Not needed for build docs

        Write-Verbose -Message '- Gathering CIFS configuration information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Cifs -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcCifsServer'
        $systemData.Cifs | Add-Member -MemberType NoteProperty -Name Server -Value (Get-NcCifsServer -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcCifsSymlink'
        $systemData.Cifs | Add-Member -MemberType NoteProperty -Name Symlink -Value (Get-NcCifsSymlink -Controller $Controller -ErrorVariable +NetAppDocsError)
        if ($systemData.System.OntapiVersion -lt [version]::Parse('1.20')) {
            foreach ($symlink in ($systemData.Cifs.Symlink | Where-Object { $_ -and [string]::IsNullOrWhiteSpace($_.Vserver) })) {
                $symlink | Add-Member -MemberType NoteProperty -Name Vserver -Value ($systemData.Cifs.Server | Where-Object { $_.CifsServer -ceq $symlink.CifsServer }).Vserver -ErrorAction Ignore -Force
            }
        }

        Write-Verbose -Message 'Calling Get-NcCifsShare'
        $systemData.Cifs | Add-Member -MemberType NoteProperty -Name Share -Value (Get-NcCifsShare -Controller $Controller -ErrorVariable +NetAppDocsError)
        if ($systemData.System.OntapiVersion -lt [version]::Parse('1.20')) {
            foreach ($share in ($systemData.Cifs.Share | Where-Object { $_ -and [string]::IsNullOrWhiteSpace($_.Vserver) })) {
                $share | Add-Member -MemberType NoteProperty -Name Vserver -Value ($systemData.Cifs.Server | Where-Object { $_.CifsServer -ceq $share.CifsServer }).Vserver -ErrorAction Ignore -Force
            }
        }

        Write-Verbose -Message 'Calling Get-NcCifsShareAcl'
        $systemData.Cifs | Add-Member -MemberType NoteProperty -Name ShareAcl -Value (Get-NcCifsShareAcl -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcCifsPreferredDomainController'
        $systemData.Cifs | Add-Member -MemberType NoteProperty -Name PreferredDomainController -Value (Get-NcCifsPreferredDomainController -Controller $Controller -ErrorVariable +NetAppDocsError)

        # Add Vserver specific properties
        ##As of 1.20+, these are cluster APIs but the TOOLKIT still enforces VserverContext
        $systemData.Cifs | Add-Member -MemberType NoteProperty -Name HomeDirectorySearchPath -Value @( )
        $systemData.Cifs | Add-Member -MemberType NoteProperty -Name Option -Value @( )
        $systemData.Cifs | Add-Member -MemberType NoteProperty -Name Security -Value @( )

        foreach ($cifsServer in ($systemData.Cifs.Server | Where-Object { $_ })) {
            Write-Verbose -Message "Calling Get-NcCifsHomeDirectorySearchPath (Vserver: $( $cifsServer.Vserver ))"
            Get-NcCifsHomeDirectorySearchPath -VserverContext $cifsServer.Vserver -Controller $Controller -ErrorVariable +NetAppDocsError | Where-Object { $_ } | ForEach-Object {
                $systemData.Cifs.HomeDirectorySearchPath += $_
                $systemData.Cifs.HomeDirectorySearchPath | Select-Object -Last 1 | Add-Member -MemberType NoteProperty -Name Vserver -Value $cifsServer.Vserver -ErrorAction Ignore -Force
            }

            Write-Verbose -Message "Calling Get-NcCifsOption (Vserver: $( $cifsServer.Vserver ))"
            Get-NcCifsOption -VserverContext $cifsServer.Vserver -Controller $Controller -ErrorVariable +NetAppDocsError | Where-Object { $_ } | ForEach-Object {
                $systemData.Cifs.Option += $_
                $systemData.Cifs.Option | Select-Object -Last 1 | Add-Member -MemberType NoteProperty -Name Vserver -Value $cifsServer.Vserver -ErrorAction Ignore -Force
            }

            Write-Verbose -Message "Calling Get-NcCifsSecurity (Vserver: $( $cifsServer.Vserver ))"
            Get-NcCifsSecurity -VserverContext $cifsServer.Vserver -Controller $Controller -ErrorVariable +NetAppDocsError | Where-Object { $_ } | ForEach-Object {
                $systemData.Cifs.Security += $_
                $systemData.Cifs.Security | Select-Object -Last 1 | Add-Member -MemberType NoteProperty -Name Vserver -Value $cifsServer.Vserver -ErrorAction Ignore -Force
            }
        }

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Gathering CIFS local users and groups -'
            $systemData.Cifs | Add-Member -MemberType NoteProperty -Name UserAndGroup -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcCifsLocalUser'
            $systemData.Cifs.UserAndGroup | Add-Member -MemberType NoteProperty -Name LocalUser -Value (Get-NcCifsLocalUser -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcCifsLocalGroup'
            $systemData.Cifs.UserAndGroup | Add-Member -MemberType NoteProperty -Name LocalGroup -Value (Get-NcCifsLocalGroup -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcCifsLocalGroupMember'
            $systemData.Cifs.UserAndGroup | Add-Member -MemberType NoteProperty -Name LocalGroupMember -Value (Get-NcCifsLocalGroupMember -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcCifsPrivilege'
            $systemData.Cifs.UserAndGroup | Add-Member -MemberType NoteProperty -Name Privilege -Value (Get-NcCifsPrivilege -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcCifsBranchCache'
            $systemData.Cifs | Add-Member -MemberType NoteProperty -Name BranchCache -Value (Get-NcCifsBranchCache -Controller $Controller -ErrorVariable +NetAppDocsError)

            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
                Write-Verbose -Message 'Calling Get-NcCifsHomeDirectoryConfig'
                $systemData.Cifs | Add-Member -MemberType NoteProperty -Name HomeDirectoryConfig -Value (Get-NcCifsHomeDirectoryConfig -Controller $Controller -ErrorVariable +NetAppDocsError)

                if ($systemData.System.OntapiVersion -ge [version]::Parse('1.130')) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (cifs-domain-server-discovery-mode-get-iter)'
                    $systemData.Cifs | Add-Member -MemberType NoteProperty -Name DomainServerDiscoveryMode -Value ((Invoke-NcSystemApi -Request '<cifs-domain-server-discovery-mode-get-iter />' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'attributes-list'.'cifs-domain-server-discovery-mode-info' | Convert-GenericXmlNode)
                }
            }
        }

        #endregion CIFS Information

        #region GPO Information

        ##Unused Cmdlet: Get-NcGpoApplied - Not needed for build docs (Gpo.Applied) (BURT 640487: -lt 8.2 loop through Cifs.Server and set -VserverContext. TOOLKIT/ZAPI BURT: ZAPI throws error when GPO list is empty-try/catch.)
        ##Unused Cmdlet: Get-NcGpoGpResult - Not needed for build docs (Gpo.GpResult) (BURT 640487: -lt 8.2 loop through Cifs.Server and set -VserverContext)
        ##Unused Cmdlet: Get-NcGpoRestrictedGroupApplied - (1.30+) - Not needed for build docs (Gpo.RestrictedGroup.Applied. TOOLKIT/ZAPI BURT: ZAPI throws error when GPO list is empty-try/catch.)
        ##Unused Cmdlet: Get-NcGpoRestrictedGroupDefined - (1.30+) - Not needed for build docs (Gpo.RestrictedGroup.Defined)

        Write-Verbose -Message '- Gathering GPO configuration information -'

        $systemData.Cifs | Add-Member -MemberType NoteProperty -Name Gpo -Value ([pscustomobject]@{ })

        ##BURT 640487
        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Calling Get-NcGpo'
            $systemData.Cifs.Gpo | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcGpo -Controller $Controller -ErrorVariable +NetAppDocsError)
        }
        else {
            $systemData.Cifs.Gpo | Add-Member -MemberType NoteProperty -Name Info -Value @( )

            foreach ($cifsServer in ($systemData.Cifs.Server | Where-Object { $_ })) {
                Write-Verbose -Message "Calling Get-NcGpo (Vserver: $( $cifsServer.Vserver ))"
                Get-NcGpo -VserverContext $cifsServer.Vserver -Controller $Controller -ErrorVariable +NetAppDocsError | Where-Object { $_ } | ForEach-Object {
                    $systemData.Cifs.Gpo.Info += $_
                    $systemData.Cifs.Gpo.Info | Select-Object -Last 1 | Add-Member -MemberType NoteProperty -Name Vserver -Value $cifsServer.Vserver -ErrorAction Ignore -Force
                }
            }
        }

        #endregion GPO Information

        #region NFS

        ##Unused Cmdlet: Get-NcNfsExport - Not needed in cDOT
        ##Unused Cmdlet: Get-NcNfsSecurityFlavor - Underlying ZAPI broken, not needed for build docs

        Write-Verbose -Message '- Gathering NFS configuration information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Nfs -Value ([pscustomobject]@{ })

        #BURT 512073
        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.17')) {
            Write-Verbose -Message 'Calling Get-NcNfsService'
            $systemData.Nfs | Add-Member -MemberType NoteProperty -Name Service -Value (Get-NcNfsService -Controller $Controller -ErrorVariable +NetAppDocsError)
        }
        else {
            $systemData.Nfs | Add-Member -MemberType NoteProperty -Name Service -Value @( )

            foreach ($vserver in ($systemData.Vserver.Info | Where-Object { $_.AllowedProtocols -contains 'nfs' -and $_.VserverType -in @( 'data', 'cluster' ) })) {
                if (-not $systemData.Nfs.Service) {
                    Write-Verbose -Message "Calling Get-NcNfsService (Vserver: $( $vserver.VserverName ))"
                    try {
                        Get-NcNfsService -VserverContext $vserver.VserverName -Controller $Controller -ErrorAction Stop | Where-Object { $_ } | ForEach-Object {
                            $systemData.Nfs.Service += $_
                            $systemData.Nfs.Service | Select-Object -Last 1 | Add-Member -MemberType NoteProperty -Name Vserver -Value $vserver.VserverName
                        }
                    }
                    catch {
                        if ($_.Exception.Message -like "*entry doesn't exist*") {
                            $Global:Error.RemoveAt(0)
                        }
                        else {
                            $NetAppDocsError += $_
                        }

                        continue
                    }
                }
            }
        }

        #endregion NFS

        #region Performance Configuration Information

        ##Unused Cmdlet: Get-NcPerfArchiveDatastore - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcPerfCounter - Not needed for build docs
        ##Unused Cmdlet: Get-NcPerfData - Not needed for build docs
        ##Unused Cmdlet: Get-NcPerfInstance - Not needed for build docs
        ##Unused Cmdlet: Get-NcPerfObject - Not needed for build docs
        ##Unused Cmdlet: Get-NcPerfPresetDetailInfo - Not needed for build docs
        ##Unused Cmdlet: Get-NcPerfArchiveConfig - (1.20+) Not needed for build docs (Perf.ArchiveConfig)
        ##Unused Cmdlet: Get-NcPerfPreset - (1.20+) Not needed for build docs (Perf.Preset)
        ##Unused Cmdlet: Get-NcPerfSamples - Not needed for build docs

        #endregion Performance Configuration Information

        #region Secd Information

        ##Unused Cmdlet: Get-NcSecdAuthClaimName - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecdAuthDcInfo - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecdAuthOntapAdminUnixCredsUsingUid - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecdAuthOntapAdminUnixCredsUsingUnixUserName - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecdConnectionTimeout - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecdDnsForwardLookup - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecdDnsSrvLookup - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecdGroupMapping - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecdNameMapping - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecdServerDiscoveryHost - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecdServerDiscoverySite - (1.140+) Not needed for build docs

        #endregion Secd Information

        #region Security Information

        ##Unused Cmdlet: Get-NcSecurityTrace - Not currently needed for build docs
        ##Unused Cmdlet: Get-NcSecurityTraceFilter - Not currently needed for build docs
        ##Unused Cmdlet: Get-NcSecurityLoginCurrentUser - (1.100+) Not needed
        ##Unused Cmdlet: Get-NcSecurityCertificateIssued - (1.20+) Not needed for build docs (Security.CertificateIssued)
        ##Unused Cmdlet: Get-NcSecuritySsl - (1.20+) Not needed for build docs (Security.Ssl)
        ##Unused Cmdlet: Get-NcSecurityConfigOcsp - (1.120+) Not currently needed for build docs (Security.Ocsp)
        ##Unused Cmdlet: Get-NcSecuritySamlSp - (1.130+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecurityLastLoginInfo - (1.150+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecurityConfig (Interface: ssl) - (1.100+) TOOLKIT: Get-NcSecurityConfig is not returning the SupportedProtocols

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.17')) {
            Write-Verbose -Message '- Gathering security related information -'
            $systemData | Add-Member -MemberType NoteProperty -Name Security -Value ([pscustomobject]@{ })

            $systemData.Security | Add-Member -MemberType NoteProperty -Name DomainTunnelVserver -Value $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (security login domain-tunnel show)'
                    ((Invoke-NcSystemApi -Request '<system-cli><args><arg>security login domain-tunnel show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliTextInstance).TunnelVserver
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (security login domain-tunnel show)'
                    (Invoke-NcSsh -Command 'security login domain-tunnel show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliTextInstance).TunnelVserver
                }
            )

            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
                Write-Verbose -Message 'Calling Get-NcSecurityCertificate'
                $systemData.Security | Add-Member -MemberType NoteProperty -Name Certificate -Value (Get-NcSecurityCertificate -Controller $Controller -ErrorVariable +NetAppDocsError)

                if ($systemData.System.OntapiVersion -ge [version]::Parse('1.21')) {
                    Write-Verbose -Message 'Calling Get-NcSecuritySsh'
                    $systemData.Security | Add-Member -MemberType NoteProperty -Name Ssh -Value (Get-NcSecuritySsh -Controller $Controller -ErrorVariable +NetAppDocsError)
                }

                if ($systemData.System.OntapiVersion -ge [version]::Parse('1.100')) {
                    #TOOLKIT: Get-NcSecurityConfig is not returning the SupportedProtocols and is also not updated to include SupportedCipherSuites
                    #Write-Verbose -Message 'Calling Get-NcSecurityConfig (Interface: ssl)'
                    #$systemData.Security | Add-Member -MemberType NoteProperty -Name Config -Value (Get-NcSecurityConfig -Interface ssl -Controller $Controller -ErrorVariable +NetAppDocsError)

                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (security-config-get)'
                    $systemData.Security | Add-Member -MemberType NoteProperty -Name Config -Value (
                        (Invoke-NcSystemApi -Request '<security-config-get><interface>ssl</interface></security-config-get>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.attributes.'security-config-info' | Convert-GenericXmlNode | Select-Object -ExcludeProperty SupportedCipherSuites -Property *, @{
                            Name       = 'SupportedCipherSuites'
                            Expression = { $_.SupportedCipherSuites }
                        }
                    )

                    $systemData.Security | Add-Member -MemberType NoteProperty -Name ConfigStatus -Value $(
                        if ($useSystemCli) {
                            Write-Verbose -Message 'Calling Invoke-NcSystemApi (security config status show (priv diag))'
                            (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -privilege diagnostic -confirmations off;</arg><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>security config status show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                        }
                        else {
                            Write-Verbose -Message 'Calling Invoke-NcSsh (security config status show (priv diag))'
                            Invoke-NcSsh -Command 'set -privilege diagnostic -confirmations off; set -showseparator ~; set -showallfields true; security config status show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                        }
                    )

                    $systemData.Security | Add-Member -MemberType NoteProperty -Name Protocol -Value @( )

                    Write-Verbose -Message 'Calling Get-NcSecurityProtocol (Application: rsh)'
                    $systemData.Security.Protocol += Get-NcSecurityProtocol -Application rsh -Controller $Controller -ErrorVariable +NetAppDocsError

                    Write-Verbose -Message 'Calling Get-NcSecurityProtocol (Application: telnet)'
                    $systemData.Security.Protocol += Get-NcSecurityProtocol -Application telnet -Controller $Controller -ErrorVariable +NetAppDocsError

                    if ($systemData.System.OntapiVersion -ge [version]::Parse('1.120')) {
                        Write-Verbose -Message 'Calling Get-NcSecurityProtocolSsh'
                        $systemData.Security | Add-Member -MemberType NoteProperty -Name ProtocolSsh -Value (Get-NcSecurityProtocolSsh -Controller $Controller -ErrorVariable +NetAppDocsError)
                    }
                }
            }
        }

        #endregion Security Information

        #region Security Key Manager Information

        ##Unused Cmdlet: Get-NcSecurityKeyManagerKeys (1.31+) Not needed for build docs (Security.KeyManager.Keys)
        ##Unused Cmdlet: Get-NcSecurityKeyManagerRestore (1.31+) Not needed for build docs (Security.KeyManager.Restore)
        ##Unused Cmdlet: Get-NcSecurityKeyManagerBackup (1.100+) Not needed for build docs (Security.KeyManager.Backup)
        ##Unused Cmdlet: Get-NcSecurityKeyManagerQueryV2 - (1.130+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecurityKeyManagerConfig - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecurityKeyManagerRestoreV2 - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcExternalRestore - (1.160+) Not needed for build docs
        ##Unused Cmdlet: Get-NcExternalStatus - (1.160+) Not needed for build docs
        ##Unused Cmdlet: Get-NcKeyQuery - (1.160+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecurityKeyManagerExternal - (1.160+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecurityKeyManagerKeyStore - (1.160+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecurityKeyManagerOnboardBackup - (1.160+) Not needed for build docs

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.31')) {
            Write-Verbose -Message '- Gathering security key manager information -'
            $systemData.Security | Add-Member -MemberType NoteProperty -Name KeyManager -Value ([pscustomobject]@{ })

            try {
                Write-Verbose -Message 'Calling Get-NcSecurityKeyManager'
                $systemData.Security.KeyManager | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcSecurityKeyManager -Controller $Controller -ErrorAction Stop)
            }
            catch {
                if ($_.Exception.Message -like '*Key manager is not enabled*' -or $_.Exception.Message -like '*Configure a key manager*' -or $_.Exception.Message -like '*This command is not supported when onboard key management is enabled*' -or $_.Exception.Message -like '*This command is not supported when the Onboard Key Manager is enabled*') {
                    $Global:Error.RemoveAt(0)
                }
                else {
                    $NetAppDocsError += $_
                }
            }
        }

        #endregion Security Key Manager Information

        #region Security Session Information

        ##Unused Cmdlet: Get-NcSecuritySessionAppStats - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecuritySessionLocStats - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecuritySessionReqStats - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecuritySessionRunningStats - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecuritySessionUserStats - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecuritySessionVserverStats - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSecuritySessionLimit - (1.100+) Info available in specific cmdlets

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.100')) {
            Write-Verbose -Message '- Gathering Security Session information -'
            $systemData | Add-Member -MemberType NoteProperty -Name SecuritySession -Value (
                [pscustomobject]@{
                    Limit = [pscustomobject]@{ }
                }
            )

            Write-Verbose -Message 'Calling Get-NcSecuritySessionAppLimit'
            $systemData.SecuritySession.Limit | Add-Member -MemberType NoteProperty -Name Application -Value (Get-NcSecuritySessionAppLimit -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcSecuritySessionLocLimit'
            $systemData.SecuritySession.Limit | Add-Member -MemberType NoteProperty -Name Location -Value (Get-NcSecuritySessionLocLimit -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcSecuritySessionReqLimit'
            $systemData.SecuritySession.Limit | Add-Member -MemberType NoteProperty -Name Request -Value (Get-NcSecuritySessionReqLimit -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcSecuritySessionUserLimit'
            $systemData.SecuritySession.Limit | Add-Member -MemberType NoteProperty -Name User -Value (Get-NcSecuritySessionUserLimit -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcSecuritySessionVserverLimit'
            $systemData.SecuritySession.Limit | Add-Member -MemberType NoteProperty -Name Vserver -Value (Get-NcSecuritySessionVserverLimit -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Security Session Information

        #region Snaplock Information

        ##Unused Cmdlet: Get-NcSnaplockRetentionTime - (1.100+) Not currently needed for build docs
        ##Unused Cmdlet: Get-NcSnaplockLogConfig - (1.100+) Not currently needed for build docs (Snaplock.LogConfig)
        ##Unused Cmdlet: Get-NcSnaplockLogFiles - (1.100+) Not currently needed for build docs (Snaplock.LogFiles)
        ##Unused Cmdlet: Get-NcSnaplockComplianceClock - (1.100+) Not needed for build docs (Snaplock.ComplianceClock)
        ##Unused Cmdlet: Get-NcSnaplockVolAttr - (1.100+) Using the CLI command instead
        ##Unused Cmdlet: Get-NcSnaplockEventRetention - (1.130+) Not currently needed for build docs (user needs explicit vsadmin-snaplock role)
        ##Unused Cmdlet: Get-NcSnaplockEventRetentionPolicy - (1.130+) Not currently needed for build docs (user needs explicit vsadmin-snaplock role)
        ##Unused Cmdlet: Get-NcSnaplockLegalHold - (1.130+) Not currently needed for build docs (user needs explicit vsadmin-snaplock role)
        ##Unused Cmdlet: Get-NcSnaplockClockNtpSync - (1.140+) Not currently needed for build docs

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.100')) {
            Write-Verbose -Message '- Gathering Snaplock information -'
            $systemData | Add-Member -MemberType NoteProperty -Name Snaplock -Value ([pscustomobject]@{ })

            $systemData.Snaplock | Add-Member -MemberType NoteProperty -Name Volume -Value $(
                $volSnaplockShow = $(
                    if ($useSystemCli) {
                        Write-Verbose -Message 'Calling Invoke-NcSystemApi (volume snaplock show)'
                        (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>volume snaplock show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                    }
                    else {
                        Write-Verbose -Message 'Calling Invoke-NcSsh (volume snaplock show)'
                        Invoke-NcSsh -Command 'set -showseparator ~; set -showallfields true; volume snaplock show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                    }
                )

                $volSnaplockShow | Select-Object -ExcludeProperty PrivilegedDelete -Property *, @{
                    Name       = 'PrivilegedDeleteState'
                    Expression = { $_.PrivilegedDelete }
                }, @{
                    Name       = 'SnaplockComplianceClock'
                    Expression = {
                        if (($compClockTime = Convert-DateTimeString -String $_.ComplianceClockTime) -or ($compClockTime = Convert-DateTimeString -String (($_.ComplianceClockTime -split '\s' | Select-Object -Index 0, 1, 2, 3, 5, 6) -join ' ') -Format 'ddd MMM d HH:mm:ss yyyy zzz')) {
                            ConvertTo-UnixTimestamp -DateTime $compClockTime.ToUniversalTime()
                        }
                    }
                }, @{
                    Name       = 'VolumeExpiryTime'
                    Expression = {
                        if (-not [string]::IsNullOrWhiteSpace($_.ExpiryTime)) {
                            if ($_.ExpiryTime -ne 'none' -and (($expiryTime = Convert-DateTimeString -String $_.ExpiryTime) -or ($expiryTime = Convert-DateTimeString -String $_.ExpiryTime -Format 'ddd MMM d HH:mm:ss GMT yyyy'))) {
                                ConvertTo-UnixTimestamp -DateTime $expiryTime.ToUniversalTime()
                            }
                            else {
                                #if we are here, this means the format of the string wasn't "GMT", so just take the string as is for now
                                $_.ExpiryTime
                            }
                        }
                    }
                }
            )
        }

        #endregion Snaplock Information

        #region NVME Information

        ##Unused Cmdlet: Get-NcNvmeNamespaceOsType - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNvmeSubsystemOsType - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNvmeSubsystemController - (1.140+) Throws 'error in XML document' (ZAPI burt)
        ##Unused Cmdlet: Get-NcNvmeFeature - (1.140+) Not currently needed for build docs (cmdlet doesn't return all API props)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.140')) {
            Write-Verbose -Message '- Gathering NVME information'
            $systemData | Add-Member -MemberType NoteProperty -Name Nvme -Value (
                [pscustomobject]@{
                    Subsystem = [pscustomobject]@{ }
                }
            )

            Write-Verbose -Message 'Calling Get-NcNvme'
            $systemData.Nvme | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcNvme -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcNvmeInterface'
            $systemData.Nvme | Add-Member -MemberType NoteProperty -Name Interface -Value (Get-NcNvmeInterface -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcNvmeNamespace'
            $systemData.Nvme | Add-Member -MemberType NoteProperty -Name Namespace -Value (Get-NcNvmeNamespace -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcNvmeSubsystem'
            $systemData.Nvme.Subsystem | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcNvmeSubsystem -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcNvmeSubsystemHost'
            $systemData.Nvme.Subsystem | Add-Member -MemberType NoteProperty -Name Host -Value (Get-NcNvmeSubsystemHost -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcNvmeSubsystemMap'
            $systemData.Nvme.Subsystem | Add-Member -MemberType NoteProperty -Name Map -Value (Get-NcNvmeSubsystemMap -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion NVME Information

        #region UCM Information

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.21')) {
            Write-Verbose -Message '- Gathering UCM information -'
            $systemData | Add-Member -MemberType NoteProperty -Name Ucm -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcUcmAdapter'
            $systemData.Ucm | Add-Member -MemberType NoteProperty -Name Adapter -Value (Get-NcUcmAdapter -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion UCM Information

        #region WAFL

        ##Unused Cmdlet: Get-NcWaflSync - Not needed for build docs

        #endregion WAFL

        #region VM/VM Services Information

        ##Unused Cmdlet: Get-NcVMProviderCred - (1.100+) Not sure what to do with this yet (deprecated in 1.120)
        ##Unused Cmdlet: Get-NcVMProviderProxy - (1.100+) Not sure what to do with this yet (deprecated in 1.120)
        ##Unused Cmdlet: Get-NcVMSVsphereCred - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcVirtualMachineMaintenanceEvent - (1.150+) Not needed for build docs
        ##Unused Cmdlet: Get-NcVMSystemDisks - (1.100+) Deprecated in 1.150
        ##Unused Cmdlet: Get-NcVmDiskObjectStore - (1.150+) Not needed for build docs
        ##Unused Cmdlet: Get-NcVmDiskObjectStoreDnsCache - (1.150+) Not needed for build docs
        ##Unused Cmdlet: Get-NcVmNetworkLoadBalancer - (1.150+) Not currently needed for build docs

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message '- Gathering VM information -'
            $systemData | Add-Member -MemberType NoteProperty -Name Vm -Value (
                [pscustomobject]@{
                    Instance   = @( )
                    HyperVisor = @( )
                }
            )

            foreach ($node in ($systemData.Node | Where-Object { $_ })) {
                if ($node.NodeModel -like 'CDvM*') {
                    Write-Verbose -Message "Calling Get-NcVMInfo (Node: $( $node.Node ))"
                    $systemData.Vm.Instance += Get-NcVMInfo -Node $node.Node -Controller $Controller -ErrorVariable +NetAppDocsError
                }
                elseif ($node.NodeModel -like 'FDvM*') {
                    $systemData.Vm.HyperVisor += $(
                        if ($useSystemCli) {
                            Write-Verbose -Message "Calling Invoke-NcSystemApi (system node virtual-machine hypervisor show (Node: $( $node.Node ))"
                            (Invoke-NcSystemApi -Request "<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>system node virtual-machine hypervisor show -node $( $node.Node )</arg></args></system-cli>" -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                        }
                        else {
                            Write-Verbose -Message "Calling Invoke-NcSsh (system node virtual-machine hypervisor show (Node: $( $node.Node ))"
                            Invoke-NcSsh -Command "set -showseparator ~; set -showallfields true; system node virtual-machine hypervisor show -node $( $node.Node )" -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                        }
                    )
                }
            }
        }

        #endregion VM/VM Services Information

        #region Cluster Switch Information

        ##Unused Cmdlet: Get-NcClusterSwt - (1.150+) Using CLI for this now (bug in PSTK 4.5.0 output)
        ##Unused Cmdlet: Get-NcClusterSwitch - (1.160+) Using CLI for this
        ##Unused Cmdlet: Get-NcClusterSwitchPoll - (1.160+) Not needed for build docs

        $systemData | Add-Member -MemberType NoteProperty -Name ClusterSwitch -Value ([pscustomobject]@{ })

        $systemData.ClusterSwitch | Add-Member -MemberType NoteProperty -Name Info -Value $(
            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.32')) {
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (system cluster-switch show-all (priv advanced))'
                    (Invoke-NcSystemApi -Request '<system-cli><priv>advanced</priv><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>system cluster-switch show-all</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText | Where-Object { $_.SwVersion -notlike 'NetApp Release*' }
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (system cluster-switch show-all (priv advanced))'
                    Invoke-NcSsh -Command 'set -privilege advanced -confirmations off; set -showseparator ~; set -showallfields true; system cluster-switch show-all' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            }
            elseif ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (system health cluster-switch show)'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>system health cluster-switch show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText | Where-Object { $_.SwVersion -notlike 'NetApp Release*' }
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (system health cluster-switch show)'
                    Invoke-NcSsh -Command 'set -showseparator ~; set -showallfields true; system health cluster-switch show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText | Where-Object { $_.SwVersion -notlike 'NetApp Release*' }
                }
            }
        )

        $systemData.ClusterSwitch | Add-Member -MemberType NoteProperty -Name Interface -Value $(
            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.32')) {
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (system cluster-switch interface show (priv diag))'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -privilege diagnostic -confirmations off;</arg><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>system cluster-switch interface show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (system cluster-switch interface show (priv diag))'
                    Invoke-NcSsh -Command 'set -privilege diagnostic -confirmations off; set -showseparator ~; set -showallfields true; system cluster-switch interface show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            }
            elseif ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (system health cluster-switch interface show (priv diag))'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -privilege diagnostic -confirmations off;</arg><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>system health cluster-switch interface show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (system health cluster-switch interface show (priv diag))'
                    Invoke-NcSsh -Command 'set -privilege diagnostic -confirmations off; set -showseparator ~; set -showallfields true; system health cluster-switch interface show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            }
        )

        #endregion Cluster Switch Information

        #region Template Information

        ##Unused Cmdlet: Get-NcTemplateManagementTemplateProvisionError - (1.130+) Not needed for build docs

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.130')) {
            Write-Verbose -Message '- Gathering template information -'
            $systemData | Add-Member -MemberType NoteProperty -Name Template -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcTemplateManagementTemplate'
            $systemData.Template | Add-Member -MemberType NoteProperty -Name ManagementTemplate -Value (Get-NcTemplateManagementTemplate -Controller $Controller -ErrorVariable +NetAppDocsError)

            #BURT 1140022: template-management-template-parameter-get-iter hangs on 9.3 systems (using CLI instead)
            ##The -iter portion of the API is broken if max-records <= parameterCount
            #Write-Verbose -Message 'Calling Get-NcTemplateManagementTemplateParameter
            #$systemData.Template | Add-Member -MemberType NoteProperty -Name ManagementTemplateParameter -Value (Get-NcTemplateManagementTemplateParameter -Controller $Controller -ErrorVariable +NetAppDocsError)

            $systemData.Template | Add-Member -MemberType NoteProperty -Name ManagementTemplateParameter -Value $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (template parameter show)'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>template parameter show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (template parameter show)'
                    Invoke-NcSsh -Command 'set -showseparator ~; set -showallfields true; template parameter show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            )
        }

        #endregion Template Information

        #region Disk Qual Information

        Write-Verbose -Message '- Gathering DQP information -'
        $dqpRegex = '(?s)^(?:Node[:]\s*(?<Node>.*?))\s+.*?(?:Datecode[:]\s*(?<PackageDate>.*?))\s+.*?(?:FileName[=]\s*(?<FileName>.*?))\s+.*?(?:FileVersion[=]\s*(?<FileVersion>.*?))\s+.*?(?:DriveRecordCount[=]\s*(?<DriveRecordCount>.*?))\s+.*?(?:AliasRecordCount[=]\s*(?<AliasRecordCount>.*?))\s+.*?(?:DeviceRecordCount[=]\s*(?<DeviceRecordCount>.*?))\s+.*?(?:SystemRecordCount[=]\s*(?<SystemRecordCount>.*?))\s+.*$'

        $systemData | Add-Member -MemberType NoteProperty -Name DiskQualPackage -Value $(
            $diskQual = $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (system node run -node * -command rdfile /etc/qual_devices_v3)'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>system node run -node * -command rdfile /etc/qual_devices_v3</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (system node run -node * -command rdfile /etc/qual_devices_v3)'
                    Invoke-NcSsh -Command 'system node run -node * -command rdfile /etc/qual_devices_v3' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false
                }
            )

            if ($diskQual) {
                $diskQual -split '(?=Node[:]\s)' | Select-Object -Skip 1 | Import-WithRegex -RegexString $dqpRegex
            }
        )

        #endregion Disk Qual Information

        #region Debug Information

        Write-Verbose -Message '- Gathering debug information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Debug -Value ([pscustomobject]@{ })

        try {
            $systemData.Debug | Add-Member -MemberType NoteProperty -Name Vreport -Value $(
                $debugVreportShow = $(
                    if ($useSystemCli) {
                        Write-Verbose -Message 'Calling Invoke-NcSystemApi (debug vreport show (priv diag))'
                        (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -privilege diagnostic -confirmations off;</arg><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>debug vreport show</arg></args></system-cli>' -Controller $Controller -ErrorAction Stop).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                    }
                    else {
                        Write-Verbose -Message 'Calling Invoke-NcSsh (debug vreport show (priv diag))'
                        Invoke-NcSsh -Command 'set -privilege diagnostic -confirmations off; set -showseparator ~; set -showallfields true; debug vreport show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                    }
                )

                if ($debugVreportShow) {
                    $debugVreportShow | Select-Object -ExcludeProperty Attributes -Property *, @{
                        Name       = 'AttributesList'
                        Expression = { $_.Attributes -split '\s*,\s*' | ForEach-Object { $_ -replace '^\s*\"|\"\s*$' } }
                    }
                }
            )
        }
        catch {
            if ($_.Exception.Message -like '*There is an error in XML document*' -or $_.Exception.Message -like '*Convert-SystemCliTextInstance*') {
                $Global:Error.RemoveAt(0)
            }
            else {
                $NetAppDocsError += $_
            }
        }

        #endregion Debug Information

        Write-Verbose -Message '- Getting current timestamp -'
        if ($systemData.System.OntapiVersion -lt [version]::Parse('1.30')) {
            $systemData | Add-Member -MemberType NoteProperty -Name ReferenceDT -Value (ConvertFrom-UnixTimestamp -Seconds (Get-NcClusterHa -Node ($systemData.Net.Interface | Where-Object { $_.Address -eq $Controller.Address.IpAddressToString }).CurrentNode -Controller $Controller -ErrorVariable +NetAppDocsError).CurrentTime)
        }
        else {
            $systemData | Add-Member -MemberType NoteProperty -Name ReferenceDT -Value (ConvertFrom-UnixTimestamp -Seconds (Get-NcTime -Controller $Controller -ErrorVariable +NetAppDocsError).UtcTime)
        }

        Write-Verbose -Message '- Identifying SystemName -'
        $systemData | Add-Member -MemberType NoteProperty -Name SystemName -Value $systemData.Cluster.Info.ClusterName

        Write-Verbose -Message '- Identifying any errors during processing -'
        $systemData | Add-Member -MemberType NoteProperty -Name DiscoveryError -Value @(
            foreach ($err in ($NetAppDocsError | Where-Object { $_ })) {
                $err.CategoryInfo | Select-Object -Property *, @{
                    Name       = 'Message'
                    Expression = { $err.Exception.Message }
                }
            }
        )

        if (($systemData.DiscoveryError | Measure-Object).Count -gt 0) {
            $msg = New-Object -TypeName System.Text.StringBuilder

            [void]$msg.AppendLine("Errors were encountered while collecting data from system: $( $Controller.Name )")
            [void]$msg.AppendLine('-------------------------------------------------------------------------------------')
            [void]$msg.AppendLine('Please review the output for accuracy and, if necessary, correct any error conditions')
            [void]$msg.AppendLine('then run these scripts again.')
            [void]$msg.AppendLine('-------------------------------------------------------------------------------------')
            [void]$msg.AppendLine()

            Write-Warning -Message $msg.ToString()
        }

        $systemData
    }
    catch {
        $Global:Error.RemoveAt(0)

        $msg = New-Object -TypeName System.Text.StringBuilder

        [void]$msg.AppendLine('An unexpected error occurred. Processing halted.')
        [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
        [void]$msg.AppendLine('As this error is not expected, please consider contacting the support staff via e-mail or by')
        [void]$msg.AppendLine('starting a thread on the community site.')
        [void]$msg.AppendLine('            E-mail: ng-NetAppDocs-support@netapp.com')
        [void]$msg.AppendLine('    Community site: https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs')
        [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
        [void]$msg.AppendLine($( $_.Exception.GetBaseException().Message ))
        [void]$msg.AppendLine("Location: $( $_.ScriptStackTrace )")
        [void]$msg.AppendLine()

        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $_.CategoryInfo.TargetName)))
    }
}

# SIG # Begin signature block
# MIIq0AYJKoZIhvcNAQcCoIIqwTCCKr0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDhOgea136Fs7gn
# xPzYijYq5UnhNiihhysw7CiCaKlVZqCCJGgwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggWNMIIEdaADAgECAhAOmxiO
# +dAt5+/bUOIIQBhaMA0GCSqGSIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYD
# VQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAi
# BgNVBAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAw
# MDBaFw0zMTExMDkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
# Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERp
# Z2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBAL/mkHNo3rvkXUo8MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsb
# hA3EMB/zG6Q4FutWxpdtHauyefLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iT
# cMKyunWZanMylNEQRBAu34LzB4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGb
# NOsFxl7sWxq868nPzaw0QF+xembud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclP
# XuU15zHL2pNe3I6PgNq2kZhAkHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCr
# VYJBMtfbBHMqbpEBfCFM1LyuGwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFP
# ObURWBf3JFxGj2T3wWmIdph2PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTv
# kpI6nj3cAORFJYm2mkQZK37AlLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWM
# cCxBYKqxYxhElRp2Yn72gLD76GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls
# 5Q5SUUd0viastkF13nqsX40/ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBR
# a2+xq4aLT8LWRV+dIPyhHsXAj6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6
# MIIBNjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qY
# rhwPTzAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8E
# BAMCAYYweQYIKwYBBQUHAQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5k
# aWdpY2VydC5jb20wQwYIKwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDig
# NoY0aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNybDARBgNVHSAECjAIMAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCg
# v0NcVec4X6CjdBs9thbX979XB72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQT
# SnovLbc47/T/gLn4offyct4kvFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh
# 65ZyoUi0mcudT6cGAxN3J0TU53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSw
# uKFWjuyk1T3osdz9HNj0d1pcVIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAO
# QGPFmCLBsln1VWvPJ6tsds5vIy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjD
# TZ9ztwGpn1eqXijiuZQwggXMMIIENKADAgECAhAg429sPxgagUb53pPffJfkMA0G
# CSqGSIb3DQEBDAUAMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBS
# MzYwHhcNMjEwOTA5MDAwMDAwWhcNMjMwOTA5MjM1OTU5WjBQMQswCQYDVQQGEwJV
# UzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UECgwMTmV0QXBwLCBJbmMuMRUw
# EwYDVQQDDAxOZXRBcHAsIEluYy4wggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGK
# AoIBgQC4kYYj/oViZD9pN03hrqFjtwOz1Gx4eDWVV8IYUYEr2qpLeNYvWz26B/+E
# mYLboAGVpkYg0Wske3hjRpooZlMORm1S4+2C2NoQCstZ+kmNTevbXHJe/w1VNJrm
# fKjpXOJEfx8GUGDqCMR30dChVpsdBEavrM7T0hnfJLv18i19SGv3a+nhvB3iOjLP
# SEg8+DFkjD5jTI6fQ83DBEiVR4UEYq7yyH578ZEylhsRfJmK+uIREycEUk/NpFTw
# g6/7lxh/WYabCrZzqI4Ep3QataRKaAlxt3BiOlZkX4WGX3DYvSYltWOM2UlCcGpP
# m/1/LN8Xuvf+YO6H472QecHe59XtXbHSMKvqDuOEJu8Wk2LFMNK732o3Fc5QIHie
# 6JtNsgNYf+Vsm5EMBD1ZHNE+C3zlmQbbWIU5uEU1nhiodBPKsu0SkS6pARyKBO05
# DSc2gWWv0aukKi04ZAn+hAdSl5s1dgmU5shBvKHqe15K9ZRN4FFO/CgHE0BoGYQS
# UQVKwa0CAwEAAaOCAZwwggGYMB8GA1UdIwQYMBaAFA8qyyCHKLjsb0iuK1SmKaoX
# pM0MMB0GA1UdDgQWBBQuH643KcBMmb/Q6IZt+H9IrnXFwDAOBgNVHQ8BAf8EBAMC
# B4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDAzARBglghkgBhvhC
# AQEEBAMCBBAwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwIwJTAjBggrBgEFBQcC
# ARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQBMEkGA1UdHwRCMEAw
# PqA8oDqGOGh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVT
# aWduaW5nQ0FSMzYuY3JsMHkGCCsGAQUFBwEBBG0wazBEBggrBgEFBQcwAoY4aHR0
# cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQVIz
# Ni5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqG
# SIb3DQEBDAUAA4IBgQCOoGdXjP8Sif0h3ZvoDsIVfrJvQUdP9pZExRJGGj/Te6ML
# XyqHNc/G2WklNv+BC4ryNJ4zp0oneHOub55ejrYPfhgP9oFWS8oOPJbrp3rAtINa
# OeKRC88WUYZSKYAxSX6kubEjnQD6cmazXjxVN6P8+6q9smeqF3xI+++c+ekMw3Xv
# 4EWRYXqN0/srfFp1WpoluwSQcZMqj9haSX0bgO6AAWH2AnVJMfXTIKz/0FQ/RW0y
# Ed5QYQqYInhA7IUz9dBziNpJlwONeF5j7xzgfYDY63WU6WrgJnGjYkQCOOsUx74j
# gUiMRe9zV48GS8Vxz22c/TQs0x4/1GmOSJvUHEk3GseBmB3v+yEhh/D6zWOfYP4X
# D/9b91CxmugTuepuaJSeRg+qUm3KarKsOCUF/CLqUrNh/JwKrWD1cghRaYLvMucs
# ScksHHe7ZDvb2OtvxWXjPk1d1NKvEwFJSS6hIVIbug9x28AJqOoP5Pn9wZvJSzvW
# uQJGCLscdGyKefdCo30wggYaMIIEAqADAgECAhBiHW0MUgGeO5B5FSCJIRwKMA0G
# CSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290
# IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFQxCzAJBgNVBAYT
# AkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28g
# UHVibGljIENvZGUgU2lnbmluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEBAQUAA4IB
# jwAwggGKAoIBgQCbK51T+jU/jmAGQ2rAz/V/9shTUxjIztNsfvxYB5UXeWUzCxEe
# AEZGbEN4QMgCsJLZUKhWThj/yPqy0iSZhXkZ6Pg2A2NVDgFigOMYzB2OKhdqfWGV
# oYW3haT29PSTahYkwmMv0b/83nbeECbiMXhSOtbam+/36F09fy1tsB8je/RV0mIk
# 8XL/tfCK6cPuYHE215wzrK0h1SWHTxPbPuYkRdkP05ZwmRmTnAO5/arnY83jeNzh
# P06ShdnRqtZlV59+8yv+KIhE5ILMqgOZYAENHNX9SJDm+qxp4VqpB3MV/h53yl41
# aHU5pledi9lCBbH9JeIkNFICiVHNkRmq4TpxtwfvjsUedyz8rNyfQJy/aOs5b4s+
# ac7IH60B+Ja7TVM+EKv1WuTGwcLmoU3FpOFMbmPj8pz44MPZ1f9+YEQIQty/NQd/
# 2yGgW+ufflcZ/ZE9o1M7a5Jnqf2i2/uMSWymR8r2oQBMdlyh2n5HirY4jKnFH/9g
# Rvd+QOfdRrJZb1sCAwEAAaOCAWQwggFgMB8GA1UdIwQYMBaAFDLrkpr/NZZILyhA
# QnAgNpFcF4XmMB0GA1UdDgQWBBQPKssghyi47G9IritUpimqF6TNDDAOBgNVHQ8B
# Af8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcD
# AzAbBgNVHSAEFDASMAYGBFUdIAAwCAYGZ4EMAQQBMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Um9vdFI0Ni5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ1Jvb3RSNDYu
# cDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQwFAAOCAgEABv+C4XdjNm57oRUgmxP/BP6YdURhw1aVcdGRP4Wh60BAscjW
# 4HL9hcpkOTz5jUug2oeunbYAowbFC2AKK+cMcXIBD0ZdOaWTsyNyBBsMLHqafvIh
# rCymlaS98+QpoBCyKppP0OcxYEdU0hpsaqBBIZOtBajjcw5+w/KeFvPYfLF/ldYp
# mlG+vd0xqlqd099iChnyIMvY5HexjO2AmtsbpVn0OhNcWbWDRF/3sBp6fWXhz7Dc
# ML4iTAWS+MVXeNLj1lJziVKEoroGs9Mlizg0bUMbOalOhOfCipnx8CaLZeVme5yE
# Lg09Jlo8BMe80jO37PU8ejfkP9/uPak7VLwELKxAMcJszkyeiaerlphwoKx1uHRz
# NyE6bxuSKcutisqmKL5OTunAvtONEoteSiabkPVSZ2z76mKnzAfZxCl/3dq3dUNw
# 4rg3sTCggkHSRqTqlLMS7gjrhTqBmzu1L90Y1KWN/Y5JKdGvspbOrTfOXyXvmPL6
# E52z1NZJ6ctuMFBQZH3pwWvqURR8AgQdULUvrxjUYbHHj95Ejza63zdrEcxWLDX6
# xWls/GDnVNueKjWUH3fTv1Y8Wdho698YADR7TNx8X8z2Bev6SivBBOHY+uqiirZt
# g0y9ShQoPzmCcn63Syatatvx157YK9hlcPmVoa1oDE5/L9Uo2bC5a4CH2Rwwggau
# MIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJ
# BgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5k
# aWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAe
# Fw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcw
# FQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3Rl
# ZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9Ge
# TKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0
# hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZl
# jZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAsh
# aG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVY
# TXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1
# biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCir
# c0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+
# DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA
# +bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42Pg
# puE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzS
# M7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU
# uhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6
# mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsG
# AQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29t
# MEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNl
# cnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3Js
# My5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAE
# GTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1Z
# jsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8d
# B+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVp
# P0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp8
# 76i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2
# nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3
# ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQ
# txMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc
# 4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+Y
# AN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZ
# vAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQr
# H4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIGwDCCBKigAwIBAgIQDE1p
# ckuU+jwqSj0pB4A9WjANBgkqhkiG9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4XDTIyMDkyMTAwMDAw
# MFoXDTMzMTEyMTIzNTk1OVowRjELMAkGA1UEBhMCVVMxETAPBgNVBAoTCERpZ2lD
# ZXJ0MSQwIgYDVQQDExtEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMiAtIDIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDP7KUmOsap8mu7jcENmtuh6BSFdDMa
# JqzQHFUeHjZtvJJVDGH0nQl3PRWWCC9rZKT9BoMW15GSOBwxApb7crGXOlWvM+xh
# iummKNuQY1y9iVPgOi2Mh0KuJqTku3h4uXoW4VbGwLpkU7sqFudQSLuIaQyIxvG+
# 4C99O7HKU41Agx7ny3JJKB5MgB6FVueF7fJhvKo6B332q27lZt3iXPUv7Y3UTZWE
# aOOAy2p50dIQkUYp6z4m8rSMzUy5Zsi7qlA4DeWMlF0ZWr/1e0BubxaompyVR4aF
# eT4MXmaMGgokvpyq0py2909ueMQoP6McD1AGN7oI2TWmtR7aeFgdOej4TJEQln5N
# 4d3CraV++C0bH+wrRhijGfY59/XBT3EuiQMRoku7mL/6T+R7Nu8GRORV/zbq5Xwx
# 5/PCUsTmFntafqUlc9vAapkhLWPlWfVNL5AfJ7fSqxTlOGaHUQhr+1NDOdBk+lbP
# 4PQK5hRtZHi7mP2Uw3Mh8y/CLiDXgazT8QfU4b3ZXUtuMZQpi+ZBpGWUwFjl5S4p
# kKa3YWT62SBsGFFguqaBDwklU/G/O+mrBw5qBzliGcnWhX8T2Y15z2LF7OF7ucxn
# EweawXjtxojIsG4yeccLWYONxu71LHx7jstkifGxxLjnU15fVdJ9GSlZA076XepF
# cxyEftfO4tQ6dwIDAQABo4IBizCCAYcwDgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB
# /wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwIAYDVR0gBBkwFzAIBgZngQwB
# BAIwCwYJYIZIAYb9bAcBMB8GA1UdIwQYMBaAFLoW2W1NhS9zKXaaL3WMaiCPnshv
# MB0GA1UdDgQWBBRiit7QYfyPMRTtlwvNPSqUFN9SnDBaBgNVHR8EUzBRME+gTaBL
# hklodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0
# MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3JsMIGQBggrBgEFBQcBAQSBgzCBgDAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFgGCCsGAQUFBzAC
# hkxodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRS
# U0E0MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3J0MA0GCSqGSIb3DQEBCwUAA4IC
# AQBVqioa80bzeFc3MPx140/WhSPx/PmVOZsl5vdyipjDd9Rk/BX7NsJJUSx4iGNV
# CUY5APxp1MqbKfujP8DJAJsTHbCYidx48s18hc1Tna9i4mFmoxQqRYdKmEIrUPwb
# tZ4IMAn65C3XCYl5+QnmiM59G7hqopvBU2AJ6KO4ndetHxy47JhB8PYOgPvk/9+d
# EKfrALpfSo8aOlK06r8JSRU1NlmaD1TSsht/fl4JrXZUinRtytIFZyt26/+YsiaV
# OBmIRBTlClmia+ciPkQh0j8cwJvtfEiy2JIMkU88ZpSvXQJT657inuTTH4YBZJwA
# wuladHUNPeF5iL8cAZfJGSOA1zZaX5YWsWMMxkZAO85dNdRZPkOaGK7DycvD+5sT
# X2q1x+DzBcNZ3ydiK95ByVO5/zQQZ/YmMph7/lxClIGUgp2sCovGSxVK05iQRWAz
# gOAj3vgDpPZFR+XOuANCR+hBNnF3rf2i6Jd0Ti7aHh2MWsgemtXC8MYiqE+bvdgc
# mlHEL5r2X6cnl7qWLoVXwGDneFZ/au/ClZpLEQLIgpzJGgV8unG1TnqZbPTontRa
# mMifv427GFxD9dAq6OJi7ngE273R+1sKqHB+8JeEeOMIA11HLGOoJTiXAdI/Otrl
# 5fbmm9x+LMz/F0xNAKLY1gEOuIvu5uByVYksJxlh9ncBjDGCBb4wggW6AgEBMGgw
# VDELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDErMCkGA1UE
# AxMiU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIFIzNgIQIONvbD8YGoFG
# +d6T33yX5DANBglghkgBZQMEAgEFAKCBhDAYBgorBgEEAYI3AgEMMQowCKACgACh
# AoAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAM
# BgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCB6HcXd5670TK7R/9yTQGprxDX/
# gAZ81kO1qhtYEhOWPzANBgkqhkiG9w0BAQEFAASCAYCB1eIbzPWTNIqeA58LvuXJ
# Kda44lefY7H37HG5al+ITACXapSJ6kPlqa6DbNay7JPq758RL6rOA8zM51eu2kpP
# +5sGWZw624AZRDbet3L0+wp0vT4PoikVJWU9SK8G2jDlIlpU9BySkydB0cYOe3ry
# 6snLDhX+z9HAY2y7NPUgXscf+BfJJnO/Lkxi5pP07u8jZIUw7vyo05X4nsEIYZq+
# II7q/0mZQ1rkaPaibvziOqNwr5VEyO2I9I0PGNROCGIEkntthaoNrYy6rIGISrcY
# dtxFFz/6GQ7Zf6fJ2w3nNzKYdLevGfHv28P4yAWrSdMh8FMXt7OJYeyECQVRI28i
# rKJLjNrNH0XV+z+e7AwdZgx9GXSh6rN2G2wtAE/HQJN20GPFdyKU3vxwbBAcuMSP
# O7IqoV9GeLCC4k6Adf6FtTZWYxVnjhTn3DnrH+NlaUsj9tPIiP9PrFP3dsoGvXq6
# qShRFqVRAu+uNQZPakwLQQAe06aBBdBVCmNwZeDF+xyhggMgMIIDHAYJKoZIhvcN
# AQkGMYIDDTCCAwkCAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
# cnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
# SEEyNTYgVGltZVN0YW1waW5nIENBAhAMTWlyS5T6PCpKPSkHgD1aMA0GCWCGSAFl
# AwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUx
# DxcNMjMwNDAyMTUyODI3WjAvBgkqhkiG9w0BCQQxIgQgtHK0/rncNjqoP2ILK4sZ
# zR7T217lQX7Mkf+rJpUcGFswDQYJKoZIhvcNAQEBBQAEggIAHKIrUjjZEsNvnIyJ
# HHyz+QiZU2fSPS4QBH/cUXbi7QTJk1PuszxgZ9dlh1FplAMn4Vy87QIOugMbz5Qs
# Ovo9HfLTPXMUyniEtF6smNsOEPDOKVAS+aoehpeShgNAYUqLXqHp8cRpB7im6CDY
# yKlM2wIfR9RriZ/luEXMgQuI320Ao8vADhfFnpZNTevbVmchz5uHUH2Od3dJWip/
# Zn0n28DevoFgIeNp4f+oiukAT6WAmZbajz7xBgzKiDYZdCv0RKShVNFiGMJkh6uy
# CGSudyfKZs1tCzbnDBtEyEWygX4Mj9900BvguPWHGfsWIbcjvqIvYauocnwM1pVI
# 7M1O84EV+G4FeAwjmm7HEm6Y+MDHzcPvDzT74gLLdUcmxGgkp8VFypzqoPu3R2uE
# 3oKy+ava9+L0XrGlKwNfTQNj4uokyvhw2kDCbfh3vesFI4wLfenFVHkqDeyWyB1R
# 9IMed3Au9CQ6uaaZBa34LlAYCrdHVCPV3THVVvJigrxjYv40t/JYuijZAhuLBj+X
# IApFi6SztSjCJLpAEWf5WY/4VYsqYzJQ15ce9uWvgy3VSSRVYrTFadS5xMfGPYYQ
# jffn2lj6m6LpuZFC8xDtEXseMzKJDcvSUJxbrRxRnRsZkOEDvXDTd3Ur7yE2xTLz
# wvl5BImcN6BFW5f9LW4H7n6lUDk=
# SIG # End signature block
