function Get-NtapFilerAsupData {
    <#
        .SYNOPSIS
            Collects data for Data ONTAP 7-Mode storage systems using ASUPs.

        .DESCRIPTION
            Uses the controller ASUP to gather as much configuration information as possible. This
            data can be used to automate build docs and health checks. It can also be used for historical
            configuration information. If the -System parameter is used, the latest ASUP of the specified
            AsupSubject type will be used. If an older or specific ASUP is desired, use the
            Find-NtapFilerAsup function and pass those results to this function. The ASUPs will be validated
            to determine that an ASUP from each node is present, complete and generated at similar times.

        .EXAMPLE
            PS> Get-NtapFilerAsupData -System 'mtharvard:30020945!mtmassive:30020946' | Export-Clixml -Path 'D:\mtharvard-mtmassive_Asup_raw.xml'

            Collects the controller information from the HA pair information specified and saves the results
            to a .xml file for further processing.

        .EXAMPLE
            PS> Get-NtapFilerAsupData -System 'mtharvard:30020945!mtmassive:30020946' -AsupSubject user_triggered | Export-Clixml -Path 'D:\mtharvard-mtmassive_Asup_raw.xml'

            Collects the controller information from the HA pair information specified, using the latest user_triggered
            ASUP set, and saves the results to a .xml file for further processing.

        .EXAMPLE
            PS> Get-NtapFilerAsupData -System 'mtharvard:30020945!mtmassive:30020946' | Format-NtapFilerData | Out-NtapDocument -WordFile 'D:\Output\mtharvard-mtmassive_Asup.docx' -ExcelFile 'D:\Output\mtharvard-mtmassive_Asup.xlsx'

            Collects the controller information from the system information specified, formats the data and creates
            the specified Word and Excel documents.

        .EXAMPLE
            PS> Get-NtapFilerAsupData -AsupId 2015011716380597!2015011716330550 | Export-Clixml -Path 'D:\Output\mtmassive_mtharvard-Asup_raw.xml'

            Perform the data collection using specific ASUPIDs.

        .EXAMPLE
            PS> Get-NtapFilerAsupData -System 'mtmassive:30020946' -Force | Export-Clixml -Path 'D:\mtmassive_Asup_raw.xml'

            Collects the controller information from the system name and serial number specified and saves the
            results to a .xml file for further processing. If this node is a member of an HA pair and, for some
            reason, you do not want the data from the other node the -Force parameter must be specified (Note: in
            almost all cases this will result in incomplete data).

        .EXAMPLE
            PS> Find-NtapFilerAsup -System 'mtharvard:30020945!mtmassive:30020946' -StartDate 10-15-2014 -EndDate 11-15-2014 | Get-NtapFilerAsupData | Export-Clixml -Path 'D:\Output\mtmassive_mtharvard-Asup_raw.xml'

            Finds the first validated weekly ASUP set for the specified HA pair by searching between the dates
            specified, performs the data collection and saves the results to a .xml file for further processing.

        .LINK
            https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs

        .LINK
            Format-NtapFilerData

        .INPUTS
            System.Xml.XmlElement

        .OUTPUTS
            System.Management.Automation.PSCustomObject

        .NOTES
            AUTHOR : Jason Cole
            REQUIRES : Internal connection to NetApp
    #>

    [CmdletBinding(DefaultParameterSetName = 'BySystemInfo')]
    [OutputType([System.Management.Automation.PSCustomObject])]
    param (
        # The system information to search for (e.g., HA pair: 'Node1Hostname:Node1SerialNumber!Node2Hostname:Node2SerialNumber'
        # and stand-alone: 'NodeName:NodeSerialNumber').
        [Parameter(Mandatory, Position = 0, ParameterSetName = 'BySystemInfo')]
        [ValidateScript({
                if ($_ -notmatch '^\s*[^!:]+[:]\d+(?:[!][^!:]+[:]\d+)?\s*$') {
                    throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]"$_ is not a valid format for the System parameter (e.g., (HA pair): 'Node1Hostname:Node1SerialNumber!Node2Hostname:Node2SerialNumber') or (stand-alone): 'NodeHostname:NodeSerialNumber'.", 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $_))
                }

                $true
            })]
        [string[]]$System,

        # The ASUPID(s) to search for. An ASUPID from each node in the HA pair is required. Each node's ASUPID must be
        # separated by an exclamation point (e.g., 2015012500120554!2015012423440127). Sets of ASUPIDs from multiple
        # HA pairs must be comma separated (e.g., 2015012500120554!2015012423440127, 2015062302150043!2015062302150041).
        [Parameter(Mandatory, Position = 0, ParameterSetName = 'ByAsupId')]
        [ValidateScript({
                if ($_ -notmatch '^\s*\d+(?:\s*[!]\s*\d+){0,1}$') {
                    throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]"$_ is not a valid format for the AsupId parameter (e.g., (HA pair): 'AsupId_Node1!AsupId_Node2') or (stand-alone): 'AsupId_Node1'.", 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $_))
                }

                $true
            })]
        [string[]]$AsupId,

        # An ASUP object passed from the Find-NtapFilerAsup function.
        [Parameter(Mandatory, Position = 0, ParameterSetName = 'ByAsupObject', ValueFromPipeline)]
        [ValidateNotNullOrEmpty()]
        [System.Xml.XmlElement[]]$AsupObject,

        # The type of ASUP to query for (weekly or user_triggered).
        [Parameter(ParameterSetName = 'BySystemInfo')]
        [ValidateSet('weekly', 'user_triggered')]
        [string]$AsupSubject = 'weekly',

        # Forces the use of a invalid ASUP (i.e., ASUP content missing or mismatched generated dates between
        # the nodes in the HA pair).
        [Parameter()]
        [switch]$Force
    )

    begin {
        if ($PSCmdlet.MyInvocation.PipelinePosition -eq 1) {
            try {
                Invoke-SsoLogin -ErrorAction Stop
            }
            catch {
                $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($_.Exception.Message, 'InvalidResult', [System.Management.Automation.ErrorCategory]::InvalidResult, 'ASUP REST Server')))
            }

            Write-Verbose -Message 'Validating connectivity to ASUP REST server'

            if (-not (Test-AsupRestConnectivity)) {
                $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('Unable to contact NetApp ASUP REST server. Please ensure you have an internal connection to NetApp.', 'ResourceUnavailable', [System.Management.Automation.ErrorCategory]::ResourceUnavailable, 'ASUP REST Server')))
            }
        }
    }
    process {
        Write-Verbose -Message "Initializing function: $( $PSCmdlet.MyInvocation.MyCommand.Name ) ($( $PSCmdlet.MyInvocation.MyCommand.ModuleName ) v$( (Get-NtapDocsVersion -AsVersion).ToString() ))"

        switch -Exact ($PSCmdlet.ParameterSetName) {
            'BySystemInfo' {
                foreach ($nodeSet in ($System | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })) {
                    if (($asupSet = Find-NtapFilerAsup -System $nodeSet -AsupSubject $AsupSubject -ValidateAsupSystemType 7-Mode)) {
                        if (($asupSet | Where-Object { $_.is_valid -eq $false } | Measure-Object).Count -eq 0 -or $Force.IsPresent) {
                            foreach ($asup in $asupSet) {
                                try {
                                    $functionStartTime = [datetime]::Now

                                    $asupData = Get-FilerAsupData -Asup $asup

                                    $asupData

                                    $statsParams = @{
                                        FunctionPSCallStack = Get-PSCallStack
                                        TargetDetails       = [pscustomobject]@{
                                            SystemUid     = [string]::Join('_', $asupData.SystemInfo.SystemId, $asupData.SystemInfo.SerialNo)
                                            SystemType    = $asupData.pstypenames[0]
                                            SystemVersion = $asupData.SystemInfo.VersionInfo.Version
                                        }
                                        ElapsedTicks        = ([datetime]::Now - $functionStartTime).Ticks
                                        ExcludeParameter    = 'System'
                                    }

                                    Write-Statistics @statsParams
                                }
                                catch {
                                    Write-Error -ErrorRecord $_
                                }
                            }
                        }
                    }
                }
            }
            'ByAsupId' {
                foreach ($haPairAsupId in ($AsupId | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })) {
                    if (($asupSet = Find-NtapFilerAsup -AsupId $haPairAsupId -ValidateAsupSystemType 7-Mode)) {
                        if (($asupSet | Where-Object { $_.is_valid -eq $false } | Measure-Object).Count -eq 0 -or $Force.IsPresent) {
                            foreach ($asup in $asupSet) {
                                try {
                                    $functionStartTime = [datetime]::Now

                                    $asupData = Get-FilerAsupData -Asup $asup

                                    $asupData

                                    $statsParams = @{
                                        FunctionPSCallStack = Get-PSCallStack
                                        TargetDetails       = [pscustomobject]@{
                                            SystemUid     = [string]::Join('_', $asupData.SystemInfo.SystemId, $asupData.SystemInfo.SerialNo)
                                            SystemType    = $asupData.pstypenames[0]
                                            SystemVersion = $asupData.SystemInfo.VersionInfo.Version
                                        }
                                        ElapsedTicks        = ([datetime]::Now - $functionStartTime).Ticks
                                        ExcludeParameter    = 'AsupId'
                                    }

                                    Write-Statistics @statsParams
                                }
                                catch {
                                    Write-Error -ErrorRecord $_
                                }
                            }
                        }
                    }
                }
            }
            'ByAsupObject' {
                if ($AsupObject) {
                    if (($AsupObject | Where-Object { $_.sys_operating_mode -eq 'Cluster-Mode' } | Measure-Object).Count -gt 0) {
                        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('One or more ASUPS in the ASUP set is from an ONTAP cluster system. Please use Get-NtapClusterAsupData for ONTAP ASUPs.', 'InvalidType', [System.Management.Automation.ErrorCategory]::InvalidType, $AsupObject)))
                    }

                    if (($AsupObject | Where-Object { $_.is_valid -eq $false } | Measure-Object).Count -eq 0 -or $Force.IsPresent) {
                        foreach ($asup in ($AsupObject | Where-Object { $_ })) {
                            try {
                                $functionStartTime = [datetime]::Now

                                $asupData = Get-FilerAsupData -Asup $asup

                                $asupData

                                $statsParams = @{
                                    FunctionPSCallStack = Get-PSCallStack
                                    TargetDetails       = [pscustomobject]@{
                                        SystemUid     = [string]::Join('_', $asupData.SystemInfo.SystemId, $asupData.SystemInfo.SerialNo)
                                        SystemType    = $asupData.pstypenames[0]
                                        SystemVersion = $asupData.SystemInfo.VersionInfo.Version
                                    }
                                    ElapsedTicks        = ([datetime]::Now - $functionStartTime).Ticks
                                    ExcludeParameter    = 'AsupObject'
                                }

                                Write-Statistics @statsParams
                            }
                            catch {
                                Write-Error -ErrorRecord $_
                            }
                        }
                    }
                }
            }
        }
    }
}

function Get-FilerAsupData {
    [CmdletBinding()]
    param (
        [Parameter(Mandatory)]
        [ValidateNotNullOrEmpty()]
        [System.Xml.XmlElement]$Asup
    )

    $systemAsupData = [pscustomobject]@{ }
    $systemAsupData.pstypenames.Insert(0, 'NetAppDocs.NaController.Asup.Data')

    $NetAppDocsError = @( )

    Write-Verbose -Message '- Capturing module version -'
    $systemAsupData | Add-Member -MemberType NoteProperty -Name NetAppDocsVersion (Get-NtapDocsVersion -AsVersion)

    Write-Verbose -Message '- Capturing PowerShell version -'
    $systemAsupData | Add-Member -MemberType NoteProperty -Name PowerShellVersion -Value $PSVersionTable.PSVersion

    Write-Verbose -Message '- Capturing ASUP MetaData -'
    $systemAsupData | Add-Member -MemberType NoteProperty -Name SourceAsup -Value (ConvertTo-AsupObject -AsupSet $Asup)

    try {
        #region System Information

        Write-Verbose -Message '- Gathering basic system information -'

        try {
            Write-Verbose -Message 'Parsing SYSTEM'
            $systemInfo = Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'SYSTEM' -ReplaceString '^Sys(?!temId)' -ErrorAction Stop
        }
        catch {
            $Global:Error.RemoveAt(0)

            if ($_.CategoryInfo.Category -eq 'ReadError') {
                $msg = New-Object -TypeName System.Text.StringBuilder

                [void]$msg.AppendLine('An unexpected error occurred. Processing halted for this system.')
                [void]$msg.AppendLine('---------------------------------------------------------------------------------------------')
                [void]$msg.AppendLine('This error is not expected, but we do know about it. Unfortunately, this is an issue with the')
                [void]$msg.AppendLine('ASUP data warehouse and (usually) the ASUP not being fully ingested. It is possible that')
                [void]$msg.AppendLine('waiting up to a few hours or so may be enough time for the ASUP DW to finishing ingesting the')
                [void]$msg.AppendLine('data. However, in some cases, the only workaround is to try a different set of ASUPs.')
                [void]$msg.AppendLine('---------------------------------------------------------------------------------------------')
                [void]$msg.AppendLine("Target: $( $_.CategoryInfo.TargetName ). Reason: $( $_.Exception.GetBaseException().Message )")
                [void]$msg.AppendLine("Location: $( $_.ScriptStackTrace )")
                [void]$msg.AppendLine()

                $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $_.CategoryInfo.TargetName)))
            }
            else {
                $NetAppDocsError += $_
            }
        }

        #region Raw Sysconfig data

        Write-Verbose -Message '- Collecting raw sysconfig data -'

        Write-Verbose -Message 'Capturing SYSCONFIG-A'
        $sysconfigA = Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'SYSCONFIG-A' -ErrorVariable +NetAppDocsError

        Write-Verbose -Message 'Capturing SYSCONFIG-AC'
        $sysconfigAc = Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'SYSCONFIG-AC' -ErrorVariable +NetAppDocsError

        #endregion Raw Sysconfig Data

        <#$systemVersionMajor, $systemVersionMinor = [regex]::Match($systemAsupData.SourceAsup.SystemVersion, '^(\d+)[.](\d+).+?$').Groups[1, 2].Value
        if ($systemVersionMajor -ge 8 -and $systemVersionMinor -ge 2) {
            Write-Verbose -Message 'Parsing MOTHERBOARD-INFO.XML'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name Motherboard -Value (
                Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'MOTHERBOARD-INFO.XML' -InnerProperty 'T_MBINFO' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                    Name       = 'BIOS'
                    Expression = { $_.BiosVersion }
                }
            )

            Write-Verbose -Message 'Parsing SYSTEM-INFO.XML'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name SystemInfo -Value (
                Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SYSTEM-INFO.XML' -InnerProperty 'T_SYSINFO' -ErrorVariable +NetAppDocsError | Select-Object -ExcludeProperty OntapVersion -Property *, @{
                    Name       = 'Hostname'
                    Expression = { $_.SystemHostname }
                }, @{
                    Name       = 'SerialNo'
                    Expression = { $_.SystemSerialNumber }
                }, @{
                    Name       = 'PartnerSystemId'
                    Expression = { $_.PartnerId }
                }, @{
                    Name       = 'Model'
                    Expression = { $systemAsupData.Motherboard.ModelName }
                }, @{
                    Name       = 'Version'
                    Expression = { ([regex]::Split(([regex]::Split($_.OntapVersion, 'Release\s(?:\w+[__])?')[1]), '\s+'))[0].Replace(':', '') }
                }
            )
        }
        else {
            $systemAsupData | Add-Member -MemberType NoteProperty -Name Motherboard -Value $(
                if ($sysconfigA -match '(?m)^\s+BIOS\s+Version[:]\s+(?<BiosVersion>\S+).*$') {
                    [pscustomobject]@{
                        BIOS = $Matches.BiosVersion
                    }

                    $Matches.Clear()
                }
            )

            $systemAsupData | Add-Member -MemberType NoteProperty -Name SystemInfo -Value (
                $systemAsupData.SourceAsup | Select-Object -Property SystemId, @{
                    Name       = 'Hostname'
                    Expression = { $_.SystemName }
                }, @{
                    Name       = 'SerialNo'
                    Expression = { $_.SerialNumber }
                }, @{
                    Name       = 'PartnerHostname'
                    Expression = { $_.HaGroup.PartnerHostname }
                }, @{
                    Name       = 'PartnerSystemId'
                    Expression = { $_.HaGroup.PartnerSystemId }
                }, @{
                    Name       = 'Model'
                    Expression = { $_.SystemModel }
                }, @{
                    Name       = 'Version'
                    Expression = { $_.SystemVersion }
                }
            )
        }

        # This options hack is purely for backwards compatiblity (used to pass the 'SYSTEM'.Options object to this as a pscustomobject)
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Option -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing OPTIONS'
        Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'OPTIONS' -ErrorVariable +NetAppDocsError | Get-NodeOptions | ForEach-Object {
            $systemAsupData.Option | Add-Member -MemberType NoteProperty -Name $_.Name -Value $_.Value
        }#>

        $systemAsupData | Add-Member -MemberType NoteProperty -Name SystemInfo -Value ($systemInfo | Select-Object -ExcludeProperty Options -Property *)
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Option -Value $systemInfo.Options

        Write-Verbose -Message 'Identifying IOXM status and PlatformConfig'
        $systemAsupData.SystemInfo | Add-Member -MemberType NoteProperty -Name IoxmPresent -Value ([bool](Select-String -InputObject $sysconfigA -Pattern 'IOXM\s+Status:\s+present' -Quiet))
        $systemAsupData.SystemInfo | Add-Member -MemberType NoteProperty -Name PlatformConfig -Value (Get-PlatformConfig -SystemModel $systemAsupData.SystemInfo.Model -IsClustered (! [string]::IsNullOrWhiteSpace($systemAsupData.SystemInfo.PartnerSystemId) -and ($systemAsupData.SystemInfo.PartnerSystemId -as [int]) -ne '0') -IoxmPresent $systemAsupData.SystemInfo.IoxmPresent)

        Write-Verbose -Message 'Identifying version information'
        $systemAsupData.SystemInfo | Add-Member -MemberType NoteProperty -Name VersionInfo -Value (
            [pscustomobject]@{
                Version       = $systemAsupData.SystemInfo.Version
                VersionTupleV = $(
                    if ([int]::TryParse($systemAsupData.SystemInfo.VersionHBranch, [ref]$null)) {
                        [version]::Parse([string]::Format('{0:N1}', $systemAsupData.SystemInfo.VersionHBranch))
                    }
                    else {
                        [version]::Parse($systemAsupData.SystemInfo.VersionHBranch)
                    }
                )
            }
        )

        Write-Verbose -Message 'Parsing MOTHERBOARD'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Motherboard -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'MOTHERBOARD' -ReplaceString '^Mb' -ErrorVariable +NetAppDocsError)

        #endregion System Information

        #region Cluster HA Information

        Write-Verbose -Message '- Gathering Cluster Info -'

        Write-Verbose -Message 'Parsing HA_GROUP'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Cluster -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'HA_GROUP' -ErrorVariable +NetAppDocsError)

        #endregion Cluster HA Information

        #region License Information

        Write-Verbose -Message '- Gathering licensing information -'

        Write-Verbose -Message 'Parsing SERVICE'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Service -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'SERVICE' -ReplaceString '^Svc' -ErrorVariable +NetAppDocsError)

        <#if ($systemVersionMajor -ge 8 -and $systemVersionMinor -ge 2) {
            Write-Verbose -Message 'Parsing LICENSES.XML'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name Service (
                Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'LICENSES.XML' -InnerProperty 'T_LIC_V2' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                    Name       = 'Name'
                    Expression = { $_.Package }
                }, @{
                    Name       = 'HasLic'
                    Expression = { $true }
                }, @{
                    Name       = 'Description'
                    Expression = { $_.Descr }
                }, @{
                    Name       = 'IsDemo'
                    Expression = { ! [string]::IsNullOrWhiteSpace($_.Expiration) }
                }
            )
        }
        else {
            Write-Verbose -Message 'Parsing SOFTWARE-LICENSES.XML'
            $softwareLicenses = Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'SOFTWARE-LICENSES.XML' -ErrorVariable +NetAppDocsError

            try {
                ($softwareLicenseXml = New-Object -TypeName System.Xml.XmlDocument).LoadXml($softwareLicenses)

                $systemAsupData | Add-Member -MemberType NoteProperty -Name Service (
                    $softwareLicenseXml.'license-list-info'.'licenses'.'license-info' | Convert-GenericXmlNode | Where-Object { $_.IsLicensed } | Select-Object -Property IsDemo, @{
                        Name       = 'Name'
                        Expression = { $_.Service }
                    }, @{
                        Name       = 'HasLic'
                        Expression = { $true }
                    }
                )
            }
            catch {
                $Global:Error.RemoveAt(0)

                Write-Verbose -Message 'Parsing SOFTWARE-LICENSES'
                $softwareLicenses = Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'SOFTWARE-LICENSES' -ErrorVariable +NetAppDocsError

                $systemAsupData | Add-Member -MemberType NoteProperty -Name Service @(
                    foreach ($line in ($softwareLicenses -split "`n" | Select-Object -Skip 4)) {
                        if ($line -match '^\s*(\S+)(?:\s+site)?\s+XXXXXXX(?:.*?(\(.+\)))?$') {
                            [pscustomobject]@{
                                Name   = $Matches[1]
                                HasLic = $true
                                IsDemo = ! [string]::IsNullOrWhiteSpace($Matches[2])
                            }
                        }
                    }
                )
            }
        }#>

        #endregion License Information

        #region Adapter Card Inventory

        Write-Verbose -Message '- Identifying adapter cards -'

        Write-Verbose -Message 'Calling Get-NtapExpansionSlotInventory'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name AdapterCard -Value (Get-NtapExpansionSlotInventory -SysconfigAc $sysconfigAc -SysconfigVorA $sysconfigA | Select-Object -Property * -ExcludeProperty NodeName)

        #endregion Adapter Card Inventory

        #region Remote Management Configuration

        Write-Verbose -Message '- Gathering remote management configuration -'

        Write-Verbose -Message 'Calling Get-NtapRemoteManagement'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name ServiceProcessor -Value (Get-NtapRemoteManagement -Sysconfig $sysconfigA | Select-Object -Property * -ExcludeProperty NodeName)

        #endregion Remote Management Configuration

        #region SNMP Configuration

        Write-Verbose -Message '- Gathering SNMP configuration -'

        Write-Verbose -Message 'Parsing SNMP'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Snmp -Value ((Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'SNMP' -ErrorVariable +NetAppDocsError) -split "`n")

        #endregion SNMP Configuration

        #region Storage Configuration

        Write-Verbose -Message '- Gathering system storage configuration and system ACP connectivity -'

        $systemAsupData | Add-Member -MemberType NoteProperty -Name Storage -Value ([pscustomobject]@{ })

        $storageConfiguration = ($sysconfigA -split "`n") | Select-String -Pattern 'System Storage Configuration:' -SimpleMatch
        $systemAsupData.Storage | Add-Member -MemberType NoteProperty -Name Configuration -Value $(
            if ($storageConfiguration) {
                ($storageConfiguration -split ':')[1].Trim()
            }
            else {
                'NA'
            }
        )

        $systemAcpConnectivity = ($sysconfigA -split "`n") | Select-String -Pattern 'System ACP Connectivity:' -SimpleMatch
        $systemAsupData.Storage | Add-Member -MemberType NoteProperty -Name SystemAcpConnectivity -Value $(
            if ($systemAcpConnectivity) {
                ($systemAcpConnectivity -split ':')[1].Trim()
            }
            else {
                'NA'
            }
        )

        #endregion Storage Configuration

        #region Disk Information

        ##Unused Section: UNOWNED-DISKS - Not needed for build docs (UnownedDisk)
        ##Unused Section: PATH - Not needed for build docs (DiskPath)

        Write-Verbose -Message '- Gathering disk information -'

        Write-Verbose -Message 'Parsing DEVICE'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Disk -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'DEVICE' -ReplaceString '^Dvc' -ErrorVariable +NetAppDocsError | Where-Object { $_.Type -eq 'DISK' -or $_.Type -eq 'LUN' })

        foreach ($disk in ($systemAsupData.Disk | Where-Object { $_ })) {
            $disk | Add-Member -MemberType NoteProperty -Name Node -Value ($systemAsupData.SystemInfo | Where-Object { ($_.SerialNo -replace '^0') -eq $disk.SysId }).Hostname
            $disk | Add-Member -MemberType NoteProperty -Name DiskUid -Value ($disk.Uid | Where-Object { $_ -ne $disk.SerialNo })
            $disk | Add-Member -MemberType NoteProperty -Name HomeNodeName -Value ($systemAsupData.SystemInfo | Where-Object { ($_.SystemId -replace '^0') -eq $disk.HomeOwner }).Hostname
            $disk | Add-Member -MemberType NoteProperty -Name OwnerNodeName -Value ($systemAsupData.SystemInfo | Where-Object { ($_.SystemId -replace '^0') -eq $disk.CurrentOwner }).Hostname
            $disk | Add-Member -MemberType NoteProperty -Name Name -Value $(
                if (-not [string]::IsNullOrWhiteSpace($disk.Node)) {
                    ([string]::Format('{0}:{1}', $disk.Node, $disk.PrimaryPath))
                }
                else {
                    $disk.PrimaryPath
                }
            )
        }

        #endregion Disk Information

        #region Shelf Information

        ##Unused Section: SHELF_ENVIRONMENT_INFO - Not needed for build docs (Shelf.Environment)

        Write-Verbose -Message '- Gathering shelf information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Shelf -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing SHELF'
        $systemAsupData.Shelf | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'SHELF' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing SHELF_MODULE'
        $systemAsupData.Shelf | Add-Member -MemberType NoteProperty -Name Module -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'SHELF_MODULE' -ReplaceString '^Smod' -ErrorVariable +NetAppDocsError)

        #endregion Shelf Information

        #region Autosupport Configuration

        ##Unused Section: AUTOSUPPORT.XML - Not needed for build docs (Autosupport.Config)
        ##Unused Section: AUTOSUPPORT-HISTORY.XML - Not needed for build docs (Autosupport.History)

        #endregion Autosupport Configuration

        #region DNS

        ##Unused Section: SYSTEM_KNOWN_HOST - Not needed for build docs (Dns.Host)

        Write-Verbose -Message '- Gathering DNS information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Dns -Value ([pscustomobject]@{ })

        $resolvConf = $(
            try {
                Write-Verbose -Message 'Parsing RESOLV-CONF'
                Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'RESOLV-CONF' -ErrorAction Stop
            }
            catch {
                if ($_.Exception.Message -like '*Invalid or No section(s) available for the request:*' -or $_.Exception.Message -eq 'No systems available for the input data:[]') {
                    $Global:Error.RemoveAt(0)

                    Write-Verbose -Message 'Parsing RESOLV-CONF.GZ'
                    Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'RESOLV-CONF.GZ' -ErrorVariable +NetAppDocsError
                }
                else {
                    $NetAppDocsError += $_
                }
            }
        )

        $systemAsupData.Dns | Add-Member -MemberType NoteProperty -Name Server -Value $(
            if (-not [string]::IsNullOrWhiteSpace($resolvConf)) {
                $resolvConf -split '\n' | Where-Object { $_ -match '^\s*nameserver' } | ForEach-Object {
                    [pscustomobject]@{
                        Server = ($_ -replace 'nameserver').Trim()
                    }
                }
            }
        )

        #endregion DNS

        #region Quotas

        $quotas = $(
            try {
                Write-Verbose -Message 'Parsing QUOTAS'
                Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'QUOTAS' -ErrorAction Stop
            }
            catch {
                if ($_.Exception.Message -like '*Invalid or No section(s) available for the request:*' -or $_.Exception.Message -eq 'No systems available for the input data:[]') {
                    $Global:Error.RemoveAt(0)

                    Write-Verbose -Message 'Parsing QUOTAS.GZ'
                    Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'QUOTAS.GZ' -ErrorVariable +NetAppDocsError
                }
                else {
                    $NetAppDocsError += $_
                }
            }
        )

        $systemAsupData | Add-Member -MemberType NoteProperty -Name QuotaEntry -Value (Convert-AsupQuota -Quotas $quotas)

        #endregion Quotas

        #region NFS

        Write-Verbose -Message '- Gathering NFS configuration -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Nfs -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing EXPORTS'
        $systemAsupData.Nfs | Add-Member -MemberType NoteProperty -Name Export -Value (Convert-AsupExport -Exports (Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'EXPORTS' -ErrorVariable +NetAppDocsError))

        #endregion NFS

        #region Iscsi

        ##Unused Section: ISCSI-STATUS - Not needed for build docs (Iscsi.Status)
        ##Unused Section: ISCSI-PORTALS - Not needed for build docs (Iscsi.Portal)
        ##Unused Section: ISCSI-ISNS - Not needed for build docs (Iscsi.Isns)

        Write-Verbose -Message '- Gathering iSCSI configuration -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Iscsi -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing ISCSI-ALIAS'
        $systemAsupData.Iscsi | Add-Member -MemberType NoteProperty -Name TargetAlias -Value ((Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'ISCSI-ALIAS' -ErrorVariable +NetAppDocsError) | Get-Match -Pattern '(?m)^\s*iSCSI target alias[:]\s+(.*?)\s*$')

        Write-Verbose -Message 'Parsing ISCSI-INTERFACE'
        $systemAsupData.Iscsi | Add-Member -MemberType NoteProperty -Name Interface -Value ((Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'ISCSI-INTERFACE' -ErrorVariable +NetAppDocsError) | Get-Match -Pattern '(?m)^\s*(?:Interface\s+)(?<InterfaceName>.*?)\s+(?<State>.*?)\s*$')

        Write-Verbose -Message 'Parsing ISCSI-NODENAME'
        $systemAsupData.Iscsi | Add-Member -MemberType NoteProperty -Name NodeName -Value ((Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'ISCSI-NODENAME' -ErrorVariable +NetAppDocsError) | Get-Match -Pattern '(?m)^\s*iSCSI target nodename[:]\s+(.*?)\s*$')

        Write-Verbose -Message 'Parsing ISCSI-SECURITY'
        $systemAsupData.Iscsi | Add-Member -MemberType NoteProperty -Name DefaultAuth -Value ((Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'ISCSI-SECURITY' -ErrorVariable +NetAppDocsError) | Get-Match -Pattern '(?m)^\s*(?:Default\ssec\sis\s(?<DefaultAuth>.*?))?\s*\binit[:]\s+(?<Initiator>(.*?))\s+auth[:]\s+(?<AuthType>.*?)\s+.*?Inbound\susername[:]\s+(?<UserName>.*?)\s+.*?Outbound\susername[:]\s*(.*?)\s*$')

        Write-Verbose -Message 'Parsing ISCSI-TARGET-PORTAL-GROUPS'
        $systemAsupData.Iscsi | Add-Member -MemberType NoteProperty -Name TargetPortalGroup -Value ((Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'ISCSI-TARGET-PORTAL-GROUPS' -ErrorVariable +NetAppDocsError) | Get-Match -Pattern '(?m)^\s*(?<TpgroupTag>\d+)\s+(?<TpgroupName>.*?)\s+(?<InterfaceList>.*?)\s*$')

        #endregion Iscsi

        #region Aggregate Configuration

        Write-Verbose -Message '- Gathering aggregate information -'

        $aggrRaidStatusAttrs = @(
            [pscustomobject]@{
                PropertyName = 'HasDegradedRg'
                StatusString = 'degraded'
            }
            [pscustomobject]@{
                PropertyName = 'HasReconsRg'
                StatusString = 'reconstruct'
            }
            [pscustomobject]@{
                PropertyName = 'HasUninitParity'
                StatusString = 'noparity'
            }
            [pscustomobject]@{
                PropertyName = 'IsSyncMirrored'
                StatusString = 'mirrored'
            }
            [pscustomobject]@{
                PropertyName = 'IsGrowing'
                StatusString = 'growing'
            }
            [pscustomobject]@{
                PropertyName = 'IsForeign'
                StatusString = 'foreign'
            }
            [pscustomobject]@{
                PropertyName = 'IsVerifiying'
                StatusString = 'verifying'
            }
            [pscustomobject]@{
                PropertyName = 'IsPartial'
                StatusString = 'partial'
            }
            [pscustomobject]@{
                PropertyName = 'IsNormal'
                StatusString = 'normal'
            }
        )

        Write-Verbose -Message 'Parsing AGGREGATE'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Aggr -Value (
            Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'AGGREGATE' -ReplaceString '^Aggr' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                Name       = 'RaidStatus'
                Expression = {
                    $(
                        foreach ($statusAttr in $aggrRaidStatusAttrs) {
                            if ($_.$( $statusAttr.PropertyName ) -eq $true) {
                                $statusAttr.StatusString
                            }
                        }
                    ) -join ', '
                }
            }
        )

        #endregion Aggregate Configuration

        #region Plex/RaidGroup Configuration

        Write-Verbose -Message '- Gathering plex/rg information -'

        Write-Verbose -Message 'Parsing PLEX'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Plex -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'PLEX' -ReplaceString '^Plex' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing RAID_GROUP'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name RaidGroup -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'RAID_GROUP' -ReplaceString '^Rg' -ErrorVariable +NetAppDocsError)

        #endregion Plex/RaidGroup Configuration

        #region Volume Configuration

        Write-Verbose -Message '- Gathering volume information -'

        Write-Verbose -Message 'Parsing VOLUME'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Vol -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'VOLUME' -ReplaceString '^Vol' -ErrorVariable +NetAppDocsError)

        #foreach ($vol in ($systemAsupData.Vol | Where-Object { $_ })) {
        #    $vol | Add-Member -MemberType NoteProperty -Name SsResPct -Value ($snapReserve | Where-Object { $_.Volume -ceq $vol.Name }).SnapshotReservePercentage
        #    $vol | Add-Member -MemberType NoteProperty -Name VolumeAllocatedKb -Value ([System.Math]::Round(($vol.AllocatedKb - ($vol.AllocatedKb * ($vol.SsResPct / 100)))))
        #}

        #endregion Volume Configuration

        #region Snapshot Information

        Write-Verbose -Message 'Parsing SNAPSHOT'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Snapshot -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'SNAPSHOT' -ReplaceString '^Snapshot' -ErrorVariable +NetAppDocsError)

        #endregion Snapshot Information

        #region Qtree Configuration

        Write-Verbose -Message '- Gathering qtree information -'

        Write-Verbose -Message 'Parsing QTREE'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Qtree -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'QTREE' -ReplaceString '^Qt' -ErrorVariable +NetAppDocsError)

        #endregion Qtree Configuration

        #region Vfiler Configuration

        Write-Verbose -Message '- Gathing Vfiler information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Vfiler -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing VFILER'
        $systemAsupData.Vfiler | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'VFILER' -ReplaceString '^Vf' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing VFILER_SERVICE'
        $systemAsupData.Vfiler | Add-Member -MemberType NoteProperty -Name Service -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'VFILER_SERVICE' -ReplaceString '^Vf' -ErrorVariable +NetAppDocsError)

        #endregion Vfiler Configuration

        #region SIS Configuration

        Write-Verbose -Message 'Parsing DEDUPE_STATUS'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Sis -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'DEDUPE_STATUS' -ErrorVariable +NetAppDocsError)

        #endregion SIS Configuration

        #region CIFS Configuration

        Write-Verbose -Message '- Gathering CIFS information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Cifs -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing CIFS_DOMAIN_INFO'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name Config -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'CIFS_DOMAIN_INFO' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS_SHARE'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name Share -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'CIFS_SHARE' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS_SHARE_ACL'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name ShareAcl -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'CIFS_SHARE_ACL' -ErrorVariable +NetAppDocsError)

        #endregion CIFS Configuration

        #region VSCAN Configuration

        Write-Verbose -Message 'Parsing VSCAN'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Vscan -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'VSCAN' -ReplaceString '^Vscan' -ErrorVariable +NetAppDocsError)

        #endregion VSCAN Configuration

        #region Lun Configuration

        Write-Verbose -Message '- Gathing Lun information -'

        Write-Verbose -Message 'Parsing LUN'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Lun -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'LUN' -ReplaceString '^Lun' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing LUN_INITIATOR_GROUP'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name LunIgroup -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'LUN_INITIATOR_GROUP' -ErrorVariable +NetAppDocsError)

        #endregion Lun Configuration

        #region Igroup Configuration

        Write-Verbose -Message '- Gathing Igroup information -'

        Write-Verbose -Message 'Parsing INITIATOR_GROUP'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Igroup -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'INITIATOR_GROUP' -ReplaceString '^Ig' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing INITIATOR_PATH'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name InitiatorPath -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'INITIATOR_PATH' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing PORTSETS'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name PortSet -Value ((Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'PORTSETS' -ErrorVariable +NetAppDocsError) | Get-Match -Pattern '(?s)^\s*(?<PortsetName>.*?)\s+[(](?<PortsetType>.*?)[)][:]\s+ports[:]\s+(?<PortsetInfo>.*?)\s+igroups[:]\s+(?<InitiatorGroupInfo>.*?)\s*$')

        #endregion Igroup Configuration

        #region Data Protection

        Write-Verbose -Message '- Gathering data protection information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name DataProtection -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing SM_SV_RELATIONS'
        $systemAsupData.DataProtection | Add-Member -MemberType NoteProperty -Name Status -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'SM_SV_RELATIONS' -ReplaceString '^(?:Sm|Sv)' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing SM_SV_DESTINATION'
        $systemAsupData.DataProtection | Add-Member -MemberType NoteProperty -Name Destination -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'SM_SV_DESTINATION' -ReplaceString '^(?:Sm|Sv)' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing SNAPVAULT_SCHEDULE'
        $systemAsupData.DataProtection | Add-Member -MemberType NoteProperty -Name SnapvaultSchedule -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'SNAPVAULT_SCHEDULE' -ErrorVariable +NetAppDocsError)

        #endregion Data Protection

        #region Network Information

        Write-Verbose -Message '- Gathering network information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Net -Value ([pscustomobject]@{ })

        $etcRc = $(
            try {
                Write-Verbose -Message 'Parsing RC'
                Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'RC' -ErrorAction Stop
            }
            catch {
                if ($_.Exception.Message -like '*Invalid or No section(s) available for the request:*' -or $_.Exception.Message -eq 'No systems available for the input data:[]') {
                    $Global:Error.RemoveAt(0)

                    Write-Verbose -Message 'Parsing RC.GZ'
                    Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'RC.GZ' -ErrorVariable +NetAppDocsError
                }
                else {
                    $NetAppDocsError += $_
                }
            }
        )

        Write-Verbose -Message 'Parsing INTERFACE'
        $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Port -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'INTERFACE' -ReplaceString '^If' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing IP'
        $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Interface -Value (Get-AsupObject -AsupObject $systemAsupData.SourceAsup -ObjectName 'IP' -ReplaceString '^If' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Identifying routes'
        $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Route -Value ($etcRc | Get-Match -Pattern '(?m)^\s*route\s+add\s+(?:(?<Destination>default)\s+(?<Gateway>.*?)|(?<Type>(net|host))\s+(?<Destination>.*?)\s+(?<Gateway>.*?))\s+(?<Metric>\d+)\s*$')

        Write-Verbose -Message 'Identifying Ifgrps'
        $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Vif -Value (
            $etcRc | Get-Match -Pattern '(?m)^\s*ifgrp\s+create\s+(?<IfgrpType>.+?)\s+(?<IfgrpName>.*?)\s+[-]b(?<IfgrpPolicy>.+?)\s+(?<Links>.*)\s*$' | Select-Object -ExcludeProperty Links -Property *, @{
                Name       = 'Links'
                Expression = { $_.Links -split '\s+' }
            }
        )

        Write-Verbose -Message 'Identifying Aliases'
        $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Alias -Value ($etcRc | Get-Match -Pattern '(?m)^\s*ifconfig\s+(?<InterfaceName>\S+).*?(?<![-])alias.*?(?<IpAddress>\b(?:[0-9]{1,3}\.){3}[0-9]{1,3}\b)\s+netmask.*$')

        #endregion Network Information

        Write-Verbose -Message '- Identifying ASUP Generation Time -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name ReferenceDT -Value $systemAsupData.SourceAsup.AsupGenerationDate.ToUniversalTime()

        Write-Verbose -Message '- Identifying SystemName -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name SystemName -Value $systemAsupData.SystemInfo.Hostname

        Write-Verbose -Message '- Identifying any errors during processing -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name DiscoveryError -Value @(
            foreach ($err in ($NetAppDocsError | Where-Object { $_ -and $_.Exception.Message -notlike 'Exception calling "LoadXml"*' -and $_.Exception.Message -notlike 'Entity not supported*' })) {
                $err.CategoryInfo | Select-Object -Property Category, Reason, @{
                    Name       = 'Activity'
                    Expression = { $_.Activity }
                }, @{
                    Name       = 'TargetName'
                    Expression = { "$( $systemAsupData.SystemName ) ($( $_.TargetName ))" }
                }, @{
                    Name       = 'TargetType'
                    Expression = { 'NaController.Asup' }
                }, @{
                    Name       = 'Message'
                    Expression = {
                        if ($err.Exception.Message -eq 'No systems available for the input data:[]') {
                            'ASUP section is missing or empty for this system'
                        }
                        else {
                            $err.Exception.Message
                        }
                    }
                }
            }
        )

        if (($systemAsupData.DiscoveryError | Where-Object { $_.Reason -ne 'NO_DATA_FOUND' } | Measure-Object).Count -gt 0) {
            $msg = New-Object -TypeName System.Text.StringBuilder

            [void]$msg.AppendLine("Errors were encountered while collecting data from ASUPID: $( $Asup.asup_id )")
            [void]$msg.AppendLine('-------------------------------------------------------------------------------------')
            [void]$msg.AppendLine('Please review the output for accuracy and, if necessary, correct any error conditions')
            [void]$msg.AppendLine('then run these scripts again.')
            [void]$msg.AppendLine('-------------------------------------------------------------------------------------')
            [void]$msg.AppendLine()

            Write-Warning -Message $msg.ToString()
        }

        $systemAsupData
    }
    catch {
        $Global:Error.RemoveAt(0)

        $msg = New-Object -TypeName System.Text.StringBuilder

        if ($_.CategoryInfo.Category -eq 'ResourceUnavailable') {
            [void]$msg.AppendLine('The connection to the ASUP REST server has failed. Please check your connection and try the capture again.')
        }
        else {
            [void]$msg.AppendLine('An unexpected error occurred. Processing halted.')
            [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
            [void]$msg.AppendLine('As this error is not expected, please consider contacting the support staff via e-mail or by')
            [void]$msg.AppendLine('starting a thread on the community site.')
            [void]$msg.AppendLine('            E-mail: ng-NetAppDocs-support@netapp.com')
            [void]$msg.AppendLine('    Community site: https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs')
            [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
            [void]$msg.AppendLine($( $_.Exception.GetBaseException().Message ))
            [void]$msg.AppendLine("Location: $( $_.ScriptStackTrace )")
            [void]$msg.AppendLine()
        }

        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $_.CategoryInfo.TargetName)))
    }
}

function Convert-AsupExport {
    [CmdletBinding()]
    param (
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$Exports
    )

    $exportRegex = '(?s)^(?:(?<Pathname>.+)\s+[-](?:actual[=](?<ActualPathname>.*?)[,])?(?<AccessString>.*?)$|(?<AccessString>.*?)(?:[,]actual[=](?<ActualPathname>.*?)?))$'

    foreach ($export in (($Exports -split "`n") | Where-Object { -not [string]::IsNullOrWhiteSpace($_) -and $_ -notmatch '^\s*#' })) {
        $outputObj = Import-WithRegex -CliOutput $export -RegexString $exportRegex
        $outputObj | Add-Member -MemberType NoteProperty -Name SecurityRules -Value @( )

        foreach ($accessString in ($outputObj.AccessString -split '(sec[=][^,sec=].*)$' | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })) {
            $rule = [pscustomobject]@{
                SecFlavor = $null
                ReadOnly  = @( )
                ReadWrite = @( )
                Root      = @( )
                Anon      = $( if ($accessString -match 'anon(?=[=](?<Anon>\d+))') { $Matches.Anon } )
                NoSuid    = $accessString -like '*nosuid*'
            }

            if ($Matches) {
                $Matches.Clear()
            }

            switch -Regex ($accessString) {
                '[^sec=]' {
                    $rule.SecFlavor = 'sys'
                }
                '(?:sec[=](?<SecFlavor>.*?))(?:[,].*)?$' {
                    $rule.SecFlavor = $Matches.SecFlavor -split ':'

                    if ($Matches) {
                        $Matches.Clear()
                    }
                }
                'ro(?![=]|ot)' {
                    $rule.ReadOnly += [pscustomobject]@{
                        AllHosts = $true
                        Name     = $null
                        Negate   = $null
                    }

                    if ($Matches) {
                        $Matches.Clear()
                    }
                }
                '(?:ro[=](?<RoRule>.*?))(?:[,].*)?$' {
                    foreach ($roRule in ($Matches.RoRule -split '\s*:\s*')) {
                        $rule.ReadOnly += [pscustomobject]@{
                            AllHosts = $null
                            Name     = $roRule -replace '^[-]'
                            Negate   = $roRule -match '^[-]'
                        }

                        if ($Matches) {
                            $Matches.Clear()
                        }
                    }
                }
                'rw(?![=])' {
                    $rule.ReadWrite += [pscustomobject]@{
                        AllHosts = $true
                        Name     = $null
                        Negate   = $null
                    }
                }
                '(?:rw[=](?<RwRule>.*?))(?:[,].*)?$' {
                    foreach ($rwRule in ($Matches.RwRule -split '\s*:\s*')) {
                        $rule.ReadWrite += [pscustomobject]@{
                            AllHosts = $null
                            Name     = $rwRule -replace '^[-]'
                            Negate   = $rwRule -match '^[-]'
                        }

                        if ($Matches) {
                            $Matches.Clear()
                        }
                    }
                }
                'root(?![=])' {
                    $rule.Root += [pscustomobject]@{
                        AllHosts = $true
                        Name     = $null
                        Negate   = $null
                    }

                    if ($Matches) {
                        $Matches.Clear()
                    }
                }
                '(?:root[=](?<RootRule>.*?))(?:[,].*)?$' {
                    foreach ($rootRule in ($Matches.RootRule -split '\s*:\s*')) {
                        $rule.Root += [pscustomobject]@{
                            AllHosts = $null
                            Name     = $rootRule -replace '^[-]'
                            Negate   = $rootRule -match '^[-]'
                        }

                        if ($Matches) {
                            $Matches.Clear()
                        }
                    }
                }
            }

            $outputObj.SecurityRules += $rule
        }

        $outputObj
    }
}

function Convert-AsupQuota {
    [CmdletBinding()]
    param (
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$Quotas
    )

    $quotaRegex = '(?<QuotaTarget>.*?)\s+(?<QuotaType>(?:tree|user|group))(?:[@]/vol/(?<Volume>.*?)/(?<Qtree>.*?))?\s+(?<DiskLimit>.*?)\s+(?<FileLimit>.*?)\s+(?<Threshold>.*?)\s+(?<SoftDiskLimit>.*?)\s+(?<SoftFileLimit>.*?)$'

    foreach ($quota in (($Quotas -split "`n") | Where-Object { -not [string]::IsNullOrWhiteSpace($_) -and $_ -notmatch '^\s*#' })) {
        $outputObj = Import-WithRegex -CliOutput $quota -RegexString $quotaRegex

        if ($outputObj.QuotaTarget -match '/vol/' -and (-not $outputObj.Volume)) {
            $vol, $qtree = ($outputObj.QuotaTarget -replace '/vol/') -split '/' | Where-Object { -not [string]::IsNullOrWhiteSpace($_) } | Select-Object -First 2

            $outputObj | Add-Member -NotePropertyMembers @{
                Volume = $vol
                Qtree  = $qtree
            }
        }

        $outputObj | Select-Object -ExcludeProperty QuotaPath -Property *
    }
}

# SIG # Begin signature block
# MIIq0AYJKoZIhvcNAQcCoIIqwTCCKr0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBuRP+mUNovaOwm
# SjuLi+10X4YICDNPyhcVLA5X0gHSPqCCJGgwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggWNMIIEdaADAgECAhAOmxiO
# +dAt5+/bUOIIQBhaMA0GCSqGSIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYD
# VQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAi
# BgNVBAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAw
# MDBaFw0zMTExMDkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
# Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERp
# Z2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBAL/mkHNo3rvkXUo8MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsb
# hA3EMB/zG6Q4FutWxpdtHauyefLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iT
# cMKyunWZanMylNEQRBAu34LzB4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGb
# NOsFxl7sWxq868nPzaw0QF+xembud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclP
# XuU15zHL2pNe3I6PgNq2kZhAkHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCr
# VYJBMtfbBHMqbpEBfCFM1LyuGwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFP
# ObURWBf3JFxGj2T3wWmIdph2PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTv
# kpI6nj3cAORFJYm2mkQZK37AlLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWM
# cCxBYKqxYxhElRp2Yn72gLD76GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls
# 5Q5SUUd0viastkF13nqsX40/ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBR
# a2+xq4aLT8LWRV+dIPyhHsXAj6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6
# MIIBNjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qY
# rhwPTzAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8E
# BAMCAYYweQYIKwYBBQUHAQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5k
# aWdpY2VydC5jb20wQwYIKwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDig
# NoY0aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNybDARBgNVHSAECjAIMAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCg
# v0NcVec4X6CjdBs9thbX979XB72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQT
# SnovLbc47/T/gLn4offyct4kvFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh
# 65ZyoUi0mcudT6cGAxN3J0TU53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSw
# uKFWjuyk1T3osdz9HNj0d1pcVIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAO
# QGPFmCLBsln1VWvPJ6tsds5vIy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjD
# TZ9ztwGpn1eqXijiuZQwggXMMIIENKADAgECAhAg429sPxgagUb53pPffJfkMA0G
# CSqGSIb3DQEBDAUAMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBS
# MzYwHhcNMjEwOTA5MDAwMDAwWhcNMjMwOTA5MjM1OTU5WjBQMQswCQYDVQQGEwJV
# UzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UECgwMTmV0QXBwLCBJbmMuMRUw
# EwYDVQQDDAxOZXRBcHAsIEluYy4wggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGK
# AoIBgQC4kYYj/oViZD9pN03hrqFjtwOz1Gx4eDWVV8IYUYEr2qpLeNYvWz26B/+E
# mYLboAGVpkYg0Wske3hjRpooZlMORm1S4+2C2NoQCstZ+kmNTevbXHJe/w1VNJrm
# fKjpXOJEfx8GUGDqCMR30dChVpsdBEavrM7T0hnfJLv18i19SGv3a+nhvB3iOjLP
# SEg8+DFkjD5jTI6fQ83DBEiVR4UEYq7yyH578ZEylhsRfJmK+uIREycEUk/NpFTw
# g6/7lxh/WYabCrZzqI4Ep3QataRKaAlxt3BiOlZkX4WGX3DYvSYltWOM2UlCcGpP
# m/1/LN8Xuvf+YO6H472QecHe59XtXbHSMKvqDuOEJu8Wk2LFMNK732o3Fc5QIHie
# 6JtNsgNYf+Vsm5EMBD1ZHNE+C3zlmQbbWIU5uEU1nhiodBPKsu0SkS6pARyKBO05
# DSc2gWWv0aukKi04ZAn+hAdSl5s1dgmU5shBvKHqe15K9ZRN4FFO/CgHE0BoGYQS
# UQVKwa0CAwEAAaOCAZwwggGYMB8GA1UdIwQYMBaAFA8qyyCHKLjsb0iuK1SmKaoX
# pM0MMB0GA1UdDgQWBBQuH643KcBMmb/Q6IZt+H9IrnXFwDAOBgNVHQ8BAf8EBAMC
# B4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDAzARBglghkgBhvhC
# AQEEBAMCBBAwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwIwJTAjBggrBgEFBQcC
# ARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQBMEkGA1UdHwRCMEAw
# PqA8oDqGOGh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVT
# aWduaW5nQ0FSMzYuY3JsMHkGCCsGAQUFBwEBBG0wazBEBggrBgEFBQcwAoY4aHR0
# cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQVIz
# Ni5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqG
# SIb3DQEBDAUAA4IBgQCOoGdXjP8Sif0h3ZvoDsIVfrJvQUdP9pZExRJGGj/Te6ML
# XyqHNc/G2WklNv+BC4ryNJ4zp0oneHOub55ejrYPfhgP9oFWS8oOPJbrp3rAtINa
# OeKRC88WUYZSKYAxSX6kubEjnQD6cmazXjxVN6P8+6q9smeqF3xI+++c+ekMw3Xv
# 4EWRYXqN0/srfFp1WpoluwSQcZMqj9haSX0bgO6AAWH2AnVJMfXTIKz/0FQ/RW0y
# Ed5QYQqYInhA7IUz9dBziNpJlwONeF5j7xzgfYDY63WU6WrgJnGjYkQCOOsUx74j
# gUiMRe9zV48GS8Vxz22c/TQs0x4/1GmOSJvUHEk3GseBmB3v+yEhh/D6zWOfYP4X
# D/9b91CxmugTuepuaJSeRg+qUm3KarKsOCUF/CLqUrNh/JwKrWD1cghRaYLvMucs
# ScksHHe7ZDvb2OtvxWXjPk1d1NKvEwFJSS6hIVIbug9x28AJqOoP5Pn9wZvJSzvW
# uQJGCLscdGyKefdCo30wggYaMIIEAqADAgECAhBiHW0MUgGeO5B5FSCJIRwKMA0G
# CSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290
# IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFQxCzAJBgNVBAYT
# AkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28g
# UHVibGljIENvZGUgU2lnbmluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEBAQUAA4IB
# jwAwggGKAoIBgQCbK51T+jU/jmAGQ2rAz/V/9shTUxjIztNsfvxYB5UXeWUzCxEe
# AEZGbEN4QMgCsJLZUKhWThj/yPqy0iSZhXkZ6Pg2A2NVDgFigOMYzB2OKhdqfWGV
# oYW3haT29PSTahYkwmMv0b/83nbeECbiMXhSOtbam+/36F09fy1tsB8je/RV0mIk
# 8XL/tfCK6cPuYHE215wzrK0h1SWHTxPbPuYkRdkP05ZwmRmTnAO5/arnY83jeNzh
# P06ShdnRqtZlV59+8yv+KIhE5ILMqgOZYAENHNX9SJDm+qxp4VqpB3MV/h53yl41
# aHU5pledi9lCBbH9JeIkNFICiVHNkRmq4TpxtwfvjsUedyz8rNyfQJy/aOs5b4s+
# ac7IH60B+Ja7TVM+EKv1WuTGwcLmoU3FpOFMbmPj8pz44MPZ1f9+YEQIQty/NQd/
# 2yGgW+ufflcZ/ZE9o1M7a5Jnqf2i2/uMSWymR8r2oQBMdlyh2n5HirY4jKnFH/9g
# Rvd+QOfdRrJZb1sCAwEAAaOCAWQwggFgMB8GA1UdIwQYMBaAFDLrkpr/NZZILyhA
# QnAgNpFcF4XmMB0GA1UdDgQWBBQPKssghyi47G9IritUpimqF6TNDDAOBgNVHQ8B
# Af8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcD
# AzAbBgNVHSAEFDASMAYGBFUdIAAwCAYGZ4EMAQQBMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Um9vdFI0Ni5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ1Jvb3RSNDYu
# cDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQwFAAOCAgEABv+C4XdjNm57oRUgmxP/BP6YdURhw1aVcdGRP4Wh60BAscjW
# 4HL9hcpkOTz5jUug2oeunbYAowbFC2AKK+cMcXIBD0ZdOaWTsyNyBBsMLHqafvIh
# rCymlaS98+QpoBCyKppP0OcxYEdU0hpsaqBBIZOtBajjcw5+w/KeFvPYfLF/ldYp
# mlG+vd0xqlqd099iChnyIMvY5HexjO2AmtsbpVn0OhNcWbWDRF/3sBp6fWXhz7Dc
# ML4iTAWS+MVXeNLj1lJziVKEoroGs9Mlizg0bUMbOalOhOfCipnx8CaLZeVme5yE
# Lg09Jlo8BMe80jO37PU8ejfkP9/uPak7VLwELKxAMcJszkyeiaerlphwoKx1uHRz
# NyE6bxuSKcutisqmKL5OTunAvtONEoteSiabkPVSZ2z76mKnzAfZxCl/3dq3dUNw
# 4rg3sTCggkHSRqTqlLMS7gjrhTqBmzu1L90Y1KWN/Y5JKdGvspbOrTfOXyXvmPL6
# E52z1NZJ6ctuMFBQZH3pwWvqURR8AgQdULUvrxjUYbHHj95Ejza63zdrEcxWLDX6
# xWls/GDnVNueKjWUH3fTv1Y8Wdho698YADR7TNx8X8z2Bev6SivBBOHY+uqiirZt
# g0y9ShQoPzmCcn63Syatatvx157YK9hlcPmVoa1oDE5/L9Uo2bC5a4CH2Rwwggau
# MIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJ
# BgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5k
# aWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAe
# Fw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcw
# FQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3Rl
# ZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9Ge
# TKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0
# hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZl
# jZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAsh
# aG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVY
# TXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1
# biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCir
# c0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+
# DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA
# +bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42Pg
# puE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzS
# M7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU
# uhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6
# mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsG
# AQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29t
# MEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNl
# cnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3Js
# My5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAE
# GTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1Z
# jsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8d
# B+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVp
# P0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp8
# 76i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2
# nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3
# ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQ
# txMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc
# 4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+Y
# AN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZ
# vAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQr
# H4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIGwDCCBKigAwIBAgIQDE1p
# ckuU+jwqSj0pB4A9WjANBgkqhkiG9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4XDTIyMDkyMTAwMDAw
# MFoXDTMzMTEyMTIzNTk1OVowRjELMAkGA1UEBhMCVVMxETAPBgNVBAoTCERpZ2lD
# ZXJ0MSQwIgYDVQQDExtEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMiAtIDIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDP7KUmOsap8mu7jcENmtuh6BSFdDMa
# JqzQHFUeHjZtvJJVDGH0nQl3PRWWCC9rZKT9BoMW15GSOBwxApb7crGXOlWvM+xh
# iummKNuQY1y9iVPgOi2Mh0KuJqTku3h4uXoW4VbGwLpkU7sqFudQSLuIaQyIxvG+
# 4C99O7HKU41Agx7ny3JJKB5MgB6FVueF7fJhvKo6B332q27lZt3iXPUv7Y3UTZWE
# aOOAy2p50dIQkUYp6z4m8rSMzUy5Zsi7qlA4DeWMlF0ZWr/1e0BubxaompyVR4aF
# eT4MXmaMGgokvpyq0py2909ueMQoP6McD1AGN7oI2TWmtR7aeFgdOej4TJEQln5N
# 4d3CraV++C0bH+wrRhijGfY59/XBT3EuiQMRoku7mL/6T+R7Nu8GRORV/zbq5Xwx
# 5/PCUsTmFntafqUlc9vAapkhLWPlWfVNL5AfJ7fSqxTlOGaHUQhr+1NDOdBk+lbP
# 4PQK5hRtZHi7mP2Uw3Mh8y/CLiDXgazT8QfU4b3ZXUtuMZQpi+ZBpGWUwFjl5S4p
# kKa3YWT62SBsGFFguqaBDwklU/G/O+mrBw5qBzliGcnWhX8T2Y15z2LF7OF7ucxn
# EweawXjtxojIsG4yeccLWYONxu71LHx7jstkifGxxLjnU15fVdJ9GSlZA076XepF
# cxyEftfO4tQ6dwIDAQABo4IBizCCAYcwDgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB
# /wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwIAYDVR0gBBkwFzAIBgZngQwB
# BAIwCwYJYIZIAYb9bAcBMB8GA1UdIwQYMBaAFLoW2W1NhS9zKXaaL3WMaiCPnshv
# MB0GA1UdDgQWBBRiit7QYfyPMRTtlwvNPSqUFN9SnDBaBgNVHR8EUzBRME+gTaBL
# hklodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0
# MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3JsMIGQBggrBgEFBQcBAQSBgzCBgDAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFgGCCsGAQUFBzAC
# hkxodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRS
# U0E0MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3J0MA0GCSqGSIb3DQEBCwUAA4IC
# AQBVqioa80bzeFc3MPx140/WhSPx/PmVOZsl5vdyipjDd9Rk/BX7NsJJUSx4iGNV
# CUY5APxp1MqbKfujP8DJAJsTHbCYidx48s18hc1Tna9i4mFmoxQqRYdKmEIrUPwb
# tZ4IMAn65C3XCYl5+QnmiM59G7hqopvBU2AJ6KO4ndetHxy47JhB8PYOgPvk/9+d
# EKfrALpfSo8aOlK06r8JSRU1NlmaD1TSsht/fl4JrXZUinRtytIFZyt26/+YsiaV
# OBmIRBTlClmia+ciPkQh0j8cwJvtfEiy2JIMkU88ZpSvXQJT657inuTTH4YBZJwA
# wuladHUNPeF5iL8cAZfJGSOA1zZaX5YWsWMMxkZAO85dNdRZPkOaGK7DycvD+5sT
# X2q1x+DzBcNZ3ydiK95ByVO5/zQQZ/YmMph7/lxClIGUgp2sCovGSxVK05iQRWAz
# gOAj3vgDpPZFR+XOuANCR+hBNnF3rf2i6Jd0Ti7aHh2MWsgemtXC8MYiqE+bvdgc
# mlHEL5r2X6cnl7qWLoVXwGDneFZ/au/ClZpLEQLIgpzJGgV8unG1TnqZbPTontRa
# mMifv427GFxD9dAq6OJi7ngE273R+1sKqHB+8JeEeOMIA11HLGOoJTiXAdI/Otrl
# 5fbmm9x+LMz/F0xNAKLY1gEOuIvu5uByVYksJxlh9ncBjDGCBb4wggW6AgEBMGgw
# VDELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDErMCkGA1UE
# AxMiU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIFIzNgIQIONvbD8YGoFG
# +d6T33yX5DANBglghkgBZQMEAgEFAKCBhDAYBgorBgEEAYI3AgEMMQowCKACgACh
# AoAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAM
# BgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCADS3q6bFiEX/z6jW7ZjQKTjJ5Z
# rPpwFrKqTKUHR4oRkTANBgkqhkiG9w0BAQEFAASCAYBdGLAR5QgcTVGBqz5FO575
# MLuQuLZ7238voLH2S59f4lbqRhb/pLtZwSY0JHsm1G6a2MJwrDzCF3/y4ijwZW2b
# fx/Ek/VuddPdIJB6O/Xk0ZDU38SKgTZFhuJsoiPwcwnm7y4/I7zTSlSk1MHRCLEn
# EyTL0b0CCrUBcTNiPWRYDT5UXOBqRMWTfdPM5wQFJQCCnvpXPA7tVogMqMmomEFz
# hvfMO8RT6yxswZmcegZDZBERV9Ts0GNclLrAJi/gfBiLtlwEuhG0VCAke96Pb30F
# vymyAhLiZjMrPRpBLfyOFJduEhblrHEeoHPNY9BDAw+yeOVFyu9+P5wvjdRkiFn5
# +RQqmhW0S5mpOmOySF0jzxXxSh/HCf7I9O0T5PkbUQhTw+NjXvI9kZdaczcYLEut
# cFu8/Uyw2o6xRD1srQqZ3LI8gwqVgs3WfghviXZT+pJApEJxzr0WjRXMq0lHKNdg
# tH1oIG7tyfoXFxGbs9BHc7u3BiBzlsqejnPb5PP5ViyhggMgMIIDHAYJKoZIhvcN
# AQkGMYIDDTCCAwkCAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
# cnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
# SEEyNTYgVGltZVN0YW1waW5nIENBAhAMTWlyS5T6PCpKPSkHgD1aMA0GCWCGSAFl
# AwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUx
# DxcNMjMwNDAyMTUyODI3WjAvBgkqhkiG9w0BCQQxIgQgBH8UnMZlKl1+m80Km4YE
# iTwaQP1WDHLf69FsGmMqG7YwDQYJKoZIhvcNAQEBBQAEggIAZ1E8F/0+VI83dldP
# AwdD6XNysSIzj4w7a991q/A1AqaHzFQLBfuHq73kRhkpeR/Cprdr6Wr1bPZb3hSv
# Vh6Wgzk+7pxdNuL2MRptkcFagqvzMeNxAOPscorXemrBZZB3OKwCsFRN+B6D76E6
# LbOC0jRrpjnvQ8AKT4Sg/Ivqq6ZYa7gJE9iKtvxeQmtRrOKQyBGowFdcrPkaUkZI
# 1OdEpYpsDP3AmlhnhMgvXEFXZ3DOwKH4D5ufkOoG0xojwZyu38MOUVaRL9zZexI9
# GgOVR7FUjPwDTLg8Ejxn4LvV3YNo3dtAWh+bScnLiiz0J+0n5JI8C75Ot4O+ekO7
# 28M3QhDmuQNCCmJ3jxnpkYJsFJ3DKIqrX1RqqIGRhy787g+xRfFpVJpUgI4EnxEg
# Kag1KxKD1shyUDR7+j7RMca7JfMyxgWWxjos4u+sgXSs4GbNDKuDLw3l4JaeUXEZ
# ko5EJhNklP/ORq0cySKkFD8gDbT2THUWxqRVZQgfg+re3unfT0X//tHSP/uoviHB
# JsLwXQocWJwPEsWaBSH4q/VXb9uSu6P3G8dgFxNIw8z89H13mtQ0tijuoCuwwemO
# strPnIB5QhBrpS2IcJC2thsRWx1vjkHAPyoJkGmXupP3K9ZBzzvnL76Ey49ohgbB
# hBKl97sR9km6SleguoLPMRrDMkA=
# SIG # End signature block
