function Get-NtapFilerData {
    <#
        .SYNOPSIS
            Collects data from Data ONTAP 7-Mode storage systems.

        .DESCRIPTION
            Uses the NetApp ONTAP PowerShell Toolkit to gather as much configuration information as possible. This
            data can be used to automate build docs and health checks. It can also be used for historical
            configuration information.

        .EXAMPLE
            PS> Get-NtapFilerData -Name controller1, controller2 | Export-Clixml -Path 'D:\Output\HaPair.xml'

            Collects the controller information from the system names specified and saves the results to a .xml file for
            further processing.

        .EXAMPLE
            PS> Get-NtapFilerData -Name controller1, controller2 -Credential (Get-Credential) | Export-Clixml -Path 'D:\Output\HaPair.xml'

            Collects the controller information from the system names specified using alternate credentials and saves the
            results to a .xml file for further processing. The Add-NaCredential cmdlet may be used to cache
            credentials for the system to be accessed.

        .EXAMPLE
            PS> Get-NtapFilerData -Name controller1, controller2 | Format-NtapFilerData | Out-NtapDocument -WordFile 'D:\Output\HaPair.docx' -ExcelFile 'D:\Output\HaPair.xlsx'

            Collects the controller information from the system names specified, formats the data and creates the specified
            Word and Excel documents.

        .LINK
            https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs

        .LINK
            Format-NtapFilerData

        .INPUTS
            NetApp.Ontapi.AbstractController

        .INPUTS
            System.String

        .OUTPUTS
            System.Management.Automation.PSCustomObject

        .NOTES
            AUTHOR : Jason Cole
    #>

    [CmdletBinding(DefaultParameterSetName = 'Name')]
    [OutputType([System.Management.Automation.PSCustomObject])]
    param (
        # The system name or IP address of the storage system to gather the data from.
        [Parameter(Mandatory, Position = 0, ParameterSetName = 'Name', ValueFromPipeline, ValueFromPipelineByPropertyName)]
        [ValidateNotNullOrEmpty()]
        [Alias('FilerName', 'SystemName')]
        [string[]]$Name,

        # A NetApp ONTAP PowerShell Toolkit controller object (NaController).
        [Parameter(Mandatory, Position = 0, ParameterSetName = 'ControllerObject', ValueFromPipeline)]
        [ValidateNotNullOrEmpty()]
        [NetApp.Ontapi.AbstractController[]]$AbstractController,

        # Specifies explicit credentials to connect to the storage system. (Alias: Cred)
        [Parameter(ParameterSetName = 'Name')]
        [ValidateNotNullOrEmpty()]
        [Alias('Cred')]
        [pscredential]
        [System.Management.Automation.Credential()]
        $Credential = [pscredential]::Empty,

        # The port on which to connect to the storage system. If not set, it defaults to 80 for HTTP and 443 for HTTPS.
        [Parameter(ParameterSetName = 'Name')]
        [ValidateRange([uint16]::MinValue, [uint16]::MaxValue)]
        [uint16]$Port,

        # Connection timeout in milliseconds. Only applies to HTTP/HTTPS connections. Defaults to 60000 milliseconds.
        [Parameter()]
        [ValidateRange(0, [int]::MaxValue)]
        [int]$Timeout,

        # Forces the use of RPC for the connection to the storage system.
        [Parameter(ParameterSetName = 'Name')]
        [switch]$ForceRPC,

        # Forces the use of HTTPS instead of HTTP for the connection to the storage system.
        [Parameter(ParameterSetName = 'Name')]
        [Alias('ForceHTTPS')]
        [switch]$ForceSecure,

        # Disables the ping test prior to connecting to the storage system (useful in the event ICMP is being blocked). (Deprecated)
        [Parameter()]
        [ValidateScript({
                Write-Warning -Message '-DisablePingCheck has been deprecated (no longer performing an ICMP ping to test connectivity).'

                $true
            })]
        [switch]$DisablePingCheck,

        # Disables the manual generation of an ASUP for the storage system. (Deprecated)
        [Parameter()]
        [ValidateScript({
                Write-Warning -Message 'The -DisableAsupInvoke switch has been deprecated. The function no longer triggers ASUPs by default (use the -InvokeAsup switch to explicitly trigger ASUPs)'

                $true
            })]
        [switch]$DisableAsupInvoke,

        # Invokes a user generated ASUP for the storage system.
        [Parameter()]
        [switch]$InvokeAsup
    )

    process {
        Write-Verbose -Message "Initializing function: $( $PSCmdlet.MyInvocation.MyCommand.Name ) ($( $PSCmdlet.MyInvocation.MyCommand.ModuleName ) v$( (Get-NtapDocsVersion -AsVersion).ToString() ))"

        switch -Exact ($PSCmdlet.ParameterSetName) {
            'Name' {
                foreach ($filer in ($Name | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })) {
                    try {
                        Write-Verbose -Message "Validating connectivity to system: $filer"

                        $ip = [System.Net.Dns]::GetHostAddresses($filer)
                        $tcpClient = New-Object -TypeName System.Net.Sockets.TCPClient

                        if ($PSCmdlet.MyInvocation.BoundParameters.ContainsKey('Port')) {
                            $tcpConnect = $tcpClient.BeginConnect($ip, $Port, $null, $null)

                            if (-not $tcpConnect.AsyncWaitHandle.WaitOne(7500, $false)) {
                                $tcpClient.Close()

                                throw 'Connection failed (ResourceUnavailable).'
                            }
                        }
                        elseif ($ForceRPC.IsPresent) {
                            $tcpConnect = $tcpClient.BeginConnect($ip, 445, $null, $null)

                            if (-not $tcpConnect.AsyncWaitHandle.WaitOne(7500, $false)) {
                                $tcpClient.Close()

                                throw 'Connection failed (ResourceUnavailable).'
                            }
                        }
                        elseif ($ForceSecure.IsPresent) {
                            $tcpConnect = $tcpClient.BeginConnect($ip, 443, $null, $null)

                            if (-not $tcpConnect.AsyncWaitHandle.WaitOne(7500, $false)) {
                                $tcpClient.Close()

                                throw 'Connection failed (ResourceUnavailable).'
                            }
                        }
                        else {
                            $tcpConnect = $tcpClient.BeginConnect($ip, 443, $null, $null)

                            if (-not $tcpConnect.AsyncWaitHandle.WaitOne(7500, $false)) {
                                $tcpClient.Close()

                                $tcpClient = New-Object -TypeName System.Net.Sockets.TCPClient
                                $tcpConnect = $tcpClient.BeginConnect($ip, 80, $null, $null)

                                if (-not $tcpConnect.AsyncWaitHandle.WaitOne(7500, $false)) {
                                    $tcpClient.Close()

                                    throw 'Connection failed (ResourceUnavailable).'
                                }
                            }
                        }

                        [void]$tcpClient.EndConnect($tcpConnect)
                        $tcpClient.Close()

                        $params = @{
                            Name        = $filer
                            ForceRPC    = $ForceRPC.IsPresent
                            ForceSecure = $ForceSecure.IsPresent
                        }

                        if ($Credential -ne [pscredential]::Empty) {
                            $params.Add('Credential', $Credential)
                        }

                        if ($PSCmdlet.MyInvocation.BoundParameters.ContainsKey('Port')) {
                            $params.Add('Port', $Port)
                        }

                        if ($PSCmdlet.MyInvocation.BoundParameters.ContainsKey('Timeout')) {
                            $params.Add('Timeout', $Timeout)
                        }

                        $controller = Connect-NtapStorageSystem @params -NaController -ErrorAction Stop

                        if ($controller.GetType().FullName -ne 'NetApp.Ontapi.Filer.NaController') {
                            throw 'Incorrect controller type specified for the Get-NtapFilerData function.'
                        }

                        if (-not (Test-NtapStorageSystemConnection -Controller $controller -ErrorAction Stop)) {
                            throw 'Unspecified error.'
                        }

                        Write-Verbose -Message "Valid connection established to system: $filer"
                    }
                    catch {
                        $Global:Error.RemoveAt(0)

                        Write-Error -ErrorRecord (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("Unable to gather data from system. Reason: $( $_.Exception.GetBaseException().Message )", $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $filer))

                        continue
                    }

                    Write-Verbose -Message "Gathering data from system: $filer"

                    try {
                        $functionStartTime = [datetime]::Now

                        $ntapData = Get-FilerData -Controller $controller

                        $ntapData

                        if ($ntapData -and $InvokeAsup.IsPresent) {
                            Write-Verbose -Message '- Generating ASUPs -'

                            try {
                                Write-Verbose -Message "Invoking ASUP on system: $( $ntapData.SystemInfo.SystemName )"
                                [void](Invoke-NaSystemApi -Request "<system-cli><args><arg>options</arg><arg>autosupport.doit</arg><arg>`"NetAppDocs_$( $ntapData.ReferenceDT.ToString('u') )`"</arg></args></system-cli>" -Controller $controller)
                            }
                            catch {
                                $Global:Error.RemoveAt(0)

                                Show-ErrorMessage -Message "Unable to invoke ASUP on system: $( $ntapData.SystemInfo.SystemName )"
                            }
                        }

                        $statsParams = @{
                            FunctionPSCallStack = Get-PSCallStack
                            TargetDetails       = [pscustomobject]@{
                                SystemUid     = [string]::Join('_', $ntapData.SystemInfo.SystemId, $ntapData.SystemInfo.SystemSerialNumber)
                                SystemType    = $ntapData.pstypenames[0]
                                SystemVersion = $ntapData.SystemVersion
                            }
                            ElapsedTicks        = ([datetime]::Now - $functionStartTime).Ticks
                            ExcludeParameter    = 'Name'
                        }

                        Write-Statistics @statsParams
                    }
                    catch {
                        Write-Error -ErrorRecord $_
                    }
                }
            }
            'ControllerObject' {
                foreach ($controller in ($AbstractController | Where-Object { $_ })) {
                    Write-Verbose -Message "Validating connectivity to system: $( $controller.Name )"

                    try {
                        if ($controller.GetType().FullName -ne 'NetApp.Ontapi.Filer.NaController') {
                            throw 'Incorrect controller type specified for the Get-NtapFilerData function.'
                        }

                        if ($controller.Protocol -like 'HTTP*' -and $PSCmdlet.MyInvocation.BoundParameters.ContainsKey('Timeout')) {
                            $controller.TimeoutMsec = $Timeout
                        }

                        if (-not (Test-NtapStorageSystemConnection -Controller $controller -ErrorAction Stop)) {
                            throw 'Unspecified error.'
                        }

                        Write-Verbose -Message "Valid connection established to system: $( $controller.Name )"
                    }
                    catch {
                        $Global:Error.RemoveAt(0)

                        Write-Error -ErrorRecord (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("Unable to gather data from system. Reason: $( $_.Exception.GetBaseException().Message )", $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $controller.Name))

                        continue
                    }

                    Write-Verbose -Message "Gathering data from system: $( $controller.Name )"

                    try {
                        $functionStartTime = [datetime]::Now

                        $ntapData = Get-FilerData -Controller $controller

                        $ntapData

                        if ($ntapData -and $InvokeAsup.IsPresent) {
                            Write-Verbose -Message '- Generating ASUPs -'

                            try {
                                Write-Verbose -Message "Invoking ASUP on system: $( $ntapData.SystemInfo.SystemName )"
                                [void](Invoke-NaSystemApi -Request "<system-cli><args><arg>options</arg><arg>autosupport.doit</arg><arg>`"NetAppDocs_$( $ntapData.ReferenceDT.ToString('u') )`"</arg></args></system-cli>" -Controller $controller)
                            }
                            catch {
                                $Global:Error.RemoveAt(0)

                                Show-ErrorMessage -Message "Unable to invoke ASUP on system: $( $ntapData.SystemInfo.SystemName )"
                            }
                        }

                        $statsParams = @{
                            FunctionPSCallStack  = Get-PSCallStack
                            TargetDetails        = [pscustomobject]@{
                                SystemUid     = [string]::Join('_', $ntapData.SystemInfo.SystemId, $ntapData.SystemInfo.SystemSerialNumber)
                                SystemType    = $ntapData.pstypenames[0]
                                SystemVersion = $ntapData.SystemVersion
                            }
                            ElapsedTicks         = ([datetime]::Now - $functionStartTime).Ticks
                            ParameterToSetAsBool = 'AbstractController'
                        }

                        Write-Statistics @statsParams
                    }
                    catch {
                        Write-Error -ErrorRecord $_
                    }
                }
            }
        }
    }
}

function Get-FilerData {
    <#
        .NOTES
            BURTS
            : 439564 - storage-shelf-list-info doesn't do Jakes
            :   Workaround: Parsing sysconfig -v output manually
            :   Fixed: RR.0
            : 457555 - storage-shelf-list-info zapi fails most of the time with shelf-id specified
            :   Workaround: Calling per channel instead of per shelf (then parsing by shelf)
            :   Fixed: RR.0
            : 174666 - vfiler-get-allowed-protocols and vfiler-get-disallowed-protocols does not return all the protocols
            :   Workaround: Using Invoke-NaSystemApi or Invoke-NaSsh to call relevant CLI command
            :   Target Release: RR.0
            : 443207 - Missing vfiler support for cifs-share-acl-list-iter-start API
            :   Workaround: None, skipping for CIFS shares present in vfiler other than vfiler0
            :   Target Release: SN.0
            : 419019 - Panic in "system-cli" vfilerized zapi session with root username changed to NULL
            :   Workaround: Using 'vfiler run <vfiler> <command>' from vfiler0 context instead
            :   Fixed: IC.5, RR.0
            : 384034 - options-list-info, options-get and options-set does not support JAF installable options eg. ndmpd.enable
            :   Workaround: Using Invoke-NaSystemApi or Invoke-NaSsh to call 'options ndmpd.enable'
            :   Target Release: RR.0
            : 389554 - Filer panicked while cleaning up an uninitialized snapvault iterator
            :   Workaround: None - system version is checked before calling
            :   Fixed: RR.0, BR.2, IC.6
            : 613219 - "fcadmin config" does not show storage HBA card information
            :   Workaround: Ignore the error messages from Get-NaFcAdapter on Vespa controllers
            :   Fixed: Not fixed in customer release of ONTAP
            : 560283 - zapidoc: Improve description for storage-disk-get-iter max-records field
            :   Workaround: Use <next-tag> in a do loop
            :   Fixed: RR.1
            : 456727 - cifs-share-list-iter-start fails: "Unable to create iterator"
            :   Workaround: None - This only occurs in a Vfiler context with the /etc volume is full
            :   Fixed: None
    #>

    [CmdletBinding()]
    param (
        [Parameter(Mandatory)]
        [ValidateNotNullOrEmpty()]
        [NetApp.Ontapi.Filer.NaController]$Controller
    )

    $systemData = [pscustomobject]@{ }
    $systemData.pstypenames.Insert(0, 'NetAppDocs.NaController.Data')

    Write-Verbose -Message '- Capturing module version -'
    $systemData | Add-Member -MemberType NoteProperty -Name NetAppDocsVersion (Get-NtapDocsVersion -AsVersion)

    Write-Verbose -Message '- Capturing PowerShell version -'
    $systemData | Add-Member -MemberType NoteProperty -Name PowerShellVersion -Value $PSVersionTable.PSVersion

    $NetAppDocsError = @( )

    try {
        #region Physical Filer Data

        #region Raw SysConfig Data

        Write-Verbose -Message '- Collecting raw sysconfig data -'

        Write-Verbose -Message 'Calling Invoke-NaSystemApi (sysconfig)'
        $sysconfig = (Invoke-NaSystemApi -Request '<system-cli><args><arg>sysconfig</arg></args></system-cli>' -Controller $Controller -ErrorAction SilentlyContinue -ErrorVariable +NetAppDocsError).results.'cli-output'

        Write-Verbose -Message 'Calling Invoke-NaSystemApi (sysconfig -ac)'
        $sysconfigAc = (Invoke-NaSystemApi -Request '<system-cli><args><arg>sysconfig</arg><arg>-ac</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'

        Write-Verbose -Message 'Calling Invoke-NaSystemApi (storage show)'
        $storageShow = (Invoke-NaSystemApi -Request '<system-cli><args><arg>storage</arg><arg>show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'

        Write-Verbose -Message 'Calling Invoke-NaSystemApi (sysconfig -v)'
        $sysconfigV = (Invoke-NaSystemApi -Request '<system-cli><args><arg>sysconfig</arg><arg>-v</arg></args></system-cli>' -Controller $Controller -ErrorAction SilentlyContinue -ErrorVariable +NetAppDocsError).results.'cli-output'

        Write-Verbose -Message 'Calling Invoke-NaSystemApi (storage show shelf)'
        $storageShowShelf = (Invoke-NaSystemApi -Request '<system-cli><args><arg>storage</arg><arg>show</arg><arg>shelf</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'

        #endregion Raw SysConfig Data

        #region Basic System Information

        ##Unused Cmdlet: Get-NaClusterPartner - Info exists in Get-NaCluster
        ##Unused Cmdlet: Get-NaSystemApi - Not needed
        ##Unused Cmdlet: Get-NaSystemAvailableReplicationTransfers - Not needed for build docs
        ##Unused Cmdlet: Get-NaSystemVendorInfo - Not needed for build docs
        ##Unused Cmdlet: Get-NaStorageAdapterInfo - Not needed for build docs
        ##Unused Cmdlet: Get-NaClusterHwAssistStats - Not needed for build docs

        Write-Verbose -Message '- Gathering basic system information -'

        Write-Verbose -Message 'Calling Get-NaSystemInfo'
        $systemData | Add-Member -MemberType NoteProperty -Name SystemInfo -Value (Get-NaSystemInfo -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Identifying IOXM status and PlatformConfig'
        $systemData.SystemInfo | Add-Member -MemberType NoteProperty -Name IoxmPresent -Value ([bool](Select-String -InputObject $sysconfig -Pattern 'IOXM\s+Status:\s+present' -Quiet))
        $systemData.SystemInfo | Add-Member -MemberType NoteProperty -Name PlatformConfig -Value (Get-PlatformConfig -SystemModel $systemData.SystemInfo.SystemModel -IsClustered (! [string]::IsNullOrWhiteSpace($systemData.SystemInfo.PartnerSystemId)) -IoxmPresent $systemData.SystemInfo.IoxmPresent)

        Write-Verbose -Message 'Identifying SystemName'
        $systemData | Add-Member -MemberType NoteProperty -Name SystemName -Value $systemData.SystemInfo.SystemName

        Write-Verbose -Message 'Calling Get-NaSystemOntapiVersion'
        $systemData | Add-Member -MemberType NoteProperty -Name OntapiVersion -Value ([version]::Parse((Get-NaSystemOntapiVersion -Controller $Controller -ErrorVariable +NetAppDocsError).ToString()))

        Write-Verbose -Message 'Calling Get-NaLicense'
        $systemData | Add-Member -MemberType NoteProperty -Name License -Value (Get-NaLicense -Controller $Controller -ErrorVariable +NetAppDocsError)


        $systemData | Add-Member -MemberType NoteProperty -Name Cluster -Value $(
            if (
                ($systemData.OntapiVersion -ge [version]::Parse('1.20')) -or
                ($systemData.License | Where-Object { $_.Service -in @( 'cluster', 'cluster_remote', 'cf', 'cf_remote' ) -and $_.IsLicensed } | Measure-Object).Count -gt 0
            ) {
                Write-Verbose -Message 'Calling Get-NaCluster'
                (Get-NaCluster -Controller $Controller -ErrorVariable +NetAppDocsError)
            }
        )

        Write-Verbose -Message 'Calling Get-NaSystemVersion'
        $systemData | Add-Member -MemberType NoteProperty -Name SystemVersion -Value ([regex]::Split(([regex]::Split((Get-NaSystemVersion -Controller $Controller -ErrorVariable +NetAppDocsError), 'Release\s(?:\w+[__])?')[1]), '\s+'))[0].Replace(':', '')

        Write-Verbose -Message 'Calling Get-NaTimeZone'
        $systemData | Add-Member -MemberType NoteProperty -Name Timezone -Value (Get-NaTimezone -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NaVolRoot'
        $systemData | Add-Member -MemberType NoteProperty -Name VolRoot -Value (Get-NaVolRoot -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion Basic System Information

        #region Node Health Validation

        if ($systemData.Cluster) {
            if ($systemData.Cluster.Partner -and $systemData.Cluster.State -ne 'CONNECTED') {
                $nodeHaStateMsg = [string]::Format(
                    'Node: {0} (HA State: {1})', $systemData.SystemName, $(
                        if ($systemData.Cluster.State) {
                            $systemData.Cluster.State
                        }
                        else {
                            'UNKNOWN'
                        }
                    )
                )

                $msg = New-Object -TypeName System.Text.StringBuilder

                [void]$msg.AppendLine("HA pair '$( $systemData.SystemName )/$( $systemData.Cluster.Partner )' is not healthy. Errors may be encountered and data may be incomplete.")
                [void]$msg.AppendLine('------------------------------------------------------------------------------')
                [void]$msg.AppendLine($nodeHaStateMsg)
                [void]$msg.AppendLine('Both nodes of the HA pair must be healthy to ensure accurate data is gathered.')
                [void]$msg.AppendLine('------------------------------------------------------------------------------')
                [void]$msg.AppendLine()

                Show-ErrorMessage -Message $msg.ToString()

                $NetAppDocsError += New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("HA pair is not healthy. Reason: $nodeHaStateMsg", 'HAPAIR_NOT_HEALTHY', [System.Management.Automation.ErrorCategory]::DeviceError, ([string]::Format('{0}/{1}', $systemData.SystemName, $systemData.Cluster.Partner)))
            }
        }

        #endregion Node Health Validation

        #region Feature Status Information

        if ($systemData.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Calling Get-NaFeatureStatus'
            $systemData | Add-Member -MemberType NoteProperty -Name FeatureStatus -Value (Get-NaFeatureStatus -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Feature Status Information

        #region Detailed System Info

        Write-Verbose -Message '- Gathering more detailed system info -'

        Write-Verbose -Message 'Identifying adapter cards'
        $systemData | Add-Member -MemberType NoteProperty -Name AdapterCard -Value (Get-NtapExpansionSlotInventory -SysconfigAc $sysconfigAc -SysconfigVorA $sysconfigV | Sort-Object -Property SlotNumber)

        Write-Verbose -Message 'Calling Get-NaSnmp'
        $systemData | Add-Member -MemberType NoteProperty -Name Snmp -Value (Get-NaSnmp -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message '- Identifying system storage configuration -'
        $systemData | Add-Member -MemberType NoteProperty -Name Storage -Value ([pscustomobject]@{ })

        $storageConfiguration = $sysconfig -split '\n' | Select-String -Pattern 'System Storage Configuration:' -SimpleMatch
        $systemData.Storage | Add-Member -MemberType NoteProperty -Name Configuration -Value $(
            if ($storageConfiguration) {
                ($storageConfiguration.ToString().Split(':'))[1].Trim()
            }
            else {
                'NA'
            }
        )

        Write-Verbose -Message '- Identifying system ACP connectivity information -'

        $systemAcpConnectivity = $sysconfig -split '\n' | Select-String -Pattern 'System ACP Connectivity:' -SimpleMatch
        $systemData.Storage | Add-Member -MemberType NoteProperty -Name SystemAcpConnectivity -Value $(
            if ($systemAcpConnectivity) {
                ($systemAcpConnectivity.ToString().Split(':'))[1].Trim()
            }
            else {
                'NA'
            }
        )

        Write-Verbose -Message '- Gathering system storage adapter configuration -'
        $systemData.Storage | Add-Member -MemberType NoteProperty -Name Adapter -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NaStorageAdapter'
        $adapterList = Get-NaStorageAdapter -Controller $Controller -ErrorVariable +NetAppDocsError

        $systemData.Storage.Adapter | Add-Member -MemberType NoteProperty -Name Info -Value @( )
        $adapterList | Where-Object { $_ } | ForEach-Object {
            Write-Verbose -Message "Calling Invoke-NaSystemApi (storage-adapter-get-adapter-info) $( $_.AdapterName )"
            $systemData.Storage.Adapter.Info += (Invoke-NaSystemApi -Request "<storage-adapter-get-adapter-info><adapter-name>$( $_.AdapterName )</adapter-name></storage-adapter-get-adapter-info>" -Controller $Controller -ErrorVariable +NetAppDocsError).results.'adapter-details'.'adapter-detail-info' | Convert-GenericXmlNode
        }

        Write-Verbose -Message 'Calling Get-NaReallocate'
        $systemData | Add-Member -MemberType NoteProperty -Name Reallocate -Value (Get-NaReallocate -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion Detailed System Info

        #region Remote Management Configuration

        Write-Verbose -Message '- Gathering remote management configuration -'

        if (-not [string]::IsNullOrWhiteSpace($sysconfigV)) {
            $systemData | Add-Member -MemberType NoteProperty -Name RemoteManagement -Value (Get-NtapRemoteManagement -Sysconfig $sysconfigV)
        }
        else {
            $sp = $(
                switch -Wildcard ($sysconfig) {
                    '*Service Processor*' {
                        Write-Verbose -Message 'Calling Invoke-NaSystemApi (sp status)'
                        (Invoke-NaSystemApi -Request '<system-cli><args><arg>sp</arg><arg>status</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'

                        break
                    }
                    '*Remote LAN Module*' {
                        Write-Verbose -Message 'Calling Invoke-NaSystemApi (rlm status)'
                        (Invoke-NaSystemApi -Request '<system-cli><args><arg>rlm</arg><arg>status</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'

                        break
                    }
                    '*Baseboard Management Controller*' {
                        Write-Verbose -Message 'Calling Invoke-NaSystemApi (bmc status)'
                        (Invoke-NaSystemApi -Request '<system-cli><args><arg>bmc</arg><arg>status</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'

                        break
                    }
                    default { $null }
                }
            )

            if (-not [string]::IsNullOrWhiteSpace($sp)) {
                Write-Verbose -Message 'Calling Get-NtapRemoteManagement'
                $systemData | Add-Member -MemberType NoteProperty -Name RemoteManagement -Value (Get-NtapRemoteManagement -Sysconfig $sp)
            }
            else {
                $NetAppDocsError += New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('Unable to capture SP/RLM/BMC information.', 'NO_SP_INFORMATION', [System.Management.Automation.ErrorCategory]::InvalidData, $systemData.SystemName)
            }
        }

        #endregion Remote Management Configuration

        #region Aggregate Status

        ##Unused Cmdlet: Get-NaAggrFilerInfo - Not needed for build docs
        ##Unused Cmdlet: Get-NaAggrMediaScrub - Not needed for build docs
        ##Unused Cmdlet: Get-NaAggrScrub - Not needed for build docs
        ##Unused Cmdlet: Get-NaAggrVerify - Not needed for build docs

        Write-Verbose -Message '- Gathering detailed aggregate information -'

        Write-Verbose -Message 'Calling Get-NaAggr'
        $systemData | Add-Member -MemberType NoteProperty -Name Aggr -Value (Get-NaAggr -Controller $Controller -ErrorVariable +NetAppDocsError)

        foreach ($aggr in ($systemData.Aggr | Where-Object { $_ })) {
            Write-Verbose -Message "Calling Get-NaAggrOption (Aggr: $( $aggr.Name ))"
            $aggr | Add-Member -MemberType NoteProperty -Name Option -Value (Get-NaAggrOption -Name $aggr.Name -Controller $Controller -ErrorVariable +NetAppDocsError)

            if ($aggr.State -eq 'online') {
                Write-Verbose -Message "Calling Get-NaAggrSpace (Aggr: $( $aggr.Name ))"
                $aggr | Add-Member -MemberType NoteProperty -Name Space -Value (Get-NaAggrSpace -Name $aggr.Name -Controller $Controller -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message "Calling Get-NaSnapshot (Aggr: $( $aggr.Name ))"
                $aggr | Add-Member -MemberType NoteProperty -Name Snapshot -Value (Get-NaSnapshot -TargetName $aggr.Name -TargetType Aggregate -Terse -Controller $Controller -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message "Calling Get-NaSnapshotReserve (Aggr: $( $aggr.Name ))"
                $aggr | Add-Member -MemberType NoteProperty -Name SnapshotReserve -Value (Get-NaSnapshotReserve -TargetName $aggr.Name -TargetType Aggregate -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false)
            }
        }

        #endregion Aggregate Status

        #region Flash Device Details

        ##Unused Cmdlet: Get-NaFlashProfile - Not needed for build docs (Flash.Profile - add Name property)
        ##Unused Cmdlet: Get-NaFlashThreshold - Not needed for build docs (Flash.Threshold)

        Write-Verbose -Message '- Gathering flash device data -'
        $systemData | Add-Member -MemberType NoteProperty -Name Flash -Value ([pscustomobject]@{ })

        if ($systemData.OntapiVersion -ge [version]::Parse('1.13')) {
            Write-Verbose -Message 'Calling Get-NaFlashDevice'
            $systemData.Flash | Add-Member -MemberType NoteProperty -Name Device -Value (Get-NaFlashDevice -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Flash Device Details

        #region Cluster HW Assist Details

        ##Unused Cmdlet: Get-NaClusterHwAssist - Not needed for build docs (if $systemData.Cluster and $systemData.Cluster.State -ne 'Error')

        #endregion Cluster HW Assist Details

        #region Priority Scheduling Data

        ##Unused Cmdlet: Get-NaPriority - Not needed for build docs (Priority.Info)
        ##Unused Cmdlet: Get-NaPriorityDefault - Not needed for build docs (Priority.Default)
        ##Unused Cmdlet: Get-NaPriorityVolume - Not needed for build docs (Priority.Volume)

        #endregion Priority Scheduling Data

        #region Disk Information

        ##Unused Cmdlet: Get-NaDiskOwnerFiler - Info exists in Get-NaDisk / Get-NaDiskOwner

        Write-Verbose -Message '- Gathering physical disk information -'

        $systemData | Add-Member -MemberType NoteProperty -Name Disk -Value $(
            if ($systemData.OntapiVersion -ge [version]::Parse('1.19')) {
                (Get-NaDisk -Controller $Controller -ErrorVariable +NetAppDocsError)
            }
            else {
                (Get-NaDisk -Old -Controller $Controller -ErrorVariable +NetAppDocsError)
            }
        )

        if ($storageShow -notlike '*SANOWN not enabled*') {
            Write-Verbose -Message 'Calling Get-NaDiskOwner'
            $systemData | Add-Member -MemberType NoteProperty -Name DiskOwner -Value (Get-NaDiskOwner -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Disk Information

        #region FC/FCP Configuration

        ##Unused Cmdlet: Get-NaFcpAdapterStatistics - Not needed for build docs
        ##Unused Cmdlet: Get-NaFcpNodeName - Data exists in Get-NcFcpAdapter
        ##Unused Cmdlet: Get-NaFcpPortName - Data exists in Get-NcFcpAdapter
        ##Unused Cmdlet: Get-NaFcpPortNameAlias - ?? Should we add? Errors if wwpn alias not set
        ##Unused Cmdlet: Get-NaFcpClusterMode - Not needed for build docs (Fcp.FcpClusterMode - if ($systemData.Cluster))
        ##Unused Cmdlet: Get-NaFcpInitiator - Not needed for build docs (Fcp.FcpInitiator)
        ##Unused Cmdlet: Get-NaPortSet - Not needed for build docs (Fcp.PortSet - if (Fcp.FcpAdapter.Count -gt 1))
        ##Unused Cmdlet: Test-NaLunConfig - Not needed for build docs (Fcp.LunConfigCheck.Info)
        ##Unused Cmdlet: Test-NaLunConfig -CfMode - Not needed for build docs (Fcp.LunConfigCheck.CfMode)
        ##Unused Cmdlet: Test-NaLunConfig -SSI - Not needed for build docs (Fcp.LunConfigCheck.SSI - if Fcp.FcpClusterMode -eq 'single_image')

        Write-Verbose -Message '- Gathering FC adapter data -'

        Write-Verbose -Message 'Calling Get-NaFcAdapter'
        #BURT 613219
        if ($systemData.SystemInfo.SystemModel -match 'FAS2[0|2][2|4|5]0') {
            $systemData | Add-Member -MemberType NoteProperty -Name FcAdapter -Value (Get-NaFcAdapter -Controller $Controller -ErrorAction SilentlyContinue)
        }
        else {
            $systemData | Add-Member -MemberType NoteProperty -Name FcAdapter -Value (Get-NaFcAdapter -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        Write-Verbose -Message '- Gathering FCP configuration data -'
        $systemData | Add-Member -MemberType NoteProperty -Name Fcp -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NaOption (licensed_feature.fcp.enable)'
        $optionLicensedFeatureFcpEnable = Get-NaOption -OptionNames 'licensed_feature.fcp.enable' -Controller $Controller -ErrorAction SilentlyContinue

        if (
            ($systemData.OntapiVersion -ge [version]::Parse('1.20') -and (($systemData.License | Where-Object { $_.Service -eq 'fcp' }).IsLicensed -and ($optionLicensedFeatureFcpEnable.Value -eq 'on'))) -or
            ($systemData.OntapiVersion -lt [version]::Parse('1.20') -and (($systemData.License | Where-Object { $_.Service -eq 'fcp' }).IsLicensed))
        ) {
            Write-Verbose -Message 'Calling Get-NaFcpAdapter'
            $systemData.Fcp | Add-Member -MemberType NoteProperty -Name FcpAdapter -Value (Get-NaFcpAdapter -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion FC/FCP Configuration

        #region iSCSI Configuration

        try {
            $iscsiInterface = Get-NaIscsiInterface -Controller $Controller -ErrorAction Stop

            Write-Verbose -Message '- Gathering iSCSI configuration data -'
            $systemData | Add-Member -MemberType NoteProperty -Name Iscsi -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NaIscsiInterface'
            $systemData.Iscsi | Add-Member -MemberType NoteProperty -Name Interface -Value $iscsiInterface
        }
        catch {
            if ($_.Exception.ErrorName -eq 'EAPILICENSE') {
                $Global:Error.RemoveAt(0)
            }
            else {
                $NetAppDocsError += $_
            }
        }

        if ($systemData.Iscsi) {
            Write-Verbose -Message 'Calling Get-NaIscsiTargetPortalGroup'
            $systemData.Iscsi | Add-Member -MemberType NoteProperty -Name TargetPortalGroup -Value (Get-NaIscsiTargetPortalGroup -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion iSCSI configuration

        #region Network Details

        ##Unused Cmdlet: Get-NaNetDcb - Not yet needed for build docs
        ##Unused Cmdlet: Get-NaNetDcbPriority - Not yet needed for build docs
        ##Unused Cmdlet: Get-NaNetActiveConfig - Not needed for build docs (Net.ActiveConfig)
        ##Unused Cmdlet: Get-NaNetVif - Not needed for build docs (Net.Vif)
        ##Unused Cmdlet: Get-NaNetVlan - Not needed for build docs (Net.Vlan)

        Write-Verbose -Message '- Gathering network details -'
        $systemData | Add-Member -MemberType NoteProperty -Name Net -Value ([pscustomobject]@{ })

        if ($systemData.OntapiVersion -ge [version]::Parse('1.11')) {
            Write-Verbose -Message 'Calling Get-NaNetInterface'
            $systemData.Net | Add-Member -MemberType NoteProperty -Name Interface -Value (Get-NaNetInterface -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        if ($systemData.OntapiVersion -ge [version]::Parse('1.11')) {
            Write-Verbose -Message 'Calling Get-NaNetPersistentConfig'
            $systemData.Net | Add-Member -MemberType NoteProperty -Name PersistentConfig -Value (Get-NaNetPersistentConfig -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        if ($systemData.OntapiVersion -ge [version]::Parse('1.11')) {
            Write-Verbose -Message 'Calling Get-NaNetRoute'
            $systemData.Net | Add-Member -MemberType NoteProperty -Name Route -Value (Get-NaNetRoute -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Network Details

        #region Shelf Details

        Write-Verbose -Message '- Gathering shelf data -'

        $systemData | Add-Member -MemberType NoteProperty -Name Shelf -Value ([pscustomobject]@{ })

        if (($systemData.Disk | Where-Object { $_.DiskType -ne 'LUN' } | Measure-Object).Count -gt 0) {
            if ($systemData.OntapiVersion -ge [version]::Parse('1.15')) {
                Write-Verbose -Message 'Calling Get-NaShelf'
                $systemData.Shelf | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NaShelf -Controller $Controller -ErrorVariable +NetAppDocsError | Where-Object { -not [string]::IsNullOrWhiteSpace($_.ShelfId) })

                Write-Verbose -Message 'Calling Invoke-NaSystemApi (storage-shelf-bay-list-info)'
                $systemData.Shelf | Add-Member -MemberType NoteProperty -Name Bay -Value ((Invoke-NaSystemApi -Request '<storage-shelf-bay-list-info />' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'shelf-bay-list'.'shelf-bay-list-info' | Convert-GenericXmlNode)
            }
            else {
                Write-Verbose -Message 'Calling Get-NaShelf'
                $shelfNonAta = Get-NaShelf -Controller $Controller -ErrorVariable +NetAppDocsError | Where-Object { -not [string]::IsNullOrWhiteSpace($_.ShelfId) }

                #BURT 439564
                $shelfAta = Get-NaShelfAta -Sysconfig $sysconfigV -StorageShowShelf $storageShowShelf

                $systemData.Shelf | Add-Member -MemberType NoteProperty -Name Info -Value (
                    ($shelfNonAta, $shelfAta) | ForEach-Object {
                        if ($_) {
                            $_
                        }
                    }
                )
            }

            Write-Verbose -Message 'Calling Get-NaShelfEnvironment'
            #BURT 457555
            $systemData.Shelf | Add-Member -MemberType NoteProperty -Name Environment -Value (
                $systemData.Shelf.Info | Where-Object { $_.ChannelName -and $_.ChannelName -ne 'UNUSED' } | Select-Object -Property ChannelName -Unique | ForEach-Object {
                    Get-NaShelfEnvironment -Channel $_.ChannelName -Controller $Controller -ErrorVariable +NetAppDocsError
                }
            )
        }

        #endregion Shelf Details

        #region Vfiler Configuration

        ##Unused Cmdlet: Get-NaVfilerDr - Not needed for build docs
        ##Unused Cmdlet: Get-NaVfilerMigrate - Not needed for build docs
        ##Unused Cmdlet: Get-NaVfilerStatus - Info exists in Get-NaVfiler

        Write-Verbose -Message '- Gathering MultiStore (vFiler) configurations -'

        try {
            Write-Verbose -Message 'Calling Get-NaVfiler'
            $systemData | Add-Member -MemberType NoteProperty -Name Vfiler -Value (Get-NaVfiler -Controller $Controller -ErrorAction Stop)
        }
        catch {
            if ($_.Exception.ErrorName -eq 'EAPILICENSE') {
                $Global:Error.RemoveAt(0)
            }
            else {
                $NetAppDocsError += $_
            }
        }

        if ($systemData.Vfiler) {
            foreach ($vfiler in ($systemData.Vfiler | Where-Object { $_ })) {
                #BURT 174666
                $vfiler | Add-Member -MemberType NoteProperty -Name Protocol -Value $(
                    if ($systemData.OntapiVersion -ge [version]::Parse('1.15')) {
                        (Get-NaVfilerProtocol -Name $vfiler.Name -Controller $Controller -ErrorVariable +NetAppDocsError)
                    }
                    else {
                        Write-Verbose -Message 'Calling Invoke-NaSystemApi (vfiler allow)'
                        $allowedProtocols = (Invoke-NaSystemApi -Request "<system-cli><args><arg>vfiler</arg><arg>allow</arg><arg>$( $vfiler.Name )</arg></args></system-cli>" -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'

                        Write-Verbose -Message 'Calling Invoke-NaSystemApi (vfiler disallow)'
                        $disallowedProtocols = (Invoke-NaSystemApi -Request "<system-cli><args><arg>vfiler</arg><arg>disallow</arg><arg>$( $vfiler.Name )</arg></args></system-cli>" -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'

                        [pscustomobject]@{
                            AllowedProtocols    = $allowedProtocols -split '\n' | ForEach-Object {
                                ($_ -split 'proto=')[1] | Where-Object { -not [string]::IsNullOrWhiteSpace($_) }
                            }
                            DisallowedProtocols = $disallowedProtocols -split '\n' | ForEach-Object {
                                ($_ -split 'proto=')[1] | Where-Object { -not [string]::IsNullOrWhiteSpace($_) }
                            }
                        }
                    }
                )
            }

            if ($systemData.OntapiVersion -ge [version]::Parse('1.14')) {
                Write-Verbose -Message 'Calling Get-NaNetIpspace'
                $systemData.Net | Add-Member -MemberType NoteProperty -Name Ipspace -Value (Get-NaNetIpspace -Controller $Controller -ErrorVariable +NetAppDocsError)
            }
        }
        else {
            Write-Verbose -Message '- Identifying default vfiler configuration -'

            Write-Verbose -Message 'Calling Read-NaFile (/etc/resolv.conf)'
            $resolvConf = Read-NaFile "/vol/$( ($systemData.VolRoot).Name )/etc/resolv.conf" -Controller $Controller -ErrorAction SilentlyContinue

            Write-Verbose -Message 'Calling Get-NaOption (dns.domainname, nis.domainname, nis.servers)'
            $systemData | Add-Member -MemberType NoteProperty -Name Vfiler -Value (
                [pscustomobject]@{
                    Name     = 'vfiler0'
                    Status   = 'running'
                    DnsInfo  = [pscustomobject]@{
                        DnsDomainname = (Get-NaOption -OptionNames 'dns.domainname' -Controller $Controller -ErrorVariable +NetAppDocsError).Value
                        DnsServers    = $(
                            if (-not [string]::IsNullOrWhiteSpace($resolvConf)) {
                                $resolvConf -split '\n' | Where-Object { $_ -match '^\s*nameserver' } | ForEach-Object {
                                    [pscustomobject]@{
                                        Server = ($_ -replace 'nameserver').Trim()
                                    }
                                }
                            }
                        )
                    }
                    NisInfo  = [pscustomobject]@{
                        NisDomainname = (Get-NaOption -OptionNames 'nis.domainname' -Controller $Controller -ErrorVariable +NetAppDocsError).Value
                        NisServers    = (Get-NaOption -OptionNames 'nis.servers' -Controller $Controller -ErrorVariable +NetAppDocsError).Value -split ',' | ForEach-Object {
                            [pscustomobject]@{
                                Server = $_.Trim()
                            }
                        }
                    }
                    Protocol = [pscustomobject]@{
                        AllowedProtocols    = 'rsh', 'ssh', 'nfs', 'cifs', 'iscsi', 'ftp', 'http'
                        DisallowedProtocols = @( )
                    }
                }
            )

            $systemData.Vfiler | Add-Member -MemberType NoteProperty -Name Ipspace -Value $null
        }

        #endregion Vfiler Configuration

        #region Volume Configuration

        Write-Verbose -Message '- Gathering detailed volume information -'

        ##Unused Cmdlet: Get-NaSnapshotDelta - Not needed for build docs
        ##Unused Cmdlet: Get-NaSnapshotReclaimable - Not needed for build docs
        ##Unused Cmdlet: Get-NaSnapshotRestoreStatus - Not needed for build docs
        ##Unused Cmdlet: Get-NaSnapshotVolumeSpace - Not needed for build docs
        ##Unused Cmdlet: Get-NaVolCharmap - Not needed for build docs
        ##Unused Cmdlet: Get-NaVolCloneSplit - Not needed for build docs
        ##Unused Cmdlet: Get-NaVolCloneSplitEstimate - Not needed for build docs
        ##Unused Cmdlet: Get-NaVolContainer - Info exists in Get-NaAggr
        ##Unused Cmdlet: Get-NaVolMove - Not needed for build docs
        ##Unused Cmdlet: Get-NaVolSis - ????
        ##Unused Cmdlet: Get-NaVolSize - Info exists in Get-NaVol
        ##Unused Cmdlet: Get-NaSnapshotAutodelete - Info exists in Get-NaVol
        ##Unused Cmdlet: Get-NaVolAutosize - Info exists in Get-NaVol

        Write-Verbose -Message 'Calling Get-NaVol'
        $systemData | Add-Member -MemberType NoteProperty -Name Vol -Value (Get-NaVol -Controller $Controller -ErrorVariable +NetAppDocsError)

        if ($systemData.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Calling Get-NaVolFootprint'
            $systemData | Add-Member -MemberType NoteProperty -Name VolFootprint -Value (Get-NaVolFootprint -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        if ($systemData.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Calling Get-NaVolSpace'
            $systemData | Add-Member -MemberType NoteProperty -Name VolSpace -Value (Get-NaVolSpace -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        foreach ($vol in ($systemData.Vol | Where-Object { $_ })) {
            Write-Verbose -Message "Calling Get-NaVolOption (Vol: $( $vol.Name ))"
            $vol | Add-Member -MemberType NoteProperty -Name Option -Value (Get-NaVolOption -Name $vol.Name -Controller $Controller -ErrorVariable +NetAppDocsError)

            if ($vol.State -eq 'online') {
                Write-Verbose -Message "Calling Get-NaSnapshot (Vol: $( $vol.Name ))"
                $vol | Add-Member -MemberType NoteProperty -Name Snapshot -Value (Get-NaSnapshot -TargetName $vol.Name -TargetType Volume -Controller $Controller -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message "Calling Get-NaSnapshotSchedule (Vol: $( $vol.Name ))"
                $vol | Add-Member -MemberType NoteProperty -Name SnapshotSchedule -Value (Get-NaSnapshotSchedule -TargetName $vol.Name -Controller $Controller -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message "Calling Get-NaSnapshotReserve (Vol: $( $vol.Name ))"
                $vol | Add-Member -MemberType NoteProperty -Name SnapshotReserve -Value (Get-NaSnapshotReserve -TargetName $vol.Name -TargetType Volume -Controller $Controller -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message "Calling Get-NaVolLanguage (Vol: $( $vol.Name ))"
                $vol | Add-Member -MemberType NoteProperty -Name Language -Value (Get-NaVolLanguage -Name $vol.Name -Controller $Controller -ErrorVariable +NetAppDocsError)

                if ($vol.RaidStatus -notlike '*read-only*' -and $vol.Type -ne 'trad') {
                    #If the volume belongs to a vfiler, this call to Get-NaQuotaStatus will fail with an error message (will get called again during vfiler loop below)
                    if (-not $vol.OwningVfiler -or $vol.OwningVfiler -ceq 'vfiler0') {
                        Write-Verbose -Message "Calling Get-NaVolQuotaStatus (Vol: $( $vol.Name ))"
                        $vol | Add-Member -MemberType NoteProperty -Name QuotaStatus -Value (Get-NaVolQuotaStatus -Volume $vol.Name -Controller $Controller -ErrorVariable +NetAppDocsError)
                    }
                }
            }
        }

        #endregion Volume Configuration

        #region SIS Configuration

        Write-Verbose -Message '- Gathering SIS information -'

        try {
            Write-Verbose -Message 'Calling Get-NaSis'
            $systemData | Add-Member -MemberType NoteProperty -Name Sis -Value (Get-NaSis -Controller $Controller -ErrorAction Stop)
        }
        catch {
            if ($_.Exception.Message -eq 'No Status entry found.') {
                $Global:Error.RemoveAt(0)
            }
            else {
                $NetAppDocsError += $_
            }
        }

        #endregion SIS Configuration

        #region Qtree Configuration

        Write-Verbose -Message '- Gathering qtree data -'

        Write-Verbose -Message 'Calling Get-NaQtree'
        $systemData | Add-Member -MemberType NoteProperty -Name Qtree -Value (Get-NaQtree -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion Qtree Configuration

        #region Snaplock Configuration

        ##Unused Cmdlet: Get-NaSnaplockComplianceClock - Currently no Snaplock support in NetAppDocs
        ##Unused Cmdlet: Get-NaSnaplockLog - Currently no Snaplock support in NetAppDocs
        ##Unused Cmdlet: Get-NaSnaplockLogVolume - Currently no Snaplock support in NetAppDocs
        ##Unused Cmdlet: Get-NaSnaplockOption - Currently no Snaplock support in NetAppDocs
        ##Unused Cmdlet: Get-NaSnaplockRetentionTime - Currently no Snaplock support in NetAppDocs

        #endregion Snaplock Configuration

        #region UCM Information

        if ($systemData.OntapiVersion -ge [version]::Parse('1.21')) {
            Write-Verbose -Message '- Gathering UCM information -'
            $systemData | Add-Member -MemberType NoteProperty -Name Ucm -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NaUcmAdapter'
            $systemData.Ucm | Add-Member -MemberType NoteProperty -Name Adapter -Value (Get-NaUcmAdapter -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion UCM Information

        #endregion Physical Filer Data

        #region Virtual Filer Data

        foreach ($vfiler in ($systemData.Vfiler | Where-Object { $_ -and $_.Status -ne 'DR backup' })) {
            $Controller.Vfiler = $vfiler.Name

            #region Basic vFiler System Info

            Write-Verbose -Message '- Identifying basic vFiler system info -'

            $vfiler | Add-Member -MemberType NoteProperty -Name RootPath -Value $(
                if ($vfiler.Name -ceq 'vfiler0') {
                    "/vol/$( $systemData.VolRoot.Name )"
                }
                else {
                    ($vfiler.VfStores | Where-Object { $_.IsEtc }).Path
                }
            )

            Write-Verbose -Message 'Calling Get-NaOption'
            $vfiler | Add-Member -MemberType NoteProperty -Name Option -Value (Get-NaOption -Controller $Controller -ErrorVariable +NetAppDocsError)

            #endregion Basic vFiler System Info

            #region Qtree Configuration
            #This is necessary in the vfiler context for the OwningVfiler property to be correct

            Write-Verbose -Message '- Gathering qtree data -'

            if ($vfiler.Name -cne 'vfiler0') {
                Write-Verbose -Message 'Calling Get-NaQtree'

                foreach ($vfQtree in (Get-NaQtree -Controller $Controller -ErrorVariable +NetAppDocsError | Where-Object { $_ })) {
                    $systemData.Qtree | Where-Object { $_.Volume -ceq $vfQtree.Volume -and $_.Qtree -ceq $vfQtree.Qtree } | Add-Member -MemberType NoteProperty -Name OwningVfiler -Value $vfiler.Name -Force
                }
            }

            #endregion Qtree Configuration

            #region Cifs

            ##Unused Cmdlet: Get-NaCifsBranchCacheHashStatistics - Not needed for build docs
            ##Unused Cmdlet: Get-NaCifsOrganizationalUnit - Not needed for build docs
            ##Unused Cmdlet: Get-NaCifsSession - Not needed for build docs
            ##Unused Cmdlet: Get-NaCifsSite - Not needed for build docs
            ##Unused Cmdlet: Get-NaCifsStatistics - Not needed for build docs
            ##Unused Cmdlet: Get-NaCifsNetBiosAlias - Not needed for build docs (vFiler.Cifs.NetBiosAlias)

            if (($systemData.License | Where-Object { $_.Service -eq 'cifs' }).IsLicensed -and (Test-NaCifs -Controller $Controller -ErrorAction Ignore) -eq 'started') {
                Write-Verbose -Message '- Gathering CIFS Configuration information -'
                $vfiler | Add-Member -MemberType NoteProperty -Name Cifs -Value ([pscustomobject]@{ })

                Write-Verbose -Message 'Calling Get-NaCifs'
                $vfiler.Cifs | Add-Member -MemberType NoteProperty -Name Config -Value (Get-NaCifs -Controller $Controller -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message 'Calling Get-NaCifsHomeDirectory'
                $vfiler.Cifs | Add-Member -MemberType NoteProperty -Name HomeDirectory -Value (Get-NaCifsHomeDirectory -Controller $Controller -ErrorVariable +NetAppDocsError)

                #BURT 456727
                try {
                    Write-Verbose -Message 'Calling Get-NaCifsShare'
                    $vfiler.Cifs | Add-Member -MemberType NoteProperty -Name Share -Value (Get-NaCifsShare -Controller $Controller -ErrorAction Stop)
                }
                catch {
                    if ($_.Exception.Message -like '*Unable to create iterator*') {
                        $Global:Error.RemoveAt(0)
                    }
                    else {
                        $NetAppDocsError += $_
                    }
                }

                #BURT 443207
                if ($vfiler.Name -ceq 'vfiler0') {
                    if ($systemData.OntapiVersion -ge [version]::Parse('1.14')) {
                        #BURT 456727
                        try {
                            Write-Verbose -Message 'Calling Get-NaCifsShareAcl'
                            $vfiler.Cifs | Add-Member -MemberType NoteProperty -Name ShareAcl -Value (Get-NaCifsShareAcl -Controller $Controller -ErrorAction Stop)
                        }
                        catch {
                            if ($_.Exception.Message -like '*Unable to create iterator*') {
                                $Global:Error.RemoveAt(0)
                            }
                            else {
                                $NetAppDocsError += $_
                            }
                        }
                    }
                    else {
                        $vfiler.Cifs | Add-Member -MemberType NoteProperty -Name ShareAcl -Value $(
                            Write-Verbose -Message 'Calling Invoke-NaSystemApi (cifs shares)'
                            Convert-VfilerCifsShareAcl -CifsShareString (Invoke-NaSystemApi -Request '<system-cli><args><arg>cifs</arg><arg>shares</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'
                        )
                    }
                }
                else {
                    if ($systemData.OntapiVersion -ge [version]::Parse('1.20')) {
                        #BURT 456727
                        try {
                            Write-Verbose -Message 'Calling Get-NaCifsShareAcl'
                            $vfiler.Cifs | Add-Member -MemberType NoteProperty -Name ShareAcl -Value (Get-NaCifsShareAcl -Controller $Controller -ErrorAction Stop)
                        }
                        catch {
                            if ($_.Exception.Message -like '*Unable to create iterator*') {
                                $Global:Error.RemoveAt(0)
                            }
                            else {
                                $NetAppDocsError += $_
                            }
                        }
                    }
                }

                Write-Verbose -Message 'Calling Get-NaOption (cifs.wins_servers)'
                $winsServers = (Get-NaOption -OptionNames 'cifs.wins_servers' -Controller $Controller -ErrorAction Ignore).Value
                $vfiler.Cifs | Add-Member -MemberType NoteProperty -Name WinsServers -Value $(
                    if ($winsServers) {
                        $winsServers -split '\s*,\s*' | Where-Object { -not [string]::IsNullOrWhiteSpace($_) } | ForEach-Object {
                            [pscustomobject]@{
                                Server = $_.Trim()
                            }
                        }
                    }
                )
            }

            #endregion Cifs

            #region FPolicy

            ##Unused Cmdlet: Get-NaFpolicyOption - Not needed for build docs (fpolicy.Option)
            ##Unused Cmdlet: Get-NaFpolicyServer - Not needed for build docs (fpolicy.Server)
            ##Unused Cmdlet: Get-NaFpolicySecondaryServer - Not needed for build docs (fpolicy.SecondaryServer)
            ##Unused Cmdlet: Get-NaFpolicyVolumeList - Not needed for build docs (fpolicy.VolumeList)

            Write-Verbose -Message 'Capturing Fpolicy configuration data'

            if (
                ($systemData.License | Where-Object { $_.Service -eq 'cifs' }).IsLicensed -and
                (Test-NaCifs -Controller $Controller -ErrorAction Ignore) -eq 'started' -and
                (Test-NaFpolicy -Controller $Controller -ErrorAction Ignore)
            ) {
                $vfiler | Add-Member -MemberType NoteProperty -Name Fpolicy -Value ([pscustomobject]@{ })

                Write-Verbose -Message 'Calling Get-NaFpolicy'
                $vfiler.Fpolicy | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NaFpolicy -Controller $Controller -ErrorVariable +NetAppDocsError)

                foreach ($fpolicy in ($vfiler.Fpolicy.Info | Where-Object { $_ })) {
                    Write-Verbose -Message "Calling Get-NaFpolicyExtensionList (FPolicy: $( $fpolicy.Name ))"
                    $fpolicy | Add-Member -MemberType NoteProperty -Name ExtensionList -Value (Get-NaFpolicyExtensionList -Name $fpolicy.Name -Controller $Controller -ErrorVariable +NetAppDocsError | Select-Object -Property Include, Exclude)
                }
            }

            #endregion FPolicy

            #region User/Group Data

            ##Unused Cmdlet: Get-NaUnixGroup - Not needed for build docs?

            Write-Verbose -Message '- Gathering local group and user data -'

            Write-Verbose -Message 'Calling Get-NaGroup'
            $vfiler | Add-Member -MemberType NoteProperty -Name Group -Value (Get-NaGroup -GetCapabilities -Controller $Controller -ErrorVariable +NetAppDocsError)
            foreach ($group in ($vfiler.Group | Where-Object { $_ })) {
                Write-Verbose -Message "Calling Get-NaDomainUser (Group: $group)"
                $group | Add-Member -MemberType NoteProperty -Name DomainUser -Value (Get-NaDomainUser -Group $group.Name -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false)
            }

            Write-Verbose -Message 'Calling Get-NaUser'
            $vfiler | Add-Member -MemberType NoteProperty -Name User -Value (Get-NaUser -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NaRole'
            $vfiler | Add-Member -MemberType NoteProperty -Name Role -Value (Get-NaRole -Controller $Controller -ErrorVariable +NetAppDocsError)

            #endregion User/Group Data

            #region iSCSI

            ##Unused Cmdlet: Get-NaIscsiConnection - Not needed for build docs
            ##Unused Cmdlet: Get-NaIscsiInitiator - Not yet needed for build docs?
            ##Unused Cmdlet: Get-NaIscsiPortal - Not needed for build docs??
            ##Unused Cmdlet: Get-NaIscsiSession - Not needed for build docs
            ##Unused Cmdlet: Get-NaIscsiStatistics - Not needed for build docs
            ##Unused Cmdlet: Get-NaIscsiInitiatorAuth - Not needed for build docs (vFiler.Iscsi.InitiatorAuth)

            if ($systemData.Iscsi) {
                Write-Verbose -Message '- Gathering iSCSI configuration -'
                $vfiler | Add-Member -MemberType NoteProperty -Name Iscsi -Value ([pscustomobject]@{ })

                Write-Verbose -Message 'Calling Test-NaIscsi'
                if (Test-NaIscsi -Controller $Controller -ErrorAction Ignore) {
                    Write-Verbose -Message 'Calling Get-NaIscsiInitiatorDefaultAuth'
                    $vfiler.Iscsi | Add-Member -MemberType NoteProperty -Name DefaultAuth -Value (Get-NaIscsiInitiatorDefaultAuth -Controller $Controller -ErrorVariable +NetAppDocsError)

                    if ($systemData.OntapiVersion -ge [version]::Parse('1.3')) {
                        Write-Verbose -Message 'Calling Get-NaIsns'
                        $vfiler.Iscsi | Add-Member -MemberType NoteProperty -Name Isns -Value (Get-NaIsns -Controller $Controller -ErrorVariable +NetAppDocsError)
                    }

                    Write-Verbose -Message 'Calling Get-NaIscsiNodeName'
                    $vfiler.Iscsi | Add-Member -MemberType NoteProperty -Name NodeName -Value (Get-NaIscsiNodeName -Controller $Controller -ErrorVariable +NetAppDocsError)

                    Write-Verbose -Message 'Calling Get-NaIscsiTargetAlias'
                    $vfiler.Iscsi | Add-Member -MemberType NoteProperty -Name TargetAlias -Value (Get-NaIscsiTargetAlias -Controller $Controller -ErrorVariable +NetAppDocsError)
                }
            }

            #endregion iSCSI

            #region LUN/iGroup Information

            ##Unused Cmdlet: Get-NaLunAttribute - Info exists in Get-NaLun
            ##Unused Cmdlet: Get-NaLunByIgroup - Not needed
            ##Unused Cmdlet: Get-NaLunClone - Not needed for build docs
            ##Unused Cmdlet: Get-NaLunCloneSplit - Not needed for build docs
            ##Unused Cmdlet: Get-NaLunComment - Info exists in Get-NaLun
            ##Unused Cmdlet: Get-NaLunGeometry - Not needed for build docs
            ##Unused Cmdlet: Get-NaLunInquiry - Not needed for build docs
            ##Unused Cmdlet: Get-NaLunMapByInitiator - Not needed
            ##Unused Cmdlet: Get-NaLunMaxSize - Not needed for build docs
            ##Unused Cmdlet: Get-NaLunMinSize - Not needed for build docs
            ##Unused Cmdlet: Get-NaLunOccupiedSize - Info exists in Get-NaLun
            ##Unused Cmdlet: Get-NaLunPersistentReservation - Not needed for build docs
            ##Unused Cmdlet: Get-NaLunSelect - Not needed for build docs
            ##Unused Cmdlet: Get-NaLunSerialNumber - Info exists in Get-NaLun
            ##Unused Cmdlet: Get-NaLunSignature - Not needed for build docs
            ##Unused Cmdlet: Get-NaLunSnapUsage - Not needed for build docs
            ##Unused Cmdlet: Get-NaLunSpaceReserved - Info exists in Get-NaLun
            ##Unused Cmdlet: Get-NaLunStatistics - Not needed for build docs
            ##Unused Cmdlet: Get-NaLunTargetDeviceId - Not needed for build docs

            Write-Verbose -Message '- Gathering LUN/iGroup information -'

            Write-Verbose -Message 'Calling Get-NaLun'
            $vfiler | Add-Member -MemberType NoteProperty -Name Lun -Value (Get-NaLun -Controller $Controller -ErrorVariable +NetAppDocsError)

            foreach ($lun in ($vfiler.Lun | Where-Object { $_ })) {
                Write-Verbose -Message "Calling Get-NaLunMap (LUN: $( $lun.Path ))"
                $lun | Add-Member -MemberType NoteProperty -Name Map -Value (Get-NaLunMap -Path $lun.Path -Controller $Controller -ErrorVariable +NetAppDocsError)
            }

            Write-Verbose -Message '- Gathering iGroup information -'

            Write-Verbose -Message 'Calling Get-NaIgroup'
            $vfiler | Add-Member -MemberType NoteProperty -Name Igroup -Value (Get-NaIgroup -Controller $Controller -ErrorVariable +NetAppDocsError)

            foreach ($igroup in ($vfiler.Igroup | Where-Object { $_.Initiators })) {
                foreach ($initiator in ($igroup.Initiators | Where-Object { $_ })) {
                    Write-Verbose -Message "Calling Confirm-NaLunInitatiorLoggedIn (Initiator: $( $initiator.InitiatorName ))"
                    $initiator | Add-Member -MemberType NoteProperty -Name IsLoggedIn -Value (Confirm-NaLunInitiatorLoggedIn -Initiator $initiator.InitiatorName -Controller $Controller -ErrorVariable +NetAppDocsError)
                }
            }

            #endregion LUN/iGroup Information

            #region NFS

            ##Unused Cmdlet: Get-NaNfsExportStoragePath - Info exists in Get-NaNfsExport
            ##Unused Cmdlet: Get-NaNfsMonitor - Not needed for build docs
            ##Unused Cmdlet: Get-NaNfsSecurityFlavor - Not needed for build docs
            ##Unused Cmdlet: Get-NaNfsStatistics - Not needed for build docs
            ##Unused Cmdlet: Get-NaNfsTopClient - Not needed for build docs

            Write-Verbose -Message '- Gathering NFS configuration -'
            $vfiler | Add-Member -MemberType NoteProperty -Name Nfs -Value ([pscustomobject]@{ })

            try {
                Write-Verbose -Message 'Calling Get-NaNfsExport'
                $vfiler.Nfs | Add-Member -MemberType NoteProperty -Name Export -Value (Get-NaNfsExport -Controller $Controller -ErrorAction Stop)

                Write-Verbose -Message 'Calling Get-NaNfsExport -Persistent'
                $vfiler.Nfs | Add-Member -MemberType NoteProperty -Name PersistentExport -Value (Get-NaNfsExport -Persistent -Controller $Controller -ErrorAction Stop)
            }
            catch {
                if ($_.Exception.ErrorName -eq 'EAPILICENSE') {
                    $Global:Error.RemoveAt(0)
                }
                else {
                    $NetAppDocsError += $_
                }
            }

            try {
                Write-Verbose -Message 'Calling Read-NaFile (/etc/netgroup)'
                $vfiler.Nfs | Add-Member -MemberType NoteProperty -Name Netgroup -Value (Convert-NetgroupEntry -Netgroup (Read-NaFile -Path "$( $vfiler.RootPath )/etc/netgroup" -Controller $Controller -ErrorAction Stop))
            }
            catch {
                if ($_.Exception.ErrorName -eq 'EONTAPI_ENOENT') {
                    $Global:Error.RemoveAt(0)
                }
                else {
                    $NetAppDocsError += $_
                }
            }

            #endregion NFS

            #region NameMapping

            Write-Verbose -Message '- Gathering name mappings information -'

            try {
                Write-Verbose -Message 'Calling Read-NaFile (/etc/usermap.cfg)'
                $vfiler | Add-Member -MemberType NoteProperty -Name NameMapping -Value (Convert-UsermapCfgEntry -UserMapCfg (Read-NaFile -Path "$( $vfiler.RootPath )/etc/usermap.cfg" -Controller $Controller -ErrorAction Stop))
            }
            catch {
                if ($_.Exception.ErrorName -eq 'EONTAPI_ENOENT') {
                    $Global:Error.RemoveAt(0)
                }
                else {
                    $NetAppDocsError += $_
                }
            }

            try {
                Write-Verbose -Message 'Calling Read-NaFile (/etc/passwd)'
                $vfiler | Add-Member -MemberType NoteProperty -Name NameMappingUnixUser -Value (Convert-PasswdEntry -Passwd (Read-NaFile -Path "$( $vfiler.RootPath )/etc/passwd" -Controller $Controller -ErrorAction Stop))
            }
            catch {
                if ($_.Exception.ErrorName -eq 'EONTAPI_ENOENT') {
                    $Global:Error.RemoveAt(0)
                }
                else {
                    $NetAppDocsError += $_
                }
            }

            #endregion NameMapping

            #region Snapmirror

            Write-Verbose -Message '- Identifying SnapMirror configuration information -'

            if (
                (
                    $systemData.OntapiVersion -ge [version]::Parse('1.20') -and
                    $null -ne ($systemData.License | Where-Object { $_.Service -eq 'snapmirror' -and ($_.Type -ne 'Demo' -or (-not [string]::IsNullOrWhiteSpace($_.ExpirationTime) -and (ConvertFrom-UnixTimestamp -Seconds $_.ExpirationTime) -gt [datetime]::UtcNow)) })
                ) -or
                (
                    $systemData.OntapiVersion -lt [version]::Parse('1.20') -and
                    ($systemData.License | Where-Object { $_.Service -eq 'snapmirror' }).IsLicensed
                )
            ) {
                $vfiler | Add-Member -MemberType NoteProperty -Name Snapmirror -Value ([pscustomobject]@{ })

                Write-Verbose -Message 'Calling Get-NaSnapmirror'
                $vfiler.Snapmirror | Add-Member -MemberType NoteProperty -Name Relationship -Value (Get-NaSnapmirror -Controller $Controller -ErrorVariable +NetAppDocsError -WarningAction SilentlyContinue)

                if ($systemData.OntapiVersion -ge [version]::Parse('1.2')) {
                    Write-Verbose -Message 'Calling Get-NaSnapmirrorSchedule'
                    $vfiler.Snapmirror | Add-Member -MemberType NoteProperty -Name Schedule -Value (Get-NaSnapmirrorSchedule -Controller $Controller -ErrorVariable +NetAppDocsError)
                }

                Write-Verbose -Message 'Calling Get-NaSnapmirrorSyncSchedule'
                $vfiler.Snapmirror | Add-Member -MemberType NoteProperty -Name SyncSchedule -Value (Get-NaSnapmirrorSyncSchedule -Controller $Controller -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message 'Calling Get-NaSnapmirrorDestination'
                $vfiler.Snapmirror | Add-Member -MemberType NoteProperty -Name Destination -Value (Get-NaSnapmirrorDestination -Controller $Controller -ErrorVariable +NetAppDocsError)

                if ($systemData.OntapiVersion -ge [version]::Parse('1.3')) {
                    Write-Verbose -Message 'Calling Get-NaSnapmirrorConnection'
                    $vfiler.Snapmirror | Add-Member -MemberType NoteProperty -Name Connection -Value (Get-NaSnapmirrorConnection -Controller $Controller -ErrorVariable +NetAppDocsError)
                }
            }

            #endregion Snapmirror

            #region Snapvault

            ##Unused Cmdlet: Get-NaSnapvaultPriSnapshotScheduleStatus - Not needed for build docs
            ##Unused Cmdlet: Get-NaSnapvaultSecSnapshotScheduleStatus - Not needed for build docs
            ##Unused Cmdlet: Get-NaSnapvaultSoftlock - Not needed for build docs
            ##Unused Cmdlet: Get-NaSnapvaultSoftlockedSnapshot - Not needed for build docs

            Write-Verbose -Message '- Gathering Snapvault configuration information -'
            $vfiler | Add-Member -MemberType NoteProperty -Name Snapvault -Value ([pscustomobject]@{ })

            if (
                (
                    $systemData.OntapiVersion -ge [version]::Parse('1.20') -and
                    $null -ne ($systemData.License | Where-Object { $_.Service -eq 'snapvault' -and ($_.Type -ne 'Demo' -or (-not [string]::IsNullOrWhiteSpace($_.ExpirationTime) -and (ConvertFrom-UnixTimestamp -Seconds $_.ExpirationTime) -gt [datetime]::UtcNow)) })
                ) -or
                (
                    $systemData.OntapiVersion -lt [version]::Parse('1.20') -and
                    ($systemData.License | Where-Object { $_.Service -eq 'sv_ontap_sec' }).IsLicensed
                )
            ) {
                Write-Verbose -Message 'Calling Get-NaSnapvaultSecStatus'
                $vfiler.Snapvault | Add-Member -MemberType NoteProperty -Name Status -Value (Get-NaSnapvaultSecStatus -Controller $Controller -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message 'Calling Get-NaSnapvault'
                $vfiler.Snapvault | Add-Member -MemberType NoteProperty -Name Config -Value (Get-NaSnapvault -Controller $Controller -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message 'Calling Get-NaSnapvaultSecSnapshotSchedule'
                $vfiler.Snapvault | Add-Member -MemberType NoteProperty -Name Schedule -Value (Get-NaSnapvaultSecSnapshotSchedule -Controller $Controller -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message 'Calling Get-NaSnapvaultSecDestinations'
                $vfiler.Snapvault | Add-Member -MemberType NoteProperty -Name Destination -Value (Get-NaSnapvaultSecDestinations -Controller $Controller -ErrorVariable +NetAppDocsError)
            }
            elseif (
                $systemData.OntapiVersion -lt [version]::Parse('1.20') -and
                ($systemData.License | Where-Object { $_.Service -eq 'sv_ontap_pri' }).IsLicensed
            ) {
                #BURT 389554
                $systemVersion = [version]::Parse($systemData.SystemVersion -replace '[A-Z].*$')

                if (
                    $systemVersion -gt [version]::Parse('8.0.2') -or
                    ($systemVersion -ge [version]::Parse('7.3.6') -and $systemVersion -lt [version]::Parse('8.0'))
                ) {
                    Write-Verbose -Message 'Calling Get-NaSnapvaultPriStatus'
                    $vfiler.Snapvault | Add-Member -MemberType NoteProperty -Name Status -Value (Get-NaSnapvaultPriStatus -Controller $Controller -ErrorVariable +NetAppDocsError)

                    Write-Verbose -Message 'Calling Get-NaSnapvaultPriSnapshotSchedule'
                    $vfiler.Snapvault | Add-Member -MemberType NoteProperty -Name Schedule -Value (Get-NaSnapvaultPriSnapshotSchedule -Controller $Controller -ErrorVariable +NetAppDocsError)

                    Write-Verbose -Message 'Calling Get-NaSnapvaultPriDestinations'
                    $vfiler.Snapvault | Add-Member -MemberType NoteProperty -Name Destination -Value (Get-NaSnapvaultPriDestinations -Controller $Controller -ErrorVariable +NetAppDocsError)
                }
            }

            #endregion Snapvault

            #region Quota Information

            ##Unused Cmdlet: Get-NaQuota - Info exists in Get-NaQuotaEntry
            ##Unused Cmdlet: Get-NaQuotaReport - Info exists in Get-NaQuotaEntry

            Write-Verbose -Message '- Gathering quota information -'

            if ($vfiler.Status -eq 'running') {
                Write-Verbose -Message 'Calling Get-NaQuotaEntry'
                $vfiler | Add-Member -MemberType NoteProperty -Name QuotaEntry -Value (Get-NaQuotaEntry -Controller $Controller -ErrorVariable +NetAppDocsError)
            }

            foreach ($vol in ($systemData.Vol | Where-Object { "/vol/$( $_.Name )" -cin (($vfiler | Where-Object { $_.Name -cne 'vfiler0' }).VfStores.Path | Where-Object { -not [string]::IsNullOrWhiteSpace($_) }) -and -not [string]::IsNullOrWhiteSpace($_.QuotaStatus) -and $_.State -eq 'online' -and $_.RaidStatus -notlike '*read-only*' })) {
                Write-Verbose -Message "Calling Get-NaVolQuotaStatus (Vol: $( $vol.Name ))"
                $vol | Add-Member -MemberType NoteProperty -Name QuotaStatus -Value (Get-NaVolQuotaStatus -Volume $vol.Name -Controller $Controller -ErrorVariable +NetAppDocsError)
            }

            #endregion Quota Information

            #region Vfiler0 Context Required

            ###The following section requires the vfiler to be running and the vfiler context be set to vfiler0.
            ###This section must remain at the end of the foreach loop
            #BURT 419019
            if ($vfiler.Status -eq 'running') {
                $Controller.Vfiler = 'vfiler0'

                #region Additional vFiler Options

                Write-Verbose -Message '- Gathering additional hidden options -'

                # BURT 384034
                $additionalOptions = @(
                    'ndmpd.enable'
                )

                foreach ($additionalOption in ($additionalOptions | Where-Object { $_ })) {
                    if (-not ($vfiler.Option | Where-Object { $_.Name -eq $additionalOption })) {
                        $optRequest = New-Object -TypeName System.Xml.XmlDocument

                        if ($vfiler.Name -ceq 'vfiler0') {
                            $optRequest.LoadXml("<system-cli><args><arg>options</arg><arg>$additionalOption</arg></args></system-cli>")
                        }
                        else {
                            $optRequest.LoadXml("<system-cli><args><arg>vfiler</arg><arg>run</arg><arg>-q</arg><arg>$( $vfiler.Name )</arg><arg>options</arg><arg>$additionalOption</arg></args></system-cli>")
                        }

                        Write-Verbose -Message 'Calling Invoke-NaSystemApi (options)'
                        $optResults = (Invoke-NaSystemApi -RequestXML $optRequest -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' -split '\s+'

                        $vfiler.Option += [pscustomobject]@{
                            ClusterConstraint = 'none'
                            Name              = $optResults[0].Trim()
                            Value             = $optResults[1].Trim()
                        }
                    }
                }

                #endregion Addition vFiler Options

                #region Cifs Share Acls

                if ($vfiler.Name -cne 'vfiler0' -and $vfiler.Cifs -and (-not $vfiler.Cifs.ShareAcl)) {
                    $vfiler.Cifs | Add-Member -MemberType NoteProperty -Name ShareAcl -Value $(
                        Write-Verbose -Message 'Calling Invoke-NaSystemApi (cifs shares)'
                        Convert-VfilerCifsShareAcl -CifsShareString (Invoke-NaSystemApi -Request "<system-cli><args><arg>vfiler</arg><arg>run</arg><arg>-q</arg><arg>$( $vfiler.Name )</arg><arg>cifs</arg><arg>shares</arg></args></system-cli>" -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'
                    )
                }

                #endregion Cifs Share Acls

                #region Vscan Settings

                Write-Verbose -Message '- Gathering Vscan settings -'

                $vfiler | Add-Member -MemberType NoteProperty -Name Vscan -Value ([pscustomobject]@{ })

                $vscanRequest = New-Object -TypeName System.Xml.XmlDocument
                $vscanOptionsRequest = New-Object -TypeName System.Xml.XmlDocument

                if ($vfiler.Name -ceq 'vfiler0') {
                    $vscanRequest.LoadXml('<system-cli><args><arg>vscan</arg></args></system-cli>')
                    $vscanOptionsRequest.LoadXml('<system-cli><args><arg>vscan</arg><arg>options</arg></args></system-cli>')
                }
                else {
                    $vscanRequest.LoadXml("<system-cli><args><arg>vfiler</arg><arg>run</arg><arg>-q</arg><arg>$( $vfiler.Name )</arg><arg>vscan</arg></args></system-cli>")
                    $vscanOptionsRequest.LoadXml("<system-cli><args><arg>vfiler</arg><arg>run</arg><arg>-q</arg><arg>$( $vfiler.Name )</arg><arg>vscan</arg><arg>options</arg></args></system-cli>")
                }

                Write-Verbose -Message 'Calling Invoke-NaSystemApi (vscan)'
                $vscanResults = (Invoke-NaSystemApi -RequestXML $vscanRequest -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'

                Write-Verbose -Message 'Calling Invoke-NaSystemApi (vscan options)'
                $vscanOptionsResults = (Invoke-NaSystemApi -RequestXML $vscanOptionsRequest -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output'

                Write-Verbose -Message 'Calling Get-NaVscan'
                $vfiler.Vscan = Get-NaVscan -Vscan $vscanResults -VscanOptions $vscanOptionsResults

                #endregion Vscan Settings
            }

            #endregion Vfiler0 Context Required
        }

        #endregion Virtual Filer Data

        $Controller.Vfiler = $null

        Write-Verbose -Message '- Getting current timestamp -'

        Write-Verbose -Message 'Calling Get-NaTime'
        $systemData | Add-Member -MemberType NoteProperty -Name ReferenceDT -Value (ConvertFrom-UnixTimestamp -Seconds (Get-NaTime -Controller $Controller -ErrorVariable +NetAppDocsError).UtcTime)

        Write-Verbose -Message '- Getting uptime -'

        Write-Verbose -Message 'Calling Get-NaPerfData -Name system -Counters uptime'
        $systemData.SystemInfo | Add-Member -MemberType NoteProperty -Name NodeUptime -Value ([long](New-TimeSpan -End ([datetime]::Now.AddSeconds(((Get-NaPerfData -Name system -Counters uptime -Controller $Controller -ErrorVariable +NetAppDocsError).Counters | Select-Object -First 1).Value))).TotalSeconds)

        Write-Verbose -Message '- Identifying any errors during processing -'
        $systemData | Add-Member -MemberType NoteProperty -Name DiscoveryError -Value @(
            foreach ($err in ($NetAppDocsError | Where-Object { $_ })) {
                $err.CategoryInfo | Select-Object -Property *, @{
                    Name       = 'Message'
                    Expression = { $err.Exception.Message }
                }
            }
        )

        if (($systemData.DiscoveryError | Measure-Object).Count -gt 0) {
            $msg = New-Object -TypeName System.Text.StringBuilder

            [void]$msg.AppendLine("Errors were encountered while collecting data from system: $( $Controller.Name )")
            [void]$msg.AppendLine('-------------------------------------------------------------------------------------')
            [void]$msg.AppendLine('Please review the output for accuracy and, if necessary, correct any error conditions')
            [void]$msg.AppendLine('then run these scripts again.')
            [void]$msg.AppendLine('-------------------------------------------------------------------------------------')
            [void]$msg.AppendLine()

            Write-Warning -Message $msg.ToString()
        }

        $systemData
    }
    catch {
        $Global:Error.RemoveAt(0)

        $msg = New-Object -TypeName System.Text.StringBuilder

        [void]$msg.AppendLine('An unexpected error occurred. Processing halted.')
        [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
        [void]$msg.AppendLine('As this error is not expected, please consider contacting the support staff via e-mail or by')
        [void]$msg.AppendLine('starting a thread on the community site.')
        [void]$msg.AppendLine('            E-mail: ng-NetAppDocs-support@netapp.com')
        [void]$msg.AppendLine('    Community site: https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs')
        [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
        [void]$msg.AppendLine($( $_.Exception.GetBaseException().Message ))
        [void]$msg.AppendLine("Location: $( $_.ScriptStackTrace )")
        [void]$msg.AppendLine()

        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $_.CategoryInfo.TargetName)))
    }
    finally {
        $Controller.Vfiler = $null
    }
}

#region Misc Unused Cmdlets

##Unused Cmdlet: Get-NaClone - Not needed for build docs
##Unused Cmdlet: Get-NaCopyOffload - Not needed for build docs
##Unused Cmdlet: Get-NaFile - Not needed for build docs
##Unused Cmdlet: Get-NaFileFingerprint - Not needed for build docs
##Unused Cmdlet: Get-NaFileSymlink - Not needed for build docs
##Unused Cmdlet: Get-NaLockStatus - Not needed for build docs
##Unused Cmdlet: Get-NaNdmpCopy - Not needed for build docs
##Unused Cmdlet: Get-NaPerfCounter - Not needed for build docs
##Unused Cmdlet: Get-NaPerfInstance - Not needed for build docs
##Unused Cmdlet: Get-NaPerfObject - Not needed for build docs
##Unused Cmdlet: Get-NaRadiusServer - Not yet needed for build docs?
##Unused Cmdlet: Get-NaRadiusStatistics - Not needed for build docs
##Unused Cmdlet: Get-NaReallocateJobs - Not needed for build docs
##Unused Cmdlet: Get-NaRshSession - Not needed for build docs
##Unused Cmdlet: Get-NaRshStatistics - Not needed for build docs
##Unused Cmdlet: Get-NaSystemLog - Not needed for build docs (generates too much data)

#endregion Unused Cmdlets

#region Functions

function Get-NaVscan {
    <#
        .SYNOPSIS
            Collects the Vscan configuration data via system-cli API calls.

        .DESCRIPTION
            This function is only necessary as there are no Vscan APIs.

        .NOTES
            CAVEAT : This function parses raw text output, so at any point this may break due to changes in the text.
    #>

    param (
        # The raw output string from the 'vscan' CLI command.
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$Vscan,

        # The raw output string from the 'vscan options' CLI command.
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$VscanOptions
    )

    $ipv4Regex = [regex]'(?<First>2[0-4]\d|25[0-5]|[01]?\d\d?)\.(?<Second>2[0-4]\d|25[0-5]|[01]?\d\d?)\.(?<Third>2[0-4]\d|25[0-5]|[01]?\d\d?)\.(?<Fourth>2[0-4]\d|25[0-5]|[01]?\d\d?)'

    $vscanSections = $Vscan -split '\n\n' | ForEach-Object { $_.Trim() }

    $vscanObj = [pscustomobject]@{
        Scanners = @( )
        Options  = @( )
    }

    (($vscanSections[1] -split '--*\n')[1] -split '\n') | ForEach-Object {
        if ($_ -match "(?<IP>$ipv4Regex)\s+\\\\(?<Name>.*)\s+(?<PriOrSec>Pri|Sec)") {
            $vscanObj.Scanners += [pscustomobject]@{
                IP       = $Matches.IP
                Name     = $Matches.Name
                PriOrSec = $Matches.PriOrSec
            }

            if ($Matches) {
                $Matches.Clear()
            }
        }
    }

    $VscanOptions -replace 'vscan options ' -replace ':' -split '\n' | Where-Object { -not [string]::IsNullOrWhiteSpace($_) } | ForEach-Object {
        $opt = $_ -split '\s+'
        $vscanObj.Options += [pscustomobject]@{
            Name  = $opt[0].Trim()
            Value = $opt[1].Trim()
        }
    }

    [pscustomobject]@{
        Status        = [regex]::Match($vscanSections[0], 'enabled|disabled').Value
        InclusionList = ($vscanSections[2] -split ':' | ForEach-Object { if ($_ -notlike '*list is empty*') { $_.Trim() } })[1]
        ExclusionList = ($vscanSections[3] -split ':' | ForEach-Object { if ($_ -notlike '*list is empty*') { $_.Trim() } })[1]
        Scanner       = $vscanObj.Scanners
        Option        = $vscanObj.Options
    }
}

function Get-NaShelfAta {
    <#
        .SYNOPSIS
            Parses the raw sysconfig -v string to gather ATA shelf information.

        .DESCRIPTION
            This function is only necessary for older version of ONTAPi that did not return ATA shelf information. Only returns
            limited information as compared to other modules output from the API.

        .NOTES
            CAVEAT : This function parses raw text output, so at any point this may break due to changes in the text.
    #>

    param (
        # The raw output string from the 'sysconfig -v' output.
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$Sysconfig,

        # The raw output string from the 'storage show shelf' output.
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$StorageShowShelf
    )

    $storageShowShelfArray = @( )

    foreach ($line in ($StorageShowShelf -split '\n' | Select-String -Pattern ': ', '\[\s*\d*\]')) {
        switch -Regex ($line) {
            '(.+):\s+(.+)' {
                $Key, $Value = $matches[1..2]

                if ($Key -like '*Shelf name*') {
                    if ($shelfObj) {
                        $storageShowShelfArray += $shelfObj
                    }

                    $shelfObj = [pscustomobject]@{
                        ShelfName = $Value.ToLower()
                        BayCount  = 0
                        Bays      = 0
                    }
                }
                else {
                    $shelfObj | Add-Member -MemberType NoteProperty -Name ((Get-Culture).TextInfo.ToTitleCase($Key) -replace '\s', '' -replace 'Termswitch', 'TerminationSwitch' -replace 'ShelfUID', 'ShelfUid') -Value $Value.ToLower()
                }

                break
            }
            '\s*\[\s*\d*\]' {
                $shelfObj.BayCount++
                $shelfObj.Bays++

                break
            }
            default { }
        }
    }

    $storageShowShelfArray += $shelfObj

    $fwRegex = [regex]'(?:\s*Shelf\s(?<ShelfId>\d+)\:\s+(?<ShelfType>.*)\s+Firmware.*[A]\:\s+(?<FirmwareRevA>\d+)\s+AT-FCX\s[B]\:\s+(?<FirmwareRevB>\d+|--)?)'
    $hbaRegex = [regex]'(?:\s*slot.*Adapter\s(?<ChannelName>\d+\w+)\s.*?)'

    foreach ($line in ($Sysconfig -split '\n' | Select-String -Pattern 'FC Host Adapter', 'Shelf.*AT-FCX')) {
        switch -Regex ($line) {
            $hbaRegex {
                $hbaPort = $Matches.ChannelName.Trim()

                break
            }
            $fwRegex {
                $shelfMatch = $Matches
                $shelfName = "$hbaPort.shelf$( $shelfMatch.ShelfId.Trim() )"
                $storageShowShelfEntry = $storageShowShelfArray | Where-Object { $_.ShelfName -eq $shelfName }

                [pscustomobject]@{
                    Name              = $shelfName
                    Status            = $storageShowShelfEntry.ShelfState
                    Channel           = $hbaPort
                    ID                = $shelfMatch.ShelfId.Trim()
                    Bays              = $storageShowShelfEntry.BayCount
                    Type              = $shelfMatch.ShelfType.Trim()
                    ChannelName       = $hbaPort
                    FirmwareRevA      = $shelfMatch.FirmwareRevA.Trim()
                    FirmwareRevB      = $shelfMatch.FirmwareRevB.Trim()
                    Module            = $storageShowShelfEntry.Module
                    ModuleState       = $storageShowShelfEntry.ModuleState
                    BayCount          = $storageShowShelfEntry.BayCount
                    ShelfId           = $shelfMatch.ShelfId.Trim()
                    ShelfName         = $shelfName
                    ShelfState        = $storageShowShelfEntry.ShelfState
                    ShelfType         = $shelfMatch.ShelfType.Trim()
                    ShelfUid          = $storageShowShelfEntry.ShelfUid
                    TerminationSwitch = $storageShowShelfEntry.TerminationSwitch
                }

                if ($Matches) {
                    $Matches.Clear()
                }

                break
            }
            default { }
        }
    }
}

function Convert-UsermapCfgEntry {
    param (
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$UserMapCfg
    )

    if (-not [string]::IsNullOrWhiteSpace($UserMapCfg)) {
        $i = 1

        $userEntryObj = [pscustomobject]@{
            Position           = 0
            WindowsIpQualifier = $null
            WindowsName        = $null
            Direction          = $null
            UnixIpQualifier    = $null
            UnixName           = $null
        }

        foreach ($line in ($UserMapCfg -split '\n' | Where-Object { -not [string]::IsNullOrWhiteSpace($_) -and $_ -notmatch '^(?:\s*|[#].*)$' })) {
            $entry = $line -split '(?s)\s+(?=(?:[^"]*"[^"]*")*[^"]*$)', 3 | Where-Object { -not [string]::IsNullOrWhiteSpace($_) }

            $outputObj = $userEntryObj.psobject.Copy()

            switch -Exact ($entry.Count) {
                '3' {
                    $outputObj.Direction = $(
                        switch -Exact ($entry[1]) {
                            '=>' { 'win_unix' }
                            '<=' { 'unix_win' }
                            '==' { 'bidirectional' }
                        }
                    )

                    $outputObj.Position = $i
                    $outputObj.WindowsName, $outputObj.WindowsIpQualifier = ($entry[0].Split(':', 2))[1, 0] | ForEach-Object { $_.Trim() }
                    $outputObj.UnixName, $outputObj.UnixIpQualifier = ($entry[2].Split(':', 2))[1, 0] | ForEach-Object { $_.Trim() }

                    $outputObj
                }
                '2' {
                    $outputObj.Direction = 'bidirectional'
                    $outputObj.Position = $i
                    $outputObj.WindowsName, $outputObj.WindowsIpQualifier = ($entry[0].Split(':', 2))[1, 0] | ForEach-Object { $_.Trim() }
                    $outputObj.UnixName, $outputObj.UnixIpQualifier = ($entry[1].Split(':', 2))[1, 0] | ForEach-Object { $_.Trim() }

                    $outputObj
                }
                default {
                    Write-Warning -Message "There is an invalid entry in the usermap.cfg file. This entry (#$i) has been skipped and the output data may be invalid."
                }
            }

            $i++
        }
    }
}

function Convert-VfilerCifsShareAcl {
    param (
        [Parameter(Mandatory)]
        [string]$CifsShareString
    )

    foreach ($line in (($CifsShareString -split "`n") | Where-Object { -not [string]::IsNullOrWhiteSpace($_) -and $_ -notmatch '^(?:Name.+|[----].+|\s*)$' })) {
        if ($line -match '^\S') {
            if ($outputObj) {
                $outputObj
            }

            $outputObj = [pscustomobject]@{
                ShareName   = ($line -split '[/]')[0].Trim()
                UserAclInfo = @( )
            }
        }
        elseif ($line -match '\s+[/]\s+') {
            $user, $accessRight = $line -split '\s+[/]\s+', 2

            if ($user -match '{g}') {
                $outputObj.UserAclInfo += [pscustomobject]@{
                    UserName      = $null
                    AccessRights  = $AccessRight.Trim()
                    UnixGroupName = ($user -replace '{g}').Trim()
                }
            }
            else {
                $outputObj.UserAclInfo += [pscustomobject]@{
                    UserName      = $user.Trim()
                    AccessRights  = $AccessRight.Trim()
                    UnixGroupName = $null
                }
            }
        }
    }

    if (($outputObj.UserAclInfo | Measure-Object).Count -gt 0) {
        $outputObj
    }
}

function Convert-PasswdEntry {
    param (
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$Passwd
    )

    #NAME:PASSWD:UID:GID:COMMENT:DIRECTORY:SHELL

    if (-not [string]::IsNullOrWhiteSpace($Passwd)) {
        foreach ($line in ($Passwd -split '\n' | Where-Object { -not [string]::IsNullOrWhiteSpace($_) -and $_ -notmatch '^(?:\s*[#]|\s*$)' })) {
            $entry = $line -split ':', 7

            [pscustomobject]@{
                UserName = $entry[0]
                UserId   = $entry[2]
                GroupId  = $entry[3]
            }
        }
    }
}

function Convert-NetgroupEntry {
    param (
        [Parameter(Mandatory)]
        [AllowEmptyString()]
        [string]$Netgroup
    )

    #groupname member-list (hostname, username, domainname)

    if (-not [string]::IsNullOrWhiteSpace($Netgroup)) {
        foreach ($line in ($Netgroup -split '\n' | Where-Object { -not [string]::IsNullOrWhiteSpace($_) -and $_ -notmatch '^(?:\s*[#]|\s*$)' })) {
            $entry = $line -split '\s+'

            foreach ($member in ($entry | Select-Object -Skip 1 | Where-Object { -not [string]::IsNullOrWhiteSpace($_) })) {
                if ($member -like '(*)') {
                    $hostName, $userName, $domainName = ($member -replace '[)]' -replace '[(]') -split '\s*,\s*', 3

                    [pscustomobject]@{
                        NetgroupName   = $entry[0]
                        MemberNetgroup = $null
                        Host           = $hostName
                        User           = $userName
                        Domain         = $domainName
                    }
                }
                else {
                    [pscustomobject]@{
                        NetgroupName   = $entry[0]
                        MemberNetgroup = $member
                        Host           = $null
                        User           = $null
                        Domain         = $null
                    }
                }
            }
        }
    }
}

#endregion Functions

# SIG # Begin signature block
# MIIq0AYJKoZIhvcNAQcCoIIqwTCCKr0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCA8GR5rI/y2ol83
# 9JkxMbq/GPfGZ1+uoX9e0CF4/x8/0KCCJGgwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggWNMIIEdaADAgECAhAOmxiO
# +dAt5+/bUOIIQBhaMA0GCSqGSIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYD
# VQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAi
# BgNVBAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAw
# MDBaFw0zMTExMDkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
# Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERp
# Z2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBAL/mkHNo3rvkXUo8MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsb
# hA3EMB/zG6Q4FutWxpdtHauyefLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iT
# cMKyunWZanMylNEQRBAu34LzB4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGb
# NOsFxl7sWxq868nPzaw0QF+xembud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclP
# XuU15zHL2pNe3I6PgNq2kZhAkHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCr
# VYJBMtfbBHMqbpEBfCFM1LyuGwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFP
# ObURWBf3JFxGj2T3wWmIdph2PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTv
# kpI6nj3cAORFJYm2mkQZK37AlLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWM
# cCxBYKqxYxhElRp2Yn72gLD76GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls
# 5Q5SUUd0viastkF13nqsX40/ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBR
# a2+xq4aLT8LWRV+dIPyhHsXAj6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6
# MIIBNjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qY
# rhwPTzAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8E
# BAMCAYYweQYIKwYBBQUHAQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5k
# aWdpY2VydC5jb20wQwYIKwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDig
# NoY0aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNybDARBgNVHSAECjAIMAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCg
# v0NcVec4X6CjdBs9thbX979XB72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQT
# SnovLbc47/T/gLn4offyct4kvFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh
# 65ZyoUi0mcudT6cGAxN3J0TU53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSw
# uKFWjuyk1T3osdz9HNj0d1pcVIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAO
# QGPFmCLBsln1VWvPJ6tsds5vIy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjD
# TZ9ztwGpn1eqXijiuZQwggXMMIIENKADAgECAhAg429sPxgagUb53pPffJfkMA0G
# CSqGSIb3DQEBDAUAMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBS
# MzYwHhcNMjEwOTA5MDAwMDAwWhcNMjMwOTA5MjM1OTU5WjBQMQswCQYDVQQGEwJV
# UzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UECgwMTmV0QXBwLCBJbmMuMRUw
# EwYDVQQDDAxOZXRBcHAsIEluYy4wggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGK
# AoIBgQC4kYYj/oViZD9pN03hrqFjtwOz1Gx4eDWVV8IYUYEr2qpLeNYvWz26B/+E
# mYLboAGVpkYg0Wske3hjRpooZlMORm1S4+2C2NoQCstZ+kmNTevbXHJe/w1VNJrm
# fKjpXOJEfx8GUGDqCMR30dChVpsdBEavrM7T0hnfJLv18i19SGv3a+nhvB3iOjLP
# SEg8+DFkjD5jTI6fQ83DBEiVR4UEYq7yyH578ZEylhsRfJmK+uIREycEUk/NpFTw
# g6/7lxh/WYabCrZzqI4Ep3QataRKaAlxt3BiOlZkX4WGX3DYvSYltWOM2UlCcGpP
# m/1/LN8Xuvf+YO6H472QecHe59XtXbHSMKvqDuOEJu8Wk2LFMNK732o3Fc5QIHie
# 6JtNsgNYf+Vsm5EMBD1ZHNE+C3zlmQbbWIU5uEU1nhiodBPKsu0SkS6pARyKBO05
# DSc2gWWv0aukKi04ZAn+hAdSl5s1dgmU5shBvKHqe15K9ZRN4FFO/CgHE0BoGYQS
# UQVKwa0CAwEAAaOCAZwwggGYMB8GA1UdIwQYMBaAFA8qyyCHKLjsb0iuK1SmKaoX
# pM0MMB0GA1UdDgQWBBQuH643KcBMmb/Q6IZt+H9IrnXFwDAOBgNVHQ8BAf8EBAMC
# B4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDAzARBglghkgBhvhC
# AQEEBAMCBBAwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwIwJTAjBggrBgEFBQcC
# ARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQBMEkGA1UdHwRCMEAw
# PqA8oDqGOGh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVT
# aWduaW5nQ0FSMzYuY3JsMHkGCCsGAQUFBwEBBG0wazBEBggrBgEFBQcwAoY4aHR0
# cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQVIz
# Ni5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqG
# SIb3DQEBDAUAA4IBgQCOoGdXjP8Sif0h3ZvoDsIVfrJvQUdP9pZExRJGGj/Te6ML
# XyqHNc/G2WklNv+BC4ryNJ4zp0oneHOub55ejrYPfhgP9oFWS8oOPJbrp3rAtINa
# OeKRC88WUYZSKYAxSX6kubEjnQD6cmazXjxVN6P8+6q9smeqF3xI+++c+ekMw3Xv
# 4EWRYXqN0/srfFp1WpoluwSQcZMqj9haSX0bgO6AAWH2AnVJMfXTIKz/0FQ/RW0y
# Ed5QYQqYInhA7IUz9dBziNpJlwONeF5j7xzgfYDY63WU6WrgJnGjYkQCOOsUx74j
# gUiMRe9zV48GS8Vxz22c/TQs0x4/1GmOSJvUHEk3GseBmB3v+yEhh/D6zWOfYP4X
# D/9b91CxmugTuepuaJSeRg+qUm3KarKsOCUF/CLqUrNh/JwKrWD1cghRaYLvMucs
# ScksHHe7ZDvb2OtvxWXjPk1d1NKvEwFJSS6hIVIbug9x28AJqOoP5Pn9wZvJSzvW
# uQJGCLscdGyKefdCo30wggYaMIIEAqADAgECAhBiHW0MUgGeO5B5FSCJIRwKMA0G
# CSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290
# IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFQxCzAJBgNVBAYT
# AkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28g
# UHVibGljIENvZGUgU2lnbmluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEBAQUAA4IB
# jwAwggGKAoIBgQCbK51T+jU/jmAGQ2rAz/V/9shTUxjIztNsfvxYB5UXeWUzCxEe
# AEZGbEN4QMgCsJLZUKhWThj/yPqy0iSZhXkZ6Pg2A2NVDgFigOMYzB2OKhdqfWGV
# oYW3haT29PSTahYkwmMv0b/83nbeECbiMXhSOtbam+/36F09fy1tsB8je/RV0mIk
# 8XL/tfCK6cPuYHE215wzrK0h1SWHTxPbPuYkRdkP05ZwmRmTnAO5/arnY83jeNzh
# P06ShdnRqtZlV59+8yv+KIhE5ILMqgOZYAENHNX9SJDm+qxp4VqpB3MV/h53yl41
# aHU5pledi9lCBbH9JeIkNFICiVHNkRmq4TpxtwfvjsUedyz8rNyfQJy/aOs5b4s+
# ac7IH60B+Ja7TVM+EKv1WuTGwcLmoU3FpOFMbmPj8pz44MPZ1f9+YEQIQty/NQd/
# 2yGgW+ufflcZ/ZE9o1M7a5Jnqf2i2/uMSWymR8r2oQBMdlyh2n5HirY4jKnFH/9g
# Rvd+QOfdRrJZb1sCAwEAAaOCAWQwggFgMB8GA1UdIwQYMBaAFDLrkpr/NZZILyhA
# QnAgNpFcF4XmMB0GA1UdDgQWBBQPKssghyi47G9IritUpimqF6TNDDAOBgNVHQ8B
# Af8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcD
# AzAbBgNVHSAEFDASMAYGBFUdIAAwCAYGZ4EMAQQBMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Um9vdFI0Ni5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ1Jvb3RSNDYu
# cDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQwFAAOCAgEABv+C4XdjNm57oRUgmxP/BP6YdURhw1aVcdGRP4Wh60BAscjW
# 4HL9hcpkOTz5jUug2oeunbYAowbFC2AKK+cMcXIBD0ZdOaWTsyNyBBsMLHqafvIh
# rCymlaS98+QpoBCyKppP0OcxYEdU0hpsaqBBIZOtBajjcw5+w/KeFvPYfLF/ldYp
# mlG+vd0xqlqd099iChnyIMvY5HexjO2AmtsbpVn0OhNcWbWDRF/3sBp6fWXhz7Dc
# ML4iTAWS+MVXeNLj1lJziVKEoroGs9Mlizg0bUMbOalOhOfCipnx8CaLZeVme5yE
# Lg09Jlo8BMe80jO37PU8ejfkP9/uPak7VLwELKxAMcJszkyeiaerlphwoKx1uHRz
# NyE6bxuSKcutisqmKL5OTunAvtONEoteSiabkPVSZ2z76mKnzAfZxCl/3dq3dUNw
# 4rg3sTCggkHSRqTqlLMS7gjrhTqBmzu1L90Y1KWN/Y5JKdGvspbOrTfOXyXvmPL6
# E52z1NZJ6ctuMFBQZH3pwWvqURR8AgQdULUvrxjUYbHHj95Ejza63zdrEcxWLDX6
# xWls/GDnVNueKjWUH3fTv1Y8Wdho698YADR7TNx8X8z2Bev6SivBBOHY+uqiirZt
# g0y9ShQoPzmCcn63Syatatvx157YK9hlcPmVoa1oDE5/L9Uo2bC5a4CH2Rwwggau
# MIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJ
# BgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5k
# aWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAe
# Fw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcw
# FQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3Rl
# ZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9Ge
# TKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0
# hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZl
# jZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAsh
# aG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVY
# TXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1
# biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCir
# c0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+
# DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA
# +bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42Pg
# puE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzS
# M7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU
# uhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6
# mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsG
# AQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29t
# MEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNl
# cnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3Js
# My5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAE
# GTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1Z
# jsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8d
# B+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVp
# P0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp8
# 76i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2
# nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3
# ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQ
# txMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc
# 4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+Y
# AN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZ
# vAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQr
# H4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIGwDCCBKigAwIBAgIQDE1p
# ckuU+jwqSj0pB4A9WjANBgkqhkiG9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4XDTIyMDkyMTAwMDAw
# MFoXDTMzMTEyMTIzNTk1OVowRjELMAkGA1UEBhMCVVMxETAPBgNVBAoTCERpZ2lD
# ZXJ0MSQwIgYDVQQDExtEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMiAtIDIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDP7KUmOsap8mu7jcENmtuh6BSFdDMa
# JqzQHFUeHjZtvJJVDGH0nQl3PRWWCC9rZKT9BoMW15GSOBwxApb7crGXOlWvM+xh
# iummKNuQY1y9iVPgOi2Mh0KuJqTku3h4uXoW4VbGwLpkU7sqFudQSLuIaQyIxvG+
# 4C99O7HKU41Agx7ny3JJKB5MgB6FVueF7fJhvKo6B332q27lZt3iXPUv7Y3UTZWE
# aOOAy2p50dIQkUYp6z4m8rSMzUy5Zsi7qlA4DeWMlF0ZWr/1e0BubxaompyVR4aF
# eT4MXmaMGgokvpyq0py2909ueMQoP6McD1AGN7oI2TWmtR7aeFgdOej4TJEQln5N
# 4d3CraV++C0bH+wrRhijGfY59/XBT3EuiQMRoku7mL/6T+R7Nu8GRORV/zbq5Xwx
# 5/PCUsTmFntafqUlc9vAapkhLWPlWfVNL5AfJ7fSqxTlOGaHUQhr+1NDOdBk+lbP
# 4PQK5hRtZHi7mP2Uw3Mh8y/CLiDXgazT8QfU4b3ZXUtuMZQpi+ZBpGWUwFjl5S4p
# kKa3YWT62SBsGFFguqaBDwklU/G/O+mrBw5qBzliGcnWhX8T2Y15z2LF7OF7ucxn
# EweawXjtxojIsG4yeccLWYONxu71LHx7jstkifGxxLjnU15fVdJ9GSlZA076XepF
# cxyEftfO4tQ6dwIDAQABo4IBizCCAYcwDgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB
# /wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwIAYDVR0gBBkwFzAIBgZngQwB
# BAIwCwYJYIZIAYb9bAcBMB8GA1UdIwQYMBaAFLoW2W1NhS9zKXaaL3WMaiCPnshv
# MB0GA1UdDgQWBBRiit7QYfyPMRTtlwvNPSqUFN9SnDBaBgNVHR8EUzBRME+gTaBL
# hklodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0
# MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3JsMIGQBggrBgEFBQcBAQSBgzCBgDAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFgGCCsGAQUFBzAC
# hkxodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRS
# U0E0MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3J0MA0GCSqGSIb3DQEBCwUAA4IC
# AQBVqioa80bzeFc3MPx140/WhSPx/PmVOZsl5vdyipjDd9Rk/BX7NsJJUSx4iGNV
# CUY5APxp1MqbKfujP8DJAJsTHbCYidx48s18hc1Tna9i4mFmoxQqRYdKmEIrUPwb
# tZ4IMAn65C3XCYl5+QnmiM59G7hqopvBU2AJ6KO4ndetHxy47JhB8PYOgPvk/9+d
# EKfrALpfSo8aOlK06r8JSRU1NlmaD1TSsht/fl4JrXZUinRtytIFZyt26/+YsiaV
# OBmIRBTlClmia+ciPkQh0j8cwJvtfEiy2JIMkU88ZpSvXQJT657inuTTH4YBZJwA
# wuladHUNPeF5iL8cAZfJGSOA1zZaX5YWsWMMxkZAO85dNdRZPkOaGK7DycvD+5sT
# X2q1x+DzBcNZ3ydiK95ByVO5/zQQZ/YmMph7/lxClIGUgp2sCovGSxVK05iQRWAz
# gOAj3vgDpPZFR+XOuANCR+hBNnF3rf2i6Jd0Ti7aHh2MWsgemtXC8MYiqE+bvdgc
# mlHEL5r2X6cnl7qWLoVXwGDneFZ/au/ClZpLEQLIgpzJGgV8unG1TnqZbPTontRa
# mMifv427GFxD9dAq6OJi7ngE273R+1sKqHB+8JeEeOMIA11HLGOoJTiXAdI/Otrl
# 5fbmm9x+LMz/F0xNAKLY1gEOuIvu5uByVYksJxlh9ncBjDGCBb4wggW6AgEBMGgw
# VDELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDErMCkGA1UE
# AxMiU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIFIzNgIQIONvbD8YGoFG
# +d6T33yX5DANBglghkgBZQMEAgEFAKCBhDAYBgorBgEEAYI3AgEMMQowCKACgACh
# AoAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAM
# BgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCBdCXb7DzHGyapFSqbztB9+Y8C/
# ZLShhVpe3/d3G6ZvrjANBgkqhkiG9w0BAQEFAASCAYAmM+k33VRs+RlOF4iWZwa/
# TxntHbktZsTC39N3RRG/sES/kaP0+or72dBRyfP+ALfkVrxNRE8KDQoBLzYN8m2w
# Cc+fRCiHVqTZjT8Lz8oPNzHZl8KCOoJfA2fDKE6fWQffilcZP8hwA/sExLXb+U2t
# c8Jy1AP29+agFSczlaFfyLw48TFxd9v7+YMAKJrDXZS3azFRO5sHOIa76sq5yiYs
# 3ihf7LRAO3RD+pXng1TcFVOJlPGoZhdj/ZvSbAPmOVNFa8OrJqYgLuwfwbiyEI65
# OcplImjH56dJeQnueJ1+ncvYINYtxR+5G2GI3eLp9od6ARoo+Y7B8+IRxQSuFqXE
# vl0+7Ohjg+3hWpRtR8ccRkX1LshNED4Ci1UHOfHMm/AsxGIn72xSesgq0T60wip5
# GzWjtO6Zmsjiya2Jhh9Jv2GjS7vIRwYL919CRkSpcrwy7HwMgEXiqqaLyCBttsyD
# KjZe4vLQjq0/xhFz9856ziJZdtAG/VhqMPL70XMotHuhggMgMIIDHAYJKoZIhvcN
# AQkGMYIDDTCCAwkCAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
# cnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
# SEEyNTYgVGltZVN0YW1waW5nIENBAhAMTWlyS5T6PCpKPSkHgD1aMA0GCWCGSAFl
# AwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUx
# DxcNMjMwNDAyMTUyODI3WjAvBgkqhkiG9w0BCQQxIgQgQO1kydBuwMOBG/iwhFTq
# 5VhupFnLGspwtHcptRKmWaswDQYJKoZIhvcNAQEBBQAEggIAigFUAm7iCgnE47pB
# U3W7Sj56+ffvIt8Ce+tMHnFrQK2z3xdXLOjdGI1vX9cBOUs0LuM5GMKcf8WQ+lUZ
# l1Ix9TlV09k3ByOl6ubuhnSMSuC2beI8xRDC/YNtRniiEIGa8+ah3EBZeXLxqwym
# aYOEsVN46G/GN/WkkNcBAbJFVkb1kEy5kccRnlGaIKoAyZrfTqSbH74ZNqHJ1ULZ
# A3Z8ZL+0F7ucTNjb/RPPMkhg6aw0OnD4CjkXqWwbe5SIJG4bk3WQZ7bpO6w3nNnB
# Y7F3CP1Xlc3mvbpOMMARaO62aqibFeMKvW/FcJYZYoiLlxHJ7e1s2YbtQ0yLjb+C
# Zf/3zPyXuWrT1Vy3lIeslYUhxalHeNG60hcFfhVG+C6PrK8b+2mDJVFM3cwWLKEo
# MfBPfzhlpcchST28QpQ4zhqnx4hJ+TJw3svnmPdpMEDPGBsQ37M5IRPzhjM+RBN5
# 67wkVq5c0YHzcCOtWV5G/BlpjhRbTz/1GK9aNR2RboQjRX5hCwvUiKrK8xuUzrGX
# L7Ks8v4OFFR3ohyVrYVE5jRKYiUmxHVrlzmU7NHa52evew6OB/CZ9zg/fiHKPx79
# aPUX7Vk0+wAIsGF+S3wImiqDzmFZD5P0G1IVYe00cWYMOql4OVaTKkJO5g+7fZLZ
# ICKjHrDuq37XZwoBSNBKKTuuECw=
# SIG # End signature block
