function Get-NtapVserverAsupData {
    <#
        .SYNOPSIS
            Collects SVM specific data for ONTAP storage systems using ASUPs.

        .DESCRIPTION
            Uses the node ASUPs to gather as much SVM configuration information as possible for the cluster. This
            data can be used to automate build docs and health checks. It can also be used for historical
            configuration information. If the -Cluster parameter is used, the latest ASUPs of the specified
            AsupSubject type will be used. If older or specified ASUPs are desired, use the Find-NtapClusterAsup
            function and pass those results to this function. The ASUPs will be validated to determine that an
            ASUP from each node in the cluster is present, complete and generated at similar times.

        .EXAMPLE
            PS> Get-NtapVserverAsupData -Cluster 'den-cdot:394e82bf-485c-11e2-86b3-123478563412' | Export-Clixml -Path 'D:\den-cdot_Asup-SVM_raw.xml'

            Collects all SVM information from the cluster name and cluster uuid specified and saves the results to a .xml file for
            further processing.

        .EXAMPLE
            PS> Get-NtapVserverAsupData -Cluster 'den-cdot:394e82bf-485c-11e2-86b3-123478563412' -Name vs0 | Export-Clixml -Path 'D:\den-cdot_Asup-vs0_raw.xml'

            Collects SVM 'vs0' information from the cluster name and cluster uuid specified and saves the results to a .xml file for
            further processing.

        .EXAMPLE
            PS> Get-NtapVserverAsupData -Cluster 'den-cdot:394e82bf-485c-11e2-86b3-123478563412' -AsupSubject user_triggered | Export-Clixml -Path 'D:\den-cdot_Asup-SVM_raw.xml'

            Collects all SVM information from the cluster name and cluster uuid specified, using the latest
            user_triggered ASUP set, and saves the results to a .xml file for further processing.

        .EXAMPLE
            PS> Get-NtapvserverAsupData -Cluster 'den-cdot:394e82bf-485c-11e2-86b3-123478563412' | Format-NtapClusterData | Out-NtapDocument -WordFile 'D:\Output\den-cdot_Asup-SVM.docx' -ExcelFile 'D:\Output\den-cdot_Asup-SVM.xlsx'

            Collects all SVM information from the cluster name and cluster uuid specified, formats the data and creates
            the specified Word and Excel documents.

        .EXAMPLE
            PS> Get-NtapVserverAsupData -AsupId 2015012500120554!2015012423440127 | Export-Clixml -Path 'D:\Output\den-cdot-Asup-SVM_raw.xml'

            Collects all SVM information using specific ASUPIDs.

        .EXAMPLE
            PS> Find-NtapClusterAsup -Cluster 'den-cdot:394e82bf-485c-11e2-86b3-123478563412' -StartDate 10-15-2014 -EndDate 11-15-2014 | Get-NtapClusterAsupData | Export-Clixml -Path 'D:\Output\den-cdot_Asup-SVM_raw.xml'

            Finds the first validated weekly ASUP set for the specified cluster by searching between the dates
            specified, performs the data collection and saves the results to a .xml file for further processing.

        .LINK
            https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs

        .LINK
            Format-NtapClusterData

        .INPUTS
            System.Xml.XmlElement

        .OUTPUTS
            System.Management.Automation.PSCustomObject

        .NOTES
            AUTHOR : Jason Cole
            REQUIRES : Internal connection to NetApp
    #>

    [CmdletBinding(DefaultParameterSetName = 'ByClusterInfo')]
    [OutputType([System.Management.Automation.PSCustomObject])]
    param (
        # Searches the ASUP database for the specified ClusterName and ClusterIdentifier
        # (e.g., den-cdot:394e82bf-485c-11e2-86b3-123478563412).
        [Parameter(Mandatory, Position = 0, ParameterSetName = 'ByClusterInfo', ValueFromPipeline)]
        [ValidateScript({
                if ($_ -notmatch '^\s*[^!:]+[:][^!:]+\s*$') {
                    throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]"$_ is not a valid format for the Cluster parameter (e.g., ClusterName:ClusterIdentifier).", 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $_))
                }

                $true
            })]
        [string]$Cluster,

        # The ASUPID(s) to search for. An ASUPID from each node in the cluster is required. Each node's ASUPID must be
        # separated by an exclamation point (e.g., 2015012500120554!2015012423440127). Sets of ASUPIDs from multiple
        # clusters must be comma separated (e.g., 2015012500120554!2015012423440127, 2015062302150043!2015062302150041).
        [Parameter(Mandatory, Position = 0, ParameterSetName = 'ByAsupId')]
        [ValidateScript({
                if ($_ -notmatch '^\s*\d+(?:\s*[!]\s*\d+)*$') {
                    throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ([System.ArgumentException]"$_ is not a valid format for the AsupId parameter (e.g., 'ClusterNode1_AsupId!ClusterNode2_AsupId') or (single node cluster): 'ClusterNode1_AsupId'.", 'ArgumentException', [System.Management.Automation.ErrorCategory]::InvalidArgument, $_))
                }

                $true
            })]
        [string]$AsupId,

        # An ASUP object passed from the Find-NtapClusterAsup function. This is used when using older ASUPs to
        # generate the data.
        [Parameter(Mandatory, Position = 0, ParameterSetName = 'ByAsupObject', ValueFromPipeline)]
        [ValidateNotNullOrEmpty()]
        [System.Xml.XmlElement[]]$AsupObject,

        # A list of data SVMs to gather the data for. If blank, the information for all data SVMs will
        # be returned.
        [Parameter()]
        [ValidateNotNullOrEmpty()]
        [Alias('VserverName', 'SvmName')]
        [string[]]$Name,

        # The type of ASUP to search for (weekly or user_triggered).
        [Parameter(ParameterSetName = 'ByClusterInfo')]
        [ValidateSet('weekly', 'user_triggered')]
        [string]$AsupSubject = 'weekly',

        # Forces the use of an invalid ASUP (i.e., ASUP content missing or mismatched generated dates between
        # the nodes in the cluster).
        [Parameter()]
        [switch]$Force
    )

    begin {
        if ($PSCmdlet.MyInvocation.PipelinePosition -eq 1) {
            try {
                Invoke-SsoLogin -ErrorAction Stop
            }
            catch {
                $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($_.Exception.Message, 'InvalidResult', [System.Management.Automation.ErrorCategory]::InvalidResult, 'ASUP REST Server')))
            }

            Write-Verbose -Message 'Validating connectivity to ASUP REST server'

            if (-not (Test-AsupRestConnectivity)) {
                $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('Unable to contact NetApp ASUP REST server. Please ensure you have an internal connection to NetApp.', 'ResourceUnavailable', [System.Management.Automation.ErrorCategory]::ResourceUnavailable, 'ASUP REST Server')))
            }
        }
    }

    process {
        Write-Verbose -Message "Initializing function: $( $PSCmdlet.MyInvocation.MyCommand.Name ) ($( $PSCmdlet.MyInvocation.MyCommand.ModuleName ) v$( (Get-NtapDocsVersion -AsVersion).ToString() ))"

        switch -Exact ($PSCmdlet.ParameterSetName) {
            'ByClusterInfo' {
                $asupSet = Find-NtapClusterAsup -Cluster $Cluster -AsupSubject $AsupSubject -ValidateAsupSystemType Vserver
            }
            'ByAsupId' {
                $asupSet = Find-NtapClusterAsup -AsupId $AsupId -ValidateAsupSystemType Vserver
            }
            'ByAsupObject' {
                if ($AsupObject) {
                    if (($AsupObject | Where-Object { $_.sys_operating_mode -ne 'Cluster-Mode' } | Measure-Object).Count -gt 0) {
                        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ('One or more ASUPS in the ASUP set is from a 7-Mode system. Please use Get-NtapFilerAsupData for 7-Mode ASUPs.', 'InvalidType', [System.Management.Automation.ErrorCategory]::InvalidType, $AsupObject)))
                    }

                    $asupSet = $AsupObject
                }
            }
        }

        if ($asupSet) {
            if (($asupSet | Where-Object { $_.is_valid -eq $false } | Measure-Object).Count -eq 0 -or $Force.IsPresent) {
                try {
                    $functionStartTime = [datetime]::Now

                    $asupData = Get-VserverAsupData -AsupSet $asupSet -VserverContext $Name

                    $asupData

                    foreach ($vsData in $asupData) {
                        $statsParams = @{
                            FunctionPSCallStack = Get-PSCallStack
                            TargetDetails       = [pscustomobject]@{
                                SystemUid     = [string]::Join('_', $vsData.Cluster.Info.ClusterIdentifier, $vsData.Vserver.Info.Uuid)
                                SystemType    = $vsData.pstypenames[0]
                                SystemVersion = ([regex]::Split(([regex]::Split($vsData.System.VersionInfo.Version, 'Release\s(?:\w+[__])?')[1]), '\s+'))[0].Replace(':', '')
                            }
                            ElapsedTicks        = ([datetime]::Now - $functionStartTime).Ticks
                            ExcludeParameter    = 'Cluster', 'AsupId', 'AsupObject', 'Name'
                        }

                        Write-Statistics @statsParams
                    }
                }
                catch {
                    Write-Error -ErrorRecord $_
                }
            }
        }
    }
}

function Get-VserverAsupData {
    [CmdletBinding()]
    param (
        [Parameter(Mandatory)]
        [ValidateNotNullOrEmpty()]
        [System.Xml.XmlElement[]]$AsupSet,

        [Parameter()]
        [string[]]$VserverContext
    )

    $systemAsupData = [pscustomobject]@{ }
    $systemAsupData.pstypenames.Insert(0, 'NetAppDocs.NcController.Asup.Data')

    $NetAppDocsError = @( )

    Write-Verbose -Message '- Capturing module version -'
    $systemAsupData | Add-Member -MemberType NoteProperty -Name NetAppDocsVersion (Get-NtapDocsVersion -AsVersion)

    Write-Verbose -Message '- Capturing PowerShell version -'
    $systemAsupData | Add-Member -MemberType NoteProperty -Name PowerShellVersion -Value $PSVersionTable.PSVersion

    Write-Verbose -Message '- Capturing ASUP MetaData -'
    $systemAsupData | Add-Member -MemberType NoteProperty -Name SourceAsup -Value (ConvertTo-AsupObject -AsupSet $AsupSet)

    try {
        try {
            Write-Verbose -Message '- Gathering list of SVMs -'

            $allVservers = Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VSERVER-INFO.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VSERV' -ReplaceString '^Vserv(?!er)' -ErrorVariable +NetAppDocsError

            if ($VserverContext) {
                $diffObject = Compare-Object -ReferenceObject $allVservers.Vserver -DifferenceObject $VserverContext -IncludeEqual

                ($diffObject | Where-Object { $_.SideIndicator -eq '=>' }).InputObject | Where-Object { $_ } | ForEach-Object {
                    Show-ErrorMessage -Message "Vserver does not exist in cluster: $_"
                }

                $vservers = $allVservers | Where-Object { $_.Vserver -cin ($diffObject | Where-Object { $_.SideIndicator -eq '==' }).InputObject }
            }
            else {
                $vservers = $allVservers
            }
        }
        catch {
            $Global:Error.RemoveAt(0)

            Write-Error -ErrorRecord (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("Unable to gather SVMs from system. Reason: $( $_.Exception.GetBaseException().Message )", $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $ClusterName))

            continue
        }

        if (-not $vservers) {
            continue
        }

        #region System/Node Information

        Write-Verbose -Message '- Gathering basic system information -'

        try {
            Write-Verbose -Message 'Parsing SYSTEM'
            $system = $systemAsupData.SourceAsup | Get-AsupObject -ObjectName 'SYSTEM' -ReplaceString '^Sys(?!temId)' -ErrorAction Stop | Select-Object -Property *, @{
                Name       = 'SystemName'
                Expression = { $_.Hostname }
            }

            Write-Verbose -Message 'Parsing CLUSTER'
            $clusterObject = $systemAsupData.SourceAsup | Get-AsupObject -ObjectName 'CLUSTER' -ReplaceString '^Cluster(?!Name|Identifier)' -ErrorAction Stop | Where-Object { -not [string]::IsNullOrWhiteSpace($_.ClusterName) -and -not [string]::IsNullOrWhiteSpace($_.ClusterIdentifier) } | Select-Object -First 1

            if (-not $clusterObject) {
                throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("ASUP 'CLUSTER' object incomplete", 'ReadError', [System.Management.Automation.ErrorCategory]::ReadError, $AsupSet[0].cluster_name))
            }

            Write-Verbose -Message 'Parsing CLUSTER-INFO.XML'
            $clusterInfoXml = Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CLUSTER-INFO.XML' -InnerProperty 'CLUSTER_INFO' -ErrorVariable +NetAppDocsError | Where-Object { $_.IsSelf }

            if (-not $clusterInfoXml) {
                throw (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("ASUP 'CLUSTER-INFO.XML' section incomplete", 'ReadError', [System.Management.Automation.ErrorCategory]::ReadError, $AsupSet[0].cluster_name))
            }

            Write-Verbose -Message 'Parsing CLUSTER-RING.XML'
            $clusterRingXml = Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CLUSTER-RING.XML' -InnerProperty 'T_RDB' -ReplaceString '^Rdb' -ErrorVariable +NetAppDocsError | Group-Object -Property Node, UnitName | ForEach-Object {
                $_.Group | Sort-Object -Property DbTrnxs | Select-Object -Last 1
            }

            Write-Verbose -Message '- Identifying Cluster Ring Master Node -'
            $masterNodeName = $(
                $systemAsupData.SourceAsup | ForEach-Object {
                    $srcAsup = $_

                    ($clusterRingXml | Where-Object { $_.Node -ceq $srcAsup.Systemname -and $_.Unitname -eq 'mgmt' -and ($_.Local -ceq $_.Node -and $_.Node -ceq $_.Master) }).Master
                } | Select-Object -First 1
            )

            Write-Verbose -Message 'Parsing MOTHERBOARD'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name Motherboard -Value $(
                foreach ($srcAsup in $systemAsupData.SourceAsup) {
                    Get-AsupObject -AsupObject $srcAsup -ObjectName 'MOTHERBOARD' -ReplaceString '^Mb' -ErrorAction Stop | Select-Object -Property *, @{
                        Name       = 'SystemName'
                        Expression = { $srcAsup.SystemName }
                    }
                }
            )
        }
        catch {
            if ($_.CategoryInfo.Category -eq 'ReadError') {
                $Global:Error.RemoveAt(0)

                $msg = New-Object -TypeName System.Text.StringBuilder

                [void]$msg.AppendLine('An unexpected error occurred. Processing halted for this system.')
                [void]$msg.AppendLine('---------------------------------------------------------------------------------------------')
                [void]$msg.AppendLine('This error is not expected, but we do know about it. Unfortunately, this is an issue with the')
                [void]$msg.AppendLine('ASUP data warehouse and (usually) the ASUP not being fully ingested. It is possible that')
                [void]$msg.AppendLine('waiting up to a few hours or so may be enough time for the ASUP DW to finishing ingesting the')
                [void]$msg.AppendLine('data. However, in some cases, the only workaround is to try a different set of ASUPs.')
                [void]$msg.AppendLine('---------------------------------------------------------------------------------------------')
                [void]$msg.AppendLine("Target: $( $_.CategoryInfo.TargetName ). Reason: $( $_.Exception.GetBaseException().Message )")
                [void]$msg.AppendLine("Location: $( $_.ScriptStackTrace )")
                [void]$msg.AppendLine()

                $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $_.CategoryInfo.TargetName)))
            }
            else {
                $NetAppDocsError += $_
            }
        }

        $systemAsupData | Add-Member -MemberType NoteProperty -Name System -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing EFFECTIVE-CLUSTER-VERSION.XML'
        $clusterVersionXml = Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'EFFECTIVE-CLUSTER-VERSION.XML' -MasterNodeName $masterNodeName -InnerProperty 'effective_cluster_version' -ReplaceString '^Cluster' -ErrorVariable +NetAppDocsError

        $systemAsupData.System | Add-Member -MemberType NoteProperty -Name VersionInfo -Value (
            [pscustomobject]@{
                Version       = $clusterVersionXml.VersionString
                VersionTupleV = [version]::Parse([string]::Format('{0}.{1}.{2}', $clusterVersionXml.Generation, $clusterVersionXml.Major, $clusterVersionXml.Minor))
            }
        )

        $systemAsupData.System | Add-Member -MemberType NoteProperty -Name OntapiVersion -Value ([version]::Parse([string]::Format('{0}.{1}', $clusterVersionXml.OntapiMajor, $clusterVersionXml.OntapiMinor)))

        Write-Verbose -Message '- Gathering node information-'

        $systemAsupData | Add-Member -MemberType NoteProperty -Name Node -Value ($system | Select-Object -Property * -ExcludeProperty Options)

        foreach ($node in ($systemAsupData.Node | Where-Object { -not $_.ClusterMemberId })) {
            $node | Add-Member -MemberType NoteProperty -Name ClusterMemberId -Value ($clusterInfoXml | Where-Object { $_.Node -ceq $node.HostName }).NodeUuid -Force
        }

        <#Write-Verbose -Message 'Parsing MOTHERBOARD-INFO.XML'
        $motherboardInfoXml = $systemAsupData.SourceAsup | ForEach-Object {
            [pscustomobject]@{
                NodeName = $_.SystemName
                Object   = Get-AsupSection -AsupObject $_ -SectionName 'MOTHERBOARD-INFO.XML' -InnerProperty 'T_MBINFO' -ErrorVariable +NetAppDocsError
            }
        }

        Write-Verbose -Message 'Parsing SYSTEM-INFO.XML'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Node -Value @(
            $systemAsupData.SourceAsup | ForEach-Object {
                $systemName = $_.SystemName
                $motherboardInfoXmlObject = ($motherboardInfoXml | Where-Object { $systemName -eq $_.NodeName }).Object

                Get-AsupSection -AsupObject $_ -SectionName 'SYSTEM-INFO.XML' -InnerProperty 'T_SYSINFO' -ErrorVariable +NetAppDocsError | Select-Object -ExcludeProperty OntapVersion -Property *, @{
                    Name       = 'Hostname'
                    Expression = { $_.SystemHostname }
                }, @{
                    Name       = 'SerialNo'
                    Expression = { $_.SystemSerialNumber }
                }, @{
                    Name       = 'PartnerSystemId'
                    Expression = { $_.PartnerId }
                }, @{
                    Name       = 'FwRev'
                    Expression = { $motherboardInfoXmlObject.BiosVersion }
                }, @{
                    Name       = 'Model'
                    Expression = { $motherboardInfoXmlObject.ModelName }
                }, @{
                    Name       = 'OntapVersion'
                    Expression = { ([regex]::Split(([regex]::Split($_.OntapVersion, 'Release\s(?:\w+[__])?')[1]), '\s+'))[0].Replace(':', '') }
                }
            }
        )#>

        #endregion System/Node Information

        #region Cluster Information

        Write-Verbose -Message '- Gathering Cluster Info -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Cluster -Value ([pscustomobject]@{ })

        $systemAsupData.Cluster | Add-Member -MemberType NoteProperty -Name Info -Value ($clusterObject | Select-Object -ExcludeProperty Version*, OntapVersion* -Property *)

        #endregion Cluster Information

        #region Vserver Configuration

        ##Unused Section: HEALTH-VSERVER.XML - Not currently needed for build docs (Vserver.Health)
        ##Unused Section: VSERVER-WEB-SERVICES.XML - Not currently needed for build docs (Vserver.WebService)

        Write-Verbose -Message '- Gathing Vserver information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Vserver -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing VSERVER-INFO.XML'
        $systemAsupData.Vserver | Add-Member -MemberType NoteProperty -Name Info -Value $vservers

        #endregion Vserver Configuration

        #region NDMP

        Write-Verbose -Message '- Gathering NDMP configuration -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Ndmp -Value ([pscustomobject]@{ })

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Parsing NDMP-NODE-SCOPE-MODE-STATUS'
            $ndmpNodeScopeMode = Get-AsupContent -AsupObject $systemAsupData.SourceAsup -SectionName 'NDMP-NODE-SCOPE-MODE-STATUS' -ErrorVariable +NetAppDocsError | Select-Object -First 1

            if ($ndmpNodeScopeMode -like '*disabled*') {
                Write-Verbose -Message 'Parsing NDMP-VSERVER-PROPERTIES.XML'
                $systemAsupData.Ndmp | Add-Member -MemberType NoteProperty -Name VserverServices -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NDMP-VSERVER-PROPERTIES.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_NDMP_VSERV' -ErrorVariable +NetAppDocsError)
            }
        }

        #endregion NDMP

        #region FlexCache Data

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Gathering FlexCache data -'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name Flexcache -Value ([pscustomobject]@{ })

            if ($systemAsupData.System.OntapiVersion -lt [version]::Parse('1.30')) {
                Write-Verbose -Message 'Parsing FLEXCACHE-VOLUME.XML'
                $systemAsupData.Flexcache | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FLEXCACHE-VOLUME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FLEXCACHE_REPORT' -ReplaceString '^Fc' -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message 'Parsing FLEXCACHE-CACHE-POLICY.XML'
                $systemAsupData.Flexcache | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FLEXCACHE-CACHE-POLICY.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FLEXCACHE_CACHE_POLICY_MGMT' -ReplaceString '^Fc' -ErrorVariable +NetAppDocsError)
            }

            if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.150')) {
                Write-Verbose -Message 'Parsing FLEXCACHE-RELATIONS.XML'
                $systemAsupData.Flexcache | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FLEXCACHE-RELATIONS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FLEXCACHE' -ErrorVariable +NetAppDocsError)
            }
        }

        #endregion FlexCache Data

        #region Export Policy Configuration

        ##Unused Section: EXPORTS-ACCESS-CACHE-CONFIG.XML - (1.100+) Not needed for build docs (Export.AccessCacheConfig)

        Write-Verbose -Message '- Gathering export policy information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Export -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing EXPORT-RULESET-UI-TABLE.XML'
        $systemAsupData.Export | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'EXPORT-RULESET-UI-TABLE.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_EXPORT_POL' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing EXPORT-RULE-TABLE.XML'
        $systemAsupData.Export | Add-Member -MemberType NoteProperty -Name Rule -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'EXPORT-RULE-TABLE.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_EXPORT_POL_RULE' -ReplaceString '^Exportpolrule' -ErrorVariable +NetAppDocsError)

        #endregion Export Policy Configuration

        #region Fpolicy Configuration

        ##Unused Section: FPOLICY-POLICY-STATUS.XML - (1.20+) Not needed for build docs (Fpolicy.Status)

        Write-Verbose -Message '- Gathering Fpolicy configuration -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Fpolicy -Value ([pscustomobject]@{ })

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Parsing FPOLICY-SERVER-STATUS.XML'
            $systemAsupData.Fpolicy | Add-Member -MemberType NoteProperty -Name ServerStatus -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FPOLICY-SERVER-STATUS.XML' -InnerProperty 'T_FPOLICY_SERVER_STATUS' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing FPOLICY-EXT-ENGINE.XML'
            $systemAsupData.Fpolicy | Add-Member -MemberType NoteProperty -Name ExternalEngine -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FPOLICY-EXT-ENGINE.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FPOLICY_EXT_ENGINE' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing FPOLICY-POLICY.XML'
            $systemAsupData.Fpolicy | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FPOLICY-POLICY.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FPOLICY_POLICY' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing FPOLICY-EVENT.XML'
            $systemAsupData.Fpolicy | Add-Member -MemberType NoteProperty -Name Event -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FPOLICY-EVENT.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FPOLICY_EVENT' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing FPOLICY-SCOPE.XML'
            $systemAsupData.Fpolicy | Add-Member -MemberType NoteProperty -Name Scope -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FPOLICY-SCOPE.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FPOLICY_SCOPE' -ErrorVariable +NetAppDocsError)
        }

        #endregion Fpolicy Configuration

        #region SIS Configuration

        Write-Verbose -Message '- Gathering SIS configuration -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Sis -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing SIS-STATUS-L.XML'
        $systemAsupData.Sis | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SIS-STATUS-L.XML' -InnerProperty 'T_SIS' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing SIS-POLICY-UI.XML'
        $systemAsupData.Sis | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SIS-POLICY-UI.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_SIS_POLICY_UI' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing SIS-STAT-L.XML'
        $sisStatL = Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SIS-STAT-L.XML' -InnerProperty 'T_SIS_STAT' -ErrorVariable +NetAppDocsError

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.150')) {
            Write-Verbose -Message 'Parsing SIS-CONFIG.XML'
            $systemAsupData.Sis | Add-Member -MemberType NoteProperty -Name Config -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SIS-CONFIG.XML' -InnerProperty 'T_SIS_CONFIG' -ErrorVariable +NetAppDocsError)
        }

        #endregion SIS Configuration

        #region Volume Configuration

        Write-Verbose -Message '- Gathering volume information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Vol -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing VOLUME.XML'
        $systemAsupData.Vol | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VOLUME.XML' -InnerProperty 'T_VOLUME' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing VOLUME'
        $volObjs = $systemAsupData.SourceAsup | Get-AsupObject -ObjectName 'VOLUME' -ReplaceString '^Vol' -ErrorVariable +NetAppDocsError

        foreach ($vol in ($systemAsupData.Vol.Info | Where-Object { $_ })) {
            $volObj = $volObjs | Where-Object { $_.VserverId -ceq $vol.Vs -and $_.InstanceUuid -eq $vol.IUuid }
            $volSisStatL = $sisStatL | Where-Object { $_.Vs -ceq $vol.Vs -and $_.Path -ceq "/vol/$( $vol.Vol )" }

            if ($volObj) {
                foreach ($prop in ($volObj | Get-Member -Name Asis*, Compress*, SAllocated*, SAvail*, Sused*, TotSaved*, IsOnline, Options)) {
                    $vol | Add-Member -MemberType NoteProperty -Name $prop.Name -Value $volObj.$( $prop.Name )
                }

                $vol | Add-Member -MemberType NoteProperty -Name IsNodeRoot -Value $volObj.IsRoot
                $vol | Add-Member -MemberType NoteProperty -Name LanguageString -Value $volObj.Language
                $vol | Add-Member -MemberType NoteProperty -Name CloneChildCount -Value $volObj.CloneCt
            }

            $vol | Add-Member -MemberType NoteProperty -Name IsSisVolume -Value $(
                if ($sisStatL) {
                    $null -ne $volSisStatL
                }
            )
        }

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Calling Get-AsupVolSpace'
            $systemAsupData.Vol | Add-Member -MemberType NoteProperty -Name Space -Value (Get-AsupVolSpace -AsupObject $systemAsupData.SourceAsup)

            if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.130')) {
                Write-Verbose -Message 'Parsing VOLUME-CLONE-SHOW.XML'
                $systemAsupData.Vol | Add-Member -MemberType NoteProperty -Name CloneVol -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VOLUME-CLONE-SHOW.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VOLUME_CLONE_SHOW' -ErrorVariable +NetAppDocsError)
            }
        }

        #endregion Volume Configuration

        #region Infinivol Repository Configuration

        ##Unused Section: REPOSITORY-INFO.XML - Not needed for build docs (Repository.Info)
        ##Unused Section: REPOSITORY-POLICY-RULES.XML - Not needed for build docs (Repository.PolicyRules)
        ##Unused Section: REPOSITORY-STORAGE-SERVICES.XML - Not needed for build docs (Repository.StorageServices)

        Write-Verbose -Message '- Gathering infinite volume repository information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Repository -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing REPOSITORY-CONSTITUENTS.XML'
        $systemAsupData.Repository | Add-Member -MemberType NoteProperty -Name Constituents -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'REPOSITORY-CONSTITUENTS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_REPOSITORY_CONSTITUENT' -ErrorVariable +NetAppDocsError)

        #endregion Infinivol Repository Configuration

        #region Qtree Configuration

        Write-Verbose -Message '- Gathering qtree information -'

        Write-Verbose -Message 'Parsing QTREE.XML'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Qtree -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'QTREE.XML' -InnerProperty 'T_QTREE' -ErrorVariable +NetAppDocsError)

        #endregion Qtree Configuration

        #region Name Mapping Data

        Write-Verbose -Message '- Gathering name mapping data -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name NameMapping -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing NAMEMAPPINGS-BYNAME.XML'
        $systemAsupData.NameMapping | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NAMEMAPPINGS-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VSERV_NAMEMAPPING' -ReplaceString '^Vservnamemap' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing UNIX-USERS-BYNAME.XML'
        $systemAsupData.NameMapping | Add-Member -MemberType NoteProperty -Name UnixUser -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'UNIX-USERS-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_UNIX_USERS' -ReplaceString '^Unixusers' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing UNIX-GROUPS-BYNAME.XML'
        $systemAsupData.NameMapping | Add-Member -MemberType NoteProperty -Name UnixGroup -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'UNIX-GROUPS-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_UNIX_GROUP' -ReplaceString '^Unixgroup' -ErrorVariable +NetAppDocsError)

        #endregion Name Mapping Data

        #region Vscan Configuration

        ##Unused Section: VSCAN-ACTIVE-SCANNER-POOL.XML - (1.20+) Not needed for build docs (Vscan.ScannerPoolActive)
        ##Unused Section: VSCAN-EVENTS.XML - (1.31+) Not needed for build docs (Vscan.Events)
        ##Unused Section: VSCAN-CONNECTION-STATUS-ALL.XML - (1.31+) Not needed for SVM docs (Vscan.Connection)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.21')) {
            Write-Verbose -Message '- Gathering Vscan information -'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name Vscan -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing VSCAN-STATUS.XML'
            $systemAsupData.Vscan | Add-Member -MemberType NoteProperty -Name Status -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VSCAN-STATUS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VSCAN_STATUS' -ReplaceString '^Vscan' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing VSCAN-ON-ACCESS-POLICY.XML'
            $systemAsupData.Vscan | Add-Member -MemberType NoteProperty -Name OnAccessPolicy -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VSCAN-ON-ACCESS-POLICY.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VSCAN_ON_ACCESS_POLICY' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing VSCAN-SCANNER-POOL.XML'
            $systemAsupData.Vscan | Add-Member -MemberType NoteProperty -Name ScannerPool -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VSCAN-SCANNER-POOL.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VSCAN_SCANNER_POOL' -ErrorVariable +NetAppDocsError)
        }

        #endregion Vscan Configuration

        #region Network Information

        ##Unused Section: IPSPACES.XML - (1.30+) Not needed for SVM build docs (Net.Ipspace)

        Write-Verbose -Message '- Gathering network information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Net -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing CLUSTER_LIF'
        $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Interface -Value ($systemAsupData.SourceAsup | Get-AsupObject -ObjectName 'CLUSTER_LIF' -ReplaceString '^Vif' -ErrorVariable +NetAppDocsError)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message 'Parsing NETWORK-ROUTES.XML'
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Route -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NETWORK-ROUTES.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_ROUTES' -ReplaceString '^Route' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing VS-FAILOVER-GROUPS.XML'
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name FailoverGroup -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VS-FAILOVER-GROUPS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VS_FAILOVER_GROUPS' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing VS-FIREWALL-POLICIES.XML'
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name FirewallPolicy -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VS-FIREWALL-POLICIES.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_vs_firewall_policies' -ErrorVariable +NetAppDocsError)
        }
        else {
            Write-Verbose -Message 'Parsing FAILOVER-GROUPS.XML'
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name FailoverGroup -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FAILOVER-GROUPS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FAILOVER_GROUP' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing NETWORK-ROUTES.XML'
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name RoutingGroupRoute -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NETWORK-ROUTES.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_ROUTES' -ErrorVariable +NetAppDocsError)

            if ($systemAsupData.System.OntapiVersion -lt [version]::Parse('1.30')) {
                Write-Verbose -Message 'Parsing NETWORK-ROUTING-GROUPS.XML'
                $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name RoutingGroup -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NETWORK-ROUTING-GROUPS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_ROUTING_GROUPS' -ErrorVariable +NetAppDocsError)
            }
        }

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.31')) {
            Write-Verbose -Message 'Parsing DDNS.XML'
            $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Ddns -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'DDNS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_DDNS' -ErrorVariable +NetAppDocsError)

            if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.150')) {
                $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name ServicePolicy -Value ([pscustomobject]@{ })

                Write-Verbose -Message 'Parsing NETWORK-SERVICE-POLICY.XML'
                $systemAsupData.Net.ServicePolicy | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NETWORK-SERVICE-POLICY.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_LIF_SERVICE_POLICIES' -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message 'Parsing NETWORK-SERVICE-POLICY-ENTRY.XML'
                $systemAsupData.Net.ServicePolicy | Add-Member -MemberType NoteProperty -Name Entry -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NETWORK-SERVICE-POLICY-ENTRY.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_LIF_SERVICE_POLICY_ENTRIES' -ErrorVariable +NetAppDocsError)
            }
        }

        Write-Verbose -Message 'Parsing DNS.XML'
        $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name Dns -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'DNS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_DNS' -ReplaceString '^Dns' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing HOSTS.XML'
        $systemAsupData.Net | Add-Member -MemberType NoteProperty -Name DnsHost -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'HOSTS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_HOSTS' -ReplaceString '^Hosts' -ErrorVariable +NetAppDocsError)

        #endregion Network Information

        #region FC/FCP Information

        Write-Verbose -Message '- Gathering FCP configuration data -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Fcp -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing FCP.XML'
        $systemAsupData.Fcp | Add-Member -MemberType NoteProperty -Name Service -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FCP.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FCP' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing FCP-INITIATOR.XML'
        $systemAsupData.Fcp | Add-Member -MemberType NoteProperty -Name Initiator -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FCP-INITIATOR.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_FCP_INITIATOR' -ErrorVariable +NetAppDocsError)

        #endregion FC/FCP Information

        #region PortSet Configuration

        Write-Verbose -Message '- Gathering PortSet information -'

        Write-Verbose -Message 'Parsing PORTSET.XML'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name PortSet -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'PORTSET.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_PORTSET' -ReplaceString '^PortSet' -ErrorVariable +NetAppDocsError)

        #endregion PortSet Configuration

        #region iSCSI Information

        ##Unused Section: ISCSI-INTERFACE.XML - Not needed for build docs (data in network-interface section) (Iscsi.Interface.Info)
        ##Unused Section: ISCSI-SECURITY.XML - Not needed for build docs (Iscsi.Interface.Access)

        Write-Verbose -Message '- Gathering iSCSI configuration data -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Iscsi -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing ISCSI.XML'
        $systemAsupData.Iscsi | Add-Member -MemberType NoteProperty -Name Service -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'ISCSI.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_ISCSI' -ReplaceString '^Iscsi' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing ISCSI-SESSION-BRIEF.XML'
        $systemAsupData.Iscsi | Add-Member -MemberType NoteProperty -Name Initiator -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'ISCSI-SESSION-BRIEF.XML' -InnerProperty 'T_ISCSI_SESSION_BRIEF' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing INITIATOR_SESSION'
        $initiatorSession = $systemAsupData.SourceAsup | Get-AsupObject -ObjectName 'INITIATOR_SESSION' -ReplaceString '^Ini' -ErrorVariable +NetAppDocsError

        foreach ($iscsiInitiator in ($systemAsupData.Iscsi.Initiator | Where-Object { $_ })) {
            $iscsiInitiator | Add-Member -MemberType NoteProperty -Name InitiatorGroupList -Value @(
                $initiatorSession | Where-Object { $_.Tsih -eq $iscsiInitiator.Tsih -and $_.Isid -eq $iscsiInitiator.Isid } | ForEach-Object {
                    [regex]::Replace($_.Igid, "^$( [regex]::Escape($iscsiInitiator.Vserver) )", '')
                } | Where-Object { $_ -ne 'null' } | Sort-Object -Unique
            )

            $iscsiInitiator | Add-Member -MemberType NoteProperty -Name TpgroupTag -Value ($systemAsupData.Net.Interface | Where-Object { $_.VserverId -ceq $iscsiInitiator.Vserver -and $_.LogicalInterface -eq $iscsiInitiator.Tpgroup }).CurrentTpgt
        }

        #endregion iSCSI Information

        #region Security Information

        Write-Verbose -Message '- Gathering security information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Security -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing CERTIFICATE.XML'
        $systemAsupData.Security | Add-Member -MemberType NoteProperty -Name Certificate -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CERTIFICATE.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CERT' -ErrorVariable +NetAppDocsError)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.100')) {
            Write-Verbose -Message 'Parsing SSH.XML'
            $systemAsupData.Security | Add-Member -MemberType NoteProperty -Name Ssh -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SSH.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_SSH' -ErrorVariable +NetAppDocsError)
        }

        #endregion Security Information

        #region Security Session Information

        ##Unused Section: SESSION-USER-LIMITS.XML - (1.100+) Not needed for build docs (SecuritySession.UserLimit)
        ##Unused Section: SESSION-VSERVER-LIMITS.XML - (1.100+) Not needed for build docs (SecuritySession.VserverLimit)

        #endregion Security Session Information

        #region Snaplock Information

        ##Unused Section: SNAPLOCK-LOG.XML - (1.100+) Not currently needed for build docs (Snaplock.LogConfig)
        ##Unused Section: SNAPLOCK-LOG-FILE.XML - (1.100+) Not currently needed for build docs (Snaplock.LogFiles)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.100')) {
            Write-Verbose -Message '- Gathering Snaplock information -'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name Snaplock -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing SNAPLOCK-VOLUME.XML'
            $systemAsupData.Snaplock | Add-Member -MemberType NoteProperty -Name Volume -Value (
                Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SNAPLOCK-VOLUME.XML' -InnerProperty 'T_VOLUME_SNAPLOCK' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                    Name       = 'SnaplockComplianceClock'
                    Expression = {
                        if (($compClockTime = Convert-DateTimeString -String $_.CompClockTime) -or ($compClockTime = Convert-DateTimeString -String (($_.CompClockTime -split '\s' | Select-Object -Index 0, 1, 2, 3, 5, 6) -join ' ') -Format 'ddd MMM d HH:mm:ss yyyy zzz')) {
                            ConvertTo-UnixTimestamp -DateTime $compClockTime.ToUniversalTime()
                        }
                    }
                }, @{
                    Name       = 'VolumeExpiryTime'
                    Expression = {
                        if (-not [string]::IsNullOrWhiteSpace($_.ExpiryTime)) {
                            if ($_.ExpiryTime -ne 'none' -and (($expiryTime = Convert-DateTimeString -String $_.ExpiryTime) -or ($expiryTime = Convert-DateTimeString -String $_.ExpiryTime -Format 'ddd MMM d HH:mm:ss GMT yyyy'))) {
                                ConvertTo-UnixTimestamp -DateTime $expiryTime.ToUniversalTime()
                            }
                            else {
                                #if we are here, this means the format of the string wasn't "GMT", so just take the string as is for now
                                $_.ExpiryTime
                            }
                        }
                    }
                } -Unique
            )
        }

        #endregion Snaplock Information

        #region NVMe Information

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.140')) {
            Write-Verbose -Message '- Gathering NVMe information'
            $systemAsupData | Add-Member -MemberType NoteProperty -Name Nvme -Value (
                [pscustomobject]@{
                    Subsystem = [pscustomobject]@{ }
                }
            )

            Write-Verbose -Message 'Parsing NVMF-SERVICE.XML'
            $systemAsupData.Nvme | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NVMF-SERVICE.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_NVMF_SERVICE' -ErrorVariable +NetAppDocsError)

            #Write-Verbose -Message 'Parsing NVMF-FEATURES.XML'
            #$systemAsupData.Nvme | Add-Member -MemberType NoteProperty -Name Feature -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NVMF-FEATURES.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_NVMF_FEATURES' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing NVMF-SUBSYSTEM.XML'
            $systemAsupData.Nvme.Subsystem | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NVMF-SUBSYSTEM.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_NVMF_SUBSYSTEM' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing NVMF-SUBSYSTEM-HOST.XML'
            $systemAsupData.Nvme.Subsystem | Add-Member -MemberType NoteProperty -Name Host -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NVMF-SUBSYSTEM-HOST.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_NVMF_SUBSYSTEM_HOST' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing NVMF-SUBSYSTEM-MAP.XML'
            $systemAsupData.Nvme.Subsystem | Add-Member -MemberType NoteProperty -Name Map -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NVMF-SUBSYSTEM-MAP.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_NVMF_SUBSYSTEM_MAP' -ErrorVariable +NetAppDocsError)
        }

        #endregion NVMe Information

        #region FileService Audit Configuration

        Write-Verbose -Message '- Gathering File Service Audit configuration -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name FileServiceAudit -Value ([pscustomobject]@{ })

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Parsing FILE-SERVICES-AUDIT.XML'
            $systemAsupData.FileServiceAudit | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'FILE-SERVICES-AUDIT.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VSERVER_AUDIT' -ReplaceString '^(?:Vserveraudit|LogFile)' -ErrorVariable +NetAppDocsError)
        }

        #endregion FileService Audit Configuration

        #region Vserver Peer Configuration

        ##Unused Section: VSERVER-PEER-TRANSITION.XML - 8.2+ Not needed for build docs (Vserver.Peer.Transition)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Gathing Vserver Peer information -'
            $systemAsupData.Vserver | Add-Member -MemberType NoteProperty -Name Peer -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Parsing VSERVER-PEER.XML'
            $systemAsupData.Vserver.Peer | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VSERVER-PEER.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VSERVERPEER_RDB' -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Parsing PEER_CLUSTER'
            $peerCluster = $systemAsupData.SourceAsup | Get-AsupObject -ObjectName 'PEER_CLUSTER' -ErrorVariable +NetAppDocsError

            foreach ($vserverPeer in ($systemAsupData.Vserver.Peer.Info | Where-Object { $_ })) {
                if ($vserverPeer.PeerClusterUUID -eq $systemAsupData.Cluster.Info.ClusterIdentifier) {
                    $clusterPeerName = $systemAsupData.Cluster.Info.ClusterName
                }
                else {
                    $clusterPeerName = ($systemAsupData.Cluster.Peer.Info | Where-Object { $_.ClusterUUID -eq $vserverPeer.PeerClusterUUID } | Select-Object -First 1).Cluster

                    if ([string]::IsNullOrWhiteSpace($clusterPeerName) -and $peerCluster) {
                        $clusterPeerName = ($peerCluster | Where-Object { $_.PeerClusterUuid -eq $vserverPeer.PeerClusterUUID } | Select-Object -First 1).PeerClusterName
                    }
                }

                $vserverPeer | Add-Member -MemberType NoteProperty -Name PeerCluster -Value $clusterPeerName -Force
            }
        }

        #endregion Vserver Peer Configuration

        #region Snapshot Information

        Write-Verbose -Message '- Gathering Snapshot information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Snapshot -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing SNAPSHOT.XML'
        $systemAsupData.Snapshot | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SNAPSHOT.XML' -InnerProperty 'T_SNAPSHOT' -ErrorVariable +NetAppDocsError | Sort-Object -Property * -Unique)

        Write-Verbose -Message 'Parsing SNAPSHOT-POLICY.XML'
        $systemAsupData.Snapshot | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SNAPSHOT-POLICY.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_SNAPSHOT_POLICY' -ReplaceString '^Snapshotpolicy' -ErrorVariable +NetAppDocsError)

        #endregion Snapshot Information

        #region Snapmirror Configuration

        Write-Verbose -Message '- Gathering SnapMirror information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Snapmirror -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing SNAPMIRROR.XML'
        $systemAsupData.Snapmirror | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SNAPMIRROR.XML' -InnerProperty 'T_SNAPMIRROR' -ErrorVariable +NetAppDocsError | Group-Object -Property SourcePath, DestinationPath, Type | ForEach-Object { $_.Group | Select-Object -First 1 })

        Write-Verbose -Message 'Parsing SNAPMIRROR-POLICY.XML'
        $systemAsupData.Snapmirror | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'SNAPMIRROR-POLICY.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_SNAPMIRROR_POLICY' -ReplaceString '^Smpolicy' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing SNAPMIRROR-DESTINATION.XML'
        $systemAsupData.Snapmirror | Add-Member -MemberType NoteProperty -Name Destination -Value $(
            foreach ($srcAsup in $systemAsupData.SourceAsup) {
                Get-AsupSection -AsupObject $srcAsup -SectionName 'SNAPMIRROR-DESTINATION.XML' -InnerProperty 'T_SNAPMIRROR_DESTINATION' -ReplaceString '^Smpolicy' -ErrorVariable +NetAppDocsError | Where-Object { $_.SourceVolumeNode -ceq $srcAsup.SystemName }
            }
        )

        #endregion Snapmirror Configuration

        #region CIFS

        ##Unused Section: CIFS-DOMAIN-DISCOVERED-SERVER.XML - Not currently needed for build docs (Cifs.DomainServer)

        Write-Verbose -Message '- Gathering CIFS information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Cifs -Value ([pscustomobject]@{ })
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name Gpo -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing CIFS-SERVER-BYNAME.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name Server -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-SERVER-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_SERVER' -ReplaceString '^CifsServer' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-SERVER-OPTION.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name Option -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-SERVER-OPTION.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_SERVER_OPTIONS' -ReplaceString '^CifsServerOptions' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-SERVER-SECURITY.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name Security -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-SERVER-SECURITY.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_SERVER_SECURITY' -ReplaceString '^CifsServerSecurity' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-SYMLINK-PATHMAPPING.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name Symlink -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-SYMLINK-PATHMAPPING.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_SYMLINK' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-SHARE-BYNAME.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name Share -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-SHARE-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_SHARE' -ReplaceString '^CifsShare' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Capturing CIFS-SHARE-ACL.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name ShareAcl -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-SHARE-ACL.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_SHARE_ACL' -ReplaceString '^CifsShareAcl' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-GPO-DB.XML'
        $systemAsupData.Cifs.Gpo | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-GPO-DB.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_GPO_DB' -ReplaceString '^GpoDb' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-HOMEDIR.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name HomeDirectoryConfig -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-HOMEDIR.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_HOMEDIR' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-HOMEDIR-SEARCHPATH.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name HomeDirectorySearchPath -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-HOMEDIR-SEARCHPATH.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_HOMEDIR_SEARCHPATH' -ReplaceString '^CifsHomedirSearchPath' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-DOMAIN-PREFERRED-DC.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name PreferredDomainController -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-DOMAIN-PREFERRED-DC.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_PREFERRED_DCS' -ReplaceString '^CifsPreferredDc' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing CIFS-DC-DISCOVERY-MODE.XML'
        $systemAsupData.Cifs | Add-Member -MemberType NoteProperty -Name DomainServerDiscoveryMode -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'CIFS-DC-DISCOVERY-MODE.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_CIFS_DISCOVERY_MODE' -ErrorVariable +NetAppDocsError)

        #endregion CIFS

        #region NFS

        Write-Verbose -Message '- Gathering NFS configuration information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Nfs -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing NFS-SERVERS-BYNAME.XML'
        $systemAsupData.Nfs | Add-Member -MemberType NoteProperty -Name Service -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NFS-SERVERS-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_VSERV_NFS' -ReplaceString '^VservNfs' -ErrorVariable +NetAppDocsError)

        #endregion NFS

        #region Lun Configuration

        Write-Verbose -Message '- Gathering Lun information -'

        $systemAsupData | Add-Member -MemberType NoteProperty -Name Lun -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing LUN.XML'
        $systemAsupData.Lun | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'LUN.XML' -InnerProperty 'T_LUN' -ReplaceString '^Lun' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing LUN-MAPS.XML'
        $systemAsupData.Lun | Add-Member -MemberType NoteProperty -Name Map -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'LUN-MAPS.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_LUN_MAP' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing IGROUP.XML'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Igroup -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'IGROUP.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_IGROUP' -ReplaceString '^Igroup' -ErrorVariable +NetAppDocsError)

        #endregion Lun Configuration

        #region Quota Configuration

        ##Unused Section: QUOTA-POLICY.XML - (1.20+) Not needed for build docs (Quota.Policy)
        ##Unused Section: QUOTA-POLICY-UI-TABLE.XML - (-lt 1.20) Not needed for build docs (Quota.Policy)

        Write-Verbose -Message '- Gathering quota configuration -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Quota -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing QUOTA-RULES.XML'
        $systemAsupData.Quota | Add-Member -MemberType NoteProperty -Name Rule -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'QUOTA-RULES.XML' -InnerProperty 'T_QUOTA_RULES' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing VOLUME-QUOTA.XML'
        $systemAsupData.Quota | Add-Member -MemberType NoteProperty -Name Status -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'VOLUME-QUOTA.XML' -InnerProperty 'T_QUOTA' -ErrorVariable +NetAppDocsError)

        #endregion Quota Configuration

        #region Name Services Information (NIS/Kerberos/LDAP)

        ##Unused Section: UNIX-GROUPS-FILEONLY.XML - (1.101+) Not needed for build docs
        ##Unused Section: UNIX-USERS-FILEONLY.XML - (1.101+) Not needed for build docs

        Write-Verbose -Message '- Gathering name services information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name NameService -Value ([pscustomobject]@{ })
        $systemAsupData.NameService | Add-Member -MemberType NoteProperty -Name Kerberos -Value ([pscustomobject]@{ })
        $systemAsupData.NameService | Add-Member -MemberType NoteProperty -Name Ldap -Value ([pscustomobject]@{ })
        $systemAsupData.NameService | Add-Member -MemberType NoteProperty -Name Nis -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing NIS-DOMAINS-BYNAME.XML'
        $systemAsupData.NameService.Nis | Add-Member -MemberType NoteProperty -Name Info -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NIS-DOMAINS-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_NIS' -ReplaceString '^Nis' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing KERBEROS-IFCONFIG-BYNAME.XML'
        $systemAsupData.NameService.Kerberos | Add-Member -MemberType NoteProperty -Name Config -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'KERBEROS-IFCONFIG-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_KERBEROS_CONFIG' -ReplaceString '^Kerberosconfig' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing KERBREALM.XML'
        $systemAsupData.NameService.Kerberos | Add-Member -MemberType NoteProperty -Name Realm -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'KERBREALM.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_KERBEROS_REALM' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing LDAP-CONFIG.XML'
        $systemAsupData.NameService.Ldap | Add-Member -MemberType NoteProperty -Name Config -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'LDAP-CONFIG.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_LDAP' -ReplaceString '^Ldapconfig' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing LDAP-CLIENT-CONFIG.XML'
        $systemAsupData.NameService.Ldap | Add-Member -MemberType NoteProperty -Name Client -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'LDAP-CLIENT-CONFIG.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_LDAP_CLIENT' -ReplaceString '^Ldapclientconfig' -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Parsing LDAP-CLIENT-SCHEMA.XML'
        $systemAsupData.NameService.Ldap | Add-Member -MemberType NoteProperty -Name ClientSchema -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'LDAP-CLIENT-SCHEMA.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_LDAP_CLIENT_SCHEMA' -ReplaceString '^Ldapclientschema' -ErrorVariable +NetAppDocsError)

        if ($systemAsupData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message 'Parsing NSSWITCH-BYNAME.XML'
            $systemAsupData.NameService | Add-Member -MemberType NoteProperty -Name NsSwitch -Value (Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'NSSWITCH-BYNAME.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_NSSWITCH' -ReplaceString '^Nsswitch' -ErrorVariable +NetAppDocsError)
        }

        #endregion Name Services Information (NIS/Kerberos/LDAP)

        #region Job Information

        Write-Verbose -Message '- Gathering job information -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name Job -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Parsing JM-SCHED.XML'
        $systemAsupData.Job | Add-Member -MemberType NoteProperty -Name Schedule -Value (
            Get-AsupSection -AsupObject $systemAsupData.SourceAsup -SectionName 'JM-SCHED.XML' -MasterNodeName $masterNodeName -InnerProperty 'T_JOB_SCHEDULE' -ReplaceString '^JobSchedule' -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                Name       = 'JobScheduleInfo'
                Expression = { ConvertFrom-JobScheduleDescription -JobScheduleDescription $_.Description }
            }
        )

        #endregion Job Information

        Write-Verbose -Message '- Identifying ASUP Generation Time -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name ReferenceDT -Value ($systemAsupData.SourceAsup | Select-Object -First 1).AsupGenerationDate.ToUniversalTime()

        Write-Verbose -Message '- Identifying SystemName -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name SystemName -Value $systemAsupData.Cluster.Info.ClusterName

        Write-Verbose -Message '- Identifying any errors during processing -'
        $systemAsupData | Add-Member -MemberType NoteProperty -Name DiscoveryError -Value @(
            foreach ($err in ($NetAppDocsError | Where-Object { $_ -and $_.Exception.Message -notlike 'Exception calling "LoadXml"*' -and $_.Exception.Message -notlike 'Entity not supported*' })) {
                $nodes, $section = $err.CategoryInfo.TargetName -split '\s*:\s*'

                foreach ($node in ($nodes -split '\s+')) {
                    $err.CategoryInfo | Select-Object -Property Category, Reason, @{
                        Name       = 'Activity'
                        Expression = { $_.Activity }
                    }, @{
                        Name       = 'TargetName'
                        Expression = { $node }
                    }, @{
                        Name       = 'TargetSection'
                        Expression = { $section }
                    }, @{
                        Name       = 'TargetType'
                        Expression = { 'NcController.Asup' }
                    }, @{
                        Name       = 'ManifestStatus'
                        Expression = {
                            (($systemAsupData.SourceAsup | Where-Object { $_.SystemName -eq $node }).Manifest | Where-Object {
                                $_.BodyFile -eq $section -or
                                $_.BodyFile -eq "$( $section ).txt" -or
                                $_.BodyFile -eq "$( $section -replace '-', '_' )"
                            }
                        ).Status }
                    }, @{
                        Name       = 'Message'
                        Expression = {
                            if ($err.Exception.Message -eq 'No systems available for the input data:[]') {
                                'ASUP section is missing or empty for this system'
                            }
                            else {
                                $err.Exception.Message -split '\s*[:]\s*' | Select-Object -First 1
                            }
                        }
                    }
                }
            }
        )

        if (($systemAsupData.DiscoveryError | Where-Object { $_.Reason -ne 'NO_DATA_FOUND' -or ($_.ManifestStatus -and $_.ManifestStatus -notin @( 'content-empty', 'content-not-collected-precheck' )) } | Measure-Object).Count -gt 0) {
            $msg = New-Object -TypeName System.Text.StringBuilder

            [void]$msg.AppendLine("Errors were encountered while collecting data from ASUPID: $( $systemAsupData.SourceAsup.AsupId )")
            [void]$msg.AppendLine('-------------------------------------------------------------------------------------')
            [void]$msg.AppendLine('Please review the output for accuracy and, if necessary, correct any error conditions')
            [void]$msg.AppendLine('then run these scripts again.')
            [void]$msg.AppendLine('-------------------------------------------------------------------------------------')
            [void]$msg.AppendLine()

            Write-Warning -Message $msg.ToString()
        }

        $systemAsupData | ConvertTo-NtapVserverData -VserverName $VserverContext
    }
    catch {
        $Global:Error.RemoveAt(0)

        $msg = New-Object -TypeName System.Text.StringBuilder

        if ($_.CategoryInfo.Category -eq 'ResourceUnavailable') {
            [void]$msg.AppendLine('The connection to the ASUP REST server has failed. Please check your connection and try the capture again.')
        }
        elseif ($_.CategoryInfo.Category -eq 'ReadError') {
            [void]$msg.AppendLine('An unexpected error occurred. Processing halted for this system.')
            [void]$msg.AppendLine('---------------------------------------------------------------------------------------------')
            [void]$msg.AppendLine('This error is not expected, but we do know about it. Unfortunately, this is an issue with the')
            [void]$msg.AppendLine('ASUP data warehouse and the only workaround at this time is to try a different set of ASUPs.')
            [void]$msg.AppendLine('---------------------------------------------------------------------------------------------')
            [void]$msg.AppendLine($( $_.Exception.GetBaseException().Message ))
            [void]$msg.AppendLine("Location: $( $_.ScriptStackTrace )")
            [void]$msg.AppendLine()
        }
        else {
            [void]$msg.AppendLine('An unexpected error occurred. Processing halted.')
            [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
            [void]$msg.AppendLine('As this error is not expected, please consider contacting the support staff via e-mail or by')
            [void]$msg.AppendLine('starting a thread on the community site.')
            [void]$msg.AppendLine('            E-mail: ng-NetAppDocs-support@netapp.com')
            [void]$msg.AppendLine('    Community site: https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs')
            [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
            [void]$msg.AppendLine($( $_.Exception.GetBaseException().Message ))
            [void]$msg.AppendLine("Location: $( $_.ScriptStackTrace )")
            [void]$msg.AppendLine()
        }

        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $_.CategoryInfo.TargetName)))
    }
}

# SIG # Begin signature block
# MIIq0AYJKoZIhvcNAQcCoIIqwTCCKr0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCD5cB6TW95OtUhH
# wqVvPNMLxwmxXt8mWaHMG6Z7mXeG4qCCJGgwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggWNMIIEdaADAgECAhAOmxiO
# +dAt5+/bUOIIQBhaMA0GCSqGSIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYD
# VQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAi
# BgNVBAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAw
# MDBaFw0zMTExMDkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
# Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERp
# Z2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBAL/mkHNo3rvkXUo8MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsb
# hA3EMB/zG6Q4FutWxpdtHauyefLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iT
# cMKyunWZanMylNEQRBAu34LzB4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGb
# NOsFxl7sWxq868nPzaw0QF+xembud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclP
# XuU15zHL2pNe3I6PgNq2kZhAkHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCr
# VYJBMtfbBHMqbpEBfCFM1LyuGwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFP
# ObURWBf3JFxGj2T3wWmIdph2PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTv
# kpI6nj3cAORFJYm2mkQZK37AlLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWM
# cCxBYKqxYxhElRp2Yn72gLD76GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls
# 5Q5SUUd0viastkF13nqsX40/ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBR
# a2+xq4aLT8LWRV+dIPyhHsXAj6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6
# MIIBNjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qY
# rhwPTzAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8E
# BAMCAYYweQYIKwYBBQUHAQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5k
# aWdpY2VydC5jb20wQwYIKwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDig
# NoY0aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNybDARBgNVHSAECjAIMAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCg
# v0NcVec4X6CjdBs9thbX979XB72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQT
# SnovLbc47/T/gLn4offyct4kvFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh
# 65ZyoUi0mcudT6cGAxN3J0TU53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSw
# uKFWjuyk1T3osdz9HNj0d1pcVIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAO
# QGPFmCLBsln1VWvPJ6tsds5vIy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjD
# TZ9ztwGpn1eqXijiuZQwggXMMIIENKADAgECAhAg429sPxgagUb53pPffJfkMA0G
# CSqGSIb3DQEBDAUAMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBS
# MzYwHhcNMjEwOTA5MDAwMDAwWhcNMjMwOTA5MjM1OTU5WjBQMQswCQYDVQQGEwJV
# UzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UECgwMTmV0QXBwLCBJbmMuMRUw
# EwYDVQQDDAxOZXRBcHAsIEluYy4wggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGK
# AoIBgQC4kYYj/oViZD9pN03hrqFjtwOz1Gx4eDWVV8IYUYEr2qpLeNYvWz26B/+E
# mYLboAGVpkYg0Wske3hjRpooZlMORm1S4+2C2NoQCstZ+kmNTevbXHJe/w1VNJrm
# fKjpXOJEfx8GUGDqCMR30dChVpsdBEavrM7T0hnfJLv18i19SGv3a+nhvB3iOjLP
# SEg8+DFkjD5jTI6fQ83DBEiVR4UEYq7yyH578ZEylhsRfJmK+uIREycEUk/NpFTw
# g6/7lxh/WYabCrZzqI4Ep3QataRKaAlxt3BiOlZkX4WGX3DYvSYltWOM2UlCcGpP
# m/1/LN8Xuvf+YO6H472QecHe59XtXbHSMKvqDuOEJu8Wk2LFMNK732o3Fc5QIHie
# 6JtNsgNYf+Vsm5EMBD1ZHNE+C3zlmQbbWIU5uEU1nhiodBPKsu0SkS6pARyKBO05
# DSc2gWWv0aukKi04ZAn+hAdSl5s1dgmU5shBvKHqe15K9ZRN4FFO/CgHE0BoGYQS
# UQVKwa0CAwEAAaOCAZwwggGYMB8GA1UdIwQYMBaAFA8qyyCHKLjsb0iuK1SmKaoX
# pM0MMB0GA1UdDgQWBBQuH643KcBMmb/Q6IZt+H9IrnXFwDAOBgNVHQ8BAf8EBAMC
# B4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDAzARBglghkgBhvhC
# AQEEBAMCBBAwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwIwJTAjBggrBgEFBQcC
# ARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQBMEkGA1UdHwRCMEAw
# PqA8oDqGOGh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVT
# aWduaW5nQ0FSMzYuY3JsMHkGCCsGAQUFBwEBBG0wazBEBggrBgEFBQcwAoY4aHR0
# cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQVIz
# Ni5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqG
# SIb3DQEBDAUAA4IBgQCOoGdXjP8Sif0h3ZvoDsIVfrJvQUdP9pZExRJGGj/Te6ML
# XyqHNc/G2WklNv+BC4ryNJ4zp0oneHOub55ejrYPfhgP9oFWS8oOPJbrp3rAtINa
# OeKRC88WUYZSKYAxSX6kubEjnQD6cmazXjxVN6P8+6q9smeqF3xI+++c+ekMw3Xv
# 4EWRYXqN0/srfFp1WpoluwSQcZMqj9haSX0bgO6AAWH2AnVJMfXTIKz/0FQ/RW0y
# Ed5QYQqYInhA7IUz9dBziNpJlwONeF5j7xzgfYDY63WU6WrgJnGjYkQCOOsUx74j
# gUiMRe9zV48GS8Vxz22c/TQs0x4/1GmOSJvUHEk3GseBmB3v+yEhh/D6zWOfYP4X
# D/9b91CxmugTuepuaJSeRg+qUm3KarKsOCUF/CLqUrNh/JwKrWD1cghRaYLvMucs
# ScksHHe7ZDvb2OtvxWXjPk1d1NKvEwFJSS6hIVIbug9x28AJqOoP5Pn9wZvJSzvW
# uQJGCLscdGyKefdCo30wggYaMIIEAqADAgECAhBiHW0MUgGeO5B5FSCJIRwKMA0G
# CSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290
# IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFQxCzAJBgNVBAYT
# AkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28g
# UHVibGljIENvZGUgU2lnbmluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEBAQUAA4IB
# jwAwggGKAoIBgQCbK51T+jU/jmAGQ2rAz/V/9shTUxjIztNsfvxYB5UXeWUzCxEe
# AEZGbEN4QMgCsJLZUKhWThj/yPqy0iSZhXkZ6Pg2A2NVDgFigOMYzB2OKhdqfWGV
# oYW3haT29PSTahYkwmMv0b/83nbeECbiMXhSOtbam+/36F09fy1tsB8je/RV0mIk
# 8XL/tfCK6cPuYHE215wzrK0h1SWHTxPbPuYkRdkP05ZwmRmTnAO5/arnY83jeNzh
# P06ShdnRqtZlV59+8yv+KIhE5ILMqgOZYAENHNX9SJDm+qxp4VqpB3MV/h53yl41
# aHU5pledi9lCBbH9JeIkNFICiVHNkRmq4TpxtwfvjsUedyz8rNyfQJy/aOs5b4s+
# ac7IH60B+Ja7TVM+EKv1WuTGwcLmoU3FpOFMbmPj8pz44MPZ1f9+YEQIQty/NQd/
# 2yGgW+ufflcZ/ZE9o1M7a5Jnqf2i2/uMSWymR8r2oQBMdlyh2n5HirY4jKnFH/9g
# Rvd+QOfdRrJZb1sCAwEAAaOCAWQwggFgMB8GA1UdIwQYMBaAFDLrkpr/NZZILyhA
# QnAgNpFcF4XmMB0GA1UdDgQWBBQPKssghyi47G9IritUpimqF6TNDDAOBgNVHQ8B
# Af8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcD
# AzAbBgNVHSAEFDASMAYGBFUdIAAwCAYGZ4EMAQQBMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Um9vdFI0Ni5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ1Jvb3RSNDYu
# cDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQwFAAOCAgEABv+C4XdjNm57oRUgmxP/BP6YdURhw1aVcdGRP4Wh60BAscjW
# 4HL9hcpkOTz5jUug2oeunbYAowbFC2AKK+cMcXIBD0ZdOaWTsyNyBBsMLHqafvIh
# rCymlaS98+QpoBCyKppP0OcxYEdU0hpsaqBBIZOtBajjcw5+w/KeFvPYfLF/ldYp
# mlG+vd0xqlqd099iChnyIMvY5HexjO2AmtsbpVn0OhNcWbWDRF/3sBp6fWXhz7Dc
# ML4iTAWS+MVXeNLj1lJziVKEoroGs9Mlizg0bUMbOalOhOfCipnx8CaLZeVme5yE
# Lg09Jlo8BMe80jO37PU8ejfkP9/uPak7VLwELKxAMcJszkyeiaerlphwoKx1uHRz
# NyE6bxuSKcutisqmKL5OTunAvtONEoteSiabkPVSZ2z76mKnzAfZxCl/3dq3dUNw
# 4rg3sTCggkHSRqTqlLMS7gjrhTqBmzu1L90Y1KWN/Y5JKdGvspbOrTfOXyXvmPL6
# E52z1NZJ6ctuMFBQZH3pwWvqURR8AgQdULUvrxjUYbHHj95Ejza63zdrEcxWLDX6
# xWls/GDnVNueKjWUH3fTv1Y8Wdho698YADR7TNx8X8z2Bev6SivBBOHY+uqiirZt
# g0y9ShQoPzmCcn63Syatatvx157YK9hlcPmVoa1oDE5/L9Uo2bC5a4CH2Rwwggau
# MIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJ
# BgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5k
# aWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAe
# Fw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcw
# FQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3Rl
# ZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9Ge
# TKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0
# hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZl
# jZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAsh
# aG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVY
# TXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1
# biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCir
# c0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+
# DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA
# +bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42Pg
# puE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzS
# M7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU
# uhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6
# mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsG
# AQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29t
# MEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNl
# cnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3Js
# My5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAE
# GTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1Z
# jsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8d
# B+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVp
# P0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp8
# 76i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2
# nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3
# ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQ
# txMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc
# 4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+Y
# AN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZ
# vAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQr
# H4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIGwDCCBKigAwIBAgIQDE1p
# ckuU+jwqSj0pB4A9WjANBgkqhkiG9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4XDTIyMDkyMTAwMDAw
# MFoXDTMzMTEyMTIzNTk1OVowRjELMAkGA1UEBhMCVVMxETAPBgNVBAoTCERpZ2lD
# ZXJ0MSQwIgYDVQQDExtEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMiAtIDIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDP7KUmOsap8mu7jcENmtuh6BSFdDMa
# JqzQHFUeHjZtvJJVDGH0nQl3PRWWCC9rZKT9BoMW15GSOBwxApb7crGXOlWvM+xh
# iummKNuQY1y9iVPgOi2Mh0KuJqTku3h4uXoW4VbGwLpkU7sqFudQSLuIaQyIxvG+
# 4C99O7HKU41Agx7ny3JJKB5MgB6FVueF7fJhvKo6B332q27lZt3iXPUv7Y3UTZWE
# aOOAy2p50dIQkUYp6z4m8rSMzUy5Zsi7qlA4DeWMlF0ZWr/1e0BubxaompyVR4aF
# eT4MXmaMGgokvpyq0py2909ueMQoP6McD1AGN7oI2TWmtR7aeFgdOej4TJEQln5N
# 4d3CraV++C0bH+wrRhijGfY59/XBT3EuiQMRoku7mL/6T+R7Nu8GRORV/zbq5Xwx
# 5/PCUsTmFntafqUlc9vAapkhLWPlWfVNL5AfJ7fSqxTlOGaHUQhr+1NDOdBk+lbP
# 4PQK5hRtZHi7mP2Uw3Mh8y/CLiDXgazT8QfU4b3ZXUtuMZQpi+ZBpGWUwFjl5S4p
# kKa3YWT62SBsGFFguqaBDwklU/G/O+mrBw5qBzliGcnWhX8T2Y15z2LF7OF7ucxn
# EweawXjtxojIsG4yeccLWYONxu71LHx7jstkifGxxLjnU15fVdJ9GSlZA076XepF
# cxyEftfO4tQ6dwIDAQABo4IBizCCAYcwDgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB
# /wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwIAYDVR0gBBkwFzAIBgZngQwB
# BAIwCwYJYIZIAYb9bAcBMB8GA1UdIwQYMBaAFLoW2W1NhS9zKXaaL3WMaiCPnshv
# MB0GA1UdDgQWBBRiit7QYfyPMRTtlwvNPSqUFN9SnDBaBgNVHR8EUzBRME+gTaBL
# hklodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0
# MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3JsMIGQBggrBgEFBQcBAQSBgzCBgDAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFgGCCsGAQUFBzAC
# hkxodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRS
# U0E0MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3J0MA0GCSqGSIb3DQEBCwUAA4IC
# AQBVqioa80bzeFc3MPx140/WhSPx/PmVOZsl5vdyipjDd9Rk/BX7NsJJUSx4iGNV
# CUY5APxp1MqbKfujP8DJAJsTHbCYidx48s18hc1Tna9i4mFmoxQqRYdKmEIrUPwb
# tZ4IMAn65C3XCYl5+QnmiM59G7hqopvBU2AJ6KO4ndetHxy47JhB8PYOgPvk/9+d
# EKfrALpfSo8aOlK06r8JSRU1NlmaD1TSsht/fl4JrXZUinRtytIFZyt26/+YsiaV
# OBmIRBTlClmia+ciPkQh0j8cwJvtfEiy2JIMkU88ZpSvXQJT657inuTTH4YBZJwA
# wuladHUNPeF5iL8cAZfJGSOA1zZaX5YWsWMMxkZAO85dNdRZPkOaGK7DycvD+5sT
# X2q1x+DzBcNZ3ydiK95ByVO5/zQQZ/YmMph7/lxClIGUgp2sCovGSxVK05iQRWAz
# gOAj3vgDpPZFR+XOuANCR+hBNnF3rf2i6Jd0Ti7aHh2MWsgemtXC8MYiqE+bvdgc
# mlHEL5r2X6cnl7qWLoVXwGDneFZ/au/ClZpLEQLIgpzJGgV8unG1TnqZbPTontRa
# mMifv427GFxD9dAq6OJi7ngE273R+1sKqHB+8JeEeOMIA11HLGOoJTiXAdI/Otrl
# 5fbmm9x+LMz/F0xNAKLY1gEOuIvu5uByVYksJxlh9ncBjDGCBb4wggW6AgEBMGgw
# VDELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDErMCkGA1UE
# AxMiU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIFIzNgIQIONvbD8YGoFG
# +d6T33yX5DANBglghkgBZQMEAgEFAKCBhDAYBgorBgEEAYI3AgEMMQowCKACgACh
# AoAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAM
# BgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCAaYZ7zm3V0kWnWegvKL8ul1lz1
# /4QR65QROTs7LupTTzANBgkqhkiG9w0BAQEFAASCAYCXAxTFwkCl7MBvcWpMQOwm
# 2zA9Zz88F6o3/yd3Gy18wgEvfa7rkYFLk1wFENFljNz4zLtYLwxkuKvTVENP6wZo
# vJfDddGiTfkePLQjM4dlvAQLpn8ueJxYU65SENqNkzOup14NR31KfmKQ0BeIeH+d
# zLs2tF2dsrI8sx/AMDUqsYNCXXYn6WyG03gqHL051TuUISlnxqWJOoO7elcUys7F
# gASMP0emDmuuJPfj3A6mBDkI2Peb/EGrO62vnojx/7o6+tbvHxjF6amAXiS6Gz6L
# diC0x1TrBDLNpda5O8pL/HyuIwkqMJ5ZBfbxDJ7IFuVyVbHqiDCGENFzD0pp0+1V
# qSzwK5yA0nCKPqzE87K1LGBG6O85YzwevJXvZpn9q2C7Re6ADUmCPn53zd3Jkj/H
# Y7lA6bAdNX6pHO8CYhwvdhxM0B8mwFds3aYl7jKbzzvgRKBjf6N4u8kQIjI/Rd4i
# XSqofyuGemNtNeKP2If+KpYbZ6RSqANFYjVHG+LsBLahggMgMIIDHAYJKoZIhvcN
# AQkGMYIDDTCCAwkCAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
# cnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
# SEEyNTYgVGltZVN0YW1waW5nIENBAhAMTWlyS5T6PCpKPSkHgD1aMA0GCWCGSAFl
# AwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUx
# DxcNMjMwNDAyMTUyODI3WjAvBgkqhkiG9w0BCQQxIgQgMD5HQfoFvIUesyl40FEk
# TLYeDyuOHmQcSE4oMD5LnA0wDQYJKoZIhvcNAQEBBQAEggIAtBZ5CZ9Cm1M75Hvm
# WMxtdO2b0PvpZ56pZM1wRG3zd6qoNHiZrxDLnxiZBrVpnK4cIJBmg6fNgYg/MbVQ
# h1ihMXzKg0juOVLHV0PfhnBUyehHTQXytbYeToz4iSQ7MuRhA4lSFHxy84t1MRz3
# DQy3wlq+X2XhRfxe1mTFJ+JOpMMX60oRuovlACTK4ZkkyXMpnBFiPLHd6chHfxFA
# HhkmxKxHSNBq8pKNAlLBaOMft/R8Qs8EvoQPNZgJgH/2a5r02ShiDDDwilXJvS6z
# 9aZTXGCdQ2Uu/lGsE0Kaau/0N94u8fVPcYdxTyL8XitwKGk+KT+/4y6e6UEeB6Ta
# zW8nWQQdwDOBAdpimSrtsZ4BYHqOcUeDpNTE724rkX9zdsneY2sQjYHRoTkMTzK0
# Ypo4GALyQogohN3kLw/VH+M1ajpxQqJN+XVd711aMxPxXICvnsMXJ6ONMYGktVlx
# U0jwqt9cyAYOCUM5EaLbszPOgZegAogCfJLZXNmotdZlu/1iAWjaw68nxPPZgyII
# c766B/HONHmNaCxJt0F77oNgxQI6cwCiR5/rp/AX5/HbLz/3JWaQl1P7qG6UDvKS
# kxDJ5T+eaqLt0tXdcqNNCdO9HLGEpJd7U7VXJa13MwvnKOqrpsFAjh7lsvDOr7nn
# HUsLRa1WBrudHFKMjA+W6xskjJA=
# SIG # End signature block
