function Get-NtapVserverData {
    <#
        .SYNOPSIS
            Collects SVM specific data from ONTAP storage systems.

        .DESCRIPTION
            Uses the NetApp ONTAP PowerShell Toolkit to gather as much SVM configuration information as possible. This
            data can be used to automate build docs. It can also be used for historical configuration information. It
            will gather information about each data SVM, or specific SVMs with the -Name parameter.

        .EXAMPLE
            PS> Get-NtapVserverData -ClusterName cluster1 | Export-Clixml -Path 'D:\Output\cluster1_AllSVMs.xml'

            Collects the configuration information for each data SVM from the cluster 'cluster1' and saves the results
            to a .xml file for further processing.

        .EXAMPLE
            PS> Get-NtapVserverData -ClusterName cluster1 -Credential (Get-Credential) | Export-Clixml -Path 'D:\Output\cluster1_AllSVMs.xml'

            Collects the configuration information for each data SVM from the cluster 'cluster1' using alternate
            credentials, and saves the results to a .xml file for further processing. The Add-NcCredential cmdlet may be
            used to cache credentials for the cluster to be accessed.

        .EXAMPLE
            PS> Get-NtapVserverData -Name vs_nfs, vs_cifs, vs_san -ClusterName cluster1 | Export-Clixml -Path 'D:\Output\cluster1_SVMs.xml'

            Collects the configuration information for the specified data SVMs from the cluster 'cluster1' and saves the
            results to a .xml file for further processing.

        .EXAMPLE
            PS> Get-NtapVserverData -ClusterName cluster1 | Format-NtapVserverData | Out-NtapDocument -WordFile 'D:\Output\cluster1_AllSVMs.xlsx' -WordFile 'D:\Output\cluster1_AllSVMs.xlsx'

            Collects the configuration information for all data SVMs in cluster 'cluster1' and produces the specified
            Word and Excel documents.

        .LINK
            https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs

        .LINK
            Format-NtapVserverData

        .INPUTS
            NetApp.Ontapi.AbstractController

        .INPUTS
            System.String

        .OUTPUTS
            System.Management.Automation.PSCustomObject

        .NOTES
            AUTHOR : Jason Cole
    #>

    [CmdletBinding(DefaultParameterSetName = 'Name')]
    [OutputType([System.Management.Automation.PSCustomObject])]
    param (
        # The system name or IP address of the cluster admin SVM to gather the data from.
        [Parameter(Mandatory, Position = 0, ValueFromPipeline, ParameterSetName = 'Name')]
        [ValidateNotNullOrEmpty()]
        [string]$ClusterName,

        # A NetApp ONTAP PowerShell Toolkit controller object (NcController).
        [Parameter(Mandatory, Position = 0, ValueFromPipeline, ParameterSetName = 'ControllerObject')]
        [ValidateNotNullOrEmpty()]
        [NetApp.Ontapi.AbstractController]$AbstractController,

        #  A list of data SVMs to gather the data for. If blank, the information for all data SVMs will be returned.
        [Parameter()]
        [ValidateNotNullOrEmpty()]
        [Alias('VserverName', 'SvmName')]
        [string[]]$Name,

        # Specifies explicit credentials to connect to the storage system. (Alias: Cred)
        [Parameter(ParameterSetName = 'Name')]
        [ValidateNotNullOrEmpty()]
        [Alias('Cred')]
        [pscredential]
        [System.Management.Automation.Credential()]
        $Credential = [pscredential]::Empty,

        # The port on which to connect to the storage system. If not set, it defaults to 80 for HTTP and 443 for HTTPS.
        [Parameter(ParameterSetName = 'Name')]
        [ValidateRange([uint16]::MinValue, [uint16]::MaxValue)]
        [uint16]$Port,

        # Connection timeout in milliseconds. Defaults to 60000 milliseconds.
        [Parameter()]
        [ValidateRange(0, [int]::MaxValue)]
        [int]$Timeout,

        # Forces the use of HTTPS instead of HTTP for the connection to the storage system.
        [Parameter(ParameterSetName = 'Name')]
        [Alias('ForceHTTPS')]
        [switch]$ForceSecure,

        # During the initial connection to the storage system, this switch disables a check that is performed to ensure that the LIF
        # being used for the connection is a cluster mgmt or a node mgmt port. This should only be used for a very specific scenario
        # where port forwarding is configured on the host connecting to the storage system.
        [Parameter()]
        [switch]$DisableLifCheck,

        # Disables the ping test prior to connecting to the storage system (useful in the event ICMP is being blocked). (Deprecated)
        [Parameter()]
        [ValidateScript({
                Write-Warning -Message '-DisablePingCheck has been deprecated (no longer performing an ICMP ping to test connectivity).'

                $true
            })]
        [switch]$DisablePingCheck
    )

    process {
        Write-Verbose -Message "Initializing function: $( $PSCmdlet.MyInvocation.MyCommand.Name ) ($( $PSCmdlet.MyInvocation.MyCommand.ModuleName ) v$( (Get-NtapDocsVersion -AsVersion).ToString() ))"

        switch -Exact ($PSCmdlet.ParameterSetName) {
            'Name' {
                Write-Verbose -Message "Validating connectivity to system: $ClusterName"

                try {
                    $ip = [System.Net.Dns]::GetHostAddresses($ClusterName)
                    $tcpClient = New-Object -TypeName System.Net.Sockets.TCPClient

                    if ($PSCmdlet.MyInvocation.BoundParameters.ContainsKey('Port')) {
                        $tcpConnect = $tcpClient.BeginConnect($ip, $Port, $null, $null)

                        if (-not $tcpConnect.AsyncWaitHandle.WaitOne(7500, $false)) {
                            $tcpClient.Close()

                            throw 'Connection failed (ResourceUnavailable).'
                        }
                    }
                    elseif ($ForceSecure.IsPresent) {
                        $tcpConnect = $tcpClient.BeginConnect($ip, 443, $null, $null)

                        if (-not $tcpConnect.AsyncWaitHandle.WaitOne(7500, $false)) {
                            $tcpClient.Close()

                            throw 'Connection failed (ResourceUnavailable).'
                        }
                    }
                    else {
                        $tcpConnect = $tcpClient.BeginConnect($ip, 443, $null, $null)

                        if (-not $tcpConnect.AsyncWaitHandle.WaitOne(7500, $false)) {
                            $tcpClient.Close()

                            $tcpClient = New-Object -TypeName System.Net.Sockets.TCPClient
                            $tcpConnect = $tcpClient.BeginConnect($ip, 80, $null, $null)

                            if (-not $tcpConnect.AsyncWaitHandle.WaitOne(7500, $false)) {
                                $tcpClient.Close()

                                throw 'Connection failed (ResourceUnavailable).'
                            }
                        }
                    }

                    [void]$tcpClient.EndConnect($tcpConnect)
                    $tcpClient.Close()

                    $params = @{
                        Name        = $ClusterName
                        ForceSecure = $ForceSecure.IsPresent
                    }

                    if ($Credential -ne [pscredential]::Empty) {
                        $params.Add('Credential', $Credential)
                    }

                    if ($PSCmdlet.MyInvocation.BoundParameters.ContainsKey('Port')) {
                        $params.Add('Port', $Port)
                    }

                    if ($PSCmdlet.MyInvocation.BoundParameters.ContainsKey('Timeout')) {
                        $params.Add('Timeout', $Timeout)
                    }

                    $controller = Connect-NtapStorageSystem @params -NcController -ErrorAction Stop

                    if ($controller.GetType().FullName -ne 'NetApp.Ontapi.Filer.C.NcController') {
                        throw 'Incorrect controller type specified for the Get-NtapVserverData function.'
                    }

                    $controller.Vserver = $null

                    if (-not (Test-NtapStorageSystemConnection -Controller $controller -DisableLifCheck:$DisableLifCheck.IsPresent -ErrorAction Stop)) {
                        throw 'Unspecified error.'
                    }

                    Write-Verbose -Message "Valid connection established to system: $ClusterName"

                    try {
                        Write-Verbose -Message '- Gathering list of SVMs -'

                        $allVservers = Get-NcVserver -Controller $controller -ErrorAction Stop | Where-Object { $_.VserverType -in @( 'data', 'cluster' ) }

                        if ($Name) {
                            $diffObject = Compare-Object -ReferenceObject $allVservers.VserverName -DifferenceObject $Name -IncludeEqual

                            ($diffObject | Where-Object { $_.SideIndicator -eq '=>' }).InputObject | Where-Object { $_ } | ForEach-Object {
                                Show-ErrorMessage -Message "Vserver does not exist in cluster: $_"
                            }

                            $vservers = $allVservers | Where-Object { $_.VserverName -cin ($diffObject | Where-Object { $_.SideIndicator -eq '==' }).InputObject }
                        }
                        else {
                            $vservers = $allVservers
                        }
                    }
                    catch {
                        $Global:Error.RemoveAt(0)

                        Write-Error -ErrorRecord (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("Unable to gather SVMs from system. Reason: $( $_.Exception.GetBaseException().Message )", $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $ClusterName))

                        continue
                    }
                }
                catch {
                    $Global:Error.RemoveAt(0)

                    Write-Error -ErrorRecord (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("Unable to gather data from system. Reason: $( $_.Exception.GetBaseException().Message )", $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $ClusterName))

                    continue
                }

                foreach ($vs in ($vservers | Where-Object { $_ })) {
                    Write-Verbose -Message "Gathering data for Vserver: $( $vs.VserverName )"

                    try {
                        $functionStartTime = [datetime]::Now

                        # Workaround for background jobs/threads failing with SSL cert errors
                        <#if ($PSVersionTable.PSVersion -gt [version]::Parse('7.0') -and [string]::IsNullOrWhiteSpace($Host.Runspace.Name) -or $Host.Runspace.Name -eq 'RemoteHost') {
                            $controller = Connect-NtapStorageSystem @params -NcController -ErrorAction Stop

                            $controller.Vserver = $null
                        }#>

                        $ntapData = Get-VserverData -VserverContext $vs.VserverName -Controller $controller

                        $ntapData

                        $statsParams = @{
                            FunctionPSCallStack = $(
                                if ($PSCmdlet.MyInvocation.InvocationName -ne $PSCmdlet.MyInvocation.MyCommand) {
                                    Get-PSCallStack | Select-Object -Skip 1
                                }
                                else {
                                    Get-PSCallStack
                                }
                            )
                            TargetDetails       = [pscustomobject]@{
                                SystemUid     = [string]::Join('_', $ntapData.Cluster.Info.ClusterUuid, $ntapData.Vserver.Info.Uuid)
                                SystemType    = $ntapData.pstypenames[0]
                                SystemVersion = ([regex]::Split(([regex]::Split($ntapData.System.VersionInfo.Version, 'Release\s(?:\w+[__])?')[1]), '\s+'))[0].Replace(':', '')
                            }
                            ElapsedTicks        = ([datetime]::Now - $functionStartTime).Ticks
                            ExcludeParameter    = 'ClusterName', 'Name'
                        }

                        Write-Statistics @statsParams
                    }
                    catch {
                        Write-Error -ErrorRecord $_
                    }
                }
            }
            'ControllerObject' {
                $controller = $AbstractController

                Write-Verbose -Message "Validating connectivity to system: $( $controller.Name )"

                try {
                    if ($controller.GetType().FullName -ne 'NetApp.Ontapi.Filer.C.NcController') {
                        throw 'Incorrect controller type specified for the Get-NtapVserverData function.'
                    }

                    $controller.Vserver = $null

                    if ($controller.Protocol -like 'HTTP*' -and $PSCmdlet.MyInvocation.BoundParameters.ContainsKey('Timeout')) {
                        $controller.TimeoutMsec = $Timeout
                    }

                    if (-not (Test-NtapStorageSystemConnection -Controller $controller -DisableLifCheck:$DisableLifCheck.IsPresent -ErrorAction Stop)) {
                        throw 'Unspecified error.'
                    }

                    Write-Verbose -Message "Valid connection established to system: $( $controller.Name )"

                    try {
                        Write-Verbose -Message '- Gathering list of SVMs -'

                        $allVservers = Get-NcVserver -Controller $controller -ErrorAction Stop | Where-Object { $_.VserverType -in @( 'data', 'cluster' ) }

                        if ($Name) {
                            $diffObject = Compare-Object -ReferenceObject $allVservers.VserverName -DifferenceObject $Name -IncludeEqual

                            ($diffObject | Where-Object { $_.SideIndicator -eq '=>' }).InputObject | Where-Object { $_ } | ForEach-Object {
                                Show-ErrorMessage -Message "Vserver does not exist in cluster: $_"
                            }

                            $vservers = $allVservers | Where-Object { $_.VserverName -cin ($diffObject | Where-Object { $_.SideIndicator -eq '==' }).InputObject }
                        }
                        else {
                            $vservers = $allVservers
                        }
                    }
                    catch {
                        $Global:Error.RemoveAt(0)

                        Write-Error -ErrorRecord (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("Unable to gather SVMs from system. Reason: $( $_.Exception.GetBaseException().Message )", $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $controller.Name))

                        continue
                    }
                }
                catch {
                    $Global:Error.RemoveAt(0)

                    Write-Error -ErrorRecord (New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ("Unable to gather data from system. Reason: $( $_.Exception.GetBaseException().Message )", $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $controller.Name))

                    continue
                }

                foreach ($vs in ($vservers | Where-Object { $_ })) {
                    Write-Verbose -Message "Gathering data for Vserver: $( $vs.VserverName )"

                    try {
                        $functionStartTime = [datetime]::Now

                        # Workaround for background jobs/threads failing with SSL cert errors
                        <#if ($PSVersionTable.PSVersion -gt [version]::Parse('7.0') -and [string]::IsNullOrWhiteSpace($Host.Runspace.Name) -or $Host.Runspace.Name -eq 'RemoteHost') {
                            $userName = $(
                                if ($controller.Credentials.Domain) {
                                    [string]::Format('{0}\{1}', $controller.Credentials.Domain, $controller.Credentials.UserName)
                                }
                                else {
                                    $controller.Credentials.UserName
                                }
                            )

                            $controller = Connect-NcController -Name $controller.name -Port $controller.Port -Credential ([System.Management.Automation.PSCredential]::new($userName, $controller.Credentials.SecurePassword)) -Transient

                            if ($controller.Protocol -like 'HTTP*' -and $PSCmdlet.MyInvocation.BoundParameters.ContainsKey('Timeout')) {
                                $controller.TimeoutMsec = $Timeout
                            }

                            $controller.Vserver = $null
                        }#>

                        $ntapData = Get-VserverData -VserverContext $vs.VserverName -Controller $controller

                        $ntapData

                        $statsParams = @{
                            FunctionPSCallStack  = $(
                                if ($PSCmdlet.MyInvocation.InvocationName -ne $PSCmdlet.MyInvocation.MyCommand) {
                                    Get-PSCallStack | Select-Object -Skip 1
                                }
                                else {
                                    Get-PSCallStack
                                }
                            )
                            TargetDetails        = [pscustomobject]@{
                                SystemUid     = [string]::Join('_', $ntapData.Cluster.Info.ClusterUuid, $ntapData.Vserver.Info.Uuid)
                                SystemType    = $ntapData.pstypenames[0]
                                SystemVersion = ([regex]::Split(([regex]::Split($ntapData.System.VersionInfo.Version, 'Release\s(?:\w+[__])?')[1]), '\s+'))[0].Replace(':', '')
                            }
                            ElapsedTicks         = ([datetime]::Now - $functionStartTime).Ticks
                            ParameterToSetAsBool = 'AbstractController'
                        }

                        Write-Statistics @statsParams
                    }
                    catch {
                        Write-Error -ErrorRecord $_
                    }
                }
            }
        }
    }
}

function Get-VserverData {
    <#
        .NOTES
            BURTS
            : 640487 - gpo-get-iter - get-iters which are associated with vservers and callable from cluster context should return the vserver name
            :   Workaround: Loop through each vserver, calling the associated cmdlet
            :   Fixed: SN.0
            : 512073 - Need nfs-status-get-iter in cluster scope (nfs-service-get-iter)
            :   Workaround: Loop through each vserver, calling the associated cmdlet
            :   Fixed: RR.1
    #>

    [CmdletBinding()]
    param (
        [Parameter(Mandatory)]
        [ValidateNotNullOrEmpty()]
        [NetApp.Ontapi.Filer.C.NcController]$Controller,

        [Parameter(Mandatory)]
        [ValidateNotNullOrEmpty()]
        [string]$VserverContext
    )

    $systemData = [pscustomobject]@{ }

    $systemData.pstypenames.Insert(
        0, $(
            if ($Controller.Credentials.UserName -eq 'fsxadmin') {
                'NetAppDocs.NcVserver.FSxOntap.Data'
            }
            else {
                'NetAppDocs.NcVserver.Data'
            }
        )
    )

    Write-Verbose -Message '- Capturing module version -'
    $systemData | Add-Member -MemberType NoteProperty -Name NetAppDocsVersion (Get-NtapDocsVersion -AsVersion)

    Write-Verbose -Message '- Capturing PowerShell version -'
    $systemData | Add-Member -MemberType NoteProperty -Name PowerShellVersion -Value $PSVersionTable.PSVersion

    $NetAppDocsError = @( )

    try {
        #region System Information

        ##Unused Cmdlet: Get-NcSystemUserCapability - Not needed for build docs
        ##Unused Cmdlet: Get-NcSystemVersion - Info in Get-NcSystemVersionInfo

        Write-Verbose -Message 'Calling Get-NcSystemApi'
        $useSystemCli = $null -ne (Get-NcSystemApi -Name 'system-cli*' -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message '- Gathering basic system information -'
        $systemData | Add-Member -MemberType NoteProperty -Name System -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcSystemVersionInfo'
        $systemData.System | Add-Member -MemberType NoteProperty -Name VersionInfo -Value (Get-NcSystemVersionInfo -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcSystemOntapiVersion'
        $systemData.System | Add-Member -MemberType NoteProperty -Name OntapiVersion -Value ([version]::Parse((Get-NcSystemOntapiVersion -Controller $Controller -ErrorVariable +NetAppDocsError).ToString()))

        #endregion System Information

        #region Timezone Information

        Write-Verbose -Message 'Calling Get-NcTimezone'
        $systemData | Add-Member -MemberType NoteProperty -Name Timezone -Value (Get-NcTimezone -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion Timezone Information

        #region Node Information

        if ($Controller.Credentials.UserName -ne 'fsxadmin') {
            Write-Verbose -Message '- Gathering node information-'

            Write-Verbose -Message 'Calling Get-NcNode'
            $systemData | Add-Member -MemberType NoteProperty -Name Node -Value (Get-NcNode -Controller $Controller -ErrorVariable +NetAppDocsError | Where-Object { -not [string]::IsNullOrWhiteSpace($_.Node) })
        }

        #endregion Node Information

        #region Cluster Information

        Write-Verbose -Message '- Gathering cluster information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Cluster -Value ([pscustomobject]@{ })

        $systemData.Cluster | Add-Member -MemberType NoteProperty -Name Info -Value $(
            if ($Controller.Credentials.UserName -eq 'fsxadmin') {
                Write-Verbose -Message 'Invoke-NcSsh (cluster identity show (priv diag))'
                Invoke-NcSsh -Command 'set -privilege diagnostic -confirmations off; set -showseparator ~; set -showallfields true; cluster identity show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText | Select-Object -Property RdbUuid, @{
                    Name       = 'ClusterName'
                    Expression = { $_.Name }
                }, @{
                    Name       = 'ClusterUuid'
                    Expression = { $_.Uuid }
                }, @{
                    Name       = 'ClusterSerialNumber'
                    Expression = { $_.SerialNumber }
                }, @{
                    Name       = 'ClusterContact'
                    Expression = { $_.Contact }
                }, @{
                    Name       = 'ClusterLocation'
                    Expression = { $_.Location }
                }
            }
            else {
                Write-Verbose -Message 'Calling Get-NcCluster'
                Get-NcCluster -Controller $Controller -ErrorVariable +NetAppDocsError
            }
        )

        #endregion Cluster Information

        #region Option Information

        Write-Verbose -Message '- Gathering options -'

        Write-Verbose -Message 'Calling Get-NcOption'
        $systemData | Add-Member -MemberType NoteProperty -Name Option -Value @( (Get-NcOption -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError) )

        #endregion Option Information

        #region Vserver Information

        ##Unused Cmdlet: Get-NcVserverAggr - (1.100+) Info in Get-NcVserver

        Write-Verbose -Message '- Gathering Vserver information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Vserver -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcVserver'
        $systemData.Vserver | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcVserver -Name $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.31')) {
            Write-Verbose -Message '- Gathering login banner/MOTD information -'

            Write-Verbose -Message 'Calling Get-NcVserverLoginBanner'
            $systemData.Vserver | Add-Member -MemberType NoteProperty -Name LoginBanner -Value (Get-NcVserverLoginBanner -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcVserverMotd'
            $systemData.Vserver | Add-Member -MemberType NoteProperty -Name Motd -Value (Get-NcVserverMotd -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Vserver Information

        #region Vserver Peer Information

        ##Unused Cmdlet: Get-NcVserverPeerTransition - (1.20+) Not needed for build docs (Vserver.Peer.Transition)
        ##Unused Cmdlet: Get-NcVserverPeerPermission - (1.120+) Not currently needed for build docs (Vserver.Peer.Permission)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Gathering Vserver Peer information -'
            $systemData.Vserver | Add-Member -MemberType NoteProperty -Name Peer -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcVserverPeer'
            $systemData.Vserver.Peer | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcVserverPeer -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Vserver Peer Information

        #region Vserver Migration Information

        ##Unused Cmdlet: Get-NcVserverMigrate - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcVserverMigrateVolume - (1.140+) Not needed for build docs

        #endregion Vserver Migration Information

        #region Volume Configuration

        ##Unused Cmdlet: Get-NcVolAutobalance - (1.30+) Not currently needed (Infinite Volumes)
        ##Unused Cmdlet: Get-NcVolAutosize - Info exists in Get-NcVol
        ##Unused Cmdlet: Get-NcVolCloneSplit - Not needed for build docs
        ##Unused Cmdlet: Get-NcVolCloneSplitEstimate - Not needed for build docs
        ##Unused Cmdlet: Get-NcVolContainer - Info exists in Get-NcAggr
        ##Unused Cmdlet: Get-NcVolLanguage - Info exists in Get-NcVol
        ##Unused Cmdlet: Get-NcVolLimit - Not currently needed (Infinite Volumes)
        ##Unused Cmdlet: Get-NcVolMoveTargetAggr - Not needed for build docs
        ##Unused Cmdlet: Get-NcVolRoot - Info exists in Get-NcVol (& Get-NcVserver)
        ##Unused Cmdlet: Get-NcVolSize - Info exists in Get-NcVol
        ##Unused Cmdlet: Get-NcVolStorageService - Not needed for build docs ?? (not sure what this is)
        ##Unused Cmdlet: Get-NcVolFlexgroupAggrLayout - (1.110+) Not needed for build docs ?? (API not found?)
        ##Unused Cmdlet: Get-NcVolFlexgroupExpandLimits - (1.110+) Not needed for build docs ?? (API not found?)
        ##Unused Cmdlet: Get-NcVolFlexgroupLimits - (1.110+) Not needed for build docs ??
        ##Unused Cmdlet: Get-NcVolFlexgroupResizeLimits - (1.110+) Not needed for build docs ??
        ##Unused Cmdlet: Get-NcVolOption - Not needed for build docs (Vol.Option - loop through each cluster volume)
        ##Unused Cmdlet: Get-NcVolumeEncryption - (1.130+) Not currently needed for build docs (API issue in RC1)
        ##Unused Cmdlet: Get-NcVolumeEncryptionConversion - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcVolumeEncryptionRekey - (1.140+) Not needed for build docs

        Write-Verbose -Message '- Gathering detailed volume information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Vol -Value ([pscustomobject]@{ })

        #ZAPI BUG: Not all attributes are returned unless we inititalize the template (flexgroup aggr-list specifically)
        Write-Verbose -Message 'Calling Get-NcVol'
        $volTemplate = Set-AllNcTemplateProperties -Template (Get-NcVol -Template -Fill -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
        $systemData.Vol | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcVol -Attributes $volTemplate -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcVolClone'
        $systemData.Vol | Add-Member -MemberType NoteProperty -Name CloneVol -Value (Get-NcVolClone -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Calling Get-NcVolSpace'
            $systemData.Vol | Add-Member -MemberType NoteProperty -Name Space -Value (Get-NcVolSpace -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcVolFootprint'
            $systemData.Vol | Add-Member -MemberType NoteProperty -Name Footprint -Value (Get-NcVolFootprint -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.140')) {
                $systemData.Vol | Add-Member -MemberType NoteProperty -Name TieringInfo -Value $(
                    if ($useSystemCli) {
                        Write-Verbose -Message 'Calling Invoke-NcSystemApi (vol show -fields vserver, vserver-uuid, instance-uuid, tiering-minimum-cooling-days (priv diag))'
                        (Invoke-NcSystemApi -Request "<system-cli><args><arg>set -privilege diagnostic -confirmations off;</arg><arg>set -showseparator ~;</arg><arg>vol show -fields vserver,vserver-uuid,instance-uuid,tiering-minimum-cooling-days -vserver $VserverContext</arg></args></system-cli>" -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                    }
                    else {
                        Write-Verbose -Message 'Calling Invoke-NcSsh (vol show -fields vserver, vserver-uuid, instance-uuid, tiering-minimum-cooling-days (priv diag))'
                        Invoke-NcSsh -Command "set -privilege diagnostic -confirmations off; set -showseparator ~; set -showallfields true; vol show -fields vserver,vserver-uuid,instance-uuid,tiering-minimum-cooling-days -vserver $VserverContext" -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                    }
                )
            }
        }

        #endregion Volume Configuration

        #region SIS Configuration

        Write-Verbose -Message '- Gathering SIS configuration -'
        $systemData | Add-Member -MemberType NoteProperty -Name Sis -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcSis'
        $systemData.Sis | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcSis -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcSisPolicy'
        $systemData.Sis | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-NcSisPolicy -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion SIS Configuration

        #region Snapshot Information

        ##Unused Cmdlet: Get-NcSnapshotAutoDelete - Info exists in Get-NcVol
        ##Unused Cmdlet: Get-NcSnapshotDelta - Not needed for build docs
        ##Unused Cmdlet: Get-NcSnapshotReclaimable - Not needed for build docs
        ##Unused Cmdlet: Get-NcSnapshotReserve - Info exists in Get-NcVol
        ##Unused Cmdlet: Get-NcSnapshotRestoreStatus - Not needed for build docs

        Write-Verbose -Message '- Gathering detailed Snapshot information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Snapshot -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcSnapshot'
        $snapshotTemplate = Set-AllNcTemplateProperties -Template (Get-NcSnapshot -Template -Controller $Controller -ErrorVariable +NetAppDocsError)
        $systemData.Snapshot | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcSnapshot -Attributes $snapshotTemplate -Vserver $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcSnapshotPolicy'
        $Controller.Vserver = $VserverContext
        $systemData.Snapshot | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-NcSnapshotPolicy -Controller $Controller -ErrorVariable +NetAppDocsError)
        $Controller.Vserver = $null

        if ($systemData.System.OntapiVersion -lt [version]::Parse('1.20')) {
            $systemData.Snapshot.Policy | Add-Member -MemberType NoteProperty -Name VserverName -Value $vserverContext -ErrorAction Ignore -Force
        }

        #endregion Snapshot Information

        #region Qtree Configuration

        Write-Verbose -Message '- Gathering qtree data -'

        Write-Verbose -Message 'Calling Get-NcQtree'
        $systemData | Add-Member -MemberType NoteProperty -Name Qtree -Value (Get-NcQtree -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion Qtree Configuration

        #region User Data

        Write-Verbose -Message '- Gathering user data -'

        $systemData | Add-Member -MemberType NoteProperty -Name User -Value $(
            if ($Controller.Credentials.UserName -eq 'fsxadmin') {
                Write-Verbose -Message 'Invoke-NcSsh (security login show)'
                Invoke-NcSsh -Command "set -showseparator ~; set -showallfields true; security login show -vserver $VserverContext" -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText | Select-Object -ExcludeProperty Role, IsAccountLocked -Property *, @{
                    Name       = 'RoleName'
                    Expression = { $_.Role }
                }, @{
                    Name       = 'IsLocked'
                    Expression = { $_.IsAccountLocked -eq 'yes' }
                }
            }
            else {
                Write-Verbose -Message 'Calling Get-NcUser'
                Get-NcUser -Vserver $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError
            }
        )

        #endregion User Data

        #region Role Data

        ##Unused Cmdlet: Get-NcRoleConfig - Using equivalent CLI command due to ZAPI issues not returning all properties

        Write-Verbose -Message '- Gathering role data -'
        $systemData | Add-Member -MemberType NoteProperty -Name Role -Value ([pscustomobject]@{ })


        $systemData.Role | Add-Member -MemberType NoteProperty -Name Info -Value $(
            if ($Controller.Credentials.UserName -eq 'fsxadmin') {
                Write-Verbose -Message 'Invoke-NcSsh (security login role show)'
                Invoke-NcSsh -Command "set -showseparator ~; set -showallfields true; security login role show -vserver $VserverContext" -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText | Select-Object -Property Vserver, @{
                    Name       = 'RoleName'
                    Expression = { $_.Role }
                }, @{
                    Name       = 'AccessLevel'
                    Expression = { $_.Access }
                }, @{
                    Name       = 'CommandDirectoryName'
                    Expression = { $_.Cmddirname }
                }, @{
                    Name       = 'RoleQuery'
                    Expression = { $_.Query }
                }
            }
            else {
                Write-Verbose -Message 'Calling Get-NcRole'
                Get-NcRole -Vserver $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError
            }
        )

        #CLI BURT: priv diag needed for this CLI call only because admin/advanced does not add the appropriate newline between records
        $securityRoleConfigCliOutput = $(
            if ($Controller.Credentials.UserName -eq 'fsxadmin' -or -not $useSystemCli) {
                Write-Verbose -Message 'Invoke-NcSsh (security login role config show (priv diag))'
                Invoke-NcSsh -Command "set -privilege diagnostic -confirmations off; set -showseparator ~; set -showallfields true; security login role config show -vserver $VserverContext" -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
            }
            else {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (security login role config show (priv diag))'
                (Invoke-NcSystemApi -Request "<system-cli><args><arg>set -privilege diagnostic -confirmations off;</arg><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>security login role config show -vserver $VserverContext</arg></args></system-cli>" -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
            }
        )

        $systemData.Role | Add-Member -MemberType NoteProperty -Name Config -Value (
            $securityRoleConfigCliOutput | Select-Object -Property Vserver, @{
                Name       = 'RoleName'
                Expression = { $_.Role }
            }, @{
                Name       = 'MinUsernameSize'
                Expression = { $_.UsernameMinlength }
            }, @{
                Name       = 'RequireUsernameAlphaNumeric'
                Expression = { $_.UsernameAlphaNum -eq 'enabled' }
            }, @{
                Name       = 'MinPasswordSize'
                Expression = { $_.PasswdMinlength }
            }, @{
                Name       = 'RequirePasswordAlphaNumeric'
                Expression = { $_.PasswdAlphanum -eq 'enabled' }
            }, @{
                Name       = 'MinPasswdSpecialchar'
                Expression = { $_.PasswdMinSpecialChars }
            }, @{
                Name       = 'PasswordExpirationDuration'
                Expression = { $_.PasswdExpiryTime }
            }, @{
                Name       = 'RequireInitialPasswordUpdate'
                Expression = { $_.RequireInitialPasswdUpdate -eq 'enabled' }
            }, @{
                Name       = 'LastPasswordsDisallowedCount'
                Expression = { $_.DisallowedReuse }
            }, @{
                Name       = 'ChangePasswordDurationInDays'
                Expression = { $_.ChangeDelay }
            }, MaxFailedLoginAttempts, LockoutDuration, DelayAfterFailedLogin, PasswdMinLowercaseChars, PasswdMinUppercaseChars, PasswdMinDigits, PasswdExpiryWarnTime, AccountExpiryTime, AccountInactiveLimit
        )

        #endregion Role Data

        #region SNMP User Data

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.100') -and $Controller.Credentials.UserName -ne 'fsxadmin') {
            Write-Verbose -Message '- Gathering SNMP user data -'

            $systemData | Add-Member -MemberType NoteProperty -Name SnmpUser -Value $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (security snmpusers)'
                    (Invoke-NcSystemApi -Request '<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>security snmpusers -vserver $VserverContext</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (security snmpusers)'
                    Invoke-NcSsh -Command "set -showseparator ~; set -showallfields true; security snmpusers -vserver $VserverContext" -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            )
        }

        #endregion SNMP User Data

        #region Network Details

        ##Unused Cmdlet: Get-NcNetArpEntry - (1.30+) Not sure if needed for build docs
        ##Unused Cmdlet: Get-NcNetPort - ZapiDoc issue, not vserver level apis
        ##Unused Cmdlet: Get-NcNetPortIfgrp - ZapiDoc issue, not vserver level apis
        ##Unused Cmdlet: Get-NcNetActiveRoutes - (1.30+) Not needed for build docs (Net.ActiveRoutes)
        ##Unused Cmdlet: Get-NcNetNdpNeighbor - (1.30+) Not needed for build docs (Net.NdpNeighbor)

        Write-Verbose -Message '- Gathering network details -'
        $systemData | Add-Member -MemberType NoteProperty -Name Net -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcNetInterface'
        $systemData.Net | Add-Member -MemberType NoteProperty -Name Interface -Value (Get-NcNetInterface -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)


        $netInterfaceCli = $(
            if ($Controller.Credentials.UserName -eq 'fsxadmin' -or -not $useSystemCli) {
                Write-Verbose -Message 'Invoke-NcSsh (network interface show (priv advanced))'
                Invoke-NcSsh -Command "set -privilege advanced -confirmations off; set -showseparator ~; set -showallfields true; network interface show -vserver $VserverContext" -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
            }
            else {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (network interface show (priv advanced))'
                (Invoke-NcSystemApi -Request "<system-cli><priv>advanced</priv><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>network interface show -vserver $VserverContext</arg></args></system-cli>" -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
            }
        )

        foreach ($lif in ($systemData.Net.Interface | Where-Object { $_ })) {
            $netInterface = $netInterfaceCli | Where-Object { $_.Vserver -ceq $lif.Vserver -and $_.Lif -ceq $lif.InterfaceName }

            $lif | Add-Member -MemberType NoteProperty -Name NumericId -Value $netinterface.NumericId -Force
            $lif | Add-Member -MemberType NoteProperty -Name ServicePolicy -Value $netinterface.ServicePolicy -Force
        }

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.31')) {
            Write-Verbose -Message 'Calling Get-NcNetDdns'
            $systemData.Net | Add-Member -MemberType NoteProperty -Name Ddns -Value (Get-NcNetDdns -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        Write-Verbose -Message 'Calling Get-NcNetDns'
        $systemData.Net | Add-Member -MemberType NoteProperty -Name Dns -Value (Get-NcNetDns -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcNetDnsHost'
        $systemData.Net | Add-Member -MemberType NoteProperty -Name DnsHost -Value (Get-NcNetDnsHost -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Identifying Net.DnsZone data'
        $systemData.Net | Add-Member -MemberType NoteProperty -Name DnsZone -Value ($netInterfaceCli | Select-Object -Property Vserver, Lif, AllowLbMigrate, LbWeight)

        $systemData.Net | Add-Member -MemberType NoteProperty -Name FailoverTarget -Value $(
            if ($useSystemCli) {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (network interface show -failover)'
                (Invoke-NcSystemApi -Request "<system-cli><args><arg>network interface show -failover -vserver $VserverContext</arg></args></system-cli>" -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-FailoverTarget
            }
            else {
                Write-Verbose -Message 'Calling Invoke-NcSsh (network interface show -failover)'
                Invoke-NcSsh -Command "network interface show -failover -vserver $VserverContext" -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-FailoverTarget
            }
        )

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message 'Calling Get-NcNetFailoverGroup'
            $systemData.Net | Add-Member -MemberType NoteProperty -Name FailoverGroup -Value (Get-NcNetFailoverGroup -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcNetRoute'
            $systemData.Net | Add-Member -MemberType NoteProperty -Name Route -Value (Get-NcNetRoute -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcNetRouteLif'
            $systemData.Net | Add-Member -MemberType NoteProperty -Name RouteLif -Value (Get-NcNetRouteLif -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcNetFirewallPolicy'
            $systemData.Net | Add-Member -MemberType NoteProperty -Name FirewallPolicy -Value (Get-NcNetFirewallPolicy -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.150')) {
                $systemData.Net | Add-Member -MemberType NoteProperty -Name ServicePolicy -Value ([pscustomobject]@{ })

                $systemData.Net.ServicePolicy | Add-Member -MemberType NoteProperty -Name Info -Value $(
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (net-interface-service-policy-get-iter)'
                    (Invoke-NcSystemApi -Request "<net-interface-service-policy-get-iter><query><net-interface-service-policy-info><vserver>$VserverContext</vserver></net-interface-service-policy-info></query></net-interface-service-policy-get-iter>" -Controller $Controller -ErrorVariable +NetAppDocsError).results.'attributes-list'.'net-interface-service-policy-info' | Convert-GenericXmlNode
                )

                if ($Controller.Credential.UserName -ne 'fsxadmin') {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (net-interface-service-policy-entry-get-iter)'
                    $systemData.Net.ServicePolicy | Add-Member -MemberType NoteProperty -Name Entry -Value ((Invoke-NcSystemApi -Request "<net-interface-service-policy-entry-get-iter><query><net-interface-service-policy-entry-info><vserver>$VserverContext</vserver></net-interface-service-policy-entry-info></query></net-interface-service-policy-entry-get-iter>" -Controller $Controller -ErrorVariable +NetAppDocsError).results.'attributes-list'.'net-interface-service-policy-entry-info' | Convert-GenericXmlNode)
                }
            }
        }
        else {
            if ($Controller.Credential.UserName -ne 'fsxadmin') {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (network routing-groups show)'
                $systemData.Net | Add-Member -MemberType NoteProperty -Name RoutingGroup -Value ((Invoke-NcSystemApi -Request '<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>network routing-groups show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText)
            }

            Write-Verbose -Message 'Calling Get-NcNetRoutingGroupRoute'
            $systemData.Net | Add-Member -MemberType NoteProperty -Name RoutingGroupRoute -Value (Get-NcNetRoutingGroupRoute -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Network Details

        #region Clone

        ##Unused Cmdlet: Get-NcCloneAutodelete - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcCloneDeletion - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcCloneLoadSplit - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcCloneToken - (1.30+) Not needed for build docs

        #endregion Clone

        #region Export Policy Configuration

        ##Unused Cmdlet: Get-NcExportAccessCacheConfig - (1.100+) Not needed for build docs (Export.AccessCacheConfig)

        Write-Verbose -Message '- Gathering export policy information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Export -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcExportPolicy'
        $systemData.Export | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-NcExportPolicy -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcExportRule'
        $systemData.Export | Add-Member -MemberType NoteProperty -Name Rule -Value (Get-NcExportRule -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion Export Policy Configuration

        #region FC/FCP Configuration

        ##Unused Cmdlet: Get-NcFcpNodeName - Info exists in Get-NcFcpInterface
        ##Unused Cmdlet: Get-NcFcpPortName - Info exists in Get-NcFcpInterface
        ##Unused Cmdlet: Get-NcFcpPingIgroup - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFcpPingInitiator - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFcpPortNameAlias - Not needed for build docs (Fcp.PortNameAlias)
        ##Unused Cmdlet: Get-NcFcpNameServer = (1.120+) Not currently needed for build docs (Fcp.NameServer)
        ##Unused Cmdlet: Get-NcFcpTopology = (1.120+) Not currently needed for build docs (Fcp.Topology)

        if ($Controller.Credentials.UserName -ne 'fsxadmin') {
            Write-Verbose -Message '- Gathering FCP configuration data -'
            $systemData | Add-Member -MemberType NoteProperty -Name Fcp -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcFcpService'
            $systemData.Fcp | Add-Member -MemberType NoteProperty -Name Service -Value (Get-NcFcpService -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcFcpInitiator'
            $systemData.Fcp | Add-Member -MemberType NoteProperty -Name Initiator -Value (Get-NcFcpInitiator -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcFcpInterface'
            $systemData.Fcp | Add-Member -MemberType NoteProperty -Name Interface -Value (Get-NcFcpInterface -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion FC/FCP Configuration

        #region iSCSI Configuration

        ##Unused Cmdlet: Get-NcIscsiConnection - Not needed for build docs
        ##Unused Cmdlet: Get-NcIscsiInitiatorAuth - Not needed for build docs ??
        ##Unused Cmdlet: Get-NcIscsiNodeName - Info in Get-NcIscsiService
        ##Unused Cmdlet: Get-NcIscsiSession - Not needed for build docs
        ##Unused Cmdlet: Get-NcIscsiStatistics - Not needed for build docs
        ##Unused Cmdlet: Get-NcIscsiTargetalias - Info in Get-NcIscsiService
        ##Unused Cmdlet: Get-NcIscsiInterfaceAccess - Not needed for build docs (Iscsi.Interface.Access)

        Write-Verbose -Message '- Gathering iSCSI configuration data -'
        $systemData | Add-Member -MemberType NoteProperty -Name Iscsi -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcIscsiService'
        $systemData.Iscsi | Add-Member -MemberType NoteProperty -Name Service -Value (Get-NcIscsiService -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        $systemData.Iscsi | Add-Member -MemberType NoteProperty -Name Interface -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcIscsiInterface'
        $systemData.Iscsi.Interface | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcIscsiInterface -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcIscsiInitiator'
        $systemData.Iscsi | Add-Member -MemberType NoteProperty -Name Initiator -Value (Get-NcIscsiInitiator -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcIscsiTargetPortalGroup'
        $systemData.Iscsi | Add-Member -MemberType NoteProperty -Name TargetPortalGroup -Value (Get-NcIscsiTargetPortalGroup -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion iSCSI configuration

        #region iSNS Configuration

        ##Unused Cmdlet: Get-NcIsns - Not needed for build docs ??

        #endregion iSNS Configuration

        #region LUN Information

        ##Unused Cmdlet: Get-NcLunAlignment - Info exists in Get-NcLun
        ##Unused Cmdlet: Get-NcLunAttribute - Info exists in Get-NcLun
        ##Unused Cmdlet: Get-NcLunByIgroup - Not needed
        ##Unused Cmdlet: Get-NcLunGeometry - Not needed for build docs
        ##Unused Cmdlet: Get-NcLunInquiry - Not needed for build docs
        ##Unused Cmdlet: Get-NcLunMapByInitiator - Not needed
        ##Unused Cmdlet: Get-NcLunMaxSize - Not needed for build docs
        ##Unused Cmdlet: Get-NcLunMinSize - Not needed for build docs
        ##Unused Cmdlet: Get-NcLunPersistentReservation - Not needed for build docs
        ##Unused Cmdlet: Get-NcLunStatistics - Not needed for build docs
        ##Unused Cmdlet: Get-NcLunSelect - Not sure if this is needed or not
        ##Unused Cmdlet: Get-NcLunSignature - Not sure if this is needed or not
        ##Unused Cmdlet: Get-NcLunSerialNumber - Info exists in Get-NcLun
        ##Unused Cmdlet: Get-NcLunCopy - (1.30+) Not needed for build docs (Lun.Copy)
        ##Unused Cmdlet: Get-NcLunMove - (1.30+) Not needed for build docs (Lun.Move)

        Write-Verbose -Message '- Gathering LUN information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Lun -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcLun'
        $systemData.Lun | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcLun -VserverContext $VserverContext -Controller $Controller -WarningAction SilentlyContinue -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcLunMap'
        $systemData.Lun | Add-Member -MemberType NoteProperty -Name Map -Value (Get-NcLunMap -VserverContext $VserverContext -Controller $Controller -WarningAction SilentlyContinue -ErrorVariable +NetAppDocsError)

        #endregion LUN Information

        #region iGroup Configuration

        Write-Verbose -Message '- Gathering iGroup information -'

        Write-Verbose -Message 'Calling Get-NcIgroup'
        $systemData | Add-Member -MemberType NoteProperty -Name Igroup -Value (Get-NcIgroup -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        $igroupInitiatorDetails = $(
            if ($useSystemCli) {
                Write-Verbose -Message 'Calling Invoke-NcSystemApi (lun igroup show -fields init-details (priv diag))'
                (Invoke-NcSystemApi -Request "<system-cli><args><arg>set -privilege diagnostic -confirmations off;</arg><arg>set -showseparator ~;</arg><arg>lun igroup show -Vserver $VserverContext -fields init-details</arg></args></system-cli>" -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
            }
            else {
                Write-Verbose -Message 'Calling Invoke-NcSsh (lun igroup show -fields init-details (priv diag))'
                Invoke-NcSsh -Command "set -privilege diagnostic -confirmations off; set -showseparator ~; set -showallfields true; lun igroup show -Vserver $VserverContext -fields init-details" -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
            }
        )

        foreach ($igroup in ($systemData.Igroup | Where-Object { $_.Initiators })) {
            $initiatorDetails = $igroupInitiatorDetails | Where-Object { $_.Igroup -ceq $igroup.InitiatorGroupName } | Select-Object -ExcludeProperty InitDetails -Property *, @{
                Name       = 'Initiators'
                Expression = {
                    @(
                        $_.InitDetails -replace '"' -split '\s*,\s*' | ForEach-Object {
                            [pscustomobject]@{
                                InitiatorName = $_ -replace '\s+\(.+\)'
                                IsLoggedIn    = $_ -like '*(logged in)'
                            }
                        }
                    )
                }
            }

            foreach ($initiator in ($igroup.Initiators | Where-Object { -not [string]::IsNullOrWhiteSpace($_.InitiatorName) })) {
                $initiator | Add-Member -MemberType NoteProperty -Name IsLoggedIn -Value $(
                    if ($initiatorDetails) {
                        ($initiatorDetails.Initiators | Where-Object { $_.InitiatorName -ceq $initiator.InitiatorName }).IsLoggedIn
                    }
                )
            }
        }

        #endregion iGroup Information

        #region PortSet Configuration

        Write-Verbose -Message '- Gathering PortSet information -'

        Write-Verbose -Message 'Calling Get-NcPortSet'
        $systemData | Add-Member -MemberType NoteProperty -Name PortSet -Value (Get-NcPortset -Vserver $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion PortSet Configuration

        #region Active Directory

        ##Unused Cmdlet: Get-NcActiveDirectoryAccount - (1.21+) Not needed for build docs (ActiveDirectoryAccount)

        #endregion Active Directory

        #region CIFS Configuration

        ##Unused Cmdlet: Get-NcCifsConnection - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcCifsNbtStat - Not needed for build docs
        ##Unused Cmdlet: Get-NcCifsSession - Not needed for build docs
        ##Unused Cmdlet: Get-NcCifsSessionFile - Not needed for build docs
        ##Unused Cmdlet: Get-NcCifsShadowCopyEmsMessage - Not needed for build docs
        ##Unused Cmdlet: Get-NcCifsUserHomeDirectory - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcCifsUserAndGroupMember - (1.100+) Not sure what this does differently
        ##Unused Cmdlet: Get-NcCifsUserAndUserMembership - (1.100+) Not sure what this does differently
        ##Unused Cmdlet: Get-NcCifsDomainTrust - (1.21+) Not needed for build docs (Cifs.DomainTrust)
        ##Unused Cmdlet: Get-Get-NcCifsTrustedDomain - (1.21+) Not needed for build docs (Cifs.TrustedDomain)
        ##Unused Cmdlet: Get-NcCifsCharacterMapping - (1.30+) Not needed for build docs (Cifs.CharacterMapping)
        ##Unused Cmdlet: Get-NcCifsDomainPasswordSchedule - (1.30+) Not needed for build docs (Cifs.DomainPasswordSchedule)
        ##Unused Cmdlet: Get-NcCifsDomainServer - Caused timeout issues in some cases (Cifs.DomainServer)
        ##Unused Cmdlet: Get-NcCifsPreferredDcStatus - (1.150+) Not needed for build docs
        ##Unused Cmdlet: Get-NcCifsServerStatus - (1.150+) Not needed for build docs

        Write-Verbose -Message '- Gathering CIFS configuration information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Cifs -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcCifsServer'
        $systemData.Cifs | Add-Member -MemberType NoteProperty -Name Server -Value (Get-NcCifsServer -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcCifsSymlink'
        $systemData.Cifs | Add-Member -MemberType NoteProperty -Name Symlink -Value (Get-NcCifsSymlink -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcCifsShare'
        $systemData.Cifs | Add-Member -MemberType NoteProperty -Name Share -Value (Get-NcCifsShare -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcCifsShareAcl'
        $systemData.Cifs | Add-Member -MemberType NoteProperty -Name ShareAcl -Value (Get-NcCifsShareAcl -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        if ($systemData.Cifs.Server) {
            Write-Verbose -Message 'Calling Get-NcCifsHomeDirectorySearchPath'
            $systemData.Cifs | Add-Member -MemberType NoteProperty -Name HomeDirectorySearchPath -Value (Get-NcCifsHomeDirectorySearchPath -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcCifsOption'
            $systemData.Cifs | Add-Member -MemberType NoteProperty -Name Option -Value (Get-NcCifsOption -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
            if ($systemData.System.OntapiVersion -lt [version]::Parse('1.20') -and $systemData.Cifs.Option) {
                $systemData.Cifs.Option | Add-Member -MemberType NoteProperty -Name Vserver -Value $systemData.Vserver.Info.VserverName -ErrorAction Ignore -Force
            }

            $Controller.Vserver = $VserverContext
            Write-Verbose -Message 'Calling Get-NcCifsPreferredDomainController'
            $systemData.Cifs | Add-Member -MemberType NoteProperty -Name PreferredDomainController -Value (Get-NcCifsPreferredDomainController -Controller $Controller -ErrorVariable +NetAppDocsError)
            $Controller.Vserver = $null

            Write-Verbose -Message 'Calling Get-NcCifsSecurity'
            $systemData.Cifs | Add-Member -MemberType NoteProperty -Name Security -Value (Get-NcCifsSecurity -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
            if ($systemData.System.OntapiVersion -lt [version]::Parse('1.20') -and $systemData.Cifs.Security) {
                $systemData.Cifs.Security | Add-Member -MemberType NoteProperty -Name Vserver -Value $systemData.Vserver.Info.VserverName -ErrorAction Ignore -Force
            }
        }

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Gathering CIFS local users and groups'
            $systemData.Cifs | Add-Member -MemberType NoteProperty -Name UserAndGroup -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcCifsLocalUser'
            $systemData.Cifs.UserAndGroup | Add-Member -MemberType NoteProperty -Name LocalUser -Value (Get-NcCifsLocalUser -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcCifsLocalGroup'
            $systemData.Cifs.UserAndGroup | Add-Member -MemberType NoteProperty -Name LocalGroup -Value (Get-NcCifsLocalGroup -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcCifsLocalGroupMember'
            $systemData.Cifs.UserAndGroup | Add-Member -MemberType NoteProperty -Name LocalGroupMember -Value (Get-NcCifsLocalGroupMember -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcCifsPrivilege'
            $systemData.Cifs.UserAndGroup | Add-Member -MemberType NoteProperty -Name Privilege -Value (Get-NcCifsPrivilege -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            #This will error if Vserver does not have a CIFS server running
            if ($systemData.Cifs.Server) {
                Write-Verbose -Message 'Calling Get-NcCifsBranchCache'
                $systemData.Cifs | Add-Member -MemberType NoteProperty -Name BranchCache -Value (Get-NcCifsBranchCache -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
            }

            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
                Write-Verbose -Message 'Calling Get-NcCifsHomeDirectoryConfig'
                $systemData.Cifs | Add-Member -MemberType NoteProperty -Name HomeDirectoryConfig -Value (Get-NcCifsHomeDirectoryConfig -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

                if ($systemData.System.OntapiVersion -ge [version]::Parse('1.130')) {
                    $systemData.Cifs | Add-Member -MemberType NoteProperty -Name DomainServerDiscoveryMode -Value $(
                        if ($useSystemCli) {
                            Write-Verbose -Message 'Calling Invoke-NcSystemApi (cifs-domain-server-discovery-mode-get-iter)'
                            (Invoke-NcSystemApi -Request "<cifs-domain-server-discovery-mode-get-iter><query><cifs-domain-server-discovery-mode-info><vserver>$VserverContext</vserver></cifs-domain-server-discovery-mode-info></query></cifs-domain-server-discovery-mode-get-iter>" -Controller $Controller -ErrorVariable +NetAppDocsError).results.'attributes-list'.'cifs-domain-server-discovery-mode-info' | Convert-GenericXmlNode
                        }
                        else {
                            Write-Verbose -Message 'Calling Invoke-NcSsh (vserver cifs domain discovered-servers show)'
                            Invoke-NcSsh -Command "set -showseparator ~; vserver cifs domain discovered-servers show -vserver $VserverContext" -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                        }
                    )
                }
            }
        }

        #endregion CIFS Information

        #region GPO Information

        ##Unused Cmdlet: Get-NcGpoApplied - Not needed for build docs (Gpo.Applied) (BURT 640487: -lt 8.2 loop through Cifs.Server and set -VserverContext. TOOLKIT/ZAPI BURT: ZAPI throws error when GPO list is empty-try/catch.)
        ##Unused Cmdlet: Get-NcGpoGpResult - Not needed for build docs (Gpo.GpResult) (BURT 640487: -lt 8.2 loop through Cifs.Server and set -VserverContext)
        ##Unused Cmdlet: Get-NcGpoRestrictedGroupApplied - (1.30+) - Not needed for build docs (Gpo.RestrictedGroup.Applied. TOOLKIT/ZAPI BURT: ZAPI throws error when GPO list is empty-try/catch.)
        ##Unused Cmdlet: Get-NcGpoRestrictedGroupDefined - (1.30+) - Not needed for build docs (Gpo.RestrictedGroup.Defined)

        Write-Verbose -Message '- Gathering GPO configuration information -'

        $systemData.Cifs | Add-Member -MemberType NoteProperty -Name Gpo -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcGpo'
        $systemData.Cifs.Gpo | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcGpo -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion GPO Information

        #region Fpolicy Configuration

        ##Unused Cmdlet: Get-NcFpolicyPassthroughReadConnection - (1.30+) Not needed for build docs (Fpolicy.PassthroughReadConnection)
        ##Unused Cmdlet: Get-NcFpolicyStatus - (1.20+) Not needed for build docs (Fpolicy.Status)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Gathering Fpolicy configuration -'
            $systemData | Add-Member -MemberType NoteProperty -Name Fpolicy -Value ([pscustomobject]@{ })

            if ($Controller.Credentials.UserName -ne 'fsxadmin') {
                Write-Verbose -Message 'Calling Get-NcFpolicyServerStatus'
                $systemData.Fpolicy | Add-Member -MemberType NoteProperty -Name ServerStatus -Value (Get-NcFpolicyServerStatus -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
            }

            Write-Verbose -Message 'Calling Get-NcFpolicyExternalEngine'
            $systemData.Fpolicy | Add-Member -MemberType NoteProperty -Name ExternalEngine -Value (Get-NcFpolicyExternalEngine -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcFpolicyPolicy'
            $systemData.Fpolicy | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-NcFpolicyPolicy -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcFpolicyEvent'
            $systemData.Fpolicy | Add-Member -MemberType NoteProperty -Name Event -Value (Get-NcFpolicyEvent -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcFpolicyScope'
            $systemData.Fpolicy | Add-Member -MemberType NoteProperty -Name Scope -Value (Get-NcFpolicyScope -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Fpolicy Configuration

        #region File Information

        ##Unused Cmdlet: Get-NcFile - Not needed for build docs
        ##Unused Cmdlet: Get-NcFileHoles - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFileInodeInfo - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFileSpaceReservationInfo - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFileUsage - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFileUsageResult - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcSymLinkTarget - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFileFingerprint - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFileFingerprintStatus - (1.100+) Not needed for build docs

        #endregion File Information

        #region Lock Information

        ##Unused Cmdlet: Get-NcLock - (1.30+) Not needed for build docs

        #endregion Lock Information

        #region NFS

        ##Unused Cmdlet: Get-NcNfsExport - Not needed in cDOT
        ##Unused Cmdlet: Get-NcNfsSecurityFlavor - Underlying ZAPI broken, not needed for build docs

        Write-Verbose -Message '- Gathering NFS configuration information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Nfs -Value ([pscustomobject]@{ })

        $nfsError = @( )
        Write-Verbose -Message 'Calling Get-NcNfsService'
        $systemData.Nfs | Add-Member -MemberType NoteProperty -Name Service -Value (Get-NcNfsService -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NfsError)

        #Ignore the error EOBJECTNOTFOUND error from ONTAP 8.1
        if ($systemData.System.OntapiVersion -gt [version]::Parse('1.15') -or $nfsError.Reason -ne 'EOBJECTNOTFOUND') {
            $NetAppDocsError += $nfsError
        }

        #BURT 512073
        if ($systemData.System.OntapiVersion -lt [version]::Parse('1.17')) {
            $systemData.Nfs.Service | Add-Member -MemberType NoteProperty -Name Vserver -Value $systemData.Vserver.Info.VserverName -ErrorAction Ignore -Force
        }

        #endregion NFS

        #region Name Services Information (NIS/Kerberos/LDAP)

        ##Unused Cmdlet: Get-NcNameServiceDNSStatistics - (1.30+) Deprecated in 9.0
        ##Unused Cmdlet: Get-NcNameServiceHostnameFromIP - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNameServiceIPFromHostname - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNameServiceNisStatistics - (1.30+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNameserviceBoundNisServer - (1.31+) Not currently needed
        ##Unused Cmdlet: Get-NcNameServiceNisBindingDetail - (1.100+) Not needed for build docs (Nameservice.Nis.BindingDetail)
        ##Unused Cmdlet: Get-NcNameServiceUnixGroupFileOnly - (1.110+) Not needed for build docs (NameService.Unix.GroupFileOnly - loop through NFS vservers and use -VserverContext)
        ##Unused Cmdlet: Get-NcNameServiceUnixUserFileOnly - (1.110+) Not needed for build docs (NameService.Unix.UserFileOnly - loop through NFS vservers and use -VserverContext)
        ##Unused Cmdlet: Get-NcKerberosRealm - Doesn't have VServerContext or use the Controller.Vserver property so using ZAPI instead
        ##Unused Cmdlet: Get-NcNameServiceCacheGroupMembershipSettings - (1.130+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNameServiceCacheHostsSettings - (1.130+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNameServiceCacheNetgroupSettings - (1.130+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNameServiceCacheUnixGroupSettings - (1.130+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNameServiceCacheUnixUserSettings - (1.130+) Not needed for build docs

        Write-Verbose -Message '- Gathering name services information -'
        $systemData | Add-Member -MemberType NoteProperty -Name NameService -Value ([pscustomobject]@{ })
        $systemData.NameService | Add-Member -MemberType NoteProperty -Name Kerberos -Value ([pscustomobject]@{ })
        $systemData.NameService | Add-Member -MemberType NoteProperty -Name Ldap -Value ([pscustomobject]@{ })
        $systemData.NameService | Add-Member -MemberType NoteProperty -Name Nis -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcNis'
        $systemData.NameService.Nis | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcNis -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcKerberosConfig'
        $systemData.NameService.Kerberos | Add-Member -MemberType NoteProperty -Name Config -Value (Get-NcKerberosConfig -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        $realms = @( )

        Write-Verbose -Message 'Calling Invoke-NcSystemApi (kerberos-realm-get-iter)'
        $results = (Invoke-NcSystemApi -Request "<kerberos-realm-get-iter><max-records>100</max-records><query><kerberos-realm><vserver-name>$VserverContext</vserver-name></kerberos-realm></query></kerberos-realm-get-iter>" -Controller $Controller -ErrorVariable +NetAppDocsError).results
        $realms += $results.'attributes-list'.'kerberos-realm'

        if ($results.'next-tag') {
            do {
                $results = (Invoke-NcSystemApi -Request "<kerberos-realm-get-iter><max-records>100</max-records><tag>$( $results.'next-tag' )</tag><query><kerberos-realm><vserver-name>$VserverContext</vserver-name></kerberos-realm></query></kerberos-realm-get-iter>" -Controller $Controller -ErrorVariable +NetAppDocsError).results
                $realms += $results.'attributes-list'.'kerberos-realm'
            }
            while ($results.'next-tag')
        }

        $systemData.NameService.Kerberos | Add-Member -MemberType NoteProperty -Name Realm -Value ($realms | Convert-GenericXmlNode)

        Write-Verbose -Message 'Calling Get-NcLdapClient'
        $systemData.NameService.Ldap | Add-Member -MemberType NoteProperty -Name Client -Value (Get-NcLdapClient -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcLdapClientSchema'
        $systemData.NameService.Ldap | Add-Member -MemberType NoteProperty -Name ClientSchema -Value (Get-NcLdapClientSchema -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcLdapConfig'
        $systemData.NameService.Ldap | Add-Member -MemberType NoteProperty -Name Config -Value (Get-NcLdapConfig -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        if ($systemData.System.OntapiVersion -lt [version]::Parse('1.20')) {
            $systemData.NameService.Ldap.Client | Add-Member -MemberType NoteProperty -Name Vserver -Value $systemData.Vserver.Info.VserverName -ErrorAction Ignore -Force
            $systemData.NameService.Ldap.ClientSchema | Add-Member -MemberType NoteProperty -Name Vserver -Value $systemData.Vserver.Info.VserverName -ErrorAction Ignore -Force
            $systemData.NameService.Ldap.Config | Add-Member -MemberType NoteProperty -Name Vserver -Value $systemData.Vserver.Info.VserverName -ErrorAction Ignore -Force
        }

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message 'Calling Get-NcNameServiceNsSwitch'
            $systemData.NameService | Add-Member -MemberType NoteProperty -Name NsSwitch -Value (Get-NcNameServiceNsSwitch -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Name Service Information (NIS/Kerberos/LDAP)

        #region Netgroup Information

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.30')) {
            Write-Verbose -Message '- Gathering netgroup information -'
            $systemData | Add-Member -MemberType NoteProperty -Name Netgroup -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcNetgroupFile'
            $systemData.Netgroup | Add-Member -MemberType NoteProperty -Name File -Value (Get-NcNetgroupFile -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Netgroup Information

        #region Name Mapping Data

        Write-Verbose -Message '- Gathering name mapping data -'
        $systemData | Add-Member -MemberType NoteProperty -Name NameMapping -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcNameMapping'
        $systemData.NameMapping | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcNameMapping -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcNameMappingUnixGroup'
        $systemData.NameMapping | Add-Member -MemberType NoteProperty -Name UnixGroup -Value (Get-NcNameMappingUnixGroup -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcNameMappingUnixUser'
        $systemData.NameMapping | Add-Member -MemberType NoteProperty -Name UnixUser -Value (Get-NcNameMappingUnixUser -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion Name Mapping Data

        #region Group Mapping Data

        Write-Verbose -Message '- Gathering group mapping data -'

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Calling Get-NcGroupMapping'
            $systemData | Add-Member -MemberType NoteProperty -Name GroupMapping -Value (Get-NcGroupMapping -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Group Mapping Data

        #region Vscan Configuration

        ##Unused Cmdlet: Get-NcVscanConnectionStats - Not needed for build docs
        ##Unused Cmdlet: Get-NcVscanOnDemandReport - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcVscanOnDemandTask - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcVscanScannerPoolActive - (1.21+) Not needed for build docs (Vscan.ScannerPoolActive)
        ##Unused Cmdlet: Get-NcVscanEvents - (1.31+) Not needed for build docs (Vscan.Events)
        ##Unused Cmdlet: Get-NcVscanConnection - (1.31+) Not needed for SVM docs (Vscan.Connection)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.21')) {
            Write-Verbose -Message '- Gathering Vscan information -'
            $systemData | Add-Member -MemberType NoteProperty -Name Vscan -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcVscanOnAccessPolicy'
            $systemData.Vscan | Add-Member -MemberType NoteProperty -Name OnAccessPolicy -Value (Get-NcVscanOnAccessPolicy -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcVscanScannerPool'
            $systemData.Vscan | Add-Member -MemberType NoteProperty -Name ScannerPool -Value (Get-NcVscanScannerPool -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcVscanStatus'
            $systemData.Vscan | Add-Member -MemberType NoteProperty -Name Status -Value (Get-NcVscanStatus -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion Vscan Configuration

        #region Job Information

        ##Unused Cmdlet: Get-NcJobCronSchedule - Not needed, info in Get-NcJobSchedule
        ##Unused Cmdlet: Get-NcJobIntervalSchedule - Not needed, info in Get-NcJobSchedule
        ##Unused Cmdlet: Get-NcJobHistory - Not needed

        Write-Verbose -Message '- Gathering job information -'
        $systemData | Add-Member -MemberType NoteProperty -Name Job -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcJob'
        $systemData.Job | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcJob -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcJobSchedule'
        $systemData.Job | Add-Member -MemberType NoteProperty -Name Schedule -Value (
            Get-NcJobSchedule -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError | Select-Object -Property *, @{
                Name       = 'JobScheduleInfo'
                Expression = { ConvertFrom-JobScheduleDescription -JobScheduleDescription $_.JobScheduleDescription }
            }
        )

        #endregion Job Information

        #region FileDirectorySecurity Configuration

        ##Unused Cmdlet: Get-NcFileDirectorySecurity - Not needed for build docs
        ##Unused Cmdlet: Get-NcFileDirectorySecurityNtfs - Not needed for build docs
        ##Unused Cmdlet: Get-NcFileDirectorySecurityNtfsDacl - Not needed for build docs
        ##Unused Cmdlet: Get-NcFileDirectorySecurityNtfsSacl - Not needed for build docs
        ##Unused Cmdlet: Get-NcFileDirectorySecurityPolicy - (1.20+) Not needed for build docs (FileDirectorySecurity.Policy)
        ##Unused Cmdlet: Get-NcFileDirectoryEffectivePermissions - (1.120+) Not needed for build docs

        Write-Verbose -Message '- Gathering File Directory Security configuration -'
        $systemData | Add-Member -MemberType NoteProperty -Name FileDirectorySecurity -Value ([pscustomobject]@{ })

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Calling Get-NcFileDirectorySecurityPolicyTask'
            $systemData.FileDirectorySecurity | Add-Member -MemberType NoteProperty -Name PolicyTask -Value (Get-NcFileDirectorySecurityPolicyTask -Vserver $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion FileDirectorySecurity Configuration

        #region FileService Audit Configuration

        Write-Verbose -Message '- Gathering File Service Audit configuration -'
        $systemData | Add-Member -MemberType NoteProperty -Name FileServiceAudit -Value ([pscustomobject]@{ })

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Calling Get-NcFileServiceAudit'
            $systemData.FileServiceAudit | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcFileserviceAudit -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion FileService Audit Configuration

        #region FlexCache Data

        ##Unused Cmdlet: Get-NcFlexcacheUsage - (1.150+) Not needed for build docs
        ##Unused Cmdlet: Get-NcFlexcacheConnectedCache - (1.150+) Not currently needed for build docs

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Gathering FlexCache data -'
            $systemData | Add-Member -MemberType NoteProperty -Name Flexcache -Value ([pscustomobject]@{ })

            if ($systemData.System.OntapiVersion -lt [version]::Parse('1.30')) {
                Write-Verbose -Message 'Calling Get-NcFlexcache'
                $systemData.Flexcache | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcFlexcache -Vserver $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

                Write-Verbose -Message 'Calling Get-NcFlexcachePolicy'
                $systemData.Flexcache | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-NcFlexcachePolicy -Vserver $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
            }

            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.150')) {
                Write-Verbose -Message 'Calling Get-NcFlexcache'
                $systemData.Flexcache | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcFlexcache -Vserver $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
            }
        }

        #endregion FlexCache Data

        #region Snapmirror Configuration

        ##Unused Cmdlet: Get-NcSnapmirrorSnapshotOwner - Not currently needed for build docs
        ##Unused Cmdlet: Get-NcSnapmirrorHistory - (1.30+) Not currently needed for build docs
        ##Unused Cmdlet: Get-NcSnapmirrorUnprotected - (1.130+) Not currently needed for build docs
        ##Unused Cmdlet: Get-NcSnapmirrorCapabilities - (1.130+) Not a vserver command (zapidoc issue)

        Write-Verbose -Message '- Gathering Snapmirror configuration -'
        $systemData | Add-Member -MemberType NoteProperty -Name Snapmirror -Value ([pscustomobject]@{ })

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message 'Calling Get-NcSnapmirror'
            $systemData.Snapmirror | Add-Member -MemberType NoteProperty -Name Info -Value $(
                if ($systemData.System.OntapiVersion -ge [version]::Parse('1.100')) {
                    Get-NcSnapmirror -Expand -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError
                }
                else {
                    Get-NcSnapmirror -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError
                }
            )

            Write-Verbose -Message 'Calling Get-NcSnapmirrorDestination'
            $systemData.Snapmirror | Add-Member -MemberType NoteProperty -Name Destination -Value (Get-NcSnapmirrorDestination -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcSnapmirrorPolicy'
            $systemData.Snapmirror | Add-Member -MemberType NoteProperty -Name Policy -Value (Get-NcSnapmirrorPolicy -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
        }
        else {
            Write-Verbose -Message 'Calling Get-NcSnapmirror (null Vserver context)'
            $systemData.Snapmirror | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcSnapmirror -Controller $Controller -ErrorVariable +NetAppDocsError | Where-Object { $_.Vserver -ceq $systemData.Vserver.Info.VserverName })
        }

        #endregion Snapmirror Configuration

        #region Quota Configuration

        ##Unused Cmdlet: Get-NcQuotaReport - Not needed for build docs
        ##Unused Cmdlet: Get-NcQuotaPolicy - (1.31+) Not needed for build docs (Quota.Policy)
        ##Unused Cmdlet: Get-NcQuotaPolicyRuleCount - (1.100+) Not currently needed for build docs

        Write-Verbose -Message '- Gathering quota configuration -'
        $systemData | Add-Member -MemberType NoteProperty -Name Quota -Value ([pscustomobject]@{ })

        Write-Verbose -Message 'Calling Get-NcQuota'
        $systemData.Quota | Add-Member -MemberType NoteProperty -Name Rule -Value (Get-NcQuota -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        Write-Verbose -Message 'Calling Get-NcQuotaStatus'
        $systemData.Quota | Add-Member -MemberType NoteProperty -Name Status -Value (Get-NcQuotaStatus -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

        #endregion Quota Configuration

        #region Performance Configuration Information

        ##Unused Cmdlet: Get-NcPerfCounter - Not needed for build docs
        ##Unused Cmdlet: Get-NcPerfData - Not needed for build docs
        ##Unused Cmdlet: Get-NcPerfInstance - Not needed for build docs
        ##Unused Cmdlet: Get-NcPerfObject - Not needed for build docs
        ##Unused Cmdlet: Get-NcPerfSamples - (1.100+) Not needed for build docs

        #endregion Performance Configuration Information

        #region Secd Information

        ##Unused Cmdlet: Get-NcSecdConnectionTimeout - (1.140+) Not needed for build docs

        #endregion Secd Information

        #region Security Information

        ##Unused Cmdlet: Get-NcSecurityTrace - Not currently needed for build docs
        ##Unused Cmdlet: Get-NcSecurityTraceFilter - Not currently needed for build docs
        ##Unused Cmdlet: Get-NcSecurityLoginCurrentUser - (1.100+) Not needed
        ##Unused Cmdlet: Get-NcSecurityCertificateIssued - (1.20+) Not needed for build docs (Security.CertificateIssued)
        ##Unused Cmdlet: Get-NcSecuritySsl - (1.20+) Not needed for build docs (Security.Ssl)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.17')) {
            Write-Verbose -Message '- Gathering security related information -'
            $systemData | Add-Member -MemberType NoteProperty -Name Security -Value ([pscustomobject]@{ })

            $systemData.Security | Add-Member -MemberType NoteProperty -Name DomainTunnelVserver -Value $(
                iF ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (security login domain-tunnel show)'
                    ((Invoke-NcSystemApi -Request '<system-cli><args><arg>security login domain-tunnel show</arg></args></system-cli>' -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliTextInstance).TunnelVserver
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (security login domain-tunnel show)'
                    (Invoke-NcSsh -Command 'security login domain-tunnel show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText).TunnelVserver
                }
            )

            if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
                Write-Verbose -Message 'Calling Get-NcSecurityCertificate'
                $systemData.Security | Add-Member -MemberType NoteProperty -Name Certificate -Value (Get-NcSecurityCertificate -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

                if ($systemData.System.OntapiVersion -ge [version]::Parse('1.21')) {
                    Write-Verbose -Message 'Calling Get-NcSecuritySsh'
                    $systemData.Security | Add-Member -MemberType NoteProperty -Name Ssh -Value (Get-NcSecuritySsh -VserverContext $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
                }
            }
        }

        #endregion Security Information

        #region Snaplock Information

        ##Unused Cmdlet: Get-NcSnaplockRetentionTime - (1.100+) Not currently needed for build docs
        ##Unused Cmdlet: Get-NcSnaplockLogConfig - (1.100+) Not currently needed for build docs (Snaplock.LogConfig)
        ##Unused Cmdlet: Get-NcSnaplockLogFiles - (1.100+) Not currently needed for build docs (Snaplock.LogFiles)
        ##Unused Cmdlet: Get-NcSnaplockVolAttr - (1.100+) Using the CLI command instead
        ##Unused Cmdlet: Get-NcSnaplockEventRetention - (1.130+) Not currently needed for build docs (user needs explicit vsadmin-snaplock role)
        ##Unused Cmdlet: Get-NcSnaplockEventRetentionPolicy - (1.130+) Not currently needed for build docs (user needs explicit vsadmin-snaplock role)
        ##Unused Cmdlet: Get-NcSnaplockLegalHold - (1.130+) Not currently needed for build docs (user needs explicit vsadmin-snaplock role)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.100')) {
            Write-Verbose -Message '- Gathering Snaplock information -'
            $systemData | Add-Member -MemberType NoteProperty -Name Snaplock -Value ([pscustomobject]@{ })

            $volumeSnaplockShow = $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (volume snaplock show)'
                    (Invoke-NcSystemApi -Request "<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>volume snaplock show -Vserver $VserverContext</arg></args></system-cli>" -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (volume snaplock show)'
                    Invoke-NcSsh -Command "volume snaplock show -Vserver $VserverContext" -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            )

            $systemData.Snaplock | Add-Member -MemberType NoteProperty -Name Volume -Value (
                $volumeSnaplockShow | Select-Object -ExcludeProperty PrivilegedDelete -Property *, @{
                    Name       = 'PrivilegedDeleteState'
                    Expression = { $_.PrivilegedDelete }
                }, @{
                    Name       = 'SnaplockComplianceClock'
                    Expression = {
                        if (($compClockTime = Convert-DateTimeString -String $_.ComplianceClockTime) -or ($compClockTime = Convert-DateTimeString -String (($_.ComplianceClockTime -split '\s' | Select-Object -Index 0, 1, 2, 3, 5, 6) -join ' ') -Format 'ddd MMM d HH:mm:ss yyyy zzz')) {
                            ConvertTo-UnixTimestamp -DateTime $compClockTime.ToUniversalTime()
                        }
                    }
                }, @{
                    Name       = 'VolumeExpiryTime'
                    Expression = {
                        if (-not [string]::IsNullOrWhiteSpace($_.ExpiryTime)) {
                            if ($_.ExpiryTime -ne 'none' -and (($expiryTime = Convert-DateTimeString -String $_.ExpiryTime) -or ($expiryTime = Convert-DateTimeString -String $_.ExpiryTime -Format 'ddd MMM d HH:mm:ss GMT yyyy'))) {
                                ConvertTo-UnixTimestamp -DateTime $expiryTime.ToUniversalTime()
                            }
                            else {
                                #if we are here, this means the format of the string wasn't "GMT", so just take the string as is for now
                                $_.ExpiryTime
                            }
                        }
                    }
                }
            )
        }

        #endregion Snaplock Information

        #region NVMe Information

        ##Unused Cmdlet: Get-NcNvmeNamespaceOsType - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNvmeSubsystemOsType - (1.140+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNvmeSubsystemController - (1.140+) Throws 'error in XML document' (ZAPI burt)

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.140') -and $Controller.Credentials.UserName -ne 'fsxadmin') {
            Write-Verbose -Message '- Gathering NVME information'
            $systemData | Add-Member -MemberType NoteProperty -Name Nvme -Value (
                [pscustomobject]@{
                    Subsystem = [pscustomobject]@{ }
                }
            )

            Write-Verbose -Message 'Calling Get-NcNvme'
            $systemData.Nvme | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcNvme -Vserver $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcNvmeInterface'
            $systemData.Nvme | Add-Member -MemberType NoteProperty -Name Interface -Value (Get-NcNvmeInterface -Vserver $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcNvmeNamespace'
            $systemData.Nvme | Add-Member -MemberType NoteProperty -Name Namespace -Value (Get-NcNvmeNamespace -Vserver $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcNvmeSubsystem'
            $systemData.Nvme.Subsystem | Add-Member -MemberType NoteProperty -Name Info -Value (Get-NcNvmeSubsystem -Vserver $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcNvmeSubsystemHost'
            $systemData.Nvme.Subsystem | Add-Member -MemberType NoteProperty -Name Host -Value (Get-NcNvmeSubsystemHost -Vserver $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcNvmeSubsystemMap'
            $systemData.Nvme.Subsystem | Add-Member -MemberType NoteProperty -Name Map -Value (Get-NcNvmeSubsystemMap -Vserver $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)
        }

        #endregion NVMe Information

        #region QOS Configuration

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20') -and $Controller.Credentials.UserName -ne 'fsxadmin') {
            Write-Verbose -Message '- Gathering QOS configuration -'
            $systemData | Add-Member -MemberType NoteProperty -Name Qos -Value ([pscustomobject]@{ })

            Write-Verbose -Message 'Calling Get-NcQosPolicyGroup'
            $systemData.Qos | Add-Member -MemberType NoteProperty -Name PolicyGroup -Value (Get-NcQosPolicyGroup -Vserver $VserverContext -Controller $Controller -ErrorVariable +NetAppDocsError)

            Write-Verbose -Message 'Calling Get-NcQosWorkload'
            $systemData.Qos | Add-Member -MemberType NoteProperty -Name Workload -Value (Get-NcQosWorkload -Controller $Controller -ErrorVariable +NetAppDocsError | Where-Object { $_.Vserver -ceq $VserverContext })
        }

        #endregion QOS Configuration

        #region NDMP

        ##Unused Cmdlet: Get-NcNdmpVserverAttr - (1.100+) Not needed for build docs
        ##Unused Cmdlet: Get-NcNdmpVserverSession - (1.100+) Not needed for build docs

        if ($systemData.System.OntapiVersion -ge [version]::Parse('1.20')) {
            Write-Verbose -Message '- Gathering NDMP configuration -'
            $systemData | Add-Member -MemberType NoteProperty -Name Ndmp -Value ([pscustomobject]@{ })

            $systemData.Ndmp | Add-Member -MemberType NoteProperty -Name VserverServices -Value $(
                if ($useSystemCli) {
                    Write-Verbose -Message 'Calling Invoke-NcSystemApi (vserver services ndmp show)'
                    (Invoke-NcSystemApi -Request "<system-cli><args><arg>set -showseparator ~;</arg><arg>set -showallfields true;</arg><arg>vserver services ndmp show -vserver $VserverContext</arg></args></system-cli>" -Controller $Controller -ErrorVariable +NetAppDocsError).results.'cli-output' | Convert-SystemCliFieldDelimitedText
                }
                else {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (vserver services ndmp show)'
                    Invoke-NcSsh -Command "set -showseparator ~; set -showallfields true; vserver services ndmp show -vserver $VserverContext" -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText
                }
            )
        }

        #endregion NDMP

        #region WAFL

        ##Unused Cmdlet: Get-NcWaflSync - Not needed for build docs

        #endregion WAFL

        Write-Verbose -Message '- Getting current timestamp -'

        if ($systemData.System.OntapiVersion -lt [version]::Parse('1.30')) {
            Write-Verbose -Message 'Calling Get-NcNetInterface (only for identifying the node to gather the system time)'
            $nodeName = (Get-NcNetInterface -Controller $Controller -ErrorVariable +NetAppDocsError | Where-Object { $_.Address -eq $Controller.Address.IpAddressToString }).CurrentNode

            $systemData | Add-Member -MemberType NoteProperty -Name ReferenceDT -Value (ConvertFrom-UnixTimestamp -Seconds (Get-NcClusterHa -Node $nodeName -Controller $Controller -ErrorVariable +NetAppDocsError).CurrentTime)
        }
        else {
            $systemData | Add-Member -MemberType NoteProperty -Name ReferenceDT -Value $(
                if ($Controller.Credentials.UserName -eq 'fsxadmin') {
                    Write-Verbose -Message 'Calling Invoke-NcSsh (cluster date show)'
                    [datetime]::Parse((Invoke-NcSsh -Command 'set -showseparator ~; set -showallfields true; cluster date show' -Controller $Controller -ErrorVariable +NetAppDocsError -Verbose:$false | Convert-SystemCliFieldDelimitedText | Select-Object -First 1).UtcDateAndTime)
                }
                else {
                    Write-Verbose -Message 'Calling Get-NcTime'
                    (ConvertFrom-UnixTimestamp -Seconds (Get-NcTime -Controller $Controller -ErrorVariable +NetAppDocsError).UtcTime)
                }
            )
        }

        Write-Verbose -Message '- Identifying SystemName -'
        $systemData | Add-Member -MemberType NoteProperty -Name SystemName -Value $systemData.Vserver.Info.VserverName

        Write-Verbose -Message '- Identifying any errors during processing -'
        $systemData | Add-Member -MemberType NoteProperty -Name DiscoveryError -Value @(
            foreach ($err in ($NetAppDocsError | Where-Object { $_ })) {
                $err.CategoryInfo | Select-Object -Property *, @{
                    Name       = 'Message'
                    Expression = { $err.Exception.Message }
                }
            }
        )

        if (($systemData.DiscoveryError | Measure-Object).Count -gt 0) {
            $msg = New-Object -TypeName System.Text.StringBuilder

            [void]$msg.AppendLine("Errors were encountered while collecting data from system: $( $vs.VserverName )")
            [void]$msg.AppendLine('-------------------------------------------------------------------------------------')
            [void]$msg.AppendLine('Please review the output for accuracy and, if necessary, correct any error conditions')
            [void]$msg.AppendLine('then run these scripts again.')
            [void]$msg.AppendLine('-------------------------------------------------------------------------------------')
            [void]$msg.AppendLine()

            Write-Warning -Message $msg.ToString()
        }

        $systemData
    }
    catch {
        $Global:Error.RemoveAt(0)

        $msg = New-Object -TypeName System.Text.StringBuilder

        [void]$msg.AppendLine('An unexpected error occurred. Processing halted.')
        [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
        [void]$msg.AppendLine('As this error is not expected, please consider contacting the support staff via e-mail or by')
        [void]$msg.AppendLine('starting a thread on the community site.')
        [void]$msg.AppendLine('            E-mail: ng-NetAppDocs-support@netapp.com')
        [void]$msg.AppendLine('    Community site: https://community.netapp.com/t5/NetAppDocs/bd-p/netappdocs')
        [void]$msg.AppendLine('--------------------------------------------------------------------------------------------------')
        [void]$msg.AppendLine($( $_.Exception.GetBaseException().Message ))
        [void]$msg.AppendLine("Location: $( $_.ScriptStackTrace )")
        [void]$msg.AppendLine()

        $PSCmdlet.ThrowTerminatingError((New-Object -TypeName System.Management.Automation.ErrorRecord -ArgumentList ($msg.ToString(), $_.FullyQualifiedErrorId, $_.CategoryInfo.Category, $_.CategoryInfo.TargetName)))
    }
    finally {
        $Controller.Vserver = $null
    }
}

#region Misc Unused Cmdlets

##Unused Cmdlet: Get-NcEfficiency - Not needed for build docs
##Unused Cmdlet: Get-NcFeatureStatus - Not needed for Vserver build docs

#endregion Unused Cmdlets

# SIG # Begin signature block
# MIIq0AYJKoZIhvcNAQcCoIIqwTCCKr0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAO3g7S4nCfwv0s
# ksx2RohQlCNBXWhKDi5K47EyRB4CnKCCJGgwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggWNMIIEdaADAgECAhAOmxiO
# +dAt5+/bUOIIQBhaMA0GCSqGSIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYD
# VQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAi
# BgNVBAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAw
# MDBaFw0zMTExMDkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
# Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERp
# Z2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBAL/mkHNo3rvkXUo8MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsb
# hA3EMB/zG6Q4FutWxpdtHauyefLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iT
# cMKyunWZanMylNEQRBAu34LzB4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGb
# NOsFxl7sWxq868nPzaw0QF+xembud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclP
# XuU15zHL2pNe3I6PgNq2kZhAkHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCr
# VYJBMtfbBHMqbpEBfCFM1LyuGwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFP
# ObURWBf3JFxGj2T3wWmIdph2PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTv
# kpI6nj3cAORFJYm2mkQZK37AlLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWM
# cCxBYKqxYxhElRp2Yn72gLD76GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls
# 5Q5SUUd0viastkF13nqsX40/ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBR
# a2+xq4aLT8LWRV+dIPyhHsXAj6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6
# MIIBNjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qY
# rhwPTzAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8E
# BAMCAYYweQYIKwYBBQUHAQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5k
# aWdpY2VydC5jb20wQwYIKwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDig
# NoY0aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNybDARBgNVHSAECjAIMAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCg
# v0NcVec4X6CjdBs9thbX979XB72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQT
# SnovLbc47/T/gLn4offyct4kvFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh
# 65ZyoUi0mcudT6cGAxN3J0TU53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSw
# uKFWjuyk1T3osdz9HNj0d1pcVIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAO
# QGPFmCLBsln1VWvPJ6tsds5vIy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjD
# TZ9ztwGpn1eqXijiuZQwggXMMIIENKADAgECAhAg429sPxgagUb53pPffJfkMA0G
# CSqGSIb3DQEBDAUAMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBS
# MzYwHhcNMjEwOTA5MDAwMDAwWhcNMjMwOTA5MjM1OTU5WjBQMQswCQYDVQQGEwJV
# UzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UECgwMTmV0QXBwLCBJbmMuMRUw
# EwYDVQQDDAxOZXRBcHAsIEluYy4wggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGK
# AoIBgQC4kYYj/oViZD9pN03hrqFjtwOz1Gx4eDWVV8IYUYEr2qpLeNYvWz26B/+E
# mYLboAGVpkYg0Wske3hjRpooZlMORm1S4+2C2NoQCstZ+kmNTevbXHJe/w1VNJrm
# fKjpXOJEfx8GUGDqCMR30dChVpsdBEavrM7T0hnfJLv18i19SGv3a+nhvB3iOjLP
# SEg8+DFkjD5jTI6fQ83DBEiVR4UEYq7yyH578ZEylhsRfJmK+uIREycEUk/NpFTw
# g6/7lxh/WYabCrZzqI4Ep3QataRKaAlxt3BiOlZkX4WGX3DYvSYltWOM2UlCcGpP
# m/1/LN8Xuvf+YO6H472QecHe59XtXbHSMKvqDuOEJu8Wk2LFMNK732o3Fc5QIHie
# 6JtNsgNYf+Vsm5EMBD1ZHNE+C3zlmQbbWIU5uEU1nhiodBPKsu0SkS6pARyKBO05
# DSc2gWWv0aukKi04ZAn+hAdSl5s1dgmU5shBvKHqe15K9ZRN4FFO/CgHE0BoGYQS
# UQVKwa0CAwEAAaOCAZwwggGYMB8GA1UdIwQYMBaAFA8qyyCHKLjsb0iuK1SmKaoX
# pM0MMB0GA1UdDgQWBBQuH643KcBMmb/Q6IZt+H9IrnXFwDAOBgNVHQ8BAf8EBAMC
# B4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDAzARBglghkgBhvhC
# AQEEBAMCBBAwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwIwJTAjBggrBgEFBQcC
# ARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQBMEkGA1UdHwRCMEAw
# PqA8oDqGOGh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVT
# aWduaW5nQ0FSMzYuY3JsMHkGCCsGAQUFBwEBBG0wazBEBggrBgEFBQcwAoY4aHR0
# cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQVIz
# Ni5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqG
# SIb3DQEBDAUAA4IBgQCOoGdXjP8Sif0h3ZvoDsIVfrJvQUdP9pZExRJGGj/Te6ML
# XyqHNc/G2WklNv+BC4ryNJ4zp0oneHOub55ejrYPfhgP9oFWS8oOPJbrp3rAtINa
# OeKRC88WUYZSKYAxSX6kubEjnQD6cmazXjxVN6P8+6q9smeqF3xI+++c+ekMw3Xv
# 4EWRYXqN0/srfFp1WpoluwSQcZMqj9haSX0bgO6AAWH2AnVJMfXTIKz/0FQ/RW0y
# Ed5QYQqYInhA7IUz9dBziNpJlwONeF5j7xzgfYDY63WU6WrgJnGjYkQCOOsUx74j
# gUiMRe9zV48GS8Vxz22c/TQs0x4/1GmOSJvUHEk3GseBmB3v+yEhh/D6zWOfYP4X
# D/9b91CxmugTuepuaJSeRg+qUm3KarKsOCUF/CLqUrNh/JwKrWD1cghRaYLvMucs
# ScksHHe7ZDvb2OtvxWXjPk1d1NKvEwFJSS6hIVIbug9x28AJqOoP5Pn9wZvJSzvW
# uQJGCLscdGyKefdCo30wggYaMIIEAqADAgECAhBiHW0MUgGeO5B5FSCJIRwKMA0G
# CSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290
# IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFQxCzAJBgNVBAYT
# AkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28g
# UHVibGljIENvZGUgU2lnbmluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEBAQUAA4IB
# jwAwggGKAoIBgQCbK51T+jU/jmAGQ2rAz/V/9shTUxjIztNsfvxYB5UXeWUzCxEe
# AEZGbEN4QMgCsJLZUKhWThj/yPqy0iSZhXkZ6Pg2A2NVDgFigOMYzB2OKhdqfWGV
# oYW3haT29PSTahYkwmMv0b/83nbeECbiMXhSOtbam+/36F09fy1tsB8je/RV0mIk
# 8XL/tfCK6cPuYHE215wzrK0h1SWHTxPbPuYkRdkP05ZwmRmTnAO5/arnY83jeNzh
# P06ShdnRqtZlV59+8yv+KIhE5ILMqgOZYAENHNX9SJDm+qxp4VqpB3MV/h53yl41
# aHU5pledi9lCBbH9JeIkNFICiVHNkRmq4TpxtwfvjsUedyz8rNyfQJy/aOs5b4s+
# ac7IH60B+Ja7TVM+EKv1WuTGwcLmoU3FpOFMbmPj8pz44MPZ1f9+YEQIQty/NQd/
# 2yGgW+ufflcZ/ZE9o1M7a5Jnqf2i2/uMSWymR8r2oQBMdlyh2n5HirY4jKnFH/9g
# Rvd+QOfdRrJZb1sCAwEAAaOCAWQwggFgMB8GA1UdIwQYMBaAFDLrkpr/NZZILyhA
# QnAgNpFcF4XmMB0GA1UdDgQWBBQPKssghyi47G9IritUpimqF6TNDDAOBgNVHQ8B
# Af8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcD
# AzAbBgNVHSAEFDASMAYGBFUdIAAwCAYGZ4EMAQQBMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Um9vdFI0Ni5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ1Jvb3RSNDYu
# cDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQwFAAOCAgEABv+C4XdjNm57oRUgmxP/BP6YdURhw1aVcdGRP4Wh60BAscjW
# 4HL9hcpkOTz5jUug2oeunbYAowbFC2AKK+cMcXIBD0ZdOaWTsyNyBBsMLHqafvIh
# rCymlaS98+QpoBCyKppP0OcxYEdU0hpsaqBBIZOtBajjcw5+w/KeFvPYfLF/ldYp
# mlG+vd0xqlqd099iChnyIMvY5HexjO2AmtsbpVn0OhNcWbWDRF/3sBp6fWXhz7Dc
# ML4iTAWS+MVXeNLj1lJziVKEoroGs9Mlizg0bUMbOalOhOfCipnx8CaLZeVme5yE
# Lg09Jlo8BMe80jO37PU8ejfkP9/uPak7VLwELKxAMcJszkyeiaerlphwoKx1uHRz
# NyE6bxuSKcutisqmKL5OTunAvtONEoteSiabkPVSZ2z76mKnzAfZxCl/3dq3dUNw
# 4rg3sTCggkHSRqTqlLMS7gjrhTqBmzu1L90Y1KWN/Y5JKdGvspbOrTfOXyXvmPL6
# E52z1NZJ6ctuMFBQZH3pwWvqURR8AgQdULUvrxjUYbHHj95Ejza63zdrEcxWLDX6
# xWls/GDnVNueKjWUH3fTv1Y8Wdho698YADR7TNx8X8z2Bev6SivBBOHY+uqiirZt
# g0y9ShQoPzmCcn63Syatatvx157YK9hlcPmVoa1oDE5/L9Uo2bC5a4CH2Rwwggau
# MIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJ
# BgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5k
# aWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAe
# Fw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcw
# FQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3Rl
# ZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9Ge
# TKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0
# hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZl
# jZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAsh
# aG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVY
# TXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1
# biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCir
# c0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+
# DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA
# +bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42Pg
# puE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzS
# M7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU
# uhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6
# mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsG
# AQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29t
# MEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNl
# cnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3Js
# My5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAE
# GTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1Z
# jsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8d
# B+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVp
# P0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp8
# 76i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2
# nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3
# ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQ
# txMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc
# 4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+Y
# AN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZ
# vAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQr
# H4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIGwDCCBKigAwIBAgIQDE1p
# ckuU+jwqSj0pB4A9WjANBgkqhkiG9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4XDTIyMDkyMTAwMDAw
# MFoXDTMzMTEyMTIzNTk1OVowRjELMAkGA1UEBhMCVVMxETAPBgNVBAoTCERpZ2lD
# ZXJ0MSQwIgYDVQQDExtEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMiAtIDIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDP7KUmOsap8mu7jcENmtuh6BSFdDMa
# JqzQHFUeHjZtvJJVDGH0nQl3PRWWCC9rZKT9BoMW15GSOBwxApb7crGXOlWvM+xh
# iummKNuQY1y9iVPgOi2Mh0KuJqTku3h4uXoW4VbGwLpkU7sqFudQSLuIaQyIxvG+
# 4C99O7HKU41Agx7ny3JJKB5MgB6FVueF7fJhvKo6B332q27lZt3iXPUv7Y3UTZWE
# aOOAy2p50dIQkUYp6z4m8rSMzUy5Zsi7qlA4DeWMlF0ZWr/1e0BubxaompyVR4aF
# eT4MXmaMGgokvpyq0py2909ueMQoP6McD1AGN7oI2TWmtR7aeFgdOej4TJEQln5N
# 4d3CraV++C0bH+wrRhijGfY59/XBT3EuiQMRoku7mL/6T+R7Nu8GRORV/zbq5Xwx
# 5/PCUsTmFntafqUlc9vAapkhLWPlWfVNL5AfJ7fSqxTlOGaHUQhr+1NDOdBk+lbP
# 4PQK5hRtZHi7mP2Uw3Mh8y/CLiDXgazT8QfU4b3ZXUtuMZQpi+ZBpGWUwFjl5S4p
# kKa3YWT62SBsGFFguqaBDwklU/G/O+mrBw5qBzliGcnWhX8T2Y15z2LF7OF7ucxn
# EweawXjtxojIsG4yeccLWYONxu71LHx7jstkifGxxLjnU15fVdJ9GSlZA076XepF
# cxyEftfO4tQ6dwIDAQABo4IBizCCAYcwDgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB
# /wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwIAYDVR0gBBkwFzAIBgZngQwB
# BAIwCwYJYIZIAYb9bAcBMB8GA1UdIwQYMBaAFLoW2W1NhS9zKXaaL3WMaiCPnshv
# MB0GA1UdDgQWBBRiit7QYfyPMRTtlwvNPSqUFN9SnDBaBgNVHR8EUzBRME+gTaBL
# hklodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0
# MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3JsMIGQBggrBgEFBQcBAQSBgzCBgDAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFgGCCsGAQUFBzAC
# hkxodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRS
# U0E0MDk2U0hBMjU2VGltZVN0YW1waW5nQ0EuY3J0MA0GCSqGSIb3DQEBCwUAA4IC
# AQBVqioa80bzeFc3MPx140/WhSPx/PmVOZsl5vdyipjDd9Rk/BX7NsJJUSx4iGNV
# CUY5APxp1MqbKfujP8DJAJsTHbCYidx48s18hc1Tna9i4mFmoxQqRYdKmEIrUPwb
# tZ4IMAn65C3XCYl5+QnmiM59G7hqopvBU2AJ6KO4ndetHxy47JhB8PYOgPvk/9+d
# EKfrALpfSo8aOlK06r8JSRU1NlmaD1TSsht/fl4JrXZUinRtytIFZyt26/+YsiaV
# OBmIRBTlClmia+ciPkQh0j8cwJvtfEiy2JIMkU88ZpSvXQJT657inuTTH4YBZJwA
# wuladHUNPeF5iL8cAZfJGSOA1zZaX5YWsWMMxkZAO85dNdRZPkOaGK7DycvD+5sT
# X2q1x+DzBcNZ3ydiK95ByVO5/zQQZ/YmMph7/lxClIGUgp2sCovGSxVK05iQRWAz
# gOAj3vgDpPZFR+XOuANCR+hBNnF3rf2i6Jd0Ti7aHh2MWsgemtXC8MYiqE+bvdgc
# mlHEL5r2X6cnl7qWLoVXwGDneFZ/au/ClZpLEQLIgpzJGgV8unG1TnqZbPTontRa
# mMifv427GFxD9dAq6OJi7ngE273R+1sKqHB+8JeEeOMIA11HLGOoJTiXAdI/Otrl
# 5fbmm9x+LMz/F0xNAKLY1gEOuIvu5uByVYksJxlh9ncBjDGCBb4wggW6AgEBMGgw
# VDELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDErMCkGA1UE
# AxMiU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIFIzNgIQIONvbD8YGoFG
# +d6T33yX5DANBglghkgBZQMEAgEFAKCBhDAYBgorBgEEAYI3AgEMMQowCKACgACh
# AoAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAM
# BgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCDy37OYFVQY8BvEOJtbP3SizyEs
# e2OZMPyHpw6UksnoSjANBgkqhkiG9w0BAQEFAASCAYBj2QCAZDVGEOHWyrSpVCwv
# Q7wj0gpfwk7jhJNMhQFu2SExZRuvbLE/uP90Pzlx5L59ieNWnliI36kgOZFVzDwK
# nOpE+NddlNBLr26CgIy+lD9DyEyKd9gzByelOOIq9I1h0oyDCRyXxVJW1yML4ekM
# zFcuWBKtrRrCqMW3khe49nRmntLzbyhyUKreQ37N84sn8ifVsYot0dHVtGjFalFV
# 0q4HoOTq1M2QXELxYiog4Si68T36ecD6XEyJeUc8bMCgj36sQKF1bf4w4p4oaHik
# p1bWKrBxB17lhzyJ5fFXemuEopjc5joHS6l3MXn/nBpAqIeVCcYW9XWSK4wmSJ8V
# yGYq2V+ryQ+jLnEoQ8saT/efG/O8hzb50Xf/mVU8LvWQVqiSfPBBEulysj3A0wKE
# 4wldIvTYTVPkMM6xzHeZWHiqFzgEd2v+QNbSnHL3g5uXt/susw2ISJIn43Khqc0K
# MSdfl35cYILIQuKXZe+Dgg2mAfolhJNfa/rLT4wfv9mhggMgMIIDHAYJKoZIhvcN
# AQkGMYIDDTCCAwkCAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
# cnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
# SEEyNTYgVGltZVN0YW1waW5nIENBAhAMTWlyS5T6PCpKPSkHgD1aMA0GCWCGSAFl
# AwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUx
# DxcNMjMwNDAyMTUyODI3WjAvBgkqhkiG9w0BCQQxIgQgxQfbhacsU13ObsXHc69J
# 4i7oW9TETS8wZq3qCuT0cIYwDQYJKoZIhvcNAQEBBQAEggIARB4FU5ubzCdxKxPd
# S9UBhXqoLt6OjhnzI7Ekv4MfXCCSaIUsRkBJjoy+qn9NG8LSm8Q94zqaf0jtnItc
# 2SH4fAq3xuqdpZNNcpKY4CgeQUxq/U38mnyCDKpFZChWu/SWLmU69ZfGRPvXynZZ
# mTaiIZAgJeTzSyOyPzgc/xY5q2NDdYyREOs0kL4FuxKigGaG5Yiz9bc/zBsEIR7t
# POaXIdK87EvidXvQJqUbEZI5SicbD0uVR2Q83fjFQUjWSEEN4cXIu6WHZPNeoTVW
# y57RrnWPcX9JMrYLTFBln25Qm8g7bc/yHvEp1SG77qkcvYbr7+KEH1Y+RKbqBWLz
# 9MNGcZTA+TMIZpifgFHfaaQQSvWlBvZnB7NmpUU98cONppAqUvb+fUmEbeAjLLZi
# 8Gtx+TtpsGjSt1m+lUZh3Ed56Ho6/olE/iJfClvf9Wy7/38/zWtTJSPzNmdqz+XC
# nxk6u7bAhjl2uAP3KzrfIczAYspoV8g1ioaLLIqUzTcGbX/Nxs3zNnONJsb5EzTc
# DW0vfkMmC/IrTNUV9x9GZjgdHtZxd4tUboUZkGSJfFjyCmqmznMobyj3O66yfhle
# eiwVY2gzSj3hU/PuqOk++3Bvc1LInYDLfU97C4pgis/HdktAiqi7ozAGu0G1I/Wr
# 9jIye9MEXfAtdL8CwKjD/nFoeMo=
# SIG # End signature block
