'''
Created on Nov 12, 2013

@author: emilymw
Copyright (c) 2013, 2015 by Cisco Systems
'''

from itertools import imap
import re

from base.simpletype import SimpleType
from base.dmlist import DMList
from base.compositetype import CompositeType, Description
from translator.state_type import State, Type
from translator.base.dmboolean import DMBoolean
from asaio.cli_interaction import ignore_info_response_parser
from translator.structured_cli import StructuredCommand
from rule.access_rule import AccessGroups
from interface_config import SecurityLevel, IPv4Addr, IPv6AddressList, IPv6Address

class BridgeGroupIntfs(DMList):
    'Container of BridgeGroupIntf'

    GROUP_NUMBER_PATTERN = None
    ACCESS_GROUP_NUMBER_PATTERN = None

    def __init__(self):
        super(BridgeGroupIntfs, self).__init__('BridgeGroupIntf',
                                               BridgeGroupIntf,
                                               'interface BVI')

    @classmethod
    def get_group_number_pattern(cls):
        if not cls.GROUP_NUMBER_PATTERN:
            cls.GROUP_NUMBER_PATTERN = re.compile('interface BVI(\d+)')
        return cls.GROUP_NUMBER_PATTERN

    @classmethod
    def get_access_group_number_pattern(cls):
        if not cls.ACCESS_GROUP_NUMBER_PATTERN:
            cls.ACCESS_GROUP_NUMBER_PATTERN = re.compile('access-group .+ interface BVI(\d+)')
        return cls.ACCESS_GROUP_NUMBER_PATTERN

    def get_translator(self, cli):
        if isinstance(cli, StructuredCommand):
            cli = cli.command
        m = self.get_group_number_pattern().match(cli.strip())
        if m:
            group_number = m.group(1)
            child = self.get_child(group_number)
            if not child:
                child = self.child_class(group_number)
                self.register_child(child)
            return child
        """
        Recognize access-group command for BVI
        """
        m = self.get_access_group_number_pattern().match(cli.strip())
        if m:
            group_number = m.group(1)
            child = self.get_child(group_number)
            return child

class BridgeGroupIntf(CompositeType):
    'Model for "interface BVI" CLI'
    def __init__(self, group_number):
        super(BridgeGroupIntf, self).__init__(
            group_number,
            asa_gen_template='interface BVI%s');
        self.register_child(BVIIPv4Addr())
        self.register_child(IPv6Addresses())
        self.register_child(IPv6Enable())
        self.register_child(SimpleType('ipv6_nd_reachable_time',
                                       'ipv6 nd reachable-time'))
        self.register_child(SimpleType('ipv6_nd_ns_interval',
                                       'ipv6 nd ns-interval'))
        self.register_child(SimpleType('ipv6_nd_dad_attempts',
                                       'ipv6 nd dad attempts'))
        self.register_child(Description())
        self.register_child(SecurityLevel())
        self.register_child(AccessGroups())
        self.response_parser = ignore_info_response_parser

    def update_references(self):
        'Rid of AccessGroups if we are in transparent mode or asa-fi-dp'
        CompositeType.update_references(self)
        if self.get_top().IS_LITE or not self.get_top().is_router_mode():
            child = self.get_child('AccessGroup')
            if child:
                self.unregister_child(child)
        if not self.get_top().is_router_mode():
            child = self.get_child('security_level')
            if child:
                self.unregister_child(child)

    @property
    def nameif(self):
        'Required by AccessGroups'
        return 'BVI' + self.ifc_key

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.delta_ifc_key = (Type.FOLDER, self.parent.ifc_key, self.ifc_key)
            self.delta_ifc_cfg_value = {'state': State.DESTROY, 'value': {}}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] = self.delta_ifc_cfg_value
        elif isinstance(cli, basestring) and cli.startswith('access-group'):
            access_groups = self.get_child('AccessGroup')
            if access_groups:
                access_group = access_groups.get_translator(cli)
                if access_group:
                    access_group.diff_ifc_asa(cli)
        else:
            super(BridgeGroupIntf, self).diff_ifc_asa(cli)

    def get_cli(self):
        return self.asa_gen_template % self.ifc_key

    def validate_configuration(self):
        if (not self.ifc_key.isdigit() or int(self.ifc_key) < 1 or
                int(self.ifc_key) > 100):
            return self.generate_fault("The name cannot be '" + self.ifc_key +
                                       "', it must be an integer between 1 and 100.")

        if self.get_state() in (State.CREATE, State.MODIFY):
            if not any(imap(lambda x: self.get_child(x).has_address(),
                            ('IPv4Address', 'IPv6Address', 'ipv6_enable'))):
                return self.generate_fault(
                    'IPv4 address or IPv6 address must be defined for a bridge group')
        result = []
        for child in self.children.itervalues():
            child_result = child.validate_configuration()
            self.report_fault(child_result, result)
        return result

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        '''Override the default implementation to generate nameif sub-command to allow it to participate in routing
        if the target ASA is router mode. We are running in IRB, Integrated Routing and Bridging, mode.
        Also make sure access-group command is issued after the nameif is set.
        '''
        if self.get_state() == State.CREATE and self.get_top().is_router_mode():
            access_group = self.get_child('AccessGroup')
            if access_group:
                self.unregister_child(access_group)
            nameif = SimpleType('nameif', 'nameif', response_parser=ignore_info_response_parser)
            nameif.populate_model((Type.PARAM, 'nameif', ''), {'state': State.CREATE, 'value': 'BVI' + self.ifc_key})
            self.register_child(nameif)
            if access_group:
                self.register_child(access_group)
        return super(BridgeGroupIntf, self).ifc2asa(no_asa_cfg_stack, asa_cfg_list)

class AddressCommon(object):
    'Has common functionality for addresses'

    def has_address(self):
        return self.has_ifc_delta_cfg() and self.get_state() != State.DESTROY

class BVIIPv4Addr(IPv4Addr, AddressCommon):
    pass

class BVIIPv6Address(IPv6Address, AddressCommon):
    pass

class IPv6Addresses(IPv6AddressList):
    'Container for IPv6 addresses'
    def __init__(self):
        IPv6AddressList.__init__(self)
        self.child_class = BVIIPv6Address

    def has_address(self):
        return any(imap(lambda x: x.has_address(), self))

class IPv6Enable(DMBoolean, AddressCommon):
    'IPv6 enable configuration'

    def __init__(self):
        super(IPv6Enable, self).__init__('ipv6_enable', 'ipv6 enable', 'enable')
