'''
Created on Jun 12, 2013

@author: dli

Copyright (c) 2013 by Cisco Systems
'''

import re
from dmobject import DMObject
from dmlist import DMList
from translator.state_type import State, Type
from utils.util import ifcize_param_dict
from translator.structured_cli import StructuredCommand

class SimpleType(DMObject):
    '''For simple configuration object, such as hostname, where the number of attribute is
    only one: hostname <name_str>.
    For CLI with more complex syntax, such as:
          http enable <interface> port <port>
    You can supply asa_gen_template in the initializer using the Python formatting string:
         'http enable %(interface)s port %(port)d'
    Assuming you have "interface" and "port" attributes defined in the device specification file for
    the http enable object.

    '''
    def __init__(self,
                 ifc_key = "",
                 asa_key = "",
                 asa_gen_template = None,
                 asa_mode_template = None,
                 defaults = None,
                 is_removable = True,
                 response_parser = None,
                 is_system_context = False):
        '''
        @see DMObject.__init__
        @param asa_gen_temlate: str
            The template for generate CLI. e.g. 'http enable %(interface)s port %(port)s'.
        @param asa_mode_temlate: str
            The template for generate CLI mode command. e.g. 'interface %(interface)s'
            for interface sub-command.
            See DataOrMgmtStandbyIP in failover.py on how to use it.
        @param is_removable: boolean
            Specifies if the configuration is removable.
            Example of non-removable configuration object: 'route management ...'
            Example of removable configuration object: 'object network'
        @param response_parser: func
            Responser parser for the CLI
        @param is_system_context: Boolean
            Indicate if the command is a system context command.
        '''
        DMObject.__init__(self, ifc_key, asa_key)
        if not asa_gen_template:
            asa_gen_template = asa_key + " %s" # one parameter CLI, e.g. hostname <host>
        self.asa_gen_template = asa_gen_template
        self.asa_mode_template = asa_mode_template
        self.defaults = defaults
        self.is_removable = is_removable
        if response_parser:
            self.response_parser = response_parser
        self.is_system_context = is_system_context

    def get_action(self):
        '''
        @return the State for this object
        '''
        return self.get_state()

    def set_action(self, state):
        'Set the state for this object'
        self.set_state(state)

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        '''Generate ASA configuration from IFC configuration delta.
        @see DMObject.ifc2asa for parameter details
        '''
        if not self.has_ifc_delta_cfg():
            return
        action = self.get_action()
        if action == State.NOCHANGE:
            return

        if action in (State.CREATE, State.MODIFY):
            self.generate_cli(asa_cfg_list, self.get_cli())
        elif action == State.DESTROY and self.is_removable:
            self.generate_cli(no_asa_cfg_stack, 'no ' + self.get_cli())
        #Otherwise action == State.NOCHANGE or action == State.DESTROY and !is_removable: do nothing.

    def diff_ifc_asa(self, cli):
        '''compare configuration from populate_model with a given CLI see if they are different, and
        @precondition:
            self.populate_model likely to have been invoked.
            cli has prefix of self.get_asa_key()
        @postcondition:
            If the CLI in asa_cfg_list differers from self.self.get_cli(),
            set self.delta_ifc_cfg_value['state'] to value State.MODIFY.

            If we cannot find the CLI from asa_cfg_list for this object,
            set self.delta_ifc_cfg_value['state'] to value State.CREATE.

            If this object is not populated by populate_model, it means
            the configuration exists in the ASA but not in the IFC, we
            therefore need to delete it from the ASA to make it in sync
            with the IFC.
        '''
        if not self.has_ifc_delta_cfg():
            if self.is_removable: #delete operation required
                self.delta_ifc_key = self.create_delta_ifc_key(cli)
                self.delta_ifc_cfg_value = {'state': State.DESTROY, 'value': self.parse_cli(cli)}

                "add it to its container's delta_ifc_cfg_value"
                ancestor = self.get_ifc_delta_cfg_ancestor()
                if ancestor:
                    ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] =  self.delta_ifc_cfg_value
            return
        'Use dictionary compare instead of CLI compare to take care of optional parameters'
        if self.is_the_same_cli(cli):
            self.set_action(State.NOCHANGE)
        else:
            self.set_action(State.MODIFY)

    def is_the_same_cli(self, cli):
        '@return if the given CLI is the same as the one represented by me'
        this_cli = self.get_cli()
        return self.parse_cli(cli) == self.parse_cli(this_cli.strip()) if this_cli else False

    def create_delta_ifc_key(self, cli):
        '@return delta_ifc_key for this object'
        if isinstance(self.parent, DMList):
            return (Type.PARAM, self.parent.ifc_key, cli)
        else:
            return (Type.PARAM, self.ifc_key, '')

    def get_cli(self):
        '''Return the CLI for the self.delta_ifc_cfg_value, used by ifc2asa.
        @precondition: self.has_iifc_detal_cfg() is True
        @note: it always returns the +ve CLI, i.e. no "no" prefix in the return value.
        '''

        value = self.get_value()

        'Take care of mode command'
        if self.asa_mode_template:
            assert isinstance(value, dict)
            self.mode_command = self.asa_mode_template % value

        'Take care of defaults'
        if self.defaults:
            #filling the default values for missing parameters
            if isinstance(value, dict):
                for name in self.defaults:
                    value[name] = value.get(name, self.defaults[name])
            elif not value: #Single parameter CLI
                value = self.defaults
        '''The following join, split, and strip, and split are used to rid of extra space characters
        that may result from empty optional parameters.
        '''
        return ' '.join((self.asa_gen_template % value).split())


    def parse_cli(self, cli):
        '''Discover the value for this object from given CLI
        @param cli: CLI
            Currently, it supports two kinds of asa_gen_templates:
            - single-parameter CLI, such as 'hostname %s', and
            - multiple-parameter CLI, such as 'http enable %(interface)s port %(port)s'
          You have to write your own if it is of other form of template.
        @return the value for this object, string for single parameter CLI,
         or a dictionary for multi-parameter CLI.
        '''
        'Take care of single parameter CLI: such as "hostname %s"'
        if len(re.compile(' %s\s?').findall(self.asa_gen_template)) == 1:
            return self.parse_single_parameter_cli(cli)

        'Take care of multi-parameter CLI, such as "http enable %(interface)s port %(port)s"'
        names = re.compile('%\((\S+)\)s').findall(self.asa_gen_template)
        if not names:
            self.log('SimpleType.parse_cli: cannot handle CLI template - %s' % self.asa_gen_template)
            return
        return self.parse_multi_parameter_cli(cli)

    def parse_single_parameter_cli(self, cli):
        '''Parse CLI's that has single parameters, such as 'hostname <hosname>'.
        The template is of the form, for example: 'hostname %s'.
        '''
        's will be of the form, for example, "hostname (\S+)"'
        s = re.sub(' (%s)', r' (\S+)', self.asa_gen_template)
        p = re.compile(s)
        if isinstance(cli, StructuredCommand):
            cli = cli.command
        m = p.match(cli.strip())
        if m:
            return m.group(1)
        return None

    def parse_multi_parameter_cli(self, cli, alternate_asa_gen_template = None):
        '''Parse CLI's that has multiple parameters, such as 'http enable <interface> port <port>'.
        The template is of the form, for example: 'http enable %(interface) port %(port)s', where
        'interface' and 'port' are the names of the their entries in IFC dictionary respectively.
        @param cli: str
            The CLI to parse
        @param alternate_asa_gen_template: str of the form, for example, 'http enable $(interface)s port %(port)s'
            The template used to convert CLI to its IFC configuration dictionary format.
            If its value is None, self.asa_gen_template will be used.
        '''
        cli_template = alternate_asa_gen_template if alternate_asa_gen_template else self.asa_gen_template
        # names is list of the entry names in IFC dictionary, such as ['interface', 'port']
        names = re.compile('%\((\S+)\)s').findall(cli_template)
        # s is the regular expression for parsing CLI,  of such form: 'http enable (\S+) port (\S+)'
        # If any of the parameter is optional, you  need to override this method to take care of it.
        s = re.sub('%\(\S+\)s', r'(\S+)', cli_template)
        pattern = re.compile(s)

        if isinstance(cli, StructuredCommand):
            cli = cli.command
        m = pattern.match(cli.strip())
        if not m:
            return None

        result = {}
        for pos, name in enumerate(names, 1):
            result[name] = m.group(pos)
        return ifcize_param_dict(result)

    def __str__(self):
        '''
        @return: the user-readable string of this configuration object
        '''
        if not self.has_ifc_delta_cfg():
            return ""

        state = self.delta_ifc_cfg_value['state']

        if (state == State.CREATE):
            state = "Create"
        elif (state == State.MODIFY):
            state = "Modify"
        elif (state == State.DESTROY):
            state = "Destroy"
        return "%s %s" % (state, self.get_cli())
