'''
Created on Aug 23, 2013

@author: feliu
'''

import re

from translator.base.dmobject import DMObject
from translator.base.dmlist import DMList
from translator.base.simpletype import SimpleType
from translator.base.dmboolean import DMBoolean
from translator.state_type import Type, State
from translator.validators import SystemContextValidator
from utils.util import normalize_interface_name, netmask_from_prefix_length
from utils.util import filter_first

class FailoverConfig(DMObject):
    '''
    This class represents the holder of all failover configuration.
    '''

    def __init__(self, is_multi_mode_asa):
        DMObject.__init__(self, FailoverConfig.__name__)
        ifc_asa_keys = (
                        ("lan_unit",          "failover lan unit"),
                        ("key_secret",        "failover key"),
                        ("key_in_hex",        "failover key hex"),
                        ("interface_policy",  "failover interface-policy")
                        )
        for ifc, asa in ifc_asa_keys:
            self.register_child(FOSimpleType(ifc, asa))
        self.register_child(FOBoolean('http_replication', 'failover replication http'))
        self.register_child(MgmtStandbyIP())
        self.register_child(FailoverLANInterface())
        self.register_child(FailoverLinkInterface())
        self.register_child(DMList('failover_ip', FailoverIP, asa_key='failover interface ip'))
        self.register_child(DMList('polltime', FailoverPolltime, asa_key='failover polltime'))
        self.register_child(FOBoolean('failover', 'failover'))
        self.response_parser = failover_response_parser

        for child in self.children.values():
            if child.ifc_key != 'mgmt_standby_ip':
                child.is_system_context = is_multi_mode_asa
        self.is_system_context = is_multi_mode_asa

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        """
        Override the default implementation so on removing failover configuration,
        we will make sure to restore the management IP address there as well. Otherwise we will lose
        connectivity with it.
        """
        DMObject.populate_model(self, delta_ifc_key, delta_ifc_cfg_value)
        'Mark this folder DESTROY if all its children are marked DESTROYED'
        children_with_config = filter(lambda child: child.has_ifc_delta_cfg(), self)
        if children_with_config and all(map(lambda child: child.get_state() == State.DESTROY, children_with_config)):
                self.set_state(State.DESTROY)
        if self.get_state() == State.DESTROY:
            mgnt_standby = self.get_child('mgmt_standby_ip')
            if mgnt_standby.has_ifc_delta_cfg():
                return
            'To make sure we will restore the management IP address on the standby device'
            mgnt_standby.populate_model((Type.FOLDER, 'mgmt_standby_ip', 'standby_ip'),
                                        {'state': State.DESTROY,
                                            'value': {
                                                (Type.PARAM, 'standby_ip', 'secondary_ip'): {
                                                    'state': State.DESTROY,
                                                    'value': 'dummy',
                                                }
                                        }})

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        """
        Override the default implementation so that the configuration of the IP address
        on management interface always comes first.
        This is to ensure that we can still reach our device through the management interface
        after failover configurations are delivered.
        """
        if not self.has_ifc_delta_cfg():
            return
        tmp_asa_cfg_list = []
        tmp_no_asa_cfg_stack = []
        DMObject.ifc2asa(self, tmp_no_asa_cfg_stack, tmp_asa_cfg_list)
        man_interface_cmd = filter_first(lambda clii: clii.command.startswith('ip address'), tmp_asa_cfg_list)
        if man_interface_cmd:
            clear_failover = False
            if self.get_state() == State.DESTROY:
                """
                Use 'clear config failover' instead of a set of 'no failover ....' command.
                """
                if tmp_no_asa_cfg_stack:
                    tmp_no_asa_cfg_stack = []
                    self.generate_cli(tmp_no_asa_cfg_stack, 'clear config failover')
                    clear_failover = True
            tmp_asa_cfg_list.remove(man_interface_cmd)
            tmp_no_asa_cfg_stack.append(man_interface_cmd)
            if clear_failover and not self.is_standby_unit():
                # If primary unit, disable failover on both units first.
                self.generate_cli(tmp_no_asa_cfg_stack, 'write standby')
                self.generate_cli(tmp_no_asa_cfg_stack, 'no failover')
        asa_cfg_list.extend(tmp_asa_cfg_list)
        no_asa_cfg_stack.extend(tmp_no_asa_cfg_stack)

    def is_standby_unit(self):
        """
        Determine if the target device is the standby unit.
        Have to be careful, the lan_unit value could be out of sync with the ASA if there has been
        a failover event taken place. So we check the output of 'show failover' to be sure.
        """
        failover_status = self.query_asa('show failover state')
        if failover_status:
            """
            Sample output of 'show failover state':
                           State          Last Failure Reason      Date/Time
            This host  -   Secondary
                           Standby Ready  None
            Other host -   Primary
                           Active         None
            """
            failover_status = failover_status.strip().split('\n')
            role = failover_status[2].split()[0]
            if role != 'Disabled':
                return 'Standby' == role
        lan_unit = self.get_child('lan_unit')
        return lan_unit.get_value() == 'secondary'

class FOBoolean(DMBoolean, SystemContextValidator):
    def __init__(self, ifc_key, asa_key):
        DMBoolean.__init__(self,
                           ifc_key = ifc_key,
                           asa_key = asa_key,
                           on_value="enable",
                           response_parser=failover_response_parser)

class FOSimpleType(SimpleType, SystemContextValidator):
    def __init__(self, ifc_key, asa_key):
        SimpleType.__init__(self,
                           ifc_key = ifc_key,
                           asa_key = asa_key,
                           response_parser=failover_response_parser)


class MgmtStandbyIP(SimpleType):
    '''
    This class represents the holder of management interface configuration of the standby IP.
    '''
    def __init__(self):
        SimpleType.__init__(self,
                            ifc_key= 'mgmt_standby_ip',
                            asa_gen_template = "ip address %(active_ip)s %(netmask)s standby %(standby_ip)s",
                            response_parser = failover_response_parser)
        self.is_critical = True #make sure it comes out before failover commands even in multi-context mode.

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.get_man_interface_information(delta_ifc_cfg_value)
        return SimpleType.populate_model(self, delta_ifc_key, delta_ifc_cfg_value)

    def is_my_cli(self, cli):
        if isinstance(cli, basestring):
            return False
        if not cli.command.startswith('interface '):
            return False
        if hasattr(self, 'mode_command'):
            return self.mode_command == cli.command
        if not filter_first(lambda cmd: str(cmd) == 'nameif management', cli.sub_commands):
            return False
        self.mode_command = cli.command
        address = filter_first(lambda cmd: str(cmd).startswith('ip address'), cli.sub_commands)
        if address:
            address = address.split()
            self.man_interface_info = {'active_ip': address[2],
                                       'netmask':   address[3]}
            'only cares if it has standby address'
            return len(address) > 4 and 'standby' == address[4]
        return False

    def is_the_same_cli(self, cli):
        address = filter_first(lambda cmd: str(cmd).startswith('ip address'), cli.sub_commands)
        return address == self.get_cli()

    def get_man_interface_information(self, delta_ifc_cfg_value):
        """Gather the physical interface name of the management interface and and its IP address 
        """
        'first check to see if we have the information cached'
        mode_command = delta_ifc_cfg_value.get('mode_command')
        if mode_command:
            self.mode_command = mode_command
            self.man_interface_info = delta_ifc_cfg_value['man_interface_info']
            return
        result = self.query_asa("show ip address management | grep management")
        if not result:
            return
        """
        The format of result looks like:
            Management0/0            management             10.22.51.207    255.255.255.0   CONFIG
            Management0/0            management             10.22.51.208    255.255.255.0   CONFIG
        For a failover-standby device, where the 2nd line has the address for accessing it.
        """
        result = result.strip().split('\n')
        if len(result) == 2:
            result = result[0].split()
            self.man_interface_info = {'active_ip': result[2],
                                       'netmask':   result[3]}
            self.mode_command = 'interface ' + result[0]
            'cache the information for later use'
            delta_ifc_cfg_value['mode_command'] = self.mode_command
            delta_ifc_cfg_value['man_interface_info'] = self.man_interface_info

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        '''Generate ASA configuration from IFC configuration delta.
        '''
        if not self.has_ifc_delta_cfg():
            return
        action = self.delta_ifc_cfg_value['state']
        if action == State.NOCHANGE:
            return
        if not hasattr(self, 'man_interface_info'):
            return

        if self.is_standby_unit() and action != State.DESTROY:
            'do not generate the CLI on standby device, it is replicated from active unit'
            return
        self.generate_cli(asa_cfg_list, self.get_cli())

    def get_cli(self):
        '''
        Override the default so that in the case of destroy, we will not issue the standby IP address
        '''
        value = self.get_value()
        if not value:
            value = {}
        value.update(self.man_interface_info)
        if self.get_action() == State.DESTROY:
            """
            For the standby device, we must restore the IP address to the device parameter to
            the call-out
            """
            if self.is_standby_unit():
                value['active_ip'] = self.get_top().get_device()['host']
            command = "ip address %(active_ip)s %(netmask)s" % value
        else:
            command = self.asa_gen_template % value
        return command

    def is_standby_unit(self):
        return self.parent.is_standby_unit()

class FailoverInterface(SimpleType, SystemContextValidator):
    '''
    Common functionality for the failover LAN and Link interfaces
    '''

    def __init__(self, name, asa_key, asa_gen_template):
        super(FailoverInterface, self).__init__(
            name,
            asa_key,
            asa_gen_template,
            response_parser = failover_response_parser)

    def get_cli(self):
        return self.asa_gen_template % self.values

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.has_ifc_delta_cfg():
            state = self.get_state()
            if state in (State.CREATE, State.MODIFY):
                # Enable the interface, but only once
                command = 'no shutdown'
                mode_command = 'interface ' + self.values['interface']
                if not filter_first(lambda cmd: cmd.command == command and
                                        cmd.mode_command == mode_command,
                                    asa_cfg_list):
                    self.generate_cli(asa_cfg_list,
                                      command,
                                      mode_command=mode_command)
                self.generate_cli(asa_cfg_list, self.get_cli())
            elif state == State.DESTROY:
                self.generate_cli(no_asa_cfg_stack, 'no ' + self.get_cli())

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        super(FailoverInterface, self).populate_model(delta_ifc_key,
                                                      delta_ifc_cfg_value)
        self.values = self.get_value()
        if 'interface' in self.values:
            self.values['interface'] = normalize_interface_name(
                self.values['interface'])

    def validate_configuration(self):
        if self.has_ifc_delta_cfg():
            faults = []

            # Check for system context access error
            self.report_fault(self.validate(self.values), faults)

            # Check for missing interface
            if not self.values.get('interface'):
                self.report_fault('Missing interface configuration.', faults)

            return faults

class FailoverLANInterface(FailoverInterface):
    '''
    This class represents the holder of failover LAN-based interface configuration.
    '''

    def __init__(self):
        super(FailoverLANInterface, self).__init__(
            'failover_lan_interface',
            'failover lan interface',
            'failover lan interface %(interface_name)s %(interface)s')

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        super(FailoverLANInterface, self).populate_model(delta_ifc_key,
                                                      delta_ifc_cfg_value)
        if 'interface' not in self.values:
            self.values['interface'] = normalize_interface_name(
                self.get_top().get_failover_lan_interface())

class FailoverLinkInterface(FailoverInterface):
    '''
    This class represents the holder of failover link (stateful) interface configuration.
    '''

    def __init__(self):
        super(FailoverLinkInterface, self).__init__(
            'failover_link_interface',
            'failover link',
            'failover link %(interface_name)s %(interface)s')

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        super(FailoverLinkInterface, self).populate_model(delta_ifc_key,
                                                      delta_ifc_cfg_value)
        if 'interface' not in self.values:
            self.values['interface'] = normalize_interface_name(
                self.get_top().get_failover_link_interface())

    def update_references(self):
        if hasattr(self, 'values'):
            if 'use_lan' in self.values:
                lan = self.parent.get_child('failover_lan_interface')
                if lan and hasattr(lan, 'values'):
                    self.values['interface'] = lan.values['interface']

class FailoverIP(SimpleType, SystemContextValidator):
    '''
    This class represents the holder of failover IP configuration.
    '''

    def __init__(self, name):
        SimpleType.__init__(self, name, is_removable=True,
                            asa_gen_template='failover interface ip %(interface_name)s %(active_ip)s',
                            response_parser = failover_response_parser)

    def get_cli(self):
        '''Generate the CLI for this single failover ip config.
        '''
        assert self.has_ifc_delta_cfg()
        config = self.get_value()
        netmask = config.get('netmask') if ':' not in config.get('active_ip') else ''
        standby_ip = config.get('standby_ip')
        result = SimpleType.get_cli(self)
        result += ' ' + netmask + ' standby ' + standby_ip
        return ' '.join(result.split())

    def create_asa_key(self):
        '''Create the the asa key identifies this object
        @return str
        '''
        assert self.has_ifc_delta_cfg()
        value = self.get_value()
        return self.asa_gen_template % value

    def parse_multi_parameter_cli(self, cli):
        '''
        Override the default implementation in case the CLI does not match asa_gen_template due to optional parameter
        '''
        # Take care of the mandatory parameters
        result = SimpleType.parse_multi_parameter_cli(self, cli, alternate_asa_gen_template = self.asa_gen_template)

        'Take care of the optional parameters'
        tokens = cli.split()

        # The number of tokens must greater than 3, i.e. 'failover interface ip ...'
        assert len(tokens) > 3

        for name  in ['interface_name', 'active_ip', 'standby_ip']:
            result[(Type.PARAM, name, '')] = {'state': State.NOCHANGE, 'value': ''}
        option = tokens[3:]
        # for ipv4, option is: %(interface_name)s %(active_ip)s %(netmask)s standby %(standby_ip)s
        # for ipv6, option is: %(interface_name)s %(active_ip)s standby %(standby_ip)s
        result[Type.PARAM, 'interface_name', '']['value'] = option[0]
        result[Type.PARAM, 'active_ip', '']['value'] = option[1]
        if ':' not in option[1]: # ipv4
            result[(Type.PARAM, 'netmask', '')] = {'state': State.NOCHANGE, 'value': ''}
            result[Type.PARAM, 'netmask', '']['value'] = option[2]
            # skip option[3], which should be the 'standby' keyword
            result[Type.PARAM, 'standby_ip', '']['value'] = option[4]
        else: # ipv6, no netmask
            # skip option[2], which should be the 'standby' keyword
            result[Type.PARAM, 'standby_ip', '']['value'] = option[3]
        return result

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.normalize_netmask(delta_ifc_cfg_value)
        return SimpleType.populate_model(self, delta_ifc_key, delta_ifc_cfg_value)

    def normalize_netmask(self, config):
        '''
        If the netmask is prefix length for IPv4, turns it into netmask
        '''
        netmask_key = filter_first(lambda key: key[1] == 'netmask', config['value'].keys())
        if not netmask_key:
            return
        netmask = config['value'][netmask_key]['value']
        if netmask.isdigit():
            config['value'][netmask_key]['value'] = netmask_from_prefix_length(netmask)

class FailoverPolltime(SimpleType, SystemContextValidator):
    '''
    This class represents the holder of failover polltime configuration.
    '''

    def __init__(self, name):
        SimpleType.__init__(self, name,
                            asa_gen_template='failover polltime',
                            response_parser = failover_response_parser)

    def get_cli(self):
        cli = []
        cli.append('failover polltime')
        config = self.get_value()
        unit_or_interface = config.get('unit_or_interface')
        if unit_or_interface == 'interface':
            cli.append(' interface')
        elif unit_or_interface == 'unit':
            cli.append(' unit')
        if config.get('interval_unit') == 'msec':
            cli.append(' msec')
        cli.append(' ' + config.get('interval_value'))
        holdtime_value = config.get('holdtime_value')
        if holdtime_value:
            cli.append(' holdtime')
            if config.get('holdtime_unit') == 'msec':
                cli.append(' msec')
            cli.append(' ' + holdtime_value)
        return ''.join(cli)

    def validate_configuration(self):
        if self.has_ifc_delta_cfg():
            faults = []
            config = self.get_value()

            # Check for system context access error
            self.report_fault(self.validate(config), faults)

            # Validate poll times
            if config.get('unit_or_interface') == 'interface':
                self.validate_poll_times(config, faults, 'Interface', 1, 15,
                                         500, 999, 5, 75)
            else:
                self.validate_poll_times(config, faults, 'Unit', 1, 15, 200,
                                         999, 1, 45, 800, 999)

            return faults

    def validate_poll_times(self, config, faults, poll_type, poll_min, poll_max,
                            poll_msec_min, poll_msec_max, hold_min, hold_max,
                            hold_msec_min=None, hold_msec_max=None):
        interval_value = config.get('interval_value')
        if interval_value:
            interval_value = int(interval_value)
            if config.get('interval_unit') == 'msec':
                if not (poll_msec_min <= interval_value <= poll_msec_max):
                    self.report_fault('interval_value must be between %s and %s milliseconds.' % (poll_msec_min, poll_msec_max),
                                      faults, 'interval_value')
            else:
                if not (poll_min <= interval_value <= poll_max):
                    self.report_fault('interval_value must be between %s and %s seconds.' % (poll_min, poll_max),
                                      faults, 'interval_value')

            holdtime_value = config.get('holdtime_value')
            if holdtime_value:
                holdtime_value = int(holdtime_value)
                if config.get('holdtime_unit') == 'msec':
                    if hold_msec_min == None:
                        self.report_fault(poll_type + ' holdtime does not support msec.',
                                          faults, 'holdtime_unit')
                    elif not (hold_msec_min <= holdtime_value <= hold_msec_max):
                        self.report_fault('holdtime_value must be between %s and %s milliseconds.' % (hold_msec_min, hold_msec_max),
                                          faults, 'holdtime_value')
                else:
                    if not (hold_min <= holdtime_value <= hold_max):
                        self.report_fault('holdtime_value must be between %s and %s seconds.' % (hold_min, hold_max),
                                          faults, 'holdtime_value')

    def parse_cli(self, cli):
        '''
        Discover the value for this object from given CLI
        '''
        # Valid examples:
        # 1.  failover polltime 5
        # 2.  failover polltime 5 holdtime 15
        # 3.  failover polltime msec 200
        # 4.  failover polltime msec 200 holdtime 1
        # 5.  failover polltime msec 200 holdtime msec 800
        # 6.  failover polltime interface|unit 5
        # 7.  failover polltime interface|unit 5 holdtime 15
        # 8.  failover polltime interface|unit msec 200
        # 9.  failover polltime interface|unit msec 200 holdtime 1
        # 10. failover polltime interface|unit msec 200 holdtime msec 800

        p1 = re.compile('^failover polltime \d+$')
        p2 = re.compile('^failover polltime \d+ holdtime \d+$')
        p3 = re.compile('^failover polltime msec \d+$')
        p4 = re.compile('^failover polltime msec \d+ holdtime \d+$')
        p5 = re.compile('^failover polltime msec \d+ holdtime msec \d+$')
        p6 = re.compile('^failover polltime (interface|unit) \d+$')
        p7 = re.compile('^failover polltime (interface|unit) \d+ holdtime \d+$')
        p8 = re.compile('^failover polltime (interface|unit) msec \d+$')
        p9 = re.compile('^failover polltime (interface|unit) msec \d+ holdtime \d+$')
        p10= re.compile('^failover polltime (interface|unit) msec \d+ holdtime msec \d+$')

        index_unit_or_interface = None
        index_interval_in_second = None
        index_interval_in_msec = None
        index_holdtime_in_second = None
        index_holdtime_in_msec = None

        if p1.match(cli):   # failover polltime 5
            index_interval_in_second = 2
        elif p2.match(cli): # failover polltime 5 holdtime 15
            index_interval_in_second = 2
            index_holdtime_in_second = 4
        elif p3.match(cli): # failover polltime msec 200
            index_interval_in_msec = 3
        elif p4.match(cli): # failover polltime msec 200 holdtime 1
            index_interval_in_msec = 3
            index_holdtime_in_second = 5
        elif p5.match(cli): # failover polltime msec 200 holdtime msec 800
            index_interval_in_msec = 3
            index_holdtime_in_msec = 6
        elif p6.match(cli): # failover polltime interface|unit 5
            index_unit_or_interface = 2
            index_interval_in_second = 3
        elif p7.match(cli): # failover polltime interface|unit 5 holdtime 15
            index_unit_or_interface = 2
            index_interval_in_second = 3
            index_holdtime_in_second = 5
        elif p8.match(cli): # failover polltime interface|unit msec 200
            index_unit_or_interface = 2
            index_interval_in_msec = 4
        elif p9.match(cli): # failover polltime interface|unit msec 200 holdtime 1
            index_unit_or_interface = 2
            index_interval_in_msec = 4
            index_holdtime_in_second = 6
        elif p10.match(cli):# failover polltime interface|unit msec 200 holdtime msec 800
            index_unit_or_interface = 2
            index_interval_in_msec = 4
            index_holdtime_in_msec = 7

        tokens = cli.split()
        result = {}
        if index_unit_or_interface:
            result[(Type.PARAM, 'unit_or_interface', '')] = {'state': State.NOCHANGE, 'value': ''}
            result[Type.PARAM, 'unit_or_interface', '']['value'] = tokens[index_unit_or_interface]
        if index_interval_in_second:
            result[(Type.PARAM, 'interval_unit', '')] = {'state': State.NOCHANGE, 'value': ''}
            result[Type.PARAM, 'interval_unit', '']['value'] = 'second'
            result[(Type.PARAM, 'interval_value', '')] = {'state': State.NOCHANGE, 'value': ''}
            result[Type.PARAM, 'interval_value', '']['value'] = tokens[index_interval_in_second]
        if index_interval_in_msec:
            result[(Type.PARAM, 'interval_unit', '')] = {'state': State.NOCHANGE, 'value': ''}
            result[Type.PARAM, 'interval_unit', '']['value'] = 'msec'
            result[(Type.PARAM, 'interval_value', '')] = {'state': State.NOCHANGE, 'value': ''}
            result[Type.PARAM, 'interval_value', '']['value'] = tokens[index_interval_in_msec]
        if index_holdtime_in_second:
            result[(Type.PARAM, 'holdtime_unit', '')] = {'state': State.NOCHANGE, 'value': ''}
            result[Type.PARAM, 'holdtime_unit', '']['value'] = 'second'
            result[(Type.PARAM, 'holdtime_value', '')] = {'state': State.NOCHANGE, 'value': ''}
            result[Type.PARAM, 'holdtime_value', '']['value'] = tokens[index_holdtime_in_second]
        if index_holdtime_in_msec:
            result[(Type.PARAM, 'holdtime_unit', '')] = {'state': State.NOCHANGE, 'value': ''}
            result[Type.PARAM, 'holdtime_unit', '']['value'] = 'msec'
            result[(Type.PARAM, 'holdtime_value', '')] = {'state': State.NOCHANGE, 'value': ''}
            result[Type.PARAM, 'holdtime_value', '']['value'] = tokens[index_holdtime_in_msec]
        return result

def failover_response_parser(response):
    '''
    Ignores INFO, WARNING, and some expected errors in the response, otherwise returns original.
    '''

    if response:
        msgs_to_ignore = ('INFO:',
                          'WARNING',
                          'Interface does not have virtual MAC',
                          'No change to the stateful interface',
                          'Waiting for the earlier webvpn instance to terminate',
                          'This unit is in syncing state',
                          'Configuration syncing is in progress')
        found_msg_to_ignore = False
        for msg in msgs_to_ignore:
            if msg in response:
                found_msg_to_ignore = True
    return None if response and found_msg_to_ignore else response
