'''
Copyright (c) 2013 by Cisco Systems, Inc.

@author: emilymw
'''

import re

from asaio.cli_interaction import ignore_info_response_parser,\
    ignore_response_parser
from asaio.cli_interaction import ignore_warning_response_parser
from base.compositetype import CompositeType, Description
from base.dmboolean import DMBoolean
from base.dmlist import DMList
from base.simpletype import SimpleType
from routing.ospf_interface import OspfCombinedInterface
from state_type import State, Type
from utils.util import netmask_from_prefix_length, filter_first,\
    ifcize_param_dict, set_cfg_state
from utils.util import normalize_ipv6_address, normalize_ipv4_address_and_mask_4_asa
from validators import IPv6AddressValidator, IPv6AddressSubnetValidator
from translator.validators import InterfaceIPv4AddressValidator
from translator.address_pool import IPv4AddressPool, IPv6AddressPool

class InterfaceConfig(CompositeType):
    def __init__(self):
        super(InterfaceConfig, self).__init__(InterfaceConfig.__name__)
        self.ipv4_addr = IPv4Addr()
        self.ospf_combined_interface = OspfCombinedInterface()

        self.register_child(MacAddress())
        self.register_child(self.ipv4_addr)
        self.register_child(BridgeGroup())
        self.register_child(SecurityLevel())
        self.register_child(IPv6AddressList())
        self.register_child(IPv6LinkLocal())
        self.register_child(DMBoolean('ipv6_autoconfig',
                                      'ipv6 address autoconfig', 'autoconfig'))
        self.register_child(DMBoolean('ipv6_enable', 'ipv6 enable', 'enable'))
        self.register_child(SimpleType('ipv6_nd_reachable_time',
                                       'ipv6 nd reachable-time'))
        self.register_child(SimpleType('ipv6_nd_ns_interval',
                                       'ipv6 nd ns-interval'))
        self.register_child(SimpleType('ipv6_nd_ra_interval',
                                       'ipv6 nd ra-interval'))
        self.register_child(SimpleType('ipv6_nd_ra_lifetime',
                                       'ipv6 nd ra-lifetime'))
        self.register_child(DMList('IPv6NeighborDiscovery',
                                   IPv6NeighborDiscovery,
                                   'ipv6 nd prefix'))
        self.register_child(SimpleType('ipv6_nd_dad_attempts',
                                       'ipv6 nd dad attempts'))
        self.register_child(self.ospf_combined_interface)
        self.register_child(Description())

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.delta_ifc_key = (Type.FOLDER, self.ifc_key, '')
            self.delta_ifc_cfg_value = {'state': State.DESTROY, 'value': {}}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] =  self.delta_ifc_cfg_value
        else:
            super(InterfaceConfig, self).diff_ifc_asa(cli)

    # This is used for the mode command of the children
    def get_cli(self):
        if hasattr(self.parent, 'mode_command'):
            return self.parent.mode_command

    def get_encapass(self):
        return self.parent.get_encapass()

    def get_ipv4_address_dict(self):
        'Get a dictionary containing the IPv4 address, netmask, and state'
        return self.ipv4_addr.get_address_dict()

    def get_ospf_area_id(self):
        return self.ospf_combined_interface.get_ospf_area_id()

    def get_ospfv3(self):
        return self.ospf_combined_interface.get_ospfv3()

    def get_ospfvif_state(self):
        return self.ospf_combined_interface.get_state()

    def ifc2asa(self, no_asa_cfg_stack,  asa_cfg_list):
        # The parent will handle destroying the interface.
        if (self.parent.get_state() != State.DESTROY and
                # Can only generate sub-commands if the interface VIF/VLAN
                # information is known.
                self.parent.allow_sub_commands()):
            super(InterfaceConfig, self).ifc2asa(no_asa_cfg_stack,
                                                 asa_cfg_list)
    @property
    def nameif(self):
        return self.parent.nameif

class SecurityLevel(SimpleType):
    'Model after "security-level <0-100>'
    def __init__(self):
        super(SecurityLevel, self).__init__('security_level', 'security-level')
        self.response_parser=ignore_info_response_parser

    def diff_ifc_asa(self, cli):
        # Security level 0 is the default, don't remove it
        if not self.has_ifc_delta_cfg() and self.parse_cli(cli) == '0':
            return
        super(SecurityLevel, self).diff_ifc_asa(cli)


class IPv4Addr(SimpleType, InterfaceIPv4AddressValidator):
    'Model after "ip address ipv4_address netmask" sub-command'
    def __init__(self):
        super(IPv4Addr, self).__init__('IPv4Address', 'ip address',
                                       asa_gen_template='ip address %(ipv4_address)s',
                                       response_parser = ignore_warning_response_parser)

    def get_address_dict(self):
        'Get a dictionary containing the IPv4 address, netmask, and state'
        if self.has_ifc_delta_cfg():
            ipaddress, netmask = self.get_value()['ipv4_address'].split('/')
            if '.' not in netmask: # Prefix length
                netmask = netmask_from_prefix_length(netmask)
            return {'ipaddress': ipaddress,
                    'netmask': netmask,
                    'state': self.get_state()}

    def get_cli(self):
        'Override the default implementation to take care / delimiter'
        value = self.get_value()
        ipv4address = value.get('ipv4_address')
        if '.' in ipv4address: #ipv4 address
            ipv4address = normalize_ipv4_address_and_mask_4_asa(ipv4address)
            value.update({'ipv4_address': ipv4address})
        if 'ipv4_standby_address' in value:
            return (self.asa_gen_template + ' standby %(ipv4_standby_address)s') % value
        elif 'ipv4_address_pool' in value:
            return (self.asa_gen_template % value) + ' cluster-pool ' + self.get_address_pool_name()
        else:
            return self.asa_gen_template % value

    def parse_multi_parameter_cli(self, cli):
        '''
        Override the default implementation in case the CLI does not match asa_gen_template due to optional parameter
        '''

        result = {}

        # Take care of the optional parameters
        tokens = cli.split()

        # The number of tokens must > 2, i.e. 'ip address IP [NETMASK] [standby IP]'
        option = tokens[2:]

        # To be more specific, the valid CLI could be of one the following:
        #     ip address 192.168.2.2 255.255.255.0
        #     ip address 192.168.2.2 255.255.255.0 standby 192.168.2.3
        #     ip address 192.168.2.2 255.255.255.0 cluster-pool pool-name
        if len(option) > 2:
            if option[2] == 'standby':
                result[(Type.PARAM, 'ipv4_standby_address', '')] = {'state': State.NOCHANGE, 'value': option[3]}
            elif option[2] == 'cluster-pool':
                result[(Type.PARAM, 'ipv4_address_pool', '')] = {'state': State.NOCHANGE, 'value': self.get_address_pool(option[3])}
        if len(option) > 1:
            # IPv4 address is in the form of IP/MASK in the device model'
            result[(Type.PARAM, 'ipv4_address', '')] = {'state': State.NOCHANGE, 'value': '/'.join([option[0], option[1]])}
        return result

    def validate(self, value):
        return None if value == None else InterfaceIPv4AddressValidator.validate(self, value.get('ipv4_address'))

    def get_address_pool_name(self):
        return IPv4AddressPool.make_name(self.parent.nameif)

    def get_address_pool(self, name):
        pool = self.get_top().get_child('IPv4AddressPool').get_child(name)
        if pool:
            return '%(address_range)s' % pool.get_value()

    def update_references(self):
        '''Override the default so that we create the address pool object
        '''
        SimpleType.update_references(self)
        if not self.has_ifc_delta_cfg():
            return
        address_pool_key = filter_first(lambda key: key[1] == 'ipv4_address_pool', self.delta_ifc_cfg_value['value'].keys())
        if address_pool_key:
            self.populate_address_pool(self.delta_ifc_cfg_value['value'][address_pool_key])

    def populate_address_pool(self, pool):
        def get_mask():
            address_key = filter_first(lambda key: key[1] == 'ipv4_address', self.delta_ifc_cfg_value['value'].keys())
            address = self.delta_ifc_cfg_value['value'][address_key]['value'].split('/')
            result = address[1]
            if '.' not in result: # Prefix length
                result = netmask_from_prefix_length(result)
            return result

        def make_pool_value():
            '''
            Return a dictionary value for IPv4AddressPool of the normal form:
             {'state': 1, 'value': {'address_range': '1.1.1.1-1.1.1.4', 'mask': '255.255.255.0'}}
            '''
            result = {'address_range': pool['value'], 'mask': get_mask()}
            result = ifcize_param_dict(result)
            set_cfg_state(result, pool['state'])
            result = {'state': pool['state'], 'value': result}
            return result

        def make_pool_key():
            return (Type.PARAM, 'IPv4AddressPool', self.get_address_pool_name())

        pools = self.get_top().get_child('IPv4AddressPool')
        pools.register_pool(make_pool_key(), make_pool_value())

class IPv6LinkLocal(SimpleType):
    'Model after  "ipv6 address x:x:x:x::x link-local'
    def __init__(self):
        super(IPv6LinkLocal, self).__init__(
            'IPv6LinkLocalAddress',
            re.compile('ipv6 address [\dabcdef:]+ link-local'),
            asa_gen_template='ipv6 address %(ipv6_link_local_address)s link-local')

    def get_cli(self):
        value = self.get_value()
        cli = []
        cli.append('ipv6 address')
        cli.append(normalize_ipv6_address(value.get('ipv6_link_local_address')))
        cli.append('link-local')
        if value.get('ipv6_link_local_standby_address'):
            cli.append('standby')
            cli.append(normalize_ipv6_address(value.get('ipv6_link_local_standby_address')))
        return ' '.join(cli)

    def parse_multi_parameter_cli(self, cli):
        '''
        Override the default implementation in case the CLI does not match asa_gen_template due to optional parameter
        '''

        result = {}

        # Take care of the optional parameters
        tokens = cli.split()

        # The number of tokens must > 2, i.e. 'ipv6 address IPV6_LINK_LOCAL_ADDRESS link-local [standby IPV6_LINK_LOCAL_STANDBY_ADDRESS]'
        option = tokens[2:]

        result[(Type.PARAM, 'ipv6_link_local_address', '')] = {'state': State.NOCHANGE, 'value': option[0]}
        if len(option) > 2: # has standby address
            result[(Type.PARAM, 'ipv6_link_local_standby_address', '')] = {'state': State.NOCHANGE, 'value': option[2]}
        return result

    def validate_configuration(self):
        if not self.has_ifc_delta_cfg():
            return
        value = self.get_value()
        result = []
        address = value.get('ipv6_link_local_address')
        error = IPv6AddressValidator().validate(address)
        if error:
            self.report_fault(error, result, 'ipv6_link_local_address')
        standby_address = value.get('ipv6_link_local_standby_address')
        if standby_address:
            error = IPv6AddressValidator().validate(standby_address)
            if error:
                self.report_fault(error, result, 'ipv6_link_local_standby_address')
        return result

class IPv6AddressList(DMList):
    def __init__(self):
        super(IPv6AddressList, self).__init__(
            'IPv6Address',
            IPv6Address,
            re.compile('ipv6 address [\dabcdef:]+/\d+'))

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        '''Override the default to provide the mini_audit_command for modification operation.
        '''
        self.mini_audit_command = 'show run %s | grep ipv6 address' % self.mode_command
        DMList.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

    def read_asa_config(self):
        '''Override the default to remove the leading space in the 'ipv6 address' sub-commands
        '''
        result = DMList.read_asa_config(self)
        if not result:
            return
        lines = result.split('\n')
        lines = map(str.strip, lines)
        result = '\n'.join(lines)
        return result

class IPv6Address(SimpleType):
    'Model after "ipv6 address ipv6_address/prefix [eui-64]" sub-command'
    def __init__(self, instance):
        super(IPv6Address, self).__init__(
            instance,
            asa_gen_template='ipv6 address %(ipv6_address_with_prefix)s')

    def create_asa_key(self):
        return self.get_cli()

    def get_cli(self):
        cli = []
        value = self.get_value()
        cli.append('ipv6 address')
        cli.append(normalize_ipv6_address(value['ipv6_address_with_prefix'],
                                          interface_addr=True))
        if 'eui64' in value:
            cli.append('eui-64')
        elif 'ipv6_standby_address' in value:
            cli.append('standby')
            cli.append(value.get('ipv6_standby_address'))
        elif 'ipv6_address_pool' in value:
            cli.append('cluster-pool ' + self.get_address_pool_name(value['ipv6_address_with_prefix']))
        return ' '.join(cli)

    def parse_multi_parameter_cli(self, cli):
        # Take care of the mandatory parameters
        result = super(IPv6Address, self).parse_multi_parameter_cli(cli)

        'Take care of the optional parameters'
        tokens = cli.split()
        if len(tokens) == 4 and tokens[3] == 'eui-64':
            result[(Type.PARAM, 'eui64', '')] = {'state': State.NOCHANGE, 'value': 'enable'}
        elif len(tokens) == 5:
            if tokens[3] == 'standby':
                # ipv6 address ipv6_address/prefix standby ipv6_standby_address
                result[(Type.PARAM, 'ipv6_standby_address', '')] = {'state': State.NOCHANGE, 'value': tokens[4]}
            elif tokens[3] == 'cluster-pool':
                # ipv6 address ipv6_address/prefix cluster-pool pool-name
                result[(Type.PARAM, 'ipv6_address_pool', '')] = {'state': State.NOCHANGE, 'value': self.get_address_pool(tokens[4])}
        return result


    def validate_configuration(self):
        if not self.has_ifc_delta_cfg():
            return
        value = self.get_value()
        result = []
        address = value.get('ipv6_address_with_prefix')
        error = IPv6AddressSubnetValidator().validate(address)
        if error:
            self.report_fault(error, result, 'ipv6_address_with_prefix')
        standby_address = value.get('ipv6_standby_address')
        if standby_address:
            error = IPv6AddressValidator().validate(standby_address)
            if error:
                self.report_fault(error, result, 'ipv6_standby_address')
        present = [value.get('eui-64'), standby_address, value.get('ipv6_address_pool')]
        if len(filter(lambda v: v, present)) > 1:
            self.report_fault("Cann specify one of eui-64, ipv6_standby_address, or ipv6_address_pool at the same time", result)
        return result

    def get_address_pool_name(self, address):
        return IPv6AddressPool.make_name(self.parent.parent.nameif, address)

    def get_address_pool(self, name):
        pool = self.get_top().get_child('IPv6AddressPool').get_child(name)
        if pool:
            return '%(address)s %(number)s' % pool.get_value()

    def update_references(self):
        '''Override the default so that we create the address pool object.
        We also normalize IPv6 address for ASA, so that during audit, we can compare the values between ASA and APIC correctly.
        '''
        SimpleType.update_references(self)
        if not self.has_ifc_delta_cfg():
            return
        address_pool_key = self.get_param_dict_key('ipv6_address_pool')
        address_key = self.get_param_dict_key('ipv6_address_with_prefix')
        address = self.delta_ifc_cfg_value['value'][address_key]
        address['value'] = normalize_ipv6_address(address['value'], True)
        if address_pool_key:
            address_pool = self.delta_ifc_cfg_value['value'][address_pool_key]
            v = address_pool['value']
            if v:
                token = v.split(' ')
                v =  normalize_ipv6_address(token[0], True) + ' ' + token[1]
                address_pool['value'] = v
            if address_pool['state'] == State.NOCHANGE and self.get_state() == State.MODIFY:
                #address change means address-pool name change
                address_pool['state'] = State.MODIFY
            self.populate_address_pool(address, address_pool)

    def populate_address_pool(self, address, pool):
        '''
        @param address: format {'value': '1::10/24'}
        @param pool: format {'value': '1::1 2'}
        '''
        def get_mask():
            return address['value'].split('/')[1]

        def make_pool_value():
            '''
            @return a dictionary value for IPv6AddressPool of the normal form:
             {'state': 1, 'value': {'address': '1::4/24', 'number: '3'}}
            '''
            token = pool['value'].split(' ')
            result = {'address':token[0] + '/' + get_mask(), 'number': token[1]}
            result = ifcize_param_dict(result)
            set_cfg_state(result, pool['state'])
            result = {'state': pool['state'], 'value': result}
            return result

        def make_pool_key():
            return (Type.PARAM, 'IPv6AddressPool', self.get_address_pool_name(address['value']))

        if pool and pool.get('value'):
            pools = self.get_top().get_child('IPv6AddressPool')
            pools.register_pool(make_pool_key(), make_pool_value())

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        '''Override the default so that in the case of changing the address-pool while keeping the address unchanged
        we need to removed the address so that we can modify the address-pool object on ASA, and then putting the address back.
        This is because not allow the overwrite the value of an existing ipv6 address pool, one has to delete the existing one, and add back
        the one with new value.
        '''
        def is_nochange_address_and_modified_pool():
            address_pool_key = self.get_param_dict_key('ipv6_address_pool')
            if not address_pool_key:
                return False
            return (self.get_state() == State.NOCHANGE and
                    self.get_param_state('ipv6_address_pool') == State.MODIFY)

        if is_nochange_address_and_modified_pool():
            self.generate_cli(no_asa_cfg_stack, 'no ' + self.get_cli(), response_parser = ignore_response_parser)
            self.set_state(State.CREATE)
        return SimpleType.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

    def get_param_dict_key(self, param_key):
        '''
        @param key: str, such as 'ipv6_address_with_prefix', or 'ipv6_address_pool'
        @return dictionary key tuple on APIC, such as (Type.PARAM, 'ipv6_address_with_prefix', '')
        '''
        return filter_first(lambda k: k[1] == param_key, self.delta_ifc_cfg_value['value'].keys())

    def get_param_state(self, param_key):
        key = self.get_param_dict_key(param_key)
        return self.delta_ifc_cfg_value['value'][key]['state'] if key else -1

class IPv6NeighborDiscovery(SimpleType):
    'Model after "ipv6 nd prefix x:x:x:x::/prefix|default [s1 s2] [at d1 d2] [no-advertise] [no-autoconfig] [off-link]" sub-command'
    def __init__(self, instance):
        super(IPv6NeighborDiscovery, self).__init__(instance,
            asa_gen_template='ipv6 nd prefix %(prefix_or_default)s')

    def create_asa_key(self):
        value = self.get_value()
        if value['prefix_or_default']  != 'default':
            value['prefix_or_default'] = normalize_ipv6_address(value['prefix_or_default'],
                                                                interface_addr=True)
        d1 = value.get('valid_lifetime_in_seconds')
        d2 = value.get('preferred_lifetime_in_seconds')
        if d1 and d2:
            t1 = d1.split()
            t2 = d2.split()
            if len(t1) > 3:
                d1 = ' '.join(t1[0:3])
            if len(t2) > 3:
                d2 = ' '.join(t2[0:3])
            value['valid_lifetime_in_seconds'] = d1
            value['preferred_lifetime_in_seconds'] = d2
        return self.asa_gen_template % value

    def get_cli(self):
        'Override the default implementation to take care / delimiter'
        config = self.get_value()
        if config['prefix_or_default']  != 'default':
            config['prefix_or_default'] = normalize_ipv6_address(config['prefix_or_default'],
                                                                 interface_addr=True)
        valid_lifetime_in_seconds = config.get('valid_lifetime_in_seconds')
        valid_lifetime_never_expire = config.get('valid_lifetime_never_expire')
        preferred_lifetime_in_seconds = config.get('preferred_lifetime_in_seconds')
        preferred_lifetime_never_expire = config.get('preferred_lifetime_never_expire')
        valid_lifetime_in_date = config.get('valid_lifetime_in_date')
        preferred_lifetime_in_date = config.get('preferred_lifetime_in_date')
        s1 = valid_lifetime_in_seconds if valid_lifetime_in_seconds else valid_lifetime_never_expire
        s2 = preferred_lifetime_in_seconds if preferred_lifetime_in_seconds else preferred_lifetime_never_expire
        d1 = valid_lifetime_in_date if valid_lifetime_in_date else ''
        d2 = preferred_lifetime_in_date if preferred_lifetime_in_date else ''
        result = 'ipv6 nd prefix ' + config['prefix_or_default']
        if s1 and s2:
            result += ' ' + s1 + ' ' + s2
        elif d1 and d2:
            d1 = self.get_date_without_year(d1)
            d2 = self.get_date_without_year(d2)
            result += ' at ' + d1 + ' ' + d2

        no_advertise = config.get('no_advertise')
        no_autoconfig = config.get('no_autoconfig')
        off_link = config.get('off_link')

        if no_advertise:
            result += ' no-advertise'
        else:
            if off_link:
                result += ' off-link'
            if no_autoconfig:
                result += ' no-autoconfig'
        return ' '.join(result.split())

    def validate_configuration(self):
        if not self.has_ifc_delta_cfg():
            return
        result = []
        config = self.get_value()
        address = config.get('prefix_or_default')
        if address != 'default':
            error = IPv6AddressSubnetValidator().validate(address)
            if error:
                self.report_fault(error, result, 'prefix_or_default')
        no_advertise = config.get('no_advertise')
        no_autoconfig = config.get('no_autoconfig')
        off_link = config.get('off_link')
        if no_advertise and (off_link or no_autoconfig):
            self.report_fault(
                'IPv6 neighbor discovery cannot support no_advertise in conjunction with off_link or no_autoconfig options.',
                result,
                'no_advertise')
        return result

    def get_date_without_year(self, date_cli):
        p1 = re.compile('^(\S+) (\d+) (\d+) (\S+)')
        p2 = re.compile('^(\d+) (\S+) (\d+) (\S+)')

        m1 = p1.match(date_cli)
        m2 = p2.match(date_cli)
        m = m1 if m1 else m2
        if m:
            return ' '.join(m.group(1, 2, 4))
        else:
            return date_cli

    def parse_multi_parameter_cli(self, cli):
        # Take care of the mandatory parameters
        result = super(IPv6NeighborDiscovery, self).parse_multi_parameter_cli(cli)

        'Take care of the optional parameters'
        tokens = cli.split()
        if ' at ' in cli:
            colon_index_list = [a for a,b in enumerate(tokens) if ':' in b]
            d1 = self.get_date_without_year(' '.join(tokens[5:colon_index_list[0]+1]))
            d2 = self.get_date_without_year(' '.join(tokens[(colon_index_list[0] + 1):colon_index_list[1]+1]))

            result[(Type.PARAM, 'valid_lifetime_in_date', '')] = {'state': State.NOCHANGE, 'value': d1}
            result[(Type.PARAM, 'preferred_lifetime_in_date', '')] = {'state': State.NOCHANGE, 'value': d2}
        elif len(tokens) > 5 and (tokens[4].isdigit() or tokens[4] == 'infinite') and \
            (tokens[5].isdigit() or tokens[5] == 'infinite'):
            s1 = 'valid_lifetime_in_seconds' if tokens[5].isdigit() else 'valid_lifetime_never_expire'
            s2 = 'preferred_lifetime_in_second' if tokens[5].isdigit() else 'preferred_lifetime_never_expire'
            result[(Type.PARAM, s1, '')] = {'state': State.NOCHANGE, 'value': tokens[4]}
            result[(Type.PARAM, s2, '')] = {'state': State.NOCHANGE, 'value': tokens[5]}
        if 'no-advertise' in cli:
            result[(Type.PARAM, 'no_advertise', '')] = {'state': State.NOCHANGE, 'value': 'enable'}
        else:
            if 'off-link' in cli:
                result[(Type.PARAM, 'off-link', '')] = {'state': State.NOCHANGE, 'value': 'enable'}
            if 'no-autoconfig' in cli:
                result[(Type.PARAM, 'no-autoconfig', '')] = {'state': State.NOCHANGE, 'value': 'enable'}
        return result

class MacAddress(SimpleType):
    '''
    This class represents the MAC address configuration of an interface.
    '''

    def __init__(self):
        super(MacAddress, self).__init__(
            ifc_key='mac_address',
            asa_key = 'mac-address',
            asa_gen_template='mac-address %(active_mac)s')

    def get_cli(self):
        value = self.get_value()
        if 'standby_mac' in value:
            return (self.asa_gen_template + ' standby %(standby_mac)s') % value
        else:
            return self.asa_gen_template % value

    def parse_multi_parameter_cli(self, cli):
        '''
        Override the default implementation in case the CLI does not match asa_gen_template due to optional parameter
        '''
        # Take care of the mandatory parameters
        result = super(MacAddress, self).parse_multi_parameter_cli(
            cli, alternate_asa_gen_template = self.asa_gen_template)

        # Take care of the optional parameters
        tokens = cli.split()

        # The number of tokens must >= 2, i.e. 'mac-address H.H.H [standby H.H.H]'
        option = tokens[3:]
        if len(option) > 0:
            result[(Type.PARAM, 'standby', '')] = {'state': State.NOCHANGE, 'value': ''}
            result[Type.PARAM, 'standby', '']['value'] = option[0]
        return result

class BridgeGroup(SimpleType):
    '''
    This class represents the bridge-group configuration of an interface.
    '''
    def __init__(self):
        SimpleType.__init__(self, 'bridge_group', 'bridge-group')

    def diff_ifc_asa(self, cli):
        'Override the default implementation to remember the old cli for deletion purpose'
        SimpleType.diff_ifc_asa(self, cli)
        self.delta_ifc_cfg_value['old_cli'] = cli

    def ifc2asa(self, no_asa_cfg_stack,  asa_cfg_list):
        '''Override the default implementation for modify action:
        Need to issue no command to delete old one and then issue command to set the new one.
        '''
        if self.get_action() == State.MODIFY:
            old_cli = self.get_old_cli()
            if old_cli:
                if old_cli == self.get_cli():
                    return
                self.generate_cli(no_asa_cfg_stack, 'no ' + old_cli)
        SimpleType.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

    def get_old_cli(self):
        '@return the old CLI for modification operation.'
        old_cli = self.delta_ifc_cfg_value.get('old_cli')
        if old_cli:
            return old_cli
        query_cmd = "show run " + self.parent.get_cli() + " | grep " + self.asa_key
        old_cli = self.query_asa(query_cmd)
        return old_cli.strip() if old_cli else None
