#
# Copyright (c) 2013 by Cisco Systems, Inc.
#
'''
Created on Jun 28, 2013

@author: feliu

This is the logging class.

'''

import asaio.cli_interaction as cli_interaction
from base.dmboolean import DMBoolean
from base.dmlist import DMList
from base.dmobject import DMObject
from base.simpletype import SimpleType
from state_type import Type, State
from validators import UnsupportedOnMultiContextASAValiator

class LoggingConfig(DMObject):
    '''
    This class represents the holder of all logging configuration.
    '''
    def __init__(self):
        DMObject.__init__(self, LoggingConfig.__name__)

        ifc_asa_keys = (
                        ("facility",                "logging facility"),
                        ("buffered_level",          "logging buffered"),
                        ("buffer_size",             "logging buffer-size"),
                        ("flash_maximum_allocation","logging flash-maximum-allocation"),
                        ("flash_minimum_free",      "logging flash-minimum-free")
                        )
        for (ifc, asa) in ifc_asa_keys:
            simple_type = FlashSimpleType if ifc.startswith('flash') else SimpleType
            self.register_child(simple_type(ifc, asa,
                                response_parser=cli_interaction.ignore_warning_response_parser))

        self.register_child(LoggingTrap())
        ifc_asa_keys_boolean_type = (
                        ("enable_logging",          "logging enable"),
                        ("standby",                 "logging standby"),
                        ("permit_hostdown",         "logging permit-hostdown")
                        )
        for (ifc, asa) in ifc_asa_keys_boolean_type:
            self.register_child(DMBoolean(ifc, asa, on_value="enable",
                                response_parser=cli_interaction.ignore_warning_response_parser))
        self.register_child(FlashBufferWrap())

        self.register_child(LoggingHosts())
        self.register_child(DMList('logging_message', LoggingMessage, asa_key='logging message',
                                   mini_audit_command='show run logging | grep logging message'))
        self.response_parser = cli_interaction.ignore_warning_response_parser

class FlashSimpleType(SimpleType, UnsupportedOnMultiContextASAValiator):
    'logging flash sub-commands not supported by ASA in multi-context mode'
    def __init__(self, ifc_key, asa_key, response_parser):
        SimpleType.__init__(self, ifc_key = ifc_key, asa_key = asa_key, response_parser = response_parser)

class FlashBufferWrap(DMBoolean, UnsupportedOnMultiContextASAValiator):
    def __init__(self):
        DMBoolean.__init__(self, 'flash_bufferwrap', 'logging flash-bufferwrap',
                           on_value="enable",
                           response_parser=cli_interaction.ignore_warning_response_parser)


class LoggingTrap(SimpleType):
    def __init__(self):
        SimpleType.__init__(self, ifc_key='trap_level', asa_key='logging trap',
                            response_parser=cli_interaction.ignore_warning_response_parser)

    def get_cli(self):
        '''Generate the CLI for this single logging trap config.
        '''
        cli = SimpleType.get_cli(self).lower()
        trap_level_list = [('0', 'emergencies'),
                           ('1', 'alerts'),
                           ('2', 'critical'),
                           ('3', 'errors'),
                           ('4', 'warnings'),
                           ('5', 'notifications'),
                           ('6', 'informational'),
                           ('7', 'debugging')]
        for item in trap_level_list:
            (num_level, desc_level) = item
            cli = cli.replace(num_level, desc_level)
        return cli

class LoggingHost(SimpleType):
    def __init__(self, name):
        SimpleType.__init__(self, name, is_removable = True,
                            asa_gen_template='logging host %(interface)s %(ip)s',
                            response_parser = cli_interaction.ignore_warning_response_parser)

    def create_asa_key(self):
        '''Create the the asa key identifies this object
        @return str
        '''
        return self.get_cli()

    def get_cli(self):
        '''Generate the CLI for this single logging host config.
        '''

        value = self.get_value_with_interface()
        cli = []
        cli.append(self.asa_gen_template % value)
        if 'protocol_port' in value:
            port = value['protocol_port']
            port = port.replace('tcp/', '6/')
            port = port.replace('udp/', '17/')
            cli.append(' ' + port)
        if 'secure' in value:
            cli.append(' secure')
        if 'emblem' in value:
            cli.append(' format emblem')
        return ''.join(cli).lower()

    def get_cli_prefix(self):
        return self.asa_gen_template % self.get_value_with_interface()

    def get_value_with_interface(self):
        'Allow to be overridden to supply interface'
        return self.get_value()

    def parse_multi_parameter_cli(self, cli):
        '''
        Override the default implementation in case the CLI does not match asa_gen_template due to optional parameter
        '''
        # Take care of the mandatory parameters
        result = SimpleType.parse_multi_parameter_cli(self, cli, alternate_asa_gen_template = self.asa_gen_template)

        'Take care of the optional parameters'
        tokens = cli.split()

        # The number of mandatory parameters is 4, i.e. 'logging host management 2.2.2.2'
        if len(tokens) == 4:
            return result # no optional parameter

        option = tokens[4:]
        if 'secure' in option: # e.g. "logging host mgmt 2.2.2.2 secure
            result[Type.PARAM, 'secure', ''] = {'state': State.NOCHANGE,
                                                'value': ''}
        if 'emblem'in option: # e.g. "logging host mgmt 2.2.2.2 format emblem
            result[Type.PARAM, 'emblem', ''] = {'state': State.NOCHANGE,
                                                'value': ''}
        if '/' in option[0]: # e.g. "logging host mgmt 2.2.2.2 6/1471'
            result[Type.PARAM, 'protocol_port', ''] = {'state': State.NOCHANGE,
                                                       'value': option[0]}
        return result

    def is_my_cli(self, cli):
        return cli == self.asa_key

class LoggingHosts(DMList):
    'Container for the LoggingHost object'

    def __init__(self):
        super(LoggingHosts, self).__init__(
            'logging_host', LoggingHost, 'logging host',
            mini_audit_command='show run logging | grep logging host')

    def is_my_cli(self, cli):
        if isinstance(cli, basestring):
            if cli.startswith(self.asa_key + ' management'):
                    return True

class LoggingMessage(SimpleType):
    def __init__(self, name):
        SimpleType.__init__(self, name, is_removable=True,
                            asa_gen_template='logging message %(message)s',
                            response_parser = cli_interaction.ignore_warning_response_parser)

    def get_cli(self):
        '''Generate the CLI for this single logging message config.
        '''
        assert self.has_ifc_delta_cfg()
        config = self.get_value()
        level = config.get('level')
        level = ' level ' + level if level else ''
        return 'logging message %(message)s' % config + level

    def create_asa_key(self):
        '''Create the the asa key identifies this object
        @return str
        '''
        assert self.has_ifc_delta_cfg()
        value = self.get_value()
        return 'logging message %(message)s' % value

    def parse_multi_parameter_cli(self, cli):
        '''
        Override the default implementation in case the CLI does not match asa_gen_template due to optional parameter
        '''
        # Take care of the mandatory parameters
        result = SimpleType.parse_multi_parameter_cli(self, cli, alternate_asa_gen_template = self.asa_gen_template)

        'Take care of the optional parameters'
        tokens = cli.split()

        # The number of mandatory parameters is 3, i.e. 'logging message 106015'
        if len(tokens) == 3:
            return result # no optional parameter

        result[(Type.PARAM, 'level', '')] = {'state': State.NOCHANGE, 'value': ''}

        option = tokens[3:]
        if 'level' in option: # e.g. "logging message 106015 level critical
            result[Type.PARAM, 'level', '']['value'] = option[1]
        return result
