'''
Copyright (c) 2013 by Cisco Systems, Inc.

@author: emilymw
'''
import re
from asaio.cli_interaction import object_response_parser
from base.simpletype import SimpleType
from base.dmlist import DMList
from base.compositetype import CompositeType, Description
from validators import IPAddressValidator, IPAddressRangeValidator, IPAddressSubnetValidator, VersionFQDNValidator
from state_type import State, Type
from utils.util import normalize_ipv4_address_and_mask_4_asa, binary_search

class _ObjectNetworks(DMList):
    'Base class for the containers'
    def __init__(self, name, child_class, asa_sub_key):
        DMList.__init__(self, name, child_class, asa_key = 'object network')
        self.asa_sub_key = asa_sub_key

    def is_my_cli(self, cli):
        if isinstance(cli, basestring):
            return False # not handled by us
        if not cli.command.startswith(self.asa_key):
            return False
        return any(re.compile(self.asa_sub_key).match(str(cmd).strip()) for cmd in  cli.sub_commands)

    def get_translator(self, cli):
        '''
        Override the default implementation to speed up CLI to MO lookup.
        The optimization is to use binary search instead of linear search.
        '''
        if not self.is_my_cli(cli):
            return

        '''
        Find the translator from its children.
        Using binary search, the result is the same as the following algorithm:

        for child in self.children.values():
            result = child.get_translator(cli)
            if result:
                break
        '''
        if not hasattr(self, '_cli2obj_dict'):
            self._cli2obj_dict = {}
            for a in self.children.values():
                self._cli2obj_dict[str(a.get_cli())] = a
            self._sorted_clis = self._cli2obj_dict.keys()
            self._sorted_clis.sort()
        cli_str = str(cli if isinstance(cli, str) else cli.command)
        result = binary_search(self._cli2obj_dict, self._sorted_clis, cli_str)
        if result:
            return result
        'Create a child that is to generate "no" cli, by the invocation of child.diff_ifc_asa'
        result = self.child_class(cli)
        self.register_child(result)
        result.has_matched_cli = True
        return result

class HostObjects(_ObjectNetworks):
    'Container for HostObject'
    def __init__(self):
        _ObjectNetworks.__init__(self, 'HostObject', HostObject, asa_sub_key = 'host')

class RangeObjects(_ObjectNetworks):
    'Container for RangeObject'
    def __init__(self):
        _ObjectNetworks.__init__(self, 'RangeObject', RangeObject, asa_sub_key = 'range')

class SubnetObjects(_ObjectNetworks):
    'Container for SubnetObject'
    def __init__(self):
        _ObjectNetworks.__init__(self, 'SubnetObject', SubnetObject, asa_sub_key = 'subnet')

class FQDNObjects(_ObjectNetworks):
    'Container for FQDNObject'
    def __init__(self):
        _ObjectNetworks.__init__(self, 'FQDNObject', FQDNObject, asa_sub_key = 'fqdn')

class _ObjectNetwork(CompositeType):
    '''Base class 'object network' CLI object. This is the base type for all rest classes in this module
    '''
    def __init__(self, name):
        SimpleType.__init__(self, name,
                            asa_gen_template='object network %(name)s',
                            response_parser = object_response_parser)
        self.register_child(Description())
        'parent_key is the key to the IFC configuration folder for deleted objects during diff_asa operation'
        self.parent_key = (Type.FOLDER, 'NetworkObject', '')

    def create_asa_key(self):
        return self.get_cli()

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        'Override the default implementation to set the value of "name" with instance ID if it is not there yet.'
        name_key = filter(lambda k: k[1] == 'name', delta_ifc_cfg_value['value'].keys())
        #name_key will be there if the config is on ASA but not in IFC
        if not name_key:
            delta_ifc_cfg_value['value'][(Type.PARAM, 'name', 'name')] = {'state': 1, 'value': delta_ifc_key[2]}
        return CompositeType.populate_model(self, delta_ifc_key, delta_ifc_cfg_value)

    def create_delta_ifc_key(self, cli):
        'override the default to return object name as the key'
        if isinstance(cli, str):
            key = cli.split()[2]
        else:
            key = cli.command.split()[2]
        return (Type.PARAM, self.parent.ifc_key, key)

    def diff_ifc_asa(self, cli):
        """
        Override the default implementation to take care of delete operation.
        """
        if self.has_ifc_delta_cfg():
            return CompositeType.diff_ifc_asa(self, cli)
        'Delete operation'
        self.delta_ifc_key = self.create_delta_ifc_key(cli)
        self.delta_ifc_cfg_value = {'state': State.DESTROY, 'value': self.parse_cli(cli)}
        """
        Add it to its container's delta_ifc_cfg_value for the entry (5, 'NetworkObject', '')".
        """
        ancestor = self.get_ifc_delta_cfg_ancestor()
        if not ancestor.delta_ifc_cfg_value['value'].has_key(self.parent_key):
            ancestor.delta_ifc_cfg_value['value'][self.parent_key] = {'state': 1, 'value': {}}
        parent_ifc_folder = ancestor.delta_ifc_cfg_value['value'][self.parent_key]['value']
        parent_ifc_folder[self.delta_ifc_key] = self.delta_ifc_cfg_value


class HostSubCommand(SimpleType, IPAddressValidator):
    'Model after "host" sub-command'
    def __init__(self):
        SimpleType.__init__(self, ifc_key='host_ip_address', asa_key = 'host')

class HostObject(_ObjectNetwork):
    '''Model after 'object network\n host' object
    '''
    def __init__(self, name):
        _ObjectNetwork.__init__(self, name);
        self.register_child(HostSubCommand())

class SubnetSubCommand(SimpleType, IPAddressSubnetValidator):
    'Model after "subnet" sub-command'
    def __init__(self):
        SimpleType.__init__(self, ifc_key='network_ip_address', asa_key = 'subnet')

    def get_cli(self):
        'Override the default implementation to take care / delimiter'
        value = self.get_value()
        if '.' in value: #ipv4 address
            value = normalize_ipv4_address_and_mask_4_asa(value)
        return self.asa_gen_template % value

    def parse_single_parameter_cli(self, cli):
        'Override the default implementation because the IPv4 network address is entered as an address and netmask tuple'
        return '/'.join(cli.split()[1:])

class SubnetObject(_ObjectNetwork):
    '''Model after 'object network\n subnet' object
    '''
    def __init__(self, name):
        _ObjectNetwork.__init__(self, name);
        self.register_child(SubnetSubCommand())

class RangeSubCommand(SimpleType, IPAddressRangeValidator):
    'Model after "range" sub-command'
    def __init__(self):
        SimpleType.__init__(self, ifc_key='ip_address_range', asa_key = 'range')

    def get_cli(self):
        'Override the default implementation to take care "-" delimiter'
        value = ' '.join(self.get_value().split('-'))
        return self.asa_gen_template % value

    def parse_single_parameter_cli(self, cli):
        'Override the default implementation because to use "-" as the delimiter'
        return '-'.join(cli.split()[1:])

class RangeObject(_ObjectNetwork):
    '''Model after 'object network\n range" object
    '''
    def __init__(self, name):
        _ObjectNetwork.__init__(self, name);
        self.register_child(RangeSubCommand())

class FqdnSubCommand(SimpleType, VersionFQDNValidator):
    "Model after 'fqdn' sub-command. The value is in the form of: {v4|v6} <fqdn-string>"
    def __init__(self):
        SimpleType.__init__(self, ifc_key='fqdn', asa_key = 'fqdn')

    def parse_single_parameter_cli(self, cli):
        'Override the default implementation because to use the multiple parameters'
        return ' '.join(cli.split()[1:])

class FQDNObject(_ObjectNetwork):
    '''Model after 'object network\n fqdn" object
    '''
    def __init__(self, name):
        _ObjectNetwork.__init__(self, name);
        self.register_child(FqdnSubCommand())
