'''
Created on Oct 22, 2013

@author: dli
Copyright (c) 2013 by Cisco Systems
'''
import re
from asaio.cli_interaction import object_response_parser
from base.simpletype import SimpleType
from base.dmlist import DMList
from base.compositetype import CompositeType, Description
from object_group import ObjectGroupList, ObjectList
from validators import IPAddressValidator, IPAddressSubnetValidator
from state_type import Type, State
from translator.structured_cli import convert_to_structured_commands,\
    StructuredCommand
from utils.util import filter_first, normalize_ipv4_address_and_mask_4_asa
from translator.connector import Connector

class NetworkObjectGroups(ObjectGroupList):
    'Container of NetworkObjectGroup'
    def __init__(self):
        super(NetworkObjectGroups, self).__init__(
            name='NetworkObjectGroup', asa_key='object-group network',
            child_class=NetworkObjectGroup)

    def get_translator(self, cli):
        '''Override the default implementation so as to ignore non-EPG network-object-group for ASA-DP lite.
        We do not support general network-object-groups in ASA-DP lite, but dynamic EPG network-object-group update.
        '''
        if self.get_top().IS_LITE:
            cmd = cli if isinstance(cli, basestring) else cli.command
            if cmd.startswith('object-group network '):
                nog_name = cmd.split(' ')[-1]
                if not Connector.is_epg_nog(nog_name):
                    return
            elif cmd == 'forward-reference enable':
                self.set_forward_reference_enabled_on_asa()
                return
        return ObjectGroupList.get_translator(self, cli)

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        n = len(no_asa_cfg_stack)
        ObjectGroupList.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)
        if self.get_top().IS_LITE and n < len(no_asa_cfg_stack) and not self.is_forward_reference_enabled_on_asa():
            '''Make sure we have 'forward-reference enable' issued before the removing EPG network-object-groups
            if it is not yet on ASA, see CSCvf21889.
            '''
            self.generate_cli(no_asa_cfg_stack, 'forward-reference enable')

    def set_forward_reference_enabled_on_asa(self):
        self.parent.delta_ifc_cfg_value['value'][(Type.PARAM, 'forward_reference_enabled', '')] = {'state': State.NOCHANGE, 'value': True}

    def is_forward_reference_enabled_on_asa(self):
        return self.parent.delta_ifc_cfg_value['value'].get((Type.PARAM, 'forward_reference_enabled', '')) != None

class NetworkObjectGroup(CompositeType):
    'Model for "object-group network" CLI'
    def __init__(self, name):
        SimpleType.__init__(self, name,
                            asa_gen_template='object-group network %(name)s');
        self.register_child(Description())
        self.register_child(OGNDMList(name='host_ip_address',    child_class = HostObject,    asa_key = 'network-object host'))
        self.register_child(OGNDMList(name='object_name',        child_class = ObjectObject,  asa_key = 'network-object object'))
        self.register_child(OGNDMList(name='network_ip_address', child_class = NetworkObject, asa_key = re.compile('^network-object [^h]')))
        self.register_child(OGNDMList(name='object_group_name',  child_class = GroupObject,   asa_key = 'group-object'))

    def create_asa_key(self):
        return self.get_cli()

    def get_name(self):
        'Used by ObjectGroupList to access the name of the object group'
        return self.get_value().get('name')

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        'Override the default implementation to set the value of "name" with instance ID if it is not there yet.'
        name_key = filter(lambda k: k[1] == 'name', delta_ifc_cfg_value['value'].keys())
        #name_key will be there if the config is on ASA but not in IFC
        if not name_key:
            delta_ifc_cfg_value['value'][(Type.PARAM, 'name', 'name')] = {'state': 1, 'value': delta_ifc_key[2]}
        return CompositeType.populate_model(self, delta_ifc_key, delta_ifc_cfg_value)

class OGNDMList(ObjectList):
    'A special DMList that takes care of mini_audit. OGN stand for ObjectGroupNetwork'
    def __init__(self, name, child_class, asa_key):
        super(OGNDMList, self).__init__(name, child_class, asa_key,
                                        'show run object-group network')

    def read_asa_config(self):
        """
        Override the default to get our config.
        ASA does not take 'show run object-group network <name>' command. So have have to figure out our network
        object-group from list of object-groups from the output of 'show run object-group network'.
        """
        if hasattr(self.parent.parent, 'cached_asa_config'):
            clis = self.parent.parent.cached_asa_config
        else:
            clis = DMList.read_asa_config(self)
            if not clis:
                return
            clis = convert_to_structured_commands(clis.split('\n'))
            self.parent.parent.cached_asa_config = clis
        mine = filter_first(lambda cli: isinstance(cli, StructuredCommand) and cli.command == self.mode_command,  clis)
        if mine:
            return '\n'.join(mine.sub_commands)

class _NetworkObject(SimpleType):
    'base class for network-object and group-object sub-command'
    def __init__(self, ifc_key, asa_gen_template):
        SimpleType.__init__(self, ifc_key = ifc_key,
                            asa_gen_template = asa_gen_template,
                            response_parser = object_response_parser)

    def create_asa_key(self):
        return self.get_cli()

class HostObject(_NetworkObject, IPAddressValidator):
    'Model after "network-object host" sub-command'
    def __init__(self, instance):
        _NetworkObject.__init__(self, ifc_key = instance, asa_gen_template='network-object host %s')

class ObjectObject(_NetworkObject):
    'Model after "network-object object" sub-command'
    def __init__(self, instance):
        _NetworkObject.__init__(self, ifc_key = instance, asa_gen_template='network-object object %s')

class NetworkObject(_NetworkObject, IPAddressSubnetValidator):
    'Model after "network-object ipv4_address netmask" and "network-object ipv6_address/prefix_length" sub-command'
    def __init__(self, instance):
        _NetworkObject.__init__(self, ifc_key = instance, asa_gen_template='network-object %s')

    def get_cli(self):
        'Override the default implementation to take care / delimiter'
        value = self.get_value()
        if '.' in value: #ipv4 address
            value = normalize_ipv4_address_and_mask_4_asa(value)
        return self.asa_gen_template % value

    def parse_single_parameter_cli(self, cli):
        'Override the default implementation because the IPv4 network address is entered as an address and netmask tuple'
        return '/'.join(cli.split()[1:])

class GroupObject(_NetworkObject):
    'Model after "group-object" sub-command'
    def __init__(self, instance):
        _NetworkObject.__init__(self, ifc_key = instance, asa_gen_template='group-object %s')
