'''
Created on Dec 20, 2013

@author: dli
Copyright (c) 2013 by Cisco Systems
'''

from base.dmlist import DMList
from structured_cli import convert_to_structured_commands, StructuredCommand
from utils.util import filter_first
from translator.state_type import State

class ObjectGroupList(DMList):
    """
    Model after object-group CLIs.
    This is a special DMList where the order of the children may not be quite right when first populated, in that
    a child referencing another child may come before the child being referenced due to the arbitrary order of
    Python dictionary. On generating CLI, we therefore, have to make sure that ASA will not reject an object-group
    with forward reference, i.e. a reference to an object object-group object yet defined. 

    The children of this list need to provide a function named get_name() that
    returns the name of the object group that is used to reference it.
    """

    def __init__(self, name, child_class, asa_key='', mini_audit_command=None,
                 object_group_key='object_group_name'):
        '''
        The object_group_key parameter is the name of the key in the object
        group that refers to another object group.
        '''
        super(ObjectGroupList, self).__init__(name, child_class, asa_key,
                                              mini_audit_command)
        self.object_group_key = object_group_key

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        'Override default implementation to take care of forward reference'
        if self.get_top().compare_asa_version(9, 3, 2, 0) < 0:
            """
            'forward-reference' CLI unavailable in ASA earlier than 9.3(2).
            So we have to sort the object-group referenced by other object-group
            comes first.
            """
            if self.has_forward_reference():
                self.sort_children()
            DMList.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)
            return
        'Use forward-reference CLI to take care of forward reference'
        stack_length = len(no_asa_cfg_stack)
        list_length = len(asa_cfg_list)
        DMList.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)
        if (stack_length < len(no_asa_cfg_stack)) or (list_length < len(asa_cfg_list)):
            if not self.has_forward_reference():
                return
            """
            Add 'forward-reference enable' as the first command, so that ASA won't reject the
            subsequent commands.
            """
            FORWARD_REFERENCE = 'forward-reference enable'
            'avoid duplicate of FORWARD_REFERENCE commands'
            hit = filter_first(lambda cmd: str(cmd) == FORWARD_REFERENCE, no_asa_cfg_stack)
            if hit:
                no_asa_cfg_stack.remove(hit)
            self.generate_cli(no_asa_cfg_stack, FORWARD_REFERENCE)

    def has_forward_reference(self):
        'return true if any of the member has forward reference to other object-group'
        def has_undefined_member(object_group, defined_object_groups):
            group_objects = filter_first(lambda child: child.ifc_key == self.object_group_key, object_group.children.values())
            def is_undefined(go):
                if go.get_state() in (State.CREATE, State.MODIFY):
                    'Check to see if go is defined ahead of object_group'
                    if go.get_value() in defined_object_groups:
                        return False
                    'check to see if the object-group go is already on ASA'
                    og = self.get_child(go.get_value())
                    return og == None or og.get_state() == State.CREATE
            undefined_member = filter_first(is_undefined, group_objects)
            return undefined_member != None
        defined_object_groups = []
        for child in self.children.values():
            if child.get_state() == State.NOCHANGE:
                'we do not care about this child, because it does not generate CLI'
                continue
            if has_undefined_member(child, defined_object_groups):
                return True
            defined_object_groups.append(child.get_name())

    def sort_children(self):
        'Make sure object-group referenced by other object-group comes in earlier in CLI'
        def reference(object_group, object_group_name):
            '@return True if object_group  directly or indirectly references an object group of name object_group_name; or False'
            def get_object_group(name):
                '@return a named object group'
                hits = filter(lambda child: child.get_name() == name, self.children.values())
                return hits[0] if hits else None

            group_objects = filter_first(lambda child: child.ifc_key == self.object_group_key, object_group.children.values())
            for group_object in group_objects.children.values():
                if group_object.get_value() == object_group_name:
                    return True #directly referenced
                object_group = get_object_group(group_object.get_value())
                if not object_group:
                    continue
                if object_group.get_state() == State.NOCHANGE:
                    'object_group is already on ASA, its position here is not important'
                    continue
                if reference(object_group, object_group_name):
                    return True #indirectly referenced
            return False

        def compare(x, y):
            if reference(x, y.get_name()):
                result = 1
            elif reference(y, x.get_name()):
                result = -1
            else:
                result = 0
            return result

        """
        @note: It would be simpler to use the builtin sorted function, like below:

            child_list = sorted(self.children.values(), cmp = compare)

        However, it does not work in case, CSCun10249, where one has three groups, such as A, B, C, and
        A references C, but B has no referential relationship with any one. Looks like for
        sorted function to work, its input must be total relation, i.e. every pair of its elements
        must be in referential relationship in this case.
        """

        child_list = []
        for child in self.children.values():
            if child.get_state() == State.NOCHANGE:
                """
                No CLI will be generated for this one, so we do not worry about its position for
                its definition, nor its reference.
                Warning: this may cause reference function to complain about ref has no definition.
                """
                continue
            if child.get_state() == State.MODIFY:
                """
                The CLI is already in ASA, so its position relative to others that referencing it is
                not important
                """
                refferer = None
            else:
                refferer = filter_first(lambda x: compare(x, child) == 1, child_list)
            if refferer:
                'place child ahead of the referrer'
                index = child_list.index(refferer)
                child_list.insert(index, child)
            else:
                child_list.append(child)

        self.children.clear()
        for child in child_list:
            self.register_child(child)

    def get_translator(self, cli):
        '''
        Override the default implementation to speed up CLI to MO lookup.
        The optimization is to not lookup a network object group which has already a matching CLI.
        '''
        if not self.is_my_cli(cli):
            return

        'Find the translator from its children'
        result = None
        for child in self.children.values():
            if hasattr(child, 'has_matched_cli'):
                continue
            result = child.get_translator(cli)
            if result:
                result.has_matched_cli = True
                break
        if result:
            return result
        'Create a child that is to generate "no" cli, by the invocation of child.diff_ifc_asa'
        result = self.child_class(cli)
        self.register_child(result)
        result.has_matched_cli = True
        return result

class ObjectList(DMList):
    '''
    A list of objects for use with ObjectGroupList.  It has special handling for
    the mini_audit function.
    '''

    def __init__(self, name, child_class, asa_key, mini_audit_command):
        super(ObjectList, self).__init__(name, child_class, asa_key,
                                         mini_audit_command)

    def read_asa_config(self):
        """
        Override the default to get our config.
        ASA does not take 'show run object-group <type> <name>' command. So have
        have to figure out our object-group from list of object-groups from the
        output of 'show run object-group <type>'.
        """
        if hasattr(self.parent.parent, 'cached_asa_config'):
            clis = self.parent.parent.cached_asa_config
        else:
            clis = super(ObjectList, self).read_asa_config()
            if not clis:
                return
            clis = convert_to_structured_commands(clis.split('\n'))
            self.parent.parent.cached_asa_config = clis
        mine = filter_first(lambda cli: isinstance(cli, StructuredCommand) and
                            cli.command == self.mode_command,  clis)
        if mine:
            return '\n'.join(mine.sub_commands)
