'''
Created on Dec 22, 2014

@author: jeffryp

Copyright (c) 2014 by Cisco Systems, Inc.
All rights reserved.

Classes used for BGP.
'''

import re

from asaio.cli_interaction import CLIInteraction
from asaio.cli_interaction import ignore_info_response_parser, ignore_warning_response_parser, ignore_response_parser
from common import Redistribute, RouterCommon, RouterID
from translator.base.compositetype import CompositeType
from translator.base.dmlist import DMList
from translator.base.dmobject import DMObject
from translator.base.simpletype import SimpleType
from translator.state_type import State, Type
from translator.structured_cli import StructuredCommand
from utils.util import netmask_from_prefix_length, prefix_length_from_netmask
from utils.util import normalize_ip_address
from utils.util import query_asa_n

class AddressFamily(RouterCommon, CompositeType):
    'Address family configuration'

    SUB_MODE_EXIT_CLI = 'exit-address-family'

    def __init__(self, ifc_key, ipv6=False):
        super(AddressFamily, self).__init__(ifc_key, 'address-family')
        self.ipv6 = ipv6

        if not ipv6:
            self.register_child(BGPRouterID())
        self.register_child(Neighbors())
        self.register_child(Networks())
        self.register_child(Redistribute('connected'))
        self.register_child(Redistribute('static'))
        self.register_child(Redistribute('ospf'))

    def get_translator(self, cli):
        self.asa_key = self.create_asa_key()
        return super(AddressFamily, self).get_translator(cli)

    def create_asa_key(self):
        return self.get_cli()

    def create_missing_ifc_delta_cfg(self):
        if 'redistribute' not in self.value:
            self.value['redistribute'] = {}

    def diff_ifc_asa(self, cli):
        if isinstance(cli, StructuredCommand):
            for cmd in cli.sub_commands:
                translator = self.get_child_translator(cmd)
                if translator:
                    translator.diff_ifc_asa(cmd)

    def get_cli(self):
        return self.asa_gen_template % self.get_family()

    def get_family(self):
        return 'ipv6' if self.is_ipv6() else 'ipv4'

    def get_sub_commands(self):
        sub_commands = super(AddressFamily, self).get_sub_commands()
        if sub_commands:
            # Add sub-mode enter and exit CLI
            sub_commands.insert(0, self.get_cli())
            sub_commands.append(self.SUB_MODE_EXIT_CLI)
        return sub_commands

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.has_ifc_delta_cfg():
            # IPv6 is supported starting with ASA version 9.3.2
            if (self.is_ipv6() and
                    self.get_top().compare_asa_version(9, 3, 2) < 0):
                return

            cli_list = []
            no_cli_stack = []
            child_mode_command = self.get_child_mode_command()
            self.ifc2asa_with_mode(no_cli_stack, cli_list,
                                   child_mode_command)
            if cli_list or no_cli_stack:
                # The no CLI goes to the asa_cfg_list, not the no_asa_cfg_list
                asa_cfg_list.extend(reversed(no_cli_stack))
                asa_cfg_list.extend(cli_list)
                # Exit address-family sub-mode
                self.generate_cli(asa_cfg_list,
                                  self.SUB_MODE_EXIT_CLI,
                                  mode_command=child_mode_command)

    def is_ipv6(self):
        return self.ipv6

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.value = delta_ifc_cfg_value['context'][self.get_family()]
        for child in self.children.itervalues():
            child.populate_model(delta_ifc_key, self.value)

    def should_generate_router_id(self):
        '''
        The router-id CLI is a sub-command of the 'address-family ipv4' command
        for ASA versions prior to 9.3.2
        '''

        return self.get_top().compare_asa_version(9, 3, 2) < 0

class BGP(CompositeType):
    'BGP configuration'

    def __init__(self, instance):
        super(BGP, self).__init__(instance, 'router bgp')
        self.mini_audit_command = 'show running-config router bgp'
        self.mode_response_parser = ignore_info_response_parser

        self.register_child(GracefulRestart())
        self.register_child(BGPRouterID())
        self.register_child(Timers())
        self.register_child(AddressFamily('ipv4'))
        self.register_child(AddressFamily('ipv6', ipv6=True))

    def create_asa_key(self):
        return self.get_cli()

    def get_translator(self, cli):
        self.asa_key = self.create_asa_key()
        return super(BGP, self).get_translator(cli)

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.delta_ifc_key = Type.BGPDEV, '', cli
            self.value = {'state': State.DESTROY,
                          'localAS': self.parse_cli(cli)}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] =  self.value
        elif self.get_state() != State.DESTROY:
            super(BGP, self).diff_ifc_asa(cli)

    def get_asn(self):
        return self.value['localAS']

    def get_cli(self):
        return self.asa_gen_template % self.value['localAS']

    def get_state(self):
        return self.value['state']

    def set_state(self, state):
        self.value['state'] = state

    def has_ifc_delta_cfg(self):
        return (hasattr(self, 'value') and
                'localAS' in self.value)

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.has_ifc_delta_cfg():
            if (self.get_state() == State.MODIFY and
                    not self.get_top().is_audit):
                self.mini_audit()

            # For multiple mode, need to create the common BGP process in the
            # system context
            if (self.get_state() == State.CREATE and
                    self.get_top().system_context_accessible()):
                self.generate_cli(asa_cfg_list,
                                  self.get_cli(),
                                  is_system_context=True)

            if self.get_state() == State.DESTROY:
                # For multiple mode, need to delete the common BGP process in
                # the system context if no other contexts are using BGP
                if self.should_remove_system_process():
                    self.generate_cli(no_asa_cfg_stack,
                                      'no ' + self.get_cli() + ' noconfirm',
                                      is_system_context=True)
                else:
                    self.generate_cli(no_asa_cfg_stack, 'no ' + self.get_cli())
                return

            cli_list = []
            no_cli_stack = []
            super(BGP, self).ifc2asa(no_cli_stack, cli_list)
            if cli_list or no_cli_stack:
                # The no CLI goes to the asa_cfg_list, not the no_asa_cfg_list
                asa_cfg_list.extend(no_cli_stack)
                asa_cfg_list.extend(cli_list)
            elif self.get_state() == State.CREATE:
                # There is no sub-command configuration so need to generate
                # the router CLI
                self.generate_cli(asa_cfg_list,
                                  self.get_cli(),
                                  response_parser=ignore_info_response_parser)

    def is_the_same_cli(self, cli):
        # Handle the case for no sub-commands
        if isinstance(cli, basestring):
            return SimpleType.is_the_same_cli(self, cli)
        return super(BGP, self).is_the_same_cli(cli)

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.delta_ifc_key = delta_ifc_key
        self.value = delta_ifc_cfg_value
        translate_ifc_cfg(self.value)
        for child in self.children.itervalues():
            child.populate_model(delta_ifc_key, self.value)

    def should_generate_router_id(self):
        '''
        The router-id CLI is a sub-command of the 'router bgp' command for ASA
        versions of 9.3.2 and above
        '''

        return self.get_top().compare_asa_version(9, 3, 2) >= 0

    def should_remove_system_process(self):
        '''
        Determine if the BGP process in the system context should be removed.

        If there is only one user context with BGP configuration, then when it
        is removed, the BGP process in the system context should be removed.

        To determine this, get a list of all the contexts.  Look for BGP
        configuration in each of the contexts.  If the only context with BGP
        configuration is the same as the current context, then return True.
        '''

        top = self.get_top()
        if top.system_context_accessible():
            create_cli = lambda name: CLIInteraction(
                'show run router bgp',
                mode_command='changeto context ' + name,
                response_parser=lambda response: name if response else None)
            results = query_asa_n(
                top.get_device(),
                map(create_cli, top.get_all_context_names()))[0]
            if results:
                bgp_contexts = filter(
                    None,
                    map(lambda cli_result: cli_result and cli_result.err_msg,
                        results))
                return bgp_contexts == [top.get_asa_context_name()]

class BGPs(DMList):
    'Container for BGP objects'

    def __init__(self):
        super(BGPs, self).__init__(BGP.__name__, BGP, 'router bgp')

    def get_asn(self):
        'Returns the autonomous system number of the active BGP process'

        for bgp in self.children.itervalues():
            if bgp.has_ifc_delta_cfg() and bgp.get_state() != State.DESTROY:
                return bgp.get_asn()

class BGPRouterID(RouterID):
    'Router ID specialized for BGP'

    def __init__(self):
        super(BGPRouterID, self).__init__('rtrId', 'bgp router-id')

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.parent.should_generate_router_id():
            super(BGPRouterID, self).ifc2asa(no_asa_cfg_stack, asa_cfg_list)

class GracefulRestart(SimpleType):
    'Graceful restart configuration'

    ASA_KEY = 'bgp graceful-restart stalepath-time'
    CLI_PATTERN = None
    IFC_KEY = 'gracefulRestart'

    def __init__(self):
        super(GracefulRestart, self).__init__(
            ifc_key=GracefulRestart.IFC_KEY,
            asa_key=GracefulRestart.ASA_KEY,
            asa_gen_template=GracefulRestart.ASA_KEY + ' %(stale)s',
            response_parser=ignore_warning_response_parser)

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.value = self.parse_cli(cli)
            self.set_state(State.DESTROY)
            self.parent.value[GracefulRestart.IFC_KEY] =  self.value
        elif self.get_state() != State.DESTROY:
            if self.is_the_same_cli(cli):
                self.set_state(State.NOCHANGE)
            else:
                self.set_state(State.MODIFY)

    @classmethod
    def get_cli_pattern(cls):
        if not cls.CLI_PATTERN:
            cls.CLI_PATTERN = re.compile(GracefulRestart.ASA_KEY + ' (\S+)')
        return cls.CLI_PATTERN

    def get_config_path(self):
        return self.parent.get_config_path()

    def get_state(self):
        return self.value['state']

    def has_ifc_delta_cfg(self):
        return hasattr(self, 'value')

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        # Graceful restart is not supported in multi-mode
        if not self.get_top().is_multi_mode_asa():
            if self.has_ifc_delta_cfg() and self.get_state() == State.CREATE:
                self.generate_cli(asa_cfg_list, 'bgp graceful-restart')
            super(GracefulRestart, self).ifc2asa(no_asa_cfg_stack, asa_cfg_list)

    def set_state(self, state):
        self.value['state'] = state

    def get_value(self):
        return self.value

    def parse_cli(self, cli):
        m = self.get_cli_pattern().match(cli)
        if m:
            result = {}
            result['stale'] = m.group(1)
            return result

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if GracefulRestart.IFC_KEY in delta_ifc_cfg_value:
            self.value = delta_ifc_cfg_value[GracefulRestart.IFC_KEY]

class Neighbor(DMObject):
    'Neighbor configuration'

    def __init__(self, instance):
        super(Neighbor, self).__init__(instance, 'neighbor')
        self.register_child(NeighborRemoteAS())
        self.register_child(NeighborActivate())
        self.register_child(NeighborNextHopSelf())

    def get_cli(self):
        return self.asa_key + ' %s' % self.get_key()

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.value = {}
            self.set_state(State.DESTROY)
            self.parent.get_value()[self.get_key()] = self.value
        elif self.get_state() != State.DESTROY:
            for child in self.children.itervalues():
                translator = child.get_translator(cli)
                if translator:
                    translator.diff_ifc_asa(cli)
                    return

    def get_config_path(self):
        return self.parent.get_config_path()

    def get_state(self):
        return self.value['state']

    def set_state(self, state):
        self.value['state'] = state

    def has_ifc_delta_cfg(self):
        return hasattr(self, 'value')

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.has_ifc_delta_cfg():
            state = self.get_state()
            if state == State.DESTROY:
                self.generate_cli(no_asa_cfg_stack, 'no ' + self.get_cli())
            elif state in (State.CREATE, State.MODIFY):
                for child in self.children.itervalues():
                    child.mode_command = self.mode_command
                    child.ifc2asa(no_asa_cfg_stack, asa_cfg_list)

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.value = delta_ifc_cfg_value
        for child in self.children.itervalues():
            child.populate_model(delta_ifc_key, self.value)

class Neighbors(DMList):
    'Container for Neighbor objects'

    CLI_PATTERN = None

    def __init__(self):
        super(Neighbors, self).__init__(Neighbor.__name__, Neighbor, 'neighbor')

    @classmethod
    def get_cli_pattern(cls):
        if not cls.CLI_PATTERN:
            cls.CLI_PATTERN = re.compile('neighbor (\S+) .*')
        return cls.CLI_PATTERN

    def get_config_path(self):
        return self.parent.get_config_path()

    def get_translator(self, cli):
        if isinstance(cli, StructuredCommand):
            cli = cli.command
        m = self.get_cli_pattern().match(cli.strip())
        if m:
            ip = m.group(1)
            neighbor = self.get_child(ip)
            if not neighbor:
                neighbor = self.child_class(ip)
                self.register_child(neighbor)
            return neighbor

    def get_value(self):
        return self.value

    def has_ifc_delta_cfg(self):
        return hasattr(self, 'value')

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.value = delta_ifc_cfg_value['neighbor']
        for key, value in self.value.iteritems():
            if isinstance(value, dict):
                child = self.child_class(key)
                self.register_child(child)
                child.populate_model(key, value)

class NeighborActivate(SimpleType):
    'Network activate configuration'

    CLI_PATTERN = None

    def __init__(self):
        super(NeighborActivate, self).__init__('adminStatus')

    def diff_ifc_asa(self, cli):
        if self.is_the_same_cli(cli):
            self.set_state(State.NOCHANGE)
        else:
            self.set_state(State.MODIFY)

    def get_cli(self):
        return 'neighbor %s activate' % self.parent.get_key()

    @classmethod
    def get_cli_pattern(cls):
        if not cls.CLI_PATTERN:
            cls.CLI_PATTERN = re.compile('neighbor \S+ activate')
        return cls.CLI_PATTERN

    def get_config_path(self):
        return self.parent.get_config_path()

    def get_state(self):
        return self.value['state']

    def set_state(self, state):
        self.value['state'] = state

    def get_value(self):
        return self.value

    def get_translator(self, cli):
        if isinstance(cli, StructuredCommand):
            cli = cli.command
        if self.get_cli_pattern().match(cli.strip()):
            return self

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        # The 'no neighbor ... activate' commands need to be put on asa_cfg_list
        # since they need to be generated after their corresponding 'neighbor
        # ... remote-as' command
        state = self.get_state()
        if state in (State.CREATE, State.MODIFY):
            if self.get_value()['adminStatus'] == 'enabled':
                self.generate_cli(asa_cfg_list, self.get_cli())
            else:
                self.generate_cli(asa_cfg_list, 'no ' + self.get_cli())
        elif state == State.DESTROY:
            self.generate_cli(asa_cfg_list, 'no ' + self.get_cli())

    def is_the_same_cli(self, cli):
        this_cli = self.get_cli()
        if self.get_value()['adminStatus'] != 'enabled':
            this_cli = StructuredCommand(this_cli, True)
        return self.parse_cli(cli) == self.parse_cli(this_cli)

    def parse_cli(self, cli):
        result = {}
        if isinstance(cli, StructuredCommand) and cli.is_no:
            result['adminStatus'] = ''
        else:
            result['adminStatus'] = 'enabled'
        return result

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if 'adminStatus' in delta_ifc_cfg_value:
            self.value = delta_ifc_cfg_value['adminStatus']

    def is_active(self):
        return (self.get_state() in [State.CREATE, State.MODIFY] and
                self.get_value()[self.ifc_key] == 'enabled')

class NeighborRemoteAS(SimpleType):
    'Network remoteAS configuration'

    CLI_PATTERN = None

    def __init__(self):
        super(NeighborRemoteAS, self).__init__('remoteAS')

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.value = self.parse_cli(cli)
            self.set_state(State.DESTROY)
            self.parent.get_value()['remoteAS'] = self.value
        elif self.get_state() != State.DESTROY:
            if self.is_the_same_cli(cli):
                self.set_state(State.NOCHANGE)
            else:
                self.set_state(State.MODIFY)

    def get_cli(self):
        return 'neighbor %s remote-as %s' % (self.parent.get_key(),
                                             self.get_value()['remoteAS'])

    @classmethod
    def get_cli_pattern(cls):
        if not cls.CLI_PATTERN:
            cls.CLI_PATTERN = re.compile('neighbor \S+ remote-as (\S+)')
        return cls.CLI_PATTERN

    def get_config_path(self):
        return self.parent.get_config_path()

    def get_state(self):
        return self.value['state']

    def set_state(self, state):
        self.value['state'] = state

    def get_value(self):
        return self.value

    def get_translator(self, cli):
        if (isinstance(cli, basestring) and
                self.get_cli_pattern().match(cli.strip())):
            return self

    def has_ifc_delta_cfg(self):
        return hasattr(self, 'value')

    def parse_cli(self, cli):
        m = self.get_cli_pattern().match(cli)
        if m:
            result = {}
            result['remoteAS'] = m.group(1)
            return result

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if 'remoteAS' in delta_ifc_cfg_value:
            self.value = delta_ifc_cfg_value['remoteAS']

class NeighborNextHopSelf(SimpleType):
    '''
    This is required for iBGP to work, CSCvb16865.
    This field is not exposed to the user. It will be set if 'activate' is set.
    '''
    def __init__(self):
        super(NeighborNextHopSelf, self).__init__('$nextHopSelf$',
                                                  asa_gen_template='neighbor %s next-hop-self',
                                                  response_parser= ignore_response_parser)

    def get_cli(self):
        return self.asa_gen_template % self.parent.get_key()

    def get_state(self):
        'Only generate the CLI if the neighbor is active'
        active = self.parent.get_child('adminStatus')
        return State.CREATE if active.is_active() else State.NOCHANGE

class Network(RouterCommon, SimpleType):
    'Network configuration'

    IPV4_CLI_PATTERN = None
    IPV6_CLI_PATTERN = None

    def __init__(self, instance):
        super(Network, self).__init__(instance, 'network')

    def create_asa_key(self):
        return self.get_cli()

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.value = self.parse_cli(cli)
            self.set_state(State.DESTROY)
            self.parent.get_value().append(self.value)
        elif self.get_state() != State.DESTROY:
            if self.is_the_same_cli(cli):
                self.set_state(State.NOCHANGE)
            else:
                self.set_state(State.MODIFY)

    def get_cli(self):
        ipaddress = normalize_ip_address(self.value['ipaddress'])
        if self.is_ipv6():
            netmask = self.value['netmask']
            return 'network %s/%s' % (ipaddress, netmask)
        else:
            netmask = netmask_from_prefix_length(self.value['netmask'])
            return 'network %s mask %s' % (ipaddress, netmask)

    def get_cli_pattern(self):
        if self.is_ipv6():
            if not self.IPV6_CLI_PATTERN:
                self.IPV6_CLI_PATTERN = re.compile('network (\S+)/(\S+)')
            return self.IPV6_CLI_PATTERN
        else:
            if not self.IPV4_CLI_PATTERN:
                self.IPV4_CLI_PATTERN = re.compile('network (\S+) mask (\S+)')
            return self.IPV4_CLI_PATTERN

    def is_ipv6(self):
        return self.parent.is_ipv6()

    def parse_cli(self, cli):
        m = self.get_cli_pattern().match(cli)
        if m:
            result = {}
            if self.is_ipv6():
                result['ipaddress'] = m.group(1)
                result['netmask'] = int(m.group(2))
            else:
                result['ipaddress'] = m.group(1)
                result['netmask'] = prefix_length_from_netmask(m.group(2))
            return result

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.value = delta_ifc_cfg_value

    @staticmethod
    def make_key(network):
        ipaddress = normalize_ip_address(network['ipaddress'])
        netmask = str(network['netmask'])
        return ipaddress + '/' + netmask

class Networks(RouterCommon, DMList):
    'Container for Network objects'

    def __init__(self):
        super(Networks, self).__init__(Network.__name__, Network, 'network')

    def is_ipv6(self):
        return self.parent.is_ipv6()

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.value = delta_ifc_cfg_value['networks']
        for network in self.value:
            key = Network.make_key(network)
            if key not in self.children:
                child = self.child_class(key)
                self.register_child(child)
                child.populate_model(None, network)

class Timers(RouterCommon, SimpleType):
    'Timer configuration'

    CLI_PATTERN = None

    def __init__(self):
        super(Timers, self).__init__(
            ifc_key='timers',
            asa_key='timers bgp',
            asa_gen_template='timers bgp %(keepalive)s %(hold)s %(neighborMinHold)s',
            defaults={'neighborMinHold': 0})
        self.response_parser = (lambda resp:
                                None if resp.upper().startswith('% WARNING:')
                                else resp)

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.value = self.parse_cli(cli)
            self.set_state(State.DESTROY)
            self.parent.value['timers'] =  self.value
        elif self.get_state() != State.DESTROY:
            if self.is_the_same_cli(cli):
                self.set_state(State.NOCHANGE)
            else:
                self.set_state(State.MODIFY)

    @classmethod
    def get_cli_pattern(cls):
        if not cls.CLI_PATTERN:
            cls.CLI_PATTERN = re.compile('timers bgp (\S+) (\S+)(?: (\S+))?')
        return cls.CLI_PATTERN

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        # Timers are not supported in multi-mode
        if not self.get_top().is_multi_mode_asa():
            super(Timers, self).ifc2asa(no_asa_cfg_stack, asa_cfg_list)

    def parse_cli(self, cli):
        m = self.get_cli_pattern().match(cli)
        if m:
            result = {}
            result['keepalive'] = m.group(1)
            result['hold'] = m.group(2)
            neighborMinHold = m.group(3)
            if neighborMinHold:
                result['neighborMinHold'] = neighborMinHold
            return result

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if 'timers' in delta_ifc_cfg_value:
            timers = delta_ifc_cfg_value['timers']
            if 'keepalive' in timers and 'hold' in timers:
                self.value = timers

def bgp_audit_init(configuration):
    '''
    Transformer to initialize the configuration for audit. It sets all the
    dictionaries to a state of CREATE
    '''

    def set_state(cfg):
        if isinstance(cfg, dict):
            if 'state' in  cfg:
                cfg['state'] = State.CREATE
            for value in cfg.itervalues():
                set_state(value)
        elif isinstance(cfg, list):
            for value in cfg:
                set_state(value)

    bgp_keys = filter(lambda (kind, key, instance):
                      kind == Type.BGPDEV,
                      configuration.values()[0]['value'].keys())
    for bgp_key in bgp_keys:
        set_state(configuration.values()[0]['value'][bgp_key])
    return configuration

def translate_ifc_cfg(ifc_cfg):
    '''
    Translate from the IFC model to the ASA model and create any missing
    configuration. The ASA model has a one-to-one correspondence to the ASA
    command hierarchy.

    - For single value keys, convert to a dictionary.
    - For dictionaries that do not have a state key, add a state using the
      common state.
    - The redistribute configuration is a list for the IFC, but for the ASA
      it will be a dictionary.

    IFC model:
        'context': {
            'ipv4': {
                'neighbor': {
                    '': {
                        'adminStatus': '',
                        'remoteAS': int,
                    },
                },
                'networks': [
                    {   'ipaddress': '',
                        'netmask': '',
                    },
                ],
                'redistribute': ['connected', 'ospf', 'static'],
            },
            'ipv6': {
                'neighbor': {
                    '': {
                        'adminStatus': '',
                        'remoteAS': int,
                    },
                },
                'networks': [
                    {   'ipaddress': '',
                        'netmask': '',
                    },
                ],
                'redistribute': ['connected', 'ospf', 'static'],
            },
        },
        'localAS': int,
        'rtrId': '',
        'timers': {
            'hold': int,
            'keepalive': int,
            'neighborMinHold': int,
            'stale': int,
        },

    ASA model: 
        'context': {
            'ipv4': {
                'neighbor': {
                    '': {
                        'adminStatus': {
                            'adminStatus': '',
                        },
                        'remoteAS': {
                            'remoteAS': int,
                        },
                    },
                },
                'networks': [
                    {   'ipaddress': '',
                        'netmask': '',
                    },
                ],
                'redistribute': {
                    'connected': {
                    },
                    'ospf': {
                    },
                    'static': {
                    },
                },
            },
            'ipv6'" {
                'neighbor': {
                    '': {
                        'adminStatus: {
                            'adminStatus: '',
                        },
                        'remoteAS': {
                            'remoteAS': int,
                        },
                    },
                },
                'networks': [
                    {   'ipaddress': '',
                        'netmask': '',
                    },
                ],
                'redistribute': {
                    'connected': {
                    },
                    'ospf': {
                    },
                    'static': {
                    },
                },
            },
        },
        'gracefulRestart': {
            'stale': int,
        },
        'localAS': int,
        'rtrId': {
            'rtrId': '',
        },
        'timers': {
            'hold': int,
            'keepalive': int,
            'neighborMinHold': int,
        },
    '''

    def make_dict(cfg, key):
        if key in cfg:
            value = cfg[key]
            if not isinstance(value, dict):
                cfg[key] = {key: value}
            return cfg[key]

    def set_default_state(cfg, default_state):
        if isinstance(cfg, dict):
            cfg.setdefault('state', default_state)
            for value in cfg.itervalues():
                set_default_state(value, default_state)
        elif isinstance(cfg, list):
            for value in cfg:
                set_default_state(value, default_state)

    # Graceful restart
    timers = ifc_cfg.get('timers')
    if timers and 'stale' in timers:
        gracefulRestart = ifc_cfg.setdefault('gracefulRestart', {})
        if not 'stale' in gracefulRestart:
            gracefulRestart['stale'] = timers['stale']

    # Router ID for ASA 9.3.2 and later
    make_dict(ifc_cfg, 'rtrId')

    # Create contexts and their contents as needed
    context = ifc_cfg.setdefault('context', {})
    for ip_key in ('ipv4', 'ipv6'):
        is_ipv6 = (ip_key == 'ipv6')
        ip_value = context.setdefault(ip_key, {})

        # Neighbors
        neighbors = ip_value.setdefault('neighbor', {})
        for neighbor in neighbors.itervalues():
            if isinstance(neighbor, dict):
                for key in ('remoteAS', 'adminStatus'):
                    key_cfg = make_dict(neighbor, key)
                    if key_cfg and 'state' in neighbor:
                        key_cfg.setdefault('state', neighbor['state'])

        # Networks
        ip_value.setdefault('networks', [])

        # Redistributions
        if 'redistribute' in ip_value:
            redistribute = ip_value['redistribute']
            if isinstance(redistribute, list):
                # Convert the list to a dictionary
                redistribute_dict = {}
                for protocol in redistribute:
                    if protocol in ('connected', 'ospf', 'static'):
                        redistribute_dict[protocol] = {}
                ip_value['redistribute'] = redistribute_dict

        # Router ID for prior to ASA 9.3.2
        if not is_ipv6 and 'rtrId' in ifc_cfg:
            ip_value.setdefault('rtrId', {'rtrId': ifc_cfg['rtrId']['rtrId']})

    set_default_state(ifc_cfg, ifc_cfg['state'])
