'''
Created on Jul 15, 2013

@author: jeffryp

Copyright (c) 2013 by Cisco Systems, Inc.
All rights reserved.

Classes used for access rules.
'''
import re

from asaio.cli_interaction import ignore_info_response_parser
from asaio.cli_interaction import ignore_response_parser
from asaio.cli_interaction import ignore_warning_response_parser
from translator.base.dmlist import DMList
from translator.base.dmobject import DMObject
from translator.base.simpletype import SimpleType
from translator.state_type import State, Type
from translator.validators import ICMPValidator
from translator.validators import ProtocolValidator, TCPUDPValidator
import utils.protocol
from utils.service import get_icmp_type_cli, get_port_cli
from utils.util import filter_first, ifcize_param_dict, read_clis
from utils.util import binary_search
from translator.connector import Connector
from translator.structured_cli import StructuredCommand


class AccessGroup(SimpleType):
    'A single access group'

    def __init__(self, ifc_key, direction):
        super(AccessGroup, self).__init__(ifc_key, 'access-group')
        self.direction = direction

    def get_cli(self):
        value = self.get_value()
        return 'access-group %s %s interface %s' % (value, self.direction,
                                                    self.parent.nameif)

    def create_asa_key(self):
        return self.get_cli()

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        # The ASA will remove the access group when the interface is removed
        if self.interface.get_state() != State.DESTROY:
            if self.get_state() == State.DESTROY:
                self.response_parser=ignore_response_parser
            elif hasattr(self, 'response_parser'):
                del self.response_parser
            super(AccessGroup, self).ifc2asa(no_asa_cfg_stack, asa_cfg_list)

    @property
    def interface(self):
        return self.parent.interface

class AccessGroups(DMObject):
    'The access groups bound to an interface'

    MY_CLI_PATTERN = None

    def __init__(self):
        super(AccessGroups, self).__init__('AccessGroup', 'access-group')
        self.register_child(AccessGroup('inbound_access_list_name', 'in'))
        self.register_child(AccessGroup('outbound_access_list_name', 'out'))

    def get_my_cli_pattern(self):
        if not self.MY_CLI_PATTERN:
            self.MY_CLI_PATTERN = re.compile('access-group \S+ \S+ interface (\S+)$')
        return self.MY_CLI_PATTERN

    def get_translator(self, cli):
        if isinstance(cli, basestring):
            m = self.get_my_cli_pattern().match(cli.strip())
            if m and m.group(1) == self.nameif:
                return super(AccessGroups, self).get_translator(cli)

    @property
    def interface(self):
        return self.parent

    @property
    def nameif(self):
        return self.interface.nameif

class AccessControlEntry(DMObject):
    'A single access control entry (ACE)'

    # CLI syntax:
    #     access-list <name> extended <action> <protocol> <source-ip>
    #         <source-mask> [<dest-operator> <dest-low-port> [<dest-high-port>]]
    #
    # Examples:
    #     access-list global_access extended deny ip any any
    #     access-list global_access extended deny object ldap_server any any
    #     access-list global_access extended deny object-group web_server any any
    #     access-list global_access extended deny tcp any any eq 5006
    #     access-list global_access extended deny tcp any any range 5001 5002
    ANY_CLI_PATTERN = None
    ICMP_CLI_PATTERN = None
    TCP_UDP_CLI_PATTERN = None

    INTERVAL_DEFAULT = '300'
    LEVEL_MAP = [
        'emergencies',   # 0
        'alerts',        # 1
        'critical',      # 2
        'errors',        # 3
        'warnings',      # 4
        'notifications', # 5
        'informational', # 6
        'debugging'      # 7
    ]
    LEVEL_DEFAULT = LEVEL_MAP[6]

    def __init__(self, instance):
        DMObject.__init__(self, ifc_key=instance)
        self.defaults = {
            'protocol': {'name_number': 'ip'},
            'source_address': {'any': 'any'},
            'destination_address': {'any': 'any'}
        }

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.delta_ifc_key = (Type.FOLDER, AccessControlEntry.__name__, cli)
            self.delta_ifc_cfg_value = {'state': State.DESTROY, 'value': ifcize_param_dict(self.values)}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] =  self.delta_ifc_cfg_value

    def equals_no_order(self, ace):
        if isinstance(ace, ACERemark):
            return False
        return self.get_pure_ace() == ace.get_pure_ace()

    def get_pure_ace(self):
        result = self.values.copy()
        for key in ('order', 'remark', 'extra-remarks'):
            if key in result:
                del result[key]
        return result

    def get_cli(self, line_no=None):
        cli = []
        cli.append('access-list ' + self.parent.ifc_key)
        if line_no:
            cli.append(' line ' + str(line_no))
        cli.append(' extended ' + self.values['action'])
        self.append_protocol(cli, self.values['protocol'])
        self.append_security_group(cli, self.values.get('source_security_group'))
        self.append_address(cli, self.values.get('source_address'))
        if self.is_tcp_udp():
            self.append_service(cli, self.values.get('source_service'))
        self.append_security_group(cli, self.values.get('destination_security_group'))
        self.append_address(cli, self.values.get('destination_address'))
        if self.is_tcp_udp():
            self.append_service(cli, self.values.get('destination_service'))
        elif self.is_icmp():
            self.append_icmp(cli, self.values.get('icmp'))
        self.append_log(cli, self.values.get('log'))
        self.append_time_range(cli, self.values.get('time_range_name'))

        return ''.join(cli)

    def is_icmp(self):
        return self.values['protocol'].get('name_number') in ('icmp', 'icmp6')

    def is_tcp_udp(self):
        return self.values['protocol'].get('name_number') in ('tcp', 'udp')

    def set_acl_changed(self):
        self.parent.set_acl_changed()

    @staticmethod
    def append_address(cli, address):
        if 'any' in address:
            cli.append(' ' + address['any'])
        elif 'object_name' in address:
            cli.append(' object ' + address['object_name'])
        elif 'object_group_name' in address:
            cli.append(' object-group ' + address['object_group_name'])
        elif 'epg_name' in address:
            cli.append(' object-group ' + Connector.epg_name2nog_name(address['epg_name']))

    @staticmethod
    def append_icmp(cli, icmp):
        if icmp:
            cli.append(' ' + icmp['type'])
            if 'code' in icmp:
                cli.append(' ' + icmp['code'])

    @classmethod
    def append_log(cls, cli, log):
        if log:
            cli.append(' log')
            if 'disable' in log:
                cli.append(' disable')
            else:
                cli.append(' ' + log['level'])
                cli.append(' interval ' + log['interval'])

    @staticmethod
    def append_protocol(cli, protocol):
        if 'name_number' in protocol:
            cli.append(' ' + protocol['name_number'])
        elif 'object_name' in protocol:
            cli.append(' object ' + protocol['object_name'])
        elif 'object_group_name' in protocol:
            cli.append(' object-group ' + protocol['object_group_name'])

    @staticmethod
    def append_security_group(cli, security_group):
        if security_group:
            if 'security_group_name' in security_group:
                cli.append(' security-group name ' +
                           security_group['security_group_name'])
            elif 'security_group_tag' in security_group:
                cli.append(' security-group tag ' +
                           security_group['security_group_tag'])
            elif 'security_object_group' in security_group:
                cli.append(' object-group-security ' +
                           security_group['security_object_group'])

    @staticmethod
    def append_service(cli, service):
        if service:
            cli.append(' ' + service['operator'] + ' ' + service['low_port'])
            if service['operator'] == 'range':
                cli.append(' ' + service['high_port'])

    @staticmethod
    def append_time_range(cli, time_range):
        if time_range:
            cli.append(' time-range ' + time_range)

    def generate_command(self, asa_cfg_list, line_no=None):
        self.generate_cli(asa_cfg_list,
                          self.get_cli(line_no),
                          response_parser=ignore_warning_response_parser)
        self.set_acl_changed()

    def generate_delete(self, asa_cfg_list):
        self.generate_cli(asa_cfg_list,
                          'no ' + self.get_cli(),
                          response_parser=ignore_response_parser)
        self.set_acl_changed()

    @classmethod
    def parse_cli(cls, cli, order=None):
        extra_remarks = None
        if isinstance(cli, StructuredCommand):
            remark = cli.sub_commands[0]
            if len(cli.sub_commands)>1:
                '''
                We only allow a single remark from APIC, extra remarks in the ASA
                for a given ACE will be removed.
                '''
                extra_remarks = cli.sub_commands[1:]
            cli = cli.command
        else:
            remark = None
        result = cls.parse_tcp_udp_cli(cli, order)
        if not result:
            result = cls.parse_icmp_cli(cli, order)
        if not result:
            result = cls.parse_any_cli(cli, order)
        if remark:
            result['remark'] = remark
            if extra_remarks:
                result['extra-remarks'] = extra_remarks
        return result

    @classmethod
    def get_any_cli_pattern(cls):
        'For any type of traffic, no ports'

        if not cls.ANY_CLI_PATTERN:
            cls.ANY_CLI_PATTERN = re.compile(
                'access-list \S+ extended' +
                ' (\S+)' +                                          # Action, group 1
                ' (?:(object|object-group) )?(\S+)' +               # Protocol, groups 2,3
                '(?: (?:(?:object-group-security (\S+))|' +         # Source security group, groups 4,5,6
                    '(?:security-group (?:(?:name (\S+))|(?:tag (\d+))))))?' + 
                ' (?:(any[46]?)|(?:(object|object-group) (\S+)))' + # Source address, groups 7,8,9
                '(?: (?:(?:object-group-security (\S+))|' +         # Destination security group, groups 10,11,12
                    '(?:security-group (?:(?:name (\S+))|(?:tag (\d+))))))?' + 
                ' (?:(any[46]?)|(?:(object|object-group) (\S+)))' + # Destination address, groups 13,14,15
                '(?: (log)(?:(?: (disable))|' +                     # Log, groups 16,17,18,19
                    '(?: (emergencies|alerts|critical|errors|warnings|notifications|informational|debugging))?' +
                    '(?: interval (\d+))?)?)?' +
                '(?: time-range (\S+))?$')                          # time-range, group 20
        return cls.ANY_CLI_PATTERN

    @classmethod
    def parse_any_cli(cls, cli, order):
        'For any type of traffic, no ports'

        m = cls.get_any_cli_pattern().match(cli)
        if m:
            result = {}
            if order:
                result['order'] = str(order)
            result['action'] = m.group(1)
            cls.parse_protocol(result, *m.group(2, 3))
            cls.parse_security_group(result, 'source_security_group', *m.group(4, 5, 6))
            cls.parse_address(result, 'source_address', *m.group(7, 8, 9))
            cls.parse_security_group(result, 'destination_security_group', *m.group(10, 11, 12))
            cls.parse_address(result, 'destination_address', *m.group(13, 14, 15))
            cls.parse_log(result, *m.group(16, 17, 18, 19))
            cls.parse_time_range(result, m.group(20))
            return result

    @classmethod
    def get_icmp_cli_pattern(cls):
        'For ICMP traffic, with ICMP type'

        if not cls.ICMP_CLI_PATTERN:
            cls.ICMP_CLI_PATTERN = re.compile(
                'access-list \S+ extended' +
                ' (\S+)' +                                          # Action, group 1
                ' (icmp[6]?)' +                                     # Protocol, group 2
                '(?: (?:(?:object-group-security (\S+))|' +         # Source security group, groups 3,4,5
                    '(?:security-group (?:(?:name (\S+))|(?:tag (\d+))))))?' + 
                ' (?:(any[46]?)|(?:(object|object-group) (\S+)))' + # Source address, groups 6,7,8
                '(?: (?:(?:object-group-security (\S+))|' +         # Destination security group, groups 9,10,11
                    '(?:security-group (?:(?:name (\S+))|(?:tag (\d+))))))?' + 
                ' (?:(any[46]?)|(?:(object|object-group) (\S+)))' + # Destination address, groups 12,13,14
                '(?: (?!log|time-range)(\S+)(?: (\S+))?)?' +                   # ICMP information, groups 15,16
                '(?: (log)(?:(?: (disable))|' +                     # Log, groups 17,18,19,20
                    '(?: (emergencies|alerts|critical|errors|warnings|notifications|informational|debugging))?' +
                    '(?: interval (\d+))?)?)?' +
                '(?: time-range (\S+))?$')                          # time-range, group 21
        return cls.ICMP_CLI_PATTERN

    @classmethod
    def parse_icmp_cli(cls, cli, order):
        'For ICMP traffic, with ICMP type'

        m = cls.get_icmp_cli_pattern().match(cli)
        if m:
            result = {}
            if order:
                result['order'] = str(order)
            result['action'] = m.group(1)
            result['protocol'] = {'name_number': m.group(2)}
            cls.parse_security_group(result, 'source_security_group', *m.group(3, 4, 5))
            cls.parse_address(result, 'source_address', *m.group(6, 7, 8))
            cls.parse_security_group(result, 'destination_security_group', *m.group(9, 10, 11))
            cls.parse_address(result, 'destination_address', *m.group(12, 13, 14))
            cls.parse_icmp(result, *m.group(15, 16))
            cls.parse_log(result, *m.group(17, 18, 19, 20))
            cls.parse_time_range(result, m.group(21))
            return result

    @classmethod
    def get_tcp_udp_cli_pattern(cls):
        'For TCP or UDP traffic, with ports'

        if not cls.TCP_UDP_CLI_PATTERN:
            cls.TCP_UDP_CLI_PATTERN = re.compile(
                'access-list \S+ extended' +
                ' (\S+)' +                                          # Action, group 1
                ' (tcp|udp)' +                                      # Protocol, group 2
                '(?: (?:(?:object-group-security (\S+))|' +         # Source security group, groups 3,4,5
                    '(?:security-group (?:(?:name (\S+))|(?:tag (\d+))))))?' + 
                ' (?:(any[46]?)|(?:(object|object-group) (\S+)))' + # Source address, groups 6,7,8
                '(?: (lt|gt|eq|neq|range) (\S+)(?: (\S+))?)?' +     # Source service, groups 9,10,11
                '(?: (?:(?:object-group-security (\S+))|' +         # Destination security group, groups 12,13,14
                    '(?:security-group (?:(?:name (\S+))|(?:tag (\d+))))))?' + 
                ' (?:(any[46]?)|(?:(object|object-group) (\S+)))' + # Destination address, groups 15,16,17
                '(?: (lt|gt|eq|neq|range) (\S+)(?: (?!log|time-range)(\S+))?)?' +     # Destination service, groups 18,19,20
                '(?: (log)(?:(?: (disable))|' +                     # Log, groups 21,22,23,24
                    '(?: (emergencies|alerts|critical|errors|warnings|notifications|informational|debugging))?' +
                    '(?: interval (\d+))?)?)?' +
                '(?: time-range (\S+))?$')                          # time-range, group 25
        return cls.TCP_UDP_CLI_PATTERN

    @classmethod
    def parse_tcp_udp_cli(cls, cli, order):
        'For TCP or UDP traffic, with ports'

        m = cls.get_tcp_udp_cli_pattern().match(cli)
        if m:
            result = {}
            if order:
                result['order'] = str(order)
            result['action'] = m.group(1)
            result['protocol'] = {'name_number': m.group(2)}
            cls.parse_security_group(result, 'source_security_group', *m.group(3, 4, 5))
            cls.parse_address(result, 'source_address', *m.group(6, 7, 8))
            cls.parse_service(result, 'source_service', *m.group(9, 10, 11))
            cls.parse_security_group(result, 'destination_security_group', *m.group(12, 13, 14))
            cls.parse_address(result, 'destination_address', *m.group(15, 16, 17))
            cls.parse_service(result, 'destination_service', *m.group(18, 19, 20))
            cls.parse_log(result, *m.group(21, 22, 23, 24))
            cls.parse_time_range(result, m.group(25))
            return result

    @staticmethod
    def parse_address(result, key, any_address, keyword, name):
        if any_address:
            result[key] = {'any': any_address}
        if keyword:
            address = {}
            if keyword == 'object':
                address['object_name'] = name
            elif keyword == 'object-group':
                if Connector.is_epg_nog(name):
                    address['epg_name'] = Connector.nog_name2epg_name(name)
                else:
                    address['object_group_name'] = name
            result[key] = address

    @staticmethod
    def parse_icmp(result, type_name, code):
        if type_name:
            icmp = {'type': type_name}
            if code:
                icmp['code'] = code
            result['icmp'] = icmp

    @classmethod
    def parse_log(cls, result, log_token, disable_token, level_token,
                  interval_token):
        if log_token:
            log = {}
            if disable_token:
                log['disable'] = None
            else:
                log['level'] = level_token if level_token else cls.LEVEL_DEFAULT
                log['interval'] = interval_token if interval_token else cls.INTERVAL_DEFAULT
            result['log'] = log

    @staticmethod
    def parse_protocol(result, keyword, name):
        protocol = {}
        if keyword:
            if keyword == 'object':
                protocol['object_name'] = name
            elif keyword == 'object-group':
                protocol['object_group_name'] = name
        else:
            protocol['name_number'] = name
        result['protocol'] = protocol

    @staticmethod
    def parse_security_group(result, key, group, name, tag):
        if group:
            result[key] = {'security_object_group': group}
        elif name:
            result[key] = {'security_group_name': name}
        elif tag:
            result[key] = {'security_group_tag': tag}

    @staticmethod
    def parse_service(result, key, operator, low_port, high_port):
        if operator:
            service = {'operator': operator, 'low_port': low_port}
            if high_port:
                service['high_port'] = high_port
            result[key] = service

    @staticmethod
    def parse_time_range(result, time_range_name):
        if time_range_name:
            result['time_range_name'] = time_range_name

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        super(AccessControlEntry, self).populate_model(
            delta_ifc_key, delta_ifc_cfg_value)
        self.values = self.get_value()

        # Populate defaults
        for name, default in self.defaults.iteritems():
            self.values.setdefault(name, default)

        self.normalize_protocol(self.values['protocol'])
        self.normalize_service(self.values['protocol'], self.values.get('source_service'))
        self.normalize_service(self.values['protocol'], self.values.get('destination_service'))
        self.normalize_icmp(self.values['protocol'], self.values.get('icmp'))
        self.normalize_log(self.values.get('log'))

    @staticmethod
    def normalize_icmp(protocol, icmp):
        'Replace icmp type with its CLI equivalent'
        if icmp and 'name_number' in protocol:
            icmp['type'] = get_icmp_type_cli(protocol['name_number'], icmp['type'])

    @classmethod
    def normalize_log(cls, log):
        if log != None:
            if 'disable' in log:
                log['disable'] = None
            else:
                # Set the defaults
                if 'interval' not in log:
                    log['interval'] = cls.INTERVAL_DEFAULT
                if 'level' not in log:
                    log['level'] = cls.LEVEL_DEFAULT

                # Replace level with its CLI equivalent
                level = log['level']
                if level.isdigit():
                    log['level'] = cls.LEVEL_MAP[int(level)]

    @staticmethod
    def normalize_protocol(protocol):
        'Replace protocol with its CLI equivalent'
        if 'name_number' in protocol:
            protocol['name_number'] = utils.protocol.get_cli(protocol['name_number'])

    @staticmethod
    def normalize_service(protocol, service):
        'Replace service port with its CLI equivalent'
        if service and 'name_number' in protocol:
            protocol_name = protocol['name_number']
            service['low_port'] = get_port_cli(protocol_name, service['low_port'])
            if 'high_port' in service:
                service['high_port'] = get_port_cli(protocol_name, service['high_port'])

    def validate_configuration(self):
        faults = []
        self.validate_protocol(faults)
        for folder in ('source_security_group', 'destination_security_group'):
            self.validate_security_group(faults, folder)
        for folder in ('source_address', 'destination_address'):
            self.validate_address(faults, folder)
        self.validate_service(faults)
        self.validate_icmp(faults)
        self.validate_log(faults)
        return faults

    def validate_address(self, faults, folder):
        address = self.values[folder]
        if len(address) > 1:
            faults.append(self.generate_fault(
                folder + ' can only contain one of any, object_name, object_group_name, or epg_name.',
                folder))

    def validate_icmp(self, faults):
        icmp = self.values.get('icmp')
        if icmp:
            if not self.is_icmp():
                faults.append(self.generate_fault(
                    'Only supported for a protocol of icmp or icmp6.', 'icmp'))
                return

            protocol = self.values['protocol'].get('name_number')
            validator = ICMPValidator(protocol, 'icmp')
            validator.generate_fault = self.generate_fault
            faults.extend(validator.validate(icmp))

    def validate_log(self, faults):
        log = self.values.get('log')
        if log and 'disable' in log:
            if 'level' in log:
                faults.append(self.generate_fault(
                    'level cannot be specified when disable is specified.',
                    ('log', 'level')))
            if 'interval' in log:
                faults.append(self.generate_fault(
                    'interval cannot be specified when disable is specified.',
                    ('log', 'interval')))

    def validate_protocol(self, faults):
        protocol = self.values['protocol']
        if len(protocol) > 1:
            faults.append(self.generate_fault(
                'protocol can only contain one of name_number, object_name, or object_group_name.',
                'protocol'))
            return
        if 'name_number' in protocol:
            err_msg = ProtocolValidator().validate(protocol['name_number'])
            if err_msg:
                faults.append(self.generate_fault(
                    err_msg, ['protocol', 'name_number']))

    def validate_security_group(self, faults, folder):
        if folder in self.values:
            security_group = self.values[folder]
            if len(security_group) > 1:
                faults.append(self.generate_fault(
                    folder + ' can only contain one of security_group_name, ' +
                        'security_group_tag, or security_object_group.',
                    folder))

    def validate_service(self, faults):
        if 'source_service' in self.values or 'destination_service' in self.values:
            if not self.is_tcp_udp():
                for key in ('source_service', 'destination_service'):
                    if key in self.values:
                        faults.append(self.generate_fault(
                            'Only supported for a protocol of tcp or udp.',
                            key))
                return

            protocol = self.values['protocol'].get('name_number')
            validator = TCPUDPValidator(protocol, 'source_service', 'destination_service')
            validator.generate_fault = self.generate_fault
            faults.extend(validator.validate(self.values))

    def get_remark(self):
        remark_key = filter_first(lambda key: key[1] == 'remark', self.delta_ifc_cfg_value['value'].keys())
        if not remark_key:
            return
        remark_entry = self.delta_ifc_cfg_value['value'][remark_key]
        return ACERemark(self, remark_entry['state'], remark_entry['value'])

    def get_extra_remarks(self):
        '''
        'extra-remarks' is not exposed to the user. It is only used to handle
        multiple remarks for an ACE on ASA that are configured out of band.
        '''
        extra_remarks_key = filter_first(lambda key: key[1] == 'extra-remarks', self.delta_ifc_cfg_value['value'].keys())
        if not extra_remarks_key:
            return []
        extra_remarks_entry = self.delta_ifc_cfg_value['value'][extra_remarks_key]
        result = []
        for remark in extra_remarks_entry['value']:
            result.append(ACERemark(self, State.DESTROY, remark))
        return result

    def require_mini_audit(self):
        '''
        Return True if one of its attributes is in the state of State.MODIFY, or
        an attribute other than 'remark' is in the state of CREATE or DESTROY.
        '''
        if self.get_state() != State.MODIFY:
            return
        modified_attributes = filter(lambda key: self.delta_ifc_cfg_value['value'][key]['state'] == State.MODIFY,
                                    self.delta_ifc_cfg_value['value'].keys())
        if modified_attributes != []:
            return True
        created_or_destroyed_attributes = filter(lambda key: self.delta_ifc_cfg_value['value'][key]['state'] in (State.CREATE, State.DESTROY),
                                                 self.delta_ifc_cfg_value['value'].keys())
        if len(created_or_destroyed_attributes) == 1:
            return created_or_destroyed_attributes[0][1] != 'remark'
        return len(created_or_destroyed_attributes) > 0

    def set_state(self, state):
        'Override the default implementation so that if input state is State.NOCHANGE, we will set it to the remark as well'
        DMObject.set_state(self, state)
        if state != State.NOCHANGE:
            return
        remark_key = filter_first(lambda key: key[1] == 'remark', self.delta_ifc_cfg_value['value'].keys())
        if not remark_key:
            return
        remark_entry = self.delta_ifc_cfg_value['value'][remark_key]
        remark_entry['state'] = state

    def set_order_number(self, n):
        order_key = filter_first(lambda key: key[1] == 'order', self.delta_ifc_cfg_value['value'].keys())
        if not order_key:
            return
        order_entry = self.delta_ifc_cfg_value['value'][order_key]
        old_order = order_entry['value']
        order_entry['value'] = str(n)
        self.values['order'] = str(n)
        return old_order

    def __str__(self):
        return self.get_cli()

class ACERemark(SimpleType):
    'For access-list remark'
    def __init__(self, ace, state, value):
        """
        @param ace: AccessControlEntry
            the AccessControlEntry that this object remarks.
        @param state: State
            configuration state of the remark
        @param value: str
            the remark text
        """
        SimpleType.__init__(self, 'remark')
        self.delta_ifc_cfg_value = {'state':state, 'value': value}
        self.delta_ifc_key = {Type.PARAM, 'remark', ''}
        self.parent = ace

    def generate_command(self, asa_cfg_list, line_no=None):
        acl_name = self.parent.parent.ifc_key
        self.asa_gen_template ='access-list ' + acl_name
        if line_no:
            self.asa_gen_template += ' line ' + str(line_no)
        self.asa_gen_template += ' remark %s'
        self.generate_cli(asa_cfg_list,
                          self.get_cli(),
                          response_parser=ignore_warning_response_parser)
        self.parent.set_acl_changed()

    def generate_delete(self, asa_cfg_list, line_no):
        acl_name = self.parent.parent.ifc_key
        self.asa_gen_template ='access-list ' + acl_name + ' line ' + str(line_no) + ' remark %s'
        self.generate_cli(asa_cfg_list,
                          'no ' + self.get_cli(),
                          response_parser=ignore_response_parser)
        self.parent.set_acl_changed()

    def equals_no_order(self, ace):
        if isinstance(ace, ACERemark):
            return self.get_value() == ace.get_value()

    def __str__(self):
        self.asa_gen_template ='access-list ' + self.parent.parent.ifc_key + ' remark %s'
        return self.get_cli()

class AccessList(DMList):
    'A single access list that contains access control entries (ACE)'

    def __init__(self, instance):
        DMList.__init__(self, instance, AccessControlEntry, 'access-list')
        self.acl_changed = False
        self.clear_translation_state = None
        self.diff_ifc_asa_ace_order = 1

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.delta_ifc_key = (Type.FOLDER, self.parent.ifc_key, self.ifc_key)
            self.delta_ifc_cfg_value = {'state': State.DESTROY, 'value': {}}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] = self.delta_ifc_cfg_value
        elif self.delta_ifc_cfg_value['state'] != State.DESTROY:
            values = AccessControlEntry.parse_cli(cli, self.diff_ifc_asa_ace_order)
            if values:
                self.set_state(State.MODIFY)

                ace = AccessControlEntry(cli)
                ace.parent = self
                ace.values = values
                self.diff_ifc_asa_ace_order += 1

                match_ace = self.find_ace(ace)
                if match_ace:
                    match_ace.set_state(State.NOCHANGE)
                else:
                    # Delete the ACE if it doesn't exist or is in a different order
                    self.register_child(ace)
                    ace.diff_ifc_asa(cli)

    def find_ace(self, ace):
        '''
        Use binary search to find the matching ACE.
        The result is the same as the following algorithm:

        for match_ace in self.children.itervalues():
            if match_ace.values == ace.values:
                return match_ace
        '''
        if not hasattr(self, '_cli2ace_dict'):
            'cache _cli2ace_dict and _sorted_ace_clis'
            self._cli2ace_dict = {}
            for a in self.children.itervalues():
                self._cli2ace_dict[a.get_cli()] = a
            self._sorted_ace_clis = self._cli2ace_dict.keys()
            self._sorted_ace_clis.sort()
        matched_ace = binary_search(self._cli2ace_dict, self._sorted_ace_clis, ace)
        if not matched_ace:
            return None
        return matched_ace if matched_ace.values == ace.values else None

    def generate_delta(self, no_asa_cfg_stack, asa_cfg_list):
        'Optimize the generated CLI by looking for ACEs that have moved'

        # Split into old and new config
        old_config = []
        new_config = []
        for ace in self.children.itervalues():
            state = ace.delta_ifc_cfg_value['state']
            if state == State.NOCHANGE:
                old_config.append(ace)
                new_config.append(ace)
            elif state == State.MODIFY and not ace.require_mini_audit():
                '''
                This is case that remark in the ACE is added or deleted.
                But no other attribute in the ACE is changed.
                self.normalize_remarks will take care of adding/removing remark CLIs
                '''
                old_config.append(ace)
                new_config.append(ace)
            elif state in (State.CREATE, State.MODIFY):
                new_config.append(ace)
            elif state == State.DESTROY:
                old_config.append(ace)

        # Sort the old and new config
        self.sort_config(old_config)
        self.sort_config(new_config)
        self.normalize_remarks(old_config, is_old_config=True)
        self.normalize_remarks(new_config, is_old_config=False)

        # List of new positions corresponding to the old positions.  e.g. if the
        # first ACE in the old config is now in position 4 of the new config,
        # then new_positions[0] == 4
        new_positions = []

        # The element at old_positions[i] is equal to the element at
        # new_positions[i].  This is used to keep track of what has changed
        # during the traversal of the old config for optimizing the CLI when
        # traversing the new config 
        old_positions = []

        # List of old positions that have been deleted
        delete_old_positions = []

        new_config_cli2ace_dict = {}
        for ace in new_config:
            new_config_cli2ace_dict[ace.get_cli()] = ace
        new_config_sorted_clis = new_config_cli2ace_dict.keys()
        new_config_sorted_clis.sort()
        for i, ace in enumerate(old_config):
            pos = AccessList.equal_ace_position(new_config, new_config_cli2ace_dict, new_config_sorted_clis, ace)
            if pos < 0:
                delete_old_positions.append(i)
            else:
                new_positions.append(pos)
                old_positions.append(i)

        # Handle the case where all the ACEs in an access list have been either
        # deleted or moved.  Normally, all the ACEs would first be deleted and
        # then the changed ones would be added back.  However, if the access
        # list is in use, then the ASA will delete all polices using the access
        # list when the last ACE is deleted.  To get around this, the first
        # changed ACE is added before any ACEs are deleted. 
        is_rebuilt = (len(delete_old_positions) == len(old_config) and
                      len(new_config) > 0)

        # Delete what was moved and what was deleted
        what_moved = AccessList.what_moved(new_positions)
        for i in what_moved:
            delete_old_positions.append(old_positions[new_positions.index(i)])
        for i in delete_old_positions:
            if isinstance(old_config[i], ACERemark):
                """
                Need to provide line number to delete remark because ASA allows the same remark string
                appears before different ACEs.
                """
                old_config[i].generate_delete(no_asa_cfg_stack, i+(2 if is_rebuilt else 1))
            else:
                old_config[i].generate_delete(no_asa_cfg_stack)

        # If the access list is rebuilt, add the first changed ACE to the end
        # of the 'no' CLI list.  After the 'no' list is reversed, the first
        # changed ACE will be before any ACEs to be removed.
        if is_rebuilt:
            new_config[0].generate_command(no_asa_cfg_stack, line_no=1)

        # Add what was added or moved
        start = 1 if is_rebuilt else 0
        for i, ace in enumerate(new_config[start:], start):
            if i not in new_positions or i in what_moved:
                ace.generate_command(asa_cfg_list, line_no=i + 1)

    def get_translator(self, cli):
        # Used for mini_audit only
        return self

    def mini_audit(self):
        'Override the default implementation to make remark sub-command of ACE'
        asa_clis = self.read_asa_config()
        if not asa_clis:
            return
        self.create_missing_ifc_delta_cfg()
        asa_clis = read_clis(asa_clis.strip().split('\n'))
        translator_mode_command = self.__dict__.get('mode_command')
        for cli in asa_clis:
            translator = self.get_translator(cli)
            if translator:
                translator.diff_ifc_asa(cli)
                translator.mode_command = translator_mode_command

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if filter_first(lambda child: child.require_mini_audit(), self):
            # Remove any entries to be deleted
            for key, ace in self.children.items():
                if ace.get_state() == State.DESTROY:
                    del(self.children[key])
            # Change the remaining entries to CREATE
            for ace in self.children.itervalues():
                ace.set_state(State.CREATE)

            self.mini_audit_command = ('show running-config access-list ' +
                                       self.ifc_key)
            self.mini_audit()

        state = self.get_state()
        if state in (State.CREATE, State.MODIFY):
            if state == State.CREATE:
                # Sort the ACEs into order
                access_control_entries = list(self.children.itervalues())
                self.sort_config(access_control_entries)
                self.normalize_remarks(access_control_entries)
                for ace in access_control_entries:
                    ace.generate_command(asa_cfg_list)
            else:
                self.generate_delta(no_asa_cfg_stack, asa_cfg_list)
        elif state == State.DESTROY:
            self.generate_cli(
                no_asa_cfg_stack,
                'clear config access-list ' + self.ifc_key,
                response_parser=ignore_response_parser)
            self.set_acl_changed()
        if (self.acl_changed and
                (self.clear_translation_state in (State.CREATE, State.MODIFY, State.NOCHANGE) or
                 (self.get_state() == State.DESTROY and self.clear_translation_state != None))):
            self.parent.set_clear_xlate()

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.delta_ifc_key = delta_ifc_key
        self.delta_ifc_cfg_value = delta_ifc_cfg_value
        for key, value in delta_ifc_cfg_value['value'].iteritems():
            if key[1] == 'AccessControlEntry':
                super(AccessList, self).populate_model(key, value)
            elif key[1] == 'clear_translation':
                self.clear_translation_state = value['state']
        if self.get_top().is_audit:
            self.normalize_ace_order()
        return

    def normalize_ace_order(self):
        '''Make sure the order number starts from 1.
           This helps to reduce the number of ACEs generated in the dictionary during audit operation.
        '''
        IS_NORMALIZED = '____CACHE_IS_NORMALIZED___'
        if self.delta_ifc_cfg_value.get(IS_NORMALIZED, False):
            return
        aces = list(self.children.itervalues())
        self.sort_config(aces)
        i = 1
        for ace in aces:
            ace.set_order_number(i)
            i += 1
        self.delta_ifc_cfg_value[IS_NORMALIZED] = True

    def set_acl_changed(self):
        self.acl_changed = True

    def validate_duplicates(self, faults):
        if self.get_state() in [State.DESTROY, State.NOCHANGE]:
            return
        aces = filter(lambda ace: ace.get_state() != State.DESTROY, self)
        new_changed_aces = filter(lambda ace: ace.get_state() != State.NOCHANGE, aces)
        if not new_changed_aces:
            return
        cli2ace_dict = {}
        for a in new_changed_aces:
            try:
                cli2ace_dict[a.get_cli()] = a
            except:
                'no-op'
        sorted_clis = cli2ace_dict.keys()
        sorted_clis.sort()
        for ace in aces:
            '''
            Use binary search to speed up. The result is the same as the following algorithm.:

            for other in new_changed_aces:
                if other != ace and other.equals_no_order(ace):
                    faults.append(other.generate_fault(
                        'This is a duplicate of the ' +
                        'AccessControlEntry named \'' +
                        ace.delta_ifc_key[2] + '\'.'))
            '''
            other = binary_search(cli2ace_dict, sorted_clis, ace)
            if other and other != ace:
                faults.append(other.generate_fault(
                    'This is a duplicate of the ' +
                    'AccessControlEntry named \'' +
                    ace.delta_ifc_key[2] + '\'.'))

    def validate_not_empty(self, faults):
        if self.get_state() != State.DESTROY:
            count = 0
            for ace in self.children.itervalues():
                # MODIFY is not actually supported, but include it to prevent an
                # error about an empty ACL.  The ACE will report an error for
                # the MODIFY.
                if ace.get_state() in (State.NOCHANGE, State.CREATE, State.MODIFY):
                    count += 1
            if count < 1:
                faults.append(self.generate_fault(
                    'An AccessList cannot be empty, it must contain at ' +
                        'least one AccessControlEntry.'))

    def validate_configuration(self):
        faults = []

        self.validate_not_empty(faults)
        self.validate_duplicates(faults)

        for child in self:
            self.report_fault(child.validate_configuration(), faults)

        return faults

    @staticmethod
    def equal_ace_position(new_config, new_config_cli2ace_dict, new_config_sorted_clis, old_ace):
        '''
        Find position of the ACE in the new config.

        Using binary search with ACE.get_cli() as the key to speed up.
        The result is the same as the following algorithm:

        for i, new_ace in enumerate(new_config):
            if new_ace.equals_no_order(old_ace):
                return i
        return -1
        '''
        new_ace = binary_search(new_config_cli2ace_dict, new_config_sorted_clis, old_ace)
        return new_config.index(new_ace) if new_ace else -1

    @staticmethod
    def longest_increasing_subsequence(x):
        '''
        Finds the longest increasing subsequence of the specified sequence.

        The subsequence is in sorted order (lowest to highest) and is as long as
        possible.  The subsequence is not necessarily contiguous.

        As an example, for the following sequence:
            1,2,0,3,4,5,6
        the longest increasing subsequence is:
            1,2,3,4,5,6
        '''

        if not x:
            return []

        best = [0]
        pred = [0] * len(x)
        for i in xrange(1, len(x)):
            if cmp(x[best[-1]], x[i]) < 0:
                pred[i] = best[-1]
                best.append(i)
                continue

            low = 0
            high = len(best) - 1
            while low < high:
                mid = (low + high) / 2
                if cmp(x[best[mid]], x[i]) < 0:
                    low = mid + 1
                else:
                    high = mid

            if cmp(x[i], x[best[low]]) < 0:
                if low > 0:
                    pred[i] = best[low -1]
                best[low] = i

        result = []
        j = best[-1]
        for i in xrange(len(best)):
            result.insert(0, x[j])
            j = pred[j]
        return result

    @staticmethod
    def sort_config(config):
        '''
        Sort by key and then sort by order.  This will ensure that the ACL
        order is always the same for ACLs that have the same order, even
        though a dictionary does not have an order.
        '''

        config.sort(key=lambda ace: ace.delta_ifc_key)
        config.sort(key=lambda ace: int(ace.values['order']))

    @staticmethod
    def what_moved(positions):
        if len(positions) <= 1:
            return []

        sequence = AccessList.longest_increasing_subsequence(positions)
        moved_positions = []
        for i in positions:
            if i not in sequence:
                moved_positions.append(i)
        return moved_positions

    @staticmethod
    def normalize_remarks(aces, is_old_config = False):
        """
        Transform the list of ACEs to another list of ACEs where remark in an ACE is converted into special ACE that is
        inserted before the one it belongs.
        @param aces: list of AccessControlEntry
        @param is_old: boolean
            it has the value True if aces is for old configuration, otherwise False.
        """
        for index in xrange(len(aces)-1, -1, -1):
            ace_remark = aces[index].get_remark();
            extra_remarks = aces[index].get_extra_remarks();
            if not ace_remark:
                continue
            state = ace_remark.get_state()
            if state == State.DESTROY:
                if is_old_config:
                    aces.insert(index, ace_remark)
            elif state in (State.CREATE, State.MODIFY):
                if not is_old_config:
                    aces.insert(index, ace_remark)
            else: # State.NOCHANGE
                aces.insert(index, ace_remark)
            if not is_old_config:
                continue
            'rid of extra remarks for the ACE'
            for i, extra_remark in enumerate(extra_remarks):
                aces.insert(index+i+1, extra_remark)

class AccessListList(DMList):
    'A list of access lists'

    NAME_PATTERN = None

    def __init__(self):
        DMList.__init__(self, 'AccessList', AccessList, 'access-list')
        self.clear_xlate = False

    def get_name_pattern(self):
        if not self.NAME_PATTERN:
            # Extract the access list name
            self.NAME_PATTERN = re.compile('access-list (\S+) .*')
        return self.NAME_PATTERN

    def get_translator(self, cli):
        'cli can be StructuredCommand, as the device package places remark(s) as sub-command(s) to the ACE it remarks'
        cli = cli.command if isinstance(cli, StructuredCommand) else cli
        m = self.get_name_pattern().match(cli.strip())
        if m:
            name = m.group(1)
            acl = self.get_child(name)
            if not acl:
                acl = self.child_class(name)
                self.register_child(acl)
            return acl

    def ifc2asa(self, no_asa_cfg_stack,  asa_cfg_list):
        cfg_stack = []
        cfg_list = []
        super(AccessListList, self).ifc2asa(cfg_stack, cfg_list)
        no_asa_cfg_stack.extend(cfg_stack)
        asa_cfg_list.extend(cfg_list)
        if (self.clear_xlate):
            self.generate_cli(asa_cfg_list,
                              'clear xlate',
                              response_parser=ignore_info_response_parser)
        if self.references_epg(cfg_stack + cfg_list):
            """
            To allow empty object-group in an ACE.
            Use no_asa_cfg_stack to make sure this command is issued ahead of any ACE creation command.
            """
            self.generate_cli(no_asa_cfg_stack,
                              'forward-reference enable')

    def set_clear_xlate(self):
        self.clear_xlate = True

    def references_epg(self, clis):
        '@return True if there is reference to an EPG in an ACE'
        return filter_first(lambda cli: str(cli).find('__$EPG$_') != -1 and not str(cli).startswith('no '), clis) != None
