'''
Created on Jan 13, 2017
Copyright (c) 2017 by Cisco Systems, Inc.

@author: dli
'''
import re
from copy import copy
from translator.base.dmlist import DMList
from translator.base.dmobject import DMObject
from translator.base.compositetype import CompositeType
from translator.base.simpletype import SimpleType
from translator.state_type import State, Type
from utils.util import filter_first, set_cfg_state, normalize_param_dict
from translator.structured_cli import StructuredCommand
from asaio import cli_interaction

class HiddenSimpleType(SimpleType):
    ''' A SimpleType that is not visible to the user in GlobalS2SVPNConfiguraiton.
    '''
    def __init__(self, ifc_key, asa_key):
        SimpleType.__init__(self, ifc_key, asa_key, asa_key)

    def create_missing_ifc_delta_cfg(self):
        '''Override the default implementation so that we create dictionary for this 'hidden' field
        '''
        if self.has_ifc_delta_cfg():
            return
        self.delta_ifc_key = Type.PARAM, self.ifc_key, ''
        self.delta_ifc_cfg_value = {'state': State.CREATE, 'value': {}}
        ancestor = self.get_ifc_delta_cfg_ancestor()
        if ancestor:
            ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] =  self.delta_ifc_cfg_value

    def is_the_same_cli(self, cli):
        return cli == self.asa_key

class HiddenCompositeType(CompositeType):
    ''' A CompositeTye that is not visible to the user in GlobalS2SVPNConfiguraiton.
    '''
    def __init__(self, ifc_key, asa_key):
        CompositeType.__init__(self, ifc_key, asa_key, asa_key)

    def create_missing_ifc_delta_cfg(self):
        '''Override the default implementation so that we create dictionary for this 'hidden' field
        '''
        if self.has_ifc_delta_cfg():
            return
        self.delta_ifc_key = Type.FOLDER, type(self).__name__, self.ifc_key
        self.delta_ifc_cfg_value = {'state': State.NOCHANGE, 'value': {}}
        ancestor = self.get_ifc_delta_cfg_ancestor()
        if ancestor:
            ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] =  self.delta_ifc_cfg_value
        for child in self.children.itervalues():
            child.create_missing_ifc_delta_cfg()

    def parse_cli(self, cli):
        return {}

    def create_delta_ifc_key(self, cli):
        return (Type.PARAM, self.parent.ifc_key, self.asa_key.split(' ')[-1])

class HiddenSimpleType2(SimpleType):
    'Model CLIs not exposed to user in S2SVPNPolicy'
    def __init__(self, ifc_key, asa_key, asa_gen_template):
        SimpleType.__init__(self, ifc_key, asa_key, asa_gen_template)

    def is_the_same_cli(self, cli):
        return cli == self.get_cli()

    def set_state(self, state):
        my_key = (Type.PARAM, self.ifc_key, '')
        self.delta_ifc_cfg_value['value'][my_key] = {'state': state, 'value': ''}

    def get_state(self):
        my_key = (Type.PARAM, self.ifc_key, '')
        if self.delta_ifc_cfg_value['value'].has_key(my_key):
            result = self.delta_ifc_cfg_value['value'][my_key]['state']
        else:
            result =  SimpleType.get_state(self)
            if result == State.MODIFY:
                result = State.NOCHANGE # during serviceModify, this is read-only
        return result

    def create_missing_ifc_delta_cfg(self):
        'Override the default implementation to represent us in the dictionary'
        my_key = (Type.PARAM, self.ifc_key, '')
        if  not self.delta_ifc_cfg_value['value'].has_key(my_key):
            delta_ifc_key = my_key
            delta_ifc_cfg_value = {'state': State.CREATE, 'value': ''}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][delta_ifc_key] =  delta_ifc_cfg_value

class PermitVPN(HiddenSimpleType):
    '''
    Model after CLI 'sysopt connection permit-vpn'. The default value on ASA is 'sysopt connection permit-vpn',
    this means it only shows up in the running config as 'no sysopt connection permit-vpn'
    '''
    def __init__(self):
        HiddenSimpleType.__init__(self, PermitVPN.__name__, 'sysopt connection permit-vpn')

    def diff_ifc_asa(self, cli):
        '''
        cli is 'no sysopt connection permit-vpn' from ASA, we need to create 'sysopt connection permit-vpn'.
        '''
        if self.has_ifc_delta_cfg():
            self.set_action(State.CREATE)
            self.delta_ifc_cfg_value['is_default'] = False

    def is_the_same_cli(self, cli):
        'is_my_cli has made sure it is the same as CLI'
        return True

    def is_my_cli(self, cli):
        'only shows up in running-config if it is the no form of the command'
        if isinstance(cli, basestring):
            return
        if not cli.is_no:
            return
        command =  cli.command.strip()
        return command == self.asa_key

    def create_missing_ifc_delta_cfg(self):
        'Take care of the fact that by default, the CLI is on the ASA'
        HiddenSimpleType.create_missing_ifc_delta_cfg(self)
        self.delta_ifc_cfg_value['is_default'] = True if self.get_top().is_audit else False

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        'Take care of the fact that by default, the CLI is on the ASA'
        if not self.has_ifc_delta_cfg():
            return
        if self.delta_ifc_cfg_value.get('is_default', False):
            self.set_action(State.NOCHANGE)
        HiddenSimpleType.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

class GlobalS2SVPNConfiguration(DMObject):
    '''
    Global configuration for site-to-site VPN:
    - by pass ACL: sysopt connection permit-vpn
    - ipsec proposals
    - ike policies
    It is implicit. i.e. not configurable by user from APIC.
    '''
    def __init__(self):
        DMObject.__init__(self, GlobalS2SVPNConfiguration.__name__)
        self.register_child(IKEv2PolicyList())
        self.register_child(IPSecProposalList())
        self.register_child(PermitVPN())

    def update_references(self):
        '''
        Override the default implementation so that the CLIs are generated if there is any newly created S2SVPN Policy
        '''
        if self.get_top().is_audit:
            return
        self.create_missing_ifc_delta_cfg()
        if hasattr(self, 'delta_ifc_cfg_value'):
            set_cfg_state(self.delta_ifc_cfg_value, State.CREATE)

    def create_missing_ifc_delta_cfg(self):
        '''
        Override default implementation to create dictionary for the 'hidden' parameters if there is any
        S2SVPNPolicy defined.
        '''
        if not self.get_top().is_audit and not self.has_new_s2svpn_policy():
            # do nothing in serviceModify if there is no new policy.
            return
        if  self.has_ifc_delta_cfg():
            return
        self.delta_ifc_key = Type.FOLDER, self.ifc_key, ''
        self.delta_ifc_cfg_value = {'state': State.NOCHANGE, 'value': {}}
        ancestor = self.get_ifc_delta_cfg_ancestor()
        ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] =  self.delta_ifc_cfg_value
        if self.get_top().is_audit:
            if not self.has_s2svpn_policy():
                #ready to clean up of global S2S VPN configuration
                return
        for child in self.children.itervalues():
            child.create_missing_ifc_delta_cfg()

    def has_new_s2svpn_policy(self):
        '''
        See if there is any new S2SVPNPolicy.
        '''
        hit = filter_first(lambda policy: policy.get_state() == State.CREATE, self.get_all_s2svpn_policies())
        return hit != None

    def has_s2svpn_policy(self):
        '''
        See if there is any S2SVPNPolicy.
        '''
        hit = filter_first(lambda policy: policy.get_state() != State.DESTROY, self.get_all_s2svpn_policies())
        return hit != None

    def get_all_s2svpn_policies(self):
        intf_list = self.parent.get_child('Interface')
        result = []
        for intf in intf_list:
            s2svpn_policy_list = intf.get_child('S2SVPNPolicy')
            s2svpn_policy_list.update_references()
            result.extend(s2svpn_policy_list)
        return result

    def get_translator(self, cli):
        if self.has_ifc_delta_cfg():
            return DMObject.get_translator(self, cli)

class IPSecProposalList(DMList):
    '''
    CLIs:
        crypto ipsec ikev2 ipsec-proposal ASA-DP
         protocol esp encryption aes-gcm-256  aes-gmac-256 aes-256 aes-gcm-192 aes-192 3des aes
         protocol esp integrity sha-512 sha-384 sha-256 sha-1 null
    '''
    def __init__(self):
        super(IPSecProposalList, self).__init__(IPSecProposal.__name__, IPSecProposal,
                                                'crypto ipsec ikev2 ipsec-proposal')
        self.register_child(IPSecProposal('ASA-DP'))

    def get_translator(self, cli):
        'Override the default to ignore any ikev2 ipsec-proposal defined by user'
        if not self.is_my_cli(cli):
            return
        'Find the translator from its children'
        result = None
        for child in self.children.values():
            if hasattr(self, 'get_translator'):
                result = child.get_translator(cli)
                if result:
                    return result

class IPSecProposal(HiddenCompositeType):
    ''' CLI:
        crypto ipsec ikev2 ipsec-proposal ASA-DP
         protocol esp encryption aes-gcm-256 aes-gcm-192 aes-gcm aes-256 aes-192 aes 3des aes-gmac-256 aes-gmac-192 aes-gmac
         protocol esp integrity sha-512 sha-384 sha-256 sha-1 null

       The order of encryption algorithms is set by ASA. Try to have the *256 come ahead of *192, but
       ASA rearranges them in the order shown here.
    '''
    def __init__(self, name):
        HiddenCompositeType.__init__(self, name,'crypto ipsec ikev2 ipsec-proposal ' + name)
        self.register_child(HiddenSimpleType('encryption', 'protocol esp encryption aes-gcm-256 aes-gcm-192 aes-gcm aes-256 aes-192 aes 3des aes-gmac-256 aes-gmac-192 aes-gmac'))
        self.register_child(HiddenSimpleType('integrity', 'protocol esp integrity sha-512 sha-384 sha-256 sha-1 null'))


class IKEv2PolicyList(DMList):
    '''
    CLIs:
        # for aes and des
         crypto ikev2 policy 10
         encryption aes-256 aes-192 aes 3des
         integrity sha512 sha384 sha256 sha
         group 21 20 19 24 14 5 2 1
         prf sha512 sha384 sha256 sha
         lifetime seconds 86400
        # for aes-gcm
        crypto ikev2 policy 20
         encryption aes-gcm-256 aes-gcm-192 aes-gcm
         integrity null
         group 21 20 19 24 14 5 2 1
         prf sha512 sha384 sha256
         lifetime seconds 86400
            '''
    def __init__(self):
        super(IKEv2PolicyList, self).__init__(IKEv2Policy.__name__, IKEv2Policy,'crypto ikev2 policy')
        for (number, encryption, integrity) in [(10, 'aes-256 aes-192 aes 3des', 'sha512 sha384 sha256 sha'),
                                                (20, 'aes-gcm-256 aes-gcm-192 aes-gcm', 'null')]:
            self.register_child(IKEv2Policy(str(number), encryption, integrity))

    def get_translator(self, cli):
        'Override the default to ignore any ikev2 policy defined by user'
        if not self.is_my_cli(cli):
            return
        'Find the translator from its children'
        result = None
        for child in self.children.values():
            if hasattr(self, 'get_translator'):
                result = child.get_translator(cli)
                if result:
                    return result

class IKEv2Policy(HiddenCompositeType):
    ''' CLI example:
         crypto ikev2 policy 10
          encryption aes-256 aes-192 aes 3des
          integrity sha512 sha384 sha256 sha
          group 21 20 19 24 14 5 2 1
          prf sha512 sha384 sha256 sha
          lifetime seconds 86400
    '''
    def __init__(self, name, encryption, integrity):
        HiddenCompositeType.__init__(self, name, 'crypto ikev2 policy ' + name)
        self.register_child(HiddenSimpleType('encryption', 'encryption ' + encryption))
        self.register_child(HiddenSimpleType('integrity', 'integrity ' + integrity))
        self.register_child(HiddenSimpleType('group', 'group 21 20 19 24 14 5 2 1'))
        self.register_child(HiddenSimpleType('prf', 'prf sha512 sha384 sha256 sha'))
        self.register_child(HiddenSimpleType('lifetime', 'lifetime seconds 86400'))

class EnableIKE2(SimpleType):
    '''
    It models the CLI ''crypto ikev2 enable <nameif>'
    This object is not exposed to the user.
    It is on if there is any active policy.
    '''
    def __init__(self):
        SimpleType.__init__(self, EnableIKE2.__name__, asa_gen_template = 'crypto ikev2 enable %s')

    def get_cli_prefixes(self):
        return set(['crypto'])

    def create_missing_ifc_delta_cfg(self):
        '''
        Override the default to create the dictionary for it.
        '''
        if not self.asa_key:
            self.asa_key = self.create_asa_key()
        if self.has_ifc_delta_cfg():
            return
        s2s_vpn_policy_list = self.get_interface().get_child(S2SVPNPolicy.__name__)
        if not s2s_vpn_policy_list.children:
            return
        self.delta_ifc_key = (Type.PARAM, self.ifc_key, '')
        self.delta_ifc_cfg_value = {'state': self.determine_state(), 'value': self.get_interface().nameif}
        self.parent.delta_ifc_cfg_value['value'][self.delta_ifc_key] = self.delta_ifc_cfg_value

    def create_asa_key(self):
        return self.asa_gen_template % self.get_interface().nameif

    def determine_state(self):
        """
        The sate depends on the whether there is any S2SVPN policy on the interface.
        """
        s2s_vpn_policy_list = self.get_interface().get_child(S2SVPNPolicy.__name__)
        states = map(lambda child: child.get_state(), s2s_vpn_policy_list.children.values())
        if filter(lambda state: state == State.CREATE, states):
            result = State.CREATE
        elif len(filter(lambda state: state == State.DESTROY, states)) == len(states):
            result = State.DESTROY
        else:
            result = State.NOCHANGE
        return result

    def get_interface(self):
        return self.parent.parent

class CryptoMapBindInterface(EnableIKE2):
    '''
    It models the CLI 'crypto map <nameif> interface <nameif>'
    This object is not exposed to the user.
    It is on if there is any active policy.
    '''
    def __init__(self):
        SimpleType.__init__(self, CryptoMapBindInterface.__name__,
                            asa_gen_template = 'crypto map %s interface %s')

    def create_asa_key(self):
        return self.asa_gen_template % (self.get_interface().nameif, self.get_interface().nameif)

    def get_cli(self):
        return self.asa_key

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        '''
        1. There is no need to generate 'no' command, as it is covered by the 'clear config crypto map <nameif>' command already.
        2.. Will still issue command if there is a replacement operation of crypto map entry.
            This happens if user changes the value of peer_ip, or order of the policy
        '''
        if self.get_action() == State.DESTROY:
            # no need to generate 'no' form of the command
            return
        if self.get_state() == State.NOCHANGE and self.is_crypto_map_binding_required():
            self.set_state(State.CREATE)
        EnableIKE2.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

    def is_crypto_map_binding_required(self):
        '''Return true if we need to issue command to bind the crypto map to the interface.
        The decision is made on whether there is any newly created or modified policy.
        '''
        for policy in self.get_interface().get_child(S2SVPNPolicy.__name__):
            if policy.get_state() == State.CREATE:
                return True
            if policy.get_order_state() == State.CREATE: #due to the creation of a new policy, or change order of an existingone.
                return True

class S2SVPNEnabler(DMObject):
    '''
    MOB to enable S2SVPN on the interface.
    This object is not exposed to the user.
    It enables S2SVPN on the interface if there is S2SVPNPolicy for the interface after
    the operation.
    The enabling consists of enable IKEv2 on the interface as well as bind the crypto map to the interface.
    '''
    def __init__(self):
        DMObject.__init__(self, S2SVPNEnabler.__name__)
        self.register_child(CryptoMapBindInterface())
        self.register_child(EnableIKE2())

    def update_references(self):
        '''
        Override the default to create the dictionary for it and its children.
        '''
        self.create_missing_ifc_delta_cfg()

class S2SVPNPolicyList(DMList):
    '''
    Site-to-site VPN configuration for a given interfaces
    '''

    def __init__(self):
        super(S2SVPNPolicyList, self).__init__(S2SVPNPolicy.__name__, S2SVPNPolicy,
                                               mini_audit_command = ['show run crypto map',
                                                                     'show run group-policy',
                                                                     'show run tunnel-group'])

    def read_asa_config(self):
        """
        Override the default because we have to get the output of multiple 'show run' commands
        """
        result = ''
        for show_cmd in self.mini_audit_command:
            resp =  self.query_asa(show_cmd)
            if resp:
                result += resp
        return result

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        is_audit = self.get_top().is_audit
        if filter_first(lambda child: child.is_mini_audit_required(), self):
            # Remove any entries to be deleted
            for key, child in self.children.items():
                if child.get_state() == State.DESTROY:
                    del(self.children[key])
                else: #set states to CREATE
                    set_cfg_state(child.delta_ifc_cfg_value, State.CREATE)
            self.mini_audit()
            is_audit = True
        if self.are_all_policies_deleted():
            '''
            We need to summmarize 'clear crypto map <name> <order' if there is the result
            of deletion results in the complete deletion of the crypto map
            '''
            tmp_no_asa_cfg_stack = []
            DMObject.ifc2asa(self, tmp_no_asa_cfg_stack, asa_cfg_list)
            tmp_no_asa_cfg_stack = self.summarize_clear_crypto_map_commands(tmp_no_asa_cfg_stack)
            no_asa_cfg_stack.extend(tmp_no_asa_cfg_stack)
        else:
            if is_audit and len(self.children) > 1:
                self.consolidate_policies(no_asa_cfg_stack)
            DMObject.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

    def get_cli_prefixes(self):
        ''' Make sure we have the cli_prefixes defined if if there no child yet defined.
        The other advantage is to only do it once rather than once every member
        '''
        return S2SVPNPolicy('dummy').get_cli_prefixes()

    def get_translator(self, cli):
        '''Override the default implementation, as there is no 1-1 corresponding between CLI
         and IFC configuration
        '''
        #look up from existing policies
        result = None
        for child in self.children.values():
            if hasattr(self, 'get_translator'):
                result = child.get_translator(cli)
                if result:
                    return result
        # if not found, create a S2SVPNPolicy that will generate CLIs to delete the policy
        dummy = S2SVPNPolicy(cli)
        dummy.parent = self
        result = dummy.get_translator(cli)
        if result:
            self.register_child(dummy)
            return result

    def validate_configuration(self):
        '''Make sure we don't have different policies using the same 'order' or 'peer_ip'
        '''
        faults = []
        checked = []
        active_polices = filter(lambda policy: policy.get_state() != State.DESTROY, self)
        make_msg = lambda name, field: 'This policy has the same %s as the policy named \'%s\'. They must be different.' % (field, name)
        for policy in active_polices:
            order = policy.get_order()
            name = policy.delta_ifc_key[2]
            peer = policy.get_peer_ip()
            checked.append(policy)
            for other in active_polices:
                if other in checked:
                    continue
                if order == other.get_order():
                    faults.append(other.generate_fault(make_msg(name, 'order number')))
                if peer == other.get_peer_ip():
                    faults.append(other.generate_fault(make_msg(name, 'peer_ip')))
        return faults

    def are_all_policies_deleted(self):
        states = map(lambda policy: policy.get_state(), self)
        result = states and all(map(lambda state: state == State.DESTROY, states))
        return result

    def summarize_clear_crypto_map_commands(self, no_asa_cfg_stack):
        '''
        Replace multiple 'clear config crypto map <nameif> <order>' commands by a single
        'clear config crypto map <nameif>' command.
        '''
        result = filter(lambda clii: not clii.command.startswith('clear config crypto map'),
                         no_asa_cfg_stack)
        self.generate_cli(result, 'clear config crypto map ' + self.parent.nameif)
        return result

    def consolidate_policies(self, no_asa_cfg_stack):
        '''
        Taking care of the case changing 'order' number of a policy during audit.
        Use 'peer_ip' number as the key.
        We will see two polcies with the same peer_id, one is in the state DESTROY,
        its 'order' number is the old number. The only thing we need to do for the old ploilcy
        is to delete the old crypto map entry by issuing:
           'clear crypto map <nameif> <older_order_number>'
        There is no need to issue 'clear config group-policy' or 'clear config tunnel-group' command.
        '''
        policies_to_be_deleted = filter(lambda policy: policy.get_state() == State.DESTROY, self)
        for policy in policies_to_be_deleted:
            if filter_first(lambda p: p != policy and p.get_peer_ip() == policy.get_peer_ip(), self):
                self.remove(policy)
                self.generate_cli(no_asa_cfg_stack, 'clear config crypto map ' + self.parent.nameif + ' ' + policy.get_order())

class S2SVPNPolicy(DMObject):
    '''
    It is a combination of TunnelGroup, GroupPolicy, CrytoMap, and some misc stuff
    '''
    def __init__(self, instance):
        DMObject.__init__(self, instance)
        self.register_child(CryptoMap())
        self.register_child(GroupPolicy())
        self.register_child(TunnelGroup())
        self.register_child(Advanced())

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        self.normalize_state()
        DMObject.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        delta_ifc_cfg_value['value'][(Type.PARAM, 'nameif', '')] = {'state': State.NOCHANGE,
                                                                    'value': self.parent.parent.nameif}
        S2SVPNPolicy.populate_model_recursively(self, delta_ifc_key, delta_ifc_cfg_value)
        advanced_key =  filter_first(lambda key: key[1] == Advanced.__name__, self.delta_ifc_cfg_value['value'].keys())
        if advanced_key:
            self.get_child(Advanced.__name__).populate_model(advanced_key, self.delta_ifc_cfg_value['value'][advanced_key])

    def create_ifc_delta_cfg(self, peer_ip, order):
        ''''
        Uses the given peer_ip to create the delta_ifc_cfg for the policy.
        This is called during audit operation when a policy is seen on ASA but on on APIC.
        '''
        delta_ifc_key = (Type.FOLDER, S2SVPNPolicy.__name__, self.ifc_key)
        delta_ifc_cfg_value = {'state': State.DESTROY,
                               'value' : {
                                          (Type.PARAM, 'order', ''):   {'state': State.DESTROY, 'value': order},
                                          (Type.PARAM, 'peer_ip', ''): {'state': State.DESTROY, 'value': peer_ip},
                                          (Type.PARAM, 'pre_shared_key_local', ''): {'state': State.DESTROY, 'value': ''},
                                          (Type.PARAM, 'pre_shared_key_remote', ''): {'state': State.DESTROY, 'value': ''},
                                          (Type.PARAM, 'traffic_selection', ''): {'state': State.DESTROY, 'value': ''},
                                          }}
        self.populate_model(delta_ifc_key, delta_ifc_cfg_value)
        self.parent.parent.delta_ifc_cfg_value['value'][delta_ifc_key] = delta_ifc_cfg_value

    def is_peer_ip_modified(self):
        return self.get_field_state('peer_ip') == State.MODIFY

    def is_order_modified(self):
        return self.get_field_state('order') == State.MODIFY

    def get_field_state(self, field_name):
        field_key = filter_first(lambda key: key[1] == field_name, self.delta_ifc_cfg_value['value'].keys())
        return self.delta_ifc_cfg_value['value'][field_key]['state']

    def is_mini_audit_required(self):
        return self.is_peer_ip_modified() or self.is_order_modified()

    def get_translator(self, cli):
        '''Override the default implementation to return this S2SVPNPolicy if the CLI belongs to it.
        CLIs:
            tunnel-group
            group-policy
            crypto
        '''
        if self.has_ifc_delta_cfg() and self.get_state() == State.DESTROY:
            '''This only happens if the policy is on ASA but not on APIC.
            In this case there is no need to process further related CLI from the ASA.
            '''
            return
        for child in self.children.itervalues():
            result = child.get_translator(cli)
            if result:
                return result

    def normalize_state(self):
        '''The state of the policy is determined by its components
        '''
        #filter out the state of 'nameif', as it is not exposed to the user.
        params = copy(self.delta_ifc_cfg_value['value'])
        params.pop((Type.PARAM, 'nameif', ''))
        states = map(lambda value: value['state'], params.values())
        peer_ip_key = filter_first(lambda key: key[1] == 'peer_ip', params.keys())
        peer_ip_state= params[peer_ip_key]['state']
        if all(map(lambda state: state == State.NOCHANGE, states)):
            self.set_state(State.NOCHANGE)
        elif DMObject.get_state(self) == State.CREATE and peer_ip_state == State.CREATE:
            #recreate everything if we have a change to 'peer_ip'
            set_cfg_state(self.delta_ifc_cfg_value, State.CREATE)
        elif DMObject.get_state(self) == State.NOCHANGE and any(map(lambda state: state != State.NOCHANGE, states)):
            self.set_state(State.MODIFY)

    def get_order(self):
        return self.get_value()['order']

    def get_order_state(self):
        key = filter_first(lambda key: key[1] == 'order', self.delta_ifc_cfg_value['value'].keys())
        return self.delta_ifc_cfg_value['value'][key]['state']

    def get_peer_ip(self):
        return self.get_value()['peer_ip']

    @staticmethod
    def populate_model_recursively(dmobject, delta_ifc_key, delta_ifc_cfg_value):
        dmobject.delta_ifc_key = delta_ifc_key
        dmobject.delta_ifc_cfg_value = delta_ifc_cfg_value
        if not dmobject.children:
            return
        for child in dmobject.children.values():
            if child.ifc_key == Advanced.__name__:
                continue
            S2SVPNPolicy.populate_model_recursively(child, delta_ifc_key, delta_ifc_cfg_value)
            if not child.asa_key and child.asa_gen_template:
                child.asa_key = child.asa_gen_template % child.get_value()

class SuperObject(DMObject):
    '''
    DMObject that consists of multiple CLIs, such as tunnel-group, which consists of
        tunnel-group <name> type ipsec-l2l, and
        tunnel-group <name> ispec-attributes
    When destroy the object, a single 'clear config' command is needed rather than mulitple
    '''

    def __init__(self, ifc_key, asa_key):
        DMObject.__init__(self, ifc_key, asa_key)

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        self.asa_key = self.asa_key % self.get_value()
        if self.get_state() == State.DESTROY:
            self.generate_cli(no_asa_cfg_stack, 'clear config ' + self.asa_key)
        else:
            DMObject.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

class GroupPolicy(SuperObject):
    '''
    CompositeType with multiple global commands. Easier to have two commands.
    CLIs
      group-policy externalIf_1.1.1.1 internal
      group-policy externalIf_1.1.1.1 attributes
        vpn-tunnel-protocol ikev2
    '''
    def __init__(self):
        DMObject.__init__(self, GroupPolicy.__name__, 'group-policy %(nameif)s_%(peer_ip)s')
        self.register_child(GroupPolicyInternal())
        self.register_child(GroupPolicyAttributes())

class GroupPolicyInternal(SimpleType):
    '''
    CLI
      group-policy 1.1.1.1 internal
    '''
    def __init__(self):
        SimpleType.__init__(self, GroupPolicyInternal.__name__,
                            asa_gen_template='group-policy %(nameif)s_%(peer_ip)s internal',
                            response_parser=cli_interaction.ignore_response_parser)

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.get_state() != State.MODIFY:
            SimpleType.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

class GroupPolicyAttributes(CompositeType):
    '''
    CLI:
      group-policy 1.1.1.1 attributes
        vpn-tunnel-protocol ikev2
    '''
    def __init__(self):
        CompositeType.__init__(self, GroupPolicyAttributes.__name__,
                               asa_gen_template='group-policy %(nameif)s_%(peer_ip)s attributes')
        self.register_child(HiddenSimpleType2('vpn-tunnel-protocol', 'vpn-tunnel-protocol', 'vpn-tunnel-protocol ikev2'))

class TunnelGroup(SuperObject):
    '''
    Three global commands, two are composite.
    CLIs:
      tunnel-group 1.1.1.1 type ipsec-l2l
      tunnel-group 1.1.1.1 general-attributes
        default-group-policy GroupPolicy_1.1.1.1
      tunnel-group 1.1.1.1 ipsec-attributes
        ikev2 local-authentication pre-shared-key key
        ikev2 remote-authentication pre-shared-key key
        isakmp keepalive threshold 10 retry 2
    '''

    def __init__(self):
        SuperObject.__init__(self, TunnelGroup.__name__, 'tunnel-group %(peer_ip)s')
        self.register_child(TunnelGroupType())
        self.register_child(TunnelGroupGeneralAttributes())
        self.register_child(TunnelGroupIPSecAttributes())

class TunnelGroupType(SimpleType):
    '''
    CLI:
      tunnel-group 1.1.1.1 type ipsec-l2l
    '''
    def __init__(self):
        SimpleType.__init__(self, TunnelGroupType.__name__,
                            asa_gen_template='tunnel-group %(peer_ip)s type ipsec-l2l',
                            response_parser=cli_interaction.ignore_response_parser)

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.get_state() != State.MODIFY:
            SimpleType.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

    def diff_ifc_asa(self, cli):
        #@todo what if the cli is different?
        return

class TunnelGroupGeneralAttributes(CompositeType):
    '''
    CLIs:
      tunnel-group 1.1.1.1 general-attributes
        default-group-policy 1.1.1.1
    '''
    def __init__(self):
        CompositeType.__init__(self, TunnelGroupGeneralAttributes.__name__,
                               asa_gen_template='tunnel-group %(peer_ip)s general-attributes')
        self.register_child(HiddenSimpleType2('default-group-policy','default-group-policy', 'default-group-policy %(nameif)s_%(peer_ip)s'))

class TunnelGroupIPSecAttributes(CompositeType):
    '''
    CLIs:
      tunnel-group 1.1.1.1 ipsec-attributes
        ikev2 local-authentication <pre_shared_key_local>
        ikev2 remote-authentication <pre_shared_key_remote>
    '''
    def __init__(self):
        CompositeType.__init__(self, TunnelGroupIPSecAttributes.__name__,
                               asa_gen_template='tunnel-group %(peer_ip)s ipsec-attributes')
        self.register_child(PresharedKey('pre_shared_key_remote', asa_gen_template='ikev2 remote-authentication pre-shared-key %(pre_shared_key_remote)s'))
        self.register_child(PresharedKey('pre_shared_key_local', asa_gen_template='ikev2 local-authentication pre-shared-key %(pre_shared_key_local)s'))

    def diff_ifc_asa(self, cli):
        if isinstance(cli, StructuredCommand):
            for cmd in cli.sub_commands:
                translator = self.get_child_translator(cmd)
                if translator:
                    translator.diff_ifc_asa(cmd)

class PresharedKey(SimpleType):
    def __init__(self, ifc_key, asa_gen_template):
        SimpleType.__init__(self, ifc_key, asa_gen_template=asa_gen_template,
                            response_parser=cli_interaction.ignore_info_response_parser)

    def get_action(self):
        dict_value = self.get_dict_value()
        if dict_value:
            return dict_value['state']

    def set_action(self, state):
        dict_value = self.get_dict_value()
        if dict_value:
            dict_value['state'] = state

    def get_dict_value(self):
        dict_key = filter_first(lambda key: key[1] == self.get_key(), self.delta_ifc_cfg_value['value'].keys())
        if dict_key:
            return self.delta_ifc_cfg_value['value'][dict_key]

class CryptoMap(SuperObject):
    '''
    CLIs:
      crypto map externalIf 1 match access-list-s2svpn
      crypto map externalIf 1 set peer  1.1.1.1
      crypto map externalIf 1 set ikev2 ipsec-proposal ASA-DP
      crypto map externalIf interface externalIf
    '''
    def __init__(self):
        DMObject.__init__(self, CryptoMap.__name__, 'crypto map %(nameif)s %(order)s')
        self.register_child(CryptoMapType('traffic_selection', 'crypto map %(nameif)s %(order)s match address %(traffic_selection)s'))
        self.register_child(CryptoMapPeerIP())
        self.register_child(CryptoMapIPsecProposal())

class CryptoMapIPsecProposal(HiddenSimpleType2):
    def __init__(self):
        HiddenSimpleType2.__init__(self, HiddenSimpleType2.__name__,
                                   '',
                                   'crypto map %(nameif)s %(order)s set ikev2 ipsec-proposal ASA-DP')

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        '''
        Override the default implementation because we need ot issue the 'no' for of the command first when changing the value
        '''
        if self.get_state() in (State.CREATE, State.MODIFY):
            self.generate_cli(asa_cfg_list, 'no crypto map %(nameif)s %(order)s set ikev2 ipsec-proposal' % self.get_value())
        return HiddenSimpleType2.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

class CryptoMapType(SimpleType):
    'Model after MO visible to the end user'
    def __init__(self, ifc_key, asa_gen_template):
        SimpleType.__init__(self, ifc_key,
                            asa_gen_template=asa_gen_template,
                            response_parser=cli_interaction.ignore_warning_response_parser)

    def get_action(self):
        dict_value = self.get_dict_value()
        if dict_value:
            return dict_value['state']

    def set_action(self, state):
        dict_value = self.get_dict_value()
        if dict_value:
            dict_value['state'] = state
            if state == State.NOCHANGE:
                '''
                'order' is NOCHANGE if any of the crypto field is NOCHANGE
                '''
                order_key = filter_first(lambda key: key[1] == 'order', self.delta_ifc_cfg_value['value'].keys())
                self.delta_ifc_cfg_value['value'][order_key]['state'] = State.NOCHANGE

    def get_dict_value(self):
        dict_key = filter_first(lambda key: key[1] == self.ifc_key, self.delta_ifc_cfg_value['value'].keys())
        if dict_key:
            return self.delta_ifc_cfg_value['value'][dict_key]

class CryptoMapPeerIP(CryptoMapType):
    '''
    Model after the CLI 'crypto map <name> <number> set peer <ip>'
    We use it to identify the existence of a S2SVPNPolicy on ASA.
    '''
    def __init__(self):
        CryptoMapType.__init__(self, 'peer_ip', 'crypto map %(nameif)s %(order)s set peer %(peer_ip)s')

    def is_my_cli(self, cli):
        if not self.asa_key:
            'We got here because the extra config on ASA during audit'
            self.asa_key = re.compile('crypto map ' + self.parent.parent.parent.parent.nameif + ' \d+ set peer')
        return CryptoMapType.is_my_cli(self, cli)

    def diff_ifc_asa(self, cli):
        if self.has_ifc_delta_cfg():
            return CryptoMapType.diff_ifc_asa(self, cli)
        'Extra policy on the ASA'
        ifc_dict = normalize_param_dict(self.parse_cli(cli))
        self.parent.parent.create_ifc_delta_cfg(ifc_dict['peer_ip'], ifc_dict['order'])

class Advanced(DMObject):
    '''The advanced parameters
    '''
    def __init__(self):
        DMObject.__init__(self, Advanced.__name__)
        self.register_child(AdvancedParameter('pfs', 'pfs'))
        self.register_child(AdvancedParameter('sa_lifet_time_in_kilobytes', 'security-association lifetime kilobytes'))
        self.register_child(AdvancedParameter('sa_lifet_time_in_seconds', 'security-association lifetime seconds'))

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        '''Override the default implementation so that we do not generate CLI in case of destroy if the parant is destroyed,
        because the parant's CryptoMap object will clean up the whole crypto map.
        '''
        if self.parent.get_state() != State.DESTROY:
            DMObject.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

class AdvancedParameter(SimpleType):
    def __init__(self, ifc_key, asa_sub_key):
        SimpleType.__init__(self, ifc_key)
        self.asa_sub_key = asa_sub_key

    def update_references(self):
        self.asa_key = 'crypto map %s %s set %s ' % (self.get_nameif(), self.get_order(), self.asa_sub_key)
        self.asa_gen_template = self.asa_key + '%s'

    def get_order(self):
        return self.parent.parent.get_value()['order']

    def get_nameif(self):
        return self.parent.parent.get_value()['nameif']
