#
# Copyright (c) 2013 by Cisco Systems, Inc.
#
'''
Created on Jun 28, 2013

@author: feliu

This is the static route class.

'''

from state_type import Type, State
from validators import IPv6AddressValidator, IPv6AddressSubnetValidator
from base.dmobject import DMObject
from base.dmlist import DMList
from base.simpletype import SimpleType
from utils.util import normalize_ipv6_address, netmask_from_prefix_length,\
    filter_first


class StaticRoute(DMObject):
    def __init__(self):
        super(StaticRoute, self).__init__(StaticRoute.__name__)
        self.register_child(RouteList('route', Route, 'route ',
                                      mini_audit_command = 'show run route '))
        self.register_child(RouteList('ipv6_route', IPv6Route, 'ipv6 route ',
                                      mini_audit_command = 'show run ipv6 | grep ^ipv6 route '))

    def get_interface_nameif(self):
        return self.parent.nameif

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.has_ifc_delta_cfg() and self.parent.get_state() == State.DESTROY:
            """
            No need to issue 'no route <nameif> ...' command if the interface is cleared.
            Because 'clear config interface ..' will clear all configurations referencing it.
            Otherwise these commands will be rejected by ASA any way, as the 'route <nameif> ..' are already gone.
            CSCuw60753
            """
            return
        DMObject.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

class RouteList(DMList):
    def get_interface_nameif(self):
        return self.parent.get_interface_nameif()

    def is_my_cli(self, cli):
        if isinstance(cli, basestring):
            return cli.startswith(self.asa_key + self.get_interface_nameif())

class Route(SimpleType):
    '''
    This class represents the holder of all IPv4 static routes.
    '''

    def __init__(self,
                 name,
                 asa_gen_template='route %(interface)s %(network)s %(netmask)s %(gateway)s %(metric)s',
                 defaults={'metric': '1'}):
        super(Route, self).__init__(
            name,
            asa_gen_template=asa_gen_template,
            defaults=defaults,
            response_parser = static_route_response_parser)

    def get_cli(self):
        value = self.get_value_with_interface()
        for name in self.defaults:
            value[name] = value.get(name, self.defaults[name])
        return self.asa_gen_template % value

    def get_value_with_interface(self):
        value = self.get_value()
        value['interface'] = self.parent.get_interface_nameif()
        return value

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        'Prevent the deletion of route on management interface or IFC will not be able to connect to the ASA'
        self.is_removable = not self.parent.get_interface_nameif().startswith('management')
        super(Route, self).ifc2asa(no_asa_cfg_stack, asa_cfg_list)

    def create_asa_key(self):
        '''
        Create the asa key that identifies this object, everything except
        'metric' makes up the key
        @return str
        '''
        return ('route %(interface)s %(network)s %(netmask)s %(gateway)s' %
                self.get_value_with_interface())

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.normalize_netmask(delta_ifc_cfg_value)
        return SimpleType.populate_model(self, delta_ifc_key, delta_ifc_cfg_value)

    def normalize_netmask(self, config):
        '''
        If the netmask is prefix length for IPv4, turns it into netmask
        '''
        netmask_key = filter_first(lambda key: key[1] == 'netmask', config['value'].keys())
        if not netmask_key:
            return
        netmask = config['value'][netmask_key]['value']
        if netmask.isdigit():
            config['value'][netmask_key]['value'] = netmask_from_prefix_length(netmask)

class IPv6Route(Route):
    '''
    This class represents the holder of all IPv6 static routes.
    '''

    def __init__(self, name):
        super(IPv6Route, self).__init__(
            name,
            asa_gen_template = 'ipv6 route %(interface)s %(prefix)s %(gateway)s %(hop_count)s %(tunneled)s',
            defaults = {'hop_count': '', 'tunneled': ''})

    def normalize_ipv6_values(self, value):
        '''
        @param value - is a dictionary that contains ipv6 prefix and gateway.
        '''
        value['prefix'] = normalize_ipv6_address(value['prefix'])
        value['gateway'] = normalize_ipv6_address(value['gateway'])
        return value

    def get_cli(self):
        value = self.get_value_with_interface()
        for name in self.defaults:
            value[name] = value.get(name, self.defaults[name])
        if value['hop_count'] and int(value['hop_count']) == 1: # Default hop count
            # Do not generate default hop count in CLI, it's not only not needed, but also
            # may cause mismatch to the CLI from ASA during audit because the default hop
            # count is not store on ASA.
            value['hop_count'] = ''
        return ' '.join(str(self.asa_gen_template % value).split())

    def get_value_with_interface(self):
        value = super(IPv6Route, self).get_value_with_interface()
        return self.normalize_ipv6_values(value)

    def create_asa_key(self):
        '''Create the the asa key identifies this object, everything except 'metric' make up the key
        @return str
        '''
        return ('ipv6 route %(interface)s %(prefix)s %(gateway)s' %
                self.get_value_with_interface())

    def parse_multi_parameter_cli(self, cli):
        '''Override the default implementation in case the CLI does not match asa_gen_template due to optional
        parameter
        '''
        'Take care of the mandatory parameters'
        result = super(IPv6Route, self).parse_multi_parameter_cli(
            cli,
            alternate_asa_gen_template='ipv6 route %(interface)s %(prefix)s %(gateway)s')
        'Take care of the optional parameters'
        for name, value  in self.defaults.iteritems():
            result[(Type.PARAM, name, '')] = {'state': State.NOCHANGE, 'value': value}
        tokens = cli.split()
        if len(tokens) == 5:
            return result #no optional parameter

        option = tokens[5]
        if option == 'tunneled':
            result[Type.PARAM, 'tunneled', '']['value'] = option
        elif option:
            result[Type.PARAM, 'hop_count', '']['value'] = option
        return result

    def validate_configuration(self):
        faults = []
        value = self.get_value()
        msg = IPv6AddressSubnetValidator().validate(value.get('prefix'))
        if msg:
            faults.append(self.generate_fault(msg, 'prefix'))
        msg = IPv6AddressValidator().validate(value.get('gateway'))
        if msg:
            faults.append(self.generate_fault(msg, 'gateway'))
        return faults

    def is_my_cli(self, cli):
        """
        Override the default implementation in order to handling command with 'tunneled' flag.
        ASA allows the following two CLIs co-exist in the running-config:
             ipv6 route <bla> tunneled
             ipv6 route <bla>
        We need to make 'tunneled' flag as part of the key.
        """
        if not Route.is_my_cli(self, cli):
            return False
        if cli.endswith('tunneled') or self.get_cli().endswith('tunneled'):
            return self.get_cli() == cli
        return True

def static_route_response_parser(response):
    '''
    Ignores expected INFO, WARNING, and error in the response, otherwise returns original.
    '''

    if response:
        msgs_to_ignore = ('INFO:',
                          'WARNING:',
                          'No matching route to delete')
        found_msg_to_ignore = False
        for msg in msgs_to_ignore:
            if msg in response:
                found_msg_to_ignore = True
    return None if response and found_msg_to_ignore else response
