'''
Created on Aug 4, 2015

@author: jeffryp

Copyright (c) 2015 by Cisco Systems, Inc.
All rights reserved.

Classes used for global and interface TCP options configuration.
'''

import re

from base.dmobject import DMObject
from state_type import State, Type
from structured_cli import StructuredCommand

class CommonReset(DMObject):
    '''
    Common reset configuration
    '''

    def __init__(self, ifc_key, asa_key):
        super(CommonReset, self).__init__(ifc_key, asa_key)

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.delta_ifc_key = (Type.PARAM, self.ifc_key, '')
            self.delta_ifc_cfg_value = {
                'state': State.CREATE,
                'value': ('enable' if isinstance(cli, StructuredCommand) and
                          cli.is_no else 'disable')}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] =  self.delta_ifc_cfg_value
        elif self.is_the_same_cli(cli):
            self.set_state(State.NOCHANGE)
        else:
            self.set_state(State.MODIFY)

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.has_ifc_delta_cfg():
            state = self.get_state()
            if self.get_value() == 'disable':
                if state in (State.CREATE, State.MODIFY):
                    state = State.DESTROY
                elif state == State.DESTROY:
                    state = State.CREATE
            if state in (State.CREATE, State.MODIFY):
                self.generate_cli(asa_cfg_list, self.get_cli())
            elif state == State.DESTROY:
                self.generate_cli(asa_cfg_list, 'no ' + self.get_cli())

    def is_my_cli(self, cli):
        if isinstance(cli, StructuredCommand):
            cli = cli.command
        return cli.strip() == self.get_cli()

    def is_the_same_cli(self, cli):
        cli_value = ('disable' if isinstance(cli, StructuredCommand) and
                     cli.is_no else 'enable')
        return self.get_value() == cli_value

class GlobalTCPOptions(DMObject):
    '''
    Global TCP options configuration
    '''

    def __init__(self):
        super(GlobalTCPOptions, self).__init__(GlobalTCPOptions.__name__)
        self.register_child(GlobalReset('outbound_reset',
                                        'service resetoutbound'))
        self.register_child(GlobalReset('inbound_reset',
                                        'service resetinbound'))

class GlobalReset(CommonReset):
    '''
    Global reset configuration
    '''

    def __init__(self, ifc_key, asa_key):
        super(GlobalReset, self).__init__(ifc_key, asa_key)

    def get_cli(self):
        return self.asa_key

class InterfaceReset(CommonReset):
    '''
    Interface reset configuration
    '''

    def __init__(self, ifc_key, asa_key):
        super(InterfaceReset, self).__init__(ifc_key, asa_key)
        self.response_parser = (
            lambda resp:
            None if re.search('is already (disabled|enabled) for all interfaces', resp)
            else resp)

    def get_cli(self):
        return self.asa_key + ' ' + self.parent.nameif

class TCPOptions(DMObject):
    '''
    Interface TCP options configuration
    '''

    def __init__(self):
        super(TCPOptions, self).__init__(TCPOptions.__name__)
        self.register_child(InterfaceReset('outbound_reset',
                                           'service resetoutbound interface'))
        self.register_child(InterfaceReset('inbound_reset',
                                           'service resetinbound interface'))

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        # The ASA will remove the resets when the interface is removed
        if self.parent.get_state() != State.DESTROY:
            super(TCPOptions, self).ifc2asa(no_asa_cfg_stack, asa_cfg_list)

    @property
    def nameif(self):
        return self.parent.nameif
