/*++

Copyright (c) 1995  Digital Equipment Corporation

Module Name:

    arc.h

Abstract:

    This module defines the ARC system firmware interface.

Author:

    Dave Richards [DEC] 25-Aug-95
   

Revision History:

    Hy Pham       [DEC] 24-Oct-96

--*/

#ifndef _ARC_H_
#define _ARC_H_

#include <types.h>


//
// Kernel Segment 0
//

#define KSEG0           0x80000000      // Superpage base


//
// Software Loading and Execution Functions
//

LONG
Load(
    CHAR *Path,
    ULONG TopAddr,
    ULONG *ExecAddr,
    ULONG *LowAddr
    );


LONG
Invoke(
    ULONG ExecAddr,
    ULONG StackAddr,
    ULONG Argc,
    CHAR *Argv[],
    CHAR *Envp[]
    );

LONG
Execute(
    CHAR *Path,
    ULONG Argc,
    CHAR *Argv[],
    CHAR *Envp[]
    );

//
// Program Termination Functions
//

void
Halt(
    void
    );

void
PowerDown(
    void
    );

void
Restart(
    void
    );

void
Reboot(
    void
    );

void
EnterInteractiveMode(
    void
    );

//
// Configuration Query Functions
//

typedef enum{
    SystemClass,
    ProcessorClass,
    CacheClass,
    AdapterClass,
    ControllerClass,
    PeripheralClass,
    MemoryClass,
    MaximumClass
} CONFIGURATION_CLASS;


#ifndef _NTDDK_
 
typedef enum {
    ArcSystem,
    CentralProcessor,
    FloatingPointProcessor,
    PrimaryIcache,
    PrimaryDcache,
    SecondaryIcache,
    SecondaryDcache,
    SecondaryCache,
    EisaAdapter,
    TcAdapter,
    ScsiAdapter,
    DtiAdapter,
    MultiFunctionAdapter,
    DiskController,
    TapeController,
    CdromController,
    WormController,
    SerialController,
    NetworkController,
    DisplayController,
    ParallelController,
    PointerController,
    KeyboardController,
    AudioController,
    OtherController,
    DiskPeripheral,
    FloppyDiskPeripheral,
    TapePeripheral,
    ModemPeripheral,
    MonitorPeripheral,
    PrinterPeripheral,
    PointerPeripheral,
    KeyboardPeripheral,
    TerminalPeripheral,
    OtherPeripheral,
    LinePeripheral,
    NetworkPeripheral,
    SystemMemory,
    MaximumType
} CONFIGURATION_TYPE,*PCONFIGURATION_TYPE;

#endif // _NTDDK_

typedef enum {
    Failed = 1,
    ReadOnly = 2,
    Removable = 4,
    ConsoleIn = 8,
    ConsoleOut = 16,
    Input = 32,
    Output = 64
} IDENTIFIERFLAG;


typedef struct {
    CONFIGURATION_CLASS Class;
    CONFIGURATION_TYPE Type;
    IDENTIFIERFLAG Flags;
    USHORT Version;
    USHORT Revision;
    ULONG Key;
    ULONG AffinityMask;
    ULONG ConfigurationDataLength;
    ULONG IdentifierLength;
    CHAR *Identifier;
} CONFIGURATION_COMPONENT,*PCONFIGURATION_COMPONENT;



CONFIGURATION_COMPONENT *
GetPeer(
    CONFIGURATION_COMPONENT *Current
    );

CONFIGURATION_COMPONENT *
GetChild(
    CONFIGURATION_COMPONENT *Current
    );

CONFIGURATION_COMPONENT *
GetParent(
    CONFIGURATION_COMPONENT *Current
    );

CONFIGURATION_COMPONENT *
AddChild(
    CONFIGURATION_COMPONENT *Current,
    CONFIGURATION_COMPONENT *Template,
    void *ConfigurationData
    );

LONG
DeleteComponent(
    CONFIGURATION_COMPONENT *ComponentToDelete
    );

CONFIGURATION_COMPONENT *
GetComponent(
    CHAR *Path
    );

LONG
GetConfigurationData(
    void *ConfigurationData,
    CONFIGURATION_COMPONENT *Component
    );

LONG
SaveConfiguration(
    void
    );

//
// System ID Query Functions
//


typedef struct {
    CHAR VendorId[8];
    CHAR ProductId[8];
} SYSTEM_ID;


SYSTEM_ID *
GetSystemId(
    void
    );

//
// Memory Query Functions
//

typedef enum {
    MemoryExceptionBlock,
    MemorySystemBlock,
    MemoryFree,
    MemoryBad,
    MemoryLoadedProgram,
    MemoryFirmwareTemporary,
    MemoryFirmwarePermanent,
    MemoryFreeContiguous,
    MemorySpecialMemory,
    MemoryMaximum
} MEMORY_TYPE;


typedef struct {
    MEMORY_TYPE Type;
    ULONG BasePage;
    ULONG PageCount;
} MEMORY_DESCRIPTOR;


MEMORY_DESCRIPTOR *
GetMemoryDescriptor(
    MEMORY_DESCRIPTOR *Current
    );

//
// Time Functions
//

#ifndef _NTDDK_

typedef struct _TIME_FIELDS{
    USHORT Year;        // range [1601...]
    USHORT Month;       // range [1..12]
    USHORT Day;         // range [1..31]
    USHORT Hour;        // range [0..23]
    USHORT Minute;      // range [0..59]
    USHORT Second;      // range [0..59]
    USHORT Milliseconds;// range [0..999]
    USHORT Weekday;     // range [0..6] == [Sunday..Saturday]
} TIME_FIELDS,*PTIME_FIELDS;

#endif  // _NTDDK_

TIME_FIELDS *
GetTime(
    void
    );

ULONG
GetRelativeTime(
    void
    );

//
// Input/Output Functions
//

#define StandardIn      0                     // = ARC_CONSOLE_INPUT
#define StandardOut     1                     // = ARC_CONSOLE_OUTPUT

#define ReadOnlyFile    1
#define HiddenFile      2
#define SystemFile      4
#define ArchiveFile     8
#define DirectoryFile   16
#define DeleteFile      32

typedef struct {
    ULONG FileNameLength;
    UCHAR FileAttribute;
    CHAR FileName[32];
} DIRECTORY_ENTRY;

typedef enum {
    OpenReadOnly,
    OpenWriteOnly,
    OpenReadWrite,
    CreateWriteOnly,
    CreateReadWrite,
    SupersedeWriteOnly,
    SupersedeReadWrite,
    OpenDirectory,
    CreateDirectory,
    OpenMaximumMode
} OPEN_MODE;

typedef enum {
    SeekAbsolute,
    SeekRelative,
    SeekMaximum
} SEEK_MODE;


typedef enum {
    MountLoadMedia,
    MountUnloadMedia,
    MountMaximum
} MOUNT_OPERATION;

typedef struct {
    LARGE_INTEGER StartingAddress;
    LARGE_INTEGER EndingAddress;
    LARGE_INTEGER CurrentAddress;
    CONFIGURATION_TYPE Type;
    ULONG FileNameLength;
    UCHAR Attributes;
    CHAR FileName[32];
} FILE_INFORMATION;

LONG
GetDirectoryEntry(
    ULONG FileId,
    DIRECTORY_ENTRY *Buffer,
    ULONG Length,
    ULONG *Count
    );

LONG
Open(
    const CHAR *Path,
    OPEN_MODE OpenMode,
    ULONG *FileId
    );

LONG
Close(
    ULONG FileId
    );

LONG
Read(
    ULONG FileId,
    void *Buffer,
    ULONG N,
    ULONG *Count
    );

LONG
GetReadStatus(
    ULONG FileId
    );

LONG
Write(
    ULONG FileId,
    void const *Buffer,
    ULONG N,
    ULONG *Count
    );

LONG
Seek(
    ULONG FileId,
    LARGE_INTEGER *Position,
    SEEK_MODE SeekMode
    );

LONG
Mount(
    CHAR *Path,
    MOUNT_OPERATION Operation
    );

LONG
GetFileInformation(
    ULONG FileId,
    FILE_INFORMATION *Information
    );

LONG
SetFileInformation(
    ULONG FileId,
    ULONG AttributeFlags,
    ULONG AttributeMask
    );

//
// Environment Functions
//

CHAR *
GetEnvironmentVariable(
    CHAR *Name
    );

LONG
SetEnvironmentVariable(
    CHAR *Name,
    CHAR *Value
    );

//
// Cache Function
//

void
FlushAllCaches (
    void
    );

//
// Test Unicode Function
//

LONG
TestUnicodeCharacter (
    ULONG FileId,
    WCHAR UnicodeCharacter
    );

//
// Get Display Status Function
//

typedef struct {
    USHORT CursorXPosition;
    USHORT CursorYPosition;
    USHORT CursorMaxXPosition;
    USHORT CursorMaxYPosition;
    UCHAR ForegroundColor;
    UCHAR BackgroundColor;
    BOOLEAN HighIntensity;
    BOOLEAN Underscored;
    BOOLEAN ReverseVideo;
} ARC_DISPLAY_STATUS;

ARC_DISPLAY_STATUS *
GetDisplayStatus (
    ULONG FileId
    );



#endif // _ARC_H_
