// Compiler notes

// CURRENT:

// Attempting to fix shadow speckle on coincident surfaces; attempting to clip away
// non-origin surfaces on shadow rays over short distances.

// GLOBAL FIXUPS:
// Currently, the interior surfaces of unions show up in differences where they shouldn't.
// This is hard to fix - is a parenting solution the answer?

// Notes:

// Verbose [1]
// 0 -> NO OUTPUT  (EXCEPT ERRORS)
// 1 -> INTERESTING / STRUCTURAL OUTPUT (WARNINGS)
// 2 -> MOST STUFF
// 3 -> RIDICULOUS STUFF
// 4 -> EVERYTHING!!!!

// Note that non-zero verbose settings can severely impare the parsing performance of the
// program for complex scenes (it will spew forth megabytes of text!)

// KNOWN BUG: In Win95, it can max out the 16Mb of DPMI memory available. So,
// if you get "Virtual memory exceeded by `new'" then alter the properties
// of the program (or DOS box it runs from) to type in 65535 DPMI memory,
// If that fails: Check you have a few 10s of Mb free on the hard drive, or
// run in DOS with CWSDPMI

// BIG PROBLEM: The parser chews up vast amounts of memory making 20-byte
// tokens out of files. It needs a total overhaul, and probably the
// abandonment of the entire system

// Look into hardcoding or #including a series of #declares to enhance
// compatibility, e.g. x = < 1, 0, 0 >, y, z, on, off, and to enhance ease of
// use e.g. map_Planar = 0, map_Spherical = 1

// Hmm... Perhaps the cameras should be treated like light_sources to allow
// transformations and nesting...? Yes, this allows their attachment to objects, a 
// desireable effect.

// Make no_shadow recursive to higher levels

// AA looks suspect - why? / fix

// Make textures work in frame {}

// Look up the total spectral reflection properties of the average surface

// A lesson from POV - the camera (after look_at) is wrong-handed

/* Add these bits...

    ? Jitter=bool
    ? Jitter_Amount=n.n
    ?? camera focal blur
    ? height_field  Can do as a group of unions..., but will take forever
    ? torus         Can I write a 4th order iterative solver?
    ?? disc         Why ?
    ? mesh          Just use to mask a union (maybe)
    ?? area lights
    ? spotlight     Make it a texture!
    ?? clipped_by
    ? IMPLEMENT REFRACTION -> ?? caustics
    ?? finish(fade_distance, fade_power)

    Display=bool
    Preview_Start_Size=n
    Preview_End_Size=n

    // By default, on, off, on , on

    Relativity=bool     // Gives all effects
    Observation=bool    // Gives observation, not sight

*/

// #define PC
// #define DISPLAY
#define DEBUG

#define PI 3.141592654

//
//  Definitions
//

#define SG_TRIANGLE          1
#define SG_SMOOTH_TRIANGLE   2
#define SG_SPHERE            3
#define SG_BOX               4
#define SG_CYLINDER          5
#define SG_CONE              6
#define SG_PLANE             7
#define SG_QUADRIC           8
#define SG_LIGHT_SOURCE      9  // Not a true object, but a convenient
                                // fiction when parsing
#define CSG_UNION           -1
#define CSG_MERGE           -2
#define CSG_DIFFERENCE      -3
#define CSG_INTERSECTION    -4

#define MT_NONE        -2
#define MT_COLOUR      -1
#define MT_PLANAR       0
#define MT_SPHERICAL    1
#define MT_CYLINDRICAL  2
//                      3
//                      4
#define MT_TOROIDAL     5
#define MT_DISK         6

// Wavelengths...
#define lUltraViolet    340.0
#define lBlue           460.0
#define lGreen          520.0
#define lRed            700.0
#define lInfraRed       1000.0

//
//  Include files
//

#include <stdlib.h>
#include <math.h>
#include <stdio.h>
//#include <dos.h>
//#include <conio.h>
//#include <sys\nearptr.h>
//#include <mem.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include "keywords.h"

//
//  Prototypes
//

// Classes

class Vector;
class Matrix;
class Bitmap;
    class iColour;
    class fColour;
class Event;
class Ray;
class Token;
class Declared;

class Scene;

    class Frame;
        class Camera;
        class Light_Source;

        class Object;
            class SG_triangle;
            class SG_smooth_triangle;
            class SG_sphere;
            class SG_box;
            class SG_cylinder;
            class SG_cone;
            class SG_plane;
            class SG_quadric;
            class CSG_union;
            class CSG_merge;
            class CSG_difference;
            class CSG_intersection;

            class Bounding_Sphere;

        class Texture;
            class Pigment;
            class Finish;
            class Normal;

// Functions

void PrintToken(Token *T);

//
//  Classes
//

class Vector {
    // Vector in 3 dimensions
    // 
    // All common operations supported, use / for cross product

    public:
    double x, y, z;
    Vector() {};
    Vector(double _x, double _y, double _z) { x = _x; y = _y; z = _z;};
    inline Vector  operator +  (Vector v);
    inline Vector  operator -  (Vector v);
    inline double   operator *  (Vector v); // Dot product
    inline Vector  operator /  (Vector v); // Cross product
    inline void    operator += (Vector v);
    inline void    operator -= (Vector v);
    inline void    operator /= (Vector v); // Cross product
    inline Vector  operator *  (double f);
    inline Vector  operator /  (double f);
    inline void    operator *= (double f);
    inline void    operator /= (double f);
    inline int     operator == (Vector v);
    inline int     operator != (Vector v);
    inline double   nrm(void);
    inline double   sqn(void);
    inline void    Normalise(double f);
           Vector  LorentzTransform(Vector v, Vector u);
           Token  *Load(Token *T);
    };

class Matrix {
    // 3d Transformation matrix
    //
    // This is not a true matrix, rather it is a 3x3 with an offset vector
    // tacked on. These matrices multiply like a 4x4 with
    // a41, a42, a43 = 0, a44 = 1

    public:
    // Not in obvious order, to allow column equivalence to 4 vectors (not
    // four-vectors !)
    double a11, a21, a31;
    double a12, a22, a32;
    double a13, a23, a33;
    double a14, a24, a34;
    Matrix() {};
    Matrix(	double _a11, double _a21, double _a31,
			double _a12, double _a22, double _a32,
			double _a13, double _a23, double _a33,
			double _a14, double _a24, double _a34
			) {
			a11 = _a11;
			a12 = _a12;
			a13 = _a13;
			a14 = _a14;
			a21 = _a21;
			a22 = _a22;
			a23 = _a23;
			a24 = _a24;
			a31 = _a31;
			a32 = _a32;
			a33 = _a33;
			a34 = _a34;
			};
    Vector TransformLocation(Vector &v);
    Vector TransformDirection(Vector v);
    Vector TransformNormal(Vector v);
    Matrix operator + (Matrix &m);
    Matrix operator - (Matrix &m);
    Matrix operator * (Matrix &m);
    void   operator += (Matrix &m);
    void   operator -= (Matrix &m);
    void   operator *= (Matrix &m);
    Matrix invert(void);
    double det(void);
    Token *Load(Token *T);
    };

class iColour {
    // Integer colour representation (used for storage)
    public:
    unsigned char b, g, r, a;
    inline fColour Convert(void);
    };

class fColour {
    // doubling point colour representation (used for calculations)
    public:
    double b, g, r, f, t;
    fColour() {};
    fColour(double _b, double _g, double _r, double _f, double _t): b(_b), g(_g), r(_r), f(_f), t(_t) {};;
    inline iColour Convert();
    inline void Clean();
    fColour Doppler(double d);
    fColour Doppler2(double d);

    inline fColour operator +  (fColour c);
    inline fColour operator -  (fColour c);
    inline fColour operator *  (fColour c);
    inline void    operator += (fColour c);
    inline void    operator -= (fColour c);
    inline void    operator *= (fColour c);

    inline fColour operator *  (double a);
    inline void    operator *= (double a);

    Token *Load(Token *);
    };

class Bitmap {
    // Bitmaps in B-G-R-A format, as in 32 bit TARGA
    public:
    unsigned long uMax, vMax;
    iColour *Pixel;
    Bitmap *Next; // Useful to avoid repetition...
    char *FileName;
    void Create(unsigned long u, unsigned long v);
    void LoadTGA(char *Name);
    void SaveTGA(char *Name);
    void Combine(unsigned long u1, unsigned long v1,
                 unsigned long u2, unsigned long v2,
                 unsigned long u3, unsigned long v3,
                 unsigned long u4, unsigned long v4,
                 Bitmap b);
    void Set(unsigned long u, unsigned v, iColour c);
    fColour GetPixel(double u, double v, long interpolate);
    void GetBump(double &u, double &v, long interpolate);
    };

class Event {
    public:
    Event() {};
    Event(double _x, double _y, double _z, double _t): x(_x, _y, _z), t(_t) {};
    Vector x;
    double t;
    Event LorentzTransform(Vector &v, Vector &u);
    };

class Ray {
    public:
    Event Origin;               // Both for the current frame
    Vector Direction;           
    double doppler, intensity;   // Both relative to camera frame,
                                // which allows adding of proper colours

    Ray LorentzTransform(Vector &v, Vector &u);

    };

class Bounding_Sphere {
    public:
    Vector o;
    double r;
    void Union(Bounding_Sphere b);
    void Intersection(Bounding_Sphere b);
    };

class Frame_Intersection {
    public:
    Frame *From_Frame;
    Object *From_Object;
    Ray From_Ray;
    Event Hit;
    };

class Object_Intersection {
    public:
    Object *From_Object;
    double t;
    };

class Object {
    public:
    long Object_Type;
    void *Geometry;
    Matrix Geometry_Transformation;
    Texture *Surface;
    Matrix Texture_Transformation;
    Bounding_Sphere bounded_by;
    long no_shadow, inverse;

    Token *Load(Token *T);
    void Paint(Texture *T);
    void Transform(Matrix &Transformation);

    void Show(void);

    Bounding_Sphere Bound(Object **L_S_L);

    Object_Intersection Intersection(Ray &Initial_Ray, double t_min, double t_max, Object *Ray_Parent);

    double Interior(Vector o);

    Vector Normal(Vector o);

    fColour GetPigment(Event o);
    Vector GetNormal(Event o);

    };

class SG_triangle {
    public:
    Vector Normal;  // Normal in frame coordinates
    };

class SG_smooth_triangle {
    public:
    Vector Normal, Normal_1, Normal_2, Normal_3;    // Normals in frame coordinates
    double u, v;                             // Useful optomisation
    };

class SG_sphere {
    public:
    };

class SG_box {
    public:
    Vector Normal;
    };

class SG_cylinder {
    public:
    long open;
    };

class SG_cone {
    public:
    double Radius_1;
    double Radius_2;
    long open;
    };

class SG_plane {
    public:
    Vector Normal;
    };

class SG_quadric {
    public:
    double cxx, cyy, czz, cxy, cxz, cyz, cx, cy, cz, c;
    };

class CSG_union {
    public:
    long Number_Objects;
    Object **Object_List;
    };

class CSG_merge {
    public:
    long Number_Objects;
    Object **Object_List;
    };

class CSG_difference {
    public:
    long Number_Objects;
    Object **Object_List;
    };

class CSG_intersection {
    public:
    long Number_Objects;
    Object **Object_List;
    };

class Light_Source {
    public:
    double fade_distance;
    double fade_power;
	long LensFlare;
    };

class Frame {
    public:
    Vector Velocity;
    long Number_Objects;
    Object **Object_List;
    long Number_Light_Sources;
    Object **Light_Source_List;

    Token *Load(Token *T, Vector v);

    void Show(void);

    void Bound(void);

    Frame_Intersection Intersection(Ray &Initial_Ray, Object *Ray_Parent);

    };

class Token {
    public:
    long ID;
    long Line;
    char *FileName;
    void *Data;
    Token *Next;
    };

class Declared {
    public:
    char *DeclaredIdentifier;
    Token *FirstToken;
    Declared *Next;
    };

class Scene {
    public:
    long Number_Cameras;
    Camera **Camera_List;
    long Number_Frames;
    Frame **Frame_List;
    Bitmap LensFlare;
	Bitmap HUD;

    long uImage, vImage;
    char *OutFile;



    void Load(char *FileName);
    void Raytrace(void);
    fColour GetColour(Frame_Intersection &Best_Intersection, long Recursion);
    Frame_Intersection CastRay(Ray &Initial_Ray, Vector &v, Object *Ray_Parent);
    fColour Antialias(fColour c00, fColour c02, fColour c20, fColour c22, double u_0, double v0_, double u_2, double v2_, Camera *C, long Recursion);
    fColour TraceRay(double u, double v, Camera *C);
    };

class Pigment {
    public:
    long map_type, interpolate;
    fColour colour;
    Bitmap *bitmap;
    double frequency_u;
    double frequency_v;
    double phase_u;
    double phase_v;
    double drift_u;
    double drift_v;
    double doppler;
    double intensity;
    long once;
    Matrix Transformation;

    Token *Load(Token *T);

    };

class Normal {
    public:
    long map_type, interpolate;
    Bitmap *bitmap;
    double frequency_u;
    double frequency_v;
    double phase_u;
    double phase_v;
    double drift_u;
    double drift_v;
    double bump_size;
    long once;
    Matrix Transformation;

    Token *Load(Token *T);

    };

class Finish {
    public:
    fColour ambient;
    fColour reflection;
    double   diffuse,
            brilliance,
            specular,
            roughness,
            refraction,
            ior;
    long    metallic;

    Token *Load(Token *T);

    };

class Texture {
    public:

    // Doesn't need a transformation - use pigment, finish instead

    Pigment P;
    Normal  N;
    Finish  F;

    Token *Load(Token *T);

    fColour GetPigment(Event o);

    };


class Camera {
    public:
    long Type;
    Event location;
    Vector look_at, right, up, direction, sky, Velocity;
    double angle;

    Token *Load(Token *T, Vector v);

    void Show(void);

    Ray GetRay(double u, double v);
    void GetCoordinates(Ray r, double &du, double &dv);

    };

class global_settings {
    public:
    double  Verbose,
            Antialias,
            Sampling_Method,
            Antialias_Threshold,
            Antialias_Depth,
            adc_bailout,
            assumed_gamma,
            max_trace_level,
            Relativity,
            Observation,
            Doppler,
            Intensity,
            Display,
			Window_l,
			Window_r,
			Window_t,
			Window_b,
			Window_Doppler,
			Window_Intensity,
			NW_Doppler,
			NW_Intensity,
			HUD;
    long    Tokens,
			MemoryUsed,
			PrimitivesUsed,
			Continue;

    };


#ifdef DISPLAY

// This code originally used for low-lwvel acess to SVGA on IBM PCs. Now extended to OpenGL
// This is VERY primitive code - the window will not handle propery! Don't try to use on
// X-Windows systems yet - probably won't work!

#ifdef PC
	#include "windows.h"
#endif

#include "gl\gl.h"
#include "gl\glaux.h"

void SetMode(long w, long h) {
	auxInitDisplayMode(AUX_SINGLE | AUX_RGBA);
	auxInitPosition(0, 0, w, h);
	auxInitWindow("Backlight Preview");

	glOrtho(0.0f, (float) w, 0.0f, (float) h, 1.0f, -1.0f);

	glClearColor(0.5f, 0.5f, 0.5f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT);

	glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
	glBegin(GL_LINES);
		glVertex2i(0, 0);
		glVertex2i(w, h);
		glVertex2i(w, 0);
		glVertex2i(0, h);
	glEnd();

	glFlush();

	glBegin(GL_POINTS);

	}

void ResetMode(long w, long h) {
	glEnd();
	
	glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
	glBegin(GL_LINES);
		glVertex2i(0, 0);
		glVertex2i(w, h);
		glVertex2i(w, 0);
		glVertex2i(0, h);
	glEnd();

	glFlush();

	glBegin(GL_POINTS);

	}

void SetPixel(long u, long v, iColour c) {
	glColor3ub(c.r, c.g, c.b);
	//glBegin(GL_POINTS);
		glVertex2i(u, v);
	//glEnd();
	//glFlush();
    }

#endif

//
//  Global Variables
//

// ?  Necessary

Frame **Current_Frame;
Camera **Current_Camera;

Texture *Default_Texture;
Bitmap *First_Bitmap;

global_settings Globals;
Texture SkySphereSurface;
Vector SkySphereVelocity;

//
//  Functions
//

// Remedy an unfortunate omission of ANSI C/C++

double fmax(double a, double b) {
    if(a >= b) {
        return a;
        }
    else {
        return b;
        }
    }

//
//  Vector functions
//

inline Vector Vector::operator + (Vector v) {
    Vector t;
    t.x = x + v.x;
    t.y = y + v.y;
    t.z = z + v.z;
    return t;
    };

inline Vector Vector::operator - (Vector v) {
    Vector t;
    t.x = x - v.x;
    t.y = y - v.y;
    t.z = z - v.z;
    return t;
    };

inline double Vector::operator * (Vector v) {
    return x*v.x + y*v.y + z*v.z;
    }

inline Vector Vector::operator / (Vector v) {
    Vector t;
    t.x = y*v.z - z*v.y;
    t.y = z*v.x - x*v.z;
    t.z = x*v.y - y*v.x;
    return t;
    }

inline void Vector::operator += (Vector v) {
    x += v.x;
    y += v.y;    
    z += v.z;
    };

inline void Vector::operator -= (Vector v) {
    x -= v.x;
    y -= v.y;    
    z -= v.z;
    };

inline void Vector::operator /= (Vector v) {
    Vector t;
    t.x = y*v.z - z*v.y;
    t.y = z*v.x - x*v.z;
    t.z = x*v.y - y*v.x;
    *this = t;
    }

inline Vector Vector::operator * (double f) {
    Vector t;
    t.x = x*f;
    t.y = y*f;
    t.z = z*f;
    return t;
    }

inline Vector Vector::operator / (double f) {
    Vector t;

    if(f == 0) {
        printf("Vector::operator / (double) : Attepted divide by zero\n");
        exit(EXIT_FAILURE);
        }

    t.x = x/f;
    t.y = y/f;
    t.z = z/f;
    return t;
    }

inline void Vector::operator *= (double f) {
    x *= f;
    y *= f;
    z *= f;
    }

inline void Vector::operator /= (double f) {

    if(f == 0) {
        printf("Vector::operator /= (double) : Attepted divide by zero\n");
        exit(EXIT_FAILURE);
        }

    x /= f;
    y /= f;
    z /= f;
    }

inline int Vector::operator == (Vector v) {
    if(x == v.x) {
        if(y == v.y) {
            if(z == v.z) {
                return 1;
                }
            }
        }
    return 0;
    }

inline int Vector::operator != (Vector v) {
    if(x != v.x) {
        return 1;
        }
    if(y != v.y) {
        return 1;
        }
    if(z != v.z) {
        return 1;
        }
    return 0;
    }

inline double Vector::nrm(void) {
    return sqrt(x*x + y*y + z*z);
    }

inline double Vector::sqn(void) {
    return x*x + y*y + z*z;
    }

inline void Vector::Normalise(double f) {

    if((x*x + y*y + z*z) == 0) {
        printf("Vector::Normalise(double) : Attempted to normalise zero vector\n");
        exit(EXIT_FAILURE);
        }

    f /= sqrt(x*x + y*y + z*z);
    x *= f;
    y *= f;
    z *= f;
    }

inline double gamma(Vector v) {

    if(v.sqn() >= 1) {
        printf("gamma(Vector) : Attepted to calculate Lorentz factor for illegal velocity\n");
        exit(EXIT_FAILURE);
        }

    return 1.0/sqrt(1 - v.sqn());
    }

Vector Vector::LorentzTransform(Vector v, Vector u) {

    // FROM frame with velocity v (relative to base reference frame)
    // TO frame with velocity u (relative to base reference frame)

    Vector e;
    Vector nu, nv;
    double t, gu, gv;

    if(!Globals.Relativity) {

        // Newtonian

        return ((*this) + v - u);
        }


    if((v.sqn() > 1) || (u.sqn() > 1)) {
        printf("Vector::LorentzTransform(Vector, Vector) : Illegal frame velocity\n");
        exit(EXIT_FAILURE);
        }

    if(u.nrm() != 0) {
        nu = u / u.nrm();
        }
    else {
        nu = u;
        }

    if(v.nrm() != 0) {
        nv = v / v.nrm();
        }
    else {
        nv = v;
        }

    gu = gamma(u);
    gv = gamma(v);

    // Derived in obvious fashion; calculate x|| = x.nv, and transform this
    // with t in the traditional fashion, then reconstruct x'. Applied twice
    // this gives a transformation between general frames. Uses -v as we
    // transform FROM v-frame. Uses event 1, v on velocity to transform
    // velocity and then renormalises time.

    t = gu*(gv*(1-(u*v)+(v*(*this)))+(u*nv)*(nv*(*this))*(1-gv) - (u*(*this)));
    e = (*this)+nv*(gv-1)*(nv*(*this))+v*gv+nu*(gu-1)*((nu*(*this))+(nu*v)*gv+(nu*nv)*(nv*(*this))*(gv-1))-u*gu*gv*((v*(*this))+1);
    e /= t;

    return e;
    }

Token * Vector::Load(Token *T) {

    switch(T->ID) {

        case ID_DOUBLE:  // A single value - so promote to vector
            x = *((double *) T->Data);
            y = *((double *) T->Data);
            z = *((double *) T->Data);
            break;

        case ID_LEFT_ANGLE: // A vector of the form < #, #, # >
            T = T->Next;
            if(T->ID == ID_DOUBLE) {
                x = *((double *) T->Data);
                }
            else {
                PrintToken(T);
                printf("found, number expected\n");
                exit(EXIT_FAILURE);
                }
            T = T->Next;
            if(T->ID != ID_COMMA) {
                PrintToken(T);
                printf("found, comma expected\n");
                exit(EXIT_FAILURE);
                }
            T = T->Next;
            if(T->ID == ID_DOUBLE) {
                y = *((double *) T->Data);
                }
            else {
                PrintToken(T);
                printf("found, number expected\n");
                exit(EXIT_FAILURE);
                }
            T = T->Next;
            if(T->ID != ID_COMMA) {
                PrintToken(T);
                printf("found, comma expected\n");
                exit(EXIT_FAILURE);
                }
            T = T->Next;
            if(T->ID == ID_DOUBLE) {
                z = *((double *) T->Data);
                }
            else {
                PrintToken(T);
                printf("found, number expected\n");
                exit(EXIT_FAILURE);
                }
            T = T->Next;
            if(T->ID != ID_RIGHT_ANGLE) {
                PrintToken(T);
                printf("found, > expected\n");
                exit(EXIT_FAILURE);
                }
            break;
        default:
            PrintToken(T);
            printf("found, vector expression expected\n");
            exit(EXIT_FAILURE);
            break;
        }
    return T;
    }


//
//  Matrix functions
//

Vector Matrix::TransformLocation(Vector &v) {
    Vector t;
    t.x = a11*v.x + a12*v.y + a13*v.z + a14;
    t.y = a21*v.x + a22*v.y + a23*v.z + a24;
    t.z = a31*v.x + a32*v.y + a33*v.z + a34;
    return t;
    }

Vector Matrix::TransformDirection(Vector v) {
    Vector t;
    t.x = a11*v.x + a12*v.y + a13*v.z;
    t.y = a21*v.x + a22*v.y + a23*v.z;
    t.z = a31*v.x + a32*v.y + a33*v.z;
    return t;
    }

Vector Matrix::TransformNormal(Vector v) {
    Vector t;
    t.x = (a22*a33 - a23*a32)*v.x + (a31*a23 - a33*a21)*v.y + (a21*a32 - a22*a31)*v.z;
    t.y = (a32*a13 - a33*a12)*v.x + (a11*a33 - a13*a31)*v.y + (a31*a12 - a32*a11)*v.z;
    t.z = (a12*a23 - a13*a22)*v.x + (a21*a13 - a23*a11)*v.y + (a11*a22 - a12*a21)*v.z;
    return t;
    }


Matrix Matrix::operator + (Matrix &m) {
    Matrix t;
    t.a11 = a11 + m.a11;
    t.a12 = a12 + m.a12;
    t.a13 = a13 + m.a13;
    t.a14 = a14 + m.a14;
    t.a21 = a21 + m.a21;
    t.a22 = a22 + m.a22;
    t.a23 = a23 + m.a23;
    t.a24 = a24 + m.a24;
    t.a31 = a31 + m.a31;
    t.a32 = a32 + m.a32;
    t.a33 = a33 + m.a33;
    t.a34 = a34 + m.a34;
    return t;
    }

Matrix Matrix::operator - (Matrix &m) {
    Matrix t;
    t.a11 = a11 - m.a11;
    t.a12 = a12 - m.a12;
    t.a13 = a13 - m.a13;
    t.a14 = a14 - m.a14;
    t.a21 = a21 - m.a21;
    t.a22 = a22 - m.a22;
    t.a23 = a23 - m.a23;
    t.a24 = a24 - m.a24;
    t.a31 = a31 - m.a31;
    t.a32 = a32 - m.a32;
    t.a33 = a33 - m.a33;
    t.a34 = a34 - m.a34;
    return t;
    }

void Matrix::operator += (Matrix &m) {
    a11 += m.a11;
    a12 += m.a12;
    a13 += m.a13;
    a14 += m.a14;
    a21 += m.a21;
    a22 += m.a22;
    a23 += m.a23;
    a24 += m.a24;
    a31 += m.a31;
    a32 += m.a32;
    a33 += m.a33;
    a34 += m.a34;
    }

void Matrix::operator -= (Matrix &m) {
    a11 -= m.a11;
    a12 -= m.a12;
    a13 -= m.a13;
    a14 -= m.a14;
    a21 -= m.a21;
    a22 -= m.a22;
    a23 -= m.a23;
    a24 -= m.a24;
    a31 -= m.a31;
    a32 -= m.a32;
    a33 -= m.a33;
    a34 -= m.a34;
    }

Matrix Matrix::operator * (Matrix &m) {
    Matrix t;
    t.a11 = a11*m.a11 + a12*m.a21 + a13*m.a31;
    t.a12 = a11*m.a12 + a12*m.a22 + a13*m.a32;
    t.a13 = a11*m.a13 + a12*m.a23 + a13*m.a33;
    t.a14 = a11*m.a14 + a12*m.a24 + a13*m.a34 + a14;
    t.a21 = a21*m.a11 + a22*m.a21 + a23*m.a31;
    t.a22 = a21*m.a12 + a22*m.a22 + a23*m.a32;
    t.a23 = a21*m.a13 + a22*m.a23 + a23*m.a33;
    t.a24 = a21*m.a14 + a22*m.a24 + a23*m.a34 + a24;
    t.a31 = a31*m.a11 + a32*m.a21 + a33*m.a31;
    t.a32 = a31*m.a12 + a32*m.a22 + a33*m.a32;
    t.a33 = a31*m.a13 + a32*m.a23 + a33*m.a33;
    t.a34 = a31*m.a14 + a32*m.a24 + a33*m.a34 + a34;

    return t;

    }

void Matrix::operator *= (Matrix &m) {
    Matrix t;
    t.a11 = a11*m.a11 + a12*m.a21 + a13*m.a31;
    t.a12 = a11*m.a12 + a12*m.a22 + a13*m.a32;
    t.a13 = a11*m.a13 + a12*m.a23 + a13*m.a33;
    t.a14 = a11*m.a14 + a12*m.a24 + a13*m.a34 + a14;
    t.a21 = a21*m.a11 + a22*m.a21 + a23*m.a31;
    t.a22 = a21*m.a12 + a22*m.a22 + a23*m.a32;
    t.a23 = a21*m.a13 + a22*m.a23 + a23*m.a33;
    t.a24 = a21*m.a14 + a22*m.a24 + a23*m.a34 + a24;
    t.a31 = a31*m.a11 + a32*m.a21 + a33*m.a31;
    t.a32 = a31*m.a12 + a32*m.a22 + a33*m.a32;
    t.a33 = a31*m.a13 + a32*m.a23 + a33*m.a33;
    t.a34 = a31*m.a14 + a32*m.a24 + a33*m.a34 + a34;

    *this = t;

    }

Matrix Matrix::invert(void) {
    // Invert matrix
    //
    // This is whipped out of symbolic Matlab - it should be correct but
    // probably isn't very efficient
    //
    Matrix t;

    // FIXUP: Make this test OPTIONAL

    if(det() == 0) {
        printf("Attempted to invert uninvertible matrix");
        exit(EXIT_FAILURE);
        }

    t.a11 = -(a22*a33-a32*a23)/(-a11*a22*a33+a11*a32*a23+a21*a12*a33-a21*a32*a13-a31*a12*a23+a31*a22*a13);
    t.a12 = (a12*a33-a32*a13)/(-a11*a22*a33+a11*a32*a23+a21*a12*a33-a21*a32*a13-a31*a12*a23+a31*a22*a13);
    t.a13 = -(a12*a23-a22*a13)/(-a11*a22*a33+a11*a32*a23+a21*a12*a33-a21*a32*a13-a31*a12*a23+a31*a22*a13);
    t.a14 = (a12*a23*a34-a12*a24*a33-a22*a13*a34+a22*a14*a33+a32*a13*a24-a32*a14*a23)/(-a11*a22*a33+a11*a32*a23+a21*a12*a33-a21*a32*a13-a31*a12*a23+a31*a22*a13);
    t.a21 = (a21*a33-a31*a23)/(-a11*a22*a33+a11*a32*a23+a21*a12*a33-a21*a32*a13-a31*a12*a23+a31*a22*a13);
    t.a22 = (-a11*a33+a31*a13)/(-a11*a22*a33+a11*a32*a23+a21*a12*a33-a21*a32*a13-a31*a12*a23+a31*a22*a13);
    t.a23 = -(-a11*a23+a21*a13)/(-a11*a22*a33+a11*a32*a23+a21*a12*a33-a21*a32*a13-a31*a12*a23+a31*a22*a13);
    t.a24 = (-a11*a23*a34+a11*a24*a33+a21*a13*a34-a21*a14*a33-a31*a13*a24+a31*a14*a23)/(-a11*a22*a33+a11*a32*a23+a21*a12*a33-a21*a32*a13-a31*a12*a23+a31*a22*a13);
    t.a31 = -(a21*a32-a31*a22)/(-a11*a22*a33+a11*a32*a23+a21*a12*a33-a21*a32*a13-a31*a12*a23+a31*a22*a13);
    t.a32 = -(-a11*a32+a31*a12)/(-a11*a22*a33+a11*a32*a23+a21*a12*a33-a21*a32*a13-a31*a12*a23+a31*a22*a13);
    t.a33 = (-a11*a22+a21*a12)/(-a11*a22*a33+a11*a32*a23+a21*a12*a33-a21*a32*a13-a31*a12*a23+a31*a22*a13);
    t.a34 = -(-a11*a22*a34+a11*a24*a32+a21*a12*a34-a21*a14*a32-a31*a12*a24+a31*a14*a22)/(-a11*a22*a33+a11*a32*a23+a21*a12*a33-a21*a32*a13-a31*a12*a23+a31*a22*a13);

    return t;
    }

double Matrix::det(void) {
    return a11*a22*a33-a11*a32*a23-a21*a12*a33+a21*a32*a13+a31*a12*a23-a31*a22*a13;
    }

Token *Matrix::Load(Token *T) {

    Matrix Rx, Ry, Rz;

    Vector v;

    // Only allow translate, scale, rotate at present.

    // May change depending on format of transformations, debugging etc.

    switch(T->ID) {
        case ID_translate:
            T = T->Next;
            T = v.Load(T);

            a11 = 1;
            a12 = 0;
            a13 = 0;
            a14 = -v.x;
            a21 = 0;
            a22 = 1;
            a23 = 0;
            a24 = -v.y;
            a31 = 0;
            a32 = 0;
            a33 = 1;
            a34 = -v.z;

            break;
    
        case ID_scale:
            T = T->Next;
            T = v.Load(T);

            if((v.x == 0) || (v.y == 0) || (v.z == 0)) {
                printf("Degenerate scale in %s line %li\n", T->FileName, T->Line);
                exit(EXIT_FAILURE);
                }

            a11 = 1/v.x;
            a12 = 0;
            a13 = 0;
            a14 = 0;
            a21 = 0;
            a22 = 1/v.y;
            a23 = 0;
            a24 = 0;
            a31 = 0;
            a32 = 0;
            a33 = 1/v.z;
            a34 = 0;

            break;

        case ID_rotate:
            T = T->Next;
            T = v.Load(T);

            // Convert degrees to radians

            v /= 57.29577951;

            // Rotation Matrices

            Rx.a11 = 1;
            Rx.a12 = 0;
            Rx.a13 = 0;
            Rx.a14 = 0;
            Rx.a21 = 0;
            Rx.a22 = cos(v.x);
            Rx.a23 = -sin(v.x);
            Rx.a24 = 0;
            Rx.a31 = 0;
            Rx.a32 = sin(v.x);
            Rx.a33 = cos(v.x);
            Rx.a34 = 0;

            Ry.a11 = cos(v.y);
            Ry.a12 = 0;
            Ry.a13 = sin(v.y);
            Ry.a14 = 0;
            Ry.a21 = 0;
            Ry.a22 = 1;
            Ry.a23 = 0;
            Ry.a24 = 0;
            Ry.a31 = -sin(v.y);
            Ry.a32 = 0;
            Ry.a33 = cos(v.y);
            Ry.a34 = 0;

            Rz.a11 = cos(v.z);
            Rz.a12 = -sin(v.z);
            Rz.a13 = 0;
            Rz.a14 = 0;
            Rz.a21 = sin(v.z);
            Rz.a22 = cos(v.z);
            Rz.a23 = 0;
            Rz.a24 = 0;
            Rz.a31 = 0;
            Rz.a32 = 0;
            Rz.a33 = 1;
            Rz.a34 = 0;

            // Combine 'em

            *this = Rx * Ry * Rz;

            break;

        case ID_matrix:
            T = T->Next;
            if(T->ID != ID_LEFT_ANGLE) { PrintToken(T); printf("found, expected <\n"); exit(EXIT_FAILURE); }
            T = T->Next;
            if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
            a11 = *((double *) T->Data);
            T = T->Next;
            if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
            T = T->Next;
            if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
            a21 = *((double *) T->Data);
            T = T->Next;
            if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
            T = T->Next;
            if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
            a31 = *((double *) T->Data);
            T = T->Next;
            if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
            T = T->Next;
            if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
            a12 = *((double *) T->Data);
            T = T->Next;
            if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
            T = T->Next;
            if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
            a22 = *((double *) T->Data);
            T = T->Next;
            if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
            T = T->Next;
            if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
            a32 = *((double *) T->Data);
            T = T->Next;
            if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
            T = T->Next;
            if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
            a13 = *((double *) T->Data);
            T = T->Next;
            if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
            T = T->Next;
            if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
            a23 = *((double *) T->Data);
            T = T->Next;
            if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
            T = T->Next;
            if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
            a33 = *((double *) T->Data);
            T = T->Next;
            if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
            T = T->Next;
            if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
            a14 = *((double *) T->Data);
            T = T->Next;
            if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
            T = T->Next;
            if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
            a24 = *((double *) T->Data);
            T = T->Next;
            if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
            T = T->Next;
            if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
            a34 = *((double *) T->Data);
            T = T->Next;
            if(T->ID != ID_RIGHT_ANGLE) { PrintToken(T); printf("found, expected >\n"); exit(EXIT_FAILURE); }

            break;

        }

    // Check

    if(det() == 0) {
        printf("Degenerate transformation in %s at line %li\n", T->FileName, T->Line);
        exit(EXIT_FAILURE);
        }

    return T;

    }


//
//  Colour functions
//

inline fColour iColour::Convert(void) {
    fColour c;
    c.b = (double) b / 255.0;
    c.g = (double) g / 255.0;
    c.r = (double) r / 255.0;
    c.f = 0;
    c.t = (double) 1.0 - ((double) a / 255.0);
    return c;
    }

inline iColour fColour::Convert(void) {
    iColour c;

    if(b < 1.0) {
        if(b > 0.0) {
            c.b = (unsigned char) ((double) b * 255.0);
            }
        else {
            c.b = 0;
            }
        }
    else {
        c.b = 255;
        }

    if(g < 1.0) {
        if(g > 0.0) {
            c.g = (unsigned char) ((double) g * 255.0);
            }
        else {
            c.g = 0;
            }
        }
    else {
        c.g = 255;
        }

    if(r < 1.0) {
        if(r > 0.0) {
            c.r = (unsigned char) ((double) r * 255.0);
            }
        else {
            c.r = 0;
            }
        }
    else {
        c.r = 255;
        }

    if(t < 1.0) {
        if(t > 0.0) {
            c.a = (unsigned char) ((double) ((double) 1.0 - t) * 255.0);
            }
        else {
            c.a = 255;
            }
        }
    else {
        c.a = 0;
        }

    return c;
    }

inline void fColour::Clean(void) {
    if(r > 1) {
        r = 1;
        }
    else {
        if(r < 0) {
            r = 0;
            }
        }
    if(g > 1) {
        g = 1;
        }
    else {
        if(g < 0) {
            g = 0;
            }
        }
    if(b > 1) {
        b = 1;
        }
    else {
        if(b < 0) {
            b = 0;
            }
        }
    }

fColour fColour::Doppler(double d) {
    fColour s;

    if(d == 1.0) {
        return *this;
        }


    // Gives a spectrum like
    //        _                              _           ____
    // BLUE _- \____ GREEN _-_/\_-_ RED ____/ -_ WHITE _-    -_
    


    // Flip d

    d = 1.0/d;

    // Work out which bit of the interpolated spectrum...

    if(lBlue < (d * lUltraViolet)) {
        s.b = lBlue * ((0.5 * b) + (0.25 * g) + (0.125 * r))/ (d * lUltraViolet);
        }
    else {
        if(lBlue < (d * lBlue)) {
            s.b = ((0.5 * b) + (0.25 * g) + (0.125 * r)) + ((lBlue - (d * lUltraViolet)) / ((d * lBlue) - (d * lUltraViolet))) * ((0.5 * b) - (0.25 * g) - (0.125 * r));
            }
        else {
            if(lBlue < (d * lGreen)) {
                s.b = b + ((lBlue - (d * lBlue)) / ((d * lGreen) - (d * lBlue))) * (-b + g);
                }
            else {
                if(lBlue < (d * lRed)) {
                    s.b = g + ((lBlue - (d * lGreen)) / ((d * lRed) - (d * lGreen))) * (-g + r);
                    }
                else {
                   if(lBlue < (d * lInfraRed)) {
                        s.b = r + ((lBlue - (d * lRed)) / ((d * lInfraRed) - (d * lRed))) * ((0.125 * b) + (0.25 * g) - (0.5 * r));
                        }
                    else {
                        // Must be this... so no test
                        s.b = ((0.125 * b) + (0.25 * g) + (0.5 * r)) * ((d * lInfraRed) / lBlue);
                        }
                    }
                }
            }
        }

    if(lGreen < (d * lUltraViolet)) {
        s.g = lGreen * ((0.5 * b) + (0.25 * g) + (0.125 * r))/ (d * lUltraViolet);
        }
    else {
        if(lGreen < (d * lBlue)) {
            s.g = ((0.5 * b) + (0.25 * g) + (0.125 * r)) + ((lGreen - (d * lUltraViolet)) / ((d * lBlue) - (d * lUltraViolet))) * ((0.5 * b) - (0.25 * g) - (0.125 * r));
            }
        else {
            if(lGreen < (d * lGreen)) {
                s.g = b + ((lGreen - (d * lBlue)) / ((d * lGreen) - (d * lBlue))) * (-b + g);
                }
            else {
                if(lGreen < (d * lRed)) {
                    s.g = g + ((lGreen - (d * lGreen)) / ((d * lRed) - (d * lGreen))) * (-g + r);
                    }
                else {
                   if(lGreen < (d * lInfraRed)) {
                        s.g = r + ((lGreen - (d * lRed)) / ((d * lInfraRed) - (d * lRed))) * ((0.125 * b) + (0.25 * g) - (0.5 * r));
                        }
                    else {
                        // Must be this... so no test
                        s.g = ((0.125 * b) + (0.25 * g) + (0.5 * r)) * ((d * lInfraRed) / lGreen);
                        }
                    }
                }
            }
        }

    if(lRed < (d * lUltraViolet)) {
        s.r = lRed * ((0.5 * b) + (0.25 * g) + (0.125 * r))/ (d * lUltraViolet);
        }
    else {
        if(lRed < (d * lBlue)) {
            s.r = ((0.5 * b) + (0.25 * g) + (0.125 * r)) + ((lRed - (d * lUltraViolet)) / ((d * lBlue) - (d * lUltraViolet))) * ((0.5 * b) - (0.25 * g) - (0.125 * r));
            }
        else {
            if(lRed < (d * lGreen)) {
                s.r = b + ((lRed - (d * lBlue)) / ((d * lGreen) - (d * lBlue))) * (-b + g);
                }
            else {
                if(lRed < (d * lRed)) {
                    s.r = g + ((lRed - (d * lGreen)) / ((d * lRed) - (d * lGreen))) * (-g + r);
                    }
                else {
                   if(lRed < (d * lInfraRed)) {
                        s.r = r + ((lRed - (d * lRed)) / ((d * lInfraRed) - (d * lRed))) * ((0.125 * b) + (0.25 * g) - (0.5 * r));
                        }
                    else {
                        // Must be this... so no test
                        s.r = ((0.125 * b) + (0.25 * g) + (0.5 * r)) * ((d * lInfraRed) / lRed);
                        }
                    }
                }
            }
        }




    s.f = f;
    s.t = t;

    return s;
    }

fColour fColour::Doppler2(double d) {
    fColour s;

    // This alternative to Doppler() has infinte wings - we can use it to
    // better model the reflectivity of surfaces, and ambient light etc. -
    // the cases when we don't want a falloff in refklectivity etc. as well
    // as illumination

    // Gives a spectrum like
    //      __                          __       ______
    // BLUE   \___ GREEN __/\__ RED ___/   WHITE
    

    if(d == 1.0) {
        return *this;
        }

    // Flip d

    d = 1.0/d;

    // Work out which bit of the interpolated spectrum...

    if(lBlue < (d * lBlue)) {
        s.b = b;
        }
    else {
        if(lBlue < (d * lGreen)) {
            s.b = b + ((lBlue - (d * lBlue)) / ((d * lGreen) - (d * lBlue))) * (-b + g);
            }
        else {
            if(lBlue < (d * lRed)) {
                s.b = g + ((lBlue - (d * lGreen)) / ((d * lRed) - (d * lGreen))) * (-g + r);
                }
             else {
                s.b = r;
                }
            }
        }

    if(lGreen < (d * lBlue)) {
        s.g = b;
        }
    else {
        if(lGreen < (d * lGreen)) {
            s.g = b + ((lGreen - (d * lBlue)) / ((d * lGreen) - (d * lBlue))) * (-b + g);
            }
        else {
            if(lGreen < (d * lRed)) {
                s.g = g + ((lGreen - (d * lGreen)) / ((d * lRed) - (d * lGreen))) * (-g + r);
                }
             else {
                s.g = r;
                }

            }
        }

    if(lRed < (d * lBlue)) {
        s.r = b;
        }
    else {
        if(lRed < (d * lGreen)) {
            s.r = b + ((lRed - (d * lBlue)) / ((d * lGreen) - (d * lBlue))) * (-b + g);
            }
        else {
            if(lRed < (d * lRed)) {
                s.r = g + ((lRed - (d * lGreen)) / ((d * lRed) - (d * lGreen))) * (-g + r);
                }
             else {
                s.r = r;
                }
            }
        }

    s.f = f;
    s.t = t;

    return s;
    }


fColour fColour::operator + (fColour c) {
    fColour s;
    s.r = r + c.r;
    s.g = g + c.g;    
    s.b = b + c.b;
    s.f = f;
    s.t = t;
    return s;
    }

fColour fColour::operator - (fColour c) {
    fColour s;
    s.r = r - c.r;
    s.g = g - c.g;    
    s.b = b - c.b;
    s.f = f;
    s.t = t;
    return s;
    }

fColour fColour::operator * (fColour c) {
    fColour s;
    s.r = r * c.r;
    s.g = g * c.g;    
    s.b = b * c.b;
    s.f = f;
    s.t = t;
    return s;
    }

void fColour::operator += (fColour c) {
    r += c.r;
    g += c.g;
    b += c.b;
    }

void fColour::operator -= (fColour c) {
    r -= c.r;
    g -= c.g;
    b -= c.b;
    }

void fColour::operator *= (fColour c) {
    r *= c.r;
    g *= c.g;
    b *= c.b;
    }

fColour fColour::operator * (double a) {
    fColour s;
    s.r = r * a;
    s.g = g * a;
    s.b = b * a;
    s.f = f;
    s.t = t;
    return s;
    }

void fColour::operator *= (double a) {
    r *= a;
    g *= a;
    b *= a;
    }

Token *fColour::Load(Token *T) {
    Token *S;

    S = T;

    do {
        switch(T->ID) {
            case ID_color:  // Either superfluous or unnecessary
            case ID_colour:
                break;

            case ID_DOUBLE:
                r = *((double *) T->Data);
                g = *((double *) T->Data);
                b = *((double *) T->Data);
                f = *((double *) T->Data);
                t = *((double *) T->Data);
                break;

            case ID_LEFT_ANGLE:
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE);; }
                r = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                g = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                b = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                f = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                t = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_RIGHT_ANGLE) { PrintToken(T); printf("found, expected >\n"); exit(EXIT_FAILURE); }
                break;

            case ID_rgb:
                T=T->Next;
                if(T->ID != ID_LEFT_ANGLE) {
                    if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected <\n"); exit(EXIT_FAILURE); }
                    // OK, is a double, so...
                    r = *((double *) T->Data);
                    g = *((double *) T->Data);
                    b = *((double *) T->Data);
                    break;
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                r = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                g = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                b = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_RIGHT_ANGLE) { PrintToken(T); printf("found, expected >\n"); exit(EXIT_FAILURE); }
                break;

            case ID_rgbt:
                T=T->Next;
                if(T->ID != ID_LEFT_ANGLE) {
                    if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected <\n"); exit(EXIT_FAILURE); }
                    // OK, is a double, so...
                    r = *((double *) T->Data);
                    g = *((double *) T->Data);
                    b = *((double *) T->Data);
                    t = *((double *) T->Data);
                    break;
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                r = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                g = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                b = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                t = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_RIGHT_ANGLE) { PrintToken(T); printf("found, expected >\n"); exit(EXIT_FAILURE); }
                break;

            case ID_rgbf:
                T=T->Next;
                if(T->ID != ID_LEFT_ANGLE) {
                    if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected <\n"); exit(EXIT_FAILURE); }
                    // OK, is a double, so...
                    r = *((double *) T->Data);
                    g = *((double *) T->Data);
                    b = *((double *) T->Data);
                    f = *((double *) T->Data);
                    break;
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                r = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                g = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                b = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                f = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_RIGHT_ANGLE) { PrintToken(T); printf("found, expected >\n"); exit(EXIT_FAILURE); }
                break;

            case ID_rgbft:
                // This is the default format - can leave it to LEFT_ANGLE
                break;

            case ID_rgbtf:
                T=T->Next;
                if(T->ID != ID_LEFT_ANGLE) {
                    if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected <\n"); exit(EXIT_FAILURE); }
                    // OK, is a double, so...
                    r = *((double *) T->Data);
                    g = *((double *) T->Data);
                    b = *((double *) T->Data);
                    t = *((double *) T->Data);
                    f = *((double *) T->Data);
                    break;
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                r = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                g = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                b = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                t = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                f = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_RIGHT_ANGLE) { PrintToken(T); printf("found, expected >\n"); exit(EXIT_FAILURE); }
                break;

            case ID_red:
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                r = *((double *) T->Data);
                break;

            case ID_green:
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                g = *((double *) T->Data);
                break;

            case ID_blue:
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                b = *((double *) T->Data);
                break;

            case ID_filter:
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                f = *((double *) T->Data);
                break;

            case ID_transmit:
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                t = *((double *) T->Data);
                break;

            default:
                if(S == T) {
                    PrintToken(T);
                    if(Globals.Verbose >= 2) {
                        printf("no colour specified here, using defaults");
                        }
                    }
                return S;
                break;
            }
        S = T;
        T = T->Next;
        } while(T->ID != ID_END);

    return T;

    }

//
//  Bitmap functions
//

void Bitmap::Create(unsigned long u, unsigned long v) {
    uMax = u;
    vMax = v;

    Pixel = new iColour[uMax * vMax];
	Globals.MemoryUsed += 4 * uMax * vMax;
    if(Pixel == NULL) {
        printf("Bitmap::Create: Can't allocate %li bytes of memory for new bitmap (%li x %li)\n", 4 * uMax * vMax, uMax, vMax);
        exit(EXIT_FAILURE);
        }

    FileName = NULL;
    Next = NULL;
    
    memset(Pixel, 0, 4 * uMax * vMax);
    }

void Bitmap::LoadTGA(char *Name) {
    FILE *Handle;
    unsigned char Header[18];

    Handle = fopen(Name, "rb");
    if(Handle == NULL) {
        printf("Bitmap::LoadTGA: Can't open %.32s\n", Name);
        exit(EXIT_FAILURE);
        }

    fseek(Handle, 0, 0);
    fread(Header, 1, 18, Handle);

    if(Header[2] != 2) {
        printf("LoadTGA: %32s is not a valid TGA file\n", Name);
        exit(EXIT_FAILURE);
        }

    if(Header[16] != 32) {
        printf("LoadTGA: %.32s is a %i bit file. Only 32 bit images (24 bit colour \n+ 8 bit alpha channel) are supported. Please convert this file\n", Name, Header[16]);
        exit(EXIT_FAILURE);
        }

    FileName = NULL;
    Next = NULL;

    uMax = ((unsigned long) Header[13] << 8) + Header[12];
    vMax = ((unsigned long) Header[15] << 8) + Header[14];

    Pixel = new iColour[uMax * vMax];
   	Globals.MemoryUsed += 4 * uMax * vMax;
	if(Pixel == NULL) {
        printf("LoadTGA: Can't allocate %li bytes of memory for %.32s (%li x %li)\n", 4 * uMax * vMax, Name, uMax, vMax);
        exit(EXIT_FAILURE);
        }
    fseek(Handle, 18, 0);
    fread(Pixel, 4, uMax*vMax, Handle);
    fclose(Handle);

    if(Globals.Verbose >= 1) {
        printf("Bitmap::LoadTGA: %.32s (%li x %li) successfully loaded\n", Name, uMax, vMax);
        }
    }

void Bitmap::SaveTGA(char *Name) {
    FILE *Handle;
    unsigned char Header[18];

    Header[ 0] = 0;
    Header[ 1] = 0;
    Header[ 2] = 2;     // Uncompressed, uninteresting
    Header[ 3] = 0;
    Header[ 4] = 0;
    Header[ 5] = 0;
    Header[ 6] = 0;
    Header[ 7] = 0;
    Header[ 8] = 0;
    Header[ 9] = 0;
    Header[10] = 0;
    Header[11] = 0;
    Header[12] = (unsigned char) uMax;  // Dimensions
    Header[13] = (unsigned char) ((unsigned long) uMax >> 8);
    Header[14] = (unsigned char) vMax;
    Header[15] = (unsigned char) ((unsigned long) vMax >> 8);
    Header[16] = 32;    // Bits per pixel
    Header[17] = 0;

    Handle = fopen(Name, "wb");
    if(Handle == NULL) {
        printf("Bitmap::SaveTGA: Can't open %.32s\n", Name);
        exit(EXIT_FAILURE);
        }
    fseek(Handle, 0, 0);
    fwrite(Header, 1, 18, Handle);
    fseek(Handle, 18, 0);
    fwrite(Pixel, 4, uMax * vMax, Handle);
    fclose(Handle);

    // FIXUP: Verbose?
    if(Globals.Verbose >= 1) {
        printf("Bitmap::SaveTGA: %.32s (%li x %li) successfully saved\n", Name, uMax, vMax);
        }
    }

void Bitmap::Set(unsigned long u, unsigned v, iColour c) {
    Pixel[u + v*uMax].b = (unsigned char) ((double) Pixel[u + v*uMax].b * c.a / 255.0 + c.b * (255.0 - c.a) / 255.0);
    Pixel[u + v*uMax].g = (unsigned char) ((double) Pixel[u + v*uMax].g * c.a / 255.0 + c.g * (255.0 - c.a) / 255.0);
    Pixel[u + v*uMax].r = (unsigned char) ((double) Pixel[u + v*uMax].r * c.a / 255.0 + c.r * (255.0 - c.a) / 255.0);
    Pixel[u + v*uMax].a = (unsigned char) ((double) Pixel[u + v*uMax].a * c.a / 255.0);
    }

fColour Bitmap::GetPixel(double u, double v, long interpolate) {
    long iu, iv;
    double du, dv;
    iColour c;
    fColour s;

	//printf("Getting pixel...\n");

    // Clip to unit square

    u = fmod(u, 1.0);
    v = fmod(v, 1.0);

	if(u < 0.0f) {
        u += 1.0f;
        }
    if(v < 0.0f) {
        v += 1.0f;
        }

	if(u < 0.0f) {
		u = 0.0f;
		}
	if(v < 0.0f) {
		v = 0.0f;
		}
	if(u >= 1.0f) {
		u = 0.0f;
		}
	if(v >= 1.0f) {
		v = 0.0f;
		}

	//printf("    (%16.16lf, %16.16lf)\n", u, v);

    iu = (long) ((double) u * uMax);
    iv = (long) ((double) v * vMax);

    c = Pixel[iu + (iv * uMax)];

    if(!interpolate) {
        s = c.Convert();
        s.f = s.t;
        return s;
        }

    du = fmod(u * uMax, 1.0);
    dv = fmod(v * vMax, 1.0);

    s.b = (double) (1.0 - du) * (1.0 - dv) * c.b / 255.0;
    s.g = (double) (1.0 - du) * (1.0 - dv) * c.g / 255.0;
    s.r = (double) (1.0 - du) * (1.0 - dv) * c.r / 255.0;
    s.t = (double) 1.0 - (1.0 - du) * (1.0 - dv) * c.a / 255.0;

    iu++;
    if(iu >= (signed long) uMax) {
        iu = 0;
        }
    c = Pixel[iu + (iv * uMax)];

    s.b += (double) du * (1.0 - dv) * c.b / 255.0;
    s.g += (double) du * (1.0 - dv) * c.g / 255.0;
    s.r += (double) du * (1.0 - dv) * c.r / 255.0;
    s.t -= (double) du * (1.0 - dv) * c.a / 255.0;

    iv++;
    if(iv >= (signed long) vMax) {
        iv = 0;
        }
    c = Pixel[iu + (iv * uMax)];

    s.b += (double) du * dv * c.b / 255.0;
    s.g += (double) du * dv * c.g / 255.0;
    s.r += (double) du * dv * c.r / 255.0;
    s.t -= (double) du * dv * c.a / 255.0;

    iu--;
    if(iu < 0) {
        iu = uMax - 1;
        }
    c = Pixel[iu + (iv * uMax)];

    s.b += (double) (1.0 - du) * dv * c.b / 255.0;
    s.g += (double) (1.0 - du) * dv * c.g / 255.0;
    s.r += (double) (1.0 - du) * dv * c.r / 255.0;
    s.t -= (double) (1.0 - du) * dv * c.a / 255.0;

    s.f = s.t;

	//printf("                ...Finished\n");
    
	return s;
    };

void Bitmap::GetBump(double &u, double &v, long interpolate) {
    long iu, iv;
    double du, dv;
    double c00, c01, c02, c10, c11, c12, c20, c21, c22;

    // Clip to unit square

    u = fmod(u, 1.0);
    v = fmod(v, 1.0);

    // Necessary?

    if(u < 0.0) {
        u += 1.0;
        }
    if(v < 0.0) {
        v += 1.0;
        }


    iu = (long) ((double) u * uMax);
    iv = (long) ((double) v * vMax);

    c00 = Pixel[iu + (iv * uMax)].r + Pixel[iu + (iv * uMax)].g + Pixel[iu + (iv * uMax)].b;


    iu++;
    if(iu >= (signed long) uMax) {
        iu = 0;
        }
    c10 = Pixel[iu + (iv * uMax)].r + Pixel[iu + (iv * uMax)].g + Pixel[iu + (iv * uMax)].b;

    iv++;
    if(iv >= (signed long) vMax) {
        iv = 0;
        }
    c11 = Pixel[iu + (iv * uMax)].r + Pixel[iu + (iv * uMax)].g + Pixel[iu + (iv * uMax)].b;

    iu--;
    if(iu < 0) {
        iu = uMax - 1;
        }
    c01 = Pixel[iu + (iv * uMax)].r + Pixel[iu + (iv * uMax)].g + Pixel[iu + (iv * uMax)].b;

    if(!interpolate) {
        u = (c10 - c00) / 255;
        v = (c01 - c00) / 255;
        return;
        }

    iv++;
    if(iv >= (signed long) vMax) {
        iv = 0;
        }
    c02 = Pixel[iu + (iv * uMax)].r + Pixel[iu + (iv * uMax)].g + Pixel[iu + (iv * uMax)].b;

    iu++;
    if(iu >= (signed long) uMax) {
        iu = 0;
        }
    c12 = Pixel[iu + (iv * uMax)].r + Pixel[iu + (iv * uMax)].g + Pixel[iu + (iv * uMax)].b;

    iu++;
    if(iu >= (signed long) uMax) {
        iu = 0;
        }
    c22 = Pixel[iu + (iv * uMax)].r + Pixel[iu + (iv * uMax)].g + Pixel[iu + (iv * uMax)].b;

    iv--;
    if(iv < 0) {
        iv = vMax - 1;
        }
    c21 = Pixel[iu + (iv * uMax)].r + Pixel[iu + (iv * uMax)].g + Pixel[iu + (iv * uMax)].b;

    iv--;
    if(iv < 0) {
        iv = vMax - 1;
        }
    c20 = Pixel[iu + (iv * uMax)].r + Pixel[iu + (iv * uMax)].g + Pixel[iu + (iv * uMax)].b;

    du = fmod(u * uMax, 1.0);
    dv = fmod(v * vMax, 1.0);

    // We now interpolate the slopes...

    u = (1 - du) * (1 - dv) * (c10 - c00)
      +      du  * (1 - dv) * (c20 - c10)
      + (1 - du) *      dv  * (c11 - c01)
      +      du  *      dv  * (c21 - c11);
    u /= (double) 255.0;
    v = (1 - du) * (1 - dv) * (c01 - c00)
      +      du  * (1 - dv) * (c11 - c10)
      + (1 - du) *      dv  * (c02 - c01)
      +      du  *      dv  * (c12 - c11);
    v /= (double) 255.0;

    return;
    };



void Bitmap::Combine(unsigned long u1, unsigned long v1,
                     unsigned long u2, unsigned long v2,
                     unsigned long u3, unsigned long v3,
                     unsigned long u4, unsigned long v4,
                     Bitmap b) {
    unsigned long iu, iv;
    double fu, fv, du, dv, su, sv;

    du = (double) (u4 - u3) / ((u2 - u1) * b.uMax);
    dv = (double) (v4 - v3) / ((v2 - v1) * b.vMax);

	su = (double) u3 / b.vMax;
	sv = (double) v3 / b.uMax;

	if(u1 < 0) {
		su -= du * u1;
		u1 = 0;
		}
	if(v1 < 0) {
		sv -= dv * v1;
		v1 = 0;
		}
	if(u2 >= uMax) {
		u2 = uMax;
		}
	if(v2 >= vMax) {
		v2 = vMax;
		}
	
    iv = v1;
    fv = (double) sv;
    do {
        iu = u1;
        fu = (double) su;
        do {
            Set(iu, iv, b.GetPixel(fu, fv, 1).Convert());
            fu += du;
            } while(++iu < u2);
        fv += dv;
        } while(++iv < v2);
    }

void RequestBitmap(Bitmap **Target, char *FileName) {
    
	Bitmap *Current;
	long i;

	if(First_Bitmap == NULL) {
		// This is the first texture bitmap we have attempted to load, so it is not
		// duplicated in memory. Thus we load it, creating as we go!

		(*Target) = new Bitmap;
		Globals.MemoryUsed += sizeof(Bitmap);
		(*Target)->LoadTGA(FileName);

		(*Target)->FileName = FileName;
		(*Target)->Next = NULL;

		First_Bitmap = *Target;
	
		return;
		}
	
	Current = First_Bitmap;

	while(Current->Next != NULL) {
		// This isn't the last bitmap
		i = 0;
		while(Current->FileName[i] == FileName[i]) {
			// OK, strings are equal in this character, so check for the string terminator
			// then try the next character
			if( (Current->FileName[i] == 0) && (FileName[i] == 0) ) {
				// Ahah! We have already loaded the file. So
				*Target = Current;

				// printf("RequestBitmap: %s already loaded, using loaded version.\n", FileName);

				return;
				}
			// Increment,
			i++;
			}
		// Increment...	
		Current = Current->Next;
		}
	
	// OK this is the last file in the sequence (With a NULL pointer to the next file...)
	
	i = 0;

	while(Current->FileName[i] == FileName[i]) {
		
		// OK, strings are equal in this character, so check for the string terminator
		// then try the next character
		if( (Current->FileName[i] == 0) && (FileName[i] == 0) ) {
			// Ahah! We have already loaded the file. So

			// printf("RequestBitmap: %s already loaded, using loaded version.\n", FileName);

			*Target = Current;
			return;
			}
		// Increment,
		i++;
		}
	
	// No such luck, the bitmap has not been loaded yet. So load it!

	*Target = new Bitmap;
	Globals.MemoryUsed += sizeof(Bitmap);
	(*Target)->LoadTGA(FileName);
	(*Target)->FileName = FileName;
	(*Target)->Next = NULL;

	Current->Next = *Target;

    }
    

//
//  Event functions
//

Event Event::LorentzTransform(Vector &v, Vector &u) {

    // FROM frame with velocity v (relative to base reference frame)
    // TO frame with velocity u (relative to base reference frame)

    Event e;
    Vector nu, nv;
    double gu, gv;

    if(!Globals.Relativity) {
        e.x = x + (v * t) - (u * t);
        e.t = t;
        return e;
        }

    if(u.nrm() != 0) {
        nu = u / u.nrm();
        }
    else {
        nu = u;
        }

    if(v.nrm() != 0) {
        nv = v / v.nrm();
        }
    else {
        nv = v;
        }

    gu = gamma(u);
    gv = gamma(v);

    // Derived in obvious fashion; calculate x|| = x.nv, and transform this
    // with t in the traditional fashion, then reconstruct x'. Applied twice
    // this gives a transformation between general frames. Uses -v as we
    // transform FROM v-frame.

    e.t = gu*(gv*(t*(1-(u*v))+(v*x))+(u*nv)*(nv*x)*(1-gv) - (u*x));
    e.x = x+nv*(gv-1)*(nv*x)+v*gv*t+nu*(gu-1)*((nu*x)+(nu*v)*gv*t+(nu*nv)*(nv*x)*(gv-1))-u*gu*gv*((v*x)+t);

    return e;
    }

Ray Ray::LorentzTransform(Vector &u, Vector &v) {

    Ray r;
    Event Temp;
    double Dratio;

    // FIXUP: this is a bit of an inefficient mess
    // Try to extract the useful bits from the geometry to speed up the
    // Doppler and intensity calculations

    // We use the Lorentz Transform (or Gallilean if Relativity == 0)

    r.Origin = Origin.LorentzTransform(u, v);

    r.Direction = Direction.LorentzTransform(u, v);

    if(Globals.Doppler || Globals.Intensity) {

        // Checked and confirmed, is frequency in v / frequency in u

        Dratio = ( 1 + (r.Direction * v)) * ( 1 - (Direction.LorentzTransform(u, Vector( 0, 0, 0 )) * u)) * gamma(u) * gamma(v);
        }

    if(Globals.Doppler) {
        r.doppler = doppler * Dratio;
        }
    else {
        r.doppler = doppler;
        }

    // Intensity combines both the solid angle ratio (obviously) and the
    // Doppler ratio - actually a dt/dt' ratio causing longer/shorter
    // exposure

     if(Globals.Intensity) {

        // Checked and confirmed, but this is shamefully wasteful

        r.intensity = intensity * Dratio / (
                                   ( 1 - (Direction.LorentzTransform(u, Vector( 0, 0, 0 )) * v)) * ( 1 - (Direction.LorentzTransform(u, Vector( 0, 0, 0 )) * v)) * ( 1 + (Direction * u)) * ( 1 + (Direction * u)) / ((1 - u.sqn()) * (1 - v.sqn()))
                                  );
        }
     else {
        r.intensity = intensity;
        }

    return r;

    }

//
// Raytracing Functions
//

fColour Texture::GetPigment(Event o) {
    double r;
    double u = 0, v = 0;
    fColour s;

    switch(P.map_type) {

        case MT_NONE:
            return fColour( 1, 1, 1, 0, 0 );
            break;

        case MT_COLOUR:
            return P.colour;
            break;

        case MT_PLANAR:
            o.x = P.Transformation.TransformLocation(o.x);
            u = o.x.x;
            v = o.x.y;
            break;

        case MT_SPHERICAL:
            o.x = P.Transformation.TransformLocation(o.x);
            if((o.x.x != 0) || (o.x.z != 0)) {
                u = atan2(o.x.x, o.x.z) / (2*PI);
                v = atan2(o.x.y, sqrt(o.x.x*o.x.x + o.x.z*o.x.z)) / PI + 0.5;
                }
            break;
        case MT_CYLINDRICAL:
            o.x = P.Transformation.TransformLocation(o.x);
            if((o.x.x != 0) || (o.x.z != 0)) {
                u = atan2(o.x.x, o.x.z) / (2*PI);
                }
            v = o.x.y;
            break;

        case MT_TOROIDAL:   // Not much point without a torus!
            o.x = P.Transformation.TransformLocation(o.x);
            if((o.x.x != 0) || (o.x.z != 0)) {
                r = sqrt(o.x.x*o.x.x + o.x.y*o.x.y) - 1;
                if((r != 0) || (o.x.y != 0)) {
                    u = atan2(o.x.x, o.x.z) / (2*PI);
                    v = atan2(o.x.y, r) / (2*PI);
                    }
                }
            break;

        case MT_DISK:
            o.x = P.Transformation.TransformLocation(o.x);
            if((o.x.x != 0) || (o.x.z != 0)) {
                u = atan2(o.x.x, o.x.z) / (2*PI);
                }
            v = sqrt((o.x.x * o.x.x) + (o.x.z * o.x.z));
            break;

        default:
            printf("Unknown map type %li in Texture::GetPigment\n", P.map_type);
            break;
        }

    u *= P.frequency_u;
    v *= P.frequency_v;
    u += P.phase_u;
    v += P.phase_v;
    u += o.t * P.drift_u;
    v += o.t * P.drift_v;

    if(P.once) {
        if((u < 0) || (v < 0) || (u > 1) || (v > 1)) {
            return fColour( 1, 1, 1, 0, 1 ); // Transparent
            }
        }

    s = P.bitmap->GetPixel(u, v, P.interpolate);
    s.f *= P.colour.f;
    s.t *= P.colour.t;

    return s;

    }



fColour Object::GetPigment(Event o) {
    double r;
    double u = 0, v = 0;
    fColour s;

    switch(Surface->P.map_type) {

        case MT_NONE:
            return fColour( 1, 1, 1, 0, 0 );
            break;

        case MT_COLOUR:
            return Surface->P.colour;
            break;

        case MT_PLANAR:
            o.x = Texture_Transformation.TransformLocation(o.x);
            o.x = Surface->P.Transformation.TransformLocation(o.x);
            u = o.x.x;
            v = o.x.y;
            break;

        case MT_SPHERICAL:

            o.x = Texture_Transformation.TransformLocation(o.x);
            o.x = Surface->P.Transformation.TransformLocation(o.x);

            if((o.x.x != 0) || (o.x.z != 0)) {
                u = atan2(o.x.x, o.x.z) / (2*PI);
                v = atan2(o.x.y, sqrt(o.x.x*o.x.x + o.x.z*o.x.z)) / PI + 0.5;
                }
            break;
        case MT_CYLINDRICAL:
            o.x = Texture_Transformation.TransformLocation(o.x);
            o.x = Surface->P.Transformation.TransformLocation(o.x);

            if((o.x.x != 0) || (o.x.z != 0)) {
                u = atan2(o.x.x, o.x.z) / (2*PI);
                }
            v = o.x.y;
            break;

        case MT_TOROIDAL:   // Not much point without a torus!
            o.x = Texture_Transformation.TransformLocation(o.x);
            o.x = Surface->P.Transformation.TransformLocation(o.x);

            if((o.x.x != 0) || (o.x.z != 0)) {
                r = sqrt(o.x.x*o.x.x + o.x.y*o.x.y) - 1;
                if((r != 0) || (o.x.y != 0)) {
                    u = atan2(o.x.x, o.x.z) / (2*PI);
                    v = atan2(o.x.y, r) / (2*PI);
                    }
                }
            break;

        case MT_DISK:
            o.x = Texture_Transformation.TransformLocation(o.x);
            o.x = Surface->P.Transformation.TransformLocation(o.x);

            if((o.x.x != 0) || (o.x.z != 0)) {
                u = atan2(o.x.x, o.x.z) / (2*PI);
                }
            v = sqrt((o.x.x * o.x.x) + (o.x.z * o.x.z));
            break;

        default:
            printf("Unknown map type %li in Object::GetPigment\n", Surface->P.map_type);
            break;
        }

    u *= Surface->P.frequency_u;
    v *= Surface->P.frequency_v;
    u += Surface->P.phase_u;
    v += Surface->P.phase_v;
    u += Surface->P.drift_u * o.t;
    v += Surface->P.drift_v * o.t;

    if(Surface->P.once) {
        if((u < 0) || (v < 0) || (u > 1) || (v > 1)) {
            return fColour( 0, 0, 0, 0, 1 ); // Transparent
            }
        }

    s = Surface->P.bitmap->GetPixel(u, v, Surface->P.interpolate);
    s.f *= Surface->P.colour.f;
    s.t *= Surface->P.colour.t;

    return s;

    }

Vector Object::GetNormal(Event o) {
    Vector du, dv;
    double u = 0, v = 0, r;

    du = Vector( 1, 0, 0 );
    dv = Vector( 0, 1, 0 );

    // FIXUP: When you want to interpolate splines & differentiate them
    // in 2D space etc.

    switch(Surface->N.map_type) {

        case MT_PLANAR:
            o.x = Texture_Transformation.TransformLocation(o.x);
            o.x = Surface->N.Transformation.TransformLocation(o.x);
            u = o.x.x;
            v = o.x.y;
            break;
        case MT_SPHERICAL:
            o.x = Texture_Transformation.TransformLocation(o.x);
            o.x = Surface->N.Transformation.TransformLocation(o.x);

            if((o.x.x != 0) || (o.x.z != 0)) {
                u = atan2(o.x.x, o.x.z) / (2*PI);
                v = atan2(o.x.y, sqrt(o.x.x*o.x.x + o.x.z*o.x.z)) / PI + 0.5;
                du = Vector( cos(u * 2 * PI), 0, sin(u * 2 * PI) );
                dv = Vector( sin(u * 2 * PI)*cos(v * PI), sin(v * PI), cos(u * 2 * PI)*cos(v * PI) );
                }
            break;
        case MT_CYLINDRICAL:
            o.x = Texture_Transformation.TransformLocation(o.x);
            o.x = Surface->N.Transformation.TransformLocation(o.x);

            if((o.x.x != 0) || (o.x.z != 0)) {
                u = atan2(o.x.x, o.x.z) / (2*PI);
                du = Vector( cos(u * 2 * PI), 0, sin(u * 2 * PI) );
                }
            v = o.x.y;
            break;

        case MT_TOROIDAL:   // Not much point without a torus!
            o.x = Texture_Transformation.TransformLocation(o.x);
            o.x = Surface->N.Transformation.TransformLocation(o.x);

            if((o.x.x != 0) || (o.x.z != 0)) {
                r = sqrt(o.x.x*o.x.x + o.x.y*o.x.y) - 1;
                if((r != 0) || (o.x.y != 0)) {
                    u = atan2(o.x.x, o.x.z) / (2*PI);
                    v = atan2(o.x.y, r) / (2*PI);

                    // FIXUP: Vectors for torus...

                    }
                }
            break;

        case MT_DISK:
            o.x = Texture_Transformation.TransformLocation(o.x);
            o.x = Surface->N.Transformation.TransformLocation(o.x);
            
            if((o.x.x != 0) || (o.x.z != 0)) {
                u = atan2(o.x.x, o.x.z) / (2*PI);
                du = Vector( cos(u * 2 * PI), 0, sin(u * 2 * PI) );
                v = sqrt((o.x.x * o.x.x) + (o.x.z * o.x.z));
                dv = Vector( o.x.x / v, 0, o.x.z / v );
                }

            break;

        default:
            return Vector( 0, 0, 0 );
            break;
        }

    u *= Surface->N.frequency_u;
    v *= Surface->N.frequency_v;
    u += Surface->N.phase_u;
    v += Surface->N.phase_v;
    u += Surface->N.drift_u * o.t;
    v += Surface->N.drift_v * o.t;


    if(Surface->N.once) {
        if((u < 0) || (v < 0) || (u > 1) || (v > 1)) {
            return Vector( 0, 0, 0 );
            }
        }

    Surface->N.bitmap->GetBump(u, v, Surface->N.interpolate);

    // The purpose of u, v has changed radically

    du = ((du * u) + (dv * v))*Surface->N.bump_size;

    return Texture_Transformation.invert().TransformDirection( Surface->N.Transformation.invert().TransformDirection( du ) );

    }

double Object::Interior(Vector o) {
    long i;
    double f;

    // Insert bounding test, IF possible

    switch(Object_Type) {

        case CSG_UNION:
            i = 0;
            f = -1;
            while(i < ((CSG_union *) Geometry)->Number_Objects) {
                if( ((CSG_union *) Geometry)->Object_List[i]->Interior(o) > 0) {
                    f = 1;
                    break;
                    }
                i++;
                }
            break;

        case CSG_MERGE:
            i = 0;
            f = -1;
            while(i < ((CSG_union *) Geometry)->Number_Objects) {
                if( ((CSG_union *) Geometry)->Object_List[i]->Interior(o) > 0) {
                    f = 1;
                    break;
                    }
                i++;
                }
            break;
        
        case CSG_INTERSECTION:
            i = 0;
            f = 1;
            while(i < ((CSG_union *) Geometry)->Number_Objects) {
                if( ((CSG_union *) Geometry)->Object_List[i]->Interior(o) < 0) {
                    f = -1;
                    break;
                    }
                i++;
                }
            break;

        case CSG_DIFFERENCE:
            i = 0;
            f = 1;
            while(i < ((CSG_union *) Geometry)->Number_Objects) {
                if(i != 0) {
                    if( ((CSG_union *) Geometry)->Object_List[i]->Interior(o) > 0) {
                        f = -1;
                        break;
                        }
                    }
                else {
                    if( ((CSG_union *) Geometry)->Object_List[i]->Interior(o) < 0) {
                        f = -1;
                        break;
                        }
                    }
                i++;
                }
            break;

        case SG_TRIANGLE:
            return 0;
            break;
        case SG_SMOOTH_TRIANGLE:
            return 0;
            break;
        case SG_SPHERE:
            o = Geometry_Transformation.TransformLocation(o);
            f = 1 - o.sqn();
            break;
        case SG_BOX:
            o = Geometry_Transformation.TransformLocation(o);
            f = -1;
            if(o.x > 0) {
                if(o.x < 1) {
                    if(o.y > 0) {
                        if(o.y < 1) {
                            if(o.z > 0) {
                                if(o.z < 1) {
                                    f = 1;
                                    }
                                }
                            }
                        }
                    }
                }
            break;
        case SG_CYLINDER:
            o = Geometry_Transformation.TransformLocation(o);
            f = -1;
            if(o.x > 0) {
                if(o.x < 1) {
                    f = 1 - o.y*o.y - o.z*o.z;
                    }
                }
            break;

        case SG_CONE:
            o = Geometry_Transformation.TransformLocation(o);
            f = -1;
            if(o.x > 0) {
                if(o.x < 1) {
                    if( (o.y*o.y + o.z*o.z) < ((1-o.x)*((SG_cone *) Geometry)->Radius_1 + o.x*((SG_cone *) Geometry)->Radius_2) * ((1-o.x)*((SG_cone *) Geometry)->Radius_1 + o.x*((SG_cone *) Geometry)->Radius_2) ) {
                        f = 1;
                        }
                    }
                }
            break;

        case SG_PLANE:
            // FIXUP: 2/3 of this is going to waste
            o = Geometry_Transformation.TransformLocation(o);
            f = o.x;
            break;

        case SG_QUADRIC:
            f = ((SG_quadric *) Geometry)->cxx * o.x*o.x
              + ((SG_quadric *) Geometry)->cyy * o.y*o.y
              + ((SG_quadric *) Geometry)->czz * o.z*o.z
              + ((SG_quadric *) Geometry)->cxy * o.x*o.y
              + ((SG_quadric *) Geometry)->cxz * o.x*o.z
              + ((SG_quadric *) Geometry)->cxx * o.x*o.y
              + ((SG_quadric *) Geometry)->cx  * o.x
              + ((SG_quadric *) Geometry)->cy  * o.y
              + ((SG_quadric *) Geometry)->cz  * o.z
              + ((SG_quadric *) Geometry)->c            ;
            break;
        default:
            printf("Object::Interior: Unknown object type %li\n", Object_Type);
            break;

        }

    if(inverse) {
        f = -f;
        }

    return f;

    }

Vector Object::Normal(Vector o) {

    Vector n;

    // Must be an SG, so

    switch(Object_Type) {

            // FIXUP: Work out triangle normal optomisation

        case SG_TRIANGLE:
            n = ((SG_triangle *) Geometry)->Normal;
            break;
        case SG_SMOOTH_TRIANGLE:
            n = (((SG_smooth_triangle *) Geometry)->Normal_1 * ((SG_smooth_triangle *) Geometry)->u)
              + (((SG_smooth_triangle *) Geometry)->Normal_2 * ((SG_smooth_triangle *) Geometry)->v)
              + (((SG_smooth_triangle *) Geometry)->Normal_3 * (1.0 - ((SG_smooth_triangle *) Geometry)->u - ((SG_smooth_triangle *) Geometry)->v));
            break;
        case SG_SPHERE:
            n = Geometry_Transformation.TransformLocation(o);
            n = (Geometry_Transformation.invert()).TransformNormal(n);
            break;

        case SG_BOX:
            o = Geometry_Transformation.TransformLocation(o) - Vector( 0.5, 0.5, 0.5 );

            if(fabs(o.x) > fabs(o.y)) {
                if(fabs(o.x) > fabs(o.z)) {
                    n = Vector( o.x, 0, 0 );
                    }
                else {
                    n = Vector( 0, 0, o.z );
                    }
                }
            else {
                if(fabs(o.y) > fabs(o.z)) {
                    n = Vector( 0, o.y, 0 );
                    }
                else {
                    n = Vector( 0, 0, o.z );
                    }
                }

            n = (Geometry_Transformation.invert()).TransformNormal(n);

            break;

        case SG_CYLINDER:
            o = Geometry_Transformation.TransformLocation(o) - Vector( 0.5, 0, 0 );

            if(4*o.x*o.x > (o.y*o.y + o.z*o.z)) {
                n = Vector( o.x, 0, 0 );
                }
            else {
                n = Vector( 0, o.y, o.z );
                }

            n = (Geometry_Transformation.invert()).TransformNormal(n);

            break;

        case SG_CONE:
            o = Geometry_Transformation.TransformLocation(o) - Vector( 0.5, 0, 0 );            

            if(o.x > 0) {
                if(4*o.x*o.x*((SG_cone *) Geometry)->Radius_2*((SG_cone *) Geometry)->Radius_2 > (o.y*o.y + o.z*o.z)) {
                    // => a cap:
                    n = Vector( o.x, 0, 0 );
                    n = (Geometry_Transformation.invert()).TransformNormal(n);
                    break;
                    }
                }
            else {
                if(4*o.x*o.x*((SG_cone *) Geometry)->Radius_1*((SG_cone *) Geometry)->Radius_1 > (o.y*o.y + o.z*o.z)) {
                    // => a cap:
                    n = Vector( o.x, 0, 0 );
                    n = (Geometry_Transformation.invert()).TransformNormal(n);
                    break;
                    }
                }

            // So, a wall:

            o.x = sqrt(o.y*o.y + o.z*o.z);

            n = Vector( ((SG_cone *) Geometry)->Radius_1 - ((SG_cone *) Geometry)->Radius_2, o.y/o.x, o.z/o.x );

            n = (Geometry_Transformation.invert()).TransformNormal(n);

            break;

        case SG_PLANE:
            n = ((SG_plane *) Geometry)->Normal;
            break;

        case SG_QUADRIC:
            o = Geometry_Transformation.TransformLocation(o);
            n.x = 2.0 * ((SG_quadric *) Geometry)->cxx * o.x
                      + ((SG_quadric *) Geometry)->cxy * o.y
                      + ((SG_quadric *) Geometry)->cxz * o.z
                      + ((SG_quadric *) Geometry)->cx;
            n.y = 2.0 * ((SG_quadric *) Geometry)->cyy * o.y
                      + ((SG_quadric *) Geometry)->cxy * o.x
                      + ((SG_quadric *) Geometry)->cyz * o.z
                      + ((SG_quadric *) Geometry)->cy;
            n.z = 2.0 * ((SG_quadric *) Geometry)->czz * o.z
                      + ((SG_quadric *) Geometry)->cxz * o.x
                      + ((SG_quadric *) Geometry)->cyz * o.y
                      + ((SG_quadric *) Geometry)->cz;
            n = (Geometry_Transformation.invert()).TransformNormal(n);
            break;

        default:
            printf("Unknown object type %li sent to Object::Normal\n", Object_Type);
            break;
        
        }

    n.Normalise(1);

    return n;

    }

// FIXUP: 
// This function SHOULD be unnecessary, but we need it to keep the
// double accuracy consistent and allow proper comparisons...

int CheckTime(double t, double t_min, double t_max) {
    if((t < t_max) && ((t_min == 0) || (t > t_min))) {
        return 1;
        }
    else {
        return 0;
        }
    }

// This function is the workhorse of the raytracer - and it's structure leaves much to be
// desired! Probably the biggest problem (and biggest departure from POV structure is the
// shadow handling here.

Object_Intersection Object::Intersection(Ray &Initial_Ray, double t_min, double t_max, Object *Ray_Parent) {

    Object_Intersection Best_Intersection, Current_Intersection;
    Object *lRay_Parent;
    Vector o, d;
    double a, b, c, t, u, v, discriminant, x1, x2, y1, y2, z1, z2, lt_max;
    long i, j;

    // Put BEFORE bounding, as we know it will intersect if it is the Parent

    // Rays originating on a surface cause massive problems, particularly
    // with glancing blows along the surface. How can we deal with this?
    // Well, one way is to get all the intersections with the
    // surface and see if the later one is further

    // So...we do this:
    // We loop through the surfaces of the object starting at t_max = 1e10
    // so we definitely get the sucker, and then continue and get the first
    // surface whose |t| from zero is greater than it's predecessor. OK...

    if(this == Ray_Parent) {
        Current_Intersection = Intersection(Initial_Ray, t_min, 1e10, NULL);
        if(Current_Intersection.From_Object == NULL) {
            return Current_Intersection; // FIXUP: Really weird if this happens
            }

        do {
            //printf("        surface clipping, t = %16.16f\n", t);
            t = fabs(Current_Intersection.t - t_max);
            lt_max = Current_Intersection.t;
            Current_Intersection = Intersection(Initial_Ray, t_min, lt_max, NULL);
            if(Current_Intersection.From_Object == NULL) {
                // Occurs if the rest of the object is behind the ray
                return Current_Intersection;
                }
            } while(fabs(Current_Intersection.t - t_max) < t); // <= strictly, but this prevents lockup

        //printf("        surface clipping, t = %16.16f\n", t);

        return Current_Intersection;
            
        }

    // DAMN! This is all very well when it's the object casting the shadow on itself, but
	// when its another coincident surface from a different shape, things get ugly!

	// To rectify this, we resort to the quick-and-dirty method of not accepting
	// intersections too close to the surface of a ray-emitting object. After all, if we can
	// see it...

    Best_Intersection.From_Object = NULL;

    // Test bounding...

    // FIXUP: Test bounding

    // Note: the t_min and t_max values are important. t_max allows us to
    // kill surfaces we know are obscured. t_min allows us to look beyond
    // surfaces we know don't exist

    // FIXUP: We can optomise algorithm for triangles greatly...
    // but make sure manual bounding works!

    if(bounded_by.r != 0) { // Object is bounded

        if((Object_Type == SG_TRIANGLE) || (Object_Type == SG_SMOOTH_TRIANGLE)) {

            // Instead of a sphere we use a circle...

            if(Object_Type == SG_TRIANGLE) {
                a = Initial_Ray.Direction * ((SG_triangle *) Geometry)->Normal;
                if(a == 0) {
                    return Best_Intersection;
                    }
                t = ((bounded_by.o - Initial_Ray.Origin.x) * ((SG_triangle *) Geometry)->Normal) / a;
                }
            else {
                // SG_SMOOTH_TRIANGLE
                a = Initial_Ray.Direction * ((SG_smooth_triangle *) Geometry)->Normal;
                if(a == 0) {
                    return Best_Intersection;
                    }
                t = ((bounded_by.o - Initial_Ray.Origin.x) * ((SG_smooth_triangle *) Geometry)->Normal) / a;
                }

            if(!CheckTime(t, t_min, t_max)) {
                return Best_Intersection;
                }

            // Check for circle...

            o = Initial_Ray.Origin.x + (Initial_Ray.Direction * t);

            if( (o - bounded_by.o).sqn() > (bounded_by.r * bounded_by.r) ) {
                // Not even close
                return Best_Intersection;
                }
            

            }
        else {

            a = Initial_Ray.Direction.sqn();
            b = Initial_Ray.Direction * (Initial_Ray.Origin.x - bounded_by.o) * 2.0;
            c = (Initial_Ray.Origin.x - bounded_by.o)*(Initial_Ray.Origin.x - bounded_by.o) - bounded_by.r*bounded_by.r;

            // Line-of-sight bounding:

            discriminant = b*b - 4*a*c;

            if(discriminant <= 0) {
                // Ray doesn't pass through sphere...
                return Best_Intersection;
                }
        
            // FIXUP: Does what follows make it faster or slower?

			

            discriminant = sqrt(discriminant);

            // Depth bounding:

            // Closest...
            t = (-b + discriminant)/(a*2.0);
            if((t < t_min) && (t_min != 0)) {
                // Too far away
                return Best_Intersection;
                }

            // Furthest...
            t = (-b - discriminant)/(a*2.0);
            if(t > t_max) {
                // Too close
                return Best_Intersection;
                }

			/*

			// New, linear testing

			// Closest...
			t = -b/(a * 2.0);
			if(((t + bounded_by.r) < t_min) && (t_min != 0)) {
				// Too far away
				return Best_Intersection;
			}
			// Furthest...
			if((t - bounded_by.r) > t_max) {
				// Too close
				return Best_Intersection;
			}

			*/

            }
        }

    // We are here => passed bounding tests

    Best_Intersection.t = t_min;

    switch(Object_Type) {
        case CSG_UNION:

            //printf("Testing union\n");

            i = 0;

            while(i < ((CSG_union *) Geometry)->Number_Objects) {

                Current_Intersection = ((CSG_union *) Geometry)->Object_List[i]->Intersection(Initial_Ray, Best_Intersection.t, t_max, Ray_Parent);

                // Now, if it has been updated, it means that YES, it is
                // a better choice than Best_Intersection, so

                if(Current_Intersection.From_Object != NULL) {
                    Best_Intersection = Current_Intersection;
                    }

                i++;
                }

            //printf("CSG_Union returns %16.16f\n", Best_Intersection.t);

            break;

        case CSG_MERGE:

            lt_max = t_max;
            lRay_Parent = Ray_Parent;
            i = 0;
            while(i < ((CSG_merge *) Geometry)->Number_Objects) {
                Current_Intersection = ((CSG_merge *) Geometry)->Object_List[i]->Intersection(Initial_Ray, Best_Intersection.t, lt_max, lRay_Parent);
                if(Current_Intersection.From_Object != NULL) {
                    o = Initial_Ray.Origin.x + (Initial_Ray.Direction * Current_Intersection.t);
                    j = 0;
                    while(j < ((CSG_merge *) Geometry)->Number_Objects) {
                        if(i != j) {
                            if( ((CSG_merge *) Geometry)->Object_List[j]->Interior(o) > 0) {
                                break;
                                }
                             }
                        j++;
                        }
                    //printf("Escaped j-loop...\n");
                    if(j == ((CSG_merge *) Geometry)->Number_Objects) {
                        Best_Intersection = Current_Intersection;
                        lt_max = t_max;
                        lRay_Parent = Ray_Parent;
                        i++;
                        }
                    else {
                        lt_max = Current_Intersection.t;
                        lRay_Parent = Current_Intersection.From_Object;
                        }
                    }
                else {
                    lt_max = t_max;
                    lRay_Parent = Ray_Parent;
                    i++;
                    }
                //printf("Continuing intersection tests...\n");
                }
            break;

        case CSG_DIFFERENCE:

            lt_max = t_max;
            lRay_Parent = Ray_Parent;
            i = 0;
            while(i < ((CSG_difference *) Geometry)->Number_Objects) {
                Current_Intersection = ((CSG_difference *) Geometry)->Object_List[i]->Intersection(Initial_Ray, Best_Intersection.t, lt_max, lRay_Parent);
                if(Current_Intersection.From_Object != NULL) {
                    o = Initial_Ray.Origin.x + (Initial_Ray.Direction * Current_Intersection.t);
                    j = 0;
                    while(j < ((CSG_difference *) Geometry)->Number_Objects) {
                        if(i != j) {
                            if(i == 0) {
                                if( ((CSG_difference *) Geometry)->Object_List[j]->Interior(o) > 0) {
                                    break;
                                    }
                                }
                            else {
                                if(j == 0) {
                                    if( ((CSG_difference *) Geometry)->Object_List[j]->Interior(o) < 0) {
                                        break;
                                        }
                                    }
                                else {
                                    if( ((CSG_difference *) Geometry)->Object_List[j]->Interior(o) > 0) {
                                        break;
                                        }
                                    }
                                }
                            }
                        j++;
                        }
                    if(j == ((CSG_difference *) Geometry)->Number_Objects) {
                        Best_Intersection = Current_Intersection;
                        lt_max = t_max;
                        lRay_Parent = Ray_Parent;
                        i++;
                        }
                    else {
                        lt_max = Current_Intersection.t;
                        lRay_Parent = Current_Intersection.From_Object;
                        }
                    }
                else {
                    lt_max = t_max;
                    lRay_Parent = Ray_Parent;
                    i++;
                    }
                }

            break;

        case CSG_INTERSECTION:

            //printf("Testing intersection\n");

            lt_max = t_max;
            lRay_Parent = Ray_Parent;
            i = 0;
            while(i < ((CSG_intersection *) Geometry)->Number_Objects) {
                Current_Intersection = ((CSG_intersection *) Geometry)->Object_List[i]->Intersection(Initial_Ray, Best_Intersection.t, lt_max, lRay_Parent);
                if(Current_Intersection.From_Object != NULL) {
                    o = Initial_Ray.Origin.x + (Initial_Ray.Direction * Current_Intersection.t);
                    j = 0;
                    while(j < ((CSG_intersection *) Geometry)->Number_Objects) {
                        if(i != j) {
                            if( ((CSG_intersection *) Geometry)->Object_List[j]->Interior(o) < 0) {
                                //printf("Breaking j-loop\n");
                                break;
                                }
                             }
                        //printf("Incrementing j...\n");
                        j++;
                        }
                    //printf("Escaped j-loop...\n");
                    if(j == ((CSG_intersection *) Geometry)->Number_Objects) {
                        Best_Intersection = Current_Intersection;
                        lt_max = t_max;
                        lRay_Parent = Ray_Parent;
                        i++;
                        }
                    else {
                        lt_max = Current_Intersection.t;
                        lRay_Parent = Current_Intersection.From_Object;
                        }
                    }
                else {
                    lt_max = t_max;
                    lRay_Parent = Ray_Parent;
                    i++;
                    }
                //printf("Continuing intersection tests...\n");
                }
            //printf("Finished testing intersecion\n");
            break;

        case SG_TRIANGLE:


            // Out with the old! We have done most of the calculations in the
            // bounding routine...

            // We don't need to calculate o.z, so we don't

            u = Geometry_Transformation.a11*o.x
              + Geometry_Transformation.a12*o.y
              + Geometry_Transformation.a13*o.z
              + Geometry_Transformation.a14;
            v = Geometry_Transformation.a21*o.x
              + Geometry_Transformation.a22*o.y
              + Geometry_Transformation.a23*o.z
              + Geometry_Transformation.a24;

            if( (u > 0) && (v > 0) && (u + v < 1) ) {

                Best_Intersection.From_Object = this;
                Best_Intersection.t = t;

                }

            break;


        case SG_SMOOTH_TRIANGLE:

            // Out with the old! We have done most of the calculations in the
            // bounding routine...

            // We don't need to calculate o.z, so don't

            u = Geometry_Transformation.a11*o.x
              + Geometry_Transformation.a12*o.y
              + Geometry_Transformation.a13*o.z
              + Geometry_Transformation.a14;
            v = Geometry_Transformation.a21*o.x
              + Geometry_Transformation.a22*o.y
              + Geometry_Transformation.a23*o.z
              + Geometry_Transformation.a24;

            if( (u > 0) && (v > 0) && (u + v < 1) ) {

                Best_Intersection.From_Object = this;
                Best_Intersection.t = t;

                ((SG_smooth_triangle *) Geometry)->u = u;
                ((SG_smooth_triangle *) Geometry)->v = v;

                }

            break;

        case SG_SPHERE:
            o = Geometry_Transformation.TransformLocation(Initial_Ray.Origin.x);
            d = Geometry_Transformation.TransformDirection(Initial_Ray.Direction);
            
            a = d.sqn();
            b = (o * d) * 2.0;
            c = o.sqn() - 1;

            discriminant = b*b - 4*a*c;
            
            if(discriminant > 0) {

                discriminant = sqrt(discriminant);
                t = (-b + discriminant) / (2.0 * a);

                if( CheckTime(t, t_min, t_max) ) {
                    Best_Intersection.From_Object = this;
                    Best_Intersection.t = t;
                    }
                else {
                    t = (-b - discriminant) / (2.0 * a);
        
                    if( CheckTime(t, t_min, t_max) ) {
                        Best_Intersection.From_Object = this;
                        Best_Intersection.t = t;
                        }
                    }
                }

            break;

        case SG_BOX:
            o = Geometry_Transformation.TransformLocation(Initial_Ray.Origin.x);
            d = Geometry_Transformation.TransformDirection(Initial_Ray.Direction);

            // FIXUP: There has to be a better way...

            if(d.x != 0) {

                // d.x != 0

                if(d.y != 0) {

                    // d.x != 0, d.y != 0

                    if(d.z != 0) {
 
                        // d.x != 0, d.y != 0, d.z != 0

                        x1 =    - o.x /d.x;
                        x2 = (1 - o.x)/d.x;
                        y1 =    - o.y /d.y;
                        y2 = (1 - o.y)/d.y;
                        z1 =    - o.z /d.z;
                        z2 = (1 - o.z)/d.z;

                        if(x1 < x2) {
                             t = x1;
                            x1 = x2;
                            x2 = t;
                            }
                        if(y1 < y2) {
                             t = y1;
                            y1 = y2;
                            y2 = t;
                            }
                        if(z1 < z2) {
                             t = z1;
                            z1 = z2;
                            z2 = t;
                            }

                        if( (y1 > x1) && (z1 > x1) && (x1 > y2) && (x1 > z2)) {
                            if(CheckTime(x1, t_min, t_max)) {
                                Best_Intersection.t = x1;
                                Best_Intersection.From_Object = this;
                                break;
                                }
                            }

                        if( (x1 > y1) && (z1 > y1) && (y1 > x2) && (y1 > z2)) {
                            if(CheckTime(y1, t_min, t_max)) {
                                Best_Intersection.t = y1;
                                Best_Intersection.From_Object = this;
                                break;
                                }
                            }

                        if( (x1 > z1) && (y1 > z1) && (z1 > x2) && (z1 > y2)) {
                            if(CheckTime(z1, t_min, t_max)) {
                                Best_Intersection.t = z1;
                                Best_Intersection.From_Object = this;
                                break;
                                }
                            }

                        if( (y1 > x2) && (z1 > x2) && (x2 > y2) && (x2 > z2)) {
                            if(CheckTime(x2, t_min, t_max)) {
                                Best_Intersection.t = x2;
                                Best_Intersection.From_Object = this;
                                break;
                                }
                            }

                        if( (x1 > y2) && (z1 > y2) && (y2 > x2) && (y2 > z2)) {
                            if(CheckTime(y2, t_min, t_max)) {
                                Best_Intersection.t = y2;
                                Best_Intersection.From_Object = this;
                                break;
                                }
                            }

                        if( (x1 > z2) && (y1 > z2) && (z2 > x2) && (z2 > y2)) {
                            if(CheckTime(z2, t_min, t_max)) {
                                Best_Intersection.t = z2;
                                Best_Intersection.From_Object = this;
                                break;
                                }
                            }


                        break;

                        }
                    else {

                        // d.x != 0, d.y != 0, d.z == 0

                        if((o.z > 0) && (o.z < 1)) {

                            x1 =    - o.x /d.x;
                         x2 = (1 - o.x)/d.x;
                            y1 =    - o.y /d.y;
                            y2 = (1 - o.y)/d.y;

                            if(x1 < x2) {
                                 t = x1;
                                x1 = x2;
                                x2 = t;
                                }
                            if(y1 < y2) {
                                 t = y1;
                                y1 = y2;
                                y2 = t;
                                }

                            if( (y1 > x1) && (x1 > y2) ) {
                                if(CheckTime(x1, t_min, t_max)) {
                                    Best_Intersection.t = x1;
                                    Best_Intersection.From_Object = this;
                                    break;
                                    }
                                }

                            if( (x1 > y1) && (y1 > x2) ) {
                                if(CheckTime(y1, t_min, t_max)) {
                                    Best_Intersection.t = y1;
                                    Best_Intersection.From_Object = this;
                                    break;
                                    }
                                }

                            if( (y1 > x2) && (x2 > y2) ) {
                                if(CheckTime(x2, t_min, t_max)) {
                                    Best_Intersection.t = x2;
                                    Best_Intersection.From_Object = this;
                                    break;
                                    }
                                }

                            if( (x1 > y2) && (y2 > x2) ) {
                                if(CheckTime(y2, t_min, t_max)) {
                                    Best_Intersection.t = y2;
                                    Best_Intersection.From_Object = this;
                                    break;
                                    }
                                }
                            }
                        break;

                        }
                    }
                else {

                    // d.x != 0, d.y == 0

                    if(d.z != 0) {

                        // d.x != 0, d.y == 0, d.z != 0

                        if((o.y > 0) && (o.y < 1)) {

                            x1 =    - o.x /d.x;
                            x2 = (1 - o.x)/d.x;
                            z1 =    - o.z /d.z;
                            z2 = (1 - o.z)/d.z;

                            if(x1 < x2) {
                                 t = x1;
                                x1 = x2;
                                x2 = t;
                                }
                            if(z1 < z2) {
                                 t = z1;
                                z1 = z2;
                                z2 = t;
                                }

                            if( (z1 > x1) && (x1 > z2) ) {
                                if(CheckTime(x1, t_min, t_max)) {
                                    Best_Intersection.t = x1;
                                    Best_Intersection.From_Object = this;
                                    break;
                                    }
                                }

                            if( (x1 > z1) && (z1 > x2) ) {
                                if(CheckTime(z1, t_min, t_max)) {
                                    Best_Intersection.t = z1;
                                    Best_Intersection.From_Object = this;
                                    break;
                                    }
                                }

                            if( (z1 > x2) && (x2 > z2) ) {
                                if(CheckTime(x2, t_min, t_max)) {
                                    Best_Intersection.t = x2;
                                    Best_Intersection.From_Object = this;
                                    break;
                                    }
                                }

                            if( (x1 > z2) && (z2 > x2) ) {
                                if(CheckTime(z2, t_min, t_max)) {
                                    Best_Intersection.t = z2;
                                    Best_Intersection.From_Object = this;
                                    break;
                                    }
                                }
                            }
                        break;
                        
                        }
                    else {

                        // d.x != 0, d.y == 0, d.z == 0

                        if( (o.y > 0) && (o.z > 0) && (o.y < 1) && (o.z < 1)) {

                            x1 =    - o.x /d.x;
                            x2 = (1 - o.x)/d.x;

                            if(x1 < x2) {
                                 t = x1;
                                x1 = x2;
                                x2 = t;
                                }

                            if(CheckTime(x1, t_min, t_max)) {
                                Best_Intersection.t = x1;
                                Best_Intersection.From_Object = this;
                                break;
                                }

                            if(CheckTime(x2, t_min, t_max)) {
                                Best_Intersection.t = x1;
                                Best_Intersection.From_Object = this;
                                break;
                                }
                            
                            }

                        break;
                        }


                    }
                }
            else {

                // d.x == 0

                if(d.y != 0) {

                    // d.x == 0, d.y != 0

                    if(d.z != 0) {

                        // d.x == 0, d.y != 0, d.z != 0

                        if((o.x > 0) && (o.x < 1)) {

                            y1 =    - o.y /d.y;
                            y2 = (1 - o.y)/d.y;
                            z1 =    - o.z /d.z;
                            z2 = (1 - o.z)/d.z;

                            if(y1 < y2) {
                                 t = y1;
                                y1 = y2;
                                y2 = t;
                                }
                            if(z1 < z2) {
                                 t = z1;
                                z1 = z2;
                                z2 = t;
                                }

                            if( (z1 > y1) && (y1 > z2) ) {
                                if(CheckTime(y1, t_min, t_max)) {
                                    Best_Intersection.t = y1;
                                    Best_Intersection.From_Object = this;
                                    break;
                                    }
                                }

                            if( (y1 > z1) && (z1 > y2) ) {
                                if(CheckTime(z1, t_min, t_max)) {
                                    Best_Intersection.t = z1;
                                    Best_Intersection.From_Object = this;
                                    break;
                                    }
                                }

                            if( (z1 > y2) && (y2 > z2) ) {
                                if(CheckTime(y2, t_min, t_max)) {
                                    Best_Intersection.t = y2;
                                    Best_Intersection.From_Object = this;
                                    break;
                                    }
                                }

                            if( (y1 > z2) && (z2 > y2) ) {
                                if(CheckTime(z2, t_min, t_max)) {
                                    Best_Intersection.t = z2;
                                    Best_Intersection.From_Object = this;
                                    break;
                                    }
                                }
                            }

                        break;
                        }
                    else {

                        // d.x == 0, d.y != 0, d.z == 0

                        if( (o.x > 0) && (o.z > 0) && (o.x < 1) && (o.z < 1)) {

                            y1 =    - o.y /d.y;
                            y2 = (1 - o.y)/d.y;

                            if(y1 < y2) {
                                 t = y1;
                                y1 = y2;
                                y2 = t;
                                }

                            if(CheckTime(y1, t_min, t_max)) {
                                Best_Intersection.t = y1;
                                Best_Intersection.From_Object = this;
                                break;
                                }

                            if(CheckTime(y2, t_min, t_max)) {
                                Best_Intersection.t = y1;
                                Best_Intersection.From_Object = this;
                                break;
                                }
                            
                            }

                        break;
                        }

                    }
                else {

                    // d.x == 0, d.y == 0

                    if(d.z != 0) {

                        // d.x == 0, d.y == 0, d.z != 0

                        if( (o.x > 0) && (o.y > 0) && (o.x < 1) && (o.y < 1)) {

                            z1 =    - o.z /d.z;
                            z2 = (1 - o.z)/d.z;

                            if(z1 < z2) {
                                 t = z1;
                                z1 = z2;
                                z2 = t;
                                }

                            if(CheckTime(z1, t_min, t_max)) {
                                Best_Intersection.t = z1;
                                Best_Intersection.From_Object = this;
                                break;
                                }

                            if(CheckTime(z2, t_min, t_max)) {
                                Best_Intersection.t = z1;
                                Best_Intersection.From_Object = this;
                                break;
                                }
                            
                            }

                        break;
                        }
                    else {

                        // d.x == 0, d.y == 0, d.z == 0

                        // Impossible! ( I hope )

                        printf("Object::Intersection [box] Null ray detected (THIS IS IMPOSSIBLE)\n");

                        break;
                        }
                    }
                }

            // As only small parts are run for a given situation,
            // is fast (hopefully

            break;

        case SG_CYLINDER:

            // printf("SG_C entry ");
            
            o = Geometry_Transformation.TransformLocation(Initial_Ray.Origin.x);
            d = Geometry_Transformation.TransformDirection(Initial_Ray.Direction);

            // FIXUP: Improve ?

            if((d.y == 0) && (d.z == 0)) {
                // parallel to axis

                if( ((SG_cylinder *) Geometry)->open ) {

                    if((o.y*o.y + o.z*o.z) < 1) {
                        t = (1-o.x)/d.x;
                        if(CheckTime(t, t_min, t_max)) {
                            Best_Intersection.t = t;
                            Best_Intersection.From_Object = this;
                            // printf("case 1");
                            break;
                            }
                        t = -o.x/d.x;
                        if(CheckTime(t, t_min, t_max)) {
                            Best_Intersection.t = t;
                            Best_Intersection.From_Object = this;
                            // printf("case 2");
                            break;
                            }
                        }
                    }
                break;
                }

            a = d.y*d.y + d.z*d.z;
            b = (d.y*o.y + d.z*o.z) * 2.0;
            c = o.y*o.y + o.z*o.z - 1.0;

            discriminant = b*b - 4*a*c;

            if(discriminant > 0) {
                discriminant = sqrt(discriminant);
                z1 = (-b + discriminant)/(2.0*a);
                x1 = o.x + d.x * z1;
                if((x1 > 0) && (x1 < 1) && CheckTime(z1, t_min, t_max) ) {
                    Best_Intersection.t = z1;
                    Best_Intersection.From_Object = this;
                    // printf("case 3");
                    break;
                    }

                z2 = (-b - discriminant)/(2.0*a);
                x2 = o.x + d.x * z2;


                if( ((SG_cylinder *) Geometry)->open ) {

                    // Hmm, a possible cap...
            
                    // Order is important here, determines first strike on
                    // ray going through both caps.

                    if((x1 > 1) && (x2 < 1)) {
                        z1 = (1-o.x)/d.x;
                        if(CheckTime(z1, t_min, t_max)) {
                            Best_Intersection.t = z1;
                            Best_Intersection.From_Object = this;
                            // printf("case 3");
                            return Best_Intersection;
                            }
                        }
        
                    if((x1 < 0) && (x2 > 0)) {
                        z1 = -o.x/d.x;
                        if(CheckTime(z1, t_min, t_max)) {
                            Best_Intersection.t = z1;
                            Best_Intersection.From_Object = this;
                            // printf("case 4");
                            break;
                            }
                        }
                
                    if((x1 < 1) && (x2 > 1)) {
                        z1 = (1-o.x)/d.x;
                        if(CheckTime(z1, t_min, t_max)) {
                            Best_Intersection.t = z1;
                            Best_Intersection.From_Object = this;
                            // printf("case 5");
                            break;
                            }
                        }
        
                    if((x1 > 0) && (x2 < 0)) {
                        z1 = -o.x/d.x;
                        if(CheckTime(z1, t_min, t_max)) {
                            Best_Intersection.t = z1;
                            Best_Intersection.From_Object = this;
                            // printf("case 6");
                            break;
                            }
                        }

                    }

                // OK, so try back surface...

                if((x2 > 0) && (x2 < 1) && CheckTime(z2, t_min, t_max)) {
                    Best_Intersection.t = z2;
                    Best_Intersection.From_Object = this;
                    // printf("case 7");
                    break;
                    }

                }

            break;

        case SG_CONE:
            o = Geometry_Transformation.TransformLocation(Initial_Ray.Origin.x);
            d = Geometry_Transformation.TransformDirection(Initial_Ray.Direction);

            // FIXUP:

            z1 = ((SG_cone *) Geometry)->Radius_1;
            z2 = ((SG_cone *) Geometry)->Radius_2;

            a = d.y*d.y
              + d.z*d.z
              - d.x*d.x*(z2-z1)*(z2-z1);

            b = 2*d.y*o.y
              + 2*d.z*o.z
              - 2*d.x*o.x*(z2-z1)*(z2-z1)
              - 2*d.x*(z2-z1)*z1;

            c = o.y*o.y
              + o.z*o.z
              - o.x*o.x*(z2-z1)*(z2-z1)
              - z1*z1
              - 2*o.x*(z2-z1)*z1;

            discriminant = b*b - 4*a*c;


            if(discriminant > 0) {

                discriminant = sqrt(discriminant);
                y1 = (-b + discriminant) / (2.0 * a);

                x1 = o.x + d.x * y1;

                if((x1 > 0) && (x1 < 1)) {
                    if(CheckTime(y1, t_min, t_max)) {
                        Best_Intersection.t = y1;
                        Best_Intersection.From_Object = this;
                        break;
                        }
                    }

                y2 = (-b - discriminant) / (2.0 * a);

                x2 = o.x + d.x * y2;


                if( ((SG_cone *) Geometry)->open ) {

                    // Order is important here...

                    // FIXUP: Do we need radius checks here as well?

                    if((x1 > 1) && (x2 < 1)) {
                        y1 = (1-o.x)/d.x;
                        if(CheckTime(y1, t_min, t_max)) {
                            if(((o.y+d.y*y1)*(o.y+d.y*y1)+(o.z+d.z*y1)*(o.z+d.z*y1)) < z2*z2) {
                                Best_Intersection.t = y1;
                                Best_Intersection.From_Object = this;
                                break;
                                }
                            }
                        }
                
                    if((x1 < 0) && (x2 > 0)) {
                        // Bottom cap...
                        y1 = -o.x/d.x;
                        if(CheckTime(y1, t_min, t_max)) {
                            if(((o.y+d.y*y1)*(o.y+d.y*y1)+(o.z+d.z*y1)*(o.z+d.z*y1)) < z1*z1) {
                                Best_Intersection.t = y1;
                                Best_Intersection.From_Object = this;
                                break;
                                }
                            }
                        }

                    if((x1 < 1) && (x2 > 1)) {
                        y1 = (1-o.x)/d.x;
                        if(CheckTime(y1, t_min, t_max)) {
                            if(((o.y+d.y*y1)*(o.y+d.y*y1)+(o.z+d.z*y1)*(o.z+d.z*y1)) < z2*z2) {
                                Best_Intersection.t = y1;
                                Best_Intersection.From_Object = this;
                                break;
                                }
                            }
                        }
                
                    if((x1 > 0) && (x2 < 0)) {
                        // Bottom cap...
                        y1 = -o.x/d.x;
                        if(CheckTime(y1, t_min, t_max)) {
                            if(((o.y+d.y*y1)*(o.y+d.y*y1)+(o.z+d.z*y1)*(o.z+d.z*y1)) < z1*z1) {
                                Best_Intersection.t = y1;
                                Best_Intersection.From_Object = this;
                                break;
                                }
                            }
                        }

                    }

                // Far wall...

                if((x2 > 0) && (x2 < 1)) {
                    if(CheckTime(y2, t_min, t_max)) {
                        Best_Intersection.t = y2;
                        Best_Intersection.From_Object = this;
                        break;
                        }
                    }

                // This is not the be-all and end-all however:
                // It can be the case that the ray intersects the surface
                // twice then barrels off to infinity through the
                // caps, never touching the sides...
                // In this case x1 & x2 are on the same side
                // of the cone...

                if((d.x != 0) && ((SG_cone *) Geometry)->open ) {

                    y1 = -o.x/d.x;
                    y2 = (1-o.x)/d.x;

                    if(x1 < x2) {
                        // Ray is moving up x axis
                        // Check radius
                        if( ((o.y+d.y*y1)*(o.y+d.y*y1)+(o.z+d.z*y1)*(o.z+d.z*y1)) < z1*z1) {
                            if(CheckTime(y1, t_min, t_max)) {
                                Best_Intersection.t = y1;
                                Best_Intersection.From_Object = this;
                                break;
                                }
                            }
                        if( ((o.y+d.y*y2)*(o.y+d.y*y2)+(o.z+d.z*y2)*(o.z+d.z*y2)) < z2*z2) {
                            if(CheckTime(y2, t_min, t_max)) {
                                Best_Intersection.t = y2;
                                Best_Intersection.From_Object = this;
                                break;
                                }
                            }
                        break;
                        }
                    else {
                        // Ray is moving down x axis
                        // Check radius
                        if( ((o.y+d.y*y2)*(o.y+d.y*y2)+(o.z+d.z*y2)*(o.z+d.z*y2)) < z2*z2) {
                            if(CheckTime(y2, t_min, t_max)) {
                                Best_Intersection.t = y2;
                                Best_Intersection.From_Object = this;
                                break;
                                }
                            }
                        if( ((o.y+d.y*y1)*(o.y+d.y*y1)+(o.z+d.z*y1)*(o.z+d.z*y1)) < z1*z1) {
                            if(CheckTime(y1, t_min, t_max)) {
                                Best_Intersection.t = y1;
                                Best_Intersection.From_Object = this;
                                break;
                                }
                            }
                        break;
                        }
                    }
                }


            break;

        case SG_PLANE:
            o = Geometry_Transformation.TransformLocation(Initial_Ray.Origin.x);
            d = Geometry_Transformation.TransformDirection(Initial_Ray.Direction);

            if(d.x != 0) {
                t = -o.x/d.x;
                if( CheckTime(t, t_min, t_max) ) {
                    Best_Intersection.From_Object = this;
                    Best_Intersection.t = t;
                    }
                }

            break;

        case SG_QUADRIC:
            o = Geometry_Transformation.TransformLocation(Initial_Ray.Origin.x);
            d = Geometry_Transformation.TransformDirection(Initial_Ray.Direction);

            // As you can guess, this is NOT a fast shape

            a = ((SG_quadric *) Geometry)->cxx * d.x * d.x
              + ((SG_quadric *) Geometry)->cyy * d.y * d.y
              + ((SG_quadric *) Geometry)->czz * d.z * d.z
              + ((SG_quadric *) Geometry)->cxy * d.x * d.y
              + ((SG_quadric *) Geometry)->cxz * d.y * d.z
              + ((SG_quadric *) Geometry)->cyz * d.y * d.z;
            b = ((SG_quadric *) Geometry)->cxx * d.x * o.x * 2
              + ((SG_quadric *) Geometry)->cyy * d.y * o.y * 2
              + ((SG_quadric *) Geometry)->czz * d.z * o.z * 2
              + ((SG_quadric *) Geometry)->cxy * d.x * o.y
              + ((SG_quadric *) Geometry)->cxy * d.y * o.x
              + ((SG_quadric *) Geometry)->cxz * d.x * o.z
              + ((SG_quadric *) Geometry)->cxz * d.z * o.x
              + ((SG_quadric *) Geometry)->cyz * d.y * o.z
              + ((SG_quadric *) Geometry)->cyz * d.z * o.y
              + ((SG_quadric *) Geometry)->cx  * d.x
              + ((SG_quadric *) Geometry)->cy  * d.y
              + ((SG_quadric *) Geometry)->cz  * d.z;
            c = ((SG_quadric *) Geometry)->cxx * o.x * o.x
              + ((SG_quadric *) Geometry)->cyy * o.y * o.y
              + ((SG_quadric *) Geometry)->czz * o.z * o.z
              + ((SG_quadric *) Geometry)->cxy * o.x * o.y
              + ((SG_quadric *) Geometry)->cxz * o.x * o.z
              + ((SG_quadric *) Geometry)->cyz * o.y * o.z
              + ((SG_quadric *) Geometry)->cx
              + ((SG_quadric *) Geometry)->cy
              + ((SG_quadric *) Geometry)->cz
              + ((SG_quadric *) Geometry)->c;

            discriminant = b*b - 4*a*c;
            
            if(discriminant > 0) {

                discriminant = sqrt(discriminant);
                t = (-b + discriminant) / (2.0 * a);

                if( CheckTime(t, t_min, t_max) ) {
                    Best_Intersection.From_Object = this;
                    Best_Intersection.t = t;
                    }
                else {
                    t = (-b - discriminant) / (2.0 * a);
        
                    if( CheckTime(t, t_min, t_max) ) {
                        Best_Intersection.From_Object = this;
                        Best_Intersection.t = t;
                        }
                    }
                }

            break;

        default:
            printf("How on earth could an unregistered shape reach the Object::Intersection routine ?! (THIS IS IMPOSSIBLE)\n");
            exit(EXIT_FAILURE);
            break;
        }

    return Best_Intersection;

    }

Frame_Intersection Frame::Intersection(Ray &Initial_Ray, Object *Ray_Parent) {
    Frame_Intersection Return_Intersection;
    Object_Intersection Best_Intersection, Current_Intersection;

    long i;

    i = 0;

    Best_Intersection.From_Object = NULL;

    while(i < Number_Objects) {

        // OK, we run through all the base level objects

		// FIXUP: There is a quick and dirty solution here, with t_max nonzero! This means
		// that surfaces not belonging to the parent surface (should it exist) that lie
		// within this difference will be clipped back, possibly resulting in shadow speckle
		// in rare cases - but believe me, it's the lesser of two evils, because otherwise
		// all coincident surfaces speckle!

		// Using a 1e-6 (a micron) as surface clip - this is about the wavelength of light
		// anyway, so diffraction starts to mess stuff up at this scale as it is

        Current_Intersection = Object_List[i]->Intersection(Initial_Ray, 0, -1e-6, Ray_Parent);

        if(Best_Intersection.From_Object == NULL) {
            Best_Intersection = Current_Intersection;
            }
        else {
            if(Current_Intersection.From_Object != NULL) {
                if(Current_Intersection.t < 0) {
                    if(Current_Intersection.t > Best_Intersection.t) {
                        Best_Intersection = Current_Intersection;
                        }
                    }
                }
            }

        i++;
        }

    Return_Intersection.From_Frame = this;
    Return_Intersection.From_Ray = Initial_Ray;
    Return_Intersection.From_Object = Best_Intersection.From_Object;

    if(Return_Intersection.From_Object == NULL) {
        return Return_Intersection;
        }

    Return_Intersection.Hit.x = Initial_Ray.Origin.x + (Initial_Ray.Direction * Best_Intersection.t);
    Return_Intersection.Hit.t = Initial_Ray.Origin.t + Best_Intersection.t;

    return Return_Intersection;
    }

fColour Scene::GetColour(Frame_Intersection &Best_Intersection, long Recursion) {

    Frame_Intersection Current_Intersection;
    Ray Current_Ray;
    Event Current_Hit, Shadow_Hit;
    Vector n, d, o, r;
    fColour c, oc, lc, lt, sc;
    long i, j;

    // NOTE:
    // GetColour() calls should NEVER be doppler shifted, as it returns
    // a preshifted colour

    if(Recursion > Globals.max_trace_level) {
        return fColour( 0, 0, 0, 0, 0 );
        }

    if(Best_Intersection.From_Object == NULL) {

        Current_Ray = Best_Intersection.From_Ray.LorentzTransform(Best_Intersection.From_Frame->Velocity, SkySphereVelocity );

        oc = SkySphereSurface.GetPigment(
                                Event( -Current_Ray.Direction.x,
                                          -Current_Ray.Direction.y,
                                          -Current_Ray.Direction.z,
                                           Current_Ray.Origin.t )
                                ).Doppler(Current_Ray.doppler * SkySphereSurface.P.doppler) * Current_Ray.intensity * SkySphereSurface.P.intensity;

        return oc;

        }


    oc = Best_Intersection.From_Object->GetPigment(Best_Intersection.Hit).Doppler2(Best_Intersection.From_Ray.doppler * Best_Intersection.From_Object->Surface->P.doppler);

    // Ambient light...

    c = oc * (Best_Intersection.From_Object->Surface->F.ambient.Doppler2(Best_Intersection.From_Ray.doppler) * Best_Intersection.From_Ray.intensity * Best_Intersection.From_Object->Surface->P.intensity);
    c.f = oc.f; // FIXUP: ? Necessary ?
    c.t = oc.t;

    // Get surface normal

    n = Best_Intersection.From_Object->Normal(Best_Intersection.Hit.x);

    // May have to modify this for refraction...

    if(n * Best_Intersection.From_Ray.Direction < 0) {
        n.x = -n.x;
        n.y = -n.y;
        n.z = -n.z;
        }

    n += Best_Intersection.From_Object->GetNormal(Best_Intersection.Hit);
    n.Normalise(1);

    // Intensity plays no part in the reflection/refraction as:
    // The ray-size


    // Reflection...

    if( (Best_Intersection.From_Object->Surface->F.reflection.r != 0) || (Best_Intersection.From_Object->Surface->F.reflection.g != 0) || (Best_Intersection.From_Object->Surface->F.reflection.b || 0) ) {

        Current_Ray = Best_Intersection.From_Ray;

        // Now -n is good, so it should be like -n

        // When n & r are ||, we want cr = -r
     
		// When n & r are _|_, we want cr = r

        Current_Ray.Direction = Best_Intersection.From_Ray.Direction - (n * 2.0 * (n * Best_Intersection.From_Ray.Direction));

        Current_Ray.Origin = Best_Intersection.Hit;

        Current_Intersection = CastRay(Current_Ray, Best_Intersection.From_Frame->Velocity, Best_Intersection.From_Object);

        // FIXUP: Reflection temporarily equispectral?

		// FIXUP: Jazz this up...

        c += GetColour(Current_Intersection, Recursion + 1)
           * Best_Intersection.From_Object->Surface->F.reflection
                .Doppler2(Best_Intersection.From_Ray.doppler)
             ;
        }

    if( (oc.t != 0) || (oc.f != 0) ) {

		// Transmit the ray

        Current_Ray = Best_Intersection.From_Ray;

        // We can mess with refraction later...

        Current_Ray.Origin = Best_Intersection.Hit;

        Current_Intersection = CastRay(Current_Ray, Best_Intersection.From_Frame->Velocity, Best_Intersection.From_Object);

        if(oc.t != 0) {
            c += GetColour(Current_Intersection, Recursion + 1) * oc.t;
            }
        if(oc.f != 0) {
            c += GetColour(Current_Intersection, Recursion + 1) * oc * oc.f;
            }
        }

    // Light_Sourcing...

    // FIXUP: his is getting RIDICULOUS - make some local pointers

    i = 0;
    while(i < Number_Frames) {

        // Work in local frame...

        Current_Hit = Best_Intersection.Hit.LorentzTransform(Best_Intersection.From_Frame->Velocity, Frame_List[i]->Velocity);

        j = 0;

        while(j < Frame_List[i]->Number_Light_Sources) {

            o = ((Vector *) &Frame_List[i]->Light_Source_List[j]->Geometry_Transformation)[3];

            o.x = -o.x;
            o.y = -o.y;
            o.z = -o.z;

            d = Current_Hit.x - o;

            // Construct ray in new frame moving back in time FROM surface,
            // TO light_source

            // We need a new way to express the ray...
            // starting in the OBJECT FRAME

            Current_Ray.Origin = Current_Hit;
            Current_Ray.Direction = d / (d.nrm());

            // FIXUP: This is a horrible, inefficient mess!

            Current_Ray = Current_Ray.LorentzTransform(Frame_List[i]->Velocity, Best_Intersection.From_Frame->Velocity);

            Current_Ray.intensity = Best_Intersection.From_Ray.intensity;
            Current_Ray.doppler = Best_Intersection.From_Ray.doppler;

            Current_Ray = Current_Ray.LorentzTransform(Best_Intersection.From_Frame->Velocity, Frame_List[i]->Velocity);

            // Now, trace the ray... remembering to include the
            // object as parent to prevent 'speckle'

            // We need to allow for transparent surfaces...

            lt = fColour( 1, 1, 1, 0, 0);

            Current_Intersection.From_Object = NULL;

  //          printf("Casting shadow ray...\n");

            if( Frame_List[i]->Light_Source_List[j]->no_shadow ) {

    //            printf("First shot\n");

                Current_Intersection = CastRay(Current_Ray, Frame_List[i]->Velocity, Best_Intersection.From_Object);

                while( (Current_Intersection.From_Object != NULL) && ((Current_Hit.t - Current_Intersection.Hit.t) < d.nrm()) ) {

                    if(Current_Intersection.From_Object->no_shadow) {

                        if(Current_Intersection.From_Object->Surface->P.map_type >= 0) {
                            Shadow_Hit = Current_Intersection.Hit.LorentzTransform(Frame_List[i]->Velocity, Current_Intersection.From_Frame->Velocity);
                            sc = Current_Intersection.From_Object->GetPigment(Shadow_Hit).Doppler2(Current_Ray.doppler * Current_Intersection.From_Object->Surface->P.doppler) * Current_Ray.intensity;
							//sc = Current_Intersection.From_Object->GetPigment(Current_Intersection.Hit).Doppler2(Current_Ray.doppler * Current_Intersection.From_Object->Surface->P.doppler) * Current_Ray.intensity;
                            }
                        else {
                            sc = Current_Intersection.From_Object->Surface->P.colour.Doppler2(Current_Ray.doppler * Current_Intersection.From_Object->Surface->P.doppler) * Current_Ray.intensity;
                            }

                        lt *= ((fColour( 1, 1, 1, 0, 0 ) * sc.t) + (sc * sc.f));

                        if((lt.r <= 0) && (lt.g <= 0) && (lt.b <= 0)) {

                            // If light is zero, stop the ray

                            break;
                            }
                        }

                    Current_Ray.Origin = Current_Intersection.Hit;
                    
      //              printf("Subsequent shot\n");
                    Current_Intersection = CastRay(Current_Ray, Frame_List[i]->Velocity, Current_Intersection.From_Object);

                    }
                }

//            printf("Completed shadow ray\n");

            if( (Current_Intersection.From_Object == NULL) || ((Current_Hit.t - Current_Intersection.Hit.t) > d.nrm()) ) {

                // We have a unobstructed path to the light_source, so

                // FIXUP: This is dangerous messing around with Current_Ray...

                Current_Ray.Direction = Current_Ray.Direction.LorentzTransform(Frame_List[i]->Velocity, Best_Intersection.From_Frame->Velocity);

                if(Current_Ray.Direction * n < 0) {

                    // OK, we give the event to the light source to allow for
                    // time-dependant textures

                    lc = Frame_List[i]->Light_Source_List[j]->GetPigment(
                                                                Event( o.x + (d.x / d.nrm()),
                                                                          o.y + (d.y / d.nrm()),
                                                                          o.z + (d.z / d.nrm()),
                                                                          Current_Hit.t - d.nrm() )  
					                               ).Doppler(Current_Ray.doppler);

                    // FIXUP: for intensity...
                    Current_Ray.intensity *= Frame_List[i]->Light_Source_List[j]->Surface->P.intensity;

                    if( ((Light_Source *) Frame_List[i]->Light_Source_List[j]->Geometry)->fade_power != 0) {
                        Current_Ray.intensity *= 2 / ( 1.0 + pow(d.nrm()/((Light_Source *) Frame_List[i]->Light_Source_List[j]->Geometry)->fade_distance, ((Light_Source *) Frame_List[i]->Light_Source_List[j]->Geometry)->fade_power) );
                        }

                    lc *= Current_Ray.intensity;

                    r.x = -Current_Ray.Direction.x;
                    r.y = -Current_Ray.Direction.y;
                    r.z = -Current_Ray.Direction.z;

                    if(Best_Intersection.From_Object->Surface->F.diffuse != 0) {
                        if(Best_Intersection.From_Object->Surface->F.brilliance == 1) {
                            c += lt * (oc * lc) * (n * r) * Best_Intersection.From_Object->Surface->F.diffuse * (1 - oc.t - oc.f);
                            }
                        else {
                            c += lt * (oc * lc) * pow(n * r, Best_Intersection.From_Object->Surface->F.brilliance) * Best_Intersection.From_Object->Surface->F.diffuse * (1 - oc.t - oc.f);
                            }
                        }

                    if(Best_Intersection.From_Object->Surface->F.specular) {
                        r += Best_Intersection.From_Ray.Direction;
                        r.Normalise(1);
                        if(Best_Intersection.From_Object->Surface->F.metallic) {
                            // FIXUP: Make metallic do something!
                            c += lt * oc * pow(n * r, 1/Best_Intersection.From_Object->Surface->F.roughness) * Best_Intersection.From_Object->Surface->F.specular;
                            }
                        else {
                            c += lt * lc * pow(n * r, 1/Best_Intersection.From_Object->Surface->F.roughness) * Best_Intersection.From_Object->Surface->F.specular;
                            }
                        }

                    }

                }
            j++;
            }
        i++;
        }

    // printf("Completed Ray\n");

    return c;

    }

Frame_Intersection Scene::CastRay(Ray &Initial_Ray, Vector &v, Object *Ray_Parent) {
    Ray Current_Ray;
    Frame_Intersection Current_Intersection, Best_Intersection;
    Event Current_Hit, Best_Hit;
    long i;

    // We now need to check all the frames.

    // printf("Scene::CastRay commencing\n");

    i = 0;

    Best_Intersection.From_Object = NULL;

    if(Initial_Ray.Direction == Vector( 0, 0, 0 ) ) {
        return Best_Intersection;
        }

    while(i < Number_Frames) {
		// Test for any objects in the frame before we start chewing up precious time!
	//	if(Frame_List[i]->Number_Objects > 0) {

        Current_Ray = Initial_Ray.LorentzTransform(v, Frame_List[i]->Velocity);
        // Current_Ray now in the frame frame (!)

        Current_Intersection = Frame_List[i]->Intersection(Current_Ray, Ray_Parent);

        Current_Hit = Current_Intersection.Hit.LorentzTransform(Frame_List[i]->Velocity, v);

        // Make sure the ray gets carried through... for SkySphere
        if(Best_Intersection.From_Object == NULL) {
            Best_Intersection.From_Frame = Current_Intersection.From_Frame;
            Best_Intersection.From_Ray = Current_Intersection.From_Ray;
            }

        if(Current_Intersection.From_Object != NULL) {
            // Hit something somwhere
            if(Best_Intersection.From_Object == NULL) {
                Best_Intersection = Current_Intersection;
                Best_Hit = Current_Hit;
                }
            else {
                if(Current_Hit.t < Initial_Ray.Origin.t) {
                    // Hit occurs before photon recieved
                    if(Current_Hit.t > Best_Hit.t) {
                        // Hit occurs after current best hit
                        Best_Intersection = Current_Intersection;
                        Best_Hit = Current_Hit;
                        }
                    }
                }
            }
	//		}	
        i++;
        }

    // printf("Scene::CastRay completed\n");

    return Best_Intersection;
    }

Ray Camera::GetRay(double u, double v) {
    Ray r;
    double f;

	if((u > Globals.Window_l) && (u < Globals.Window_r) && (v > Globals.Window_t) && (v < Globals.Window_b)) {
		Globals.Doppler = Globals.Window_Doppler;
		Globals.Intensity = Globals.Window_Intensity;
		}
	else {
		Globals.Doppler = Globals.NW_Doppler;
		Globals.Intensity = Globals.NW_Intensity;
		}
    r.doppler = 1.0;
    r.intensity = 1.0;

    switch(Type) {
        case 0:
            // Perspective
            r.Origin = location;
            r.Direction = (direction + right*u + up*v) * -1;
            r.Direction.Normalise(1);
            break;
        case 1:
            // Orthographic
            r.Direction = direction * -1;
            r.Origin.x = location.x + right*u + up*v;
            r.Origin.t = location.t;
            r.Direction.Normalise(1);
            break;
        case 2:
            // Fisheye
            r.Origin = location;

            if( (u == 0) && (v == 0)) {
                r.Direction = direction * -1;
                r.Direction.Normalise(1);
                break;
                }
            else {
                f = sqrt(u*u + v*v);
                if(f > 0.5) {
                    r.Direction = Vector( 0, 0, 0 );
                    break;
                    }

                r.Direction = (direction * cos(f * angle) + (right * (u/f) + up * (v/f)) * sin(f * angle) ) * -1;
                r.Direction.Normalise(1);
                break;
                }
            break;
        case 3:
            r.Origin = location;
            r.Direction = (direction*cos(u * 2.0 * PI) - right*sin(u * 2.0 * PI)) * -cos(v*PI) + up * -sin(v*PI);
            break;
        default:
            printf("Camera::GetRay: Unrecognised camera type! (THIS IS IMPOSSIBLE) \n");
            exit(EXIT_FAILURE);
            break;
        }

    return r;

    }

void Camera::GetCoordinates(Ray r, double &du, double &dv) {
    double f;
    switch(Type) {
        case 0:
            // Perspective
            f = direction * r.Direction;
            if(f >= 0) {
                // The light source is behind the camera
                du = 10;
                dv = 10;
                }
            else {
                du = right * r.Direction;
                dv = up * r.Direction;
				du /= right.sqn();
                dv /= up.sqn();
				du /= f;
                dv /= f;
                }
            break;
        default:
            // FIXUP:
            // No lens flare for these yet!
            // Put pixel outside the range
            du = 10;
            dv = 10;
            break;
        }

	if((du > Globals.Window_l) && (du < Globals.Window_r) && (dv > Globals.Window_t) && (dv < Globals.Window_b)) {
		Globals.Doppler = Globals.Window_Doppler;
		Globals.Intensity = Globals.Window_Intensity;
		}
	else {
		Globals.Doppler = Globals.NW_Doppler;
		Globals.Intensity = Globals.NW_Intensity;
		}
    }

fColour Scene::TraceRay(double u, double v, Camera *C) {
    Frame_Intersection Best_Intersection;
    Ray Current_Ray;
    fColour c;

    Current_Ray = C->GetRay(u, v);

    if(Current_Ray.Direction.sqn() != 0) {

        Best_Intersection = CastRay(Current_Ray, C->Velocity, (Object *) NULL);

        c = GetColour(Best_Intersection, 0);
        c.Clean();
        return c;

        }
    else {
        return fColour( 0, 0, 0, 0, 0 );
        }
    }


fColour Scene::Antialias(fColour c00, fColour c02, fColour c20, fColour c22, double u_0, double v0_, double u_2, double v2_, Camera *C, long Recursion) {
    fColour c01, c10, c11, c12, c21;
    fColour Average;
    long u, v;

    // OK, first, do we need to antialias ?

    if(Recursion >= Globals.Antialias_Depth) {
        return c22;
        }

    if( (fabs(c22.r - c02.r) + fabs(c22.g - c02.g) + fabs(c22.b - c02.b)) < Globals.Antialias_Threshold) {
        if( (fabs(c22.r - c20.r) + fabs(c22.g - c20.g) + fabs(c22.b - c20.b)) < Globals.Antialias_Threshold) {
            if( (fabs(c00.r - c02.r) + fabs(c00.g - c02.g) + fabs(c00.b - c02.b)) < Globals.Antialias_Threshold) {
                if( (fabs(c00.r - c20.r) + fabs(c00.g - c20.g) + fabs(c00.b - c20.b)) < Globals.Antialias_Threshold) {
                    return c22;
                    }
                }
            }
        }

    // So, we DO need to antialias...


    if(Globals.Sampling_Method == 2) {
        c01 = TraceRay((u_0 + u_2)/2.0, v0_, C);
        c10 = TraceRay(u_0, (v0_ + v2_)/2.0, C);
        c11 = TraceRay((u_0 + u_2)/2.0, (v0_ + v2_)/2.0, C);
        c12 = TraceRay(u_2, (v0_ + v2_)/2.0, C);
        c21 = TraceRay((u_0 + u_2)/2.0, v2_, C);

        Average = fColour( 0, 0, 0, 0, 0 );

        Average += Antialias(c00, c01, c10, c11, u_0, v0_, (u_0 + u_2)/2.0, (v0_ + v2_) / 2.0, C, Recursion + 1) * 0.25;

        Average += Antialias(c01, c02, c11, c12, (u_0 + u_2)/2.0, v0_, u_2, (v0_ + v2_) / 2.0, C, Recursion + 1) * 0.25;

        Average += Antialias(c10, c11, c20, c21, u_0, (v0_ + v2_)/2, (u_0 + u_2)/2.0, v2_, C, Recursion + 1) * 0.25;

        Average += Antialias(c11, c12, c21, c22, (u_0 + u_2)/2, (v0_ + v2_)/2, u_2, v2_, C, Recursion + 1) * 0.25;
        }
    else {
        // Non adaptive...
        
        Average = c22;

        v = 0;
        while(v < Globals.Antialias_Depth) {
            u = 0;
            while(u < Globals.Antialias_Depth) {

                if(u || v) {

                    Average += TraceRay(
                        u_2 - (u_2 - u_0)*u/Globals.Antialias_Depth,
                        v2_ - (v2_ - v0_)*v/Globals.Antialias_Depth,
                        C);
                    
                    }

                u++;
                }
            v++;
            }

        if(Globals.Antialias_Depth) {
            Average *= (1/(Globals.Antialias_Depth*Globals.Antialias_Depth));
            }        

        }

    return Average;

    }

void DrawFlare(Bitmap Current_Bitmap, Bitmap LensFlare, fColour lc, long u, long v) {
    double Scale, Width, sx, x, y, dx, dy;
    long u1, u2, v1, v2, iu, iv;
    iColour c, *p;
    
    // OK, we first establish the dimensions of the flare. We take a 100%
    // colour to form a flare of total width 1/4 the minimum image dimension

    // FIXUP: Give global control on flare size

    if(Current_Bitmap.uMax >= Current_Bitmap.vMax) {
        Scale = Current_Bitmap.uMax / 2;
        }
    else {
        Scale = Current_Bitmap.vMax / 2;
        }

    // First the red flare

    Width = sqrt(lc.r) * Scale;

    u1 = (long) ((double) u - Width / 2);
    u2 = (long) ((double) u + Width / 2);
    v1 = (long) ((double) v - Width / 2);
    v2 = (long) ((double) v + Width / 2);

    dx = 1.0 / Width;
    dy = dx;
    
    sx = dx / 2;
    y = dy / 2;

    if(u1 < 0) {
        sx += dx * -u1;
        u1 = 0;
        }
    if(v1 < 0) {
        y += dy * -v1;
        v1 = 0;
        }
    if(u2 > (signed) Current_Bitmap.uMax) {
        u2 = Current_Bitmap.uMax;
        }
    if(v2 > (signed) Current_Bitmap.vMax) {
        v2 = Current_Bitmap.vMax;
        }
    
    // Now we blit it...

    iv = v1;
    while(iv < v2) {
        x = sx;
        iu = u1;
        while(iu < u2) {

            c = LensFlare.GetPixel(x, y, 1).Convert();

            p = Current_Bitmap.Pixel + iu + (iv * Current_Bitmap.uMax);

            if( (long) p->r + c.b < 256 ) { p->r += c.b; } else { p->r = 255; }
            if( (long) p->g + c.r < 256 ) { p->g += c.r; } else { p->g = 255; }
            if( (long) p->b + c.r < 256 ) { p->b += c.r; } else { p->b = 255; }

            x += dx;
            iu++;
            }
        y += dy;
        iv++;
        }

    // Now the green flare

    Width = sqrt(lc.g) * Scale;

    u1 = (long) ((double) u - Width / 2);
    u2 = (long) ((double) u + Width / 2);
    v1 = (long) ((double) v - Width / 2);
    v2 = (long) ((double) v + Width / 2);

    dx = 1.0 / Width;
    dy = dx;
    
    sx = dx / 2;
    y = dy / 2;

    // Smooth if we are inflating the image

    if(u1 < 0) {
        sx += dx * -u1;
        u1 = 0;
        }
    if(v1 < 0) {
        y += dy * -v1;
        v1 = 0;
        }
    if(u2 > (signed) Current_Bitmap.uMax) {
        u2 = Current_Bitmap.uMax;
        }
    if(v2 > (signed) Current_Bitmap.vMax) {
        v2 = Current_Bitmap.vMax;
        }
    
    // Now we blit it...

    iv = v1;
    while(iv < v2) {
        x = sx;
        iu = u1;
        while(iu < u2) {

            c = LensFlare.GetPixel(x, y, 1).Convert();

            p = Current_Bitmap.Pixel + iu + (iv * Current_Bitmap.uMax);

            if( (long) p->r + c.r < 256 ) { p->r += c.r; } else { p->r = 255; }
            if( (long) p->g + c.b < 256 ) { p->g += c.b; } else { p->g = 255; }
            if( (long) p->b + c.r < 256 ) { p->b += c.r; } else { p->b = 255; }

            x += dx;
            iu++;
            }
        y += dy;
        iv++;
        }

    // Finally the blue flare

    Width = sqrt(lc.b) * Scale;

    u1 = (long) ((double) u - Width / 2);
    u2 = (long) ((double) u + Width / 2);
    v1 = (long) ((double) v - Width / 2);
    v2 = (long) ((double) v + Width / 2);

    dx = 1.0 / Width;
    dy = dx;
    
    sx = dx / 2;
    y = dy / 2;

    // Smooth if we are inflating the image

    if(u1 < 0) {
        sx += dx * -u1;
        u1 = 0;
        }
    if(v1 < 0) {
        y += dy * -v1;
        v1 = 0;
        }
    if(u2 > (signed) Current_Bitmap.uMax) {
        u2 = Current_Bitmap.uMax;
        }
    if(v2 > (signed) Current_Bitmap.vMax) {
        v2 = Current_Bitmap.vMax;
        }
    
    // Now we blit it...

    iv = v1;
    while(iv < v2) {
        x = sx;
        iu = u1;
        while(iu < u2) {

            c = LensFlare.GetPixel(x, y, 1).Convert();

            p = Current_Bitmap.Pixel + iu + (iv * Current_Bitmap.uMax);

            if( (long) p->r + c.r < 256 ) { p->r += c.r; } else { p->r = 255; }
            if( (long) p->g + c.r < 256 ) { p->g += c.r; } else { p->g = 255; }
            if( (long) p->b + c.b < 256 ) { p->b += c.b; } else { p->b = 255; }

            x += dx;
            iu++;
            }
        y += dy;
        iv++;
        }

    // Finished!

    }

void Scene::Raytrace(void) {
    Bitmap Current_Bitmap;
    long Current_Camera;
	// Obsolete:
    // long Current_Light_Source;
    long u, v, i, j, v_t, v_b, u_l, u_r;
    double du, dv;
    Ray Current_Ray;
    fColour RayColour;
    char FileName[256];
    time_t Start, Current;


//  LENS FLARE STUFF --------------------------------------------------------

    Frame_Intersection Current_Intersection;
    Event Camera_Hit, Shadow_Hit;
    Vector n, d, o, r;
    fColour c, oc, lc, lt, sc;

//  END LENS FLARE STUFF ----------------------------------------------------

    Current_Bitmap.Create(uImage, vImage);
    
    Current_Camera = Globals.Continue;

	// Print stats

	if(Globals.Verbose > 0) {
		printf("Memory Consumed: %li bytes = %lg kb = %lg Mb\n", Globals.MemoryUsed, (double) Globals.MemoryUsed / 1024.0f, (double) Globals.MemoryUsed / 1048576.0f);
		printf("Primitives Used: %li\n", Globals.PrimitivesUsed);
	}

    // Start the clock...

    time(&Start);

    while(Current_Camera < Number_Cameras) {

        #ifdef PC

        if(Globals.Display) {
			if(Current_Camera == 0) {
				SetMode(uImage, vImage);
				}
			else {
				ResetMode(uImage, vImage);
				}
            }

        #endif

        v = 0;
        while(v < vImage) {

            // The aspect is taken care of by the up & right vectors in the
            // camera definition

            dv =  ((double) ((double) v + 0.5) / vImage) - 0.5;

            u = 0;
            while(u < uImage) {

                du = ((double) ((double) u + 0.5) / uImage) - 0.5;

                RayColour = TraceRay(du, dv, Camera_List[Current_Camera]);

                // Insert AA here...

                // Blit regardless of alpha


                if(Globals.Antialias && (u != 0) && (v != 0)) {
                    Current_Bitmap.Pixel[u + (v * uImage)] = Antialias(
                                                                Current_Bitmap.Pixel[(u - 1) + ((v - 1) * uImage)].Convert(),
                                                                Current_Bitmap.Pixel[u + ((v - 1) * uImage)].Convert(),
                                                                Current_Bitmap.Pixel[(u - 1) + (v * uImage)].Convert(),
                                                                RayColour,
                                                                du - (double) 1.0/uImage,
                                                                dv - (double) 1.0/vImage,
                                                                du,
                                                                dv,
                                                                Camera_List[Current_Camera],
                                                                0).Convert();
                                                                
                    }
                else {
                    Current_Bitmap.Pixel[u + (v * uImage)] = RayColour.Convert();
                    }

                #ifdef PC

                if(Globals.Display) {
                    SetPixel(u, v, Current_Bitmap.Pixel[u + (v * uImage)] );
                    }

                #endif

                u++;
                }
            if((Globals.Verbose == 3) && ((v & 0x0000000F) == 0)) {
				//if(!Globals.Display) {
                    printf("Completed line %li\n", v);
                    //getch();
                //    }
				}
			if(Globals.Verbose > 3) {
                //if(!Globals.Display) {
                    printf("Completed line %li\n", v);
                    //getch();
                //    }
                }
            v++;
            }

        // OK, we now get the lens flare

// ---- LENS FLARE STUFF ----------------------------------------------------

/* FIXUP:
    This is currently ripped wholesale out of the illumination/shadow code
    in GetColour, and as such the two routines should be combined in a
    seperate function, or at least be identically updated in the future */


    i = 0;
    while(i < Number_Frames) {

        // Calculate camera event in local frame of light source

        Camera_Hit = Camera_List[Current_Camera]->location.LorentzTransform(Camera_List[Current_Camera]->Velocity, Frame_List[i]->Velocity);

        j = 0;

        // Cyscle through the light sources

        while(j < Frame_List[i]->Number_Light_Sources) {

			if(	((Light_Source *) Frame_List[i]->Light_Source_List[j]->Geometry)->LensFlare ) {

            // Extract the light source coordinates (at rest)

            o = ((Vector *) &Frame_List[i]->Light_Source_List[j]->Geometry_Transformation)[3];

            o.x = -o.x;
            o.y = -o.y;
            o.z = -o.z;

            // Establish direction vector

            d = Camera_Hit.x - o;

            // Construct ray in new frame moving back in time FROM camera,
            // TO light_source

            // We need a new way to express the ray...
            // starting in the OBJECT FRAME

            // We construct the ray in the light_source frame

            Current_Ray.Origin = Camera_Hit;
            Current_Ray.Direction = d / (d.nrm());
            Current_Ray.intensity = 1;
            Current_Ray.doppler = 1;

            // Then transform it back to the camera frame

            Current_Ray = Current_Ray.LorentzTransform(Frame_List[i]->Velocity, Camera_List[Current_Camera]->Velocity);

            // Now the ray is in the camera frame - so we check that it is
            // actually visible!

            Camera_List[Current_Camera]->GetCoordinates(Current_Ray, du, dv);

            if((du > -0.5) && (du < 0.5) && (dv > -0.5) && (dv < 0.5)) {

            // START EXISTENCE TEST

            u = (long) ((double) (du + 0.5) * uImage + 1.5);
            v = (long) ((double) (dv + 0.5) * vImage + 1.5);

            // Spot check

            // Current_Bitmap.Pixel[u + (v * uImage)].g = 255;

            // Reset the intensity / doppler values

            Current_Ray.intensity = 1;
            Current_Ray.doppler = 1;

            // And transform it back...

            Current_Ray = Current_Ray.LorentzTransform(Camera_List[Current_Camera]->Velocity, Frame_List[i]->Velocity);

            // Now, trace the ray... we don't have an
            // object as parent to prevent 'speckle' as the camera is invisible

            // We need to allow for transparent surfaces...

            lt = fColour( 1, 1, 1, 0, 0);

            Current_Intersection.From_Object = NULL;

  //          printf("Casting shadow ray...\n");

            if( Frame_List[i]->Light_Source_List[j]->no_shadow ) {

    //            printf("First shot\n");

                Current_Intersection = CastRay(Current_Ray, Frame_List[i]->Velocity, NULL);

                while( (Current_Intersection.From_Object != NULL) && ((Camera_Hit.t - Current_Intersection.Hit.t) < d.nrm()) ) {

                    if(Current_Intersection.From_Object->no_shadow) {

                        if(Current_Intersection.From_Object->Surface->P.map_type >= 0) {
                            Shadow_Hit = Current_Intersection.Hit.LorentzTransform(Frame_List[i]->Velocity, Current_Intersection.From_Frame->Velocity);
                            sc = Current_Intersection.From_Object->GetPigment(Shadow_Hit).Doppler2(Current_Ray.doppler * Current_Intersection.From_Object->Surface->P.doppler) * Current_Ray.intensity;
                            }
                        else {
                            sc = Current_Intersection.From_Object->Surface->P.colour.Doppler2(Current_Ray.doppler * Current_Intersection.From_Object->Surface->P.doppler) * Current_Ray.intensity;
                            }

                        lt *= ((fColour( 1, 1, 1, 0, 0 ) * sc.t) + (sc * sc.f));

                        if((lt.r <= 0) && (lt.g <= 0) && (lt.b <= 0)) {

                            // If light is zero, stop the ray

                            break;
                            }
                        }

                    Current_Ray.Origin = Current_Intersection.Hit;
                    
      //              printf("Subsequent shot\n");
                    Current_Intersection = CastRay(Current_Ray, Frame_List[i]->Velocity, Current_Intersection.From_Object);

                    }
                }

//            printf("Completed shadow ray\n");

            if( (Current_Intersection.From_Object == NULL) || ((Camera_Hit.t - Current_Intersection.Hit.t) > d.nrm()) ) {

                // We have a (relatively) unobstructed path to the light_source, so

                // FIXUP: This is dangerous messing around with Current_Ray...

                Current_Ray.Direction = Current_Ray.Direction.LorentzTransform(Frame_List[i]->Velocity, Camera_List[Current_Camera]->Velocity);

                    // OK, we give the event to the light source to allow for
                    // time-dependant textures

                    lc = Frame_List[i]->Light_Source_List[j]->GetPigment(
                                                                Event( o.x + (d.x / d.nrm()),
                                                                          o.y + (d.y / d.nrm()),
                                                                          o.z + (d.z / d.nrm()),
                                                                          Camera_Hit.t - d.nrm() )  
					                               ).Doppler(Current_Ray.doppler);

                    // FIXUP: for intensity...
                    Current_Ray.intensity *= Frame_List[i]->Light_Source_List[j]->Surface->P.intensity;

                    if( ((Light_Source *) Frame_List[i]->Light_Source_List[j]->Geometry)->fade_power != 0) {
                        Current_Ray.intensity *= 2 / ( 1.0 + pow(d.nrm()/((Light_Source *) Frame_List[i]->Light_Source_List[j]->Geometry)->fade_distance, ((Light_Source *) Frame_List[i]->Light_Source_List[j]->Geometry)->fade_power) );
                        }

                    lc *= Current_Ray.intensity;

                    // OK we draw the flare here...

                    // Current_Bitmap.Pixel[u + (v * uImage)] = lc.Convert();

                    // Now we consider the bitmap...

                    // Create a function to do this...

                    DrawFlare(Current_Bitmap, LensFlare, lc, u, v);




                }


            // END EXISTENCE TEST

            }

			}

            j++;
            }
        i++;
        }


// ---- END LENS FLARE STUFF ------------------------------------------------

// ---- START HUD STUFF

		if(Globals.HUD) {
	        u_l = (long) ((double) (Globals.Window_l + 0.5) * uImage + 0.5);
		    v_t = (long) ((double) (Globals.Window_t + 0.5) * vImage + 0.5);
			u_r = (long) ((double) (Globals.Window_r + 0.5) * uImage + 0.5);
	        v_b = (long) ((double) (Globals.Window_b + 0.5) * vImage + 0.5);

			//Bottom Left
			Current_Bitmap.Combine(u_l - 32, v_t - 32, u_l + 128, v_t + 128, 0, 0, 160, 160, HUD);
			// Top Left
			Current_Bitmap.Combine(u_l - 32, v_b - 128, u_l + 128, v_b + 32, 0, 160, 160, 320, HUD);
			// Bottom Right
			Current_Bitmap.Combine(u_r - 128, v_t - 32, u_r + 32, v_t + 128, 160, 0, 320, 160, HUD);
			// Top Right
			Current_Bitmap.Combine(u_r - 128, v_b - 128, u_r + 32, v_b + 32, 160, 160, 320, 320, HUD);
			}

// ---- END HUD STUFF

		if(Number_Cameras > 1) {
			sprintf(FileName, "%s%4.4li.tga", OutFile, Current_Camera);
			}
		else {
			sprintf(FileName, "%s.tga", OutFile);
			}

        Current_Bitmap.SaveTGA(FileName);

        time(&Current);

        printf("%g seconds elapsed\n", difftime(Current, Start) );

        Current_Camera++;
        }
    }

//
// Ripple functions
//

void Object::Transform(Matrix &Transformation) {

    long i;

    switch(Object_Type) {
        case SG_TRIANGLE:
        case SG_SMOOTH_TRIANGLE:
        case SG_SPHERE:
        case SG_BOX:
        case SG_CYLINDER:
        case SG_CONE:
        case SG_PLANE:
        case SG_QUADRIC:
        case SG_LIGHT_SOURCE:

            Geometry_Transformation *= Transformation;
            Texture_Transformation *= Transformation;

            break;

        case CSG_UNION:
            i = 0;
            while(i < ((CSG_union *) Geometry)->Number_Objects) {
                ((CSG_union *) Geometry)->Object_List[i]->Transform(Transformation);
                i++;
                }
            break;

        case CSG_MERGE:
            i = 0;
            while(i < ((CSG_merge *) Geometry)->Number_Objects) {
                ((CSG_merge *) Geometry)->Object_List[i]->Transform(Transformation);
                i++;
                }
            break;

        case CSG_DIFFERENCE:
            i = 0;
            while(i < ((CSG_difference *) Geometry)->Number_Objects) {
                ((CSG_difference *) Geometry)->Object_List[i]->Transform(Transformation);
                i++;
                }
            break;

        case CSG_INTERSECTION:
            i = 0;
            while(i < ((CSG_intersection *) Geometry)->Number_Objects) {
                ((CSG_intersection *) Geometry)->Object_List[i]->Transform(Transformation);
                i++;
                }
            break;

        }

    }

void Object::Paint(Texture *T) {
    long i;

    switch(Object_Type) {
        case SG_TRIANGLE:
        case SG_SMOOTH_TRIANGLE:
        case SG_SPHERE:
        case SG_BOX:
        case SG_CYLINDER:
        case SG_CONE:
        case SG_PLANE:
        case SG_QUADRIC:
        case SG_LIGHT_SOURCE:

            if(Surface == Default_Texture) {
                Surface = T;
                Texture_Transformation = Matrix( 1, 0, 0,  0, 1, 0,  0, 0, 1,  0, 0, 0 );
                }

            break;

        case CSG_UNION:
            i = 0;
            while(i < ((CSG_union *) Geometry)->Number_Objects) {

                ((CSG_union *) Geometry)->Object_List[i]->Paint(T);
                i++;
                }
            break;

        case CSG_MERGE:
            i = 0;
            while(i < ((CSG_merge *) Geometry)->Number_Objects) {
                ((CSG_merge *) Geometry)->Object_List[i]->Paint(T);
                i++;
                }
            break;

        case CSG_DIFFERENCE:
            i = 0;
            while(i < ((CSG_difference *) Geometry)->Number_Objects) {
                ((CSG_difference *) Geometry)->Object_List[i]->Paint(T);
                i++;
                }
            break;

        case CSG_INTERSECTION:
            i = 0;
            while(i < ((CSG_intersection *) Geometry)->Number_Objects) {
                ((CSG_intersection *) Geometry)->Object_List[i]->Paint(T);
                i++;
                }
            break;

        }

    }


//
// Parsing (Load) Functions
//

void Bounding_Sphere::Union(Bounding_Sphere b) {
    Vector no;
    double nr;

    if(r == 0) {
        return;
        }
    if(b.r == 0) {
        *this = b;
        return;
        }

    if(r >= b.r) {
        if( (o - b.o).nrm() + b.r <= r ) {
            // b inside this => return this
            return;
            }
        }

    if(r <= b.r) {
        if( (o - b.o).nrm() + r <= b.r) {
            // this inside b => return b
            o = b.o;
            r = b.r;

            return;
            }
        }

    // Neither inside the other. So...

    no = (o + b.o + (o - b.o)*(r - b.r)/((o - b.o).nrm())) / 2;
    nr = (r + b.r + (o - b.o).nrm()) / 2;

    o = no;
    r = nr;

    return;

    }

void Bounding_Sphere::Intersection(Bounding_Sphere b) {
    Vector no;
    // Obsolete:
	// double nr;
    
    // OK, simplest option is to return smallest sphere
    // There are better solutions, however

    // FIXUP: Improve solution...

	if(r == 0) {
		// We are unbounded, so anything is better!
		o = b.o;
		r = b.r;
		}

    if(b.r < r) {
        o = b.o;
        r = b.r;
        return;
        }

    return;
        
    }


void Camera::Show(void) {
    printf("Camera\n");
    }

Token *Camera::Load(Token *T, Vector v) {
    Matrix Transformation;
    long Depth;
    double f;

    Type = 0;

    location.x = Vector( 0, 0, 0 );
    location.t = 0;
    look_at    = Vector( 0, 0, 0 );
    right      = Vector( 1.33, 0, 0 );
    up         = Vector( 0, 1, 0 );
    direction  = Vector( 0, 0, 1 );
    sky        = Vector( 0, 1, 0 );
    angle      = 0;

    Velocity   = v;

    Depth = 0;

    do {

        T = T->Next;

        switch(T->ID) {

            case ID_velocity:

                T = T->Next;

                T = Velocity.Load(T);

                // We need the velocity relative to the global zero velocity,
                // not the parent frame as given

                Velocity = Velocity.LorentzTransform(v, Vector( 0.0, 0.0, 0.0 ) );

                break;

            case ID_camera:
                break;
            case ID_LEFT_BRACE:
                Depth++;
                break;
            case ID_RIGHT_BRACE:
                Depth--;
                break;

            case ID_perspective:
                Type = 0;
                break;
            case ID_orthographic:
                Type = 1;
                break;
            case ID_fisheye:
                Type = 2;
                break;
            case ID_mercator:
                Type = 3;
                break;

            case ID_angle:
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                angle = *((double *) T->Data) * (PI / 180);

                if(angle ==  0) {
                    PrintToken(T);
                    printf(" is invalid value for angle\n");
                    exit(EXIT_FAILURE);
                    }
                if(Type == 0) {

                    if(angle >= PI) {
                        PrintToken(T);
                        printf("is invalid value for angle\n");
                        exit(EXIT_FAILURE);
                        }

					f = tan(angle / 2) * direction.nrm() / right.nrm();

                    right *= f;
                    up    *= f;
                    }

                break;

            case ID_location:
                T = T->Next;
                T = location.x.Load(T);
                T = T->Next;
                if(T->ID != ID_COMMA) {
                    PrintToken(T);
                    printf("found, expected comma\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                location.t = *((double *) T->Data);

                break;
            case ID_look_at:
                T = T->Next;
                T = look_at.Load(T);

                // We rotate the camera around...

                direction = (look_at - location.x) * direction.nrm() / (look_at - location.x).nrm();
                // FXUP: Keep handedness?
                right = (direction / sky) * right.nrm() / (direction / sky).nrm();
                up = (direction / right) * up.nrm() / -(direction / right).nrm();

                break;
            case ID_right:
                T = T->Next;
                T = right.Load(T);
                break;
            case ID_up:
                T = T->Next;
                T = up.Load(T);
                break;
            case ID_direction:
                T = T->Next;
                T = direction.Load(T);
                break;
            case ID_sky:
                T = T->Next;
                T = sky.Load(T);
                break;

            case ID_translate:
            case ID_scale:
            case ID_rotate:
            case ID_matrix:
                T = Transformation.Load(T);

                location.x = Transformation.TransformLocation (location.x);
                look_at    = Transformation.TransformLocation (look_at   );
                right      = Transformation.TransformDirection(right     );
                up         = Transformation.TransformDirection(up        );
                direction  = Transformation.TransformDirection(direction );
                sky        = Transformation.TransformDirection(sky       );

                break;

            default:
                PrintToken(T);
                printf("found unexpectedly\n");
                exit(EXIT_FAILURE);
                break;
            }

        } while(Depth > 0);

    // Sort out vectors for the weird projections...

    switch(Type) {

        case 0:
            // Force orthogonal ?
            break;
        case 1:
            break;
        case 2:
        case 3:
            // Spherical-like projections get their vectors normalised...
            direction.Normalise(1);
            right.Normalise(1);
            up.Normalise(1);
            break;
        default:
            break;
        }
    return T;

    }

Token *Pigment::Load(Token *T) {
    Matrix Trans;

    T = T->Next;
    if(T->ID != ID_LEFT_BRACE) {
        PrintToken(T);
        printf("found, { expected\n");
        exit(EXIT_FAILURE);
        }

    T = T->Next;

    while(T->ID != ID_RIGHT_BRACE) {
        switch(T->ID) {

            case ID_image_map:
                // We don't really need to worry too much about syntax,
                // so sweep it under the carpet...
                T = Load(T);
                break;

            case ID_map_type:
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, number expected\n");
                    exit(EXIT_FAILURE);
                    }
                map_type = (long int) *((double *) T->Data);

                break;

            case ID_colour:
            case ID_color:
            case ID_red:
            case ID_blue:
            case ID_green:
            case ID_rgb:
            case ID_rgbf:
            case ID_rgbt:
            case ID_rgbft:
            case ID_rgbtf:
                map_type = MT_COLOUR;
                T = colour.Load(T);
                break;

            // Tricky part: We load any filter/transmit values into the
            // colour, but DON'T change the type: allows us to set values
            // for the image maps...

            case ID_filter:
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, number expected\n");
                    exit(EXIT_FAILURE);
                    }
                colour.f = *((double *) T->Data);
                break;
            case ID_transmit:
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, number expected\n");
                    exit(EXIT_FAILURE);
                    }
                colour.t = *((double *) T->Data);
                break;

            case ID_tga:
                T = T->Next;
                if(T->ID != ID_STRING) {
                    PrintToken(T);
                    printf("found, expected \"filename.tga\"\n");
                    exit(EXIT_FAILURE);
                    }

                RequestBitmap(&bitmap, (char *) T->Data);

                if(map_type < 0) {
                    map_type = MT_PLANAR; // default mapping
                    }

                break;

            case ID_interpolate:
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, number expected\n");
                    exit(EXIT_FAILURE);
                    }
                interpolate = (long int) *((double *) T->Data);
                break;

            case ID_doppler:
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, number expected\n");
                    exit(EXIT_FAILURE);
                    }
                doppler = *((double *) T->Data);
                break;

            case ID_intensity:
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, number expected\n");
                    exit(EXIT_FAILURE);
                    }
                intensity = *((double *) T->Data);
                break;

            case ID_once:
                once = !once;
                break;

            case ID_frequency:
                T = T->Next;
                if(T->ID != ID_LEFT_ANGLE) {
                    if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected <\n"); exit(EXIT_FAILURE); }
                    // is a double...
                    frequency_u = *((double *) T->Data);
                    frequency_v = *((double *) T->Data);
                    break;
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                frequency_u = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                frequency_v = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_RIGHT_ANGLE) { PrintToken(T); printf("found, expected >\n"); exit(EXIT_FAILURE); }
                break;
            case ID_phase:
                T = T->Next;
                if(T->ID != ID_LEFT_ANGLE) {
                    if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected <\n"); exit(EXIT_FAILURE); }
                    // is a double...
                    phase_u = *((double *) T->Data);
                    phase_v = *((double *) T->Data);
                    break;
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                phase_u = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                phase_v = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_RIGHT_ANGLE) { PrintToken(T); printf("found, expected >\n"); exit(EXIT_FAILURE); }
                break;
            case ID_drift:
                T = T->Next;
                if(T->ID != ID_LEFT_ANGLE) {
                    if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected <\n"); exit(EXIT_FAILURE); }
                    // is a double...
                    drift_u = *((double *) T->Data);
                    drift_v = *((double *) T->Data);
                    break;
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                drift_u = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                drift_v = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_RIGHT_ANGLE) { PrintToken(T); printf("found, expected >\n"); exit(EXIT_FAILURE); }
                break;
            case ID_translate:
            case ID_scale:
            case ID_rotate:
            case ID_matrix:
                T = Trans.Load(T);
                Transformation *= Trans;

                break;

            default:
                PrintToken(T);
                printf("illegal in pigment definition\n");
                exit(EXIT_FAILURE);
                break;
            }
        T = T->Next;
        }

    return T;

    }



Token *Normal::Load(Token *T) {
    Matrix Trans;

    T = T->Next;
    if(T->ID != ID_LEFT_BRACE) {
        PrintToken(T);
        printf("found, { expected\n");
        exit(EXIT_FAILURE);
        }

    T = T->Next;

    while(T->ID != ID_RIGHT_BRACE) {
        switch(T->ID) {

            case ID_bump_map:
                // We don't really need to worry too much about syntax,
                // so sweep it under the carpet...
                T = Load(T);
                break;

            case ID_map_type:
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, number expected\n");
                    exit(EXIT_FAILURE);
                    }
                map_type = (long int) *((double *) T->Data);

                    // This has to be an integer. 0, 1, 2, 5 are POV
                    // -2 is NOTHING, -1 is colour (constant) and >= 10 are
                    // yet to be specified, though they will be
                    // used for more exotic mappings

                break;

            case ID_tga:
                T = T->Next;
                if(T->ID != ID_STRING) {
                    PrintToken(T);
                    printf("found, expected \"filename.tga\"\n");
                    exit(EXIT_FAILURE);
                    }

                RequestBitmap(&bitmap, (char *) T->Data);

                if(map_type < 0) {
                    map_type = 0;
                    }

                break;

            case ID_interpolate:
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, number expected\n");
                    exit(EXIT_FAILURE);
                    }
                interpolate = (long int) *((double *) T->Data);
                break;
                
            case ID_bump_size:
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, number expected\n");
                    exit(EXIT_FAILURE);
                    }
                bump_size = *((double *) T->Data);
                break;

            case ID_once:
                once = !once;
                break;

            case ID_frequency:
                T = T->Next;
                if(T->ID != ID_LEFT_ANGLE) {
                    if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected <\n"); exit(EXIT_FAILURE); }
                    // is a double...
                    frequency_u = *((double *) T->Data);
                    frequency_v = *((double *) T->Data);
                    break;
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                frequency_u = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                frequency_v = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_RIGHT_ANGLE) { PrintToken(T); printf("found, expected >\n"); exit(EXIT_FAILURE); }
                break;
            case ID_phase:
                T = T->Next;
                if(T->ID != ID_LEFT_ANGLE) {
                    if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected <\n"); exit(EXIT_FAILURE); }
                    // is a double...
                    phase_u = *((double *) T->Data);
                    phase_v = *((double *) T->Data);
                    break;
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                phase_u = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                phase_v = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_RIGHT_ANGLE) { PrintToken(T); printf("found, expected >\n"); exit(EXIT_FAILURE); }
                break;
            case ID_drift:
                T = T->Next;
                if(T->ID != ID_LEFT_ANGLE) {
                    if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected <\n"); exit(EXIT_FAILURE); }
                    // is a double...
                    drift_u = *((double *) T->Data);
                    drift_v = *((double *) T->Data);
                    break;
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                drift_u = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_COMMA) { PrintToken(T); printf("found, expected comma\n"); exit(EXIT_FAILURE); }
                T = T->Next;
                if(T->ID != ID_DOUBLE) { PrintToken(T); printf("found, expected number\n"); exit(EXIT_FAILURE); }
                drift_v = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_RIGHT_ANGLE) { PrintToken(T); printf("found, expected >\n"); exit(EXIT_FAILURE); }
                break;

            case ID_translate:
            case ID_scale:
            case ID_rotate:
            case ID_matrix:
                T = Trans.Load(T);
                Transformation *= Trans;

                break;

            default:
                PrintToken(T);
                printf("illegal in normal definition\n");
                exit(EXIT_FAILURE);
                break;
            }
        T = T->Next;
        }

    return T;

    }

Token *Finish::Load(Token *T) {

    T = T->Next;
    if(T->ID != ID_LEFT_BRACE) {
        PrintToken(T);
        printf("found, { expected\n");
        exit(EXIT_FAILURE);
        }

    T = T->Next;

    while(T->ID != ID_RIGHT_BRACE) {
        switch(T->ID) {

            case ID_ambient:
                T = T->Next;
                T = ambient.Load(T);
                break;                

            case ID_diffuse:
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, number expected\n");
                    exit(EXIT_FAILURE);
                    }
                diffuse = *((double *) T->Data);
                break;

            case ID_brilliance:
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, number expected\n");
                    exit(EXIT_FAILURE);
                    }
                brilliance = *((double *) T->Data);
                break;

            case ID_phong:
            case ID_specular:
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, number expected\n");
                    exit(EXIT_FAILURE);
                    }
                specular = *((double *) T->Data);
                break;

            case ID_roughness:
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, number expected\n");
                    exit(EXIT_FAILURE);
                    }
                roughness = *((double *) T->Data);
                break;

            case ID_phong_size:
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, number expected\n");
                    exit(EXIT_FAILURE);
                    }
                roughness = 1 / (*((double *) T->Data));
                break;

            case ID_reflection:
                T = T->Next;
                T = reflection.Load(T);
                break;

            case ID_metallic:
                //FIXUP:
                break;

            case ID_refraction:
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, number expected\n");
                    exit(EXIT_FAILURE);
                    }
                refraction = *((double *) T->Data);
                break;

            case ID_ior:
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, number expected\n");
                    exit(EXIT_FAILURE);
                    }
                ior = *((double *) T->Data);
                break;

            default:
                PrintToken(T);
                printf("found in finish definition\n");
                exit(EXIT_FAILURE);
                break;
            }
        T = T->Next;
        }

    return T;

    }

Token *Texture::Load(Token *T) {
    Matrix Transformation;

    *this = *Default_Texture;

    switch(T->ID) {
        case ID_texture:

            T = T->Next;

            if(T->ID != ID_LEFT_BRACE) {
                PrintToken(T);
                printf("found, { expected\n");
                exit(EXIT_FAILURE);
                }

            T = T->Next;    

            while(T->ID != ID_RIGHT_BRACE) {
                switch(T->ID) {

                    case ID_colour:
                    case ID_color:
                        T = P.colour.Load(T);
                        break;
                    case ID_pigment:
                        T = P.Load(T);
                        break;

                    case ID_normal:
                        T = N.Load(T);
                        break;

                    case ID_finish:
                        T = F.Load(T);
                        break;

                    case ID_texture:
                        // Allow the recursion which #definitions cause
                        T = this->Load(T);
                        break;

                    case ID_translate:
                    case ID_scale:
                    case ID_rotate:
                    case ID_matrix:
                        T = Transformation.Load(T);

                        P.Transformation *= Transformation;
                        N.Transformation *= Transformation;

                        break;

                    default:
                        PrintToken(T);
                        printf("illegal in texture definition");
                        exit(EXIT_FAILURE);
                        break;
                    }
                T = T->Next;    
                } 

            break;

        case ID_colour:
        case ID_color:
            T = P.colour.Load(T);
            break;
        case ID_pigment:
            T = P.Load(T);
            break;
        case ID_normal:
            T = N.Load(T);
            break;
        case ID_finish:
            T = F.Load(T);
            break;
        default:
            PrintToken(T);
            printf("found, expected texture definition\n");
            exit(EXIT_FAILURE);
            break;
        }

    return T;
    }


void Object::Show(void) {
    long i = 0;

    switch(Object_Type) {

        case CSG_UNION:
            printf("union {\n");
            i = 0;
            while(i < ((CSG_union *) Geometry)->Number_Objects) {
                ((CSG_union *) Geometry)->Object_List[i]->Show();
                i++;
                }
            printf("    }, bounded_by < %g, %g, %g >, %g\n", bounded_by.o.x, bounded_by.o.y, bounded_by.o.z, bounded_by.r);
            break;
        case CSG_MERGE:
            i = 0;
            printf("merge {\n");
            while(i < ((CSG_merge *) Geometry)->Number_Objects) {
                ((CSG_merge *) Geometry)->Object_List[i]->Show();
                i++;
                }
            printf("    }, bounded_by < %g, %g, %g >, %g\n", bounded_by.o.x, bounded_by.o.y, bounded_by.o.z, bounded_by.r);
            break;
        case CSG_INTERSECTION:
            i = 0;
            printf("intersection {\n");
            while(i < ((CSG_intersection *) Geometry)->Number_Objects) {
                ((CSG_intersection *) Geometry)->Object_List[i]->Show();
                i++;
                }
            printf("    }, bounded_by < %g, %g, %g >, %g\n", bounded_by.o.x, bounded_by.o.y, bounded_by.o.z, bounded_by.r);
            break;
        case CSG_DIFFERENCE:
            i = 0;
            printf("difference {\n");
            while(i < ((CSG_difference *) Geometry)->Number_Objects) {
                ((CSG_difference *) Geometry)->Object_List[i]->Show();
                i++;
                }
            printf("    }, bounded_by < %g, %g, %g >, %g\n", bounded_by.o.x, bounded_by.o.y, bounded_by.o.z, bounded_by.r);
            break;
        case SG_TRIANGLE:
            printf("triangle");
            printf(", bounded by < %g, %g, %g >, %g\n", bounded_by.o.x, bounded_by.o.y, bounded_by.o.z, bounded_by.r);
            break;
        case SG_SMOOTH_TRIANGLE:
            printf("smooth_triangle");
            printf(", bounded by < %g, %g, %g >, %g\n", bounded_by.o.x, bounded_by.o.y, bounded_by.o.z, bounded_by.r);
            break;
        case SG_SPHERE:
            printf("sphere");
            printf(", bounded by < %g, %g, %g >, %g\n", bounded_by.o.x, bounded_by.o.y, bounded_by.o.z, bounded_by.r);
            break;
        case SG_BOX:
            printf("box");
            printf(", bounded by < %g, %g, %g >, %g\n", bounded_by.o.x, bounded_by.o.y, bounded_by.o.z, bounded_by.r);
            break;
        case SG_CYLINDER:
            printf("cylinder");
            printf(", bounded by < %g, %g, %g >, %g\n", bounded_by.o.x, bounded_by.o.y, bounded_by.o.z, bounded_by.r);
            break;
        case SG_CONE:
            printf("cone");
            printf(", bounded by < %g, %g, %g >, %g\n", bounded_by.o.x, bounded_by.o.y, bounded_by.o.z, bounded_by.r);
            break;
        case SG_PLANE:
            printf("plane");
            printf(", bounded by < %g, %g, %g >, %g\n", bounded_by.o.x, bounded_by.o.y, bounded_by.o.z, bounded_by.r);
            break;
        case SG_QUADRIC:
            printf("quadric");
            printf(", bounded by < %g, %g, %g >, %g\n", bounded_by.o.x, bounded_by.o.y, bounded_by.o.z, bounded_by.r);
            break;
        case SG_LIGHT_SOURCE:
            printf("light_source < %g, %g, %g >\n", 
					Geometry_Transformation.a14, 
					Geometry_Transformation.a24, 
					Geometry_Transformation.a34
					);
            break;
        default:
            printf("Unknown object type!\n");
            break;
        }
	}

Bounding_Sphere Object::Bound(Object **L_S_L) {
    Matrix Transformation;
    Vector v1, v2, v3;
    Object **Hierarchy;
    long i, j, k;
    double a, b;

    // FIXUP: This will cause manually bounded CSGs to
    // have unbounded components!!!

    if(bounded_by.r != 0) {
        // In case the bounding has already been done manually
        return bounded_by;
        }

	switch(Object_Type) {
        
        case CSG_UNION:
            i = 0;
            while(i < ((CSG_union *) Geometry)->Number_Objects) {
                if( ((CSG_union *) Geometry)->Object_List[i]->Object_Type == SG_LIGHT_SOURCE ) {
                    // Transfer light_source to Light_Source_List;
                    ((CSG_union *) Geometry)->Number_Objects--;
                    j = 0;
                    while(L_S_L[j] != NULL) {
                        j++;
                        }
                    L_S_L[j] = ((CSG_union *) Geometry)->Object_List[i];
                    j = i;
                    while(j < ((CSG_union *) Geometry)->Number_Objects) {
                        ((CSG_union *) Geometry)->Object_List[j] = ((CSG_union *) Geometry)->Object_List[j + 1];
                        j++;
                        }
                    // Don't increment i, as it now indexes a new object
                    }
                else {
                    i++;
                    }
                }

            i = 0;
            while(i < ((CSG_union *) Geometry)->Number_Objects) {
                if(i == 0) {
                    bounded_by = ((CSG_union *) Geometry)->Object_List[i]->Bound(L_S_L);
                    }
                else {
                    bounded_by.Union( ((CSG_union *) Geometry)->Object_List[i]->Bound(L_S_L) );
                    }
                i++;
                }

            // Octree big unions down to little unions

            // FIXUP: Threshold for hierarchy... - experiment with
            // optimal value

            if( ((CSG_union *) Geometry)->Number_Objects > 4) {

                if(Globals.Verbose > 1) {
					printf("Subdividing large union...\n");
				}

                // We create a new series of unions

                Hierarchy = new Object *[8];
				Globals.MemoryUsed += sizeof(Object *) * 8;
	
                i = 0;
                do {

                    Hierarchy[i] = new Object;
					Globals.MemoryUsed += sizeof(Object);
	
                    Hierarchy[i]->Object_Type = CSG_UNION;
                    Hierarchy[i]->Geometry = new CSG_union;
					Globals.MemoryUsed += sizeof(CSG_union);
	                Hierarchy[i]->bounded_by.o = Vector( 0, 0, 0 );
                    Hierarchy[i]->bounded_by.r = 0;

                    Hierarchy[i]->Geometry_Transformation = Matrix( 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0 );
                    Hierarchy[i]->Texture_Transformation = Matrix( 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0 );
                    Hierarchy[i]->Surface = Default_Texture;
                    Hierarchy[i]->no_shadow = 1;
                    Hierarchy[i]->inverse = 0;

                    // Now, we setup the union...

                    // Count members...

                    j = 0;
                    k = 0;

                    do {
                        switch(i) {
                            case 0:
                                if(
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).x < 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).y < 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).z < 0 ) ) {
                                   k++; 
                                   }
                                break;
                            case 1:
                                if(
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).x < 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).y < 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).z >= 0 ) ) {
                                   k++; 
                                   }
                                break;
                            case 2:
                                if(
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).x < 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).y >= 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).z < 0 ) ) {
                                   k++; 
                                   }
                                break;
                            case 3:
                                if(
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).x < 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).y >= 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).z >= 0 ) ) {
                                   k++; 
                                   }
                                break;
                            case 4:
                                if(
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).x >= 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).y < 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).z < 0 ) ) {
                                   k++; 
                                   }
                                break;
                            case 5:
                                if(
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).x >= 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).y < 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).z >= 0 ) ) {
                                   k++; 
                                   }
                                break;
                            case 6:
                                if(
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).x >= 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).y >= 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).z < 0 ) ) {
                                   k++; 
                                   }
                                break;
                            case 7:
                                if(
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).x >= 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).y >= 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).z >= 0 ) ) {
                                   k++; 
                                   }
                                break;
                            }
                        j++;
                        } while(j < ((CSG_union *) Geometry)->Number_Objects);

                    if(k == j) {
                        // All in one!
                        // will cause infinite recursion, so pull the
                        // plug NOW!
                        return bounded_by;
                        }

                    ((CSG_union *) Hierarchy[i]->Geometry)->Number_Objects = k;
                    ((CSG_union *) Hierarchy[i]->Geometry)->Object_List = new Object *[k];
					Globals.MemoryUsed += sizeof(Object *) * k;
	
                    // And now we fill up the new union

                    j = 0;
                    k = 0;

                    do {
                        switch(i) {
                            case 0:
                                if(
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).x < 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).y < 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).z < 0 ) ) {
                                   ((CSG_union *) Hierarchy[i]->Geometry)->Object_List[k] = ((CSG_union *) Geometry)->Object_List[j];
                                   k++;
                                   }
                                break;
                            case 1:
                                if(
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).x < 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).y < 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).z >= 0 ) ) {
                                   ((CSG_union *) Hierarchy[i]->Geometry)->Object_List[k] = ((CSG_union *) Geometry)->Object_List[j];
                                   k++; 
                                   }
                                break;
                            case 2:
                                if(
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).x < 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).y >= 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).z < 0 ) ) {
                                   ((CSG_union *) Hierarchy[i]->Geometry)->Object_List[k] = ((CSG_union *) Geometry)->Object_List[j];
                                   k++; 
                                   }
                                break;
                            case 3:
                                if(
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).x < 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).y >= 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).z >= 0 ) ) {
                                   ((CSG_union *) Hierarchy[i]->Geometry)->Object_List[k] = ((CSG_union *) Geometry)->Object_List[j];
                                   k++; 
                                   }
                                break;
                            case 4:
                                if(
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).x >= 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).y < 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).z < 0 ) ) {
                                   ((CSG_union *) Hierarchy[i]->Geometry)->Object_List[k] = ((CSG_union *) Geometry)->Object_List[j];
                                   k++; 
                                   }
                                break;
                            case 5:
                                if(
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).x >= 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).y < 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).z >= 0 ) ) {
                                   ((CSG_union *) Hierarchy[i]->Geometry)->Object_List[k] = ((CSG_union *) Geometry)->Object_List[j];
                                   k++; 
                                   }
                                break;
                            case 6:
                                if(
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).x >= 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).y >= 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).z < 0 ) ) {
                                   ((CSG_union *) Hierarchy[i]->Geometry)->Object_List[k] = ((CSG_union *) Geometry)->Object_List[j];
                                   k++; 
                                   }
                                break;
                            case 7:
                                if(
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).x >= 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).y >= 0 ) &&
                                   ( ( ((CSG_union *) Geometry)->Object_List[j]->bounded_by.o - bounded_by.o ).z >= 0 ) ) {
                                   ((CSG_union *) Hierarchy[i]->Geometry)->Object_List[k] = ((CSG_union *) Geometry)->Object_List[j];
                                   k++; 
                                   }
                                break;
                            }
                        j++;
                        } while(j < ((CSG_union *) Geometry)->Number_Objects);

                    // So, each Hierarchy[i] is now a complete object/union,
                    // except for bounding

                    i++;
                    } while(i < 8);

                // And now, we

                free( ((CSG_union *) Geometry)->Object_List );

                ((CSG_union *) Geometry)->Number_Objects = 8;

                ((CSG_union *) Geometry)->Object_List = Hierarchy;

                i = 0;
                do {
                    Hierarchy[i]->Bound(L_S_L);
                    i++;
                    } while(i < 8);
                    
                // We keep precalculated bound for the union.

                }
         break;

        case CSG_MERGE:
            i = 0;
            while(i < ((CSG_merge *) Geometry)->Number_Objects) {
                if( ((CSG_merge *) Geometry)->Object_List[i]->Object_Type == SG_LIGHT_SOURCE ) {
                    // Transfer light_source to Light_Source_List;
                    ((CSG_merge *) Geometry)->Number_Objects--;
                    j = 0;
                    while(L_S_L[j] != NULL) {
                        j++;
                        }
                    L_S_L[j] = ((CSG_merge *) Geometry)->Object_List[i];
                    j = i;
                    while(j < ((CSG_merge *) Geometry)->Number_Objects) {
                        ((CSG_merge *) Geometry)->Object_List[j] = ((CSG_merge *) Geometry)->Object_List[j + 1];
                        j++;
                        }
                    // Don't increment i, as it now indexes a new object
                    }
                else {
                    i++;
                    }
                }
             i = 0;
             while(i < ((CSG_merge *) Geometry)->Number_Objects) {
                if(i == 0) {
                    bounded_by = ((CSG_merge *) Geometry)->Object_List[i]->Bound(L_S_L);
                    }
                else {
                    bounded_by.Union( ((CSG_merge *) Geometry)->Object_List[i]->Bound(L_S_L) );
                    }
                i++;
                }
         break;

        case CSG_INTERSECTION:
            i = 0;
            while(i < ((CSG_intersection *) Geometry)->Number_Objects) {
                if( ((CSG_intersection *) Geometry)->Object_List[i]->Object_Type == SG_LIGHT_SOURCE ) {
                    // Transfer light_source to Light_Source_List;
                    ((CSG_intersection *) Geometry)->Number_Objects--;
                    j = 0;
                    while(L_S_L[j] != NULL) {
                        j++;
                        }
                    L_S_L[j] = ((CSG_intersection *) Geometry)->Object_List[i];
                    j = i;
                    while(j < ((CSG_intersection *) Geometry)->Number_Objects) {
                        ((CSG_intersection *) Geometry)->Object_List[j] = ((CSG_intersection *) Geometry)->Object_List[j + 1];
                        j++;
                        }
                    // Don't increment i, as it now indexes a new object
                    }
                else {
                    i++;
                    }
                }
             i = 0;
             while(i < ((CSG_intersection *) Geometry)->Number_Objects) {
                if(i == 0) {
                    bounded_by = ((CSG_intersection *) Geometry)->Object_List[i]->Bound(L_S_L);
                    }
                else {
                    bounded_by.Intersection( ((CSG_intersection *) Geometry)->Object_List[i]->Bound(L_S_L) );
                    }
                i++;
                }
         break;

        case CSG_DIFFERENCE:
            i = 0;
            while(i < ((CSG_difference *) Geometry)->Number_Objects) {
                if( ((CSG_difference *) Geometry)->Object_List[i]->Object_Type == SG_LIGHT_SOURCE ) {
                    // Transfer light_source to Light_Source_List;
                    ((CSG_difference *) Geometry)->Number_Objects--;
                    j = 0;
                    while(L_S_L[j] != NULL) {
                        j++;
                        }
                    L_S_L[j] = ((CSG_difference *) Geometry)->Object_List[i];
                    j = i;
                    while(j < ((CSG_difference *) Geometry)->Number_Objects) {
                        ((CSG_difference *) Geometry)->Object_List[j] = ((CSG_difference *) Geometry)->Object_List[j + 1];
                        j++;
                        }
                    // Don't increment i, as it now indexes a new object
                    }
                else {
                    i++;
                    }
                }

                // Will be less than first object...

                bounded_by = ((CSG_difference *) Geometry)->Object_List[0]->Bound(L_S_L);

                i = 1;

                // Even though we don't need them to calculate the bound,
                // we have to initialise their bounding

                while(i < ((CSG_difference *) Geometry)->Number_Objects) {
                    ((CSG_difference *) Geometry)->Object_List[i]->Bound(L_S_L);
                    i++;
                    }


            break;

        case SG_TRIANGLE:
        case SG_SMOOTH_TRIANGLE:
            Transformation = Geometry_Transformation.invert();

            // OK, we use the transformation to get the 3 vertices, and then
            // calculate the containing sphere

            // FIXUP: Improve

            v1 = ((Vector *) &Transformation)[0];
            v2 = ((Vector *) &Transformation)[1];
            v3 = ((Vector *) &Transformation)[3];

            // First check if the triangle has an obtuse angle...

            if((v1 * v2) <= 0) {
                bounded_by.o = v3 + ((v1 + v2) / 2);
                bounded_by.r = (v2 - v1).nrm() / 2;
                }
            else {
                if((v1 * (v1 - v2)) <= 0) {
                    bounded_by.o = v3 + (v2 / 2);
                    bounded_by.r = v2.nrm() / 2;
                    }
                else {
                    if((v2 * (v2 - v1)) <= 0) {
                        bounded_by.o = v3 + (v1 / 2);
                        bounded_by.r = v1.nrm() / 2;
                        }
                    else {
                        // Thus, all angles are acute, and all vertices
                        // will lie on the bounding sphere

                        // We calculate the position on the triangle
                        // where the difference to all vertices is equal
                        // (and minimal):

                        a = (v2.sqn() / 2) * (v1.sqn() + (v1 * v2)) / ((v1.sqn() * v2.sqn()) + (v1 * v2));
                        b = 0.5 - (a * (v1*v2) / v2.sqn());

                        bounded_by.o = (v1 * a) + (v2 * b) + v3;
                        bounded_by.r = ((v1 * a) + (v2 * b)).nrm();

                        }
                    }
                }

            // Sort out normals

            if(Object_Type == SG_TRIANGLE) {
                // FLAT triangle

                ((SG_triangle *) Geometry)->Normal = (Geometry_Transformation.invert()).TransformNormal( Vector( 0, 0, 1 ) );

                ((SG_triangle *) Geometry)->Normal.Normalise(1);
                }
            else {
                // PHONG Normal-Interpolated triangle

                ((SG_smooth_triangle *) Geometry)->Normal = (Geometry_Transformation.invert()).TransformNormal( Vector( 0, 0, 1 ) );
                ((SG_smooth_triangle *) Geometry)->Normal.Normalise(1);

                ((SG_smooth_triangle *) Geometry)->Normal_1 = (Geometry_Transformation.invert()).TransformNormal( ((SG_smooth_triangle *) Geometry)->Normal_1 );
                ((SG_smooth_triangle *) Geometry)->Normal_2 = (Geometry_Transformation.invert()).TransformNormal( ((SG_smooth_triangle *) Geometry)->Normal_2 );
                ((SG_smooth_triangle *) Geometry)->Normal_3 = (Geometry_Transformation.invert()).TransformNormal( ((SG_smooth_triangle *) Geometry)->Normal_3 );

                ((SG_smooth_triangle *) Geometry)->Normal_1.Normalise(1);
                ((SG_smooth_triangle *) Geometry)->Normal_2.Normalise(1);
                ((SG_smooth_triangle *) Geometry)->Normal_3.Normalise(1);
                }

			Globals.PrimitivesUsed++;

            break;

        case SG_SPHERE:
            Transformation = Geometry_Transformation.invert();

            // This will be ugly...

            // By linearity and symmetry, we can set

            bounded_by.o = ((Vector *) &Transformation)[3];

            // We need to find the maximum radius of the transformed surface
            // FIXUP: Improve

            bounded_by.r = (((Vector *) &Transformation)[0]
                          + ((Vector *) &Transformation)[1]
                          + ((Vector *) &Transformation)[2]).nrm();

            // Obviously true, but NEVER optimal

			Globals.PrimitivesUsed++;

            break;

        case SG_BOX:
            Transformation = Geometry_Transformation.invert();

            // By linearity and symmetry we can set

            bounded_by.o = ((Vector *) &Transformation)[0] * 0.5
                         + ((Vector *) &Transformation)[1] * 0.5
                         + ((Vector *) &Transformation)[2] * 0.5
                         + ((Vector *) &Transformation)[3];

            // Now, test to find longest diagonal

            bounded_by.r = fmax(
                                fmax( (((Vector *) &Transformation)[0] + ((Vector *) &Transformation)[1] + ((Vector *) &Transformation)[2]).nrm() / 2,
                                      (((Vector *) &Transformation)[0] + ((Vector *) &Transformation)[1] - ((Vector *) &Transformation)[2]).nrm() / 2 ),
                                fmax( (((Vector *) &Transformation)[0] - ((Vector *) &Transformation)[1] + ((Vector *) &Transformation)[2]).nrm() / 2,
                                      (((Vector *) &Transformation)[0] - ((Vector *) &Transformation)[1] - ((Vector *) &Transformation)[2]).nrm() / 2 ) );

			Globals.PrimitivesUsed++;

            break;

        case SG_CYLINDER:
            Transformation = Geometry_Transformation.invert();

            // By linearity and symmetry we get

            bounded_by.o = ((Vector *) &Transformation)[0] * 0.5
                         + ((Vector *) &Transformation)[3];


            // Test to find longest section

            // FIXUP: Improve (approximates cube)

            bounded_by.r = fmax(
                                fmax( (((Vector *) &Transformation)[0] * 0.5 + ((Vector *) &Transformation)[1] + ((Vector *) &Transformation)[2]).nrm(),
                                      (((Vector *) &Transformation)[0] * 0.5 + ((Vector *) &Transformation)[1] - ((Vector *) &Transformation)[2]).nrm() ),
                                fmax( (((Vector *) &Transformation)[0] * 0.5 - ((Vector *) &Transformation)[1] + ((Vector *) &Transformation)[2]).nrm(),
                                      (((Vector *) &Transformation)[0] * 0.5 - ((Vector *) &Transformation)[1] - ((Vector *) &Transformation)[2]).nrm() ) );

			Globals.PrimitivesUsed++;

            break;

        case SG_CONE:
            Transformation = Geometry_Transformation.invert();
            
            // FIXUP: Really bad approximation! Also, doesn't seem to work!

            bounded_by.o = ((Vector *) &Transformation)[0] * 0.5
                         + ((Vector *) &Transformation)[3];

            bounded_by.r = fmax(
                                fmax( (((Vector *) &Transformation)[0] * 0.5 + (((Vector *) &Transformation)[1] + ((Vector *) &Transformation)[2]) * fmax( ((SG_cone *) Geometry)->Radius_1, ((SG_cone *) Geometry)->Radius_2)).nrm(),
                                      (((Vector *) &Transformation)[0] * 0.5 + (((Vector *) &Transformation)[1] - ((Vector *) &Transformation)[2]) * fmax( ((SG_cone *) Geometry)->Radius_1, ((SG_cone *) Geometry)->Radius_2)).nrm() ),
                                fmax( (((Vector *) &Transformation)[0] * 0.5 - (((Vector *) &Transformation)[1] - ((Vector *) &Transformation)[2]) * fmax( ((SG_cone *) Geometry)->Radius_1, ((SG_cone *) Geometry)->Radius_2)).nrm(),
                                      (((Vector *) &Transformation)[0] * 0.5 - (((Vector *) &Transformation)[1] + ((Vector *) &Transformation)[2]) * fmax( ((SG_cone *) Geometry)->Radius_1, ((SG_cone *) Geometry)->Radius_2)).nrm() ) );

			Globals.PrimitivesUsed++;
            
            break;

        case SG_PLANE:
            ((SG_plane *) Geometry)->Normal = (Geometry_Transformation.invert()).TransformNormal( Vector( 1, 0, 0 ) );
            ((SG_plane *) Geometry)->Normal.Normalise(1);

			Globals.PrimitivesUsed++;

            break;
        case SG_QUADRIC:

			Globals.PrimitivesUsed++;

            break;
        default:
            printf("Unexpected/unknown object type %li in bounding routine!\n", Object_Type);
            exit(EXIT_FAILURE);
            break;

        }

	if(inverse == 1) {
		// It all falls apart...
		bounded_by.r = 0;
		}

    return bounded_by;
    }

Token *Object::Load(Token *T) {
    Object **First_Object;
    Token *S;
    Texture *Current_Texture;
    Matrix Transformation;
    Vector v1, v2, v3, n1, n2, n3;
    double r1;
    long Number_Objects, Depth, i, Current_Object;

    Object_Type = 0;
    Geometry = NULL;
    Surface = Default_Texture;
    Geometry_Transformation = Matrix( 1, 0, 0,  0, 1, 0,  0, 0, 1,  0, 0, 0 );
    Texture_Transformation = Matrix( 1, 0, 0,  0, 1, 0,  0, 0, 1,  0, 0, 0 );
    no_shadow = 1; // is really a test of if set, cast shadow
    inverse = 0;
    bounded_by.o = Vector( 0, 0, 0 );
    bounded_by.r = 0;

    // Loading will be contained within the object class - this allows
    // a minimum of duplication

    switch(T->ID) {
        case ID_union:
            Object_Type = CSG_UNION;
            Geometry = new CSG_union;
			Globals.MemoryUsed += sizeof(CSG_union);
	        ((CSG_union *) Geometry)->Number_Objects = 0;
            ((CSG_union *) Geometry)->Object_List = NULL;
            break;
        case ID_merge:
            Object_Type = CSG_MERGE;
            Geometry = new CSG_merge;
			Globals.MemoryUsed += sizeof(CSG_merge);
            ((CSG_merge *) Geometry)->Number_Objects = 0;
            ((CSG_merge *) Geometry)->Object_List = NULL;
            break;
        case ID_difference:
            Object_Type = CSG_DIFFERENCE;
            Geometry = new CSG_difference;
			Globals.MemoryUsed += sizeof(CSG_difference);
            ((CSG_difference *) Geometry)->Number_Objects = 0;
            ((CSG_difference *) Geometry)->Object_List = NULL;
            break;
        case ID_intersection:
            Object_Type = CSG_INTERSECTION;
            Geometry = new CSG_intersection;
			Globals.MemoryUsed += sizeof(CSG_intersection);
            ((CSG_intersection *) Geometry)->Number_Objects = 0;
            ((CSG_intersection *) Geometry)->Object_List = NULL;
            break;
        case ID_triangle:
            Object_Type = SG_TRIANGLE;
            Geometry = new SG_triangle;
			Globals.MemoryUsed += sizeof(SG_triangle);
            ((SG_triangle *) Geometry)->Normal = Vector( 0, 0, 1 );
            break;
        case ID_smooth_triangle:
            Object_Type = SG_SMOOTH_TRIANGLE;
            Geometry = new SG_smooth_triangle;
			Globals.MemoryUsed += sizeof(SG_smooth_triangle);
            ((SG_smooth_triangle *) Geometry)->Normal_1 = Vector( 0, 0, 1 );
            ((SG_smooth_triangle *) Geometry)->Normal_2 = Vector( 0, 0, 1 );
            ((SG_smooth_triangle *) Geometry)->Normal_3 = Vector( 0, 0, 1 );
            break;
        case ID_sphere:
            Object_Type = SG_SPHERE;
            Geometry = new SG_sphere;
			Globals.MemoryUsed += sizeof(SG_sphere);
            break;
        case ID_box:
            Object_Type = SG_BOX;
            Geometry = new SG_box;
			Globals.MemoryUsed += sizeof(SG_box);
            break;
        case ID_cylinder:
            Object_Type = SG_CYLINDER;
            Geometry = new SG_cylinder;
			Globals.MemoryUsed += sizeof(SG_cylinder);
            ((SG_cylinder *) Geometry)->open = 1;
            break;
        case ID_cone:
            Object_Type = SG_CONE;
            Geometry = new SG_cone;
			Globals.MemoryUsed += sizeof(SG_cone);
            ((SG_cone *) Geometry)->Radius_1 = 1;
            ((SG_cone *) Geometry)->Radius_2 = 1;
            ((SG_cone *) Geometry)->open = 1;
            break;
        case ID_plane:
            Object_Type = SG_PLANE;
            Geometry = new SG_plane;
			Globals.MemoryUsed += sizeof(SG_plane);
            ((SG_plane *) Geometry)->Normal = Vector( 0, 0, 0 );
            break;
        case ID_quadric:
            Object_Type = SG_QUADRIC;
            Geometry = new SG_quadric;
			Globals.MemoryUsed += sizeof(SG_quadric);
            ((SG_quadric *) Geometry)->cxx =  1;
            ((SG_quadric *) Geometry)->cyy =  1;
            ((SG_quadric *) Geometry)->czz =  1;
            ((SG_quadric *) Geometry)->cxy =  0;
            ((SG_quadric *) Geometry)->cxz =  0;
            ((SG_quadric *) Geometry)->cyz =  0;
            ((SG_quadric *) Geometry)->cx  =  0;
            ((SG_quadric *) Geometry)->cy  =  0;
            ((SG_quadric *) Geometry)->cz  =  0;
            ((SG_quadric *) Geometry)->c   = -1;
            break;
        case ID_light_source:
            Object_Type = SG_LIGHT_SOURCE;
            Geometry = new Light_Source;
			Globals.MemoryUsed += sizeof(Light_Source);
            ((Light_Source *) Geometry)->fade_power = 0;
            ((Light_Source *) Geometry)->fade_distance = 1;
			((Light_Source *) Geometry)->LensFlare = 0;
            break;
        default:
            PrintToken(T);
            printf("found, object definition expected\n");
            break;
        }

    T = T->Next;

    if(T->ID != ID_LEFT_BRACE) {
        PrintToken(T);
        printf(" found, { expected\n");
        exit(EXIT_FAILURE);
        }

    if(Object_Type < 0) {
        // If CSG, then count members

        S = T->Next;
        Depth = 1;
        Number_Objects = 0;

        do {
            switch(S->ID) {
                case ID_LEFT_BRACE:
                    Depth++;
                    break;
                case ID_RIGHT_BRACE:
                    Depth--;
                    break;
                case ID_union:
                case ID_merge:
                case ID_intersection:
                case ID_difference:
                case ID_triangle:
                case ID_smooth_triangle:
                case ID_sphere:
                case ID_box:
                case ID_cylinder:
                case ID_cone:
                case ID_plane:
                case ID_quadric:
                    if(Depth == 1) {
                        Number_Objects++;
                        }
                    break;
                case ID_END:
                    printf("End of file found in object definition %s at line %li\n", T->FileName, T->Line);
                    exit(EXIT_FAILURE);
                    break;
                default:
                    // Not interested
                    break;
                }
            S = S->Next;
            } while(Depth > 0);

        if(Number_Objects == 0) {
            printf("Empty CSG object in %s at line %li, please remove\n", T->FileName, T->Line);
            exit(EXIT_FAILURE);
            }

        if(Number_Objects == 1) {
            if(Globals.Verbose >= 1) {
                printf("Warning: Should be more than 1 object in CSG in %s at line %li\n", T->FileName, T->Line);
                }
            }

        switch(Object_Type) {
            case CSG_UNION:
                ((CSG_union *) Geometry)->Number_Objects = Number_Objects;
                ((CSG_union *) Geometry)->Object_List = new Object *[Number_Objects];
				Globals.MemoryUsed += sizeof(Object *) * Number_Objects;
                First_Object = ((CSG_union *) Geometry)->Object_List;
                break;
            case CSG_MERGE:
                ((CSG_merge *) Geometry)->Number_Objects = Number_Objects;
                ((CSG_merge *) Geometry)->Object_List = new Object *[Number_Objects];
				Globals.MemoryUsed += sizeof(Object *) * Number_Objects;
                First_Object = ((CSG_merge *) Geometry)->Object_List;
                break;
            case CSG_INTERSECTION:
                ((CSG_intersection *) Geometry)->Number_Objects = Number_Objects;
                ((CSG_intersection *) Geometry)->Object_List = new Object *[Number_Objects];
				Globals.MemoryUsed += sizeof(Object *) * Number_Objects;
                First_Object = ((CSG_intersection *) Geometry)->Object_List;
                break;
            case CSG_DIFFERENCE:
                ((CSG_difference *) Geometry)->Number_Objects = Number_Objects;
                ((CSG_difference *) Geometry)->Object_List = new Object *[Number_Objects];
				Globals.MemoryUsed += sizeof(Object *) * Number_Objects;
                First_Object = ((CSG_difference *) Geometry)->Object_List;
                break;
            default:
                printf("Object::Load: CSG is not CSG, apparently ?!\n");
                exit(EXIT_FAILURE);
                break;
            }


//        printf("Successfully allocated CSG memory\n");

        Current_Object = 0;

        i = 0;
        while(i < Number_Objects) {
            First_Object[i] = NULL;
            i++;
            }

        T = T->Next;

        do {

//            PrintToken(T);
//            printf("now parsing\n");

            switch(T->ID) {

                case ID_union:
                case ID_merge:
                case ID_intersection:
                case ID_difference:
                case ID_triangle:
                case ID_smooth_triangle:
                case ID_sphere:
                case ID_box:
                case ID_cylinder:
                case ID_cone:
                case ID_plane:
                case ID_quadric:
                    First_Object[Current_Object] = new Object;
					Globals.MemoryUsed += sizeof(Object);
                    T = First_Object[Current_Object]->Load(T);
                    Current_Object++;
                    break;

                case ID_texture:
                case ID_colour:
                case ID_color:
                case ID_pigment:
                case ID_normal:
                case ID_finish:
                    Current_Texture = new Texture;
					Globals.MemoryUsed += sizeof(Texture);
                    T = Current_Texture->Load(T);

                    i = 0;
                    while(i < Current_Object) {
                        First_Object[i]->Paint(Current_Texture);
                        i++;
                        }

                    break;

                case ID_translate:
                case ID_scale:
                case ID_rotate:
                case ID_matrix:

                    T = Transformation.Load(T);

//                    printf("Have loaded transformation\n");

                    // Ripple transformation through objects:

                    i = 0;
                    while((First_Object[i] != NULL) && (i < Number_Objects)) {
//                        printf("Transforming member %li\n", i);
                        First_Object[i]->Transform(Transformation);
                        i++;
                        }
                    break;

                case ID_inverse:
                    inverse = !inverse;
                    break;
                case ID_bounded_by:
                    T = T->Next;
                    if(T->ID != ID_LEFT_BRACE) {
                        PrintToken(T);
                        printf("found, expected {\n");
                        exit(EXIT_FAILURE);
                        }
                    T = T->Next;
                    T = bounded_by.o.Load(T);
                    T = T->Next;
                    if(T->ID != ID_COMMA) {
                        PrintToken(T);
                        printf("found, expected comma\n");
                        exit(EXIT_FAILURE);
                        }
                    T = T->Next;
                    if(T->ID != ID_DOUBLE) {
                        PrintToken(T);
                        printf("found, expected number\n");
                        exit(EXIT_FAILURE);
                        }
                    bounded_by.r = *((double *) T->Data);
                    T = T->Next;
                    if(T->ID != ID_RIGHT_BRACE) {
                        PrintToken(T);
                        printf("found, expected }\n");
                        exit(EXIT_FAILURE);
                        }
                    break;
                default:
                    PrintToken(T);
                    printf("found in object definition\n");
                    exit(EXIT_FAILURE);
                    break;
                }

            T = T->Next;

            } while(T->ID != ID_RIGHT_BRACE);

        // Loaded

        return T;

        }

    // We are here => solid geometry (or light_source);

    Depth = 1;

    do {

        T = T->Next;

        switch(T->ID) {

            case ID_DOUBLE:
            case ID_LEFT_ANGLE:

                // We are about to read in data, which can be variously
                // interpreted depending on the object type

                switch(Object_Type) {
                    
                    // A lot of the matrices produced in this section may
                    // simply be wrong. Will require extensive debugging

                    case SG_TRIANGLE:

                        T = v1.Load(T);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        T = v2.Load(T);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        T = v3.Load(T);

                        // We now need to establish a transformation which
                        // takes v1 to {1,0,0}, v2 to {0,1,0},
                        // and v3 to {0,0,0}. By way of weeding out
                        // degenerate triangles we also impose the condition
                        // that (v1 - v3) / (v2 - v3) (Cross product) goes to
                        // {0,0,1}.
                        // To do this, we create the simple inverse first,
                        // then attempt to invert it. Failure => degenerate
                        // triangle.
                        // Later versions may make this a NULL object
                        // if degenerate rather than stopping program. So...

                        // Mask the matrix as four vectors.

                        ((Vector *) &Transformation)[0] = v1 - v3;
                        ((Vector *) &Transformation)[1] = v2 - v3;
                        ((Vector *) &Transformation)[2] = (v1 - v3) / (v2 - v3);
                        ((Vector *) &Transformation)[3] = v3;

                        // So, Transformation is now loaded.

                        if(Transformation.det() == 0) {
                            printf("Degenerate triangle in %s at line %li, please remove\n", T->FileName, T->Line);
                            exit(EXIT_FAILURE);
                            }

                        Geometry_Transformation = Transformation.invert();

                        // FIXUP:
                        // The normal changes with any transformations, so
                        // don't work it out until after all transformations
                        // i.e. in bounding routine
                        
                        break;

                    case SG_SMOOTH_TRIANGLE:

                        T = v1.Load(T);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        T = n1.Load(T);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        T = v2.Load(T);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        T = n2.Load(T);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        T = v3.Load(T);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        T = n3.Load(T);

                        // We now need to establish a transformation which
                        // takes v1 to {1,0,0}, v2 to {0,1,0},
                        // and v3 to {0,0,0}. By way of weeding out
                        // degenerate triangles we also impose the condition
                        // that (v1 - v3) / (v2 - v3) (Cross product) goes to
                        // {0,0,1}.
                        // To do this, we create the simple inverse first,
                        // then attempt to invert it. Failure => degenerate
                        // triangle.
                        // Later versions may make this a NULL object
                        // if degenerate rather than stopping program. So...

                        // Mask the matrix as four vectors.

                        ((Vector *) &Transformation)[0] = v1 - v3;
                        ((Vector *) &Transformation)[1] = v2 - v3;
                        ((Vector *) &Transformation)[2] = (v1 - v3) / (v2 - v3);
                        ((Vector *) &Transformation)[3] = v3;

                        // So, Transformation is now loaded.

                        if(Transformation.det() == 0) {
                            printf("Degenerate triangle in %s at line %li, please remove\n", T->FileName, T->Line);
                            exit(EXIT_FAILURE);
                            }

                        Geometry_Transformation = Transformation.invert();

                        ((SG_smooth_triangle *) Geometry)->Normal_1 = Geometry_Transformation.TransformNormal( n1 );
                        ((SG_smooth_triangle *) Geometry)->Normal_2 = Geometry_Transformation.TransformNormal( n2 );
                        ((SG_smooth_triangle *) Geometry)->Normal_3 = Geometry_Transformation.TransformNormal( n3 );

                        break;

                    case SG_SPHERE:

                        T = v1.Load(T);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        if(T->ID != ID_DOUBLE) {
                            PrintToken(T);
                            printf("found, number expected\n");
                            exit(EXIT_FAILURE);
                            }
                        r1 = *((double *) T->Data);

                        if(r1 == 0) {
                            printf("Degenerate sphere in %s at line %li, please remove\n", T->FileName, T->Line);
                            exit(EXIT_FAILURE);
                            }

                        // Setup matrix transformation

                        Transformation = Matrix( r1, 0, 0,  0, r1, 0,  0, 0, r1,  v1.x, v1.y, v1.z );

                        Geometry_Transformation = Transformation.invert();

                        break;
                        
                    case SG_BOX:
                        T = v1.Load(T);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        T = v2.Load(T);
                        
                        if((v1.x == v2.x) || (v1.y == v2.y) || (v1.z == v2.z)) {
                            printf("Degenerate box in %s at line %li, please remove\n", T->FileName, T->Line);
                            exit(EXIT_FAILURE);
                            }

                        Transformation = Matrix( v2.x - v1.x, 0, 0,  0, v2.y - v1.y, 0,  0, 0, v2.z - v1.z,  v1.x, v1.y, v1.z );

                        Geometry_Transformation = Transformation.invert();

                        break;

                    case SG_CYLINDER:
                        // This will be nasty...

                        T = v1.Load(T);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        T = v2.Load(T);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        if(T->ID != ID_DOUBLE) {
                            PrintToken(T);
                            printf("found, number expected\n");
                            exit(EXIT_FAILURE);
                            }
                        r1 = *((double *) T->Data);

                        // We need a transformation to take an arbitrary
                        // cylinder to a { 0, 0, 0 }, { 1, 0, 0 }, 1
                        // primitive cylinder.

                        // Start with the inverse:
                        // { 0, 0, 0 } -> v1
                        // { 1, 0, 0 } -> v2
                        // And radius increased by factor of r1. So...

                        // We seek two vectors to act like a coordinate
                        // basis in the new space...

                        n1 = (v1 - v2);

                        if((n1 == Vector( 0, 0, 0 )) || (r1 == 0)) {
                            printf("Degenerate cylinder in %s at line %li, please remove\n", T->FileName, T->Line);
                            exit(EXIT_FAILURE);
                            }

                        n1.Normalise(1);


                        // Create vectors with as many zero entries as possible
                        
                        if(n1.x != 0) {
                            // We can define a perpendicular vector thus
                            n2 = n1 / (Vector( 0, 0, 1 ));
                            n3 = n1 / n2;
                            }
                        else {
                            if(n1.y != 0) {
                                // We can define a perpendicular vector thus
                                n2 = Vector( 1, 0, 0 );
                                n3 = n1 / n2;
                                }
                            else {
                                // By non-zero, must have (n1.z != 0)
                                // We can define a perpendicular vector thus
                                n2 = Vector( 1, 0, 0 );
                                n3 = Vector( 0, 1, 0 );
                                }
                            }

                        n2.Normalise(r1);
                        n3.Normalise(r1);

                        ((Vector *) &Transformation)[0] = v2 - v1;
                        ((Vector *) &Transformation)[1] = n2;
                        ((Vector *) &Transformation)[2] = n3;
                        ((Vector *) &Transformation)[3] = v1;

                        // Can't be degenerate... (I hope!)

                        Geometry_Transformation = Transformation.invert();

                        break;
    
                    case SG_CONE:

                        // This is a bit trickier - as the cone can't be
                        // reduced to a linearly transformed primitive
                        // We perform the same transformation as the
                        // cylinder and store the radii for intersection testing.

                        T = v1.Load(T);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        if(T->ID != ID_DOUBLE) {
                            PrintToken(T);
                            printf("found, number expected\n");
                            exit(EXIT_FAILURE);
                            }
                        ((SG_cone *) Geometry)->Radius_1 = *((double *) T->Data);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        T = v2.Load(T);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        if(T->ID != ID_DOUBLE) {
                            PrintToken(T);
                            printf("found, number expected\n");
                            exit(EXIT_FAILURE);
                            }
                        ((SG_cone *) Geometry)->Radius_2 = *((double *) T->Data);

                        // As for a radius 1 cylinder...

                        n1 = (v1 - v2);

                        if((n1 == Vector( 0, 0, 0 )) || (( ((SG_cone *) Geometry)->Radius_1 == 0) && ( ((SG_cone *) Geometry)->Radius_2 == 0))) {
                            printf("Degenerate cone in %s at line %li, please remove\n", T->FileName, T->Line);
                            exit(EXIT_FAILURE);
                            }

                        // Create vectors with as many zero entries as possible
                        
                        if(n1.x != 0) {
                            // We can define a perpendicular vector thus
                            n2 = n1 / Vector( 0, 0, 1 );
                            n3 = n1 / n2;
                            }
                        else {
                            if(n1.y != 0) {
                                // We can define a perpendicular vector thus
                                n2 = Vector( 1, 0, 0 );
                                n3 = n1 / n2;
                                }
                            else {
                                // By non-zero, must have (n1.z != 0)
                                // We can define a perpendicular vector thus
                                n2 = Vector( 1, 0, 0 );
                                n3 = Vector( 0, 1, 0 );
                                }
                            }

                        n2.Normalise(1); // Keep radial distances untouched,
                        n3.Normalise(1); // use radii in intersection model

                        ((Vector *) &Transformation)[0] = v2 - v1;
                        ((Vector *) &Transformation)[1] = n2;
                        ((Vector *) &Transformation)[2] = n3;
                        ((Vector *) &Transformation)[3] = v1;

                        // Can't be degenerate... (I hope!)

                        Geometry_Transformation = Transformation.invert();

                        break;
                    

                    case SG_PLANE:
                        
                        T = n1.Load(T);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        if(T->ID != ID_DOUBLE) {
                            PrintToken(T);
                            printf("found, number expected\n");
                            exit(EXIT_FAILURE);
                            }
                        r1 = *((double *) T->Data);

                        if( n1 == Vector( 0, 0, 0 ) ) {
                            printf("Degenerate plane in %s at line %li, please remove\n", T->FileName, T->Line);
                            exit(EXIT_FAILURE);
                            }

                        // Transform to an y-z plane through the origin
                        // Work again on inverse transformation:
                        // For this we again need a new basis; recycle
                        // code from cylinder...

                        if(n1.x != 0) {
                            // We can define a perpendicular vector thus
                            n2 = n1 / Vector( 0, 0, 1 );
                            n3 = n1 / n2;
                            }
                        else {
                            if(n1.y != 0) {
                                // We can define a perpendicular vector thus
                                n2 = Vector( 1, 0, 0 );
                                n3 = n1 / n2;
                                }
                            else {
                                // By non-zero, must have (n1.z != 0)
                                // We can define a perpendicular vector thus
                                n2 = Vector( 1, 0, 0 );
                                n3 = Vector( 0, 1, 0 );
                                }
                            }

                        n2.Normalise(1);
                        n3.Normalise(1);

                        ((Vector *) &Transformation)[0] = n1;
                        ((Vector *) &Transformation)[1] = n2;
                        ((Vector *) &Transformation)[2] = n3;
                        ((Vector *) &Transformation)[3] = n1*r1;

                        // Can't be degenerate... (I hope!)

                        Geometry_Transformation = Transformation.invert();

                        break;

                    case SG_QUADRIC:
                        T = v1.Load(T);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        T = v2.Load(T);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        T = v3.Load(T);
                        T = T->Next;
                        if(T->ID != ID_COMMA) {
                            PrintToken(T);
                            printf("found, comma expected\n");
                            exit(EXIT_FAILURE);
                            }
                        T = T->Next;
                        if(T->ID != ID_DOUBLE) {
                            PrintToken(T);
                            printf("found, number expected\n");
                            exit(EXIT_FAILURE);
                            }
                        ((SG_quadric *) Geometry)->c = *((double *) T->Data);

                        if((v1.sqn() == 0) && (v2.sqn() == 0)) {
                            printf("Degenerate quadric in %s at line %li, please remove (or convert to plane)\n", T->FileName, T->Line);
                            exit(EXIT_FAILURE);
                            }

                        // Put values in...

                        ((SG_quadric *) Geometry)->cxx = v1.x;
                        ((SG_quadric *) Geometry)->cyy = v1.y;
                        ((SG_quadric *) Geometry)->czz = v1.z;

                        ((SG_quadric *) Geometry)->cxy = v2.x;
                        ((SG_quadric *) Geometry)->cxz = v2.y;
                        ((SG_quadric *) Geometry)->cyz = v2.z;

                        ((SG_quadric *) Geometry)->cx  = v3.x;
                        ((SG_quadric *) Geometry)->cy  = v3.y;
                        ((SG_quadric *) Geometry)->cz  = v3.z;

                        // Leave it be
                        Geometry_Transformation = Matrix( 1, 0, 0,  0, 1, 0,  0, 0, 1,  0, 0, 0 );

                        break;

                    case SG_LIGHT_SOURCE:

                        T = v1.Load(T);

                        // Just to allow it to be transformed with the
                        // rest of the objects

                        Geometry_Transformation = Matrix( 1, 0, 0,  0, 1, 0,  0, 0, 1,  -v1.x, -v1.y, -v1.z );
                        
						// FIXUP: This apparently useless piece of code is the only thing
						// that stops some weird floating point problems in MSVC++6.0 with fast opt!
						
						/*
						printf("Loaded light source from Token < %g, %g, %g > into matrix with offset < %g, %g, %g >\n",
							v1.x,
							v1.y,
							v1.z,
							Geometry_Transformation.a14,
							Geometry_Transformation.a24,
							Geometry_Transformation.a34
							);

						*/	
																		
                        break;                    
                    }

                break;

            case ID_texture:
            case ID_colour:
            case ID_color:
            case ID_pigment:
            case ID_normal:
            case ID_finish:
                Current_Texture = new Texture;
				Globals.MemoryUsed += sizeof(Texture);
                T = Current_Texture->Load(T);

                Surface = Current_Texture;
                Texture_Transformation = Matrix( 1, 0, 0,  0, 1, 0,  0, 0, 1,  0, 0, 0 );

                break;

            case ID_translate:
            case ID_scale:
            case ID_rotate:
            case ID_matrix:

                T = Transformation.Load(T);
                Geometry_Transformation *= Transformation;
                Texture_Transformation *= Transformation;

                break;

            case ID_inverse:
                inverse = !inverse;
                break;
            case ID_bounded_by:
                T = T->Next;
                if(T->ID != ID_LEFT_BRACE) {
                    PrintToken(T);
                    printf("found, expected {\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                T = bounded_by.o.Load(T);
                T = T->Next;
                if(T->ID != ID_COMMA) {
                    PrintToken(T);
                    printf("found, expected comma\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                bounded_by.r = *((double *) T->Data);
                T = T->Next;
                if(T->ID != ID_RIGHT_BRACE) {
                    PrintToken(T);
                    printf("found, expected }\n");
                    exit(EXIT_FAILURE);
                    }
                break;

            case ID_fade_power:
                if(Object_Type != SG_LIGHT_SOURCE) {
                    PrintToken(T);
                    printf("only legal in light_source { }\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                ((Light_Source *) Geometry)->fade_power = *((double *) T->Data);
                break;
            case ID_fade_distance:
                if(Object_Type != SG_LIGHT_SOURCE) {
                    PrintToken(T);
                    printf("only legal in light_source { }\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                ((Light_Source *) Geometry)->fade_distance = *((double *) T->Data);
                break;
			case ID_lens_flare:
				if(Object_Type != SG_LIGHT_SOURCE) {
                    PrintToken(T);
                    printf("only legal in light_source { }\n");
                    exit(EXIT_FAILURE);
                    }
				((Light_Source *) Geometry)->LensFlare = !((Light_Source *) Geometry)->LensFlare;
				break;
            case ID_open:
                if((Object_Type != SG_CYLINDER) && (Object_Type != SG_CONE)) {
                    PrintToken(T);
                    printf("only legal in cylinder or cone { }\n");
                    exit(EXIT_FAILURE);
                    }
                if(Object_Type == SG_CYLINDER) {
                    ((SG_cylinder *) Geometry)->open = !((SG_cylinder *) Geometry)->open;
                    }
                else {
                    ((SG_cone *) Geometry)->open = !((SG_cone *) Geometry)->open;
                    }
                break;
            case ID_shadowless:
            case ID_no_shadow:
                no_shadow = !no_shadow;
                break;

            // Allow recusion in object definitions

            case ID_triangle:
                if(Object_Type != SG_TRIANGLE) {
                    PrintToken(T);
                    printf("found unexpectedly\n");
                    exit(EXIT_FAILURE);
                    }
                break;
            case ID_smooth_triangle:
                if(Object_Type != SG_SMOOTH_TRIANGLE) {
                    PrintToken(T);
                    printf("found unexpectedly\n");
                    exit(EXIT_FAILURE);
                    }
                break;
            case ID_sphere:
                if(Object_Type != SG_SPHERE) {
                    PrintToken(T);
                    printf("found unexpectedly\n");
                    exit(EXIT_FAILURE);
                    }
                break;
            case ID_box:
                if(Object_Type != SG_BOX) {
                    PrintToken(T);
                    printf("found unexpectedly\n");
                    exit(EXIT_FAILURE);
                    }
                break;
            case ID_cylinder:
                if(Object_Type != SG_CYLINDER) {
                    PrintToken(T);
                    printf("found unexpectedly\n");
                    exit(EXIT_FAILURE);
                    }
                break;
            case ID_cone:
                if(Object_Type != SG_CONE) {
                    PrintToken(T);
                    printf("found unexpectedly\n");
                    exit(EXIT_FAILURE);
                    }
                break;
            case ID_plane:
                if(Object_Type != SG_PLANE) {
                    PrintToken(T);
                    printf("found unexpectedly\n");
                    exit(EXIT_FAILURE);
                    }
                break;
            case ID_quadric:
                if(Object_Type != SG_QUADRIC) {
                    PrintToken(T);
                    printf("found unexpectedly\n");
                    exit(EXIT_FAILURE);
                    }
                break;
            case ID_light_source:
                if(Object_Type != SG_LIGHT_SOURCE) {
                    PrintToken(T);
                    printf("found unexpectedly\n");
                    exit(EXIT_FAILURE);
                    }
                break;

            case ID_LEFT_BRACE:
                Depth++;
                break;

            case ID_RIGHT_BRACE:
                Depth--;
                break;

            default:
                PrintToken(T);
                printf("found in object definition\n");
                exit(EXIT_FAILURE);
                break;

            }

        } while(Depth > 0);

    return T;

    }
       
void Frame::Show(void) {
    long i = 0;

    printf("frame {\n");

    while(i < Number_Objects) {
        Object_List[i]->Show();
        i++;
        }

    i = 0;

    printf("    Contains %li light_source(s)\n", Number_Light_Sources);

    while(i < Number_Light_Sources) {
        Light_Source_List[i]->Show();
        i++;
        }

    printf("    }\n");

    }


void Frame::Bound(void) {
    // This routine both bounds shapes and extracts light sources

    long i = 0, j;
    

    while(i < Number_Objects) {
        if( Object_List[i]->Object_Type == SG_LIGHT_SOURCE ) {

            // Transfer light_source to Light_Source_List;

            Number_Objects--;

            j = 0;

            while(Light_Source_List[j] != NULL) {
                j++;
                }

            Light_Source_List[j] = Object_List[i];

            j = i;

            while(j < Number_Objects) {
                Object_List[j] = Object_List[j + 1];
                j++;
                }
            // Don't increment i, as it now indexes a new object
            }
        else {
            i++;
            }
        }

    i = 0;

    while(i < Number_Objects) {
        Object_List[i]->Bound(Light_Source_List);
        i++;
        }

    }

Token *Frame::Load(Token *T, Vector v) {
    Token *S;
    Texture *Current_Texture;
    
    long Current_Object;

    Matrix Transformation;

    long Depth, Frame_Depth, i;

    T = T->Next;

    // Check syntax

    if(T->ID != ID_LEFT_BRACE) {
        PrintToken(T);
        printf(" found, { expected");
        exit(EXIT_FAILURE);
        }

    T = T->Next;

    Depth = 1;

    // Initialise variables:

    Velocity = v;
    Number_Objects = 0;
    Number_Light_Sources = 0;
    Object_List = NULL;
    Light_Source_List = NULL;

    // Count components of the frame

    S = T;

    do {
        switch(S->ID) {
            case ID_LEFT_BRACE:
                Depth++;
                break;
            case ID_RIGHT_BRACE:
                Depth--;
                break;
            case ID_light_source:   // Counts as a light_source and an object
                Number_Light_Sources++;
                // Flow into objects
            case ID_union:
            case ID_merge:
            case ID_intersection:
            case ID_difference:
            case ID_triangle:
            case ID_smooth_triangle:
            case ID_sphere:
            case ID_box:
            case ID_cylinder:
            case ID_cone:
            case ID_plane:
            case ID_quadric:
                if(Depth == 1) {    // i.e. if we are not in a CSG
                    Number_Objects++;
                    }
                break;

            case ID_frame:

                S = S->Next;

                if(S->ID != ID_LEFT_BRACE) {
                    PrintToken(S);
                    printf("found, expected {\n");
                    exit(EXIT_FAILURE);
                    }

                Frame_Depth = 1;

                do {
                    S = S->Next;
                    switch(S->ID) {
                        case ID_LEFT_BRACE: 
                            Frame_Depth++;
                            break;
                        case ID_RIGHT_BRACE:
                            Frame_Depth--;
                            break;
                        case ID_END:
                            PrintToken(S);
                            printf("found, expected continued frame definition\n");
                            exit(EXIT_FAILURE);
                            break;
                        default:
                            break;
                        }
                    } while(Frame_Depth > 0);

                break;

            case ID_END:
                PrintToken(S);
                printf("found unexpectedly\n");
                exit(EXIT_FAILURE);
                break;

            default:
                break;

            }

        S = S->Next;

        } while(Depth > 0);

    // Leaves loop when we get to sublevel

    Object_List = new Object *[Number_Objects];
	Globals.MemoryUsed += sizeof(Object *) * Number_Objects;
    Light_Source_List = new Object *[Number_Light_Sources];
	Globals.MemoryUsed += sizeof(Object *) * Number_Light_Sources;

    Current_Object = 0;

    i = 0;
    while(i < Number_Objects) {
        Object_List[i] = NULL;
        i++;
        }

    i = 0;
    while(i < Number_Light_Sources) {
        Light_Source_List[i] = NULL;
        i++;
        }

    do {

        switch(T->ID) {
            case ID_velocity:

                T = T->Next;

                T = Velocity.Load(T);

                // We need the velocity relative to the global zero velocity,
                // not the parent frame as given

                Velocity = Velocity.LorentzTransform(v, Vector( 0.0, 0.0, 0.0 ) );

                break;

            case ID_frame:
                (*Current_Frame) = new Frame;
				Globals.MemoryUsed += sizeof(Frame);
                T = (*Current_Frame)->Load(T, Velocity);
                Current_Frame++;
                break;

            case ID_camera:
                (*Current_Camera) = new Camera;
				Globals.MemoryUsed += sizeof(Camera);
                T = (*Current_Camera)->Load(T, Velocity);
                Current_Camera++;
                break;

            case ID_triangle:
            case ID_smooth_triangle:
            case ID_sphere:
            case ID_box:
            case ID_cylinder:
            case ID_cone:
            case ID_plane:
            case ID_quadric:
            case ID_light_source:
            case ID_union:
            case ID_merge:
            case ID_difference:
            case ID_intersection:
                Object_List[Current_Object] = new Object;
				Globals.MemoryUsed += sizeof(Object);
                T = Object_List[Current_Object]->Load(T);
                Current_Object++;
                break;

            case ID_texture:
            case ID_colour:
            case ID_color:
            case ID_pigment:
            case ID_normal:
            case ID_finish:
                Current_Texture = new Texture;
				Globals.MemoryUsed += sizeof(Texture);
                T = Current_Texture->Load(T);

                i = 0;
                while(i < Current_Object) {
                    Object_List[i]->Paint(Current_Texture);
                    i++;
                    }

                break;

            case ID_translate:
            case ID_scale:
            case ID_rotate:
            case ID_matrix:
                T = Transformation.Load(T);

                // Ripple transformation through objects:

                i = 0;
                while(i < Current_Object) {
                    Object_List[i]->Transform(Transformation);
                    i++;
                    }

                break;

            case ID_RIGHT_BRACE:    // Only possible for first interior token
                printf("Empty frame definition in %s at line %li, please remove\n", T->FileName, T->Line);
                exit(EXIT_FAILURE);
                break;

            default:
                PrintToken(T);
                printf(" not allowed in frame definition\n");
                exit(EXIT_FAILURE);
                break;
            }

        T = T->Next;

        } while(T->ID != ID_RIGHT_BRACE);

    return T;

    }


//
//  Tokeniser functions
//

int Test(char **p, char *t) {
    long i = 0;

    // Run through characters until end of either string

    while(((*p)[i] == t[i]) && ((*p)[i] != 0)) {
        ++i;
        }

    if((t[i] == 0) && ((!(isalnum((*p)[i]) || ((*p)[i] == '_'))) || ((*p)[i] == 0))) { 

        // First deviation occurs at t's terminator AND there is space after
        // => t occurs at *p

        (*p) += i;
        return 1; 
        }
    else {
        return 0;
        }
    }


Declared *FirstDeclare;

Token *IdentifyKeyword(Token *T, char **p) {
    Declared *d;
    Token *S;
    long i = 0;

    T->ID = -1;

    do {
        if(Test(p, Keyword[i].String)) {
            T->ID = Keyword[i].ID;
            T->Data = NULL;
            T->Next = new Token;
			Globals.MemoryUsed += sizeof(Token);
            return T->Next;
            }
        } while(++i < ID_ARRAY_SIZE);

    // We are here => not a keyword. So we now parse for #declared
    // identifiers.
    
    d = FirstDeclare;

    while(d->Next != NULL) {
        if(Test(p, d->DeclaredIdentifier)) {

            // Copy token string from declare:

            S = d->FirstToken;

            while(S->ID != ID_END) {

                T->ID        = S->ID      ;
                T->Line      = S->Line    ;
                T->FileName  = S->FileName;
                T->Data      = S->Data    ;   // This means we can't free 'Data'
                                            // when updating declares
                T->Next = new Token; 
				Globals.MemoryUsed += sizeof(Token);
                S = S->Next;
                T = T->Next;
                }
            
            return T;
            }
        d = d->Next;
        }

    // We are here => Does not match any identifier

    printf("Undeclared identifier in %s at line %li: %.32s\n", T->FileName, T->Line, *p);
    exit(EXIT_FAILURE);

	return T;

    }



// Prototype:
Token *TokenizeFile(char *Input, Token *FirstToken);
Token *GetNextToken(Token *T, char **p, long *Current_Line);

Token *IdentifyDirective(Token *T, char **p, long *Current_Line) {
    Token *Q, *R, *S;
    char *NewName;
    Declared *d;
    long i;

    if(Test(p,"include")) {

        do {
            (*p)++;
            if(*(*p) == '\n') {
                (*Current_Line)++;
                }
            } while(isspace(*(*p)));
        
        if(*(*p) != '\"') {
            printf("Expected [\"] in %s at line %li, instead found %.32s\n", T->FileName, *Current_Line, *p);
            exit(EXIT_FAILURE);
            }

        i = -1;

        do {
            (*p)++;
            i++;
            } while((*(*p) != '\"') && (*(*p) != 0));

        if(*(*p) == 0) {
            printf("Unterminated string constant in %s starting at line %li: %.32s\n", T->FileName, *Current_Line, *p - i - 1);
            exit(EXIT_FAILURE);
            }

        if(i == 0) {
            printf("Null string constant in %s at line %li: %.32s\n", T->FileName, *Current_Line, *p - i - 1);
            exit(EXIT_FAILURE);
            }

        NewName = new char[i + 1];
		Globals.MemoryUsed += i + 1;
        memcpy(NewName, *p - i, i);
        NewName[i] = 0;

        T = TokenizeFile(NewName, T); // Tokenises the include file

        (*p) += 2;

        // Don't free NewName - leave it for future generations of tokens
        // to refer to

        return T;

        }

    if(Test(p, "declare")) {

        do {
            (*p)++;
            if(*(*p) == '\n') {
                (*Current_Line)++;
                }
            } while(isspace(*(*p)));

        // Prevent definition of unaccessable identifiers

        if(!isalpha(**p)) {
            printf("Expected letter to start identifier in %s at line %li, instead found %.32s\n", T->FileName, *Current_Line, *p);
            exit(EXIT_FAILURE);
            }

        i = 0;
        do {
            (*p)++;
            i++;
            } while(isalnum(*(*p)) || (*(*p) == '_'));

        NewName = new char[i + 1];
		Globals.MemoryUsed += i + 1;
        memcpy(NewName, *p - i, i);
        NewName[i] = 0;

        do {
            (*p)++;
            if(*(*p) == '\n') {
                (*Current_Line)++;
                }
            } while(isspace(*(*p)));

        if(*(*p) != '=') {
            printf("Expected [=] in %s at line %li, instead found %.32s\n", T->FileName, *Current_Line, *p);
            exit(EXIT_FAILURE);
            }
            
        (*p)++;

        // Now, read in the tokens...

        // FIXUP: To define colours, transformations etc. we need the
        // ability to read until depth = 0 on a new line, NOT
        // the retrictive # and # { # # # } forms allowed at the moment

        S = new Token; 
		Globals.MemoryUsed += sizeof(Token);

        S->Line = *Current_Line;
        S->FileName = T->FileName;

        R = GetNextToken(S, p, Current_Line);

        R->Line = *Current_Line;
        R->FileName = T->FileName;


        if(**p == 0) {
            printf("Expected declared item in %s at line %li, instead found end of file\n", T->FileName, *Current_Line);
            exit(EXIT_FAILURE);
            }
        
        switch(S->ID) {

            case ID_DOUBLE:
                // #declare NUMBER
                // This is a one-token linked list...
                R->ID = ID_END;
                R->Line = *Current_Line;
                R->FileName = T->FileName;
                R->Data = NULL;
                R->Next = NULL;
                break;
            case ID_LEFT_ANGLE:
                // #declare < NUMBER, .... NUMBER >
                // Go until we get a '>'

                Q = R;

                do {
                    R = Q;

                    Q = GetNextToken(R, p, Current_Line);

                    Q->Line = *Current_Line;
                    Q->FileName = T->FileName;

                    } while((R->ID != ID_RIGHT_ANGLE) && (**p != 0));

                if(**p == 0) {
                    printf("Expected continued declaration in %s at line %li, instead found end of file\n", T->FileName, *Current_Line);
                    exit(EXIT_FAILURE);
                    }

                // End the list

                Q->ID = ID_END;
                Q->Line = *Current_Line;
                Q->FileName = T->FileName;
                Q->Data = NULL;
                Q->Next = NULL;
                break;
            default:
                     // #declare Identifier = Previously_Declared OR
                     // #declare Identifier = Something { Stuff }

                long i = 0;

                Q = R;
                
                do {
                    R = Q;

                    Q = GetNextToken(R, p, Current_Line);

                    Q->Line = *Current_Line;
                    Q->FileName = T->FileName;

                    // Use i to keep track of how deep we are in

                    if(R->ID == ID_LEFT_BRACE) {
                        i++;
                        }
                    if(R->ID == ID_RIGHT_BRACE) {
                        i--;
                        }

                    } while((i > 0) && (**p != 0));

                if(**p == 0) {
                    printf("Expected continued declaration in %s at line %li, instead found end of file\n", T->FileName, *Current_Line);
                    exit(EXIT_FAILURE);
                    }

                // End the list

                if(R->ID == ID_RIGHT_BRACE) {
                    // Simple
                    Q->ID = ID_END;
                    Q->Line = *Current_Line;
                    Q->FileName = T->FileName;
                    Q->Data = NULL;
                    Q->Next = NULL;
                    }
                else {
                    // Oops - we have read something we shouldn't have.
                    // We have to reinsert it into the main stream

                    T->ID = R->ID;
                    T->Line = R->Line;
                    T->FileName = T->FileName;
                    T->Data = R->Data;
                    T->Next = new Token; 
					Globals.MemoryUsed += sizeof(Token);
                    T = T->Next;

                    // Rid ourselves of the useless Q;

                    free(Q);

                    // Cap the declare

                    R->ID = ID_END;
                    R->Line = *Current_Line;
                    R->FileName = T->FileName;
                    R->Data = NULL;
                    R->Next = NULL;
                    }

                break;
  

            }

        

        // Now, we insert the newly declared token string into the list of
        // declared token strings

        d = FirstDeclare;

        while(d->Next != NULL) {
            if(Test(&NewName, d->DeclaredIdentifier)) {
                // free the old declare

				printf("Warning: Redefining identifier %s\n", NewName);

                R = d->FirstToken;

                while(R != NULL) {
                    Q = R->Next;
                    free(R);
                    R = Q;
                    }

                d->FirstToken = S;
                d->Next = new Declared;
				Globals.MemoryUsed += sizeof(Declared);
                d = d->Next;
                d->Next = NULL;
                return T;
                }
            d = d->Next;
            }

        // Does not appear in the list, so we add a new part.

        d->DeclaredIdentifier = NewName;
        d->FirstToken = S;
        d->Next = new Declared;
		Globals.MemoryUsed += sizeof(Declared);
        d = d->Next;
        d->Next = NULL;
                
        // So, the result of the above is:
        //      We a have a new or updated string of tokens Declared
        //      We have moved *p to the end of the declare
        //      We have not changed T at all, so will continue as though
        //          the declare did not exist

        return T;

        }
 
    // Neither #include or #declare

    printf("Undefined prepocessor directive in %s at line %li: %.32s\n", T->FileName, T->Line, *p - 1);
    exit(EXIT_FAILURE);    

	return T;

    }


Token *GetNextToken(Token *T, char **p, long *Current_Line) {
    long i;

    // This is massive - tests for all the possible first letters
    // individually (!)

        // Move to next text

        while(isspace(*(*p))) {
            if(*(*p) == '\n') {
                (*Current_Line)++;
                T->Line++;
                }
            (*p)++;
            }

        switch((*p)[0]) {

            case 0:
                // We do nothing: the token remains unused, and the
                // 0 character breaks the calling loop
                break;

            case '/':   // Comments

                switch((*p)[1]) {

                    case '/':  // For these comments, nesting is unimportant
                        (*p)++;
                        do {
                            (*p)++;
                            } while((*(*p) != '\n') && (*(*p) != 0));
                        break;

                    case '*':  // These comments can be nested
                               // Ironically, DJGPP doesn't seem to
                               // actually support nested comments
                               // of this type.
                        i = 1;
                        (*p) += 2;
                        do {
                            switch((*p)[0]) {
                                case '/':
                                    switch((*p)[1]) {
                                        case '*':
                                            i++;
                                            (*p) += 2;
                                            break;
                                        default:
                                            (*p)++;
                                            break;
                                        }
                                    break;
                                case '*':
                                    switch((*p)[1]) {
                                        case '/':
                                            i--;
                                            (*p) += 2;
                                            break;
                                        default:
                                            (*p)++;
                                            break;
                                        }
                                    break;
                                default:
                                    (*p)++;
                                    break;
                                }       

                            if(*(*p) == '\n') {
                                (*Current_Line)++;
                                T->Line++;
                                }

                            } while((*(*p) != 0) && (i != 0));

                        if(i != 0) {
                            printf("Unterminated comment in %s\n", T->FileName);
                            exit(EXIT_FAILURE);
                            }
                        break;
                    default:
                        printf("Unknown identifier [/] in %s line %li : %.32s ...\n", T->FileName, (*Current_Line), (*p));
                        exit(EXIT_FAILURE);
                        break;
                    }
                break;

            case '0':   // doubles. Cases overflow to produce an OR effect;
            case '1':
            case '2':                
            case '3':                        
            case '4':
            case '5':
            case '6':
            case '7':
            case '8':
            case '9':
            case '.':
            case '+':
            case '-':

                T->ID = ID_DOUBLE;
                T->Data = new double;  // Read in double
				Globals.MemoryUsed += sizeof(double);
                    *((double *) T->Data) = strtod((*p), p);
                T->Next = new Token; 
				Globals.MemoryUsed += sizeof(Token);
       
                T = T->Next;

                break;

            //Punctuation: Each case sets
            case '{':
                T->ID = ID_LEFT_BRACE;
                T->Data = NULL;
                T->Next = new Token; 
				Globals.MemoryUsed += sizeof(Token);
                T = T->Next;
                (*p)++;
                break;
            case '}':
                T->ID = ID_RIGHT_BRACE;
                T->Data = NULL;
                T->Next = new Token; 
				Globals.MemoryUsed += sizeof(Token);
                T = T->Next;
                (*p)++;
                break;
            case '<':
                T->ID = ID_LEFT_ANGLE;
                T->Data = NULL;
                T->Next = new Token; 
				Globals.MemoryUsed += sizeof(Token);
                T = T->Next;
                (*p)++;
                break;
            case '>':
                T->ID = ID_RIGHT_ANGLE;
                T->Data = NULL;
                T->Next = new Token; 
				Globals.MemoryUsed += sizeof(Token);
                T = T->Next;
                (*p)++;
                break;
            case ',':
                T->ID = ID_COMMA;
                T->Data = NULL;
                T->Next = new Token; 
				Globals.MemoryUsed += sizeof(Token);
                T = T->Next;
                (*p)++;
                break;
            case '=':
                T->ID = ID_EQUALS;
                T->Data = NULL;
                T->Next = new Token; 
				Globals.MemoryUsed += sizeof(Token);
                T = T->Next;
                (*p)++;
                break;
            case '\"':
                T->ID = ID_STRING;
                (*p)++;
                i = 0;
               
                while((**p != '\"') && (**p != 0)) {

                    if(**p == '\n') {
                        (*Current_Line)++;
                        }

                    i++;
                    (*p)++;
                    }

                if(**p == 0) {
                    printf("Unterminated string constant in %s starting line %li", T->FileName, T->Line);
                    exit(EXIT_FAILURE);
                    }

                T->Line = *Current_Line;

                if(i == 0) {
                    printf("Null string constant in %s starting line %li", T->FileName, T->Line);
                    exit(EXIT_FAILURE);
                    }

                T->Data = new char *[i + 1];
				Globals.MemoryUsed += i + 1;

                memcpy(T->Data, *p - i, i);

                ((char *) T->Data)[i] = 0;

                T->Next = new Token; 
				Globals.MemoryUsed += sizeof(Token);
                T = T->Next;
                (*p)++;

                break;

            // Preprocessor directives:

            case '#':
                (*p)++;

                T = IdentifyDirective(T, p, Current_Line);
                break;

            // Keywords or identifiers. Note that while identifiers can
            // contain anything EXCEPT whitespace, they can only start with
            // an upper or lower case letter

            case 'a':
            case 'b':
            case 'c':
            case 'd':
            case 'e':
            case 'f':
            case 'g':
            case 'h':
            case 'i':
            case 'j':
            case 'k':
            case 'l':
            case 'm':
            case 'n':
            case 'o':
            case 'p':
            case 'q':
            case 'r':
            case 's':
            case 't':
            case 'u':
            case 'v':
            case 'w':
            case 'x':
            case 'y':
            case 'z':
            case 'A':
            case 'B':
            case 'C':
            case 'D':
            case 'E':
            case 'F':
            case 'G':
            case 'H':
            case 'I':
            case 'J':
            case 'K':
            case 'L':
            case 'M':
            case 'N':
            case 'O':
            case 'P':
            case 'Q':
            case 'R':
            case 'S':
            case 'T':
            case 'U':
            case 'V':
            case 'W':
            case 'X':
            case 'Y':
            case 'Z':

                T = IdentifyKeyword(T, p);

                break;
                        
            default:
                printf("Disallowed character %.1s in %.32s\n", *p, T->FileName);
                exit(EXIT_FAILURE);
                break;

            }
    

    // A precaution...
    
    T->FileName = NULL;

    return T;

    }

Token *TokenizeFile(char *Input, Token *FirstToken) {
    FILE *Handle;
    char *Text, *p;
    Token *T;
    long FileLength, Current_Line;

    if(Globals.Verbose >= 1) {
        printf ("Opening %.32s\n", Input);
        }

    Handle = fopen(Input, "rt");

    if(Handle == NULL) {
        printf("Fatal Error: Can't open %s\n", Input);
        exit(EXIT_FAILURE);
        }

    FileLength = 0;

    do {
        FileLength++;
        } while(getc(Handle) != EOF);

    if(Globals.Verbose >= 2) {
        printf("File length is %li bytes\n", FileLength);
        }

    Text = new char[FileLength];
	Globals.MemoryUsed += FileLength;

    fseek(Handle, 0, 0);
    fread(Text, 1, FileLength, Handle);

    // Don't close until the file is finished - this means it can't be
    // opened again by itself or it's #include files - and so is safe
    // from cyclic effects ?

    Text[FileLength-1] = 0;

    T = FirstToken;
    Current_Line = 1;
    p = Text;

    // Parsing...

    do {

        T->Line = Current_Line;
        T->FileName = Input;

        T = GetNextToken(T, &p, &Current_Line);

        } while(*p != 0);

    free(Text);

    // We can now safely allow other subroutines to access the file

    fclose(Handle);

    if(Globals.Verbose >= 2) {
        printf("File read and closed\n");
        }

    // Return the final token

    return T;

    }

void PrintToken(Token *T) {

    printf("%s:%li: ", T->FileName, T->Line);

    switch(T->ID) {

        case ID_LEFT_BRACE:
            printf("{ ");
            break;
        case ID_RIGHT_BRACE:
            printf("} ");
            break;
        case ID_LEFT_ANGLE:
            printf("< ");
            break;
        case ID_RIGHT_ANGLE:
            printf("> ");
            break;
        case ID_COMMA:
            printf(", ");
            break;
        case ID_EQUALS:
            printf("= ");
            break;
        case ID_DOUBLE:
            printf("%g ", *((double *) T->Data));
            break;
        case ID_STRING:
            //printf("String = \"%s\" ", ((char *) T->Data));
            printf("String\n");
            break;
        case ID_END:
            printf("End of file ");
            break;
        default:
            if(T->ID < ID_ARRAY_SIZE) {
                printf("%s ", Keyword[T->ID].String);
                }
            else {
                printf("Weird T->ID: %li\n", T->ID);
                exit(EXIT_FAILURE);
                }
            break;
        }

    }

void Scene::Load(char *FileName) {
    Token *FirstToken;
    Token *T;

    long i;

    // Initialise linked list of tokens

    FirstToken = new Token;
	Globals.MemoryUsed += sizeof(Token);

    // Initialise linked list of Declares;
    FirstDeclare = new Declared;
	Globals.MemoryUsed += sizeof(Declared);
    FirstDeclare->Next = NULL;

    T = TokenizeFile(FileName, FirstToken);

    T->ID = ID_END;
    T->Line = 0;
    T->FileName = FileName;
    T->Data = NULL;
    T->Next = NULL;

    // We now have a list of tokens, each corresponding to a directive in the
    // source file.

    // FIXUP: We can't check this until Globals.Verbose is actually parsed!

    if(Globals.Verbose >= 3) {
        // Display list...
        T = FirstToken;
        do {
            PrintToken(T);
            printf("\n");
            T = T->Next;
            } while(T->ID != ID_END);
        }

    // Setup lists of frames and cameras

    Number_Cameras = 0;
    Number_Frames = 0;

    T = FirstToken;

    do {
        // Count scene components

        if(T->ID == ID_camera) {
            Number_Cameras++;
            }
        if(T->ID == ID_frame) {
            Number_Frames++;
            }
        T = T->Next;
        } while(T->ID != ID_END);

    if(Globals.Verbose >= 1) {
        printf("Scene contains %li camera(s) and %li inertial frame(s)\n", Number_Cameras, Number_Frames);
        }

    if((Number_Cameras == 0) || (Number_Frames == 0)) {
        printf("Scene must contain at least one camera and one inertial frame\n");
        exit(EXIT_FAILURE);
        }

    Camera_List = new Camera *[Number_Cameras];
	Globals.MemoryUsed += sizeof(Camera *) * Number_Cameras;
    Frame_List = new Frame *[Number_Frames];
	Globals.MemoryUsed += sizeof(Frame) * Number_Frames;

    Current_Frame = Frame_List;
    Current_Camera = Camera_List;

    // Kick off parsing list...

    // Initialise properties...

    uImage = 160;
    vImage = 120;
    OutFile = "Out";

    T = FirstToken;

    do {
        switch(T->ID) {
            case ID_frame:
                (*Current_Frame) = new Frame;
				Globals.MemoryUsed += sizeof(Frame);
                T = (*Current_Frame)->Load(T, Vector( 0.0, 0.0, 0.0 ) );
                Current_Frame++;
                break;
            case ID_camera:
                (*Current_Camera) = new Camera;
				Globals.MemoryUsed += sizeof(Camera);
                T = (*Current_Camera)->Load(T, Vector( 0.0, 0.0, 0.0 ) );
                Current_Camera++;
                break;

            case ID_Height:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                vImage = (long int) ((double) *((double *) T->Data));
                break;
            case ID_Width:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                uImage = (long int) ((double) *((double *) T->Data));
                break;
            case ID_Output_File_Name:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_STRING) {
                    PrintToken(T);
                    printf("found, expected string\n");
                    exit(EXIT_FAILURE);
                    }
                OutFile = (char *) T->Data;
                break;
            case ID_Verbose:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                Globals.Verbose = *((double *) T->Data);
                break;
            case ID_Display:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                Globals.Display = *((double *) T->Data);
                break;
            case ID_Doppler:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                Globals.Doppler = *((double *) T->Data);
				Globals.NW_Doppler = *((double *) T->Data);
                break;
            case ID_Intensity:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                Globals.Intensity = *((double *) T->Data);
                Globals.NW_Intensity = *((double *) T->Data);
                break;
            case ID_Antialias:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                Globals.Antialias = *((double *) T->Data);
                break;
            case ID_Sampling_Method:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                Globals.Sampling_Method = *((double *) T->Data);
                break;
            case ID_Antialias_Threshold:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                Globals.Antialias_Threshold = *((double *) T->Data);
                break;
            case ID_Antialias_Depth:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                Globals.Antialias_Depth = *((double *) T->Data);
                break;
            case ID_Window_t:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                Globals.Window_t = *((double *) T->Data);
                break;
            case ID_Window_b:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                Globals.Window_b = *((double *) T->Data);
                break;
            case ID_Window_r:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                Globals.Window_r = *((double *) T->Data);
                break;
            case ID_Window_l:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                Globals.Window_l = *((double *) T->Data);
                break;
            case ID_Window_Doppler:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                Globals.Window_Doppler = *((double *) T->Data);
                break;
            case ID_Window_Intensity:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                Globals.Window_Intensity = *((double *) T->Data);
                break;
            case ID_HUD:
                T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                Globals.HUD = *((double *) T->Data);
                break;
			case ID_Continue:
				T = T->Next;
                if(T->ID != ID_EQUALS) {
                    PrintToken(T);
                    printf("found, expected =\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                if(T->ID != ID_DOUBLE) {
                    PrintToken(T);
                    printf("found, expected number\n");
                    exit(EXIT_FAILURE);
                    }
                Globals.Continue = (long) *((double *) T->Data);
                break;

            case ID_default:
                if(T->ID != ID_LEFT_BRACE) {
                    PrintToken(T);
                    printf("found, expected {\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                T = Default_Texture->Load(T);
                T = T->Next;
                break;

            case ID_background:
            case ID_sky_sphere:
                T = T->Next;
                if(T->ID != ID_LEFT_BRACE) {
                    PrintToken(T);
                    printf("found, expected {\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                while(T->ID != ID_RIGHT_BRACE) {
                    if( (T->ID != ID_texture) && (T->ID != ID_pigment) && (T->ID != ID_colour) && (T->ID != ID_color) && (T->ID != ID_velocity)) {
                        PrintToken(T);
                        printf("found, expected texture definition\n");
                        exit(EXIT_FAILURE);
                        }
                    if(T->ID == ID_velocity) {
                        T = T->Next;
                        T = SkySphereVelocity.Load(T);
                        }
                    else {
                        T = SkySphereSurface.Load(T);
                        }
                    T = T->Next;
                    }
                break;
            case ID_global_settings:
                T = T->Next;
                if(T->ID != ID_LEFT_BRACE) {
                    PrintToken(T);
                    printf("found, expected {\n");
                    exit(EXIT_FAILURE);
                    }
                T = T->Next;
                while(T->ID != ID_RIGHT_BRACE) {
                    switch(T->ID) {
                        case ID_adc_bailout:
                            T = T->Next;
                            if(T->ID != ID_DOUBLE) {
                                PrintToken(T);
                                printf("found, expected number\n");
                                exit(EXIT_FAILURE);
                                }
                            Globals.adc_bailout = *((double *) T->Data);
                            break;
                        case ID_assumed_gamma:
                            T = T->Next;
                            if(T->ID != ID_DOUBLE) {
                                PrintToken(T);
                                printf("found, expected number\n");
                                exit(EXIT_FAILURE);
                                }
                            Globals.assumed_gamma = *((double *) T->Data);
                            break;
                        case ID_max_trace_level:
                            T = T->Next;
                            if(T->ID != ID_DOUBLE) {
                                PrintToken(T);
                                printf("found, expected number\n");
                                exit(EXIT_FAILURE);
                                }
                            Globals.max_trace_level = *((double *) T->Data);
                            break;
                        default:
                            PrintToken(T);
                            printf("is not permitted in global_settings { }\n");
                            exit(EXIT_FAILURE);
                            break;
                        }
                    T = T->Next;
                    }
                break;
            default:
                PrintToken(T);
                printf("found, illegal at top level\n");
                exit(EXIT_FAILURE);
                break;
            }
        T = T->Next;
        } while(T->ID != ID_END);

    if(Globals.Verbose >= 2) {
        printf("Parsed successfully\n");
        }

    // We now need to traverse the heierachy to extract the light sources,
    // and put in place bounding shapes

	if(Globals.Verbose > 0) {
		printf("Entering bounding routine...\n");
	}

    i = 0;

    while(i < Number_Frames) {
        Frame_List[i]->Bound();
        i++;
        }


    if(Globals.Verbose >= 2) {
        printf("Showing heierachy...\n");

        i = 0;

        while(i < Number_Frames) {
            Frame_List[i]->Show();
            i++;
            }

        }

    LensFlare.LoadTGA("LensFlare.tga");
    HUD.LoadTGA("HUD.tga");

    }

int main(int argc, char *argv[]) {
    Scene World;
    Texture Default;

    printf("\n    BACKLIGHT Relativistic Raytracer v2.9\n\n    Copyright (c) Antony Searle 1997-8\n\n");

    // Look at arguments...

    if(argc != 2) {
        printf("Syntax: BackLight [input file name]\n");
        exit(EXIT_FAILURE);
        }

    // Setup some globals...

    First_Bitmap = NULL;

    Default_Texture = &Default;

    Default.P.map_type = MT_COLOUR;
    Default.P.interpolate = 0;
    Default.P.colour = fColour( 0, 0, 0, 0, 0 );
    Default.P.bitmap = NULL;
    Default.P.frequency_u = 1;
    Default.P.frequency_v = 1;
    Default.P.phase_u = 0;
    Default.P.phase_v = 0;
    Default.P.drift_u = 0;
    Default.P.drift_v = 0;
    Default.P.once = 0;
    Default.P.doppler = 1.0;
    Default.P.intensity = 1.0;
    Default.P.Transformation = Matrix( 1, 0, 0,  0, 1, 0,  0, 0, 1,  0, 0, 0 );

    Default.N.map_type = MT_NONE;
    Default.N.interpolate = 0;
    Default.N.frequency_u = 1;
    Default.N.frequency_v = 1;
    Default.N.phase_u = 0;
    Default.N.phase_v = 0;
    Default.N.drift_u = 0;
    Default.N.drift_v = 0;
    Default.N.once = 0;
    Default.N.bitmap = NULL;
    Default.N.bump_size = 1;
    Default.N.Transformation = Matrix( 1, 0, 0,  0, 1, 0,  0, 0, 1,  0, 0, 0 );

    Default.F.ambient = fColour( 0, 0, 0, 0, 0 );
    Default.F.diffuse = 1;
    Default.F.brilliance = 1;
    Default.F.specular = 0;
    Default.F.roughness = 0.1;
    Default.F.reflection = fColour( 0, 0, 0, 0, 0 );
    Default.F.refraction = 0;
    Default.F.ior = 1;
    Default.F.metallic = 0;

    SkySphereSurface = Default;
    SkySphereVelocity = Vector( 0, 0, 0 );

    Globals.Verbose = 1;
    Globals.Antialias = 0;
    Globals.Sampling_Method = 1;
    Globals.Antialias_Threshold = 0.3;
    Globals.Antialias_Depth = 2;
    Globals.adc_bailout = 1/255;
    Globals.assumed_gamma = 1;
    Globals.max_trace_level = 5;
    Globals.Display = 0;

    Globals.Doppler = 1;
    Globals.Intensity = 1;
    Globals.Relativity = 1;
    Globals.Observation = 0;

	Globals.NW_Intensity = 1;
	Globals.NW_Doppler = 1;

	Globals.Window_t = 1;
	Globals.Window_b = 1;
	Globals.Window_l = 1;
	Globals.Window_r = 1;

	Globals.Window_Doppler = 0;
	Globals.Window_Intensity = 0;

	Globals.HUD = 0;

    Globals.Tokens = 0;

	Globals.MemoryUsed = 0;
	Globals.PrimitivesUsed = 0;

	Globals.Continue = 0;

    World.Load(argv[1]);

    World.Raytrace();

	return(EXIT_SUCCESS);

    }


