//////////////////////////////////////////////////////////////////////////////////
// Project Name:   [ CDX Class Library - CDX.lib ]
// Source File:    [ CDXSurface Implementation ]
// Author:         [ Bil Simser - bsimser@home.com ]
// Contributions:  [ Dan Farley, Chris Barnes, Alan Simons ]
// Revision:       [ 1.99a ]
//////////////////////////////////////////////////////////////////////////////////
#include "CDX.h"
#include "math.h"

//<JJH>
//const float PI = (float)3.1415927;
//<JJH>

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Constructor
//////////////////////////////////////////////////////////////////////////////////
CDXSurface::CDXSurface(void)
{
	m_lpDDS = NULL;
	m_DC = NULL;
	m_Font = NULL;
	Screen = NULL;
	m_FontLoaded = FALSE;

//<JJH>
	m_pFilename = NULL;
//<JJH>

	m_ColorKey = -1;
	DestRect.top = 0;
	DestRect.left = 0;
	DestRect.bottom = 0;
	DestRect.right = 0;

//<JJH>
	m_MMX = IsMMX();
//<JJH>

}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Constructor
//////////////////////////////////////////////////////////////////////////////////
CDXSurface::CDXSurface(CDXScreen *pScreen, int Width, int Height, BOOL memoryType)
{
	Create(pScreen, Width, Height, memoryType);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Constructor
//////////////////////////////////////////////////////////////////////////////////
CDXSurface::CDXSurface(CDXScreen *pScreen, const char *szFilename, BOOL memoryType)
{
	Create(pScreen, szFilename, memoryType);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Destructor
//////////////////////////////////////////////////////////////////////////////////
CDXSurface::~CDXSurface()
{
//<JJH>
	if(m_Font != NULL) {
		DeleteObject(m_Font);
	}

	if(m_DC != NULL) {
		ReleaseDC();
	}
//<JJH>

//<WPS>
	if(m_FontLoaded) {
		RemoveFontResource(m_FontName);
	}
//<WPS>

	RELEASE(m_lpDDS);

	if (Screen != NULL)
	    RELEASE(Screen->m_lpDD);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Create
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXSurface::Create(CDXScreen *pScreen, int Width, int Height, BOOL memoryType)
{
	HRESULT rval;

	ZeroMemory(&m_DDSD, sizeof(m_DDSD));
	m_DDSD.dwSize = sizeof(m_DDSD);
	m_DDSD.dwFlags = DDSD_CAPS | DDSD_HEIGHT |DDSD_WIDTH;

//<JJH>
	switch(memoryType) {
	    case CDXMEM_SYSTEMONLY:
	        m_DDSD.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;
			break;

		case CDXMEM_VIDEOONLY:
			m_DDSD.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_VIDEOMEMORY;
			break;
		
        case CDXMEM_VIDTHENSYS:
		default:
	        m_DDSD.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
			break;
	}
//<JJH>

	m_DDSD.dwWidth = Width;
	m_DDSD.dwHeight = Height;

	rval = pScreen->m_lpDD->CreateSurface(&m_DDSD, &m_lpDDS, NULL);
	if(rval != DD_OK)
	{
		DDError(rval, pScreen->m_hWnd, __FILE__, __LINE__);
		return FALSE;
	}

	m_DC = NULL;
	m_Font = NULL;
	m_pFilename = NULL;
	Screen = pScreen;
	m_ColorKey = -1;

	m_PixelWidth = Width;
	m_PixelHeight = Height;

	DestRect.top = 0;
	DestRect.left = 0;
	DestRect.bottom = m_PixelHeight;
	DestRect.right = m_PixelWidth;

	SrcRect.top = 0;
	SrcRect.left = 0;
	SrcRect.bottom = m_PixelHeight;
	SrcRect.right = m_PixelWidth;

//<JJH>
	m_ClipRect.top = 0;
	m_ClipRect.left = 0;
	m_ClipRect.bottom = m_PixelHeight - 1;
	m_ClipRect.right = m_PixelWidth - 1;

	GetRGBFormat(m_lpDDS, &m_RGB);
	m_MMX = IsMMX();
	ODS(" AddRef\n");
	pScreen->m_lpDD->AddRef();
//<JJH>
    
	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Create
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXSurface::Create(CDXScreen *pScreen, const char *szFilename, BOOL memoryType)
{
	if(szFilename == NULL) 
		return FALSE;

	m_lpDDS = DDLoadSizeBitmap(pScreen->m_lpDD, szFilename, &m_PixelWidth, &m_PixelHeight, memoryType);
	if(m_lpDDS == NULL) 
		return FALSE;

	m_DC = NULL;
	m_Font = NULL;
	m_pFilename = szFilename;
	Screen = pScreen;
	m_ColorKey = 0;

	DestRect.top = 0;
	DestRect.left = 0;
	DestRect.bottom = m_PixelHeight;
	DestRect.right = m_PixelWidth;

	SrcRect.top = 0;
	SrcRect.left = 0;
	SrcRect.bottom = m_PixelHeight;
	SrcRect.right = m_PixelWidth;

//<JJH>
	m_ClipRect.top		= 0;
	m_ClipRect.left		= 0;
	m_ClipRect.bottom	= m_PixelHeight - 1;
	m_ClipRect.right	= m_PixelWidth - 1;

	GetRGBFormat(m_lpDDS,&m_RGB);
	m_MMX = IsMMX();
	ODS(" AddRef\n");
	pScreen->m_lpDD->AddRef();
//<JJH>
	
	ZeroMemory(&m_DDSD, sizeof(m_DDSD));
	m_DDSD.dwSize = sizeof(m_DDSD);
	m_DDSD.dwFlags = DDSD_CAPS | DDSD_HEIGHT |DDSD_WIDTH;

//<JJH>
	switch (memoryType)
	{
		case CDXMEM_SYSTEMONLY:
			m_DDSD.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;
			break;

		case CDXMEM_VIDEOONLY:
			m_DDSD.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_VIDEOMEMORY;
			break;

		case CDXMEM_VIDTHENSYS:
		default:
			m_DDSD.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
			break;
	}
//<JJH>

	m_DDSD.dwWidth = m_PixelWidth;
	m_DDSD.dwHeight = m_PixelHeight;

	return TRUE;
}

//<JJH>
//////////////////////////////////////////////////////////////////////////////////
// CDXSurface ValdidateBlt
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXSurface::ValidateBlt(CDXSurface* lpCDXS, LONG *lDestX, LONG *lDestY, RECT *srcRect)
{
	RECT destRect;
	RECT tmpRect;

	// Test if srcRect lies somewhere on this surface's clipRect
	if (!(ClipRect(srcRect)))
		return FALSE;

	// From srcRect build the destRect
	destRect.top    = *lDestY;
	destRect.left   = *lDestX;
	destRect.bottom = *lDestY + (srcRect->bottom - srcRect->top);
	destRect.right  = *lDestX + (srcRect->right - srcRect->left);

	// Save off the destRect
	tmpRect.top    = destRect.top;
	tmpRect.left   = destRect.left;
	tmpRect.bottom = destRect.bottom;
	tmpRect.right  = destRect.right;

	// Test if destRect lies somewhere on lpCDXS surface's clipRect
	if (!(lpCDXS->ClipRect(&destRect)))
		return FALSE;

	// Compare the returned destRect to the saved tmpRect
	if (destRect.top != tmpRect.top)
		srcRect->top += destRect.top - tmpRect.top;

	if (destRect.left != tmpRect.left)
		srcRect->left += destRect.left - tmpRect.left;

	if (destRect.bottom != tmpRect.bottom)
		srcRect->bottom += destRect.bottom - tmpRect.bottom;

	if (destRect.right != tmpRect.right)
		srcRect->right += destRect.right - tmpRect.right;

	*lDestX = destRect.left;
	*lDestY = destRect.top;

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface::ClipRect - clips a rectangle to m_clipRect 
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXSurface::ClipRect(RECT *Rect)
{
	// Check if Rect is completely outside of the m_clipRect (Screen Space)
	if (Rect->top >= m_ClipRect.bottom)
		return FALSE;

	if (Rect->left >= m_ClipRect.right)
		return FALSE;

	if (Rect->bottom <= m_ClipRect.top)
		return FALSE;

	if (Rect->right <= m_ClipRect.left)
		return FALSE;


	// Clip rect to the surface's clipRect
	if (Rect->top < m_ClipRect.top)
		Rect->top = m_ClipRect.top;

	if (Rect->left < m_ClipRect.left)
		Rect->left = m_ClipRect.left;

	if (Rect->bottom > m_ClipRect.bottom)
		Rect->bottom = m_ClipRect.bottom;

	if (Rect->right > m_ClipRect.right)
		Rect->right = m_ClipRect.right;

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface::GetClipRect - Fills out the passed RECT with m_clipRect
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::GetClipRect(RECT *clipRect)
{
	clipRect->top = m_ClipRect.top;
	clipRect->left = m_ClipRect.left;
	clipRect->bottom = m_ClipRect.bottom;
	clipRect->right = m_ClipRect.right;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface::SetClipRect - Set m_clipRect to the passed in RECT values
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::SetClipRect(RECT *clipRect)
{
	if ((m_ClipRect.top = clipRect->top) < 0)
		m_ClipRect.top = 0;

	if ((m_ClipRect.left = clipRect->left) < 0)
		m_ClipRect.left = 0;

	if ((m_ClipRect.bottom = clipRect->bottom) > m_PixelHeight)
		m_ClipRect.bottom = m_PixelHeight;

	if ((m_ClipRect.right = clipRect->right) > m_PixelWidth)
		m_ClipRect.right = m_PixelWidth;
}


//<JJH>

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface CDXBLT_Blk
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_Blk(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect)
{
	HRESULT rval;

	if (ValidateBlt(lpCDXS, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;
	
	rval = lpCDXS->m_lpDDS->BltFast(lDestX, lDestY, m_lpDDS, &srcRect, DDBLTFAST_WAIT);

	if(rval == DDERR_SURFACELOST) 
		Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface CDXBLT_Trans
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_Trans(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect)
{
	HRESULT rval;

	if (ValidateBlt(lpCDXS, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;
	
	rval = lpCDXS->m_lpDDS->BltFast(lDestX, lDestY, m_lpDDS, &srcRect, DDBLTFAST_WAIT | DDBLTFAST_SRCCOLORKEY);

	if(rval == DDERR_SURFACELOST) 
		Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface CDXBLT_BlkHFlip
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_BlkHFlip(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect)
{
	HRESULT rval;
	DDBLTFX ddBltFx;
	RECT destRect;
	DWORD Pixel;
	LONG width, height;
	LONG tmpLeft, tmpRight;

	tmpLeft = lDestX;  // equal to or smaller then lDestX
	tmpRight = srcRect.right;

	if (ValidateBlt(lpCDXS, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

	if (tmpLeft != lDestX)
	{
		srcRect.left -= lDestX - tmpLeft;
		srcRect.right -= lDestX - tmpLeft;
		tmpRight -= lDestX - tmpLeft;
	}

	if (tmpRight != srcRect.right)
	{
		srcRect.left += tmpRight - srcRect.right;
		srcRect.right += tmpRight - srcRect.right;
	}

	ddBltFx.dwSize = sizeof(DDBLTFX);
	ddBltFx.dwDDFX = DDBLTFX_MIRRORLEFTRIGHT;

	width           = srcRect.right - srcRect.left;
	height          = srcRect.bottom - srcRect.top;
	destRect.top    = lDestY;
	destRect.left   = lDestX;
	destRect.bottom = lDestY + height;
	destRect.right  = lDestX + width;

	rval = lpCDXS->m_lpDDS->Blt(&destRect, m_lpDDS, &srcRect, DDBLT_DDFX | DDBLT_WAIT, &ddBltFx);
	if (rval == DD_OK)
		return rval;

	for(int j = 0; j < height; j++)
	{
		for(int i = 0; i < width; i++)
		{
			Pixel = GetPixel(srcRect.left + i, srcRect.top + j);
			lpCDXS->PutPixel((destRect.left + width) - i, destRect.top + j, Pixel);
		}
	}
	return DD_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface CDXBLT_TransHFlip
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_TransHFlip(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect)
{
	HRESULT rval;
	DDBLTFX ddBltFx;
	RECT destRect;
	DWORD Pixel;
	LONG width, height;
	LONG tmpLeft, tmpRight;

	tmpLeft = lDestX;  // equal to or smaller then lDestX
	tmpRight = srcRect.right;

	if (ValidateBlt(lpCDXS, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

	if (tmpLeft != lDestX)
	{
		srcRect.left -= lDestX - tmpLeft;
		srcRect.right -= lDestX - tmpLeft;
		tmpRight -= lDestX - tmpLeft;
	}

	if (tmpRight != srcRect.right)
	{
		srcRect.left += tmpRight - srcRect.right;
		srcRect.right += tmpRight - srcRect.right;
	}

	ddBltFx.dwSize = sizeof(DDBLTFX);
	ddBltFx.dwDDFX = DDBLTFX_MIRRORLEFTRIGHT;

	width           = srcRect.right - srcRect.left;
	height          = srcRect.bottom - srcRect.top;
	destRect.top    = lDestY;
	destRect.left   = lDestX;
	destRect.bottom = lDestY + height;
	destRect.right  = lDestX + width;

	rval = lpCDXS->m_lpDDS->Blt(&destRect, m_lpDDS, &srcRect, DDBLT_DDFX | DDBLT_WAIT | DDBLT_KEYSRC, &ddBltFx);
	if (rval == DD_OK)
		return rval;

	for(int j = 0; j < height; j++)
	{
		for(int i = 0; i < width; i++)
		{
			Pixel = GetPixel(srcRect.left + i, srcRect.top + j);
			if (Pixel != m_ColorKey)
				lpCDXS->PutPixel((destRect.left + width) - i, destRect.top + j, Pixel);
		}
	}
	return DD_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface CDXBLT_BlkVFlip
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_BlkVFlip(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect)
{
	HRESULT rval;
	DDBLTFX ddBltFx;
	RECT destRect;
	DWORD Pixel;
	LONG width, height;
	LONG tmpTop, tmpBottom;

	tmpTop = lDestY;  
	tmpBottom = srcRect.bottom;

	if (ValidateBlt(lpCDXS, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

	if (tmpTop != lDestY)
	{
		srcRect.top -= lDestY - tmpTop;
		srcRect.bottom -= lDestY - tmpTop;
		tmpBottom -= lDestY - tmpTop;
	}

	if (tmpBottom != srcRect.bottom)
	{
		srcRect.top += tmpBottom - srcRect.bottom;
		srcRect.bottom += tmpBottom - srcRect.bottom;
	}

	ddBltFx.dwSize = sizeof(DDBLTFX);
	ddBltFx.dwDDFX = DDBLTFX_MIRRORUPDOWN;

	width           = srcRect.right - srcRect.left;
	height          = srcRect.bottom - srcRect.top;
	destRect.top    = lDestY;
	destRect.left   = lDestX;
	destRect.bottom = lDestY + height;
	destRect.right  = lDestX + width;

	rval = lpCDXS->m_lpDDS->Blt(&destRect, m_lpDDS, &srcRect, DDBLT_DDFX | DDBLT_WAIT, &ddBltFx);
	if (rval == DD_OK)
		return rval;

	for(int j = 0; j < height; j++)
	{
		for(int i = 0; i < width; i++)
		{
			Pixel = GetPixel(srcRect.left + i, srcRect.top + j);
			lpCDXS->PutPixel(destRect.left + i, (destRect.top + height) - j, Pixel);
		}
	}
	return DD_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface CDXBLT_TransVFlip
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_TransVFlip(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect)
{
	HRESULT rval;
	DDBLTFX ddBltFx;
	RECT destRect;
	DWORD Pixel;
	LONG width, height;
	LONG tmpTop, tmpBottom;

	tmpTop = lDestY;  
	tmpBottom = srcRect.bottom;

	if (ValidateBlt(lpCDXS, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

	if (tmpTop != lDestY)
	{
		srcRect.top -= lDestY - tmpTop;
		srcRect.bottom -= lDestY - tmpTop;
		tmpBottom -= lDestY - tmpTop;
	}

	if (tmpBottom != srcRect.bottom)
	{
		srcRect.top += tmpBottom - srcRect.bottom;
		srcRect.bottom += tmpBottom - srcRect.bottom;
	}

	ddBltFx.dwSize = sizeof(DDBLTFX);
	ddBltFx.dwDDFX = DDBLTFX_MIRRORUPDOWN;

	width           = srcRect.right - srcRect.left;
	height          = srcRect.bottom - srcRect.top;
	destRect.top    = lDestY;
	destRect.left   = lDestX;
	destRect.bottom = lDestY + height;
	destRect.right  = lDestX + width;

	rval = lpCDXS->m_lpDDS->Blt(&destRect, m_lpDDS, &srcRect, DDBLT_DDFX | DDBLT_WAIT | DDBLT_KEYSRC, &ddBltFx);
	if (rval == DD_OK)
		return rval;

	for(int j = 0; j < height; j++)
	{
		for(int i = 0; i < width; i++)
		{
			Pixel = GetPixel(srcRect.left + i, srcRect.top + j);
			if (Pixel != m_ColorKey)
				lpCDXS->PutPixel(destRect.left + i, (destRect.top + height) - j, Pixel);
		}
	}
	return DD_OK;
}

////////////////////////////////////////////////////////////////////////
// CDXSprite CDXBLT_BlkAlphaFast - 
////////////////////////////////////////////////////////////////////////

HRESULT CDXSurface::CDXBLT_BlkAlphaFast(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect)
{
    int register i,j;
	int height,width;
    BYTE* lpSprite;
    BYTE* lpDest;
    WORD dPitch, SpritePitch;
    DWORD sTemp,dTemp;
    WORD sbuf,dbuf;
    DWORD Result;
	BOOL oddWidth = FALSE;

	// Validate the inputs and clip them to m_clipRect
	if (ValidateBlt(dest, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

    // Set height and width of SPRITE
    height = srcRect.bottom - srcRect.top;
    width = srcRect.right - srcRect.left;

    // Lock down both surfaces for read and write
    Lock();
    dest->Lock();

    // Set the pitch for both surfaces
    SpritePitch = m_DDSD.lPitch;
    dPitch = dest->m_DDSD.lPitch;

    // Initialize the pointers to the upper left hand corner of the surface
    lpSprite = (BYTE*)m_DDSD.lpSurface;
    lpDest = (BYTE*)dest->m_DDSD.lpSurface;

	// Start RGB Format switch statement
	switch(dest->m_RGB.bpp)
	{
	case 8:
		// IMHO paletized modes are a thing of the past please feel free to 
		// implement this if you so desire.
		break;

	case 15:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.


		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);
				dTemp = *((WORD*)lpDest);

				*((WORD*)lpDest) = ((sTemp & 0x7BDE) >> 1) + ((dTemp & 0x7BDE) >> 1);

				lpDest += 2;
				lpSprite += 2;
			}

			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);

				Result = ((sTemp & 0x7BDE) >> 1) + ((dTemp & 0x7BDE) >> 1);
				Result |= ((sTemp & 0x7BDE0000) >> 1) + ((dTemp & 0x7BDE0000) >> 1);

				*((DWORD*)lpDest) = Result;

				lpDest += 4;
				lpSprite += 4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 16:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.

		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);
				dTemp = *((WORD*)lpDest);

				*((WORD*)lpDest) = ((sTemp & 0xF7DE) >> 1) + ((dTemp & 0xF7DE) >> 1);

				lpDest += 2;
				lpSprite += 2;
			}

			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);

				Result = ((sTemp & 0xF7DE) >> 1) + ((dTemp & 0xF7DE) >> 1);
				Result |= ((sTemp & 0xF7DE0000) >> 1) + ((dTemp & 0xF7DE0000) >> 1);

				*((DWORD*)lpDest) = Result;

				lpDest += 4;
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;

	case 24:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 3);
		lpDest += (lDestY * dPitch) + (lDestX * 3);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (3 * width));
		dbuf = (WORD)(dPitch - (3 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);

				Result = ((sTemp & 0xFEFEFE) >> 1) + ((dTemp & 0xFEFEFE) >> 1);

				*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
				lpDest += 2;
				*lpDest = (BYTE)(Result >> 16);

				lpDest++;
				lpSprite += 3;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		
		break;

	case 32:
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 4);
		lpDest += (lDestY * dPitch) + (lDestX * 4);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (4 * width));
		dbuf = (WORD)(dPitch - (4 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);

				Result = ((sTemp & 0xFEFEFE) >> 1) + ((dTemp & 0xFEFEFE) >> 1);

				*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
				lpDest += 2;
				*lpDest = (BYTE)(Result >> 16);

				lpDest += 2;
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;
	} // End RGB Format switch statement


    UnLock();
    dest->UnLock();

	return DD_OK;
}

////////////////////////////////////////////////////////////////////////
// CDXSprite CDXBLT_TRANSALPHAFAST - 
////////////////////////////////////////////////////////////////////////

HRESULT CDXSurface::CDXBLT_TransAlphaFast(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect)
{
    int register i,j;
	int height,width;
    BYTE* lpSprite;
    BYTE* lpDest;
    WORD dPitch, SpritePitch;
    DWORD sColorKey;
    DWORD sTemp,dTemp;
    WORD sbuf,dbuf;
    DWORD Result;
	BOOL oddWidth = FALSE;

	// Validate the inputs and clip them to m_clipRect
	if (ValidateBlt(dest, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

    // Set height and width of SPRITE
    height = srcRect.bottom - srcRect.top;
    width = srcRect.right - srcRect.left;

    // Lock down both surfaces for read and write
    Lock();
    dest->Lock();

    // Set the pitch for both surfaces
    SpritePitch = m_DDSD.lPitch;
    dPitch = dest->m_DDSD.lPitch;

    // Initialize the pointers to the upper left hand corner of the surface
    lpSprite = (BYTE*)m_DDSD.lpSurface;
    lpDest = (BYTE*)dest->m_DDSD.lpSurface;

	// Get the color key for sprite surface
	sColorKey = (DWORD)m_ColorKey;

	// Start RGB Format switch statement
	switch(dest->m_RGB.bpp)
	{
	case 8:
		// IMHO paletized modes are a thing of the past please feel free to 
		// implement this if you so desire.
		break;

	case 15:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.


		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					*((WORD*)lpDest) = ((sTemp & 0x7BDE) >> 1) + ((dTemp & 0x7BDE) >> 1);
				}

				lpDest += 2;
				lpSprite += 2;
			}

			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);
				Result = dTemp;

				if ((sTemp & 0xFFFF) != sColorKey)
				{
					Result &= 0xFFFF0000;
					Result |= ((sTemp & 0x7BDE) >> 1) + ((dTemp & 0x7BDE) >> 1);
				}
				if ((sTemp >> 16) != sColorKey)
				{
					Result &= 0xFFFF;
					Result |= ((sTemp & 0x7BDE0000) >> 1) + ((dTemp & 0x7BDE0000) >> 1);
				}

				*((DWORD*)lpDest) = Result;
				lpDest += 4;
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 16:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.

		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					*((WORD*)lpDest) = ((sTemp & 0xF7DE) >> 1) + ((dTemp & 0xF7DE) >> 1);
				}

				lpDest += 2;
				lpSprite += 2;
			}

			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);
				Result = dTemp;

				if ((sTemp & 0xFFFF) != sColorKey)
				{
					Result &= 0xFFFF0000;
					Result |= ((sTemp & 0xF7DE) >> 1) + ((dTemp & 0xF7DE) >> 1);
				}
				if ((sTemp >> 16) != sColorKey)
				{
					Result &= 0xFFFF;
					Result |= ((sTemp & 0xF7DE0000) >> 1) + ((dTemp & 0xF7DE0000) >> 1);
				}

				*((DWORD*)lpDest) = Result;
				lpDest += 4;
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;

	case 24:
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 3);
		lpDest += (lDestY * dPitch) + (lDestX * 3);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (3 * width));
		dbuf = (WORD)(dPitch - (3 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if ((sTemp &= 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);
					Result = ((sTemp & 0xFEFEFE) >> 1) + ((dTemp & 0xFEFEFE) >> 1);

					*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
					lpDest += 2;
					*lpDest = (BYTE)(Result >> 16);
					lpDest++;
				}
				else
				{
					lpDest += 3;
				}

				lpSprite += 3;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;

	case 32:
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 4);
		lpDest += (lDestY * dPitch) + (lDestX * 4);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (4 * width));
		dbuf = (WORD)(dPitch - (4 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if ((sTemp &= 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);
					Result = ((sTemp & 0xFEFEFE) >> 1) + ((dTemp & 0xFEFEFE) >> 1);

					*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
					lpDest += 2;
					*lpDest = (BYTE)(Result >> 16);
					lpDest += 2;
				}
				else
				{
					lpDest += 4;
				}

				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;
	} // End RGB Format switch statement


    UnLock();
    dest->UnLock();

	return DD_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSprite CDXBLT_BLKALPHA - 
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_BlkAlpha(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect, WORD ALPHA)
{
	int register i,j;
	int height,width;
	BYTE* lpSprite;
	BYTE* lpDest;
	WORD dPitch, SpritePitch;
	DWORD sTemp,dTemp;
	WORD sr,sg,sb,dr,dg,db;
	WORD sbuf,dbuf;
	DWORD Result;
	BOOL oddWidth = FALSE;

	// Validate the inputs and clip them to m_clipRect
	if (ValidateBlt(dest, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

	// Set height and width of SPRITE
    height = srcRect.bottom - srcRect.top;
    width = srcRect.right - srcRect.left; 

    // Lock down both surfaces for read and write
    Lock();
    dest->Lock();

	// Set the pitch for both surfaces
    SpritePitch = m_DDSD.lPitch;
    dPitch = dest->m_DDSD.lPitch;

    // Initialize the pointers to the upper left hand corner of surface
    lpSprite = (BYTE*)m_DDSD.lpSurface;
    lpDest = (BYTE*)dest->m_DDSD.lpSurface;


	switch(dest->m_RGB.bpp)
	{
	case 8:
		// IMHO paletized modes are a thing of the past please feel free to 
		// implement this if you so desire.
		break;

	case 15:

		// Check the ALPHA value
		if (ALPHA < 0)
			ALPHA = 0;
		else if (ALPHA > 255)
			ALPHA = 255;
		
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.

		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);
				dTemp = *((WORD*)lpDest);
				sb = sTemp & 0x1f;
				db = dTemp & 0x1f;
				sg = (sTemp >> 5) & 0x1f;
				dg = (dTemp >> 5) & 0x1f;
				sr = (sTemp >> 10) & 0x1f;
				dr = (dTemp >> 10) & 0x1f;

				*((WORD*)lpDest) = (WORD)((ALPHA * (db - sb) >> 8) + sb |
					((ALPHA * (dg - sg) >> 8) + sg) << 5 |
					((ALPHA * (dr - sr) >> 8) + sr) << 10);

				lpDest += 2;
				lpSprite += 2;
			}
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);

				sb = sTemp & 0x1f;
				db = dTemp & 0x1f;
				sg = (sTemp >> 5) & 0x1f;
				dg = (dTemp >> 5) & 0x1f;
				sr = (sTemp >> 10) & 0x1f;
				dr = (dTemp >> 10) & 0x1f;

				Result = (DWORD)((ALPHA * (db - sb) >> 8) + sb |
					((ALPHA * (dg - sg) >> 8) + sg) << 5 |
					((ALPHA * (dr - sr) >> 8) + sr) << 10);

				sb = (sTemp >> 16) & 0x1f;
				db = (dTemp >> 16) & 0x1f;
				sg = (sTemp >> 21) & 0x1f;
				dg = (dTemp >> 21) & 0x1f;
				sr = (sTemp >> 26) & 0x1f;
				dr = (dTemp >> 26) & 0x1f;

				Result |= (DWORD)( (ALPHA * (db - sb) >> 8) + sb |
					((ALPHA * (dg - sg) >> 8) + sg) << 5 |
					((ALPHA * (dr - sr) >> 8) + sr) << 10 ) << 16;

				*((DWORD*)lpDest) = Result;

				lpDest += 4;
				lpSprite += 4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 16:

		// Check the ALPHA value
		if (ALPHA < 0)
			ALPHA = 0;
		else if (ALPHA > 255)
			ALPHA = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.


		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);
				dTemp = *((WORD*)lpDest);

				sb = sTemp & 0x1f;
				db = dTemp & 0x1f;
				sg = (sTemp >> 5) & 0x3f;
				dg = (dTemp >> 5) & 0x3f;
				sr = (sTemp >> 11) & 0x1f;
				dr = (dTemp >> 11) & 0x1f;

				*((WORD*)lpDest) = (WORD)((ALPHA * (db - sb) >> 8) + sb |
					((ALPHA * (dg - sg) >> 8) + sg) << 5 |
					((ALPHA * (dr - sr) >> 8) + sr) << 11);

				lpDest += 2;
				lpSprite += 2;
			}
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);

				sb = sTemp & 0x1f;
				db = dTemp & 0x1f;
				sg = (sTemp >> 5) & 0x3f;
				dg = (dTemp >> 5) & 0x3f;
				sr = (sTemp >> 11) & 0x1f;
				dr = (dTemp >> 11) & 0x1f;

				Result = (DWORD)((ALPHA * (db - sb) >> 8) + sb |
					((ALPHA * (dg - sg) >> 8) + sg) << 5 |
					((ALPHA * (dr - sr) >> 8) + sr) << 11);

				sb = (sTemp >> 16) & 0x1f;
				db = (dTemp >> 16) & 0x1f;
				sg = (sTemp >> 21) & 0x3f;
				dg = (dTemp >> 21) & 0x3f;
				sr = (sTemp >> 27) & 0x1f;
				dr = (dTemp >> 27) & 0x1f;

				Result |= (DWORD)( (ALPHA * (db - sb) >> 8) + sb |
					((ALPHA * (dg - sg) >> 8) + sg) << 5 |
					((ALPHA * (dr - sr) >> 8) + sr) << 11 ) << 16;

				*((DWORD*)lpDest) = Result;

				lpDest += 4;
				lpSprite +=4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 24:
		// Check the ALPHA value
		if (ALPHA < 0)
			ALPHA = 0;
		else if (ALPHA > 255)
			ALPHA = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 3);
		lpDest += (lDestY * dPitch) + (lDestX * 3);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (3 * width));
		dbuf = (WORD)(dPitch - (3 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);

				sb = sTemp & 0xFF;
				db = dTemp & 0xFF;
				sg = (sTemp >> 8) & 0xFF;
				dg = (dTemp >> 8) & 0xFF;
				sr = (sTemp >> 16) & 0xFF;
				dr = (dTemp >> 16) & 0xFF;

				Result = (DWORD)(ALPHA * (db - sb) >> 8) + sb |
					(DWORD)((ALPHA * (dg - sg) >> 8) + sg) << 8 |
					(DWORD)((ALPHA * (dr - sr) >> 8) + sr) << 16;

				*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
				lpDest += 2;
				*lpDest = (BYTE)(Result >> 16);

				lpDest++;
				lpSprite += 3;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 32:
		// Check the ALPHA value
		if (ALPHA < 0)
			ALPHA = 0;
		else if (ALPHA > 255)
			ALPHA = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 4);
		lpDest += (lDestY * dPitch) + (lDestX * 4);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (4 * width));
		dbuf = (WORD)(dPitch - (4 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);

				sb = sTemp & 0xFF;
				db = dTemp & 0xFF;
				sg = (sTemp >> 8) & 0xFF;
				dg = (dTemp >> 8) & 0xFF;
				sr = (sTemp >> 16) & 0xFF;
				dr = (dTemp >> 16) & 0xFF;

				Result = (DWORD)((ALPHA * (db - sb) >> 8) + sb) |
					(DWORD)((ALPHA * (dg - sg) >> 8) + sg) << 8 |
					(DWORD)((ALPHA * (dr - sr) >> 8) + sr) << 16;

				*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
				lpDest += 2;
				*lpDest = (BYTE)(Result >> 16);

				lpDest += 2;
				lpSprite += 4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;
	} // End RGB Format switch statement


	UnLock();
	dest->UnLock();

	return DD_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface CDXBLT_TransAlpha - 
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_TransAlpha(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect, WORD ALPHA)
{
	int register i,j;
	int height,width;
	BYTE* lpSprite;
	BYTE* lpDest;
	WORD dPitch, SpritePitch;
	DWORD sColorKey;
	DWORD sTemp,dTemp;
	WORD sr,sg,sb,dr,dg,db;
	WORD sbuf,dbuf;
	DWORD Result;
	BOOL oddWidth = FALSE;

	// Validate the inputs and clip them to m_clipRect
	if (ValidateBlt(dest, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

	// Set height and width of SPRITE
    height = srcRect.bottom - srcRect.top;
    width = srcRect.right - srcRect.left; 

    // Get the color key for sprite surface
    sColorKey = (DWORD)m_ColorKey;

    // Lock down both surfaces for read and write
    Lock();
    dest->Lock();

	// Set the pitch for both surfaces
    SpritePitch = m_DDSD.lPitch;
    dPitch = dest->m_DDSD.lPitch;

    // Initialize the pointers to the upper left hand corner of surface
    lpSprite = (BYTE*)m_DDSD.lpSurface;
    lpDest = (BYTE*)dest->m_DDSD.lpSurface;


	switch(dest->m_RGB.bpp)
	{
	case 8:
		// IMHO paletized modes are a thing of the past please feel free to 
		// implement this if you so desire.
		break;

	case 15:

		// Check the ALPHA value
		if (ALPHA < 0)
			ALPHA = 0;
		else if (ALPHA > 255)
			ALPHA = 255;
		
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.

		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					sb = sTemp & 0x1f;
					db = dTemp & 0x1f;
					sg = (sTemp >> 5) & 0x1f;
					dg = (dTemp >> 5) & 0x1f;
					sr = (sTemp >> 10) & 0x1f;
					dr = (dTemp >> 10) & 0x1f;

					*((WORD*)lpDest) = (DWORD)((ALPHA * (db - sb) >> 8) + sb |
						((ALPHA * (dg - sg) >> 8) + sg) << 5 |
						((ALPHA * (dr - sr) >> 8) + sr) << 10);
				}

				lpDest += 2;
				lpSprite += 2;
			}
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if (sTemp != ( (DWORD)sColorKey | ((DWORD)sColorKey << 16)) )
				{
					dTemp = *((DWORD*)lpDest);
					Result = dTemp;

					if ((sTemp & 0xffff) != sColorKey)
					{
						sb = sTemp & 0x1f;
						db = dTemp & 0x1f;
						sg = (sTemp >> 5) & 0x1f;
						dg = (dTemp >> 5) & 0x1f;
						sr = (sTemp >> 10) & 0x1f;
						dr = (dTemp >> 10) & 0x1f;

						Result = Result & 0xffff0000;
						Result |= (DWORD)((ALPHA * (db - sb) >> 8) + sb |
							((ALPHA * (dg - sg) >> 8) + sg) << 5 |
							((ALPHA * (dr - sr) >> 8) + sr) << 10);
					}
					if (((sTemp >> 16) & 0xffff) != sColorKey)
					{
						sb = (sTemp >> 16) & 0x1f;
						db = (dTemp >> 16) & 0x1f;
						sg = (sTemp >> 21) & 0x1f;
						dg = (dTemp >> 21) & 0x1f;
						sr = (sTemp >> 26) & 0x1f;
						dr = (dTemp >> 26) & 0x1f;

						Result = Result & 0xffff;
						Result |= (DWORD)( (ALPHA * (db - sb) >> 8) + sb |
							((ALPHA * (dg - sg) >> 8) + sg) << 5 |
							((ALPHA * (dr - sr) >> 8) + sr) << 10 ) << 16;
					}

					*((DWORD*)lpDest) = Result;
				}
				lpDest += 4;
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;
		}while (--i > 0);

		break;

	case 16:

		// Check the ALPHA value
		if (ALPHA < 0)
			ALPHA = 0;
		else if (ALPHA > 255)
			ALPHA = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.


		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					sb = sTemp & 0x1f;
					db = dTemp & 0x1f;
					sg = (sTemp >> 5) & 0x3f;
					dg = (dTemp >> 5) & 0x3f;
					sr = (sTemp >> 11) & 0x1f;
					dr = (dTemp >> 11) & 0x1f;

					*((WORD*)lpDest) = (DWORD)((ALPHA * (db - sb) >> 8) + sb |
						((ALPHA * (dg - sg) >> 8) + sg) << 5 |
						((ALPHA * (dr - sr) >> 8) + sr) << 11);
				}

				lpDest += 2;
				lpSprite += 2;
			}
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if (sTemp != ( (DWORD)sColorKey | ((DWORD)sColorKey << 16)) )
				{
					dTemp = *((DWORD*)lpDest);
					Result = dTemp;

					if ((sTemp & 0xffff) != sColorKey)
					{
						sb = sTemp & 0x1f;
						db = dTemp & 0x1f;
						sg = (sTemp >> 5) & 0x3f;
						dg = (dTemp >> 5) & 0x3f;
						sr = (sTemp >> 11) & 0x1f;
						dr = (dTemp >> 11) & 0x1f;

						Result = Result & 0xffff0000;
						Result |= (DWORD)((ALPHA * (db - sb) >> 8) + sb |
							((ALPHA * (dg - sg) >> 8) + sg) << 5 |
							((ALPHA * (dr - sr) >> 8) + sr) << 11);
					}
					if (((sTemp >> 16) & 0xffff) != sColorKey)
					{
						sb = (sTemp >> 16) & 0x1f;
						db = (dTemp >> 16) & 0x1f;
						sg = (sTemp >> 21) & 0x3f;
						dg = (dTemp >> 21) & 0x3f;
						sr = (sTemp >> 27) & 0x1f;
						dr = (dTemp >> 27) & 0x1f;

						Result = Result & 0xffff;
						Result |= (DWORD)( (ALPHA * (db - sb) >> 8) + sb |
							((ALPHA * (dg - sg) >> 8) + sg) << 5 |
							((ALPHA * (dr - sr) >> 8) + sr) << 11 ) << 16;
					}

					*((DWORD*)lpDest) = Result;
				}
				lpDest += 4;
				lpSprite +=4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;
		}while (--i > 0);

		break;

	case 24:

		// Check the ALPHA value
		if (ALPHA < 0)
			ALPHA = 0;
		else if (ALPHA > 255)
			ALPHA = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 3);
		lpDest += (lDestY * dPitch) + (lDestX * 3);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (3 * width));
		dbuf = (WORD)(dPitch - (3 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if ((sTemp & 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);
					sb = sTemp & 0xFF;
					db = dTemp & 0xFF;
					sg = (sTemp >> 8) & 0xFF;
					dg = (dTemp >> 8) & 0xFF;
					sr = (sTemp >> 16) & 0xFF;
					dr = (dTemp >> 16) & 0xFF;

					Result = (DWORD)((ALPHA * (db - sb) >> 8) + sb |
						((ALPHA * (dg - sg) >> 8) + sg) << 8 |
						((ALPHA * (dr - sr) >> 8) + sr) << 16);

					*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
					lpDest += 2;
					*lpDest = (BYTE)(Result >> 16);
					lpDest++;
				}
				else
				{
					lpDest += 3;
				}

				lpSprite += 3;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;
		}while (--i > 0);
		break;

	case 32:
		// Check the ALPHA value
		if (ALPHA < 0)
			ALPHA = 0;
		else if (ALPHA > 255)
			ALPHA = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 4);
		lpDest += (lDestY * dPitch) + (lDestX * 4);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (4 * width));
		dbuf = (WORD)(dPitch - (4 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if ((sTemp & 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);
					sb = sTemp & 0xFF;
					db = dTemp & 0xFF;
					sg = (sTemp >> 8) & 0xFF;
					dg = (dTemp >> 8) & 0xFF;
					sr = (sTemp >> 16) & 0xFF;
					dr = (dTemp >> 16) & 0xFF;

					Result = (DWORD)((ALPHA * (db - sb) >> 8) + sb |
						((ALPHA * (dg - sg) >> 8) + sg) << 8 |
						((ALPHA * (dr - sr) >> 8) + sr) << 16);

					*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
					lpDest += 2;
					*lpDest = (BYTE)(Result >> 16);
					lpDest += 2;
				}
				else
				{
					lpDest += 4;
				}

				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;
		}while (--i > 0);
		break;
	} // End RGB Format switch statement


	UnLock();
	dest->UnLock();

	return DD_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface CDXBLT_BlkShadow - 
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_BlkShadow(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect, WORD SHADOW)
{
	int register i,j;
	int height,width;
	BYTE* lpSprite;
	BYTE* lpDest;
	WORD dPitch, SpritePitch;
	DWORD dTemp;
	WORD dr,dg,db;
	WORD sbuf,dbuf;
	DWORD Result;
	BOOL oddWidth = FALSE;

	// Validate the inputs and clip them to m_clipRect
	if (ValidateBlt(dest, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

	// Set height and width of SPRITE
    height = srcRect.bottom - srcRect.top;
    width = srcRect.right - srcRect.left; 

    // Lock down both surfaces for read and write
    Lock();
    dest->Lock();

	// Set the pitch for both surfaces
    SpritePitch = m_DDSD.lPitch;
    dPitch = dest->m_DDSD.lPitch;

    // Initialize the pointers to the upper left hand corner of surface
    lpSprite = (BYTE*)m_DDSD.lpSurface;
    lpDest = (BYTE*)dest->m_DDSD.lpSurface;


	switch(dest->m_RGB.bpp)
	{
	case 8:
		// IMHO paletized modes are a thing of the past please feel free to 
		// implement this if you so desire.
		break;

	case 15:

		// Check the ALPHA value
		if (SHADOW < 0)
			SHADOW = 0;
		else if (SHADOW > 255)
			SHADOW = 255;
		
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.

		i = height;
		do
		{
			if (oddWidth)
			{
				dTemp = *((WORD*)lpDest);
				db = dTemp & 0x1f;
				dg = (dTemp >> 5) & 0x1f;
				dr = (dTemp >> 10) & 0x1f;

				*((WORD*)lpDest) = (WORD)((SHADOW * db) >> 8 |
					((SHADOW * dg) >> 8) << 5 |
					((SHADOW * dr) >> 8) << 10);

				lpDest += 2;
				lpSprite += 2;
			}
			j = width;
			do
			{
				dTemp = *((DWORD*)lpDest);

				db = dTemp & 0x1f;
				dg = (dTemp >> 5) & 0x1f;
				dr = (dTemp >> 10) & 0x1f;

				Result = (DWORD)((SHADOW * db) >> 8 |
					((SHADOW * dg) >> 8) << 5 |
					((SHADOW * dr) >> 8) << 10);

				db = (dTemp >> 16) & 0x1f;
				dg = (dTemp >> 21) & 0x1f;
				dr = (dTemp >> 26) & 0x1f;

				Result |= (DWORD)( (SHADOW * db) >> 8 |
					((SHADOW * dg) >> 8) << 5 |
					((SHADOW * dr) >> 8) << 10 ) << 16;

				*((DWORD*)lpDest) = Result;

				lpDest += 4;
				lpSprite += 4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 16:

		// Check the ALPHA value
		if (SHADOW < 0)
			SHADOW = 0;
		else if (SHADOW > 255)
			SHADOW = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.


		i = height;
		do
		{
			if (oddWidth)
			{
				dTemp = *((WORD*)lpDest);

				db = dTemp & 0x1f;
				dg = (dTemp >> 5) & 0x3f;
				dr = (dTemp >> 11) & 0x1f;

				*((WORD*)lpDest) = (WORD)((SHADOW * db) >> 8 |
					((SHADOW * dg) >> 8) << 5 |
					((SHADOW * dr) >> 8) << 11);

				lpDest += 2;
				lpSprite += 2;
			}
			j = width;
			do
			{
				dTemp = *((DWORD*)lpDest);

				db = dTemp & 0x1f;
				dg = (dTemp >> 5) & 0x3f;
				dr = (dTemp >> 11) & 0x1f;

				Result = (DWORD)((SHADOW * db) >> 8 |
					((SHADOW * dg) >> 8) << 5 |
					((SHADOW * dr) >> 8) << 11);

				db = (dTemp >> 16) & 0x1f;
				dg = (dTemp >> 21) & 0x3f;
				dr = (dTemp >> 27) & 0x1f;

				Result |= (DWORD)( (SHADOW * db) >> 8 |
					((SHADOW * dg) >> 8) << 5 |
					((SHADOW * dr) >> 8) << 11 ) << 16;

				*((DWORD*)lpDest) = Result;

				lpDest += 4;
				lpSprite +=4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 24:

		// Check the ALPHA value
		if (SHADOW < 0)
			SHADOW = 0;
		else if (SHADOW > 255)
			SHADOW = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 3);
		lpDest += (lDestY * dPitch) + (lDestX * 3);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (3 * width));
		dbuf = (WORD)(dPitch - (3 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				dTemp = *((DWORD*)lpDest);

				db = dTemp & 0xFF;
				dg = (dTemp >> 8) & 0xFF;
				dr = (dTemp >> 16) & 0xFF;

				Result = (DWORD)((SHADOW * db) >> 8 |
					((SHADOW * dg) >> 8) << 8 |
					((SHADOW * dr) >> 8) << 16);

				*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
				lpDest += 2;
				*lpDest = (BYTE)(Result >> 16);
				lpDest++;

				lpSprite += 3;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;

	case 32:

		// Check the ALPHA value
		if (SHADOW < 0)
			SHADOW = 0;
		else if (SHADOW > 255)
			SHADOW = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 4);
		lpDest += (lDestY * dPitch) + (lDestX * 4);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (4 * width));
		dbuf = (WORD)(dPitch - (4 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				dTemp = *((DWORD*)lpDest);

				db = dTemp & 0xFF;
				dg = (dTemp >> 8) & 0xFF;
				dr = (dTemp >> 16) & 0xFF;

				Result = (DWORD)((SHADOW * db) >> 8 |
					((SHADOW * dg) >> 8) << 8 |
					((SHADOW * dr) >> 8) << 16);

				*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
				lpDest += 2;
				*lpDest = (BYTE)(Result >> 16);
				lpDest += 2;

				lpSprite += 4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;
	} // End RGB Format switch statement


	UnLock();
	dest->UnLock();

	return DD_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface CDXBLT_TransShadow - 
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_TransShadow(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect, WORD SHADOW)
{
	int register i,j;
	int height,width;
	BYTE* lpSprite;
	BYTE* lpDest;
	WORD dPitch, SpritePitch;
	DWORD sColorKey;
	DWORD sTemp,dTemp;
	WORD dr,dg,db;
	WORD sbuf,dbuf;
	DWORD Result;
	BOOL oddWidth = FALSE;

	// Validate the inputs and clip them to m_clipRect
	if (ValidateBlt(dest, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

	// Set height and width of SPRITE
    height = srcRect.bottom - srcRect.top;
    width = srcRect.right - srcRect.left; 

    // Get the color key for sprite surface
    sColorKey = (DWORD)m_ColorKey;

    // Lock down both surfaces for read and write
    Lock();
    dest->Lock();

	// Set the pitch for both surfaces
    SpritePitch = m_DDSD.lPitch;
    dPitch = dest->m_DDSD.lPitch;

    // Initialize the pointers to the upper left hand corner of surface
    lpSprite = (BYTE*)m_DDSD.lpSurface;
    lpDest = (BYTE*)dest->m_DDSD.lpSurface;


	switch(dest->m_RGB.bpp)
	{
	case 8:
		// IMHO paletized modes are a thing of the past please feel free to 
		// implement this if you so desire.
		break;

	case 15:

		// Check the ALPHA value
		if (SHADOW < 0)
			SHADOW = 0;
		else if (SHADOW > 255)
			SHADOW = 255;
		
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.

		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					db = dTemp & 0x1f;
					dg = (dTemp >> 5) & 0x1f;
					dr = (dTemp >> 10) & 0x1f;

					*((WORD*)lpDest) = (DWORD)( (SHADOW * db) >> 8 |
						((SHADOW * dg) >> 8) << 5 |
						((SHADOW * dr) >> 8) << 10 );
				}

				lpDest += 2;
				lpSprite += 2;
			}
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if (sTemp != ( (DWORD)sColorKey | ((DWORD)sColorKey << 16)) )
				{
					dTemp = *((DWORD*)lpDest);
					Result = dTemp;

					if ((sTemp & 0xffff) != sColorKey)
					{
						db = dTemp & 0x1f;
						dg = (dTemp >> 5) & 0x1f;
						dr = (dTemp >> 10) & 0x1f;

						Result = Result & 0xffff0000;
						Result |= (DWORD)( (SHADOW * db) >> 8 |
							((SHADOW * dg) >> 8) << 5 |
							((SHADOW * dr) >> 8) << 10 );
					}
					if (((sTemp >> 16) & 0xffff) != sColorKey)
					{
						db = (dTemp >> 16) & 0x1f;
						dg = (dTemp >> 21) & 0x1f;
						dr = (dTemp >> 26) & 0x1f;

						Result = Result & 0xffff;
						Result |= (DWORD)( (SHADOW * db) >> 8 |
							((SHADOW * dg) >> 8) << 5 |
							((SHADOW * dr) >> 8) << 10 ) << 16;
					}

					*((DWORD*)lpDest) = Result;
				}
				lpDest += 4;
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;
		}while (--i > 0);

		break;

	case 16:

		// Check the ALPHA value
		if (SHADOW < 0)
			SHADOW = 0;
		else if (SHADOW > 255)
			SHADOW = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.


		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					db = dTemp & 0x1f;
					dg = (dTemp >> 5) & 0x3f;
					dr = (dTemp >> 11) & 0x1f;

					*((WORD*)lpDest) = (DWORD)( (SHADOW * db) >> 8 |
						((SHADOW * dg) >> 8) << 5 |
						((SHADOW * dr) >> 8) << 11 );
				}

				lpDest += 2;
				lpSprite += 2;
			}
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);

				if (sTemp != ( (DWORD)sColorKey | ((DWORD)sColorKey << 16)) )
				{
					dTemp = *((DWORD*)lpDest);
					Result = dTemp;

					if ((sTemp & 0xffff) != sColorKey)
					{
						db = dTemp & 0x1f;
						dg = (dTemp >> 5) & 0x3f;
						dr = (dTemp >> 11) & 0x1f;

						Result = Result & 0xffff0000;
						Result |= (DWORD)( (SHADOW * db) >> 8 |
							((SHADOW * dg) >> 8) << 5 |
							((SHADOW * dr) >> 8) << 11 );
					}
					if (((sTemp >> 16) & 0xffff) != sColorKey)
					{
						db = (dTemp >> 16) & 0x1f;
						dg = (dTemp >> 21) & 0x3f;
						dr = (dTemp >> 27) & 0x1f;

						Result = Result & 0xffff;
						Result |= (DWORD)( (SHADOW * db) >> 8 |
							((SHADOW * dg) >> 8) << 5 |
							((SHADOW * dr) >> 8) << 11 ) << 16;
					}

					*((DWORD*)lpDest) = Result;
				}
				lpDest += 4;
				lpSprite +=4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;
		}while (--i > 0);

		break;

	case 24:

		// Check the ALPHA value
		if (SHADOW < 0)
			SHADOW = 0;
		else if (SHADOW > 255)
			SHADOW = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 3);
		lpDest += (lDestY * dPitch) + (lDestX * 3);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (3 * width));
		dbuf = (WORD)(dPitch - (3 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				
				if ((sTemp & 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);

					db = dTemp & 0xFF;
					dg = (dTemp >> 8) & 0xFF;
					dr = (dTemp >> 16) & 0xFF;

					Result = (DWORD)( (SHADOW * db) >> 8 |
						((SHADOW * dg) >> 8) << 8 |
						((SHADOW * dr) >> 8) << 16 );

					*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
					lpDest += 2;
					*lpDest = (BYTE)(Result >> 16);
					lpDest++;
				}
				else
				{
					lpDest += 3;
				}
				
				lpSprite += 3;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;
		}while (--i > 0);
		break;

	case 32:

		// Check the ALPHA value
		if (SHADOW < 0)
			SHADOW = 0;
		else if (SHADOW > 255)
			SHADOW = 255;

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 4);
		lpDest += (lDestY * dPitch) + (lDestX * 4);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (4 * width));
		dbuf = (WORD)(dPitch - (4 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				
				if ((sTemp & 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);

					db = dTemp & 0xFF;
					dg = (dTemp >> 8) & 0xFF;
					dr = (dTemp >> 16) & 0xFF;

					Result = (DWORD)( (SHADOW * db) >> 8 |
						((SHADOW * dg) >> 8) << 8 |
						((SHADOW * dr) >> 8) << 16 );

					*((WORD*)lpDest) = (WORD)(Result & 0xFFFF);
					lpDest += 2;
					*lpDest = (BYTE)(Result >> 16);
					lpDest += 2;
				}
				else
				{
					lpDest += 4;
				}
				
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;
		}while (--i > 0);
		break;
	} // End RGB Format switch statement


	UnLock();
	dest->UnLock();

	return DD_OK;

}

////////////////////////////////////////////////////////////////////////
// CDXSurface CDXBLT_BlkShadowFast - 
////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_BlkShadowFast(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect)
{
    int register i,j;
	int height,width;
    BYTE* lpSprite;
    BYTE* lpDest;
    WORD dPitch, SpritePitch;
    DWORD dTemp;
    WORD sbuf,dbuf;
    DWORD Result;
	BOOL oddWidth = FALSE;

	// Validate the inputs and clip them to m_clipRect
	if (ValidateBlt(dest, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

    // Set height and width of SPRITE
    height = srcRect.bottom - srcRect.top;
    width = srcRect.right - srcRect.left;

    // Lock down both surfaces for read and write
    Lock();
    dest->Lock();

    // Set the pitch for both surfaces
    SpritePitch = m_DDSD.lPitch;
    dPitch = dest->m_DDSD.lPitch;

    // Initialize the pointers to the upper left hand corner of the surface
    lpSprite = (BYTE*)m_DDSD.lpSurface;
    lpDest = (BYTE*)dest->m_DDSD.lpSurface;

	// Start RGB Format switch statement
	switch(dest->m_RGB.bpp)
	{
	case 8:
		// IMHO paletized modes are a thing of the past please feel free to 
		// implement this if you so desire.
		break;

	case 15:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.


		i = height;
		do
		{
			if (oddWidth)
			{
				dTemp = *((WORD*)lpDest);
				*((WORD*)lpDest) = (dTemp & 0x7BDE) >> 1;
				lpDest += 2;
			}

			j = width;
			do
			{
				dTemp = *((DWORD*)lpDest);
				Result = (dTemp & 0x7BDE) >> 1;
				Result |= (dTemp & 0x7BDE0000) >> 1;
				*((DWORD*)lpDest) = Result;
				lpDest += 4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 16:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.

		i = height;
		do
		{
			if (oddWidth)
			{
				dTemp = *((WORD*)lpDest);
				*((WORD*)lpDest) = (dTemp & 0xF7DE) >> 1;
				lpDest += 2;
			}

			j = width;
			do
			{
				dTemp = *((DWORD*)lpDest);
				Result = (dTemp & 0xF7DE) >> 1;
				Result |= (dTemp & 0xF7DE0000) >> 1;
				*((DWORD*)lpDest) = Result;
				lpDest += 4;

			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;

	case 24:
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 3);
		lpDest += (lDestY * dPitch) + (lDestX * 3);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (3 * width));
		dbuf = (WORD)(dPitch - (3 * width));
		i = height;
		do
		{
			j = width;
			do
			{
				dTemp = *((DWORD*)lpDest);
				*((WORD*)lpDest) = (WORD)((dTemp & 0xFEFE) >> 1);
				lpDest += 2;
				*lpDest = (BYTE)(((dTemp >> 16) & 0xFE) >> 1);
				lpDest++;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;

	case 32:
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 4);
		lpDest += (lDestY * dPitch) + (lDestX * 4);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (4 * width));
		dbuf = (WORD)(dPitch - (4 * width));
		i = height;
		do
		{
			j = width;
			do
			{
				dTemp = *((DWORD*)lpDest);
				*((WORD*)lpDest) = (WORD)((dTemp & 0xFEFE) >> 1);
				lpDest += 2;
				*lpDest = (BYTE)(((dTemp >> 16) & 0xFE) >> 1);
				lpDest += 2;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;
	} // End RGB Format switch statement


    UnLock();
    dest->UnLock();

	return DD_OK;
}

////////////////////////////////////////////////////////////////////////
// CDXSurface CDXBLT_TransShadowFast 
////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CDXBLT_TransShadowFast(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect)
{
    int register i,j;
	int height,width;
    BYTE* lpSprite;
    BYTE* lpDest;
    WORD dPitch, SpritePitch;
    DWORD sColorKey;
    DWORD sTemp,dTemp;
    WORD sbuf,dbuf;
    DWORD Result;
	BOOL oddWidth = FALSE;

	// Validate the inputs and clip them to m_clipRect
	if (ValidateBlt(dest, &lDestX, &lDestY, &srcRect) == FALSE)
		return DDERR_GENERIC;

    // Set height and width of SPRITE
    height = srcRect.bottom - srcRect.top;
    width = srcRect.right - srcRect.left;

    // Lock down both surfaces for read and write
    Lock();
    dest->Lock();

    // Set the pitch for both surfaces
    SpritePitch = m_DDSD.lPitch;
    dPitch = dest->m_DDSD.lPitch;

    // Initialize the pointers to the upper left hand corner of the surface
    lpSprite = (BYTE*)m_DDSD.lpSurface;
    lpDest = (BYTE*)dest->m_DDSD.lpSurface;

	// Get the color key for sprite surface
	sColorKey = (DWORD)m_ColorKey;

	// Start RGB Format switch statement
	switch(dest->m_RGB.bpp)
	{
	case 8:
		// IMHO paletized modes are a thing of the past please feel free to 
		// implement this if you so desire.
		break;

	case 15:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.


		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					*((WORD*)lpDest) = (dTemp & 0x7BDE) >> 1;
				}

				lpDest += 2;
				lpSprite += 2;
			}

			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);
				Result = dTemp;

				if ((sTemp & 0xFFFF) != sColorKey)
				{
					Result &= 0xFFFF0000;
					Result |= (dTemp & 0x7BDE) >> 1;
				}
				if ((sTemp >> 16) != sColorKey)
				{
					Result &= 0xFFFF;
					Result |= (dTemp & 0x7BDE0000) >> 1;
				}

				*((DWORD*)lpDest) = Result;
				lpDest += 4;
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);

		break;

	case 16:

		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 2);
		lpDest += (lDestY * dPitch) + (lDestX * 2);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (2 * width));
		dbuf = (WORD)(dPitch - (2 * width));

		// Is the Sprite width odd or even?
		if (width % 2 == 1)
		{
			oddWidth = TRUE;
			width = (width - 1) / 2; //div by 2, processing 2 pixels at a time.
		}
		else
			width = width / 2;  //div by 2, processing 2 pixels at a time.

		i = height;
		do
		{
			if (oddWidth)
			{
				sTemp = *((WORD*)lpSprite);

				if (sTemp != sColorKey)
				{
					dTemp = *((WORD*)lpDest);
					*((WORD*)lpDest) = (dTemp & 0xF7DE) >> 1;
				}

				lpDest += 2;
				lpSprite += 2;
			}

			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				dTemp = *((DWORD*)lpDest);
				Result = dTemp;

				if ((sTemp & 0xFFFF) != sColorKey)
				{
					Result &= 0xFFFF0000;
					Result |= (dTemp & 0xF7DE) >> 1;
				}
				if ((sTemp >> 16) != sColorKey)
				{
					Result &= 0xFFFF;
					Result |= (dTemp & 0xF7DE0000) >> 1;
				}

				*((DWORD*)lpDest) = Result;
				lpDest += 4;
				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;

	case 24:
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 3);
		lpDest += (lDestY * dPitch) + (lDestX * 3);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (3 * width));
		dbuf = (WORD)(dPitch - (3 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				
				if ((sTemp & 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);

					*((WORD*)lpDest) = (WORD)((dTemp & 0xFEFE) >> 1);
					lpDest += 2;
					*lpDest = (BYTE)(((dTemp >> 16) & 0xFE) >> 1);
					lpDest++;
				}
				else
				{
					lpDest += 3;
				}

				lpSprite += 3;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;

	case 32:
		// Initialize the pointers to the first pixel in the rectangle
		lpSprite += (srcRect.top * SpritePitch) + (srcRect.left * 4);
		lpDest += (lDestY * dPitch) + (lDestX * 4);

		// Set the horizontal padding
		sbuf = (WORD)(SpritePitch - (4 * width));
		dbuf = (WORD)(dPitch - (4 * width));

		i = height;
		do
		{
			j = width;
			do
			{
				sTemp = *((DWORD*)lpSprite);
				
				if ((sTemp & 0xFFFFFF) != sColorKey)
				{
					dTemp = *((DWORD*)lpDest);

					*((WORD*)lpDest) = (WORD)((dTemp & 0xFEFE) >> 1);
					lpDest += 2;
					*lpDest = (BYTE)(((dTemp >> 16) & 0xFE) >> 1);
					lpDest += 2;
				}
				else
				{
					lpDest += 4;
				}

				lpSprite += 4;
			}while (--j > 0);
			lpDest += dbuf;
			lpSprite += sbuf;

		}while (--i > 0);
		break;
	} // End RGB Format switch statement


    UnLock();
    dest->UnLock();

	return DD_OK;
}
//<JJH>

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Draw
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::Draw(CDXSurface* lpDDest)
{
	HRESULT rval;

  rval = lpDDest->m_lpDDS->Blt(&DestRect, m_lpDDS, &SrcRect, DDBLT_WAIT, NULL);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawFast
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawFast(int X, int Y, CDXSurface* lpDDest)
{
	HRESULT rval;

	rval = lpDDest->m_lpDDS->BltFast(X, Y, m_lpDDS, &SrcRect, DDBLTFAST_WAIT);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawTrans
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawTrans(int X, int Y, CDXSurface* lpDDest)
{
	HRESULT rval;

	rval = lpDDest->m_lpDDS->BltFast(X, Y, m_lpDDS, &SrcRect, DDBLTFAST_WAIT | DDBLTFAST_SRCCOLORKEY);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawClipped
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawClipped(int X, int Y, CDXSurface* lpDDest, LPRECT ClipRect)
{
	HRESULT rval;
	RECT ModSrc = SrcRect;
	Clip(&X, &Y, &ModSrc, ClipRect);

	rval = lpDDest->m_lpDDS->BltFast(X, Y, m_lpDDS, &ModSrc, DDBLTFAST_WAIT | DDBLTFAST_SRCCOLORKEY);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawWindowed
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawWindowed(CDXSurface* lpDDest)
{
	HRESULT rval;
	RECT Window;
	GetClientRect((HWND)Screen->m_hWnd, &Window);
	ClientToScreen((HWND)Screen->m_hWnd, (LPPOINT)&Window);
	ClientToScreen((HWND)Screen->m_hWnd, (LPPOINT)&Window+1);

	rval = lpDDest->m_lpDDS->Blt(&Window, m_lpDDS, NULL, DDBLT_WAIT, NULL);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawScaled
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawScaled(int X, int Y, float Factor, CDXSurface* lpDDS)
{
	HRESULT rval;

	DestRect.top = Y;
	DestRect.left = X;
	DestRect.bottom = Y + (m_PixelHeight * Factor);
	DestRect.right = X + (m_PixelWidth * Factor);

	rval = lpDDS->m_lpDDS->Blt(&DestRect, m_lpDDS, &SrcRect, DDBLT_WAIT | DDBLT_KEYSRC, NULL);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawScaled
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawScaled(int X, int Y, int Width, int Height, CDXSurface* lpDDS)
{
	HRESULT rval;

	DestRect.top = Y;
	DestRect.left = X;
	DestRect.bottom = Y + (m_PixelHeight * Height);
	DestRect.right = X + (m_PixelWidth * Width);

	rval = lpDDS->m_lpDDS->Blt(&DestRect, m_lpDDS, &SrcRect, DDBLT_WAIT | DDBLT_KEYSRC, NULL);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}

#define DEG2RAD(x) (x*(float)PI/(float)180)

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawRotated
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::DrawRotated(int X, int Y, double Angle, CDXSurface* Dest)
{
	// NOT IMPLEMENTED
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawHFlip
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawHFlip(int X, int Y, CDXSurface* Dest)
{
	DDBLTFX ddBltFx;

	ddBltFx.dwSize = sizeof(DDBLTFX);
	ddBltFx.dwDDFX = DDBLTFX_MIRRORLEFTRIGHT;
	return m_lpDDS->Blt(NULL, NULL, NULL, DDBLT_DDFX | DDBLT_WAIT, &ddBltFx);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawVFlip
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawVFlip(int X, int Y, CDXSurface* Dest)
{
	DDBLTFX ddBltFx;

	ddBltFx.dwSize = sizeof(DDBLTFX);
	ddBltFx.dwDDFX = DDBLTFX_MIRRORUPDOWN;
	return m_lpDDS->Blt(NULL, NULL, NULL, DDBLT_DDFX | DDBLT_WAIT, &ddBltFx);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface SetDest
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::SetDest(int t, int l, int b, int r)
{
	DestRect.top	= t;
	DestRect.left	= l;
	DestRect.bottom = b;
	DestRect.right	= r;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface SetSrc
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::SetSrc(int t, int l, int b, int r)
{
	SrcRect.top		= t;
	SrcRect.left	= l;
	SrcRect.bottom	= b;
	SrcRect.right	= r;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface ColorKey
//////////////////////////////////////////////////////////////////////////////////
//<JJH>
void CDXSurface::SetColorKey(DWORD col)
//<JJH>
{
	DDCOLORKEY ddck;

	m_ColorKey = col;
	ddck.dwColorSpaceLowValue = col;
	ddck.dwColorSpaceHighValue = col;

	m_lpDDS->SetColorKey(DDCKEY_SRCBLT, &ddck);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface ColorKey
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::SetColorKey(void)
{
	DDCOLORKEY		ddck;
	DWORD			dw = CLR_INVALID;
#if DIRECTDRAW_VERSION >= CDX_DDVER
	DDSURFACEDESC2	ddsd;
#else
	DDSURFACEDESC	ddsd;
#endif
	HRESULT			hres;

	ddsd.dwSize = sizeof(ddsd);
	while((hres = m_lpDDS->Lock(NULL, &ddsd, 0, NULL)) == DDERR_WASSTILLDRAWING);

	if(SUCCEEDED(hres)) {
		dw = *(DWORD *)ddsd.lpSurface;
		if(ddsd.ddpfPixelFormat.dwRGBBitCount < 32)
			dw &= (1 << ddsd.ddpfPixelFormat.dwRGBBitCount) - 1;
		m_lpDDS->Unlock(NULL);
	}

	m_ColorKey = dw;
	ddck.dwColorSpaceLowValue = dw;
	ddck.dwColorSpaceHighValue = dw;

	m_lpDDS->SetColorKey(DDCKEY_SRCBLT, &ddck);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Restore
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::Restore(void)
{
	m_lpDDS->Restore();
	DDReLoadBitmap(m_lpDDS, m_pFilename);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Lock
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::Lock(void)
{
	ZeroMemory(&m_DDSD, sizeof(m_DDSD));
	m_DDSD.dwSize = sizeof(m_DDSD);
	return m_lpDDS->Lock(NULL, &m_DDSD, DDLOCK_WAIT, NULL);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface UnLock
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::UnLock(void)
{
	return m_lpDDS->Unlock(NULL);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface GetDC
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::GetDC()
{
	m_lpDDS->GetDC(&m_DC);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface ReleaseDC
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::ReleaseDC()
{
	m_lpDDS->ReleaseDC(m_DC);
	m_DC = NULL;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface ChangeFont
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::ChangeFont(const char* FontName, int Width, int Height, int Attributes)
{
//<JJH>
	if (m_Font != NULL)
		DeleteObject(m_Font);
//<JJH>

	m_Font = CreateFont(Height, Width,
		0, 0,
		Attributes,
		FALSE,
		FALSE,
		FALSE,
		ANSI_CHARSET,
		OUT_DEFAULT_PRECIS,
		CLIP_DEFAULT_PRECIS,
		NONANTIALIASED_QUALITY,
		VARIABLE_PITCH,
		FontName);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface SelectFont
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::SetFont(void)
{
	SelectObject(m_DC, m_Font);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface TextOut
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::TextXY(int x, int y, COLORREF col, LPCTSTR pString)
{
	SetBkMode(m_DC, TRANSPARENT);
	SetTextColor(m_DC, col);
	TextOut(m_DC, x, y, pString, strlen(pString));
}

//////////////////////////////////////////////////////////////////////////////////
// Swap - swaps two integers
//////////////////////////////////////////////////////////////////////////////////
void Swap(int *a, int *b)
{
	int Temp = *a;
	*a = *b;
	*b = Temp;
}

//<JJH>
//////////////////////////////////////////////////////////////////////////////////
// CDXSurface PutPixel
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::PutPixel(int X, int Y, DWORD Col)
{
    BYTE* Bitmap = (BYTE*)m_DDSD.lpSurface;
    int colorBitDepth = Screen->m_BPP;

	if (X < m_ClipRect.left || X > m_ClipRect.right)
		return;
	if (Y < m_ClipRect.top || Y > m_ClipRect.bottom)
		return;

    switch(colorBitDepth)
    {
    case 8:
			Bitmap += Y * m_DDSD.lPitch + X;
			*Bitmap = (BYTE)Col;
			break;
    case 15:
    case 16:
            Bitmap += Y * m_DDSD.lPitch + X * 2;
            *((WORD*)Bitmap) = (WORD)Col;
            break;
    case 24:
            Bitmap += Y * m_DDSD.lPitch + X * 3;
            *((WORD*)Bitmap) = (WORD)(Col & 0xFFFF);
            Bitmap += 2;
            *Bitmap = (BYTE)((Col >> 16) & 0xFF);
            break;
    case 32:
            Bitmap += Y * m_DDSD.lPitch + X * 4;
			// This would be faster but I'm not sure if it is correct?
			//*((DWORD*)Bitmap) = Col; 
            *((WORD*)Bitmap) = (WORD)(Col & 0xFFFF);
            Bitmap += 2;
            *Bitmap = (BYTE)((Col >> 16) & 0xFF);
            break;
    default:
            break;
    }
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface GetPixel
//////////////////////////////////////////////////////////////////////////////////
DWORD CDXSurface::GetPixel(int X, int Y)
{
	DWORD Pixel;
	BYTE* Bitmap = (BYTE*)m_DDSD.lpSurface;

    switch(Screen->m_BPP)
    {
    case 8:
            Bitmap += Y * m_DDSD.lPitch + X;
			Pixel = (DWORD)(*Bitmap);
            return Pixel;
            break;
    case 15:
    case 16:
            Bitmap += Y * m_DDSD.lPitch + X * 2;
			Pixel = (DWORD)(*((WORD*)Bitmap));
            return Pixel;
            break;
    case 24:
            Bitmap += Y * m_DDSD.lPitch + X * 3;
            Pixel = (DWORD)(*((WORD*)Bitmap));
            Bitmap += 2;
			Pixel |= (DWORD)(*Bitmap) << 16;
            return Pixel;
            break;
    case 32:
            Bitmap += Y * m_DDSD.lPitch + X * 4;
            Pixel = *((DWORD*)Bitmap);
            return Pixel;
            break;
	default:
		return NULL;
    }
}

////////////////////////////////////////////////////////////////////////
// CDXSurface Rect
////////////////////////////////////////////////////////////////////////
void CDXSurface::Rect(int X1, int Y1, int X2, int Y2, DWORD Col)
{
	HLine(X1,X2,Y1,Col);
	HLine(X1,X2,Y2,Col);

	VLine(Y1,Y2,X1,Col);
	VLine(Y1,Y2,X2,Col);
}

////////////////////////////////////////////////////////////////////////
// CDXSurface FillRect
////////////////////////////////////////////////////////////////////////
void CDXSurface::FillRect(int X1, int Y1, int X2, int Y2, DWORD Col)
{
	DDBLTFX ddbltfx;
	RECT Rect = { X1, Y1, X2, Y2 };
	
	ddbltfx.dwSize = sizeof(ddbltfx);
	ddbltfx.dwFillColor = Col;

	m_lpDDS->Blt(&Rect, NULL, NULL, DDBLT_WAIT | DDBLT_COLORFILL, &ddbltfx);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Line
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::Line(int X1, int Y1, int X2, int Y2, DWORD Col)
{
	double xStep, yStep, X, Y;
	int xLength, yLength, xCount, yCount;

	// If the line is horizontal or vertical use the fast version.
	if (X1 == X2)
	{
		VLine(Y1,Y2,X1,Col);
		return;
	}
	else if (Y1 == Y2)
	{
		HLine(X1,X2,Y1,Col);
		return;
	}

	xLength = abs(X2 - X1);
	yLength = abs(Y2 - Y1);

	if(xLength == 0) VLine(Y1, Y2, X1, Col);
	else if(yLength == 0) HLine(X1, X2, Y1, Col);

	else if(xLength > yLength)
	{
		if(X1 > X2)
		{
			Swap(&X1, &X2);
			Swap(&Y1, &Y2);
		}

		yStep = (double)(Y2 - Y1) / (double)(X2 - X1);
		Y = Y1;

		for(xCount = X1; xCount <= X2; xCount++)
		{
			PutPixel(xCount, (int)Y, Col);
			Y += yStep;
		}
	}
	else
	{
		if(Y1 > Y2)
		{
			Swap(&X1, &X2);
			Swap(&Y1, &Y2);
		}

		xStep = (double)(X2 - X1) / (double)(Y2 - Y1);
		X = X1;

		for(yCount = Y1; yCount <= Y2; yCount++)
		{
			PutPixel((int)X, yCount, Col);
			X += xStep;
		}
	}
}

////////////////////////////////////////////////////////////////////////
// CDXSurface VLine
////////////////////////////////////////////////////////////////////////
void CDXSurface::VLine(int Y1, int Y2, int X, DWORD Col)
{
	// Notice: It is assumed that the surface will be locked prior
	//         to calling this function.  This is done for speed.
    int Length,top,bottom;
	DWORD dwCol;
    BYTE *Pixel = (BYTE*)m_DDSD.lpSurface;

	// Do some quick bounds checking
    if (X < m_ClipRect.left)
        return;

    if (X > m_ClipRect.right)
        return;


	// Determine which is top and bottom by the values
    if (Y1 > Y2)
    {
        top = Y2;
        bottom = Y1 + 1;
    }
    else
    {
        top = Y1;
        bottom = Y2 + 1;
    }

    // Clip the line
    if (top < m_ClipRect.top)
        top = m_ClipRect.top;

    if (bottom > m_ClipRect.bottom)
        bottom = m_ClipRect.bottom;

    Pixel += top * m_DDSD.lPitch;
    Length = bottom - top;

    // Draw the line

	switch (Screen->m_BPP)
	{
	case 8:
		Pixel += X;
		do
		{
			*Pixel = (BYTE)Col;
			Pixel += m_DDSD.lPitch;
			Length--;
		}while (Length > 0);
		break;

	case 15:
	case 16:

		Pixel += X << 1;

		do
		{
			*((WORD*)Pixel) = (WORD)Col;
			Pixel += m_DDSD.lPitch;
			Length--;
		}while(Length > 0);

		break;

	case 24:

		Pixel += X + X + X;

		dwCol = Col & 0xFFFF;
		Col = (Col >> 16) & 0xFF;
		do
		{
			*((WORD*)Pixel) = (WORD)dwCol;
			Pixel += 2;
			*Pixel = (BYTE)Col;
			Pixel += m_DDSD.lPitch - 2;
			Length--;
		}while(Length > 0);

		break;

	case 32:

		Pixel += X << 2;

		dwCol = Col & 0xFFFF;
		Col = (Col >> 16) & 0xFF;
		do
		{
			// This would be faster but I'm not sure if it is correct?
			//*((DWORD*)Pixel) = dwCol;
			//Pixel += m_DDSD.lPitch;
			*((WORD*)Pixel) = (WORD)dwCol;
			Pixel += 2;
			*Pixel = (BYTE)Col;
			Pixel += m_DDSD.lPitch - 2;
			Length--;
		}while(Length > 0);

		break;
	}
}

////////////////////////////////////////////////////////////////////////
// CDXSurface HLine
////////////////////////////////////////////////////////////////////////
void CDXSurface::HLine(int X1, int X2, int Y, DWORD Col)
{
	// Notice: It is assumed that the surface will be locked prior
	//         to calling this function.  This is done for speed.
    int Length,left,right;
	int i;
    DWORD dwCol;
    BYTE *Pixel = (BYTE*)m_DDSD.lpSurface;

	// Do some quick bounds checking
    if (Y < m_ClipRect.top)
        return;

    if (Y > m_ClipRect.bottom)
        return;


	// Determine which is left and right by value
    if (X1 > X2)
    {
        left = X2;
        right = X1 + 1;
    }
    else
    {
        left = X1;
        right = X2 + 1;
    }

    // Clip the line
    if (left < m_ClipRect.left)
            left = m_ClipRect.left;

    if (right > m_ClipRect.right)
            right = m_ClipRect.right;

	// Calculate the length of the line
    Length = right - left;
    Pixel += Y * m_DDSD.lPitch;

	switch (Screen->m_BPP)
	{
	case 8:
		Pixel += left;
		i = Length % 4;
		Length -= i;
		for (i; i > 0; i--)
		{
			*Pixel = (BYTE)Col;
			Pixel++;
		}
		if (Length > 3)
		{
			dwCol = Col | (Col << 8) | (Col << 16) | (Col << 24);
			do
			{
				*((DWORD*)Pixel) = dwCol;
				Pixel += 4;
				Length -= 4;
			}while(Length > 0);
		}
		break;

	case 15:
	case 16:

		Pixel += left << 1;
		i = Length % 2;
		Length -= i;
		for (i; i > 0; i--)
		{
			*((WORD*)Pixel) = (WORD)Col;
			Pixel += 2;
		}
		if (Length > 1)
		{
			dwCol = Col | (Col << 16);
			do
			{
				*((DWORD*)Pixel) = dwCol;
				Pixel += 4;
				Length -= 2;
			}while(Length > 0);
		}

		break;

	case 24:

		Pixel += left + left + left;

		dwCol = Col & 0xFFFF;
		Col = (Col >> 16) & 0xFF;
		do
		{
			*((WORD*)Pixel) = (WORD)dwCol;
			Pixel += 2;
			*Pixel = (BYTE)Col;
			Pixel++;
			Length--;
		}while(Length > 0);

		break;

	case 32:

		Pixel += left << 2;

		dwCol = Col & 0xFFFF;
		Col = (Col >> 16) & 0xFF;
		do
		{
			// This would be faster but I'm not sure if it is correct?
			//*((DWORD*)Pixel) = dwCol;
			//Pixel += m_DDSD.lPitch + 4;
			*((WORD*)Pixel) = (WORD)dwCol;
			Pixel += 2;
			*Pixel = (BYTE)Col;
			Pixel += 2;
			Length--;
		}while(Length > 0);

		break;
	}
}
//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Circle
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::Circle(int X, int Y, int Radius, DWORD Col)
{
	// Notice: It is assumed that the surface will be locked prior
	//         to calling this function.  This is done for speed.
	// Thank Micheal Abrash for this routine.
	int majorAxis, minorAxis;
	double radiusSqrd = (double)Radius * Radius;

	majorAxis = 0;
	minorAxis = Radius;

	do
	{
		PutPixel(X+majorAxis, Y-minorAxis, Col);
		PutPixel(X-majorAxis, Y-minorAxis, Col);
		PutPixel(X+majorAxis, Y+minorAxis, Col);
		PutPixel(X-majorAxis, Y+minorAxis, Col);
		PutPixel(X+minorAxis, Y-majorAxis, Col);
		PutPixel(X-minorAxis, Y-majorAxis, Col);
		PutPixel(X+minorAxis, Y+majorAxis, Col);
		PutPixel(X-minorAxis, Y+majorAxis, Col);

		majorAxis++;
		minorAxis = sqrt(radiusSqrd - ((double)majorAxis * majorAxis)) + 0.5;

	}while (majorAxis <= minorAxis);

}
////////////////////////////////////////////////////////////////////////
// CDXSurface FillCircle
////////////////////////////////////////////////////////////////////////
void CDXSurface::FillCircle(int X, int Y, int Radius, DWORD Col)
{
	// Notice: It is assumed that the surface will be locked prior
	//         to calling this function.  This is done for speed.
    int Y1,dx;
    int top, bot;
	double R;

    R = (double)Radius * Radius;

	// Perform a little bounds checking
    if ((top = Y - Radius) < m_ClipRect.top)
        top = m_ClipRect.top;
    else if (top > m_ClipRect.bottom)
        return;

    if ((bot = Y + Radius) > m_ClipRect.bottom)
        bot = m_ClipRect.bottom;
    else if (bot < m_ClipRect.top )
        return;

    for(Y1=top; Y1 < bot; Y1++)
    {
        dx = sqrt(R - ((double)(Y-Y1)*(Y-Y1))) + 0.5;
		HLine(X-dx, X+dx, Y1, Col);
    }
}
//<JJH>

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Fill
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::Fill(DWORD FillColor)
{
	DDBLTFX ddBltFx;

	ddBltFx.dwSize = sizeof(DDBLTFX);
	ddBltFx.dwFillColor = FillColor;
	m_lpDDS->Blt(NULL, NULL, NULL, DDBLT_WAIT | DDBLT_COLORFILL, &ddBltFx);
}

//////////////////////////////////////////////////////////////////////////////////
// Clip - clips a destination rectange and modifys X,Y coords appropriatly
//////////////////////////////////////////////////////////////////////////////////
void Clip(int *DestX, int *DestY, RECT *SrcRect, RECT *DestRect)
{
	// If it's partly off the right side of the screen
	if(*DestX + (SrcRect->right - SrcRect->left) > DestRect->right)
		SrcRect->right -= *DestX + (SrcRect->right-SrcRect->left) - DestRect->right;

	// Partly off the left side of the screen
	if(*DestX < DestRect->left)
	{
		SrcRect->left += DestRect->left - *DestX;
		*DestX = DestRect->left;
	}

	// Partly off the top of the screen
	if(*DestY < DestRect->top)
	{
		SrcRect->top += DestRect->top - *DestY;
		*DestY = DestRect->top;
	}

	// If it's partly off the bottom side of the screen
	if(*DestY + (SrcRect->bottom - SrcRect->top) > DestRect->bottom)
	SrcRect->bottom -= ((SrcRect->bottom-SrcRect->top)+*DestY) - DestRect->bottom;

	return;
}
