//////////////////////////////////////////////////////////////////////////////////
// Project Name:   [ CDX Class Library - CDX.lib ]
// Source File:    [ DirectDraw Utility Functions ]
// Author:         [ Bil Simser - bsimser@home.com ]
// Contributions:  [ Lennart Steinke ]
// Revision:       [ 1.99a ]
//////////////////////////////////////////////////////////////////////////////////
#undef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN

#include "CDX.h"

////////////////////////////////////////////////////////////////////////
//
//  DDLoadBitmap
//
//  Create a DirectDrawSurface from a bitmap resource.
//
////////////////////////////////////////////////////////////////////////
#if DIRECTDRAW_VERSION >= CDX_DDVER
extern "C" IDirectDrawSurface4 * DDLoadBitmap(IDirectDraw4 *pdd, LPCSTR szBitmap, int dx, int dy, BOOL memoryType)
#else
extern "C" IDirectDrawSurface * DDLoadBitmap(IDirectDraw *pdd, LPCSTR szBitmap, int dx, int dy, BOOL memoryType)
#endif
{
    HBITMAP             hbm;
    BITMAP              bm;
#if DIRECTDRAW_VERSION >= CDX_DDVER
    DDSURFACEDESC2      ddsd;
    IDirectDrawSurface4 *pdds;
#else
    DDSURFACEDESC      ddsd;
    IDirectDrawSurface *pdds;
#endif
    //
    //  try to load the bitmap as a resource, if that fails, try it as a file
    //
    hbm = (HBITMAP)LoadImage(GetModuleHandle(NULL), szBitmap, IMAGE_BITMAP, dx, dy, LR_CREATEDIBSECTION);

    if (hbm == NULL)
        hbm = (HBITMAP)LoadImage(NULL, szBitmap, IMAGE_BITMAP, dx, dy, LR_LOADFROMFILE|LR_CREATEDIBSECTION);

    if (hbm == NULL)
        return NULL;

    //
    // get size of the bitmap
    //
    GetObject(hbm, sizeof(bm), &bm);      // get size of bitmap

    //
    // create a DirectDrawSurface for this bitmap
    //
    ZeroMemory(&ddsd, sizeof(ddsd));
    ddsd.dwSize = sizeof(ddsd);
    ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT |DDSD_WIDTH;

//<JJH>
	switch(memoryType) {
		case CDXMEM_SYSTEMONLY:
			ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;
			break;

		case CDXMEM_VIDEOONLY:
			ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_VIDEOMEMORY;
			break;

		case CDXMEM_VIDTHENSYS:
		default:
			ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
			break;
	}
//<JJH>

    ddsd.dwWidth = bm.bmWidth;
    ddsd.dwHeight = bm.bmHeight;

    if (pdd->CreateSurface(&ddsd, &pdds, NULL) != DD_OK)
        return NULL;

    DDCopyBitmap(pdds, hbm, 0, 0, 0, 0);

    DeleteObject(hbm);

    return pdds;
}

////////////////////////////////////////////////////////////////////////
//
//  DDCreateSurface
//
//  Create a DirectDrawSurface of a specified size
//  If ddsd is not NULL, it will be filled with the desc of the new surface
//
////////////////////////////////////////////////////////////////////////
#if DIRECTDRAW_VERSION >= CDX_DDVER
IDirectDrawSurface4 *DDCreateSurface(IDirectDraw4 *pdd, DWORD width, DWORD height, DDSURFACEDESC2 *ddsd)
#else
IDirectDrawSurface *DDCreateSurface(IDirectDraw *pdd, DWORD width, DWORD height, DDSURFACEDESC *ddsd)
#endif
{
	BOOL free_ddsd;
#if DIRECTDRAW_VERSION >= CDX_DDVER
	IDirectDrawSurface4 *pdds;
#else
	IDirectDrawSurface *pdds;
#endif

	// Define what we want (surface description)
	if (ddsd == NULL)
	{
		free_ddsd = TRUE;
#if DIRECTDRAW_VERSION >= CDX_DDVER
		ddsd = (DDSURFACEDESC2*) malloc(sizeof(DDSURFACEDESC2));
#else
		ddsd = (DDSURFACEDESC*) malloc(sizeof(DDSURFACEDESC));
#endif
		if (ddsd == NULL)
			return NULL;
	}
	else
	{
		free_ddsd = FALSE;
	}

#if DIRECTDRAW_VERSION >= CDX_DDVER
	ZeroMemory(ddsd, sizeof(DDSURFACEDESC2));
	ddsd->dwSize = sizeof(DDSURFACEDESC2);
#else
	ZeroMemory(ddsd, sizeof(DDSURFACEDESC));
	ddsd->dwSize = sizeof(DDSURFACEDESC);
#endif

	ddsd->dwFlags = DDSD_CAPS | DDSD_HEIGHT |DDSD_WIDTH;
	ddsd->ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
	ddsd->dwWidth = width;
	ddsd->dwHeight = height;

	// If we can't create the surface, return NULL
	if (pdd->CreateSurface(ddsd, &pdds, NULL) != DD_OK){
		return NULL;
	}


	if (free_ddsd)
	{
		free(ddsd);
	}

	// Success: return the surface
	return pdds;
}


////////////////////////////////////////////////////////////////////////
//
//  DDLoadSizeBitmap
//
//  Create a DirectDrawSurface from a sized bitmap resource.
//
////////////////////////////////////////////////////////////////////////
#if DIRECTDRAW_VERSION >= CDX_DDVER
IDirectDrawSurface4 * DDLoadSizeBitmap(IDirectDraw4 *pdd, LPCSTR szBitmap, int *dx, int *dy, BOOL memoryType)
#else
IDirectDrawSurface * DDLoadSizeBitmap(IDirectDraw *pdd, LPCSTR szBitmap, int *dx, int *dy, BOOL memoryType)
#endif
{
	HBITMAP             hbm;
	BITMAP              bm;
#if DIRECTDRAW_VERSION >= CDX_DDVER
	DDSURFACEDESC2       ddsd;
	IDirectDrawSurface4 *pdds;
#else
	DDSURFACEDESC       ddsd;
	IDirectDrawSurface *pdds;
#endif

	hbm = (HBITMAP)LoadImage(GetModuleHandle(NULL), szBitmap, IMAGE_BITMAP, 0, 0, LR_CREATEDIBSECTION);

	if (hbm == NULL)
	hbm = (HBITMAP)LoadImage(NULL, szBitmap, IMAGE_BITMAP, 0, 0, LR_LOADFROMFILE|LR_CREATEDIBSECTION);

	if (hbm == NULL)
		 return NULL;

	//
	// get size of the bitmap
	//
	GetObject(hbm, sizeof(bm), &bm);      // get size of bitmap

	//
	// create a DirectDrawSurface for this bitmap
	//
	ZeroMemory(&ddsd, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT |DDSD_WIDTH;

//<JJH>
	switch(memoryType) {
		case CDXMEM_SYSTEMONLY:
			ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;
			break;

		case CDXMEM_VIDEOONLY:
			ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_VIDEOMEMORY;
			break;

		case CDXMEM_VIDTHENSYS:
		default:
			ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
			break;
	}
//<JJH>
	
	ddsd.dwWidth = bm.bmWidth;
	ddsd.dwHeight = bm.bmHeight;

	if (pdd->CreateSurface(&ddsd, &pdds, NULL) != DD_OK){
		return NULL;
	}
	DDCopyBitmap(pdds, hbm, 0, 0, 0, 0);

	DeleteObject(hbm);

	*dx = ddsd.dwWidth;
	*dy = ddsd.dwHeight;

	return pdds;
}

////////////////////////////////////////////////////////////////////////
//
//  DDReLoadBitmap
//
//  Load a bitmap from a file or resource into a directdraw surface.
//  Normaly used to re-load a surface after a restore.
//
////////////////////////////////////////////////////////////////////////
#if DIRECTDRAW_VERSION >= CDX_DDVER
HRESULT DDReLoadBitmap(IDirectDrawSurface4 *pdds, LPCSTR szBitmap)
#else
HRESULT DDReLoadBitmap(IDirectDrawSurface *pdds, LPCSTR szBitmap)
#endif
{
    HBITMAP             hbm;
    HRESULT             hr;

    //
    //  try to load the bitmap as a resource, if that fails, try it as a file
    //
    hbm = (HBITMAP)LoadImage(GetModuleHandle(NULL), szBitmap, IMAGE_BITMAP, 0, 0, LR_CREATEDIBSECTION);

    if (hbm == NULL)
        hbm = (HBITMAP)LoadImage(NULL, szBitmap, IMAGE_BITMAP, 0, 0, LR_LOADFROMFILE|LR_CREATEDIBSECTION);

    if (hbm == NULL)
    {
        OutputDebugString("handle is null\n");
        return E_FAIL;
    }

    hr = DDCopyBitmap(pdds, hbm, 0, 0, 0, 0);
    if (hr != DD_OK)
    {
        OutputDebugString("ddcopybitmap failed\n");
    }


    DeleteObject(hbm);
    return hr;
}

////////////////////////////////////////////////////////////////////////
//
//  DDCopyBitmap
//
//  Draw a bitmap into a DirectDrawSurface
//
////////////////////////////////////////////////////////////////////////
#if DIRECTDRAW_VERSION >= CDX_DDVER
extern "C" HRESULT DDCopyBitmap(IDirectDrawSurface4 *pdds, HBITMAP hbm, int x, int y, int dx, int dy)
#else
extern "C" HRESULT DDCopyBitmap(IDirectDrawSurface *pdds, HBITMAP hbm, int x, int y, int dx, int dy)
#endif
{
    HDC                 hdcImage;
    HDC                 hdc;
    BITMAP              bm;
#if DIRECTDRAW_VERSION >= CDX_DDVER
    DDSURFACEDESC2       ddsd;
#else
    DDSURFACEDESC       ddsd;
#endif

    HRESULT             hr;

    if (hbm == NULL || pdds == NULL)
        return E_FAIL;

    //
    // make sure this surface is restored.
    //
    pdds->Restore();

    //
    //  select bitmap into a memoryDC so we can use it.
    //
    hdcImage = CreateCompatibleDC(NULL);
    if (!hdcImage)
        OutputDebugString("createcompatible dc failed\n");
    SelectObject(hdcImage, hbm);

    //
    // get size of the bitmap
    //
    GetObject(hbm, sizeof(bm), &bm);    // get size of bitmap
    dx = dx == 0 ? bm.bmWidth  : dx;    // use the passed size, unless zero
    dy = dy == 0 ? bm.bmHeight : dy;

    //
    // get size of surface.
    //
    ddsd.dwSize = sizeof(ddsd);
    ddsd.dwFlags = DDSD_HEIGHT | DDSD_WIDTH;
    pdds->GetSurfaceDesc(&ddsd);

    if ((hr = pdds->GetDC(&hdc)) == DD_OK)
    {
        StretchBlt(hdc, 0, 0, ddsd.dwWidth, ddsd.dwHeight, hdcImage, x, y, dx, dy, SRCCOPY);
        pdds->ReleaseDC(hdc);
    }

    DeleteDC(hdcImage);

    return hr;
}

////////////////////////////////////////////////////////////////////////
//
//  DDLoadPalette
//
//  Create a DirectDraw palette object from a bitmap resoure.
//
//  If the resource does not exist or NULL is passed create a
//  default 332 palette.
//
////////////////////////////////////////////////////////////////////////
#if DIRECTDRAW_VERSION >= CDX_DDVER
extern "C" IDirectDrawPalette * DDLoadPalette(IDirectDraw4 *pdd, LPCSTR szBitmap)
#else
extern "C" IDirectDrawPalette * DDLoadPalette(IDirectDraw *pdd, LPCSTR szBitmap)
#endif
{
    IDirectDrawPalette* ddpal;
    int                 i;
    int                 n;
    int                 fh;
    HRSRC               h;
    LPBITMAPINFOHEADER  lpbi;
    PALETTEENTRY        ape[256];
    RGBQUAD *           prgb;

    //
    // build a 332 palette as the default.
    //
    for (i=0; i<256; i++)
    {
        ape[i].peRed   = (BYTE)(((i >> 5) & 0x07) * 255 / 7);
        ape[i].peGreen = (BYTE)(((i >> 2) & 0x07) * 255 / 7);
        ape[i].peBlue  = (BYTE)(((i >> 0) & 0x03) * 255 / 3);
        ape[i].peFlags = (BYTE)0;
    }

    //
    // get a pointer to the bitmap resource.
    //
    if (szBitmap && (h = FindResource(NULL, szBitmap, RT_BITMAP)))
    {
        lpbi = (LPBITMAPINFOHEADER)LockResource(LoadResource(NULL, h));
        if (!lpbi)
            OutputDebugString("lock resource failed\n");
        prgb = (RGBQUAD*)((BYTE*)lpbi + lpbi->biSize);

        if (lpbi == NULL || lpbi->biSize < sizeof(BITMAPINFOHEADER))
            n = 0;
        else if (lpbi->biBitCount > 8)
            n = 0;
        else if (lpbi->biClrUsed == 0)
            n = 1 << lpbi->biBitCount;
        else
            n = lpbi->biClrUsed;

        //
        //  a DIB color table has its colors stored BGR not RGB
        //  so flip them around.
        //
        for(i=0; i<n; i++ )
        {
            ape[i].peRed   = prgb[i].rgbRed;
            ape[i].peGreen = prgb[i].rgbGreen;
            ape[i].peBlue  = prgb[i].rgbBlue;
            ape[i].peFlags = 0;
        }
    }
    else if (szBitmap && (fh = _lopen(szBitmap, OF_READ)) != -1)
    {
        BITMAPFILEHEADER bf;
        BITMAPINFOHEADER bi;

        _lread(fh, &bf, sizeof(bf));
        _lread(fh, &bi, sizeof(bi));
        _lread(fh, ape, sizeof(ape));
        _lclose(fh);

        if (bi.biSize != sizeof(BITMAPINFOHEADER))
            n = 0;
        else if (bi.biBitCount > 8)
            n = 0;
        else if (bi.biClrUsed == 0)
            n = 1 << bi.biBitCount;
        else
            n = bi.biClrUsed;

        //
        //  a DIB color table has its colors stored BGR not RGB
        //  so flip them around.
        //
        for(i=0; i<n; i++ )
        {
            BYTE r = ape[i].peRed;
            ape[i].peRed  = ape[i].peBlue;
            ape[i].peBlue = r;
        }
    }

    pdd->CreatePalette(DDPCAPS_8BIT, ape, &ddpal, NULL);

    return ddpal;
}

/*
 * DDColorMatch
 *
 * convert a RGB color to a pysical color.
 *
 * we do this by leting GDI SetPixel() do the color matching
 * then we lock the memory and see what it got mapped to.
 */
#if DIRECTDRAW_VERSION >= CDX_DDVER
extern "C" DWORD DDColorMatch(IDirectDrawSurface4 *pdds, COLORREF rgb)
#else
extern "C" DWORD DDColorMatch(IDirectDrawSurface *pdds, COLORREF rgb)
#endif
{
    COLORREF rgbT;
    HDC hdc;
    DWORD dw = CLR_INVALID;
#if DIRECTDRAW_VERSION >= CDX_DDVER
    DDSURFACEDESC2 ddsd;
#else
    DDSURFACEDESC ddsd;
#endif
    HRESULT hres;

    //
    //  use GDI SetPixel to color match for us
    //
    if (rgb != CLR_INVALID && pdds->GetDC(&hdc) == DD_OK)
    {
        rgbT = GetPixel(hdc, 0, 0);             // save current pixel value
        SetPixel(hdc, 0, 0, rgb);               // set our value
        pdds->ReleaseDC(hdc);
    }

    //
    // now lock the surface so we can read back the converted color
    //
    ddsd.dwSize = sizeof(ddsd);
    while ((hres = pdds->Lock(NULL, &ddsd, 0, NULL)) == DDERR_WASSTILLDRAWING)
        ;

    if (hres == DD_OK)
    {
        dw  = *(DWORD *)ddsd.lpSurface;                     // get DWORD
        dw &= (1 << ddsd.ddpfPixelFormat.dwRGBBitCount)-1;  // mask it to bpp
        pdds->Unlock(NULL);
    }

    //
    //  now put the color that was there back.
    //
    if (rgb != CLR_INVALID && pdds->GetDC(&hdc) == DD_OK)
    {
        SetPixel(hdc, 0, 0, rgbT);
        pdds->ReleaseDC(hdc);
    }

    return dw;
}

/*
 * DDSetColorKey
 *
 * set a color key for a surface, given a RGB.
 * if you pass CLR_INVALID as the color key, the pixel
 * in the upper-left corner will be used.
 */
#if DIRECTDRAW_VERSION >= CDX_DDVER
extern "C" HRESULT DDSetColorKey(IDirectDrawSurface4 *pdds, COLORREF rgb)
#else
extern "C" HRESULT DDSetColorKey(IDirectDrawSurface *pdds, COLORREF rgb)
#endif
{
    DDCOLORKEY          ddck;

    ddck.dwColorSpaceLowValue  = DDColorMatch(pdds, rgb);
    ddck.dwColorSpaceHighValue = ddck.dwColorSpaceLowValue;
    return pdds->SetColorKey(DDCKEY_SRCBLT, &ddck);
}

//<JJH>

////////////////////////////////////////////////////////////////////////
//
//  GetRGBFormat
//
//  Fills out a RGBFORMAT structure.
//
////////////////////////////////////////////////////////////////////////
#if DIRECTDRAW_VERSION >= CDX_DDVER
BOOL GetRGBFormat ( IDirectDrawSurface4* Surface, RGBFORMAT* rgb)
#else
BOOL GetRGBFormat ( IDirectDrawSurface* Surface, RGBFORMAT* rgb)
#endif
{
#if DIRECTDRAW_VERSION >= CDX_DDVER
    DDSURFACEDESC2 ddsd;
#else
    DDSURFACEDESC ddsd;
#endif
    BYTE shiftcount;

	//get a surface description
    ddsd.dwSize = sizeof( ddsd );
    ddsd.dwFlags = DDSD_PIXELFORMAT;

    if (Surface->GetSurfaceDesc ( &ddsd ) != DD_OK )
        return FALSE;

	// Check if the surface is 4 or 8 bpp
	if (ddsd.ddpfPixelFormat.dwRGBBitCount <= 8)
	{
		rgb->bpp = ddsd.ddpfPixelFormat.dwRGBBitCount;
		
		rgb->depth.rgbRed      = 0;
		rgb->Position.rgbRed   = 0;

		rgb->depth.rgbGreen    = 0;
		rgb->Position.rgbGreen = 0;

		rgb->depth.rgbBlue     = 0;
		rgb->Position.rgbBlue  = 0;
		
		return TRUE;
	}

	//Get the RED channel data for 15, 16, 24, or 32 bpp surface
    shiftcount = 0;
    while ( !(ddsd.ddpfPixelFormat.dwRBitMask & 1) && (shiftcount < 32))
    {
        ddsd.ddpfPixelFormat.dwRBitMask >>= 1;
        shiftcount++;
    }

	// depth is the highest possible value that this channel could
	// be set to. It is cast to a BYTE because 255 is the highest
	// value for all modes.
    rgb->depth.rgbRed = (BYTE) ddsd.ddpfPixelFormat.dwRBitMask;

	// Position represents where the color channel starts, bitwise, 
	// in the pixel. Another way of saying this that Position is 
	// eqaul to the number of zeros to the right of the channel's
	// bitmask.
    rgb->Position.rgbRed = shiftcount;

    //Get the GREEN channel data for 15, 16, 24, or 32 bpp surface
    shiftcount = 0;
    while ( !(ddsd.ddpfPixelFormat.dwGBitMask & 1) && (shiftcount < 32))
    {
        ddsd.ddpfPixelFormat.dwGBitMask >>= 1;
        shiftcount++;
    }
    rgb->depth.rgbGreen =(BYTE)ddsd.ddpfPixelFormat.dwGBitMask;
    rgb->Position.rgbGreen = shiftcount;

    //Get the BLUE channel data for 15, 16, 24, or 32 bpp surface
    shiftcount = 0;
    while ( !(ddsd.ddpfPixelFormat.dwBBitMask & 1) && (shiftcount < 32))
    {
        ddsd.ddpfPixelFormat.dwBBitMask >>= 1;
        shiftcount++;
    }
    rgb->depth.rgbBlue =(BYTE)ddsd.ddpfPixelFormat.dwBBitMask;
    rgb->Position.rgbBlue = shiftcount;

	//  Determine the bpp for the surface. (15, 16, 24, or 32)
	if (ddsd.ddpfPixelFormat.dwFlags & DDPF_RGB)
	{
		if (ddsd.ddpfPixelFormat.dwRGBBitCount == 16)
		{
			if ((rgb->depth.rgbBlue + rgb->depth.rgbGreen + rgb->depth.rgbRed) == 93)
				rgb->bpp = 15;
			else
				rgb->bpp = 16;
		}
		else if (ddsd.ddpfPixelFormat.dwRGBBitCount == 24)
			rgb->bpp = 24;
		else if (ddsd.ddpfPixelFormat.dwRGBBitCount == 32)
			rgb->bpp = 32;
		else
			rgb->bpp = 0;
	}
	else
	{
		rgb->bpp = 0;
		return FALSE;
	}

	return TRUE;
}
//<JJH>

////////////////////////////////////////////////////////////////////////
//
//  IsMMX
//
//  Is the machine capable of MMX instructions
//
////////////////////////////////////////////////////////////////////////
#ifdef _M_ALPHA
// The Alpha processor doesn't support MMX, it has MVI
BOOL __cdecl IsMMX(void)
{
	return FALSE;
}

#else		/* must be a different processor type */

#ifdef _MSC_VER
BOOL __cdecl IsMMX(void)
{
    SYSTEM_INFO si;
    int nCPUFeatures=0;
    GetSystemInfo(&si);
    if (si.dwProcessorType != PROCESSOR_INTEL_386 && si.dwProcessorType != PROCESSOR_INTEL_486)
    {
        try
        {
            __asm
            {
                ; we must push/pop the registers << CPUID>>  writes to, as the
				; optimiser doesn't know about << CPUID>> , and so doesn't expect
				; these registers to change.
                push eax
                push ebx
                push ecx
                push edx

                ; << CPUID>> 
                ; eax=0,1,2 -> CPU info in eax,ebx,ecx,edx
                mov eax,1
                _emit 0x0f
                _emit 0xa2
                mov nCPUFeatures,edx

                pop edx
                pop ecx
                pop ebx
                pop eax
            }
        }
        catch(...) // just to be sure...
        {
			return false;
        }
    }
    return (nCPUFeatures & 0x00800000) != 0;
}
#else
typedef void (*ASMProc)(); 
BOOL __cdecl IsMMX(void)
{

    SYSTEM_INFO si;
    int nCPUFeatures=0;
   
    char xMMXCode[2] = {0x0f, 0xa2};
    ASMProc function_xMMXCode;

    GetSystemInfo(&si);
    if (si.dwProcessorType != PROCESSOR_INTEL_386 && si.dwProcessorType != PROCESSOR_INTEL_486)
    {
        try
        {
            __asm
            {
                push eax
                push ebx
                push ecx
                push edx

                mov eax,1
            }
            function_xMMXCode = (ASMProc)xMMXCode;
            function_xMMXCode();
            __asm {
                mov nCPUFeatures,edx

                pop edx
                pop ecx
                pop ebx
                pop eax
            }
        }
        catch(...) // just to be sure...
        {
            return false;
        }
    }
    return (nCPUFeatures & 0x00800000) != 0;
}
#endif

#endif			/* ifdef _M_ALPHA */
