/* Copyright (C) 1997-1998, Russell Lang.  All rights reserved.
  
  This file is part of RedMon.
  
  This program is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the RedMon Free Public Licence 
  (the "Licence") for full details.
  
  Every copy of RedMon must include a copy of the Licence, normally in a 
  plain ASCII text file named LICENCE.  The Licence grants you the right 
  to copy, modify and redistribute RedMon, but only under certain conditions 
  described in the Licence.  Among other things, the Licence requires that 
  the copyright notice and this notice be preserved on all copies.
*/

/* redmon.c */

/*
 * This is a Port Monitor for Windows 95 and Windows NT.
 * The monitor name is "Redirected Port" .
 * A write to any port provided by this monitor will be
 * redirected to a program using a pipe to stdin.
 *
 * An example is redirecting the output from the 
 * PostScript printer driver to Ghostscript, which then
 * writes to a non-PostScript printer.
 *
 * For efficiency reasons, don't use the C run time library. 
 *
 * The Windows NT version must use Unicode, so Windows NT
 * specific code is conditionally compiled with 
 *   #ifdef UNICODE
 *   #endif
 */

#define STRICT
#include <windows.h>
#include "redmon.h"
#ifdef BETA
#include <time.h>
#endif

typedef struct redata {
    BOOL started;
    BOOL error;
    TCHAR portname[MAXSHORTSTR];
    TCHAR command[1024];
    DWORD delay;
    DWORD show;
    TCHAR pPrinterName[MAXSTR];
    DWORD JobId;
    HANDLE hChildStdinRd;
    HANDLE hChildStdinWrDup;	/* We write to this one */
    HANDLE hChildStdoutRd; 	/* We read from this one */
    HANDLE hChildStdoutWr;
    HANDLE hChildStderrRd; 	/* We read from this one */
    HANDLE hChildStderrWr;
#ifdef SAVESTD
    HANDLE hSaveStdin;
    HANDLE hSaveStdout;
    HANDLE hSaveStderr;
#endif
    HANDLE hLogFile;
    PROCESS_INFORMATION piProcInfo;
    BOOL debug;
    HANDLE hmutex;	/* To control access to pipe and file handles */
    BOOL watch;		/* TRUE if watch thread should keep running */
    HANDLE hthread;
    DWORD threadid;
} REDATA;


typedef struct add_param {
	LPTSTR portname;
	LPTSTR portdesc;
} ADDPARAM;

BOOL start_redirect(REDATA * prd);
void reset_redata(REDATA *prd);
BOOL CALLBACK AddDlgProc(HWND hDlg, UINT message, 
        WPARAM wParam, LPARAM lParam);
BOOL CALLBACK ConfigDlgProc(HWND hDlg, UINT message, 
	WPARAM wParam, LPARAM lParam);
BOOL CALLBACK LogfileDlgProc(HWND hDlg, UINT message, 
	WPARAM wParam, LPARAM lParam);

/* These are our only global variables */
TCHAR rekey[MAXSTR];   /* Registry key name for our use */
HINSTANCE hdll;
TCHAR helpfile[MAXSTR];
TCHAR copyright[] = COPYRIGHT;
TCHAR version[] = VERSION;

#define PORTSNAME TEXT("Ports")
#define DESCKEY TEXT("Description")
#define COMMANDKEY TEXT("Command")
#define ARGKEY TEXT("Arguments")
#define SHOWKEY TEXT("ShowWindow")
#define LOGUSEKEY TEXT("LogFileUse")
#define LOGNAMEKEY TEXT("LogFileName")
#define LOGDELAYKEY TEXT("LogFileDelay")
#define LOGDEBUGKEY TEXT("LogFileDebug")
#define BACKSLASH TEXT("\\")
#define DEFAULT_DELAY 15   /* seconds */

/*
 * Required functions for a Port Monitor are:
 *   AddPort
 *   AddPortEx              (NT only)
 *   ClosePort
 *   ConfigurePort
 *   DeletePort
 *   EndDocPort
 *   EnumPorts
 *   GetPrinterDataFromPort
 *   InitializeMonitor      (NT 3.51 only)
 *   InitializeMonitorEx    (95 only)
 *   InitializePrintMonitor (NT only)
 *   OpenPort
 *   ReadPort
 *   SetPortOpenTimeouts
 *   StartDocPort
 *   WritePort
 */


void
request_mutex(REDATA *prd)
{
    if ((prd->hmutex != NULL) && (prd->hmutex != INVALID_HANDLE_VALUE))
        WaitForSingleObject(prd->hmutex, 30000);
}

void
release_mutex(REDATA *prd)
{
    if ((prd->hmutex != NULL) && (prd->hmutex != INVALID_HANDLE_VALUE))
	ReleaseMutex(prd->hmutex);
}

#ifdef BETA
int
beta_expired(void)
{
  time_t today = time(NULL);
  struct tm *t;
  t = localtime(&today);
  if (t->tm_year+1900 < BETA_YEAR)
    return 0;
  if (t->tm_year+1900 > BETA_YEAR)
    return 1;    /* beta copy has expired */
  if (t->tm_mon+1 < BETA_MONTH)
    return 0;
  if (t->tm_mon+1 > BETA_MONTH)
    return 1;    /* beta copy has expired */
  if (t->tm_mday < BETA_DAY)
    return 0;
  return 1;    /* beta copy has expired */
}

int beta(void)
{
  if (beta_expired()) {
    TCHAR buf[MAXSTR];
    TCHAR title[MAXSTR];
    LoadString(hdll, IDS_BETAEXPIRED, buf, sizeof(buf)/sizeof(TCHAR)-1);
    LoadString(hdll, IDS_TITLE, title, sizeof(title)/sizeof(TCHAR)-1);
    MessageBox(HWND_DESKTOP, buf, title, MB_OK | MB_ICONHAND);
    return 1;
  }
  return 0;
}
#endif /* BETA */


BOOL WINAPI rEnumPorts(LPTSTR pName,DWORD Level,LPBYTE  pPorts, 
         DWORD cbBuf, LPDWORD pcbNeeded, LPDWORD pcReturned)
{
    HKEY hkey, hsubkey;
    TCHAR portname[MAXSHORTSTR], portdesc[MAXSHORTSTR];
    TCHAR monitorname[MAXSHORTSTR];
    TCHAR buf[MAXSTR];
    int needed;
    DWORD cbData, keytype;
    PORT_INFO_1 *pi1;
    PORT_INFO_2 *pi2;
    LPTSTR pstr;
    LONG rc;
    int i;

    *pcbNeeded = 0;
    *pcReturned = 0;

    if (pName != NULL) {
	return FALSE;
    }

    if ((Level < 1) || (Level > 2)) {
	SetLastError(ERROR_INVALID_LEVEL);
	return FALSE;
    }

    lstrcpy(buf, rekey);
    lstrcat(buf, BACKSLASH);
    lstrcat(buf, PORTSNAME);
    rc = RegOpenKeyEx(HKEY_LOCAL_MACHINE, buf, 0, KEY_READ, &hkey);
    if (rc != ERROR_SUCCESS) {
	return TRUE;	/* There are no ports */
    }

    LoadString(hdll, IDS_MONITORNAME, monitorname, 
	sizeof(monitorname)/sizeof(TCHAR)-1);

    /* First pass is to calculate the number of bytes needed */
    needed = 0;
    i = 0;
    rc = RegEnumKey(hkey, 0, portname, sizeof(portname));
    while (rc == ERROR_SUCCESS) {
	needed += (lstrlen(portname) + 1) * sizeof(TCHAR);
	if (Level == 1) {
	    needed += sizeof(PORT_INFO_1);
	}
	else if (Level == 2) {
	    needed += sizeof(PORT_INFO_2);
	    needed += (lstrlen(monitorname) + 1) * sizeof(TCHAR);
	    rc = RegOpenKeyEx(hkey, portname, 0, KEY_READ, &hsubkey);
	    if (rc == ERROR_SUCCESS) {
		cbData = sizeof(portdesc);
		keytype = REG_SZ;
		RegQueryValueEx(hsubkey, DESCKEY, 0, &keytype, 
			(LPBYTE)portdesc, &cbData);
		if (rc == ERROR_SUCCESS)
		    needed += (lstrlen(portdesc) + 1) * sizeof(TCHAR);
		else
		    needed += 1 * sizeof(TCHAR);	/* empty string */
		RegCloseKey(hsubkey);
	    }
	    else {
		needed += 1 * sizeof(TCHAR);	/* empty string */
	    }
	}
	i++;
        rc = RegEnumKey(hkey, i, portname, sizeof(portname));
    }
    *pcbNeeded = needed;
   
    if ((pPorts == NULL) || (needed > cbBuf)) {
	RegCloseKey(hkey);
	SetLastError(ERROR_INSUFFICIENT_BUFFER);
	return FALSE;
    }

    /* Second pass to copy the data to the buffer */

    /* PORT_INFO_x structures must be placed at the beginning
     * of the buffer, and strings at the end of the buffer.
     * This is important!  It appears that one buffer is 
     * allocated, then each port monitor is called in turn to
     * add its entries to the buffer, between previous entries.
     */

    i = 0;
    pi1 = (PORT_INFO_1 *)pPorts;
    pi2 = (PORT_INFO_2 *)pPorts;
    pstr = (LPTSTR)(pPorts + cbBuf);
    rc = RegEnumKey(hkey, 0, portname, sizeof(portname));
    while (rc == ERROR_SUCCESS) {
	if (Level == 1) {
	    pstr -= lstrlen(portname) + 1;
	    lstrcpy(pstr, portname);
	    pi1[i].pName = pstr;
	}
	else if (Level == 2){
	    pstr -= lstrlen(portname) + 1;
	    lstrcpy(pstr, portname);
	    pi2[i].pPortName = pstr;

	    pstr -= lstrlen(monitorname) + 1;
	    lstrcpy(pstr, monitorname);
	    pi2[i].pMonitorName = pstr;

	    rc = RegOpenKeyEx(hkey, portname, 0, KEY_READ, &hsubkey);
	    if (rc == ERROR_SUCCESS) {
		cbData = sizeof(portdesc);
		keytype = REG_SZ;
		RegQueryValueEx(hsubkey, DESCKEY, 0, &keytype, 
			(LPBYTE)portdesc, &cbData);
		if (rc != ERROR_SUCCESS)
		    portdesc[0] = '\0';
		RegCloseKey(hsubkey);
	    }
	    else {
		portdesc[0] = '\0';
	    }

	    pstr -= lstrlen(portdesc) + 1;
	    lstrcpy(pstr, portdesc);
	    pi2[i].pDescription = pstr;

	    /* Say that writing to this port is supported, */
	    /* but reading is not. */
	    /* Using fPortType = 3 is wrong. */
	    /* Options are PORT_TYPE_WRITE=1, PORT_TYPE_READ=2, */
	    /*   PORT_TYPE_REDIRECTED=4, PORT_TYPE_NET_ATTACHED=8 */
	    pi2[i].fPortType = PORT_TYPE_WRITE;
	    pi2[i].Reserved = 0;
	}
	i++;
        rc = RegEnumKey(hkey, i, portname, sizeof(portname));
    }
    *pcReturned = i;
    RegCloseKey(hkey);

    return TRUE;
}


BOOL WINAPI rAddPort(LPTSTR pName, HWND hWnd, LPTSTR pMonitorName)
{
    int i, j;
    TCHAR portname[MAXSHORTSTR], portdesc[MAXSHORTSTR];
    TCHAR buf[MAXSTR], str[MAXSTR];
    HGLOBAL hglobal;
    LPBYTE ports;
    DWORD cbBuf, needed, returned;
    PORT_INFO_2 *pi2;
    BOOL unique;
    LONG rc;
    HKEY hkey, hsubkey;
    ADDPARAM ap;

    if (pName != NULL)
	return FALSE;

    /* Get the list of existing ports. */
    /* Ask for all ports, not just ours.
     * If we ask for only RedMon ports by calling rEnumPorts, a user
     * can create a second LPT1: This isn't useful because it doesn't 
     * get serviced before the Local Port, and we can't delete it
     * with the Delete Port button.  Instead we have to use the
     * Registry Editor to clean up the mess.
     * To avoid this, stop users from entering any existing port name.
     */
    needed = 0;
    /* 1998-01-08
     * The TELES.FaxMon monitor is buggy - it returns an incorrect
     * value for "needed" if we use the following call.
     *    EnumPorts(NULL, 2, NULL, 0, &needed, &returned);
     * To dodge this bug, pass it buffer of non-zero size.
     */
   
    cbBuf = 4096;
    hglobal = GlobalAlloc(GPTR, (DWORD)cbBuf);  
    ports = GlobalLock(hglobal);
    if (ports == NULL) {
	SetLastError(ERROR_NOT_ENOUGH_MEMORY);
	return FALSE;
    }
    if (!EnumPorts(NULL, 2, ports, cbBuf, &needed, &returned)) {
	GlobalUnlock(hglobal);
	GlobalFree(hglobal);
	if (GetLastError() != ERROR_INSUFFICIENT_BUFFER) {
	    return FALSE;	/* give up */
	}
	else {
	    /* try again, with requested size */
            cbBuf = needed;
            /* If there are no ports installed, avoid trying to 
	     * allocate 0 bytes
	     */
	    hglobal = GlobalAlloc(GPTR, (DWORD)cbBuf+4);  
	    ports = GlobalLock(hglobal);
	    if (ports == NULL) {
		SetLastError(ERROR_NOT_ENOUGH_MEMORY);
		return FALSE;
	    }
	    if (!EnumPorts(NULL, 2, ports, cbBuf, &needed, &returned)) {
		GlobalUnlock(hglobal);
		GlobalFree(hglobal);
		return FALSE;
	    }
	}
    }

    pi2 = (PORT_INFO_2 *)ports;

    unique = TRUE;
    do {
	if (!unique) {
	    LoadString(hdll, IDS_NOTUNIQUE, str, sizeof(str)/sizeof(TCHAR)-1);
	    wsprintf(buf, str, portname);
	    LoadString(hdll, IDS_ADDPORT, str, sizeof(str)/sizeof(TCHAR)-1);
	    if (MessageBox(hWnd, buf, str, MB_OKCANCEL) == IDCANCEL) {
		GlobalUnlock(hglobal);
		GlobalFree(hglobal);
		return FALSE;
	    }
	}

	/* Suggest a unique port name */
	for (i=1; i<100; i++) {
	    wsprintf(portname, TEXT("RPT%d:"), i);
	    for (j=0; j<returned; j++) {
		if (lstrcmp(portname, pi2[j].pPortName) == 0)
		    break;
	    }
	    if (j >= returned)
		break;	/* didn't find a match so portname is unique */
	}
        LoadString(hdll, IDS_MONITORNAME, portdesc, 
		sizeof(portdesc)/sizeof(TCHAR)-1);

	ap.portname = portname;
	ap.portdesc = portdesc;
	if (!DialogBoxParam(hdll, MAKEINTRESOURCE(IDD_ADDPORT), 
		hWnd, AddDlgProc, (LPARAM)&ap)) {
	    GlobalUnlock(hglobal);
	    GlobalFree(hglobal);
	    return FALSE;
	}

	if (lstrlen(portname) && portname[lstrlen(portname)-1] != ':')
	    lstrcat(portname, TEXT(":"));	/* append ':' if not present */

	for (j=0; j<returned; j++) {
	    if (lstrcmp(portname, pi2[j].pPortName) == 0) {
		unique = FALSE;
		break;
	    }
	}
    } while (!unique);

    GlobalUnlock(hglobal);
    GlobalFree(hglobal);

    /* store new port name in registry */
    rc = RegOpenKeyEx(HKEY_LOCAL_MACHINE, rekey, 0, KEY_ALL_ACCESS, &hkey);
    if (rc == ERROR_SUCCESS) {
	lstrcpy(buf, PORTSNAME);
	lstrcat(buf, BACKSLASH);
	lstrcat(buf, portname);
	rc = RegCreateKey(hkey, buf, &hsubkey);
	if (rc == ERROR_SUCCESS) {
	    RegSetValueEx(hsubkey, DESCKEY, 0, REG_SZ,
		(CONST BYTE *)portdesc, (lstrlen(portdesc)+1)*sizeof(TCHAR));
	}
    } 

   return TRUE;
}

#ifdef UNICODE
/* Windows NT only */
/* untested */
BOOL WINAPI rAddPortEx(LPWSTR pName, DWORD Level, LPBYTE lpBuffer, 
     LPWSTR lpMonitorName)
{
    int j;
    TCHAR *portname;
    TCHAR portdesc[MAXSHORTSTR];
    TCHAR buf[MAXSTR];
    HGLOBAL hglobal;
    LPBYTE ports;
    DWORD cbBuf, needed, returned;
    PORT_INFO_2 *pi2;
    LONG rc;
    HKEY hkey, hsubkey;

    if (pName != NULL)
	return FALSE;

    if ((Level != 1) && (Level != 2)) {
	SetLastError(ERROR_INVALID_LEVEL);
	return FALSE;
    }

    /* Get the list of existing ports. */
    EnumPorts(NULL, 2, NULL, 0, &needed, &returned);
    cbBuf = needed;
    hglobal = GlobalAlloc(GPTR, (DWORD)cbBuf);
    ports = GlobalLock(hglobal);
    if (ports == NULL) {
	SetLastError(ERROR_NOT_ENOUGH_MEMORY);
	return FALSE;
    }

    if (!EnumPorts(NULL, 2, ports, cbBuf, &needed, &returned)) {
	GlobalUnlock(hglobal);
	GlobalFree(hglobal);
	return FALSE;
    }
    pi2 = (PORT_INFO_2 *)ports;

    if (Level == 1) {
	PORT_INFO_1 *addpi1 = (PORT_INFO_1 *)lpBuffer;
	portname = addpi1->pName;
        LoadString(hdll, IDS_MONITORNAME, portdesc, 
		sizeof(portdesc)/sizeof(TCHAR)-1);
    }
    if (Level == 2) {
	PORT_INFO_2 *addpi2 = (PORT_INFO_2 *)lpBuffer;
	portname = addpi2->pPortName;
	lstrcpyn(portdesc, addpi2->pDescription, 
		sizeof(portdesc)/sizeof(TCHAR)-1);
    }

    /* Make sure portname is unique */
    for (j=0; j<returned; j++) {
	if (lstrcmp(portname, pi2[j].pPortName) == 0) {
	    /* fail because port already exists */
	    GlobalUnlock(hglobal);
	    GlobalFree(hglobal);
	    return FALSE;
	}
    }

    GlobalUnlock(hglobal);
    GlobalFree(hglobal);

    /* store new port name in registry */
    rc = RegOpenKeyEx(HKEY_LOCAL_MACHINE, rekey, 0, KEY_ALL_ACCESS, &hkey);
    if (rc == ERROR_SUCCESS) {
	lstrcpy(buf, PORTSNAME);
	lstrcat(buf, BACKSLASH);
	lstrcat(buf, portname);
	rc = RegCreateKey(hkey, buf, &hsubkey);
	if (rc == ERROR_SUCCESS) {
	    RegSetValueEx(hsubkey, DESCKEY, 0, REG_SZ,
		(CONST BYTE *)portdesc, (lstrlen(portdesc)+1)*sizeof(TCHAR));
	}
    } 
	
   return TRUE;
}
#endif

BOOL WINAPI rDeletePort(LPTSTR pName, HWND hWnd, LPTSTR pPortName)
{
    HKEY hkey;
    LONG rc;
    TCHAR buf[MAXSTR];
    if (pName != NULL)
	return FALSE;

    lstrcpy(buf, rekey);
    lstrcat(buf, BACKSLASH);
    lstrcat(buf, PORTSNAME);
    rc = RegOpenKeyEx(HKEY_LOCAL_MACHINE, buf, 0, KEY_ALL_ACCESS, &hkey);
    if (rc != ERROR_SUCCESS) {
	SetLastError(REGDB_E_KEYMISSING);
	return FALSE;	/* There are no ports */
    }
    RegDeleteKey(hkey, pPortName); 
    RegCloseKey(hkey);

    return TRUE;
}


BOOL WINAPI rConfigurePort(LPTSTR pName, HWND hWnd, LPTSTR pPortName)
{
    if (pName != NULL)
	return FALSE;

    (void)DialogBoxParam(hdll, MAKEINTRESOURCE(IDD_CONFIGPORT), 
		hWnd, ConfigDlgProc, (LPARAM)pPortName);
    return TRUE;
}



BOOL WINAPI rOpenPort(LPTSTR pName, PHANDLE pHandle)
{
    HGLOBAL hglobal;
    REDATA *prd;

    TCHAR buf[MAXSTR];
#ifdef BETA
     if (beta())
	return FALSE;
#endif

    hglobal = GlobalAlloc(GPTR, (DWORD)sizeof(REDATA));
    prd = (REDATA *)GlobalLock(hglobal);
    if (prd == (REDATA *)NULL) {
	SetLastError(ERROR_NOT_ENOUGH_MEMORY);
	return FALSE;
    }
    FillMemory((PVOID)prd, sizeof(REDATA), 0);
    reset_redata(prd);
    lstrcpy(prd->portname, pName);

    /* Do the rest of the opening in rStartDocPort() */

    GlobalUnlock(hglobal);
    *pHandle = (HANDLE)hglobal;
    return TRUE;
}

#ifdef UNUSED
/* Don't use OpenPortEx in a Port Monitor */
BOOL WINAPI rOpenPortEx(LPTSTR  pPortName, 
        LPTSTR  pPrinterName, PHANDLE pHandle, struct _MONITOR 
        FAR *pMonitor);
#endif


/* The log file is single byte characters only */
/* Write a single character or wide character string to the log file,
 * converting it to single byte characters */
void write_string_to_log(REDATA *prd, LPTSTR buf)
{
int count;
CHAR cbuf[256];
BOOL UsedDefaultChar;
DWORD cbWritten;
    if (prd->hLogFile == INVALID_HANDLE_VALUE)
	return;

    request_mutex(prd);
#ifdef UNICODE
    while (lstrlen(buf)) {
	count = min(lstrlen(buf), sizeof(cbuf));
	WideCharToMultiByte(CP_ACP, 0, buf, count,
		cbuf, sizeof(cbuf), NULL, &UsedDefaultChar);
	buf += count;
	WriteFile(prd->hLogFile, cbuf, count, &cbWritten, NULL);
    }
#else
    WriteFile(prd->hLogFile, buf, lstrlen(buf), &cbWritten, NULL);
#endif
    FlushFileBuffers(prd->hLogFile);
    release_mutex(prd);
}

void
reset_redata(REDATA *prd)
{
    prd->started = FALSE;
    prd->error = FALSE;
    prd->command[0] = '\0';
    prd->delay = DEFAULT_DELAY;	/* delay if log file not used */
    prd->show = 0;
    prd->pPrinterName[0] = '\0';
    prd->JobId = 0;
    prd->hChildStdinRd = INVALID_HANDLE_VALUE;
    prd->hChildStdinWrDup = INVALID_HANDLE_VALUE;
    prd->hChildStdoutRd = INVALID_HANDLE_VALUE;
    prd->hChildStdoutWr = INVALID_HANDLE_VALUE;
    prd->hChildStderrWr = INVALID_HANDLE_VALUE;
    prd->hChildStderrWr = INVALID_HANDLE_VALUE;
#ifdef SAVESTD
    prd->hSaveStdin = INVALID_HANDLE_VALUE;
    prd->hSaveStdout = INVALID_HANDLE_VALUE;
    prd->hSaveStderr = INVALID_HANDLE_VALUE;
#endif
    prd->hLogFile = INVALID_HANDLE_VALUE;
    prd->piProcInfo.hProcess = INVALID_HANDLE_VALUE;
    prd->hmutex = INVALID_HANDLE_VALUE;
    prd->watch = FALSE;
    prd->hthread = INVALID_HANDLE_VALUE;
    prd->threadid = 0;
}

/* copy stdout and stderr to log file, if open */
void
flush_stdout(REDATA *prd)
{
    BYTE buf[MAXSTR];
    DWORD bytes_available, dwRead, dwWritten;
    BOOL result;

    request_mutex(prd);

    /* copy anything on stdout to log file */
    bytes_available = 0;
    result = PeekNamedPipe(prd->hChildStdoutRd, NULL, 0, NULL, 
	    &bytes_available, NULL);
    while (result && bytes_available) {
	if (!ReadFile(prd->hChildStdoutRd, buf, sizeof(buf), &dwRead, NULL) ||
	    dwRead == 0) break;
	if (prd->hLogFile != INVALID_HANDLE_VALUE) {
	    WriteFile(prd->hLogFile, buf, dwRead, &dwWritten, NULL);
	    FlushFileBuffers(prd->hLogFile);
	}
	result = PeekNamedPipe(prd->hChildStdoutRd, NULL, 0, NULL, 
	    &bytes_available, NULL);
    }

    /* copy anything on stderr to log file */
    bytes_available = 0;
    result = PeekNamedPipe(prd->hChildStderrRd, NULL, 0, NULL, 
	    &bytes_available, NULL);
    while (result && bytes_available) {
	if (!ReadFile(prd->hChildStderrRd, buf, sizeof(buf), &dwRead, NULL) ||
	    dwRead == 0) break;
	if (prd->hLogFile != INVALID_HANDLE_VALUE) {
	    WriteFile(prd->hLogFile, buf, dwRead, &dwWritten, NULL);
	    FlushFileBuffers(prd->hLogFile);
	}
	result = PeekNamedPipe(prd->hChildStderrRd, NULL, 0, NULL, 
	    &bytes_available, NULL);
    }

    release_mutex(prd);
}

/* Watch to see if the process ends prematurely */
DWORD WINAPI WatchThread(LPVOID lpThreadParameter)
{
    HANDLE hPort = (HANDLE)lpThreadParameter;
    REDATA *prd = GlobalLock((HGLOBAL)hPort);
    DWORD exit_status;

    if (prd == (REDATA *)NULL)
	return 1;

    if (prd->debug)
	write_string_to_log(prd, TEXT("\r\nREDMON WatchThread: started\r\n"));

    while (prd->watch && !prd->error) {
	/* If WriteFile to the pipe has blocked because the program 
	 * has terminated, close down the  write pipe to unblock WriteFile.
	 */
	if (prd->piProcInfo.hProcess == INVALID_HANDLE_VALUE)
	    prd->error = TRUE;
	if (!prd->error 
	    && GetExitCodeProcess(prd->piProcInfo.hProcess, &exit_status)
	    && (exit_status != STILL_ACTIVE))
	    prd->error = TRUE;

	if (prd->error) {
	    if (prd->debug)
		write_string_to_log(prd, 
		  TEXT("REDMON WatchThread: process isn't running.\r\n\
    Closing both ends of stdin pipe\r\n"));
	    request_mutex(prd);
	    if (prd->hChildStdinWrDup != INVALID_HANDLE_VALUE) {
		CloseHandle(prd->hChildStdinWrDup);
		prd->hChildStdinWrDup = INVALID_HANDLE_VALUE;
	    }
	    if (prd->hChildStdinRd != INVALID_HANDLE_VALUE) {
		CloseHandle(prd->hChildStdinRd);
		prd->hChildStdinRd = INVALID_HANDLE_VALUE;
	    }
	    release_mutex(prd);
	    break;
	}
	Sleep(2000);  /* Pause a little to avoid wasting CPU time */
    }

    if (prd->debug)
	write_string_to_log(prd, TEXT("\r\nREDMON WatchThread: ending\r\n"));

    GlobalUnlock(hPort);
    return 0;
}

BOOL WINAPI rStartDocPort(HANDLE  hPort, LPTSTR  pPrinterName, 
        DWORD   JobId, DWORD   Level, LPBYTE  pDocInfo)
{
    TCHAR buf[MAXSTR];
    TCHAR command[MAXSTR], args[MAXSTR];
    HKEY hkey;
    REDATA *prd = GlobalLock((HGLOBAL)hPort);
    BOOL flag;
    LONG rc;
    DWORD cbData, keytype;
    DWORD uselog;

    if (prd == (REDATA *)NULL) {
	SetLastError(ERROR_INVALID_HANDLE);
	return FALSE;
    }

    reset_redata(prd);
    lstrcpy(prd->pPrinterName, pPrinterName);
    prd->JobId = JobId;

    /* Launch application */

    lstrcpy(buf, rekey);
    lstrcat(buf, BACKSLASH);
    lstrcat(buf, PORTSNAME);
    lstrcat(buf, BACKSLASH);
    lstrcat(buf, prd->portname);
    rc = RegOpenKeyEx(HKEY_LOCAL_MACHINE, buf, 0, KEY_READ, &hkey);
    if (rc != ERROR_SUCCESS) {
	GlobalUnlock((HGLOBAL)hPort);
	SetLastError(REGDB_E_KEYMISSING);
	return FALSE;	/* There are no ports */
    }

    /* Build command line */
    lstrcpy(prd->command, TEXT("\042"));
    cbData = sizeof(prd->command) - (1 * sizeof(TCHAR));
    keytype = REG_SZ;
    RegQueryValueEx(hkey, COMMANDKEY, 0, &keytype, 
	(LPBYTE)(prd->command+1), &cbData);
    lstrcat(prd->command, TEXT("\042 "));
    cbData = sizeof(prd->command) -
	(1 + lstrlen(prd->command)) * sizeof(TCHAR);
    RegQueryValueEx(hkey, ARGKEY, 0, &keytype, 
	(LPBYTE)(prd->command + lstrlen(prd->command)), &cbData);

    /* Find out if window is Normal, Minimized or Hidden */
    cbData = sizeof(DWORD);
    keytype = REG_DWORD;
    RegQueryValueEx(hkey, SHOWKEY, 0, &keytype, 
	(LPBYTE)&prd->show, &cbData);

    /* query if log is to be used */
    cbData = sizeof(DWORD);
    keytype = REG_DWORD;
    uselog = 0;
    RegQueryValueEx(hkey, LOGUSEKEY, 0, &keytype, 
	(LPBYTE)&uselog, &cbData);
    if (uselog) {
	/* Open optional log file */
	prd->hLogFile = INVALID_HANDLE_VALUE;
	buf[0] = '\0';
	cbData = sizeof(buf);
	keytype = REG_SZ;
	RegQueryValueEx(hkey, LOGNAMEKEY, 0, &keytype, (LPBYTE)buf, &cbData);
	if (lstrlen(buf)) {
	    prd->hLogFile = CreateFile(buf, GENERIC_WRITE, FILE_SHARE_READ, 
			NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
	}
	/* set delay */
	cbData = sizeof(DWORD);
	keytype = REG_DWORD;
        RegQueryValueEx(hkey, LOGDELAYKEY, 0, &keytype, 
	    (LPBYTE)&prd->delay, &cbData);
	/* find out if debug messages are required */
	cbData = sizeof(DWORD);
	keytype = REG_DWORD;
	RegQueryValueEx(hkey, LOGDEBUGKEY, 0, &keytype, 
	    (LPBYTE)&prd->debug, &cbData);
	if (prd->debug) {
	    LoadString(hdll, IDS_TITLE, buf, sizeof(buf)/sizeof(TCHAR)-1);
	    write_string_to_log(prd, buf);
	    write_string_to_log(prd, TEXT("\n"));
	    write_string_to_log(prd, copyright);
	    write_string_to_log(prd, version);
	}
    }


    RegCloseKey(hkey);

    prd->hmutex = CreateMutex(NULL, FALSE, NULL);
    flag = start_redirect(prd);
    if (flag) {
        WaitForInputIdle(prd->piProcInfo.hProcess, 5000);

	/* Create thread to monitor process and close pipe if
	 * process finishes.  This is needed to unblock a
 	 * WriteFile to the pipe from WritePort.
	 */
	prd->watch = TRUE;
	prd->hthread = CreateThread(NULL, 0, &WatchThread, 
		hPort, 0, &prd->threadid);
    }
    else {
	DWORD cbWritten;
	DWORD err = GetLastError();
	/* ENGLISH */
	wsprintf(buf, 
	   TEXT("StartDocPort: failed to start process\r\n  Port = %s\r\n  Command = %s\r\n  Error = %ld\r\n"),
	   prd->portname, prd->command, err);
	switch(err) {
	    case ERROR_FILE_NOT_FOUND:
		lstrcat(buf, TEXT("  File not found\r\n"));
		break;
	    case ERROR_PATH_NOT_FOUND:
		lstrcat(buf, TEXT("  Path not found\r\n"));
		break;
	    case ERROR_BAD_PATHNAME:
		lstrcat(buf, TEXT("  Bad path name\r\n"));
		break;
	}
	write_string_to_log(prd, buf);

	/* clean up pipes and redirected stdio */
	if (prd->hChildStderrRd)
	    CloseHandle(prd->hChildStderrRd);
	if (prd->hChildStderrWr)
	    CloseHandle(prd->hChildStderrWr);
	if (prd->hChildStdoutRd)
	    CloseHandle(prd->hChildStdoutRd);
	if (prd->hChildStdoutWr)
	    CloseHandle(prd->hChildStdoutWr);
	if (prd->hChildStdinRd)
	    CloseHandle(prd->hChildStdinRd);
	if (prd->hChildStdinWrDup)
	    CloseHandle(prd->hChildStdinWrDup);
	prd->hChildStderrRd   = INVALID_HANDLE_VALUE;
	prd->hChildStderrWr   = INVALID_HANDLE_VALUE;
	prd->hChildStdoutRd   = INVALID_HANDLE_VALUE;
	prd->hChildStdoutWr   = INVALID_HANDLE_VALUE;
	prd->hChildStdinRd    = INVALID_HANDLE_VALUE;
	prd->hChildStdinWrDup = INVALID_HANDLE_VALUE;
#ifdef SAVESTD
	SetStdHandle(STD_INPUT_HANDLE, prd->hSaveStdin);
	SetStdHandle(STD_OUTPUT_HANDLE, prd->hSaveStdout);
	SetStdHandle(STD_ERROR_HANDLE, prd->hSaveStdout);
#endif
    }

    if (prd->debug) {
	wsprintf(buf, 
	  TEXT("REDMON StartDocPort: returning %d\r\n\
  %s\r\n\
  Printer=%s\r\n\
  JobId=%d\r\n"), 
	  flag, prd->command, prd->pPrinterName, prd->JobId);
	write_string_to_log(prd, buf);
	if ((Level == 1) && pDocInfo) {
	    DOC_INFO_1 *dci1 = (DOC_INFO_1 *)pDocInfo;
	    wsprintf(buf, TEXT("\
  Level=1\r\n\
    DocumentName=\042%s\042\r\n\
    OutputFile=\042%s\042\r\n\
    Datatype=\042%s\042\r\n"),
		dci1->pDocName ? dci1->pDocName : TEXT("(null)"), 
		dci1->pOutputFile ? dci1->pOutputFile : TEXT("(null)"),
		dci1->pDatatype ? dci1->pDatatype : TEXT("(null)"));
	    write_string_to_log(prd, buf);
	}
	else if ((Level == 2) && pDocInfo) {
	    DOC_INFO_2 *dci2 = (DOC_INFO_2 *)pDocInfo;
	    wsprintf(buf, TEXT("\
  Level=2\r\n\
    DocumentName=\042%s\042\r\n\
    OutputFile=\042%s\042\r\n\
    Datatype=\042%s\042\r\n\
    Mode=%d\r\n\
    JobId=%d\r\n"),
		dci2->pDocName ? dci2->pDocName : TEXT("(null)"), 
		dci2->pOutputFile ? dci2->pOutputFile : TEXT("(null)"),
		dci2->pDatatype ? dci2->pDatatype : TEXT("(null)"),
		dci2->dwMode, dci2->JobId);
	    write_string_to_log(prd, buf);
	}
	else {
	    wsprintf(buf, TEXT("  Level=%d pDocInfo=%d\r\n"),
		Level, pDocInfo);
	    write_string_to_log(prd, buf);
	}
    }

    if (!flag) {
	CloseHandle(prd->hLogFile);
	prd->hLogFile = INVALID_HANDLE_VALUE;
	CloseHandle(prd->hmutex);
	prd->hmutex    = INVALID_HANDLE_VALUE;
    }

    prd->started = flag;

    GlobalUnlock((HGLOBAL)hPort);

    return flag;
}


/* WritePort is normally called between StartDocPort and EndDocPort,
 * but can be called outside this pair for bidirectional printers.
 */
BOOL WINAPI rWritePort(HANDLE  hPort, LPBYTE  pBuffer, 
        DWORD   cbBuf, LPDWORD pcbWritten)
{
    TCHAR buf[MAXSTR];
    BOOL flag;
    DWORD dwRead, dwWritten;
    DWORD exit_status;
    HANDLE hPrinter;
    REDATA *prd = GlobalLock((HGLOBAL)hPort);

    if (prd == (REDATA *)NULL) {
	SetLastError(ERROR_INVALID_HANDLE);
	return FALSE;
    }

    *pcbWritten = 0;

    if (!prd->started) {
	if (prd->debug) {
	    wsprintf(buf, 
	      TEXT("REDMON WritePort: called outside Start/EndDocPort.  Returning FALSE\r\n"));
	    MessageBox(NULL, buf, MONITORNAME, MB_OK);
	}
	GlobalUnlock((HGLOBAL)hPort);
	return FALSE;
    }

    /* Make sure process is still running */
    if (!prd->error && (prd->piProcInfo.hProcess == INVALID_HANDLE_VALUE))
	prd->error = TRUE;
    if (!prd->error 
	&& GetExitCodeProcess(prd->piProcInfo.hProcess, &exit_status)
	&& (exit_status != STILL_ACTIVE))
	prd->error = TRUE;
    if (prd->hChildStdinWrDup == INVALID_HANDLE_VALUE)
	prd->error = TRUE;

    if (prd->error) {
	/* The process is no longer running, probably due to an error. */
	/* If we return an error from WritePort, the spooler crashes.
	 * To avoid this mess, don't return an error from WritePort.
	 * Instead carry on as if the error didn't occur.
	 * The only evidence of an error will be the log file.
	 */
	*pcbWritten = cbBuf;	/* say we wrote it all */
	if (prd->debug && (prd->hLogFile != INVALID_HANDLE_VALUE)) {
	    DWORD cbWritten;
	    wsprintf(buf, 
	      TEXT("\r\nREDMON WritePort: Process not running. \
Returning TRUE.\r\n    Ignoring %d bytes\r\n"), cbBuf);
	    write_string_to_log(prd, buf);
	}

	/* Cancel the print job */
	if (OpenPrinter(prd->pPrinterName, &hPrinter, NULL)) {
	    DWORD dwNeeded = 0;
	    HGLOBAL hglobal = GlobalAlloc(GPTR, (DWORD)4096);
	    JOB_INFO_1 *pjob = (JOB_INFO_1 *)GlobalLock(hglobal);
	    if ((pjob != (JOB_INFO_1 *)NULL) && 
	         GetJob(hPrinter, prd->JobId, 1, (LPBYTE)pjob, 
		    4096, &dwNeeded)) {
		pjob->Status = JOB_STATUS_ERROR;
		SetJob(hPrinter, prd->JobId, 1, (LPBYTE)pjob, 
		    JOB_CONTROL_CANCEL);
		if (prd->debug && (prd->hLogFile != INVALID_HANDLE_VALUE)) {
		    DWORD cbWritten;
		    wsprintf(buf, 
	      TEXT("\r\nREDMON WritePort: Cancelling print job method 1\r\n"));
		    write_string_to_log(prd, buf);
		}
		GlobalUnlock(hglobal);
		GlobalFree(hglobal);
	    }
	    else {
		SetJob(hPrinter, prd->JobId, 0, NULL, JOB_CONTROL_CANCEL);
		if (prd->debug && (prd->hLogFile != INVALID_HANDLE_VALUE)) {
		    DWORD cbWritten;
		    wsprintf(buf, 
	      TEXT("\r\nREDMON WritePort: Cancelling print job method 2\r\n"));
		    write_string_to_log(prd, buf);
		}
	    }
	    ClosePrinter(hPrinter);
	}

	GlobalUnlock((HGLOBAL)hPort);
	return TRUE;	/* say we wrote it all */
    }

    if (prd->debug && (prd->hLogFile != INVALID_HANDLE_VALUE)) {
	DWORD cbWritten;
	wsprintf(buf, 
	 TEXT("\r\nREDMON WritePort: about to write %d bytes to port.\r\n"),
	 cbBuf);
	write_string_to_log(prd, buf);
    }

    /* write to pipe */
    flag = WriteFile(prd->hChildStdinWrDup, pBuffer, cbBuf, pcbWritten, NULL);

    if (prd->debug && (prd->hLogFile != INVALID_HANDLE_VALUE)) {
	DWORD cbWritten;
	request_mutex(prd);
        WriteFile(prd->hLogFile, pBuffer, cbBuf, &cbWritten, NULL);
	FlushFileBuffers(prd->hLogFile);
	release_mutex(prd);
	wsprintf(buf, 
	  TEXT("\r\nREDMON WritePort: returning %d, count=%d written=%d\r\n"), 
	  flag, cbBuf, *pcbWritten);
	write_string_to_log(prd, buf);
    }

    flush_stdout(prd);

    GlobalUnlock((HGLOBAL)hPort);

    return flag;
}

/* ReadPort can be called within a Start/EndDocPort pair,
 * and also outside this pair.
 */
BOOL WINAPI rReadPort(HANDLE hPort, LPBYTE pBuffer, 
        DWORD  cbBuffer, LPDWORD pcbRead)
{
    REDATA *prd = GlobalLock((HGLOBAL)hPort);
    DWORD exit_status;

    /* we don't support reading */
    *pcbRead = 0;

    if (prd == (REDATA *)NULL) {
	SetLastError(ERROR_INVALID_HANDLE);
	return FALSE;
    }

    Sleep(1000);	/* Pause a little */

    if (prd->debug) {
	TCHAR buf[MAXSTR];
	wsprintf(buf, TEXT("REDMON ReadPort: returning TRUE. Process %s\r\n"),
	   prd->error ? TEXT("has an ERROR.") : TEXT("is OK."));
	write_string_to_log(prd, buf);
	wsprintf(buf, TEXT("REDMON ReadPort: You must disable bi-directional printer support for this printer\r\n"));
	write_string_to_log(prd, buf);
    }

    GlobalUnlock((HGLOBAL)hPort);

    return TRUE;
}

BOOL WINAPI rEndDocPort(HANDLE hPort)
{
    TCHAR buf[MAXSTR];
    BOOL flag;
    DWORD bytes_available;
    DWORD dwRead, dwWritten;
    REDATA *prd = GlobalLock((HGLOBAL)hPort);
    DWORD exit_status;
    HANDLE hPrinter;
    int i;

    if (prd == (REDATA *)NULL) {
	SetLastError(ERROR_INVALID_HANDLE);
	return FALSE;
    }

    /* wait until watch thread ends */
    prd->watch = FALSE;
    while (GetExitCodeThread(prd->hthread, &exit_status)
	 && (exit_status == STILL_ACTIVE)) {
	    if (prd->debug) {
		wsprintf(buf, 
	        TEXT("REDMON EndDocPort: waiting for WatchThread to end\r\n")); 
		write_string_to_log(prd, buf);
	    }
	    Sleep(1000);
	}
    CloseHandle(prd->hthread);

    /* Close stdin to signal EOF */
    if (prd->hChildStdinWrDup != INVALID_HANDLE_VALUE)
	CloseHandle(prd->hChildStdinWrDup);

    /* wait here for up to 'delay' seconds until process ends */
    /* so that process has time to write stdout/err */
    exit_status = 0;
    for (i=0; i<prd->delay; i++) {
	if (prd->piProcInfo.hProcess == INVALID_HANDLE_VALUE)
	    break;
	if (GetExitCodeProcess(prd->piProcInfo.hProcess, &exit_status)) {
	    if (exit_status != STILL_ACTIVE)
	        break;
	}
	else {
	    /* process doesn't exist */
	    break;
	}
        Sleep(1000);
    }

    if (prd->debug) {
	wsprintf(buf, 
	    TEXT("REDMON EndDocPort: process %s after %d second%s\r\n"), 
	    (exit_status == STILL_ACTIVE) ? 
		TEXT("still running") : TEXT("finished"), 
	    i, (i != 1) ? TEXT("s") : TEXT(""));
	write_string_to_log(prd, buf);
    }

    /* Close the other end of the stdout & stderr pipes to flush them */
    CloseHandle(prd->hChildStdoutWr);
    CloseHandle(prd->hChildStderrWr);

    /* copy anything on stdout/err to log file */
    flush_stdout(prd);

    /* Close the read end of the stdio pipes */
    CloseHandle(prd->hChildStderrRd);
    CloseHandle(prd->hChildStdoutRd);
    CloseHandle(prd->hChildStdinRd);

    /* NT documentation says *we* should cancel the print job. */
    /* 95 documentation says nothing about this. */
    if (OpenPrinter(prd->pPrinterName, &hPrinter, NULL)) {
	SetJob(hPrinter, prd->JobId, 0, NULL, JOB_CONTROL_CANCEL);
	ClosePrinter(hPrinter);
    }

#ifdef UNUSED
    /* elsewhere, the NT documentation says we should use */
    /* JOB_CONTROL_SENT_TO_PRINTER */
    if (OpenPrinter(prd->pPrinterName, &hPrinter, NULL)) {
	SetJob(hPrinter, prd->JobId, 0, NULL, JOB_CONTROL_SENT_TO_PRINTER);
	ClosePrinter(hPrinter);
    }
#endif

    if (prd->hLogFile != INVALID_HANDLE_VALUE)
	CloseHandle(prd->hLogFile);

    if ((prd->hmutex != NULL) && (prd->hmutex != INVALID_HANDLE_VALUE))
	CloseHandle(prd->hmutex);

    reset_redata(prd);

    GlobalUnlock((HGLOBAL)hPort);

    return TRUE;
}


BOOL WINAPI rClosePort(HANDLE  hPort)
{
    TCHAR buf[MAXSTR];

    /* assume files were all closed in rEndDocPort() */

    if (hPort)
	GlobalFree((HGLOBAL)hPort);

    return TRUE;
}


BOOL WINAPI rSetPortTimeOuts(HANDLE  hPort, LPCOMMTIMEOUTS lpCTO, DWORD reserved)
{
    TCHAR buf[MAXSTR];
    /* Do nothing */

#ifdef DEBUG
    {
	TCHAR buf[MAXSTR];
	REDATA *prd = GlobalLock((HGLOBAL)hPort);
	if (prd == (REDATA *)NULL) {
	    SetLastError(ERROR_INVALID_HANDLE);
	    return FALSE;
	}
	if (prd->debug) {
	    wsprintf(buf, TEXT("REDMON SetPortTimeOuts: returning TRUE\r\n\
	    values = %d %d %d %d %d\r\n"),
		lpCTO->ReadIntervalTimeout,
		lpCTO->ReadTotalTimeoutMultiplier,
		lpCTO->ReadTotalTimeoutConstant,
		lpCTO->WriteTotalTimeoutMultiplier,
		lpCTO->WriteTotalTimeoutConstant);
	    write_string_to_log(prd, buf);
	}
	GlobalUnlock((HGLOBAL)hPort);
    }
#endif

    return TRUE;
}
 

/* Exported functions */


#ifdef UNICODE
#ifdef NT35
/* Windows NT 3.51 */
BOOL WINAPI _export
InitializeMonitor(LPWSTR pRegisterRoot)
{
    TCHAR buf[MAXSTR];
    LPTSTR p;
    if (lstrlen(pRegisterRoot) + 1 > sizeof(rekey) / sizeof(TCHAR)) 
	return FALSE;

    lstrcpy(rekey, pRegisterRoot);

    /* get help file name */
    GetModuleFileName(hdll, helpfile, sizeof(helpfile)/sizeof(TCHAR));
    p = helpfile + lstrlen(helpfile) - 1;
    while (p >= helpfile) {
	if (*p == '\\') {
	    p++;
	    break;
        }
	p--;
    }
    LoadString(hdll, IDS_HELPFILE, p, 
	sizeof(helpfile)/sizeof(TCHAR) - (int)(p-helpfile));

    return TRUE;
}

#else
/* Windows NT4 version */
MONITOREX mex;

LPMONITOREX WINAPI _export
InitializePrintMonitor(LPWSTR pRegisterRoot)
{
    TCHAR buf[MAXSTR];
    LPTSTR p;
#ifdef UNUSED
    wsprintf(buf, TEXT("InitializePrintMonitor: registrykey=%s"), 
		pRegisterRoot);
    MessageBox(NULL, buf, MONITORNAME, MB_OK);
#endif

    if (lstrlen(pRegisterRoot) + 1 > sizeof(rekey) / sizeof(TCHAR)) 
	return FALSE;

    lstrcpy(rekey, pRegisterRoot);

    mex.dwMonitorSize = sizeof(MONITOR);
    mex.Monitor.pfnEnumPorts = rEnumPorts;
    mex.Monitor.pfnOpenPort = rOpenPort;
    mex.Monitor.pfnOpenPortEx = NULL;
    mex.Monitor.pfnStartDocPort = rStartDocPort;
    mex.Monitor.pfnWritePort = rWritePort;
    mex.Monitor.pfnReadPort = rReadPort;
    mex.Monitor.pfnEndDocPort = rEndDocPort;
    mex.Monitor.pfnClosePort = rClosePort;
    mex.Monitor.pfnAddPort = rAddPort;
    mex.Monitor.pfnAddPortEx = rAddPortEx;
    mex.Monitor.pfnConfigurePort = rConfigurePort;
    mex.Monitor.pfnDeletePort = rDeletePort;
    mex.Monitor.pfnGetPrinterDataFromPort = NULL;
    mex.Monitor.pfnSetPortTimeOuts = rSetPortTimeOuts;

    /* get help file name */
    GetModuleFileName(hdll, helpfile, sizeof(helpfile)/sizeof(TCHAR));
    p = helpfile + lstrlen(helpfile) - 1;
    while (p >= helpfile) {
	if (*p == '\\') {
	    p++;
	    break;
        }
	p--;
    }
    LoadString(hdll, IDS_HELPFILE, p, 
	sizeof(helpfile)/sizeof(TCHAR) - (int)(p-helpfile));

    return &mex;
}
#endif	/* !NT35 */
#else
/* Windows 95 version */
BOOL WINAPI 
InitializeMonitorEx(LPTSTR pRegisterRoot, LPMONITOR pMonitor)
{
    TCHAR buf[MAXSTR];
    LPTSTR p;
#ifdef UNUSED
    wsprintf(buf, TEXT("InitializeMonitorEx: registrykey=%s pmonitor=%d"), 
		pRegisterRoot, pMonitor);
    MessageBox(NULL, buf, MONITORNAME, MB_OK);
#endif

    if (lstrlen(pRegisterRoot) + 1 > sizeof(rekey) / sizeof(TCHAR)) 
	return FALSE;

    lstrcpy(rekey, pRegisterRoot);

    pMonitor->pfnEnumPorts = rEnumPorts;
    pMonitor->pfnOpenPort = rOpenPort;
    pMonitor->pfnOpenPortEx = NULL;
    pMonitor->pfnStartDocPort = rStartDocPort;
    pMonitor->pfnWritePort = rWritePort;
    pMonitor->pfnReadPort = rReadPort;
    pMonitor->pfnEndDocPort = rEndDocPort;
    pMonitor->pfnClosePort = rClosePort;
    pMonitor->pfnAddPort = rAddPort;
    pMonitor->pfnConfigurePort = rConfigurePort;
    pMonitor->pfnDeletePort = rDeletePort;
    pMonitor->pfnGetPrinterDataFromPort = NULL;
    pMonitor->pfnSetPortTimeOuts = rSetPortTimeOuts;

    /* get help file name */
    GetModuleFileName(hdll, helpfile, sizeof(helpfile)/sizeof(TCHAR));
    p = helpfile + lstrlen(helpfile) - 1;
    while (p >= helpfile) {
	if (*p == '\\') {
	    p++;
	    break;
        }
	p--;
    }
    LoadString(hdll, IDS_HELPFILE, p, 
	sizeof(helpfile)/sizeof(TCHAR) - (int)(p-helpfile));

    return TRUE;

}
#endif


/* DLL entry point for Borland C++ */
#ifdef __BORLANDC__
#pragma argsused
#endif
BOOL WINAPI _export
DllEntryPoint(HINSTANCE hInst, DWORD fdwReason, LPVOID lpReserved)
{
    hdll = hInst;
    return TRUE;
}

#ifdef __BORLANDC__
#pragma argsused
#endif
/* DLL entry point for Microsoft Visual C++ */
BOOL WINAPI
DllMain(HINSTANCE hInst, DWORD fdwReason, LPVOID lpReserved)
{
    return DllEntryPoint(hInst, fdwReason, lpReserved);
}

BOOL
browse(HWND hwnd, UINT control, UINT filter, BOOL save)
{
OPENFILENAME ofn;
TCHAR szFilename[MAXSTR];	/* filename for OFN */
TCHAR szFilter[MAXSTR];
TCHAR szDir[MAXSTR];
TCHAR cReplace;
int i;

BOOL flag;
	/* setup OPENFILENAME struct */
	FillMemory((PVOID)&ofn, sizeof(ofn), 0);
	ofn.lStructSize = sizeof(OPENFILENAME);
	ofn.hwndOwner = hwnd;
	ofn.lpstrFile = szFilename;
	ofn.nMaxFile = sizeof(szFilename);
	ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

	GetDlgItemText(hwnd, control, szFilename, 
		sizeof(szFilename)/sizeof(TCHAR) - 1);

	if ((lstrlen(szFilename) == 2) && (szFilename[1] == ':'))
	    lstrcat(szFilename, BACKSLASH);

	if (lstrlen(szFilename)) {
	    lstrcpy(szDir, szFilename);
	    for (i=lstrlen(szDir)-1; i; i--) {
		if (szDir[i] == '\\') {
		    lstrcpy(szFilename, szDir+i+1);
		    szDir[i+1] = '\0';
		    ofn.lpstrInitialDir = szDir;
		    break;
		}
	    }
	}

	if (LoadString(hdll, filter, szFilter, 
	    sizeof(szFilter)/sizeof(TCHAR) -1 )) {
	    cReplace = szFilter[lstrlen(szFilter)-1];
	    for (i=0; szFilter[i] != '\0'; i++)
	        if (szFilter[i] == cReplace)
		    szFilter[i] = '\0';
	    ofn.lpstrFilter = szFilter;
	    ofn.nFilterIndex = 0;
	}

	/* call the common dialog box */
	if (save)
	    flag = GetSaveFileName(&ofn);
	else
	    flag = GetOpenFileName(&ofn);

	if (flag)
	    SetDlgItemText(hwnd, control, szFilename);
	return flag;
}





/* Add Port dialog box */
BOOL CALLBACK 
AddDlgProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch(message) {
        case WM_INITDIALOG:
	    {
	    TCHAR buf[MAXSTR];
	    ADDPARAM *ap;

	    /* save pointer to port name */
	    SetWindowLong(hDlg, GWL_USERDATA, (LONG)lParam);
	    
	    ap = (ADDPARAM *)lParam;

	    SetDlgItemText(hDlg, IDC_PORTNAME, ap->portname);
	    SetDlgItemText(hDlg, IDC_PORTDESC, ap->portdesc);
	    SetForegroundWindow(hDlg);
	    }
            return( TRUE);
        case WM_COMMAND:
            switch(LOWORD(wParam)) {
	      case IDC_HELPBUTTON:
		  {TCHAR buf[MAXSTR];
		  LoadString(hdll, IDS_HELPADD, buf, 
			sizeof(buf)/sizeof(TCHAR) - 1);
		  WinHelp(hDlg, helpfile, HELP_KEY, (DWORD)buf);
		  }
                  return(TRUE);
              case IDC_PORTNAME:
              case IDC_PORTDESC:
                  return(TRUE);
	      case IDOK:
		{ ADDPARAM *ap = (ADDPARAM *)GetWindowLong(hDlg, GWL_USERDATA);
		  GetDlgItemText(hDlg, IDC_PORTNAME, ap->portname, 
			MAXSHORTSTR-1);
		  GetDlgItemText(hDlg, IDC_PORTDESC, ap->portdesc, 
			MAXSHORTSTR-1);
		}
		EndDialog(hDlg, TRUE);
		return(TRUE);
	      case IDCANCEL:
                EndDialog(hDlg, FALSE);
                return(TRUE);
              default:
                return(FALSE);
            }
        default:
            return(FALSE);
    }
}

/* Log file dialog box */
BOOL CALLBACK 
LogfileDlgProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch(message) {
        case WM_INITDIALOG:
	    {
	    HKEY hkey;
	    TCHAR buf[MAXSTR];
	    TCHAR str[MAXSTR];
	    DWORD cbData, keytype;
	    DWORD dwtemp;

	    /* save pointer to port name */
	    SetWindowLong(hDlg, GWL_USERDATA, (LONG)lParam);

	    lstrcpy(buf, rekey);
	    lstrcat(buf, BACKSLASH);
	    lstrcat(buf, PORTSNAME);
	    lstrcat(buf, BACKSLASH);
	    lstrcat(buf, (LPTSTR)lParam);
	    if ( RegOpenKeyEx(HKEY_LOCAL_MACHINE, buf, 0, 
		    KEY_ALL_ACCESS, &hkey) != ERROR_SUCCESS ) {
		/* we can't do anything */
		LoadString(hdll, IDS_CONFIGUNKNOWN, buf, 
		    sizeof(buf)/sizeof(TCHAR) - 1);
		SetWindowText(hDlg, buf);
		return TRUE;
	    }

	    keytype = REG_SZ;
	    buf[0] = '\0';
	    cbData = sizeof(buf);
	    RegQueryValueEx(hkey, LOGNAMEKEY, 0, &keytype, 
		(LPBYTE)buf, &cbData);
	    SetDlgItemText(hDlg, IDC_LOGNAME, buf);

	    dwtemp = DEFAULT_DELAY;
	    keytype = REG_DWORD;
	    cbData = sizeof(dwtemp);
	    RegQueryValueEx(hkey, LOGDELAYKEY, 0, &keytype, 
		(LPBYTE)&dwtemp, &cbData);
	    SetDlgItemInt(hDlg, IDC_LOGDELAY, dwtemp, FALSE);

#ifdef BETA
	    dwtemp = 1;	/* beta versions default to debug enabled */
#else
	    dwtemp = 0;
#endif
	    keytype = REG_DWORD;
	    cbData = sizeof(dwtemp);
	    RegQueryValueEx(hkey, LOGDEBUGKEY, 0, &keytype, 
		(LPBYTE)&dwtemp, &cbData);
	    SendDlgItemMessage(hDlg, IDC_LOGDEBUG, BM_SETCHECK, dwtemp, 0);

	    dwtemp = 0;
	    keytype = REG_DWORD;
	    cbData = sizeof(dwtemp);
	    RegQueryValueEx(hkey, LOGUSEKEY, 0, &keytype, 
		(LPBYTE)&dwtemp, &cbData);
	    
	    SendDlgItemMessage(hDlg, IDC_LOGUSE, BM_SETCHECK, dwtemp, 0);
	    EnableWindow(GetDlgItem(hDlg, IDC_LOGNAMEPROMPT), dwtemp);
	    EnableWindow(GetDlgItem(hDlg, IDC_LOGNAME), dwtemp);
	    EnableWindow(GetDlgItem(hDlg, IDC_LOGDELAYPROMPT1), dwtemp);
	    EnableWindow(GetDlgItem(hDlg, IDC_LOGDELAYPROMPT2), dwtemp);
	    EnableWindow(GetDlgItem(hDlg, IDC_LOGDELAY), dwtemp);
	    EnableWindow(GetDlgItem(hDlg, IDC_LOGDEBUG), dwtemp);
	    EnableWindow(GetDlgItem(hDlg, IDC_BROWSE), dwtemp);

	    LoadString(hdll, IDS_CONFIGLOGFILE, str, 
		sizeof(str)/sizeof(TCHAR) - 1);
	    wsprintf(buf, str, (LPSTR)lParam);
	    SetWindowText(hDlg, buf);
	    }
            return( TRUE);
        case WM_COMMAND:
            switch(LOWORD(wParam)) {
	      case IDC_HELPBUTTON:
		  {TCHAR buf[MAXSTR];
		  LoadString(hdll, IDS_HELPLOG, buf, 
			sizeof(buf)/sizeof(TCHAR) - 1);
		  WinHelp(hDlg, helpfile, HELP_KEY, (DWORD)buf);
		  }
                  return(TRUE);
	      case IDC_BROWSE:
		  browse(hDlg, IDC_LOGNAME, IDS_FILTER_TXT, TRUE);
                  return(TRUE);
	      case IDC_LOGUSE:
		  if (HIWORD(wParam) == BN_CLICKED) {
		    BOOL enabled = (BOOL)SendDlgItemMessage(hDlg, 
				IDC_LOGUSE, BM_GETCHECK, 0, 0);
		    enabled = !enabled;
		    SendDlgItemMessage(hDlg, IDC_LOGUSE, 
			BM_SETCHECK, enabled, 0);
		    EnableWindow(GetDlgItem(hDlg, IDC_LOGNAMEPROMPT), enabled);
		    EnableWindow(GetDlgItem(hDlg, IDC_LOGNAME), enabled);
		    EnableWindow(GetDlgItem(hDlg, IDC_LOGDELAYPROMPT1),enabled);
		    EnableWindow(GetDlgItem(hDlg, IDC_LOGDELAYPROMPT2),enabled);
		    EnableWindow(GetDlgItem(hDlg, IDC_LOGDELAY), enabled);
		    EnableWindow(GetDlgItem(hDlg, IDC_LOGDEBUG), enabled);
		    EnableWindow(GetDlgItem(hDlg, IDC_BROWSE), enabled);
		   }
		   return TRUE;
	      case IDC_LOGFILE:
                    return(TRUE);
/* should we fall through to IDOK */
	      case IDOK:
		{
		HKEY hkey;
		TCHAR buf[MAXSTR];
		BOOL enabled;
		DWORD delay;
		BOOL success;
		lstrcpy(buf, rekey);
		lstrcat(buf, BACKSLASH);
		lstrcat(buf, PORTSNAME);
		lstrcat(buf, BACKSLASH);
		lstrcat(buf, (LPTSTR)GetWindowLong(hDlg, GWL_USERDATA));
		if ( RegOpenKeyEx(HKEY_LOCAL_MACHINE, buf, 0, 
			KEY_ALL_ACCESS, &hkey) == ERROR_SUCCESS ) {
		    enabled = (BOOL)SendDlgItemMessage(hDlg, IDC_LOGUSE,
			BM_GETCHECK, 0, 0);
		    delay = (DWORD)enabled;
		    RegSetValueEx(hkey, LOGUSEKEY, 0, REG_DWORD,
			(CONST BYTE *)&delay, sizeof(DWORD));
		    if (enabled) {
			/* get log filename */
		        GetDlgItemText(hDlg, IDC_LOGNAME, buf, 
			    sizeof(buf)/sizeof(TCHAR) - 1);
		        RegSetValueEx(hkey, LOGNAMEKEY, 0, REG_SZ,
			    (CONST BYTE *)buf, (lstrlen(buf)+1)*sizeof(TCHAR));
			/* get shutdown delay */
			delay = (DWORD)GetDlgItemInt(hDlg, IDC_LOGDELAY, 
				&success, FALSE);
			if (success)
		            RegSetValueEx(hkey, LOGDELAYKEY, 0, REG_DWORD,
			        (CONST BYTE *)&delay, sizeof(DWORD));
			/* get debug flag */
		        delay = SendDlgItemMessage(hDlg, IDC_LOGDEBUG,
			    BM_GETCHECK, 0, 0);
			RegSetValueEx(hkey, LOGDEBUGKEY, 0, REG_DWORD,
			    (CONST BYTE *)&delay, sizeof(DWORD));
		    }
		}
		}
		EndDialog(hDlg, TRUE);
		return(TRUE);
	      case IDCANCEL:
                EndDialog(hDlg, FALSE);
                return(TRUE);
              default:
                return(FALSE);
            }
        default:
            return(FALSE);
    }
}


/* Configure Port dialog box */
BOOL CALLBACK 
ConfigDlgProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch(message) {
        case WM_INITDIALOG:
	    {
	    HKEY hkey;
	    TCHAR buf[MAXSTR];
	    TCHAR str[MAXSTR];
	    DWORD cbData, keytype;
	    DWORD show;
	    int i;

	    /* save pointer to port name */
	    SetWindowLong(hDlg, GWL_USERDATA, (LONG)lParam);

	    lstrcpy(buf, rekey);
	    lstrcat(buf, BACKSLASH);
	    lstrcat(buf, PORTSNAME);
	    lstrcat(buf, BACKSLASH);
	    lstrcat(buf, (LPTSTR)lParam);
	    if ( RegOpenKeyEx(HKEY_LOCAL_MACHINE, buf, 0, 
		    KEY_ALL_ACCESS, &hkey) != ERROR_SUCCESS ) {
		/* we can't do anything */
		LoadString(hdll, IDS_CONFIGUNKNOWN, buf, 
		    sizeof(buf)/sizeof(TCHAR) - 1);
		SetWindowText(hDlg, buf);
		return TRUE;
	    }

	    keytype = REG_SZ;
	    buf[0] = '\0';
	    cbData = sizeof(buf);
	    RegQueryValueEx(hkey, COMMANDKEY, 0, &keytype, 
		(LPBYTE)buf, &cbData);
	    SetDlgItemText(hDlg, IDC_COMMAND, buf);
	    buf[0] = '\0';
	    cbData = sizeof(buf);
	    RegQueryValueEx(hkey, ARGKEY, 0, &keytype, 
		(LPBYTE)buf, &cbData);
	    SetDlgItemText(hDlg, IDC_ARGS, buf);

	    cbData = sizeof(DWORD);
	    keytype = REG_DWORD;
	    show = 0;
	    RegQueryValueEx(hkey, SHOWKEY, 0, &keytype, 
		(LPBYTE)&show, &cbData);
	    SendDlgItemMessage(hDlg, IDC_SHOW, CB_RESETCONTENT,
		(WPARAM)0, (LPARAM)0);
	    for (i=0; i<=SHOW_HIDE; i++) {
		LoadString(hdll, IDS_SHOWBASE + i, buf, 
		    sizeof(buf)/sizeof(TCHAR) - 1);
		SendDlgItemMessage(hDlg, IDC_SHOW, CB_ADDSTRING,
		    (WPARAM)0, (LPARAM)buf);
	    }
	    SendDlgItemMessage(hDlg, IDC_SHOW, CB_SETCURSEL,
		(WPARAM)show, (LPARAM)0);

	    LoadString(hdll, IDS_CONFIGPROP, str, 
		sizeof(str)/sizeof(TCHAR) - 1);
	    wsprintf(buf, str, (LPSTR)lParam);
	    SetWindowText(hDlg, buf);
	    SetForegroundWindow(hDlg);
	    }
            return( TRUE);
        case WM_COMMAND:
            switch(LOWORD(wParam)) {
	      case IDC_HELPBUTTON:
		  {TCHAR buf[MAXSTR];
		  LoadString(hdll, IDS_HELPCONFIG, buf, 
			sizeof(buf)/sizeof(TCHAR) - 1);
		  WinHelp(hDlg, helpfile, HELP_KEY, (DWORD)buf);
		  }
                  return(TRUE);
	      case IDC_BROWSE:
		  browse(hDlg, IDC_COMMAND, IDS_FILTER_EXE, FALSE);
                  return(TRUE);
	      case IDC_LOGFILE:
		  DialogBoxParam(hdll, MAKEINTRESOURCE(IDD_CONFIGLOG), hDlg, 
			LogfileDlgProc, 
			(LPARAM)GetWindowLong(hDlg, GWL_USERDATA));
                  return(TRUE);
              case IDC_COMMAND:
              case IDC_ARGS:
                    return(TRUE);
/* should we fall through to IDOK */
	      case IDOK:
		{
		HKEY hkey;
		TCHAR buf[MAXSTR];
		DWORD show;
		lstrcpy(buf, rekey);
		lstrcat(buf, BACKSLASH);
		lstrcat(buf, PORTSNAME);
		lstrcat(buf, BACKSLASH);
		lstrcat(buf, (LPTSTR)GetWindowLong(hDlg, GWL_USERDATA));
		if ( RegOpenKeyEx(HKEY_LOCAL_MACHINE, buf, 0, 
			KEY_ALL_ACCESS, &hkey) == ERROR_SUCCESS ) {
		    GetDlgItemText(hDlg, IDC_COMMAND, buf, 
			sizeof(buf)/sizeof(TCHAR) - 1);
		    RegSetValueEx(hkey, COMMANDKEY, 0, REG_SZ,
			(CONST BYTE *)buf, (lstrlen(buf)+1)*sizeof(TCHAR));
		    GetDlgItemText(hDlg, IDC_ARGS, buf, 
			sizeof(buf)/sizeof(TCHAR) - 1);
		    RegSetValueEx(hkey, ARGKEY, 0, REG_SZ,
			(CONST BYTE *)buf, (lstrlen(buf)+1)*sizeof(TCHAR));
		    show = SendDlgItemMessage(hDlg, IDC_SHOW, CB_GETCURSEL,
			(WPARAM)0, (LPARAM)0);
		    RegSetValueEx(hkey, SHOWKEY, 0, REG_DWORD,
			(CONST BYTE *)&show, sizeof(DWORD));
		}
		}
		EndDialog(hDlg, TRUE);
		return(TRUE);
	      case IDCANCEL:
                EndDialog(hDlg, FALSE);
                return(TRUE);
              default:
                return(FALSE);
            }
        default:
            return(FALSE);
    }
}

/* start_redirect() was originally based on an example in the Win32 SDK
 * which used GetStdHandle() and SetStdHandle() to redirect stdio.
 * The example works under Windows 95, but not under NT.
 * For NT, we need to use 
 *  siStartInfo.dwFlags = STARTF_USESTDHANDLES;
 *  siStartInfo.hStdInput = prd->hChildStdinRd;
 *  siStartInfo.hStdOutput = prd->hChildStdoutWr;
 *  siStartInfo.hStdError = prd->hChildStderrWr;
 * The SDK example does NOT include these.  Most strange for an 
 * example that was written before Windows 95 existed!
 * STARTF_USESTDHANDLES also works for Windows 95, so the original 
 * code is commented out with #ifdef SAVESTD / #endif
 */

/* Start child program with redirected standard input and output */
BOOL start_redirect(REDATA * prd)
{
    SECURITY_ATTRIBUTES saAttr;
    STARTUPINFO siStartInfo;
    HANDLE hChildStdinWr;
    BOOL fSuccess;

    /* Set the bInheritHandle flag so pipe handles are inherited. */

    saAttr.nLength = sizeof(SECURITY_ATTRIBUTES);
    saAttr.bInheritHandle = TRUE;
    saAttr.lpSecurityDescriptor = NULL;

#ifdef SAVESTD
    /* Save the current standard handles . */
    prd->hSaveStdout = GetStdHandle(STD_OUTPUT_HANDLE);
    prd->hSaveStderr = GetStdHandle(STD_ERROR_HANDLE);
    prd->hSaveStdin = GetStdHandle(STD_INPUT_HANDLE);
#endif

    /* 	Create anonymous inheritable pipes for STDIN, STDOUT and STDERR
     *  for child
     */

    if (!CreatePipe(&prd->hChildStdoutRd, &prd->hChildStdoutWr, &saAttr, 0))
	return FALSE;
    if (!CreatePipe(&prd->hChildStderrRd, &prd->hChildStderrWr, &saAttr, 0))
	return FALSE;
    if (!CreatePipe(&prd->hChildStdinRd, &hChildStdinWr, &saAttr, 0))
	return FALSE;

    /* For STDIN, create a noninheritable duplicate of write handle,
     * then close the inheritable write handle.
     */

    fSuccess = DuplicateHandle(GetCurrentProcess(), hChildStdinWr,
        GetCurrentProcess(), &prd->hChildStdinWrDup, 0,
        FALSE,       /* not inherited */
        DUPLICATE_SAME_ACCESS);
    CloseHandle(hChildStdinWr);

    if (!fSuccess)
	return FALSE;

#ifdef SAVESTD
    if (!SetStdHandle(STD_OUTPUT_HANDLE, prd->hChildStdoutWr))
	return FALSE;
    if (!SetStdHandle(STD_ERROR_HANDLE, prd->hChildStderrWr))
	return FALSE;
    if (!SetStdHandle(STD_INPUT_HANDLE, prd->hChildStdinRd))
	return FALSE;
#endif

    /* Now create the child process. */

    /* Set up members of STARTUPINFO structure. */

    siStartInfo.cb = sizeof(STARTUPINFO);
    siStartInfo.lpReserved = NULL;
    siStartInfo.lpDesktop = NULL;
    siStartInfo.lpTitle = NULL;  /* use executable name as title */
    siStartInfo.dwX = siStartInfo.dwY = CW_USEDEFAULT;		/* ignored */
    siStartInfo.dwXSize = siStartInfo.dwYSize = CW_USEDEFAULT;	/* ignored */
    siStartInfo.dwXCountChars = 80;
    siStartInfo.dwYCountChars = 25;
    siStartInfo.dwFillAttribute = 0;			/* ignored */
    siStartInfo.dwFlags = STARTF_USESTDHANDLES;
    siStartInfo.wShowWindow = SW_SHOWNORMAL;		/* ignored */
    if (prd->show != SHOW_NORMAL)
        siStartInfo.dwFlags |= STARTF_USESHOWWINDOW;
    if (prd->show == SHOW_MIN)
        siStartInfo.wShowWindow = SW_SHOWMINNOACTIVE;	
    else if (prd->show == SHOW_HIDE)
        siStartInfo.wShowWindow = SW_HIDE;	
    siStartInfo.cbReserved2 = 0;
    siStartInfo.lpReserved2 = NULL;
    siStartInfo.hStdInput = prd->hChildStdinRd;
    siStartInfo.hStdOutput = prd->hChildStdoutWr;
    siStartInfo.hStdError = prd->hChildStderrWr;

    /* Create the child process. */

    if (!CreateProcess(NULL,
        prd->command,       /* command line                       */
        NULL,          /* process security attributes        */
        NULL,          /* primary thread security attributes */
        TRUE,          /* handles are inherited              */
        0,             /* creation flags                     */
        NULL,          /* use parent's environment           */
        NULL,          /* use parent's current directory     */
        &siStartInfo,  /* STARTUPINFO pointer                */
        &prd->piProcInfo))  /* receives PROCESS_INFORMATION       */
	  return FALSE;

    /* After process creation, restore the saved STDIN and STDOUT. */

#ifdef SAVESTD
    if (!SetStdHandle(STD_INPUT_HANDLE, prd->hSaveStdin))
	return FALSE;
    if (!SetStdHandle(STD_OUTPUT_HANDLE, prd->hSaveStdout))
	return FALSE;
    if (!SetStdHandle(STD_ERROR_HANDLE, prd->hSaveStderr))
	return FALSE;
#endif

    return TRUE;
}



#ifdef MAKEEXE
BYTE pPorts[4096];
DWORD cbBuf, cbNeeded, cReturned;
TCHAR mess[MAXSTR];
#define QUICKBROWN "The quick brown fox jumps over the lazy dog."
#define PSTEST "(c:/rjl/monitor/test.txt) (w) file\n dup (Hello, world\\n) writestring closefile\n(Hello, world\\n) print flush\n"

int PASCAL 
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpszCmdLine, int cmdShow)
{
    DWORD written;
    HANDLE hport;
    PORT_INFO_2 *pi2;
    int i;
#ifdef UNICODE
    LPMONITOREX pmon;
#else
    MONITOR mon;
#endif
    TCHAR monitorname[256];
    TCHAR mess[256];
    hdll = hInstance;
#ifdef UNICODE
    pmon = InitializePrintMonitor(TEXT("System\\CurrentControlSet\\control\\Print\\Monitors\\Redirected Port"));
#else
    InitializeMonitorEx(TEXT("System\\CurrentControlSet\\control\\Print\\Monitors\\Redirected Port"), &mon);
#endif

    LoadString(hdll, IDS_MONITORNAME, monitorname, 
	sizeof(monitorname)/sizeof(TCHAR)-1);

#ifdef UNUSED
    rAddPort(NULL, HWND_DESKTOP, monitorname);
    rAddPort(NULL, HWND_DESKTOP, monitorname);
#endif
    rAddPort(NULL, HWND_DESKTOP, monitorname);
    rConfigurePort(NULL, HWND_DESKTOP, TEXT("RPT1:"));

    cbBuf = 0;
    rEnumPorts(NULL, 2, pPorts, cbBuf, &cbNeeded, &cReturned);
    cbBuf = cbNeeded;
    rEnumPorts(NULL, 2, pPorts, cbBuf, &cbNeeded, &cReturned);
    pi2 = (PORT_INFO_2 *)pPorts;

    for (i=0; i<cReturned; i++) {
	wsprintf(mess, TEXT("\042%s\042 \042%s\042 \042%s\042 %d\n"), 
	   pi2[i].pPortName,
	   pi2[i].pMonitorName,
	   pi2[i].pDescription,
	   pi2[i].fPortType
	   );
        MessageBox(NULL, mess, TEXT("EXE for Redirect Monitor"), MB_OK);
    }
#ifdef UNUSED
#endif

    rOpenPort(TEXT("RPT1:"), &hport);
    rStartDocPort(hport, TEXT("Dummy printer"), 1, 1, NULL);
//    rWritePort(hport, (LPBYTE)&QUICKBROWN, strlen(QUICKBROWN), &written);
    rWritePort(hport, (LPBYTE)&PSTEST, strlen(PSTEST), &written);
    rWritePort(hport, (LPBYTE)&PSTEST, strlen(PSTEST), &written);
    rEndDocPort(hport);
    rClosePort(hport);

    rDeletePort(NULL, HWND_DESKTOP, TEXT("RPT2:"));

    return 0;
}
#endif

/* end of redmon.c */
