/**/
/*
/* HINDEX.C, 29-SEP-2003, Michael D. Duffy
/*
/* ABSTRACT:
/*
/*   This program will search an OpenVMS online HELP library file for a 
/*   specific word and report which help entries contain it.
/*
/* DEVELOPMENT ENVIRONMENT:
/*
/*   This program was developed using Compaq C V6.4-005 on OpenVMS VAX V7.3.
/*   It has also been built and tested on OpenVMS Alpha V7.3 using Compaq
/*   C V6.5-001-48BCD.  No source code modifications were required between
/*   The VAX and Alpha versions.
/*
/*   On 21-OCT-2004, this program was built successfully with no modifications
/*   on Itanium (HP c X7.1-056 on OpenVMS IA64 E8.2)
/*
/* TO BUILD:
/*
/*   Compile and link the program:
/*
/*     $ cc helpindex    
/*     $ link helpindex
/*
/*   Modify the HINDEX.CLD File.  Change the IMAGE line to specify the proper
/*   disk and directory for your system.
/*
/* TO RUN:
/*
/*   $ set command hindex  ! Or add to system-wide command tables
/*   $ hindex searchstring [optional-qualifiers]
/*
/*   Optional Qualifiers:
/*
/*     /LIBRARY=filename   Specifies that the program should search a help 
/*                         library other than the default of 
/*                         sys$help:helplib.hlb.
/*
/*     /WINDOW=(x,y)       Controls how many help text lines will be displayed
/*                         surrounding the line in which a match was found.
/*                         "x" specifies the number of preceeding lines,
/*                         while "y" specifies the number of subsequent lines.
/*                         If you omit /WINDOW, only the line(s) containing
/*                         the match will be shown.
/*
/*     /NOWINDOW           Specifies that no help text lines at all should
/*                         be displayed.  Only matching level headings will
/*                         be shown.
/*
/*     /DEBUG=number       Sets the internal debugging level.  Trace messages 
/*                         are enabled at various debugging levels.  This
/*                         feature is including for the convienience of
/*                         any programmers who wish to modify this program.
/*
/*     /INCLUDE=(module, module [...])
/*                         Defines a list of modules in the help library
/*                         that will be searched.  If used, only the
/*                         listed modules will be searched.  All others
/*                         will be skipped.  If both /INCLUDE and /EXCLUDE
/*                         are specified, /INCLUDE will take precedence.
/*                         Example:  /INCLUDE=(TYPE, WAIT)  This specifies
/*                         that only the TYPE and WAIT top-level help
/*                         topics will be searched.
/*
/*     /EXCLUDE=(module, module [...])
/*                         Specifies that all modules in the library
/*                         will be searched except those listed.  
/*                         If both /INCLUDE and /EXCLUDE are specified,
/*                         /INCLUDE will take precedence.  Example:
/*                         /EXCLUDE=(CC, HINTS)  This specifies that
/*                         the CC and HINTS top-level help topics will
/*                         be skipped and all others will be searched.
/*
/*     /EXTRA              Controls whether an "extra" heading level is
/*                         displayed.  Some help text records begin at
/*                         the first byte.  Of these, some are appropriate
/*                         to be treated as another sub-level, and some
/*                         are not.  This qualifier, if present, requests
/*                         that such records will be included in headings.
/*                         Futute versions of this program will be smart 
/*                         enough to tell the difference and take the 
/*                         appropriate action automatically.
/*
/*     /LISTMATCH=[EXACT,ABBREVIATED]
/*                         Controls whether values supplied with /INCLUDE
/*                         and /EXCLUDE must be spelled out fully
/*                         (/LISTMATCH=EXACT) or may be abbreviated
/*                         (/LISTMATCH=ABBREVIATED) in order to match a
/*                         module name.  The default is ABBREVIATED.
/*
/* PRIVILEGES REQUIRED:
/*
/*   This program requires no elevated privileges.
/*
/* KNOWN PROBLEMS AND RESTRICTIONS:
/*
/*   No problems are known at this time, but there is a restristion:
/*
/*   This program looks for keywords, not phrases.  Do not attempt to use
/*   a phrase enclosed in quotes, as in:
/*
/*        $ HINDEX "DCL Command"
/*
/*   The quotation marks will become part of the search string and fail
/*   to match occurances.  If you were to modify the source to strip
/*   the quotation marks, the program would still fail to locate the phrase
/*   if it were to span lines in the help library.  A future version may
/*   remove this restriction.
/*
/* CONTACTING THE AUTHOR:
/*
/*   Comments, suggestions, bug reports and questions about this software may 
/*   be directed to the author.  At the time of this writing, the author may 
/*   be contacted at duffy@process.com.  Also, Duffy regularly monitors the 
/*   comp.os.vms newsgroup.  
/*
/* COPYRIGHT NOTICE:
/*
/*   This software is COPYRIGHT  2003 MICHAEL D, DUFFY. ALL RIGHTS RESERVED. 
/*   Permission is granted for not-for-profit redistribution, provided
/*   all source and object code remain unchanged from the original distribution,
/*   and that all copyright notices remain intact.  You may modify your own 
/*   copy of the source code for your individual use, but you may not
/*   redistribute a modified version.
/*
/* DISCLAIMER:
/*
/*   This software is provided "AS IS".  The author, Michael Duffy, 
/*   the archive maintainer, Hunter Goatley, and Process Software make no 
/*   representations or warranties with repsect to the software and 
/*   specifically disclaim any implied warranties of merchantability or 
/*   fitness for any particular purpose.
/*
/* MODIFICATION HISTORY:
/*
/*   05-NOV-2003,	M. Duffy	V1.0
/*
/*	First release.
/*
/*   29-SEP-2003,	M. Duffy	X0.0
/*
/*	Original development version.
/*
/**/

typedef char string80[80];

#include stdio       /* printf(), etc... */
#include string      /* strlen(), etc... */
#include ssdef       /* VMS status return codes */
#include climsgdef   /* DCL status codes */
#include lbrdef      /* Librarian services (to manipulate help libraries */
#include descrip     /* String descriptors */
#include rmsdef      /* Record Management Services status codes */
#include stdlib      /* atoi(), etc... */
#include ctype       /* toupper, etc. */

/* Things relating to the help library files. */

char library[1024] = "sys$help:helplib.hlb";
char searchstring[1024] = "\0";
unsigned long int libstat;
unsigned long int library_index;  /* set by ini_control, read by lbr$xxx */
$DESCRIPTOR(library_filespec,""); /* don't worry about having too little string
                                     space.  Pointer field will be directed
                                     at an entirely different string.  This
                                     technique is used at multiple locations
                                     throughout this program. */
$DESCRIPTOR(library_record,"");   /* receives library data records.*/
$DESCRIPTOR(library_record_up,"");/* receives upcase library data records.*/
$DESCRIPTOR(outbufdes,"");        /* receives length of data record*/

/* Librarian routine definitions*/

int lbr$ini_control(); /* Initialize control structure. */
int lbr$open();        /* Open library file. */
int lbr$get_index();   /* Step through each key, call user-routine for each. */
int lbr$lookup_key();  /* call inpreparation for lbr$get_record. */
int lbr$get_record();  /* Retrieve data records from library */

globalvalue int lbr$_normal;
globalvalue int lbr$_typmismch;

/* Other VMS, VMSRTL, etc. routines */

int STR$UPCASE();
int LIB$FIND_FILE();

/* Output buffer stuff.  We store each line of help text in a round-robin */
/* array.  This allows us to support the /WINDOW qualifier, where we can  */
/* display some number of lines before and after the line containing a    */
/* match. */

#define OUT_BUF_SIZE 100
#define MAX_WINDOW 45     /* Must always be less than half OUT_BUF_SIZE */
struct {
  int needs_printing;
  int serial;            /* Tracks lines belonging to different help topics.*/
  char text[255];
  } outbuffer[OUT_BUF_SIZE];
int bufferslot_in = 0;
int bufferslot_out;
int serial_ref = 0;      /* Source for outbuffer.serial field */

/* Command-line processing routines and variables. */

int cli$present();     /* See if a command line element is present. */
int cli$get_value();   /* Get value associated with command line element. */
int clistat;

$DESCRIPTOR(cli_include_desc,"INCLUDE");
$DESCRIPTOR(cli_include_ret_desc," ");
short cli_include_ret_len;
char include_in[80];
string80 include_spec[100];
int include_seen[100];
int num_includes;

$DESCRIPTOR(cli_exclude_desc,"EXCLUDE");
$DESCRIPTOR(cli_exclude_ret_desc," ");
short cli_exclude_ret_len;
char exclude_in[80];
string80 exclude_spec[100];
int exclude_seen[100];
int num_excludes;

$DESCRIPTOR(cli_debug_desc,"DEBUG");
$DESCRIPTOR(cli_debug_ret_desc," ");
short cli_debug_ret_len;
char debuglevel_ascii[16];

$DESCRIPTOR(cli_listmatch_desc,"LISTMATCH");
$DESCRIPTOR(cli_listmatch_ret_desc," ");
short cli_listmatch_ret_len;
char listmatch_ascii[80];

$DESCRIPTOR(cli_p1_desc,"P1");
$DESCRIPTOR(cli_p1_ret_desc," ");
short cli_p1_ret_len;

$DESCRIPTOR(cli_library_desc,"LIBRARY");
$DESCRIPTOR(cli_library_ret_desc," ");
short cli_library_ret_len;

$DESCRIPTOR(cli_window_desc,"WINDOW");
$DESCRIPTOR(cli_window_ret_desc," ");
short cli_window_ret_len;
char window_ascii[255];

$DESCRIPTOR(cli_extra_desc,"EXTRA");
$DESCRIPTOR(cli_extra_ret_desc," ");
int extra = 0;

/* Internal routines */

int process_one_key(); /* called by lbr$get_index, once for each library key*/
int eval_commandline(); 
int check_include_list();
int flush_lines();
int mark_lines();
int trim();
int switch_subtopics();

/* Miscellaneous */

int temploop;
int debug = 0; /* Set this to a nonzero value to trace certain functions*/
int preceeding_lines = 0;  /* corresponds to first /window value*/
int subsequent_lines = 0;  /* corresponds to second /window value*/
int no_window = 0; /* will be set if /nowindow was specified*/
int sys$exit();
int heading_printed = 0;
int current_nesting_level;
string80 level_heading[11];
char lib_record_str[1024];
int loop;
int listmatch_exact = 0;

/* Begin main */

int main(int argc, char *argv[])
{

/* Evaluate the command line */

clistat = eval_commandline();
if (clistat != SS$_NORMAL) return(clistat);

/* Set up the library control structure.  It's used by the LBR$ routines. */

libstat = lbr$ini_control(&library_index, &LBR$C_READ, &LBR$C_TYP_HLP, 0);
if (debug) printf("Libstat(lbr$ini_control) return value: %d\n",libstat);

if (libstat != lbr$_normal)
  {
  printf("**Error** lbr$ini_control returned an unexpected value %d\n",libstat);
  return(libstat);
  }

/* Now open the library */

library_filespec.dsc$a_pointer = &library[0]; /* point descriptor to filename*/
library_filespec.dsc$w_length = strlen(library); /* and set length */
libstat = lbr$open(&library_index, &library_filespec, 0, 0, 0, 0, 0);
if (debug) printf("Libstat(lbr$open) return value: %d\n",libstat);

if (libstat != lbr$_normal)
  {
  printf("**Error** lbr$open returned an unexpected value %d\n",libstat);
  printf("  Filespec: '%s'\n",library_filespec.dsc$a_pointer);
  if (libstat == lbr$_typmismch)
     printf("Library file appears not to be a help library\n");
  return(libstat);
  }

/* Use LBR$GET_INDEX to call a user routine for each key in the index. */
/* LBR$GET_INDEX will call process_one_key() multiple times, once per  */
/* top-level help topic, before returning here. */

libstat = lbr$get_index(&library_index,&1,process_one_key,0);
if (debug) printf("Libstat(lbr$get_index) return value: %d\n",libstat);
if (libstat != SS$_NORMAL)
  {
  printf("lbr$get_index returned an unexpected value %d\n",libstat);
  }

/* Now notify the user if any modules specified by /INCLUDE or /EXCLUDE */
/* were not present in the library. */

for (loop = 1 ; loop <= num_excludes ; loop++)
  {
  if (!exclude_seen[loop-1])
     printf("Module %s not found in library\n",exclude_spec[loop-1]);
  }

for (loop = 1 ; loop <= num_includes ; loop++)
  {
  if (!include_seen[loop-1])
     printf("Module %s not found in library\n",include_spec[loop-1]);
  }

/* And we're done. */

return(libstat);

} /* end main */

/***************************************************************************/

int eval_commandline()
{

int temp;
int loop;
int libret;
unsigned long int rmsreturn;

$DESCRIPTOR(lib_find_ret_desc,"");
$DESCRIPTOR(default_spec_desc,"*.HLB;");

char returned_filespec[1024];
lib_find_ret_desc.dsc$a_pointer = &returned_filespec[0];
lib_find_ret_desc.dsc$w_length = 255;

/* Get debug value from command line */

clistat = cli$present(&cli_debug_desc);
if (clistat == CLI$_PRESENT)
   {
   cli_debug_ret_desc.dsc$a_pointer = &debuglevel_ascii[0];
   cli_debug_ret_desc.dsc$w_length = 16;
   clistat = cli$get_value(&cli_debug_desc,&cli_debug_ret_desc,
               &cli_debug_ret_len);
   debuglevel_ascii[cli_debug_ret_len] = 0;
   debug = atoi(debuglevel_ascii);
   if (debug == 0) debug = 1; /* Minimum meaningful value is 1 */
   printf("debuglevel is %d\n",debug);
   }

/* Check for the /EXTRA qualifier */

clistat = cli$present(&cli_extra_desc);
if (clistat == CLI$_PRESENT)
   {
   extra = 1;
   }

/* Get search string from command line. */

clistat = cli$present(&cli_p1_desc);
if (clistat != CLI$_PRESENT)
   {
   printf("Please supply a search string as P1 on the command line.\n");
   printf("P1 should be defined value=(type=$file,required) in .CLD file.\n");
   return(clistat);
   }
cli_p1_ret_desc.dsc$a_pointer = &searchstring[0];
cli_p1_ret_desc.dsc$w_length = 1023;
clistat = cli$get_value(&cli_p1_desc,&cli_p1_ret_desc,&cli_p1_ret_len);
searchstring[cli_p1_ret_len] = 0;

/* Convert search string to upper case (for comparisons against uppercase */
/* versions of library records. */

for (temploop = 0 ; temploop <= strlen(searchstring) ; temploop++)
  {
  searchstring[temploop] = toupper(searchstring[temploop]);
  }
if (debug) printf ("P1 (search_string): '%s'\n",searchstring);

/**/
/*
/* Set up which help library to search.  The default is sys$help:helplib.hlb,
/* but the user may have overridden this by using /library
/*
/**/

clistat = cli$present(&cli_library_desc);
if (clistat == CLI$_NEGATED)
  {
  printf("/NOLIBRARY was specified.  No operation will be performed.\n");
  return(SS$_ABORT);
  }

if (clistat == CLI$_PRESENT)
  {
  if (debug) printf("/LIBRARY was specified.\n");
  cli_library_ret_desc.dsc$a_pointer = &library[0];
  cli_library_ret_desc.dsc$w_length = 1023;
  clistat = cli$get_value(&cli_library_desc,&cli_library_ret_desc,
            &cli_library_ret_len);
  if (clistat == CLI$_ABSENT)
     {
     printf("/LIBRARY was specified, but no file specification was supplied.\n");
     printf("  - No action will be performed.  \n");
     return(SS$_ABORT);
     }
  library[cli_library_ret_len] = 0;

  /* Use LIB$FIND_FILE to locate the actual libray file corresponding to */
  /* the /LIBRARY value. */

  libret = LIB$FIND_FILE(&cli_library_ret_desc,&lib_find_ret_desc,&0,
               &default_spec_desc,&0,
               &rmsreturn,&0);
  returned_filespec[254] = 0;
  temp = trim(returned_filespec);
  if (libret == RMS$_NORMAL)
     {
       sprintf(library,"%s\0",returned_filespec);
     } else {
       printf("Error searching for %s\n",returned_filespec);
       libret = sys$exit(libret);
     }
  }

if (debug) printf("Library: '%s'\n",library);

/* Get the values for /WINDOW, if specified. */

clistat = cli$present(&cli_window_desc);
if (debug) printf("cli$present(window) return value is %d\n",clistat);

if (clistat == CLI$_NEGATED)
  {
  if (debug) printf("/NOWINDOW was specified.\n");
  no_window = 1;
  }

if (clistat == CLI$_PRESENT)
  {
  if (debug) printf("/WINDOW was specified.\n");

  cli_window_ret_desc.dsc$a_pointer = &window_ascii[0];
  cli_window_ret_desc.dsc$w_length = 254;

  clistat = cli$get_value(&cli_window_desc,&cli_window_ret_desc,
            &cli_window_ret_len);

  if (clistat == CLI$_ABSENT)
     {
     printf("/WINDOW was specified, but no value was supplied.\n");
     return(CLI$_ABSENT);
     }
  window_ascii[cli_window_ret_len] = 0;
  if (debug) printf("first value found for /window is %s\n",window_ascii);
  preceeding_lines = atoi(window_ascii);
  if (debug) printf("preceeding_lines = %d\n",preceeding_lines);

  clistat = cli$get_value(&cli_window_desc,&cli_window_ret_desc,
            &cli_window_ret_len);

  if (clistat == CLI$_ABSENT)
     {
       if (debug) printf("Only one /WINDOW value found; subsequent lines defaults to preceeding_lines\n");
       subsequent_lines = preceeding_lines;
     } else {
       window_ascii[cli_window_ret_len] = 0;
       if (debug) printf("second value found for /window is %s\n",window_ascii);
       subsequent_lines = atoi(window_ascii);
     }
  if (debug) printf("subsequent_lines = %d\n",subsequent_lines);

  /* Now make sure that the user didn't specify three or more values */

  clistat = cli$get_value(&cli_window_desc,&cli_window_ret_desc,
            &cli_window_ret_len);
  if (clistat != CLI$_ABSENT)
     {
     printf("Too many values specified for /WINDOW.  Extra values ignored.\n");
     }
  } /* end if /window is present */

if (debug) printf("window values: %d,%d\n",preceeding_lines,subsequent_lines);

if ((preceeding_lines > MAX_WINDOW) || (subsequent_lines > MAX_WINDOW))
   {
   printf("/WINDOW value cannot exceed %d; using that value\n",MAX_WINDOW);
   if (preceeding_lines > MAX_WINDOW) preceeding_lines = MAX_WINDOW;
   if (subsequent_lines > MAX_WINDOW) subsequent_lines = MAX_WINDOW;
   if (debug) printf("new window values: %d,%d\n",preceeding_lines,subsequent_lines);
   }

/* Look for the /INCLUDE qualifier */

clistat = cli$present(&cli_include_desc);
if (debug) printf("cli$present(include) return value is %d\n",clistat);

if (clistat == CLI$_PRESENT)
   {
   cli_include_ret_desc.dsc$a_pointer = &include_in[0];
   cli_include_ret_desc.dsc$w_length = 79;

   clistat = cli$get_value(&cli_include_desc,&cli_include_ret_desc,
             &cli_include_ret_len);

   while (
         ((clistat == CLI$_COMMA) || (clistat == SS$_NORMAL))
         && (num_includes < 100))
     {
     include_in[cli_include_ret_len] = 0;
     num_includes++;
     if (num_includes >= 100)
       {
       printf("Too many values supplied for /INCLUDE.  Extra values ignored.\n");
       }
     for (loop = 0 ; loop <= strlen(include_in) ; loop++)
       {
       include_in[loop] = toupper(include_in[loop]);
       }
     sprintf(include_spec[num_includes-1],"%s\0",include_in);
     include_seen[num_includes-1] = 0;
     if (debug) printf("include_in: '%s'\n",include_in);
     clistat = cli$get_value(&cli_include_desc,&cli_include_ret_desc,
               &cli_include_ret_len);
     }
   if (!num_includes)
     {
     printf("/INCLUDE was specified, but no value was supplied.\n");
     clistat = sys$exit(CLI$_VALREQ);
     }
   }

/* Look for the /EXCLUDE qualifier */

clistat = cli$present(&cli_exclude_desc);
if (debug) printf("cli$present(exclude) return value is %d\n",clistat);

if (clistat == CLI$_PRESENT)
   {
   cli_exclude_ret_desc.dsc$a_pointer = &exclude_in[0];
   cli_exclude_ret_desc.dsc$w_length = 79;

   clistat = cli$get_value(&cli_exclude_desc,&cli_exclude_ret_desc,
             &cli_exclude_ret_len);

   while (
         ((clistat == CLI$_COMMA) || (clistat == SS$_NORMAL)) &&
         (num_excludes < 100))
     {
     exclude_in[cli_exclude_ret_len] = 0;
     num_excludes++;
     if (num_excludes >= 100)
       {
       printf("Too many values supplied for /EXCLUDE.  Extra values ignored.\n");
       }
     for (loop = 0 ; loop <= strlen(exclude_in) ; loop++)
       {
       exclude_in[loop] = toupper(exclude_in[loop]);
       }
     sprintf(exclude_spec[num_excludes-1],"%s\0",exclude_in);
     exclude_seen[num_excludes-1] = 0;
     if (debug) printf("exclude_in: '%s'\n",exclude_in);
     clistat = cli$get_value(&cli_exclude_desc,&cli_exclude_ret_desc,
               &cli_exclude_ret_len);
     }
   if (!num_excludes)
     {
     printf("/EXCLUDE was specified, but no value was supplied.\n");
     clistat = sys$exit(CLI$_VALREQ);
     }
   }

/* Ensure that is /INCLUDE and /EXCLUDE were noth specified, that /INCLUDE */
/* will take precedence */

if (num_includes && num_excludes)
  {
  printf("/INCLUDE and /EXCLUDE were both specified.\n");
  printf("  Only /INCLUDE will be processed.\n");
  num_excludes = 0;
  }

/* Process the /LISTMATCH qualifier */

clistat = cli$present(&cli_listmatch_desc);
if (debug) printf("cli$present(listmatch): %d\n",clistat);
if (clistat == CLI$_PRESENT)
   {
   cli_listmatch_ret_desc.dsc$a_pointer = &listmatch_ascii[0];
   cli_listmatch_ret_desc.dsc$w_length = 79;

   clistat = cli$get_value(&cli_listmatch_desc,&cli_listmatch_ret_desc,
             &cli_listmatch_ret_len);
   listmatch_ascii[cli_listmatch_ret_len] = 0;
   if (debug) printf("Listmatch value: '%s'\n",listmatch_ascii);
   if (listmatch_ascii[0] == 69)
       listmatch_exact = 1;
   }

/* Command line processing done.  If anything bad happens during command  */
/* line processing, we will have returned something other than SS$_NORMAL */
/* before reaching this point. */

return(SS$_NORMAL);

} /* end eval_commandline() */

/***************************************************************************/
/*
/* This routine processes one module (top-level help topic) in the library.
/* This routine is called from lbr$get_index, once for each key value (topic).
/*
/**/

int process_one_key(char *key_name_desc, char *rfa_addr)
{

int libstat;
char key_name[255];
char lib_record_up[1024];

int temploop;
int temp;

/* Define an rfa structure that lbr$lookup_key can write. */

struct 
  {
  unsigned long int lword0;
  unsigned long int lword1;
  } lookup_rfa;

/* Define the key descriptor structure.  We'll be entered with the address */
/* of such a structure, and this provides a means of interpreting its */
/* contents.*/

struct KEY_DESC_STRUCT
  {
  unsigned long int lword0;
  unsigned long int lword1;
  } *key_desc_struct;

key_desc_struct = (struct KEY_DESC_STRUCT *)key_name_desc;

strncpy(key_name,(char *)key_desc_struct->lword1,key_desc_struct->lword0);
key_name[key_desc_struct->lword0] = 0;

if (debug)
   printf("process_one_key entered: %08X, %08X, '%s'\n",
      key_desc_struct->lword0,key_desc_struct->lword1,key_name);

/* First, check the include/exclude lists to see if we're supposed to */
/* search this particular module.  If not, simply return to caller.   */

if (!check_include_list(key_name)) return(SS$_NORMAL);

/* Now call lbr$lookup_key for the RFA with which this routine was entered.*/
/* This is to set the proper context for LBR$GET_RECORD, below. */

/*Note: I'm passing key_name_desc "by value" because the "value" is */
/* actually an address.*/

libstat = lbr$lookup_key(&library_index,key_name_desc,&lookup_rfa);
if (debug) printf("Libstat(lbr$lookup_key) return value: %d\n",libstat);
if (libstat != SS$_NORMAL)
  {
  printf("**Error** lbr$lookup_key returned an unexpected value %d\n",libstat);
  printf("  Current key is: '%s'\n",key_name);
  return(libstat);
  }

/* Now use lbr$get_record to retrieve each record in the help topic.*/
/* We'll have to keep track of nesting levels ourselves. */

library_record.dsc$a_pointer = &lib_record_str[0]; 
library_record.dsc$w_length = 256;

library_record_up.dsc$a_pointer = &lib_record_up[0]; 
library_record_up.dsc$w_length = 256;

/* Read each record in the top-level help topic, keep track of subtopic */
/* levels, and tell the user of any matches on the search string. */

current_nesting_level = 0;
for (temploop = 0 ; temploop <= 9 ; temploop++)
  {
  level_heading[temploop][0] = 0;
  }
memset(lib_record_str,0,1023);
libstat = lbr$get_record(&library_index,&library_record,&outbufdes);
while (libstat == SS$_NORMAL)
  {

  /* Terminate string returned from lbr$get_record */

  lib_record_str[outbufdes.dsc$w_length] = 0; 

  /* See if we're switching suptopics.*/

  if ((lib_record_str[0] != 32) && (lib_record_str[0] != 0))
     {
     temp = switch_subtopics();
     if (!temp) return(0);
     }

  /* Unless the user specified /NOWINDOW, we'll handle the output buffer */
  /* as appropriate (move hands, store record, display records, etc.) */

  if (!no_window)
    {

    /* Step the two hands (input slot, output slot) through the buffer */

    bufferslot_in++;
    if (bufferslot_in > (OUT_BUF_SIZE-1)) bufferslot_in -= OUT_BUF_SIZE;
    bufferslot_out = bufferslot_in - preceeding_lines - 1;
    if (bufferslot_out < 0) bufferslot_out += OUT_BUF_SIZE;

    /* Set the attributes of the current buffer slot*/
  
    outbuffer[bufferslot_in].serial = serial_ref;

    /* And put the current line into the table */

    strcpy(outbuffer[bufferslot_in].text,lib_record_str);

    if (debug>=10) printf("(Store %d) %s\n",bufferslot_in,outbuffer[bufferslot_in].text);
  
    /* Display bufferslot_out line, if appropriate*/

    if ((outbuffer[bufferslot_out].needs_printing) &&
        (outbuffer[bufferslot_out].serial == serial_ref))
       {
       printf("%s\n",outbuffer[bufferslot_out].text);
       outbuffer[bufferslot_out].needs_printing = 0;
       }

  } /*end if !no_window */

  /* Look for the search string in the library record */

  temp = STR$UPCASE(&library_record_up,&library_record);

  if (strstr(lib_record_up,searchstring))
     {
     if (debug) printf("Match found(%d)\n",bufferslot_in);
     if (!no_window) temp = mark_lines();
     if (!heading_printed)
        {
        printf("\n**********\n%s %s %s %s %s %s %s %s %s\n\n",
                level_heading[1],level_heading[2],level_heading[3],
                level_heading[4],level_heading[5],level_heading[6],
                level_heading[7],level_heading[8],level_heading[9]);
        heading_printed = 1;
        }
     }

  /* And get the next record. */

  libstat = lbr$get_record(&library_index,&library_record,&outbufdes);

  } /* end while (libstat == SS$_NORMAL) */

if (libstat != RMS$_EOF)
   {
   printf("**ERROR** lbr$get_record returned an unexpected value : %d\n",libstat);
   printf("  Current key is: '%s'\n",key_name);
   return(libstat);
   }

return(1);

} /* end process_one_key() */

/***************************************************************************/
/*
/* Switch subtopics.  This routine acknowledges when the help text
/* changes from topic to topic.
/*
/**/

int switch_subtopics()
{

int temp;
char first_two_bytes[] = "  \0";
int first_two_bytes_int;
char level_match_string[] = "1 2 3 4 5 6 7 8 9 ";

/* If we get here, the first character in a text line was not a space.  This */
/* usually means we've stepped down into a subtopic, but not always.  Check  */
/* here to see if the first two bytes are a heading level. */

/* Here, we'll record the nested topic levels.  If we should subsequently */ 
/* find a match on a searchstring, the fields set below will be used */
/* at that time. */

strncpy(first_two_bytes,lib_record_str,2);
if (strstr(level_match_string,first_two_bytes))
   {
   heading_printed = 0;
   serial_ref++;
   flush_lines(); /* flush remaining buffer lines when changing topics*/
   /*printf(" '%s'\n",lib_record_str);*/
   first_two_bytes_int = atoi(first_two_bytes);
   if (first_two_bytes_int >= 0)
      {
        if (first_two_bytes_int > current_nesting_level + 1)
           {
             printf("One or more nesting levels was skipped upward!  That's bad.\n");
             return(0);
           } else {
             current_nesting_level = first_two_bytes_int;
             lib_record_str[0] = 32; lib_record_str[1] = 32;
             temp = trim(lib_record_str);
             sprintf(level_heading[current_nesting_level],
                     "%s\0",lib_record_str);
             for (temploop = current_nesting_level+1 ; 
                  temploop <= 10 ; temploop++)
               {
               level_heading[temploop][0] = 0;
               }
           }
      } else {
        printf("first_two_bytes_int is less than zero!  That's bad.\n");
        return(0);
      }
   } else { /* if first two bytes are not a nesting level number*/
      /*
      /* Switch topics here only if /EXTRA was specified 
      */
     if (extra)
        {
        serial_ref++;
        flush_lines(); /* flush remaining buffer lines when changing topics*/
        temp = trim(lib_record_str);
        sprintf(level_heading[current_nesting_level+1],
                "%s\0",lib_record_str);
        heading_printed = 0;
        }
   }

return(1);

} /* end switch_subtopics() */

/***************************************************************************/
/*
/* Trim.  This will trim tabs and spaces from both ends of a string
/*
/**/

int trim(char *instring)
{
char workstring[32768];
int position;

position = 0;
while ((instring[position] == 32) || (instring[position] == 9))
  {
  position++;
  }

sprintf(workstring,"%s\0",instring+position);

while ((workstring[strlen(workstring)-1] == 32) ||
          (workstring[strlen(workstring)-1] == 9))
  {
  workstring[strlen(workstring)-1] = 0;
  }

sprintf(instring,"%s\0",workstring);

return(1);

} /* end trim */

/***************************************************************************/
/*
/* This routine will check whether the current key should be searched.
/* This is the implementation of the /INCLUDE and /EXCLUDE qualifiers.
/* This routine will be called once for each help topic (TYPE, ASSIGN, etc.)
/* It will return one if the key should be searched, zero if not.
/*
/**/

int check_include_list(char *key_name)
{
int loop;
char local_key_name[255];

if (debug >= 2) printf("check_include_list entered: '%s'\n",key_name);

sprintf(local_key_name,"%s\0",key_name);
for (loop = 0 ; loop <= strlen(local_key_name) ; loop++)
  {
  local_key_name[loop] = toupper(local_key_name[loop]);
  }

if (num_excludes)
  {
  for (loop = 1 ; loop <= num_excludes ; loop++)
    {
    if (debug >= 2) printf("  checking exclude: '%s'\n",exclude_spec[loop-1]);

    if (
        (
         (!listmatch_exact) 
         &&
         (!strncmp(exclude_spec[loop-1],local_key_name,
          strlen(exclude_spec[loop-1])))
        )
        ||
        (
         (listmatch_exact) &&
         (!strcmp(exclude_spec[loop-1],local_key_name))
        )
       )
      {
      if (debug >= 2) printf("Exclude list match on %s\n",
              exclude_spec[loop-1]);
      exclude_seen[loop-1] = 1;
      return(0);
      }
    }
  }

if (num_includes)
  {
  for (loop = 1 ; loop <= num_includes ; loop++)
    {
    if (debug >= 2) printf("  checking include: '%s'\n",include_spec[loop-1]);

    if (
        (
         (!listmatch_exact) 
         &&
         (!strncmp(include_spec[loop-1],local_key_name,
          strlen(include_spec[loop-1])))
        )
        ||
        (
         (listmatch_exact) &&
         (!strcmp(include_spec[loop-1],local_key_name))
        )
       )
      {
      if (debug >= 2) printf("Include list match on %s\n",
              include_spec[loop-1]);
      include_seen[loop-1] = 1;
      return(1);
      }
    }
  return(0);
  }

return(1);

} /* end check_include_list */

/***************************************************************************/
/*
/* This routine will mark lines in the output buffer (slots that are
/* already filled and slots that are yet to be filled.)  The serial field
/* will prevent us from accidentally bleeding into the next topic if the
/* current topic ends before we've printed the appropriate number of lines.
/*
/**/

int mark_lines()
{
int lines_to_mark;
int loop;
int target_line;

lines_to_mark = preceeding_lines + subsequent_lines + 1;
target_line = bufferslot_in - preceeding_lines;
if (target_line < 0) target_line += (OUT_BUF_SIZE-1);

for (loop = 0 ; loop < lines_to_mark ; loop++)
  {
  if (debug >= 10) printf("Mark(%d)\n",target_line);
  outbuffer[target_line].needs_printing = 1;
  target_line++;
  if (target_line > (OUT_BUF_SIZE-1)) target_line -= (OUT_BUF_SIZE-1);
  }

return(1);
}

/***************************************************************************/

int flush_lines()
{
int loop;
int target_line;
int lines_to_check;

if (debug>=5) printf("flush_lines() called.\n");

target_line = bufferslot_out;

for (loop = 0 ; loop <= (preceeding_lines+1) ; loop++)
  {
  if (debug>=10) printf("flush:check(%d)\n",target_line);
  if ((outbuffer[target_line].needs_printing) ||
     (outbuffer[target_line].serial == serial_ref))
     printf("%s\n",outbuffer[target_line].text);
  outbuffer[target_line].needs_printing = 0;
  target_line++;
  if (target_line > (OUT_BUF_SIZE-1)) target_line -= (OUT_BUF_SIZE-1);
  }

return(1);

}
