/*
++

 Title:
	FDEL (Fast File DELete)

 Version:
	2.0

 Facility:
	System Management Tool

 Abstract:
	This program performs a fast delete of files in a very large directory (>20,000 files). The directory
        file is read into memory and each entry (file) is examined. If the file should be deleted, a delete-by-file-id 
        operation is performed. If the file should be kept, the entry is moved to a buffer that will be used to overlay
        the original directory.

        Usage:

		$ FDEL [/SINCE=yyyymmmdd:hh:m:ss.hh]
		       [/BEFORE=yyyymmmdd:hh:m:ss.hh] 
		       [/INPUT=file-spec]               [device:[directory]file-name.type;version]

	Wildcards are supported in the "file-name" and "type".

	the "/INPUT" is used to supply a list of files to be deleted. The list must be arranged alphabetically
	on "file-name" and "type", and in reverse version number order.

	FDEL works on just one directory at a time. It does not recurse to process subdirectories.

 Environment:
	Requires exclusive access directory file and all files in that directory. 

 Author:
	Mark Oakley	AirTouch Cellular	30-Dec-1999

 Modified:
	
	21-Nov-2002	Mark Oakley	Read entire directory file instead of mapping, and overwite
					with modifications to avoid corrupting back-link pointers.
					Also increase amount of information we put into a directory
					block (MAX_BYTES_USED_IN_BLK) to 460 bytes.

--
*/

#define   __NEW_STARLET    1

#define   NULL                      0
#define   BLANK                    32
#define   MINUS                    45
#define   PERIOD                   46
#define   ZERO                     48
#define   COLON                    58
#define   SEMICOLON                59
#define   LEFT_BRACKET             91
#define   RIGHT_BRACKET            93
#define   MAX_BYTES_USED_IN_BLK   460
#define   REC_TERMINATOR        65535

#define   SS$_NORDACC            9882

#include  <atrdef.h>
#include  <climsgdef.h>
#include  <cli$routines.h>
#include  <descrip.h>
#include  <fatdef.h>
#include  <fibdef.h>
#include  <fchdef.h>
#include  <iodef.h>
#include  <iosbdef.h>
#include  <lib$routines.h>
#include  <libdef.h>
#include  <rms.h>
#include  <secdef.h>
#include  <ssdef.h>
#include  <starlet.h>
#include  <stdio.h>
#include  <stdlib.h>
#include  <str$routines.h>
#include  <strdef.h>
#include  <string.h>
#include  <va_rangedef.h>

#define   check_status(status)\
          if ( ! (status & SS$_NORMAL)) \
          {\
            lib$signal(status);\
          }

/*
  Macro to move another version of a file from the old directory buffer
  to the new directory buffer.
*/
#define   move_file_version \
          new_dir_ptr->rec_len = new_dir_ptr->rec_len + 8;\
          new_ver_ptr          = new_ver_ptr_end;\
          Increment_PTR (&new_ver_ptr_end, new_ver_ptr_end, 8);\
          new_ver_ptr->version = ver_ptr->version;\
          new_ver_ptr->fid     = ver_ptr->fid;\
          new_ver_ptr->seq     = ver_ptr->seq;\
          new_ver_ptr->rvn     = ver_ptr->rvn; \
          Increment_PTR (&new_dir_ptr_end, new_dir_ptr, new_dir_ptr->rec_len+2);

/*
  Macro to move a directory entry (file) from the old directory buffer to
  the new directory buffer.
*/
#define   move_directory_entry\
          new_dir_ptr           = new_dir_ptr_end;\
          new_dir_ptr->rec_len  = 4 + 2 * ((strlen (fname) + 1) / 2) + 8;\
          new_dir_ptr->vlim     = dir_ptr->vlim;\
          new_dir_ptr->flags    = 0;\
          new_dir_ptr->name_len = dir_ptr->name_len;\
          strcpy (new_dir_ptr->name, fname);\
          Increment_PTR (&new_ver_ptr, new_dir_ptr, 6 + 2 * ((strlen (fname) + 1) / 2));\
          new_ver_ptr->version  = ver_ptr->version;\
          new_ver_ptr->fid      = ver_ptr->fid;\
          new_ver_ptr->seq      = ver_ptr->seq;\
          new_ver_ptr->rvn      = ver_ptr->rvn;\
          new_ver_ptr_end       = new_ver_ptr + 1;\
          Increment_PTR (&new_dir_ptr_end, new_dir_ptr, new_dir_ptr->rec_len+2);

typedef   struct desc_struct
{ unsigned int   len;
  char           *addr; } desc_struct_typedef;

typedef   struct item_list_struct
{ short          buff;
  short          code;
  char           *addr;
  char           *retl; } item_list_struct_typedef;

typedef   struct iosb_struct
{ unsigned short status;
  unsigned short q1;
  unsigned short q2; } iosb_struct_typedef;

/*
  Layout of an entry in the directory file,
  except for version number info.
*/
typedef   struct dir_ptr_struct
{ unsigned short int  rec_len;
  unsigned short int  vlim;
  unsigned char       flags;
  unsigned char       name_len;
  char                name[]; } dir_ptr_struct_typedef;

/*
  Layout of version number info in the
  directory file.
*/
typedef struct ver_ptr_struct
{ unsigned short int  version;
  unsigned short int  fid;
  unsigned short int  seq;
  unsigned short int  rvn;} ver_ptr_struct_typedef;

  unsigned int        before_flag, input_flag, log_flag, since_flag, wild_flag;
  unsigned int        before_time[2], since_time[2];

  struct desc_struct  input_file_spec_desc, dir_spec_desc, tmp_spec_desc;
  char                input_file_spec[255], dir_spec[255], tmp_spec[255];
  struct desc_struct  file_spec_desc, tmp_name_desc;
  char                file_spec[255], tmp_name[40], input_file_rec[255];
  struct desc_struct  input_rec_desc;

  struct FAB          input_file_fab;
  struct RAB          input_file_rab;

  char                       *blk_ptr, *blk_ptr_end, *new_blk_ptr, *new_blk_ptr_end, *new_blk_ptr_start;
  struct dir_ptr_struct      *dir_ptr, *dir_ptr_end, *new_dir_ptr, *new_dir_ptr_end;
  struct ver_ptr_struct      *ver_ptr, *ver_ptr_end, *new_ver_ptr, *new_ver_ptr_end;

  char                dir_file_buf[255];
  struct desc_struct  dir_file_desc;

  unsigned int        files_processed  =    0;
  unsigned int        files_deleted    =    0;
  unsigned int        files_preserved  =    0;
  unsigned int        interval_cnt     =    0;
  unsigned int        interval_limit   = 5000;

/*
  This is starting procedure for FDEL.
*/
main (int argc, char *argv[])
{
  unsigned int   create_tmp_name       ();
  unsigned int   parse_command_line    ();
  unsigned int   open_input_file       ();
  unsigned int   create_tmp_dir_name   ();
  unsigned int   process_dir           ();
  void           do_interval_reporting ();

  unsigned int   status;
  unsigned int   i;

  input_file_spec_desc.addr = (char *) &input_file_spec;  input_file_spec_desc.len = sizeof (input_file_spec);
  dir_spec_desc.addr        = (char *) &dir_spec;         dir_spec_desc.len        = sizeof (dir_spec);
  file_spec_desc.addr       = (char *) &file_spec;        file_spec_desc.len       = sizeof (file_spec);
  tmp_spec_desc.addr        = (char *) &tmp_spec;         tmp_spec_desc.len        = sizeof (tmp_spec);
  tmp_name_desc.addr        = (char *) &tmp_name;         tmp_name_desc.len        = sizeof (tmp_name);
  
  printf ("\n");

/*
  Create a file name based on current date/time. We will use this name to create a directory that
  is a copy of the one we work on.
*/
  status = create_tmp_name (&tmp_name);
  check_status (status);
  tmp_name_desc.len         = strlen (tmp_name);

/*
  See what qualifiers and parameters were specified.
*/
  status = parse_command_line (&before_flag, &input_flag, 
                               &log_flag,    &since_flag,
                               &before_time, &since_time,
                               &wild_flag,   &input_file_spec_desc,
                               &dir_spec_desc, &file_spec_desc);
  check_status (status);

  if (input_flag)
  { status = open_input_file ();
    check_status (status); }

  status = create_tmp_dir_name (&dir_spec_desc, &tmp_name_desc, &tmp_spec_desc);
  check_status (status);

/*
  Examine the directory entries to see what should be kept and what should be deleted.
*/
  status = process_dir (&dir_spec_desc);
  check_status (status);

/*
  Report the results.
*/
  printf ("\n\n      *** Processing completed ***\n");
  do_interval_reporting ();
  tmp_spec[tmp_spec_desc.len]     = 0;
  dir_file_buf[dir_file_desc.len] = 0;
  printf ("\n %s copied to", dir_file_buf);
  printf ("\n %s\n", tmp_spec);
}

/*
  This procedure does most of the work to examine the old directory, delete files,
  create a new directory, etc.
*/
unsigned int process_dir (struct desc_struct *dir_spec_desc)
{
  unsigned int        open_dir_file         ();
  unsigned int        get_dev_chan          ();
  unsigned int        criteria_check        ();
  unsigned int        preserve_file         ();
  unsigned int        delete_file           ();
  unsigned int        create_new_dir        ();
  void                Increment_PTR         ();
  void                do_interval_reporting ();

  unsigned int        chan, dev_chan, new_chan;
  unsigned int        i, k, offset, eob;
  unsigned int        status;

  VA_RANGE                   retadr;

  struct FAB                 dir_fab, new_dir_fab;
  struct XABALL              dir_xaball;
  struct XABFHC              dir_xabfhc;
  struct XABPRO              dir_xabpro;

  struct _iosb               iosb;

  char                       fn[255], fname[255], new_fname[255] = {0};

  struct desc_struct         file_fib_desc;

  union
  { struct _fibdef           file_fib;
    char                     fib_overlay[FIB$K_LENGTH]; } fib_union;

  enum   dispositions {DELETE, PRESERVE} disposition;

  unsigned int               blks_to_alloc, blks_to_use;
  char                       *src_ptr, *dst_ptr, *end_ptr;

  struct _atrdef             attrib[3];
  struct _fat                recattr;
  struct _fch                uchar;

  file_fib_desc.len     = FIB$K_LENGTH;
  file_fib_desc.addr    = (char *) &fib_union.file_fib;

/*
  Get a channel to the directory and capture various directory attributes.
*/
  dir_fab               = cc$rms_fab;
  dir_xaball            = cc$rms_xaball;
  dir_xabfhc            = cc$rms_xabfhc;
  dir_xabpro            = cc$rms_xabpro;
  dir_fab.fab$l_xab     = (char *) &dir_xaball;
  dir_xaball.xab$l_nxt  = (char *) &dir_xabfhc;
  dir_xabfhc.xab$l_nxt  = (char *) &dir_xabpro;
  status = open_dir_file (dir_spec_desc, &dir_fab);
  check_status (status);

  status = get_dev_chan (dir_spec_desc, &dev_chan);
  check_status (status);

/*
  Read the directory into memory.
*/
  chan                  = dir_fab.fab$l_stv;
  status                = sys$expreg (dir_xabfhc.xab$l_ebk - 1, &retadr, 0, 0);
  check_status (status);
  blk_ptr               = retadr.va_range$ps_start_va;
  blk_ptr_end           = blk_ptr + (512 * (dir_xabfhc.xab$l_ebk - 1));
  status                = sys$qiow (0, chan, IO$_READVBLK, &iosb, 0, 0, blk_ptr, 512 * dir_xabfhc.xab$l_ebk, 1, 0, 0, 0);
  check_status (status);
  check_status (iosb.iosb$w_status);

/*
  Allocate some space that will hold the contents of the directory for
  any files not deleted.
*/
  status                = sys$expreg (dir_xabfhc.xab$l_ebk + 2, &retadr, 0, 0);
  check_status (status);
  new_blk_ptr_start     = retadr.va_range$ps_start_va;
  new_blk_ptr           = new_blk_ptr_start;
  new_blk_ptr_end       = retadr.va_range$ps_end_va;
  new_dir_ptr           = (struct dir_ptr_struct *) new_blk_ptr;
  new_dir_ptr_end       = (struct dir_ptr_struct *) new_blk_ptr;

/*
  Create a copy of the directory file.
*/
  blks_to_use           = dir_xabfhc.xab$l_ebk;
  blks_to_alloc         = dir_xabfhc.xab$l_ebk;
  status = create_new_dir (&new_dir_fab, &dir_xaball, &dir_xabfhc, &dir_xabpro, &blks_to_use, &blks_to_alloc, chan);
  check_status (status);

  new_chan   = new_dir_fab.fab$l_stv;
  status = sys$qiow (0, new_chan, IO$_WRITEVBLK, &iosb, 0, 0, blk_ptr, 512 * (dir_xabfhc.xab$l_ebk - 1), 1, 0, 0, 0);
  check_status (status);
  check_status (iosb.iosb$w_status);

/*
  Acquire various file attributes about the directory copy as we will
  need to modify a few of them.
*/
  attrib[0].atr$w_size = ATR$S_UCHAR;
  attrib[0].atr$w_type = ATR$C_UCHAR;
  attrib[0].atr$l_addr = &uchar;
  attrib[1].atr$w_size = ATR$S_RECATTR;
  attrib[1].atr$w_type = ATR$C_RECATTR;
  attrib[1].atr$l_addr = &recattr;
  attrib[2].atr$w_size = 0;
  attrib[2].atr$w_type = 0;
  status = sys$qiow (0, new_chan, IO$_ACCESS , &iosb, 0, 0,
                     0, 0, 0, 0, (__int64) &attrib, 0);
  check_status (status);
  check_status (iosb.iosb$w_status);

/* 
  Enable the "Directory" and "MoveFile Diabled" attributes. Set the version number limit
  and end block indicator on the copy.
*/
  uchar.fch$v_directory  = 1;
  uchar.fch$v_nomove     = 1;
  recattr.fat$w_versions = 0;
  recattr.fat$w_efblkl   = dir_xabfhc.xab$l_ebk;
  status = sys$qiow (0, new_chan, IO$_DEACCESS , &iosb, 0, 0,
                     0, 0, 0, 0, (__int64) &attrib, 0);
  check_status (status);
  check_status (iosb.iosb$w_status);

/*
  Loop through the old directory to examine each entry. The inner most loop
  is based on version number, the next outer loop is file name.type, and the
  outer most loop is block.
*/
  while (blk_ptr < blk_ptr_end)
  { dir_ptr     = (struct dir_ptr_struct *) blk_ptr;
    dir_ptr_end = dir_ptr + 512;
    while (dir_ptr < dir_ptr_end)
    { offset      = (unsigned int) dir_ptr + (2 * ((dir_ptr->name_len + 6 + 1) / 2));
      ver_ptr     = (struct ver_ptr_struct *) offset;
      offset      = (unsigned int) dir_ptr;
      ver_ptr_end = (struct ver_ptr_struct *) (offset + dir_ptr->rec_len);
      for (i=0; i<dir_ptr->name_len; i++) 
      { fname[i] = dir_ptr->name[i];
        fn[i]    = dir_ptr->name[i]; 
      }
      fn[i]    = NULL;
      fname[i] = SEMICOLON; i++; fname[i] = NULL;
      while (ver_ptr < ver_ptr_end)
      { sprintf ((char *) &fname[i], "%d", ver_ptr->version);
        for (k=0; k<FIB$K_LENGTH; k++) fib_union.fib_overlay[k] = 0;
        fib_union.file_fib.fib$w_fid[0] = ver_ptr->fid;
        fib_union.file_fib.fib$w_fid[1] = ver_ptr->seq;
        fib_union.file_fib.fib$w_fid[2] = ver_ptr->rvn;
/*
  We have identified another file. Determine if we delete it or keep it.
*/
        status = criteria_check (&fname, ver_ptr, &disposition, dev_chan, &file_fib_desc);
        check_status (status);
        if (disposition == DELETE)
        { status = delete_file (&file_fib_desc, dev_chan);
          check_status (status); 
        }
        if (disposition == PRESERVE) 
        { status = preserve_file (&fn, new_fname);
          check_status (status); 
        }
        files_processed++;  interval_cnt++;
        if (interval_cnt >= interval_limit)  do_interval_reporting ();
        offset  = (int) ver_ptr; offset = offset + 8; ver_ptr = (struct ver_ptr_struct *) offset; 
      } 
      dir_ptr = (struct dir_ptr_struct *) ver_ptr; 
      if (dir_ptr->rec_len == REC_TERMINATOR) dir_ptr_end = dir_ptr - 1;
    } 
    blk_ptr = blk_ptr + 512;
  }

/*
  The file scanning is finished. Mark the end of the in-memory buffer.
*/
  new_dir_ptr_end->rec_len = REC_TERMINATOR;  

/*
  Overwrite the directory file with the in-memory buffer.
*/
  status = sys$qiow (0, chan, IO$_WRITEVBLK, &iosb, 0, 0, new_blk_ptr_start, new_blk_ptr - new_blk_ptr_start + 512, 1, 0, 0, 0);
  check_status (status);
  check_status (iosb.iosb$w_status);

  attrib[0].atr$w_size = ATR$S_UCHAR;
  attrib[0].atr$w_type = ATR$C_UCHAR;
  attrib[0].atr$l_addr = &uchar;
  attrib[1].atr$w_size = ATR$S_RECATTR;
  attrib[1].atr$w_type = ATR$C_RECATTR;
  attrib[1].atr$l_addr = &recattr;
  attrib[2].atr$w_size = 0;
  attrib[2].atr$w_type = 0;
  status = sys$qiow (0, chan, IO$_ACCESS , &iosb, 0, 0,
                     0, 0, 0, 0, (__int64) &attrib, 0);
  check_status (status);
  check_status (iosb.iosb$w_status);

/*
  Fix the end-of-blk field as needed.
*/
  recattr.fat$w_efblkl   = 2 + (new_blk_ptr - new_blk_ptr_start) / 512;
  status = sys$qiow (0, chan, IO$_DEACCESS , &iosb, 0, 0, 0, 0, 0, 0, (__int64) &attrib, 0);
  check_status (status);
  check_status (iosb.iosb$w_status);

  status = sys$dassgn (chan);
  check_status (status);

  return (1);
}

/*
  Use attributes from the directory file to create the new one.
*/
unsigned int  create_new_dir (struct FAB *new_dir_fab,   struct XABALL *dir_xaball,
                              struct XABFHC *dir_xabfhc, struct XABPRO *dir_xabpro,
                              unsigned int *blks_to_use, unsigned int *blks_to_alloc,
                              unsigned int chan)
{
  unsigned int     status;

  *new_dir_fab           = cc$rms_fab;
  new_dir_fab->fab$l_fna = tmp_spec_desc.addr;
  new_dir_fab->fab$b_fns = tmp_spec_desc.len;
  new_dir_fab->fab$v_ufo =   1;
  new_dir_fab->fab$v_nil =   1;
  new_dir_fab->fab$v_blk =   1;
  new_dir_fab->fab$w_mrs = 512;
  new_dir_fab->fab$l_xab = (char *) dir_xaball;

  dir_xabfhc->xab$l_sbn = NULL;

  dir_xabpro->xab$l_aclsts = NULL;

  status = sys$create (new_dir_fab, 0, 0);
  check_status (status);

  return (1);
}

/*
  This routine controls moving an entry from the directory to the in-memory buffer.
  It will detect when we reach then end of the current block so that a new block 
  can be started.
*/
unsigned int preserve_file (char *fname, char *new_fname)
{
  void               Increment_PTR ();
  unsigned int       i, bptr, ptr;
  unsigned char      not_first_instance_of_file;
  unsigned char      not_near_end_of_block;
  unsigned char      near_end_of_block;
  char               resp[5];

  files_preserved++;
  bptr = (unsigned int) new_dir_ptr_end;  bptr = bptr & 0x000001ff;

  not_first_instance_of_file = (strcmp (new_fname, fname) == 0);
  near_end_of_block          = (bptr > MAX_BYTES_USED_IN_BLK);
  not_near_end_of_block      = (bptr < MAX_BYTES_USED_IN_BLK);

  if (not_first_instance_of_file & not_near_end_of_block)
  { move_file_version }
   else
  { if (near_end_of_block)
    { new_dir_ptr_end->rec_len = REC_TERMINATOR;
      new_blk_ptr              = new_blk_ptr + 512;
      new_dir_ptr              = (struct dir_ptr_struct *) new_blk_ptr;
      new_dir_ptr_end          = (struct dir_ptr_struct *) new_blk_ptr; }
    move_directory_entry
  }

  strcpy (new_fname, fname);
  return (1);
}

/*
  This routine is used when we want to increase the value of a pointer
  by an amount that does not match a multiple of the size of the data 
  structure that the pointer references.
*/
void Increment_PTR (unsigned int  *ptr1, void *ptr2, void *inc)
{
  unsigned int     i, j, k;

  j     = (unsigned int) ptr2; k = (unsigned int) inc;
  i     = j + k;
  *ptr1 = i;
}

/*
  Assign a channel to a disk device given a complete file-spec.
*/
unsigned int get_dev_chan (struct desc_struct *dir_spec_desc, unsigned int *dev_chan)
{
  char                *cptr;
  unsigned int        status, slen = 0;
  struct desc_struct  dev_desc;

  dev_desc.addr = dir_spec_desc->addr;
  cptr          = dir_spec_desc->addr;
  while (*cptr != COLON) { cptr++; slen++;}
  dev_desc.len  = slen;

  status = sys$assign (&dev_desc, (unsigned short *) dev_chan, 0, 0);
  check_status (status);

  return (1);
}

/*
  Based on the command line qualifiers and parameters, determine is this file should
  be deleted or kept for the new directory.
*/
unsigned int criteria_check (char *fname, struct ver_ptr_struct *ver_ptr, enum dispositions {DELETE, PRESERVE} *disposition,
                             unsigned int dev_chan, struct desc_struct *file_fib_desc)
{
  unsigned int        read_next_input_rec ();

  int                 result;
  unsigned int        i, j, func;
  unsigned int        status;
  unsigned int        input_check;
  struct desc_struct  file_name_desc;

  struct _atrdef      atr_blk[3];
  struct _iosb        *iosb;

  unsigned int        date_buf[2]    = {0,0};
  unsigned int        result_time[2] = {0,0};
  unsigned int        wild_check;
  unsigned int        since_check;
  unsigned int        before_check;

  file_name_desc.addr = fname;
  file_name_desc.len  = strlen (fname);

/*
  Retrieve the creation date/time for this file if we need it.
*/
  if (before_flag || since_flag)
  { atr_blk[0].atr$w_size = sizeof (date_buf);
    atr_blk[0].atr$w_type = ATR$C_CREDATE;
    atr_blk[0].atr$l_addr = (char *) &date_buf;
    atr_blk[1].atr$w_size = 0;
    atr_blk[1].atr$w_type = 0;
    iosb                  = calloc (1, sizeof (*iosb));

    func   = IO$_ACCESS | IO$M_ACCESS;
    status = sys$qiow (0, dev_chan, func, iosb, 0, 0, file_fib_desc, 0, 0, 0, (__int64) &atr_blk, 0);
    check_status (status);
    check_status (iosb->iosb$w_status);

    func   = IO$_DEACCESS;
    status = sys$qiow (0, dev_chan, func, iosb, 0, 0, file_fib_desc, 0, 0, 0, 0, 0);
    check_status (status);
    check_status (iosb->iosb$w_status); }

  input_check = 0;
  if (input_flag)
  { while ((result = str$compare (&file_name_desc, &input_rec_desc)) > 0) 
    { status = read_next_input_rec (); 
      check_status (status); }
    if (result == 0) input_check = 1; }
  else
    input_check = 1;

  since_check = 0;
  if (since_flag)
  { status = lib$subx (since_time, (unsigned int *) &date_buf, (unsigned int *) &result_time);
    check_status (status);
    if (result_time[1] & 0x80000000) since_check = 1; }
  else since_check = 1;

  before_check = 0;
  if (before_flag)
  { status = lib$subx ((unsigned int *) &date_buf, before_time, (unsigned int *) &result_time);
    check_status (status);
    if (result_time[1] & 0x80000000) before_check = 1; }
  else before_check = 1;

  wild_check = 0;
  if (wild_flag)
  { status = str$match_wild (&file_name_desc, &file_spec_desc);
    if (status == STR$_MATCH)  wild_check = 1; }
  else wild_check = 1;

  *disposition = PRESERVE;
  if (input_check && before_check && since_check && wild_check) *disposition = DELETE;
  return (1);
}

/*
  Get a channel to the old directory.
*/
unsigned int open_dir_file (struct desc_struct *dir_spec_desc,
                            struct FAB         *dir_fab)
{
  unsigned int        status;

  char                *src_ptr, *dst_ptr, *ptr;

/*
  Convert the spec from something like "dev:[dir1.dir2]" to
  "dev:[dir1]dev2.dir".
*/
  ptr = dir_spec_desc->addr + dir_spec_desc->len - 1;
  while (*ptr != PERIOD && *ptr != LEFT_BRACKET) ptr--;
  src_ptr = dir_spec_desc->addr; dst_ptr = (char *) &dir_file_buf;
  while (src_ptr != ptr) 
  { *dst_ptr = *src_ptr; dst_ptr++; src_ptr++; };
  if (*ptr == LEFT_BRACKET) 
    { strcpy (dst_ptr, "[000000]"); dst_ptr = dst_ptr + 8; }
   else
    { *dst_ptr = RIGHT_BRACKET;     dst_ptr = dst_ptr + 1; }
  src_ptr++;
  while (*src_ptr != RIGHT_BRACKET)
  { *dst_ptr = *src_ptr; dst_ptr++; src_ptr++;}
  strcpy (dst_ptr, ".DIR"); dst_ptr = dst_ptr + 4; *dst_ptr = NULL;

  dir_fab->fab$l_fna = (char *) &dir_file_buf;
  dir_fab->fab$b_fns = strlen (dir_file_buf);
  dir_fab->fab$v_nil = 1;
  dir_fab->fab$v_ufo = 1;
  dir_fab->fab$v_get = 1;
  dir_fab->fab$v_put = 1;
  status = sys$open (dir_fab, 0, 0);
  check_status (status);

  dir_file_desc.len  = strlen (dir_file_buf);
  dir_file_desc.addr = (char *) &dir_file_buf;

  return (1);
}

/*
  Create a directory name using the current date/time string.
*/
unsigned int create_tmp_dir_name (struct desc_struct *dir_spec_desc, 
                                  struct desc_struct *tmp_name_desc,
                                  struct desc_struct *tmp_spec_desc)
{
  unsigned int        i, j;
  unsigned int        status;
  unsigned int        tmp_spec_len;
  char                dir_spec[255];
  char                *end_ptr;
  char                *tmp_spec;
  char                file_name[39];
  struct FAB          dir_fab;
  struct NAM          dir_nam;
  
  dir_fab           = cc$rms_fab;
  dir_fab.fab$l_fna = dir_spec_desc->addr;
  dir_fab.fab$b_fns = dir_spec_desc->len;
  dir_fab.fab$l_nam = &dir_nam;
  dir_nam           = cc$rms_nam;
  dir_nam.nam$l_esa = (char *) &dir_spec;
  dir_nam.nam$b_ess = sizeof (dir_spec);

  status = sys$parse (&dir_fab, 0, 0);
  check_status (status);
  dir_spec[dir_nam.nam$b_esl] = 0;

  strcpy (tmp_spec_desc->addr, dir_spec);
  tmp_spec_len = strlen (tmp_spec_desc->addr) - 3;
  tmp_spec = tmp_spec_desc->addr + tmp_spec_len;

  while (*tmp_spec != LEFT_BRACKET && *tmp_spec != PERIOD) tmp_spec--;
  if (*tmp_spec == LEFT_BRACKET)
  { tmp_spec++; 
    for (i=0; i<6; i++)
    { *tmp_spec = ZERO; tmp_spec++; } }

  *tmp_spec = RIGHT_BRACKET; tmp_spec++;
  *tmp_spec = 0;  strcat (tmp_spec, tmp_name_desc->addr);
  strcat (tmp_spec, ".DIR");

  tmp_spec = tmp_spec_desc->addr;
  tmp_spec_desc->len  = strlen (tmp_spec);

  return (1);
}

/*
  Create a string with the current date/time. Omit 
  blanks, colons, dashes, and periods.
*/
unsigned int create_tmp_name (char *tmp_name)
{
  unsigned int        i, j;
  unsigned int        status;

  char                tim_buf_asc[23];
  struct desc_struct  tim_buf_desc;

  tim_buf_desc.len  = sizeof (tim_buf_asc);
  tim_buf_desc.addr = (char *) &tim_buf_asc;
  status = sys$asctim (0, &tim_buf_desc, 0, 0);
  check_status (status);

  for (j=0; j<tim_buf_desc.len; j++)
    if (tim_buf_asc[j] != MINUS  &&  tim_buf_asc[j] != BLANK  &&  tim_buf_asc[j] != PERIOD  &&  tim_buf_asc[j] != COLON)
    {*tmp_name = tim_buf_asc[j];
     tmp_name++; }

  *tmp_name = 0;

  return (1);
}

/*
  Process the command line using CLI$ routines. Note that a CLD file must be
  applied before this program is invoked.
*/
unsigned int parse_command_line (unsigned int        *before_flag, 
                                 unsigned int        *input_flag, 
                                 unsigned int        *log_flag, 
                                 unsigned int        *since_flag,
                                 unsigned int        *before_time,
                                 unsigned int        *since_time,
                                 unsigned int        *wild_flag,
                                 struct desc_struct  *input_file_spec_desc,
                                 struct desc_struct  *dir_spec_desc,
                                 struct desc_struct  *file_spec_desc)
{
  unsigned int   status, buff[2];

  unsigned int        file_len = 0;

  char                *log_str      = "LOG";
  char                *before_str   = "BEFORE";
  char                *since_str    = "SINCE";
  char                *input_str    = "INPUT";
  char                *spec_str     = "SPEC";
  char                *def_spec     = "SYS$DISK:[]*.*;*";

  struct desc_struct  before_desc, log_desc, input_desc, since_desc;
  struct desc_struct  before_time_desc, since_time_desc;
  struct desc_struct  spec_desc, spec_buff_desc;
  struct desc_struct  def_spec_desc, str_desc;

  char                before_time_asc[23], since_time_asc[23], spec_buff[255];

  struct FAB          spec_fab;
  struct NAM          spec_nam;
  char                esa[255] = {255*0};

  before_desc.len   = strlen (before_str);  before_desc.addr   = before_str;
  input_desc.len    = strlen (input_str);   input_desc.addr    = input_str;
  log_desc.len      = strlen (log_str);     log_desc.addr      = log_str;
  since_desc.len    = strlen (since_str);   since_desc.addr    = since_str;
  spec_desc.len     = strlen (spec_str);    spec_desc.addr     = spec_str;
  def_spec_desc.len = strlen (def_spec);    def_spec_desc.addr = def_spec;
  
  *before_flag = 0;  *input_flag  = 0;  *log_flag    = 0;  *since_flag  = 0;

  before_time_desc.len   = sizeof (before_time_asc);
  before_time_desc.addr  = (char *) &before_time_asc;
  since_time_desc.len    = sizeof (since_time_asc);
  since_time_desc.addr   = (char *) &since_time_asc;
  spec_buff_desc.len     = sizeof (spec_buff);
  spec_buff_desc.addr    = (char *) &spec_buff;

  if ((status = cli$present (&before_desc)) == CLI$_PRESENT) 
    { *before_flag = 1;
      status = cli$get_value (&before_desc, &before_time_desc, 0);
      check_status (status);
      status = sys$bintim (&before_time_desc, (struct _generic_64 *) before_time);
      check_status (status); }

  if ((status = cli$present (&input_desc))  == CLI$_PRESENT) 
    { *input_flag  = 1;
      status = cli$get_value (&input_desc, input_file_spec_desc, &file_len);
      check_status (status); 
      input_file_spec_desc->len  = file_len;}

  if ((status = cli$present (&log_desc))    == CLI$_PRESENT) *log_flag    = 1;

  if ((status = cli$present (&since_desc))  == CLI$_PRESENT)
    { *since_flag  = 1;
      status = cli$get_value (&since_desc, &since_time_desc, 0);
      check_status (status);
      status = sys$bintim (&since_time_desc, (struct _generic_64 *) since_time);
      check_status (status); }

  if ((status = cli$present (&spec_desc)) == CLI$_PRESENT)
    { status = cli$get_value (&spec_desc, &spec_buff_desc, &file_len);
      check_status (status);
      spec_buff_desc.len = file_len; }
  else spec_buff_desc.len = 0;

  spec_fab           = cc$rms_fab;
  spec_fab.fab$l_nam = &spec_nam;
  spec_fab.fab$l_fna = spec_buff_desc.addr;
  spec_fab.fab$b_fns = spec_buff_desc.len;
  spec_fab.fab$l_dna = def_spec;
  spec_fab.fab$b_dns = strlen (def_spec);
  spec_nam           = cc$rms_nam;
  spec_nam.nam$l_esa = (char *) &esa;
  spec_nam.nam$b_ess = sizeof (esa);
  status = sys$parse (&spec_fab, 0, 0);
  check_status (status);

  str_desc.addr = spec_nam.nam$l_dev;
  str_desc.len  = spec_nam.nam$b_dev + spec_nam.nam$b_dir;
  status = str$copy_dx (dir_spec_desc, &str_desc);
  check_status (status);
  dir_spec_desc->len = str_desc.len;

  str_desc.addr = spec_nam.nam$l_name;
  str_desc.len  = spec_nam.nam$b_name + spec_nam.nam$b_type + spec_nam.nam$b_ver;
  status = str$copy_dx (file_spec_desc, &str_desc);
  check_status (status);
  file_spec_desc->len = str_desc.len;

  *wild_flag = 1;
  if (strcmp (spec_nam.nam$l_name, "*.*;*") == 0) *wild_flag = 0;

  return (1);

}

/*
  If we were provided with a list of files (via the "/INPUT" qualifier),
  open the file that contains the list.
*/
unsigned int open_input_file ()
{
  unsigned int    status;

  input_file_fab           = cc$rms_fab;
  input_file_fab.fab$l_fna = input_file_spec_desc.addr;
  input_file_fab.fab$b_fns = input_file_spec_desc.len;
  status = sys$open (&input_file_fab, 0, 0);
  check_status (status);

  input_file_rab           = cc$rms_rab;
  input_file_rab.rab$l_fab = &input_file_fab;
  input_file_rab.rab$l_ubf = (char *) &input_file_rec;
  input_file_rab.rab$w_usz = sizeof (input_file_rec);
  status = sys$connect (&input_file_rab, 0, 0);
  check_status (status);

  status = sys$get (&input_file_rab, 0, 0);
  check_status (status);

  input_rec_desc.addr = (char *) &input_file_rec;
  input_rec_desc.len  = input_file_rab.rab$w_rsz;

  return (1);
}

/*
  If the "/INPUT" qualifier was specified, this routine
  will read the next file-spec in the list.
*/
unsigned int read_next_input_rec ()
{
  unsigned int     status;
  unsigned int     i;

  status = sys$get (&input_file_rab, 0, 0);
  if (status != RMS$_EOF) 
  { check_status (status);
    input_rec_desc.addr = (char *) &input_file_rec;
    input_rec_desc.len  = input_file_rab.rab$w_rsz;}
  else
  { for (i=0; i<sizeof(input_file_rec); i++) input_file_rec[i] = 0xFF;
    input_rec_desc.addr = (char *) &input_file_rec;
    input_rec_desc.len  = sizeof(input_file_rec);}

  return (1);
}

/*
  Perform a delete given the file-id.
*/
unsigned int delete_file (struct desc_struct *file_fib_desc, unsigned int chan)
{
  unsigned int   status;
  struct _iosb   iosb;

  files_deleted++;

  status = sys$qiow (0, chan, IO$_DELETE | IO$M_DELETE, &iosb, 0, 0, file_fib_desc, 0, 0, 0, 0, 0);
  check_status (status);
  check_status (iosb.iosb$w_status);

  return (1);
}

/*
  Call this routine periodically to report on the progress
  of the delete operation.
*/
void do_interval_reporting (void)
{
  unsigned int        status;
  unsigned short      tim_len;
  char                tim_asc[24];
  struct desc_struct  tim_asc_desc;

  tim_asc_desc.len  = sizeof (tim_asc);
  tim_asc_desc.addr = (char *) &tim_asc;

  status = sys$asctim (&tim_len, &tim_asc_desc, 0, 0);
  check_status (status);
  tim_asc[tim_len] = NULL;

  printf ("%s  deleted: %6d,  preserved: %6d,  total: %6d\n", 
          tim_asc, files_deleted, files_preserved, files_processed);

  interval_cnt = 0;
}
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          