/****************************************************************************
 * COMPRESS.H
 *
 *      This header file contains the function and constant declarations for
 *      the data compression functions in the COMPRESS.DLL dynamic link
 *      library.
 *
 *      Modifications : 05-Mar-94 : Initial version.
 ****************************************************************************/

#if !defined(OS2DEF_INCLUDED)
#error The header file OS2.H must be included before this header file.
#endif

#if !defined(_COMPRESS_INCLUDED)
#define _COMPRESS_INCLUDED

#ifdef __cplusplus
extern "C" {
#endif

/*****************************************************************************/
/* Definition of compression constants                                       */
/*****************************************************************************/
#define METHOD_STORE    0           /* store object with no compression */
#define METHOD_DEFLATE  1           /* use ZIP style deflation */

#define METHOD_BEST     255         /* use best available compression method */
/*****************************************************************************/


/*****************************************************************************/
/* Error constants                                                           */
/*****************************************************************************/
#define COMPRESS_INVALID_METHOD     ((APIRET)0xF0000000)
#define COMPRESS_INVALID_LEVEL      ((APIRET)0xF0000001)
#define COMPRESS_INVALID_PARAMETER  ((APIRET)0xF0000002)
#define COMPRESS_INTERNAL_ERROR     ((APIRET)0xF0000003)

#define EXPAND_INVALID_PARAMETER    ((APIRET)0xFF000000)
#define EXPAND_INVALID_METHOD       ((APIRET)0xFF000001)
#define EXPAND_INTERNAL_ERROR       ((APIRET)0xFF000002)
#define EXPAND_CRC                  ((APIRET)0xFF000003)
/*****************************************************************************/

/*****************************************************************************/
/* Definition of compressed data object data structure                       */
/*****************************************************************************/
#pragma pack(1)
typedef struct _COMPRESSED_DATA {
    ULONG  crc;                 /* 32 bit CRC of rest of structure + compressed data */
    ULONG  ulCompressedSize;    /* compressed size of data */
    ULONG  crcOriginal;         /* 32 bit CRC of the original uncompressed data */
    ULONG  ulOriginalSize;      /* original uncompressed size of data */
    USHORT version;             /* version of COMPRESS.DLL used to compress data */
    BYTE   method;              /* compression method */
    BYTE   reserved;            /* currently unused, set to 0 */
    BYTE   data[1];             /* compressed data starts here */
    } COMPRESSED_DATA;

typedef COMPRESSED_DATA * PCOMPRESSED_DATA;
#pragma pack()

#define COMPRESSED_HEADER_SIZE  ( sizeof(COMPRESSED_DATA) - 1 )     /* 20 bytes */
/*****************************************************************************/

/*****************************************************************************
 * Function declarations
 *****************************************************************************/
extern APIRET EXPENTRY CompressObject( PBYTE pData,
                                       ULONG dataSize,
                                       PCOMPRESSED_DATA * ppCompressed,
                                       BYTE method,
                                       BYTE level );
/* Description          This function compresses the designated memory object.
 *
 * Parameters           pData points to the data object that is to be compressed.
 *
 *                      dataSize contains the size in bytes of the input data object.
 *
 *                      ppCompressed points to the location where the pointer to
 *                      the compressed data object is to be stored. The memory
 *                      used to store the compressed data object is sparse allocated
 *                      by the DosAllocMem function.
 *
 *                      method contains the compression method to be used to
 *                      compress the data object. This parameter can be set to
 *                      one of the following values:
 *
 *                          METHOD_STORE    ..... store with no compression.
 *                          METHOD_DEFLATE  ..... use ZIP style deflation.
 *                          METHOD_BEST     ..... use best available method.
 *
 *                      level contains a value from 0 - 9 indicating the compression
 *                      level to be used. A value of 0 forces the data to be STORED.
 *                      A value of 1 gives fast but minimal compression. A value of
 *                      9 gives slow but maximal compression. Usually a compression
 *                      level of 5 provides a decent speed/compression tradeoff.
 *
 * Return Value         A return value of 0 indicates that no error occured, otherwise
 *                      CompressObject returns one of the following error codes:
 *
 *                      Possible OS/2 errors
 *                      --------------------
 *                      ERROR_NOT_ENOUGH_MEMORY
 *                      ERROR_INVALID_PARAMETER
 *                      ERROR_INTERRUPT
 *                      ERROR_INVALID_ADDRESS
 *                      ERROR_ACCESS_DENIED
 *                      ERROR_LOCKED
 *                      ERROR_CROSSES_OBJECT_BOUNDARY
 *
 *                      Compression library specific errors
 *                      -----------------------------------
 *                      COMPRESS_INVALID_METHOD     ..... The specified compression method
 *                                                        is not supported by this version of
 *                                                        the library.
 *
 *                      COMPRESS_INVALID_LEVEL      ..... The compression level parameter
 *                                                        contained an invalid value.
 *
 *                      COMPRESS_INVALID_PARAMETER  ..... The pData, dataSize, or ppCompressed
 *                                                        parameter contained an invalid
 *                                                        value.
 *
 *                      COMPRESS_INTERNAL_ERROR     ..... An internal error occured in the
 *                                                        compression function.
 */

extern APIRET EXPENTRY ExpandObject( PCOMPRESSED_DATA pData,
                                     PBYTE * ppExpanded );
/* Purpose              This function decompresses the designated compressed data
 *                      object.
 *
 * Parameters           pData points to the compressed data object to be
 *                      expanded.
 *
 *                      ppExpanded points to the location where the pointer to the
 *                      expanded data object is to be stored. The memory used to
 *                      store the expanded data object is allocated by the
 *                      DosAllocMem function.
 *
 * Return Value         A return value of 0 indicates that no error occured, otherwise
 *                      ExpandObject returns one of the following error codes:
 *
 *                      Possible OS/2 errors
 *                      --------------------
 *                      ERROR_NOT_ENOUGH_MEMORY
 *                      ERROR_INVALID_PARAMETER
 *                      ERROR_INTERRUPT
 *                      ERROR_ACCESS_DENIED
 *
 *                      Compression library specific errors
 *                      -----------------------------------
 *                      EXPAND_INVALID_PARAMETER    ..... The pData or ppExpanded parameters
 *                                                        contain a pointer to an invalid
 *                                                        address.
 *
 *                      EXPAND_INVALID_METHOD       ..... The compressed data object was
 *                                                        compressed using a method not
 *                                                        supported by this version of the
 *                                                        library.
 *
 *                      EXPAND_CRC                  ..... The contents of the input compressed
 *                                                        data object appear to have been
 *                                                        corrupted.
 *
 *                      EXPAND_INTERNAL_ERROR       ..... An internal error occurred in the
 *                                                        data expansion function.
 */

extern USHORT EXPENTRY CompressVersion( void );
/* Purpose              This function retrieves the currently installed version
 *                      of the data compression library.
 *
 * Return Value         The major version number is stored in the high order
 *                      byte. The minor version number is stored in the low
 *                      order byte.
 */
/*****************************************************************************/

#ifdef __cplusplus
}
#endif

#endif /* _COMPRESS_INCLUDED */
