/****************************************************************************

    PROGRAM: Dtc_ex.c

    PURPOSE: Demonstrates how to use the DoubleTalk C support library

    FUNCTIONS:

    DT_Init()        - Locates and initializes DoubleTalk
    DT_RomVer()      - Returns DoubleTalk's firmware version number
    DT_SerNum()      - Returns DoubleTalk's serial number
    DT_Sts()         - Returns the status of the TTS and LPC synthesizers
    DT_Intgt()       - Returns the current settings of the TTS synthesizer
    DT_Xlt()         - Converts a text string to a phoneme string
    DT_Say()         - Speaks a text string through the TTS synthesizer
    DT_Lpc()         - Speaks one or more words from an LPC data file
    DT_LpcRpt()      - Repeats the last output LPC word or phrase
    DT_Pcm()         - Plays one or more sounds from a PCM/ADPCM data file
    DT_Tgn1()        - Plays back a tone generator data file

    COMMENTS:

    All library functions use standard C calling conventions. See dtc.h
    file for function prototypes.

    To compile and link with Microsoft C:
      cl /c /AM dtc_ex.c
      link dtc_ex.obj,dtc_ex.exe,,dtc.lib;

    To compile and link with Borland C:
      bcc -c -ml -l-c dtc_ex.c
      bcc -ml -l-c dtc_ex.obj dtc.lib


****************************************************************************/


#include <stdio.h>
#include "dtc.h"

#define MAXLPCDATLEN   8000         /* 8000 bytes for lpc data area */
#define MAXPCMDATLEN  30000         /* 30000 bytes for pcm data area */
#define MAXTGNDATLEN    200         /* 200 bytes for tgn data area */

#define NORMSPD           0         /* LPC playback speeds */
#define FASTSPD           1

#define BUFOFF            0         /* TTS and LPC buffer control */
#define BUFON             1

char lpcdata[MAXLPCDATLEN];         /* area for loading lpc data */
char pcmdata[MAXPCMDATLEN];         /* area for loading pcm data */
char tgndata[MAXTGNDATLEN];         /* area for loading tgn data */

short dtparms[10];                  /* holds results of DT_Intgt */
short lpclist[20];                  /* arrays used for passing word & sound */
short pcmlist[20];                  /*  indexes to lpc/pcm functions */


FILE *fd;


void main()
{

/*  ==== Misc examples ====  */
/*

DT_Init

Syntax: unsigned short _far DT_Init( void );

  This function initializes the DTC library and DoubleTalk for use with your
  applications. Any application program that uses DTC must use this function
  prior to calling any other DTC function.

  DT_Init determines the base I/O address of DoubleTalk (set by the jumper
  block on DoubleTalk PC, or the COM port DoubleTalk LT is connected to),
  which is used by most of the other library functions. If a DoubleTalk was
  found, its command character is reset to ^A, then initialized with the ^A@
  command. DoubleTalk's base I/O (or COM port) address is returned by the
  function, which can be used to determine whether or not a DoubleTalk was
  found, and/or to perform direct hardware I/O with DoubleTalk. The function
  returns a value of zero if a DoubleTalk was not found.

  Example:
*/

        printf( "DoubleTalk base address: %Xh\n", DT_Init() );




/*

DT_RomVer

Syntax: char _far * _far DT_RomVer( void );

  This function returns DoubleTalk's ROM version as a char string.

  Example:
*/

        printf( "DoubleTalk ROM version: %s\n", DT_RomVer() );




/*

DT_SerNum

Syntax: unsigned short _far DT_SerNum( void );

  This function returns DoubleTalk's serial number as an integer. The
  function is useful for determining if the DoubleTalk being used supports
  newer features not supported in earlier models, and for "locking" a
  program to run on just one computer.

  Example:
*/

        printf( "DoubleTalk serial number: %u\n", DT_SerNum() );




/*

DT_Sts

Syntax: unsigned short _far DT_Sts( void );

  This function returns the status of DoubleTalk's TTS and LPC synthesizers.
  The status information can be used by an application program to determine
  when either synthesizer is finished speaking, allowing other events (such
  as animation) to be synchronized with the speech. The TTS status is returned
  in the lower eight bits of the function return value, with the LPC status in
  the upper eight bits. The TTS flags are also valid in the PCM and TGN modes
  (see "readme" file for status flag definitions).

  Example:
*/

        printf( "TTS status: %Xh   ", DT_Sts() & 0xFF );
        printf( "LPC status: %Xh\n", DT_Sts() >> 8 );




/*

DT_Intgt

Syntax: void _far DT_Intgt( short _far *array );

  This function returns the current settings of DoubleTalk's TTS synthesizer
  in an 11-element integer array. The order in which the values are returned
  is shown in the table below. The table also shows the command(s) that
  control each parameter, and the parameter's range (as returned by the
  DT_Intgt routine). For detailed command descriptions, refer to the
  DoubleTalk User's Manual.

  Parameter        Command   Range
  =========        =======   =====
  Mode             C/D/T     0=Character; 1=Phoneme; 2=Text
  Punc level       mB        0-7
  Formant freq     mF        0-9
  Pitch            mP        0-99
  Speed            mS        0-9
  Volume           mV        0-9
  Tone             mX        0-2
  Intonation       E/M       0=disabled (monotone); 1=enabled
  Exc Dict loaded  L         1=exception dictionary loaded; 0 otherwise
  Exc Dict status  U         1=exception dictionary enabled; 0 otherwise
  Free RAM         L         # pages (truncated) remaining for text buffer -
                             function of dictionary size and RAM chip installed

  Example:
*/

        DT_Intgt( dtparms );
        printf( "Mode     = %d\n"
                "Punc Lvl = %d\n"
                "Fmt Freq = %d\n"
                "Pitch    = %d\n"
                "Speed    = %d\n"
                "Volume   = %d\n"
                "Tone     = %d\n",
                dtparms[0], dtparms[1], dtparms[2], dtparms[3],
                dtparms[4], dtparms[5], dtparms[6] );




/*  ==== TTS examples ====  */
/*

DT_Say

Syntax: void _far DT_Say( char _far *string, short bufctl );

  As its name implies, DT_Say literally speaks the string 'string'. DoubleTalk
  commands can also be issued by sending them as text strings by themselves,
  or embedding them within the text of a string. Bufctl is an integer value of
  0 or 1 which disables or enables DoubleTalk's hardware TTS data buffer,
  which allows the synthesizer to speak in the background. (The text buffer
  is not actually disabled in the hardware; rather, the TTS driver simply
  waits for the TTS synthesizer to finish speaking before returning control.)

  Example:
*/

        DT_Say( "The speech you are now hearing is from "
                "the DoubleTalk speech card.", BUFON );




/*

DT_Xlt

Syntax: char _far * _far DT_Xlt( char _far *string );

  This function converts ASCII text string 'string' into a phoneme string.
  Note that the phoneme string returned is in tokenized form (that used
  internally by DoubleTalk). The program txtphm.bas demonstrates how the
  tokenized phoneme string can be decompiled into the standard ASCII format.
  NOTE: You must make certain that DoubleTalk is in Text mode before using
  this function. If you want the exception dictionary to be used in the
  translation, issue the command ^AU before using DT_Xlt (see example).

  Example:
*/

        DT_Say( "\1T\1U", BUFON );           /* force Text mode (enabling
                                                 dictionary is optional) */
        printf( "The phonetics for 'medicine' is %s\n", DT_Xlt( "medicine" ) );




/*  ==== LPC examples ====  */
/*

DT_Lpc

Syntax: void _far DT_Lpc( char _far *data_array, short _far *index_array,
                          short speed, short bufctl );

  This function activates DoubleTalk's LPC synthesizer. Data_array is a char
  array containing the LPC data file. Index_array holds an indexed list of
  words and phrases to be spoken from the LPC data file. Speed is an integer
  of value 0 or 1 which determines the output speed (normal or fast). Bufctl
  is an integer value of 0 or 1 which disables or enables DoubleTalk's
  hardware LPC data buffer, which allows the synthesizer to speak in the
  background for up to 30 seconds without any intervention from the
  application program. (Disabling the buffer does not actually disable the
  hardware buffer; rather, the LPC driver simply waits for the synthesizer to
  finish talking before returning control. Also note that the hardware buffer
  only buffers one word or phrase at a time; it does not queue multiple
  indexes. The setting of bufctl does not affect the operation of the
  DT_LpcRpt function.)

  Index values are limited to the range 1 thru 16383 decimal. DT_Lpc returns
  control when it encounters an index number of 0. Refer to the"readme" file
  on this disk for information about LPC file data formats.

  Example:
*/

        fd = fopen( "words2.lpc", "rb" );    /* load LPC data file */
        fread( lpcdata, 1, MAXLPCDATLEN, fd );
        fclose( fd );

        lpclist[0] = 31, lpclist[1] = 32;    /* define word no's to speak */
        lpclist[2] = 35, lpclist[3] = 35;     
        lpclist[4] = 1, lpclist[5] = 2;
        lpclist[6] = 3, lpclist[7] = 15;
        lpclist[8] = 19, lpclist[9] = 4;
        lpclist[10] = 0;
    
        DT_Lpc( lpcdata, lpclist, NORMSPD,   /* say the list at normal */
                BUFON );                     /*  speed with buffering */




/*
DT_LpcRpt

Syntax: void _far DT_LpcRpt( short speed, short bufctl );

  The repeat function causes the last word or phrase sent to the LPC
  synthesizer to be repeated. This may seem to be somewhat of a useless
  function, since the application program could simply call the LPC function
  again with the same word/phrase number. The repeat function, however,
  causes DoubleTalk to repeat the last word or phrase in its hardware buffer
  (sort of a "replay"). This speeds up program execution because the
  application doesn't have to re-send the LPC data to DoubleTalk. The values
  of speed and bufctl need not be the same values used in the original DT_Lpc
  call.

  Example:
*/

        DT_LpcRpt( FASTSPD, BUFOFF );




/*  ==== PCM example ====  */
/*

DT_Pcm

Syntax: void _far DT_Pcm( char _far *data_array, short _far *index_array,
                          short rate );

  This function activates DoubleTalk's PCM synthesizer. Data_array is a char
  array containing the PCM data file. Index_array holds an indexed list of
  sounds to be played from the PCM data file. Rate is an integer of value 0
  to 99 decimal which determines the output sampling rate fs, based on the
  following relationship:

        fs = 617/(155 - rate)   (kHz)

  Thus, sampling rates from 4 kHz (rate = 0) to 11 kHz (rate = 99) can be
  supported. The playback volume can be adjusted in 10 steps, using the TTS
  volume command.
  
  Normally, the sampling rate should be set to the same rate the PCM data
  was originally sampled at, but playing it back faster or slower can have
  some pretty interesting effects!

  Index values are limited to the range 1 thru 255 decimal. Both PCM and
  ADPCM (compressed) formats are supported - refer to the "readme" file on
  this disk for file format information. Control is returned when DT_Pcm
  encounters a sound index number of 0.

  Example:
*/

        fd = fopen( "portal.pcm", "rb" );    /* load PCM data file */
        fread( pcmdata, 1, MAXPCMDATLEN, fd );
        fclose( fd );

        pcmlist[0] = 1, pcmlist[1] = 0;      /* define sounds to play */

        DT_Pcm( pcmdata, pcmlist, 78 );      /* play them at 8 kHz rate */




/*  ==== TGN example ====  */
/*

DT_Tgn1

Syntax: void _far DT_Tgn1( char _far *data_array, short volume,
                          short tempo);

  This function activates DoubleTalk's tone generators. Data_array is a char
  array containing the TGN data file. Volume determines the volume level at
  which the tone file will be played, between 1 and 255. Tempo sets the speed
  of play, between 1 and 65535.

  Example:
*/

        fd = fopen( "groovy.tgn", "rb" );    /* load TGN data file */
        fread( tgndata, 1, MAXTGNDATLEN, fd );
        fclose( fd );

        DT_Tgn1( tgndata, 200, 86 );         /* volume = 200, tempo = 86 */

}
