; DT_TTS.ASM
;
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;                               NOTICE                                      ;
;                                                                           ;
; This software is proprietary to RC Systems, Inc., and is provided as a    ;
; development tool for third party software developers who are developing   ;
; software for the DoubleTalk speech synthesizers. Disclosure of the source ;
; or object code in any form to any other party is strictly prohibited.     ;
;                                                                           ;
;                 Copyright (C) 1992 RC Systems, Inc.                       ;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
;
; Description and use:
;
; DT_Init
;    This routine finds and initializes DoubleTalk.
;
;    The routine first looks for an external DoubleTalk (DoubleTalk
;    LT), returning in AX the base address of the COM port that
;    it is connected to. Looking for the external unit first allows
;    systems containing both internal and external DoubleTalks to
;    work with either, if the user simply turns off the external
;    unit's power. (If the internal card were searched for first and
;    found, the external unit would never be checked for.)
;
;    If the external unit is not found, the internal card (DoubleTalk
;    PC) is checked for. On return, the AX register is set to the
;    base (LPC port) address, or 0 if the internal card wasn't found
;    at any of the six valid base addresses (set by the jumper block
;    on the card).
;
;    NOTE: DoubleTalk PC/104, an OEM version of DoubleTalk PC for the
;    PC/104 bus standard, is functionally identical to DoubleTalk PC
;    except that it does not have an LPC synthesizer. What would be
;    the LPC port in the PC/104 is mapped to the TTS port, i.e.,
;    accesses to the LPC port will actually access the TTS port in
;    the DoubleTalk PC/104. For example, if an attempt is made to read
;    the LPC port status, the TTS port status will be returned; the
;    LPC and TTS status port values will always be the same.
;
;    To enable application programs to detect the PC/104 card, the
;    DT_Init routine sets the LPC_Port address variable to 0 if it
;    determined DoubleTalk PC/104 is present. The AX register is still
;    set to the LPC/TTS port address, so that the calling program won't
;    mistakenly think DoubleTalk is not present.
;
; DT_Tts
;    This routine outputs ASCII text and commands to DoubleTalk's
;    TTS synthesizer via the AH register. The routine can also be
;    used to output tone generator and PCM data (e.g., everything but
;    LPC data; see the DT_LPC.ASM file for information about using
;    DoubleTalk's LPC synthesizer).
;
; DT_Rd
;    This routine is used by DT_INTGT.ASM and DT_XLT.ASM for reading
;    data out of DoubleTalk.
;
;
;
	  PUBLIC  LPC_Port, TTS_Port, DT_Init, DT_Tts, DT_Rd
	  PUBLIC  CnfgCom, Com_Rd, Com_Wr, Internal

	  .DATA
Cmd0      DB      1eh,1,'@',0   ; reset cmd char & reinitialize
C0Len     EQU     $-Cmd0
LPC_Port  DW      0             ; base address of DTPC, or COM port if DTLT
TTS_Port  DW      0             ;  [LPC_Port] + 1 if DTPC, COM port if DTLT
Internal  DB      0             ; 1 if internal (PC) card; 0 if external (LT)


	  .CODE
; *********************************************************
;         DTInit: Find and initialize internal/external DT
;
;         Input: nothing
;         Output: AX = DTPC base (LPC) port address, or
;                      DTLT COM port base address (THR/RBR)
;                      0 means neither found
; *********************************************************

DT_Init   PROC    FAR

	  cmp     BYTE PTR TTS_Port,0
	  jne     DI0           ; already found; just reinitialize

	  call    FindExt       ; check for external first
	  cmp     BYTE PTR TTS_Port,0
	  jne     DI0

	  call    FindInt       ; external not found; check for internal
	  cmp     BYTE PTR TTS_Port,0
	  je      DI2           ; found neither
	  mov     Internal,1    ; found internal

DI0:      mov     si,OFFSET Cmd0 ; send initialization string to DT,
	  mov     cx,C0Len      ; just in case previous application
DI1:      mov     ah,[si]       ; screwed things up
	  inc     si
	  call    FAR PTR DT_Tts
	  loop    DI1

DI2:      mov     ax,LPC_Port   ; return DT address to caller
	  or      ax,ax         ; ck for PC/104
	  jnz     DI3
	  mov     ax,TTS_Port   ; it is
	  or      ax,ax
	  jz      DI3
	  dec     ax
DI3:      ret

DT_Init   ENDP



; *********************************************************
;         DT_Tts: Write a character to TTS port
;
;         Input: AH = byte to output (text, PCM data, etc.)
;         Output: nothing
; *********************************************************

DT_Tts    PROC    FAR

	  mov     dx,TTS_Port
	  cmp     Internal,1
	  jne     DT10

DT0:      in      al,dx         ; (internal)
	  test    al,10h        ; make sure DT is ready (RDY = 1)
	  jz      DT0
	  mov     al,ah
	  pushf                 ; save 'rupts mask
	  cli                   ; disable 'rupts
	  out     dx,al
DT1:      in      al,dx         ; wait for acknowledgement (RDY = 0)
	  test    al,10h        ; (10 uS to 50 uS after write)
	  jnz     DT1
	  popf                  ; restore 'rupts mask
	  ret

DT10:     push    ax            ; (external)
	  call    Com_Wr
	  pop     ax
	  ret

DT_Tts    ENDP



; *********************************************************
;         DT_Rd: Read a character from TTS port
;
;         Input: nothing
;         Output: AL = character read
; *********************************************************

DT_Rd     PROC    FAR

	  mov     dx,TTS_Port
	  cmp     Internal,1
	  jne     DR10

DR0:      in      al,dx         ; (internal)
	  or      al,al
	  jns     DR0
	  and     al,7fh
	  push    ax
	  pushf
	  cli
	  out     dx,al
DR1:      in      al,dx
	  or      al,al
	  js      DR1
	  popf
	  pop     ax
	  ret

DR10:     call    Com_Rd        ; (external)
	  and     al,7fh
	  ret

DT_Rd     ENDP



; *********************************************************
;         FindExt: Looks for external DT (DTLT)
;
;         Input: nothing
;         Output: nothing
; *********************************************************

FindExt   PROC    NEAR

	  mov     ax,40h        ; check BIOS Data Area for
	  mov     es,ax         ; installed port addresses
	  xor     si,si

FE0:      mov     dx,es:[si]    ; try next COM port (1-4)
	  mov     ax,dx
	  and     ax,0feefh     ; valid bases are 3f8, 2f8, 3e8, 2e8
	  cmp     ax,2e8h       ; does port exist?
	  jne     FE1
	  call    TryComN       ; yes: try it
	  cmp     BYTE PTR TTS_Port,0
	  jne     FE2           ; found it!
FE1:      inc     si
	  inc     si
	  cmp     si,6
	  jna     FE0
FE2:      ret

FindExt   ENDP



; *********************************************************
;         FindInt: Looks for internal DT (DTPC)
;
;         Input: nothing
;         Output: nothing
; *********************************************************

FindInt   PROC    NEAR

	  mov     dx,25eh       ; [nnn] = 7fh; [nnn+1] = 10h;
FI0:      in      ax,dx         ;  search from 25eh to 39eh step 40h
	  and     ax,0fbffh     ; mask AE flag for compatibility w/older cards
	  cmp     ax,107fh
	  je      FI1           ; DTPC (107fh)
	  and     al,0fbh
	  cmp     ax,1010h
	  je      FI2           ; PC/104 (1010h)
	  add     dx,40h
	  cmp     dx,3deh
	  jb      FI0
	  jae     FI3           ; DT not found
FI1:      mov     LPC_Port,dx   ; found it!
FI2:      inc     dx
	  mov     TTS_Port,dx
FI3:      ret

FindInt   ENDP



; *********************************************************
;         TryComN: Attempts to find external DT on COM port
;
;         Input: DX = base address of COM port to try
;         Output: nothing
; *********************************************************

TryComN   PROC    NEAR

	  mov     LPC_Port,dx   ; in case it works
	  mov     TTS_Port,dx

	  inc     dx            ; save port's settings:
	  in      ax,dx
	  push    ax            ; IER/IIR
	  inc     dx
	  inc     dx
	  in      ax,dx
	  push    ax            ; LCR/MCR
	  mov     al,83h
	  out     dx,al
	  sub     dx,3
	  in      ax,dx
	  push    ax            ; baud rate

	  mov     ah,12         ; configure COM port (9600 baud)
	  call    CnfgCom
	  xor     al,al         ; ID function within DTLT
	  mov     dx,TTS_Port
	  out     dx,al
	  call    Com_Rd        ; listen for response...
	  cmp     al,'C'+80h    ; 1st char = 0c3h (always)
	  jne     TC1
	  call    Com_Rd
	  cmp     al,'A'+80h    ; 2nd char = version letter (A-Z) + 80h
	  jb      TC1
	  call    Com_Rd
	  cmp     al,8dh        ; 3rd (last) char = 8dh (always)
	  jne     TC1
	  add     sp,6          ; connected to this port; leave port
	  ret                   ; configured for DTLT

TC1:      mov     dx,TTS_Port   ; not connected to this port
	  xor     ax,ax
	  mov     LPC_Port,ax
	  mov     TTS_Port,ax
	  add     dx,3
	  mov     al,83h
	  out     dx,al
	  sub     dx,3
	  pop     ax
	  out     dx,ax         ; restore baud rate
	  add     dx,3
	  pop     ax
	  out     dx,ax         ; restore LCR/MCR
	  dec     dx
	  dec     dx
	  pop     ax
	  out     dx,ax         ; restore IER/IIR
	  ret

TryComN   ENDP



; *********************************************************
;         CnfgCom: Configures COM port for DTLT
;
;         Input: AH = baud rate divisor
;         Output: nothing
; *********************************************************

CnfgCom   PROC    NEAR

	  cmp     Internal,1
	  je      SB1           ; branch if internal

	  mov     dx,TTS_Port   ; wait for any chars to be
	  add     dx,5          ;  transmitted before changing
CWt:      in      al,dx
	  and     al,60h
	  cmp     al,60h
	  jne     CWt

	  sub     dx,4          ; IER
	  xor     al,al         ; no 'rupts
	  out     dx,al
	  inc     dx
	  inc     dx            ; LCR
	  mov     al,83h
	  out     dx,al
	  sub     dx,3          ; RX/TX (baud rate)
	  mov     al,ah
	  xor     ah,ah
	  out     dx,ax
	  add     dx,3          ; LCR
	  mov     al,3          ; n,8,1
	  out     dx,al
	  inc     dx            ; MCR
	  mov     al,3          ; RTS, DTR active
	  out     dx,al
	  sub     dx,4
	  in      al,dx         ; clear rx hold register
	  push    cx            ; wait for new divisor to settle
	  mov     cx,3000
SB0:      loop    SB0
	  pop     cx
SB1:      ret

CnfgCom   ENDP



; *********************************************************
;         Com_Wr: Write byte to COM port
;
;         Input: AH = character to output
;         Output: nothing
; *********************************************************

Com_Wr    PROC    NEAR

	  mov     dx,TTS_Port
	  add     dx,5          ; LSR
CW1:      in      al,dx
	  test    al,20h        ; wait for THR empty
	  jz      CW1
	  inc     dx            ; MSR
CW2:      in      al,dx
	  test    al,20h        ; wait for DTLT ready (DSR line)
	  jz      CW2
	  sub     dx,6
	  mov     al,ah
	  out     dx,al
	  ret

Com_Wr    ENDP



; *********************************************************
;         Com_Rd: Read byte from COM port
;
;         Input: nothing
;         Output: AL = character read (0dh if timeout)
; *********************************************************

Com_Rd    PROC    NEAR

	  push    cx
	  mov     dx,TTS_Port
	  add     dx,5          ; LSR
	  xor     cx,cx         ; timeout timer
CR1:      dec     cx
	  mov     al,0dh
	  jz      CR2
	  in      al,dx
	  test    al,1          ; char received?
	  jz      CR1
	  sub     dx,5          ; yes: get it
	  in      al,dx
CR2:      pop     cx
	  ret

Com_Rd    ENDP
