(*********** Microsoft Pascal NI-488 Example Program - Board Level *************)

(* This sample program is for reference only. It can only be expected to *)
(* function with a Fluke 8840 Digital Multimeter that has been properly  *)
(* configured in IBCONF.						 *)


program main (input,output);

{$include: 'decl.pas'}

{* Variable declarations taken from DECL.PAS:				     *}

var
  cmd	 : cbuf;		      (* command buffer 		     *)
  mask	 : integer;		      (* events to be waited for	     *)
  rd	 : cbuf;		      (* read data buffer		     *)
  wrt	 : cbuf;		      (* write data buffer		     *)
  bname  : nbuf;		      (* device name buffer		     *)
  bd	 : integer;		      (* device number			     *)


(*****************************************************************************)
(* This routine would notify you that the IBFIND call failed, and refer you  *)
(* to the driver software configuration procedures.                          *)
(*****************************************************************************)

procedure finderr;

begin
  writeln (' Find error');
end;

(*****************************************************************************)
(* This routine would, among other things, check IBERR to determine the      *)
(* exact cause of the error condition & then take action appropriate to the  *)
(* application.  For errors during data transfers, IBCNT may be examined to  *)
(* determine the actual number of bytes transferred.                         *)
(*****************************************************************************)

procedure error;

begin
  writeln (' Error');
  writeln ('ibsta=',ibsta,'iberr=',iberr,'ibcnt=',ibcnt);
end;

(*****************************************************************************)
(* This routine would analyze the fault code returned in the device's status *)
(* byte and then take appropriate action.                                    *)
(*****************************************************************************)

procedure deverr;

begin
  writeln (' device error');
  writeln ('ibsta=',ibsta,'iberr=',iberr,'ibcnt=',ibcnt);
end;

(*****************************************************************************)

begin

(* Assign a unique identifier to the interface board "GPIB0":                *)

  bname:= 'gpib0   ';
  bd := ibfind (bname);

(* Check for IBFIND error:						     *)

  if (bd < 0) then finderr;

(* Send the Interface Clear (IFC) message to all devices:                    *)

  ibsic (bd);

(* Check for an error on each GPIB call to be safe...                        *)

  if ((ibsta AND ERR) <> 0) then error;

(* Turn on the Remote Enable (REN) signal:                                   *)

  ibsre (bd,1);
  if ((ibsta AND ERR) <> 0) then error;

(* Inhibit front panel control with the Local Lockout (LLO) command (hex 11).	*)
(* Place the device in remote mode by addressing it to listen. The listen	*)
(* address of device three is its primary address, 3, plus hex 20.  This is an	*)
(* ASCII "#".  Send the Device Clear (DCL) message (hex 14) to clear internal	*)
(* device functions.  Finally, address the GPIB0 to talk by sending its talk	*)
(* address, the pad 0 plus hex 40, or ASCII "@".  These commands can be found	*)
(* in Appendix A of the Software Reference Manual.				*)

  cmd[1] := chr(LLO);
  cmd[2] := '#';
  cmd[3] := chr(DCL);
  cmd[4] := '@';
  ibcmd (bd,cmd,4);
  if ((ibsta AND ERR) <> 0) then error;

(* Write the function, range, and trigger source instructions to the device:	*)
(* This string is meaningful to the Fluke 8840 Digital Multimeter.	     *)

  wrt[1] := 'F';
  wrt[2] := '3';
  wrt[3] := 'R';
  wrt[4] := '7';
  wrt[5] := 'T';
  wrt[6] := '3';
  ibwrt (bd,wrt,6);
  if ((ibsta AND ERR) <> 0) then error;

(* Send the GET (hex 8) message to trigger a measurement reading:	    *)

  cmd[1] := chr(GET);
  ibcmd (bd,cmd,1);
  if ((ibsta AND ERR) <> 0) then error;

(* Wait for the DVM to set SRQ (hex 1000) or for a timeout (hex 4000).	    *)
(* These status bits are listed in Section Four of the Software Reference   *)
(* manual.  If the current time limit is too short, use IBTMO to change it. *)

  mask := TIMO OR SRQI;
  ibwait (bd,mask);
  if ((ibsta AND (TIMO OR ERR)) <> 0) then error;

(* If neither a timeout nor an error occurred, IBWAIT must have returned       *)
(* on SRQ.  Next do a serial poll.  First unaddress bus devices by sending     *)
(* the untalk (UNT) command (ASCII "_", or hex 5F) and the unlisten (UNL)      *)
(* command (ASCII "?", or hex 3F).  Then send the Serial Poll Enable (SPE)     *)
(* command (hex 18) and the DVM's talk address (device three's pad, 3, plus    *)
(* hex 40, or ASCII "C") and the GPIB0 listen address (pad 0 plus hex 20,      *)
(* or ASCII space).  These commands can be found in Appendix A of the Software *)
(* Reference Manual.							       *)

  cmd[1] := chr(UNT);
  cmd[2] := chr(UNL);
  cmd[3] := chr(SPE);
  cmd[4] := 'C';
  cmd[5] := ' ';
  ibcmd (bd,cmd,5);
  if ((ibsta AND ERR) <> 0) then error;

(* Now read the status byte. If it is 16#C0, the device has valid data to    *)
(* send, otherwise it has a fault condition to report.			     *)

  ibrd (bd,rd,1);
  if ((ibsta AND ERR) <> 0) then error;
  if (ord(rd[1]) <> 16#C0) then deverr;

(* If more than one device were attached to the bus, it would be necessary to *)
(* explicitly check the &H40 bit of the device status word to be sure that    *)
(* another device had not been responsible for asserting SRQ.  Complete the   *)
(* serial poll by sending the Serial Poll Disable (SPD) message (hex 19).     *)

  cmd[1] := chr(SPD);
  ibcmd (bd,cmd,1);
  if ((ibsta AND ERR) <> 0) then error;

(* Since the device and GPIB board are still addressed to talk and listen,   *)
(* the measurement can be read as follows:				     *)

  ibrd (bd,rd,16);
  if ((ibsta AND ERR) <> 0) then error;

(* To close out a programming sequence, send IFC to initialize the bus and   *)
(* call the IBONL function to disable the hardware and software:            *)

  ibsic (bd);
  ibonl (bd,0);

end.

