/* C Example Program - NI Device Level Functions       */
/*
  This sample program is for reference only and can only be expected to
  function with a Fluke 8840 Digital Multimeter that has been properly
  configured in IBCONF.
*/
#define INCL_DOSFILEMGR
#define INCL_DOSDEVICES
#include <fcntl.h>
#include <stdio.h>
#include <os2.h>
#include "nicode.h"

#define DISABLE_ERRORPOPUPS 0
void gpiberr(HFILE dvm, char *msg);


main()
{
	HFILE	dvm;			   /* device handle	     */
   ULONG ActionTaken;
	APIRET	RetCode;		   /* API call return code   */
	int	cnt;			   /* bytes read	     */
	short	spr;		/* serial poll response   */
	char	rd[16]; 		   /* buffer for read	     */
	nidev	nid;		      /* device config structure*/
	short	mask ;	      /* wait mask 	     */
   ULONG BytesWritten;
   ULONG	ParmLength;
   ULONG DataLength;


/* Assign a unique identifier to the DVM ("Digital Voltmeter") and store in
 * variable dvm. Exit on error.
 */
   RetCode = DosOpen("DVM",  &dvm,  &ActionTaken,
                 0L,          /* file's new size */
                 FILE_READONLY,   /* file attribute is read only */
                 OPEN_ACTION_OPEN_IF_EXISTS,   /* open the file if it already exists */
                 OPEN_FLAGS_WRITE_THROUGH | OPEN_FLAGS_FAIL_ON_ERROR | 
           		  OPEN_FLAGS_NOINHERIT | OPEN_SHARE_DENYNONE | 
					  OPEN_ACCESS_READWRITE,
                 0);
	if (RetCode != 0) {
		printf("Unable to open device");
		exit(1);
	}

/* Clear the device, check for error.
 */
   ParmLength = 0;
   DataLength = 0;
	RetCode = DosDevIOCtl(dvm,CATEGORY,DCLEAR,0,ParmLength,&ParmLength,0,DataLength,&DataLength);
   if (RetCode != 0) {
     gpiberr(dvm,"DCLEAR Error");
   }

/* Write the function, range, and trigger source instructions to the device.
 * This string is meaningful to the Fluke 8840 Digital Multimeter.
 */
   BytesWritten = 0;
	RetCode = DosWrite(dvm,"F3R7T3",6, &BytesWritten);
   if (RetCode != 0) {
     gpiberr(dvm,"DosWrite Error");
   }

/* Trigger the device.
 */
   ParmLength = 0;
   DataLength = 0;
	RetCode = DosDevIOCtl(dvm,CATEGORY,TRIGGER,0,ParmLength,&ParmLength,0,DataLength,&DataLength);
   if (RetCode != 0)	{
     gpiberr(dvm,"TRIGGER Error");
   }

/* Change the device timeout & enable GPIB errors.
 */

/* Wait for the DVM to Request Service (BRQS) or for a timeout (BTIMO).
 * These status constants are defined in NICODE.H.
 */
   mask = BTIMO | BRQS;

   ParmLength = sizeof(mask);
   DataLength = 0;
	RetCode = DosDevIOCtl(dvm,CATEGORY,BWAIT,&mask,ParmLength,&ParmLength,0,DataLength,&DataLength);
   if (RetCode != 0)	 {
     gpiberr(dvm,"BWAIT Error");
   }

/*  Next, serial poll the device.
 */

   ParmLength = sizeof(spr);
   DataLength = 0;
	RetCode = DosDevIOCtl(dvm,CATEGORY,SPOLL,&spr,ParmLength,&ParmLength,0,DataLength,&DataLength);
   if (RetCode != 0)	 {
     gpiberr(dvm,"SPOLL Error");
   }

/* Now test the status byte (response).  If response is 0xC0, the device has
  valid data to send; otherwise, it has a fault condition to report.
*/

	if (spr != 0xC0) {
      printf("Fluke has Fault condition.");
   }

/*
 * If the data is valid, read the measurement.
 */
   BytesWritten = 0;
	RetCode = DosRead(dvm, rd, 16, &BytesWritten);
   if (RetCode != 0)	 {
     gpiberr(dvm,"DosRead Error");
   }
   /* Call OFFLINE to disable the hardware and software. */
   ParmLength = 0;
   DataLength = 0;
	RetCode = DosDevIOCtl(dvm, CATEGORY, OFFLINE, 0, ParmLength, &ParmLength,
                         0, DataLength, &DataLength); 


  /* Return the handle to the system. */
	DosClose(dvm);
}

/*
 * ===========================================================================
 *                      Function GPIBERR
 *  This function will notify you that a API function failed by
 *  printing an error message.  The status variable status will also be
 *  printed in hexadecimal along with the mnemonic meaning of the bit position. 
 *  The status variable error will be printed in decimal along with the
 *  mnemonic meaning of the decimal value.  
 *
 *  The API function OFFLINE is called to disable the hardware and software.
 *
 *  The EXIT function will terminate this program.
 * ===========================================================================
 */

void gpiberr(HFILE dvm, char *msg)
{
   unsigned short status;
   unsigned short error;
   unsigned short error;
   APIRET RetCode;
   ULONG ParmLength;
   ULONG DataLength; 

   printf("%s\n", msg);
   ParmLength = sizeof(status);
   DataLength = 0;
   RetCode = DosDevIOCtl(dvm, CATEGORY, STATUS, &status, ParmLength, &ParmLength,
                         0, DataLength, &DataLength);

   printf("status = 0x%x   <", status);
    if (status & BERR )  printf (" ERR");
    if (status & BTIMO)  printf (" TIMO");
    if (status & BEND )  printf (" END");
    if (status & BSRQI)  printf (" SRQI");
    if (status & BRQS )  printf (" RQS");
    if (status & BCMPL)  printf (" CMPL");
    if (status & BLOK )  printf (" LOK");
    if (status & BREM )  printf (" REM");
    if (status & BCIC )  printf (" CIC");
    if (status & BATN )  printf (" ATN");
    if (status & BTACS)  printf (" TACS");
    if (status & BLACS)  printf (" LACS");
    if (status & BDTAS)  printf (" DTAS");
    if (status & BDCAS)  printf (" DCAS");
    printf (" >\n");             

   /* The lower byte of the return code holds the GPIB specific error. */
    error = RetCode & 0x00FF ;
    if (error == EDVR) printf (" EDVR <DOS Error>\n");
    if (error == ECIC) printf (" ECIC <Not CIC>\n");
    if (error == ENOL) printf (" ENOL <No Listener>\n");
    if (error == EADR) printf (" EADR <Address error>\n");
    if (error == EARG) printf (" EARG <Invalid argument>\n");
    if (error == ESAC) printf (" ESAC <Not Sys Ctrlr>\n");
    if (error == EABO) printf (" EABO <Op. aborted>\n");
    if (error == ENEB) printf (" ENEB <No GPIB board>\n");
    if (error == EOIP) printf (" EOIP <Async I/O in prg>\n");
    if (error == ECAP) printf (" ECAP <No capability>\n");
    if (error == EFSO) printf (" EFSO <File sys. error>\n");
    if (error == EBUS) printf (" EBUS <Command error>\n");
    if (error == ESTB) printf (" ESTB <Status byte lost>\n");
    if (error == ESRQ) printf (" ESRQ <SRQ stuck on>\n");
    if (error == ETAB) printf (" ETAB <Table Overflow>\n");

   /* Call OFFLINE to disable the hardware and software. */
   ParmLength = 0;
   DataLength = 0;
	RetCode = DosDevIOCtl(dvm, CATEGORY, OFFLINE, 0, ParmLength, &ParmLength,
                         0, DataLength, &DataLength); 
  /* Return the handle to the system. */
   DosClose(dvm);
   exit (1);
}


