/*------------------------------------------------------------------------------
   SAMPLE.C -- Sample Driver for JPEG OS/2, Windows DLL and DOS LIB
               Single processing.

Copyright International Business Machines Corp. 1991
All Rights Reserved
Refer to "LICENSE.DOC" for information regarding the use of this file.

------------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <io.h>

#ifndef PROTOTYPE
#include "sample.fun"
#include "testone.h"
#include "parser.h"
#endif

#if defined(TIME) && defined(local) && defined(MS_OS2)
#define TIMED
#endif

#ifdef MS_OS2
#define INCL_PM

#ifdef TIMED
#define INCL_DOSINFOSEG
#endif

#include <os2.h>
#else
#include "bitmap.h"
#endif
#if defined(MS_WIN)
#include <windows.h>
#endif

#include "jpegstr.h"

#define BITMAP

/******************************************************************************/
/* Main Routine                                                               */
/*      Get command line arguments                                            */
/*      Open parameter file and compressed image file                         */
/*      Open pixel file                                                       */
/*      Process data                                                          */
/*      Close files                                                           */
/******************************************************************************/

void main (argc,argv)
char *argv[];
int   argc;
{
   int         i;
   int         bytes_read_written, ret;
   char        infil[50], outfil[50], bitmp[50];
   COM_AREA    iotyp;
   int         PixelHandle;
   int         CoefHandle;
   int         JPEGHandle;
   BUFFER far *PixelBuffer;

   USHORT      first_write;

#ifdef MS_OS2
   BITMAPFILEHEADER Hdr;
#else
   PMBITMAPFILEHEADER Hdr;
#endif

#ifdef TIMED
   ULONG       CumTimes[NUM_TIME_STAMP];
   ULONG       timestamp;
   PGINFOSEG   globalInfoSeg;
   SEL         pSelGlobal, pSelLocal;

   if ( (i = DosGetInfoSeg(&pSelGlobal, &pSelLocal)) != 0) {
      printf("Unable to get global and local Info Segs. RC =  %x\n",i);
   }
   globalInfoSeg = MAKEPGINFOSEG(pSelGlobal);
   for (i = 0;i < NUM_TIME_STAMP;i++) {
      CumTimes[i] = 0;
   } /* endfor */
#endif

   first_write = TRUE;

/**********************************/
/* Process command line arguments */
/**********************************/
   cmdlin(argc, argv, infil, outfil, bitmp, &iotyp);

   if (iotyp.operation & BIT_MAP) {          /* Add BMP header on decompress */
      first_write = TRUE;
   }

/************************************************/
/* Open compressed file and JPEG parameter file */
/************************************************/
   if ((ret = openbuff(infil, outfil, &iotyp, &JPEGHandle, &CoefHandle)) == ERROR_RETURN) {
      exit(-1);
   }

/***********************************/
/* Open file containing pixel data */
/***********************************/
   if ((iotyp.operation & COMPRESS) != 0) {
      if ((ret = openPIX(bitmp,O_RDONLY,&PixelHandle)) == ERROR_RETURN) {
         close(JPEGHandle);
         close(CoefHandle);
         exit(-1);
      }
      if (iotyp.operation & BIT_MAP) {
          if ((ret = ReadBmpHeader(PixelHandle)) == ERROR_RETURN) {
             exit(-1);
          }
      }
   } else /* DeCompress */ {
      if ((ret = openPIX(bitmp,O_WRONLY|O_CREAT,&PixelHandle)) == ERROR_RETURN) {
         exit(-1);
      }
   }


#ifdef TIMED
   timestamp = globalInfoSeg->msecs;
#endif


/**********************************************************************/
/* While there are no errors and end of image has not been reached... */
/**********************************************************************/
    for (strcpy(iotyp.BegAreaID,"ComArea"),
         strcpy(iotyp.EndAreaID,"ComArea"),
         iotyp.operation |= START_IMAGE,
         ret = SUCCESS,bytes_read_written = 0;
         ((iotyp.operation & CALL_FIELD) != END_OF_IMAGE) &&
             (bytes_read_written != -1) && (ret != ERROR_RETURN); ) {

      /*******************************************/
      /* Compression: Read data to be compressed */
      /*******************************************/
        if ((iotyp.operation & COMPRESS) != 0) {
            /* Compress */
            if ((iotyp.operation & CALL_FIELD) != START_IMAGE) {
                /* Not initial call.  On initial call, jpeg_base allocates
                   pixel buffer */
                bytes_read_written = PixelRead(PixelHandle,PixelBuffer,&iotyp);
            } /* endif Initial call */
        }

      /**************************/
      /* Compress or Decompress */
      /**************************/
        if (bytes_read_written != -1) {
            /* Decompress or no error reading during compress */
#ifdef TIMED
            ret = JPEG_base(JPEGHandle,CoefHandle,&PixelBuffer,&iotyp,&Hdr,CumTimes);
#else
            ret = JPEG_base(JPEGHandle,CoefHandle,&PixelBuffer,&iotyp,&Hdr);
#endif
        }

      /********************************************/
      /* Decompression: Write out compressed data */
      /********************************************/
        if (ret != ERROR_RETURN) {
            if ((iotyp.operation & COMPRESS) == 0) {
                /* DeCompress */
                if (first_write) {
                    if (iotyp.operation & BIT_MAP) {
#ifdef MS_OS2
                        bytes_read_written = WritePixelHdr(&Hdr,
                                          sizeof(BITMAPFILEHEADER),PixelHandle);
#else
                        bytes_read_written = WritePixelHdr(&Hdr,
                                        sizeof(PMBITMAPFILEHEADER),PixelHandle);
#endif
                    }
                    first_write = FALSE;
                }
                bytes_read_written = PixelWrite(PixelHandle,PixelBuffer,&iotyp);
            }
        } else {
            printf("Error Code:  %d, data element:  %d, data value:  %x\n",
                    iotyp.error.WarnErrorCode,
                    iotyp.error.data_element,
                    iotyp.error.data_value);
        }

    } /* endfor */

/******************************
*                             *
*     Write out warnings      *
*                             *
******************************/
    for (i = 0;i < 9;i++) {
      switch(i) {
         case 0:
             if (iotyp.warning_flags & IMAGE_FRAME_WARN) {
                 printf("Image Frame Warnings:\n");
                 WriteWarnings(iotyp.warnings.ImageFrame,iotyp.warnings.NumImageFrame);
             } /* endif */
             break;
         case 1:
             if (iotyp.warning_flags & COMPONENT) {
                 printf("Component Warnings:\n");
                 WriteWarnings(iotyp.warnings.Component,iotyp.warnings.NumComponent);
             } /* endif */
             break;
         case 2:
             if (iotyp.warning_flags & SAMPLES) {
                 printf("Samples Warnings:\n");
                 WriteWarnings(iotyp.warnings.Samples,iotyp.warnings.NumSamples);
             } /* endif */
             break;
         case 3:
             if (iotyp.warning_flags & SCAN) {
                 printf("Scan Warnings:\n");
                 WriteWarnings(iotyp.warnings.Scan,iotyp.warnings.NumScan);
             } /* endif */
             break;
         case 4:
             if (iotyp.warning_flags & JPEG_TAB) {
                 printf("JPEG Table Warnings:\n");
                 WriteWarnings(iotyp.warnings.JPEG_Tab,iotyp.warnings.NumJPEG_Tab);
             } /* endif */
             break;
         case 5:
             if (iotyp.warning_flags & HUFF_TAB) {
                 printf("Huffman Table Warnings:\n");
                 WriteWarnings(iotyp.warnings.HuffTab,iotyp.warnings.NumHuffTab);
             } /* endif */
             break;
         case 6:
             if (iotyp.warning_flags & QUANT_TAB) {
                 printf("Quantization Table Warnings:\n");
                 WriteWarnings(iotyp.warnings.QuantTab,iotyp.warnings.NumQuantTab);
             } /* endif */
             break;
         case 7:
             if (iotyp.warning_flags & DNL_WARN) {
                 printf("DNL Warnings:\n");
                 WriteWarnings(iotyp.warnings.DNL,iotyp.warnings.NumDNL);
             } /* endif */
             break;
         case 8:
             if (iotyp.warning_flags & PARS) {
                 printf("Parser Warnings:\n");
                 WriteWarnings(iotyp.warnings.Pars,iotyp.warnings.NumPars);
             } /* endif */
             break;
      }
    } /* endfor */

/***************/
/* Close files */
/***************/
    close(JPEGHandle);
    close(CoefHandle);
    close(PixelHandle);

#ifdef TIMED
    timestamp = globalInfoSeg->msecs - timestamp;
    if ((iotyp.operation & COMPRESS) != 0) {
        printf("Grand Total:                         %.3f seconds\n",
                ((float)timestamp/1000.0));
    } else {
        printf("Grand Total:                         %.3f seconds\n",
                ((float)timestamp/1000.0));
    }
    for (i = 0;i < NUM_TIME_STAMP;i++) {
         switch (i) {
            case TIME_STAMP_TOTAL :
                printf("Total time in Compress/Decompress:   %.3f seconds\n",
                     ((float)CumTimes[i]/1000.0));
                break;
            default :
                printf("Unrecognized time stamp define:  %d\n",i);
                break;
         } /* endswitch */
    } /* endfor */
#endif

   exit(0);

}




void WriteWarnings(WARN_ELE warnings[],USHORT Num) {
   USHORT i;

   for (i = 0;i < Num;i++) {
      printf("%d ",warnings[i]);
   }
}





#ifdef EXAMPLES
/*******************************************************************************
********************************************************************************
********************************************************************************
/******************************************************************************/
/* cmdlin() and cmdHelp()                                                     */
/*      Get command line arguments                                            */
/******************************************************************************/

void cmdlin(int argc,char *argv[],char *pcInfil,char *pcOutFil,char *pcBitmp,COM_AREA *IoType,USHORT *pfusScale)
{
register int i;
char c;
int sBitmap = 0;
int sInFile = 0;
int sOutFile = 0;
int exit_flag;
char cMsg[200];

/*  initialize prameters */
 exit_flag = FALSE;
 strcpy(pcInfil, "\0");
 strcpy(pcOutFil, "\0");

  if (argc == 1) {
     strcpy(cMsg,"You must specify a parameter. Enter ");
     strcat(cMsg,argv[0]);
     strcat(cMsg,"? for help.");
     TestWarnMsg("",cMsg);
     cmdHelp();
     exit(1);
  }
  IoType->operation = 0;
  /* argv[0] is the execution file name                        */
  /* argv[1] and sucssive odd values are option type specifier */
  /* argv[2] and sucessive even values are option strings      */
  for (i = 1; i <= (argc-1);i++) {
         c = *argv[i];
         if (c == '/') { /* check for option specifier         */
                   c = *(argv[i]+1);
                   switch (c) {
                      case 'i': /* INFILE string                */
                      case 'I':
                             i++;
                             strcpy(pcInfil,argv[i]);
                             sInFile = 1;
                             break;
                      case 'o': /* OUTFILE string               */
                      case 'O':
                             i++;
                             strcpy(pcOutFil,argv[i]);
                             sOutFile = 1;
                             break;
                      case 'b': /* BITMAP FILE In/Out           */
                      case 'B':
                             i++;
                             strcpy(pcBitmp,argv[i]);
                             sBitmap = 1;
                             break;
                      case 's': /* INVDCT scale factor */
                      case 'S':
                             i++;
                             *pfusScale = atoi(argv[i]);
                             break;
                      case 'e': /* Encode */
                      case 'E':
                             IoType->operation |= COMPRESS;
                             break;
                      case 'd': /* Decode */
                      case 'D':
                             IoType->operation &= ~COMPRESS;
                             break;
                      case 'c': /* Colour space convert data */
                      case 'C':
                             IoType->operation |= COLOR_CONVERT;
                             i++;
                             if (stricmp(argv[i],"RGB888") == 0) {
                                 IoType->InputColorSpace = RGB888;
                             } else if (stricmp(argv[i],"YYUV") == 0) {
                                 IoType->InputColorSpace = YYUV;
                             } else if (stricmp(argv[i],"RGB565") == 0) {
                                 IoType->InputColorSpace = RGB565;
                             } else if (stricmp(argv[i],"INT565") == 0) {
                                 IoType->InputColorSpace = INT565;
                             } else if (stricmp(argv[i],"YUV") == 0) {
                                 IoType->InputColorSpace = YUV;
                             } else {
                                 strcpy(cMsg,"Color Convert Option [");
                                 strcat(cMsg,argv[i]);
                                 strcat(cMsg,"]  not recognized");
                                 TestWarnMsg("",cMsg);
                                 cmdHelp();
                                 exit_flag = TRUE;
                             }
                             i++;
                             if (stricmp(argv[i],"RGB888") == 0) {
                                 IoType->OutputColorSpace = RGB888;
                             } else if (stricmp(argv[i],"YYUV") == 0) {
                                 IoType->OutputColorSpace = YYUV;
                             } else if (stricmp(argv[i],"RGB565") == 0) {
                                 IoType->OutputColorSpace = RGB565;
                             } else if (stricmp(argv[i],"INT565") == 0) {
                                 IoType->OutputColorSpace = INT565;
                             } else if (stricmp(argv[i],"YUV") == 0) {
                                 IoType->OutputColorSpace = YUV;
                             } else {
                                 strcpy(cMsg,"Color Convert Option [");
                                 strcat(cMsg,argv[i]);
                                 strcat(cMsg,"]  not recognized");
                                 TestWarnMsg("",cMsg);
                                 cmdHelp();
                                 exit_flag = TRUE;
                             }
                             break;
                      default : /* unrecognized option          */
                             strcpy(cMsg,"Option [");
                             strcat(cMsg,argv[i]);
                             strcat(cMsg,"]  not recognized");
                             TestWarnMsg("",cMsg);
                             cmdHelp();
                             exit_flag = TRUE;
                             break;
                   }
         } else {  /* option specifier not found exit program */
              cmdHelp();
              exit_flag = TRUE;
         }
  } /* end for */
  if (sInFile == 0) {
     strcpy(cMsg,"You must specify an input file name.");
     TestWarnMsg("",cMsg);
  }
  if (sBitmap == 0) {
     strcpy(cMsg,"You must specify a bitmap file name.");
     TestWarnMsg("",cMsg);
  }
  if (sOutFile == 0) {
     strcpy(pcOutFil,"samp.out\0");
     sOutFile = 1;
  }
  if (((sInFile | sBitmap) == 0) || (exit_flag == TRUE)) {
      exit(2);
  }
} /* End of cmdlin */

void cmdHelp(void)
{
   char     cMsg[200];

    strcpy(cMsg," Error in command line arguments.");
    TestWarnMsg("",cMsg);
    strcpy(cMsg,"sample /i /o /b [/e|/d] [/c <From Space> <To Space>]  ");
    TestWarnMsg("",cMsg);
    strcpy(cMsg,"/i INFILE");
    TestWarnMsg("",cMsg);
    strcpy(cMsg,"/o OUTFILE");
    TestWarnMsg("",cMsg);
    strcpy(cMsg,"/b Raw BITMAP File(s)");
    TestWarnMsg("",cMsg);
    strcpy(cMsg,"Optional Parameters:");
    TestWarnMsg("",cMsg);
    strcpy(cMsg,"/e encode");
    TestWarnMsg("",cMsg);
    strcpy(cMsg,"/d decode");
    TestWarnMsg("",cMsg);
    strcpy(cMsg,"/c Apply a colour space conversion to image data.  Possible conversions are:");
    TestWarnMsg("",cMsg);
    strcpy(cMsg,"RGB888 YUV, RGB888 YYUV, RGB888 RGB565, RGB888 INT565");
    TestWarnMsg("",cMsg);
    strcpy(cMsg,"YUV RGB888, YYUV RGB888, RGB565 RGB888, INT565 RGB888");
    TestWarnMsg("",cMsg);
    strcpy(cMsg,"Items in [] are optional. ");
    TestWarnMsg("",cMsg);
    strcpy(cMsg,"Choose only 1 item in a group separated by |.");
    TestWarnMsg("",cMsg);
} /* End of cmdHelp */

/******************************************************************************/
/* openbuff()                                                                 */
/*      Open parameter file and compressed image file                         */
/******************************************************************************/

int openbuff(
char      *fpvSource,
char      *fpvDestination,
COM_AREA  *IoType,
USHORT    *JPEGHandle,
USHORT    *CoefHandle
)
{
   int rc;

   rc = SUCCESS;
   if ((IoType->operation & COMPRESS) != 0) {
      if ((*JPEGHandle = open((char *)fpvSource, O_BINARY|O_RDONLY,0666)) == -1 ) {
          printf("unable to open input file:  %s, return code:  %d\n",
                  (char *)fpvSource,*JPEGHandle);
          rc = ERROR_RETURN;
      }

      if (fpvDestination !=  NULL) {
          if ((*CoefHandle = open((char *)fpvDestination,O_BINARY|O_WRONLY|O_CREAT,0666))  ==  -1 )
          {
              printf("unable to open output file:  %s, return code:  %d\n",
                      (char *)fpvDestination,*CoefHandle);
              rc = ERROR_RETURN;
          }
      } else {
          if ((*CoefHandle = open("enc.out",O_BINARY|O_WRONLY|O_CREAT,0666))  ==  -1 )
          {
              printf("unable to open file:  enc.out, return code:  %d\n",
                      *CoefHandle);
              rc = ERROR_RETURN;
          }
          printf("No output file specified, output placed in [ENC.OUT]\n");
          rc = ERROR_RETURN;
      }
   } else { /* DeCompress */
      if ((*CoefHandle = open((char *)fpvSource,O_BINARY|O_RDONLY,0666)) == -1) {
          printf("unable to open input file:  %s, return code:  %d\n",
                  (char *)fpvSource,*CoefHandle);
          rc = ERROR_RETURN;
      }

      if ((*JPEGHandle = open((char *)fpvDestination,O_BINARY|O_WRONLY|O_CREAT,0666)) == -1 ) {
          printf("unable to open output file:  %s, return code:  %d\n",
                 (char *)fpvDestination,*JPEGHandle);
          rc = ERROR_RETURN;
      }
   }
   return(rc);
} /* End of openbuff() */

/******************************************************************************/
/* openPIX()                                                                  */
/*      Open pixel file                                                       */
/******************************************************************************/

int openPIX(file,options,file_handle)
char   *file;
USHORT  options;
int    *file_handle;
{
   if ((*file_handle = open(file,O_BINARY|options,0666)) == -1) {
      printf("Couldn't open Pixel file:  %s, return code:  %d\n", file,
                                                                  *file_handle);
      return(ERROR_RETURN);
   }
   return(SUCCESS);
} /* End of openPIX() */

/******************************************************************************/
/* PixelRead() and PixelWrite()                                               */
/*      Read and write a pixel buffer from/to a file                          */
/******************************************************************************/

int PixelRead(PixelHandle,PixelBuffer,ComArea)
int       PixelHandle;
BUFFER   *PixelBuffer;
COM_AREA *ComArea;
{
   int     k;
   char cMsg[100], buf[100];

   /* For all but the last group of blocks, DCT_ROW_COL rows of maxvsamp MDUs are
      acquired */
   k = read(PixelHandle, PixelBuffer->bufad, PixelBuffer->buflen);
   if (k == -1) {
      strcpy(cMsg,"Invalid read into pixel buffer, error:  ");
      itoa(errno,buf,16);
      strcat(cMsg,buf);
      strcpy(cMsg,".  Buffer length:  ");
      itoa(PixelBuffer->buflen,buf,16);
      strcat(cMsg,buf);
      TestErrMsg("PixelRead",cMsg,0);
      return(ERROR_RETURN);
   }
   return(k);

} /* End of PixelRead() */

int PixelWrite(PixelHandle,PixelBuffer,ComArea)
int       PixelHandle;
BUFFER   *PixelBuffer;
COM_AREA *ComArea;
{
   int k;
   char cMsg[100], buf[100];

   /* For all but the last group of blocks, DCT_ROW_COL rows of maxvsamp MDUs are
      acquired */
   k = write(PixelHandle, PixelBuffer->bufad,PixelBuffer->bufused);
   if (k == -1) {
      strcpy(cMsg,"Invalid write from pixel buffer, error:  ");
      itoa(errno,buf,16);
      strcat(cMsg,buf);
      strcpy(cMsg,".  Buffer length:  ");
      itoa(PixelBuffer->buflen,buf,16);
      strcat(cMsg,buf);
      TestErrMsg("PixelWrite",cMsg,0);
      return(ERROR_RETURN);
   }
   return (k);

} /* End of PixelWrite() */

/*******************************************************************************
********************************************************************************
********************************************************************************
To use multiple buffering, use AsyncPixelRead() and AsyncPixelWrite() instead
of PixelRead() and PixelWrite()

int AsyncPixelRead(PixelHandle,PixelBuffer,ComArea,write_in_progress,io_rc,async_bytes_read_written)
int       PixelHandle;
BUFFER   *PixelBuffer;
COM_AREA *ComArea;
unsigned long *write_in_progress;
int *io_rc;
int *async_bytes_read_written;
{
   int     k;
   char cMsg[100], buf[100];

      ** For all but the last group of blocks, DCT_ROW_COL rows of maxvsamp MDUs are
         acquired **
      if (write_in_progress != NULL) {
         k = DosReadAsync(PixelHandle,   ** Read asyncronously from    **
              write_in_progress,         **  the file.  Clears         **
              io_rc,PixelBuffer->bufad,  **  semaphore when read       **
              PixelBuffer->buflen,       **  actually completes.       **
              async_bytes_read_written); **                            **
         return(k);                      **                            **
      } else {
         k = read(PixelHandle, PixelBuffer->bufad, PixelBuffer->buflen);
         if (k == -1) {
            strcpy(cMsg,"Invalid read into pixel buffer, error:  ");
            itoa(errno,buf,16);
            strcat(cMsg,buf);
            strcpy(cMsg,".  Buffer length:  ");
            itoa(PixelBuffer->buflen,buf,16);
            strcat(cMsg,buf);
            TestErrMsg("AsyncPixelRead",cMsg,0);
            return(ERROR_RETURN);
         }
      } ** endif **
      return(k);
} ** End of AsyncPixelRead **

int AsyncPixelWrite(PixelHandle,PixelBuffer,ComArea,write_in_progress,io_rc,async_bytes_read_written)
int       PixelHandle;
BUFFER   *PixelBuffer;
COM_AREA *ComArea;
unsigned long *write_in_progress;
int *io_rc;
int *async_bytes_read_written;
{
   int k;
   char cMsg[100], buf[100];

   ** For all but the last group of blocks, DCT_ROW_COL rows of maxvsamp MDUs are
      acquired **
      if (write_in_progress != NULL) {
         k = DosWriteAsync(PixelHandle,  ** Write asyncronously to     **
              write_in_progress,         **  the file.  Clears         **
              io_rc,PixelBuffer->bufad,  **  semaphore when write      **
              PixelBuffer->bufused,      **  actually completes.       **
              async_bytes_read_written); **                            **
         return(k);                      **                            **
      } else {
        k = write(PixelHandle, PixelBuffer->bufad,PixelBuffer->bufused);
        if (k == -1) {
           strcpy(cMsg,"Invalid write from pixel buffer, error:  ");
           itoa(errno,buf,16);
           strcat(cMsg,buf);
           strcpy(cMsg,".  Buffer length:  ");
           itoa(PixelBuffer->buflen,buf,16);
           strcat(cMsg,buf);
           TestErrMsg("AsyncPixelWrite",cMsg,0);
           return(ERROR_RETURN);
      } ** endif **
   }
   return (k);
} ** End of AsyncPixelWrite() **
********************************************************************************
********************************************************************************
*******************************************************************************/

/******************************************************************************/
/* TestErrMsg() and TestWarnMsg()                                             */
/*      Notify users of errors and warnings                                   */
/******************************************************************************/

void  TestErrMsg(char *mod, char *mess, int rc)
{
   printf("%s: RC=%u Error: %s\n",mod,rc,mess);
} /* End of TestErrMsg() */

int  TestWarnMsg(char *mod, char *mess)
{
   printf("%s: Warning: %s\n",mod,mess);
   return(0);
} /* End of TestWarnMsg() */
#endif
