/*********************************************************************/
/*								     */
/*  MODULE NAME = 2PLAY.C					     */
/*								     */
/*  DESCRIPTIVE NAME = Audio Application Programming Interface	     */
/*		       Sample Two Track Play Program		     */
/*								     */
/*  COPYRIGHT = (C) Copyright by IBM Corp. 1991.		     */
/*		All rights reserved.				     */
/*		Refer to "LICENSE.DOC" for information regarding     */
/*		the use of this file.				     */
/*								     */
/*  FUNCTION = Provides the following functions/tools:		     */
/*								     */
/*	       2play - Provides a working example of		     */
/*		       playing two tracks using the Audio API.	     */
/*								     */
/*  CONVENTIONS = See Below					     */
/*								     */
/*    Constants - upper case					     */
/*    Internal variables - lower case				     */
/*    External variables - as required				     */
/*    Internal functions - initial caps 			     */
/*    External functions - as required				     */
/*								     */
/*  CHANGE ACTIVITY =  See Below				     */
/*								     */
/*		       01/25/91, M-ACPA AAPI Release 2.0	     */
/*								     */
/*********************************************************************/
/*								     */
/*************************SYSTEM INCLUDES*****************************/
/*								     */
#define LINT_ARGS		       /* Perform type checking on   */
				       /*  library functions	     */
/*								     */
#include <sys\types.h>		       /*			     */
#include <sys\stat.h>		       /*			     */
#include <conio.h>		       /*			     */
#include <io.h> 		       /*			     */
#include <stdio.h>		       /*			     */
#include <string.h>		       /*			     */
#include <time.h>		       /*			     */
#include <malloc.h>		       /*			     */
/*								     */
/**************************AAPI INCLUDES******************************/
/*								     */
#include "2play.h"                     /* Messages                   */
#include "eapidbas.cnc"                /* Audio file structures      */
#include "eapiacb.cnc"                 /* Audio constants/declares   */
/*								     */
/*********************CONSTANT DEFINITIONS****************************/
/*								     */
#define TRUE			1      /* True constant 	     */
#define FALSE			0      /* False constant	     */
#define SUCCESS 		0      /* Function call successful   */
#define NT			2      /* Number of tracks	     */
#define FADE_TIME		7000L  /* Milliseconds for pan,fade  */
#define FADE_PERC		5      /* % to fade,pan at a time    */
/*								     */
				       /* Key scan codes	     */
				       /*			     */
#define F1			59     /* Start/Pause/Resume - Trk 1 */
#define F2			60     /*			   2 */
#define F3			61     /*			both */
#define NR			77     /* Track 1 - Balance 5% right */
#define NL			75     /*		       left  */
#define NH			71     /*  Pan right over 5 seconds  */
#define NE			79     /*      left		     */
#define CR			116    /* Track 2 - Balance 5% right */
#define CL			115    /*		       left  */
#define CH			119    /*  Pan right over N seconds  */
#define CE			117    /*      left		     */
#define NU			72     /* Track 1 - Volume up 5%     */
#define ND			80     /*		   down      */
#define NP			73     /*  Fade up over N seconds    */
#define NW			81     /*	down		     */
#define CU			141    /* Track 2 - Volume up 5%     */
#define CD			145    /*		   down      */
#define CP			132    /*  Fade up over N seconds    */
#define CW			118    /*	down		     */
/*								     */
/*								     */
#define AVC_EXT  "._AU"                /* AVC Audio file extension   */
#define RIFF_EXT ".WAV"                /* RIFF Audio file extension  */
/*								     */
/*****************VARIABLE DECLARATION/DEFINITIONS********************/
/*								     */
unsigned char master_volume[NT] = {100,100};  /* Default control     */
unsigned short	volume[NT]	= {100,100};  /*  settings	     */
unsigned long volume_rate[NT]	= {0L,0L};    /*		     */
unsigned char balance[NT]	= {50,50};    /*		     */
unsigned long balance_rate[NT]	= {0L,0L};    /*		     */
unsigned char channel[NT]	= {255,255};  /*		     */
short	  source_mix[NT]	= {0,0};      /*		     */
unsigned char input_source[NT]	= {3,3};      /*		     */
short	  speaker[NT]		= {0,0};      /*		     */
unsigned long start_pos[NT]	= {0L,0L};    /*		     */
unsigned long end_pos[NT]	= {0L,0L};    /*		     */
short	  loop[NT]		= {0,0};      /*		     */
/*								     */
short	track_status[NT]  ={-1,-1};    /* Track Status - 1/Started   */
				       /*   0/Initialized	     */
				       /*  -1/Not initialized	     */
unsigned long cur_pos[NT] = {0L,0L};   /* Current position (seconds) */
char huge *lalloc[NT] ={NULL,NULL};    /* Ptr to data buffer >64K    */
unsigned short trk_paused[NT]= {0,0};  /* Track is paused	     */
short  close_file[NT] = {0,0};	       /* File needs to be closed    */
short  term_audio[NT] = {0,0};	       /* AAPIs need to be terminated*/
unsigned short	type[NT] = {0,0};      /* File type to play	     */
short atype[NT] = {0,0};	       /* Audio type		     */
unsigned long buf_time[NT];	       /* Size of data buffers in    */
				       /*  milliseconds 	     */
char oname[NT][81];		       /* File name		     */
AAPI_ACB   acb[NT];		       /* Audio Control Block	     */
AAPI_FMT fmt[NT];		       /* PCM format data	     */
AAPI_LST list[NT];		       /* List for looping	     */
FAB_STRUCT *fabp[NT];		       /* Ptr to File Access Block   */
/*								     */
/******************RESIDENT FUNCTION DECLARATIONS*********************/
/*								     */
static void Get_File_Name(char*,short);/* Get output file name	     */
static short Open_File(short);	       /* Open audio file	     */
static short Init_Track(short);        /* Initialize audio track     */
static void Term_Track(short);	       /* Terminate track	     */
static short Init_ACB(AAPI_ACB *,      /* Initialize Audio Control   */
	     FAB_STRUCT *,AAPI_FMT*,   /*  Block		     */
	      short);		       /*			     */
static short Check_IO(AAPI_ACB *,      /* Check if I/O needed	     */
	     unsigned long);	       /*			     */
static short Chk_User(short*);	       /* Check user input	     */
static short Get_Vol(short,short,      /* Process volume control     */
		   unsigned short);    /*			     */
static short Get_Bal(short,short,      /*	  balance control    */
		   unsigned short);    /*			     */
static short Pause_Resume(short);      /*	  start/pause/resume */
static void Write_Error_Message(short);/* Put error message to screen*/
static void Set_Options(short);        /* Get/Set Vol/Bal Parms      */
static short Query_User(char *,        /* Ask user for input	     */
  unsigned long, unsigned long,        /*			     */
  unsigned long *);		       /*			     */
static short Get_Audio_Type(	       /* Get compression type	     */
			FAB_STRUCT *); /*			     */
static short FABO_Find(FAB_STRUCT*,    /* Find File Access Block     */
   unsigned short,unsigned short,      /*  Object in list of objects */
   FABO_STRUCT * *,unsigned short *);  /*			     */
/*								     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = 2play					     */
/*								     */
/* FUNCTION = Initializes the audio card and open one or more audio  */
/*	      files and begins reading and playing audio data.	     */
/*								     */
/* INPUT =    None - The user is prompted for the name of a file     */
/*	      that contains the input audio data.		     */
/*								     */
/*	      The file must exist and be one of the following	     */
/*	       types of audio files:				     */
/*								     */
/*	      - AVC						     */
/*	      - RIFF WAVE					     */
/*								     */
/*	      If no extension is used the file is searched for	     */
/*	      without an extension.  If not found the default AVC    */
/*	      extension (_AU) is appended and the search repeated.   */
/*	      If still not successful the default RIFF WAVE	     */
/*	      extension (WAV) is appended.			     */
/*								     */
/*	      User hits "F1" or "F2" or "F3" to start/pause/resume   */
/*	       (toggles) the first, second, or both tracks.	     */
/*	      The cursor keys control volume and balance on the      */
/*	       first track (+-5%).  Home/End will pan Left/Right     */
/*	       over 7 seconds. PageUp/PageDown will fade up/down over*/
/*	       7 seconds.					     */
/*	      Ctrl+ the keys above control the second track.	     */
/*								     */
/*	      User hits any other key to stop operation.	     */
/*								     */
/* OUTPUT =   If successful					     */
/*		Audio file(s) contents played until user hits any key*/
/*								     */
/*	      Else error message written to screen		     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
main()				       /* Begin Function	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short rc;			       /* Return code		     */
short keep_going;		       /*			     */
short update_screen;		       /*			     */
short  num_trks = 0;		       /* Number of tracks requested */
short  t;			       /* For loop index	     */
/*********************************************************************/
  cprintf(M_44); cprintf(M_45);        /* Program title 	     */
  for(t=0;(t<NT);t++)		       /* For both tracks	     */
  {				       /*			     */
    rc = -1;			       /*			     */
    while (rc != SUCCESS)	       /* While user hasn't responded*/
    {				       /*  without error	     */
      rc = SUCCESS;		       /*			     */
      Get_File_Name(oname[t],t);       /* Get name of file	     */
      if (oname[t][0])		       /* If user typed a name	     */
      { 			       /*			     */
	if((rc=Open_File(t))==SUCCESS) /* If file opened	     */
	{			       /*			     */
	  num_trks++;		       /* Track will be used	     */
	}			       /*			     */
      } 			       /*			     */
    }				       /*			     */
  }				       /*			     */
  for(t=0;(t<num_trks)&&(rc==SUCCESS); /* For each active track      */
				 t++)  /*			     */
  {				       /*			     */
    rc = Init_Track(t); 	       /* Initialize the track	     */
  }				       /*			     */
  if (rc == SUCCESS)		       /* If no error setting up     */
  {				       /*  audio files		     */
    cprintf(M_24); cprintf(M_26);      /* Tell user how to pause     */
    cprintf(M_27); cprintf(M_28);      /*  and about volume and      */
    cprintf(M_29); cprintf(M_30);      /*  balance controls	     */
    cprintf(M_2);  cprintf(M_14);      /*  and how to stop	     */
    cprintf(M_31,		       /*  and current position,     */
     cur_pos[0],volume[0],balance[0],  /*   volume, and balance      */
     cur_pos[1],volume[1],balance[1]); /*			     */
  }				       /*			     */
  keep_going = TRUE;		       /*			     */
  update_screen = FALSE;	       /*			     */
  while ((keep_going) && (rc==SUCCESS) /* While user hasn't quit     */
   && ((acb[0].state==AAPI_PLAYING) || /*  and no error and	     */
       (acb[1].state==AAPI_PLAYING) || /*  still playing	     */
       (track_status[0] == 0) ||       /*  either track or	     */
       (track_status[1] == 0)))        /*  haven't started           */
  {				       /*  one of the tracks	     */
    if (kbhit())		       /* If user hit a key	     */
    {				       /*			     */
      rc = Chk_User(&keep_going);      /* Process it (volume,	     */
      update_screen = TRUE;	       /*  balance, stopping)	     */
    }				       /*			     */
    if((rc==SUCCESS) && (keep_going))  /* If continuing 	     */
    {				       /*			     */
      for(t=0;t<num_trks;t++)	       /* For each track	     */
      { 			       /*			     */
	if((((acb[t].position -        /* If playing next second     */
	     start_pos[t])	       /*  yet or control values     */
	     /1000L) > cur_pos[t]) ||  /*  might have changed	     */
	     (update_screen))	       /*			     */
	{			       /*			     */
	  cur_pos[t] = (short)	       /* Tell user elasped time     */
	     ((acb[t].position -       /*			     */
	       start_pos[t])/1000L);   /*			     */
	  update_screen = FALSE;       /*			     */
	  cprintf(M_14);	       /*  in seconds		     */
	  cprintf(M_31,cur_pos[0],     /*  and current position,     */
	   volume[0],balance[0],       /*   volume, and balance      */
	   cur_pos[1],volume[1],       /*			     */
	   balance[1]); 	       /*			     */
	}			       /*			     */
      } 			       /*			     */
      for(t=0;t<num_trks;t++)	       /* For each track	     */
      { 			       /*			     */
	rc = Check_IO(&acb[t],	       /* Check if I/O is necessary  */
		    buf_time[t]);      /*			     */
      } 			       /*			     */
    }				       /*			     */
  }				       /* Enddo playing loop	     */
  for(t=0;t<num_trks;t++)	       /* For each track	     */
  {				       /*			     */
    Term_Track(t);		       /* Terminate processing	     */
    if ((rc == SUCCESS) &&	       /* If an error occurred	     */
	(acb[t].backrc != SUCCESS))    /*  in background while	     */
    {				       /*  playing		     */
       rc = (short) acb[t].backrc;     /* Tell caller		     */
    }				       /*			     */
  }				       /*			     */
  if (rc != SUCCESS)		       /* If error occurred	     */
  {				       /*			     */
    Write_Error_Message(rc);	       /* Tell user		     */
  }				       /*			     */
}				       /*			     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Open_File					     */
/*								     */
/* FUNCTION = Query user for output file name and other parameters.  */
/*	      Open the file.					     */
/*								     */
/* INPUT =    Track to process					     */
/*								     */
/* OUTPUT =							     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static Open_File(t)		       /* Begin Function	     */
short  t;			       /* Track 		     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short rc;			       /* Return code		     */
/*********************************************************************/
  rc = fab_type(oname[t],&type[t]);    /* Find out its type	     */
  if((t) && (type[0] != type[1]) &&    /* If file types conflict     */
     ((type[1] == AAFB_AVCF) ||        /*  AVC/WAVE		     */
      (type[1] == AAFB_WAVE)))	       /*			     */
  {				       /*			     */
    rc = 2;			       /* Tell caller		     */
  }				       /*			     */
  if (rc == SUCCESS)		       /* If no error		     */
  {				       /*			     */
    switch (type[t])		       /* Case of file type	     */
    {				       /*			     */
      case AAFB_AVCF  : 	       /* AVC			     */
      rc = fab_open(oname[t],	       /* Call AAPI file function    */
	     AAFB_OPEN,AAFB_EXNO,0,    /*  to initialize file and    */
	     &fabp[t],0,0,0,0);        /*  associated structures     */
      break;			       /*			     */
      case AAFB_WAVE  : 	       /* RIFF WAVE		     */
      rc = fab_ropn(oname[t],	       /*			     */
       AAFB_OPEN,&fabp[t],&fmt[t]);    /*			     */
      break;			       /*			     */
      case AAFB_NONE  : 	       /* File does not exist	     */
      rc = AAFB_RC_02;		       /*			     */
      break;			       /*			     */
      case AAFB_UKNW  : 	       /* Unknown file type	     */
      rc = 1;			       /*			     */
      break;			       /*			     */
      default : 		       /* Else case		     */
      rc = AAFB_RC_07;		       /*			     */
      break;			       /*			     */
    }				       /* End Switch		     */
    if (rc == SUCCESS)		       /* If no error		     */
    {				       /*			     */
      atype[t] =		       /* Get audio compression      */
	 Get_Audio_Type(fabp[t]);      /*  type 		     */
      if (t)			       /* If processing 2nd file     */
      { 			       /*			     */
	if (type[t] == AAFB_AVCF)      /* If AVC type file	     */
	{			       /*			     */
	  if(((atype[0] == AUDO_AVCM)&&/* If trying to play MIDI     */
	    (atype[1] != AUDO_ML05))|| /*  and audio type other than */
	   ((atype[1] == AUDO_AVCM) && /*  voice		     */
	    (atype[0] != AUDO_ML05)))  /*			     */
	  {			       /*			     */
	    rc = 3;		       /* Invalid combination	     */
	  }			       /*			     */
	  if((atype[1] == AUDO_ML22)|| /* If trying to play stereo   */
	     (atype[1] == AUDO_FM22))  /*  or high quality music and */
	  {			       /*  other type		     */
	    rc = AAPI_RC_29;	       /* Invalid combination	     */
	  }			       /*			     */
	}			       /*			     */
	else			       /* RIFF Wave file	     */
	{			       /*			     */
	  if (fmt[1].channels !=1)     /* If this is a stereo track  */
	  {			       /*			     */
	    rc = AAPI_RC_29;	       /* Invalid combination	     */
	  }			       /*			     */
	  else			       /*			     */
	  {			       /*			     */
	    if (fmt[1].format!=        /*			     */
		AAPI_FMT_LPCM_W)       /* If track is mu_law/a_law   */
	    {			       /*			     */
	      rc = AAPI_RC_29;	       /* Invalid combination	     */
	    }			       /*			     */
	  }			       /*			     */
	}			       /*			     */
      } 			       /*			     */
    }				       /*			     */
    if (rc == SUCCESS)		       /* If file valid 	     */
    {				       /*			     */
      Set_Options(t);		       /* Get other user input	     */
    }				       /*			     */
    else			       /* Else			     */
    {				       /*			     */
      Write_Error_Message(rc);	       /* Tell user		     */
    }				       /*			     */
  }				       /*			     */
  else				       /* Else error when checking   */
  {				       /*  type 		     */
    Write_Error_Message(rc);	       /* Tell user		     */
  }				       /*			     */
  return(rc);			       /*			     */
}				       /*			     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Init_Track					     */
/*								     */
/* FUNCTION = Open the file and prepare to begin playing audio	     */
/*	      on this track.					     */
/*								     */
/* INPUT =    Track to process					     */
/*								     */
/* OUTPUT =							     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static Init_Track(t)		       /* Begin Function	     */
short  t;			       /* Track 		     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short rc;			       /* Return code		     */
/*********************************************************************/
  close_file[t] = TRUE; 	       /* Remember to close file     */
  rc =				       /* Initialize Audio CB	     */
    Init_ACB(&acb[t],fabp[t],&fmt[t],  /*			     */
	      t);		       /*			     */
  if (rc == SUCCESS)		       /* If no error on set up      */
  {				       /*			     */
    buf_time[t]=acb[t].timeleft>=100L  /* Save max time before	     */
      ? acb[t].timeleft : 100L;        /*  I/O is required	     */
    term_audio[t] = TRUE;	       /* Audio is initialized	     */
    track_status[t] = 0;	       /* Track is initialized	     */
  }				       /*			     */
  return(rc);			       /* Return result to caller    */
}				       /*			     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Term_Track					     */
/*								     */
/* FUNCTION = Finish processing for a track			     */
/*								     */
/* INPUT =    Track to process					     */
/*								     */
/* OUTPUT =							     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static void Term_Track(t)	       /* Begin Function	     */
short  t;			       /* Track 		     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
long stime;			       /* Wait time start	     */
short  w;			       /* Index 		     */
/*********************************************************************/
  if (acb[t].state != AAPI_STOPPED)    /* If not already stopped     */
  {				       /*			     */
    acb[t].controls = AAPI_STOP;       /* Set stop position so	     */
    acb[t].stoppos  = 0L;	       /*  an immediate stop	     */
    acb[t].ctlparms = AAPI_STPI;       /*  will be executed	     */
    aud_ctrl(&acb[t]);		       /* Execute the stop	     */
    stime = clock();		       /* Wait for stop to	     */
    while(			       /*  complete before	     */
    (acb[t].state!=AAPI_STOPPED) &&    /*  continuing, do not	     */
    ((clock() - stime) < 30000))       /*  badger OS while	     */
    for(w=0;w<=1000;w++);	       /*  waiting		     */
  }				       /*			     */
  if (term_audio[t])		       /* If AAPI initialized	     */
  {				       /*			     */
    aud_term(&acb[t]);		       /* Terminate AAPI	     */
  }				       /*			     */
  if (close_file[t])		       /* If file was opened	     */
  {				       /*			     */
    fab_close(fabp[t]); 	       /* Close audio file	     */
  }				       /*			     */
  if (lalloc[t] != NULL)	       /* If allocated buffer	     */
  {				       /*			     */
    hfree(lalloc[t]);		       /* Free it		     */
  }				       /*			     */
}				       /*			     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Get_File_Name				     */
/*								     */
/* FUNCTION = Query user for output file name.	Append extension     */
/*	      to name if needed.				     */
/*								     */
/* INPUT =    Ptr to area for name				     */
/*								     */
/* OUTPUT =   None - name returned in callers area		     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static void Get_File_Name(name,t)      /*			     */
char *name;			       /* Output file name	     */
short  t;			       /* Track 		     */
{				       /* 010			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
char *end;			       /* Ptr to end of name	     */
char input_buffer[80];		       /*			     */
unsigned long value;		       /*			     */
short nofile;			       /*			     */
/*********************************************************************/
  nofile = FALSE;		       /*			     */
  if (t)			       /* If processing 2nd file     */
  {				       /*			     */
    if (type[0] == AAFB_AVCF)	       /* If AVC type file	     */
    {				       /*			     */
      if((atype[0] == AUDO_ML22)||     /* If trying to play stereo   */
	 (atype[0] == AUDO_FM22))      /*  or high quality music on  */
      { 			       /*  first track		     */
	nofile = TRUE;		       /* No second file allowed     */
      } 			       /*			     */
    }				       /*			     */
    else			       /* RIFF Wave file	     */
    {				       /*			     */
      if (fmt[0].channels !=1)	       /* If first track is a	     */
      { 			       /*  stereo file		     */
	nofile =TRUE;		       /* No second file allowed     */
      } 			       /*			     */
      else			       /*			     */
      { 			       /*			     */
	if (fmt[0].format!=	       /*			     */
	      AAPI_FMT_LPCM_W)	       /* If first track is mu_law   */
	{			       /*  or a-law		     */
	  nofile =TRUE; 	       /* No second file allowed     */
	}			       /*			     */
      } 			       /*			     */
    }				       /*			     */
  }				       /*			     */
  if (!nofile)			       /* If 2nd file allowed	     */
  {				       /*			     */
    t ? cprintf(M_1) : cprintf(M_0);   /* Query for file name	     */
    gets(input_buffer); 	       /*  file name		     */
    if(sscanf(input_buffer,"%s",name)  /* If name was typed          */
       ==1)			       /*			     */
    {				       /*			     */
      if(!(((end=strrchr(name,'.'))    /* If name has no             */
	!=0) && 		       /*  extension		     */
	(strrchr(end,'\\')==0)))       /*                            */
      { 			       /*			     */
	if(access(name,0) != SUCCESS)  /* Check for existance	     */
	{			       /* If not found try RIFF      */
	  strcat(name,RIFF_EXT);       /*  extension		     */
	  if(access(name,0) != SUCCESS)/* Check for existance	     */
	  {			       /* If not found try AVC	     */
	    strcpy(&name[strlen(name)  /*  extension		     */
	     -4],AVC_EXT);	       /*			     */
	  }			       /*			     */
	}			       /*			     */
      } 			       /*			     */
    }				       /*			     */
    else			       /* Else user hit enter asking */
    {				       /*  for Source Mix	     */
      if (t==0) 		       /* If track 0		     */
      { 			       /*			     */
	strcpy(name,AAFB_SMIX);        /* Use source mix device name */
	source_mix[0] = TRUE;	       /*			     */
	if (Query_User(M_35,0L,4L,     /* Input Source		     */
			    &value))   /*			     */
	{			       /*			     */
	  input_source[0] =	       /*			     */
	   (unsigned char)	       /*			     */
	   ((value==0) || (value==3) ||/* Line left and right	     */
	   (value==4)) ? value : 3;    /*  not valid for source mix  */
	  input_source[1] =	       /* Both tracks must be same   */
	    input_source[0];	       /*  input source 	     */
	}			       /*			     */
      } 			       /*			     */
      else			       /* Else no name on 2nd track  */
      { 			       /*			     */
	name[0] = 0;		       /*			     */
	if ((!source_mix[0]) &&        /* If not already doing source*/
	    (type[0] == AAFB_WAVE))    /*  mix and doing PCM	     */
	{			       /*			     */
	  if (Query_User(M_33,1L,1L,   /* Did user want source mix   */
			      &value)) /*  or no second file	     */
	  {			       /*			     */
	    strcpy(name,AAFB_SMIX);    /* Use source mix device name */
	    source_mix[t] = 1;	       /*			     */
	    if (Query_User(M_35,0L,4L, /* Input Source		     */
			    &value))   /*			     */
	    {			       /*			     */
	      input_source[0] =        /*			     */
	       (unsigned char)	       /*			     */
	       ((value==0)||(value==3) /* Line left and right	     */
	       ||(value==4))?value:3;  /*  not valid for source mix  */
	      input_source[1] =        /* Both tracks must be same   */
		input_source[0];       /*  input source 	     */
	    }			       /*			     */
	  }			       /*			     */
	}			       /*			     */
      } 			       /*			     */
    }				       /*			     */
  }				       /*			     */
  else				       /* Else no second file	     */
  {				       /*			     */
    name[0] = 0;		       /*			     */
  }				       /*			     */
}				       /*			     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Init_ACB					     */
/*								     */
/* FUNCTION = Initialize the Audio Control Block (ACB) and call      */
/*	      the AAPI to set up for a play operation.		     */
/*								     */
/* INPUT =    Ptr to Audio Control Block			     */
/*	      Ptr to File Access Block				     */
/*	      Ptr to PCM Control Block				     */
/*	      Track to process					     */
/*								     */
/* OUTPUT =   0/ Success - ACB initialized for use		     */
/*	   3224/ Failed - No audio device installed		     */
/*	   3225/ Failed - Audio device installed, but ints disabled  */
/*	   3226/ Failed - Device driver not responding (OS/2 only)   */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Init_ACB(acbp,fabp,fmtp,  /*			     */
		    t)		       /*			     */
AAPI_ACB  *acbp;		       /* Ptr to Audio Control Block */
FAB_STRUCT *fabp;		       /* Ptr to file information    */
AAPI_FMT *fmtp; 		       /* Ptr to PCM information     */
short t;			       /* Track 		     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short	     rc;		       /* Return code		     */
AAPI_DEV   adcb;		       /* Audio Device Control Block */
short i;			       /* Loop index		     */
/*********************************************************************/
  rc = SUCCESS; 		       /*			     */
  acbp->acb2ptr = 0L;		       /* No secondary ACB	     */
  aud_init(acbp);		       /* Initialize control blk     */
  acbp->channel  = channel[t];	       /* Set input channel	     */
  acbp->dspmode  = AAPI_PLAY;	       /* Operation is playing	     */
  acbp->seektype = AAPI_INIT;	       /* First call this file	     */
  acbp->controls = AAPI_MVOL+	       /* Set for vol,bal changes    */
    AAPI_TVO1+AAPI_TVO2+AAPI_CBAL;     /*			     */
  acbp->masvol	 = master_volume[t];   /* Master volume 	     */
  acbp->trkvol1  = 0;		       /* Track volume - Set to zero */
  acbp->trkvol1s = 0L;		       /*  initially		     */
  acbp->trkvol1e = 0L;		       /*			     */
  acbp->trkvol2  = volume[t];	       /* Now set user's request     */
  acbp->trkvol2s = 0L;		       /*  with fade in over time    */
  acbp->trkvol2e = volume_rate[t];     /*			     */
  acbp->chnbal	 = balance[t];	       /* Balance		     */
  acbp->chnbals  = 0;		       /*	fade		     */
  acbp->chnbale  = balance_rate[t];    /*			     */
  acbp->outchan  = AAPI_OTBA;	       /*			     */
  acbp->fileptr  = fabp;	       /* Ptr to FAB (file info)     */
  if (speaker[t])		       /* If microchannel speaker    */
  {				       /*  requested		     */
    acbp->oprparms |= AAPI_SPKO;       /* Turn it on		     */
  }				       /*			     */
  if(((atype[0] == AUDO_AVCM) ||       /* If playing voice and MIDI  */
     (atype[1] == AUDO_AVCM)) &&       /*			     */
     (atype[t] == AUDO_ML05))	       /*			     */
  {				       /*			     */
    acbp->oprparms |= AAPI_LDVM;       /* Set up DSP for voice/MIDI  */
  }				       /*			     */
  acbp->audstart = start_pos[t];       /* Starting position	     */
  acbp->audend	 = end_pos[t];	       /* Ending position	     */
  if (loop[t])			       /* If looping requested	     */
  {				       /*			     */
    acbp->listptr = &list[t];	       /*			     */
    list[t].audstart = start_pos[t];   /* Use users requested	     */
    list[t].audend  =  end_pos[t];     /*  start and stop position   */
    list[t].nextlist = &list[t];       /* Loop it		     */
  }				       /*			     */
  acbp->fmtptr= (type[t]==AAFB_WAVE)   /* Pass additional PCM	     */
	       ? fmtp : NULL;	       /*  info if doing PCM	     */
  acbp->inpsrce = input_source[t];     /* Set source		     */
  if (source_mix[t])		       /* If source mixing	     */
  {				       /*			     */
    fmtp->fmtflag |= AAPI_FMT_SMIX;    /*			     */
  }				       /*			     */
  acbp->memid	 = AAPI_MAIN;	       /* Pass an IO buffer	     */
  if ((type[t] == AAFB_WAVE) &&        /*			     */
      ((fmtp->samples_per_second *     /* If doing high data	     */
	(unsigned long)fmtp->channels* /*  rate PCM, then	     */
	(unsigned long) 	       /*  allocate large buffers    */
	(fmtp->bits_per_sample/8)) >=  /*  else use 64K 	     */
	 AAPI_FMT_44KH_W))	       /* Buffers can be allocated   */
				       /*  from 0-64K and then	     */
  {				       /*  in 64K increments	     */
    #ifdef O_2			       /* If in OS/2		     */
    acbp->bufflen  = 0X100000L;        /* Allocate 1meg buffer	     */
    #else			       /* Else in DOS		     */
    acbp->bufflen  = 0X40000L;	       /* Allocate 256 K buffer      */
    #endif			       /*			     */
    if (fmtp->channels ==1)	       /* If mono, may need a buffer */
    {				       /*  for a possible 2nd track  */
      acbp->bufflen /= 2L;	       /* Only use half 	     */
    }				       /*			     */
  }				       /*			     */
  else				       /* Else low data rate file    */
  {				       /*			     */
    acbp->bufflen  = 0X10000L;	       /* Allocate 64 K buffer	     */
  }				       /*			     */
  for(i=0;(i<=2)&&(!lalloc[t]);i++)    /* Make three attempts to     */
  {				       /*  allocate the buffer	     */
    lalloc[t] = (char huge *)	       /*			     */
    halloc(acbp->bufflen,sizeof(char));/*			     */
    if	(lalloc[t] == NULL)	       /* If not enough memory	     */
    {				       /*			     */
      acbp->bufflen /=2L;	       /* Try asking for half	     */
    }				       /*			     */
  }				       /*			     */
  acbp->buffptr = (unsigned char *)    /*			     */
		  lalloc[t];	       /*			     */
  if  (lalloc[t] != NULL)	       /* If allocated buffer	     */
  {				       /*			     */
    if (!t)			       /* If haven't already checked */
    {				       /*  on previous track	     */
      rc = aud_cfig(&adcb);	       /* Get hardware config	     */
    }				       /*			     */
  }				       /*			     */
  else				       /* Else tell caller	     */
  {				       /*			     */
    rc = AAPI_RC_13;		       /* Not enough memory	     */
  }				       /*			     */
  if (rc == SUCCESS)		       /* If card is installed	     */
  {				       /*  and active		     */
    acbp->intlevel = (unsigned char)   /* Set interrupt level	     */
		     adcb.intlev;      /*			     */
    acbp->piobase  = adcb.iobase;      /*  and PIO base address      */
    i = TRUE;			       /*			     */
    while(i)			       /* While not severe error     */
    {				       /*			     */
      rc = aud_set(&acb[t]);	       /* Setup audio operation      */
      if (rc==AAPI_RC_10)	       /* If users stop position     */
      { 			       /*  invalid		     */
	acbp->audend   = 0L;	       /* Use end of file	     */
	if (loop[t])		       /*			     */
	{			       /* Also use entire file for   */
	  list[t].audend = 0L;	       /*  looping		     */
	}			       /*			     */
      } 			       /*			     */
      else			       /* Else			     */
      { 			       /*			     */
       i = FALSE;		       /* Operation ready to go      */
      } 			       /*			     */
    }				       /*			     */
    acbp->controls = 0; 	       /* Reset flags		     */
    acbp->oprparms = 0; 	       /*			     */
  }				       /*			     */
  return(rc);			       /* Return result to caller    */
}				       /*			     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Check I/O					     */
/*								     */
/* FUNCTION = If over half the I/O buffer is full then queue a	     */
/*	      command to do I/O to fill the audio data buffer.	     */
/*								     */
/* INPUT =  Ptr to ACB						     */
/*								     */
/* OUTPUT =   0/ Success					     */
/*	   3207/ Error on file read				     */
/*	   3211/ Error on file seek				     */
/*								     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Check_IO(acbp,tuf_time)   /* 010 Begin Function	     */
AAPI_ACB  *acbp;		       /* Audio Control Block	     */
unsigned long tuf_time; 	       /* Size of data buffers in    */
{				       /* 010			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short rc;			       /* Return code		     */
/*********************************************************************/
  rc =	SUCCESS;		       /*			     */
  if (acbp->timeleft <= (tuf_time/2L)) /* If data buffers are at     */
  {				       /*  least half empty	     */
    acbp->controls = AAPI_PFIO;        /* Set to do I/O only	     */
    acbp->iotime = 0;		       /* Set maximum amount	     */
    rc = aud_ctrl(acbp);	       /* Execute the controls	     */
    acbp->controls = 0; 	       /* Reset flag		     */
    if ((rc == AAPI_RC_17) ||	       /* If I/O not ready, or	     */
	(rc == AAPI_RC_20))	       /*   ctrl ignored, Then	     */
    {				       /*			     */
      rc = SUCCESS;		       /* Non-fatal error	     */
    }				       /*			     */
  }				       /*			     */
  return(rc);			       /* Return to caller	     */
}				       /* End Function		     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Check User					     */
/*								     */
/* FUNCTION = Check user input and process the request. 	     */
/*								     */
/* INPUT =  Ptr to ACBs, termination flag			     */
/*								     */
/* OUTPUT =   0/ Success					     */
/*	      Errors from aud_start or aud_ctrl calls		     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Chk_User(keep_going)      /* Begin Function	     */
short *keep_going;		       /* Main loop flag	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short rc, cloop;		       /*			     */
short ch, pc;			       /* User input, previous input */
unsigned short count;		       /* # of identical input chars */
/*********************************************************************/
  rc =	SUCCESS;		       /*			     */
  pc = ch = 0;			       /*			     */
  count = 0;			       /*			     */
  cloop =1;			       /*			     */
  while((cloop)&&(pc==ch)&&(kbhit()))  /* While more input and the   */
  {				       /*  same character	     */
    ch = getch();		       /*			     */
    if ((ch==0) || (ch==224))	       /* If first byte of extended  */
    {				       /*  code 		     */
      ch = getch();		       /* Get the next byte	     */
    }				       /*			     */
    if (((count == 0) || (pc==ch)) &&  /* If 1st or same character   */
	(count <= 20))		       /*  and not at max	     */
    {				       /*			     */
      count++;			       /* Keep looking for more      */
      pc=ch;			       /*			     */
    }				       /*			     */
    else			       /* Else need to put last      */
    {				       /*  char back		     */
      ungetch(ch);		       /*			     */
      cloop=0;			       /*			     */
      ch=pc;			       /*			     */
    }				       /*			     */
  }				       /*			     */
  switch (ch)			       /* Case of user input	     */
  {				       /*			     */
    case F1 :			       /* Start/Pause/Resume of      */
    case F2 :			       /*  one or both tracks	     */
    case F3 :			       /*   (F1,F2,F3)		     */
      rc = Pause_Resume(ch);	       /*			     */
    break;			       /*			     */
    case NR:			       /* 1st track balance request  */
    case NL:			       /*			     */
    case NH:			       /*			     */
    case NE:			       /*			     */
      if (track_status[0]==1)	       /* If track started	     */
      { 			       /*			     */
	rc = Get_Bal(ch,0,count);      /* Process request else ignore*/
      } 			       /*			     */
    break;			       /*			     */
    case CR:			       /* 2nd track balance request  */
    case CL:			       /*			     */
    case CH:			       /*			     */
    case CE:			       /*			     */
      if (track_status[1]==1)	       /* If track started	     */
      { 			       /*			     */
	rc = Get_Bal(ch,1,count);      /* Process request else ignore*/
      } 			       /*			     */
    break;			       /*			     */
    case NU:			       /* 1st track volume request   */
    case ND:			       /*			     */
    case NP:			       /*			     */
    case NW:			       /*			     */
      if (track_status[0]==1)	       /* If track started	     */
      { 			       /*			     */
	rc = Get_Vol(ch,0,count);      /* Process request else ignore*/
      } 			       /*			     */
    break;			       /*			     */
    case CU:			       /* 2nd track volume request   */
    case CD:			       /*			     */
    case CP:			       /*			     */
    case CW:			       /*			     */
      if (track_status[1]==1)	       /* If track started	     */
      { 			       /*			     */
	rc = Get_Vol(ch,1,count);      /* Process request else ignore*/
      } 			       /*			     */
    break;			       /*			     */
    default :			       /* Else case		     */
      *keep_going = FALSE;	       /* Shut it down		     */
    break;			       /*			     */
  }				       /* End Switch		     */
  return(rc);			       /* Return to caller	     */
}				       /* End Function		     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Get_Vol					     */
/*								     */
/* FUNCTION = Process volume control				     */
/*								     */
/* INPUT =  User request = A   = 1st track - Volume N% up	     */
/*			   V   =		       down	     */
/*		   Ctrl    A   = 2nd track - Volume N% up	     */
/*		   Ctrl    V   =		       down	     */
/*		     Page Up   = 1st track-Volume to 100% over M secs*/
/*		     Page Down =		       0%	     */
/*		Ctrl Page Up   = 2nd track-Volume to 100% over M secs*/
/*		Ctrl Page Down =		       0%	     */
/*								     */
/* OUTPUT =   0/ Success					     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Get_Vol(ch,t,cnt)	       /* Begin Function	     */
short ch;			       /* User input character	     */
short t;			       /* Track being processed      */
unsigned short cnt;		       /* # of chars input	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short rc;			       /* Return code		     */
unsigned short send_control;	       /* Control necessary	     */
/*********************************************************************/
  rc =	SUCCESS;		       /*			     */
  send_control = TRUE;		       /* Don't send control         */
  if((ch==NU) || (ch==CU))	       /* If moving up 5%	     */
  {				       /*			     */
    if (volume[t] < 100)	       /* If not at 100% already     */
    {				       /*			     */
      volume[t]=(volume[t]	       /* Set volume +N% higher      */
		<= 100-(FADE_PERC*cnt))/*			     */
       ? volume[t]+(FADE_PERC*cnt):100;/*			     */
    }				       /*			     */
    else			       /* Else volume at 100%	     */
    {				       /*			     */
      send_control = FALSE;	       /*  Don't send control        */
    }				       /*			     */
  }				       /*			     */
  else				       /* Else may be moving down    */
  {				       /*			     */
    if((ch==ND) || (ch==CD))	       /* If moving down N%	     */
    {				       /*			     */
      if (volume[t] > 0)	       /* If volume not at 0%	     */
      { 			       /*  already		     */
	 volume[t]=(volume[t]	       /* Set Volume -N% down	     */
		>= (FADE_PERC*cnt))    /*			     */
	 ? volume[t]-(FADE_PERC*cnt):0;/*			     */
      } 			       /*			     */
      else			       /* Else volume at 0%	     */
      { 			       /*			     */
	send_control = FALSE;	       /*  Don't send control        */
      } 			       /*			     */
    }				       /*			     */
  }				       /*			     */
  if (send_control)		       /* *f control still necessary */
  {				       /*			     */
    if((ch==NP) || (ch==CP) ||	       /* If ramping over time	     */
       (ch==NW) || (ch==CW))	       /*			     */
    {				       /*			     */
      acb[t].trkvol1e = FADE_TIME;     /* Set time for fade	     */
      if((ch==NP) || (ch==CP))	       /* If moving up		     */
      { 			       /*			     */
	volume[t]=100;		       /* Set max		     */
      } 			       /*			     */
      else			       /* Else			     */
      { 			       /*			     */
	volume[t]=0;		       /* Set min		     */
      } 			       /*			     */
    }				       /*			     */
    else			       /* Else			     */
    {				       /*			     */
      acb[t].trkvol1e = 0L;	       /* Do immediately	     */
    }				       /*			     */
    acb[t].controls = AAPI_TVO1;       /* Set up for volume control  */
    acb[t].trkvol1 = volume[t];        /*			     */
    acb[t].trkvol1s = 0L;	       /* Start at time 0 in file    */
    rc = aud_ctrl(&acb[t]);	       /* Execute the control	     */
    acb[t].controls=0;		       /* Reset control 	     */
    if	(rc == AAPI_RC_20)	       /* If control queue full,     */
    {				       /*			     */
      rc = SUCCESS;		       /* Non-fatal error	     */
    }				       /*			     */
  }				       /*			     */
  return(rc);			       /* Return to caller	     */
}				       /* End Function		     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Get_Bal					     */
/*								     */
/* FUNCTION = Process balance control				     */
/*								     */
/* INPUT =  User request = ->  = 1st track - Balance N% to the right */
/*			   <-  =			       left  */
/*		     Ctrl  ->  = 2nd track - Balance N% to the right */
/*		     Ctrl  <-  =			       left  */
/*			  End  = 1st track-100% to right over M secs */
/*			 Home  =		   left 	     */
/*		    Ctrl  End  = 2nd track-100% to right over M secs */
/*		    Ctrl Home  =		   left 	     */
/*								     */
/* OUTPUT =   0/ Success					     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Get_Bal(ic,t,cnt)	       /* Begin Function	     */
short ic;			       /* User input character	     */
short t;			       /* Track being processed      */
unsigned short cnt;		       /* # of chars input	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short rc;			       /* Return code		     */
unsigned short send_control;	       /* Control necessary	     */
/*********************************************************************/
  rc =	SUCCESS;		       /*			     */
  send_control = TRUE;		       /*			     */
  if((ic==NR) || (ic==CR))	       /* If moving to right N%      */
  {				       /*			     */
    if (balance[t] < 100)	       /* If balance not all the way */
    {				       /*  already		     */
      balance[t]=(balance[t] <=        /* Set balance +N% to right   */
		 100-(FADE_PERC*cnt))  /*			     */
      ? balance[t]+(FADE_PERC*cnt):100;/*			     */
    }				       /*			     */
    else			       /* Else balance all the	     */
    {				       /*  way to the right	     */
      send_control = FALSE;	       /*  Don't send control        */
    }				       /*			     */
  }				       /*			     */
  else				       /* Else moving to left	     */
  {				       /*			     */
    if((ic==NL) || (ic==CL))	       /* If moving to left N%	     */
    {				       /*			     */
      if (balance[t] > 0)	       /* If not all the to the left */
      { 			       /*  already		     */
	balance[t]=(balance[t] >=      /*			     */
		     (cnt*FADE_PERC))  /*			     */
	? balance[t]-(cnt*FADE_PERC):0;/*			     */
      } 			       /*			     */
      else			       /* Else balance all the	     */
      { 			       /*  way to the right	     */
	send_control = FALSE;	       /*  Don't send control        */
      } 			       /*			     */
    }				       /*			     */
  }				       /*			     */
  if (send_control)		       /* If control still necessary */
  {				       /*			     */
    if((ic==NH) || (ic==CH)  ||        /* If panning over time	     */
       (ic==NE) || (ic==CE))	       /*			     */
    {				       /*			     */
      acb[t].chnbale = FADE_TIME;      /* Set time for pan	     */
      if((ic==NE) || (ic==CE))	       /* If going right	     */
      { 			       /*			     */
	balance[t]=100; 	       /* Full right		     */
      } 			       /*			     */
      else			       /* Else			     */
      { 			       /*			     */
	balance[t]=0;		       /* Full Left		     */
      } 			       /*			     */
    }				       /*			     */
    else			       /* Else not panning over time */
    {				       /*			     */
      acb[t].chnbale  = 0L;	       /* Execute immediately	     */
    }				       /*			     */
    acb[t].controls = AAPI_CBAL;       /* Set up for balance control */
    acb[t].chnbal = balance[t];        /*			     */
    acb[t].chnbals = 0L;	       /*			     */
    acb[t].outchan = AAPI_OTBA;        /*			     */
    rc = aud_ctrl(&acb[t]);	       /* Execute the control	     */
    acb[t].controls=0;		       /* Reset control 	     */
    if	(rc == AAPI_RC_20)	       /* If control queue full,     */
    {				       /*			     */
      rc = SUCCESS;		       /* Non-fatal error	     */
    }				       /*			     */
  }				       /*			     */
  return(rc);			       /* Return to caller	     */
}				       /*			     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Pause_Resume 				     */
/*								     */
/* FUNCTION = Initially start or pause/resume one or more tracks.    */
/*								     */
/* INPUT =  User request = F1 = 1st track			     */
/*			   F2 = 2nd track			     */
/*			   F3 = Both tracks			     */
/*								     */
/* OUTPUT =   0/ Success					     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Pause_Resume(ch)	       /*			     */
short ch;			       /* Audio Control Blocks	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short rc;			       /* Return code		     */
short  t,ts,te; 		       /* Track index start, end     */
short  start_both;		       /* Start both tracks T/F      */
/*********************************************************************/
  rc =	SUCCESS;		       /*			     */
  start_both = FALSE;		       /*			     */
  switch (ch)			       /* Case of user input	     */
  {				       /*			     */
    case 59:			       /* 1st track only	     */
      ts = 0;			       /* Set up indexes	     */
      te = 0;			       /*			     */
    break;			       /*			     */
    case 60:			       /* 2nd track only	     */
      ts = 1;			       /*			     */
      te = 1;			       /*			     */
    break;			       /*			     */
    case 61:			       /* Both tracks		     */
      ts = 0;			       /*			     */
      te = 1;			       /*			     */
      if((track_status[0]==0) &&       /* If both tracks need to be  */
	 (track_status[1]==0))	       /*  started?		     */
      { 			       /*			     */
	start_both = TRUE;	       /* Special case		     */
      } 			       /*			     */
    break;			       /*			     */
    default :			       /* Else case		     */
    break;			       /*			     */
  }				       /* End Switch		     */
  if(!start_both)		       /* If not starting both tracks*/
  {				       /*			     */
    for(t=ts;t<=te;t++) 	       /* For each track requested   */
    {				       /*			     */
      if (track_status[t] == 1)        /* If track has been started  */
      { 			       /*			     */
	if (trk_paused[t])	       /* If already paused	     */
	{			       /*			     */
	  acb[t].controls = AAPI_RESM; /* Toggle to resume	     */
	  trk_paused[t] = 0;	       /*			     */
	}			       /*			     */
	else			       /* Else already running	     */
	{			       /*			     */
	  acb[t].controls=AAPI_PAUS;   /* Toggle to pause	     */
	  trk_paused[t] = 1;	       /*			     */
	}			       /*			     */
	rc = aud_ctrl(&acb[t]);        /* Execute the controls	     */
	acb[t].controls=0;	       /* Reset controls	     */
	if  (rc == AAPI_RC_20)	       /* If control queue full,     */
	{			       /*			     */
	  rc = SUCCESS; 	       /* Non-fatal error	     */
	}			       /*			     */
      } 			       /*			     */
      else			       /* else track inactive or     */
      { 			       /*  hasn't been started       */
	if (track_status[t] == 0)      /* If track must be started   */
	{			       /*			     */
	  rc = aud_strt(&acb[t]);      /* Start it		     */
	  if((rc == SUCCESS) ||        /* If no error or incompatible*/
	     (rc == AAPI_RC_29))       /*  file types		     */
	  {			       /*			     */
	    if (rc==SUCCESS)	       /* If no error		     */
	    {			       /*			     */
	      track_status[t] = 1;     /* Remember its started	     */
	    }			       /*			     */
	    else		       /* Else incompatible files    */
	    {			       /*			     */
	      cprintf(M_22);	       /* Tell user but 	     */
	      rc = SUCCESS;	       /*  don't stop other file     */
	    }			       /*			     */
	  }			       /*			     */
	}			       /*			     */
      } 			       /*			     */
    }				       /* Endfor for each track      */
  }				       /*			     */
  else				       /* Else need to start both    */
  {				       /*  tracks		     */
    acb[0].acb2ptr=&acb[1];	       /* Nest the ACBs 	     */
    rc = aud_strt(&acb[0]);	       /* Start then both	     */
    acb[0].acb2ptr=NULL;	       /* Unnest		     */
    if (rc==SUCCESS)		       /* If first start ok	     */
    {				       /*			     */
      track_status[0] = 1;	       /* Set 1st track started      */
      if((acb[0].acb2rc == SUCCESS) || /* If second start ok or      */
	 (acb[0].acb2rc == AAPI_RC_29))/*  files can't play at same  */
      { 			       /*  time 		     */
	if (acb[0].acb2rc==SUCCESS)    /* If no error		     */
	{			       /*			     */
	  track_status[1] = 1;	       /* Remember its started	     */
	}			       /*			     */
	else			       /* Else incompatible files    */
	{			       /*			     */
	  cprintf(M_22);	       /* Tell user but 	     */
	  rc = SUCCESS; 	       /*  don't stop other file     */
	}			       /*			     */
      } 			       /*			     */
      else			       /* Else error on second start */
      { 			       /*			     */
	rc = acb[0].acb2rc;	       /* Return the error	     */
      } 			       /*			     */
    }				       /*			     */
  }				       /*			     */
  return(rc);			       /* Return to caller	     */
}				       /*			     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Write_Error_Message				     */
/*								     */
/* FUNCTION = Determine and write error message to user 	     */
/*								     */
/* INPUT =  AAPI return code					     */
/*								     */
/* OUTPUT = Message written to standard out (user)		     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static void Write_Error_Message(rc)    /*			     */
short rc;			       /* Return code		     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
char *msg;			       /* Ptr to error message	     */
char default_error;		       /* Default error case T/F     */
/*********************************************************************/
  default_error = FALSE;	       /*			     */
  switch (rc)			       /* Case of error code	     */
  {				       /*			     */
    case 1 :			       /* Not a known audio file     */
      msg = M_16;		       /*			     */
    break;			       /*			     */
    case 2 :			       /* Conflicting file types     */
      msg = M_25;		       /*			     */
    break;			       /*			     */
    case 3 :			       /* Conflicting audio types    */
      msg = M_46;		       /*			     */
    break;			       /*			     */
    case AAPI_RC_08 :		       /*			     */
    case AAPI_RC_13 :		       /* Insufficient storage	     */
    case AAFB_RC_04 :		       /*  to complete operation     */
    case AAFB_RC_05 :		       /*			     */
    case AAPI_RC_37 :		       /*			     */
      msg = M_3;		       /*			     */
    break;			       /*			     */
    case AAFB_RC_01 :		       /* File already exists	     */
      msg = M_4;		       /*  and a create requested    */
    break;			       /*			     */
    case AAFB_RC_02 :		       /* File doesn't exist         */
      msg = M_5;		       /*  and an open requested     */
    break;			       /*			     */
    case AAFB_RC_03 :		       /* I/O error on audio file    */
    case AAFB_RC_06 :		       /*			     */
    case AAFB_RC_08 :		       /*			     */
    case AAPI_RC_01 :		       /*			     */
    case AAPI_RC_02 :		       /*			     */
    case AAPI_RC_03 :		       /*			     */
    case AAPI_RC_04 :		       /*			     */
    case AAPI_RC_05 :		       /*			     */
    case AAPI_RC_06 :		       /*			     */
      msg = M_6;		       /*			     */
    break;			       /*			     */
    case AAFB_RC_07 :		       /* I/O error on escape file   */
    case AAPI_RC_07 :		       /*			     */
    case AAPI_RC_11 :		       /*			     */
    case AAPI_RC_19 :		       /*			     */
      if (type[0] == AAFB_AVCF)        /* If AVC type file	     */
      { 			       /*			     */
	msg = M_7;		       /* Escape file message	     */
      } 			       /*			     */
      else			       /* else			     */
      { 			       /*			     */
	msg = M_6;		       /* File message		     */
      } 			       /*			     */
    break;			       /*			     */
    case AAPI_RC_12 :		       /* DSP not responding	     */
      msg = M_8;		       /*			     */
    break;			       /*			     */
    case AAPI_RC_14 :		       /* DSP code not found	     */
      msg = M_9;		       /*			     */
    break;			       /*			     */
    case AAPI_RC_15 :		       /* DSP code file invalid      */
      msg = M_10;		       /*			     */
    break;			       /*			     */
    case AAPI_RC_24 :		       /*			     */
    case AAPI_RC_25 :		       /* Audio card not	     */
      msg = M_11;		       /*  installed		     */
    break;			       /*			     */
    case AAPI_RC_26 :		       /* Device driver not	     */
      msg = M_12;		       /*  responding		     */
    break;			       /*			     */
    case AAPI_RC_34 :		       /* MIDI not allowed in	     */
      msg = M_15;		       /*  DOS or on channel A	     */
    break;			       /*			     */
    case AAPI_RC_29 :		       /* Can not play stereo or     */
      msg = M_22;		       /*  HQ music with other types */
    break;			       /*			     */
    case AAPI_RC_33 :		       /* DSP running and requested  */
      msg = M_23;		       /*  oper requires DSP reload  */
    break;			       /*			     */
    case AAPI_RC_09 :		       /* Start or stop position     */
    case AAPI_RC_10 :		       /* Invalid		     */
      msg = M_40;		       /*			     */
    break;			       /*			     */
    default :			       /* Else case		     */
      msg = M_13;		       /*			     */
      default_error = TRUE;	       /*			     */
    break;			       /*			     */
  }				       /* End Switch		     */
  if (default_error)		       /* If error that we don't     */
  {				       /*  have a message for,	     */
    cprintf(msg,rc);		       /* Tell user the return code  */
  }				       /*			     */
  else				       /* Else			     */
  {				       /*			     */
    cprintf(msg);		       /* Normal message	     */
  }				       /*			     */
}				       /*			     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Set_Options					     */
/*								     */
/* FUNCTION = Query user for volume parameters. 		     */
/*								     */
/* INPUT =    Track being processed				     */
/*								     */
/* OUTPUT =   None.						     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static void Set_Options(t)	       /*			     */
short t;			       /* Track to process	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
unsigned long value;		       /*			     */
/*********************************************************************/
  if (Query_User(M_36,0L,1L,&value))   /* Query if user wants to set */
  {				       /*  volume/balance controls   */
    if(Query_User(M_17,0L,120L,&value))/*			     */
    {				       /* Master volume 	     */
      master_volume[t]= 	       /*			     */
	  (unsigned char) value;       /*			     */
    }				       /*			     */
    if(Query_User(M_18,0L,100L,&value))/*			     */
    {				       /* Track Volume		     */
      volume[t]=(unsigned short)value; /*			     */
    }				       /*			     */
    if(Query_User(M_19,0L,	       /*			     */
		     64L,&value))      /*			     */
    {				       /* Track Volume fade time     */
      volume_rate[t] = 1000L * value;  /*			     */
    }				       /*			     */
    if(Query_User(M_20,0L,100L,&value))/*			     */
    {				       /* Balance		     */
      balance[t]=(unsigned char) value;/*			     */
    }				       /*			     */
    if(Query_User(M_21,0L,	       /*			     */
		     64L,&value))      /*			     */
    {				       /* Balance fade time	     */
	balance_rate[t]=1000L * value; /*			     */
    }				       /*			     */
  }				       /*			     */
  if((!source_mix[0]) &&	       /* If not doing source mix    */
       (!source_mix[1]) &&	       /*  on either track	     */
	(type[t]==AAFB_WAVE))	       /*  and doing PCM	     */
  {				       /*			     */
    if(Query_User(M_34,0L,1L,&value))  /* Query for source mix	     */
    {				       /*			     */
      source_mix[t] = 1;	       /*			     */
      if (Query_User(M_35,0L,4L,       /* Input Source		     */
			  &value))     /*			     */
      { 			       /*			     */
	input_source[0]=(unsigned char)/*			     */
	 ((value==0) || (value==3) ||  /* Line left and right	     */
	 (value==4)) ? value : 3;      /*  not valid for source mix  */
	input_source[1] =	       /* Both tracks must be same   */
		input_source[0];       /*  input source 	     */
      } 			       /*			     */
    }				       /*			     */
  }				       /*			     */
  if (Query_User(M_37,0L,1L,&value))   /* Query if user wants to set */
  {				       /*  other options	     */
    if (!(strcmp(oname[t],AAFB_SMIX)   /* If not doing source mix    */
			   ==0))       /*  only 		     */
    {				       /*			     */
     if(Query_User(M_41,0L,0xFFFFFFFFL,/*			     */
		    &value))	       /* Start Position	     */
     {				       /*			     */
       start_pos[t] = value;	       /*			     */
     }				       /*			     */
     if(Query_User(M_42,0L,0xFFFFFFFFL,/*			     */
		    &value))	       /* End Position		     */
     {				       /*			     */
       end_pos[t]=(value>=start_pos[t])/* If invalid end position    */
	 ? value : 0L;		       /*  use end of file	     */
     }				       /*			     */
     if(Query_User(M_43,0L,1L,&value)) /* Loop the file 	     */
     {				       /*			     */
       loop[t] = (short)value;	       /*			     */
     }				       /*			     */
    }				       /*			     */
    if(Query_User(M_32,1L,2L,&value))  /*			     */
    {				       /* Track Number		     */
      channel[t]=(unsigned char)       /*			     */
		   (value-1L);	       /*			     */
      if((channel[0]==channel[1]) &&   /* If user requested to run   */
	 (channel[0]!=AAPI_CHNS))      /*  both tracks on same	     */
      { 			       /*  signal processor channel  */
	channel[t] = AAPI_CHNS;        /* Ignore his request	     */
      } 			       /*			     */
    }				       /*			     */
    if(Query_User(M_38,1L,1L,&value))  /* Microchannel speaker	     */
    {				       /*			     */
      speaker[t]=(short) value;        /*			     */
    }				       /*			     */
  }				       /*			     */
}				       /*			     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Query_User					     */
/*								     */
/* FUNCTION = Query user for volume parameters. 		     */
/*								     */
/* INPUT =    Message to be processed				     */
/*	      Parameter minimum and maximum values		     */
/*	      Ptr to area to return user response		     */
/*								     */
/* OUTPUT =   rc = 0, User hit Enter				     */
/*	      rc = 1, User input value returned 		     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Query_User(msg,minv,maxv, /*			     */
		       value)	       /*			     */
char * msg;			       /* Query message 	     */
unsigned long minv, maxv, *value;      /* Minimum, maximum,  value   */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
char input_buffer[80];		       /*			     */
short rc;			       /*			     */
/*********************************************************************/
  rc = -1;			       /*			     */
  while (rc==-1)		       /* While user hasn't responded*/
  {				       /*			     */
    cprintf(msg);		       /* Query the user	     */
    gets(input_buffer); 	       /*			     */
    if (sscanf(input_buffer,"%lu",     /* If user typed value        */
	    value) == 1)	       /*			     */
    {				       /*			     */
      if ((*value >= minv) &&	       /* If inputted value is valid */
	   (*value <= maxv))	       /*			     */
      { 			       /*			     */
	rc = 1; 		       /* Use it		     */
      } 			       /*			     */
      else			       /* Else invalid		     */
      { 			       /*			     */
	cprintf(M_39);		       /* Tell user try again	     */
      } 			       /*			     */
    }				       /*			     */
    else			       /* Else user hit enter to     */
    {				       /*  accept default	     */
      rc = 0;			       /*			     */
    }				       /*			     */
  }				       /*			     */
  return(rc);			       /*			     */
}				       /*			     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Get_Audio_Type				     */
/*								     */
/* FUNCTION = Get the audio object and return the audio compression  */
/*	      type.						     */
/*								     */
/* INPUT =    File Access Block Pointer 			     */
/*								     */
/* OUTPUT =   Compression type					     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Get_Audio_Type(fab_ptr)   /*			     */
FAB_STRUCT *fab_ptr;		       /* Pointer to FAB	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
FABO_STRUCT *afop;		       /* Ptr to Audio FABO	     */
unsigned short aud_idx; 	       /* Ptr to Audio FABO index    */
AUDO_HDR_STRUCT *aud_hdrp;	       /* Audio header pointer	     */
/*********************************************************************/
  FABO_Find(fab_ptr,		       /* Find ptr and index to      */
	    FABO_AOBJ,		       /*  FABO of audio object      */
	    0,			       /*			     */
	    &(afop),		       /*			     */
	    &(aud_idx));	       /*			     */
  aud_hdrp= (AUDO_HDR_STRUCT *)        /* Locate audio hdr ptr	     */
	     afop->fohdptr;	       /*			     */
  return(			       /*			     */
   aud_hdrp->aud_comp!=AAPI_DEFT ?     /* Return the audio	     */
  (short)aud_hdrp->aud_comp:AAPI_SM11);/*  compression type	     */
}				       /*			     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = FABO_Find					     */
/*								     */
/* FUNCTION = Searchs a FABO list and returns an index and a	     */
/*	      pointer to a FABO with requested type and subtype.     */
/*								     */
/* INPUT =    Pointer to File Access Block			     */
/*	      Object type to match				     */
/*	      Object sub-type to match				     */
/*	      Pointer to field to place pointer to FABO 	     */
/*	      Pointer to field to place index of FABO		     */
/*								     */
/* OUTPUT =   rc = 0 - Successful, object found 		     */
/*		   1 - Object type found, subtype not found	     */
/*		   2 - Object type not found			     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
short FABO_Find(fab_ptr,type,subtype,  /*			     */
	      obj_ptr_ptr,idx_ptr)     /*			     */
FAB_STRUCT *fab_ptr;		       /* Pointer to FAB	     */
unsigned short type;		       /* FABO type to look for      */
unsigned short subtype; 	       /* Sub type to look for	     */
FABO_STRUCT * *obj_ptr_ptr;	       /* Pointer to ptr for FABO    */
unsigned short *idx_ptr;	       /* Pointer to FABO index      */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short rc;			       /* Return code from operation */
short fabo_cnt; 		       /* # of FABOs		     */
FABO_STRUCT *fabo_ptr;		       /* Ptr to FABO		     */
/*********************************************************************/
  rc = 2;			       /* Nothing found so far	     */
  fabo_cnt = 0; 		       /* Init index #		     */
  while ((fabo_cnt < fab_ptr->fabcnt)  /* While still FABOs to	     */
	&& (rc != SUCCESS))	       /*  search and haven't        */
  {				       /*  found requested FABO      */
    fabo_ptr=((FABO_STRUCT *)	       /* Get ptr to  FABO	     */
	      (fab_ptr + 1)) +	       /*			     */
	      fabo_cnt; 	       /*			     */
    if (fabo_ptr->fotype == type)      /* If types match	     */
    {				       /*			     */
      rc = 1;			       /* Set types match	     */
      if (fabo_ptr->fosub == subtype)  /* If subtype matches	     */
      { 			       /*			     */
	rc = SUCCESS;		       /* Set object found	     */
	*idx_ptr = fabo_cnt;	       /* Set index to FABO	     */
	*obj_ptr_ptr = fabo_ptr;       /* Set ptr to FABO	     */
      } 			       /*			     */
    }				       /*			     */
    fabo_cnt++; 		       /* Get next index	     */
  }				       /* Enddo 		     */
  return(rc);			       /*			     */
}				       /*			     */
/*********************************************************************/
