/*********************************************************************/
/*								     */
/*  MODULE NAME = PLAY.C					     */
/*								     */
/*  DESCRIPTIVE NAME = Audio Application Programming Interface	     */
/*		       Sample Play Program			     */
/*								     */
/*  COPYRIGHT = (C) Copyright by IBM Corp. 1989, 1991.		     */
/*		All rights reserved.				     */
/*		Refer to "LICENSE.DOC" for information regarding     */
/*		the use of this file.				     */
/*								     */
/*  FUNCTION = Provides the following functions/tools:		     */
/*								     */
/*	       play - Provides a simple working example of	     */
/*		      playing using the AVC Audio API.		     */
/*								     */
/*  CONVENTIONS = See Below					     */
/*								     */
/*    Constants - upper case					     */
/*    Internal variables - lower case				     */
/*    External variables - as required				     */
/*    Internal functions - initial caps 			     */
/*    External functions - as required				     */
/*								     */
/*								     */
/*  CHANGE ACTIVITY =  See Below				     */
/*								     */
/*		       01/11/90, AVC Release 1.02		     */
/*		       07/02/90, AVC Release 1.03		     */
/*			 Added MIDI error message		     */
/*		       01/25/91, M-ACPA Release 2.0		     */
/*			 Added RIFF WAVE file and source mix support */
/*								     */
/*********************************************************************/
/*								     */
/*************************SYSTEM INCLUDES*****************************/
/*								     */
#define LINT_ARGS		       /* Perform type checking on   */
				       /*  library functions	     */
/*								     */
/*								     */
#include <fcntl.h>		       /*			     */
#include <sys\types.h>		       /*			     */
#include <sys\stat.h>		       /*			     */
#include <conio.h>		       /*			     */
#include <io.h> 		       /*			     */
#include <stdio.h>		       /*			     */
#include <errno.h>		       /*			     */
#include <stdlib.h>		       /*			     */
#include <string.h>		       /*			     */
#include <time.h>		       /*			     */
#include <malloc.h>		       /*			     */
/*								     */
/**************************AAPI INCLUDES******************************/
/*								     */
#include "play.h"                      /* Program messages           */
#include "eapidbas.cnc"                /* Audio file structures      */
#include "eapiacb.cnc"                 /* Audio constants/declares   */
/*								     */
/*********************CONSTANT DEFINITIONS****************************/
/*								     */
#define TRUE			1      /* True constant 	     */
#define FALSE			0      /* False constant	     */
#define SUCCESS 		0      /* Function call successful   */
/*								     */
#define AVC_EXT  "._AU"                /* AVC audio file extension   */
#define RIFF_EXT ".WAV"                /* RIFF Audio file extension  */
/*								     */
/*****************VARIABLE DECLARATION/DEFINITIONS********************/
/*								     */
short	 master_volume	 =	100;	/* Default volume and	      */
short	 track_1_volume  =	100;	/*  balance settings	      */
short	 track_1_rate	 =	 0;	/*			      */
short	 balance	 =	 50;	/*			      */
short	 balance_rate	 =	 0;	/*			      */
char huge *lalloc	=	NULL;  /* Ptr to data buffer > 64K   */
unsigned long buf_time; 	       /* Size of data buffers in    */
				       /*  milliseconds 	     */
/*								     */
/******************RESIDENT FUNCTION DECLARATIONS*********************/
/*								     */
static void Get_File_Name(char *);     /* Get output file name	     */
/*								     */
static short Init_ACB(AAPI_ACB *,      /* Initialize Audio Control   */
 FAB_STRUCT*,AAPI_FMT*,unsigned short);/*  Block		     */
/*								     */
static short Check_IO(AAPI_ACB *);     /* Check if I/O needed	     */
/*								     */
static void Write_Error_Message(short);/* Put error message to screen*/
/*								     */
static void Query_Settings();	       /* Query user for settings    */
/*								     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = play 					     */
/*								     */
/* FUNCTION = Initializes the audio card and opens an audio file     */
/*	      and begins reading and playing audio data.	     */
/*								     */
/* INPUT =    None - The user is prompted for the name of a file     */
/*	      that contains the input audio data.  The user is	     */
/*	      also prompted for the initial volume and balance	     */
/*	      controls. 					     */
/*								     */
/*	      The file must exist and be one of the following	     */
/*	       types of audio files:				     */
/*								     */
/*	      - AVC						     */
/*	      - RIFF WAVE					     */
/*								     */
/*	      If no extension is used the default AVC extension      */
/*	      of _AU is appended to the name.			     */
/*								     */
/*	      If no file name is inputted the default is to use      */
/*	      the "source mix" capability.  This allows control      */
/*	      of the M-ACPA input source as if it were a PCM file.   */
/*								     */
/*	      User hits any key to stop operation.		     */
/*								     */
/* OUTPUT =   If successful					     */
/*		Audio file contents played until user hits any key   */
/*								     */
/*	      Else error message written to screen		     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*				PDL				     */
/*********************************************************************/
/*********************************************************************/
/* 010 MAIN FUNCTION						     */
/* 020 | Get an audio file name from the user			     */
/* 030 | Determine the type of file (AVC, RIFF WAVE)		     */
/* 040 | IF no error, THEN					     */
/* 050 | | CALL the AAPI to open and initialize the audio file	     */
/* +   | |    and any associated objects			     */
/* 060 | ENDIF							     */
/* 070 | IF no error, THEN					     */
/* 080 | | Query the user for volume and balance settings	     */
/* 090 | | Initialize the Audio Control Block for playing	     */
/* 100 | | IF no error, THEN					     */
/* 110 | | | CALL the AAPI to set up the play operation 	     */
/* 120 | | ENDIF						     */
/* 130 | | IF no error, THEN					     */
/* 140 | | | CALL the AAPI to start playing			     */
/* 150 | | | IF no error, THEN					     */
/* +   | | | | (The AAPI is now running in the background.  The      */
/* +   | | | |	application can now monitor the AAPI or do whatever  */
/* +   | | | |	application specific work that needs to be done.     */
/* +   | | | |	This program has no real work to do so it will sit   */
/* +   | | | |	in a loop waiting for the user to hit a key.	     */
/* +   | | | |	The AAPI will do file I/O and keep the card playing  */
/* +   | | | |	until there is no more audio data in the file	     */
/* +   | | | |	or it hits the stop point you requested.	     */
/* +   | | | |	You can stop it at any point or after it	     */
/* +   | | | |	has run out of data.  You can control things like    */
/* +   | | | |	volume and balance while it is playing by doing      */
/* +   | | | |	"audio control" calls.)                              */
/* 160 | | | | WHILE user has not quit (hit any key)		     */
/* +   | | | | |      and still playing successfully		     */
/* 170 | | | | | Write elapsed time to screen			     */
/* 180 | | | | | Check if I/O needed for card.			     */
/* +   | | | | | (This is optional.  The AAPI will do this for you   */
/* +   | | | | |  if you don't request that file I/O be done.        */
/* +   | | | | |  One reason you might want to control when I/O is   */
/* +   | | | | |  done is if you are writing (dissolving) images to  */
/* +   | | | | |  the screen and you want this to look smooth not    */
/* +   | | | | |  jerky. You can request that I/O be done just	     */
/* +   | | | | |  before you begin the dissolve so that no I/O is    */
/* +   | | | | |  done during the dissolve to disrupt it. When	     */
/* +   | | | | |  playing high data rate audio (> 44KBS) it may      */
/* +   | | | | |  always be necessary to make this call).	     */
/* 190 | | | | ENDDO						     */
/* 200 | | | | Check background return code			     */
/* 210 | | | | IF not already stopped, THEN			     */
/* 220 | | | | | CALL the AAPI to stop playing immediately	     */
/* 230 | | | | | Wait for stop to complete			     */
/* 240 | | | | ENDIF						     */
/* 250 | | | ENDIF						     */
/* 260 | | ENDIF						     */
/* 270 | | IF AAPI was initialized, THEN			     */
/* 280 | | | CALL the AAPI to terminate audio operation 	     */
/* 290 | | ENDIF						     */
/* 300 | | IF file was opened, THEN				     */
/* 310 | | | CALL the AAPI to close the audio file and		     */
/* +   | | |   deallocate the associated audio objects		     */
/* 320 | | ENDIF						     */
/* 330 | | IF data buffer allocated, THEN			     */
/* 340 | | | Free the buffer					     */
/* 350 | | ENDIF						     */
/* 360 | ENDIF							     */
/* 370 | IF error occurred, THEN				     */
/* 380 | | Write the appropriate error message to user		     */
/* 390 | ENDIF							     */
/* 400 ENDFUNCTION MAIN 					     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
main()				       /* 010 Begin Function	     */
{				       /* 010			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short rc;			       /* Return code		     */
char aname[81]; 		       /* Audio file name	     */
FAB_STRUCT *fabp;		       /* Ptr to File Access Block   */
AAPI_ACB   acb; 		       /* Audio Control Block	     */
short cur_pos;			       /* Current position (seconds) */
long stime;			       /* Wait time start	     */
short  w;			       /* Wait loop index	     */
short  close_file;		       /* File needs to be closed    */
short  term_audio;		       /* AAPI need to be terminated */
unsigned short	type;		       /* File type to play	     */
AAPI_FMT fmt;			       /* PCM format data	     */
/*********************************************************************/
  cprintf(M_22); cprintf(M_23);        /* 020 Program title	     */
  close_file = FALSE;		       /* 020 No open file yet	     */
  term_audio = FALSE;		       /* 020 Audio not initialized  */
  Get_File_Name(aname); 	       /* 020 Get name of audio file */
  rc = fab_type(aname,&type);	       /* 030 Find out its type      */
  if (rc == SUCCESS)		       /* 040 If no error	     */
  {				       /* 040			     */
    switch (type)		       /* 050 Case of file type      */
    {				       /* 050			     */
      case AAFB_AVCF  : 	       /* 050 AVC		     */
      rc = fab_open(aname,	       /* 050 Call AAPI file function*/
	     AAFB_OPEN,AAFB_EXNO,0,    /* 050  to initialize file and*/
	     &fabp,0,0,0,0);	       /* 050  associated structures */
      break;			       /* 050			     */
      case AAFB_WAVE  : 	       /* 050 RIFF WAVE 	     */
      rc = fab_ropn(aname,	       /* 050			     */
	     AAFB_OPEN,&fabp,&fmt);    /* 050			     */
      break;			       /* 050			     */
      case AAFB_NONE  : 	       /* 050 File does not exist    */
      rc = AAFB_RC_02;		       /* 050			     */
      break;			       /* 050			     */
      case AAFB_UKNW  : 	       /* 050 Unknown file type      */
      default : 		       /* 050 Else case 	     */
      rc = AAFB_RC_07;		       /* 050			     */
      break;			       /* 050			     */
    }				       /* 050 End Switch	     */
  }				       /* 060			     */
  if (rc == SUCCESS)		       /* 070 If no error opening    */
  {				       /* 070  the audio file	     */
    close_file = TRUE;		       /* 080 Remember to close file */
    Query_Settings();		       /* 080 Get user vol/bal	     */
    rc = Init_ACB(&acb,fabp,&fmt,type);/* 090 Initialize Audio CB    */
    if (rc == SUCCESS)		       /* 100 If no error	     */
    {				       /* 100			     */
      rc = aud_set(&acb);	       /* 110 Set up audio operation */
    }				       /* 120			     */
    if (rc == SUCCESS)		       /* 130 If no error on set up  */
    {				       /* 130			     */
      buf_time = acb.timeleft >= 100L  /* 140 Save max time before   */
	? acb.timeleft : 100L;	       /* 140  I/O is required	     */
      term_audio = TRUE;	       /* 140 Audio is initialized   */
      cprintf(M_2);		       /* 140 Tell user how to stop  */
      cprintf(M_14,0);		       /* 140  and current position  */
      rc = aud_strt(&acb);	       /* 140 Start playing	     */
      if (rc==SUCCESS)		       /* 150 If started playing     */
      { 			       /* 150			     */
	cur_pos = acb.position/1000L;  /* 160 Nothing played yet     */
	while ((!kbhit()) &&	       /* 160 While user hasn't quit */
	  (acb.state==AAPI_PLAYING) && /* 160  and still playing     */
	  (rc == SUCCESS))	       /* 160  and no errors yet     */
	{			       /* 160			     */
	  if((short)(acb.position      /* 170 If playing next second */
	     /1000L) > cur_pos)        /* 170  yet		     */
	  {			       /* 170			     */
	    cur_pos++;		       /* 170 Tell user elapsed time */
	    cprintf(M_14,cur_pos);     /* 170  in seconds	     */
	  }			       /* 170			     */
	  for(w=0;w<=4000;w++);        /* 170 Don't badger OS        */
	  rc = Check_IO(&acb);	       /* 180 Optional call to	     */
				       /* 180  request I/O. If high  */
				       /* 180  data rate PCM is      */
				       /* 180  being played then this*/
				       /* 180  call should be done.  */
	}			       /* 190 Enddo		     */
	if (acb.state != AAPI_STOPPED) /* 210 If not already stopped */
	{			       /* 210			     */
	  acb.controls = AAPI_STOP;    /* 220 Set stop position so   */
	  acb.stoppos  = 0L;	       /* 220  an immediate stop     */
	  acb.ctlparms = AAPI_STPI;    /* 220  will be executed      */
	  aud_ctrl(&acb);	       /* 220 Execute the stop	     */
	  stime = clock();	       /* 230 Wait for stop to	     */
	  while(		       /* 230  complete before	     */
	  (acb.state!=AAPI_STOPPED) && /* 230  continuing, do not    */
	  ((clock() - stime) < 30000)) /* 230  badger OS while	     */
	  for(w=0;w<=1000;w++);        /* 230  waiting		     */
	}			       /* 240			     */
	if ((rc == SUCCESS) &&	       /* 200 If an error occurred   */
	     (acb.backrc != SUCCESS))  /* 200  in background while   */
	{			       /* 200  playing		     */
	   rc = acb.backrc;	       /* 200 Tell caller	     */
	}			       /* 200			     */
      } 			       /* 250			     */
    }				       /* 260			     */
    if (term_audio)		       /* 270 If no error occurred   */
    {				       /* 270			     */
      aud_term(&acb);		       /* 280 Terminate AAPI	     */
    }				       /* 290			     */
    if (close_file)		       /* 300 If file was opened     */
    {				       /* 300			     */
      fab_close(fabp);		       /* 310 Close audio file	     */
    }				       /* 320			     */
    if (lalloc != NULL) 	       /* 330 If allocated buffer    */
    {				       /* 330			     */
      hfree(lalloc);		       /* 340 Free it		     */
    }				       /* 350			     */
  }				       /* 360			     */
  if (rc != SUCCESS)		       /* 370 If error occurred      */
  {				       /* 370			     */
    Write_Error_Message(rc);	       /* 380 Tell user 	     */
  }				       /* 390			     */
}				       /* 400 End Function	     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Init_ACB					     */
/*								     */
/* FUNCTION = Initialize the Audio Control Block (ACB) for a	     */
/*	      play operation.					     */
/*								     */
/* INPUT =    Ptr to ACB					     */
/*	      Ptr to FAB					     */
/*								     */
/* OUTPUT =   0/ Success - ACB initialized for use		     */
/*	   3224/ Failed - No audio device installed		     */
/*	   3225/ Failed - Audio device installed, but ints disabled  */
/*	   3226/ Failed - Device driver not responding (OS/2 only)   */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*				PDL				     */
/*********************************************************************/
/*********************************************************************/
/* 010 Init_ACB FUNCTION					     */
/* 020 | CALL AAPI to initialize Audio Control Block		     */
/* 030 | Set ACB fields needed for playing operation		     */
/* 040 | Allocate and pass data buffers to AAPI 		     */
/* 050 | IF no error, THEN					     */
/* 060 | | Call AAPI to get the current hardware configuration	     */
/* 070 | ELSE out of memory					     */
/* 080 | | Tell caller						     */
/* 090 | ENDIF							     */
/* 100 | IF valid configuration and card/device driver responding    */
/* 110 | | Put configuration information into ACB fields	     */
/* 120 | ENDIF							     */
/* 130 | RETURN result to caller				     */
/* 140 ENDFUNCTION Init_ACB					     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Init_ACB(acbp,fabp,fmtp,  /* 010 Begin Function	     */
			      type)    /*			     */
AAPI_ACB  *acbp;		       /* Audio Control Block	     */
FAB_STRUCT *fabp;		       /* Ptr to File Access Block   */
AAPI_FMT *fmtp; 		       /* Ptr to PCM format info     */
unsigned short type;		       /* Type of file requested     */
{				       /* 010			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short	     rc;		       /* Return code		     */
AAPI_DEV   adcb;		       /* Audio Device Control Block */
short i;			       /* Loop index		     */
/*********************************************************************/
  acbp->acb2ptr = 0L;		       /* 020 No secondary ACB	     */
  aud_init(acbp);		       /* 020 Initialize control blk */
  acbp->channel  = AAPI_CHNS;	       /* 030 Set input channel      */
  acbp->dspmode  = AAPI_PLAY;	       /* 030 Operation is playing   */
  acbp->seektype = AAPI_INIT;	       /* 030 First call this file   */
  acbp->controls = AAPI_MVOL+AAPI_TVO1 /* 030 Set for volume and     */
			   +AAPI_CBAL; /* 030  balance changes	     */
  acbp->masvol	 = master_volume;      /* 030 Set master volume,     */
  acbp->trkvol1  = track_1_volume;     /* 030	  track volume	     */
  acbp->trkvol1s = 0;		       /* 030	  with fade request, */
  acbp->trkvol1e = track_1_rate;       /* 030			     */
  acbp->chnbal	 = balance;	       /* 030	  balance with	     */
  acbp->chnbals  = 0;		       /* 030	  fade request	     */
  acbp->chnbale  = balance_rate;       /* 030			     */
  acbp->outchan  = AAPI_OTAB;	       /* 030			     */
  acbp->oprparms = 0;		       /* 030 No operation controls  */
  acbp->fileptr  = fabp;	       /* 030 Ptr to FAB (file info) */
  acbp->subtype  = 0;		       /* 030 Subtype to use	     */
  acbp->audstart = 0L;		       /* 030 Start at beginning     */
  acbp->audend	 = 0L;		       /* 030 No stop position	     */
  acbp->fmtptr	 = (type == AAFB_WAVE) /* 030 Pass additional PCM    */
		   ? fmtp : NULL;      /* 030  info if doing PCM     */
  acbp->memid	 = AAPI_MAIN;	       /* 040 Pass an IO buffer      */
  if ((type == AAFB_WAVE) &&	       /* 040 If doing high data     */
      ((fmtp->samples_per_second *     /* 040  rate PCM, then	     */
	(unsigned long)fmtp->channels* /* 040  allocate large buffers*/
	(unsigned long) 	       /* 040  else use 64K	     */
	(fmtp->bits_per_sample/8)) >   /* 040			     */
	 AAPI_FMT_44KH_W))	       /* 040			     */
  {				       /* 040			     */
				       /* 040 Buffrs can be allocated*/
				       /* 040  from 0-64K and then   */
				       /* 040  in 64K increments     */
				       /* 040			     */
    #ifdef O_2			       /* 040 If in OS/2	     */
    acbp->bufflen  = 0X100000L;        /* 040 Allocate 1 meg buffer  */
    #else			       /* 040 Else in DOS	     */
    acbp->bufflen  = 0X40000L;	       /* 040 Allocate 256 K buffer  */
    #endif			       /* 040			     */
  }				       /* 040			     */
  else				       /* 040 Else low data rate file*/
  {				       /* 040			     */
    acbp->bufflen  = 0X10000L;	       /* 040 Allocate 64 K buffer   */
  }				       /* 040			     */
  for(i=0;(i<=2)&&(!lalloc);i++)       /* 040 Make three attempts to */
  {				       /* 040  allocate the buffer   */
    lalloc = (char huge *)	       /* 040			     */
    halloc(acbp->bufflen,sizeof(char));/* 040			     */
    if	(lalloc == NULL)	       /* 040 If not enough memory   */
    {				       /* 040			     */
      acbp->bufflen /=2L;	       /* 040 Try asking for half    */
    }				       /* 040			     */
  }				       /* 040			     */
  acbp->buffptr = (unsigned char *)    /* 040			     */
		  lalloc;	       /* 040			     */
  if (lalloc != NULL)		       /* 050 If allocated buffer    */
  {				       /* 050			     */
    rc = aud_cfig(&adcb);	       /* 060 Get hardware config    */
  }				       /* 070			     */
  else				       /* 070 Else tell caller	     */
  {				       /* 070			     */
    rc = AAPI_RC_13;		       /* 080 Not enough memory      */
  }				       /* 090 Endif		     */
  if (rc == SUCCESS)		       /* 100 If card is installed   */
  {				       /* 100  and active	     */
    acbp->intlevel = (unsigned char)   /* 110 Set interrupt level    */
		     adcb.intlev;      /* 110			     */
    acbp->piobase  = adcb.iobase;      /* 110  and PIO base address  */
  }				       /* 120			     */
  return(rc);			       /* 130 Return result to caller*/
}				       /* 140 End Function	     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Check I/O					     */
/*								     */
/* FUNCTION = If over half the I/O buffer is empty then queue a      */
/*	      command to do I/O to fill audio data buffer.	     */
/*								     */
/* INPUT =  Ptr to ACB						     */
/*								     */
/* OUTPUT =   0/ Success					     */
/*	   3207/ Error on file read				     */
/*	   3211/ Error on file seek				     */
/*	   3220/ Error on file write				     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Check_IO(acbp)	       /* Begin Function	     */
AAPI_ACB  *acbp;		       /* Audio Control Block	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short rc;			       /* Return code		     */
/*********************************************************************/
  rc =	SUCCESS;		       /*			     */
  if (acbp->timeleft <= (buf_time/2L)) /* If data buffers is at      */
  {				       /*  least half empty	     */
    acbp->controls = AAPI_PFIO;        /* Set to do I/O only	     */
    acbp->iotime = 0;		       /* Set maximum amount	     */
    rc = aud_ctrl(acbp);	       /* Execute the controls	     */
    if ((rc == AAPI_RC_17) ||	       /* If I/O not ready, or	     */
	(rc == AAPI_RC_20))	       /*   ctrl ignored, Then	     */
    {				       /*			     */
      rc = SUCCESS;		       /* Non-fatal error	     */
    }				       /*			     */
  }				       /*			     */
  return(rc);			       /* Return to caller	     */
}				       /* End Function		     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Get_File_Name				     */
/*								     */
/* FUNCTION = Query user for audio file name. Append default	     */
/*	      AVC extension if no extension if given.		     */
/*	      If no name was typed, then the default is the Source   */
/*	      Mix device.					     */
/*								     */
/* INPUT =    Ptr to area for name				     */
/*								     */
/* OUTPUT =   None - name returned in callers area		     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static void Get_File_Name(name)        /* Begin Function	     */
char *name;			       /* Audio File name	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
char *end;			       /* Ptr to end of name	     */
char input_buffer[80];		       /*			     */
/*********************************************************************/
  cprintf(M_1); 		       /* Query and get the	     */
  gets(input_buffer);		       /*  file name		     */
  if(sscanf(input_buffer,"%s",name)    /* If name was typed          */
     ==1)			       /*			     */
  {				       /*			     */
    if(!(((end=strrchr(name,'.'))      /* If name has no             */
      !=0) &&			       /*  extension		     */
      (strrchr(end,'\\')==0)))         /*                            */
    {				       /*			     */
      if(access(name,0) != SUCCESS)    /* Check for existance	     */
      { 			       /* If not found try RIFF      */
	strcat(name,RIFF_EXT);	       /*  extension		     */
	if(access(name,0) != SUCCESS)  /* Check for existance	     */
	{			       /* If not found try AVC	     */
	  strcpy(&name[strlen(name)    /*  extension		     */
	   -4],AVC_EXT);	       /*			     */
	}			       /*			     */
      } 			       /*			     */
    }				       /*			     */
  }				       /*			     */
  else				       /* Else user hit enter asking */
  {				       /*  for Source Mix	     */
    strcpy(name,AAFB_SMIX);	       /* Use source mix device name */
  }				       /*			     */
  return;			       /* Return result to caller    */
}				       /* End Function		     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Query_Settings				     */
/*								     */
/* FUNCTION = Query user for volume and balance settings.	     */
/*								     */
/* INPUT =    None						     */
/*								     */
/* OUTPUT =   None						     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static void Query_Settings()	       /* Begin Function	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
char		input_buffer[80];      /*			     */
short		input_value;	       /*			     */
/*********************************************************************/
  cprintf(M_17);		       /* Query and get the master   */
  gets(input_buffer);		       /*  volume value 	     */
  if (sscanf(input_buffer,	       /* If user typed value then   */
      "%d",&input_value) == 1)         /*   use it else use default  */
  {				       /*			     */
    if ((input_value >= 0) &&	       /* If inputted value is valid */
	(input_value <= 120))	       /*			     */
    {				       /*			     */
      master_volume = input_value;     /* Use it		     */
    }				       /*			     */
  }				       /*			     */
  cprintf(M_20);		       /* Query and get the balance  */
  gets(input_buffer);		       /*  setting		     */
  if (sscanf(input_buffer,	       /*			     */
      "%d",&input_value) == 1)         /*                            */
  {				       /*			     */
    if ((input_value >= 0) &&	       /*			     */
	(input_value <= 100))	       /*			     */
    {				       /*			     */
      balance = 100 - input_value;     /*			     */
    }				       /*			     */
  }				       /*			     */
  return;			       /*			     */
}				       /* End Function		     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Write_Error_Message				     */
/*								     */
/* FUNCTION = Determine and write error message to user 	     */
/*								     */
/* INPUT =  AAPI return code					     */
/*								     */
/* OUTPUT = Message written to standard out (user)		     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static void Write_Error_Message(rc)    /* Begin Function	     */
short rc;			       /* Return code		     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
char *msg;			       /* Ptr to error message	     */
char default_error;		       /* Default error case T/F     */
/*********************************************************************/
    default_error = FALSE;	       /*			     */
    switch (rc) 		       /* Case of error code	     */
    {				       /*			     */
      case AAPI_RC_08 : 	       /*			     */
      case AAPI_RC_13 : 	       /* Insufficient storage	     */
      case AAFB_RC_04 : 	       /*  to complete operation     */
      case AAFB_RC_05 : 	       /*			     */
      case AAPI_RC_37 : 	       /*			     */
	msg = M_3;		       /*			     */
      break;			       /*			     */
      case AAFB_RC_01 : 	       /* File already exists	     */
	msg = M_4;		       /*  and a create requested    */
      break;			       /*			     */
      case AAFB_RC_02 : 	       /* File doesn't exist         */
	msg = M_5;		       /*  and an open requested     */
      break;			       /*			     */
      case AAFB_RC_03 : 	       /* I/O error on audio file    */
      case AAFB_RC_06 : 	       /*			     */
      case AAFB_RC_08 : 	       /*			     */
      case AAPI_RC_01 : 	       /*			     */
      case AAPI_RC_02 : 	       /*			     */
      case AAPI_RC_03 : 	       /*			     */
      case AAPI_RC_04 : 	       /*			     */
      case AAPI_RC_05 : 	       /*			     */
      case AAPI_RC_06 : 	       /*			     */
	msg = M_6;		       /*			     */
      break;			       /*			     */
      case AAFB_RC_07 : 	       /* I/O error on escape file   */
      case AAPI_RC_07 : 	       /*			     */
      case AAPI_RC_11 : 	       /*			     */
      case AAPI_RC_19 : 	       /*			     */
	msg = M_7;		       /*			     */
      break;			       /*			     */
      case AAPI_RC_12 : 	       /* DSP not responding	     */
	msg = M_8;		       /*			     */
      break;			       /*			     */
      case AAPI_RC_14 : 	       /* DSP code not found	     */
	msg = M_9;		       /*			     */
      break;			       /*			     */
      case AAPI_RC_15 : 	       /* DSP code file invalid      */
	msg = M_10;		       /*			     */
      break;			       /*			     */
      case AAPI_RC_24 : 	       /*			     */
      case AAPI_RC_25 : 	       /* Audio card not	     */
	msg = M_11;		       /*  installed		     */
      break;			       /*			     */
      case AAPI_RC_26 : 	       /* Device driver not	     */
	msg = M_12;		       /*  responding		     */
      break;			       /*			     */
      case AAPI_RC_34 : 	       /* MIDI not allowed in	     */
	msg = M_15;		       /*  DOS or on channel A	     */
      break;			       /*			     */
      case AAPI_RC_29 : 	       /* Can not play stereo or     */
	msg = M_19;		       /*  HQ music with other types */
      break;			       /*			     */
      case AAPI_RC_33 : 	       /* DSP running and requested  */
	msg = M_21;		       /*  oper requires DSP reload  */
      break;			       /*			     */
      case AAPI_RC_09 : 	       /* Start or stop position     */
      case AAPI_RC_10 : 	       /* Invalid		     */
	msg = M_18;		       /*			     */
      break;			       /*			     */
      default : 		       /* Else case		     */
	msg = M_13;		       /*			     */
	default_error = TRUE;	       /*			     */
      break;			       /*			     */
    }				       /* End Switch		     */
    if (default_error)		       /* If error that we don't     */
    {				       /*  have a message for,	     */
      cprintf(msg,rc);		       /* Tell user the return code  */
    }				       /*			     */
    else			       /* Else			     */
    {				       /*			     */
      cprintf(msg);		       /* Normal message	     */
    }				       /* Endif 		     */
    return;			       /* Return to caller	     */
}				       /* End Function		     */
/*********************************************************************/
