/*********************************************************************/
/*								     */
/*  MODULE NAME = RECORD.C					     */
/*								     */
/*  DESCRIPTIVE NAME = Audio Application Programming Interface	     */
/*		       Sample Recording Program 		     */
/*								     */
/*  COPYRIGHT = (C) Copyright by IBM Corp. 1989, 1991.		     */
/*		All rights reserved.				     */
/*		Refer to "LICENSE.DOC" for information regarding     */
/*		the use of this file.				     */
/*								     */
/*  FUNCTION = Provides the following functions/tools:		     */
/*								     */
/*	       record - Provides a working example of recording      */
/*			using the Audio API.			     */
/*								     */
/*  CONVENTIONS = See Below					     */
/*								     */
/*    Constants - upper case					     */
/*    Internal variables - lower case				     */
/*    External variables - as required				     */
/*    Internal functions - initial caps 			     */
/*    External functions - as required				     */
/*								     */
/*  CHANGE ACTIVITY =  See Below				     */
/*								     */
/*		       05/25/91, M-ACPA Release 2.0		     */
/*								     */
/*********************************************************************/
/*								     */
/*************************SYSTEM INCLUDES*****************************/
/*								     */
#define LINT_ARGS		       /* Perform type checking on   */
				       /*  library functions	     */
/*								     */
#include <sys\types.h>		       /*			     */
#include <sys\stat.h>		       /*			     */
#include <conio.h>		       /*			     */
#include <io.h> 		       /*			     */
#include <stdio.h>		       /*			     */
#include <string.h>		       /*			     */
#include <time.h>		       /*			     */
#include <malloc.h>		       /*			     */
/*								     */
/**************************AAPI INCLUDES******************************/
/*								     */
#include "record.h"                    /* Messages                   */
#include "eapidbas.cnc"                /* Audio file structures      */
#include "eapiacb.cnc"                 /* Audio constants/declares   */
/*								     */
/*********************CONSTANT DEFINITIONS****************************/
/*								     */
#define TRUE			1      /* True constant 	     */
#define FALSE			0      /* False constant	     */
#define SUCCESS 		0      /* Function call successful   */
#define FADE_TIME		7000L  /* Milliseconds for pan,fade  */
#define FADE_PERC		5      /* % to fade,pan at a time    */
				       /* Key scan codes	     */
#define F1			59     /* Start/Pause/Resume - Trk 1 */
#define NR			77     /* Balance 5% right	     */
#define NL			75     /*	     left	     */
#define NH			71     /* Pan right over 5 seconds   */
#define NE			79     /*     left		     */
#define NU			72     /* Volume up 5%		     */
#define ND			80     /*	 down		     */
#define NP			73     /* Fade up over N seconds     */
#define NW			81     /*      down		     */
/*								     */
/*								     */
#define AVC_EXT  "._AU"                /* AVC Audio file extension   */
#define RIFF_EXT ".WAV"                /* RIFF Audio file extension  */
/*								     */
/*****************VARIABLE DECLARATION/DEFINITIONS********************/
/*								     */
unsigned char master_volume = 100;     /* Default control	     */
unsigned short	volume = 100;	       /*  settings		     */
unsigned long volume_rate = 0L;        /*			     */
unsigned char balance = 50;	       /*			     */
unsigned long balance_rate = 0L;       /*			     */
unsigned char channel = AAPI_CHNS;     /*  Any available	     */
unsigned char input_source = AAPI_INLB;/*  Line in both 	     */
short speaker = 0;		       /*  Off			     */
unsigned short file_type = AAFB_WAVE;  /*  RIFF WAVE		     */
unsigned short audio_type = AAPI_WAVE; /*  Audio type = PCM	     */
unsigned short monitor = AAPI_MNPC;    /*  Uncompressed 	     */
AAPI_FMT fmt = {1,11025,8,1,2,33,0};   /*  Linear PCM,11K,8 bit,mono */
/*								     */
unsigned long cur_pos = 0L;	       /* Current position (seconds) */
char huge *lalloc = NULL;	       /* Ptr to data buffer >64K    */
unsigned short trk_paused = 1;	       /* Track is paused	     */
short  close_file = 0;		       /* File needs to be closed    */
short  term_audio = 0;		       /* AAPIs need to be terminated*/
unsigned long buf_time; 	       /* Size of data buffers in    */
				       /*  milliseconds 	     */
char name[81];			       /* File name		     */
AAPI_ACB   acb; 		       /* Audio Control Block	     */
FAB_STRUCT *fabp;		       /* Ptr to File Access Block   */
/*								     */
/******************RESIDENT FUNCTION DECLARATIONS*********************/
/*								     */
static short Get_File_Name(void);      /* Get file name and type     */
static short Init_Record(void);        /* Setup recording operation  */
static void Term_Record(short*);       /* Terminate recording	     */
static void Set_Options(void);	       /* Set recording options      */
static short Init_ACB(AAPI_ACB *,      /* Initialize Audio Cntrl Blk */
  FAB_STRUCT *, AAPI_FMT *);	       /*			     */
static short Check_IO(AAPI_ACB *);     /* Check if I/O needed	     */
static short Chk_User(short*);	       /* Check user input	     */
static short Get_Vol(short);	       /* Process volume control     */
static short Get_Bal(short);	       /*	  balance control    */
static short Pause_Resume(void);       /*	  start/pause/resume */
static void Write_Error_Message(short);/* Put error message to screen*/
static short Query_User(char *,        /* Ask user for input	     */
  unsigned long, unsigned long,        /*			     */
  unsigned long *);		       /*			     */
/*								     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = record					     */
/*								     */
/* FUNCTION = Initializes the audio card and begins recording	     */
/*	      and saving digital audio data into a file.	     */
/*								     */
/* INPUT =    None - The user is prompted for the name and type      */
/*	      of the file (AVC or RIFF WAVE) and other		     */
/*	      recording parameters.				     */
/*								     */
/*	      User hits "F1" to start/pause/resume the recording.    */
/*	       Monitoring (if enabled) is turned on immediately.     */
/*	      The following keys are valid when recording RIFF	     */
/*	       WAVE PCM files.					     */
/*	       The cursor keys control volume and balance (+-5%).    */
/*	       Home/End will pan Left/Right over 7 seconds.	     */
/*	       PageUp/PageDown will fade up/down over 7 seconds.     */
/*	      User hits any other key to stop operation.	     */
/*								     */
/* OUTPUT =   If successful					     */
/*		AVC						     */
/*		    Audio file with users name + ._AU extension      */
/*		      Contains information about the audio data      */
/*		    Escape file with users name + ._AD extension     */
/*		      Contains the actual digitized audio data	     */
/*								     */
/*		RIFF WAVE -					     */
/*		    Audio file with users name + .WAV extension      */
/*								     */
/*	      Else error message written to screen		     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
main()				       /* Begin Function	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short rc;			       /* Return code		     */
short  w;			       /* Wait loop index	     */
short keep_going;		       /*			     */
short update_screen;		       /*			     */
/*********************************************************************/
  cprintf(M_47); cprintf(M_48);        /* Program title 	     */
  while(rc = Get_File_Name());	       /* Get name, type of file     */
  rc = Init_Record();		       /* Initialize for recording   */
  if (rc == SUCCESS)		       /* If no error setting up     */
  {				       /*  audio file		     */
    if (file_type == AAFB_WAVE)        /* If RIFF WAVE PCM, allow    */
    {				       /*  volume/balance controls   */
      cprintf(M_38); cprintf(M_39);    /* Tell user how to pause     */
      cprintf(M_40); cprintf(M_41);    /*  and control volume and    */
      cprintf(M_42); cprintf(M_2);     /*  balance and how to stop   */
      cprintf(M_14); cprintf(M_43,     /*  and current position,     */
       cur_pos,volume,balance);        /*   volume, and balance      */
    }				       /*			     */
    else			       /* Else AVC ADPCM, volume and */
    {				       /*  balance controls not valid*/
      cprintf(M_38); cprintf(M_2);     /* Tell user how to pause     */
      cprintf(M_45); cprintf(M_46,     /*  and current position      */
			  cur_pos);    /*			     */
    }				       /*			     */
    rc = aud_strt(&acb);	       /* Start (but only monitor,   */
  }				       /*  no data being saved yet)  */
  keep_going = TRUE;		       /*			     */
  update_screen = FALSE;	       /*			     */
  while ((keep_going) && (rc==SUCCESS) /* While user hasn't quit     */
      && (acb.state==AAPI_RECORDING))  /*  and no error and still    */
  {				       /*  recording		     */
    if (kbhit())		       /* If user hit a key	     */
    {				       /*			     */
      rc = Chk_User(&keep_going);      /* Process it (volume,	     */
      update_screen = TRUE;	       /*  balance, stopping)	     */
    }				       /*			     */
    if((rc==SUCCESS) && (keep_going))  /* If continuing 	     */
    {				       /*			     */
      if(((short)(acb.position/1000L)  /* If playing next second     */
	> cur_pos) || (update_screen)) /*  yet or control values     */
      { 			       /*  may have changed	     */
	cur_pos = (short)	       /*			     */
	      (acb.position/1000L);    /*			     */
	update_screen = FALSE;	       /*			     */
	if (file_type == AAFB_WAVE)    /* If RIFF WAVE PCM, allow    */
	{			       /*  volume/balance controls   */
	  cprintf(M_14);	       /* Tell user time	     */
	  cprintf(M_43, 	       /*  and current position,     */
	  cur_pos,volume,balance);     /*   volume, and balance      */
	}			       /*			     */
	else			       /* Else ADPCM file, no	     */
	{			       /*  volume/balance controls   */
	  cprintf(M_45);	       /*			     */
	  cprintf(M_46,cur_pos);       /* Tell user time	     */
	}			       /*			     */
      } 			       /*			     */
      rc = Check_IO(&acb);	       /* Check if I/O needed	     */
    }				       /*			     */
  }				       /* Enddo playing loop	     */
  Term_Record(&rc);		       /* Terminate operation	     */
  if (rc != SUCCESS)		       /* If error occurred	     */
  {				       /*			     */
    Write_Error_Message(rc);	       /* Tell user		     */
  }				       /*			     */
}				       /* End Function		     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Init_Record					     */
/*								     */
/* FUNCTION = Query user for recording parameters.		     */
/*	      Open the file and prepare to begin recording audio.    */
/*								     */
/* INPUT =							     */
/*								     */
/* OUTPUT =							     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Init_Record()	       /* Begin Function	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short rc;			       /* Return code		     */
/*********************************************************************/
  Set_Options();		       /* Get other user input	     */
  switch (file_type)		       /* Case of file type	     */
  {				       /*			     */
    case AAFB_AVCF  :		       /* AVC			     */
     rc = fab_open(name,	       /* Call AAPI file function    */
       AAFB_CREA,AAFB_EXMX,AAFB_PVOL,  /*  to initialize file and    */
	   &fabp,0,0,0,0);	       /*  associated structures     */
    break;			       /*			     */
    case AAFB_WAVE  :		       /* RIFF WAVE		     */
    rc = fab_ropn(name, 	       /*			     */
	   AAFB_CREA,&fabp,&fmt);      /*			     */
    break;			       /*			     */
    default :			       /* Else case		     */
    break;			       /*			     */
  }				       /*			     */
  if (rc == SUCCESS)		       /* If no error setting up     */
  {				       /*  audio file		     */
    close_file = TRUE;		       /* Remember to close file     */
    rc = Init_ACB(&acb,fabp,&fmt);     /* Initialize Audio CB	     */
    if (rc == SUCCESS)		       /* If no error on set up      */
    {				       /*			     */
      buf_time=acb.timeleft>=100L      /* Save max time before	     */
	? acb.timeleft : 100L;	       /*  I/O is required	     */
      term_audio = TRUE;	       /* Audio is initialized	     */
    }				       /*			     */
  }				       /*			     */
  return(rc);			       /* Return result to caller    */
}				       /* End Function		     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Term_Record					     */
/*								     */
/* FUNCTION = Finish processing for a track			     */
/*								     */
/* INPUT =							     */
/*								     */
/* OUTPUT =							     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static void Term_Record(rc)	       /* Begin Function	     */
short  *rc;			       /* Return code		     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
long stime;			       /* Wait time start	     */
short  w;			       /* Index 		     */
/*********************************************************************/
  if (acb.state != AAPI_STOPPED)       /* If not already stopped     */
  {				       /*			     */
    acb.controls = AAPI_STOP;	       /* Set stop position so	     */
    acb.stoppos  = 0L;		       /*  an immediate stop	     */
    acb.ctlparms = AAPI_STPI;	       /*  will be executed	     */
    aud_ctrl(&acb);		       /* Execute the stop	     */
    stime = clock();		       /* Wait for stop to	     */
    while(			       /*  complete before	     */
    (acb.state!=AAPI_STOPPED) &&       /*  continuing, do not	     */
    ((clock() - stime) < 60000))       /*  badger OS while	     */
    for(w=0;w<=1000;w++);	       /*  waiting		     */
  }				       /*			     */
  if ((*rc == SUCCESS) &&	       /* If an error occurred	     */
       (acb.backrc != SUCCESS))        /*  in background while	     */
  {				       /*  recording		     */
     *rc = acb.backrc;		       /* Tell caller		     */
  }				       /*			     */
  if ((*rc==SUCCESS) || 	       /* If recorded		     */
      (*rc==AAPI_RC_28))	       /*  successfully 	     */
  {				       /*			     */
    *rc = fab_save(fabp,	       /* Save the changes to	     */
	   acb.oupdates);	       /*  the file		     */
  }				       /*			     */
  if (term_audio)		       /* If no error occurred	     */
  {				       /*			     */
    aud_term(&acb);		       /* Terminate AAPI	     */
  }				       /*			     */
  if (close_file)		       /* If file was opened	     */
  {				       /*			     */
    fab_close(fabp);		       /* Close audio file	     */
  }				       /*			     */
  if (lalloc != NULL)		       /* If allocated buffer	     */
  {				       /*			     */
    hfree(lalloc);		       /* Free it		     */
  }				       /*			     */
}				       /* End Function		     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Init_ACB					     */
/*								     */
/* FUNCTION = Initialize the Audio Control Block (ACB) for a	     */
/*	      general record operation. 			     */
/*								     */
/* INPUT =    Ptr to Audio Control Block			     */
/*	      Ptr to File Access Block				     */
/*	      Ptr to PCM Control Block				     */
/*								     */
/* OUTPUT =   0/ Success - ACB initialized for use		     */
/*	      N/ aud_cfig or aud_set return codes		     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Init_ACB(acbp,fabp,fmtp)  /* Begin Function	     */
AAPI_ACB  *acbp;		       /* Ptr to Audio Control Block */
FAB_STRUCT *fabp;		       /* Ptr to file information    */
AAPI_FMT *fmtp; 		       /* Ptr to PCM information     */
{				       /* 010			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short	     rc;		       /* Return code		     */
short	     i; 		       /* Loop variable 	     */
AAPI_DEV   adcb;		       /* Audio Device Control Block */
/*********************************************************************/
  acbp->acb2ptr = 0L;		       /* No secondary ACB	     */
  aud_init(acbp);		       /* Initialize control blk     */
  acbp->channel  = channel;	       /* Set input channel	     */
  acbp->seektype = AAPI_INIT;	       /* First call this file	     */
  acbp->controls = AAPI_MVOL+	       /* Set for vol,bal changes    */
    AAPI_TVO1+AAPI_TVO2+AAPI_CBAL+     /*  and			     */
    AAPI_PAUS;			       /*  to pause (monitor only)   */
  acbp->masvol	 = master_volume;      /* Master volume 	     */
  acbp->trkvol1  = 0;		       /* Track volume - Set to zero */
  acbp->trkvol1s = 0L;		       /*  initially		     */
  acbp->trkvol1e = 0L;		       /*			     */
  acbp->trkvol2  = volume;	       /* Now set user's request     */
  acbp->trkvol2s = 0L;		       /*  with fade in over time    */
  acbp->trkvol2e = volume_rate;        /*			     */
  acbp->chnbal	 = balance;	       /* Balance		     */
  acbp->chnbals  = 0;		       /*	fade		     */
  acbp->chnbale  = balance_rate;       /*			     */
  acbp->outchan  = AAPI_OTBA;	       /*			     */
  acbp->audstart = 0L;		       /* Start at beginning	     */
  acbp->audend	 = 0L;		       /* No stop position	     */
  acbp->fileptr = fabp; 	       /* Pass file info to AAPI     */
  acbp->fmtptr= (audio_type==AAPI_WAVE)/* Pass additional PCM	     */
	       ? fmtp : NULL;	       /*  info if doing PCM	     */
  acbp->fmtptr = fmtp;		       /* PCM parameters	     */
  acbp->audtype = audio_type;	       /* Type of audio 	     */
  acbp->inpsrce = input_source;        /* Input source		     */
  if (monitor)			       /* If monitor during	     */
  {				       /*  record requested	     */
    acbp->dspmode = AAPI_MON;	       /* Set post or pre	     */
    acbp->oprparms |= monitor;	       /*  compression		     */
  }				       /*			     */
  else				       /* Else no monitor	     */
  {				       /*			     */
    acbp->dspmode  = AAPI_RECD;        /* Set for recording only     */
  }				       /*			     */
  if (speaker)			       /* If microchannel speaker    */
  {				       /*  requested		     */
    acbp->oprparms |= AAPI_SPKO;       /* Turn it on		     */
  }				       /*			     */
  acbp->memid	 = AAPI_MAIN;	       /*			     */
  if ((audio_type == AAPI_WAVE) &&     /* If doing high data	     */
      ((fmtp->samples_per_second *     /*  rate PCM, then	     */
	(unsigned long)fmtp->channels* /*  allocate large buffers    */
	(unsigned long) 	       /*  else use 64K 	     */
	(fmtp->bits_per_sample/8)) >   /*			     */
	 AAPI_FMT_44KH_W))	       /*			     */
  {				       /*			     */
				       /* Buffrs can be allocated    */
				       /*  from 0-64K and then	     */
				       /*  in 64K increments	     */
				       /*			     */
    #ifdef O_2			       /* If in OS/2		     */
    acbp->bufflen  = 0X300000L;        /* Allocate 3 meg buffer      */
    #else			       /* Else in DOS		     */
    acbp->bufflen  = 0X40000L;	       /* Allocate 256 K buffer      */
    #endif			       /*			     */
  }				       /*			     */
  else				       /* Else low data rate file    */
  {				       /*			     */
    acbp->bufflen  = 0X10000L;	       /* Allocate 64 K buffer	     */
  }				       /*			     */
  for(i=0;(i<=2)&&(!lalloc);i++)       /* Make three attempts to     */
  {				       /*  allocate the buffer	     */
    lalloc = (char huge *)	       /*			     */
    halloc(acbp->bufflen,sizeof(char));/*			     */
    if	(lalloc == NULL)	       /* If not enough memory	     */
    {				       /*			     */
      acbp->bufflen /=2L;	       /* Try asking for half	     */
    }				       /*			     */
  }				       /*			     */
  acbp->buffptr = (unsigned char *)    /*			     */
		  lalloc;	       /*			     */
  if (lalloc != NULL)		       /* If allocated buffer	     */
  {				       /*			     */
    rc = aud_cfig(&adcb);	       /* Get hardware config	     */
  }				       /*			     */
  else				       /* Else tell caller	     */
  {				       /*			     */
    rc = AAPI_RC_13;		       /* Not enough memory	     */
  }				       /*			     */
  if (rc == SUCCESS)		       /* If microchannel and	     */
  {				       /*  card is responding	     */
    acbp->intlevel = (unsigned char)   /* Set interrupt level	     */
		     adcb.intlev;      /*			     */
    acbp->piobase  = adcb.iobase;      /*  and PIO base address      */
    rc = aud_set(&acb); 	       /* Set up record operation    */
    acbp->controls = 0; 	       /* Reset flags		     */
    acbp->oprparms = 0; 	       /*			     */
  }				       /*			     */
  return(rc);			       /*			     */
}				       /* End Function		     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Get_File_Name				     */
/*								     */
/* FUNCTION = Query user for output file name and type of file.      */
/*								     */
/* INPUT =							     */
/*								     */
/* OUTPUT =   rc = 0,						     */
/*		name and type returned in global variables	     */
/*		Default extension (based on type appended to name)   */
/*	      rc = 3301 					     */
/*		file already exists				     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Get_File_Name()	       /* Begin Function	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
char input_buffer[80];		       /*			     */
unsigned long value;		       /*			     */
char *end;			       /*			     */
short rc;			       /*			     */
/*********************************************************************/
  rc = 0;			       /*			     */
  cprintf(M_1); 		       /* Query and		     */
  gets(input_buffer);		       /*  get the file name	     */
  while 			       /* While name not entered     */
    (sscanf(input_buffer,"%s",name)<1) /*                            */
  {				       /*			     */
    cprintf(M_1);		       /* Keep asking		     */
    gets(input_buffer); 	       /*			     */
  }				       /*			     */
  if(Query_User(M_19,0L,1L,&value))    /* Query and		     */
  {				       /*  get the type 	     */
     file_type = (value == 1L)	       /*			     */
	   ? AAFB_AVCF : AAFB_WAVE;    /*			     */
  }				       /*			     */
  if(!(((end=strrchr(name,'.'))        /* If name has no             */
	  !=0) &&		       /*  extension		     */
	(strrchr(end,'\\')==0)))       /*                            */
  {				       /*			     */
    if(file_type == AAFB_WAVE)	       /* If Wave file type	     */
    {				       /*			     */
      strcat(name,RIFF_EXT);	       /* Add WAV extension	     */
    }				       /*			     */
    else			       /* Else			     */
    {				       /*			     */
      strcat(name,AVC_EXT);	       /* Add _AU extension	     */
    }				       /*			     */
  }				       /*			     */
  if(access(name,0) == SUCCESS)        /* If file exists	     */
  {				       /*			     */
    rc = AAFB_RC_01;		       /* Tell user		     */
  }				       /*			     */
  if(rc != SUCCESS)		       /* If error occurred	     */
  {				       /*			     */
    Write_Error_Message(rc);	       /* Tell user		     */
  }				       /*			     */
  return(rc);			       /*			     */
}				       /* End Function		     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Check I/O					     */
/*								     */
/* FUNCTION = If over half the I/O buffer is full then queue a	     */
/*	      command to do I/O to empty the audio data buffer.      */
/*								     */
/* INPUT =  Ptr to ACB						     */
/*								     */
/* OUTPUT =   0/ Success					     */
/*	   3211/ Error on file seek				     */
/*	   3220/ Error on file write				     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Check_IO(acbp)	       /* Begin Function	     */
AAPI_ACB  *acbp;		       /* Audio Control Block	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short rc;			       /* Return code		     */
/*********************************************************************/
  rc =	SUCCESS;		       /*			     */
  if (acbp->timeleft <= (buf_time/2L)) /* If data buffers is at      */
  {				       /*  least half full	     */
    acbp->controls = AAPI_PFIO;        /* Set to do I/O only	     */
    acbp->iotime = 0;		       /* Set maximum amount	     */
    rc = aud_ctrl(acbp);	       /* Execute the controls	     */
    acbp->controls = 0; 	       /* Reset flag		     */
    if ((rc == AAPI_RC_17) ||	       /* If I/O not ready, or	     */
	(rc == AAPI_RC_20))	       /*   ctrl ignored, Then	     */
    {				       /*			     */
      rc = SUCCESS;		       /* Non-fatal error	     */
    }				       /*			     */
  }				       /*			     */
  return(rc);			       /* Return to caller	     */
}				       /* End Function		     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Check User					     */
/*								     */
/* FUNCTION = Check user input and process the request. 	     */
/*								     */
/* INPUT =  Ptr to ACBs, termination flag			     */
/*								     */
/* OUTPUT =   0/ Success					     */
/*	      Errors from aud_start or aud_ctrl calls		     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Chk_User(keep_going)      /* Begin Function	     */
short *keep_going;		       /* Main loop flag	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short ch;			       /* User input		     */
short rc;			       /* Return code		     */
/*********************************************************************/
  rc =	SUCCESS;		       /*			     */
  ch = getch(); 		       /* Get user input	     */
  if ((ch==0) || (ch==224))	       /* If first byte of extended  */
  {				       /*  code 		     */
    ch = getch();		       /* Get the next byte	     */
  }				       /*			     */
  switch (ch)			       /* Case of user input	     */
  {				       /*			     */
    case F1 :			       /* Start/Pause/Resume	     */
      rc = Pause_Resume();	       /*			     */
    break;			       /*			     */
    case NR:			       /* Balance request	     */
    case NL:			       /*			     */
    case NH:			       /*			     */
    case NE:			       /*			     */
      rc = Get_Bal(ch); 	       /*			     */
    break;			       /*			     */
    case NU:			       /* Volume request	     */
    case ND:			       /*			     */
    case NP:			       /*			     */
    case NW:			       /*			     */
	rc = Get_Vol(ch);	       /* Process request else ignore*/
    break;			       /*			     */
    default :			       /* Else case		     */
      *keep_going = FALSE;	       /* Shut it down		     */
    break;			       /*			     */
  }				       /* End Switch		     */
  return(rc);			       /* Return to caller	     */
}				       /* End Function		     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Pause_Resume 				     */
/*								     */
/* FUNCTION = Pause/resume recording.				     */
/*								     */
/* INPUT =  User request = F1 = 1st track			     */
/*			   F2 = 2nd track			     */
/*			   F3 = Both tracks			     */
/*								     */
/* OUTPUT =   0/ Success					     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Pause_Resume()	       /* Begin Function	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short rc;			       /* Return code		     */
/*********************************************************************/
  if (trk_paused)		       /* If already paused	     */
  {				       /*			     */
   acb.controls = AAPI_RESM;	       /* Toggle to resume	     */
   trk_paused = 0;		       /*			     */
  }				       /*			     */
  else				       /* Else already running	     */
  {				       /*			     */
    acb.controls=AAPI_PAUS;	       /* Toggle to pause	     */
    trk_paused = 1;		       /*			     */
  }				       /*			     */
  rc = aud_ctrl(&acb);		       /* Execute the control	     */
  acb.controls=0;		       /* Reset flag		     */
  if  (rc == AAPI_RC_20)	       /* If control queue full,     */
  {				       /*			     */
    rc = SUCCESS;		       /* Non-fatal error	     */
  }				       /*			     */
  return(rc);			       /* Return to caller	     */
}				       /* End Function		     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Get_Bal					     */
/*								     */
/* FUNCTION = Process balance control				     */
/*								     */
/* INPUT =  User request = ->  = 1st track - Balance N% to the right */
/*			   <-  =			       left  */
/*			  End  = 1st track-100% to right over M secs */
/*			 Home  =		   left 	     */
/*								     */
/* OUTPUT =   0/ Success					     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Get_Bal(ic)	       /* Begin Function	     */
short ic;			       /* User input character	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short rc;			       /* Return code		     */
unsigned short send_control;	       /* Control necessary	     */
/*********************************************************************/
  rc =	SUCCESS;		       /*			     */
  send_control = TRUE;		       /*			     */
  if (ic==NR)			       /* If moving to right N%      */
  {				       /*			     */
    if (balance < 100)		       /* If balance not all the way */
    {				       /*  already		     */
      balance =(balance <=	       /* Set balance +N% to right   */
		 100-FADE_PERC)        /*			     */
	  ? balance+FADE_PERC : 100;   /*			     */
    }				       /*			     */
    else			       /* Else balance all the	     */
    {				       /*  way to the right	     */
      send_control = FALSE;	       /*  Don't send control        */
    }				       /*			     */
  }				       /*			     */
  else				       /* Else moving to left	     */
  {				       /*			     */
    if (ic==NL) 		       /* If moving to left N%	     */
    {				       /*			     */
      if (balance > 0)		       /* If not all the to the left */
      { 			       /*  already		     */
	balance = (balance >=	       /*			     */
			   FADE_PERC)  /*			     */
	    ? balance-FADE_PERC : 0;   /*			     */
      } 			       /*			     */
      else			       /* Else balance all the	     */
      { 			       /*  way to the right	     */
	send_control = FALSE;	       /*  Don't send control        */
      } 			       /*			     */
    }				       /*			     */
  }				       /*			     */
  if (send_control)		       /* If control still necessary */
  {				       /*			     */
    if((ic==NH) || (ic==NE))	       /* If panning over time	     */
    {				       /*			     */
      acb.chnbale = FADE_TIME;	       /* Set time for pan	     */
      if (ic==NE)		       /* If going right	     */
      { 			       /*			     */
	balance = 100;		       /* Full right		     */
      } 			       /*			     */
      else			       /* Else			     */
      { 			       /*			     */
	balance = 0;		       /* Full Left		     */
      } 			       /*			     */
    }				       /*			     */
    else			       /* Else not panning over time */
    {				       /*			     */
      acb.chnbale  = 0L;	       /* Execute immediately	     */
    }				       /*			     */
    acb.controls = AAPI_CBAL;	       /* Set up for balance control */
    acb.chnbal = balance;	       /*			     */
    acb.chnbals = 0L;		       /*			     */
    acb.outchan = AAPI_OTBA;	       /*			     */
    rc = aud_ctrl(&acb);	       /* Execute the control	     */
    acb.controls=0;		       /* Reset control 	     */
    if	(rc == AAPI_RC_20)	       /* If control queue full,     */
    {				       /*			     */
      rc = SUCCESS;		       /* Non-fatal error	     */
    }				       /*			     */
  }				       /*			     */
  return(rc);			       /* Return to caller	     */
}				       /* End Function		     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Get_Vol					     */
/*								     */
/* FUNCTION = Process volume control				     */
/*								     */
/* INPUT =  User request = A   = 1st track - Volume N% up	     */
/*			   V   =		       down	     */
/*		     Page Up   = 1st track-Volume to 100% over M secs*/
/*		     Page Down =		       0%	     */
/*								     */
/* OUTPUT =   0/ Success					     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Get_Vol(ch)	       /* Begin Function	     */
short ch;			       /* User input character	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
short rc;			       /* Return code		     */
unsigned short send_control;	       /* Control necessary	     */
/*********************************************************************/
  rc =	SUCCESS;		       /*			     */
  send_control = TRUE;		       /* Don't send control         */
  if (ch==NU)			       /* If moving up 5%	     */
  {				       /*			     */
    if (volume < 100)		       /* If not at 100% already     */
    {				       /*			     */
      volume=(volume		       /* Set volume +N% higher      */
		<= 100-FADE_PERC)      /*			     */
	  ? volume+FADE_PERC : 100;    /*			     */
    }				       /*			     */
    else			       /* Else volume at 100%	     */
    {				       /*			     */
      send_control = FALSE;	       /*  Don't send control        */
    }				       /*			     */
  }				       /*			     */
  else				       /* Else may be moving down    */
  {				       /*			     */
    if (ch==ND) 		       /* If moving down N%	     */
    {				       /*			     */
      if (volume > 0)		       /* If volume not at 0%	     */
      { 			       /*  already		     */
	 volume=(volume 	       /* Set Volume -N% down	     */
		>= FADE_PERC)	       /*			     */
	   ? volume-FADE_PERC:0;       /*			     */
      } 			       /*			     */
      else			       /* Else volume at 0%	     */
      { 			       /*			     */
	send_control = FALSE;	       /*  Don't send control        */
      } 			       /*			     */
    }				       /*			     */
  }				       /*			     */
  if (send_control)		       /* *f control still necessary */
  {				       /*			     */
    if((ch==NP) || (ch==NW))	       /* If ramping over time	     */
    {				       /*			     */
      acb.trkvol1e = FADE_TIME;        /* Set time for fade	     */
      if (ch==NP)		       /* If moving up		     */
      { 			       /*			     */
	volume=100;		       /* Set max		     */
      } 			       /*			     */
      else			       /* Else			     */
      { 			       /*			     */
	volume=0;		       /* Set min		     */
      } 			       /*			     */
    }				       /*			     */
    else			       /* Else			     */
    {				       /*			     */
      acb.trkvol1e = 0L;	       /* Do immediately	     */
    }				       /*			     */
    acb.controls = AAPI_TVO1;	       /* Set up for volume control  */
    acb.trkvol1 = volume;	       /*			     */
    acb.trkvol1s = 0L;		       /* Start at time 0 in file    */
    rc = aud_ctrl(&acb);	       /* Execute the control	     */
    acb.controls=0;		       /* Reset control 	     */
    if	(rc == AAPI_RC_20)	       /* If control queue full,     */
    {				       /*			     */
      rc = SUCCESS;		       /* Non-fatal error	     */
    }				       /*			     */
  }				       /*			     */
  return(rc);			       /* Return to caller	     */
}				       /* End Function		     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Write_Error_Message				     */
/*								     */
/* FUNCTION = Determine and write error message to user 	     */
/*								     */
/* INPUT =  AAPI return code					     */
/*								     */
/* OUTPUT = Message written to standard out (user)		     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static void Write_Error_Message(rc)    /* Begin Function	     */
short rc;			       /* Return code		     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
char *msg;			       /* Ptr to error message	     */
char default_error;		       /* Default error case T/F     */
/*********************************************************************/
    default_error = FALSE;	       /*			     */
    switch (rc) 		       /* Case of error code	     */
    {				       /*			     */
      case AAPI_RC_08 : 	       /*			     */
      case AAPI_RC_13 : 	       /* Insufficient storage	     */
      case AAFB_RC_04 : 	       /*  to complete operation     */
      case AAFB_RC_05 : 	       /*			     */
      case AAPI_RC_37 : 	       /*			     */
	msg = M_3;		       /*			     */
      break;			       /*			     */
      case AAFB_RC_01 : 	       /* File already exists	     */
	msg = M_4;		       /*  and a create requested    */
      break;			       /*			     */
      case AAFB_RC_02 : 	       /* File doesn't exist         */
	msg = M_5;		       /*  and an open requested     */
      break;			       /*			     */
      case AAFB_RC_03 : 	       /* I/O error on audio file    */
      case AAFB_RC_06 : 	       /*			     */
      case AAFB_RC_08 : 	       /*			     */
      case AAPI_RC_01 : 	       /*			     */
      case AAPI_RC_02 : 	       /*			     */
      case AAPI_RC_03 : 	       /*			     */
      case AAPI_RC_04 : 	       /*			     */
      case AAPI_RC_05 : 	       /*			     */
      case AAPI_RC_06 : 	       /*			     */
	msg = M_6;		       /*			     */
      break;			       /*			     */
      case AAFB_RC_07 : 	       /* I/O error on escape file   */
      case AAPI_RC_07 : 	       /*			     */
      case AAPI_RC_11 : 	       /*			     */
      case AAPI_RC_19 : 	       /*			     */
	if(file_type == AAFB_WAVE)     /* If Wave file type	     */
	{			       /*			     */
	  msg = M_6;		       /*			     */
	}			       /*			     */
	else			       /*			     */
	{			       /*			     */
	  msg = M_7;		       /*			     */
	}			       /*			     */
      break;			       /*			     */
      case AAPI_RC_12 : 	       /* DSP not responding	     */
	msg = M_8;		       /*			     */
      break;			       /*			     */
      case AAPI_RC_14 : 	       /* DSP code not found	     */
	msg = M_9;		       /*			     */
      break;			       /*			     */
      case AAPI_RC_15 : 	       /* DSP code file invalid      */
	msg = M_10;		       /*			     */
      break;			       /*			     */
      case AAPI_RC_24 : 	       /*			     */
      case AAPI_RC_25 : 	       /* Audio card not	     */
	msg = M_11;		       /*  installed		     */
      break;			       /*			     */
      case AAPI_RC_26 : 	       /* Device driver not	     */
	msg = M_12;		       /*  responding		     */
      break;			       /*			     */
      case AAPI_RC_27 : 	       /* Disk full		     */
	msg = M_15;		       /*			     */
      break;			       /*			     */
      case AAPI_RC_28 : 	       /* Audio object at	     */
	msg = M_16;		       /*  maximum size 	     */
      break;			       /*			     */
      case AAPI_RC_30 : 	       /* Media too slow or	     */
	msg = M_17;		       /*  fragmented		     */
      break;			       /*			     */
      case AAPI_RC_09 : 	       /* Start or stop position     */
      case AAPI_RC_10 : 	       /* Invalid		     */
	msg = M_18;		       /*			     */
      break;			       /*			     */
      default : 		       /* Else case		     */
	msg = M_13;		       /*			     */
	default_error = TRUE;	       /*			     */
      break;			       /*			     */
    }				       /* End Switch		     */
    if (default_error)		       /* If error that we don't     */
    {				       /*  have a message for,	     */
      cprintf(msg,rc);		       /* Tell user the return code  */
    }				       /*			     */
    else			       /* Else			     */
    {				       /*			     */
      cprintf(msg);		       /* Normal message	     */
    }				       /* Endif 		     */
    return;			       /* Return to caller	     */
}				       /* End Function		     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Set_Options					     */
/*								     */
/* FUNCTION = Query user for volume parameters. 		     */
/*								     */
/* INPUT =    Track being processed				     */
/*								     */
/* OUTPUT =   None.						     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static void Set_Options()	       /* Begin Function	     */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
unsigned long value;		       /*			     */
/*********************************************************************/
  if (file_type == AAFB_AVCF)	       /* If AVC file		     */
  {				       /*			     */
    audio_type= AAPI_SM11;	       /* Reset default to AVC music */
    if (Query_User(M_23,1L,4L,&value)) /* Query user for audio type  */
    {				       /*			     */
      audio_type=(unsigned short)value;/*			     */
    }				       /*			     */
    if (Query_User(M_25,0L,2L,&value)) /* Monitor		     */
    {				       /*			     */
      monitor=((value==0)||(value==1)) /*			     */
	 ? (value ? AAPI_MNPS : 0)     /*			     */
	 : AAPI_MNPC;		       /*			     */
    }				       /*			     */
  }				       /*			     */
  else				       /* Else RIFF Wave file	     */
  {				       /*			     */
    if (Query_User(M_27,1L,3L,&value)) /* Query user for audio type  */
    {				       /*			     */
      fmt.format=(unsigned short)value;/*			     */
    }				       /*			     */
    if (Query_User(M_20,0L,3L,&value)) /* Sample Rate		     */
    {				       /*			     */
      fmt.samples_per_second =	       /*			     */
       (value ==1) ? AAPI_FMT_11KH_W : /*			     */
       (value ==2) ? AAPI_FMT_22KH_W : /*			     */
       (value ==3) ? AAPI_FMT_44KH_W : /*			     */
		     AAPI_FMT_8000;    /*			     */
    }				       /*			     */
    if (fmt.format == AAPI_FMT_LPCM_W) /* If Linear PCM, (mu-law and */
    {				       /*  a-law always 8 bit)	     */
      if(Query_User(M_21,0L,1L,&value))/*			     */
      { 			       /* Sample Width		     */
	fmt.bits_per_sample =	       /*			     */
	 (value ==0L) ? 8 : 16;        /*			     */
      } 			       /*			     */
    }				       /*			     */
    fmt.sample_number_format =	       /* Set format of sample,      */
      (fmt.bits_per_sample == 8)       /*  unsigned = 8 bits	     */
      ? AAPI_FMT_USGN_W 	       /*  2's compliment = 16 bits  */
      : AAPI_FMT_2CMP_W;	       /*			     */
    if((fmt.format==AAPI_FMT_LPCM_W) ||/* If Linear PCM or not	     */
       (fmt.samples_per_second !=      /*  44K then allow mono or    */
       AAPI_FMT_44KH_W))	       /*  stereo else just mono     */
    {				       /*			     */
      if(Query_User(M_22,1L,2L,&value))/* Channels		     */
      { 			       /*			     */
	fmt.channels=(unsigned short)  /*			     */
		      value;	       /*			     */
      } 			       /*			     */
    }				       /*			     */
    if(Query_User(M_28,0L,100L,&value))/* Dither		     */
    {				       /*			     */
      fmt.dither_percent =	       /*			     */
	    (unsigned short)value;     /*			     */
    }				       /*			     */
    if (Query_User(M_26,0L,1L,&value)) /* Monitor		     */
    {				       /*			     */
      monitor = (value==1L)	       /*			     */
	 ? AAPI_MNPC : 0;	       /*			     */
    }				       /*			     */
  }				       /* Endif type dependent values*/
  if (Query_User(M_24,0L,4L,&value))   /* Input Source		     */
  {				       /*			     */
    input_source=(unsigned char)value; /*			     */
  }				       /*			     */
  if ((file_type == AAFB_WAVE) &&      /* If PCM file		     */
      (Query_User(M_34,0L,1L,&value))) /* Query if user wants to set */
  {				       /*  volume/balance controls   */
    if(Query_User(M_29,0L,120L,&value))/*			     */
    {				       /* Master volume 	     */
      master_volume =		       /*			     */
	  (unsigned char) value;       /*			     */
    }				       /*			     */
    if(Query_User(M_30,0L,100L,&value))/*			     */
    {				       /* Track Volume		     */
	volume = (unsigned short)value;/*			     */
    }				       /*			     */
    if(Query_User(M_31,0L,64L,&value)) /*			     */
    {				       /* Track Volume fade time     */
      volume_rate = 1000L * value;     /*			     */
    }				       /*			     */
    if(Query_User(M_32,0L,100L,&value))/*			     */
    {				       /* Balance		     */
      balance=(unsigned char) value;   /*			     */
    }				       /*			     */
    if(Query_User(M_33,0L,64L,&value)) /*			     */
    {				       /* Balance fade time	     */
      balance_rate = 1000L * value;    /*			     */
    }				       /*			     */
  }				       /* Endif type dependent values*/
  if (Query_User(M_35,1L,1L,&value))   /* Query if user wants to set */
  {				       /*  other options	     */
    if(Query_User(M_37,1L,2L,&value))  /*			     */
    {				       /* Track Number		     */
      channel=(unsigned char)	       /*			     */
		   (value-1L);	       /*			     */
    }				       /*			     */
    if(Query_User(M_36,1L,1L,&value))  /* Microchannel speaker	     */
    {				       /*			     */
      speaker = (short) value;	       /*			     */
    }				       /*			     */
  }				       /*			     */
}				       /* End Function		     */
/*********************************************************************/
/*								     */
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
/*								     */
/* FUNCTION NAME = Query_User					     */
/*								     */
/* FUNCTION = Query user for volume parameters. 		     */
/*								     */
/* INPUT =    Message to be processed				     */
/*	      Parameter minimum and maximum values		     */
/*	      Ptr to area to return user response		     */
/*								     */
/* OUTPUT =   rc = 0, User hit Enter				     */
/*	      rc = 1, User input value returned 		     */
/*								     */
/*********************************************************************/
/*********************************************************************/
/*			       CODE				     */
/*********************************************************************/
/*********************************************************************/
static short Query_User(msg,minv,maxv, /* Begin Function	     */
		       value)	       /*			     */
char * msg;			       /* Query message 	     */
unsigned long minv, maxv, *value;      /* Minimum, maximum,  value   */
{				       /*			     */
/********************LOCAL VARIABLE DEFINITIONS***********************/
char input_buffer[80];		       /*			     */
short rc;			       /*			     */
/*********************************************************************/
  rc = -1;			       /*			     */
  while (rc==-1)		       /* While user hasn't responded*/
  {				       /*			     */
    cprintf(msg);		       /* Query the user	     */
    gets(input_buffer); 	       /*			     */
    if (sscanf(input_buffer,"%lu",     /* If user typed value        */
	    value) == 1)	       /*			     */
    {				       /*			     */
      if ((*value >= minv) &&	       /* If inputted value is valid */
	   (*value <= maxv))	       /*			     */
      { 			       /*			     */
	rc = 1; 		       /* Use it		     */
      } 			       /*			     */
      else			       /* Else invalid		     */
      { 			       /*			     */
	cprintf(M_44);		       /* Tell user try again	     */
      } 			       /*			     */
    }				       /*			     */
    else			       /* Else user hit enter to     */
    {				       /*  accept default	     */
      rc = 0;			       /*			     */
    }				       /*			     */
  }				       /*			     */
  return(rc);			       /*			     */
}				       /* End Function		     */
/*********************************************************************/
