/****************************************************************************
 *
 *  Licensed Materials - Property of IBM
 *
 *  IBM TCP/IP for OS/2
 *  Copyright IBM Corp. 1995, 1996. All rights reserved.
 *  US Government Users Restricted Rights - Use, duplication or
 *  disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 ****************************************************************************
 *
 *    IBM-specific DHCP Option Instantiation Script for IBM TCP/IP 3.0 
 *
 *                             DHCIPIBM.CMD     
 *
 *            - - - - - - - - - - - - - - - - - - - - - - - - - 
 *
 * This script may be executed by the DHCP client, DHCPCD.EXE, after new 
 * DHCP options have been returned by the DHCP server.  The script
 * processes IBM-supported DHCP options that are not instantiated within
 * the DHCP client itself.                                                    
 *
 * This script is invoked as follows:
 *
 *      dhcpibm <option tag> <option data>
 *
 *      - option tag is decimal
 *      - option data is a hex string
 *
 * EXIT return values:
 *
 *    successful                   0
 *    input error                  1
 *      e.g., wrong number of arguments
 *            invalid option length
 *            unsupported option
 *            file not found
 *            improper IP address
 *            ETC environment variable not set
 *            data is wrong type                              
 *    system error                -1
 *      e.g., failed to write INI file
 *
 * Written by Janet Brewer
 *
 * 4/25/96:  Last Revision
 *
 ****************************************************************************/


/****************************************************************************
 *                             Main Script
 ****************************************************************************/
   return_code = 0

   /* add RexxUtil functions */
   call RxFuncAdd 'SysIni', 'RexxUtil', 'SysIni'
   call RxFuncAdd 'SysTempFileName', 'RexxUtil', 'SysTempFileName'
   call RxFuncAdd 'SysFileDelete', 'RexxUtil', 'SysFileDelete'
   call RxFuncAdd 'SysFileTree', 'RexxUtil', 'SysFileTree'

   /* get option number and data from the command line */
   parse arg option_tag option_data 

   if ((option_tag = '') | (option_data = '')) then do
      return_code = 1      /* wrong number of arguments */
      exit return_code 
   end

   /* strip leading and trailing spaces */
   option_tag = strip(option_tag)
   option_data = strip(option_data)

   /* for debugging, write the data to a text file */
/*   date = date('N')
     time = time('N')
     outfile = 'c:\tcpip\etc\dhcpibm.out' 
     return_code = write_debug_file(option_tag, option_data, outfile)
     call lineout outfile
*/
   /* instantiate the option */
   Select

      when (option_tag = 9) then do          /* Default LPR Server */
         if (check_IPaddr(option_data)) then 
            return_code =  write_ini('\tcpos2.ini', 'LPR', 'LPR_SERVER',,
                                     option_data) 
         else return_code =  1               /* improper IP address */
      end

      when (option_tag = 71) then do         /* NewsReader/2 Server */
         if (check_IPaddr(option_data)) then 
            return_code =  write_ini('\tcpos2.ini', 'LAN', 'NEWS',,
                                     option_data) 
         else return_code =  1              /* improper IP address */
      end

      when (option_tag = 200) then do        /* Default LPR Printer */
         data = hex2ascii_string(option_data)
         if (data \= '') then
            return_code = write_ini('\tcpos2.ini', 'LPR', 'LPR_PRINTER', data)
         else return_code = 1                /* invalid hex string */
      end

      when (option_tag = 201) then do        /* Default Gopher Server */
         data = hex2ascii_string(option_data)
         if ((data = '') | (check_IPaddr(data)) = 0) then 
            return_code = 1                  /* improper IP address */
         else
            return_code = write_ini('\tcpos2.ini', 'LAN', 'GOPHER', data) 
      end

      when (option_tag = 202) then do        /* Default WWW Home Page */
         data = hex2ascii_string(option_data)
         if (data \= '') then
            return_code =  write_file('\explore', 'ini', '[network]',,
                                      'HomePage=', data)
         else return_code = 1                /* invalid hex string */
      end

      when (option_tag = 203) then do        /* Default WWW Proxy Gateway */
         data = hex2ascii_string(option_data)
         if (data \= '') then
            return_code =  write_file('\explore', 'ini', '[network]',,
                                      'Proxy=', data)
         else return_code = 1                /* invalid hex string */
      end

      when (option_tag = 204) then do        /* Default WWW News Server */
         data = hex2ascii_string(option_data)
         if ((data = '') | (check_IPaddr(data)) = 0) then 
            return_code = 1                  /* improper IP address */
         else
            return_code = write_file('\explore', 'ini', '[network]',,
                                     'News=', data)
      end

      when (option_tag = 205) then do        /* Default SOCKS Server */
         data = hex2ascii_string(option_data)
         if (data \= '') then
            return_code = socks_server(data)
         else return_code = 1                /* invalid hex string */
      end

      when (option_tag = 206) then do        /* NFS Mount Points */
         data = hex2ascii_string(option_data)
         if (data \= '') then
            return_code = nfs_mount_points(data)
         else return_code = 1                /* invalid hex string */
      end

      when (option_tag = 207) then do        /* Default X Font Servers */
         data = hex2ascii_string(option_data)
         if (data \= '') then
            return_code = write_ini('\pmx.ini', 'DHCP',,
                                    'defaultFontServer', data)
         else return_code = 1               /* invalid hex string */
      end

      when (option_tag = 208) then do        /* X Display Manager Servers */
         data = hex2ascii_string(option_data)
         if (data \= '') then 
            return_code = write_ini('\pmx.ini', 'DHCP', 'xdmcpHosts', data)
         else return_code = 1                /* invalid hex string */
      end

      otherwise
         return_code = 1                     /* unsupported_option */
   end

   exit return_code


/**************************************************************************** 
 * socks_server(data)
 *                                                      
 * Instantiate the IP address pair returned for SOCKS.  The first address
 * is the socks server (SOCKS_SERVER) and the second is the SOCKS name
 * server (SOCKS_NS).  They are currently the same address, but in the future 
 * this may change.  If the file etc\SOCKS.ENV does not exist, we assume
 * that SOCKS is not in use and we return immediately with -1.  The data 
 * is written to a text file, ETC\SOCKS.ENV, in this format:
 *
 *    ...
 *    SOCKS_SERVER 9.67.100.1
 *    SOCKS_NS 9.67.100.2
 *    ...
 *
 * Input:  One IP address, or two IP addresses separated by spaces or tabs.
 *
 * Output: Returns 0 when SOCKS.ENV file write succeeds.  Returns 1 when the
 *         IP address is malformed.  Returns -1 if the file write fails
 *         for any reason.
 *
 ****************************************************************************/
socks_server:  Procedure

   return_code = 0
   tab_position = 1

   etc = value('ETC',,OS2ENVIRONMENT)             /* get path to ETC dir */
   file = etc||'\socks.env'                       /* filename */
   backupfile = etc||'\socks.bak'                 /* backup filename */
   tempfile = SysTempFileName(etc||'\temp.???')   /* temp file */

   /* SOCKS is not being used if SOCKS.ENV file doesn't exist, so quit */
   call SysFileTree file, 'files', 'F'
   if (files.0 = 0) then do
      return -1
   end

   /* replace tabs with spaces and get address pair */
   parse arg data
   do until tab_position = 0
     tab_position = pos('09'X, data, tab_position) 
     if (tab_position \= 0) then do
        data = delstr(data, tab_position, 1)
        data = insert(' ', data, (tab_position - 1))
     end
   end
   parse var data socks_server socks_ns

   if (check_IPaddr(socks_server) = 0) then 
      return 1             /* improper IP address */

   if (socks_ns \= '') then do
      if (check_IPaddr(socks_ns) = 0) then 
         return 1        /* improper IP address */
   end

   /* backup file in case something goes wrong */
   "@copy" file backupfile
   if RC \= 0 then do                             /* disk is probably full */
      call SysFileDelete backupfile
      return -1
   end

   /* read the entire socks.env file */
   if (etc \= '') then do
      do i = 1 until (lines(file) = 0)
         text.i = linein(file)
      end
      text.0 = i                                /* number of lines in file */
      call lineout(file)
   end
   else return -1

   /* look for SOCKS_SERVER and SOCKS_NS keynames */
   i = 1
   do while i <= text.0
      text.i = strip(text.i)      /* strip leading and trailing spaces */
      parse upper var text.i key data
      if (key = 'SOCKS_SERVER') then do
         text.i = 'SOCKS_SERVER'||' '||socks_server
         found_server = 1
      end 
      else if (socks_ns \= '') then do
         if (key = 'SOCKS_NS') then do
            text.i = 'SOCKS_NS'||' '||socks_ns
            found_nameserver = 1
         end
      end
      i = i + 1
   end

   /* keyname not found so add keyname and keydata */
   if (found_server \= 1) then do
      text.i = 'SOCKS_SERVER'||' '||socks_server
      i = i + 1
   end
   if (socks_ns \= '') then do
      if (found_nameserver \= 1) then do
         text.i = 'SOCKS_NS'||' '||socks_ns
         i = i + 1
      end 
   end

   /* write the altered file */
   do j = 1 to (i - 1)
      err = lineout(tempfile, text.j)
      if (err \= 0) then leave
   end
   if (err \= 0) then return_code = -1 

   /* if error, restore backup file, otherwise replace with the new file */
   call lineout tempfile
   if (err = 0) then 
      "@copy" tempfile file 
   else 
      "@copy" backupfile file
   if RC \= 0 then 
      return_code = -1 
   else 
      return_code = 0
   call SysFileDelete backupfile
   call SysFileDelete tempfile

   return return_code

/* end socks_server function */


/**************************************************************************** 
 * nfs_server(data)                                                           
 *
 *  Instantiate the nfs mount points.  Mount commands are instantiated until   
 *  either the string is exhaused or an error occurs.
 *
 *  Input:  Data is a hex string of mount commands with no blanks.  Mount
 *          commands are separated by ASCII ';'.
 *
 *  Output: Returns 0 when .ini file write succeeds.  Returns 1 when the
 *          a mount command does not start with 'mount' or 'mvslogin'.
 *          Returns the value returned by write_ini function if the attempt
 *          to write the .ini fails.
 *
 ****************************************************************************/
nfs_mount_points:  Procedure

   done = 0
   n = 1
   return_code = 0

   parse arg data

   /* delete previous entries in .ini file */
   return_code = write_ini('\fstab.ini', 'MOUNT_COMMANDS',,
                          'DELETE:', '') 
   if (return_code \= 0) then
      return return_code

   do until done

      /* extract a single mount command from the string of commands */
      separator = pos(';', data)
      if (separator \= 0) then do
         /* extract a mount command terminated with ';' */
         mount_point = left(data, separator - 1)
         mount_point = strip(mount_point)
         data = delstr(data, 1, separator)
      end 
      else do
        if (data \= '') then do
           /* this is the last mount command */
           mount_point = strip(data)
           data = delstr(data, 1)
           done = 1
        end
        else
           /* no more data left */ 
           leave
      end 

      /* sanity check: require 'mount' or 'mvslogin' to be first word */
      keyword = word(mount_point, 1)
      parse upper var keyword keyword
      /* if sanity check passes, delete the previous entries and write
         new entries to the .ini file */
      if ((keyword = 'MOUNT') | (keyword = 'MVSLOGIN')) then do
         return_code = write_ini('\fstab.ini', 'MOUNT_COMMANDS',,
                                 'MOUNT'n, mount_point) 
         if (return_code \= 0) then
            leave
         n = n + 1
      end
      else do
         return_code = 1
         leave
      end
   end /* do */

   return return_code

/* end nfs_server function */


/**************************************************************************** 
 *  check_IPaddr (addr)
 *
 *  Check a dotted decimal IP address for valid format.
 *
 *  Return 0 if there are not 4 octets or the data is out of range or
 *  there are embedded blanks in the address, else return 1.
 *
 ****************************************************************************/
check_IPaddr:  Procedure

   parse arg addr
   addr = strip(addr)
   parse var addr octet.1 '.' octet.2 '.' octet.3 '.' octet.4 

   do pos = 1 to 4
      if (verify(octet.pos, '0123456789', N) \= 0) then
         return 0     /* not an unsigned number */
      if ((octet.pos < 0) | (octet.pos > 255))  then
         return 0
   end

   return 1

/* end check_IPaddr function */


/****************************************************************************
 *  hex2ascii_string (option_data)
 *
 *  Take a hex string in the form 'hex"nn nn nn ... ", verify it's a valid 
 *  hex string, and convert it to an ASCII character string.
 *
 *  Returns the string if the data is valid, a null string if not.    
 *
 ****************************************************************************/
hex2ascii_string:  Procedure
   parse arg 'hex' '"' data '"'
   if (\ DATATYPE(data, X)) then return ""
   data = strip(data, 'B', '09'x)

   return X2C(data)

/* hex2ascii_string function */


/**************************************************************************** 
 *  write_ini (inifilename, application, keyname, keydata) 
 *
 *  Write the specified key data to the INI file under the specified
 *  application and key name.
 *
 *  The INI file must already exist in the ETC directory as defined by the
 *  ETC environment variable.
 *
 ****************************************************************************/
write_ini:  Procedure

   parse arg inifile, app, keyname, keydata

   etc = value('ETC',,OS2ENVIRONMENT)
   if (etc = '') then 
      return 1
   else do
      if (keyname = 'DELETE:') then 
         Result = SysIni(etc||inifile, app, 'DELETE:') 
      else 
         Result = SysIni(etc||inifile, app, keyname, keydata) 
      if Result = 'ERROR:' then 
         return -1
      else 
         return 0
   end  

/* end write_ini function */


/**************************************************************************** 
 *  write_file (filename, file_extension, appname, keyname, keydata) 
 *
 *  Write a text file with a keydata string immediately following the
 *  keyname string.
 *
 *  This function is used to write a text profile file which has a format
 *  similar to a binary .INI file.
 *
 *  Example:     ...
 *               [network]
 *               HomePage=http://www.ibm.com/
 *               ...
 *
 *     [network] is like a .INI application
 *     HomePage  is like a .INI key name
 *     http://www.ibm.com is like .INI key data 
 *
 ****************************************************************************/
write_file:  Procedure

   found_appname = 0
   found_keyname = 0
   added_line_number = 0
   return_code = 0

   parse arg filename, file_extension, appname, keyname, keydata

   etc = value('ETC',,OS2ENVIRONMENT)             /* get path to ETC dir */
   file = etc||filename'.'file_extension          /* filename */
   backupfile = etc||filename'.bak'               /* backup filename */
   tempfile = SysTempFileName(etc||'\temp.???')   /* temp file */

   /* backup file in case something goes wrong */
   "@copy" file backupfile
   if RC \= 0 then do                           /* disk is probably full */
      call SysFileDelete backupfile
      return -1
   end

   /* read the entire file */
   if (etc \= '') then do
      do i = 1 until (lines(file) = 0)
         text.i = linein(file)
      end
      text.0 = i                                /* number of lines in file */
      call lineout(file)
   end
   else return -1

   /* search for appname */
   i = 1
   do while i <= text.0
      current_line = space(strip(text.i),0)     /* strip all spaces */
      i = i + 1
      if (appname = left(current_line, length(appname))) then do
         found_appname = 1
         leave
      end
   end    

   /* if appname not found, add it and the keydata */
   if (found_appname \= 1) then do
      text.i = appname
      i = i + 1
      text.i = keyname'='keydata
      text.0 = text.0 + 2
   end
  
   /* if appname found, add or replace the keydata */
   else do 

      /* look for keyname */
      do while i <= text.0
         current_line = space(strip(text.i),0)      /* strip all spaces */

         /* keyname not found before next appname */
         parse var current_line first_char +1 .
         if (first_char = '[') then do
            leave
         end

         /* keyname found before next appname or EOF */
         if (keyname = left(current_line, length(keyname))) then do
            found_keyname = 1
            leave
         end 

         i = i + 1
      end

      /* if keyname is found, insert new keydata while preserving any */
      /*  leading blanks */
      if (found_keyname) then do
         parse var text.i key '=' olddata  
         number_of_spaces = verify(olddata,' ',NOMATCH)
         if (number_of_spaces \= 0) then
            spaces = left(olddata, number_of_spaces - 1)
         else
            spaces = ''
         text.i = key||'='||spaces||keydata
      end
 
      /* keyname not found so add keyname and keydata */
      else do
         added_line = keyname'='keydata
         do j = (i - 1) to 1 by -1    /* back up to non-blank line */
            if (text.j \= '') then do
               added_line_number = j
               leave 
            end
         end
         if (added_line_number = 0) then
            added_line_number = i
      end
   end

   /* write the altered file */
   i = 1
   if (added_line_number \= 0) then do
      do i = 1 to added_line_number
         err = lineout(tempfile, text.i)
         if (err \= 0) then leave
      end
      if (err = 0) then err = lineout(tempfile, added_line)
      if (err \=0) then return_code = -1 
   end
   if (return_code \= -1) then 
      do j = i to text.0
         err = lineout(tempfile, text.j)
         if (err \= 0) then leave
      end

   /* if error, restore backup file, otherwise replace with the new file */
   call lineout tempfile
   if (err = 0) then 
      "@copy" tempfile file 
   else 
      "@copy" backupfile file
   if RC \= 0 then 
      return_code = -1 
   else 
      return_code = 0
   call SysFileDelete backupfile
   call SysFileDelete tempfile 

   return return_code

/* end write_file function */


/**************************************************************************** 
 *  write_debug_file (option, option_data, outfile)
 *
 *  Write option number and data to a file.  For debugging use only.
 *
 ****************************************************************************/
write_debug_file:  Procedure

   parse arg option, data, outfile

   /* write the option number and data to a file */
   line = date('n') time('n')   'option: '||option||'   data:  '||data

   err = lineout(outfile, line)

   return err

/* end write_debug_file function */
