/****************************************************************************
 *
 *  Licensed Materials - Property of IBM
 *
 *  IBM TCP/IP for OS/2
 *  Copyright IBM Corp. 1995, 1996. All rights reserved.
 *  US Government Users Restricted Rights - Use, duplication or
 *  disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 ****************************************************************************
 *
 *   DHCP Option Instantiation Script for IBM TCP/IP 5.0 (for LPR DHCP support)
 *
 *                             DHCPLPR.CMD     
 *
 *            - - - - - - - - - - - - - - - - - - - - - - - - - 
 *
 * This script may be executed by the DHCP client, DHCPCD.EXE, after new 
 * DHCP options have been returned by the DHCP server.  The script
 * processes IBM-supported DHCP options that are not instantiated within
 * the DHCP client itself.                                                    
 *
 * This script is invoked as follows:
 *
 *      dhcplpr <option tag> <option data>
 *
 *      - option tag is decimal
 *      - option data is a hex string
 *
 * EXIT return values:
 *
 *    successful                   0
 *    input error                  1
 *      e.g., wrong number of arguments
 *            invalid option length
 *            unsupported option
 *            file not found
 *            improper IP address
 *            ETC environment variable not set
 *            data is wrong type                              
 *    system error                -1
 *      e.g., failed to write INI file
 *
 * 01/08/96 LSW 20152 Created this exec by copying DHCPIBM.CMD and deleting
 *                    all processing related to applications other than LPR.
 *
 ****************************************************************************/


/****************************************************************************
 *                             Main Script
 ****************************************************************************/
   return_code = 0

   /* add RexxUtil functions */
   call RxFuncAdd 'SysIni', 'RexxUtil', 'SysIni'
   call RxFuncAdd 'SysTempFileName', 'RexxUtil', 'SysTempFileName'
   call RxFuncAdd 'SysFileDelete', 'RexxUtil', 'SysFileDelete'
   call RxFuncAdd 'SysFileTree', 'RexxUtil', 'SysFileTree'

   /* get option number and data from the command line */
   parse arg option_tag option_data 

   if ((option_tag = '') | (option_data = '')) then do
      return_code = 1      /* wrong number of arguments */
      exit return_code 
   end

   /* strip leading and trailing spaces */
   option_tag = strip(option_tag)
   option_data = strip(option_data)

   /* for debugging, write the data to a text file */
/*   date = date('N')
     time = time('N')
     outfile = 'c:\tcpip\etc\dhcpibm.out' 
     return_code = write_debug_file(option_tag, option_data, outfile)
     call lineout outfile
*/
   /* instantiate the option */
   Select

      when (option_tag = 9) then do          /* Default LPR Server */
         if (check_IPaddr(option_data)) then 
            return_code =  write_ini('\tcpos2.ini', 'LPR', 'LPR_SERVER',,
                                     option_data) 
         else return_code =  1               /* improper IP address */
      end

      when (option_tag = 200) then do        /* Default LPR Printer */
         data = hex2ascii_string(option_data)
         if (data \= '') then
            return_code = write_ini('\tcpos2.ini', 'LPR', 'LPR_PRINTER', data)
         else return_code = 1                /* invalid hex string */
      end

      otherwise 
         return_code = 1                     /* unsupported_option */
   end

   exit return_code


/**************************************************************************** 
 *  check_IPaddr (addr)
 *
 *  Check a dotted decimal IP address for valid format.
 *
 *  Return 0 if there are not 4 octets or the data is out of range or
 *  there are embedded blanks in the address, else return 1.
 *
 ****************************************************************************/
check_IPaddr:  Procedure

   parse arg addr
   addr = strip(addr)
   parse var addr octet.1 '.' octet.2 '.' octet.3 '.' octet.4 

   do pos = 1 to 4
      if (verify(octet.pos, '0123456789', N) \= 0) then
         return 0     /* not an unsigned number */
      if ((octet.pos < 0) | (octet.pos > 255))  then
         return 0
   end

   return 1

/* end check_IPaddr function */


/****************************************************************************
 *  hex2ascii_string (option_data)
 *
 *  Take a hex string in the form 'hex"nn nn nn ... ", verify it's a valid 
 *  hex string, and convert it to an ASCII character string.
 *
 *  Returns the string if the data is valid, a null string if not.    
 *
 ****************************************************************************/
hex2ascii_string:  Procedure
   parse arg 'hex' '"' data '"'
   if (\ DATATYPE(data, X)) then return ""
   data = strip(data, 'B', '09'x)

   return X2C(data)

/* hex2ascii_string function */


/**************************************************************************** 
 *  write_ini (inifilename, application, keyname, keydata) 
 *
 *  Write the specified key data to the INI file under the specified
 *  application and key name.
 *
 *  The INI file must already exist in the ETC directory as defined by the
 *  ETC environment variable.
 *
 ****************************************************************************/
write_ini:  Procedure

   parse arg inifile, app, keyname, keydata

   etc = value('ETC',,OS2ENVIRONMENT)
   if (etc = '') then 
      return 1
   else do
      if (keyname = 'DELETE:') then 
         Result = SysIni(etc||inifile, app, 'DELETE:') 
      else 
         Result = SysIni(etc||inifile, app, keyname, keydata) 
      if Result = 'ERROR:' then 
         return -1
      else 
         return 0
   end  

/* end write_ini function */


/**************************************************************************** 
 *  write_debug_file (option, option_data, outfile)
 *
 *  Write option number and data to a file.  For debugging use only.
 *
 ****************************************************************************/
write_debug_file:  Procedure

   parse arg option, data, outfile

   /* write the option number and data to a file */
   line = date('n') time('n')   'option: '||option||'   data:  '||data

   err = lineout(outfile, line)

   return err

/* end write_debug_file function */
