/* RegDump.cmd: Dump trap registers screens from dump image.
   Dump image may be diskette 1 of a trap set or a trapdump file on disk

    Copyright (c) 2000 Steven Levine and Associates, Inc.
    All rights reserved.

    $TLIB$: $ &(#) %n - Ver %v, %f $
    TLIB: $ $

    Revisions	14 Nov 00 SHL - Release

*/

signal on Error
signal on FAILURE name Error
signal on Halt
signal on NOTREADY name Error
signal on NOVALUE name Error
signal on SYNTAX name Error

call Initialize

Main:

  parse arg szFile szRest

  szMatch = xrange(' ', '7e'x) || '0d'x || '0a'x
  szCrLf = '0d'x || '0a'x
  szLfCr = '0a'x || '0d'x

  call DoWork szFile

  exit

/* end main */

/*=== DoWork(szInFile): Read dump file. Find trap screens. Write to stdout. ===*/

DoWork:

  parse arg szInFile

  if szInFile = '' then
    call Usage

  say 'Reading' szInFile
  say

  /* Read in 1st 512KB of dump image */

  if stream(szInFile, 'C', 'QUERY EXISTS') == '' then do
    return szInFile 'does not exist.'
  end

  call stream szInFile, 'C', 'OPEN READ'

  szBuf = charin(szInFile,,1024 * 512)

  call stream szInFile, 'C', 'CLOSE'

  /* Set to 1 to capture buffer image */

  if 0 then do
    szOutFile = 'tmpdump.out'
    call SysFileDelete szOutFile
    call stream szOutFile, 'C', 'OPEN WRITE'
    call charout szOutFile, szBuf
    call stream szOutFile, 'C', 'CLOSE'
    say szOutFile 'written'
    exit
  end

  say 'Trap screen 1'
  say

  call DumpLines 'P1='
  call DumpLines ''

  say 'Trap screen 2'
  say

  c = pos('P1=', szBuf)

  c = c - 18
  if c < 1 then
     call Fatal 'Cannot find trap screen 2'

  szBuf = substr(szBuf, c)

  call DumpLines ''
  call DumpLines ''
  call DumpLines 'P1='

  say 'Trap screen 3'
  say

  call DumpLines '##'
  call DumpLines '', 'N'        /* This seems to be a dup */
  call DumpLines '', 'N'
  call FixLfCr
  call DumpLines ''

  return

/* end DoWork */

/*=== DumpLines(Tag, Noprint): Find printable area and write to stdout ===*/

DumpLines:

  szTag = arg(1)
  fNoPrint = arg(2)

  /* Find screen image start */

  if szTag == '' then
    c = 1                               /* Start here */
  else do
    c = pos(szTag, szBuf)
    if c = 0 then
     call Fatal 'Cannot find tag' szTag
  end

  if c > 1 then
    szBuf = substr(szBuf, c)

  c = verify(szBuf, szMatch)    /* Print printable range */

  c = c - 1
  if c < 1 then
   call Fatal 'Can not size print area'

  sz = substr(szBuf, 1, c)

  if fNoPrint = '' then
    say sz

  szBuf = substr(szBuf, c + 1)

  c = verify(szBuf, szMatch, 'M')
  if c \= 0 then
    szBuf = substr(szBuf, c)

  return

/* end DumpLines */

/*=== FixLfCr(void): Correct backwards Lf/Cr ===*/

FixLfCr:

  c = pos(szLfCr, szBuf)
  do while c \= 0
    szBuf = delstr(szBuf, c, 2)
    szBuf = insert(szCrLf, szBuf, c - 1)
    c = pos(szLfCr, szBuf, c + 2)
  end

  return

/* end FixLfCr */

/*=== Initialize: Intialize globals ===*/

Initialize:

  call LoadFuncs

  /* Set:
      Gbl.szCmdName
      Gbl.szTmpDir - default = TMP or TEMP, trailing \ forced

      Opt.fDebug - default = 0
      Opt.fVerbose - default = 0
  */

  parse source . . Gbl.szCmdName
  Gbl.szCmdName = filespec('N', Gbl.szCmdName)
  c = pos('.', Gbl.szCmdName)
  if c > 1 then
    Gbl.szCmdName = substr(Gbl.szCmdName, 1, c - 1)

  Gbl.szTmpDir = value('TMP',,'OS2ENVIRONMENT')
  if Gbl.szTmpDir \= '' & right(Gbl.szTmpDir, 1) \= ':' & right(Gbl.szTmpDir, 1) \= '\' then
    Gbl.szTmpDir = Gbl.szTmpDir'\'

  Opt.fDebug = 0
  Opt.fVerbose = 0

  return

/* end Initialize */

/*=== UsageErr(szMsg): ... ===*/

UsageErr:

  parse arg szMsg

  say szMsg

  /* Drop thru to Usage */

/* end UsageErr */

/*=== Usage: Display help ===*/

Usage:

  say 'Usage:' Gbl.szCmdName 'trapdumpfilename'

  exit 255

/* end Usage */

/*========================================================================== */
/*=== Standard functions - Delete unused.  Move modified above this mark === */
/*========================================================================== */

/*=== Error: Trap ERROR, FAILURE etc. conditions ===*/

Error:

  /*=== Returns szCondition or Exits ===*/

  say
  parse source . . szThisCmd
  say condition('C') 'signaled at' SIGL 'of' szThisCmd
  drop szThisCmd
  say 'Source =' sourceline(SIGL)
  call SysSleep 2
  if condition('I') == 'CALL' then do
    szCondition = condition('C')
    say 'Returning'
    return
  end
  else do
    trace '?A'
    say 'Exiting'
    call SysSleep 2
    exit 255
  end

/* end Error */

/*=== Fatal: Report fatal error and exit ===*/

Fatal:

  parse arg szMsg

  say
  say Gbl.szCmdName':' szMsg

  exit 254

/* end Fatal */

/*=== Halt: Trap HALT condition ===*/

Halt:

  /*=== Returns szCondition or Exits ===*/

  say
  parse source . . szThisCmd
  say condition('C') 'signaled at' SIGL 'of' szThisCmd
  drop szThisCmd
  say 'Source = ' sourceline(SIGL)
  call SysSleep 2
  if condition('I') == 'CALL' then do
    szCondition = condition('C')
    say 'Returning'
    return
  end
  else do
   say 'Exiting'
   exit
  end

/* end Halt */

/*=== LoadFuncs: Load fuctions ===*/

LoadFuncs:

  /* Add all Rexx functions */
  if RxFuncQuery('SysLoadFuncs') then do
    call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
    if RESULT then do
      say 'Cannot load SysLoadFuncs'
      exit
    end
    call SysLoadFuncs
  end /* end do */

  return

/* end LoadFuncs */

/* The end */
