/* ===========================================================================
 *  C Example Program : NI-488 Direct-Entry Points
 *
 *  This sample program is for reference only. It can only be expected to
 *  function with a Fluke 45 Digital Multimeter. 
 *
 *  This program reads 10 measurements from the Fluke 45 and averages
 *  the sum.
 * ===========================================================================
 */

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <conio.h>

#include "decl-32.h"

void gpiberr(char *msg);                 /* Error function                  */
void dvmerr(char *msg, char spr);        /* Fluke 45 error function         */

  char     rd[512],                      /* read data buffer                */
           spr;                          /* serial poll response byte       */
  int      dvm,                          /* device number                   */
           m;                            /* FOR loop counter                */
  double   sum;                          /* Accumulator of measurements     */

/****************************************
 ********  CODE TO ACCESS GPIB-32.DLL
 ***********************************************/

static PROC Pibclr ;
static PROC Pibdev ;
static PROC Pibonl ;
static PROC Pibrd  ;
static PROC Pibrsp ;
static PROC Pibtrg ;
static PROC Pibwait;
static PROC Pibwrt ;

/*
 *    This is private data for the language interface only so it is
 *    defined as 'static'.
 */
static HINSTANCE Gpib32Lib = NULL;
static int  *Pibsta;
static int  *Piberr;
static long *Pibcntl;

                                
static BOOLEAN LoadDll (void)   
{                               
   /*
    *  Call LoadLibrary to load the 32-bit GPIB DLL.  Save the handle
    *  into the global 'Gpib32Lib'.
    */
   Gpib32Lib = LoadLibrary ("GPIB-32.DLL");

   if (!Gpib32Lib)  {
      /*
       *    The LoadLibrary call failed, return with an error.
       */
      return FALSE;
   }
   

   /*
    *    OK, the GPIB library is loaded.  Let's get a pointer to the
    *    requested function.  If the GetProcAddress call fails, then
    *    return with an error.
    */
   Pibsta  = (int *) GetProcAddress(Gpib32Lib, (LPCSTR)"user_ibsta");
   Piberr  = (int *) GetProcAddress(Gpib32Lib, (LPCSTR)"user_iberr"); 
   Pibcntl = (long *)GetProcAddress(Gpib32Lib, (LPCSTR)"user_ibcnt"); 
   Pibclr  = (PROC)GetProcAddress(Gpib32Lib, (LPCSTR)"ibclr"); 
   Pibdev  = (PROC)GetProcAddress(Gpib32Lib, (LPCSTR)"ibdev"); 
   Pibonl  = (PROC)GetProcAddress(Gpib32Lib, (LPCSTR)"ibonl"); 
   Pibrd   = (PROC)GetProcAddress(Gpib32Lib, (LPCSTR)"ibrd"); 
   Pibrsp  = (PROC)GetProcAddress(Gpib32Lib, (LPCSTR)"ibrsp"); 
   Pibtrg  = (PROC)GetProcAddress(Gpib32Lib, (LPCSTR)"ibtrg"); 
   Pibwait = (PROC)GetProcAddress(Gpib32Lib, (LPCSTR)"ibwait"); 
   Pibwrt  = (PROC)GetProcAddress(Gpib32Lib, (LPCSTR)"ibwrt"); 


   if ((Pibsta  == NULL) || 
       (Piberr  == NULL) || 
       (Pibcntl == NULL) || 
       (Pibclr  == NULL) || 
       (Pibdev  == NULL) || 
       (Pibonl  == NULL) || 
       (Pibrd   == NULL) || 
       (Pibrsp  == NULL) || 
       (Pibtrg  == NULL) || 
       (Pibwait == NULL) || 
       (Pibwrt  == NULL))  {

      FreeLibrary (Gpib32Lib);
      Gpib32Lib = NULL;
      return FALSE;
   }
   else  {
      return TRUE;
   }

}  /* end of LoadDll */


static void FreeDll (void)   
{
   FreeLibrary (Gpib32Lib);
   Gpib32Lib = NULL;
   return;
}


/*****************************
 ************  BEGIN MAIN
 *************************************/

void __cdecl main() {

    if (!LoadDll())  {
       printf ("Unable to correctly access the 32-bit GPIB DLL.\n");
       return;
    }

    printf("Read 10 measurements from the Fluke 45...\n");
    printf("\n");

/*
 *  Assign a unique identifier to the Fluke 45 and store in the variable
 *  DVM.  IBDEV opens an available device and assigns it to access GPIB0
 *  with a primary address of 1, a secondary address of 0, a timeout of
 *  10 seconds, the END message enabled, and the EOS mode disabled.
 *  If DVM is less than zero, call GPIBERR with an error message.
 */

    dvm = (*Pibdev) (0, 1, 0, T10s, 1, 0);
    if (dvm < 0) gpiberr("ibdev Error");

/*
 *  Clear the internal or device functions of the Fluke 45.
 */

    (*Pibclr) (dvm);
    if ((*Pibsta) & ERR) gpiberr("ibclr Error");

/*
 *  Reset the Fluke 45 by issuing the reset (*RST) command.  Instruct the
 *  Fluke 45 to measure the volts alternating current (VAC) using auto-ranging
 *  (AUTO), to wait for a trigger from the GPIB interface board (TRIGGER 2),
 *  and to assert the IEEE-488 Service Request line, SRQ, when the measurement
 *  has been completed and the Fluke 45 is ready to send the result (*SRE 16).
 */

    (*Pibwrt) (dvm, "*RST; VAC; AUTO; TRIGGER 2; *SRE 16", 35L);
    if ((*Pibsta) & ERR) gpiberr("ibwrt Error");

/*  Initialize the accumulator of the 10 measurements to zero.              */

    sum = 0.0;

/*
 *  Establish FOR loop to read the 10 measuements.  The variable m will
 *  serve as the counter of the FOR loop.
 */

    for (m=0; m < 10 ; m++) {

     /*
      *  Trigger the Fluke 45.
      */

         (*Pibtrg) (dvm);
         if ((*Pibsta) & ERR) gpiberr("ibtrg Error");

     /*
      *  Request the triggered measurement by sending the instruction
      *  "VAL1?".
      */

         (*Pibwrt) (dvm, "VAL1?", 5L);
         if ((*Pibsta) & ERR) gpiberr("ibwrt Error");

     /*
      *  Wait for the Fluke 45 to request service (RQS) or wait for the
      *  Fluke 45 to timeout(TIMO).  The default timeout period is 10 seconds.
      *  RQS is detected by bit position 11 (hex 800).   TIMO is detected
      *  by bit position 14 (hex 4000).  These status bits are listed under
      *  the NI-488 function IBWAIT in the Software Reference Manual.
      */

         (*Pibwait) (dvm, TIMO|RQS);
         if ((*Pibsta) & (ERR|TIMO)) gpiberr("ibwait Error");

     /*
      *  Read the Fluke 45 serial poll status byte.
      */

         (*Pibrsp) (dvm, &spr);
         if ((*Pibsta) & ERR) gpiberr("ibrsp Error");

     /*
      *  If the returned status byte is hex 50, the Fluke 45 has valid data
      *  to send; otherwise, it has a fault condition to report.
      */

         if (spr != 0x50) dvmerr("Fluke 45 Error", spr);

     /*
      *  Read the Fluke 45 measurement.
      */

         (*Pibrd) (dvm, rd, 10L);
         if ((*Pibsta) & ERR) gpiberr("ibrd Error");

     /*
      *  Use the null character to mark the end of the data received
      *  in the array RD.  Print the measurement received from the
      *  Fluke 45.
      */

         rd[(*Pibcntl)] = '\0';
         printf("Reading :  %s\n", rd);

     /*  Convert the variable RD to its numeric value and add to the
      *  accumulator.
      */

         sum = sum + atof(rd);

    }  /*  Continue FOR loop until 10 measurements are read.   */

/*  Print the average of the 10 readings.                                   */

    printf("   The average of the 10 readings is : %f\n", sum/10);

/*  Call the ibonl function to disable the hardware and software.           */

    (*Pibonl) (dvm, 0);

    FreeDll();
}


void gpiberr(char *msg) {

    printf ("%s\n", msg);
    printf ("iberr = %d \n", (*Piberr));

    (*Pibonl) (0, 0); /* Disable hardware and software */

    FreeDll();
    exit(1);                                  /* Abort program */

}


void dvmerr(char *msg,char spr) {

    printf ("%s\n", msg);
    printf("Status byte = %x\n", spr);

    (*Pibonl) (0, 0); /* Disable hardware and software */

    FreeDll();
    exit(1);                                  /* Abort program */
}



