/* C Example Program - NI-488 Board Level Functions	  */
/*
 This sample program is for reference only and can only be expected to
 function with a Fluke 8840 Digital Multimeter that has been properly
 configured in IBCONF.
*/

#include <stdio.h>
#include "decl.h"

/* Application program variables passed to GPIB functions */

char rd[512];                 /* read data buffer         */
int  bd;                      /* board or device number   */
char cmd[4];		      /* command buffer 	  */

main() {

 /* Assign unique identifier to board 0 and store in variable bd.  Check for
    error. (IBFIND error = negative value returned.)	  */

    if ((bd = ibfind ("GPIB0")) < 0) finderr();

 /* Send the Interface Clear (IFC) message to all devices.  Check for error on
    each GPIB call to be safe.	Note that GPIB status is available both through
    global variable ibsta and through the return values of all GPIB functions
    except IBFIND.							      */

    if (ibsic (bd) & ERR) error();

 /* Turn on the Remote Enable (REN) signal.        */

    if (ibsre (bd,1) & ERR) error();

/* Inhibit front panel control with the Local Lockout (LLO) command (hex 11).
 * Place the device in remote mode by addressing it to listen. The listen
 * address of device three is its primary address, 3, plus hex 20.  This is an
 * ASCII "#".  Send the Device Clear (DCL) message (hex 14) to clear internal
 * device functions.  Finally, address the GPIB0 to talk by sending its talk
 * address, the pad 0 plus hex 40, or ASCII "@".  These commands can be found
 * in Appendix A of the Software Reference Manual.
 */

    ibcmd (bd,"\021#\024@",4);
    if (ibsta & ERR) error();

 /* Write the function, range, and trigger source instructions to the device.
  * This string is meaningful to the Fluke 8840 Digital Multimeter.
  */
    ibwrt (bd,"F3R7T3",6);
    if (ibsta & ERR) error();

/* Send the GET message (hex 8) to trigger a measurement reading.

    ibcmd (bd,"\010",1);
    if (ibsta & ERR) error();

/* Wait for the device to set SRQ (hex 1000) or for a timeout (hex 4000).
 * These status bits are listed in Section Four of the Software Reference
 * manual.  If the current time limit is too short, use IBTMO to change it.
 */

   if (ibwait (bd,TIMO|SRQI) & (ERR|TIMO)) error();

/* If neither a timeout nor an error occurred, IBWAIT must have returned
 * on SRQ.  Next do a serial poll.  First unaddress bus devices by sending
 * the untalk (UNT) command (ASCII "_", or hex 5F) and the unlisten (UNL)
 * command (ASCII "?", or hex 3F).  Then send the Serial Poll Enable (SPE)
 * command (hex 18) and the device's talk address (device three's pad, 3,
 * plus hex 40, or ASCII "C") and the GPIB0 listen address (pad 0 plus hex 20,
 * or ASCII space).  These commands can be found in Appendix A of the Software
 * Reference Manual.
 */

    ibcmd (bd,"?_\030C ",5);		/* UNT UNL SPE TAD MLA */
    if (ibsta & ERR) error();

 /* Now read the status byte.  If it is 0xC0, the device has valid data to send;
    otherwise it has a fault condition to report.			      */

    ibrd (bd,rd,1);
    if (ibsta & ERR) error();
    if ((rd[0] & 0xFF) != 0xC0)  deverr();

 /* If more than one device were attached to the bus, it would be necessary to
    explicitly check the 0x40 bit of the device status word to be sure that
    another device had not been responsible for asserting SRQ.		      */

/* Complete the serial poll by sending the Serial Poll Disable (SPD) message
 * (hex 19). This command can be found in Appendix A of the Software
 * Reference Manual.
 */

    if (ibcmd (bd,"\031",1) & ERR) error();

 /* Since the device and GPIB board are still addressed to talk and listen, the
    measurement can be read as follows. 				      */

    if (ibrd (bd,rd,16) & ERR) error();

 /* To close out a programming sequence, send IFC to initialize the bus and call
    the IBONL function to disable the hardware and software. */

    ibsic (bd);
    ibonl (bd,0);
}


finderr() {
/* This routine would notify you that the IBFIND call failed, and refer you to
   the handler software configuration procedures.			      */

   printf("Ibfind error; does device or board\n");
   printf("name given match configuration name?\n");
   }


error() {
/* An error checking routine at this location would, among other things, check
   iberr to determine the exact cause of the error condition and then take
   action appropriate to the application.  For errors during data transfers,
   ibcnt may be examined to determine the actual number of bytes transferred. */

   printf("GPIB function call error:\n");
   printf("ibsta=0x%x, iberr=%d, ibcnt=%d\n",ibsta,iberr,ibcnt);
   }

deverr() {
/* A routine at this location would analyze the fault code returned in the
   device's status byte and take appropriate action.                          */

   printf("Device error\n");
   printf("ibsta=0x%x, iberr=%d, ibcnt=%d\n",ibsta,iberr,ibcnt);
   }
